/**
 * Represents a length of time.
 *
 * The amount can be specified either as a literal value (e.g: `10`) which
 * cannot be negative, or as an unresolved number token.
 *
 * When the amount is passed as a token, unit conversion is not possible.
 *
 * @stability stable
 */
export declare class Duration {
    /**
     * Create a Duration representing an amount of milliseconds.
     *
     * @param amount the amount of Milliseconds the `Duration` will represent.
     * @returns a new `Duration` representing `amount` ms.
     * @stability stable
     */
    static millis(amount: number): Duration;
    /**
     * Create a Duration representing an amount of seconds.
     *
     * @param amount the amount of Seconds the `Duration` will represent.
     * @returns a new `Duration` representing `amount` Seconds.
     * @stability stable
     */
    static seconds(amount: number): Duration;
    /**
     * Create a Duration representing an amount of minutes.
     *
     * @param amount the amount of Minutes the `Duration` will represent.
     * @returns a new `Duration` representing `amount` Minutes.
     * @stability stable
     */
    static minutes(amount: number): Duration;
    /**
     * Create a Duration representing an amount of hours.
     *
     * @param amount the amount of Hours the `Duration` will represent.
     * @returns a new `Duration` representing `amount` Hours.
     * @stability stable
     */
    static hours(amount: number): Duration;
    /**
     * Create a Duration representing an amount of days.
     *
     * @param amount the amount of Days the `Duration` will represent.
     * @returns a new `Duration` representing `amount` Days.
     * @stability stable
     */
    static days(amount: number): Duration;
    /**
     * Parse a period formatted according to the ISO 8601 standard.
     *
     * @param duration an ISO-formtted duration to be parsed.
     * @returns the parsed `Duration`.
     * @see https://www.iso.org/fr/standard/70907.html
     * @stability stable
     */
    static parse(duration: string): Duration;
    private readonly amount;
    private readonly unit;
    private constructor();
    /**
     * Add two Durations together.
     *
     * @stability stable
     */
    plus(rhs: Duration): Duration;
    /**
     * Return the total number of milliseconds in this Duration.
     *
     * @returns the value of this `Duration` expressed in Milliseconds.
     * @stability stable
     */
    toMilliseconds(opts?: TimeConversionOptions): number;
    /**
     * Return the total number of seconds in this Duration.
     *
     * @returns the value of this `Duration` expressed in Seconds.
     * @stability stable
     */
    toSeconds(opts?: TimeConversionOptions): number;
    /**
     * Return the total number of minutes in this Duration.
     *
     * @returns the value of this `Duration` expressed in Minutes.
     * @stability stable
     */
    toMinutes(opts?: TimeConversionOptions): number;
    /**
     * Return the total number of hours in this Duration.
     *
     * @returns the value of this `Duration` expressed in Hours.
     * @stability stable
     */
    toHours(opts?: TimeConversionOptions): number;
    /**
     * Return the total number of days in this Duration.
     *
     * @returns the value of this `Duration` expressed in Days.
     * @stability stable
     */
    toDays(opts?: TimeConversionOptions): number;
    /**
     * Return an ISO 8601 representation of this period.
     *
     * @returns a string starting with 'P' describing the period
     * @see https://www.iso.org/fr/standard/70907.html
     * @stability stable
     */
    toIsoString(): string;
    /**
     * (deprecated) Return an ISO 8601 representation of this period.
     *
     * @returns a string starting with 'P' describing the period
     * @see https://www.iso.org/fr/standard/70907.html
     * @deprecated Use `toIsoString()` instead.
     */
    toISOString(): string;
    /**
     * Turn this duration into a human-readable string.
     *
     * @stability stable
     */
    toHumanString(): string;
    /**
     * Returns a string representation of this `Duration` that is also a Token that cannot be successfully resolved.
     *
     * This
     * protects users against inadvertently stringifying a `Duration` object, when they should have called one of the
     * `to*` methods instead.
     *
     * @stability stable
     */
    toString(): string;
    /**
     * Return the duration in a set of whole numbered time components, ordered from largest to smallest
     *
     * Only components != 0 will be returned.
     *
     * Can combine millis and seconds together for the benefit of toIsoString,
     * makes the logic in there simpler.
     */
    private components;
}
/**
 * Options for how to convert time to a different unit.
 *
 * @stability stable
 */
export interface TimeConversionOptions {
    /**
     * If `true`, conversions into a larger time unit (e.g. `Seconds` to `Minutes`) will fail if the result is not an integer.
     *
     * @default true
     * @stability stable
     */
    readonly integral?: boolean;
}
