"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Lazy = void 0;
const debug_1 = require("./debug");
const stack_trace_1 = require("./stack-trace");
const token_1 = require("./token");
/**
 * Lazily produce a value.
 *
 * Can be used to return a string, list or numeric value whose actual value
 * will only be calculated later, during synthesis.
 *
 * @stability stable
 */
class Lazy {
    /**
     * (deprecated) Defer the calculation of a string value to synthesis time.
     *
     * Use this if you want to render a string to a template whose actual value depends on
     * some state mutation that may happen after the construct has been created.
     *
     * If you are simply looking to force a value to a `string` type and don't need
     * the calculation to be deferred, use `Token.asString()` instead.
     *
     * @deprecated Use `Lazy.string()` or `Lazy.uncachedString()` instead.
     */
    static stringValue(producer, options = {}) {
        return token_1.Token.asString(new LazyString(producer, false), options);
    }
    /**
     * Defer the one-time calculation of a string value to synthesis time.
     *
     * Use this if you want to render a string to a template whose actual value depends on
     * some state mutation that may happen after the construct has been created.
     *
     * If you are simply looking to force a value to a `string` type and don't need
     * the calculation to be deferred, use `Token.asString()` instead.
     *
     * The inner function will only be invoked once, and the resolved value
     * cannot depend on the Stack the Token is used in.
     *
     * @stability stable
     */
    static string(producer, options = {}) {
        return token_1.Token.asString(new LazyString(producer, true), options);
    }
    /**
     * Defer the calculation of a string value to synthesis time.
     *
     * Use of this function is not recommended; unless you know you need it for sure, you
     * probably don't. Use `Lazy.string()` instead.
     *
     * The inner function may be invoked multiple times during synthesis. You
     * should only use this method if the returned value depends on variables
     * that may change during the Aspect application phase of synthesis, or if
     * the value depends on the Stack the value is being used in. Both of these
     * cases are rare, and only ever occur for AWS Construct Library authors.
     *
     * @stability stable
     */
    static uncachedString(producer, options = {}) {
        return token_1.Token.asString(new LazyString(producer, false), options);
    }
    /**
     * (deprecated) Defer the one-time calculation of a number value to synthesis time.
     *
     * Use this if you want to render a number to a template whose actual value depends on
     * some state mutation that may happen after the construct has been created.
     *
     * If you are simply looking to force a value to a `number` type and don't need
     * the calculation to be deferred, use `Token.asNumber()` instead.
     *
     * @deprecated Use `Lazy.number()` or `Lazy.uncachedNumber()` instead.
     */
    static numberValue(producer) {
        return token_1.Token.asNumber(new LazyNumber(producer, false));
    }
    /**
     * Defer the one-time calculation of a number value to synthesis time.
     *
     * Use this if you want to render a number to a template whose actual value depends on
     * some state mutation that may happen after the construct has been created.
     *
     * If you are simply looking to force a value to a `number` type and don't need
     * the calculation to be deferred, use `Token.asNumber()` instead.
     *
     * The inner function will only be invoked once, and the resolved value
     * cannot depend on the Stack the Token is used in.
     *
     * @stability stable
     */
    static number(producer) {
        return token_1.Token.asNumber(new LazyNumber(producer, true));
    }
    /**
     * Defer the calculation of a number value to synthesis time.
     *
     * Use of this function is not recommended; unless you know you need it for sure, you
     * probably don't. Use `Lazy.number()` instead.
     *
     * The inner function may be invoked multiple times during synthesis. You
     * should only use this method if the returned value depends on variables
     * that may change during the Aspect application phase of synthesis, or if
     * the value depends on the Stack the value is being used in. Both of these
     * cases are rare, and only ever occur for AWS Construct Library authors.
     *
     * @stability stable
     */
    static uncachedNumber(producer) {
        return token_1.Token.asNumber(new LazyNumber(producer, false));
    }
    /**
     * (deprecated) Defer the one-time calculation of a list value to synthesis time.
     *
     * Use this if you want to render a list to a template whose actual value depends on
     * some state mutation that may happen after the construct has been created.
     *
     * If you are simply looking to force a value to a `string[]` type and don't need
     * the calculation to be deferred, use `Token.asList()` instead.
     *
     * @deprecated Use `Lazy.list()` or `Lazy.uncachedList()` instead.
     */
    static listValue(producer, options = {}) {
        return token_1.Token.asList(new LazyList(producer, false, options), options);
    }
    /**
     * Defer the calculation of a list value to synthesis time.
     *
     * Use of this function is not recommended; unless you know you need it for sure, you
     * probably don't. Use `Lazy.list()` instead.
     *
     * The inner function may be invoked multiple times during synthesis. You
     * should only use this method if the returned value depends on variables
     * that may change during the Aspect application phase of synthesis, or if
     * the value depends on the Stack the value is being used in. Both of these
     * cases are rare, and only ever occur for AWS Construct Library authors.
     *
     * @stability stable
     */
    static uncachedList(producer, options = {}) {
        return token_1.Token.asList(new LazyList(producer, false, options), options);
    }
    /**
     * Defer the one-time calculation of a list value to synthesis time.
     *
     * Use this if you want to render a list to a template whose actual value depends on
     * some state mutation that may happen after the construct has been created.
     *
     * If you are simply looking to force a value to a `string[]` type and don't need
     * the calculation to be deferred, use `Token.asList()` instead.
     *
     * The inner function will only be invoked once, and the resolved value
     * cannot depend on the Stack the Token is used in.
     *
     * @stability stable
     */
    static list(producer, options = {}) {
        return token_1.Token.asList(new LazyList(producer, true, options), options);
    }
    /**
     * (deprecated) Defer the one-time calculation of an arbitrarily typed value to synthesis time.
     *
     * Use this if you want to render an object to a template whose actual value depends on
     * some state mutation that may happen after the construct has been created.
     *
     * @deprecated Use `Lazy.any()` or `Lazy.uncachedAny()` instead.
     */
    static anyValue(producer, options = {}) {
        return new LazyAny(producer, false, options);
    }
    /**
     * Defer the one-time calculation of an arbitrarily typed value to synthesis time.
     *
     * Use this if you want to render an object to a template whose actual value depends on
     * some state mutation that may happen after the construct has been created.
     *
     * The inner function will only be invoked one time and cannot depend on
     * resolution context.
     *
     * @stability stable
     */
    static any(producer, options = {}) {
        return new LazyAny(producer, true, options);
    }
    /**
     * Defer the calculation of an untyped value to synthesis time.
     *
     * Use of this function is not recommended; unless you know you need it for sure, you
     * probably don't. Use `Lazy.any()` instead.
     *
     * The inner function may be invoked multiple times during synthesis. You
     * should only use this method if the returned value depends on variables
     * that may change during the Aspect application phase of synthesis, or if
     * the value depends on the Stack the value is being used in. Both of these
     * cases are rare, and only ever occur for AWS Construct Library authors.
     *
     * @stability stable
     */
    static uncachedAny(producer, options = {}) {
        return new LazyAny(producer, false, options);
    }
    constructor() {
    }
}
exports.Lazy = Lazy;
class LazyBase {
    constructor(producer, cache) {
        this.producer = producer;
        this.cache = cache;
        // Stack trace capture is conditionned to `debugModeEnabled()`, because
        // lazies can be created in a fairly thrashy way, and the stack traces are
        // large and slow to obtain; but are mostly useful only when debugging a
        // resolution issue.
        this.creationStack = debug_1.debugModeEnabled()
            ? stack_trace_1.captureStackTrace(this.constructor)
            : [`Execute again with ${debug_1.CDK_DEBUG}=true to capture stack traces`];
    }
    resolve(context) {
        var _a;
        if (this.cache) {
            return (_a = this._cached) !== null && _a !== void 0 ? _a : (this._cached = this.producer.produce(context));
        }
        else {
            return this.producer.produce(context);
        }
    }
    toString() {
        return token_1.Token.asString(this);
    }
    /**
     * Turn this Token into JSON
     *
     * Called automatically when JSON.stringify() is called on a Token.
     */
    toJSON() {
        return '<unresolved-lazy>';
    }
}
class LazyString extends LazyBase {
}
class LazyNumber extends LazyBase {
}
class LazyList extends LazyBase {
    constructor(producer, cache, options = {}) {
        super(producer, cache);
        this.options = options;
    }
    resolve(context) {
        const resolved = super.resolve(context);
        if ((resolved === null || resolved === void 0 ? void 0 : resolved.length) === 0 && this.options.omitEmpty) {
            return undefined;
        }
        return resolved;
    }
}
class LazyAny extends LazyBase {
    constructor(producer, cache, options = {}) {
        super(producer, cache);
        this.options = options;
    }
    resolve(context) {
        const resolved = super.resolve(context);
        if (Array.isArray(resolved) && resolved.length === 0 && this.options.omitEmptyArray) {
            return undefined;
        }
        return resolved;
    }
}
//# sourceMappingURL=data:application/json;base64,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