"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.TokenizedStringFragments = void 0;
const token_1 = require("./token");
/**
 * Fragments of a concatenated string containing stringified Tokens.
 *
 * @stability stable
 */
class TokenizedStringFragments {
    constructor() {
        this.fragments = new Array();
    }
    /**
     * @stability stable
     */
    get firstToken() {
        const first = this.fragments[0];
        if (first.type === 'token') {
            return first.token;
        }
        return undefined;
    }
    /**
     * @stability stable
     */
    get firstValue() {
        return fragmentValue(this.fragments[0]);
    }
    /**
     * @stability stable
     */
    get length() {
        return this.fragments.length;
    }
    /**
     * @stability stable
     */
    addLiteral(lit) {
        this.fragments.push({ type: 'literal', lit });
    }
    /**
     * @stability stable
     */
    addToken(token) {
        this.fragments.push({ type: 'token', token });
    }
    /**
     * @stability stable
     */
    addIntrinsic(value) {
        this.fragments.push({ type: 'intrinsic', value });
    }
    /**
     * Return all Tokens from this string.
     *
     * @stability stable
     */
    get tokens() {
        const ret = new Array();
        for (const f of this.fragments) {
            if (f.type === 'token') {
                ret.push(f.token);
            }
        }
        return ret;
    }
    /**
     * Apply a transformation function to all tokens in the string.
     *
     * @stability stable
     */
    mapTokens(mapper) {
        const ret = new TokenizedStringFragments();
        for (const f of this.fragments) {
            switch (f.type) {
                case 'literal':
                    ret.addLiteral(f.lit);
                    break;
                case 'token':
                    const mapped = mapper.mapToken(f.token);
                    if (token_1.isResolvableObject(mapped)) {
                        ret.addToken(mapped);
                    }
                    else {
                        ret.addIntrinsic(mapped);
                    }
                    break;
                case 'intrinsic':
                    ret.addIntrinsic(f.value);
                    break;
            }
        }
        return ret;
    }
    /**
     * Combine the string fragments using the given joiner.
     *
     * If there are any
     *
     * @stability stable
     */
    join(concat) {
        if (this.fragments.length === 0) {
            return concat.join(undefined, undefined);
        }
        if (this.fragments.length === 1) {
            return this.firstValue;
        }
        const values = this.fragments.map(fragmentValue);
        while (values.length > 1) {
            const prefix = values.splice(0, 2);
            values.splice(0, 0, concat.join(prefix[0], prefix[1]));
        }
        return values[0];
    }
}
exports.TokenizedStringFragments = TokenizedStringFragments;
/**
 * Resolve the value from a single fragment
 *
 * If the fragment is a Token, return the string encoding of the Token.
 */
function fragmentValue(fragment) {
    switch (fragment.type) {
        case 'literal': return fragment.lit;
        case 'token': return fragment.token.toString();
        case 'intrinsic': return fragment.value;
    }
}
//# sourceMappingURL=data:application/json;base64,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