# -*- coding: utf-8 -*-
import json
import logging
import logging.handlers
import socket
import sys
import time

from .reporter import Reporter

DEFAULT_SYSLOG_ADDRESS = "/dev/log"
DEFAULT_SYSLOG_SOCKTYPE = socket.SOCK_DGRAM
DEFAULT_SYSLOG_FACILITY = logging.handlers.SysLogHandler.LOG_USER


class SysLogReporter(Reporter):

    """
    Syslog is a way for network devices to send event messages to a logging server
    """

    def __init__(
        self,
        registry=None,
        reporting_interval=5,
        tag="applipy_metrics",
        clock=time,
        address=DEFAULT_SYSLOG_ADDRESS,
        socktype=DEFAULT_SYSLOG_SOCKTYPE,
        facility=DEFAULT_SYSLOG_FACILITY,
    ):
        super(SysLogReporter, self).__init__(registry, reporting_interval)
        self.clock = clock

        handler = logging.handlers.SysLogHandler(
            address=address, facility=facility, socktype=socktype
        )
        handler.append_nul = False

        if tag is not None and tag != "":
            if sys.version_info >= (3, 3):
                handler.ident = tag + ": "
            else:
                formatter = logging.Formatter("{}: %(message)s".format(tag))
                handler.setFormatter(formatter)

        logger = logging.getLogger("applipy_metrics")
        logger.setLevel(logging.INFO)
        logger.addHandler(handler)
        self.logger = logger

    def report_now(self, registry=None, timestamp=None):
        metrics = self._collect_metrics(registry or self.registry, timestamp)
        if metrics:
            self.logger.info(metrics)

    def _collect_metrics(self, registry, timestamp=None):
        timestamp = timestamp or int(round(self.clock.time()))

        metrics_data = {"timestamp": timestamp}
        metrics = registry.dump_metrics()
        for metric_name, metric in metrics.items():
            for metric_key, metric_value in metric.items():
                metrics_data["{}.{}".format(metric_name, metric_key)] = metric_value
        result = json.dumps(metrics_data, sort_keys=True)
        return result
