import numpy as np

import materia

from .dataseries import RelativeSPDSpectrum
from .simmons_d_series import SimmonsDSeriesPCA0, SimmonsDSeriesPCA1, SimmonsDSeriesPCA2


class CIEIlluminantA(RelativeSPDSpectrum):
    def __init__(self, normalize_to=100):
        # data taken from https://web.archive.org/web/20170131100357/http://files.cie.co.at/204.xls
        x = materia.Qty(value=np.linspace(300, 780, 97), unit=materia.nanometer)

        spd = [
            0.930483,
            1.128210,
            1.357690,
            1.622190,
            1.925080,
            2.269800,
            2.659810,
            3.098610,
            3.589680,
            4.136480,
            4.742380,
            5.410700,
            6.144620,
            6.947200,
            7.821350,
            8.769800,
            9.795100,
            10.899600,
            12.085300,
            13.354300,
            14.708000,
            16.148000,
            17.675300,
            19.290700,
            20.995000,
            22.788300,
            24.670900,
            26.642500,
            28.702700,
            30.850800,
            33.085900,
            35.406800,
            37.812100,
            40.300200,
            42.869300,
            45.517400,
            48.242300,
            51.041800,
            53.913200,
            56.853900,
            59.861100,
            62.932000,
            66.063500,
            69.252500,
            72.495900,
            75.790300,
            79.132600,
            82.519300,
            85.947000,
            89.412400,
            92.912000,
            96.442300,
            100.000000,
            103.582000,
            107.184000,
            110.803000,
            114.436000,
            118.080000,
            121.731000,
            125.386000,
            129.043000,
            132.697000,
            136.346000,
            139.988000,
            143.618000,
            147.235000,
            150.836000,
            154.418000,
            157.979000,
            161.516000,
            165.028000,
            168.510000,
            171.963000,
            175.383000,
            178.769000,
            182.118000,
            185.429000,
            188.701000,
            191.931000,
            195.118000,
            198.261000,
            201.359000,
            204.409000,
            207.411000,
            210.365000,
            213.268000,
            216.120000,
            218.920000,
            221.667000,
            224.361000,
            227.000000,
            229.585000,
            232.115000,
            234.589000,
            237.008000,
            239.370000,
            241.675000,
        ]
        y = materia.Qty(value=spd, unit=materia.unitless)

        super().__init__(x=x, y=y, normalize_to=normalize_to)


class CIEIlluminantD65(RelativeSPDSpectrum):
    def __init__(self, normalize_to=100):
        # data taken from https://web.archive.org/web/20170131100357/http://files.cie.co.at/204.xls
        x = materia.Qty(value=np.linspace(300, 830, 107), unit=materia.nanometer)
        spd = [
            0.034100,
            1.664300,
            3.294500,
            11.765200,
            20.236000,
            28.644700,
            37.053500,
            38.501100,
            39.948800,
            42.430200,
            44.911700,
            45.775000,
            46.638300,
            49.363700,
            52.089100,
            51.032300,
            49.975500,
            52.311800,
            54.648200,
            68.701500,
            82.754900,
            87.120400,
            91.486000,
            92.458900,
            93.431800,
            90.057000,
            86.682300,
            95.773600,
            104.865000,
            110.936000,
            117.008000,
            117.410000,
            117.812000,
            116.336000,
            114.861000,
            115.392000,
            115.923000,
            112.367000,
            108.811000,
            109.082000,
            109.354000,
            108.578000,
            107.802000,
            106.296000,
            104.790000,
            106.239000,
            107.689000,
            106.047000,
            104.405000,
            104.225000,
            104.046000,
            102.023000,
            100.000000,
            98.167100,
            96.334200,
            96.061100,
            95.788000,
            92.236800,
            88.685600,
            89.345900,
            90.006200,
            89.802600,
            89.599100,
            88.648900,
            87.698700,
            85.493600,
            83.288600,
            83.493900,
            83.699200,
            81.863000,
            80.026800,
            80.120700,
            80.214600,
            81.246200,
            82.277800,
            80.281000,
            78.284200,
            74.002700,
            69.721300,
            70.665200,
            71.609100,
            72.979000,
            74.349000,
            67.976500,
            61.604000,
            65.744800,
            69.885600,
            72.486300,
            75.087000,
            69.339800,
            63.592700,
            55.005400,
            46.418200,
            56.611800,
            66.805400,
            65.094100,
            63.382800,
            63.843400,
            64.304000,
            61.877900,
            59.451900,
            55.705400,
            51.959000,
            54.699800,
            57.440600,
            58.876500,
            60.312500,
        ]
        y = materia.Qty(value=spd, unit=materia.unitless)

        super().__init__(x=x, y=y, normalize_to=normalize_to)


class CIEIlluminantDSeries(RelativeSPDSpectrum):
    def __init__(self, T, normalize_to=100):
        # data taken from https://web.archive.org/web/20170131100357/http://files.cie.co.at/204.xls
        wavs = materia.Qty(value=np.linspace(300, 830, 107), unit=materia.nanometer)

        # equations taken from http://www.brucelindbloom.com/Eqn_T_to_xy.html
        if T >= 4000 and T <= 7000:
            x = (
                -4.6070e9 / T ** 3 + 2.9678e6 / T ** 2 + 0.09911e3 / T + 0.244063
            )  # materia.Qty(value=-4.6070e9/T**3 + 2.9678e6/T**2 + 0.09911e3/T + 0.244063,unit=materia.unitless)
        elif T > 7000 and T <= 25000:
            x = (
                -2.0064e9 / T ** 3 + 1.9018e6 / T ** 2 + 0.24748e3 / T + 0.237040
            )  # materia.Qty(value=-2.0064e9/T**3 + 1.9018e6/T**2 + 0.24748e3/T + 0.237040,unit=materia.unitless)

        y = -3.000 * x ** 2 + 2.870 * x - 0.275

        # equations taken from http://www.brucelindbloom.com/index.html?Eqn_DIlluminant.html
        M = 0.0241 + 0.2562 * x - 0.7341 * y
        M1 = (-1.3515 - 1.7703 * x + 5.9114 * y) / M
        M2 = (0.0300 - 31.4424 * x + 30.0717 * y) / M
        vals = (
            SimmonsDSeriesPCA0().y
            + M1 * SimmonsDSeriesPCA1().y
            + M2 * SimmonsDSeriesPCA2().y
        )

        super().__init__(x=wavs, y=vals, normalize_to=normalize_to)


class CIEIlluminantF4(RelativeSPDSpectrum):
    def __init__(self, normalize_to=100):
        # data taken from https://web.archive.org/web/20100612203959/http://www.cis.rit.edu/mcsl/online/CIE/Fluorescents.xls
        x = materia.Qty(value=np.linspace(380, 780, 81), unit=materia.nanometer)
        spd = [
            0.57,
            0.7,
            0.87,
            0.98,
            2.01,
            13.75,
            1.95,
            1.59,
            1.76,
            1.93,
            2.1,
            30.28,
            8.03,
            2.55,
            2.7,
            2.82,
            2.91,
            2.99,
            3.04,
            3.08,
            3.09,
            3.09,
            3.14,
            3.06,
            3,
            2.98,
            3.01,
            3.14,
            3.41,
            3.9,
            4.69,
            5.81,
            7.32,
            22.59,
            15.11,
            13.88,
            16.33,
            18.68,
            20.64,
            24.28,
            26.26,
            23.28,
            22.94,
            22.14,
            20.91,
            19.43,
            17.74,
            16,
            14.42,
            12.56,
            10.93,
            9.52,
            8.18,
            7.01,
            6,
            5.11,
            4.36,
            3.69,
            3.13,
            2.64,
            2.24,
            1.91,
            1.7,
            1.39,
            1.18,
            1.03,
            0.88,
            0.74,
            0.64,
            0.54,
            0.49,
            0.46,
            0.42,
            0.37,
            0.37,
            0.33,
            0.35,
            0.36,
            0.31,
            0.26,
            0.19,
        ]
        y = materia.Qty(value=spd, unit=materia.unitless)

        super().__init__(x=x, y=spd, normalize_to=normalize_to)
