"use strict";
// --------------------------------------------------------------------------------
// This file defines context keys that enable certain features that are
// implemented behind a flag in order to preserve backwards compatibility for
// existing apps. When a new app is initialized through `cdk init`, the CLI will
// automatically add enable these features by adding them to the generated
// `cdk.json` file. In the next major release of the CDK, these feature flags
// will be removed and will become the default behavior.
// See https://github.com/aws/aws-cdk-rfcs/blob/master/text/0055-feature-flags.md
// --------------------------------------------------------------------------------
Object.defineProperty(exports, "__esModule", { value: true });
exports.futureFlagDefault = exports.FUTURE_FLAGS_EXPIRED = exports.FUTURE_FLAGS = exports.CLOUDFRONT_DEFAULT_SECURITY_POLICY_TLS_V1_2_2021 = exports.LAMBDA_RECOGNIZE_VERSION_PROPS = exports.EFS_DEFAULT_ENCRYPTION_AT_REST = exports.APIGATEWAY_USAGEPLANKEY_ORDERINSENSITIVE_ID = exports.RDS_LOWERCASE_DB_IDENTIFIER = exports.ECS_REMOVE_DEFAULT_DESIRED_COUNT = exports.S3_GRANT_WRITE_WITHOUT_ACL = exports.KMS_DEFAULT_KEY_POLICIES = exports.SECRETS_MANAGER_PARSE_OWNED_SECRET_NAME = exports.DOCKER_IGNORE_SUPPORT = exports.STACK_RELATIVE_EXPORTS_CONTEXT = exports.NEW_STYLE_STACK_SYNTHESIS_CONTEXT = exports.ENABLE_DIFF_NO_FAIL = exports.ENABLE_DIFF_NO_FAIL_CONTEXT = exports.ENABLE_STACK_NAME_DUPLICATES_CONTEXT = void 0;
/**
 * If this is set, multiple stacks can use the same stack name (e.g. deployed to
 * different environments). This means that the name of the synthesized template
 * file will be based on the construct path and not on the defined `stackName`
 * of the stack.
 *
 * This is a "future flag": the feature is disabled by default for backwards
 * compatibility, but new projects created using `cdk init` will have this
 * enabled through the generated `cdk.json`.
 */
exports.ENABLE_STACK_NAME_DUPLICATES_CONTEXT = '@aws-cdk/core:enableStackNameDuplicates';
/**
 * IF this is set, `cdk diff` will always exit with 0.
 *
 * Use `cdk diff --fail` to exit with 1 if there's a diff.
 */
exports.ENABLE_DIFF_NO_FAIL_CONTEXT = 'aws-cdk:enableDiffNoFail';
/** @deprecated use `ENABLE_DIFF_NO_FAIL_CONTEXT` */
exports.ENABLE_DIFF_NO_FAIL = exports.ENABLE_DIFF_NO_FAIL_CONTEXT;
/**
 * Switch to new stack synthesis method which enable CI/CD
 */
exports.NEW_STYLE_STACK_SYNTHESIS_CONTEXT = '@aws-cdk/core:newStyleStackSynthesis';
/**
 * Name exports based on the construct paths relative to the stack, rather than the global construct path
 *
 * Combined with the stack name this relative construct path is good enough to
 * ensure uniqueness, and makes the export names robust against refactoring
 * the location of the stack in the construct tree (specifically, moving the Stack
 * into a Stage).
 */
exports.STACK_RELATIVE_EXPORTS_CONTEXT = '@aws-cdk/core:stackRelativeExports';
/**
 * DockerImageAsset properly supports `.dockerignore` files by default
 *
 * If this flag is not set, the default behavior for `DockerImageAsset` is to use
 * glob semantics for `.dockerignore` files. If this flag is set, the default behavior
 * is standard Docker ignore semantics.
 *
 * This is a feature flag as the old behavior was technically incorrect but
 * users may have come to depend on it.
 */
exports.DOCKER_IGNORE_SUPPORT = '@aws-cdk/aws-ecr-assets:dockerIgnoreSupport';
/**
 * Secret.secretName for an "owned" secret will attempt to parse the secretName from the ARN,
 * rather than the default full resource name, which includes the SecretsManager suffix.
 *
 * If this flag is not set, Secret.secretName will include the SecretsManager suffix, which cannot be directly
 * used by SecretsManager.DescribeSecret, and must be parsed by the user first (e.g., Fn:Join, Fn:Select, Fn:Split).
 */
exports.SECRETS_MANAGER_PARSE_OWNED_SECRET_NAME = '@aws-cdk/aws-secretsmanager:parseOwnedSecretName';
/**
 * KMS Keys start with a default key policy that grants the account access to administer the key,
 * mirroring the behavior of the KMS SDK/CLI/Console experience. Users may override the default key
 * policy by specifying their own.
 *
 * If this flag is not set, the default key policy depends on the setting of the `trustAccountIdentities`
 * flag. If false (the default, for backwards-compatibility reasons), the default key policy somewhat
 * resemebles the default admin key policy, but with the addition of 'GenerateDataKey' permissions. If
 * true, the policy matches what happens when this feature flag is set.
 *
 * Additionally, if this flag is not set and the user supplies a custom key policy, this will be appended
 * to the key's default policy (rather than replacing it).
 */
exports.KMS_DEFAULT_KEY_POLICIES = '@aws-cdk/aws-kms:defaultKeyPolicies';
/**
 * Change the old 's3:PutObject*' permission to 's3:PutObject' on Bucket,
 * as the former includes 's3:PutObjectAcl',
 * which could be used to grant read/write object access to IAM principals in other accounts.
 * Use a feature flag to make sure existing customers who might be relying
 * on the overly-broad permissions are not broken.
 */
exports.S3_GRANT_WRITE_WITHOUT_ACL = '@aws-cdk/aws-s3:grantWriteWithoutAcl';
/**
 * ApplicationLoadBalancedServiceBase, ApplicationMultipleTargetGroupServiceBase,
 * NetworkLoadBalancedServiceBase, NetworkMultipleTargetGroupServiceBase, and
 * QueueProcessingServiceBase currently determine a default value for the desired count of
 * a CfnService if a desiredCount is not provided.
 *
 * If this flag is not set, the default behaviour for CfnService.desiredCount is to set a
 * desiredCount of 1, if one is not provided. If true, a default will not be defined for
 * CfnService.desiredCount and as such desiredCount will be undefined, if one is not provided.
 *
 * This is a feature flag as the old behavior was technically incorrect, but
 * users may have come to depend on it.
 */
exports.ECS_REMOVE_DEFAULT_DESIRED_COUNT = '@aws-cdk/aws-ecs-patterns:removeDefaultDesiredCount';
/**
 * ServerlessCluster.clusterIdentifier currently can has uppercase letters,
 * and ServerlessCluster pass it through to CfnDBCluster.dbClusterIdentifier.
 * The identifier is saved as lowercase string in AWS and is resolved as original string in CloudFormation.
 *
 * If this flag is not set, original value that one set to ServerlessCluster.clusterIdentifier
 * is passed to CfnDBCluster.dbClusterIdentifier.
 * If this flag is true, ServerlessCluster.clusterIdentifier is converted into a string containing
 * only lowercase characters by the `toLowerCase` function and passed to CfnDBCluster.dbClusterIdentifier.
 *
 * This feature flag make correct the ServerlessCluster.clusterArn when
 * clusterIdentifier contains a Upper case letters.
 */
exports.RDS_LOWERCASE_DB_IDENTIFIER = '@aws-cdk/aws-rds:lowercaseDbIdentifier';
/**
 * The UsagePlanKey resource connects an ApiKey with a UsagePlan. API Gateway does not allow more than one UsagePlanKey
 * for any given UsagePlan and ApiKey combination. For this reason, CloudFormation cannot replace this resource without
 * either the UsagePlan or ApiKey changing.
 *
 * The feature addition to support multiple UsagePlanKey resources - 142bd0e2 - recognized this and attempted to keep
 * existing UsagePlanKey logical ids unchanged.
 * However, this intentionally caused the logical id of the UsagePlanKey to be sensitive to order. That is, when
 * the 'first' UsagePlanKey resource is removed, the logical id of the 'second' assumes what was originally the 'first',
 * which again is disallowed.
 *
 * In effect, there is no way to get out of this mess in a backwards compatible way, while supporting existing stacks.
 * This flag changes the logical id layout of UsagePlanKey to not be sensitive to order.
 */
exports.APIGATEWAY_USAGEPLANKEY_ORDERINSENSITIVE_ID = '@aws-cdk/aws-apigateway:usagePlanKeyOrderInsensitiveId';
/**
 * Enable this feature flag to have elastic file systems encrypted at rest by default.
 *
 * Encryption can also be configured explicitly using the `encrypted` property.
 */
exports.EFS_DEFAULT_ENCRYPTION_AT_REST = '@aws-cdk/aws-efs:defaultEncryptionAtRest';
/**
 * Enable this feature flag to opt in to the updated logical id calculation for Lambda Version created using the
 * `fn.currentVersion`.
 *
 * The previous calculation incorrectly considered properties of the `AWS::Lambda::Function` resource that did
 * not constitute creating a new Version.
 *
 * See 'currentVersion' section in the aws-lambda module's README for more details.
 */
exports.LAMBDA_RECOGNIZE_VERSION_PROPS = '@aws-cdk/aws-lambda:recognizeVersionProps';
/**
 * Enable this feature flag to have cloudfront distributions use the security policy TLSv1.2_2021 by default.
 *
 * The security policy can also be configured explicitly using the `minimumProtocolVersion` property.
 */
exports.CLOUDFRONT_DEFAULT_SECURITY_POLICY_TLS_V1_2_2021 = '@aws-cdk/aws-cloudfront:defaultSecurityPolicyTLSv1.2_2021';
/**
 * This map includes context keys and values for feature flags that enable
 * capabilities "from the future", which we could not introduce as the default
 * behavior due to backwards compatibility for existing projects.
 *
 * New projects generated through `cdk init` will include these flags in their
 * generated `cdk.json` file.
 *
 * When we release the next major version of the CDK, we will flip the logic of
 * these features and clean up the `cdk.json` generated by `cdk init`.
 *
 * Tests must cover the default (disabled) case and the future (enabled) case.
 */
exports.FUTURE_FLAGS = {
    [exports.APIGATEWAY_USAGEPLANKEY_ORDERINSENSITIVE_ID]: true,
    [exports.ENABLE_STACK_NAME_DUPLICATES_CONTEXT]: 'true',
    [exports.ENABLE_DIFF_NO_FAIL_CONTEXT]: 'true',
    [exports.STACK_RELATIVE_EXPORTS_CONTEXT]: 'true',
    [exports.DOCKER_IGNORE_SUPPORT]: true,
    [exports.SECRETS_MANAGER_PARSE_OWNED_SECRET_NAME]: true,
    [exports.KMS_DEFAULT_KEY_POLICIES]: true,
    [exports.S3_GRANT_WRITE_WITHOUT_ACL]: true,
    [exports.ECS_REMOVE_DEFAULT_DESIRED_COUNT]: true,
    [exports.RDS_LOWERCASE_DB_IDENTIFIER]: true,
    [exports.EFS_DEFAULT_ENCRYPTION_AT_REST]: true,
    [exports.LAMBDA_RECOGNIZE_VERSION_PROPS]: true,
    [exports.CLOUDFRONT_DEFAULT_SECURITY_POLICY_TLS_V1_2_2021]: true,
};
/**
 * The list of future flags that are now expired. This is going to be used to identify
 * and block usages of old feature flags in the new major version of CDK.
 */
exports.FUTURE_FLAGS_EXPIRED = [];
/**
 * The set of defaults that should be applied if the feature flag is not
 * explicitly configured.
 */
const FUTURE_FLAGS_DEFAULTS = {
    [exports.APIGATEWAY_USAGEPLANKEY_ORDERINSENSITIVE_ID]: false,
    [exports.ENABLE_STACK_NAME_DUPLICATES_CONTEXT]: false,
    [exports.ENABLE_DIFF_NO_FAIL_CONTEXT]: false,
    [exports.STACK_RELATIVE_EXPORTS_CONTEXT]: false,
    [exports.NEW_STYLE_STACK_SYNTHESIS_CONTEXT]: false,
    [exports.DOCKER_IGNORE_SUPPORT]: false,
    [exports.SECRETS_MANAGER_PARSE_OWNED_SECRET_NAME]: false,
    [exports.KMS_DEFAULT_KEY_POLICIES]: false,
    [exports.S3_GRANT_WRITE_WITHOUT_ACL]: false,
    [exports.ECS_REMOVE_DEFAULT_DESIRED_COUNT]: false,
    [exports.RDS_LOWERCASE_DB_IDENTIFIER]: false,
    [exports.EFS_DEFAULT_ENCRYPTION_AT_REST]: false,
    [exports.LAMBDA_RECOGNIZE_VERSION_PROPS]: false,
    [exports.CLOUDFRONT_DEFAULT_SECURITY_POLICY_TLS_V1_2_2021]: false,
};
function futureFlagDefault(flag) {
    return FUTURE_FLAGS_DEFAULTS[flag];
}
exports.futureFlagDefault = futureFlagDefault;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZmVhdHVyZXMuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJmZWF0dXJlcy50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiO0FBQUEsbUZBQW1GO0FBQ25GLHVFQUF1RTtBQUN2RSw2RUFBNkU7QUFDN0UsZ0ZBQWdGO0FBQ2hGLDBFQUEwRTtBQUMxRSw2RUFBNkU7QUFDN0Usd0RBQXdEO0FBQ3hELGlGQUFpRjtBQUNqRixtRkFBbUY7OztBQUVuRjs7Ozs7Ozs7O0dBU0c7QUFDVSxRQUFBLG9DQUFvQyxHQUFHLHlDQUF5QyxDQUFDO0FBRTlGOzs7O0dBSUc7QUFDVSxRQUFBLDJCQUEyQixHQUFHLDBCQUEwQixDQUFDO0FBQ3RFLG9EQUFvRDtBQUN2QyxRQUFBLG1CQUFtQixHQUFHLG1DQUEyQixDQUFDO0FBRS9EOztHQUVHO0FBQ1UsUUFBQSxpQ0FBaUMsR0FBRyxzQ0FBc0MsQ0FBQztBQUV4Rjs7Ozs7OztHQU9HO0FBQ1UsUUFBQSw4QkFBOEIsR0FBRyxvQ0FBb0MsQ0FBQztBQUVuRjs7Ozs7Ozs7O0dBU0c7QUFDVSxRQUFBLHFCQUFxQixHQUFHLDZDQUE2QyxDQUFDO0FBRW5GOzs7Ozs7R0FNRztBQUNVLFFBQUEsdUNBQXVDLEdBQUcsa0RBQWtELENBQUM7QUFFMUc7Ozs7Ozs7Ozs7OztHQVlHO0FBQ1UsUUFBQSx3QkFBd0IsR0FBRyxxQ0FBcUMsQ0FBQztBQUU5RTs7Ozs7O0dBTUc7QUFDVSxRQUFBLDBCQUEwQixHQUFHLHNDQUFzQyxDQUFDO0FBRWpGOzs7Ozs7Ozs7Ozs7R0FZRztBQUNVLFFBQUEsZ0NBQWdDLEdBQUcscURBQXFELENBQUM7QUFFdEc7Ozs7Ozs7Ozs7OztHQVlHO0FBQ1UsUUFBQSwyQkFBMkIsR0FBRyx3Q0FBd0MsQ0FBQztBQUVwRjs7Ozs7Ozs7Ozs7OztHQWFHO0FBQ1UsUUFBQSwyQ0FBMkMsR0FBRyx3REFBd0QsQ0FBQztBQUVwSDs7OztHQUlHO0FBQ1UsUUFBQSw4QkFBOEIsR0FBRywwQ0FBMEMsQ0FBQztBQUV6Rjs7Ozs7Ozs7R0FRRztBQUNVLFFBQUEsOEJBQThCLEdBQUcsMkNBQTJDLENBQUM7QUFFMUY7Ozs7R0FJRztBQUNVLFFBQUEsZ0RBQWdELEdBQUcsMkRBQTJELENBQUM7QUFFNUg7Ozs7Ozs7Ozs7OztHQVlHO0FBQ1UsUUFBQSxZQUFZLEdBQTJCO0lBQ2xELENBQUMsbURBQTJDLENBQUMsRUFBRSxJQUFJO0lBQ25ELENBQUMsNENBQW9DLENBQUMsRUFBRSxNQUFNO0lBQzlDLENBQUMsbUNBQTJCLENBQUMsRUFBRSxNQUFNO0lBQ3JDLENBQUMsc0NBQThCLENBQUMsRUFBRSxNQUFNO0lBQ3hDLENBQUMsNkJBQXFCLENBQUMsRUFBRSxJQUFJO0lBQzdCLENBQUMsK0NBQXVDLENBQUMsRUFBRSxJQUFJO0lBQy9DLENBQUMsZ0NBQXdCLENBQUMsRUFBRSxJQUFJO0lBQ2hDLENBQUMsa0NBQTBCLENBQUMsRUFBRSxJQUFJO0lBQ2xDLENBQUMsd0NBQWdDLENBQUMsRUFBRSxJQUFJO0lBQ3hDLENBQUMsbUNBQTJCLENBQUMsRUFBRSxJQUFJO0lBQ25DLENBQUMsc0NBQThCLENBQUMsRUFBRSxJQUFJO0lBQ3RDLENBQUMsc0NBQThCLENBQUMsRUFBRSxJQUFJO0lBQ3RDLENBQUMsd0RBQWdELENBQUMsRUFBRSxJQUFJO0NBSXpELENBQUM7QUFFRjs7O0dBR0c7QUFDVSxRQUFBLG9CQUFvQixHQUFhLEVBQzdDLENBQUM7QUFFRjs7O0dBR0c7QUFDSCxNQUFNLHFCQUFxQixHQUErQjtJQUN4RCxDQUFDLG1EQUEyQyxDQUFDLEVBQUUsS0FBSztJQUNwRCxDQUFDLDRDQUFvQyxDQUFDLEVBQUUsS0FBSztJQUM3QyxDQUFDLG1DQUEyQixDQUFDLEVBQUUsS0FBSztJQUNwQyxDQUFDLHNDQUE4QixDQUFDLEVBQUUsS0FBSztJQUN2QyxDQUFDLHlDQUFpQyxDQUFDLEVBQUUsS0FBSztJQUMxQyxDQUFDLDZCQUFxQixDQUFDLEVBQUUsS0FBSztJQUM5QixDQUFDLCtDQUF1QyxDQUFDLEVBQUUsS0FBSztJQUNoRCxDQUFDLGdDQUF3QixDQUFDLEVBQUUsS0FBSztJQUNqQyxDQUFDLGtDQUEwQixDQUFDLEVBQUUsS0FBSztJQUNuQyxDQUFDLHdDQUFnQyxDQUFDLEVBQUUsS0FBSztJQUN6QyxDQUFDLG1DQUEyQixDQUFDLEVBQUUsS0FBSztJQUNwQyxDQUFDLHNDQUE4QixDQUFDLEVBQUUsS0FBSztJQUN2QyxDQUFDLHNDQUE4QixDQUFDLEVBQUUsS0FBSztJQUN2QyxDQUFDLHdEQUFnRCxDQUFDLEVBQUUsS0FBSztDQUMxRCxDQUFDO0FBRUYsU0FBZ0IsaUJBQWlCLENBQUMsSUFBWTtJQUM1QyxPQUFPLHFCQUFxQixDQUFDLElBQUksQ0FBQyxDQUFDO0FBQ3JDLENBQUM7QUFGRCw4Q0FFQyIsInNvdXJjZXNDb250ZW50IjpbIi8vIC0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tXG4vLyBUaGlzIGZpbGUgZGVmaW5lcyBjb250ZXh0IGtleXMgdGhhdCBlbmFibGUgY2VydGFpbiBmZWF0dXJlcyB0aGF0IGFyZVxuLy8gaW1wbGVtZW50ZWQgYmVoaW5kIGEgZmxhZyBpbiBvcmRlciB0byBwcmVzZXJ2ZSBiYWNrd2FyZHMgY29tcGF0aWJpbGl0eSBmb3Jcbi8vIGV4aXN0aW5nIGFwcHMuIFdoZW4gYSBuZXcgYXBwIGlzIGluaXRpYWxpemVkIHRocm91Z2ggYGNkayBpbml0YCwgdGhlIENMSSB3aWxsXG4vLyBhdXRvbWF0aWNhbGx5IGFkZCBlbmFibGUgdGhlc2UgZmVhdHVyZXMgYnkgYWRkaW5nIHRoZW0gdG8gdGhlIGdlbmVyYXRlZFxuLy8gYGNkay5qc29uYCBmaWxlLiBJbiB0aGUgbmV4dCBtYWpvciByZWxlYXNlIG9mIHRoZSBDREssIHRoZXNlIGZlYXR1cmUgZmxhZ3Ncbi8vIHdpbGwgYmUgcmVtb3ZlZCBhbmQgd2lsbCBiZWNvbWUgdGhlIGRlZmF1bHQgYmVoYXZpb3IuXG4vLyBTZWUgaHR0cHM6Ly9naXRodWIuY29tL2F3cy9hd3MtY2RrLXJmY3MvYmxvYi9tYXN0ZXIvdGV4dC8wMDU1LWZlYXR1cmUtZmxhZ3MubWRcbi8vIC0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tXG5cbi8qKlxuICogSWYgdGhpcyBpcyBzZXQsIG11bHRpcGxlIHN0YWNrcyBjYW4gdXNlIHRoZSBzYW1lIHN0YWNrIG5hbWUgKGUuZy4gZGVwbG95ZWQgdG9cbiAqIGRpZmZlcmVudCBlbnZpcm9ubWVudHMpLiBUaGlzIG1lYW5zIHRoYXQgdGhlIG5hbWUgb2YgdGhlIHN5bnRoZXNpemVkIHRlbXBsYXRlXG4gKiBmaWxlIHdpbGwgYmUgYmFzZWQgb24gdGhlIGNvbnN0cnVjdCBwYXRoIGFuZCBub3Qgb24gdGhlIGRlZmluZWQgYHN0YWNrTmFtZWBcbiAqIG9mIHRoZSBzdGFjay5cbiAqXG4gKiBUaGlzIGlzIGEgXCJmdXR1cmUgZmxhZ1wiOiB0aGUgZmVhdHVyZSBpcyBkaXNhYmxlZCBieSBkZWZhdWx0IGZvciBiYWNrd2FyZHNcbiAqIGNvbXBhdGliaWxpdHksIGJ1dCBuZXcgcHJvamVjdHMgY3JlYXRlZCB1c2luZyBgY2RrIGluaXRgIHdpbGwgaGF2ZSB0aGlzXG4gKiBlbmFibGVkIHRocm91Z2ggdGhlIGdlbmVyYXRlZCBgY2RrLmpzb25gLlxuICovXG5leHBvcnQgY29uc3QgRU5BQkxFX1NUQUNLX05BTUVfRFVQTElDQVRFU19DT05URVhUID0gJ0Bhd3MtY2RrL2NvcmU6ZW5hYmxlU3RhY2tOYW1lRHVwbGljYXRlcyc7XG5cbi8qKlxuICogSUYgdGhpcyBpcyBzZXQsIGBjZGsgZGlmZmAgd2lsbCBhbHdheXMgZXhpdCB3aXRoIDAuXG4gKlxuICogVXNlIGBjZGsgZGlmZiAtLWZhaWxgIHRvIGV4aXQgd2l0aCAxIGlmIHRoZXJlJ3MgYSBkaWZmLlxuICovXG5leHBvcnQgY29uc3QgRU5BQkxFX0RJRkZfTk9fRkFJTF9DT05URVhUID0gJ2F3cy1jZGs6ZW5hYmxlRGlmZk5vRmFpbCc7XG4vKiogQGRlcHJlY2F0ZWQgdXNlIGBFTkFCTEVfRElGRl9OT19GQUlMX0NPTlRFWFRgICovXG5leHBvcnQgY29uc3QgRU5BQkxFX0RJRkZfTk9fRkFJTCA9IEVOQUJMRV9ESUZGX05PX0ZBSUxfQ09OVEVYVDtcblxuLyoqXG4gKiBTd2l0Y2ggdG8gbmV3IHN0YWNrIHN5bnRoZXNpcyBtZXRob2Qgd2hpY2ggZW5hYmxlIENJL0NEXG4gKi9cbmV4cG9ydCBjb25zdCBORVdfU1RZTEVfU1RBQ0tfU1lOVEhFU0lTX0NPTlRFWFQgPSAnQGF3cy1jZGsvY29yZTpuZXdTdHlsZVN0YWNrU3ludGhlc2lzJztcblxuLyoqXG4gKiBOYW1lIGV4cG9ydHMgYmFzZWQgb24gdGhlIGNvbnN0cnVjdCBwYXRocyByZWxhdGl2ZSB0byB0aGUgc3RhY2ssIHJhdGhlciB0aGFuIHRoZSBnbG9iYWwgY29uc3RydWN0IHBhdGhcbiAqXG4gKiBDb21iaW5lZCB3aXRoIHRoZSBzdGFjayBuYW1lIHRoaXMgcmVsYXRpdmUgY29uc3RydWN0IHBhdGggaXMgZ29vZCBlbm91Z2ggdG9cbiAqIGVuc3VyZSB1bmlxdWVuZXNzLCBhbmQgbWFrZXMgdGhlIGV4cG9ydCBuYW1lcyByb2J1c3QgYWdhaW5zdCByZWZhY3RvcmluZ1xuICogdGhlIGxvY2F0aW9uIG9mIHRoZSBzdGFjayBpbiB0aGUgY29uc3RydWN0IHRyZWUgKHNwZWNpZmljYWxseSwgbW92aW5nIHRoZSBTdGFja1xuICogaW50byBhIFN0YWdlKS5cbiAqL1xuZXhwb3J0IGNvbnN0IFNUQUNLX1JFTEFUSVZFX0VYUE9SVFNfQ09OVEVYVCA9ICdAYXdzLWNkay9jb3JlOnN0YWNrUmVsYXRpdmVFeHBvcnRzJztcblxuLyoqXG4gKiBEb2NrZXJJbWFnZUFzc2V0IHByb3Blcmx5IHN1cHBvcnRzIGAuZG9ja2VyaWdub3JlYCBmaWxlcyBieSBkZWZhdWx0XG4gKlxuICogSWYgdGhpcyBmbGFnIGlzIG5vdCBzZXQsIHRoZSBkZWZhdWx0IGJlaGF2aW9yIGZvciBgRG9ja2VySW1hZ2VBc3NldGAgaXMgdG8gdXNlXG4gKiBnbG9iIHNlbWFudGljcyBmb3IgYC5kb2NrZXJpZ25vcmVgIGZpbGVzLiBJZiB0aGlzIGZsYWcgaXMgc2V0LCB0aGUgZGVmYXVsdCBiZWhhdmlvclxuICogaXMgc3RhbmRhcmQgRG9ja2VyIGlnbm9yZSBzZW1hbnRpY3MuXG4gKlxuICogVGhpcyBpcyBhIGZlYXR1cmUgZmxhZyBhcyB0aGUgb2xkIGJlaGF2aW9yIHdhcyB0ZWNobmljYWxseSBpbmNvcnJlY3QgYnV0XG4gKiB1c2VycyBtYXkgaGF2ZSBjb21lIHRvIGRlcGVuZCBvbiBpdC5cbiAqL1xuZXhwb3J0IGNvbnN0IERPQ0tFUl9JR05PUkVfU1VQUE9SVCA9ICdAYXdzLWNkay9hd3MtZWNyLWFzc2V0czpkb2NrZXJJZ25vcmVTdXBwb3J0JztcblxuLyoqXG4gKiBTZWNyZXQuc2VjcmV0TmFtZSBmb3IgYW4gXCJvd25lZFwiIHNlY3JldCB3aWxsIGF0dGVtcHQgdG8gcGFyc2UgdGhlIHNlY3JldE5hbWUgZnJvbSB0aGUgQVJOLFxuICogcmF0aGVyIHRoYW4gdGhlIGRlZmF1bHQgZnVsbCByZXNvdXJjZSBuYW1lLCB3aGljaCBpbmNsdWRlcyB0aGUgU2VjcmV0c01hbmFnZXIgc3VmZml4LlxuICpcbiAqIElmIHRoaXMgZmxhZyBpcyBub3Qgc2V0LCBTZWNyZXQuc2VjcmV0TmFtZSB3aWxsIGluY2x1ZGUgdGhlIFNlY3JldHNNYW5hZ2VyIHN1ZmZpeCwgd2hpY2ggY2Fubm90IGJlIGRpcmVjdGx5XG4gKiB1c2VkIGJ5IFNlY3JldHNNYW5hZ2VyLkRlc2NyaWJlU2VjcmV0LCBhbmQgbXVzdCBiZSBwYXJzZWQgYnkgdGhlIHVzZXIgZmlyc3QgKGUuZy4sIEZuOkpvaW4sIEZuOlNlbGVjdCwgRm46U3BsaXQpLlxuICovXG5leHBvcnQgY29uc3QgU0VDUkVUU19NQU5BR0VSX1BBUlNFX09XTkVEX1NFQ1JFVF9OQU1FID0gJ0Bhd3MtY2RrL2F3cy1zZWNyZXRzbWFuYWdlcjpwYXJzZU93bmVkU2VjcmV0TmFtZSc7XG5cbi8qKlxuICogS01TIEtleXMgc3RhcnQgd2l0aCBhIGRlZmF1bHQga2V5IHBvbGljeSB0aGF0IGdyYW50cyB0aGUgYWNjb3VudCBhY2Nlc3MgdG8gYWRtaW5pc3RlciB0aGUga2V5LFxuICogbWlycm9yaW5nIHRoZSBiZWhhdmlvciBvZiB0aGUgS01TIFNESy9DTEkvQ29uc29sZSBleHBlcmllbmNlLiBVc2VycyBtYXkgb3ZlcnJpZGUgdGhlIGRlZmF1bHQga2V5XG4gKiBwb2xpY3kgYnkgc3BlY2lmeWluZyB0aGVpciBvd24uXG4gKlxuICogSWYgdGhpcyBmbGFnIGlzIG5vdCBzZXQsIHRoZSBkZWZhdWx0IGtleSBwb2xpY3kgZGVwZW5kcyBvbiB0aGUgc2V0dGluZyBvZiB0aGUgYHRydXN0QWNjb3VudElkZW50aXRpZXNgXG4gKiBmbGFnLiBJZiBmYWxzZSAodGhlIGRlZmF1bHQsIGZvciBiYWNrd2FyZHMtY29tcGF0aWJpbGl0eSByZWFzb25zKSwgdGhlIGRlZmF1bHQga2V5IHBvbGljeSBzb21ld2hhdFxuICogcmVzZW1lYmxlcyB0aGUgZGVmYXVsdCBhZG1pbiBrZXkgcG9saWN5LCBidXQgd2l0aCB0aGUgYWRkaXRpb24gb2YgJ0dlbmVyYXRlRGF0YUtleScgcGVybWlzc2lvbnMuIElmXG4gKiB0cnVlLCB0aGUgcG9saWN5IG1hdGNoZXMgd2hhdCBoYXBwZW5zIHdoZW4gdGhpcyBmZWF0dXJlIGZsYWcgaXMgc2V0LlxuICpcbiAqIEFkZGl0aW9uYWxseSwgaWYgdGhpcyBmbGFnIGlzIG5vdCBzZXQgYW5kIHRoZSB1c2VyIHN1cHBsaWVzIGEgY3VzdG9tIGtleSBwb2xpY3ksIHRoaXMgd2lsbCBiZSBhcHBlbmRlZFxuICogdG8gdGhlIGtleSdzIGRlZmF1bHQgcG9saWN5IChyYXRoZXIgdGhhbiByZXBsYWNpbmcgaXQpLlxuICovXG5leHBvcnQgY29uc3QgS01TX0RFRkFVTFRfS0VZX1BPTElDSUVTID0gJ0Bhd3MtY2RrL2F3cy1rbXM6ZGVmYXVsdEtleVBvbGljaWVzJztcblxuLyoqXG4gKiBDaGFuZ2UgdGhlIG9sZCAnczM6UHV0T2JqZWN0KicgcGVybWlzc2lvbiB0byAnczM6UHV0T2JqZWN0JyBvbiBCdWNrZXQsXG4gKiBhcyB0aGUgZm9ybWVyIGluY2x1ZGVzICdzMzpQdXRPYmplY3RBY2wnLFxuICogd2hpY2ggY291bGQgYmUgdXNlZCB0byBncmFudCByZWFkL3dyaXRlIG9iamVjdCBhY2Nlc3MgdG8gSUFNIHByaW5jaXBhbHMgaW4gb3RoZXIgYWNjb3VudHMuXG4gKiBVc2UgYSBmZWF0dXJlIGZsYWcgdG8gbWFrZSBzdXJlIGV4aXN0aW5nIGN1c3RvbWVycyB3aG8gbWlnaHQgYmUgcmVseWluZ1xuICogb24gdGhlIG92ZXJseS1icm9hZCBwZXJtaXNzaW9ucyBhcmUgbm90IGJyb2tlbi5cbiAqL1xuZXhwb3J0IGNvbnN0IFMzX0dSQU5UX1dSSVRFX1dJVEhPVVRfQUNMID0gJ0Bhd3MtY2RrL2F3cy1zMzpncmFudFdyaXRlV2l0aG91dEFjbCc7XG5cbi8qKlxuICogQXBwbGljYXRpb25Mb2FkQmFsYW5jZWRTZXJ2aWNlQmFzZSwgQXBwbGljYXRpb25NdWx0aXBsZVRhcmdldEdyb3VwU2VydmljZUJhc2UsXG4gKiBOZXR3b3JrTG9hZEJhbGFuY2VkU2VydmljZUJhc2UsIE5ldHdvcmtNdWx0aXBsZVRhcmdldEdyb3VwU2VydmljZUJhc2UsIGFuZFxuICogUXVldWVQcm9jZXNzaW5nU2VydmljZUJhc2UgY3VycmVudGx5IGRldGVybWluZSBhIGRlZmF1bHQgdmFsdWUgZm9yIHRoZSBkZXNpcmVkIGNvdW50IG9mXG4gKiBhIENmblNlcnZpY2UgaWYgYSBkZXNpcmVkQ291bnQgaXMgbm90IHByb3ZpZGVkLlxuICpcbiAqIElmIHRoaXMgZmxhZyBpcyBub3Qgc2V0LCB0aGUgZGVmYXVsdCBiZWhhdmlvdXIgZm9yIENmblNlcnZpY2UuZGVzaXJlZENvdW50IGlzIHRvIHNldCBhXG4gKiBkZXNpcmVkQ291bnQgb2YgMSwgaWYgb25lIGlzIG5vdCBwcm92aWRlZC4gSWYgdHJ1ZSwgYSBkZWZhdWx0IHdpbGwgbm90IGJlIGRlZmluZWQgZm9yXG4gKiBDZm5TZXJ2aWNlLmRlc2lyZWRDb3VudCBhbmQgYXMgc3VjaCBkZXNpcmVkQ291bnQgd2lsbCBiZSB1bmRlZmluZWQsIGlmIG9uZSBpcyBub3QgcHJvdmlkZWQuXG4gKlxuICogVGhpcyBpcyBhIGZlYXR1cmUgZmxhZyBhcyB0aGUgb2xkIGJlaGF2aW9yIHdhcyB0ZWNobmljYWxseSBpbmNvcnJlY3QsIGJ1dFxuICogdXNlcnMgbWF5IGhhdmUgY29tZSB0byBkZXBlbmQgb24gaXQuXG4gKi9cbmV4cG9ydCBjb25zdCBFQ1NfUkVNT1ZFX0RFRkFVTFRfREVTSVJFRF9DT1VOVCA9ICdAYXdzLWNkay9hd3MtZWNzLXBhdHRlcm5zOnJlbW92ZURlZmF1bHREZXNpcmVkQ291bnQnO1xuXG4vKipcbiAqIFNlcnZlcmxlc3NDbHVzdGVyLmNsdXN0ZXJJZGVudGlmaWVyIGN1cnJlbnRseSBjYW4gaGFzIHVwcGVyY2FzZSBsZXR0ZXJzLFxuICogYW5kIFNlcnZlcmxlc3NDbHVzdGVyIHBhc3MgaXQgdGhyb3VnaCB0byBDZm5EQkNsdXN0ZXIuZGJDbHVzdGVySWRlbnRpZmllci5cbiAqIFRoZSBpZGVudGlmaWVyIGlzIHNhdmVkIGFzIGxvd2VyY2FzZSBzdHJpbmcgaW4gQVdTIGFuZCBpcyByZXNvbHZlZCBhcyBvcmlnaW5hbCBzdHJpbmcgaW4gQ2xvdWRGb3JtYXRpb24uXG4gKlxuICogSWYgdGhpcyBmbGFnIGlzIG5vdCBzZXQsIG9yaWdpbmFsIHZhbHVlIHRoYXQgb25lIHNldCB0byBTZXJ2ZXJsZXNzQ2x1c3Rlci5jbHVzdGVySWRlbnRpZmllclxuICogaXMgcGFzc2VkIHRvIENmbkRCQ2x1c3Rlci5kYkNsdXN0ZXJJZGVudGlmaWVyLlxuICogSWYgdGhpcyBmbGFnIGlzIHRydWUsIFNlcnZlcmxlc3NDbHVzdGVyLmNsdXN0ZXJJZGVudGlmaWVyIGlzIGNvbnZlcnRlZCBpbnRvIGEgc3RyaW5nIGNvbnRhaW5pbmdcbiAqIG9ubHkgbG93ZXJjYXNlIGNoYXJhY3RlcnMgYnkgdGhlIGB0b0xvd2VyQ2FzZWAgZnVuY3Rpb24gYW5kIHBhc3NlZCB0byBDZm5EQkNsdXN0ZXIuZGJDbHVzdGVySWRlbnRpZmllci5cbiAqXG4gKiBUaGlzIGZlYXR1cmUgZmxhZyBtYWtlIGNvcnJlY3QgdGhlIFNlcnZlcmxlc3NDbHVzdGVyLmNsdXN0ZXJBcm4gd2hlblxuICogY2x1c3RlcklkZW50aWZpZXIgY29udGFpbnMgYSBVcHBlciBjYXNlIGxldHRlcnMuXG4gKi9cbmV4cG9ydCBjb25zdCBSRFNfTE9XRVJDQVNFX0RCX0lERU5USUZJRVIgPSAnQGF3cy1jZGsvYXdzLXJkczpsb3dlcmNhc2VEYklkZW50aWZpZXInO1xuXG4vKipcbiAqIFRoZSBVc2FnZVBsYW5LZXkgcmVzb3VyY2UgY29ubmVjdHMgYW4gQXBpS2V5IHdpdGggYSBVc2FnZVBsYW4uIEFQSSBHYXRld2F5IGRvZXMgbm90IGFsbG93IG1vcmUgdGhhbiBvbmUgVXNhZ2VQbGFuS2V5XG4gKiBmb3IgYW55IGdpdmVuIFVzYWdlUGxhbiBhbmQgQXBpS2V5IGNvbWJpbmF0aW9uLiBGb3IgdGhpcyByZWFzb24sIENsb3VkRm9ybWF0aW9uIGNhbm5vdCByZXBsYWNlIHRoaXMgcmVzb3VyY2Ugd2l0aG91dFxuICogZWl0aGVyIHRoZSBVc2FnZVBsYW4gb3IgQXBpS2V5IGNoYW5naW5nLlxuICpcbiAqIFRoZSBmZWF0dXJlIGFkZGl0aW9uIHRvIHN1cHBvcnQgbXVsdGlwbGUgVXNhZ2VQbGFuS2V5IHJlc291cmNlcyAtIDE0MmJkMGUyIC0gcmVjb2duaXplZCB0aGlzIGFuZCBhdHRlbXB0ZWQgdG8ga2VlcFxuICogZXhpc3RpbmcgVXNhZ2VQbGFuS2V5IGxvZ2ljYWwgaWRzIHVuY2hhbmdlZC5cbiAqIEhvd2V2ZXIsIHRoaXMgaW50ZW50aW9uYWxseSBjYXVzZWQgdGhlIGxvZ2ljYWwgaWQgb2YgdGhlIFVzYWdlUGxhbktleSB0byBiZSBzZW5zaXRpdmUgdG8gb3JkZXIuIFRoYXQgaXMsIHdoZW5cbiAqIHRoZSAnZmlyc3QnIFVzYWdlUGxhbktleSByZXNvdXJjZSBpcyByZW1vdmVkLCB0aGUgbG9naWNhbCBpZCBvZiB0aGUgJ3NlY29uZCcgYXNzdW1lcyB3aGF0IHdhcyBvcmlnaW5hbGx5IHRoZSAnZmlyc3QnLFxuICogd2hpY2ggYWdhaW4gaXMgZGlzYWxsb3dlZC5cbiAqXG4gKiBJbiBlZmZlY3QsIHRoZXJlIGlzIG5vIHdheSB0byBnZXQgb3V0IG9mIHRoaXMgbWVzcyBpbiBhIGJhY2t3YXJkcyBjb21wYXRpYmxlIHdheSwgd2hpbGUgc3VwcG9ydGluZyBleGlzdGluZyBzdGFja3MuXG4gKiBUaGlzIGZsYWcgY2hhbmdlcyB0aGUgbG9naWNhbCBpZCBsYXlvdXQgb2YgVXNhZ2VQbGFuS2V5IHRvIG5vdCBiZSBzZW5zaXRpdmUgdG8gb3JkZXIuXG4gKi9cbmV4cG9ydCBjb25zdCBBUElHQVRFV0FZX1VTQUdFUExBTktFWV9PUkRFUklOU0VOU0lUSVZFX0lEID0gJ0Bhd3MtY2RrL2F3cy1hcGlnYXRld2F5OnVzYWdlUGxhbktleU9yZGVySW5zZW5zaXRpdmVJZCc7XG5cbi8qKlxuICogRW5hYmxlIHRoaXMgZmVhdHVyZSBmbGFnIHRvIGhhdmUgZWxhc3RpYyBmaWxlIHN5c3RlbXMgZW5jcnlwdGVkIGF0IHJlc3QgYnkgZGVmYXVsdC5cbiAqXG4gKiBFbmNyeXB0aW9uIGNhbiBhbHNvIGJlIGNvbmZpZ3VyZWQgZXhwbGljaXRseSB1c2luZyB0aGUgYGVuY3J5cHRlZGAgcHJvcGVydHkuXG4gKi9cbmV4cG9ydCBjb25zdCBFRlNfREVGQVVMVF9FTkNSWVBUSU9OX0FUX1JFU1QgPSAnQGF3cy1jZGsvYXdzLWVmczpkZWZhdWx0RW5jcnlwdGlvbkF0UmVzdCc7XG5cbi8qKlxuICogRW5hYmxlIHRoaXMgZmVhdHVyZSBmbGFnIHRvIG9wdCBpbiB0byB0aGUgdXBkYXRlZCBsb2dpY2FsIGlkIGNhbGN1bGF0aW9uIGZvciBMYW1iZGEgVmVyc2lvbiBjcmVhdGVkIHVzaW5nIHRoZVxuICogYGZuLmN1cnJlbnRWZXJzaW9uYC5cbiAqXG4gKiBUaGUgcHJldmlvdXMgY2FsY3VsYXRpb24gaW5jb3JyZWN0bHkgY29uc2lkZXJlZCBwcm9wZXJ0aWVzIG9mIHRoZSBgQVdTOjpMYW1iZGE6OkZ1bmN0aW9uYCByZXNvdXJjZSB0aGF0IGRpZFxuICogbm90IGNvbnN0aXR1dGUgY3JlYXRpbmcgYSBuZXcgVmVyc2lvbi5cbiAqXG4gKiBTZWUgJ2N1cnJlbnRWZXJzaW9uJyBzZWN0aW9uIGluIHRoZSBhd3MtbGFtYmRhIG1vZHVsZSdzIFJFQURNRSBmb3IgbW9yZSBkZXRhaWxzLlxuICovXG5leHBvcnQgY29uc3QgTEFNQkRBX1JFQ09HTklaRV9WRVJTSU9OX1BST1BTID0gJ0Bhd3MtY2RrL2F3cy1sYW1iZGE6cmVjb2duaXplVmVyc2lvblByb3BzJztcblxuLyoqXG4gKiBFbmFibGUgdGhpcyBmZWF0dXJlIGZsYWcgdG8gaGF2ZSBjbG91ZGZyb250IGRpc3RyaWJ1dGlvbnMgdXNlIHRoZSBzZWN1cml0eSBwb2xpY3kgVExTdjEuMl8yMDIxIGJ5IGRlZmF1bHQuXG4gKlxuICogVGhlIHNlY3VyaXR5IHBvbGljeSBjYW4gYWxzbyBiZSBjb25maWd1cmVkIGV4cGxpY2l0bHkgdXNpbmcgdGhlIGBtaW5pbXVtUHJvdG9jb2xWZXJzaW9uYCBwcm9wZXJ0eS5cbiAqL1xuZXhwb3J0IGNvbnN0IENMT1VERlJPTlRfREVGQVVMVF9TRUNVUklUWV9QT0xJQ1lfVExTX1YxXzJfMjAyMSA9ICdAYXdzLWNkay9hd3MtY2xvdWRmcm9udDpkZWZhdWx0U2VjdXJpdHlQb2xpY3lUTFN2MS4yXzIwMjEnO1xuXG4vKipcbiAqIFRoaXMgbWFwIGluY2x1ZGVzIGNvbnRleHQga2V5cyBhbmQgdmFsdWVzIGZvciBmZWF0dXJlIGZsYWdzIHRoYXQgZW5hYmxlXG4gKiBjYXBhYmlsaXRpZXMgXCJmcm9tIHRoZSBmdXR1cmVcIiwgd2hpY2ggd2UgY291bGQgbm90IGludHJvZHVjZSBhcyB0aGUgZGVmYXVsdFxuICogYmVoYXZpb3IgZHVlIHRvIGJhY2t3YXJkcyBjb21wYXRpYmlsaXR5IGZvciBleGlzdGluZyBwcm9qZWN0cy5cbiAqXG4gKiBOZXcgcHJvamVjdHMgZ2VuZXJhdGVkIHRocm91Z2ggYGNkayBpbml0YCB3aWxsIGluY2x1ZGUgdGhlc2UgZmxhZ3MgaW4gdGhlaXJcbiAqIGdlbmVyYXRlZCBgY2RrLmpzb25gIGZpbGUuXG4gKlxuICogV2hlbiB3ZSByZWxlYXNlIHRoZSBuZXh0IG1ham9yIHZlcnNpb24gb2YgdGhlIENESywgd2Ugd2lsbCBmbGlwIHRoZSBsb2dpYyBvZlxuICogdGhlc2UgZmVhdHVyZXMgYW5kIGNsZWFuIHVwIHRoZSBgY2RrLmpzb25gIGdlbmVyYXRlZCBieSBgY2RrIGluaXRgLlxuICpcbiAqIFRlc3RzIG11c3QgY292ZXIgdGhlIGRlZmF1bHQgKGRpc2FibGVkKSBjYXNlIGFuZCB0aGUgZnV0dXJlIChlbmFibGVkKSBjYXNlLlxuICovXG5leHBvcnQgY29uc3QgRlVUVVJFX0ZMQUdTOiB7IFtrZXk6IHN0cmluZ106IGFueSB9ID0ge1xuICBbQVBJR0FURVdBWV9VU0FHRVBMQU5LRVlfT1JERVJJTlNFTlNJVElWRV9JRF06IHRydWUsXG4gIFtFTkFCTEVfU1RBQ0tfTkFNRV9EVVBMSUNBVEVTX0NPTlRFWFRdOiAndHJ1ZScsXG4gIFtFTkFCTEVfRElGRl9OT19GQUlMX0NPTlRFWFRdOiAndHJ1ZScsXG4gIFtTVEFDS19SRUxBVElWRV9FWFBPUlRTX0NPTlRFWFRdOiAndHJ1ZScsXG4gIFtET0NLRVJfSUdOT1JFX1NVUFBPUlRdOiB0cnVlLFxuICBbU0VDUkVUU19NQU5BR0VSX1BBUlNFX09XTkVEX1NFQ1JFVF9OQU1FXTogdHJ1ZSxcbiAgW0tNU19ERUZBVUxUX0tFWV9QT0xJQ0lFU106IHRydWUsXG4gIFtTM19HUkFOVF9XUklURV9XSVRIT1VUX0FDTF06IHRydWUsXG4gIFtFQ1NfUkVNT1ZFX0RFRkFVTFRfREVTSVJFRF9DT1VOVF06IHRydWUsXG4gIFtSRFNfTE9XRVJDQVNFX0RCX0lERU5USUZJRVJdOiB0cnVlLFxuICBbRUZTX0RFRkFVTFRfRU5DUllQVElPTl9BVF9SRVNUXTogdHJ1ZSxcbiAgW0xBTUJEQV9SRUNPR05JWkVfVkVSU0lPTl9QUk9QU106IHRydWUsXG4gIFtDTE9VREZST05UX0RFRkFVTFRfU0VDVVJJVFlfUE9MSUNZX1RMU19WMV8yXzIwMjFdOiB0cnVlLFxuXG4gIC8vIFdlIHdpbGwgYWR2ZXJ0aXNlIHRoaXMgZmxhZyB3aGVuIHRoZSBmZWF0dXJlIGlzIGNvbXBsZXRlXG4gIC8vIFtORVdfU1RZTEVfU1RBQ0tfU1lOVEhFU0lTX0NPTlRFWFRdOiAndHJ1ZScsXG59O1xuXG4vKipcbiAqIFRoZSBsaXN0IG9mIGZ1dHVyZSBmbGFncyB0aGF0IGFyZSBub3cgZXhwaXJlZC4gVGhpcyBpcyBnb2luZyB0byBiZSB1c2VkIHRvIGlkZW50aWZ5XG4gKiBhbmQgYmxvY2sgdXNhZ2VzIG9mIG9sZCBmZWF0dXJlIGZsYWdzIGluIHRoZSBuZXcgbWFqb3IgdmVyc2lvbiBvZiBDREsuXG4gKi9cbmV4cG9ydCBjb25zdCBGVVRVUkVfRkxBR1NfRVhQSVJFRDogc3RyaW5nW10gPSBbXG5dO1xuXG4vKipcbiAqIFRoZSBzZXQgb2YgZGVmYXVsdHMgdGhhdCBzaG91bGQgYmUgYXBwbGllZCBpZiB0aGUgZmVhdHVyZSBmbGFnIGlzIG5vdFxuICogZXhwbGljaXRseSBjb25maWd1cmVkLlxuICovXG5jb25zdCBGVVRVUkVfRkxBR1NfREVGQVVMVFM6IHsgW2tleTogc3RyaW5nXTogYm9vbGVhbiB9ID0ge1xuICBbQVBJR0FURVdBWV9VU0FHRVBMQU5LRVlfT1JERVJJTlNFTlNJVElWRV9JRF06IGZhbHNlLFxuICBbRU5BQkxFX1NUQUNLX05BTUVfRFVQTElDQVRFU19DT05URVhUXTogZmFsc2UsXG4gIFtFTkFCTEVfRElGRl9OT19GQUlMX0NPTlRFWFRdOiBmYWxzZSxcbiAgW1NUQUNLX1JFTEFUSVZFX0VYUE9SVFNfQ09OVEVYVF06IGZhbHNlLFxuICBbTkVXX1NUWUxFX1NUQUNLX1NZTlRIRVNJU19DT05URVhUXTogZmFsc2UsXG4gIFtET0NLRVJfSUdOT1JFX1NVUFBPUlRdOiBmYWxzZSxcbiAgW1NFQ1JFVFNfTUFOQUdFUl9QQVJTRV9PV05FRF9TRUNSRVRfTkFNRV06IGZhbHNlLFxuICBbS01TX0RFRkFVTFRfS0VZX1BPTElDSUVTXTogZmFsc2UsXG4gIFtTM19HUkFOVF9XUklURV9XSVRIT1VUX0FDTF06IGZhbHNlLFxuICBbRUNTX1JFTU9WRV9ERUZBVUxUX0RFU0lSRURfQ09VTlRdOiBmYWxzZSxcbiAgW1JEU19MT1dFUkNBU0VfREJfSURFTlRJRklFUl06IGZhbHNlLFxuICBbRUZTX0RFRkFVTFRfRU5DUllQVElPTl9BVF9SRVNUXTogZmFsc2UsXG4gIFtMQU1CREFfUkVDT0dOSVpFX1ZFUlNJT05fUFJPUFNdOiBmYWxzZSxcbiAgW0NMT1VERlJPTlRfREVGQVVMVF9TRUNVUklUWV9QT0xJQ1lfVExTX1YxXzJfMjAyMV06IGZhbHNlLFxufTtcblxuZXhwb3J0IGZ1bmN0aW9uIGZ1dHVyZUZsYWdEZWZhdWx0KGZsYWc6IHN0cmluZyk6IGJvb2xlYW4ge1xuICByZXR1cm4gRlVUVVJFX0ZMQUdTX0RFRkFVTFRTW2ZsYWddO1xufVxuIl19