"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SdkProvider = void 0;
const https = require("https");
const os = require("os");
const path = require("path");
const cxapi = require("@aws-cdk/cx-api");
const AWS = require("aws-sdk");
const fs = require("fs-extra");
const logging_1 = require("../../logging");
const functions_1 = require("../../util/functions");
const credential_plugins_1 = require("../aws-auth/credential-plugins");
const awscli_compatible_1 = require("./awscli-compatible");
const sdk_1 = require("./sdk");
// Some configuration that can only be achieved by setting
// environment variables.
process.env.AWS_STS_REGIONAL_ENDPOINTS = 'regional';
process.env.AWS_NODEJS_CONNECTION_REUSE_ENABLED = '1';
const CACHED_ACCOUNT = Symbol('cached_account');
const CACHED_DEFAULT_CREDENTIALS = Symbol('cached_default_credentials');
/**
 * Creates instances of the AWS SDK appropriate for a given account/region.
 *
 * Behavior is as follows:
 *
 * - First, a set of "base" credentials are established
 *   - If a target environment is given and the default ("current") SDK credentials are for
 *     that account, return those; otherwise
 *   - If a target environment is given, scan all credential provider plugins
 *     for credentials, and return those if found; otherwise
 *   - Return default ("current") SDK credentials, noting that they might be wrong.
 *
 * - Second, a role may optionally need to be assumed. Use the base credentials
 *   established in the previous process to assume that role.
 *   - If assuming the role fails and the base credentials are for the correct
 *     account, return those. This is a fallback for people who are trying to interact
 *     with a Default Synthesized stack and already have right credentials setup.
 *
 *     Typical cases we see in the wild:
 *     - Credential plugin setup that, although not recommended, works for them
 *     - Seeded terminal with `ReadOnly` credentials in order to do `cdk diff`--the `ReadOnly`
 *       role doesn't have `sts:AssumeRole` and will fail for no real good reason.
 */
class SdkProvider {
    constructor(defaultChain, 
    /**
     * Default region
     */
    defaultRegion, sdkOptions = {}) {
        this.defaultChain = defaultChain;
        this.defaultRegion = defaultRegion;
        this.sdkOptions = sdkOptions;
        this.plugins = new credential_plugins_1.CredentialPlugins();
    }
    /**
     * Create a new SdkProvider which gets its defaults in a way that behaves like the AWS CLI does
     *
     * The AWS SDK for JS behaves slightly differently from the AWS CLI in a number of ways; see the
     * class `AwsCliCompatible` for the details.
     */
    static async withAwsCliCompatibleDefaults(options = {}) {
        var _a;
        const sdkOptions = parseHttpOptions((_a = options.httpOptions) !== null && _a !== void 0 ? _a : {});
        const chain = await awscli_compatible_1.AwsCliCompatible.credentialChain({
            profile: options.profile,
            ec2instance: options.ec2creds,
            containerCreds: options.containerCreds,
            httpOptions: sdkOptions.httpOptions,
        });
        const region = await awscli_compatible_1.AwsCliCompatible.region({
            profile: options.profile,
            ec2instance: options.ec2creds,
        });
        return new SdkProvider(chain, region, sdkOptions);
    }
    /**
     * Return an SDK which can do operations in the given environment
     *
     * The `environment` parameter is resolved first (see `resolveEnvironment()`).
     */
    async forEnvironment(environment, mode, options) {
        const env = await this.resolveEnvironment(environment);
        const baseCreds = await this.obtainBaseCredentials(env.account, mode);
        // At this point, we need at least SOME credentials
        if (baseCreds.source === 'none') {
            throw new Error(fmtObtainCredentialsError(env.account, baseCreds));
        }
        // Simple case is if we don't need to "assumeRole" here. If so, we must now have credentials for the right
        // account.
        if ((options === null || options === void 0 ? void 0 : options.assumeRoleArn) === undefined) {
            if (baseCreds.source === 'incorrectDefault') {
                throw new Error(fmtObtainCredentialsError(env.account, baseCreds));
            }
            return new sdk_1.SDK(baseCreds.credentials, env.region, this.sdkOptions);
        }
        // We will proceed to AssumeRole using whatever we've been given.
        const sdk = await this.withAssumedRole(baseCreds, options.assumeRoleArn, options.assumeRoleExternalId, env.region);
        // Exercise the AssumeRoleCredentialsProvider we've gotten at least once so
        // we can determine whether the AssumeRole call succeeds or not.
        try {
            await sdk.forceCredentialRetrieval();
            return sdk;
        }
        catch (e) {
            // AssumeRole failed. Proceed and warn *if and only if* the baseCredentials were already for the right account
            // or returned from a plugin. This is to cover some current setups for people using plugins or preferring to
            // feed the CLI credentials which are sufficient by themselves. Prefer to assume the correct role if we can,
            // but if we can't then let's just try with available credentials anyway.
            if (baseCreds.source === 'correctDefault' || baseCreds.source === 'plugin') {
                logging_1.debug(e.message);
                logging_1.warning(`${fmtObtainedCredentials(baseCreds)} could not be used to assume '${options.assumeRoleArn}', but are for the right account. Proceeding anyway.`);
                return new sdk_1.SDK(baseCreds.credentials, env.region, this.sdkOptions);
            }
            throw e;
        }
    }
    /**
     * Return the partition that base credentials are for
     *
     * Returns `undefined` if there are no base credentials.
     */
    async baseCredentialsPartition(environment, mode) {
        const env = await this.resolveEnvironment(environment);
        const baseCreds = await this.obtainBaseCredentials(env.account, mode);
        if (baseCreds.source === 'none') {
            return undefined;
        }
        return (await new sdk_1.SDK(baseCreds.credentials, env.region, this.sdkOptions).currentAccount()).partition;
    }
    /**
     * Resolve the environment for a stack
     *
     * Replaces the magic values `UNKNOWN_REGION` and `UNKNOWN_ACCOUNT`
     * with the defaults for the current SDK configuration (`~/.aws/config` or
     * otherwise).
     *
     * It is an error if `UNKNOWN_ACCOUNT` is used but the user hasn't configured
     * any SDK credentials.
     */
    async resolveEnvironment(env) {
        var _a;
        const region = env.region !== cxapi.UNKNOWN_REGION ? env.region : this.defaultRegion;
        const account = env.account !== cxapi.UNKNOWN_ACCOUNT ? env.account : (_a = (await this.defaultAccount())) === null || _a === void 0 ? void 0 : _a.accountId;
        if (!account) {
            throw new Error('Unable to resolve AWS account to use. It must be either configured when you define your CDK Stack, or through the environment');
        }
        return {
            region,
            account,
            name: cxapi.EnvironmentUtils.format(account, region),
        };
    }
    /**
     * The account we'd auth into if we used default credentials.
     *
     * Default credentials are the set of ambiently configured credentials using
     * one of the environment variables, or ~/.aws/credentials, or the *one*
     * profile that was passed into the CLI.
     *
     * Might return undefined if there are no default/ambient credentials
     * available (in which case the user should better hope they have
     * credential plugins configured).
     *
     * Uses a cache to avoid STS calls if we don't need 'em.
     */
    defaultAccount() {
        return functions_1.cached(this, CACHED_ACCOUNT, async () => {
            try {
                const creds = await this.defaultCredentials();
                const accessKeyId = creds.accessKeyId;
                if (!accessKeyId) {
                    throw new Error('Unable to resolve AWS credentials (setup with "aws configure")');
                }
                return await new sdk_1.SDK(creds, this.defaultRegion, this.sdkOptions).currentAccount();
            }
            catch (e) {
                logging_1.debug('Unable to determine the default AWS account:', e);
                return undefined;
            }
        });
    }
    /**
     * Get credentials for the given account ID in the given mode
     *
     * 1. Use the default credentials if the destination account matches the
     *    current credentials' account.
     * 2. Otherwise try all credential plugins.
     * 3. Fail if neither of these yield any credentials.
     * 4. Return a failure if any of them returned credentials
     */
    async obtainBaseCredentials(accountId, mode) {
        var _a;
        // First try 'current' credentials
        const defaultAccountId = (_a = (await this.defaultAccount())) === null || _a === void 0 ? void 0 : _a.accountId;
        if (defaultAccountId === accountId) {
            return { source: 'correctDefault', credentials: await this.defaultCredentials() };
        }
        // Then try the plugins
        const pluginCreds = await this.plugins.fetchCredentialsFor(accountId, mode);
        if (pluginCreds) {
            return { source: 'plugin', ...pluginCreds };
        }
        // Fall back to default credentials with a note that they're not the right ones yet
        if (defaultAccountId !== undefined) {
            return {
                source: 'incorrectDefault',
                accountId: defaultAccountId,
                credentials: await this.defaultCredentials(),
                unusedPlugins: this.plugins.availablePluginNames,
            };
        }
        // Apparently we didn't find any at all
        return {
            source: 'none',
            unusedPlugins: this.plugins.availablePluginNames,
        };
    }
    /**
     * Resolve the default chain to the first set of credentials that is available
     */
    defaultCredentials() {
        return functions_1.cached(this, CACHED_DEFAULT_CREDENTIALS, () => {
            logging_1.debug('Resolving default credentials');
            return this.defaultChain.resolvePromise();
        });
    }
    /**
     * Return an SDK which uses assumed role credentials
     *
     * The base credentials used to retrieve the assumed role credentials will be the
     * same credentials returned by obtainCredentials if an environment and mode is passed,
     * otherwise it will be the current credentials.
     */
    async withAssumedRole(masterCredentials, roleArn, externalId, region) {
        logging_1.debug(`Assuming role '${roleArn}'.`);
        region = region !== null && region !== void 0 ? region : this.defaultRegion;
        const creds = new AWS.ChainableTemporaryCredentials({
            params: {
                RoleArn: roleArn,
                ...externalId ? { ExternalId: externalId } : {},
                RoleSessionName: `aws-cdk-${safeUsername()}`,
            },
            stsConfig: {
                region,
                ...this.sdkOptions,
            },
            masterCredentials: masterCredentials.credentials,
        });
        return new sdk_1.SDK(creds, region, this.sdkOptions, {
            assumeRoleCredentialsSourceDescription: fmtObtainedCredentials(masterCredentials),
        });
    }
}
exports.SdkProvider = SdkProvider;
const DEFAULT_CONNECTION_TIMEOUT = 10000;
const DEFAULT_TIMEOUT = 300000;
/**
 * Get HTTP options for the SDK
 *
 * Read from user input or environment variables.
 *
 * Returns a complete `ConfigurationOptions` object because that's where
 * `customUserAgent` lives, but `httpOptions` is the most important attribute.
 */
function parseHttpOptions(options) {
    var _a;
    const config = {};
    config.httpOptions = {};
    config.httpOptions.connectTimeout = DEFAULT_CONNECTION_TIMEOUT;
    config.httpOptions.timeout = DEFAULT_TIMEOUT;
    let userAgent = options.userAgent;
    if (userAgent == null) {
        // Find the package.json from the main toolkit
        const pkg = JSON.parse((_a = readIfPossible(path.join(__dirname, '..', '..', '..', 'package.json'))) !== null && _a !== void 0 ? _a : '{}');
        userAgent = `${pkg.name}/${pkg.version}`;
    }
    config.customUserAgent = userAgent;
    const proxyAddress = options.proxyAddress || httpsProxyFromEnvironment();
    const caBundlePath = options.caBundlePath || caBundlePathFromEnvironment();
    if (proxyAddress && caBundlePath) {
        throw new Error(`At the moment, cannot specify Proxy (${proxyAddress}) and CA Bundle (${caBundlePath}) at the same time. See https://github.com/aws/aws-cdk/issues/5804`);
        // Maybe it's possible after all, but I've been staring at
        // https://github.com/TooTallNate/node-proxy-agent/blob/master/index.js#L79
        // a while now trying to figure out what to pass in so that the underlying Agent
        // object will get the 'ca' argument. It's not trivial and I don't want to risk it.
    }
    if (proxyAddress) { // Ignore empty string on purpose
        // https://aws.amazon.com/blogs/developer/using-the-aws-sdk-for-javascript-from-behind-a-proxy/
        logging_1.debug('Using proxy server: %s', proxyAddress);
        // eslint-disable-next-line @typescript-eslint/no-require-imports
        const ProxyAgent = require('proxy-agent');
        config.httpOptions.agent = new ProxyAgent(proxyAddress);
    }
    if (caBundlePath) {
        logging_1.debug('Using CA bundle path: %s', caBundlePath);
        config.httpOptions.agent = new https.Agent({
            ca: readIfPossible(caBundlePath),
            keepAlive: true,
        });
    }
    return config;
}
/**
 * Find and return the configured HTTPS proxy address
 */
function httpsProxyFromEnvironment() {
    if (process.env.https_proxy) {
        return process.env.https_proxy;
    }
    if (process.env.HTTPS_PROXY) {
        return process.env.HTTPS_PROXY;
    }
    return undefined;
}
/**
 * Find and return a CA certificate bundle path to be passed into the SDK.
 */
function caBundlePathFromEnvironment() {
    if (process.env.aws_ca_bundle) {
        return process.env.aws_ca_bundle;
    }
    if (process.env.AWS_CA_BUNDLE) {
        return process.env.AWS_CA_BUNDLE;
    }
    return undefined;
}
/**
 * Read a file if it exists, or return undefined
 *
 * Not async because it is used in the constructor
 */
function readIfPossible(filename) {
    try {
        if (!fs.pathExistsSync(filename)) {
            return undefined;
        }
        return fs.readFileSync(filename, { encoding: 'utf-8' });
    }
    catch (e) {
        logging_1.debug(e);
        return undefined;
    }
}
/**
 * Return the username with characters invalid for a RoleSessionName removed
 *
 * @see https://docs.aws.amazon.com/STS/latest/APIReference/API_AssumeRole.html#API_AssumeRole_RequestParameters
 */
function safeUsername() {
    return os.userInfo().username.replace(/[^\w+=,.@-]/g, '@');
}
/**
 * Isolating the code that translates calculation errors into human error messages
 *
 * We cover the following cases:
 *
 * - No credentials are available at all
 * - Default credentials are for the wrong account
 */
function fmtObtainCredentialsError(targetAccountId, obtainResult) {
    const msg = [`Need to perform AWS calls for account ${targetAccountId}`];
    switch (obtainResult.source) {
        case 'incorrectDefault':
            msg.push(`but the current credentials are for ${obtainResult.accountId}`);
            break;
        case 'none':
            msg.push('but no credentials have been configured');
    }
    if (obtainResult.unusedPlugins.length > 0) {
        msg.push(`and none of these plugins found any: ${obtainResult.unusedPlugins.join(', ')}`);
    }
    return msg.join(', ');
}
/**
 * Format a message indicating where we got base credentials for the assume role
 *
 * We cover the following cases:
 *
 * - Default credentials for the right account
 * - Default credentials for the wrong account
 * - Credentials returned from a plugin
 */
function fmtObtainedCredentials(obtainResult) {
    switch (obtainResult.source) {
        case 'correctDefault':
            return 'current credentials';
        case 'plugin':
            return `credentials returned by plugin '${obtainResult.pluginName}'`;
        case 'incorrectDefault':
            const msg = [];
            msg.push(`current credentials (which are for account ${obtainResult.accountId}`);
            if (obtainResult.unusedPlugins.length > 0) {
                msg.push(`, and none of the following plugins provided credentials: ${obtainResult.unusedPlugins.join(', ')}`);
            }
            msg.push(')');
            return msg.join('');
    }
}
//# sourceMappingURL=data:application/json;base64,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