"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.StackCollection = exports.CloudAssembly = exports.ExtendedStackSelection = exports.DefaultSelection = void 0;
const cxapi = require("@aws-cdk/cx-api");
const colors = require("colors/safe");
const minimatch = require("minimatch");
const semver = require("semver");
const logging_1 = require("../../logging");
const util_1 = require("../../util");
const version_1 = require("../../version");
var DefaultSelection;
(function (DefaultSelection) {
    /**
     * Returns an empty selection in case there are no selectors.
     */
    DefaultSelection["None"] = "none";
    /**
     * If the app includes a single stack, returns it. Otherwise throws an exception.
     * This behavior is used by "deploy".
     */
    DefaultSelection["OnlySingle"] = "single";
    /**
     * Returns all stacks in the main (top level) assembly only.
     */
    DefaultSelection["MainAssembly"] = "main";
    /**
     * If no selectors are provided, returns all stacks in the app,
     * including stacks inside nested assemblies.
     */
    DefaultSelection["AllStacks"] = "all";
})(DefaultSelection = exports.DefaultSelection || (exports.DefaultSelection = {}));
/**
 * When selecting stacks, what other stacks to include because of dependencies
 */
var ExtendedStackSelection;
(function (ExtendedStackSelection) {
    /**
     * Don't select any extra stacks
     */
    ExtendedStackSelection[ExtendedStackSelection["None"] = 0] = "None";
    /**
     * Include stacks that this stack depends on
     */
    ExtendedStackSelection[ExtendedStackSelection["Upstream"] = 1] = "Upstream";
    /**
     * Include stacks that depend on this stack
     */
    ExtendedStackSelection[ExtendedStackSelection["Downstream"] = 2] = "Downstream";
})(ExtendedStackSelection = exports.ExtendedStackSelection || (exports.ExtendedStackSelection = {}));
/**
 * A single Cloud Assembly and the operations we do on it to deploy the artifacts inside
 */
class CloudAssembly {
    constructor(assembly) {
        this.assembly = assembly;
        this.directory = assembly.directory;
    }
    async selectStacks(selector, options) {
        var _a;
        const asm = this.assembly;
        const topLevelStacks = asm.stacks;
        const stacks = semver.major(asm.version) < 10 ? asm.stacks : asm.stacksRecursively;
        const allTopLevel = (_a = selector.allTopLevel) !== null && _a !== void 0 ? _a : false;
        const patterns = sanitizePatterns(selector.patterns);
        if (stacks.length === 0) {
            throw new Error('This app contains no stacks');
        }
        if (allTopLevel) {
            return this.selectTopLevelStacks(stacks, topLevelStacks, options.extend);
        }
        else if (patterns.length > 0) {
            return this.selectMatchingStacks(stacks, patterns, options.extend);
        }
        else {
            return this.selectDefaultStacks(stacks, topLevelStacks, options.defaultBehavior);
        }
    }
    selectTopLevelStacks(stacks, topLevelStacks, extend = ExtendedStackSelection.None) {
        if (topLevelStacks.length > 0) {
            return this.extendStacks(topLevelStacks, stacks, extend);
        }
        else {
            throw new Error('No stack found in the main cloud assembly. Use "list" to print manifest');
        }
    }
    selectMatchingStacks(stacks, patterns, extend = ExtendedStackSelection.None) {
        // cli tests use this to ensure tests do not depend on legacy behavior
        // (otherwise they will fail in v2)
        const disableLegacy = process.env.CXAPI_DISABLE_SELECT_BY_ID === '1';
        const matchingPattern = (pattern) => (stack) => {
            if (minimatch(stack.hierarchicalId, pattern)) {
                return true;
            }
            else if (!disableLegacy && stack.id === pattern && semver.major(version_1.versionNumber()) < 2) {
                logging_1.warning('Selecting stack by identifier "%s". This identifier is deprecated and will be removed in v2. Please use "%s" instead.', colors.bold(stack.id), colors.bold(stack.hierarchicalId));
                logging_1.warning('Run "cdk ls" to see a list of all stack identifiers');
                return true;
            }
            return false;
        };
        const matchedStacks = util_1.flatten(patterns.map(pattern => stacks.filter(matchingPattern(pattern))));
        return this.extendStacks(matchedStacks, stacks, extend);
    }
    selectDefaultStacks(stacks, topLevelStacks, defaultSelection) {
        switch (defaultSelection) {
            case DefaultSelection.MainAssembly:
                return new StackCollection(this, topLevelStacks);
            case DefaultSelection.AllStacks:
                return new StackCollection(this, stacks);
            case DefaultSelection.None:
                return new StackCollection(this, []);
            case DefaultSelection.OnlySingle:
                if (topLevelStacks.length === 1) {
                    return new StackCollection(this, topLevelStacks);
                }
                else {
                    throw new Error('Since this app includes more than a single stack, specify which stacks to use (wildcards are supported) or specify `--all`\n' +
                        `Stacks: ${stacks.map(x => x.hierarchicalId).join(' · ')}`);
                }
            default:
                throw new Error(`invalid default behavior: ${defaultSelection}`);
        }
    }
    extendStacks(matched, all, extend = ExtendedStackSelection.None) {
        const allStacks = new Map();
        for (const stack of all) {
            allStacks.set(stack.hierarchicalId, stack);
        }
        const index = indexByHierarchicalId(matched);
        switch (extend) {
            case ExtendedStackSelection.Downstream:
                includeDownstreamStacks(index, allStacks);
                break;
            case ExtendedStackSelection.Upstream:
                includeUpstreamStacks(index, allStacks);
                break;
        }
        // Filter original array because it is in the right order
        const selectedList = all.filter(s => index.has(s.hierarchicalId));
        return new StackCollection(this, selectedList);
    }
    /**
     * Select a single stack by its ID
     */
    stackById(stackId) {
        return new StackCollection(this, [this.assembly.getStackArtifact(stackId)]);
    }
}
exports.CloudAssembly = CloudAssembly;
/**
 * A collection of stacks and related artifacts
 *
 * In practice, not all artifacts in the CloudAssembly are created equal;
 * stacks can be selected independently, but other artifacts such as asset
 * bundles cannot.
 */
class StackCollection {
    constructor(assembly, stackArtifacts) {
        this.assembly = assembly;
        this.stackArtifacts = stackArtifacts;
    }
    get stackCount() {
        return this.stackArtifacts.length;
    }
    get firstStack() {
        if (this.stackCount < 1) {
            throw new Error('StackCollection contains no stack artifacts (trying to access the first one)');
        }
        return this.stackArtifacts[0];
    }
    get stackIds() {
        return this.stackArtifacts.map(s => s.id);
    }
    reversed() {
        const arts = [...this.stackArtifacts];
        arts.reverse();
        return new StackCollection(this.assembly, arts);
    }
    filter(predicate) {
        return new StackCollection(this.assembly, this.stackArtifacts.filter(predicate));
    }
    concat(other) {
        return new StackCollection(this.assembly, this.stackArtifacts.concat(other.stackArtifacts));
    }
    /**
     * Extracts 'aws:cdk:warning|info|error' metadata entries from the stack synthesis
     */
    processMetadataMessages(options = {}) {
        let warnings = false;
        let errors = false;
        for (const stack of this.stackArtifacts) {
            for (const message of stack.messages) {
                switch (message.level) {
                    case cxapi.SynthesisMessageLevel.WARNING:
                        warnings = true;
                        printMessage(logging_1.warning, 'Warning', message.id, message.entry);
                        break;
                    case cxapi.SynthesisMessageLevel.ERROR:
                        errors = true;
                        printMessage(logging_1.error, 'Error', message.id, message.entry);
                        break;
                    case cxapi.SynthesisMessageLevel.INFO:
                        printMessage(logging_1.print, 'Info', message.id, message.entry);
                        break;
                }
            }
        }
        if (errors && !options.ignoreErrors) {
            throw new Error('Found errors');
        }
        if (options.strict && warnings) {
            throw new Error('Found warnings (--strict mode)');
        }
        function printMessage(logFn, prefix, id, entry) {
            logFn(`[${prefix} at ${id}] ${entry.data}`);
            if (options.verbose && entry.trace) {
                logFn(`  ${entry.trace.join('\n  ')}`);
            }
        }
    }
}
exports.StackCollection = StackCollection;
function indexByHierarchicalId(stacks) {
    const result = new Map();
    for (const stack of stacks) {
        result.set(stack.hierarchicalId, stack);
    }
    return result;
}
/**
 * Calculate the transitive closure of stack dependents.
 *
 * Modifies `selectedStacks` in-place.
 */
function includeDownstreamStacks(selectedStacks, allStacks) {
    const added = new Array();
    let madeProgress;
    do {
        madeProgress = false;
        for (const [id, stack] of allStacks) {
            // Select this stack if it's not selected yet AND it depends on a stack that's in the selected set
            if (!selectedStacks.has(id) && (stack.dependencies || []).some(dep => selectedStacks.has(dep.id))) {
                selectedStacks.set(id, stack);
                added.push(id);
                madeProgress = true;
            }
        }
    } while (madeProgress);
    if (added.length > 0) {
        logging_1.print('Including depending stacks: %s', colors.bold(added.join(', ')));
    }
}
/**
 * Calculate the transitive closure of stack dependencies.
 *
 * Modifies `selectedStacks` in-place.
 */
function includeUpstreamStacks(selectedStacks, allStacks) {
    const added = new Array();
    let madeProgress = true;
    while (madeProgress) {
        madeProgress = false;
        for (const stack of selectedStacks.values()) {
            // Select an additional stack if it's not selected yet and a dependency of a selected stack (and exists, obviously)
            for (const dependencyId of stack.dependencies.map(x => { var _a; return (_a = x.manifest.displayName) !== null && _a !== void 0 ? _a : x.id; })) {
                if (!selectedStacks.has(dependencyId) && allStacks.has(dependencyId)) {
                    added.push(dependencyId);
                    selectedStacks.set(dependencyId, allStacks.get(dependencyId));
                    madeProgress = true;
                }
            }
        }
    }
    if (added.length > 0) {
        logging_1.print('Including dependency stacks: %s', colors.bold(added.join(', ')));
    }
}
function sanitizePatterns(patterns) {
    let sanitized = patterns.filter(s => s != null); // filter null/undefined
    sanitized = [...new Set(sanitized)]; // make them unique
    return sanitized;
}
//# sourceMappingURL=data:application/json;base64,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