"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.execProgram = void 0;
const childProcess = require("child_process");
const path = require("path");
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const cxapi = require("@aws-cdk/cx-api");
const fs = require("fs-extra");
const logging_1 = require("../../logging");
const settings_1 = require("../../settings");
const version_1 = require("../../version");
/** Invokes the cloud executable and returns JSON output */
async function execProgram(aws, config) {
    var _a, _b, _c, _d, _e, _f;
    const env = {};
    const context = config.context.all;
    await populateDefaultEnvironmentIfNeeded(aws, env);
    const debugMode = (_a = config.settings.get(['debug'])) !== null && _a !== void 0 ? _a : true;
    if (debugMode) {
        env.CDK_DEBUG = 'true';
    }
    const pathMetadata = (_b = config.settings.get(['pathMetadata'])) !== null && _b !== void 0 ? _b : true;
    if (pathMetadata) {
        context[cxapi.PATH_METADATA_ENABLE_CONTEXT] = true;
    }
    const assetMetadata = (_c = config.settings.get(['assetMetadata'])) !== null && _c !== void 0 ? _c : true;
    if (assetMetadata) {
        context[cxapi.ASSET_RESOURCE_METADATA_ENABLED_CONTEXT] = true;
    }
    const versionReporting = (_d = config.settings.get(['versionReporting'])) !== null && _d !== void 0 ? _d : true;
    if (versionReporting) {
        context[cxapi.ANALYTICS_REPORTING_ENABLED_CONTEXT] = true;
    }
    // We need to keep on doing this for framework version from before this flag was deprecated.
    if (!versionReporting) {
        context['aws:cdk:disable-version-reporting'] = true;
    }
    const stagingEnabled = (_e = config.settings.get(['staging'])) !== null && _e !== void 0 ? _e : true;
    if (!stagingEnabled) {
        context[cxapi.DISABLE_ASSET_STAGING_CONTEXT] = true;
    }
    const bundlingStacks = (_f = config.settings.get(['bundlingStacks'])) !== null && _f !== void 0 ? _f : ['*'];
    context[cxapi.BUNDLING_STACKS] = bundlingStacks;
    logging_1.debug('context:', context);
    env[cxapi.CONTEXT_ENV] = JSON.stringify(context);
    const app = config.settings.get(['app']);
    if (!app) {
        throw new Error(`--app is required either in command-line, in ${settings_1.PROJECT_CONFIG} or in ${settings_1.USER_DEFAULTS}`);
    }
    // bypass "synth" if app points to a cloud assembly
    if (await fs.pathExists(app) && (await fs.stat(app)).isDirectory()) {
        logging_1.debug('--app points to a cloud assembly, so we bypass synth');
        return createAssembly(app);
    }
    const commandLine = await guessExecutable(appToArray(app));
    const outdir = config.settings.get(['output']);
    if (!outdir) {
        throw new Error('unexpected: --output is required');
    }
    await fs.mkdirp(outdir);
    logging_1.debug('outdir:', outdir);
    env[cxapi.OUTDIR_ENV] = outdir;
    // Send version information
    env[cxapi.CLI_ASM_VERSION_ENV] = cxschema.Manifest.version();
    env[cxapi.CLI_VERSION_ENV] = version_1.versionNumber();
    logging_1.debug('env:', env);
    await exec();
    return createAssembly(outdir);
    function createAssembly(appDir) {
        try {
            return new cxapi.CloudAssembly(appDir);
        }
        catch (error) {
            if (error.message.includes(cxschema.VERSION_MISMATCH)) {
                // this means the CLI version is too old.
                // we instruct the user to upgrade.
                throw new Error(`This CDK CLI is not compatible with the CDK library used by your application. Please upgrade the CLI to the latest version.\n(${error.message})`);
            }
            throw error;
        }
    }
    async function exec() {
        return new Promise((ok, fail) => {
            // We use a slightly lower-level interface to:
            //
            // - Pass arguments in an array instead of a string, to get around a
            //   number of quoting issues introduced by the intermediate shell layer
            //   (which would be different between Linux and Windows).
            //
            // - Inherit stderr from controlling terminal. We don't use the captured value
            //   anyway, and if the subprocess is printing to it for debugging purposes the
            //   user gets to see it sooner. Plus, capturing doesn't interact nicely with some
            //   processes like Maven.
            const proc = childProcess.spawn(commandLine[0], commandLine.slice(1), {
                stdio: ['ignore', 'inherit', 'inherit'],
                detached: false,
                shell: true,
                env: {
                    ...process.env,
                    ...env,
                },
            });
            proc.on('error', fail);
            proc.on('exit', code => {
                if (code === 0) {
                    return ok();
                }
                else {
                    return fail(new Error(`Subprocess exited with error ${code}`));
                }
            });
        });
    }
}
exports.execProgram = execProgram;
/**
 * If we don't have region/account defined in context, we fall back to the default SDK behavior
 * where region is retrieved from ~/.aws/config and account is based on default credentials provider
 * chain and then STS is queried.
 *
 * This is done opportunistically: for example, if we can't access STS for some reason or the region
 * is not configured, the context value will be 'null' and there could failures down the line. In
 * some cases, synthesis does not require region/account information at all, so that might be perfectly
 * fine in certain scenarios.
 *
 * @param context The context key/value bash.
 */
async function populateDefaultEnvironmentIfNeeded(aws, env) {
    var _a;
    env[cxapi.DEFAULT_REGION_ENV] = aws.defaultRegion;
    logging_1.debug(`Setting "${cxapi.DEFAULT_REGION_ENV}" environment variable to`, env[cxapi.DEFAULT_REGION_ENV]);
    const accountId = (_a = (await aws.defaultAccount())) === null || _a === void 0 ? void 0 : _a.accountId;
    if (accountId) {
        env[cxapi.DEFAULT_ACCOUNT_ENV] = accountId;
        logging_1.debug(`Setting "${cxapi.DEFAULT_ACCOUNT_ENV}" environment variable to`, env[cxapi.DEFAULT_ACCOUNT_ENV]);
    }
}
/**
 * Make sure the 'app' is an array
 *
 * If it's a string, split on spaces as a trivial way of tokenizing the command line.
 */
function appToArray(app) {
    return typeof app === 'string' ? app.split(' ') : app;
}
/**
 * Execute the given file with the same 'node' process as is running the current process
 */
function executeNode(scriptFile) {
    return [process.execPath, scriptFile];
}
/**
 * Mapping of extensions to command-line generators
 */
const EXTENSION_MAP = new Map([
    ['.js', executeNode],
]);
/**
 * Guess the executable from the command-line argument
 *
 * Only do this if the file is NOT marked as executable. If it is,
 * we'll defer to the shebang inside the file itself.
 *
 * If we're on Windows, we ALWAYS take the handler, since it's hard to
 * verify if registry associations have or have not been set up for this
 * file type, so we'll assume the worst and take control.
 */
async function guessExecutable(commandLine) {
    if (commandLine.length === 1) {
        let fstat;
        try {
            fstat = await fs.stat(commandLine[0]);
        }
        catch (error) {
            logging_1.debug(`Not a file: '${commandLine[0]}'. Using '${commandLine}' as command-line`);
            return commandLine;
        }
        // eslint-disable-next-line no-bitwise
        const isExecutable = (fstat.mode & fs.constants.X_OK) !== 0;
        const isWindows = process.platform === 'win32';
        const handler = EXTENSION_MAP.get(path.extname(commandLine[0]));
        if (handler && (!isExecutable || isWindows)) {
            return handler(commandLine[0]);
        }
    }
    return commandLine;
}
//# sourceMappingURL=data:application/json;base64,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