"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ToolkitInfo = exports.DEFAULT_TOOLKIT_STACK_NAME = void 0;
const colors = require("colors/safe");
const logging_1 = require("../logging");
const bootstrap_1 = require("./bootstrap");
const cloudformation_1 = require("./util/cloudformation");
exports.DEFAULT_TOOLKIT_STACK_NAME = 'CDKToolkit';
/**
 * The bootstrap template version that introduced ssm:GetParameter
 */
const BOOTSTRAP_TEMPLATE_VERSION_INTRODUCING_GETPARAMETER = 5;
/**
 * Information on the Bootstrap stack of the environment we're deploying to.
 *
 * This class serves to:
 *
 * - Inspect the bootstrap stack, and return various properties of it for successful
 *   asset deployment (in case of legacy-synthesized stacks).
 * - Validate the version of the target environment, and nothing else (in case of
 *   default-synthesized stacks).
 *
 * An object of this type might represent a bootstrap stack that could not be found.
 * This is not an issue unless any members are used that require the bootstrap stack
 * to have been found, in which case an error is thrown (default-synthesized stacks
 * should never run into this as they don't need information from the bootstrap
 * stack, all information is already encoded into the Cloud Assembly Manifest).
 *
 * Nevertheless, an instance of this class exists to serve as a cache for SSM
 * parameter lookups (otherwise, the "bootstrap stack version" parameter would
 * need to be read repeatedly).
 *
 * Called "ToolkitInfo" for historical reasons.
 *
 */
class ToolkitInfo {
    constructor(sdk) {
        this.sdk = sdk;
        this.ssmCache = new Map();
    }
    static determineName(overrideName) {
        return overrideName !== null && overrideName !== void 0 ? overrideName : exports.DEFAULT_TOOLKIT_STACK_NAME;
    }
    static async lookup(environment, sdk, stackName) {
        const cfn = sdk.cloudFormation();
        const stack = await cloudformation_1.stabilizeStack(cfn, stackName !== null && stackName !== void 0 ? stackName : exports.DEFAULT_TOOLKIT_STACK_NAME);
        if (!stack) {
            logging_1.debug('The environment %s doesn\'t have the CDK toolkit stack (%s) installed. Use %s to setup your environment for use with the toolkit.', environment.name, stackName, colors.blue(`cdk bootstrap "${environment.name}"`));
            return ToolkitInfo.bootstrapStackNotFoundInfo(sdk);
        }
        if (stack.stackStatus.isCreationFailure) {
            // Treat a "failed to create" bootstrap stack as an absent one.
            logging_1.debug('The environment %s has a CDK toolkit stack (%s) that failed to create. Use %s to try provisioning it again.', environment.name, stackName, colors.blue(`cdk bootstrap "${environment.name}"`));
            return ToolkitInfo.bootstrapStackNotFoundInfo(sdk);
        }
        return new ExistingToolkitInfo(stack, sdk);
    }
    static fromStack(stack, sdk) {
        return new ExistingToolkitInfo(stack, sdk);
    }
    static bootstraplessDeploymentsOnly(sdk) {
        return new BootstrapStackNotFoundInfo(sdk, 'Trying to perform an operation that requires a bootstrap stack; you should not see this error, this is a bug in the CDK CLI.');
    }
    static bootstrapStackNotFoundInfo(sdk) {
        return new BootstrapStackNotFoundInfo(sdk, 'This deployment requires a bootstrap stack with a known name; pass \'--toolkit-stack-name\' or switch to using the \'DefaultStackSynthesizer\' (see https://docs.aws.amazon.com/cdk/latest/guide/bootstrapping.html)');
    }
    /**
     * Read a version from an SSM parameter, cached
     */
    async versionFromSsmParameter(parameterName) {
        var _a, _b;
        const existing = this.ssmCache.get(parameterName);
        if (existing !== undefined) {
            return existing;
        }
        const ssm = this.sdk.ssm();
        try {
            const result = await ssm.getParameter({ Name: parameterName }).promise();
            const asNumber = parseInt(`${(_a = result.Parameter) === null || _a === void 0 ? void 0 : _a.Value}`, 10);
            if (isNaN(asNumber)) {
                throw new Error(`SSM parameter ${parameterName} not a number: ${(_b = result.Parameter) === null || _b === void 0 ? void 0 : _b.Value}`);
            }
            this.ssmCache.set(parameterName, asNumber);
            return asNumber;
        }
        catch (e) {
            if (e.code === 'ParameterNotFound') {
                throw new Error(`SSM parameter ${parameterName} not found. Has the environment been bootstrapped? Please run \'cdk bootstrap\' (see https://docs.aws.amazon.com/cdk/latest/guide/bootstrapping.html)`);
            }
            throw e;
        }
    }
}
exports.ToolkitInfo = ToolkitInfo;
/**
 * Returned when a bootstrap stack is found
 */
class ExistingToolkitInfo extends ToolkitInfo {
    constructor(bootstrapStack, sdk) {
        super(sdk);
        this.bootstrapStack = bootstrapStack;
        this.found = true;
    }
    get bucketUrl() {
        return `https://${this.requireOutput(bootstrap_1.BUCKET_DOMAIN_NAME_OUTPUT)}`;
    }
    get bucketName() {
        return this.requireOutput(bootstrap_1.BUCKET_NAME_OUTPUT);
    }
    get version() {
        var _a;
        return parseInt((_a = this.bootstrapStack.outputs[bootstrap_1.BOOTSTRAP_VERSION_OUTPUT]) !== null && _a !== void 0 ? _a : '0', 10);
    }
    get parameters() {
        var _a;
        return (_a = this.bootstrapStack.parameters) !== null && _a !== void 0 ? _a : {};
    }
    get terminationProtection() {
        var _a;
        return (_a = this.bootstrapStack.terminationProtection) !== null && _a !== void 0 ? _a : false;
    }
    /**
     * Validate that the bootstrap stack version matches or exceeds the expected version
     *
     * Use the SSM parameter name to read the version number if given, otherwise use the version
     * discovered on the bootstrap stack.
     *
     * Pass in the SSM parameter name so we can cache the lookups an don't need to do the same
     * lookup again and again for every artifact.
     */
    async validateVersion(expectedVersion, ssmParameterName) {
        let version = this.version; // Default to the current version, but will be overwritten by a lookup if required.
        if (ssmParameterName !== undefined) {
            try {
                version = await this.versionFromSsmParameter(ssmParameterName);
            }
            catch (e) {
                if (e.code !== 'AccessDeniedException') {
                    throw e;
                }
                // This is a fallback! The bootstrap template that goes along with this change introduces
                // a new 'ssm:GetParameter' permission, but when run using the previous bootstrap template we
                // won't have the permissions yet to read the version, so we won't be able to show the
                // message telling the user they need to update! When we see an AccessDeniedException, fall
                // back to the version we read from Stack Outputs; but ONLY if the version we discovered via
                // outputs is legitimately an old version. If it's newer than that, something else must be broken,
                // so let it fail as it would if we didn't have this fallback.
                if (this.version >= BOOTSTRAP_TEMPLATE_VERSION_INTRODUCING_GETPARAMETER) {
                    throw e;
                }
                logging_1.warning(`Could not read SSM parameter ${ssmParameterName}: ${e.message}`);
                // Fall through on purpose
            }
        }
        if (expectedVersion > version) {
            throw new Error(`This CDK deployment requires bootstrap stack version '${expectedVersion}', found '${version}'. Please run 'cdk bootstrap'.`);
        }
    }
    /**
     * Prepare an ECR repository for uploading to using Docker
     *
     */
    async prepareEcrRepository(repositoryName) {
        var _a, _b;
        if (!this.sdk) {
            throw new Error('ToolkitInfo needs to have been initialized with an sdk to call prepareEcrRepository');
        }
        const ecr = this.sdk.ecr();
        // check if repo already exists
        try {
            logging_1.debug(`${repositoryName}: checking if ECR repository already exists`);
            const describeResponse = await ecr.describeRepositories({ repositoryNames: [repositoryName] }).promise();
            const existingRepositoryUri = (_a = describeResponse.repositories[0]) === null || _a === void 0 ? void 0 : _a.repositoryUri;
            if (existingRepositoryUri) {
                return { repositoryUri: existingRepositoryUri };
            }
        }
        catch (e) {
            if (e.code !== 'RepositoryNotFoundException') {
                throw e;
            }
        }
        // create the repo (tag it so it will be easier to garbage collect in the future)
        logging_1.debug(`${repositoryName}: creating ECR repository`);
        const assetTag = { Key: 'awscdk:asset', Value: 'true' };
        const response = await ecr.createRepository({ repositoryName, tags: [assetTag] }).promise();
        const repositoryUri = (_b = response.repository) === null || _b === void 0 ? void 0 : _b.repositoryUri;
        if (!repositoryUri) {
            throw new Error(`CreateRepository did not return a repository URI for ${repositoryUri}`);
        }
        // configure image scanning on push (helps in identifying software vulnerabilities, no additional charge)
        logging_1.debug(`${repositoryName}: enable image scanning`);
        await ecr.putImageScanningConfiguration({ repositoryName, imageScanningConfiguration: { scanOnPush: true } }).promise();
        return { repositoryUri };
    }
    requireOutput(output) {
        if (!(output in this.bootstrapStack.outputs)) {
            throw new Error(`The CDK toolkit stack (${this.bootstrapStack.stackName}) does not have an output named ${output}. Use 'cdk bootstrap' to correct this.`);
        }
        return this.bootstrapStack.outputs[output];
    }
}
/**
 * Returned when a bootstrap stack could not be found
 *
 * This is not an error in principle, UNTIL one of the members is called that requires
 * the bootstrap stack to have been found, in which case the lookup error is still thrown
 * belatedly.
 *
 * The errors below serve as a last stop-gap message--normally calling code should have
 * checked `toolkit.found` and produced an appropriate error message.
 */
class BootstrapStackNotFoundInfo extends ToolkitInfo {
    constructor(sdk, errorMessage) {
        super(sdk);
        this.errorMessage = errorMessage;
        this.found = false;
    }
    get bootstrapStack() {
        throw new Error(this.errorMessage);
    }
    get bucketUrl() {
        throw new Error(this.errorMessage);
    }
    get bucketName() {
        throw new Error(this.errorMessage);
    }
    get version() {
        throw new Error(this.errorMessage);
    }
    async validateVersion(expectedVersion, ssmParameterName) {
        if (ssmParameterName === undefined) {
            throw new Error(this.errorMessage);
        }
        let version;
        try {
            version = await this.versionFromSsmParameter(ssmParameterName);
        }
        catch (e) {
            if (e.code !== 'AccessDeniedException') {
                throw e;
            }
            // This is a fallback! The bootstrap template that goes along with this change introduces
            // a new 'ssm:GetParameter' permission, but when run using a previous bootstrap template we
            // won't have the permissions yet to read the version, so we won't be able to show the
            // message telling the user they need to update! When we see an AccessDeniedException, fall
            // back to the version we read from Stack Outputs.
            logging_1.warning(`Could not read SSM parameter ${ssmParameterName}: ${e.message}`);
            throw new Error(`This CDK deployment requires bootstrap stack version '${expectedVersion}', found an older version. Please run 'cdk bootstrap'.`);
        }
        if (expectedVersion > version) {
            throw new Error(`This CDK deployment requires bootstrap stack version '${expectedVersion}', found '${version}'. Please run 'cdk bootstrap'.`);
        }
    }
    prepareEcrRepository() {
        throw new Error(this.errorMessage);
    }
}
//# sourceMappingURL=data:application/json;base64,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