"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const bootstrap_1 = require("../lib/api/bootstrap");
const cloudformation_deployments_1 = require("../lib/api/cloudformation-deployments");
const cdk_toolkit_1 = require("../lib/cdk-toolkit");
const util_1 = require("./util");
let cloudExecutable;
let bootstrapper;
beforeEach(() => {
    bootstrapper = util_1.instanceMockFrom(bootstrap_1.Bootstrapper);
    bootstrapper.bootstrapEnvironment.mockResolvedValue({ noOp: false, outputs: {} });
    cloudExecutable = new util_1.MockCloudExecutable({
        stacks: [
            MockStack.MOCK_STACK_A,
            MockStack.MOCK_STACK_B,
        ],
        nestedAssemblies: [{
                stacks: [MockStack.MOCK_STACK_C],
            }],
    });
});
function defaultToolkitSetup() {
    return new cdk_toolkit_1.CdkToolkit({
        cloudExecutable,
        configuration: cloudExecutable.configuration,
        sdkProvider: cloudExecutable.sdkProvider,
        cloudFormation: new FakeCloudFormation({
            'Test-Stack-A': { Foo: 'Bar' },
            'Test-Stack-B': { Baz: 'Zinga!' },
            'Test-Stack-C': { Baz: 'Zinga!' },
        }),
    });
}
describe('deploy', () => {
    describe('makes correct CloudFormation calls', () => {
        test('without options', async () => {
            // GIVEN
            const toolkit = defaultToolkitSetup();
            // WHEN
            await toolkit.deploy({ selector: { patterns: ['Test-Stack-A', 'Test-Stack-B'] } });
        });
        test('with stacks all stacks specified as double wildcard', async () => {
            // GIVEN
            const toolkit = defaultToolkitSetup();
            // WHEN
            await toolkit.deploy({ selector: { patterns: ['**'] } });
        });
        test('with one stack specified', async () => {
            // GIVEN
            const toolkit = defaultToolkitSetup();
            // WHEN
            await toolkit.deploy({ selector: { patterns: ['Test-Stack-A'] } });
        });
        test('with stacks all stacks specified as wildcard', async () => {
            // GIVEN
            const toolkit = defaultToolkitSetup();
            // WHEN
            await toolkit.deploy({ selector: { patterns: ['*'] } });
        });
        test('with sns notification arns', async () => {
            // GIVEN
            const notificationArns = ['arn:aws:sns:::cfn-notifications', 'arn:aws:sns:::my-cool-topic'];
            const toolkit = new cdk_toolkit_1.CdkToolkit({
                cloudExecutable,
                configuration: cloudExecutable.configuration,
                sdkProvider: cloudExecutable.sdkProvider,
                cloudFormation: new FakeCloudFormation({
                    'Test-Stack-A': { Foo: 'Bar' },
                    'Test-Stack-B': { Baz: 'Zinga!' },
                }, notificationArns),
            });
            // WHEN
            await toolkit.deploy({
                selector: { patterns: ['Test-Stack-A', 'Test-Stack-B'] },
                notificationArns,
            });
        });
        test('globless bootstrap uses environment without question', async () => {
            // GIVEN
            const toolkit = defaultToolkitSetup();
            // WHEN
            await toolkit.bootstrap(['aws://56789/south-pole'], bootstrapper, {});
            // THEN
            expect(bootstrapper.bootstrapEnvironment).toHaveBeenCalledWith({
                account: '56789',
                region: 'south-pole',
                name: 'aws://56789/south-pole',
            }, expect.anything(), expect.anything());
            expect(bootstrapper.bootstrapEnvironment).toHaveBeenCalledTimes(1);
        });
        test('globby bootstrap uses whats in the stacks', async () => {
            // GIVEN
            const toolkit = defaultToolkitSetup();
            cloudExecutable.configuration.settings.set(['app'], 'something');
            // WHEN
            await toolkit.bootstrap(['aws://*/bermuda-triangle-1'], bootstrapper, {});
            // THEN
            expect(bootstrapper.bootstrapEnvironment).toHaveBeenCalledWith({
                account: '123456789012',
                region: 'bermuda-triangle-1',
                name: 'aws://123456789012/bermuda-triangle-1',
            }, expect.anything(), expect.anything());
            expect(bootstrapper.bootstrapEnvironment).toHaveBeenCalledTimes(1);
        });
        test('bootstrap can be invoked without the --app argument', async () => {
            // GIVEN
            cloudExecutable.configuration.settings.clear();
            const mockSynthesize = jest.fn();
            cloudExecutable.synthesize = mockSynthesize;
            const toolkit = defaultToolkitSetup();
            // WHEN
            await toolkit.bootstrap(['aws://123456789012/west-pole'], bootstrapper, {});
            // THEN
            expect(bootstrapper.bootstrapEnvironment).toHaveBeenCalledWith({
                account: '123456789012',
                region: 'west-pole',
                name: 'aws://123456789012/west-pole',
            }, expect.anything(), expect.anything());
            expect(bootstrapper.bootstrapEnvironment).toHaveBeenCalledTimes(1);
            expect(cloudExecutable.hasApp).toEqual(false);
            expect(mockSynthesize).not.toHaveBeenCalled();
        });
    });
});
describe('synth', () => {
    test('with no stdout option', async () => {
        // GIVE
        const toolkit = defaultToolkitSetup();
        // THEN
        await expect(toolkit.synth(['Test-Stack-A'], false, true)).resolves.toBeUndefined();
    });
    describe('post-synth validation', () => {
        beforeEach(() => {
            cloudExecutable = new util_1.MockCloudExecutable({
                stacks: [
                    MockStack.MOCK_STACK_A,
                    MockStack.MOCK_STACK_B,
                ],
                nestedAssemblies: [{
                        stacks: [MockStack.MOCK_STACK_WITH_ERROR],
                    }],
            });
        });
    });
    afterEach(() => {
        process.env.STACKS_TO_VALIDATE = undefined;
    });
    describe('stack with error and flagged for validation', () => {
        beforeEach(() => {
            cloudExecutable = new util_1.MockCloudExecutable({
                stacks: [
                    MockStack.MOCK_STACK_A,
                    MockStack.MOCK_STACK_B,
                ],
                nestedAssemblies: [{
                        stacks: [
                            { properties: { validateOnSynth: true }, ...MockStack.MOCK_STACK_WITH_ERROR },
                        ],
                    }],
            });
        });
        test('causes synth to fail if autoValidate=true', async () => {
            const toolkit = defaultToolkitSetup();
            const autoValidate = true;
            await expect(toolkit.synth([], false, true, autoValidate)).rejects.toBeDefined();
        });
        test('causes synth to succeed if autoValidate=false', async () => {
            const toolkit = defaultToolkitSetup();
            const autoValidate = false;
            await expect(toolkit.synth([], false, true, autoValidate)).resolves.toBeUndefined();
        });
    });
    test('stack has error and was explicitly selected', async () => {
        cloudExecutable = new util_1.MockCloudExecutable({
            stacks: [
                MockStack.MOCK_STACK_A,
                MockStack.MOCK_STACK_B,
            ],
            nestedAssemblies: [{
                    stacks: [
                        { properties: { validateOnSynth: false }, ...MockStack.MOCK_STACK_WITH_ERROR },
                    ],
                }],
        });
        const toolkit = defaultToolkitSetup();
        await expect(toolkit.synth(['Test-Stack-A/witherrors'], false, true)).rejects.toBeDefined();
    });
    test('stack has error, is not flagged for validation and was not explicitly selected', async () => {
        cloudExecutable = new util_1.MockCloudExecutable({
            stacks: [
                MockStack.MOCK_STACK_A,
                MockStack.MOCK_STACK_B,
            ],
            nestedAssemblies: [{
                    stacks: [
                        { properties: { validateOnSynth: false }, ...MockStack.MOCK_STACK_WITH_ERROR },
                    ],
                }],
        });
        const toolkit = defaultToolkitSetup();
        await toolkit.synth([], false, true);
    });
    test('stack has dependency and was explicitly selected', async () => {
        cloudExecutable = new util_1.MockCloudExecutable({
            stacks: [
                MockStack.MOCK_STACK_C,
                MockStack.MOCK_STACK_D,
            ],
        });
        const toolkit = defaultToolkitSetup();
        await expect(toolkit.synth([MockStack.MOCK_STACK_D.stackName], true, false)).resolves.toBeDefined();
    });
});
class MockStack {
}
MockStack.MOCK_STACK_A = {
    stackName: 'Test-Stack-A',
    template: { Resources: { TemplateName: 'Test-Stack-A' } },
    env: 'aws://123456789012/bermuda-triangle-1',
    metadata: {
        '/Test-Stack-A': [
            {
                type: cxschema.ArtifactMetadataEntryType.STACK_TAGS,
                data: [
                    { key: 'Foo', value: 'Bar' },
                ],
            },
        ],
    },
};
MockStack.MOCK_STACK_B = {
    stackName: 'Test-Stack-B',
    template: { Resources: { TemplateName: 'Test-Stack-B' } },
    env: 'aws://123456789012/bermuda-triangle-1',
    metadata: {
        '/Test-Stack-B': [
            {
                type: cxschema.ArtifactMetadataEntryType.STACK_TAGS,
                data: [
                    { key: 'Baz', value: 'Zinga!' },
                ],
            },
        ],
    },
};
MockStack.MOCK_STACK_C = {
    stackName: 'Test-Stack-C',
    template: { Resources: { TemplateName: 'Test-Stack-C' } },
    env: 'aws://123456789012/bermuda-triangle-1',
    metadata: {
        '/Test-Stack-C': [
            {
                type: cxschema.ArtifactMetadataEntryType.STACK_TAGS,
                data: [
                    { key: 'Baz', value: 'Zinga!' },
                ],
            },
        ],
    },
    displayName: 'Test-Stack-A/Test-Stack-C',
};
MockStack.MOCK_STACK_D = {
    stackName: 'Test-Stack-D',
    template: { Resources: { TemplateName: 'Test-Stack-D' } },
    env: 'aws://123456789012/bermuda-triangle-1',
    metadata: {
        '/Test-Stack-D': [
            {
                type: cxschema.ArtifactMetadataEntryType.STACK_TAGS,
                data: [
                    { key: 'Baz', value: 'Zinga!' },
                ],
            },
        ],
    },
    depends: [MockStack.MOCK_STACK_C.stackName],
};
MockStack.MOCK_STACK_WITH_ERROR = {
    stackName: 'witherrors',
    env: 'aws://123456789012/bermuda-triangle-1',
    template: { resource: 'errorresource' },
    metadata: {
        '/resource': [
            {
                type: cxschema.ArtifactMetadataEntryType.ERROR,
                data: 'this is an error',
            },
        ],
    },
    displayName: 'Test-Stack-A/witherrors',
};
class FakeCloudFormation extends cloudformation_deployments_1.CloudFormationDeployments {
    constructor(expectedTags = {}, expectedNotificationArns) {
        super({ sdkProvider: undefined });
        this.expectedTags = {};
        for (const [stackName, tags] of Object.entries(expectedTags)) {
            this.expectedTags[stackName] =
                Object.entries(tags).map(([Key, Value]) => ({ Key, Value }))
                    .sort((l, r) => l.Key.localeCompare(r.Key));
        }
        if (expectedNotificationArns) {
            this.expectedNotificationArns = expectedNotificationArns;
        }
    }
    deployStack(options) {
        expect([MockStack.MOCK_STACK_A.stackName, MockStack.MOCK_STACK_B.stackName, MockStack.MOCK_STACK_C.stackName])
            .toContain(options.stack.stackName);
        expect(options.tags).toEqual(this.expectedTags[options.stack.stackName]);
        expect(options.notificationArns).toEqual(this.expectedNotificationArns);
        return Promise.resolve({
            stackArn: `arn:aws:cloudformation:::stack/${options.stack.stackName}/MockedOut`,
            noOp: false,
            outputs: { StackName: options.stack.stackName },
            stackArtifact: options.stack,
        });
    }
    readCurrentTemplate(stack) {
        switch (stack.stackName) {
            case MockStack.MOCK_STACK_A.stackName:
                return Promise.resolve({});
            case MockStack.MOCK_STACK_B.stackName:
                return Promise.resolve({});
            case MockStack.MOCK_STACK_C.stackName:
                return Promise.resolve({});
            default:
                return Promise.reject(`Not an expected mock stack: ${stack.stackName}`);
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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