"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const AWS = require("aws-sdk-mock");
const vpcs_1 = require("../../lib/context-providers/vpcs");
const mock_sdk_1 = require("../util/mock-sdk");
AWS.setSDK(require.resolve('aws-sdk'));
const mockSDK = new mock_sdk_1.MockSdkProvider();
afterEach(done => {
    AWS.restore();
    done();
});
test('looks up the requested VPC', async () => {
    // GIVEN
    const filter = { foo: 'bar' };
    const provider = new vpcs_1.VpcNetworkContextProviderPlugin(mockSDK);
    mockVpcLookup({
        subnets: [
            { SubnetId: 'sub-123456', AvailabilityZone: 'bermuda-triangle-1337', MapPublicIpOnLaunch: true },
            { SubnetId: 'sub-789012', AvailabilityZone: 'bermuda-triangle-1337', MapPublicIpOnLaunch: false },
        ],
        routeTables: [
            { Associations: [{ SubnetId: 'sub-123456' }], RouteTableId: 'rtb-123456' },
            { Associations: [{ SubnetId: 'sub-789012' }], RouteTableId: 'rtb-789012' },
        ],
        vpnGateways: [{ VpnGatewayId: 'gw-abcdef' }],
    });
    // WHEN
    const result = await provider.getValue({
        account: '1234',
        region: 'us-east-1',
        filter,
    });
    // THEN
    expect(result).toEqual({
        vpcId: 'vpc-1234567',
        vpcCidrBlock: '1.1.1.1/16',
        availabilityZones: ['bermuda-triangle-1337'],
        isolatedSubnetIds: undefined,
        isolatedSubnetNames: undefined,
        isolatedSubnetRouteTableIds: undefined,
        privateSubnetIds: ['sub-789012'],
        privateSubnetNames: ['Private'],
        privateSubnetRouteTableIds: ['rtb-789012'],
        publicSubnetIds: ['sub-123456'],
        publicSubnetNames: ['Public'],
        publicSubnetRouteTableIds: ['rtb-123456'],
        vpnGatewayId: 'gw-abcdef',
        subnetGroups: undefined,
    });
});
test('throws when no such VPC is found', async () => {
    // GIVEN
    const filter = { foo: 'bar' };
    const provider = new vpcs_1.VpcNetworkContextProviderPlugin(mockSDK);
    AWS.mock('EC2', 'describeVpcs', (params, cb) => {
        expect(params.Filters).toEqual([{ Name: 'foo', Values: ['bar'] }]);
        return cb(null, {});
    });
    // WHEN
    await expect(provider.getValue({
        account: '1234',
        region: 'us-east-1',
        filter,
    })).rejects.toThrow(/Could not find any VPCs matching/);
});
test('throws when multiple VPCs are found', async () => {
    // GIVEN
    const filter = { foo: 'bar' };
    const provider = new vpcs_1.VpcNetworkContextProviderPlugin(mockSDK);
    AWS.mock('EC2', 'describeVpcs', (params, cb) => {
        expect(params.Filters).toEqual([{ Name: 'foo', Values: ['bar'] }]);
        return cb(null, { Vpcs: [{ VpcId: 'vpc-1' }, { VpcId: 'vpc-2' }] });
    });
    // WHEN
    await expect(provider.getValue({
        account: '1234',
        region: 'us-east-1',
        filter,
    })).rejects.toThrow(/Found 2 VPCs matching/);
});
test('uses the VPC main route table when a subnet has no specific association', async () => {
    // GIVEN
    const filter = { foo: 'bar' };
    const provider = new vpcs_1.VpcNetworkContextProviderPlugin(mockSDK);
    mockVpcLookup({
        subnets: [
            { SubnetId: 'sub-123456', AvailabilityZone: 'bermuda-triangle-1337', MapPublicIpOnLaunch: true },
            { SubnetId: 'sub-789012', AvailabilityZone: 'bermuda-triangle-1337', MapPublicIpOnLaunch: false },
        ],
        routeTables: [
            { Associations: [{ SubnetId: 'sub-123456' }], RouteTableId: 'rtb-123456' },
            { Associations: [{ Main: true }], RouteTableId: 'rtb-789012' },
        ],
        vpnGateways: [{ VpnGatewayId: 'gw-abcdef' }],
    });
    // WHEN
    const result = await provider.getValue({
        account: '1234',
        region: 'us-east-1',
        filter,
    });
    // THEN
    expect(result).toEqual({
        vpcId: 'vpc-1234567',
        vpcCidrBlock: '1.1.1.1/16',
        availabilityZones: ['bermuda-triangle-1337'],
        isolatedSubnetIds: undefined,
        isolatedSubnetNames: undefined,
        isolatedSubnetRouteTableIds: undefined,
        privateSubnetIds: ['sub-789012'],
        privateSubnetNames: ['Private'],
        privateSubnetRouteTableIds: ['rtb-789012'],
        publicSubnetIds: ['sub-123456'],
        publicSubnetNames: ['Public'],
        publicSubnetRouteTableIds: ['rtb-123456'],
        vpnGatewayId: 'gw-abcdef',
        subnetGroups: undefined,
    });
});
test('Recognize public subnet by route table', async () => {
    // GIVEN
    const filter = { foo: 'bar' };
    const provider = new vpcs_1.VpcNetworkContextProviderPlugin(mockSDK);
    mockVpcLookup({
        subnets: [
            { SubnetId: 'sub-123456', AvailabilityZone: 'bermuda-triangle-1337', MapPublicIpOnLaunch: false },
        ],
        routeTables: [
            {
                Associations: [{ SubnetId: 'sub-123456' }],
                RouteTableId: 'rtb-123456',
                Routes: [
                    {
                        DestinationCidrBlock: '10.0.2.0/26',
                        Origin: 'CreateRoute',
                        State: 'active',
                        VpcPeeringConnectionId: 'pcx-xxxxxx',
                    },
                    {
                        DestinationCidrBlock: '10.0.1.0/24',
                        GatewayId: 'local',
                        Origin: 'CreateRouteTable',
                        State: 'active',
                    },
                    {
                        DestinationCidrBlock: '0.0.0.0/0',
                        GatewayId: 'igw-xxxxxx',
                        Origin: 'CreateRoute',
                        State: 'active',
                    },
                ],
            },
        ],
    });
    // WHEN
    const result = await provider.getValue({
        account: '1234',
        region: 'us-east-1',
        filter,
    });
    // THEN
    expect(result).toEqual({
        vpcId: 'vpc-1234567',
        vpcCidrBlock: '1.1.1.1/16',
        availabilityZones: ['bermuda-triangle-1337'],
        isolatedSubnetIds: undefined,
        isolatedSubnetNames: undefined,
        isolatedSubnetRouteTableIds: undefined,
        privateSubnetIds: undefined,
        privateSubnetNames: undefined,
        privateSubnetRouteTableIds: undefined,
        publicSubnetIds: ['sub-123456'],
        publicSubnetNames: ['Public'],
        publicSubnetRouteTableIds: ['rtb-123456'],
        vpnGatewayId: undefined,
        subnetGroups: undefined,
    });
});
function mockVpcLookup(options) {
    const VpcId = 'vpc-1234567';
    AWS.mock('EC2', 'describeVpcs', (params, cb) => {
        expect(params.Filters).toEqual([{ Name: 'foo', Values: ['bar'] }]);
        return cb(null, { Vpcs: [{ VpcId, CidrBlock: '1.1.1.1/16' }] });
    });
    AWS.mock('EC2', 'describeSubnets', (params, cb) => {
        expect(params.Filters).toEqual([{ Name: 'vpc-id', Values: [VpcId] }]);
        return cb(null, { Subnets: options.subnets });
    });
    AWS.mock('EC2', 'describeRouteTables', (params, cb) => {
        expect(params.Filters).toEqual([{ Name: 'vpc-id', Values: [VpcId] }]);
        return cb(null, { RouteTables: options.routeTables });
    });
    AWS.mock('EC2', 'describeVpnGateways', (params, cb) => {
        expect(params.Filters).toEqual([
            { Name: 'attachment.vpc-id', Values: [VpcId] },
            { Name: 'attachment.state', Values: ['attached'] },
            { Name: 'state', Values: ['available'] },
        ]);
        return cb(null, { VpnGateways: options.vpnGateways });
    });
}
//# sourceMappingURL=data:application/json;base64,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