"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Docker = void 0;
const fs = require("fs");
const os = require("os");
const path = require("path");
const docker_credentials_1 = require("./docker-credentials");
const shell_1 = require("./shell");
class Docker {
    constructor(logger) {
        this.logger = logger;
        this.configDir = undefined;
    }
    /**
     * Whether an image with the given tag exists
     */
    async exists(tag) {
        try {
            await this.execute(['inspect', tag], { quiet: true });
            return true;
        }
        catch (e) {
            if (e.code !== 'PROCESS_FAILED' || e.exitCode !== 1) {
                throw e;
            }
            return false;
        }
    }
    async build(options) {
        const buildCommand = [
            'build',
            ...flatten(Object.entries(options.buildArgs || {}).map(([k, v]) => ['--build-arg', `${k}=${v}`])),
            '--tag', options.tag,
            ...options.target ? ['--target', options.target] : [],
            ...options.file ? ['--file', options.file] : [],
            '.',
        ];
        await this.execute(buildCommand, { cwd: options.directory });
    }
    /**
     * Get credentials from ECR and run docker login
     */
    async login(ecr) {
        const credentials = await docker_credentials_1.obtainEcrCredentials(ecr);
        // Use --password-stdin otherwise docker will complain. Loudly.
        await this.execute(['login',
            '--username', credentials.username,
            '--password-stdin',
            credentials.endpoint], {
            input: credentials.password,
            // Need to quiet otherwise Docker will complain
            // 'WARNING! Your password will be stored unencrypted'
            // doesn't really matter since it's a token.
            quiet: true,
        });
    }
    async tag(sourceTag, targetTag) {
        await this.execute(['tag', sourceTag, targetTag]);
    }
    async push(tag) {
        await this.execute(['push', tag]);
    }
    /**
     * If a CDK Docker Credentials file exists, creates a new Docker config directory.
     * Sets up `docker-credential-cdk-assets` to be the credential helper for each domain in the CDK config.
     * All future commands (e.g., `build`, `push`) will use this config.
     *
     * See https://docs.docker.com/engine/reference/commandline/login/#credential-helpers for more details on cred helpers.
     *
     * @returns true if CDK config was found and configured, false otherwise
     */
    configureCdkCredentials() {
        const config = docker_credentials_1.cdkCredentialsConfig();
        if (!config) {
            return false;
        }
        this.configDir = fs.mkdtempSync(path.join(os.tmpdir(), 'cdkDockerConfig'));
        const domains = Object.keys(config.domainCredentials);
        const credHelpers = domains.reduce((map, domain) => {
            map[domain] = 'cdk-assets'; // Use docker-credential-cdk-assets for this domain
            return map;
        }, {});
        fs.writeFileSync(path.join(this.configDir, 'config.json'), JSON.stringify({ credHelpers }), { encoding: 'utf-8' });
        return true;
    }
    /**
     * Removes any configured Docker config directory.
     * All future commands (e.g., `build`, `push`) will use the default config.
     *
     * This is useful after calling `configureCdkCredentials` to reset to default credentials.
     */
    resetAuthPlugins() {
        this.configDir = undefined;
    }
    async execute(args, options = {}) {
        const configArgs = this.configDir ? ['--config', this.configDir] : [];
        try {
            await shell_1.shell(['docker', ...configArgs, ...args], { logger: this.logger, ...options });
        }
        catch (e) {
            if (e.code === 'ENOENT') {
                throw new Error('Unable to execute \'docker\' in order to build a container asset. Please install \'docker\' and try again.');
            }
            throw e;
        }
    }
}
exports.Docker = Docker;
function flatten(x) {
    return Array.prototype.concat([], ...x);
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZG9ja2VyLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiZG9ja2VyLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBLHlCQUF5QjtBQUN6Qix5QkFBeUI7QUFDekIsNkJBQTZCO0FBQzdCLDZEQUFrRjtBQUNsRixtQ0FBc0Q7QUF3QnRELE1BQWEsTUFBTTtJQUlqQixZQUE2QixNQUFlO1FBQWYsV0FBTSxHQUFOLE1BQU0sQ0FBUztRQUZwQyxjQUFTLEdBQXVCLFNBQVMsQ0FBQztJQUdsRCxDQUFDO0lBRUQ7O09BRUc7SUFDSSxLQUFLLENBQUMsTUFBTSxDQUFDLEdBQVc7UUFDN0IsSUFBSTtZQUNGLE1BQU0sSUFBSSxDQUFDLE9BQU8sQ0FBQyxDQUFDLFNBQVMsRUFBRSxHQUFHLENBQUMsRUFBRSxFQUFFLEtBQUssRUFBRSxJQUFJLEVBQUUsQ0FBQyxDQUFDO1lBQ3RELE9BQU8sSUFBSSxDQUFDO1NBQ2I7UUFBQyxPQUFPLENBQUMsRUFBRTtZQUNWLElBQUksQ0FBQyxDQUFDLElBQUksS0FBSyxnQkFBZ0IsSUFBSSxDQUFDLENBQUMsUUFBUSxLQUFLLENBQUMsRUFBRTtnQkFBRSxNQUFNLENBQUMsQ0FBQzthQUFFO1lBQ2pFLE9BQU8sS0FBSyxDQUFDO1NBQ2Q7SUFDSCxDQUFDO0lBRU0sS0FBSyxDQUFDLEtBQUssQ0FBQyxPQUFxQjtRQUN0QyxNQUFNLFlBQVksR0FBRztZQUNuQixPQUFPO1lBQ1AsR0FBRyxPQUFPLENBQUMsTUFBTSxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsU0FBUyxJQUFJLEVBQUUsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxDQUFDLGFBQWEsRUFBRSxHQUFHLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUM7WUFDakcsT0FBTyxFQUFFLE9BQU8sQ0FBQyxHQUFHO1lBQ3BCLEdBQUcsT0FBTyxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsQ0FBQyxVQUFVLEVBQUUsT0FBTyxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsQ0FBQyxFQUFFO1lBQ3JELEdBQUcsT0FBTyxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsQ0FBQyxRQUFRLEVBQUUsT0FBTyxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsQ0FBQyxFQUFFO1lBQy9DLEdBQUc7U0FDSixDQUFDO1FBQ0YsTUFBTSxJQUFJLENBQUMsT0FBTyxDQUFDLFlBQVksRUFBRSxFQUFFLEdBQUcsRUFBRSxPQUFPLENBQUMsU0FBUyxFQUFFLENBQUMsQ0FBQztJQUMvRCxDQUFDO0lBRUQ7O09BRUc7SUFDSSxLQUFLLENBQUMsS0FBSyxDQUFDLEdBQVk7UUFDN0IsTUFBTSxXQUFXLEdBQUcsTUFBTSx5Q0FBb0IsQ0FBQyxHQUFHLENBQUMsQ0FBQztRQUVwRCwrREFBK0Q7UUFDL0QsTUFBTSxJQUFJLENBQUMsT0FBTyxDQUFDLENBQUMsT0FBTztZQUN6QixZQUFZLEVBQUUsV0FBVyxDQUFDLFFBQVE7WUFDbEMsa0JBQWtCO1lBQ2xCLFdBQVcsQ0FBQyxRQUFRLENBQUMsRUFBRTtZQUN2QixLQUFLLEVBQUUsV0FBVyxDQUFDLFFBQVE7WUFFM0IsK0NBQStDO1lBQy9DLHNEQUFzRDtZQUN0RCw0Q0FBNEM7WUFDNUMsS0FBSyxFQUFFLElBQUk7U0FDWixDQUFDLENBQUM7SUFDTCxDQUFDO0lBRU0sS0FBSyxDQUFDLEdBQUcsQ0FBQyxTQUFpQixFQUFFLFNBQWlCO1FBQ25ELE1BQU0sSUFBSSxDQUFDLE9BQU8sQ0FBQyxDQUFDLEtBQUssRUFBRSxTQUFTLEVBQUUsU0FBUyxDQUFDLENBQUMsQ0FBQztJQUNwRCxDQUFDO0lBRU0sS0FBSyxDQUFDLElBQUksQ0FBQyxHQUFXO1FBQzNCLE1BQU0sSUFBSSxDQUFDLE9BQU8sQ0FBQyxDQUFDLE1BQU0sRUFBRSxHQUFHLENBQUMsQ0FBQyxDQUFDO0lBQ3BDLENBQUM7SUFFRDs7Ozs7Ozs7T0FRRztJQUNJLHVCQUF1QjtRQUM1QixNQUFNLE1BQU0sR0FBRyx5Q0FBb0IsRUFBRSxDQUFDO1FBQ3RDLElBQUksQ0FBQyxNQUFNLEVBQUU7WUFBRSxPQUFPLEtBQUssQ0FBQztTQUFFO1FBRTlCLElBQUksQ0FBQyxTQUFTLEdBQUcsRUFBRSxDQUFDLFdBQVcsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxNQUFNLEVBQUUsRUFBRSxpQkFBaUIsQ0FBQyxDQUFDLENBQUM7UUFFM0UsTUFBTSxPQUFPLEdBQUcsTUFBTSxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsaUJBQWlCLENBQUMsQ0FBQztRQUN0RCxNQUFNLFdBQVcsR0FBRyxPQUFPLENBQUMsTUFBTSxDQUFDLENBQUMsR0FBMkIsRUFBRSxNQUFNLEVBQUUsRUFBRTtZQUN6RSxHQUFHLENBQUMsTUFBTSxDQUFDLEdBQUcsWUFBWSxDQUFDLENBQUMsbURBQW1EO1lBQy9FLE9BQU8sR0FBRyxDQUFDO1FBQ2IsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBQ1AsRUFBRSxDQUFDLGFBQWEsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLEVBQUUsYUFBYSxDQUFDLEVBQUUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxFQUFFLFdBQVcsRUFBRSxDQUFDLEVBQUUsRUFBRSxRQUFRLEVBQUUsT0FBTyxFQUFFLENBQUMsQ0FBQztRQUVuSCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7T0FLRztJQUNJLGdCQUFnQjtRQUNyQixJQUFJLENBQUMsU0FBUyxHQUFHLFNBQVMsQ0FBQztJQUM3QixDQUFDO0lBRU8sS0FBSyxDQUFDLE9BQU8sQ0FBQyxJQUFjLEVBQUUsVUFBd0IsRUFBRTtRQUM5RCxNQUFNLFVBQVUsR0FBRyxJQUFJLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxDQUFDLFVBQVUsRUFBRSxJQUFJLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQztRQUV0RSxJQUFJO1lBQ0YsTUFBTSxhQUFLLENBQUMsQ0FBQyxRQUFRLEVBQUUsR0FBRyxVQUFVLEVBQUUsR0FBRyxJQUFJLENBQUMsRUFBRSxFQUFFLE1BQU0sRUFBRSxJQUFJLENBQUMsTUFBTSxFQUFFLEdBQUcsT0FBTyxFQUFFLENBQUMsQ0FBQztTQUN0RjtRQUFDLE9BQU8sQ0FBQyxFQUFFO1lBQ1YsSUFBSSxDQUFDLENBQUMsSUFBSSxLQUFLLFFBQVEsRUFBRTtnQkFDdkIsTUFBTSxJQUFJLEtBQUssQ0FBQyw0R0FBNEcsQ0FBQyxDQUFDO2FBQy9IO1lBQ0QsTUFBTSxDQUFDLENBQUM7U0FDVDtJQUNILENBQUM7Q0FDRjtBQTNHRCx3QkEyR0M7QUFFRCxTQUFTLE9BQU8sQ0FBQyxDQUFhO0lBQzVCLE9BQU8sS0FBSyxDQUFDLFNBQVMsQ0FBQyxNQUFNLENBQUMsRUFBRSxFQUFFLEdBQUcsQ0FBQyxDQUFDLENBQUM7QUFDMUMsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGZzIGZyb20gJ2ZzJztcbmltcG9ydCAqIGFzIG9zIGZyb20gJ29zJztcbmltcG9ydCAqIGFzIHBhdGggZnJvbSAncGF0aCc7XG5pbXBvcnQgeyBjZGtDcmVkZW50aWFsc0NvbmZpZywgb2J0YWluRWNyQ3JlZGVudGlhbHMgfSBmcm9tICcuL2RvY2tlci1jcmVkZW50aWFscyc7XG5pbXBvcnQgeyBMb2dnZXIsIHNoZWxsLCBTaGVsbE9wdGlvbnMgfSBmcm9tICcuL3NoZWxsJztcblxuaW50ZXJmYWNlIEJ1aWxkT3B0aW9ucyB7XG4gIHJlYWRvbmx5IGRpcmVjdG9yeTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUYWcgdGhlIGltYWdlIHdpdGggYSBnaXZlbiByZXBvTmFtZTp0YWcgY29tYmluYXRpb25cbiAgICovXG4gIHJlYWRvbmx5IHRhZzogc3RyaW5nO1xuICByZWFkb25seSB0YXJnZXQ/OiBzdHJpbmc7XG4gIHJlYWRvbmx5IGZpbGU/OiBzdHJpbmc7XG4gIHJlYWRvbmx5IGJ1aWxkQXJncz86IFJlY29yZDxzdHJpbmcsIHN0cmluZz47XG59XG5cbmV4cG9ydCBpbnRlcmZhY2UgRG9ja2VyQ3JlZGVudGlhbHNDb25maWcge1xuICByZWFkb25seSB2ZXJzaW9uOiBzdHJpbmc7XG4gIHJlYWRvbmx5IGRvbWFpbkNyZWRlbnRpYWxzOiBSZWNvcmQ8c3RyaW5nLCBEb2NrZXJEb21haW5DcmVkZW50aWFscz47XG59XG5cbmV4cG9ydCBpbnRlcmZhY2UgRG9ja2VyRG9tYWluQ3JlZGVudGlhbHMge1xuICByZWFkb25seSBzZWNyZXRzTWFuYWdlclNlY3JldElkPzogc3RyaW5nO1xuICByZWFkb25seSBlY3JSZXBvc2l0b3J5Pzogc3RyaW5nO1xufVxuXG5leHBvcnQgY2xhc3MgRG9ja2VyIHtcblxuICBwcml2YXRlIGNvbmZpZ0Rpcjogc3RyaW5nIHwgdW5kZWZpbmVkID0gdW5kZWZpbmVkO1xuXG4gIGNvbnN0cnVjdG9yKHByaXZhdGUgcmVhZG9ubHkgbG9nZ2VyPzogTG9nZ2VyKSB7XG4gIH1cblxuICAvKipcbiAgICogV2hldGhlciBhbiBpbWFnZSB3aXRoIHRoZSBnaXZlbiB0YWcgZXhpc3RzXG4gICAqL1xuICBwdWJsaWMgYXN5bmMgZXhpc3RzKHRhZzogc3RyaW5nKSB7XG4gICAgdHJ5IHtcbiAgICAgIGF3YWl0IHRoaXMuZXhlY3V0ZShbJ2luc3BlY3QnLCB0YWddLCB7IHF1aWV0OiB0cnVlIH0pO1xuICAgICAgcmV0dXJuIHRydWU7XG4gICAgfSBjYXRjaCAoZSkge1xuICAgICAgaWYgKGUuY29kZSAhPT0gJ1BST0NFU1NfRkFJTEVEJyB8fCBlLmV4aXRDb2RlICE9PSAxKSB7IHRocm93IGU7IH1cbiAgICAgIHJldHVybiBmYWxzZTtcbiAgICB9XG4gIH1cblxuICBwdWJsaWMgYXN5bmMgYnVpbGQob3B0aW9uczogQnVpbGRPcHRpb25zKSB7XG4gICAgY29uc3QgYnVpbGRDb21tYW5kID0gW1xuICAgICAgJ2J1aWxkJyxcbiAgICAgIC4uLmZsYXR0ZW4oT2JqZWN0LmVudHJpZXMob3B0aW9ucy5idWlsZEFyZ3MgfHwge30pLm1hcCgoW2ssIHZdKSA9PiBbJy0tYnVpbGQtYXJnJywgYCR7a309JHt2fWBdKSksXG4gICAgICAnLS10YWcnLCBvcHRpb25zLnRhZyxcbiAgICAgIC4uLm9wdGlvbnMudGFyZ2V0ID8gWyctLXRhcmdldCcsIG9wdGlvbnMudGFyZ2V0XSA6IFtdLFxuICAgICAgLi4ub3B0aW9ucy5maWxlID8gWyctLWZpbGUnLCBvcHRpb25zLmZpbGVdIDogW10sXG4gICAgICAnLicsXG4gICAgXTtcbiAgICBhd2FpdCB0aGlzLmV4ZWN1dGUoYnVpbGRDb21tYW5kLCB7IGN3ZDogb3B0aW9ucy5kaXJlY3RvcnkgfSk7XG4gIH1cblxuICAvKipcbiAgICogR2V0IGNyZWRlbnRpYWxzIGZyb20gRUNSIGFuZCBydW4gZG9ja2VyIGxvZ2luXG4gICAqL1xuICBwdWJsaWMgYXN5bmMgbG9naW4oZWNyOiBBV1MuRUNSKSB7XG4gICAgY29uc3QgY3JlZGVudGlhbHMgPSBhd2FpdCBvYnRhaW5FY3JDcmVkZW50aWFscyhlY3IpO1xuXG4gICAgLy8gVXNlIC0tcGFzc3dvcmQtc3RkaW4gb3RoZXJ3aXNlIGRvY2tlciB3aWxsIGNvbXBsYWluLiBMb3VkbHkuXG4gICAgYXdhaXQgdGhpcy5leGVjdXRlKFsnbG9naW4nLFxuICAgICAgJy0tdXNlcm5hbWUnLCBjcmVkZW50aWFscy51c2VybmFtZSxcbiAgICAgICctLXBhc3N3b3JkLXN0ZGluJyxcbiAgICAgIGNyZWRlbnRpYWxzLmVuZHBvaW50XSwge1xuICAgICAgaW5wdXQ6IGNyZWRlbnRpYWxzLnBhc3N3b3JkLFxuXG4gICAgICAvLyBOZWVkIHRvIHF1aWV0IG90aGVyd2lzZSBEb2NrZXIgd2lsbCBjb21wbGFpblxuICAgICAgLy8gJ1dBUk5JTkchIFlvdXIgcGFzc3dvcmQgd2lsbCBiZSBzdG9yZWQgdW5lbmNyeXB0ZWQnXG4gICAgICAvLyBkb2Vzbid0IHJlYWxseSBtYXR0ZXIgc2luY2UgaXQncyBhIHRva2VuLlxuICAgICAgcXVpZXQ6IHRydWUsXG4gICAgfSk7XG4gIH1cblxuICBwdWJsaWMgYXN5bmMgdGFnKHNvdXJjZVRhZzogc3RyaW5nLCB0YXJnZXRUYWc6IHN0cmluZykge1xuICAgIGF3YWl0IHRoaXMuZXhlY3V0ZShbJ3RhZycsIHNvdXJjZVRhZywgdGFyZ2V0VGFnXSk7XG4gIH1cblxuICBwdWJsaWMgYXN5bmMgcHVzaCh0YWc6IHN0cmluZykge1xuICAgIGF3YWl0IHRoaXMuZXhlY3V0ZShbJ3B1c2gnLCB0YWddKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBJZiBhIENESyBEb2NrZXIgQ3JlZGVudGlhbHMgZmlsZSBleGlzdHMsIGNyZWF0ZXMgYSBuZXcgRG9ja2VyIGNvbmZpZyBkaXJlY3RvcnkuXG4gICAqIFNldHMgdXAgYGRvY2tlci1jcmVkZW50aWFsLWNkay1hc3NldHNgIHRvIGJlIHRoZSBjcmVkZW50aWFsIGhlbHBlciBmb3IgZWFjaCBkb21haW4gaW4gdGhlIENESyBjb25maWcuXG4gICAqIEFsbCBmdXR1cmUgY29tbWFuZHMgKGUuZy4sIGBidWlsZGAsIGBwdXNoYCkgd2lsbCB1c2UgdGhpcyBjb25maWcuXG4gICAqXG4gICAqIFNlZSBodHRwczovL2RvY3MuZG9ja2VyLmNvbS9lbmdpbmUvcmVmZXJlbmNlL2NvbW1hbmRsaW5lL2xvZ2luLyNjcmVkZW50aWFsLWhlbHBlcnMgZm9yIG1vcmUgZGV0YWlscyBvbiBjcmVkIGhlbHBlcnMuXG4gICAqXG4gICAqIEByZXR1cm5zIHRydWUgaWYgQ0RLIGNvbmZpZyB3YXMgZm91bmQgYW5kIGNvbmZpZ3VyZWQsIGZhbHNlIG90aGVyd2lzZVxuICAgKi9cbiAgcHVibGljIGNvbmZpZ3VyZUNka0NyZWRlbnRpYWxzKCk6IGJvb2xlYW4ge1xuICAgIGNvbnN0IGNvbmZpZyA9IGNka0NyZWRlbnRpYWxzQ29uZmlnKCk7XG4gICAgaWYgKCFjb25maWcpIHsgcmV0dXJuIGZhbHNlOyB9XG5cbiAgICB0aGlzLmNvbmZpZ0RpciA9IGZzLm1rZHRlbXBTeW5jKHBhdGguam9pbihvcy50bXBkaXIoKSwgJ2Nka0RvY2tlckNvbmZpZycpKTtcblxuICAgIGNvbnN0IGRvbWFpbnMgPSBPYmplY3Qua2V5cyhjb25maWcuZG9tYWluQ3JlZGVudGlhbHMpO1xuICAgIGNvbnN0IGNyZWRIZWxwZXJzID0gZG9tYWlucy5yZWR1Y2UoKG1hcDogUmVjb3JkPHN0cmluZywgc3RyaW5nPiwgZG9tYWluKSA9PiB7XG4gICAgICBtYXBbZG9tYWluXSA9ICdjZGstYXNzZXRzJzsgLy8gVXNlIGRvY2tlci1jcmVkZW50aWFsLWNkay1hc3NldHMgZm9yIHRoaXMgZG9tYWluXG4gICAgICByZXR1cm4gbWFwO1xuICAgIH0sIHt9KTtcbiAgICBmcy53cml0ZUZpbGVTeW5jKHBhdGguam9pbih0aGlzLmNvbmZpZ0RpciwgJ2NvbmZpZy5qc29uJyksIEpTT04uc3RyaW5naWZ5KHsgY3JlZEhlbHBlcnMgfSksIHsgZW5jb2Rpbmc6ICd1dGYtOCcgfSk7XG5cbiAgICByZXR1cm4gdHJ1ZTtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZW1vdmVzIGFueSBjb25maWd1cmVkIERvY2tlciBjb25maWcgZGlyZWN0b3J5LlxuICAgKiBBbGwgZnV0dXJlIGNvbW1hbmRzIChlLmcuLCBgYnVpbGRgLCBgcHVzaGApIHdpbGwgdXNlIHRoZSBkZWZhdWx0IGNvbmZpZy5cbiAgICpcbiAgICogVGhpcyBpcyB1c2VmdWwgYWZ0ZXIgY2FsbGluZyBgY29uZmlndXJlQ2RrQ3JlZGVudGlhbHNgIHRvIHJlc2V0IHRvIGRlZmF1bHQgY3JlZGVudGlhbHMuXG4gICAqL1xuICBwdWJsaWMgcmVzZXRBdXRoUGx1Z2lucygpIHtcbiAgICB0aGlzLmNvbmZpZ0RpciA9IHVuZGVmaW5lZDtcbiAgfVxuXG4gIHByaXZhdGUgYXN5bmMgZXhlY3V0ZShhcmdzOiBzdHJpbmdbXSwgb3B0aW9uczogU2hlbGxPcHRpb25zID0ge30pIHtcbiAgICBjb25zdCBjb25maWdBcmdzID0gdGhpcy5jb25maWdEaXIgPyBbJy0tY29uZmlnJywgdGhpcy5jb25maWdEaXJdIDogW107XG5cbiAgICB0cnkge1xuICAgICAgYXdhaXQgc2hlbGwoWydkb2NrZXInLCAuLi5jb25maWdBcmdzLCAuLi5hcmdzXSwgeyBsb2dnZXI6IHRoaXMubG9nZ2VyLCAuLi5vcHRpb25zIH0pO1xuICAgIH0gY2F0Y2ggKGUpIHtcbiAgICAgIGlmIChlLmNvZGUgPT09ICdFTk9FTlQnKSB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcignVW5hYmxlIHRvIGV4ZWN1dGUgXFwnZG9ja2VyXFwnIGluIG9yZGVyIHRvIGJ1aWxkIGEgY29udGFpbmVyIGFzc2V0LiBQbGVhc2UgaW5zdGFsbCBcXCdkb2NrZXJcXCcgYW5kIHRyeSBhZ2Fpbi4nKTtcbiAgICAgIH1cbiAgICAgIHRocm93IGU7XG4gICAgfVxuICB9XG59XG5cbmZ1bmN0aW9uIGZsYXR0ZW4oeDogc3RyaW5nW11bXSkge1xuICByZXR1cm4gQXJyYXkucHJvdG90eXBlLmNvbmNhdChbXSwgLi4ueCk7XG59XG4iXX0=