"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.FileAssetHandler = void 0;
const fs_1 = require("fs");
const path = require("path");
const cloud_assembly_schema_1 = require("@aws-cdk/cloud-assembly-schema");
const mime = require("mime");
const progress_1 = require("../../progress");
const archive_1 = require("../archive");
const fs_extra_1 = require("../fs-extra");
const placeholders_1 = require("../placeholders");
const shell_1 = require("../shell");
class FileAssetHandler {
    constructor(workDir, asset, host) {
        this.workDir = workDir;
        this.asset = asset;
        this.host = host;
        this.fileCacheRoot = path.join(workDir, '.cache');
    }
    async publish() {
        const destination = await placeholders_1.replaceAwsPlaceholders(this.asset.destination, this.host.aws);
        const s3Url = `s3://${destination.bucketName}/${destination.objectKey}`;
        const s3 = await this.host.aws.s3Client(destination);
        this.host.emitMessage(progress_1.EventType.CHECK, `Check ${s3Url}`);
        // A thunk for describing the current account. Used when we need to format an error
        // message, not in the success case.
        const account = async () => { var _a; return (_a = (await this.host.aws.discoverCurrentAccount())) === null || _a === void 0 ? void 0 : _a.accountId; };
        switch (await bucketOwnership(s3, destination.bucketName)) {
            case BucketOwnership.MINE:
                break;
            case BucketOwnership.DOES_NOT_EXIST:
                throw new Error(`No bucket named '${destination.bucketName}'. Is account ${await account()} bootstrapped?`);
            case BucketOwnership.SOMEONE_ELSES_OR_NO_ACCESS:
                throw new Error(`Bucket named '${destination.bucketName}' exists, but not in account ${await account()}. Wrong account?`);
        }
        if (await objectExists(s3, destination.bucketName, destination.objectKey)) {
            this.host.emitMessage(progress_1.EventType.FOUND, `Found ${s3Url}`);
            return;
        }
        if (this.host.aborted) {
            return;
        }
        const publishFile = this.asset.source.executable ?
            await this.externalPackageFile(this.asset.source.executable) : await this.packageFile(this.asset.source);
        this.host.emitMessage(progress_1.EventType.UPLOAD, `Upload ${s3Url}`);
        await s3.upload({
            Bucket: destination.bucketName,
            Key: destination.objectKey,
            Body: fs_1.createReadStream(publishFile.packagedPath),
            ContentType: publishFile.contentType,
        }).promise();
    }
    async packageFile(source) {
        var _a;
        if (!source.path) {
            throw new Error(`'path' is expected in the File asset source, got: ${JSON.stringify(source)}`);
        }
        const fullPath = path.resolve(this.workDir, source.path);
        if (source.packaging === cloud_assembly_schema_1.FileAssetPackaging.ZIP_DIRECTORY) {
            const contentType = 'application/zip';
            await fs_1.promises.mkdir(this.fileCacheRoot, { recursive: true });
            const packagedPath = path.join(this.fileCacheRoot, `${this.asset.id.assetId}.zip`);
            if (await fs_extra_1.pathExists(packagedPath)) {
                this.host.emitMessage(progress_1.EventType.CACHED, `From cache ${path}`);
                return { packagedPath, contentType };
            }
            this.host.emitMessage(progress_1.EventType.BUILD, `Zip ${fullPath} -> ${path}`);
            await archive_1.zipDirectory(fullPath, packagedPath);
            return { packagedPath, contentType };
        }
        else {
            const contentType = (_a = mime.getType(fullPath)) !== null && _a !== void 0 ? _a : 'application/octet-stream';
            return { packagedPath: fullPath, contentType };
        }
    }
    async externalPackageFile(executable) {
        this.host.emitMessage(progress_1.EventType.BUILD, `Building asset source using command: '${executable}'`);
        return {
            packagedPath: (await shell_1.shell(executable, { quiet: true })).trim(),
            contentType: 'application/zip',
        };
    }
}
exports.FileAssetHandler = FileAssetHandler;
var BucketOwnership;
(function (BucketOwnership) {
    BucketOwnership[BucketOwnership["DOES_NOT_EXIST"] = 0] = "DOES_NOT_EXIST";
    BucketOwnership[BucketOwnership["MINE"] = 1] = "MINE";
    BucketOwnership[BucketOwnership["SOMEONE_ELSES_OR_NO_ACCESS"] = 2] = "SOMEONE_ELSES_OR_NO_ACCESS";
})(BucketOwnership || (BucketOwnership = {}));
async function bucketOwnership(s3, bucket) {
    try {
        await s3.getBucketLocation({ Bucket: bucket }).promise();
        return BucketOwnership.MINE;
    }
    catch (e) {
        if (e.code === 'NoSuchBucket') {
            return BucketOwnership.DOES_NOT_EXIST;
        }
        if (['AccessDenied', 'AllAccessDisabled'].includes(e.code)) {
            return BucketOwnership.SOMEONE_ELSES_OR_NO_ACCESS;
        }
        throw e;
    }
}
async function objectExists(s3, bucket, key) {
    /*
     * The object existence check here refrains from using the `headObject` operation because this
     * would create a negative cache entry, making GET-after-PUT eventually consistent. This has been
     * observed to result in CloudFormation issuing "ValidationError: S3 error: Access Denied", for
     * example in https://github.com/aws/aws-cdk/issues/6430.
     *
     * To prevent this, we are instead using the listObjectsV2 call, using the looked up key as the
     * prefix, and limiting results to 1. Since the list operation returns keys ordered by binary
     * UTF-8 representation, the key we are looking for is guaranteed to always be the first match
     * returned if it exists.
     */
    const response = await s3.listObjectsV2({ Bucket: bucket, Prefix: key, MaxKeys: 1 }).promise();
    return response.Contents != null && response.Contents.some(object => object.Key === key);
}
//# sourceMappingURL=data:application/json;base64,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