"""Check if a host is in the Google Chrome HSTS Preload list"""

import functools
import os
import typing

__version__ = "2020.7.17"
__checksum__ = "9e03ec3e1933ce414487db974c064889048a61c0f52e0a0959b57242d237c2e1"
__all__ = ["in_hsts_preload"]

# fmt: off
_GTLD_INCLUDE_SUBDOMAINS = {b'android', b'app', b'bank', b'chrome', b'dev', b'foo', b'gle', b'gmail', b'google', b'hangout', b'insurance', b'meet', b'new', b'page', b'play', b'search', b'youtube'}  # noqa: E501
_JUMPTABLE = [[(0, 11), (11, 5), None, (16, 57), (73, 26), (99, 12), None, (111, 19), (130, 11), (141, 7), (148, 20), (168, 18), None, (186, 22), (208, 45), (253, 7), (260, 9), (269, 36), (305, 10), (315, 10), (325, 21), None, (346, 50), (396, 8), (404, 9), (413, 19), (432, 13), (445, 14), (459, 14), None, None, (473, 29), (502, 16), (518, 35), (553, 14), (567, 24), (591, 9), None, (600, 25), (625, 20), (645, 8), (653, 13), (666, 10), None, (676, 11), (687, 6), (693, 26), (719, 5), (724, 5), (729, 10), (739, 10), (749, 11), (760, 12), (772, 27), None, (799, 11), (810, 11), (821, 7), (828, 29), (857, 18), (875, 27), (902, 46), (948, 25), (973, 16), (989, 8), (997, 5), (1002, 22), (1024, 18), None, (1042, 36), (1078, 15), (1093, 8), (1101, 5), None, (1106, 5), (1111, 16), (1127, 14), (1141, 18), None, (1159, 14), (1173, 18), (1191, 48), (1239, 19), (1258, 5), (1263, 46), (1309, 14), (1323, 14), (1337, 20), None, (1357, 10), (1367, 13), (1380, 10), (1390, 19), None, (1409, 13), (1422, 19), (1441, 5), (1446, 4), (1450, 22), (1472, 10), (1482, 7), (1489, 14), (1503, 21), (1524, 11), (1535, 10), (1545, 12), (1557, 32), None, (1589, 10), (1599, 14), (1613, 12), (1625, 45), (1670, 15), None, (1685, 11), (1696, 23), (1719, 21), (1740, 26), (1766, 6), (1772, 6), (1778, 7), (1785, 5), (1790, 20), (1810, 23), (1833, 24), (1857, 13), (1870, 15), (1885, 19), (1904, 6), (1910, 61), (1971, 44), (2015, 12), (2027, 23), (2050, 16), (2066, 38), (2104, 6), (2110, 12), (2122, 44), (2166, 6), (2172, 41), (2213, 13), (2226, 23), (2249, 30), (2279, 16), (2295, 8), (2303, 6), (2309, 12), (2321, 19), (2340, 21), (2361, 15), None, (2376, 35), (2411, 21), (2432, 17), (2449, 19), (2468, 26), (2494, 5), (2499, 37), (2536, 30), (2566, 16), (2582, 10), (2592, 17), (2609, 23), (2632, 14), (2646, 17), (2663, 8), (2671, 4), (2675, 7), (2682, 29), (2711, 6), (2717, 18), (2735, 27), (2762, 20), (2782, 17), (2799, 19), (2818, 12), (2830, 40), (2870, 40), (2910, 12), (2922, 48), (2970, 25), (2995, 12), None, (3007, 8), (3015, 20), (3035, 19), (3054, 6), (3060, 23), None, (3083, 23), (3106, 33), (3139, 14), (3153, 12), (3165, 27), None, (3192, 26), (3218, 31), (3249, 50), (3299, 15), (3314, 20), (3334, 15), (3349, 21), (3370, 32), (3402, 24), (3426, 20), (3446, 13), (3459, 60), (3519, 19), (3538, 9), (3547, 12), (3559, 12), (3571, 11), (3582, 10), (3592, 48), (3640, 32), None, (3672, 25), (3697, 12), None, (3709, 8), (3717, 8), (3725, 7), None, (3732, 25), (3757, 17), None, (3774, 21), (3795, 35), (3830, 12), (3842, 10), (3852, 36), (3888, 20), (3908, 22), (3930, 23), (3953, 19), (3972, 12), (3984, 5), (3989, 30), (4019, 24), (4043, 14), (4057, 14), (4071, 47), (4118, 46), None, None, (4164, 51), (4215, 42), None, (4257, 14), None, (4271, 15), (4286, 8), (4294, 21), (4315, 6), (4321, 16), (4337, 17)], [(4354, 6037), (10391, 6485), (16876, 6911), (23787, 5772), (29559, 6134), (35693, 5824), (41517, 6824), (48341, 5968), (54309, 6607), (60916, 5939), (66855, 6938), (73793, 6221), (80014, 6584), (86598, 6955), (93553, 6434), (99987, 6297), (106284, 6926), (113210, 5854), (119064, 6168), (125232, 6466), (131698, 6722), (138420, 6445), (144865, 6754), (151619, 5955), (157574, 6072), (163646, 6543), (170189, 6426), (176615, 6698), (183313, 6160), (189473, 6395), (195868, 6712), (202580, 6304), (208884, 6284), (215168, 6893), (222061, 5989), (228050, 6689), (234739, 6177), (240916, 6865), (247781, 6579), (254360, 6876), (261236, 7284), (268520, 6233), (274753, 6206), (280959, 5992), (286951, 6214), (293165, 5971), (299136, 6149), (305285, 6895), (312180, 6186), (318366, 5590), (323956, 6365), (330321, 6456), (336777, 6397), (343174, 6627), (349801, 6670), (356471, 6547), (363018, 6671), (369689, 5700), (375389, 6680), (382069, 5674), (387743, 6436), (394179, 6294), (400473, 6212), (406685, 6688), (413373, 6492), (419865, 6428), (426293, 5960), (432253, 6839), (439092, 6551), (445643, 6688), (452331, 6281), (458612, 6328), (464940, 5609), (470549, 6879), (477428, 6888), (484316, 6712), (491028, 5977), (497005, 7049), (504054, 6885), (510939, 5959), (516898, 6548), (523446, 5649), (529095, 6242), (535337, 6457), (541794, 6355), (548149, 6274), (554423, 6459), (560882, 6570), (567452, 6564), (574016, 6312), (580328, 7067), (587395, 5938), (593333, 6064), (599397, 6478), (605875, 6372), (612247, 6910), (619157, 6698), (625855, 6253), (632108, 6140), (638248, 6026), (644274, 6099), (650373, 6671), (657044, 6117), (663161, 6350), (669511, 5990), (675501, 6696), (682197, 6478), (688675, 6833), (695508, 7860), (703368, 6963), (710331, 6753), (717084, 6332), (723416, 6139), (729555, 6576), (736131, 6785), (742916, 6341), (749257, 6133), (755390, 6276), (761666, 6266), (767932, 6734), (774666, 6630), (781296, 6637), (787933, 7005), (794938, 6698), (801636, 7665), (809301, 6292), (815593, 5654), (821247, 6766), (828013, 6499), (834512, 7831), (842343, 6797), (849140, 6087), (855227, 6595), (861822, 6739), (868561, 6237), (874798, 6631), (881429, 5900), (887329, 6654), (893983, 6281), (900264, 6400), (906664, 6381), (913045, 7118), (920163, 6078), (926241, 6143), (932384, 6448), (938832, 6370), (945202, 6330), (951532, 6714), (958246, 6048), (964294, 6942), (971236, 6490), (977726, 6611), (984337, 6692), (991029, 6378), (997407, 6377), (1003784, 6317), (1010101, 6235), (1016336, 6248), (1022584, 6081), (1028665, 5607), (1034272, 5989), (1040261, 6424), (1046685, 7081), (1053766, 5997), (1059763, 6423), (1066186, 6831), (1073017, 6225), (1079242, 6018), (1085260, 6643), (1091903, 6342), (1098245, 5808), (1104053, 6392), (1110445, 7431), (1117876, 5884), (1123760, 6052), (1129812, 6536), (1136348, 6130), (1142478, 6569), (1149047, 6222), (1155269, 5907), (1161176, 7230), (1168406, 6556), (1174962, 6231), (1181193, 6839), (1188032, 7151), (1195183, 7202), (1202385, 6092), (1208477, 6834), (1215311, 6157), (1221468, 6478), (1227946, 6614), (1234560, 5993), (1240553, 6721), (1247274, 6830), (1254104, 6453), (1260557, 6319), (1266876, 6168), (1273044, 6290), (1279334, 6613), (1285947, 6139), (1292086, 6486), (1298572, 5760), (1304332, 6990), (1311322, 6806), (1318128, 6530), (1324658, 6778), (1331436, 5596), (1337032, 6538), (1343570, 6326), (1349896, 6506), (1356402, 6659), (1363061, 6982), (1370043, 6423), (1376466, 6521), (1382987, 6602), (1389589, 6306), (1395895, 6308), (1402203, 6361), (1408564, 6453), (1415017, 6127), (1421144, 6327), (1427471, 5886), (1433357, 7293), (1440650, 6506), (1447156, 6233), (1453389, 6624), (1460013, 6488), (1466501, 5692), (1472193, 6548), (1478741, 6391), (1485132, 7471), (1492603, 6346), (1498949, 5718), (1504667, 6886), (1511553, 6299), (1517852, 6889), (1524741, 6012), (1530753, 6071), (1536824, 5759), (1542583, 6550), (1549133, 6365), (1555498, 6747), (1562245, 6087), (1568332, 6469), (1574801, 6260), (1581061, 6909), (1587970, 6270), (1594240, 5695), (1599935, 6422), (1606357, 6101), (1612458, 6509), (1618967, 6660), (1625627, 7024), (1632651, 6139), (1638790, 6135), (1644925, 6560)], [(1651485, 703), (1652188, 625), (1652813, 628), (1653441, 663), (1654104, 523), (1654627, 611), (1655238, 644), (1655882, 808), (1656690, 640), (1657330, 627), (1657957, 509), (1658466, 544), (1659010, 758), (1659768, 886), (1660654, 936), (1661590, 714), (1662304, 1207), (1663511, 589), (1664100, 846), (1664946, 640), (1665586, 733), (1666319, 708), (1667027, 802), (1667829, 709), (1668538, 684), (1669222, 631), (1669853, 940), (1670793, 1033), (1671826, 767), (1672593, 657), (1673250, 896), (1674146, 728), (1674874, 557), (1675431, 671), (1676102, 732), (1676834, 761), (1677595, 619), (1678214, 674), (1678888, 692), (1679580, 929), (1680509, 683), (1681192, 823), (1682015, 660), (1682675, 671), (1683346, 728), (1684074, 362), (1684436, 763), (1685199, 857), (1686056, 673), (1686729, 525), (1687254, 789), (1688043, 633), (1688676, 763), (1689439, 935), (1690374, 918), (1691292, 464), (1691756, 661), (1692417, 486), (1692903, 578), (1693481, 701), (1694182, 734), (1694916, 753), (1695669, 1008), (1696677, 836), (1697513, 602), (1698115, 692), (1698807, 753), (1699560, 444), (1700004, 561), (1700565, 487), (1701052, 692), (1701744, 786), (1702530, 540), (1703070, 725), (1703795, 634), (1704429, 671), (1705100, 552), (1705652, 680), (1706332, 768), (1707100, 428), (1707528, 672), (1708200, 607), (1708807, 828), (1709635, 623), (1710258, 588), (1710846, 282), (1711128, 597), (1711725, 703), (1712428, 756), (1713184, 663), (1713847, 816), (1714663, 1074), (1715737, 788), (1716525, 773), (1717298, 674), (1717972, 436), (1718408, 938), (1719346, 858), (1720204, 564), (1720768, 580), (1721348, 666), (1722014, 843), (1722857, 839), (1723696, 541), (1724237, 632), (1724869, 697), (1725566, 363), (1725929, 467), (1726396, 924), (1727320, 863), (1728183, 792), (1728975, 774), (1729749, 610), (1730359, 771), (1731130, 643), (1731773, 683), (1732456, 689), (1733145, 433), (1733578, 639), (1734217, 588), (1734805, 914), (1735719, 653), (1736372, 789), (1737161, 404), (1737565, 703), (1738268, 656), (1738924, 835), (1739759, 883), (1740642, 738), (1741380, 904), (1742284, 747), (1743031, 524), (1743555, 757), (1744312, 583), (1744895, 737), (1745632, 715), (1746347, 639), (1746986, 682), (1747668, 589), (1748257, 640), (1748897, 594), (1749491, 691), (1750182, 691), (1750873, 637), (1751510, 437), (1751947, 549), (1752496, 641), (1753137, 556), (1753693, 683), (1754376, 594), (1754970, 761), (1755731, 520), (1756251, 490), (1756741, 656), (1757397, 551), (1757948, 607), (1758555, 639), (1759194, 796), (1759990, 594), (1760584, 609), (1761193, 841), (1762034, 826), (1762860, 522), (1763382, 695), (1764077, 796), (1764873, 587), (1765460, 659), (1766119, 480), (1766599, 580), (1767179, 621), (1767800, 714), (1768514, 598), (1769112, 898), (1770010, 676), (1770686, 794), (1771480, 683), (1772163, 649), (1772812, 580), (1773392, 646), (1774038, 699), (1774737, 1277), (1776014, 514), (1776528, 641), (1777169, 608), (1777777, 944), (1778721, 754), (1779475, 729), (1780204, 546), (1780750, 565), (1781315, 808), (1782123, 553), (1782676, 569), (1783245, 826), (1784071, 650), (1784721, 871), (1785592, 772), (1786364, 668), (1787032, 671), (1787703, 783), (1788486, 620), (1789106, 889), (1789995, 631), (1790626, 722), (1791348, 558), (1791906, 689), (1792595, 449), (1793044, 753), (1793797, 779), (1794576, 648), (1795224, 901), (1796125, 775), (1796900, 759), (1797659, 903), (1798562, 1077), (1799639, 815), (1800454, 586), (1801040, 839), (1801879, 662), (1802541, 483), (1803024, 443), (1803467, 692), (1804159, 749), (1804908, 406), (1805314, 974), (1806288, 462), (1806750, 758), (1807508, 844), (1808352, 712), (1809064, 769), (1809833, 626), (1810459, 743), (1811202, 665), (1811867, 753), (1812620, 566), (1813186, 566), (1813752, 408), (1814160, 601), (1814761, 437), (1815198, 740), (1815938, 815), (1816753, 764), (1817517, 718), (1818235, 621), (1818856, 568), (1819424, 808), (1820232, 422), (1820654, 573), (1821227, 780), (1822007, 513), (1822520, 839), (1823359, 2071), (1825430, 507), (1825937, 602), (1826539, 875), (1827414, 833), (1828247, 510)], [(1828757, 48), None, (1828805, 35), (1828840, 42), None, None, None, None, None, None, None, None, None, None, None, None, None, (1828882, 42), None, (1828924, 25), (1828949, 16), None, None, None, None, None, None, (1828965, 26), None, None, None, None, (1828991, 21), (1829012, 25), None, None, (1829037, 26), None, None, None, None, (1829063, 44), (1829107, 21), (1829128, 23), None, None, None, None, (1829151, 48), None, None, None, None, None, (1829199, 31), None, None, None, None, (1829230, 42), None, (1829272, 22), None, (1829294, 21), None, (1829315, 26), (1829341, 42), None, None, (1829383, 77), None, None, None, None, None, (1829460, 21), (1829481, 21), None, None, (1829502, 34), (1829536, 42), None, None, None, (1829578, 25), None, None, (1829603, 21), None, None, None, None, None, (1829624, 24), (1829648, 21), None, None, (1829669, 26), None, (1829695, 18), None, (1829713, 54), None, None, None, None, None, None, (1829767, 26), None, (1829793, 19), None, (1829812, 20), None, None, (1829832, 42), (1829874, 42), (1829916, 17), None, (1829933, 26), None, (1829959, 26), None, None, None, (1829985, 26), (1830011, 20), (1830031, 26), None, (1830057, 42), (1830099, 63), None, None, None, (1830162, 40), (1830202, 48), None, None, None, (1830250, 47), None, None, None, None, None, None, None, (1830297, 42), None, (1830339, 55), None, (1830394, 9), None, (1830403, 21), (1830424, 42), None, None, (1830466, 42), (1830508, 82), None, None, (1830590, 42), None, None, None, None, None, None, None, None, None, (1830632, 42), (1830674, 21), (1830695, 21), None, (1830716, 42), (1830758, 25), None, None, (1830783, 21), (1830804, 42), None, None, (1830846, 21), (1830867, 19), (1830886, 26), None, None, None, (1830912, 21), None, None, (1830933, 38), None, (1830971, 22), (1830993, 21), (1831014, 21), None, None, (1831035, 63), None, (1831098, 21), (1831119, 42), None, (1831161, 17), None, None, None, None, (1831178, 21), (1831199, 21), None, None, (1831220, 21), None, None, (1831241, 21), None, (1831262, 26), None, (1831288, 50), None, None, None, (1831338, 50), (1831388, 26), (1831414, 21), (1831435, 21), (1831456, 19), None, (1831475, 35), (1831510, 26), (1831536, 23), (1831559, 21), (1831580, 42), None, None, None, None, None, None, (1831622, 21), None, None, None, (1831643, 21), None, None, (1831664, 90), None, (1831754, 239), (1831993, 38), None, None, None, None]]  # noqa: E501
_CRC8_TABLE = [
    0x00, 0x07, 0x0e, 0x09, 0x1c, 0x1b, 0x12, 0x15,
    0x38, 0x3f, 0x36, 0x31, 0x24, 0x23, 0x2a, 0x2d,
    0x70, 0x77, 0x7e, 0x79, 0x6c, 0x6b, 0x62, 0x65,
    0x48, 0x4f, 0x46, 0x41, 0x54, 0x53, 0x5a, 0x5d,
    0xe0, 0xe7, 0xee, 0xe9, 0xfc, 0xfb, 0xf2, 0xf5,
    0xd8, 0xdf, 0xd6, 0xd1, 0xc4, 0xc3, 0xca, 0xcd,
    0x90, 0x97, 0x9e, 0x99, 0x8c, 0x8b, 0x82, 0x85,
    0xa8, 0xaf, 0xa6, 0xa1, 0xb4, 0xb3, 0xba, 0xbd,
    0xc7, 0xc0, 0xc9, 0xce, 0xdb, 0xdc, 0xd5, 0xd2,
    0xff, 0xf8, 0xf1, 0xf6, 0xe3, 0xe4, 0xed, 0xea,
    0xb7, 0xb0, 0xb9, 0xbe, 0xab, 0xac, 0xa5, 0xa2,
    0x8f, 0x88, 0x81, 0x86, 0x93, 0x94, 0x9d, 0x9a,
    0x27, 0x20, 0x29, 0x2e, 0x3b, 0x3c, 0x35, 0x32,
    0x1f, 0x18, 0x11, 0x16, 0x03, 0x04, 0x0d, 0x0a,
    0x57, 0x50, 0x59, 0x5e, 0x4b, 0x4c, 0x45, 0x42,
    0x6f, 0x68, 0x61, 0x66, 0x73, 0x74, 0x7d, 0x7a,
    0x89, 0x8e, 0x87, 0x80, 0x95, 0x92, 0x9b, 0x9c,
    0xb1, 0xb6, 0xbf, 0xb8, 0xad, 0xaa, 0xa3, 0xa4,
    0xf9, 0xfe, 0xf7, 0xf0, 0xe5, 0xe2, 0xeb, 0xec,
    0xc1, 0xc6, 0xcf, 0xc8, 0xdd, 0xda, 0xd3, 0xd4,
    0x69, 0x6e, 0x67, 0x60, 0x75, 0x72, 0x7b, 0x7c,
    0x51, 0x56, 0x5f, 0x58, 0x4d, 0x4a, 0x43, 0x44,
    0x19, 0x1e, 0x17, 0x10, 0x05, 0x02, 0x0b, 0x0c,
    0x21, 0x26, 0x2f, 0x28, 0x3d, 0x3a, 0x33, 0x34,
    0x4e, 0x49, 0x40, 0x47, 0x52, 0x55, 0x5c, 0x5b,
    0x76, 0x71, 0x78, 0x7f, 0x6a, 0x6d, 0x64, 0x63,
    0x3e, 0x39, 0x30, 0x37, 0x22, 0x25, 0x2c, 0x2b,
    0x06, 0x01, 0x08, 0x0f, 0x1a, 0x1d, 0x14, 0x13,
    0xae, 0xa9, 0xa0, 0xa7, 0xb2, 0xb5, 0xbc, 0xbb,
    0x96, 0x91, 0x98, 0x9f, 0x8a, 0x8d, 0x84, 0x83,
    0xde, 0xd9, 0xd0, 0xd7, 0xc2, 0xc5, 0xcc, 0xcb,
    0xe6, 0xe1, 0xe8, 0xef, 0xfa, 0xfd, 0xf4, 0xf3
]
# fmt: on

_IS_LEAF = 0x80
_INCLUDE_SUBDOMAINS = 0x40


try:
    from importlib.resources import open_binary

    def open_pkg_binary(path: str) -> typing.BinaryIO:
        return open_binary("hstspreload", path)


except ImportError:

    def open_pkg_binary(path: str) -> typing.BinaryIO:
        return open(
            os.path.join(os.path.dirname(os.path.abspath(__file__)), path), "rb",
        )


@functools.lru_cache(maxsize=1024)
def in_hsts_preload(host: typing.AnyStr) -> bool:
    """Determines if an IDNA-encoded host is on the HSTS preload list"""

    if isinstance(host, str):
        host = host.encode("ascii")
    labels = host.lower().split(b".")

    # Fast-branch for gTLDs that are registered to preload all sub-domains.
    if labels[-1] in _GTLD_INCLUDE_SUBDOMAINS:
        return True

    with open_pkg_binary("hstspreload.bin") as f:
        for layer, label in enumerate(labels[::-1]):
            # None of our layers are greater than 4 deep.
            if layer > 3:
                return False

            # Read the jump table for the layer and label
            jump_info = _JUMPTABLE[layer][_crc8(label)]
            if jump_info is None:
                # No entry: host is not preloaded
                return False

            # Read the set of entries for that layer and label
            f.seek(jump_info[0])
            data = bytearray(jump_info[1])
            f.readinto(data)

            for is_leaf, include_subdomains, ent_label in _iter_entries(data):
                # We found a potential leaf
                if is_leaf:
                    if ent_label == host:
                        return True
                    if include_subdomains and host.endswith(b"." + ent_label):
                        return True

                # Continue traversing as we're not at a leaf.
                elif label == ent_label:
                    break
            else:
                return False
    return False


def _iter_entries(data: bytes) -> typing.Iterable[typing.Tuple[int, int, bytes]]:
    while data:
        flags = data[0]
        size = data[1]
        label = bytes(data[2 : 2 + size])
        yield (flags & _IS_LEAF, flags & _INCLUDE_SUBDOMAINS, label)
        data = data[2 + size :]


def _crc8(value: bytes) -> int:
    # CRC8 reference implementation: https://github.com/niccokunzmann/crc8
    checksum = 0x00
    for byte in value:
        checksum = _CRC8_TABLE[checksum ^ byte]
    return checksum
