#!/usr/bin/python

#
# % $Id$
#
#
# Copyright (C) 2002-2007
# The MeqTree Foundation &
# ASTRON (Netherlands Foundation for Research in Astronomy)
# P.O.Box 2, 7990 AA Dwingeloo, The Netherlands
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, see <http://www.gnu.org/licenses/>,
# or write to the Free Software Foundation, Inc.,
# 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
try:
    from PyQt4.Qt import SIGNAL, QPixmap, QIcon
except ImportError:
    class QPixmap:
        def __init__(self, *args, **kwargs):
            pass

from .utils import verbosity

import sys

_dbg = verbosity(0, name='pixmaps')
_dprint = _dbg.dprint
_dprintf = _dbg.dprintf

# define a catch-all case for all missing icons
_missing_icon_xpm = ["16 16 14 1",
                     "# c None",
                     "k c None",
                     "d c None",
                     "c c None",
                     "b c None",
                     "f c None",
                     "l c None",
                     "i c None",
                     "j c None",
                     "h c None",
                     ". c None",
                     "a c #000000",
                     "g c #808080",
                     "e c #ff0000",
                     "....##aaabb#c#.#",
                     "##db#aeeea##c###",
                     "f#gg#aeeeahgg#d#",
                     ".#gggaeeeaggg#c#",
                     "c##ggaeeeaggc#.#",
                     "##hhgaeeeag#c###",
                     "##h##aeeeahh##f#",
                     "##hhhaeeeaihhh#.",
                     ".c#hhaeeeahih#dc",
                     "#jf#gaaaaagh#dfh",
                     "##kggg#hbggglchh",
                     "##gggbaaahggg#d#",
                     "c.gg#aeeeahgg###",
                     "###hhaeeea#h####",
                     "#####aeeea######",
                     "f#.#.jaaad.cccdj"]


class QPixmapWrapper(object):
    """QPixmapWrapper can defer initialization of a pixmap until the pixmap
    is actually retrieved with the pm() or iconset() method for the first time.
    This gets around the problem of not being able to create QPixmaps until a Qt
    application has been initialized.
    """

    def __init__(self, pm=_missing_icon_xpm):
        """Initialize wrapper with a pixmap or with an xpm string array"""
        self.assign(pm)

    def assign(self, pm):
        """Reassign pixmap or xpm string array to wrapper"""
        if isinstance(pm, QPixmap):
            self._pm = pm
        else:  # assume xpm string list to be decoded on-demand
            self._xpmstr = pm
            self._pm = None
        self._icon = None

    def pm(self):
        """Get QPixmap from wrapper"""
        if self._pm is None:
            self._pm = QPixmap(self._xpmstr)
        return self._pm

    def icon(self):
        """Get QIcon from wrapper"""
        if self._icon is None:
            self._icon = QIcon(self.pm())
        return self._icon

    # for compatibility with old code
    iconset = icon


exclaim = QPixmapWrapper(["14 14 3 1",
                          "       c None",
                          ".      c red",
                          "X      c yellow",
                          "              ",
                          "     ....     ",
                          "    ......    ",
                          "    ......    ",
                          "    ......    ",
                          "    ......    ",
                          "    ......    ",
                          "     ....     ",
                          "     ....     ",
                          "      ..      ",
                          "              ",
                          "      ..      ",
                          "     ....     ",
                          "      ..      "])

cancel = QPixmapWrapper(["16 16 5 1",
                         "p c #800000",
                         ". c #400000",
                         "X c #FF0000",
                         "o c #C00000",
                         "  c None",
                         "                ",
                         "                ",
                         "            X   ",
                         "    XX     Xoo  ",
                         "   XooXX  Xoo.  ",
                         "     pooXooo.   ",
                         "      poooo.    ",
                         "       XooX     ",
                         "      XoooX     ",
                         "     Xo. poX    ",
                         "     X.   po    ",
                         "    Xo     po   ",
                         "    X.      o   ",
                         "    X           ",
                         "                ",
                         "                "])

check = QPixmapWrapper(["16 14 8 1",
                        "  c #000000",
                        ". c #400000",
                        "X c None",
                        "o c #00C000",
                        "O c #008000",
                        "+ c #C0FFC0",
                        "@ c #004000",
                        "# c None",
                        "XXXXXXXXXXXXXXX+",
                        "XXXXXXXXXXXXXX+o",
                        "XXXXXXXXXXXXX+oO",
                        "XXXXXXXXXXXX+oO@",
                        "XXXXXXXXXXX+oO@ ",
                        "XX+OXXXXXX+oO@ X",
                        "X+ooOXXXX+oO@.XX",
                        " @OooOXX+oO@ XXX",
                        "X @OooO+oO@.XXXX",
                        "XX.@OoooO@.XXXXX",
                        "XXX @OoO@ XXXXXX",
                        "XXXX @O@ XXXXXXX",
                        "XXXXX . XXXXXXXX",
                        "XXXXXXXXXXXXXXXX"])

pause_normal = QPixmapWrapper(["16 16 95 2",
                               "   c None",
                               " . c #CDDAF4",
                               " X c #3A64B3",
                               " o c #4887FC",
                               " O c #3B66B7",
                               " + c #3269CF",
                               " @ c #6696EE",
                               " # c #5183E0",
                               " $ c #6494EC",
                               " % c #C4D4F1",
                               " & c #6C94E0",
                               " * c #3269D2",
                               " = c #3B80FF",
                               " - c #90B1EC",
                               " ; c #3B65B3",
                               " : c #3472E4",
                               " > c #656565",
                               " , c #5E88D5",
                               " < c #4686FC",
                               " 1 c #4A7FE2",
                               " 2 c #A3BCEA",
                               " 3 c #3569CA",
                               " 4 c #4987F8",
                               " 5 c #4E6286",
                               " 6 c #B8CCEE",
                               " 7 c #3C81FF",
                               " 8 c #3471E3",
                               " 9 c #3C66B6",
                               " 0 c #A8C2F5",
                               " q c #3E70CF",
                               " w c #326CD7",
                               " e c #486190",
                               " r c #7EA3E6",
                               " t c #5789E4",
                               " y c #3665BD",
                               " u c #8FADE6",
                               " i c #3269D0",
                               " p c #DCE6F6",
                               " a c #4D648E",
                               " s c #3D82FF",
                               " d c #8EADE8",
                               " f c #4A80E4",
                               " g c #356ACC",
                               " h c #3368CA",
                               " j c #5A8AE3",
                               " k c #3D64AC",
                               " l c #4980E6",
                               " z c #6E95DC",
                               " x c #3268CC",
                               " c c #4888FC",
                               " v c #4686FA",
                               " b c #3467C4",
                               " n c #D4E0F4",
                               " m c #3472E5",
                               " M c #EBF0FA",
                               " N c #3776EB",
                               " B c #8CACE5",
                               " V c #326AD2",
                               " C c #A1BAE9",
                               " Z c #B9CCF0",
                               " A c #B7CAEE",
                               " S c #3765B9",
                               " D c #3B66B6",
                               " F c #5B88DC",
                               " G c #6E99E8",
                               " H c #3E62A5",
                               " J c #3269CE",
                               " K c #4687FC",
                               " L c #4F6184",
                               " P c #202020",
                               " I c #3269D1",
                               " U c #799FE5",
                               " Y c #E4EBF8",
                               " T c #326BD4",
                               " R c #346ACC",
                               " E c #3268CA",
                               " W c #658EDB",
                               " Q c #3572E4",
                               " ! c #CBD9F3",
                               " ~ c #3E71CF",
                               " ^ c #E1E9F8",
                               " / c #3268CD",
                               " ( c #4083FF",
                               " ) c #3766BE",
                               " _ c #3873E0",
                               " ` c #97B3E4",
                               " ' c #8EACE5",
                               " ] c #447CE5",
                               " [ c #3676EB",
                               " { c #F2F5FB",
                               " } c #749DE9",
                               " | c #8DACE7",
                               ".  c #326AD3",
                               ".. c #3469CB",
                               ".X c None",
                               "           I + / x E y          ",
                               "       I * w m # F : T h S      ",
                               "     I.  l 0 Y { { M Z t J D    ",
                               "   i V $ n ^ ^ ^ ^ ^ ^ p U J H  ",
                               "   + l % u u u . . u u u ! j b >",
                               " x T - 6 P P P 6 6 P P P A `.  e",
                               " x _ C 2 P P P 2 2 P P P 4 = 8 X",
                               " x 1 B u P P P ' ' P P P 7 = [ O",
                               " x ] G } P P P 7 v P P P 7 7 N 9",
                               " E Q < < P P P 7 7 P P P s ( : k",
                               " y T o c P P P 7 7 P P P K o.  L",
                               "   h f @ P P P @ @ P P P @ f ) >",
                               "   S R & r r r r r r r r &.. 5 >",
                               "     ; g W | d d d d | W 3 a >  ",
                               "       H b ~ , z z , q ) 5 >    ",
                               "         > e X O 9 k L > >      "])

pause_green = QPixmapWrapper(["16 16 94 2",
                              "Qt c None",
                              ".P c #ffff00",
                              ".d c #3268ca",
                              ".c c #3268cc",
                              ".b c #3268cd",
                              ".w c #3269ce",
                              ".a c #3269cf",
                              ".y c #3269d0",
                              ".# c #3269d1",
                              ".f c #3269d2",
                              ".z c #326ad2",
                              ".o c #326ad3",
                              ".l c #326bd4",
                              ".g c #326cd7",
                              ".m c #3368ca",
                              ".L c #3467c4",
                              "#p c #3469cb",
                              "#m c #346acc",
                              ".Y c #3471e3",
                              ".k c #3472e4",
                              ".h c #3472e5",
                              "#w c #3569ca",
                              "#s c #356acc",
                              "#a c #3572e4",
                              ".e c #3665bd",
                              ".4 c #3676eb",
                              ".n c #3765b9",
                              "#l c #3766be",
                              "#. c #3776eb",
                              ".T c #3873e0",
                              ".Z c #3a64b3",
                              "#r c #3b65b3",
                              ".x c #3b66b6",
                              ".5 c #3b66b7",
                              ".X c #3b80ff",
                              "## c #3c66b6",
                              ".3 c #3c81ff",
                              "#e c #3d64ac",
                              "#c c #3d82ff",
                              ".F c #3e62a5",
                              "#B c #3e70cf",
                              "#y c #3e71cf",
                              "#d c #4083ff",
                              ".6 c #447ce5",
                              ".9 c #4686fa",
                              "#b c #4686fc",
                              "#h c #4687fc",
                              ".S c #486190",
                              "#f c #4887fc",
                              "#g c #4888fc",
                              ".p c #4980e6",
                              ".W c #4987f8",
                              ".0 c #4a7fe2",
                              "#j c #4a80e4",
                              "#x c #4d648e",
                              "#q c #4e6286",
                              "#i c #4f6184",
                              ".i c #5183e0",
                              ".v c #5789e4",
                              ".K c #5a8ae3",
                              ".j c #5b88dc",
                              "#z c #5e88d5",
                              ".A c #6494ec",
                              ".M c #656565",
                              "#t c #658edb",
                              "#k c #6696ee",
                              "#n c #6c94e0",
                              "#A c #6e95dc",
                              ".7 c #6e99e8",
                              ".8 c #749de9",
                              ".E c #799fe5",
                              "#o c #7ea3e6",
                              ".1 c #8cace5",
                              "#u c #8dace7",
                              ".2 c #8eace5",
                              "#v c #8eade8",
                              ".H c #8fade6",
                              ".N c #90b1ec",
                              ".R c #97b3e4",
                              ".U c #a1bae9",
                              ".V c #a3bcea",
                              ".q c #a8c2f5",
                              ".Q c #b7caee",
                              ".O c #b8ccee",
                              ".u c #b9ccf0",
                              ".G c #c4d4f1",
                              ".J c #cbd9f3",
                              ".I c #cddaf4",
                              ".B c #d4e0f4",
                              ".D c #dce6f6",
                              ".C c #e1e9f8",
                              ".r c #e4ebf8",
                              ".t c #ebf0fa",
                              ".s c #f2f5fb",
                              "QtQtQtQtQt.#.a.b.c.d.eQtQtQtQtQt",
                              "QtQtQt.#.f.g.h.i.j.k.l.m.nQtQtQt",
                              "QtQt.#.o.p.q.r.s.s.t.u.v.w.xQtQt",
                              "Qt.y.z.A.B.C.C.C.C.C.C.D.E.w.FQt",
                              "Qt.a.p.G.H.H.H.I.I.H.H.H.J.K.L.M",
                              ".c.l.N.O.P.P.P.O.O.P.P.P.Q.R.o.S",
                              ".c.T.U.V.P.P.P.V.V.P.P.P.W.X.Y.Z",
                              ".c.0.1.H.P.P.P.2.2.P.P.P.3.X.4.5",
                              ".c.6.7.8.P.P.P.3.9.P.P.P.3.3#.##",
                              ".d#a#b#b.P.P.P.3.3.P.P.P#c#d.k#e",
                              ".e.l#f#g.P.P.P.3.3.P.P.P#h#f.o#i",
                              "Qt.m#j#k.P.P.P#k#k.P.P.P#k#j#l.M",
                              "Qt.n#m#n#o#o#o#o#o#o#o#o#n#p#q.M",
                              "QtQt#r#s#t#u#v#v#v#v#u#t#w#x.MQt",
                              "QtQtQt.F.L#y#z#A#A#z#B#l#q.MQtQt",
                              "QtQtQtQt.M.S.Z.5###e#i.M.MQtQtQt"])

refresh = QPixmapWrapper(["16 16 7 1",
                          "  c #000000",
                          ". c #00FF00",
                          "X c None",
                          "o c #00C000",
                          "O c #008000",
                          "+ c #004000",
                          "@ c None",
                          "XXXXXX     XXXXX",
                          "XXXX  .oo  + XXX",
                          "XXX .ooO XX   XX",
                          "XXX ooO XXXXX XX",
                          "XX .oO XXXXXXX X",
                          "XX ooO XXXXXXXXX",
                          " ..ooooO XX  XXX",
                          "X .oooO XX    XX",
                          "XX .oO XX      X",
                          "XXX O XX        ",
                          "XXXX XXXXX    XX",
                          "X XXXXXXXX    XX",
                          "XX XXXXXX    XXX",
                          "XX   XX      XXX",
                          "XXX         XXXX",
                          "XXXXX     XXXXXX"])

pin_up = QPixmapWrapper(["16 16 5 1",
                         "  c None",
                         ". c #A0A0A0",
                         "X c #707070",
                         "o c #FFFFFF",
                         "O c None",
                         "                ",
                         "                ",
                         "                ",
                         "      XX     X  ",
                         "      XoX   XX  ",
                         "      XooXXXoX  ",
                         "ooooooX.o.o.oX  ",
                         "......X.o.o.oX  ",
                         "XXXXXXX..X.X.X  ",
                         "      X.XXXXXX  ",
                         "      XXX   XX  ",
                         "      XX     X  ",
                         "                ",
                         "                ",
                         "                ",
                         "                "])

pin_down = QPixmapWrapper(["16 16 5 1",
                           "  c None",
                           ". c #A0A0A0",
                           "X c #707070",
                           "o c #FFFFFF",
                           "O c None",
                           "                ",
                           "                ",
                           "        XXXX    ",
                           "      XXooooX   ",
                           "    XXXooooooX  ",
                           "    XoXooo...X  ",
                           "   XooXoo...XX  ",
                           "   XooXXo..XXX  ",
                           "   XooooXXXXX   ",
                           "   Xooo....XX   ",
                           "    Xo....XXX   ",
                           "    X....XXX    ",
                           "     XXXXXX     ",
                           "                ",
                           "                ",
                           "                "])

matrix = QPixmapWrapper(["16 16 4 1",
                         "  c None",
                         ". c None",
                         "X c #800080",
                         "o c None",
                         "... ... ....... ",
                         ".XXXX XXXX.XXXX ",
                         ".X. X.X .X.X..X ",
                         " X  X X  X X  X ",
                         ".XXXX.XXXX.XXXX ",
                         ". . . . . . . . ",
                         ".XXXX.XXXX.XXXX ",
                         " X  X X  X.X  X ",
                         ".X..X.X..X.X..X ",
                         ".XXXX XXXX.XXXX.",
                         "... ... ... ... ",
                         " XXXX XXXX XXXX ",
                         ".X. X.X .X.X..X ",
                         ".X. X.X .X.X. X ",
                         ".XXXX.XXXX.XXXX ",
                         "                "
                         ])

view_right = QPixmapWrapper(["16 16 48 1",
                             "  c None",
                             ". c #ADADAD",
                             "X c #A5A5A5",
                             "o c #A1A1A1",
                             "O c #C1C10A",
                             "+ c #FFFF85",
                             "@ c #494949",
                             "# c #FAFAFA",
                             "$ c #F6F6F6",
                             "% c #F2F2F2",
                             "& c #F0F0F0",
                             "* c #EEEEEE",
                             "= c #ECECEC",
                             "- c #EAEAEA",
                             "; c #E8E8E8",
                             ": c #E6E6E6",
                             "> c #E4E4E4",
                             ", c #E2E2E2",
                             "< c #E0E0E0",
                             "1 c #DADADA",
                             "2 c #D6D6D6",
                             "3 c #CCCCCC",
                             "4 c #C8C8C8",
                             "5 c #C6C6C6",
                             "6 c #C0C000",
                             "7 c #AEAEAE",
                             "8 c #ACACAC",
                             "9 c #A0A0A0",
                             "0 c #585858",
                             "q c #FFFFFF",
                             "w c #FFFF00",
                             "e c #FFFFC0",
                             "r c #F9F9F9",
                             "t c #F5F5F5",
                             "y c #F3F3F3",
                             "u c #F1F1F1",
                             "i c #EDEDED",
                             "p c #E9E9E9",
                             "a c #E5E5E5",
                             "s c #303030",
                             "d c #E1E1E1",
                             "f c #DFDFDF",
                             "g c #D9D9D9",
                             "h c #D3D3D3",
                             "j c #CFCFCF",
                             "k c #C9C9C9",
                             "l c #C7C7C7",
                             "z c None",
                             "                ",
                             " 0000000000000@ ",
                             " 02gg1q0ohj3k5s ",
                             " 0gyt$q08=;><5s ",
                             " 0gtr#q0.ipadls ",
                             " 01$#qq07*-:,4s ",
                             " 0gtr#q0.ipadls ",
                             " 0gyt$q08= +<5s ",
                             " 02&u%q +- wf + ",
                             " 0h=i*q0 w6w6ws ",
                             " 0j;p-q0XOe+e6s ",
                             " 03>a: +ww+q+ww+",
                             " 0k<d,q096e+e6s ",
                             " 055l4q0 w6w6ws ",
                             " @sssss +s ws + ",
                             "           +    "])

view_split = QPixmapWrapper(["16 16 40 1",
                             "  c None",
                             ". c #A1A1A1",
                             "X c #979797",
                             "o c #494949",
                             "O c #FAFAFA",
                             "+ c #F6F6F6",
                             "@ c #EEEEEE",
                             "# c #ECECEC",
                             "$ c #EAEAEA",
                             "% c #E8E8E8",
                             "& c #E6E6E6",
                             "* c #E4E4E4",
                             "= c #E2E2E2",
                             "- c #E0E0E0",
                             "; c #DEDEDE",
                             ": c #DCDCDC",
                             "> c #DADADA",
                             ", c #D6D6D6",
                             "< c #CCCCCC",
                             "1 c #C8C8C8",
                             "2 c #C6C6C6",
                             "3 c #ACACAC",
                             "4 c #585858",
                             "5 c #FFFFFF",
                             "6 c #F9F9F9",
                             "7 c #F5F5F5",
                             "8 c #F3F3F3",
                             "9 c #EDEDED",
                             "0 c #E9E9E9",
                             "q c #E7E7E7",
                             "w c #E5E5E5",
                             "e c #303030",
                             "r c #E1E1E1",
                             "t c #DDDDDD",
                             "y c #D9D9D9",
                             "u c #D3D3D3",
                             "i c #CFCFCF",
                             "p c #C9C9C9",
                             "a c #C7C7C7",
                             "s c None",
                             "                ",
                             " 4444444444444o ",
                             " 4,yy>543ui<p2e ",
                             " 4y87+54.#%*-2e ",
                             " 4y76O54.90wrae ",
                             " 4>+O554.@$&=1e ",
                             " 4555554.55555e ",
                             " 4444444444444e ",
                             " 43....4.....3e ",
                             " 4u#9@54.q*rt2e ",
                             " 4i%0$54.*r;:2e ",
                             " 4<*w&54.r;::2e ",
                             " 4p-r=54.t:::2e ",
                             " 422a154X22222e ",
                             " oeeeeeeeeeeeee ",
                             "                "])

remove = QPixmapWrapper(["16 16 15 1",
                         "  c #000000",
                         ". c #E4E4E4",
                         "X c #DEDEDE",
                         "o c #D4D4D4",
                         "O c #CECECE",
                         "+ c #BEBEBE",
                         "@ c #6E6E6E",
                         "# c #FFFFFF",
                         "$ c #E9E9E9",
                         "% c #D9D9D9",
                         "& c #C9C9C9",
                         "* c #C3C3C3",
                         "= c #B9B9B9",
                         "- c #B7B7B7",
                         "; c None",
                         "---------------#",
                         "-@@@@@@@@@@@@@@#",
                         "-@###########-@#",
                         "-@#$$$$$$$$$$-@#",
                         "-@#.. .... ..-@#",
                         "-@#X   XX   X-@#",
                         "-@#%%      %%-@#",
                         "-@#ooo    ooo-@#",
                         "-@#OOO    OOO-@#",
                         "-@#&&      &&-@#",
                         "-@#*   **   *-@#",
                         "-@#++ ++++ ++-@#",
                         "-@#==========-@#",
                         "-@------------@#",
                         "-@@@@@@@@@@@@@@#",
                         "################"])

eventnew = QPixmapWrapper(["16 16 7 1",
                           "  c None",
                           ". c #87852B",
                           "X c #918F2E",
                           "o c #FFFFFF",
                           "O c #FFFF00",
                           "+ c #FFFFC0",
                           "@ c None",
                           "                ",
                           "                ",
                           "        O       ",
                           "   O    O.   O  ",
                           "    O. +O. OO   ",
                           "    OO.XOX+O.   ",
                           "     +OOOOO.    ",
                           "   ..XOo+oOX+   ",
                           "  OOOOO+o+OOOOO ",
                           "    +XOo+oOX..  ",
                           "     .OOOOO+    ",
                           "    .O+XOX.OO   ",
                           "    OO .O+ .O   ",
                           "   O   .O    O  ",
                           "        O       ",
                           "                "])

view_tree = QPixmapWrapper(["16 16 5 1",
                            "  c #000000",
                            ". c None",
                            "X c #303030",
                            "o c #FFDCA8",
                            "O c None",
                            " ...............",
                            " ...............",
                            " ...XXX ........",
                            " ...X.. ........",
                            "   .X.o ..  .   ",
                            " ...    ........",
                            " ...............",
                            " ...............",
                            " ...............",
                            " ...XXX ........",
                            " ...X.. ........",
                            "   .X.o ..  .   ",
                            "....    ........",
                            "................",
                            "................",
                            "................"])

precplus = QPixmapWrapper(["16 16 3 1",
                           "# c #000000",
                           ". c None",
                           "a c #ffffff",
                           "........##...##.",
                           "..#....##.#.##.#",
                           ".##....##.#.##.#",
                           "#####..##.#.##.#",
                           ".##....##.#.##.#",
                           "..#....##.#.##.#",
                           "....##..##...##.",
                           "....##..........",
                           ".....#..########",
                           "........#aaaaaaa",
                           "........#aaa#aaa",
                           "........#aaa#aaa",
                           "........#a#####a",
                           "........#aaa#aaa",
                           "........#aaa#aaa",
                           "........#aaaaaaa"])

precminus = QPixmapWrapper(["16 16 3 1",
                            "# c #000000",
                            ". c None",
                            "a c #ffffff",
                            ".........##.....",
                            "...#....##.#....",
                            "...##...##.#....",
                            "######..##.#....",
                            "...##...##.#....",
                            "...#....##.#....",
                            ".....##..##.....",
                            ".....##.........",
                            "......#.########",
                            "........#aaaaaaa",
                            "........#aaaaaaa",
                            "........#aaaaaaa",
                            "........#a#####a",
                            "........#aaaaaaa",
                            "........#aaaaaaa",
                            "........#aaaaaaa"])

bars3d = QPixmapWrapper(["16 16 6 1",
                         "# c #000000",
                         "b c #00d300",
                         "a c #00ff00",
                         "d c #9595d6",
                         "c c #c0c0ff",
                         ". c None",
                         "................",
                         ".......###......",
                         "......#aa#.###..",
                         ".....#aa###aa#..",
                         ".....###b#aa##..",
                         ".....#a#####b#..",
                         ".....##cc#a#b###",
                         ".....#cc##a##aa#",
                         "...#####d#a#aa##",
                         "..#cc#c#d#a###b#",
                         ".#cc##c#d##cc#b#",
                         ".###d#c#d#cc##b#",
                         ".#c#d#c#d###d##.",
                         ".#c#d#c#d#c#d#..",
                         ".#c#d#c#d#c##...",
                         ".###########...."])

areas3d = QPixmapWrapper(["16 16 6 1",
                          "# c #000000",
                          "a c #00d300",
                          "b c #00ff00",
                          "c c #9595d6",
                          "d c #c0c0ff",
                          ". c None",
                          "......#.........",
                          ".....#a#........",
                          "....#aaa#.......",
                          "...#b#aaa###....",
                          "...#bb#aaaaa##..",
                          "..#c#bb###aaaa#.",
                          ".#ccc#bbbb##aa#.",
                          "#d#ccc#bbbbb#a#.",
                          "#dd#ccc#bbbb#aa#",
                          "#ddd#ccc###b#aaa",
                          "#dddd#ccccc###aa",
                          "#ddddd###cccc##a",
                          "#dddddddd##ccc##",
                          "#dddddddddd#ccc#",
                          "#ddddddddddd#c#.",
                          "##############.."])

treeviewoblique = QPixmapWrapper(["16 16 44 1",
                                  "h c #000000",
                                  ". c #FAF7F2",
                                  "X c #E8D4BC",
                                  "o c #FDFBF8",
                                  "O c #F0E3D4",
                                  "+ c #F7F0E8",
                                  "@ c #EBDAC5",
                                  "# c #F3E9DD",
                                  "$ c #FAF6F1",
                                  "% c #EAD7C0",
                                  "& c #FDFCFA",
                                  "* c #EDDDC9",
                                  "= c #F3E8DC",
                                  "- c #F9F5EF",
                                  "; c #E7D2B9",
                                  ": c #DCDCDC",
                                  "> c #F6EEE5",
                                  ", c #EAD8C2",
                                  "< c #E6CFB4",
                                  "1 c #F5EBE0",
                                  "2 c #E9D5BD",
                                  "3 c #FCFAF7",
                                  "4 c #A0A0A0",
                                  "5 c #F2E6D9",
                                  "6 c #F8F3EC",
                                  "7 c #808080",
                                  "8 c #EFE1CF",
                                  "9 c #F5ECE2",
                                  "0 c #E5CDB1",
                                  "q c #FEFDFB",
                                  "w c #585858",
                                  "e c #F1E5D7",
                                  "r c #F8F2EB",
                                  "t c #FFFFFF",
                                  "y c #EFE0CE",
                                  "u c #FBF8F4",
                                  "i c #303030",
                                  "p c #EFE2D1",
                                  "a c #F1E4D6",
                                  "s c #F7F1E9",
                                  "d c #FEFEFD",
                                  "f c #C3C3C3",
                                  "g c #EEDFCC",
                                  "  c None",
                                  "         ii     ",
                                  "       iw.ri    ",
                                  "     iwt3-+w    ",
                                  "   iwttqu6>#i   ",
                                  " iw:tttous7hw   ",
                                  "itthtowsshfe8i  ",
                                  "itt7:hhw1==Ogw  ",
                                  " wtthhhf+15p*,i ",
                                  " itt7:u6>#yy7 w ",
                                  "  wtth$ow#*h4X<i",
                                  "  itd7:hhwg%%;0i",
                                  "   w&.hhhfg@2wi ",
                                  "   i3-+15p*wi   ",
                                  "    w6>#awi     ",
                                  "    is9wi       ",
                                  "     ii         "])

reload_slick = QPixmapWrapper(["16 16 143 2",
                               "   c None",
                               " . c #CDDAF4",
                               " X c #3A64B3",
                               " o c #E4ECFA",
                               " O c #4887FC",
                               " + c #9CB6E8",
                               " @ c #3B66B7",
                               " # c #8AACEA",
                               " $ c #BAD0F8",
                               " % c #3269CF",
                               " & c #B4CAF2",
                               " * c #6696EE",
                               " = c #5183E0",
                               " - c #6494EC",
                               " ; c #E9ECF1",
                               " : c #C4D4F1",
                               " > c #6C94E0",
                               " , c #3269D2",
                               " < c #D6E0F2",
                               " 1 c #3B80FF",
                               " 2 c #90B1EC",
                               " 3 c #3B65B3",
                               " 4 c #C6D3EC",
                               " 5 c #FAFBFE",
                               " 6 c #CAD4E6",
                               " 7 c #3472E4",
                               " 8 c #6C9BF0",
                               " 9 c #656565",
                               " 0 c #5E88D5",
                               " q c #636363",
                               " w c #5F5F5F",
                               " e c #C2D4F5",
                               " r c #4686FC",
                               " t c #B1CAF8",
                               " y c #699CFA",
                               " u c #A3B7DD",
                               " i c #4A7FE2",
                               " p c #A3BCEA",
                               " a c #3569CA",
                               " s c #4E6286",
                               " d c #B8CCEE",
                               " f c #3C81FF",
                               " g c #3765B8",
                               " h c #3663B7",
                               " j c #81A5E6",
                               " k c #3471E3",
                               " l c #3C66B6",
                               " z c #3B66B5",
                               " x c #A8C2F5",
                               " c c #92AFE6",
                               " v c #3E70CF",
                               " b c #326CD7",
                               " n c #CBD5E9",
                               " m c #F6F8FC",
                               " M c #486190",
                               " N c #7EA3E6",
                               " B c #5789E4",
                               " V c #83ACF8",
                               " C c #EEF0F4",
                               " Z c #9ABEFE",
                               " A c #3269D0",
                               " S c #DCE6F6",
                               " D c #4D648E",
                               " F c #D8E2F2",
                               " G c #8EADE8",
                               " H c #4A80E4",
                               " J c #356ACC",
                               " K c #D9E4F6",
                               " L c #3368CA",
                               " P c #719FF3",
                               " I c #5A8AE3",
                               " U c #B5C9ED",
                               " Y c #3D64AC",
                               " T c #4980E6",
                               " R c #3E61A3",
                               " E c #6E95DC",
                               " W c #3268CC",
                               " Q c #4888FC",
                               " ! c #3467C4",
                               " ~ c #D4E0F4",
                               " ^ c #3472E5",
                               " / c #EBF0FA",
                               " ( c #7AA6F8",
                               " ) c #DEE7F7",
                               " _ c #A4BCE9",
                               " ` c #3776EB",
                               " ' c #465D86",
                               " ] c #8CACE5",
                               " [ c #FDFEFE",
                               " { c #A5BEED",
                               " } c #326AD2",
                               " | c #3A5FA5",
                               ".  c #A1BAE9",
                               ".. c #B9CCF0",
                               ".X c #A3B9E1",
                               ".o c #B7CAEE",
                               ".O c #3765B9",
                               ".+ c #CEDCF4",
                               ".@ c #C9D3E5",
                               ".# c #5B88DC",
                               ".$ c #6E99E8",
                               ".% c #9DC0FF",
                               ".& c #5891FA",
                               ".* c #3E62A5",
                               ".= c #3269CE",
                               ".- c #585858",
                               ".; c #4F6184",
                               ".: c #7CA1E5",
                               ".> c #525252",
                               "., c #3B63AF",
                               ".< c #FFFFFF",
                               ".1 c #3269D1",
                               ".2 c #84A6E6",
                               ".3 c #799FE5",
                               ".4 c #EDF0F7",
                               ".5 c #B2C5E8",
                               ".6 c #E4EBF8",
                               ".7 c #326BD4",
                               ".8 c #E5E8EF",
                               ".9 c #346ACC",
                               ".0 c #3268CA",
                               ".q c #658EDB",
                               ".w c #3572E4",
                               ".e c #D4DEF2",
                               ".r c #CBD9F3",
                               ".t c #3E71CF",
                               ".y c #E1E9F8",
                               ".u c #F7F9FD",
                               ".i c #3268CD",
                               ".p c #4083FF",
                               ".a c #3766BE",
                               ".s c #9AB5E7",
                               ".d c #3873E0",
                               ".f c #97B3E4",
                               ".g c #447CE5",
                               ".h c #3676EB",
                               ".j c #F2F5FB",
                               ".k c #6C98EB",
                               ".l c #8DACE7",
                               ".z c #326AD3",
                               ".x c #3469CB",
                               ".c c #4F8AF6",
                               ".v c None",
                               "          .1 %.i W.0 h 9        ",
                               "      .1 , b ^ =.# 7.7 L g.-    ",
                               "    .1.z T x.6.j.j /.. B.= z.>  ",
                               "   A } - ~.y F n < ).y S.3.= R.>",
                               "   % T : . 6.8.< C.e . ..r I ! 9",
                               " W.7 2 d.@.<.<.<.< K d d.o.f.z M",
                               " W.d. .X.< [.o.s _ + p U ( 1 k X",
                               " W i ].5.<.+ c c u 4.<.<.% 1.h @",
                               " W.g.$ {.< t.c #.<.<.<.< Z f ` l",
                               ".0.w r 8.< m y.&.4.<.<.< V.p 7 Y",
                               " h.7 O Q $.<.< ;.<.<.<.< P O.z.;",
                               " 9 L H * * &.u.< 5 e o.u.k H.a 9",
                               "  .O.9 > N N N.2 N N.: j >.x s 9",
                               "   9 3 J.q.l G G G G.l.q a D 9 9",
                               "     9.* !.t 0 E E 0 v.a s 9 9  ",
                               "      .> q ' |., l Y.; w.>.>    "])

publish = QPixmapWrapper(["16 16 13 1",
                          "j c #000000",
                          "# c #000000",
                          "k c #303030",
                          "a c #404000",
                          "c c #585858",
                          "e c #800000",
                          "h c #808080",
                          "i c #a0a0a0",
                          "f c #c00000",
                          "g c #c3c3c3",
                          "d c #dcdcdc",
                          ". c None",
                          "b c #ffffff",
                          ".....##.........",
                          ".....abc#.......",
                          "....#dbbdc#.....",
                          "....cbbebbdc#...",
                          "...#dbbbfebbdc#.",
                          "..#db##bbbfebbg#",
                          ".#db#bb##bbbbdh#",
                          ".cbbb##bb##bbai#",
                          "#bb##bb##bbbdc#h",
                          "#dbbb##bb#bbhic#",
                          "#hgdbbb##bbdcc#j",
                          ".#cigdbbbbbhhhc#",
                          "...#cigdbbdcca#.",
                          ".....#cigdacc#..",
                          ".......#chk##...",
                          ".........##....."])

publish_active = QPixmapWrapper(["16 16 13 1",
                                 "j c #000000",
                                 "# c #000000",
                                 "k c #303030",
                                 "a c #404000",
                                 "c c #585858",
                                 "e c #800000",
                                 "h c #808080",
                                 "i c #a0a0a0",
                                 "f c #c00000",
                                 "g c #c3c3c3",
                                 "d c #dcdc00",
                                 ". c None",
                                 "b c #ffff00",
                                 ".....##.........",
                                 ".....abc#.......",
                                 "....#dbbdc#.....",
                                 "....cbbebbdc#...",
                                 "...#dbbbfebbdc#.",
                                 "..#db##bbbfebbg#",
                                 ".#db#bb##bbbbdh#",
                                 ".cbbb##bb##bbai#",
                                 "#bb##bb##bbbdc#h",
                                 "#dbbb##bb#bbhic#",
                                 "#hgdbbb##bbdcc#j",
                                 ".#cigdbbbbbhhhc#",
                                 "...#cigdbbdcca#.",
                                 ".....#cigdacc#..",
                                 ".......#chk##...",
                                 ".........##....."])

node_any = QPixmapWrapper(["16 16 5 1",
                           "  c None",
                           ". c #808080",
                           "X c #FFFF00",
                           "o c #0000FF",
                           "O c None",
                           "                ",
                           " o         o    ",
                           "  o       o .   ",
                           "   o     o .    ",
                           "    o   o .     ",
                           "   ooooooo      ",
                           "   oXXXXXo.     ",
                           "   oXXXXXo.     ",
                           "   oXXXXXo.     ",
                           "   ooooooo.     ",
                           "    ..o....     ",
                           "      o.        ",
                           "      o.        ",
                           "      o.        ",
                           "       .        ",
                           "                "])

node_command = QPixmapWrapper(["16 16 7 1",
                               "  c None",
                               ". c #00FF00",
                               "X c #000040",
                               "o c #808080",
                               "O c #A8DCFF",
                               "+ c #0000FF",
                               "@ c #000000",
                               "                ",
                               " +         +    ",
                               "  +       + o   ",
                               "   +     + o    ",
                               "    +   + o     ",
                               "   +++++++      ",
                               "   +OOOOO+o     ",
                               "   +OOOOO+....  ",
                               "   +OOOOO..@@.. ",
                               "   ++++++.@@.@.o",
                               "    oo+oo.@@...o",
                               "      +o .@@...o",
                               "      +o .@@.@.o",
                               "      +o ..@@..o",
                               "      +o  ....oo",
                               "       o   oooo "])

node_eval = QPixmapWrapper(["16 16 7 1",
                            "  c None",
                            ". c #00FF00",
                            "X c #000040",
                            "o c #808080",
                            "O c #A8DCFF",
                            "+ c #0000FF",
                            "@ c None",
                            "                ",
                            " +         +    ",
                            "  +       + o   ",
                            "   +     + o    ",
                            "    + XXXXX     ",
                            "   +++X...Xo    ",
                            "   +OOX...Xo    ",
                            "   +OXX...XX    ",
                            "   +OX.....Xo   ",
                            "   +++X...Xoo   ",
                            "    oo+X.Xoo    ",
                            "      +oXoo     ",
                            "      +o o      ",
                            "      +o        ",
                            "       o        ",
                            "                "])

node_idle = QPixmapWrapper(["16 16 4 1",
                            "  c None",
                            ". c #000040",
                            "X c #808080",
                            "o c None",
                            "                ",
                            " .         .    ",
                            "  .       . X   ",
                            "   .     . X    ",
                            "    .   . X     ",
                            "   .......      ",
                            "   .XXXXX.X     ",
                            "   .X    .X     ",
                            "   .X    .X     ",
                            "   .......X     ",
                            "    XX.XXXX     ",
                            "      .X        ",
                            "      .X        ",
                            "      .X        ",
                            "       X        ",
                            "                "])

node_poll = QPixmapWrapper(["16 16 7 1",
                            "  c None",
                            ". c #00FF00",
                            "X c #000040",
                            "o c #808080",
                            "O c #A8DCFF",
                            "+ c #0000FF",
                            "@ c None",
                            "                ",
                            "XXXXX   XXXXX   ",
                            "X..Xoo   X..Xo  ",
                            "X...X   X...Xo  ",
                            "XX...X X...XXo  ",
                            "XoX.XX+XX.XoXo  ",
                            " o XXOOOXXo  o  ",
                            "   +OOOOO+o     ",
                            "   +OOOOO+o     ",
                            "   +++++++o     ",
                            "    oo+oooo     ",
                            "      +o        ",
                            "      +o        ",
                            "      +o        ",
                            "       o        ",
                            "                "])

node_request = QPixmapWrapper(["16 16 7 1",
                               "  c None",
                               ". c #00FF00",
                               "X c #000040",
                               "o c #808080",
                               "O c #A8DCFF",
                               "+ c #0000FF",
                               "@ c None",
                               "                ",
                               " +         +    ",
                               "  +       + o   ",
                               "   +     + o    ",
                               "    +   + o     ",
                               "   +++++++      ",
                               "   +OOOOO+o     ",
                               "   +OOXOO+o     ",
                               "   +OX.XO+o     ",
                               "   +X...X+o     ",
                               "   X.....Xo     ",
                               "   XX...XXo     ",
                               "    X...Xoo     ",
                               "    X...Xo      ",
                               "    XXXXXo      ",
                               "     ooooo      "])

##breakpoint = QPixmapWrapper(["16 16 69 1",
##                             "  c None",
##                             ". c #D77570",
##                             "X c #DF7F7B",
##                             "o c #D27E78",
##                             "O c #911414",
##                             "+ c #6A3936",
##                             "@ c #7D1414",
##                             "# c #8E1414",
##                             "$ c #411414",
##                             "% c #7A1414",
##                             "& c #B51616",
##                             "* c #CA7169",
##                             "= c #541A16",
##                             "- c #C2271F",
##                             "; c #D9A8A2",
##                             ": c #642722",
##                             "> c #3B1414",
##                             ", c #B81515",
##                             "< c #5C1F1A",
##                             "1 c #EABEB9",
##                             "2 c #C56460",
##                             "3 c #961414",
##                             "4 c #D97570",
##                             "5 c #F1C8C6",
##                             "6 c #F3D7D5",
##                             "7 c #A71414",
##                             "8 c #DB7975",
##                             "9 c #D98680",
##                             "0 c #BE2727",
##                             "q c #6B1414",
##                             "w c #1E1414",
##                             "e c #901414",
##                             "r c #612621",
##                             "t c #D46C67",
##                             "y c #B51414",
##                             "u c #2F1414",
##                             "i c #F6ECEB",
##                             "p c #DA7D7A",
##                             "a c #EABFBB",
##                             "s c #AE7269",
##                             "d c #C56058",
##                             "f c #CB6661",
##                             "g c #BB4744",
##                             "h c #761414",
##                             "j c #F3D8D7",
##                             "k c #5C1D18",
##                             "l c #291414",
##                             "z c #F6E7E7",
##                             "x c #E7B8B4",
##                             "c c #F3DADA",
##                             "v c #DD7772",
##                             "b c #DA807C",
##                             "n c #341414",
##                             "m c #A61414",
##                             "M c #A15C54",
##                             "N c #B71414",
##                             "B c #6A1414",
##                             "V c #FFFFFF",
##                             "C c #E6B9B5",
##                             "Z c #DF8580",
##                             "A c #B41414",
##                             "S c #671414",
##                             "D c #1A1414",
##                             "F c #D7A29C",
##                             "G c #C0231F",
##                             "H c #672F2A",
##                             "J c #CA655D",
##                             "K c #703B36",
##                             "L c None",
##                             "    +HKr<k=     ",
##                             "  V+zc61CFM>    ",
##                             " V+V8b9o*2gen   ",
##                             " +V4XvZ----7#l  ",
##                             "+ip&&&&&&&&&&SD ",
##                             "+j.VVVVVVVVVV%w ",
##                             "+5tV--V&V-V&V@w ",
##                             ":afVV-VAV&VVV@w ",
##                             "<xJ&V-VAV-V,m@w ",
##                             "k;dVV-VAOVV-m%w ",
##                             "=s-&&-&A&&&m3BD ",
##                             " $e7G-0-NAm3hu  ",
##                             "  n#7y&ymmOhu   ",
##                             "   nq%@@@%Bu    ",
##                             "    DwwwwwD     ",
##                             "                "])
##
##breakpoint_ss = QPixmapWrapper(["16 16 64 1",
##                                "  c None",
##                                ". c #D77570",
##                                "X c #DF7F7B",
##                                "o c #911414",
##                                "O c #6A3936",
##                                "+ c #7D1414",
##                                "@ c #8E1414",
##                                "# c #411414",
##                                "$ c #7A1414",
##                                "% c #B51616",
##                                "& c #541A16",
##                                "* c #C2271F",
##                                "= c #D9A8A2",
##                                "- c #642722",
##                                "; c #3B1414",
##                                ": c #5C1F1A",
##                                "> c #EABEB9",
##                                ", c #C56460",
##                                "< c #961414",
##                                "1 c #D97570",
##                                "2 c #F1C8C6",
##                                "3 c #F3D7D5",
##                                "4 c #A71414",
##                                "5 c #DB7975",
##                                "6 c #D98680",
##                                "7 c #BE2727",
##                                "8 c #6B1414",
##                                "9 c #1E1414",
##                                "0 c #901414",
##                                "q c #612621",
##                                "w c #D46C67",
##                                "e c #B51414",
##                                "r c #2F1414",
##                                "t c #F6ECEB",
##                                "y c #DA7D7A",
##                                "u c #EABFBB",
##                                "i c #AE7269",
##                                "p c #C56058",
##                                "a c #CB6661",
##                                "s c #BB4744",
##                                "d c #761414",
##                                "f c #F3D8D7",
##                                "g c #5C1D18",
##                                "h c #291414",
##                                "j c #F6E7E7",
##                                "k c #E7B8B4",
##                                "l c #F3DADA",
##                                "z c #DA807C",
##                                "x c #341414",
##                                "c c #A61414",
##                                "v c #A15C54",
##                                "b c #B71414",
##                                "n c #6A1414",
##                                "m c #FFFFFF",
##                                "M c #E6B9B5",
##                                "N c #B41414",
##                                "B c #671414",
##                                "V c #1A1414",
##                                "C c #D7A29C",
##                                "Z c #C0231F",
##                                "A c #672F2A",
##                                "S c #CA655D",
##                                "D c #703B36",
##                                "F c None",
##                                "    OADq:g&     ",
##                                "  mOjl3>MCv;    ",
##                                " mOm5z6pp,s0x   ",
##                                " Om1Xppp***4@h  ",
##                                "Otyppp%m%%%%%BV ",
##                                "Of.ppemmeeeee$9 ",
##                                "O2wp*eemeee%e+9 ",
##                                "-uape*emeeeee+9 ",
##                                ":kS%e*emeZeec+9 ",
##                                "g=peeemmmZe*c$9 ",
##                                "&ip%%*%N%%%c<nV ",
##                                " #04Z*7*bNc<dr  ",
##                                "  x@4e%eccodr   ",
##                                "   x8$+++$nr    ",
##                                "    V99999V     ",
##                                "                "])
##
##breakpoint_delete = QPixmapWrapper(["16 16 68 1",
##                                    "  c None",
##                                    ". c #D77570",
##                                    "X c #DF7F7B",
##                                    "o c #D27E78",
##                                    "O c #911414",
##                                    "+ c #6A3936",
##                                    "@ c #7D1414",
##                                    "# c #8E1414",
##                                    "$ c #411414",
##                                    "% c #7A1414",
##                                    "& c #B51616",
##                                    "* c #CA7169",
##                                    "= c #541A16",
##                                    "- c #C2271F",
##                                    "; c #D9A8A2",
##                                    ": c #642722",
##                                    "> c #3B1414",
##                                    ", c #B81515",
##                                    "< c #5C1F1A",
##                                    "1 c #EABEB9",
##                                    "2 c #C56460",
##                                    "3 c #961414",
##                                    "4 c #F1C8C6",
##                                    "5 c #F3D7D5",
##                                    "6 c #A71414",
##                                    "7 c #DB7975",
##                                    "8 c #D98680",
##                                    "9 c #BE2727",
##                                    "0 c #6B1414",
##                                    "q c #1E1414",
##                                    "w c #901414",
##                                    "e c #612621",
##                                    "r c #D46C67",
##                                    "t c #B51414",
##                                    "y c #2F1414",
##                                    "u c #F6ECEB",
##                                    "i c #DA7D7A",
##                                    "p c #EABFBB",
##                                    "a c #AE7269",
##                                    "s c #C56058",
##                                    "d c #CB6661",
##                                    "f c #BB4744",
##                                    "g c #761414",
##                                    "h c #F3D8D7",
##                                    "j c #5C1D18",
##                                    "k c #291414",
##                                    "l c #F6E7E7",
##                                    "z c #E7B8B4",
##                                    "x c #F3DADA",
##                                    "c c #DD7772",
##                                    "v c #DA807C",
##                                    "b c #341414",
##                                    "n c #A61414",
##                                    "m c #A15C54",
##                                    "M c #B71414",
##                                    "N c #6A1414",
##                                    "B c #FFFFFF",
##                                    "V c #E6B9B5",
##                                    "C c #DF8580",
##                                    "Z c #B41414",
##                                    "A c #671414",
##                                    "S c #1A1414",
##                                    "D c #D7A29C",
##                                    "F c #C0231F",
##                                    "G c #672F2A",
##                                    "H c #CA655D",
##                                    "J c #703B36",
##                                    "K c #000000",
##                                    "K   +GJe<j=   K ",
##                                    " KB+lx51VDm> K  ",
##                                    " BKB7v8o*2fwK   ",
##                                    " +BKXcC----K#k  ",
##                                    "+ui&K&&&&&K&&AS ",
##                                    "+h.BBKBBBKBBB%q ",
##                                    "+4rB--K&K-B&B@q ",
##                                    ":pdBB-BKB&BBB@q ",
##                                    "<zH&B-KZK-B,n@q ",
##                                    "j;sBBKBZOKB-n%q ",
##                                    "=a-&K-&Z&&Kn3NS ",
##                                    " $wKF-9-MZnKgy  ",
##                                    "  K#6t&tnnOgK   ",
##                                    " K b0%@@@%Ny K  ",
##                                    "K   SqqqqqS   K ",
##                                    "                "])

breakpoint = QPixmapWrapper(["16 16 43 1",
                             ". c None",
                             "c c #380202",
                             "K c #410202",
                             "# c #630000",
                             "o c #690200",
                             "O c #810000",
                             "N c #890400",
                             "M c #900700",
                             "L c #980a00",
                             "G c #9e6864",
                             "H c #9f0d00",
                             "E c #a71000",
                             "D c #ac1404",
                             "B c #ae1300",
                             "x c #b41600",
                             "z c #b51700",
                             "t c #bb1900",
                             "C c #bca5a5",
                             "v c #bd1a00",
                             "A c #bd2414",
                             "k c #bfa9a9",
                             "r c #c41c00",
                             "p c #c41d00",
                             "y c #c7b4b4",
                             "q c #c7b5b5",
                             "n c #c82004",
                             "u c #cab8b8",
                             "m c #cc2000",
                             "w c #cc3626",
                             "h c #d22300",
                             "I c #d2c3c3",
                             "e c #d32300",
                             "i c #d83922",
                             "b c #da2700",
                             "s c #dd4a3c",
                             "d c #e0442e",
                             "F c #e2d7d7",
                             "f c #e4dddd",
                             "g c #e8e2e2",
                             "a c #ea5a4d",
                             "J c #f4f0f0",
                             "l c #f5f3f3",
                             "j c #ffffff",
                             ".....######.....",
                             "...##aaaaaa##...",
                             "..#aabccbbbad#..",
                             ".#aeccfgcceehi#.",
                             ".#acjkllkjcmmno.",
                             "#apcjqllqjcpprp#",
                             "#stcjullujcvcvv#",
                             "#wxcjyllyjccjcz#",
                             "#ABcjyyyCjcjjcB#",
                             "#DEcyjjjFujjGEE#",
                             "#HHcjjjIJjjjKHH#",
                             ".#LcjjjyjjjcLL#.",
                             ".#MMcjjjjjcMMM#.",
                             "..#NNcccccNNN#..",
                             "...##OOOOOO##...",
                             ".....######....."])

forward_to = QPixmapWrapper(["16 15 52 1",
                             ". c None",
                             "a c None",
                             "X c None",
                             "W c None",
                             "T c None",
                             "Q c None",
                             "M c None",
                             "I c None",
                             "E c None",
                             "A c None",
                             "v c None",
                             "q c None",
                             "k c None",
                             "h c None",
                             "e c None",
                             "c c None",
                             "# c #024266",
                             "U c #034367",
                             "V c #1b5f8e",
                             "S c #266995",
                             "P c #30739c",
                             "O c #34769f",
                             "L c #3b7da3",
                             "R c #3f7fa6",
                             "K c #4181a7",
                             "H c #4586aa",
                             "G c #4c8cad",
                             "N c #4f8eb2",
                             "D c #4f90b1",
                             "J c #5998b9",
                             "y c #5999b8",
                             "C c #5a97b5",
                             "z c #5a9ab8",
                             "F c #63a2c0",
                             "u c #64a4bf",
                             "x c #6aa5be",
                             "B c #6eabc7",
                             "p c #6eadc6",
                             "s c #6faec6",
                             "o c #6faec7",
                             "w c #79b5ce",
                             "j c #79b7cd",
                             "m c #7cb9ce",
                             "t c #7db4c8",
                             "g c #83c1d4",
                             "r c #87c0d5",
                             "d c #8ecadc",
                             "n c #93c6d6",
                             "l c #94ccdc",
                             "i c #a1d2e0",
                             "f c #a9d9e5",
                             "b c #aedde9",
                             "..##.aa.........",
                             ".#bb#cca....aaa.",
                             ".#bdb#eca..a###a",
                             ".#fggb#hca.#bbb#",
                             ".#ijjjb#kca#lmm#",
                             ".#nopppb#qc#rss#",
                             ".#tuuuuub#v#wuu#",
                             ".#xyzzzzz#A#Bzz#",
                             ".#CDDDDDD#E#FDD#",
                             ".#GHHHHH#II#JHH#",
                             ".#KLLLL#MMa#NLL#",
                             ".#OPPP#QQa.#RPP#",
                             ".#SSS#TTa..aU##a",
                             ".#VV#WWa....Xaa.",
                             "..##.aa........."])

breakpoint_delete = QPixmapWrapper(["16 16 39 1",
                                    "# c None",
                                    "d c #380202",
                                    "G c #410202",
                                    "a c #630000",
                                    "m c #690200",
                                    "K c #810000",
                                    "J c #890400",
                                    "I c #900700",
                                    "H c #980a00",
                                    "C c #9e6864",
                                    "D c #9f0d00",
                                    "B c #a71000",
                                    "A c #ac1404",
                                    "z c #ae1300",
                                    "v c #b41600",
                                    "x c #b51700",
                                    "r c #bb1900",
                                    "t c #bd1a00",
                                    "y c #bd2414",
                                    "i c #bfa9a9",
                                    "p c #c41c00",
                                    "n c #c41d00",
                                    "w c #c7b4b4",
                                    "o c #c7b5b5",
                                    "l c #c82004",
                                    "s c #cab8b8",
                                    "u c #cc3626",
                                    "E c #d2c3c3",
                                    "h c #d32300",
                                    "c c #da2700",
                                    "q c #dd4a3c",
                                    "e c #e0442e",
                                    "f c #e4dddd",
                                    "g c #e8e2e2",
                                    "b c #ea5a4d",
                                    "F c #f4f0f0",
                                    "j c #f5f3f3",
                                    ". c #ffff00",
                                    "k c #ffffff",
                                    ".####aaaaaa####.",
                                    "..#aabbbbbbaa#..",
                                    "#..bbcddcccbe..#",
                                    "#a..ddfgddhh..a#",
                                    "#ab..ijjikd..lm#",
                                    "abnd..jjok..npna",
                                    "aqrdk..js..tdtta",
                                    "auvdkw....ddkdxa",
                                    "ayzdkww..kdkkdza",
                                    "aABdwk....kkCBBa",
                                    "aDDdk..EF..kGDDa",
                                    "#aHd..kwkk..HHa#",
                                    "#aI..kkkkkd..Ia#",
                                    "##..JdddddJJ..##",
                                    "#..aaKKKKKKaa..#",
                                    "..###aaaaaa###.."])

right_triangle = QPixmapWrapper([
    "16 15 26 1",
    ". c None",
    "# c #024266",
    "x c #1b5f8e",
    "w c #266995",
    "v c #30739c",
    "u c #34769f",
    "t c #3b7da3",
    "s c #4181a7",
    "r c #4586aa",
    "q c #4c8cad",
    "p c #4f90b1",
    "m c #5999b8",
    "o c #5a97b5",
    "n c #5a9ab8",
    "k c #64a4bf",
    "l c #6aa5be",
    "i c #6eadc6",
    "h c #6faec7",
    "f c #79b7cd",
    "j c #7db4c8",
    "d c #83c1d4",
    "b c #8ecadc",
    "g c #93c6d6",
    "e c #a1d2e0",
    "c c #a9d9e5",
    "a c #aedde9",
    ".....##.........",
    "....#aa#........",
    "....#aba#.......",
    "....#cdda#......",
    "....#efffa#.....",
    "....#ghiiia#....",
    "....#jkkkkka#...",
    "....#lmnnnnn#...",
    "....#opppppp#...",
    "....#qrrrrr#....",
    "....#stttt#.....",
    "....#uvvv#......",
    "....#www#.......",
    "....#xx#........",
    ".....##........."])

right_2triangles = QPixmapWrapper([
    "16 15 21 1",
    ". c None",
    "# c #024266",
    "s c #266995",
    "r c #30739c",
    "q c #30739d",
    "p c #3b7da3",
    "o c #3c7da4",
    "n c #4586aa",
    "m c #4a88aa",
    "l c #4f90b1",
    "k c #5994b1",
    "j c #5a9ab8",
    "h c #64a4bf",
    "i c #6ba2ba",
    "f c #6eadc6",
    "d c #79b7cd",
    "g c #7fb3c7",
    "b c #83c1d4",
    "e c #94c5d5",
    "c c #a5d5e2",
    "a c #aedde9",
    "................",
    "..##.....##.....",
    ".#aa#...#aa#....",
    ".#aba#..#aba#...",
    ".#cdda#.#cdda#..",
    ".#efffa##efffa#.",
    ".#ghhhha#ghhhha#",
    ".#ijjjjj#ijjjjj#",
    ".#klllll#klllll#",
    ".#mnnnn##mnnnn#.",
    ".#oppp#.#oppp#..",
    ".#qrr#..#qrr#...",
    ".#ss#...#ss#....",
    "..##.....##.....",
    "................"])

down_triangle = QPixmapWrapper([
    "15 16 17 1",
    ". c None",
    "# c #024266",
    "o c #1a5f8e",
    "n c #256895",
    "m c #2f729c",
    "l c #3a7ca3",
    "k c #4485aa",
    "j c #4f8fb1",
    "i c #68a3bf",
    "h c #79b0ca",
    "g c #85bbd1",
    "f c #93c5d7",
    "e c #9ccedd",
    "d c #a3d4e0",
    "c c #a7d7e3",
    "b c #abdae6",
    "a c #aedde9",
    "...............",
    "...............",
    "...............",
    "...............",
    ".#############.",
    "#aaaaabcdefghi#",
    "#jjjjjjjjjjjjj#",
    ".#kkkkkkkkkkk#.",
    "..#lllllllll#..",
    "...#mmmmmmm#...",
    "....#nnnnn#....",
    ".....#ooo#.....",
    "......###......",
    "...............",
    "...............",
    "..............."])

down_2triangles = QPixmapWrapper([
    "15 16 15 1",
    ". c None",
    "# c #024266",
    "m c #1a5f8e",
    "l c #256895",
    "k c #2f729c",
    "j c #3a7ca3",
    "i c #4485aa",
    "h c #79b0ca",
    "g c #85bbd1",
    "f c #93c5d7",
    "e c #9ccedd",
    "d c #a3d4e0",
    "c c #a7d7e3",
    "b c #abdae6",
    "a c #aedde9",
    "...............",
    "..###########..",
    ".#aaaabcdefgh#.",
    ".#iiiiiiiiiii#.",
    "..#jjjjjjjjj#..",
    "...#kkkkkkk#...",
    "....#lllll#....",
    ".....#mmm#.....",
    "..###########..",
    ".#aaaabcdefgh#.",
    ".#iiiiiiiiiii#.",
    "..#jjjjjjjjj#..",
    "...#kkkkkkk#...",
    "....#lllll#....",
    ".....#mmm#.....",
    "......###......"])

pause = QPixmapWrapper([
    "15 15 20 1",
    ". c None",
    "# c #024266",
    "r c #034367",
    "q c #30739c",
    "o c #3b7da3",
    "p c #3f7fa6",
    "m c #4586aa",
    "n c #4f8eb2",
    "k c #4f90b1",
    "l c #5998b9",
    "i c #5a9ab8",
    "j c #63a2c0",
    "g c #64a4bf",
    "h c #6eabc7",
    "e c #6faec6",
    "f c #79b5ce",
    "c c #7cb9ce",
    "d c #87c0d5",
    "b c #94ccdc",
    "a c #aedde9",
    "...............",
    "...............",
    "...###...###...",
    "..#aaa#.#aaa#..",
    "..#bcc#.#bcc#..",
    "..#dee#.#dee#..",
    "..#fgg#.#fgg#..",
    "..#hii#.#hii#..",
    "..#jkk#.#jkk#..",
    "..#lmm#.#lmm#..",
    "..#noo#.#noo#..",
    "..#pqq#.#pqq#..",
    "...r##...r##...",
    "...............",
    "..............."])

stop = QPixmapWrapper(["16 16 132 2",
                       "Qt c None",
                       ".# c #630000",
                       ".D c #690200",
                       "ab c #810000",
                       "a# c #890300",
                       "#7 c #890400",
                       "#8 c #8a0501",
                       "aa c #8c0804",
                       "a. c #8c0a07",
                       "#9 c #8e0e0b",
                       "#Z c #8f0600",
                       "#Y c #900700",
                       "#4 c #920700",
                       "#N c #930e05",
                       "#3 c #96110b",
                       "#D c #971508",
                       "#s c #971c10",
                       "#5 c #981a14",
                       "#0 c #981c16",
                       "#M c #990b01",
                       "#X c #9a0e04",
                       "#T c #9b1f17",
                       "#i c #9e2011",
                       "#B c #9f0d00",
                       "#S c #9f180e",
                       "#C c #a20e01",
                       "#L c #a20f01",
                       "#r c #a51504",
                       "#2 c #a5332e",
                       "#1 c #a53d38",
                       "#A c #a71000",
                       "#K c #a72114",
                       "#6 c #a73e39",
                       "#x c #a92617",
                       "#q c #aa1100",
                       "#z c #ab1100",
                       "#y c #ac1200",
                       "#p c #ac1404",
                       "#h c #ad1804",
                       "#R c #ad3a32",
                       "#W c #ad4039",
                       "#o c #ae1300",
                       ".K c #af2711",
                       "#H c #af4840",
                       ".7 c #b02c19",
                       "#f c #b31500",
                       "#g c #b31600",
                       ".y c #b32910",
                       "#n c #b41600",
                       "#d c #b51700",
                       "#m c #b7220e",
                       ".5 c #bb1900",
                       "#l c #bb4e40",
                       ".6 c #bc1a00",
                       ".U c #bc2b14",
                       ".3 c #bd1a00",
                       "#e c #bd2414",
                       ".J c #be3119",
                       "#c c #bf230b",
                       "#G c #bf6860",
                       ".S c #c01b00",
                       ".s c #c02407",
                       "#Q c #c0716b",
                       ".F c #c12104",
                       ".T c #c31d01",
                       ".Q c #c41d00",
                       ".2 c #c42207",
                       ".E c #c61f03",
                       ".i c #c62707",
                       ".o c #c82908",
                       "#b c #c84632",
                       ".C c #ca2205",
                       ".P c #ca270b",
                       ".v c #ca3214",
                       ".x c #cb2504",
                       ".p c #cc2705",
                       ".4 c #cc3626",
                       "#a c #cd6e60",
                       ".1 c #ce452e",
                       ".h c #d02301",
                       ".j c #d02804",
                       ".B c #d03516",
                       "#t c #d0a7a3",
                       ".O c #d15038",
                       ".q c #d22300",
                       "#E c #d2aba8",
                       ".n c #d32904",
                       ".w c #d62400",
                       ".0 c #d67b6b",
                       ".L c #d7aaa3",
                       ".r c #d83922",
                       "#j c #d9b7b3",
                       ".X c #dbb8b3",
                       ".b c #dc2903",
                       ".z c #dcafa8",
                       ".m c #dd2800",
                       ".k c #dd2801",
                       ".R c #dd4a3c",
                       ".l c #de2800",
                       ".f c #de2901",
                       ".g c #e0442e",
                       ".c c #e22a00",
                       ".e c #e32a00",
                       "#U c #e4c9c7",
                       ".d c #e62b00",
                       "#J c #e6cac7",
                       ".8 c #e7d0cd",
                       "#I c #e7d1d0",
                       ".V c #e8c9c5",
                       ".W c #e8d0cd",
                       "#v c #e8d6d4",
                       "#w c #e9d2d0",
                       ".a c #ea5a4d",
                       ".I c #eacac5",
                       "#k c #ead7d4",
                       ".G c #ecd1cc",
                       ".t c #ecd5d1",
                       ".u c #edd1cc",
                       "#O c #f2e7e6",
                       ".9 c #f2eeee",
                       ".H c #f5f2f2",
                       ".A c #f6e8e6",
                       "#F c #f6ecec",
                       "#. c #f6f2f2",
                       "#P c #f7efee",
                       "#u c #f7f4f4",
                       ".Z c #f8edec",
                       ".Y c #f8f4f4",
                       ".M c #f8f6f6",
                       ".N c #f9f0ee",
                       "## c #f9f5f5",
                       "#V c #fcfcfc",
                       "QtQtQtQtQt.#.#.#.#.#.#QtQtQtQtQt",
                       "QtQtQt.#.#.a.a.a.a.a.a.#.#QtQtQt",
                       "QtQt.#.a.a.b.c.d.d.e.f.a.g.#QtQt",
                       "Qt.#.a.h.i.j.k.l.m.n.o.p.q.r.#Qt",
                       "Qt.#.a.s.t.u.v.w.x.y.z.A.B.C.DQt",
                       ".#.a.E.F.G.H.I.J.K.L.M.N.O.P.Q.#",
                       ".#.R.S.T.U.V.H.W.X.Y.Z.0.1.2.3.#",
                       ".#.4.5.6.6.7.8.9#.###a#b#c.5#d.#",
                       ".##e#f#g#h#i#j#..H#k#l#m#n#f#o.#",
                       ".##p#q#r#s#t#u###v.H#w#x#y#z#A.#",
                       ".##B#C#D#E.M#F#G#H#I#u#J#K#L#B.#",
                       "Qt.##M#N#O#P#Q#R#S#T#U#V#W#X.#Qt",
                       "Qt.##Y#Z#0#1#2#3#4#4#5#6#5#Y.#Qt",
                       "QtQt.##7#8#9a.a#a#a##8aa#7.#QtQt",
                       "QtQtQt.#.#abababababab.#.#QtQtQt",
                       "QtQtQtQtQt.#.#.#.#.#.#QtQtQtQtQt"])

roadsign_nolimit = QPixmapWrapper(["16 16 4 1",
                                   "b c None",
                                   ". c None",
                                   "# c #585858",
                                   "a c #ffffff",
                                   ".....######.....",
                                   "...##########...",
                                   "..####aaaab###..",
                                   ".###aaaaaa#a###.",
                                   ".##aaaaaa#a#a##.",
                                   "###aaaaa#a#a#b##",
                                   "##aaaaa#a#a#aa##",
                                   "##aaaa#a#a#aaa##",
                                   "##aaa#a#a#aaaa##",
                                   "##aa#a#a#aaaaa##",
                                   "##b#a#a#aaaaa###",
                                   ".##a#a#aaaaaa##.",
                                   ".###a#aaaaaa###.",
                                   "..###baaaa####..",
                                   "...##########...",
                                   ".....######....."])

eject = QPixmapWrapper(["15 16 25 1",
                        ". c None",
                        "h c None",
                        "# c #024266",
                        "w c #074769",
                        "u c #2d7099",
                        "v c #2f729c",
                        "s c #3a7ca2",
                        "r c #3b7da3",
                        "p c #4486a9",
                        "o c #4788a9",
                        "n c #4a8bac",
                        "t c #4b8bab",
                        "m c #4e8eaf",
                        "l c #5292b2",
                        "k c #5796b5",
                        "q c #5998b7",
                        "j c #5a99b7",
                        "i c #5d9cba",
                        "g c #5e9dbb",
                        "f c #67a6c2",
                        "e c #71afc8",
                        "d c #7ab8cf",
                        "c c #84c1d5",
                        "b c #8ec9dc",
                        "a c #aedde9",
                        "...............",
                        "......###......",
                        ".....#aaa#.....",
                        "....#abbbb#....",
                        "...#acccccc#...",
                        "..#adddddddd#..",
                        ".#aeeeeeeeeee#.",
                        "#affffffffffff#",
                        "#ggggggggggggg#",
                        "h#############h",
                        "hhhhhhhhhhhhhhh",
                        ".#############.",
                        "#gijklmnoppppp#",
                        "#qrsssssssssss#",
                        "#tuvvvvvvvvvvv#",
                        ".w############."])

file_open = QPixmapWrapper(["16 16 223 2",
                            "bB c None",
                            "bC c None",
                            "bm c None",
                            "ba c None",
                            "aY c None",
                            ".x c None",
                            ".y c None",
                            ".O c None",
                            ".N c None",
                            ".w c None",
                            ".M c None",
                            ".o c None",
                            ".d c None",
                            ".# c None",
                            ".v c None",
                            ".m c None",
                            ".n c None",
                            ".c c None",
                            ".e c None",
                            ".b c None",
                            ".l c None",
                            ".a c None",
                            ".g c None",
                            ".p c None",
                            ".f c None",
                            "Qt c None",
                            "aI c #0013a9",
                            "bl c #000ca4",
                            ".E c #4e64c0",
                            ".h c #5d68bb",
                            "#f c #0014ae",
                            "#g c #001aaf",
                            ".k c #565fb7",
                            ".2 c #011daf",
                            "as c #0016a9",
                            ".L c #414caf",
                            ".q c #5d65b8",
                            "b# c #000da6",
                            "ac c #0017ac",
                            "#e c #1a33ac",
                            "aX c #0011a8",
                            ".F c #6e75c0",
                            "#Y c #0016ae",
                            ".i c #6872c0",
                            ".u c #5a63bb",
                            ".D c #114acf",
                            "bA c #0012a6",
                            ".z c #215cd3",
                            "aH c #0012ab",
                            ".C c #4b83e7",
                            ".B c #4e84e7",
                            ".A c #588de8",
                            "#s c #082ab8",
                            "#r c #2f58d1",
                            ".r c #8288cc",
                            ".1 c #5d6bc3",
                            "#I c #041fb6",
                            "ar c #0012b0",
                            "bk c #0f29ae",
                            ".j c #727ac5",
                            "aq c #000bbc",
                            "aa c #000bbb",
                            ".U c #6f82cf",
                            "ab c #0010b5",
                            "bn c #0431bf",
                            "aG c #000cb9",
                            "#q c #3a62d0",
                            ".G c #a3a8dd",
                            ".K c #9ba1db",
                            "bb c #0333c3",
                            "aZ c #083fca",
                            "aJ c #2963d7",
                            "#Z c #447cdd",
                            "#J c #467fde",
                            "ad c #4880de",
                            "at c #4881de",
                            "#t c #4980e0",
                            "#h c #4b83e0",
                            ".3 c #4f86e0",
                            "b. c #3951bb",
                            "bz c #1633aa",
                            "by c #1d3aae",
                            "bx c #2140b2",
                            "bw c #2547b8",
                            "bv c #2b4fbd",
                            "bu c #3156c4",
                            "bt c #375dca",
                            "bo c #3967d3",
                            "bs c #3c65d0",
                            "br c #4169d2",
                            "bq c #416bd5",
                            "bp c #446ed5",
                            "#H c #265de7",
                            "aW c #152bb7",
                            "#W c #1d59eb",
                            "ap c #0015c0",
                            "#X c #0018bc",
                            "aF c #0019c8",
                            "ao c #001ec9",
                            "aE c #0024d2",
                            "aD c #0029da",
                            "an c #0630da",
                            "a# c #0726c9",
                            "aC c #083ce4",
                            "a. c #164be3",
                            "am c #164fed",
                            "#9 c #2767f6",
                            "aB c #2864f2",
                            "#V c #2e7dfc",
                            "al c #306af3",
                            "aV c #3148c4",
                            "#U c #3786fe",
                            "aU c #3853ce",
                            "#1 c #3870db",
                            "#D c #396bdb",
                            "aT c #3b5eda",
                            "#8 c #3c80fb",
                            "aS c #4067e1",
                            "#o c #4670da",
                            "#F c #4690fe",
                            "aA c #4784f9",
                            ".P c #4981e0",
                            "#G c #4b96fe",
                            "ak c #4c88fb",
                            "#L c #4f8ee7",
                            "aK c #5288e4",
                            "au c #5590e3",
                            "#E c #5798f9",
                            "aR c #587de8",
                            "#7 c #5998fc",
                            "#C c #5a7bd8",
                            "az c #5e9cfc",
                            "#p c #6087e2",
                            "af c #618fe7",
                            "#M c #648de2",
                            "ay c #64a4f9",
                            "#S c #6599f3",
                            "#T c #65a4fe",
                            "a0 c #6c96e6",
                            "#w c #6e8fdd",
                            "#R c #6fa3f2",
                            "aj c #6fabfe",
                            "#n c #7491de",
                            ".T c #76abee",
                            "aQ c #789ff3",
                            "#6 c #7db8fd",
                            "#Q c #80aff3",
                            "ae c #85bff5",
                            "ax c #85ccfe",
                            "#B c #86a0e7",
                            "#A c #86a3e8",
                            "aP c #88aef7",
                            "#d c #8995d7",
                            "#z c #89a8e8",
                            "#y c #8dabeb",
                            "#P c #8fbcf5",
                            "#v c #8fbdf1",
                            "ai c #8fc4fb",
                            "#x c #92b2f0",
                            ".6 c #95bbec",
                            "#j c #97b6e8",
                            "aO c #97c0fc",
                            ".S c #9bdafc",
                            "av c #9fc2f0",
                            "#O c #9fc5f7",
                            ".7 c #a1aadd",
                            "ah c #a2d5f8",
                            ".R c #a6defc",
                            "#N c #a7c2f5",
                            "#5 c #a9defe",
                            "bj c #aaaaca",
                            "aN c #afdafe",
                            ".5 c #afe6fc",
                            "bc c #b1c5f1",
                            "bi c #b5b2cf",
                            "a9 c #b5b2d6",
                            "aL c #b8d8f8",
                            ".Q c #b9ecfc",
                            "aw c #b9edfe",
                            "a8 c #bbb8da",
                            "bh c #bfbdd7",
                            "#0 c #bff2fe",
                            "#k c #c4c3e4",
                            "a7 c #c5c3e0",
                            "#c c #c8cef3",
                            "aM c #c8edfe",
                            "#i c #c8fbfe",
                            ".4 c #cafcfe",
                            "#u c #cbfdfe",
                            "bg c #cecae1",
                            ".s c #ced0f0",
                            "#K c #cffefe",
                            "#3 c #d0e4f8",
                            "a6 c #d1cee6",
                            ".V c #d1d3ed",
                            "#4 c #d5fbfd",
                            "#2 c #daebfc",
                            "a5 c #dbd7ec",
                            "bf c #dcd8e9",
                            ".t c #dcdef7",
                            ".0 c #dedef7",
                            "ag c #e1fefe",
                            "a4 c #e2def1",
                            "#b c #e3e2f8",
                            ".Z c #e6e4fb",
                            ".Y c #e6e6fb",
                            "#a c #e8e8fb",
                            ".H c #e9e9fb",
                            "be c #ebe7f2",
                            "a3 c #ece9f7",
                            ".J c #ecebfc",
                            "## c #eeeefe",
                            ".8 c #f1f1fb",
                            ".I c #f2f1fe",
                            ".X c #f2f2fe",
                            "a2 c #f6f3fc",
                            "#. c #f7f7fe",
                            "bd c #f9f6fb",
                            "#m c #fbf6fe",
                            ".W c #fcfcfe",
                            "#l c #fdf8fe",
                            "a1 c #fefdfe",
                            ".9 c #fefefe",
                            "QtQtQtQtQtQtQtQtQtQtQtQtQtQtQtQt",
                            ".#.a.b.c.d.e.f.g.h.i.j.k.l.m.n.o",
                            ".#.a.b.c.d.e.p.q.r.s.t.u.v.w.x.y",
                            ".z.A.B.C.D.E.F.G.H.I.J.K.L.M.N.O",
                            ".P.Q.R.S.T.U.V.W.X.Y.Z.0.1.2.N.O",
                            ".3.4.5.6.7.8.9#.###a#b#c#d#e#f#g",
                            "#h#i#j#k.9.9.9.9#l#m#n#o#p#q#r#s",
                            "#t#u#v#w#x#y#z#A#B#C#D#E#F#G#H#I",
                            "#J#K#L#M#N#O#P#Q#R#S#T#U#V#W#X#Y",
                            "#Z#0#1#2#3#4#5#6#7#8#9a.a#aaabac",
                            "adaeafagahaiajakalamanaoapaqaras",
                            "atauavawaxayazaAaBaCaDaEaFaGaHaI",
                            "aJaKaLaMaNaOaPaQaRaSaTaUaVaWaXaY",
                            "aZa0a1.9a1a2a3a4a5a6a7a8a9b.b#ba",
                            "bbbc.9.9.9.9bdbebfbgbhbibjbkblbm",
                            "bnbobpbqbrbsbtbubvbwbxbybzbAbBbC"])

file_save = QPixmapWrapper(["16 16 179 2",
                            "Qt c None",
                            ".m c #000028",
                            "aW c #000028",
                            "aK c #0e0e47",
                            ".# c #a3a4c3",
                            ".l c #101149",
                            ".v c #070637",
                            "aV c #000023",
                            "av c #000028",
                            "ah c #00002a",
                            "#5 c #00002c",
                            "#R c #01012f",
                            "#D c #020230",
                            "#r c #040433",
                            "aU c #050530",
                            "#d c #060636",
                            ".1 c #080839",
                            ".N c #0a0a3b",
                            "aL c #100f46",
                            "aT c #10103f",
                            "aM c #101041",
                            "aS c #121241",
                            "aR c #131342",
                            "aQ c #161643",
                            "aP c #161645",
                            "aO c #191947",
                            "aN c #1b1b49",
                            "ai c #3d4073",
                            ".k c #3f4074",
                            "#6 c #494c7d",
                            ".j c #515180",
                            "#S c #545786",
                            ".i c #5a5a87",
                            "#E c #5e608e",
                            ".h c #64648f",
                            "#s c #696b96",
                            ".g c #6e6e98",
                            "#e c #73769f",
                            ".f c #7979a0",
                            ".2 c #7f81a8",
                            ".e c #8585aa",
                            ".O c #8b8db1",
                            ".d c #9595b6",
                            ".F c #9697b9",
                            ".c c #a1a0bf",
                            ".w c #a1a2c3",
                            ".b c #a3a3c3",
                            ".a c #acaec9",
                            ".E c #0f0f41",
                            "aJ c #000025",
                            "aw c #313267",
                            ".n c #aeafcb",
                            "aI c #000224",
                            "au c #020b32",
                            "ag c #060f39",
                            "#4 c #0c1643",
                            "#Q c #131d4c",
                            "#C c #1b2756",
                            "#q c #2b3765",
                            ".A c #344f85",
                            "aH c #353a64",
                            "#c c #394473",
                            ".0 c #3e4b79",
                            "#b c #435482",
                            "at c #454f79",
                            "#p c #455582",
                            "#a c #495b88",
                            ".M c #4c5987",
                            "af c #4f5880",
                            ".r c #546b9b",
                            "#3 c #59628b",
                            "ax c #5b6690",
                            "#o c #5b6892",
                            "## c #5b6e9a",
                            ".z c #5e75a1",
                            "#n c #627099",
                            "#P c #646d96",
                            ".D c #6878a4",
                            "aj c #6b7da4",
                            "#. c #6d83ab",
                            "#m c #6e7ca4",
                            "#B c #6f799e",
                            ".u c #737da8",
                            "#7 c #7b8db1",
                            ".9 c #7b8fb7",
                            "#l c #7c8db1",
                            "ay c #7f84a7",
                            "aG c #8180a4",
                            "aF c #8181a3",
                            ".Z c #818faf",
                            ".8 c #8599be",
                            "ar c #8787b0",
                            "as c #8988b2",
                            "aE c #8b8baa",
                            "#k c #8d9bbe",
                            ".7 c #8ea1c4",
                            "#T c #8fa1c3",
                            "aq c #9191b8",
                            "aD c #9292b2",
                            "ad c #9595b9",
                            "ak c #969fc0",
                            ".q c #96a6c5",
                            "ae c #9796bb",
                            ".6 c #98a9ca",
                            "ap c #9a9ac0",
                            "#j c #9baaca",
                            "aC c #9c9cb9",
                            ".J c #9caecc",
                            "ac c #9e9ec2",
                            "#1 c #9e9ec3",
                            "#F c #9eaece",
                            "#2 c #a1a0c4",
                            "#8 c #a1abca",
                            ".5 c #a1b1d1",
                            "ao c #a3a3c8",
                            "aB c #a6a6c0",
                            "#i c #a6b6d2",
                            "ab c #a7a7c9",
                            ".I c #a7b5d0",
                            "#t c #a7b7d5",
                            "#0 c #a8a8ca",
                            "#N c #ababce",
                            "#g c #abbbd6",
                            "an c #acacd0",
                            ".4 c #acbbd7",
                            "#O c #afaecf",
                            "#h c #afbdd8",
                            "aA c #b0b0c9",
                            "aa c #b1b1d1",
                            "#Z c #b2b2d2",
                            "#U c #b3bdd5",
                            "#M c #b6b6d5",
                            "am c #b6b6d7",
                            "#f c #b7c4de",
                            ".L c #b8c4d8",
                            "a# c #bbbbd8",
                            "#Y c #bcbcda",
                            "#L c #c0c0dd",
                            ".Y c #c0c9db",
                            ".C c #c0cce0",
                            "#G c #c2cbe0",
                            "az c #c4c3d7",
                            "a. c #c4c4e0",
                            ".3 c #c4d0e7",
                            "#X c #c5c5e1",
                            ".B c #c5cee1",
                            ".s c #c8d1e2",
                            "#u c #c9d3e3",
                            "#K c #cac9e4",
                            "al c #cbcae7",
                            ".P c #cbd6eb",
                            ".X c #ccd6e6",
                            "#A c #cecee2",
                            "#z c #cecee3",
                            "#W c #cfcfe9",
                            ".W c #d1d8e8",
                            ".Q c #d1daeb",
                            "#J c #d3d3ec",
                            ".V c #d3dbe9",
                            ".t c #d5deed",
                            ".G c #d5e0f1",
                            ".U c #d6deec",
                            "#9 c #dad8ee",
                            "#y c #dadaeb",
                            ".T c #dbe2f0",
                            "#I c #dcdcf1",
                            ".S c #e0e6f2",
                            ".x c #e2ebf9",
                            ".R c #e3e9f5",
                            "#V c #e4e3f8",
                            "#x c #e4e4f3",
                            "#H c #e8e7f7",
                            ".H c #ebf0f8",
                            "#w c #eeeefb",
                            ".K c #f0f3fb",
                            ".o c #f2f8fe",
                            ".y c #f5f9fe",
                            "#v c #fbfbfe",
                            ".p c #fefefe",
                            "QtQtQtQtQtQtQtQtQtQtQtQtQtQtQtQt",
                            "Qt.#.a.b.c.d.e.f.g.h.i.j.k.l.mQt",
                            "Qt.n.o.p.q.r.s.p.p.p.p.p.t.u.vQt",
                            "Qt.w.x.y.z.A.B.p.p.p.p.p.C.D.EQt",
                            "Qt.F.G.H.I.J.K.p.p.p.p.p.L.M.NQt",
                            "Qt.O.P.Q.R.S.T.U.V.W.X.Y.Z.0.1Qt",
                            "Qt.2.3.4.5.6.7.8.9#.###a#b#c#dQt",
                            "Qt#e#f#g#h#i#j#k#l#m#n#o#p#q#rQt",
                            "Qt#s#t#u.p.p#v#w#x#y#z#A#B#C#DQt",
                            "Qt#E#F#G#H#I#J#K#L#M#N#O#P#Q#RQt",
                            "Qt#S#T#U#V#W#X#Y#Z#0#1#2#3#4#5Qt",
                            "Qt#6#7#8#9a.a#aaabacadaeafagahQt",
                            "QtaiajakalamanaoapaqarasatauavQt",
                            "QtawaxayazaAaBaCaDaEaFaGaHaIaJQt",
                            "QtaKaLaMaNaOaPaQaRaSaTaTaUaVaWQt",
                            "QtQtQtQtQtQtQtQtQtQtQtQtQtQtQtQt"])

reexecute = QPixmapWrapper(["16 16 23 1",
                            "b c None",
                            "d c None",
                            "a c None",
                            "# c None",
                            ". c None",
                            "c c #024266",
                            "u c #1b5f8e",
                            "t c #34769f",
                            "s c #4182a7",
                            "r c #4e8eb0",
                            "o c #5696b5",
                            "q c #5a9ab9",
                            "p c #5c9cbb",
                            "m c #67a6c2",
                            "k c #74b2ca",
                            "l c #75b1ca",
                            "h c #81bed3",
                            "j c #84bfd3",
                            "g c #8ac4d7",
                            "i c #91c8d9",
                            "n c #98ccdd",
                            "f c #a3d5e3",
                            "e c #aedde9",
                            "................",
                            ".....##.........",
                            "...abcccc.......",
                            "..dcceeeecc.....",
                            ".acefghhhijc..cc",
                            ".cekkkkkkkkkccec",
                            "bclmccccmmmmmnmc",
                            "copc....cqqqqqqc",
                            "crc......crrrrrc",
                            "csc......csssssc",
                            "acc.....cttttttc",
                            ".ctc....ccccccc.",
                            "..cuc..........a",
                            "...ccc..........",
                            ".....ccc........",
                            "................"])

reexecute_loop = QPixmapWrapper(["16 16 24 1",
                                 "v c None",
                                 "b c None",
                                 "d c None",
                                 "a c None",
                                 "# c None",
                                 ". c None",
                                 "c c #024266",
                                 "u c #1b5f8e",
                                 "t c #34769f",
                                 "s c #4182a7",
                                 "r c #4e8eb0",
                                 "o c #5696b5",
                                 "q c #5a9ab9",
                                 "p c #5c9cbb",
                                 "m c #67a6c2",
                                 "k c #74b2ca",
                                 "l c #75b1ca",
                                 "h c #81bed3",
                                 "j c #84bfd3",
                                 "g c #8ac4d7",
                                 "i c #91c8d9",
                                 "n c #98ccdd",
                                 "f c #a3d5e3",
                                 "e c #aedde9",
                                 "................",
                                 ".....##.........",
                                 "...abcccc.......",
                                 "..dcceeeecc.....",
                                 ".acefghhhijc..cc",
                                 ".cekkkkkkkkkccec",
                                 "bclmccccmmmmmnmc",
                                 "copc....cqqqqqqc",
                                 "crc......crrrrrc",
                                 "csc......csssssc",
                                 "acc.....cttttttc",
                                 ".ctc....ccccccc.",
                                 "..cuc.....vccv.a",
                                 "...ccc...cccv...",
                                 ".....cccccvv....",
                                 "................"])

spigot = QPixmapWrapper(["16 16 177 2",
                         ".i c None",
                         ".v c None",
                         ".u c None",
                         "aF c None",
                         "aL c None",
                         "#c c None",
                         "#b c None",
                         ".F c None",
                         ".G c None",
                         ".R c None",
                         ".k c None",
                         ".j c None",
                         ".# c None",
                         "Qt c None",
                         ".t c #000000",
                         ".H c #bebfc9",
                         "aC c #000000",
                         "#1 c #000000",
                         "ar c #000000",
                         ".h c #0d0300",
                         ".Q c #000000",
                         "#a c #000000",
                         "aU c #000000",
                         "aD c #000000",
                         "aE c #000000",
                         ".1 c #000000",
                         ".w c #170d00",
                         ".2 c #cacad3",
                         "aK c #000000",
                         ".I c #555763",
                         ".E c #000000",
                         "aQ c #000000",
                         "ax c #000000",
                         ".S c #dbdbe3",
                         "ay c #1a395b",
                         ".J c #2c2f40",
                         "am c #02050c",
                         ".P c #000000",
                         "aM c #29537c",
                         "al c #000000",
                         ".g c #3c1800",
                         "aR c #060d23",
                         "aT c #000000",
                         ".a c #f9a302",
                         "#B c #05050d",
                         "#2 c #131321",
                         "as c #0b0b13",
                         "aq c #030203",
                         "aG c #63bdfe",
                         ".f c #432502",
                         "aw c #040407",
                         "#p c #29293d",
                         "aS c #020726",
                         "ae c #202034",
                         ".x c #4d3100",
                         ".K c #7d8292",
                         "#d c #b7b7c5",
                         "#M c #0d0d18",
                         ".l c #e9a60b",
                         "aB c #000326",
                         "#0 c #0d0d18",
                         "ad c #0d0d18",
                         ".d c #7f520b",
                         "ak c #0b0b13",
                         "## c #292932",
                         ".s c #4b2801",
                         ".e c #af792d",
                         ".D c #2a1600",
                         ".y c #513804",
                         "#N c #4d4d6d",
                         "ap c #0d0f26",
                         "at c #1a1a2a",
                         "av c #161625",
                         ".O c #16162d",
                         ".0 c #1f2032",
                         ".T c #dddde6",
                         "aP c #000534",
                         "a# c #19192a",
                         ".c c #cf8f09",
                         "au c #1f1f30",
                         "#q c #a9a9b9",
                         ".b c #eeaa0f",
                         "#n c #868699",
                         "#o c #737392",
                         "aJ c #030f5d",
                         "#C c #87879f",
                         ".3 c #dddde3",
                         "an c #576da4",
                         "a. c #363651",
                         ".z c #aa8e5c",
                         ".C c #996709",
                         "az c #a2d8fe",
                         "#m c #9f9fb1",
                         "aj c #38385a",
                         "aN c #649eed",
                         ".U c #dadae6",
                         "#A c #7b7b98",
                         "#9 c #3e3e64",
                         ".V c #bcbccf",
                         "af c #57577b",
                         ".B c #aa925a",
                         ".W c #c2c2d2",
                         "aa c #666482",
                         "#3 c #6c6c90",
                         "#8 c #3c3c64",
                         "#7 c #46466a",
                         "ai c #474770",
                         "#T c #4e4e72",
                         "#W c #4f4f74",
                         "ah c #4f4f76",
                         "ao c #525b98",
                         "ac c #55557a",
                         "aO c #567ddd",
                         "#Z c #58587b",
                         ".Z c #58587d",
                         "#6 c #5a5a7b",
                         "ag c #5a5a80",
                         "#U c #606081",
                         "#L c #626284",
                         "#5 c #636383",
                         "#S c #676786",
                         ".N c #686991",
                         "aI c #6ba9fc",
                         "#R c #6c6c8b",
                         "#V c #727291",
                         "#4 c #757592",
                         "#. c #7c7c9a",
                         "#X c #7d7d98",
                         "ab c #83839b",
                         "#P c #8686a0",
                         "#Q c #8787a1",
                         "#O c #9292aa",
                         "#G c #9595ab",
                         "#K c #9c9cb1",
                         "aA c #9cb8eb",
                         "aH c #9fd8fe",
                         "#Y c #a0a0b3",
                         ".6 c #a2a2b6",
                         ".Y c #a4a4b7",
                         ".9 c #a8a8bc",
                         "#J c #a9a9bc",
                         "#F c #aaaabd",
                         ".7 c #acacbe",
                         "#k c #afafc0",
                         "#l c #afafc2",
                         "#h c #b1b1c2",
                         "#H c #b2b2c3",
                         "#u c #b9b9c8",
                         ".M c #bdc2d8",
                         "#g c #c3c3d0",
                         "#E c #c4c4d1",
                         "#I c #cacad6",
                         ".5 c #cbcbd7",
                         "#r c #d0d0db",
                         "#v c #d2d2dc",
                         "#z c #d3d3de",
                         ".A c #d8bd89",
                         "#s c #dadae1",
                         "#D c #dadae2",
                         "#y c #dcdce6",
                         "#i c #e7e7ec",
                         "#e c #e7e7ed",
                         ".4 c #e9e9ed",
                         "#f c #ebebf0",
                         ".X c #ececf0",
                         "#j c #ececf1",
                         ".L c #f1f6fe",
                         ".r c #f2aa0f",
                         ".q c #f3b816",
                         "#t c #f7f7f8",
                         "#x c #f7f7f9",
                         ".8 c #f8f8f9",
                         ".o c #fbb92a",
                         ".p c #fdd061",
                         ".n c #fecf17",
                         ".m c #fee025",
                         "#w c #fefefe",
                         "Qt.#.#.#.a.b.c.d.e.f.g.h.i.i.i.i",
                         "Qt.j.j.k.l.m.n.o.p.q.r.s.t.i.i.i",
                         "Qt.u.u.v.w.x.y.z.A.B.C.D.E.i.i.i",
                         "Qt.F.G.H.I.J.K.L.M.N.O.P.Q.i.i.i",
                         "Qt.R.S.T.U.V.W.X.Y.Z.0.1.i.i.i.i",
                         "Qt.2.3.4.5.6.7.8.9#.###a#b#c.i.i",
                         "Qt#d#e#f#g#h#i#j#k#l#m#n#o#p.i.i",
                         "Qt#q#r#g#s#t#f#u#v#w#x#y#z#A#B.i",
                         "Qt#C#h.6#D#E#F#G#H#I#J#K#F#L#M.i",
                         "Qt#N#O#P#Q#R#S#T#U#V#W#X#Y#Z#0#1",
                         "Qt#2#3#4#5#6#7#8#9a.a#aaabacad#1",
                         "Qt.iaeafagahaiajakalamanaoapaqar",
                         "Qt.i.iasatauavawax.iayazaAaBaC.i",
                         "Qt.i.i.i.taDaE.Q.iaFaGaHaIaJaK.i",
                         "Qt.i.i.i.i.i.i.i.iaLaMaNaOaPaQ.i",
                         "Qt.i.i.i.i.i.i.i.i.i.iaRaSaTaU.i"])

grey_cross = QPixmapWrapper(["16 16 97 2",
                             "Qt c None",
                             ".# c #2c2c2c",
                             ".l c #313131",
                             "#D c #393939",
                             "#C c #1a1a1a",
                             "#E c #1f1f1f",
                             "#x c #212121",
                             ".z c #232323",
                             "#q c #262626",
                             ".I c #272727",
                             ".k c #292929",
                             "#m c #2a2a2a",
                             ".a c #2b2b2b",
                             "#i c #2e2e2e",
                             "#f c #313131",
                             "#d c #343434",
                             ".9 c #383838",
                             ".3 c #3b3b3b",
                             ".x c #404040",
                             ".X c #414141",
                             ".L c #484848",
                             ".A c #4a4a4a",
                             ".j c #4b4b4b",
                             ".i c #4d4d4d",
                             ".h c #4f4f4f",
                             ".g c #505050",
                             ".B c #525252",
                             ".f c #555555",
                             ".e c #575757",
                             ".d c #595959",
                             ".M c #5a5a5a",
                             ".c c #5c5c5c",
                             ".N c #5f5f5f",
                             ".C c #606060",
                             ".J c #636363",
                             ".b c #656565",
                             ".O c #676767",
                             ".D c #6b6b6b",
                             ".n c #6f6f6f",
                             ".P c #717171",
                             ".E c #737373",
                             "#B c #777777",
                             ".Q c #787878",
                             ".F c #797979",
                             ".Y c #7c7c7c",
                             ".R c #7d7d7d",
                             ".o c #7e7e7e",
                             ".G c #7f7f7f",
                             ".K c #818181",
                             ".S c #828282",
                             ".p c #858585",
                             "#w c #868686",
                             ".T c #888888",
                             ".q c #8b8b8b",
                             ".r c #8d8d8d",
                             ".U c #8f8f8f",
                             ".s c #919191",
                             ".V c #929292",
                             ".t c #959595",
                             ".8 c #979797",
                             ".u c #999999",
                             "#v c #9c9c9c",
                             "#a c #9e9e9e",
                             ".v c #a0a0a0",
                             "#b c #a4a4a4",
                             ".W c #a8a8a8",
                             ".2 c #aaaaaa",
                             "#e c #ababab",
                             "#c c #afafaf",
                             "#k c #b0b0b0",
                             ".H c #b4b4b4",
                             "#n c #b9b9b9",
                             "#h c #bababa",
                             ".w c #bcbcbc",
                             "#y c #bdbdbd",
                             "#l c #bfbfbf",
                             "#p c #c3c3c3",
                             "#A c #c6c6c6",
                             "#u c #c8c8c8",
                             "#o c #c9c9c9",
                             ".0 c #cccccc",
                             "#t c #d2d2d2",
                             ".1 c #d3d3d3",
                             ".Z c #d5d5d5",
                             ".y c #d7d7d7",
                             ".m c #d9d9d9",
                             "#z c #dadada",
                             ".5 c #dcdcdc",
                             ".7 c #dddddd",
                             "#. c #dfdfdf",
                             ".6 c #e2e2e2",
                             "#r c #e4e4e4",
                             "## c #e5e5e5",
                             "#g c #e7e7e7",
                             "#s c #e8e8e8",
                             "#j c #ebebeb",
                             ".4 c #ffffff",
                             "Qt.#.a.b.c.d.e.f.g.h.i.j.i.k.lQt",
                             ".#.c.m.n.o.p.p.q.r.s.t.u.v.w.x.l",
                             ".a.y.z.A.B.e.C.b.D.E.F.G.p.q.H.I",
                             ".J.K.L.M.N.O.D.P.Q.R.S.T.U.V.W.X",
                             ".c.Y.B.N.Z.0.P.Q.R.S.m.1.V.u.2.3",
                             ".d.p.e.O.y.4.5.R.S.6.4.7.8.v.2.9",
                             ".e.p.C.D.P.5.4#..6.4##.8#a#b#c#d",
                             ".f.q.b.P.Q.R#..4.4##.8#a.v#e.H#f",
                             ".g.r.D.Q.R.S.6.4.4#g#a.v.W.H#h#i",
                             ".h.s.E.R.S.6.4###g.4#j.W#k#l#l#m",
                             ".i.t.F.S.m.4##.8#a#j.4#j#n#o#p#q",
                             ".j.u.G.T#r.7.8#a.v.W#j#s#p#t#u.z",
                             ".i#v#w.U.V.8#a.v.W#k#n#p#o.5#u#x",
                             ".k#y.q.V.u.v#b#e.H#l#o#t#z.4.F.I",
                             ".l.x.H.W.2.2#c.H#h#l#p#A#z#B#C#D",
                             "Qt.l.I.X.3.9#d#f#i#m#q.z#E.I#DQt"])

# Acknowledgement: the following open_lock and close_lock xpm files are
# conversions from Joseph North's Sweetie BasePack 24-security-lock-open.png
# and 24-security-lock.png icons

open_lock = QPixmapWrapper(["24 24 24 1",
                            " 	c None",
                            ".	c #838383",
                            "+	c #6D6D6D",
                            "@	c #A5A5A5",
                            "#	c #E9E9E9",
                            "$	c #EBEBEB",
                            "%	c #A6A6A6",
                            "&	c #7E7E7E",
                            "*	c #B8B7AD",
                            "=	c #F3F3F3",
                            "-	c #F1F1F1",
                            ";	c #EEEEEE",
                            ">	c #ECECEC",
                            ",	c #9F9F9F",
                            "'	c #E7E7E7",
                            ")	c #B2B2B2",
                            "!	c #DFDFDF",
                            "~	c #AEAEAE",
                            "{	c #DADADA",
                            "]	c #AAAAAA",
                            "^	c #D5D5D5",
                            "/	c #D1D1D1",
                            "(	c #777777",
                            "_	c #A4A4A4",
                            "        .+++++++.       ",
                            "       .@#######@.      ",
                            "      .@$$$$$$$$$%.     ",
                            "      +#$@+++++@$$+     ",
                            "      &$$&*****+==+     ",
                            "      &$$&     +--+     ",
                            "       &&&     +;;+     ",
                            "               +>>+     ",
                            "               +##+     ",
                            "     ++,,+++++++''+     ",
                            "    +,,++,,,,,,,++,+    ",
                            "    +))))))))))))))+    ",
                            "    +!!!!!!!!!!!!!!+    ",
                            "    +~~~~~~~~~~~~~~+    ",
                            "    +{{{{{{{{{{{{{{+    ",
                            "    +]]]]]]]]]]]]]]+    ",
                            "    +^^^^^^^^^^^^^^+    ",
                            "    +%%%%%%%%%%%%%%+    ",
                            "    +//////////////+    ",
                            "    +(____________(+    ",
                            "    *++++++++++++++*    ",
                            "     **************     ",
                            "                        ",
                            "                        "])

close_lock = QPixmapWrapper(["24 24 24 1",
                             " 	c None",
                             ".	c #838383",
                             "+	c #6D6D6D",
                             "@	c #A7A7A7",
                             "#	c #F8F8F8",
                             "$	c #EBEBEB",
                             "%	c #A5A5A5",
                             "&	c #565656",
                             "*	c #B8B7AD",
                             "=	c #EFEFEF",
                             "-	c #808080",
                             ";	c #A1A1A1",
                             ">	c #E9E9E9",
                             ",	c #656565",
                             "'	c #B5B5B5",
                             ")	c #E3E3E3",
                             "!	c #B1B1B1",
                             "~	c #DDDDDD",
                             "{	c #ACACAC",
                             "]	c #D7D7D7",
                             "^	c #A8A8A8",
                             "/	c #D2D2D2",
                             "(	c #777777",
                             "_	c #A4A4A4",
                             "                        ",
                             "                        ",
                             "                        ",
                             "       .++++++++.       ",
                             "      .@########@.      ",
                             "     .@$$$$$$$$$$@.     ",
                             "     +$$.++++++%$$+     ",
                             "     +$$&******&$$+     ",
                             "     +=$+      +$$+     ",
                             "     +$$-@@@@@@-$$+     ",
                             "    +;>>>,,,,,,>>>;+    ",
                             "    +'++''''''''++'+    ",
                             "    +))))))))))))))+    ",
                             "    +!!!!!!!!!!!!!!+    ",
                             "    +~~~~~~~~~~~~~~+    ",
                             "    +{{{{{{{{{{{{{{+    ",
                             "    +]]]]]]]]]]]]]]+    ",
                             "    +^^^^^^^^^^^^^^+    ",
                             "    +//////////////+    ",
                             "    +(____________(+    ",
                             "    *++++++++++++++*    ",
                             "     **************     ",
                             "                        ",
                             "                        "])

import os
import os.path

__icons_loaded = False


def load_icons(appname, package=""):
    """
    load all icons found in path, subdirs '<package>/icons/<appname>'.
    Package is optional.
    """
    # loop over system path
    global __icons_loaded
    if __icons_loaded:
        return
    icon_paths = ['/usr/local/share/meqtrees'] + sys.path
    for path in icon_paths:
        path = path or '.'
        # for each entry, try <entry>/icons/<appname>'
        trydir = os.path.join(path, package, 'icons', appname)
        _dprint(3, 'trying icon path', trydir)
        try:
            files = os.listdir(trydir)
        except:
            continue
        _dprint(3, len(files), 'entries in', trydir)
        # loop over all files
        nicons = 0
        for f in files:
            (name, ext) = os.path.splitext(f);  # check extension
            if ext in ('.png', '.xpm', '.gif'):
                f = os.path.join(trydir, f)
                try:
                    pm = QPixmap(f)
                except:
                    _dprint(3, 'error loading icon', name, sys.exc_value())
                    continue
                # register pixmap as global symbol using the supplied name
                if name in globals():
                    globals()[name].assign(pm)
                else:
                    globals()[name] = QPixmapWrapper(pm)
                nicons += 1
                _dprint(4, 'loaded icon', f)
            else:
                _dprint(4, 'ignoring entry', f)
        _dprint(1, nicons, 'icons loaded from ', trydir)
        __icons_loaded = True


# define a pixmap access hook
# if name is undefined, it will be inserted into globals
# as a default QPixmapWrapper containing the missing icon xpm. This may be overridden
# later by load_icons() above. This allows apps to refer to icons before they're loaded.
class __PixmapHook(object):
    def __getattr__(self, name):
        _dprint(4, 'returning icon', name)
        return globals().setdefault(name, QPixmapWrapper())


pixmaps = __PixmapHook()


# define a Pixmap cache
class PixmapCache(object):
    def __init__(self, appname):
        self._appname = appname
        self._loaded = None
        self._pixmaps = {}
        self._pixmaps['_default'] = QPixmapWrapper()

    def __getattr__(self, name):
        # try to access attribute anyway, to see if we have one
        try:
            return dict.__getattr__(self, name)
        except AttributeError:
            pass
        # try to load pixmaps
        self._load()
        # try to access pixmap
        pm = self._pixmaps.get(name, None)
        if pm:
            return pm
        else:
            return QPixmapWrapper()

    def _load(self):
        """load all icons found in path, subdirs 'icons/appname'"""
        # loop over system path
        if self._loaded:
            return
        icon_paths = ['/usr/local/share/meqtrees'] + sys.path
        for path in icon_paths:
            path = path or '.'
            # for each entry, try <entry>/icons/<appname>'
            for a, b in [('icons', self._appname), (self._appname, 'icons')]:
                trydir = os.path.join(path, a, b)
                _dprint(3, 'trying icon path', trydir)
                try:
                    files = os.listdir(trydir)
                except:
                    continue
                _dprint(3, len(files), 'entries in', trydir)
                # loop over all files
                nicons = 0
                for f in files:
                    (name, ext) = os.path.splitext(f);  # check extension
                    if ext in ('.png', '.xpm', '.gif'):
                        f = os.path.join(trydir, f)
                        try:
                            pm = QPixmap(f)
                        except:
                            _dprint(3, 'error loading icon', name, sys.exc_value())
                            continue
                        # register pixmap
                        self._pixmaps[name] = QPixmapWrapper(pm)
                        nicons += 1
                        _dprint(4, 'loaded icon', f)
                    else:
                        _dprint(4, 'ignoring entry', f)
                _dprint(1, nicons, 'icons loaded from ', trydir)
                self._loaded = True
