"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const cloudfront = require("@aws-cdk/aws-cloudfront");
const s3 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const origins = require("../lib");
let stack;
let bucket;
let primaryOrigin;
beforeEach(() => {
    stack = new core_1.Stack();
    bucket = new s3.Bucket(stack, 'Bucket');
    primaryOrigin = new origins.S3Origin(bucket);
});
describe('Origin Groups', () => {
    test('correctly render the OriginGroups property of DistributionConfig', () => {
        const failoverOrigin = new origins.S3Origin(s3.Bucket.fromBucketName(stack, 'ImportedBucket', 'imported-bucket'));
        const originGroup = new origins.OriginGroup({
            primaryOrigin,
            fallbackOrigin: failoverOrigin,
            fallbackStatusCodes: [500],
        });
        new cloudfront.Distribution(stack, 'Distribution', {
            defaultBehavior: { origin: originGroup },
        });
        const primaryOriginId = 'DistributionOrigin13547B94F';
        const failoverOriginId = 'DistributionOrigin2C85CC43B';
        expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
            DistributionConfig: {
                Origins: [
                    {
                        Id: primaryOriginId,
                        DomainName: {
                            'Fn::GetAtt': ['Bucket83908E77', 'RegionalDomainName'],
                        },
                        S3OriginConfig: {
                            OriginAccessIdentity: {
                                'Fn::Join': ['', [
                                        'origin-access-identity/cloudfront/',
                                        { Ref: 'DistributionOrigin1S3Origin5F5C0696' },
                                    ]],
                            },
                        },
                    },
                    {
                        Id: failoverOriginId,
                        DomainName: {
                            'Fn::Join': ['', [
                                    'imported-bucket.s3.',
                                    { Ref: 'AWS::Region' },
                                    '.',
                                    { Ref: 'AWS::URLSuffix' },
                                ]],
                        },
                        S3OriginConfig: {
                            OriginAccessIdentity: {
                                'Fn::Join': ['', [
                                        'origin-access-identity/cloudfront/',
                                        { Ref: 'DistributionOrigin2S3OriginE484D4BF' },
                                    ]],
                            },
                        },
                    },
                ],
                OriginGroups: {
                    Items: [
                        {
                            FailoverCriteria: {
                                StatusCodes: {
                                    Items: [500],
                                    Quantity: 1,
                                },
                            },
                            Id: 'DistributionOriginGroup1A1A31B49',
                            Members: {
                                Items: [
                                    { OriginId: primaryOriginId },
                                    { OriginId: failoverOriginId },
                                ],
                                Quantity: 2,
                            },
                        },
                    ],
                    Quantity: 1,
                },
            },
        });
    });
    test('cannot have an Origin with their own failover configuration as the primary Origin', () => {
        const failoverOrigin = new origins.S3Origin(s3.Bucket.fromBucketName(stack, 'ImportedBucket', 'imported-bucket'));
        const originGroup = new origins.OriginGroup({
            primaryOrigin,
            fallbackOrigin: failoverOrigin,
        });
        const groupOfGroups = new origins.OriginGroup({
            primaryOrigin: originGroup,
            fallbackOrigin: failoverOrigin,
        });
        expect(() => {
            new cloudfront.Distribution(stack, 'Distribution', {
                defaultBehavior: { origin: groupOfGroups },
            });
        }).toThrow(/An OriginGroup cannot use an Origin with its own failover configuration as its primary origin!/);
    });
    test('cannot have an Origin with their own failover configuration as the fallback Origin', () => {
        const originGroup = new origins.OriginGroup({
            primaryOrigin,
            fallbackOrigin: new origins.S3Origin(s3.Bucket.fromBucketName(stack, 'ImportedBucket', 'imported-bucket')),
        });
        const groupOfGroups = new origins.OriginGroup({
            primaryOrigin,
            fallbackOrigin: originGroup,
        });
        expect(() => {
            new cloudfront.Distribution(stack, 'Distribution', {
                defaultBehavior: { origin: groupOfGroups },
            });
        }).toThrow(/An Origin cannot use an Origin with its own failover configuration as its fallback origin!/);
    });
    test('cannot have an empty array of fallbackStatusCodes', () => {
        const failoverOrigin = new origins.S3Origin(s3.Bucket.fromBucketName(stack, 'ImportedBucket', 'imported-bucket'));
        const originGroup = new origins.OriginGroup({
            primaryOrigin,
            fallbackOrigin: failoverOrigin,
            fallbackStatusCodes: [],
        });
        expect(() => {
            new cloudfront.Distribution(stack, 'Distribution', {
                defaultBehavior: { origin: originGroup },
            });
        }).toThrow(/fallbackStatusCodes cannot be empty/);
    });
});
//# sourceMappingURL=data:application/json;base64,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