import { RemovalPolicy, ResourceEnvironment, SecretValue, Stack } from 'aws-cdk-lib';
import { IGrantable, Grant, PolicyStatement, AddToResourcePolicyResult } from 'aws-cdk-lib/aws-iam';
import { IKey } from 'aws-cdk-lib/aws-kms';
import { SingletonFunction } from 'aws-cdk-lib/aws-lambda';
import { ISecret, ISecretAttachmentTarget, RotationSchedule, RotationScheduleOptions, SecretProps } from 'aws-cdk-lib/aws-secretsmanager';
import { Construct } from 'constructs';
export declare enum UploadType {
    /**
     * Pass the secret data inline (base64 encoded and compressed)
     */
    INLINE = "INLINE",
    /**
     * Uplaod the secert data as asset
     */
    ASSET = "ASSET"
}
/**
 * Configuration options for the SopsSync
 */
export interface SopsSyncOptions {
    /**
     * The custom resource provider to use. If you don't specify any, a new
     * provider will be created - or if already exists within this stack - reused.
     *
     * @default - A new singleton provider will be created
     */
    readonly sopsProvider?: SopsSyncProvider;
    /**
     * The filepath to the sops file
     */
    readonly sopsFilePath: string;
    /**
     * How should the secret be passed to the CustomResource?
     * @default INLINE
     */
    readonly uploadType?: UploadType;
    /**
     * The format of the sops file.
     *
     * @default - The fileformat will be derived from the file ending
     */
    readonly sopsFileFormat?: undefined | 'json' | 'yaml';
    /**
     * The kmsKey used to encrypt the sops file. Encrypt permissions
     * will be granted to the custom resource provider.
     *
     * @default - The key will be derived from the sops file
     */
    readonly sopsKmsKey?: IKey[];
    /**
     * The age key that should be used for encryption.
     */
    readonly sopsAgeKey?: SecretValue;
    /**
     * Should the encrypted sops value should be converted to JSON?
     * Only JSON can be handled by cloud formations dynamic references.
     *
     * @default true
     */
    readonly convertToJSON?: boolean;
    /**
     * Should the structure be flattened? The result will be a flat structure and all
     * object keys will be replaced with the full jsonpath as key.
     * This is usefull for dynamic references, as those don't support nested objects.
     *
     * @default true
     */
    readonly flatten?: boolean;
    /**
     * Shall all values be flattened? This is usefull for dynamic references, as there
     * are lookup errors for certain float types
     */
    readonly stringifyValues?: boolean;
}
/**
 * The configuration options extended by the target Secret
 */
export interface SopsSyncProps extends SopsSyncOptions {
    /**
     * The secret that will be populated with the encrypted sops file content.
     */
    readonly secret: ISecret;
}
/**
 * The custom resource, that is syncing the content from a sops file to a secret.
 */
export declare class SopsSync extends Construct {
    /**
     * The current versionId of the secret populated via this resource
     */
    readonly versionId: string;
    /**
     * The format of the input file
     */
    readonly sopsFileFormat: 'json' | 'yaml';
    /**
     * Was the format converted to json?
     */
    readonly converToJSON: boolean;
    /**
     * Was the structure flattened?
     */
    readonly flatten: boolean;
    /**
     * Were the values stringified?
     */
    readonly stringifiedValues: boolean;
    constructor(scope: Construct, id: string, props: SopsSyncProps);
}
export declare class SopsSyncProvider extends SingletonFunction implements IGrantable {
    private sopsAgeKeys;
    constructor(scope: Construct, id?: string);
    addAgeKey(key: SecretValue): void;
}
/**
 * The configuration options of the SopsSecret
 */
export interface SopsSecretProps extends SecretProps, SopsSyncOptions {
}
/**
 * A drop in replacement for the normal Secret, that is populated with the encrypted
 * content of the given sops file.
 */
export declare class SopsSecret extends Construct implements ISecret {
    private readonly secret;
    readonly encryptionKey?: IKey | undefined;
    readonly secretArn: string;
    readonly secretFullArn?: string | undefined;
    readonly secretName: string;
    readonly stack: Stack;
    readonly env: ResourceEnvironment;
    readonly sync: SopsSync;
    constructor(scope: Construct, id: string, props: SopsSecretProps);
    /**
     * Returns the current versionId that was created via the SopsSync
     */
    currentVersionId(): string;
    grantRead(grantee: IGrantable, versionStages?: string[]): Grant;
    grantWrite(_grantee: IGrantable): Grant;
    addRotationSchedule(id: string, options: RotationScheduleOptions): RotationSchedule;
    addToResourcePolicy(statement: PolicyStatement): AddToResourcePolicyResult;
    denyAccountRootDelete(): void;
    attach(target: ISecretAttachmentTarget): ISecret;
    applyRemovalPolicy(policy: RemovalPolicy): void;
    secretValueFromJson(jsonField: string): SecretValue;
    get secretValue(): SecretValue;
}
