"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SopsSecret = exports.SopsSyncProvider = exports.SopsSync = exports.UploadType = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const path = require("path");
//import * as crypto from 'crypto';
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_kms_1 = require("aws-cdk-lib/aws-kms");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const aws_s3_assets_1 = require("aws-cdk-lib/aws-s3-assets");
const aws_secretsmanager_1 = require("aws-cdk-lib/aws-secretsmanager");
const constructs_1 = require("constructs");
var UploadType;
(function (UploadType) {
    /**
     * Pass the secret data inline (base64 encoded and compressed)
     */
    UploadType["INLINE"] = "INLINE";
    /**
     * Uplaod the secert data as asset
     */
    UploadType["ASSET"] = "ASSET";
})(UploadType = exports.UploadType || (exports.UploadType = {}));
/**
 * The custom resource, that is syncing the content from a sops file to a secret.
 */
class SopsSync extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.converToJSON = props.convertToJSON ?? true;
        this.flatten = props.flatten ?? true;
        this.stringifiedValues = props.stringifyValues ?? true;
        const sopsFileFormat = props.sopsFileFormat ?? props.sopsFilePath.split('.').pop();
        switch (sopsFileFormat) {
            case 'json': {
                this.sopsFileFormat = 'json';
                break;
            }
            case 'yaml': {
                this.sopsFileFormat = 'yaml';
                break;
            }
            case 'yml': {
                this.sopsFileFormat = 'yaml';
                break;
            }
            default: {
                throw new Error(`Unsupported sopsFileFormat ${sopsFileFormat}`);
            }
        }
        const provider = props.sopsProvider ?? new SopsSyncProvider(scope);
        if (!fs.existsSync(props.sopsFilePath)) {
            throw new Error(`File ${props.sopsFilePath} does not exist!`);
        }
        /**
         * Handle uploadType INLINE or ASSET
         */
        const uploadType = props.uploadType ?? UploadType.INLINE;
        let sopsAsset = undefined;
        let sopsInline = undefined;
        let sopsS3File = undefined;
        switch (uploadType) {
            case UploadType.INLINE: {
                sopsInline = {
                    Content: fs.readFileSync(props.sopsFilePath).toString('base64'),
                    // We calculate the hash the same way as it would be done by new Asset(..) - so we can ensure stable version names even if switching from INLINE to ASSET and viceversa.
                    Hash: aws_cdk_lib_1.FileSystem.fingerprint(props.sopsFilePath),
                };
                break;
            }
            case UploadType.ASSET: {
                sopsAsset = new aws_s3_assets_1.Asset(this, 'Asset', {
                    path: props.sopsFilePath,
                });
                sopsS3File = {
                    Bucket: sopsAsset.bucket.bucketName,
                    Key: sopsAsset.s3ObjectKey,
                };
                break;
            }
        }
        if (provider.role !== undefined) {
            if (props.sopsKmsKey !== undefined) {
                props.sopsKmsKey.forEach((key) => key.grantDecrypt(provider.role));
            }
            const fileContent = fs.readFileSync(props.sopsFilePath);
            const regex = /arn:aws:kms:[a-z0-9-]+:[\d]+:key\/[a-z0-9-]+/gm;
            const results = regex.exec(fileContent.toString());
            if (results !== undefined) {
                results?.forEach((result, index) => aws_kms_1.Key.fromKeyArn(this, `SopsKey${index}`, result).grantDecrypt(provider.role));
            }
            props.secret.grantWrite(provider);
            if (sopsAsset !== undefined) {
                sopsAsset.bucket.grantRead(provider);
            }
            /**
             * fixes #234
             * If the kms key for secrets encryption is an IKey
             * there will be no permissions otherwise
             */
            if (props.secret.encryptionKey !== undefined &&
                !(props.secret.encryptionKey instanceof aws_kms_1.Key)) {
                props.secret.encryptionKey.grantEncryptDecrypt(provider);
            }
        }
        else {
            aws_cdk_lib_1.Annotations.of(this).addWarning(`Please ensure propper permissions for the passed lambda function:\n  - write Access to the secret\n  - encrypt with the sopsKmsKey${uploadType === UploadType.ASSET
                ? '\n  - download from asset bucket'
                : ''}`);
        }
        if (props.sopsAgeKey !== undefined) {
            provider.addAgeKey(props.sopsAgeKey);
        }
        const cr = new aws_cdk_lib_1.CustomResource(this, 'Resource', {
            serviceToken: provider.functionArn,
            resourceType: 'Custom::SopsSync',
            properties: {
                SecretARN: props.secret.secretArn,
                SopsS3File: sopsS3File,
                SopsInline: sopsInline,
                ConvertToJSON: this.converToJSON,
                Flatten: this.flatten,
                Format: this.sopsFileFormat,
                StringifiedValues: this.stringifiedValues,
            },
        });
        this.versionId = cr.getAttString('VersionId');
    }
}
exports.SopsSync = SopsSync;
_a = JSII_RTTI_SYMBOL_1;
SopsSync[_a] = { fqn: "cdk-sops-secrets.SopsSync", version: "1.2.36" };
class SopsSyncProvider extends aws_lambda_1.SingletonFunction {
    constructor(scope, id) {
        super(scope, id ?? 'SopsSyncProvider', {
            code: aws_lambda_1.Code.fromAsset(scope.node.tryGetContext('sops_sync_provider_asset_path') ||
                path.join(__dirname, '../assets/cdk-sops-lambda.zip')),
            runtime: aws_lambda_1.Runtime.GO_1_X,
            handler: 'cdk-sops-secrets',
            uuid: 'SopsSyncProvider',
            timeout: aws_cdk_lib_1.Duration.seconds(60),
            environment: {
                SOPS_AGE_KEY: aws_cdk_lib_1.Lazy.string({
                    produce: () => (this.sopsAgeKeys.map((secret) => secret.toString()) ?? []).join('\n'),
                }),
            },
        });
        this.sopsAgeKeys = [];
    }
    addAgeKey(key) {
        this.sopsAgeKeys.push(key);
    }
}
exports.SopsSyncProvider = SopsSyncProvider;
_b = JSII_RTTI_SYMBOL_1;
SopsSyncProvider[_b] = { fqn: "cdk-sops-secrets.SopsSyncProvider", version: "1.2.36" };
/**
 * A drop in replacement for the normal Secret, that is populated with the encrypted
 * content of the given sops file.
 */
class SopsSecret extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.secret = new aws_secretsmanager_1.Secret(this, 'Resource', props);
        // Fullfill secret Interface
        this.encryptionKey = this.secret.encryptionKey;
        this.secretArn = this.secret.secretArn;
        this.secretName = this.secret.secretName;
        this.stack = aws_cdk_lib_1.Stack.of(scope);
        this.env = {
            account: this.stack.account,
            region: this.stack.region,
        };
        this.sync = new SopsSync(this, 'SopsSync', {
            secret: this.secret,
            ...props,
        });
    }
    /**
     * Returns the current versionId that was created via the SopsSync
     */
    currentVersionId() {
        return this.sync.versionId;
    }
    grantRead(grantee, versionStages) {
        return this.secret.grantRead(grantee, versionStages);
    }
    grantWrite(_grantee) {
        throw new Error(`Method grantWrite(...) not allowed as this secret is managed by SopsSync`);
    }
    addRotationSchedule(id, options) {
        throw new Error(`Method addTotationSchedule('${id}', ${JSON.stringify(options)}) not allowed as this secret is managed by SopsSync`);
    }
    addToResourcePolicy(statement) {
        return this.secret.addToResourcePolicy(statement);
    }
    denyAccountRootDelete() {
        return this.secret.denyAccountRootDelete();
    }
    attach(target) {
        return this.secret.attach(target);
    }
    applyRemovalPolicy(policy) {
        return this.secret.applyRemovalPolicy(policy);
    }
    secretValueFromJson(jsonField) {
        return aws_cdk_lib_1.SecretValue.secretsManager(this.secretArn, {
            jsonField,
            versionId: this.sync.versionId,
        });
    }
    get secretValue() {
        return this.secretValueFromJson('');
    }
}
exports.SopsSecret = SopsSecret;
_c = JSII_RTTI_SYMBOL_1;
SopsSecret[_c] = { fqn: "cdk-sops-secrets.SopsSecret", version: "1.2.36" };
//# sourceMappingURL=data:application/json;base64,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