# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['splunk_otel', 'splunk_otel.cmd', 'splunk_otel.site']

package_data = \
{'': ['*']}

install_requires = \
['opentelemetry-api==1.0.0rc1',
 'opentelemetry-exporter-jaeger==1.0.0rc1',
 'opentelemetry-instrumentation>=0.18b0,<0.19',
 'opentelemetry-propagator-b3==1.0.0rc1',
 'opentelemetry-sdk==1.0.0rc1']

entry_points = \
{'console_scripts': ['splk-py-trace = splunk_otel.cmd.trace:run',
                     'splk-py-trace-bootstrap = splunk_otel.cmd.bootstrap:run']}

setup_kwargs = {
    'name': 'splunk-opentelemetry',
    'version': '0.8.0',
    'description': 'The Splunk distribution of OpenTelemetry Python Instrumentation provides a Python agent that automatically instruments your Python application to capture and report distributed traces to SignalFx APM.',
    'long_description': "# Splunk distribution of OpenTelemetry Python\n\n[![CircleCI](https://circleci.com/gh/signalfx/splunk-otel-python.svg?style=svg)](https://circleci.com/gh/signalfx/splunk-otel-python)\n[![codecov](https://codecov.io/gh/signalfx/splunk-otel-python/branch/main/graph/badge.svg?token=XKXjEQKGaK)](https://codecov.io/gh/signalfx/splunk-otel-python)\n\nThe Splunk distribution of [OpenTelemetry\nPython](https://github.com/open-telemetry/opentelemetry-python) provides\nmultiple installable packages that automatically instruments your Python\napplication to capture and report distributed traces to Splunk APM.\n\nThis Splunk distribution comes with the following defaults:\n\n- [B3 context propagation](https://github.com/openzipkin/b3-propagation).\n- [Jaeger thrift\n  exporter](https://opentelemetry-python.readthedocs.io/en/stable/exporter/jaeger/jaeger.html)\n  configured to send spans to a locally running [SignalFx Smart\n  Agent](https://docs.signalfx.com/en/latest/apm/apm-getting-started/apm-smart-agent.html)\n  (`http://localhost:9080/v1/trace`).\n- Unlimited default limits for [configuration options](#trace-configuration) to\n  support full-fidelity traces.\n\nIf you're currently using the SignalFx Tracing Library for Python and want to\nmigrate to the Splunk Distribution of OpenTelemetry Python, see [Migrate from\nthe SignalFx Tracing Library for Python](migration.md).\n\n> :construction: This project is currently in **BETA**.\n\n## Getting Started\n\nThe instrumentation works with Python version 3.6+. Supported libraries are\nlisted\n[here](https://github.com/open-telemetry/opentelemetry-python-contrib/tree/master/instrumentation).\n\nTo get started, install the `splunk-opentelemetry` package, run the bootstrap\nscript and wrap your run command with `splk-py-trace`.\n\nFor example, if the runtime parameters were:\n\n```\npython main.py --port=8000\n```\n\nThen the runtime parameters should be updated to:\n\n```\n$ pip install splunk-opentelemetry\n$ splk-py-trace-bootstrap\n$ SPLK_SERVICE_NAME=my-python-app \\\n    splk-py-trace python main.py --port=8000\n```\n\nThe service name is the only configuration option that typically needs to be\nspecified. A couple other configuration options that may need to be changed or\nset are:\n\n- Endpoint if not sending to a locally running Smart Agent with default\n  configuration\n- Environment attribute (example:\n  `OTEL_RESOURCE_ATTRIBUTES=environment=production`) to specify what\n  environment the span originated from.\n\nInstrumentation works by patching supported libraries at runtime with an\nOpenTelemetry-compatible tracer to capture and export trace spans. The agent\nalso registers an OpenTelemetry `get_tracer` so you can support existing custom\ninstrumentation or add custom instrumentation to your application later.\n\nTo see the Python instrumentation in action with sample applications, see our\n[examples](https://github.com/signalfx/tracing-examples/tree/master/signalfx-tracing/splunk-otel-python).\n\n## All configuration options\n\n### Jaeger exporter\n\n| Environment variable          | Default value                        | Notes                                                                  |\n| ----------------------------- | ------------------------------------ | ---------------------------------------------------------------------- |\n| SPLK_TRACE_EXPORTER_URL       | `http://localhost:9080/v1/trace`     | The jaeger endpoint to connect to. Currently only HTTP is supported.   |\n| SPLK_SERVICE_NAME             | `unnamed-python-service`             | The service name of this JVM instance.                                 |\n| SPLK_ACCESS_TOKEN             |                                      | The optional organization access token for trace submission requests.  |\n\n### Trace configuration\n\n| Environment variable          | Default value  | Purpose                                                                                                                                                                                                                                                                                                                                                                                                   |\n| ----------------------------- | -------------- | ------------------------------------------------------------------------------------                                                                                                                                                                                                                                                                                                                      |\n| SPLK_MAX_ATTR_LENGTH          | 1200            | Maximum length of string attribute value in characters. Longer values are truncated.                                                                                                                                                                                                                                                                                                                      |\n| OTEL_RESOURCE_ATTRIBUTES      | unset          | Comma-separated list of resource attributes added to every reported span. <details><summary>Example</summary>`key1=val1,key2=val2`</details>\n| OTEL_TRACE_ENABLED            | `true`         | Globally enables tracer creation and auto-instrumentation.                                                                                                                                                                                                                                                                                                                                                |\n\n## Advanced Getting Started\n\n### Alternative: List requirements instead of installing them\n\nThe `splk-py-trace-bootstrap` command can optionally print out the list of\npackages it would install if you chose. In order to do so, pass\n`-a=requirements` CLI argument to it. For example,\n\n```\nsplk-py-trace-bootstrap -a requirements\n```\n\nWill output something like the following:\n\n```\nopentelemetry-instrumentation-falcon>=0.15b0\nopentelemetry-instrumentation-jinja2>=0.15b0\nopentelemetry-instrumentation-requests>=0.15b0\nopentelemetry-instrumentation-sqlite3>=0.15b0\nopentelemetry-exporter-jaeger>=0.15b0\n```\n\nYou can pipe the output of this command to append the new packages to your\nrequirements.txt file or to something like `poetry add`.\n\n### Alternative: Instrument and configure by adding code\n\nIf you cannot use `splk-py-trace` command, you can also add a couple of lines\nof code to your Python application to achieve the same result.\n\n```python\nfrom splunk_otel.tracing import start_tracing\n\nstart_tracing()\n\n# rest of your python application's entrypoint script\n```\n\n## Exporting to Smart Agent, Otel collector or SignalFx ingest\n\nThis package exports spans in Jaeger Thrift format over HTTP and supports\nexporting to the SignalFx Smart Agent, OpenTelemetry collector and directly to\nSignalFx ingest API. You can use `SPLK_TRACE_EXPORTER_URL` environment variable\nto specify an export URL. The value must be a full URL including scheme and\npath.\n\n### Smart Agent\n\nThis is the default option. You do not need to set any config options if you\nwant to export to the Smart Agent and you are running the agent on the default\nport (`9080`). The exporter will default to `http://localhost:9080/v1/trace`\nwhen the environment variable is not specified.\n\n### OpenTelemetry Collector\n\nIn order to do this, you'll need to enable Jaeger Thrift HTTP receiver on\nOpenTelemetry Collector and set `SPLK_TRACE_EXPORTER_URL` to\n`http://localhost:14268/api/traces` assuming the collector is reachable via\nlocalhost.\n\n### SignalFx Ingest API\n\nIn order to send traces directly to SignalFx ingest API, you need to:\n\n1. Set `SPLK_TRACE_EXPORTER_URL` to\n   `https://ingest.<realm>.signalfx.com/v2/trace` where `realm` is your\n   SignalFx realm e.g, `https://ingest.us0.signalfx.com/v2/trace`.\n2. Set `SPLK_ACCESS_TOKEN` to one of your SignalFx APM access tokens.\n\n\n### Special Cases\n\n#### Celery\n\nTracing Celery workers works out of the box when you use the `splk-py-trace`\ncommand to start your Python application. However, if you are instrumenting\nyour celery workers with code, you'll need to make sure you setup tracing for\neach worker by using Celery's `celery.signalfx.worker_process_init` signal.\n\nFor example:\n\n```python\nfrom splunk_otel.tracing import start_tracing\nfrom celery.signals import worker_process_init\n\n@worker_process_init.connect(weak=False)\ndef on_worker_process_init(*args, **kwargs):\n    start_tracing()\n\n# rest of your python application's entrypoint script\n```\n\n#### Django\n\nAutomatically instrumenting Django requires `DJANGO_SETTINGS_MODULE`\nenvironment variable to be set. The value should be the same as set in your\n`manage.py` or `wsgi.py` modules. For example, if your manage.py file sets this\nenvironment variable to `mydjangoproject.settings` and you start your Django\nproject as `./manage.py runserver`, then you can automatically instrument your\nDjango project as follows:\n\n```\nexport DJANGO_SETTINGS_MODULE=mydjangoproject.settings\nsplk-py-trace ./manage.py runserver\n```\n\n#### Gunicorn\n\nLike Celery, we'll also need to setup tracing per Gunicorn worker. This can be\ndone by setting up tracing inside Gunicorn's `post_fork()` handler.\n\nFor example:\n\n```python\n# gunicorn.config.py\nfrom splunk_otel.tracing import start_tracing\n\ndef post_fork(server, worker):\n    start_tracing()\n```\n\nThen add `-c gunicorn.config.py` CLI flag to your gunicorn command.\n\n#### UWSGI \n\nWhen using UWSGI, tracing must be setup as a response to the `post_fork` signal. \n\nFor example:\n\n```python\nimport uwsgidecorators\nfrom splunk_otel.tracing import start_tracing\n\n@uwsgidecorators.postfork\ndef setup_tracing():\n    start_tracing()\n\n```\n\n##### Running with uwsgi\n```\nuwsgi --http :9090 --wsgi-file <your_app.py> --callable <your_wsgi_callable> --master --enable-threads\n```\n\nThe above snippet should be placed in the main python script that uwsgi imports and loads.\n\n#### UWSGI and Flask\n\nUsing USWGI with Flask requires one additional little step. Calling `start_tracing()` does not auto-instrument pre-existing flask app instances but only flask instances created after. When running flask with uwsgi, we need to create a new flask app instance before the post_fork signal is emitted. This means your flask app will not be auto-instrumented. However, you can still auto-instrument an existing flask app explicitly by importing and calling the flask instrumentor. \n\nFor example:\n```python\n# app.py\nimport uwsgidecorators\nfrom splunk_otel.tracing import start_tracing\nfrom opentelemetry.instrumentation.flask import FlaskInstrumentor\nfrom flask import Flask\n\napp = Flask(__name__)\n\n@uwsgidecorators.postfork\ndef setup_tracing():\n    start_tracing()\n    # instrument our flask app instance eplicitly\n    FlaskInstrumentor().instrument_app(app)\n\n@app.route('/')\ndef hello_world():\n    return 'Hello, World!'\n```\n\n##### Running with uWSGI:\n```\nuwsgi --http :9090 --wsgi-file app.py --callable app --master --enable-threads\n```\n\n\n## Manually instrument an application\n\nDocumentation on how to manually instrument a Python application is available\n[here](https://opentelemetry-python.readthedocs.io/en/stable/getting-started.html).\n\n## Troubleshooting\n\nEnable debug logging like you would for any Python application.\n\n```python\nimport logging\n\nlogging.basicConfig(level=logging.DEBUG)\n```\n\n> :warning: Debug logging is extremely verbose and resource intensive. Enable\n> debug logging only when needed and disable when done.\n\n# License and versioning\n\nThe Splunk distribution of OpenTelemetry Python Instrumentation is a\ndistribution of the [OpenTelemetry Python\nproject](https://github.com/open-telemetry/opentelemetry-python). It is\nreleased under the terms of the Apache Software License version 2.0. See [the\nlicense file](./LICENSE) for more details.\n",
    'author': 'Splunk',
    'author_email': 'splunk-oss@splunk.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': None,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.5,<4.0',
}


setup(**setup_kwargs)
