"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Appflow = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [appflow](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonappflow.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Appflow extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [appflow](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonappflow.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'appflow';
        this.actionList = {
            "CreateConnectorProfile": {
                "url": "https://docs.aws.amazon.com/appflow/1.0/APIReference/API_CreateConnectorProfile.html",
                "description": "Grants permission to create a login profile to be used with Amazon AppFlow flows",
                "accessLevel": "Write"
            },
            "CreateFlow": {
                "url": "https://docs.aws.amazon.com/appflow/1.0/APIReference/API_CreateFlow.html",
                "description": "Grants permission to create an Amazon AppFlow flow",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "DeleteConnectorProfile": {
                "url": "https://docs.aws.amazon.com/appflow/1.0/APIReference/API_DescribeConnectorProfiles.html",
                "description": "Grants permission to delete a login profile configured in Amazon AppFlow",
                "accessLevel": "Write",
                "resourceTypes": {
                    "connectorprofile": {
                        "required": true
                    }
                }
            },
            "DeleteFlow": {
                "url": "https://docs.aws.amazon.com/appflow/1.0/APIReference/API_DeleteFlow.html",
                "description": "Grants permission to delete an Amazon AppFlow flow",
                "accessLevel": "Write",
                "resourceTypes": {
                    "flow": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "DescribeConnectorEntity": {
                "url": "https://docs.aws.amazon.com/appflow/1.0/APIReference/API_DescribeConnectorEntity.html",
                "description": "Grants permission to describe all fields for an object in a login profile configured in Amazon AppFlow",
                "accessLevel": "Read",
                "resourceTypes": {
                    "connectorprofile": {
                        "required": true
                    }
                }
            },
            "DescribeConnectorFields": {
                "url": "https://docs.aws.amazon.com/appflow/latest/userguide/identity-access-management.html#appflow-api-actions",
                "description": "Grants permission to describe all fields for an object in a login profile configured in Amazon AppFlow (Console Only)",
                "accessLevel": "Read",
                "resourceTypes": {
                    "connectorprofile": {
                        "required": true
                    }
                }
            },
            "DescribeConnectorProfiles": {
                "url": "https://docs.aws.amazon.com/appflow/1.0/APIReference/API_DescribeConnectorProfiles.html",
                "description": "Grants permission to describe all login profiles configured in Amazon AppFlow",
                "accessLevel": "Read"
            },
            "DescribeConnectors": {
                "url": "https://docs.aws.amazon.com/appflow/1.0/APIReference/API_DescribeConnectors.html",
                "description": "Grants permission to describe all connectors supported by Amazon AppFlow",
                "accessLevel": "Read"
            },
            "DescribeFlow": {
                "url": "https://docs.aws.amazon.com/appflow/1.0/APIReference/API_DescribeFlow.html",
                "description": "Grants permission to describe a specific flow configured in Amazon AppFlow",
                "accessLevel": "Read"
            },
            "DescribeFlowExecution": {
                "url": "https://docs.aws.amazon.com/appflow/latest/userguide/identity-access-management.html#appflow-api-actions",
                "description": "Grants permission to describe all flow executions for a flow configured in Amazon AppFlow (Console Only)",
                "accessLevel": "Read",
                "resourceTypes": {
                    "flow": {
                        "required": true
                    }
                }
            },
            "DescribeFlowExecutionRecords": {
                "url": "https://docs.aws.amazon.com/appflow/1.0/APIReference/API_DescribeFlowExecutionRecords.html",
                "description": "Grants permission to describe all flow executions for a flow configured in Amazon AppFlow",
                "accessLevel": "Read",
                "resourceTypes": {
                    "flow": {
                        "required": true
                    }
                }
            },
            "DescribeFlows": {
                "url": "https://docs.aws.amazon.com/appflow/latest/userguide/identity-access-management.html#appflow-api-actions",
                "description": "Grants permission to describe all flows configured in Amazon AppFlow (Console Only)",
                "accessLevel": "Read"
            },
            "ListConnectorEntities": {
                "url": "https://docs.aws.amazon.com/appflow/1.0/APIReference/API_ListConnectorEntities.html",
                "description": "Grants permission to list all objects for a login profile configured in Amazon AppFlow",
                "accessLevel": "List",
                "resourceTypes": {
                    "connectorprofile": {
                        "required": true
                    }
                }
            },
            "ListConnectorFields": {
                "url": "https://docs.aws.amazon.com/appflow/latest/userguide/identity-access-management.html#appflow-api-actions",
                "description": "Grants permission to list all objects for a login profile configured in Amazon AppFlow (Console Only)",
                "accessLevel": "Read",
                "resourceTypes": {
                    "connectorprofile": {
                        "required": true
                    }
                }
            },
            "ListFlows": {
                "url": "https://docs.aws.amazon.com/appflow/1.0/APIReference/API_ListFlows.html",
                "description": "Grants permission to list all flows configured in Amazon AppFlow",
                "accessLevel": "List",
                "resourceTypes": {
                    "flow": {
                        "required": true
                    }
                }
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/appflow/1.0/APIReference/API_ListTagsForResource.html",
                "description": "Grants permission to list tags for a flow",
                "accessLevel": "List",
                "resourceTypes": {
                    "flow": {
                        "required": true
                    }
                }
            },
            "RunFlow": {
                "url": "https://docs.aws.amazon.com/appflow/latest/userguide/identity-access-management.html#appflow-api-actions",
                "description": "Grants permission to run a flow configured in Amazon AppFlow (Console Only)",
                "accessLevel": "Write",
                "resourceTypes": {
                    "flow": {
                        "required": true
                    }
                }
            },
            "StartFlow": {
                "url": "https://docs.aws.amazon.com/appflow/1.0/APIReference/API_StartFlow.html",
                "description": "Grants permission to activate (for scheduled and event-triggered flows) or run (for on-demand flows) a flow configured in Amazon AppFlow",
                "accessLevel": "Write",
                "resourceTypes": {
                    "flow": {
                        "required": true
                    }
                }
            },
            "StopFlow": {
                "url": "https://docs.aws.amazon.com/appflow/1.0/APIReference/API_StopFlow.html",
                "description": "Grants permission to deactivate a scheduled or event-triggered flow configured in Amazon AppFlow",
                "accessLevel": "Write",
                "resourceTypes": {
                    "flow": {
                        "required": true
                    }
                }
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/appflow/1.0/APIReference/API_TagResource.html",
                "description": "Grants permission to tag a flow",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "flow": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/appflow/1.0/APIReference/API_UntagResource.html",
                "description": "Grants permission to untag a flow",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "flow": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "UpdateConnectorProfile": {
                "url": "https://docs.aws.amazon.com/appflow/1.0/APIReference/API_UpdateConnectorProfile.html",
                "description": "Grants permission to update a login profile configured in Amazon AppFlow",
                "accessLevel": "Write",
                "resourceTypes": {
                    "flow": {
                        "required": true
                    }
                }
            },
            "UpdateFlow": {
                "url": "https://docs.aws.amazon.com/appflow/1.0/APIReference/API_UpdateFlow.html",
                "description": "Grants permission to update a flow configured in Amazon AppFlow",
                "accessLevel": "Write",
                "resourceTypes": {
                    "flow": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "connectorprofile": {
                "name": "connectorprofile",
                "url": "https://docs.aws.amazon.com/appflow/1.0/APIReference/API_ConnectorProfile.html",
                "arn": "arn:${Partition}:appflow:${Region}:${Account}:connectorprofile/${ProfileName}",
                "conditionKeys": []
            },
            "flow": {
                "name": "flow",
                "url": "https://docs.aws.amazon.com/appflow/1.0/APIReference/API_FlowDefinition.html",
                "arn": "arn:${Partition}:appflow:${Region}:${Account}:flow/${FlowName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Grants permission to create a login profile to be used with Amazon AppFlow flows
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appflow/1.0/APIReference/API_CreateConnectorProfile.html
     */
    toCreateConnectorProfile() {
        this.add('appflow:CreateConnectorProfile');
        return this;
    }
    /**
     * Grants permission to create an Amazon AppFlow flow
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/appflow/1.0/APIReference/API_CreateFlow.html
     */
    toCreateFlow() {
        this.add('appflow:CreateFlow');
        return this;
    }
    /**
     * Grants permission to delete a login profile configured in Amazon AppFlow
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appflow/1.0/APIReference/API_DescribeConnectorProfiles.html
     */
    toDeleteConnectorProfile() {
        this.add('appflow:DeleteConnectorProfile');
        return this;
    }
    /**
     * Grants permission to delete an Amazon AppFlow flow
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/appflow/1.0/APIReference/API_DeleteFlow.html
     */
    toDeleteFlow() {
        this.add('appflow:DeleteFlow');
        return this;
    }
    /**
     * Grants permission to describe all fields for an object in a login profile configured in Amazon AppFlow
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appflow/1.0/APIReference/API_DescribeConnectorEntity.html
     */
    toDescribeConnectorEntity() {
        this.add('appflow:DescribeConnectorEntity');
        return this;
    }
    /**
     * Grants permission to describe all fields for an object in a login profile configured in Amazon AppFlow (Console Only)
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appflow/latest/userguide/identity-access-management.html#appflow-api-actions
     */
    toDescribeConnectorFields() {
        this.add('appflow:DescribeConnectorFields');
        return this;
    }
    /**
     * Grants permission to describe all login profiles configured in Amazon AppFlow
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appflow/1.0/APIReference/API_DescribeConnectorProfiles.html
     */
    toDescribeConnectorProfiles() {
        this.add('appflow:DescribeConnectorProfiles');
        return this;
    }
    /**
     * Grants permission to describe all connectors supported by Amazon AppFlow
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appflow/1.0/APIReference/API_DescribeConnectors.html
     */
    toDescribeConnectors() {
        this.add('appflow:DescribeConnectors');
        return this;
    }
    /**
     * Grants permission to describe a specific flow configured in Amazon AppFlow
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appflow/1.0/APIReference/API_DescribeFlow.html
     */
    toDescribeFlow() {
        this.add('appflow:DescribeFlow');
        return this;
    }
    /**
     * Grants permission to describe all flow executions for a flow configured in Amazon AppFlow (Console Only)
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appflow/latest/userguide/identity-access-management.html#appflow-api-actions
     */
    toDescribeFlowExecution() {
        this.add('appflow:DescribeFlowExecution');
        return this;
    }
    /**
     * Grants permission to describe all flow executions for a flow configured in Amazon AppFlow
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appflow/1.0/APIReference/API_DescribeFlowExecutionRecords.html
     */
    toDescribeFlowExecutionRecords() {
        this.add('appflow:DescribeFlowExecutionRecords');
        return this;
    }
    /**
     * Grants permission to describe all flows configured in Amazon AppFlow (Console Only)
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appflow/latest/userguide/identity-access-management.html#appflow-api-actions
     */
    toDescribeFlows() {
        this.add('appflow:DescribeFlows');
        return this;
    }
    /**
     * Grants permission to list all objects for a login profile configured in Amazon AppFlow
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/appflow/1.0/APIReference/API_ListConnectorEntities.html
     */
    toListConnectorEntities() {
        this.add('appflow:ListConnectorEntities');
        return this;
    }
    /**
     * Grants permission to list all objects for a login profile configured in Amazon AppFlow (Console Only)
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appflow/latest/userguide/identity-access-management.html#appflow-api-actions
     */
    toListConnectorFields() {
        this.add('appflow:ListConnectorFields');
        return this;
    }
    /**
     * Grants permission to list all flows configured in Amazon AppFlow
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/appflow/1.0/APIReference/API_ListFlows.html
     */
    toListFlows() {
        this.add('appflow:ListFlows');
        return this;
    }
    /**
     * Grants permission to list tags for a flow
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/appflow/1.0/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.add('appflow:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to run a flow configured in Amazon AppFlow (Console Only)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appflow/latest/userguide/identity-access-management.html#appflow-api-actions
     */
    toRunFlow() {
        this.add('appflow:RunFlow');
        return this;
    }
    /**
     * Grants permission to activate (for scheduled and event-triggered flows) or run (for on-demand flows) a flow configured in Amazon AppFlow
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appflow/1.0/APIReference/API_StartFlow.html
     */
    toStartFlow() {
        this.add('appflow:StartFlow');
        return this;
    }
    /**
     * Grants permission to deactivate a scheduled or event-triggered flow configured in Amazon AppFlow
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appflow/1.0/APIReference/API_StopFlow.html
     */
    toStopFlow() {
        this.add('appflow:StopFlow');
        return this;
    }
    /**
     * Grants permission to tag a flow
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/appflow/1.0/APIReference/API_TagResource.html
     */
    toTagResource() {
        this.add('appflow:TagResource');
        return this;
    }
    /**
     * Grants permission to untag a flow
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/appflow/1.0/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        this.add('appflow:UntagResource');
        return this;
    }
    /**
     * Grants permission to update a login profile configured in Amazon AppFlow
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appflow/1.0/APIReference/API_UpdateConnectorProfile.html
     */
    toUpdateConnectorProfile() {
        this.add('appflow:UpdateConnectorProfile');
        return this;
    }
    /**
     * Grants permission to update a flow configured in Amazon AppFlow
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appflow/1.0/APIReference/API_UpdateFlow.html
     */
    toUpdateFlow() {
        this.add('appflow:UpdateFlow');
        return this;
    }
    /**
     * Adds a resource of type connectorprofile to the statement
     *
     * https://docs.aws.amazon.com/appflow/1.0/APIReference/API_ConnectorProfile.html
     *
     * @param profileName - Identifier for the profileName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onConnectorprofile(profileName, account, region, partition) {
        var arn = 'arn:${Partition}:appflow:${Region}:${Account}:connectorprofile/${ProfileName}';
        arn = arn.replace('${ProfileName}', profileName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type flow to the statement
     *
     * https://docs.aws.amazon.com/appflow/1.0/APIReference/API_FlowDefinition.html
     *
     * @param flowName - Identifier for the flowName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onFlow(flowName, account, region, partition) {
        var arn = 'arn:${Partition}:appflow:${Region}:${Account}:flow/${FlowName}';
        arn = arn.replace('${FlowName}', flowName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Appflow = Appflow;
//# sourceMappingURL=data:application/json;base64,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