"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Chime = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [chime](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonchime.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Chime extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [chime](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonchime.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'chime';
        this.actionList = {
            "AcceptDelegate": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/control-access.html",
                "description": "Grants permission to accept the delegate invitation to share management of an Amazon Chime account with another AWS Account",
                "accessLevel": "Write"
            },
            "ActivateUsers": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/manage-access.html",
                "description": "Grants permission to activate users in an Amazon Chime Enterprise account",
                "accessLevel": "Write"
            },
            "AddDomain": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/claim-domain.html",
                "description": "Grants permission to add a domain to your Amazon Chime account",
                "accessLevel": "Write"
            },
            "AddOrUpdateGroups": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/manage-chime-account.html",
                "description": "Grants permission to add new or update existing Active Directory or Okta user groups associated with your Amazon Chime Enterprise account",
                "accessLevel": "Write"
            },
            "AssociatePhoneNumberWithUser": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_AssociatePhoneNumberWithUser.html",
                "description": "Grants permission to associate a phone number with an Amazon Chime user",
                "accessLevel": "Write"
            },
            "AssociatePhoneNumbersWithVoiceConnector": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_AssociatePhoneNumbersWithVoiceConnector.html",
                "description": "Grants permission to associate multiple phone numbers with an Amazon Chime Voice Connector",
                "accessLevel": "Write"
            },
            "AssociatePhoneNumbersWithVoiceConnectorGroup": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_AssociatePhoneNumbersWithVoiceConnectorGroup.html",
                "description": "Grants permission to associate multiple phone numbers with an Amazon Chime Voice Connector Group",
                "accessLevel": "Write"
            },
            "AssociateSigninDelegateGroupsWithAccount": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_AssociateSigninDelegateGroupsWithAccount.html",
                "description": "Grants permission to associate the specified sign-in delegate groups with the specified Amazon Chime account.",
                "accessLevel": "Write"
            },
            "AuthorizeDirectory": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/control-access.html",
                "description": "Grants permission to authorize an Active Directory for your Amazon Chime Enterprise account",
                "accessLevel": "Write"
            },
            "BatchCreateAttendee": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_BatchCreateAttendee.html",
                "description": "Grants permission to create new attendees for an active Amazon Chime SDK meeting",
                "accessLevel": "Write",
                "resourceTypes": {
                    "meeting": {
                        "required": true
                    }
                }
            },
            "BatchCreateRoomMembership": {
                "url": "",
                "description": "Grants permission to batch add room members",
                "accessLevel": "Write"
            },
            "BatchDeletePhoneNumber": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_BatchDeletePhoneNumber.html",
                "description": "Grants permission to move up to 50 phone numbers to the deletion queue",
                "accessLevel": "Write"
            },
            "BatchSuspendUser": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_BatchSuspendUser.html",
                "description": "Grants permission to suspend up to 50 users from a Team or EnterpriseLWA Amazon Chime account",
                "accessLevel": "Write"
            },
            "BatchUnsuspendUser": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_BatchUnsuspendUser.html",
                "description": "Grants permission to remove the suspension from up to 50 previously suspended users for the specified Amazon Chime EnterpriseLWA account",
                "accessLevel": "Write"
            },
            "BatchUpdatePhoneNumber": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_BatchUpdatePhoneNumber.html",
                "description": "Grants permission to update phone number details within the UpdatePhoneNumberRequestItem object for up to 50 phone numbers",
                "accessLevel": "Write"
            },
            "BatchUpdateUser": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_BatchUpdateUser.html",
                "description": "Grants permission to update user details within the UpdateUserRequestItem object for up to 20 users for the specified Amazon Chime account",
                "accessLevel": "Write"
            },
            "ConnectDirectory": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/active_directory.html",
                "description": "Grants permission to connect an Active Directory to your Amazon Chime Enterprise account",
                "accessLevel": "Write",
                "dependentActions": [
                    "ds:ConnectDirectory"
                ]
            },
            "CreateAccount": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_CreateAccount.html",
                "description": "Grants permission to create an Amazon Chime account under the administrator's AWS account",
                "accessLevel": "Write"
            },
            "CreateApiKey": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/okta_sso.html",
                "description": "Grants permission to create a new SCIM access key for your Amazon Chime account and Okta configuration",
                "accessLevel": "Write"
            },
            "CreateAttendee": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_CreateAttendee.html",
                "description": "Grants permission to create a new attendee for an active Amazon Chime SDK meeting",
                "accessLevel": "Write",
                "resourceTypes": {
                    "meeting": {
                        "required": true
                    }
                }
            },
            "CreateBot": {
                "url": "",
                "description": "Grants permission to create a bot for an Amazon Chime Enterprise account",
                "accessLevel": "Write"
            },
            "CreateBotMembership": {
                "url": "",
                "description": "Grants permission to add a bot to a chat room in your Amazon Chime Enterprise account",
                "accessLevel": "Write"
            },
            "CreateCDRBucket": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/manage-access.html",
                "description": "Grants permission to create a new Call Detail Record S3 bucket",
                "accessLevel": "Write",
                "dependentActions": [
                    "s3:CreateBucket",
                    "s3:ListAllMyBuckets"
                ]
            },
            "CreateMeeting": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_CreateMeeting.html",
                "description": "Grants permission to create a new Amazon Chime SDK meeting in the specified media Region, with no initial attendees",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateMeetingWithAttendees": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_CreateMeetingWithAttendees.html",
                "description": "Grants permission to create a new Amazon Chime SDK meeting in the specified media Region, with a set of attendees",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreatePhoneNumberOrder": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_CreatePhoneNumberOrder.html",
                "description": "Grants permission to create a phone number order with the Carriers",
                "accessLevel": "Write"
            },
            "CreateProxySession": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_CreateProxySession.html",
                "description": "Grants permission to create a proxy session for the specified Amazon Chime Voice Connector",
                "accessLevel": "Write"
            },
            "CreateRoom": {
                "url": "",
                "description": "Grants permission to create a room",
                "accessLevel": "Write"
            },
            "CreateRoomMembership": {
                "url": "",
                "description": "Grants permission to add a room member",
                "accessLevel": "Write"
            },
            "CreateUser": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_CreateUser.html",
                "description": "Grants permission to create a user under the specified Amazon Chime account.",
                "accessLevel": "Write"
            },
            "CreateVoiceConnector": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_CreateVoiceConnector.html",
                "description": "Grants permission to create a Amazon Chime Voice Connector under the administrator's AWS account",
                "accessLevel": "Write"
            },
            "CreateVoiceConnectorGroup": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_CreateVoiceConnectorGroup.html",
                "description": "Grants permission to create a Amazon Chime Voice Connector Group under the administrator's AWS account",
                "accessLevel": "Write"
            },
            "DeleteAccount": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteAccount.html",
                "description": "Grants permission to delete the specified Amazon Chime account",
                "accessLevel": "Write"
            },
            "DeleteAccountOpenIdConfig": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/okta_sso.html",
                "description": "Grants permission to delete the OpenIdConfig attributes from your Amazon Chime account",
                "accessLevel": "Write"
            },
            "DeleteApiKey": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/okta_sso.html",
                "description": "Grants permission to delete the specified SCIM access key associated with your Amazon Chime account and Okta configuration",
                "accessLevel": "Write"
            },
            "DeleteAttendee": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteAttendee.html",
                "description": "Grants permission to delete the specified attendee from an Amazon Chime SDK meeting",
                "accessLevel": "Write",
                "resourceTypes": {
                    "meeting": {
                        "required": true
                    }
                }
            },
            "DeleteCDRBucket": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/control-access.html",
                "description": "Grants permission to delete a Call Detail Record S3 bucket from your Amazon Chime account",
                "accessLevel": "Write",
                "dependentActions": [
                    "s3:DeleteBucket"
                ]
            },
            "DeleteDelegate": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/control-access.html",
                "description": "Grants permission to delete delegated AWS account management from your Amazon Chime account",
                "accessLevel": "Write"
            },
            "DeleteDomain": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/claim-domain.html",
                "description": "Grants permission to delete a domain from your Amazon Chime account",
                "accessLevel": "Write"
            },
            "DeleteEventsConfiguration": {
                "url": "",
                "description": "Grants permission to delete an events configuration for a bot to receive outgoing events",
                "accessLevel": "Write"
            },
            "DeleteGroups": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/control-access.html",
                "description": "Grants permission to delete Active Directory or Okta user groups from your Amazon Chime Enterprise account",
                "accessLevel": "Write"
            },
            "DeleteMeeting": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteMeeting.html",
                "description": "Grants permission to delete the specified Amazon Chime SDK meeting",
                "accessLevel": "Write",
                "resourceTypes": {
                    "meeting": {
                        "required": true
                    }
                }
            },
            "DeletePhoneNumber": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_DeletePhoneNumber.html",
                "description": "Grants permission to move a phone number to the deletion queue",
                "accessLevel": "Write"
            },
            "DeleteProxySession": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteProxySession.html",
                "description": "Grants permission to delete a proxy session for the specified Amazon Chime Voice Connector",
                "accessLevel": "Write"
            },
            "DeleteRoom": {
                "url": "",
                "description": "Grants permission to delete a room",
                "accessLevel": "Write"
            },
            "DeleteRoomMembership": {
                "url": "",
                "description": "Grants permission to remove a room member",
                "accessLevel": "Write"
            },
            "DeleteVoiceConnector": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteVoiceConnector.html",
                "description": "Grants permission to delete the specified Amazon Chime Voice Connector",
                "accessLevel": "Write"
            },
            "DeleteVoiceConnectorEmergencyCallingConfiguration": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteVoiceConnectorEmergencyCallingConfiguration.html",
                "description": "Grants permission to delete emergency calling configuration for the specified Amazon Chime Voice Connector",
                "accessLevel": "Write"
            },
            "DeleteVoiceConnectorGroup": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteVoiceConnectorGroup.html",
                "description": "Grants permission to delete the specified Amazon Chime Voice Connector Group",
                "accessLevel": "Write"
            },
            "DeleteVoiceConnectorOrigination": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteVoiceConnectorOrigination.html",
                "description": "Grants permission to delete the origination settings for the specified Amazon Chime Voice Connector",
                "accessLevel": "Write"
            },
            "DeleteVoiceConnectorProxy": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteVoiceConnectorProxy.html",
                "description": "Grants permission to delete proxy configuration for the specified Amazon Chime Voice Connector",
                "accessLevel": "Write"
            },
            "DeleteVoiceConnectorStreamingConfiguration": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteVoiceConnectorStreamingConfiguration.html",
                "description": "Grants permission to delete streaming configuration for the specified Amazon Chime Voice Connector",
                "accessLevel": "Write"
            },
            "DeleteVoiceConnectorTermination": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteVoiceConnectorTermination.html",
                "description": "Grants permission to delete the termination settings for the specified Amazon Chime Voice Connector",
                "accessLevel": "Write"
            },
            "DeleteVoiceConnectorTerminationCredentials": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteVoiceConnectorTerminationCredentials.html",
                "description": "Grants permission to delete SIP termination credentials for the specified Amazon Chime Voice Connector",
                "accessLevel": "Write"
            },
            "DisassociatePhoneNumberFromUser": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_DisassociatePhoneNumberFromUser.html",
                "description": "Grants permission to disassociate the primary provisioned number from the specified Amazon Chime user",
                "accessLevel": "Write"
            },
            "DisassociatePhoneNumbersFromVoiceConnector": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_DisassociatePhoneNumbersFromVoiceConnector.html",
                "description": "Grants permission to disassociate multiple phone numbers from the specified Amazon Chime Voice Connector",
                "accessLevel": "Write"
            },
            "DisassociatePhoneNumbersFromVoiceConnectorGroup": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_DisassociatePhoneNumbersFromVoiceConnectorGroup.html",
                "description": "Grants permission to disassociate multiple phone numbers from the specified Amazon Chime Voice Connector Group",
                "accessLevel": "Write"
            },
            "DisassociateSigninDelegateGroupsFromAccount": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_DisassociateSigninDelegateGroupsFromAccount.html",
                "description": "Grants permission to disassociate the specified sign-in delegate groups from the specified Amazon Chime account.",
                "accessLevel": "Write"
            },
            "DisconnectDirectory": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/control-access.html",
                "description": "Grants permission to disconnect the Active Directory from your Amazon Chime Enterprise account",
                "accessLevel": "Write"
            },
            "GetAccount": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_GetAccount.html",
                "description": "Grants permission to get details for the specified Amazon Chime account",
                "accessLevel": "Read"
            },
            "GetAccountResource": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/control-access.html",
                "description": "Grants permission to get details for the account resource associated with your Amazon Chime account",
                "accessLevel": "Read"
            },
            "GetAccountSettings": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_GetAccountSettings.html",
                "description": "Grants permission to get account settings for the specified Amazon Chime account ID",
                "accessLevel": "Read"
            },
            "GetAccountWithOpenIdConfig": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/okta_sso.html",
                "description": "Grants permission to get the account details and OpenIdConfig attributes for your Amazon Chime account",
                "accessLevel": "Read"
            },
            "GetAttendee": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_GetAttendee.html",
                "description": "Grants permission to get attendee details for a specified meeting ID and attendee ID",
                "accessLevel": "Read",
                "resourceTypes": {
                    "meeting": {
                        "required": true
                    }
                }
            },
            "GetBot": {
                "url": "",
                "description": "Grants permission to retrieve details for the specified bot",
                "accessLevel": "Read"
            },
            "GetCDRBucket": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/control-access.html",
                "description": "Grants permission to get details of a Call Detail Record S3 bucket associated with your Amazon Chime account",
                "accessLevel": "Read",
                "dependentActions": [
                    "s3:GetBucketAcl",
                    "s3:GetBucketLocation",
                    "s3:GetBucketLogging",
                    "s3:GetBucketVersioning",
                    "s3:GetBucketWebsite"
                ]
            },
            "GetDomain": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/claim-domain.html",
                "description": "Grants permission to get domain details for a domain associated with your Amazon Chime account",
                "accessLevel": "Read"
            },
            "GetEventsConfiguration": {
                "url": "",
                "description": "Grants permission to retrieve details for an events configuration for a bot to receive outgoing events",
                "accessLevel": "Read"
            },
            "GetGlobalSettings": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_GetGlobalSettings.html",
                "description": "Grants permission to get global settings related to Amazon Chime for the AWS account",
                "accessLevel": "Read"
            },
            "GetMeeting": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_GetMeeting.html",
                "description": "Grants permission to get the meeting record for a specified meeting ID",
                "accessLevel": "Read",
                "resourceTypes": {
                    "meeting": {
                        "required": true
                    }
                }
            },
            "GetMeetingDetail": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/control-access.html",
                "description": "Grants permission to get attendee, connection, and other details for a meeting",
                "accessLevel": "Read"
            },
            "GetPhoneNumber": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_GetPhoneNumber.html",
                "description": "Grants permission to get details for the specified phone number",
                "accessLevel": "Read"
            },
            "GetPhoneNumberOrder": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_GetPhoneNumberOrder.html",
                "description": "Grants permission to get details for the specified phone number order",
                "accessLevel": "Read"
            },
            "GetPhoneNumberSettings": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_GetPhoneNumberSettings.html",
                "description": "Grants permission to get phone number settings related to Amazon Chime for the AWS account",
                "accessLevel": "Read"
            },
            "GetProxySession": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_GetProxySession.html",
                "description": "Grants permission to get details of the specified proxy session for the specified Amazon Chime Voice Connector",
                "accessLevel": "Read"
            },
            "GetRetentionSettings": {
                "url": "",
                "description": "Gets the retention settings for the specified Amazon Chime account.",
                "accessLevel": "Read"
            },
            "GetRoom": {
                "url": "",
                "description": "Grants permission to retrieve a room",
                "accessLevel": "Read"
            },
            "GetTelephonyLimits": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/phone-numbers.html",
                "description": "Grants permission to get telephony limits for the AWS account",
                "accessLevel": "Read"
            },
            "GetUser": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_GetUser.html",
                "description": "Grants permission to get details for the specified user ID",
                "accessLevel": "Read"
            },
            "GetUserActivityReportData": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/user-details.html",
                "description": "Grants permission to get a summary of user activity on the user details page",
                "accessLevel": "Read"
            },
            "GetUserByEmail": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/user-details.html",
                "description": "Grants permission to get user details for an Amazon Chime user based on the email address in an Amazon Chime Enterprise or Team account",
                "accessLevel": "Read"
            },
            "GetUserSettings": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_GetUserSettings.html",
                "description": "Grants permission to get user settings related to the specified Amazon Chime user",
                "accessLevel": "Read"
            },
            "GetVoiceConnector": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_GetVoiceConnector.html",
                "description": "Grants permission to get details for the specified Amazon Chime Voice Connector",
                "accessLevel": "Read"
            },
            "GetVoiceConnectorEmergencyCallingConfiguration": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_GetVoiceConnectorEmergencyCallingConfiguration.html",
                "description": "Grants permission to get details of the emergency calling configuration for the specified Amazon Chime Voice Connector",
                "accessLevel": "Read"
            },
            "GetVoiceConnectorGroup": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_GetVoiceConnectorGroup.html",
                "description": "Grants permission to get details for the specified Amazon Chime Voice Connector Group",
                "accessLevel": "Read"
            },
            "GetVoiceConnectorLoggingConfiguration": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_GetVoiceConnectorLoggingConfiguration.html",
                "description": "Grants permission to get details of the logging configuration for the specified Amazon Chime Voice Connector",
                "accessLevel": "Read"
            },
            "GetVoiceConnectorOrigination": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_GetVoiceConnectorOrigination.html",
                "description": "Grants permission to get details of the origination settings for the specified Amazon Chime Voice Connector",
                "accessLevel": "Read"
            },
            "GetVoiceConnectorProxy": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_GetVoiceConnectorProxy.html",
                "description": "Grants permission to get details of the proxy configuration for the specified Amazon Chime Voice Connector",
                "accessLevel": "Read"
            },
            "GetVoiceConnectorStreamingConfiguration": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_GetVoiceConnectorStreamingConfiguration.html",
                "description": "Grants permission to get details of the streaming configuration for the specified Amazon Chime Voice Connector",
                "accessLevel": "Read"
            },
            "GetVoiceConnectorTermination": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_GetVoiceConnectorTermination.html",
                "description": "Grants permission to get details of the termination settings for the specified Amazon Chime Voice Connector",
                "accessLevel": "Read"
            },
            "GetVoiceConnectorTerminationHealth": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_GetVoiceConnectorTerminationHealth.html",
                "description": "Grants permission to get details of the termination health for the specified Amazon Chime Voice Connector",
                "accessLevel": "Read"
            },
            "InviteDelegate": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/control-access.html",
                "description": "Grants permission to send an invitation to accept a request for AWS account delegation for an Amazon Chime account",
                "accessLevel": "Write"
            },
            "InviteUsers": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_InviteUsers.html",
                "description": "Grants permission to invite as many as 50 users to the specified Amazon Chime account",
                "accessLevel": "Write"
            },
            "InviteUsersFromProvider": {
                "url": "",
                "description": "Grants permission to invite users from a third party provider to your Amazon Chime account",
                "accessLevel": "Write"
            },
            "ListAccountUsageReportData": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/view-reports.html",
                "description": "Grants permission to list Amazon Chime account usage reporting data",
                "accessLevel": "List"
            },
            "ListAccounts": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_ListAccounts.html",
                "description": "Grants permission to list the Amazon Chime accounts under the administrator's AWS account",
                "accessLevel": "List"
            },
            "ListApiKeys": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/okta_sso.html",
                "description": "Grants permission to list the SCIM access keys defined for your Amazon Chime account and Okta configuration",
                "accessLevel": "List"
            },
            "ListAttendeeTags": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_ListAttendeeTags.html",
                "description": "Grants permission to list the tags applied to an Amazon Chime SDK attendee resource",
                "accessLevel": "Read",
                "resourceTypes": {
                    "meeting": {
                        "required": true
                    }
                }
            },
            "ListAttendees": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_ListAttendees.html",
                "description": "Grants permission to list up to 100 attendees for a specified Amazon Chime SDK meeting",
                "accessLevel": "Read",
                "resourceTypes": {
                    "meeting": {
                        "required": true
                    }
                }
            },
            "ListBots": {
                "url": "",
                "description": "Grants permission to list the bots associated with the administrator's Amazon Chime Enterprise account",
                "accessLevel": "List"
            },
            "ListCDRBucket": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/control-access.html",
                "description": "Grants permission to list Call Detail Record S3 buckets",
                "accessLevel": "List",
                "dependentActions": [
                    "s3:ListAllMyBuckets",
                    "s3:ListBucket"
                ]
            },
            "ListCallingRegions": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/phone-numbers.html",
                "description": "Grants permission to list the calling regions available for the administrator's AWS account",
                "accessLevel": "List"
            },
            "ListDelegates": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/control-access.html",
                "description": "Grants permission to list account delegate information associated with your Amazon Chime account",
                "accessLevel": "List"
            },
            "ListDirectories": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/control-access.html",
                "description": "Grants permission to list active Active Directories hosted in the Directory Service of your AWS account",
                "accessLevel": "List"
            },
            "ListDomains": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/claim-domain.html",
                "description": "Grants permission to list domains associated with your Amazon Chime account",
                "accessLevel": "List"
            },
            "ListGroups": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/control-access.html",
                "description": "Grants permission to list Active Directory or Okta user groups associated with your Amazon Chime Enterprise account",
                "accessLevel": "List"
            },
            "ListMeetingEvents": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/view-reports.html",
                "description": "Grants permission to list all events that occurred for a specified meeting",
                "accessLevel": "List"
            },
            "ListMeetingTags": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_ListMeetingTags.html",
                "description": "Grants permission to list the tags applied to an Amazon Chime SDK meeting resource.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "meeting": {
                        "required": true
                    }
                }
            },
            "ListMeetings": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_ListMeetings.html",
                "description": "Grants permission to list up to 100 active Amazon Chime SDK meetings",
                "accessLevel": "Read"
            },
            "ListMeetingsReportData": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/view-reports.html",
                "description": "Grants permission to list meetings ended during the specified date range",
                "accessLevel": "List"
            },
            "ListPhoneNumberOrders": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_ListPhoneNumberOrders.html",
                "description": "Grants permission to list the phone number orders under the administrator's AWS account",
                "accessLevel": "List"
            },
            "ListPhoneNumbers": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_ListPhoneNumbers.html",
                "description": "Grants permission to list the phone numbers under the administrator's AWS account",
                "accessLevel": "List"
            },
            "ListProxySessions": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_ListProxySessions.html",
                "description": "Grants permission to list proxy sessions for the specified Amazon Chime Voice Connector",
                "accessLevel": "List"
            },
            "ListRoomMemberships": {
                "url": "",
                "description": "Grants permission to list all room members",
                "accessLevel": "Read"
            },
            "ListRooms": {
                "url": "",
                "description": "Grants permission to list rooms",
                "accessLevel": "Read"
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_ListTagsForResource.html",
                "description": "Grants permission to list the tags applied to an Amazon Chime SDK meeting resource.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "meeting": {
                        "required": false
                    }
                }
            },
            "ListUsers": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_ListUsers.html",
                "description": "Grants permission to list the users that belong to the specified Amazon Chime account",
                "accessLevel": "List"
            },
            "ListVoiceConnectorGroups": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_ListVoiceConnectorGroups.html",
                "description": "Grants permission to list the Amazon Chime Voice Connector Groups under the administrator's AWS account",
                "accessLevel": "List"
            },
            "ListVoiceConnectorTerminationCredentials": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_ListVoiceConnectorTerminationCredentials.html",
                "description": "Grants permission to list the SIP termination credentials for the specified Amazon Chime Voice Connector",
                "accessLevel": "List"
            },
            "ListVoiceConnectors": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_ListVoiceConnectors.html",
                "description": "Grants permission to list the Amazon Chime Voice Connectors under the administrator's AWS account",
                "accessLevel": "List"
            },
            "LogoutUser": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_LogoutUser.html",
                "description": "Grants permission to log out the specified user from all of the devices they are currently logged into",
                "accessLevel": "Write"
            },
            "PutEventsConfiguration": {
                "url": "",
                "description": "Grants permission to update details for an events configuration for a bot to receive outgoing events",
                "accessLevel": "Write"
            },
            "PutRetentionSettings": {
                "url": "",
                "description": "Puts retention settings for the specified Amazon Chime account",
                "accessLevel": "Write"
            },
            "PutVoiceConnectorEmergencyCallingConfiguration": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_PutVoiceConnectorEmergencyCallingConfiguration.html",
                "description": "Grants permission to add emergency calling configuration for the specified Amazon Chime Voice Connector",
                "accessLevel": "Write"
            },
            "PutVoiceConnectorLoggingConfiguration": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_PutVoiceConnectorLoggingConfiguration.html",
                "description": "Grants permission to add logging configuration for the specified Amazon Chime Voice Connector",
                "accessLevel": "Write",
                "dependentActions": [
                    "logs:CreateLogDelivery",
                    "logs:CreateLogGroup",
                    "logs:DeleteLogDelivery",
                    "logs:DescribeLogGroups",
                    "logs:GetLogDelivery",
                    "logs:ListLogDeliveries"
                ]
            },
            "PutVoiceConnectorOrigination": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_PutVoiceConnectorOrigination.html",
                "description": "Grants permission to update the origination settings for the specified Amazon Chime Voice Connector",
                "accessLevel": "Write"
            },
            "PutVoiceConnectorProxy": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_PutVoiceConnectorProxy.html",
                "description": "Grants permission to add proxy configuration for the specified Amazon Chime Voice Connector",
                "accessLevel": "Write"
            },
            "PutVoiceConnectorStreamingConfiguration": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_PutVoiceConnectorStreamingConfiguration.html",
                "description": "Grants permission to add streaming configuration for the specified Amazon Chime Voice Connector",
                "accessLevel": "Write"
            },
            "PutVoiceConnectorTermination": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_PutVoiceConnectorTermination.html",
                "description": "Grants permission to update the termination settings for the specified Amazon Chime Voice Connector",
                "accessLevel": "Write"
            },
            "PutVoiceConnectorTerminationCredentials": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_PutVoiceConnectorTerminationCredentials.html",
                "description": "Grants permission to add SIP termination credentials for the specified Amazon Chime Voice Connector",
                "accessLevel": "Write"
            },
            "RedactConversationMessage": {
                "url": "",
                "description": "Redacts the specified Chime conversation Message",
                "accessLevel": "Write"
            },
            "RedactRoomMessage": {
                "url": "",
                "description": "Redacts the specified Chime room Message",
                "accessLevel": "Write"
            },
            "RegenerateSecurityToken": {
                "url": "",
                "description": "Grants permission to regenerate the security token for the specified bot",
                "accessLevel": "Write"
            },
            "RenameAccount": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/rename-account.html",
                "description": "Grants permission to modify the account name for your Amazon Chime Enterprise or Team account",
                "accessLevel": "Write"
            },
            "RenewDelegate": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/control-access.html",
                "description": "Grants permission to renew the delegation request associated with an Amazon Chime account",
                "accessLevel": "Write"
            },
            "ResetAccountResource": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/control-access.html",
                "description": "Grants permission to reset the account resource in your Amazon Chime account",
                "accessLevel": "Write"
            },
            "ResetPersonalPIN": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_ResetPersonalPIN.html",
                "description": "Grants permission to reset the personal meeting PIN for the specified user on an Amazon Chime account",
                "accessLevel": "Write"
            },
            "RestorePhoneNumber": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_RestorePhoneNumber.html",
                "description": "Grants permission to restore the specified phone number from the deltion queue back to the phone number inventory",
                "accessLevel": "Write"
            },
            "RetrieveDataExports": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/request-attachments.html",
                "description": "Grants permission to download the file containing links to all user attachments returned as part of the \"Request attachments\" action",
                "accessLevel": "List"
            },
            "SearchAvailablePhoneNumbers": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_SearchAvailablePhoneNumbers.html",
                "description": "Grants permission to search phone numbers that can be ordered from the carrier",
                "accessLevel": "Read"
            },
            "StartDataExport": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/request-attachments.html",
                "description": "Grants permission to submit the \"Request attachments\" request",
                "accessLevel": "Write"
            },
            "SubmitSupportRequest": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/chime-getting-admin-support.html",
                "description": "Grants permission to submit a customer service support request",
                "accessLevel": "Write"
            },
            "SuspendUsers": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/manage-access.html",
                "description": "Grants permission to suspend users from an Amazon Chime Enterprise account",
                "accessLevel": "Write"
            },
            "TagAttendee": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_TagAttendee.html",
                "description": "Grants permission to apply the specified tags to the specified Amazon Chime SDK attendee",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "meeting": {
                        "required": true
                    }
                }
            },
            "TagMeeting": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_TagMeeting.html",
                "description": "Grants permission to apply the specified tags to the specified Amazon Chime SDK meeting.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "meeting": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_TagResource.html",
                "description": "Grants permission to apply the specified tags to the specified Amazon Chime SDK meeting resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "meeting": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "UnauthorizeDirectory": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/control-access.html",
                "description": "Grants permission to unauthorize an Active Directory from your Amazon Chime Enterprise account",
                "accessLevel": "Write"
            },
            "UntagAttendee": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_UntagAttendee.html",
                "description": "Grants permission to untag the specified tags from the specified Amazon Chime SDK attendee.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "meeting": {
                        "required": true
                    }
                }
            },
            "UntagMeeting": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_UntagMeeting.html",
                "description": "Grants permission to untag the specified tags from the specified Amazon Chime SDK meeting.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "meeting": {
                        "required": true
                    }
                }
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_UntagResource.html",
                "description": "Grants permission to untag the specified tags from the specified Amazon Chime SDK meeting resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "meeting": {
                        "required": false
                    }
                }
            },
            "UpdateAccount": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdateAccount.html",
                "description": "Grants permission to update account details for the specified Amazon Chime account",
                "accessLevel": "Write"
            },
            "UpdateAccountOpenIdConfig": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/okta_sso.html",
                "description": "Grants permission to update the OpenIdConfig attributes for your Amazon Chime account",
                "accessLevel": "Write"
            },
            "UpdateAccountResource": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/control-access.html",
                "description": "Grants permission to update the account resource in your Amazon Chime account",
                "accessLevel": "Write"
            },
            "UpdateAccountSettings": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdateAccountSettings.html",
                "description": "Grants permission to update the settings for the specified Amazon Chime account",
                "accessLevel": "Write"
            },
            "UpdateBot": {
                "url": "",
                "description": "Grants permission to update the status of the specified bot",
                "accessLevel": "Write"
            },
            "UpdateCDRSettings": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/control-access.html",
                "description": "Grants permission to update your Call Detail Record S3 bucket",
                "accessLevel": "Write",
                "dependentActions": [
                    "s3:CreateBucket",
                    "s3:DeleteBucket",
                    "s3:ListAllMyBuckets"
                ]
            },
            "UpdateGlobalSettings": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdateGlobalSettings.html",
                "description": "Grants permission to update the global settings related to Amazon Chime for the AWS account",
                "accessLevel": "Write"
            },
            "UpdatePhoneNumber": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdatePhoneNumber.html",
                "description": "Grants permission to update phone number details for the specified phone number",
                "accessLevel": "Write"
            },
            "UpdatePhoneNumberSettings": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdatePhoneNumberSettings.html",
                "description": "Grants permission to update phone number settings related to Amazon Chime for the AWS account",
                "accessLevel": "Write"
            },
            "UpdateProxySession": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdateProxySession.html",
                "description": "Grants permission to update a proxy session for the specified Amazon Chime Voice Connector",
                "accessLevel": "Write"
            },
            "UpdateRoom": {
                "url": "",
                "description": "Grants permission to update a room",
                "accessLevel": "Write"
            },
            "UpdateRoomMembership": {
                "url": "",
                "description": "Grants permission to update room membership role",
                "accessLevel": "Write"
            },
            "UpdateSupportedLicenses": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/manage-access.html",
                "description": "Grants permission to update the supported license tiers available for users in your Amazon Chime account",
                "accessLevel": "Write"
            },
            "UpdateUser": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdateUser.html",
                "description": "Grants permission to update user details for a specified user ID",
                "accessLevel": "Write"
            },
            "UpdateUserLicenses": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/manage-access.html",
                "description": "Grants permission to update the licenses for your Amazon Chime users",
                "accessLevel": "Write"
            },
            "UpdateUserSettings": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdateUserSettings.html",
                "description": "Grants permission to update user settings related to the specified Amazon Chime user",
                "accessLevel": "Write"
            },
            "UpdateVoiceConnector": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdateVoiceConnector.html",
                "description": "Grants permission to update Amazon Chime Voice Connector details for the specified Amazon Chime Voice Connector",
                "accessLevel": "Write"
            },
            "UpdateVoiceConnectorGroup": {
                "url": "https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdateVoiceConnectorGroup.html",
                "description": "Grants permission to update Amazon Chime Voice Connector Group details for the specified Amazon Chime Voice Connector Group",
                "accessLevel": "Write"
            },
            "ValidateAccountResource": {
                "url": "https://docs.aws.amazon.com/chime/latest/ag/control-access.html",
                "description": "Grants permission to validate the account resource in your Amazon Chime account",
                "accessLevel": "Read"
            }
        };
        this.resourceTypes = {
            "meeting": {
                "name": "meeting",
                "url": "",
                "arn": "arn:${Partition}:chime::${AccountId}:meeting/${MeetingId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Grants permission to accept the delegate invitation to share management of an Amazon Chime account with another AWS Account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    toAcceptDelegate() {
        this.add('chime:AcceptDelegate');
        return this;
    }
    /**
     * Grants permission to activate users in an Amazon Chime Enterprise account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/manage-access.html
     */
    toActivateUsers() {
        this.add('chime:ActivateUsers');
        return this;
    }
    /**
     * Grants permission to add a domain to your Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/claim-domain.html
     */
    toAddDomain() {
        this.add('chime:AddDomain');
        return this;
    }
    /**
     * Grants permission to add new or update existing Active Directory or Okta user groups associated with your Amazon Chime Enterprise account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/manage-chime-account.html
     */
    toAddOrUpdateGroups() {
        this.add('chime:AddOrUpdateGroups');
        return this;
    }
    /**
     * Grants permission to associate a phone number with an Amazon Chime user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_AssociatePhoneNumberWithUser.html
     */
    toAssociatePhoneNumberWithUser() {
        this.add('chime:AssociatePhoneNumberWithUser');
        return this;
    }
    /**
     * Grants permission to associate multiple phone numbers with an Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_AssociatePhoneNumbersWithVoiceConnector.html
     */
    toAssociatePhoneNumbersWithVoiceConnector() {
        this.add('chime:AssociatePhoneNumbersWithVoiceConnector');
        return this;
    }
    /**
     * Grants permission to associate multiple phone numbers with an Amazon Chime Voice Connector Group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_AssociatePhoneNumbersWithVoiceConnectorGroup.html
     */
    toAssociatePhoneNumbersWithVoiceConnectorGroup() {
        this.add('chime:AssociatePhoneNumbersWithVoiceConnectorGroup');
        return this;
    }
    /**
     * Grants permission to associate the specified sign-in delegate groups with the specified Amazon Chime account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_AssociateSigninDelegateGroupsWithAccount.html
     */
    toAssociateSigninDelegateGroupsWithAccount() {
        this.add('chime:AssociateSigninDelegateGroupsWithAccount');
        return this;
    }
    /**
     * Grants permission to authorize an Active Directory for your Amazon Chime Enterprise account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    toAuthorizeDirectory() {
        this.add('chime:AuthorizeDirectory');
        return this;
    }
    /**
     * Grants permission to create new attendees for an active Amazon Chime SDK meeting
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_BatchCreateAttendee.html
     */
    toBatchCreateAttendee() {
        this.add('chime:BatchCreateAttendee');
        return this;
    }
    /**
     * Grants permission to batch add room members
     *
     * Access Level: Write
     */
    toBatchCreateRoomMembership() {
        this.add('chime:BatchCreateRoomMembership');
        return this;
    }
    /**
     * Grants permission to move up to 50 phone numbers to the deletion queue
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_BatchDeletePhoneNumber.html
     */
    toBatchDeletePhoneNumber() {
        this.add('chime:BatchDeletePhoneNumber');
        return this;
    }
    /**
     * Grants permission to suspend up to 50 users from a Team or EnterpriseLWA Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_BatchSuspendUser.html
     */
    toBatchSuspendUser() {
        this.add('chime:BatchSuspendUser');
        return this;
    }
    /**
     * Grants permission to remove the suspension from up to 50 previously suspended users for the specified Amazon Chime EnterpriseLWA account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_BatchUnsuspendUser.html
     */
    toBatchUnsuspendUser() {
        this.add('chime:BatchUnsuspendUser');
        return this;
    }
    /**
     * Grants permission to update phone number details within the UpdatePhoneNumberRequestItem object for up to 50 phone numbers
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_BatchUpdatePhoneNumber.html
     */
    toBatchUpdatePhoneNumber() {
        this.add('chime:BatchUpdatePhoneNumber');
        return this;
    }
    /**
     * Grants permission to update user details within the UpdateUserRequestItem object for up to 20 users for the specified Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_BatchUpdateUser.html
     */
    toBatchUpdateUser() {
        this.add('chime:BatchUpdateUser');
        return this;
    }
    /**
     * Grants permission to connect an Active Directory to your Amazon Chime Enterprise account
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ds:ConnectDirectory
     *
     * https://docs.aws.amazon.com/chime/latest/ag/active_directory.html
     */
    toConnectDirectory() {
        this.add('chime:ConnectDirectory');
        return this;
    }
    /**
     * Grants permission to create an Amazon Chime account under the administrator's AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_CreateAccount.html
     */
    toCreateAccount() {
        this.add('chime:CreateAccount');
        return this;
    }
    /**
     * Grants permission to create a new SCIM access key for your Amazon Chime account and Okta configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/okta_sso.html
     */
    toCreateApiKey() {
        this.add('chime:CreateApiKey');
        return this;
    }
    /**
     * Grants permission to create a new attendee for an active Amazon Chime SDK meeting
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_CreateAttendee.html
     */
    toCreateAttendee() {
        this.add('chime:CreateAttendee');
        return this;
    }
    /**
     * Grants permission to create a bot for an Amazon Chime Enterprise account
     *
     * Access Level: Write
     */
    toCreateBot() {
        this.add('chime:CreateBot');
        return this;
    }
    /**
     * Grants permission to add a bot to a chat room in your Amazon Chime Enterprise account
     *
     * Access Level: Write
     */
    toCreateBotMembership() {
        this.add('chime:CreateBotMembership');
        return this;
    }
    /**
     * Grants permission to create a new Call Detail Record S3 bucket
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - s3:CreateBucket
     * - s3:ListAllMyBuckets
     *
     * https://docs.aws.amazon.com/chime/latest/ag/manage-access.html
     */
    toCreateCDRBucket() {
        this.add('chime:CreateCDRBucket');
        return this;
    }
    /**
     * Grants permission to create a new Amazon Chime SDK meeting in the specified media Region, with no initial attendees
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_CreateMeeting.html
     */
    toCreateMeeting() {
        this.add('chime:CreateMeeting');
        return this;
    }
    /**
     * Grants permission to create a new Amazon Chime SDK meeting in the specified media Region, with a set of attendees
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_CreateMeetingWithAttendees.html
     */
    toCreateMeetingWithAttendees() {
        this.add('chime:CreateMeetingWithAttendees');
        return this;
    }
    /**
     * Grants permission to create a phone number order with the Carriers
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_CreatePhoneNumberOrder.html
     */
    toCreatePhoneNumberOrder() {
        this.add('chime:CreatePhoneNumberOrder');
        return this;
    }
    /**
     * Grants permission to create a proxy session for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_CreateProxySession.html
     */
    toCreateProxySession() {
        this.add('chime:CreateProxySession');
        return this;
    }
    /**
     * Grants permission to create a room
     *
     * Access Level: Write
     */
    toCreateRoom() {
        this.add('chime:CreateRoom');
        return this;
    }
    /**
     * Grants permission to add a room member
     *
     * Access Level: Write
     */
    toCreateRoomMembership() {
        this.add('chime:CreateRoomMembership');
        return this;
    }
    /**
     * Grants permission to create a user under the specified Amazon Chime account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_CreateUser.html
     */
    toCreateUser() {
        this.add('chime:CreateUser');
        return this;
    }
    /**
     * Grants permission to create a Amazon Chime Voice Connector under the administrator's AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_CreateVoiceConnector.html
     */
    toCreateVoiceConnector() {
        this.add('chime:CreateVoiceConnector');
        return this;
    }
    /**
     * Grants permission to create a Amazon Chime Voice Connector Group under the administrator's AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_CreateVoiceConnectorGroup.html
     */
    toCreateVoiceConnectorGroup() {
        this.add('chime:CreateVoiceConnectorGroup');
        return this;
    }
    /**
     * Grants permission to delete the specified Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteAccount.html
     */
    toDeleteAccount() {
        this.add('chime:DeleteAccount');
        return this;
    }
    /**
     * Grants permission to delete the OpenIdConfig attributes from your Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/okta_sso.html
     */
    toDeleteAccountOpenIdConfig() {
        this.add('chime:DeleteAccountOpenIdConfig');
        return this;
    }
    /**
     * Grants permission to delete the specified SCIM access key associated with your Amazon Chime account and Okta configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/okta_sso.html
     */
    toDeleteApiKey() {
        this.add('chime:DeleteApiKey');
        return this;
    }
    /**
     * Grants permission to delete the specified attendee from an Amazon Chime SDK meeting
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteAttendee.html
     */
    toDeleteAttendee() {
        this.add('chime:DeleteAttendee');
        return this;
    }
    /**
     * Grants permission to delete a Call Detail Record S3 bucket from your Amazon Chime account
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - s3:DeleteBucket
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    toDeleteCDRBucket() {
        this.add('chime:DeleteCDRBucket');
        return this;
    }
    /**
     * Grants permission to delete delegated AWS account management from your Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    toDeleteDelegate() {
        this.add('chime:DeleteDelegate');
        return this;
    }
    /**
     * Grants permission to delete a domain from your Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/claim-domain.html
     */
    toDeleteDomain() {
        this.add('chime:DeleteDomain');
        return this;
    }
    /**
     * Grants permission to delete an events configuration for a bot to receive outgoing events
     *
     * Access Level: Write
     */
    toDeleteEventsConfiguration() {
        this.add('chime:DeleteEventsConfiguration');
        return this;
    }
    /**
     * Grants permission to delete Active Directory or Okta user groups from your Amazon Chime Enterprise account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    toDeleteGroups() {
        this.add('chime:DeleteGroups');
        return this;
    }
    /**
     * Grants permission to delete the specified Amazon Chime SDK meeting
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteMeeting.html
     */
    toDeleteMeeting() {
        this.add('chime:DeleteMeeting');
        return this;
    }
    /**
     * Grants permission to move a phone number to the deletion queue
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DeletePhoneNumber.html
     */
    toDeletePhoneNumber() {
        this.add('chime:DeletePhoneNumber');
        return this;
    }
    /**
     * Grants permission to delete a proxy session for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteProxySession.html
     */
    toDeleteProxySession() {
        this.add('chime:DeleteProxySession');
        return this;
    }
    /**
     * Grants permission to delete a room
     *
     * Access Level: Write
     */
    toDeleteRoom() {
        this.add('chime:DeleteRoom');
        return this;
    }
    /**
     * Grants permission to remove a room member
     *
     * Access Level: Write
     */
    toDeleteRoomMembership() {
        this.add('chime:DeleteRoomMembership');
        return this;
    }
    /**
     * Grants permission to delete the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteVoiceConnector.html
     */
    toDeleteVoiceConnector() {
        this.add('chime:DeleteVoiceConnector');
        return this;
    }
    /**
     * Grants permission to delete emergency calling configuration for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteVoiceConnectorEmergencyCallingConfiguration.html
     */
    toDeleteVoiceConnectorEmergencyCallingConfiguration() {
        this.add('chime:DeleteVoiceConnectorEmergencyCallingConfiguration');
        return this;
    }
    /**
     * Grants permission to delete the specified Amazon Chime Voice Connector Group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteVoiceConnectorGroup.html
     */
    toDeleteVoiceConnectorGroup() {
        this.add('chime:DeleteVoiceConnectorGroup');
        return this;
    }
    /**
     * Grants permission to delete the origination settings for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteVoiceConnectorOrigination.html
     */
    toDeleteVoiceConnectorOrigination() {
        this.add('chime:DeleteVoiceConnectorOrigination');
        return this;
    }
    /**
     * Grants permission to delete proxy configuration for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteVoiceConnectorProxy.html
     */
    toDeleteVoiceConnectorProxy() {
        this.add('chime:DeleteVoiceConnectorProxy');
        return this;
    }
    /**
     * Grants permission to delete streaming configuration for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteVoiceConnectorStreamingConfiguration.html
     */
    toDeleteVoiceConnectorStreamingConfiguration() {
        this.add('chime:DeleteVoiceConnectorStreamingConfiguration');
        return this;
    }
    /**
     * Grants permission to delete the termination settings for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteVoiceConnectorTermination.html
     */
    toDeleteVoiceConnectorTermination() {
        this.add('chime:DeleteVoiceConnectorTermination');
        return this;
    }
    /**
     * Grants permission to delete SIP termination credentials for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DeleteVoiceConnectorTerminationCredentials.html
     */
    toDeleteVoiceConnectorTerminationCredentials() {
        this.add('chime:DeleteVoiceConnectorTerminationCredentials');
        return this;
    }
    /**
     * Grants permission to disassociate the primary provisioned number from the specified Amazon Chime user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DisassociatePhoneNumberFromUser.html
     */
    toDisassociatePhoneNumberFromUser() {
        this.add('chime:DisassociatePhoneNumberFromUser');
        return this;
    }
    /**
     * Grants permission to disassociate multiple phone numbers from the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DisassociatePhoneNumbersFromVoiceConnector.html
     */
    toDisassociatePhoneNumbersFromVoiceConnector() {
        this.add('chime:DisassociatePhoneNumbersFromVoiceConnector');
        return this;
    }
    /**
     * Grants permission to disassociate multiple phone numbers from the specified Amazon Chime Voice Connector Group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DisassociatePhoneNumbersFromVoiceConnectorGroup.html
     */
    toDisassociatePhoneNumbersFromVoiceConnectorGroup() {
        this.add('chime:DisassociatePhoneNumbersFromVoiceConnectorGroup');
        return this;
    }
    /**
     * Grants permission to disassociate the specified sign-in delegate groups from the specified Amazon Chime account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_DisassociateSigninDelegateGroupsFromAccount.html
     */
    toDisassociateSigninDelegateGroupsFromAccount() {
        this.add('chime:DisassociateSigninDelegateGroupsFromAccount');
        return this;
    }
    /**
     * Grants permission to disconnect the Active Directory from your Amazon Chime Enterprise account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    toDisconnectDirectory() {
        this.add('chime:DisconnectDirectory');
        return this;
    }
    /**
     * Grants permission to get details for the specified Amazon Chime account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetAccount.html
     */
    toGetAccount() {
        this.add('chime:GetAccount');
        return this;
    }
    /**
     * Grants permission to get details for the account resource associated with your Amazon Chime account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    toGetAccountResource() {
        this.add('chime:GetAccountResource');
        return this;
    }
    /**
     * Grants permission to get account settings for the specified Amazon Chime account ID
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetAccountSettings.html
     */
    toGetAccountSettings() {
        this.add('chime:GetAccountSettings');
        return this;
    }
    /**
     * Grants permission to get the account details and OpenIdConfig attributes for your Amazon Chime account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/ag/okta_sso.html
     */
    toGetAccountWithOpenIdConfig() {
        this.add('chime:GetAccountWithOpenIdConfig');
        return this;
    }
    /**
     * Grants permission to get attendee details for a specified meeting ID and attendee ID
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetAttendee.html
     */
    toGetAttendee() {
        this.add('chime:GetAttendee');
        return this;
    }
    /**
     * Grants permission to retrieve details for the specified bot
     *
     * Access Level: Read
     */
    toGetBot() {
        this.add('chime:GetBot');
        return this;
    }
    /**
     * Grants permission to get details of a Call Detail Record S3 bucket associated with your Amazon Chime account
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - s3:GetBucketAcl
     * - s3:GetBucketLocation
     * - s3:GetBucketLogging
     * - s3:GetBucketVersioning
     * - s3:GetBucketWebsite
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    toGetCDRBucket() {
        this.add('chime:GetCDRBucket');
        return this;
    }
    /**
     * Grants permission to get domain details for a domain associated with your Amazon Chime account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/ag/claim-domain.html
     */
    toGetDomain() {
        this.add('chime:GetDomain');
        return this;
    }
    /**
     * Grants permission to retrieve details for an events configuration for a bot to receive outgoing events
     *
     * Access Level: Read
     */
    toGetEventsConfiguration() {
        this.add('chime:GetEventsConfiguration');
        return this;
    }
    /**
     * Grants permission to get global settings related to Amazon Chime for the AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetGlobalSettings.html
     */
    toGetGlobalSettings() {
        this.add('chime:GetGlobalSettings');
        return this;
    }
    /**
     * Grants permission to get the meeting record for a specified meeting ID
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetMeeting.html
     */
    toGetMeeting() {
        this.add('chime:GetMeeting');
        return this;
    }
    /**
     * Grants permission to get attendee, connection, and other details for a meeting
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    toGetMeetingDetail() {
        this.add('chime:GetMeetingDetail');
        return this;
    }
    /**
     * Grants permission to get details for the specified phone number
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetPhoneNumber.html
     */
    toGetPhoneNumber() {
        this.add('chime:GetPhoneNumber');
        return this;
    }
    /**
     * Grants permission to get details for the specified phone number order
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetPhoneNumberOrder.html
     */
    toGetPhoneNumberOrder() {
        this.add('chime:GetPhoneNumberOrder');
        return this;
    }
    /**
     * Grants permission to get phone number settings related to Amazon Chime for the AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetPhoneNumberSettings.html
     */
    toGetPhoneNumberSettings() {
        this.add('chime:GetPhoneNumberSettings');
        return this;
    }
    /**
     * Grants permission to get details of the specified proxy session for the specified Amazon Chime Voice Connector
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetProxySession.html
     */
    toGetProxySession() {
        this.add('chime:GetProxySession');
        return this;
    }
    /**
     * Gets the retention settings for the specified Amazon Chime account.
     *
     * Access Level: Read
     */
    toGetRetentionSettings() {
        this.add('chime:GetRetentionSettings');
        return this;
    }
    /**
     * Grants permission to retrieve a room
     *
     * Access Level: Read
     */
    toGetRoom() {
        this.add('chime:GetRoom');
        return this;
    }
    /**
     * Grants permission to get telephony limits for the AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/ag/phone-numbers.html
     */
    toGetTelephonyLimits() {
        this.add('chime:GetTelephonyLimits');
        return this;
    }
    /**
     * Grants permission to get details for the specified user ID
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetUser.html
     */
    toGetUser() {
        this.add('chime:GetUser');
        return this;
    }
    /**
     * Grants permission to get a summary of user activity on the user details page
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/ag/user-details.html
     */
    toGetUserActivityReportData() {
        this.add('chime:GetUserActivityReportData');
        return this;
    }
    /**
     * Grants permission to get user details for an Amazon Chime user based on the email address in an Amazon Chime Enterprise or Team account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/ag/user-details.html
     */
    toGetUserByEmail() {
        this.add('chime:GetUserByEmail');
        return this;
    }
    /**
     * Grants permission to get user settings related to the specified Amazon Chime user
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetUserSettings.html
     */
    toGetUserSettings() {
        this.add('chime:GetUserSettings');
        return this;
    }
    /**
     * Grants permission to get details for the specified Amazon Chime Voice Connector
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetVoiceConnector.html
     */
    toGetVoiceConnector() {
        this.add('chime:GetVoiceConnector');
        return this;
    }
    /**
     * Grants permission to get details of the emergency calling configuration for the specified Amazon Chime Voice Connector
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetVoiceConnectorEmergencyCallingConfiguration.html
     */
    toGetVoiceConnectorEmergencyCallingConfiguration() {
        this.add('chime:GetVoiceConnectorEmergencyCallingConfiguration');
        return this;
    }
    /**
     * Grants permission to get details for the specified Amazon Chime Voice Connector Group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetVoiceConnectorGroup.html
     */
    toGetVoiceConnectorGroup() {
        this.add('chime:GetVoiceConnectorGroup');
        return this;
    }
    /**
     * Grants permission to get details of the logging configuration for the specified Amazon Chime Voice Connector
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetVoiceConnectorLoggingConfiguration.html
     */
    toGetVoiceConnectorLoggingConfiguration() {
        this.add('chime:GetVoiceConnectorLoggingConfiguration');
        return this;
    }
    /**
     * Grants permission to get details of the origination settings for the specified Amazon Chime Voice Connector
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetVoiceConnectorOrigination.html
     */
    toGetVoiceConnectorOrigination() {
        this.add('chime:GetVoiceConnectorOrigination');
        return this;
    }
    /**
     * Grants permission to get details of the proxy configuration for the specified Amazon Chime Voice Connector
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetVoiceConnectorProxy.html
     */
    toGetVoiceConnectorProxy() {
        this.add('chime:GetVoiceConnectorProxy');
        return this;
    }
    /**
     * Grants permission to get details of the streaming configuration for the specified Amazon Chime Voice Connector
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetVoiceConnectorStreamingConfiguration.html
     */
    toGetVoiceConnectorStreamingConfiguration() {
        this.add('chime:GetVoiceConnectorStreamingConfiguration');
        return this;
    }
    /**
     * Grants permission to get details of the termination settings for the specified Amazon Chime Voice Connector
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetVoiceConnectorTermination.html
     */
    toGetVoiceConnectorTermination() {
        this.add('chime:GetVoiceConnectorTermination');
        return this;
    }
    /**
     * Grants permission to get details of the termination health for the specified Amazon Chime Voice Connector
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_GetVoiceConnectorTerminationHealth.html
     */
    toGetVoiceConnectorTerminationHealth() {
        this.add('chime:GetVoiceConnectorTerminationHealth');
        return this;
    }
    /**
     * Grants permission to send an invitation to accept a request for AWS account delegation for an Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    toInviteDelegate() {
        this.add('chime:InviteDelegate');
        return this;
    }
    /**
     * Grants permission to invite as many as 50 users to the specified Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_InviteUsers.html
     */
    toInviteUsers() {
        this.add('chime:InviteUsers');
        return this;
    }
    /**
     * Grants permission to invite users from a third party provider to your Amazon Chime account
     *
     * Access Level: Write
     */
    toInviteUsersFromProvider() {
        this.add('chime:InviteUsersFromProvider');
        return this;
    }
    /**
     * Grants permission to list Amazon Chime account usage reporting data
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/ag/view-reports.html
     */
    toListAccountUsageReportData() {
        this.add('chime:ListAccountUsageReportData');
        return this;
    }
    /**
     * Grants permission to list the Amazon Chime accounts under the administrator's AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_ListAccounts.html
     */
    toListAccounts() {
        this.add('chime:ListAccounts');
        return this;
    }
    /**
     * Grants permission to list the SCIM access keys defined for your Amazon Chime account and Okta configuration
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/ag/okta_sso.html
     */
    toListApiKeys() {
        this.add('chime:ListApiKeys');
        return this;
    }
    /**
     * Grants permission to list the tags applied to an Amazon Chime SDK attendee resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_ListAttendeeTags.html
     */
    toListAttendeeTags() {
        this.add('chime:ListAttendeeTags');
        return this;
    }
    /**
     * Grants permission to list up to 100 attendees for a specified Amazon Chime SDK meeting
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_ListAttendees.html
     */
    toListAttendees() {
        this.add('chime:ListAttendees');
        return this;
    }
    /**
     * Grants permission to list the bots associated with the administrator's Amazon Chime Enterprise account
     *
     * Access Level: List
     */
    toListBots() {
        this.add('chime:ListBots');
        return this;
    }
    /**
     * Grants permission to list Call Detail Record S3 buckets
     *
     * Access Level: List
     *
     * Dependent actions:
     * - s3:ListAllMyBuckets
     * - s3:ListBucket
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    toListCDRBucket() {
        this.add('chime:ListCDRBucket');
        return this;
    }
    /**
     * Grants permission to list the calling regions available for the administrator's AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/ag/phone-numbers.html
     */
    toListCallingRegions() {
        this.add('chime:ListCallingRegions');
        return this;
    }
    /**
     * Grants permission to list account delegate information associated with your Amazon Chime account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    toListDelegates() {
        this.add('chime:ListDelegates');
        return this;
    }
    /**
     * Grants permission to list active Active Directories hosted in the Directory Service of your AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    toListDirectories() {
        this.add('chime:ListDirectories');
        return this;
    }
    /**
     * Grants permission to list domains associated with your Amazon Chime account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/ag/claim-domain.html
     */
    toListDomains() {
        this.add('chime:ListDomains');
        return this;
    }
    /**
     * Grants permission to list Active Directory or Okta user groups associated with your Amazon Chime Enterprise account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    toListGroups() {
        this.add('chime:ListGroups');
        return this;
    }
    /**
     * Grants permission to list all events that occurred for a specified meeting
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/ag/view-reports.html
     */
    toListMeetingEvents() {
        this.add('chime:ListMeetingEvents');
        return this;
    }
    /**
     * Grants permission to list the tags applied to an Amazon Chime SDK meeting resource.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_ListMeetingTags.html
     */
    toListMeetingTags() {
        this.add('chime:ListMeetingTags');
        return this;
    }
    /**
     * Grants permission to list up to 100 active Amazon Chime SDK meetings
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_ListMeetings.html
     */
    toListMeetings() {
        this.add('chime:ListMeetings');
        return this;
    }
    /**
     * Grants permission to list meetings ended during the specified date range
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/ag/view-reports.html
     */
    toListMeetingsReportData() {
        this.add('chime:ListMeetingsReportData');
        return this;
    }
    /**
     * Grants permission to list the phone number orders under the administrator's AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_ListPhoneNumberOrders.html
     */
    toListPhoneNumberOrders() {
        this.add('chime:ListPhoneNumberOrders');
        return this;
    }
    /**
     * Grants permission to list the phone numbers under the administrator's AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_ListPhoneNumbers.html
     */
    toListPhoneNumbers() {
        this.add('chime:ListPhoneNumbers');
        return this;
    }
    /**
     * Grants permission to list proxy sessions for the specified Amazon Chime Voice Connector
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_ListProxySessions.html
     */
    toListProxySessions() {
        this.add('chime:ListProxySessions');
        return this;
    }
    /**
     * Grants permission to list all room members
     *
     * Access Level: Read
     */
    toListRoomMemberships() {
        this.add('chime:ListRoomMemberships');
        return this;
    }
    /**
     * Grants permission to list rooms
     *
     * Access Level: Read
     */
    toListRooms() {
        this.add('chime:ListRooms');
        return this;
    }
    /**
     * Grants permission to list the tags applied to an Amazon Chime SDK meeting resource.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.add('chime:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to list the users that belong to the specified Amazon Chime account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_ListUsers.html
     */
    toListUsers() {
        this.add('chime:ListUsers');
        return this;
    }
    /**
     * Grants permission to list the Amazon Chime Voice Connector Groups under the administrator's AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_ListVoiceConnectorGroups.html
     */
    toListVoiceConnectorGroups() {
        this.add('chime:ListVoiceConnectorGroups');
        return this;
    }
    /**
     * Grants permission to list the SIP termination credentials for the specified Amazon Chime Voice Connector
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_ListVoiceConnectorTerminationCredentials.html
     */
    toListVoiceConnectorTerminationCredentials() {
        this.add('chime:ListVoiceConnectorTerminationCredentials');
        return this;
    }
    /**
     * Grants permission to list the Amazon Chime Voice Connectors under the administrator's AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_ListVoiceConnectors.html
     */
    toListVoiceConnectors() {
        this.add('chime:ListVoiceConnectors');
        return this;
    }
    /**
     * Grants permission to log out the specified user from all of the devices they are currently logged into
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_LogoutUser.html
     */
    toLogoutUser() {
        this.add('chime:LogoutUser');
        return this;
    }
    /**
     * Grants permission to update details for an events configuration for a bot to receive outgoing events
     *
     * Access Level: Write
     */
    toPutEventsConfiguration() {
        this.add('chime:PutEventsConfiguration');
        return this;
    }
    /**
     * Puts retention settings for the specified Amazon Chime account
     *
     * Access Level: Write
     */
    toPutRetentionSettings() {
        this.add('chime:PutRetentionSettings');
        return this;
    }
    /**
     * Grants permission to add emergency calling configuration for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_PutVoiceConnectorEmergencyCallingConfiguration.html
     */
    toPutVoiceConnectorEmergencyCallingConfiguration() {
        this.add('chime:PutVoiceConnectorEmergencyCallingConfiguration');
        return this;
    }
    /**
     * Grants permission to add logging configuration for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - logs:CreateLogDelivery
     * - logs:CreateLogGroup
     * - logs:DeleteLogDelivery
     * - logs:DescribeLogGroups
     * - logs:GetLogDelivery
     * - logs:ListLogDeliveries
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_PutVoiceConnectorLoggingConfiguration.html
     */
    toPutVoiceConnectorLoggingConfiguration() {
        this.add('chime:PutVoiceConnectorLoggingConfiguration');
        return this;
    }
    /**
     * Grants permission to update the origination settings for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_PutVoiceConnectorOrigination.html
     */
    toPutVoiceConnectorOrigination() {
        this.add('chime:PutVoiceConnectorOrigination');
        return this;
    }
    /**
     * Grants permission to add proxy configuration for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_PutVoiceConnectorProxy.html
     */
    toPutVoiceConnectorProxy() {
        this.add('chime:PutVoiceConnectorProxy');
        return this;
    }
    /**
     * Grants permission to add streaming configuration for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_PutVoiceConnectorStreamingConfiguration.html
     */
    toPutVoiceConnectorStreamingConfiguration() {
        this.add('chime:PutVoiceConnectorStreamingConfiguration');
        return this;
    }
    /**
     * Grants permission to update the termination settings for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_PutVoiceConnectorTermination.html
     */
    toPutVoiceConnectorTermination() {
        this.add('chime:PutVoiceConnectorTermination');
        return this;
    }
    /**
     * Grants permission to add SIP termination credentials for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_PutVoiceConnectorTerminationCredentials.html
     */
    toPutVoiceConnectorTerminationCredentials() {
        this.add('chime:PutVoiceConnectorTerminationCredentials');
        return this;
    }
    /**
     * Redacts the specified Chime conversation Message
     *
     * Access Level: Write
     */
    toRedactConversationMessage() {
        this.add('chime:RedactConversationMessage');
        return this;
    }
    /**
     * Redacts the specified Chime room Message
     *
     * Access Level: Write
     */
    toRedactRoomMessage() {
        this.add('chime:RedactRoomMessage');
        return this;
    }
    /**
     * Grants permission to regenerate the security token for the specified bot
     *
     * Access Level: Write
     */
    toRegenerateSecurityToken() {
        this.add('chime:RegenerateSecurityToken');
        return this;
    }
    /**
     * Grants permission to modify the account name for your Amazon Chime Enterprise or Team account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/rename-account.html
     */
    toRenameAccount() {
        this.add('chime:RenameAccount');
        return this;
    }
    /**
     * Grants permission to renew the delegation request associated with an Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    toRenewDelegate() {
        this.add('chime:RenewDelegate');
        return this;
    }
    /**
     * Grants permission to reset the account resource in your Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    toResetAccountResource() {
        this.add('chime:ResetAccountResource');
        return this;
    }
    /**
     * Grants permission to reset the personal meeting PIN for the specified user on an Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_ResetPersonalPIN.html
     */
    toResetPersonalPIN() {
        this.add('chime:ResetPersonalPIN');
        return this;
    }
    /**
     * Grants permission to restore the specified phone number from the deltion queue back to the phone number inventory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_RestorePhoneNumber.html
     */
    toRestorePhoneNumber() {
        this.add('chime:RestorePhoneNumber');
        return this;
    }
    /**
     * Grants permission to download the file containing links to all user attachments returned as part of the "Request attachments" action
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chime/latest/ag/request-attachments.html
     */
    toRetrieveDataExports() {
        this.add('chime:RetrieveDataExports');
        return this;
    }
    /**
     * Grants permission to search phone numbers that can be ordered from the carrier
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_SearchAvailablePhoneNumbers.html
     */
    toSearchAvailablePhoneNumbers() {
        this.add('chime:SearchAvailablePhoneNumbers');
        return this;
    }
    /**
     * Grants permission to submit the "Request attachments" request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/request-attachments.html
     */
    toStartDataExport() {
        this.add('chime:StartDataExport');
        return this;
    }
    /**
     * Grants permission to submit a customer service support request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/chime-getting-admin-support.html
     */
    toSubmitSupportRequest() {
        this.add('chime:SubmitSupportRequest');
        return this;
    }
    /**
     * Grants permission to suspend users from an Amazon Chime Enterprise account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/manage-access.html
     */
    toSuspendUsers() {
        this.add('chime:SuspendUsers');
        return this;
    }
    /**
     * Grants permission to apply the specified tags to the specified Amazon Chime SDK attendee
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_TagAttendee.html
     */
    toTagAttendee() {
        this.add('chime:TagAttendee');
        return this;
    }
    /**
     * Grants permission to apply the specified tags to the specified Amazon Chime SDK meeting.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_TagMeeting.html
     */
    toTagMeeting() {
        this.add('chime:TagMeeting');
        return this;
    }
    /**
     * Grants permission to apply the specified tags to the specified Amazon Chime SDK meeting resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        this.add('chime:TagResource');
        return this;
    }
    /**
     * Grants permission to unauthorize an Active Directory from your Amazon Chime Enterprise account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    toUnauthorizeDirectory() {
        this.add('chime:UnauthorizeDirectory');
        return this;
    }
    /**
     * Grants permission to untag the specified tags from the specified Amazon Chime SDK attendee.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_UntagAttendee.html
     */
    toUntagAttendee() {
        this.add('chime:UntagAttendee');
        return this;
    }
    /**
     * Grants permission to untag the specified tags from the specified Amazon Chime SDK meeting.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_UntagMeeting.html
     */
    toUntagMeeting() {
        this.add('chime:UntagMeeting');
        return this;
    }
    /**
     * Grants permission to untag the specified tags from the specified Amazon Chime SDK meeting resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        this.add('chime:UntagResource');
        return this;
    }
    /**
     * Grants permission to update account details for the specified Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdateAccount.html
     */
    toUpdateAccount() {
        this.add('chime:UpdateAccount');
        return this;
    }
    /**
     * Grants permission to update the OpenIdConfig attributes for your Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/okta_sso.html
     */
    toUpdateAccountOpenIdConfig() {
        this.add('chime:UpdateAccountOpenIdConfig');
        return this;
    }
    /**
     * Grants permission to update the account resource in your Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    toUpdateAccountResource() {
        this.add('chime:UpdateAccountResource');
        return this;
    }
    /**
     * Grants permission to update the settings for the specified Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdateAccountSettings.html
     */
    toUpdateAccountSettings() {
        this.add('chime:UpdateAccountSettings');
        return this;
    }
    /**
     * Grants permission to update the status of the specified bot
     *
     * Access Level: Write
     */
    toUpdateBot() {
        this.add('chime:UpdateBot');
        return this;
    }
    /**
     * Grants permission to update your Call Detail Record S3 bucket
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - s3:CreateBucket
     * - s3:DeleteBucket
     * - s3:ListAllMyBuckets
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    toUpdateCDRSettings() {
        this.add('chime:UpdateCDRSettings');
        return this;
    }
    /**
     * Grants permission to update the global settings related to Amazon Chime for the AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdateGlobalSettings.html
     */
    toUpdateGlobalSettings() {
        this.add('chime:UpdateGlobalSettings');
        return this;
    }
    /**
     * Grants permission to update phone number details for the specified phone number
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdatePhoneNumber.html
     */
    toUpdatePhoneNumber() {
        this.add('chime:UpdatePhoneNumber');
        return this;
    }
    /**
     * Grants permission to update phone number settings related to Amazon Chime for the AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdatePhoneNumberSettings.html
     */
    toUpdatePhoneNumberSettings() {
        this.add('chime:UpdatePhoneNumberSettings');
        return this;
    }
    /**
     * Grants permission to update a proxy session for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdateProxySession.html
     */
    toUpdateProxySession() {
        this.add('chime:UpdateProxySession');
        return this;
    }
    /**
     * Grants permission to update a room
     *
     * Access Level: Write
     */
    toUpdateRoom() {
        this.add('chime:UpdateRoom');
        return this;
    }
    /**
     * Grants permission to update room membership role
     *
     * Access Level: Write
     */
    toUpdateRoomMembership() {
        this.add('chime:UpdateRoomMembership');
        return this;
    }
    /**
     * Grants permission to update the supported license tiers available for users in your Amazon Chime account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/manage-access.html
     */
    toUpdateSupportedLicenses() {
        this.add('chime:UpdateSupportedLicenses');
        return this;
    }
    /**
     * Grants permission to update user details for a specified user ID
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdateUser.html
     */
    toUpdateUser() {
        this.add('chime:UpdateUser');
        return this;
    }
    /**
     * Grants permission to update the licenses for your Amazon Chime users
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/ag/manage-access.html
     */
    toUpdateUserLicenses() {
        this.add('chime:UpdateUserLicenses');
        return this;
    }
    /**
     * Grants permission to update user settings related to the specified Amazon Chime user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdateUserSettings.html
     */
    toUpdateUserSettings() {
        this.add('chime:UpdateUserSettings');
        return this;
    }
    /**
     * Grants permission to update Amazon Chime Voice Connector details for the specified Amazon Chime Voice Connector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdateVoiceConnector.html
     */
    toUpdateVoiceConnector() {
        this.add('chime:UpdateVoiceConnector');
        return this;
    }
    /**
     * Grants permission to update Amazon Chime Voice Connector Group details for the specified Amazon Chime Voice Connector Group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chime/latest/APIReference/API_UpdateVoiceConnectorGroup.html
     */
    toUpdateVoiceConnectorGroup() {
        this.add('chime:UpdateVoiceConnectorGroup');
        return this;
    }
    /**
     * Grants permission to validate the account resource in your Amazon Chime account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chime/latest/ag/control-access.html
     */
    toValidateAccountResource() {
        this.add('chime:ValidateAccountResource');
        return this;
    }
    /**
     * Adds a resource of type meeting to the statement
     *
     * @param meetingId - Identifier for the meetingId.
     * @param accountId - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onMeeting(meetingId, accountId, partition) {
        var arn = 'arn:${Partition}:chime::${AccountId}:meeting/${MeetingId}';
        arn = arn.replace('${MeetingId}', meetingId);
        arn = arn.replace('${AccountId}', accountId || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Chime = Chime;
//# sourceMappingURL=data:application/json;base64,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