"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Cloudfront = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [cloudfront](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoncloudfront.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Cloudfront extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [cloudfront](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoncloudfront.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'cloudfront';
        this.actionList = {
            "CreateCachePolicy": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreateCachePolicy.html",
                "description": "This action adds a new cache policy to CloudFront.",
                "accessLevel": "Write"
            },
            "CreateCloudFrontOriginAccessIdentity": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreateCloudFrontOriginAccessIdentity.html",
                "description": "This action creates a new CloudFront origin access identity.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "origin-access-identity": {
                        "required": true
                    }
                }
            },
            "CreateDistribution": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreateDistribution.html",
                "description": "This action creates a new web distribution.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "distribution": {
                        "required": true
                    }
                }
            },
            "CreateDistributionWithTags": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreateDistributionWithTags.html",
                "description": "This action creates a new web distribution with tags.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "distribution": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateFieldLevelEncryptionConfig": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreateFieldLevelEncryptionConfig.html",
                "description": "This action creates a new field-level encryption configuration.",
                "accessLevel": "Write"
            },
            "CreateFieldLevelEncryptionProfile": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreateFieldLevelEncryptionProfile.html",
                "description": "This action creates a field-level encryption profile.",
                "accessLevel": "Write"
            },
            "CreateInvalidation": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreateInvalidation.html",
                "description": "This action creates a new invalidation batch request.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "distribution": {
                        "required": true
                    }
                }
            },
            "CreateOriginRequestPolicy": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreateOriginRequestPolicy.html",
                "description": "This action adds a new origin request policy to CloudFront.",
                "accessLevel": "Write"
            },
            "CreatePublicKey": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreatePublicKey.html",
                "description": "This action adds a new public key to CloudFront.",
                "accessLevel": "Write"
            },
            "CreateStreamingDistribution": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreateStreamingDistribution.html",
                "description": "This action creates a new RTMP distribution.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "streaming-distribution": {
                        "required": true
                    }
                }
            },
            "CreateStreamingDistributionWithTags": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreateStreamingDistributionWithTags.html",
                "description": "This action creates a new RTMP distribution with tags.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "streaming-distribution": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "DeleteCachePolicy": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_DeleteCachePolicy.html",
                "description": "This action deletes a cache policy.",
                "accessLevel": "Write"
            },
            "DeleteCloudFrontOriginAccessIdentity": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_DeleteCloudFrontOriginAccessIdentity.html",
                "description": "This action deletes a CloudFront origin access identity.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "origin-access-identity": {
                        "required": true
                    }
                }
            },
            "DeleteDistribution": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_DeleteDistribution.html",
                "description": "This action deletes a web distribution.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "distribution": {
                        "required": true
                    }
                }
            },
            "DeleteFieldLevelEncryptionConfig": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_DeleteFieldLevelEncryptionConfig.html",
                "description": "This action deletes a field-level encryption configuration.",
                "accessLevel": "Write"
            },
            "DeleteFieldLevelEncryptionProfile": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_DeleteFieldLevelEncryptionProfile.html",
                "description": "This action deletes a field-level encryption profile.",
                "accessLevel": "Write"
            },
            "DeleteOriginRequestPolicy": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_DeleteOriginRequestPolicy.html",
                "description": "This action deletes an origin request policy.",
                "accessLevel": "Write"
            },
            "DeletePublicKey": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_DeletePublicKey.html",
                "description": "This action deletes a public key from CloudFront.",
                "accessLevel": "Write"
            },
            "DeleteStreamingDistribution": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_DeleteStreamingDistribution.html",
                "description": "This action deletes an RTMP distribution.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "streaming-distribution": {
                        "required": true
                    }
                }
            },
            "GetCachePolicy": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetCachePolicy.html",
                "description": "Get the cache policy",
                "accessLevel": "Read"
            },
            "GetCachePolicyConfig": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetCachePolicyConfig.html",
                "description": "Get the cache policy configuration",
                "accessLevel": "Read"
            },
            "GetCloudFrontOriginAccessIdentity": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetCloudFrontOriginAccessIdentity.html",
                "description": "Get the information about a CloudFront origin access identity.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "origin-access-identity": {
                        "required": true
                    }
                }
            },
            "GetCloudFrontOriginAccessIdentityConfig": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetCloudFrontOriginAccessIdentityConfig.html",
                "description": "Get the configuration information about a Cloudfront origin access identity.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "origin-access-identity": {
                        "required": true
                    }
                }
            },
            "GetDistribution": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetDistribution.html",
                "description": "Get the information about a web distribution.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "distribution": {
                        "required": true
                    }
                }
            },
            "GetDistributionConfig": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetDistributionConfig.html",
                "description": "Get the configuration information about a distribution.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "distribution": {
                        "required": true
                    }
                }
            },
            "GetFieldLevelEncryption": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetFieldLevelEncryption.html",
                "description": "Get the field-level encryption configuration information.",
                "accessLevel": "Read"
            },
            "GetFieldLevelEncryptionConfig": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetFieldLevelEncryptionConfig.html",
                "description": "Get the field-level encryption configuration information.",
                "accessLevel": "Read"
            },
            "GetFieldLevelEncryptionProfile": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetFieldLevelEncryptionProfile.html",
                "description": "Get the field-level encryption configuration information.",
                "accessLevel": "Read"
            },
            "GetFieldLevelEncryptionProfileConfig": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetFieldLevelEncryptionProfileConfig.html",
                "description": "Get the field-level encryption profile configuration information.",
                "accessLevel": "Read"
            },
            "GetInvalidation": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetInvalidation.html",
                "description": "Get the information about an invalidation.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "distribution": {
                        "required": true
                    }
                }
            },
            "GetOriginRequestPolicy": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetOriginRequestPolicy.html",
                "description": "Get the origin request policy",
                "accessLevel": "Read"
            },
            "GetOriginRequestPolicyConfig": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetOriginRequestPolicyConfig.html",
                "description": "Get the origin request policy configuration",
                "accessLevel": "Read"
            },
            "GetPublicKey": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetPublicKey.html",
                "description": "Get the public key information.",
                "accessLevel": "Read"
            },
            "GetPublicKeyConfig": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetPublicKeyConfig.html",
                "description": "Get the public key configuration information.",
                "accessLevel": "Read"
            },
            "GetStreamingDistribution": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetStreamingDistribution.html",
                "description": "Get the information about an RTMP distribution.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "streaming-distribution": {
                        "required": true
                    }
                }
            },
            "GetStreamingDistributionConfig": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetStreamingDistributionConfig.html",
                "description": "Get the configuration information about a streaming distribution.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "streaming-distribution": {
                        "required": true
                    }
                }
            },
            "ListCachePolicies": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListCachePolicies.html",
                "description": "List all cache policies that have been created in CloudFront for this account.",
                "accessLevel": "List"
            },
            "ListCloudFrontOriginAccessIdentities": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListCloudFrontOriginAccessIdentities.html",
                "description": "List your CloudFront origin access identities.",
                "accessLevel": "List"
            },
            "ListDistributions": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListDistributions.html",
                "description": "List the distributions associated with your AWS account.",
                "accessLevel": "List"
            },
            "ListDistributionsByCachePolicyId": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListDistributionsByCachePolicyId.html",
                "description": "List distribution IDs for distributions that have a cache behavior that's associated with the specified cache policy.",
                "accessLevel": "List"
            },
            "ListDistributionsByOriginRequestPolicyId": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListDistributionsByOriginRequestPolicyId.html",
                "description": "List distribution IDs for distributions that have a cache behavior that's associated with the specified origin request policy.",
                "accessLevel": "List"
            },
            "ListDistributionsByWebACLId": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListDistributionsByWebACLId.html",
                "description": "List the distributions associated with your AWS account with given AWS WAF web ACL.",
                "accessLevel": "List"
            },
            "ListFieldLevelEncryptionConfigs": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListFieldLevelEncryptionConfigs.html",
                "description": "List all field-level encryption configurations that have been created in CloudFront for this account.",
                "accessLevel": "List"
            },
            "ListFieldLevelEncryptionProfiles": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListFieldLevelEncryptionProfiles.html",
                "description": "List all field-level encryption profiles that have been created in CloudFront for this account.",
                "accessLevel": "List"
            },
            "ListInvalidations": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListInvalidations.html",
                "description": "List your invalidation batches.",
                "accessLevel": "List",
                "resourceTypes": {
                    "distribution": {
                        "required": true
                    }
                }
            },
            "ListOriginRequestPolicies": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListOriginRequestPolicies.html",
                "description": "List all origin request policies that have been created in CloudFront for this account.",
                "accessLevel": "List"
            },
            "ListPublicKeys": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListPublicKeys.html",
                "description": "List all public keys that have been added to CloudFront for this account.",
                "accessLevel": "List"
            },
            "ListStreamingDistributions": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListStreamingDistributions.html",
                "description": "List your RTMP distributions.",
                "accessLevel": "List"
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListTagsForResource.html",
                "description": "List tags for a CloudFront resource.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "distribution": {
                        "required": false
                    },
                    "streaming-distribution": {
                        "required": false
                    }
                }
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_TagResource.html",
                "description": "Add tags to a CloudFront resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "distribution": {
                        "required": false
                    },
                    "streaming-distribution": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_UntagResource.html",
                "description": "Remove tags from a CloudFront resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "distribution": {
                        "required": false
                    },
                    "streaming-distribution": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "UpdateCachePolicy": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_UpdateCachePolicy.html",
                "description": "This action updates a cache policy.",
                "accessLevel": "Write"
            },
            "UpdateCloudFrontOriginAccessIdentity": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_UpdateCloudFrontOriginAccessIdentity.html",
                "description": "This action sets the configuration for a CloudFront origin access identity.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "origin-access-identity": {
                        "required": true
                    }
                }
            },
            "UpdateDistribution": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_UpdateDistribution.html",
                "description": "This action updates the configuration for a web distribution.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "distribution": {
                        "required": true
                    }
                }
            },
            "UpdateFieldLevelEncryptionConfig": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_UpdateFieldLevelEncryptionConfig.html",
                "description": "This action updates a field-level encryption configuration.",
                "accessLevel": "Write"
            },
            "UpdateFieldLevelEncryptionProfile": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_UpdateFieldLevelEncryptionProfile.html",
                "description": "This action updates a field-level encryption profile.",
                "accessLevel": "Write"
            },
            "UpdateOriginRequestPolicy": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_UpdateOriginRequestPolicy.html",
                "description": "This action updates an origin request policy.",
                "accessLevel": "Write"
            },
            "UpdatePublicKey": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_UpdatePublicKey.html",
                "description": "This action updates public key information.",
                "accessLevel": "Write"
            },
            "UpdateStreamingDistribution": {
                "url": "https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_UpdateStreamingDistribution.html",
                "description": "This action updates the configuration for an RTMP distribution.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "streaming-distribution": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "distribution": {
                "name": "distribution",
                "url": "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/distribution-working-with.html",
                "arn": "arn:${Partition}:cloudfront::${Account}:distribution/${DistributionId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "streaming-distribution": {
                "name": "streaming-distribution",
                "url": "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/distribution-working-with.html",
                "arn": "arn:${Partition}:cloudfront::${Account}:streaming-distribution/${DistributionId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "origin-access-identity": {
                "name": "origin-access-identity",
                "url": "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/private-content-restricting-access-to-s3.html#private-content-restricting-access-to-s3-overview",
                "arn": "arn:${Partition}:cloudfront::${Account}:origin-access-identity/${Id}",
                "conditionKeys": []
            },
            "field-level-encryption": {
                "name": "field-level-encryption",
                "url": "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/field-level-encryption.html",
                "arn": "arn:${Partition}:cloudfront::${Account}:field-level-encryption/${Id}",
                "conditionKeys": []
            },
            "field-level-encryption-profile": {
                "name": "field-level-encryption-profile",
                "url": "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/field-level-encryption.html",
                "arn": "arn:${Partition}:cloudfront::${Account}:field-level-encryption-profile/${Id}",
                "conditionKeys": []
            },
            "cache-policy": {
                "name": "cache-policy",
                "url": "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/working-with-policies.html",
                "arn": "arn:${Partition}:cloudfront::${Account}:cache-policy/${Id}",
                "conditionKeys": []
            },
            "origin-request-policy": {
                "name": "origin-request-policy",
                "url": "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/working-with-policies.html",
                "arn": "arn:${Partition}:cloudfront::${Account}:origin-request-policy/${Id}",
                "conditionKeys": []
            }
        };
    }
    /**
     * This action adds a new cache policy to CloudFront.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreateCachePolicy.html
     */
    toCreateCachePolicy() {
        this.add('cloudfront:CreateCachePolicy');
        return this;
    }
    /**
     * This action creates a new CloudFront origin access identity.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreateCloudFrontOriginAccessIdentity.html
     */
    toCreateCloudFrontOriginAccessIdentity() {
        this.add('cloudfront:CreateCloudFrontOriginAccessIdentity');
        return this;
    }
    /**
     * This action creates a new web distribution.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreateDistribution.html
     */
    toCreateDistribution() {
        this.add('cloudfront:CreateDistribution');
        return this;
    }
    /**
     * This action creates a new web distribution with tags.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreateDistributionWithTags.html
     */
    toCreateDistributionWithTags() {
        this.add('cloudfront:CreateDistributionWithTags');
        return this;
    }
    /**
     * This action creates a new field-level encryption configuration.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreateFieldLevelEncryptionConfig.html
     */
    toCreateFieldLevelEncryptionConfig() {
        this.add('cloudfront:CreateFieldLevelEncryptionConfig');
        return this;
    }
    /**
     * This action creates a field-level encryption profile.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreateFieldLevelEncryptionProfile.html
     */
    toCreateFieldLevelEncryptionProfile() {
        this.add('cloudfront:CreateFieldLevelEncryptionProfile');
        return this;
    }
    /**
     * This action creates a new invalidation batch request.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreateInvalidation.html
     */
    toCreateInvalidation() {
        this.add('cloudfront:CreateInvalidation');
        return this;
    }
    /**
     * This action adds a new origin request policy to CloudFront.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreateOriginRequestPolicy.html
     */
    toCreateOriginRequestPolicy() {
        this.add('cloudfront:CreateOriginRequestPolicy');
        return this;
    }
    /**
     * This action adds a new public key to CloudFront.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreatePublicKey.html
     */
    toCreatePublicKey() {
        this.add('cloudfront:CreatePublicKey');
        return this;
    }
    /**
     * This action creates a new RTMP distribution.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreateStreamingDistribution.html
     */
    toCreateStreamingDistribution() {
        this.add('cloudfront:CreateStreamingDistribution');
        return this;
    }
    /**
     * This action creates a new RTMP distribution with tags.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreateStreamingDistributionWithTags.html
     */
    toCreateStreamingDistributionWithTags() {
        this.add('cloudfront:CreateStreamingDistributionWithTags');
        return this;
    }
    /**
     * This action deletes a cache policy.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_DeleteCachePolicy.html
     */
    toDeleteCachePolicy() {
        this.add('cloudfront:DeleteCachePolicy');
        return this;
    }
    /**
     * This action deletes a CloudFront origin access identity.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_DeleteCloudFrontOriginAccessIdentity.html
     */
    toDeleteCloudFrontOriginAccessIdentity() {
        this.add('cloudfront:DeleteCloudFrontOriginAccessIdentity');
        return this;
    }
    /**
     * This action deletes a web distribution.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_DeleteDistribution.html
     */
    toDeleteDistribution() {
        this.add('cloudfront:DeleteDistribution');
        return this;
    }
    /**
     * This action deletes a field-level encryption configuration.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_DeleteFieldLevelEncryptionConfig.html
     */
    toDeleteFieldLevelEncryptionConfig() {
        this.add('cloudfront:DeleteFieldLevelEncryptionConfig');
        return this;
    }
    /**
     * This action deletes a field-level encryption profile.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_DeleteFieldLevelEncryptionProfile.html
     */
    toDeleteFieldLevelEncryptionProfile() {
        this.add('cloudfront:DeleteFieldLevelEncryptionProfile');
        return this;
    }
    /**
     * This action deletes an origin request policy.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_DeleteOriginRequestPolicy.html
     */
    toDeleteOriginRequestPolicy() {
        this.add('cloudfront:DeleteOriginRequestPolicy');
        return this;
    }
    /**
     * This action deletes a public key from CloudFront.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_DeletePublicKey.html
     */
    toDeletePublicKey() {
        this.add('cloudfront:DeletePublicKey');
        return this;
    }
    /**
     * This action deletes an RTMP distribution.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_DeleteStreamingDistribution.html
     */
    toDeleteStreamingDistribution() {
        this.add('cloudfront:DeleteStreamingDistribution');
        return this;
    }
    /**
     * Get the cache policy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetCachePolicy.html
     */
    toGetCachePolicy() {
        this.add('cloudfront:GetCachePolicy');
        return this;
    }
    /**
     * Get the cache policy configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetCachePolicyConfig.html
     */
    toGetCachePolicyConfig() {
        this.add('cloudfront:GetCachePolicyConfig');
        return this;
    }
    /**
     * Get the information about a CloudFront origin access identity.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetCloudFrontOriginAccessIdentity.html
     */
    toGetCloudFrontOriginAccessIdentity() {
        this.add('cloudfront:GetCloudFrontOriginAccessIdentity');
        return this;
    }
    /**
     * Get the configuration information about a Cloudfront origin access identity.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetCloudFrontOriginAccessIdentityConfig.html
     */
    toGetCloudFrontOriginAccessIdentityConfig() {
        this.add('cloudfront:GetCloudFrontOriginAccessIdentityConfig');
        return this;
    }
    /**
     * Get the information about a web distribution.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetDistribution.html
     */
    toGetDistribution() {
        this.add('cloudfront:GetDistribution');
        return this;
    }
    /**
     * Get the configuration information about a distribution.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetDistributionConfig.html
     */
    toGetDistributionConfig() {
        this.add('cloudfront:GetDistributionConfig');
        return this;
    }
    /**
     * Get the field-level encryption configuration information.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetFieldLevelEncryption.html
     */
    toGetFieldLevelEncryption() {
        this.add('cloudfront:GetFieldLevelEncryption');
        return this;
    }
    /**
     * Get the field-level encryption configuration information.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetFieldLevelEncryptionConfig.html
     */
    toGetFieldLevelEncryptionConfig() {
        this.add('cloudfront:GetFieldLevelEncryptionConfig');
        return this;
    }
    /**
     * Get the field-level encryption configuration information.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetFieldLevelEncryptionProfile.html
     */
    toGetFieldLevelEncryptionProfile() {
        this.add('cloudfront:GetFieldLevelEncryptionProfile');
        return this;
    }
    /**
     * Get the field-level encryption profile configuration information.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetFieldLevelEncryptionProfileConfig.html
     */
    toGetFieldLevelEncryptionProfileConfig() {
        this.add('cloudfront:GetFieldLevelEncryptionProfileConfig');
        return this;
    }
    /**
     * Get the information about an invalidation.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetInvalidation.html
     */
    toGetInvalidation() {
        this.add('cloudfront:GetInvalidation');
        return this;
    }
    /**
     * Get the origin request policy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetOriginRequestPolicy.html
     */
    toGetOriginRequestPolicy() {
        this.add('cloudfront:GetOriginRequestPolicy');
        return this;
    }
    /**
     * Get the origin request policy configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetOriginRequestPolicyConfig.html
     */
    toGetOriginRequestPolicyConfig() {
        this.add('cloudfront:GetOriginRequestPolicyConfig');
        return this;
    }
    /**
     * Get the public key information.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetPublicKey.html
     */
    toGetPublicKey() {
        this.add('cloudfront:GetPublicKey');
        return this;
    }
    /**
     * Get the public key configuration information.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetPublicKeyConfig.html
     */
    toGetPublicKeyConfig() {
        this.add('cloudfront:GetPublicKeyConfig');
        return this;
    }
    /**
     * Get the information about an RTMP distribution.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetStreamingDistribution.html
     */
    toGetStreamingDistribution() {
        this.add('cloudfront:GetStreamingDistribution');
        return this;
    }
    /**
     * Get the configuration information about a streaming distribution.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetStreamingDistributionConfig.html
     */
    toGetStreamingDistributionConfig() {
        this.add('cloudfront:GetStreamingDistributionConfig');
        return this;
    }
    /**
     * List all cache policies that have been created in CloudFront for this account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListCachePolicies.html
     */
    toListCachePolicies() {
        this.add('cloudfront:ListCachePolicies');
        return this;
    }
    /**
     * List your CloudFront origin access identities.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListCloudFrontOriginAccessIdentities.html
     */
    toListCloudFrontOriginAccessIdentities() {
        this.add('cloudfront:ListCloudFrontOriginAccessIdentities');
        return this;
    }
    /**
     * List the distributions associated with your AWS account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListDistributions.html
     */
    toListDistributions() {
        this.add('cloudfront:ListDistributions');
        return this;
    }
    /**
     * List distribution IDs for distributions that have a cache behavior that's associated with the specified cache policy.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListDistributionsByCachePolicyId.html
     */
    toListDistributionsByCachePolicyId() {
        this.add('cloudfront:ListDistributionsByCachePolicyId');
        return this;
    }
    /**
     * List distribution IDs for distributions that have a cache behavior that's associated with the specified origin request policy.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListDistributionsByOriginRequestPolicyId.html
     */
    toListDistributionsByOriginRequestPolicyId() {
        this.add('cloudfront:ListDistributionsByOriginRequestPolicyId');
        return this;
    }
    /**
     * List the distributions associated with your AWS account with given AWS WAF web ACL.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListDistributionsByWebACLId.html
     */
    toListDistributionsByWebACLId() {
        this.add('cloudfront:ListDistributionsByWebACLId');
        return this;
    }
    /**
     * List all field-level encryption configurations that have been created in CloudFront for this account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListFieldLevelEncryptionConfigs.html
     */
    toListFieldLevelEncryptionConfigs() {
        this.add('cloudfront:ListFieldLevelEncryptionConfigs');
        return this;
    }
    /**
     * List all field-level encryption profiles that have been created in CloudFront for this account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListFieldLevelEncryptionProfiles.html
     */
    toListFieldLevelEncryptionProfiles() {
        this.add('cloudfront:ListFieldLevelEncryptionProfiles');
        return this;
    }
    /**
     * List your invalidation batches.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListInvalidations.html
     */
    toListInvalidations() {
        this.add('cloudfront:ListInvalidations');
        return this;
    }
    /**
     * List all origin request policies that have been created in CloudFront for this account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListOriginRequestPolicies.html
     */
    toListOriginRequestPolicies() {
        this.add('cloudfront:ListOriginRequestPolicies');
        return this;
    }
    /**
     * List all public keys that have been added to CloudFront for this account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListPublicKeys.html
     */
    toListPublicKeys() {
        this.add('cloudfront:ListPublicKeys');
        return this;
    }
    /**
     * List your RTMP distributions.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListStreamingDistributions.html
     */
    toListStreamingDistributions() {
        this.add('cloudfront:ListStreamingDistributions');
        return this;
    }
    /**
     * List tags for a CloudFront resource.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.add('cloudfront:ListTagsForResource');
        return this;
    }
    /**
     * Add tags to a CloudFront resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        this.add('cloudfront:TagResource');
        return this;
    }
    /**
     * Remove tags from a CloudFront resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        this.add('cloudfront:UntagResource');
        return this;
    }
    /**
     * This action updates a cache policy.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_UpdateCachePolicy.html
     */
    toUpdateCachePolicy() {
        this.add('cloudfront:UpdateCachePolicy');
        return this;
    }
    /**
     * This action sets the configuration for a CloudFront origin access identity.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_UpdateCloudFrontOriginAccessIdentity.html
     */
    toUpdateCloudFrontOriginAccessIdentity() {
        this.add('cloudfront:UpdateCloudFrontOriginAccessIdentity');
        return this;
    }
    /**
     * This action updates the configuration for a web distribution.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_UpdateDistribution.html
     */
    toUpdateDistribution() {
        this.add('cloudfront:UpdateDistribution');
        return this;
    }
    /**
     * This action updates a field-level encryption configuration.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_UpdateFieldLevelEncryptionConfig.html
     */
    toUpdateFieldLevelEncryptionConfig() {
        this.add('cloudfront:UpdateFieldLevelEncryptionConfig');
        return this;
    }
    /**
     * This action updates a field-level encryption profile.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_UpdateFieldLevelEncryptionProfile.html
     */
    toUpdateFieldLevelEncryptionProfile() {
        this.add('cloudfront:UpdateFieldLevelEncryptionProfile');
        return this;
    }
    /**
     * This action updates an origin request policy.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_UpdateOriginRequestPolicy.html
     */
    toUpdateOriginRequestPolicy() {
        this.add('cloudfront:UpdateOriginRequestPolicy');
        return this;
    }
    /**
     * This action updates public key information.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_UpdatePublicKey.html
     */
    toUpdatePublicKey() {
        this.add('cloudfront:UpdatePublicKey');
        return this;
    }
    /**
     * This action updates the configuration for an RTMP distribution.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_UpdateStreamingDistribution.html
     */
    toUpdateStreamingDistribution() {
        this.add('cloudfront:UpdateStreamingDistribution');
        return this;
    }
    /**
     * Adds a resource of type distribution to the statement
     *
     * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/distribution-working-with.html
     *
     * @param distributionId - Identifier for the distributionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDistribution(distributionId, account, partition) {
        var arn = 'arn:${Partition}:cloudfront::${Account}:distribution/${DistributionId}';
        arn = arn.replace('${DistributionId}', distributionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type streaming-distribution to the statement
     *
     * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/distribution-working-with.html
     *
     * @param distributionId - Identifier for the distributionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onStreamingDistribution(distributionId, account, partition) {
        var arn = 'arn:${Partition}:cloudfront::${Account}:streaming-distribution/${DistributionId}';
        arn = arn.replace('${DistributionId}', distributionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type origin-access-identity to the statement
     *
     * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/private-content-restricting-access-to-s3.html#private-content-restricting-access-to-s3-overview
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onOriginAccessIdentity(id, account, partition) {
        var arn = 'arn:${Partition}:cloudfront::${Account}:origin-access-identity/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type field-level-encryption to the statement
     *
     * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/field-level-encryption.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onFieldLevelEncryption(id, account, partition) {
        var arn = 'arn:${Partition}:cloudfront::${Account}:field-level-encryption/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type field-level-encryption-profile to the statement
     *
     * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/field-level-encryption.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onFieldLevelEncryptionProfile(id, account, partition) {
        var arn = 'arn:${Partition}:cloudfront::${Account}:field-level-encryption-profile/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type cache-policy to the statement
     *
     * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/working-with-policies.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onCachePolicy(id, account, partition) {
        var arn = 'arn:${Partition}:cloudfront::${Account}:cache-policy/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type origin-request-policy to the statement
     *
     * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/working-with-policies.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onOriginRequestPolicy(id, account, partition) {
        var arn = 'arn:${Partition}:cloudfront::${Account}:origin-request-policy/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Cloudfront = Cloudfront;
//# sourceMappingURL=data:application/json;base64,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