"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Cloudsearch = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [cloudsearch](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoncloudsearch.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Cloudsearch extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [cloudsearch](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoncloudsearch.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'cloudsearch';
        this.actionList = {
            "AddTags": {
                "url": "https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_AddTags.html",
                "description": "Attaches resource tags to an Amazon CloudSearch domain.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "BuildSuggesters": {
                "url": "https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_BuildSuggesters.html",
                "description": "Indexes the search suggestions.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "CreateDomain": {
                "url": "https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_CreateDomain.html",
                "description": "Creates a new search domain.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "DefineAnalysisScheme": {
                "url": "https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DefineAnalysisScheme.html",
                "description": "Configures an analysis scheme that can be applied to a text or text-array field to define language-specific text processing options.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "DefineExpression": {
                "url": "https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DefineExpression.html",
                "description": "Configures an Expression for the search domain.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "DefineIndexField": {
                "url": "https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DefineIndexField.html",
                "description": "Configures an IndexField for the search domain.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "DefineSuggester": {
                "url": "https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DefineSuggester.html",
                "description": "Configures a suggester for a domain.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "DeleteAnalysisScheme": {
                "url": "https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DeleteAnalysisScheme.html",
                "description": "Deletes an analysis scheme.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "DeleteDomain": {
                "url": "https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DeleteDomain.html",
                "description": "Permanently deletes a search domain and all of its data.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "DeleteExpression": {
                "url": "https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DeleteExpression.html",
                "description": "Removes an Expression from the search domain.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "DeleteIndexField": {
                "url": "https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DeleteIndexField.html",
                "description": "Removes an IndexField from the search domain.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "DeleteSuggester": {
                "url": "https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DeleteSuggester.html",
                "description": "Deletes a suggester.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "DescribeAnalysisSchemes": {
                "url": "https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DescribeAnalysisSchemes.html",
                "description": "Gets the analysis schemes configured for a domain.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "DescribeAvailabilityOptions": {
                "url": "https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DescribeAvailabilityOptions.html",
                "description": "Gets the availability options configured for a domain.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "DescribeDomainEndpointOptions": {
                "url": "https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DescribeDomainEndpointOptions.html",
                "description": "Gets the domain endpoint options configured for a domain.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "DescribeDomains": {
                "url": "https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DescribeDomains.html",
                "description": "Gets information about the search domains owned by this account.",
                "accessLevel": "List",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "DescribeExpressions": {
                "url": "https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DescribeExpressions.html",
                "description": "Gets the expressions configured for the search domain.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "DescribeIndexFields": {
                "url": "https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DescribeIndexFields.html",
                "description": "Gets information about the index fields configured for the search domain.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "DescribeScalingParameters": {
                "url": "https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DescribeScalingParameters.html",
                "description": "Gets the scaling parameters configured for a domain.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "DescribeServiceAccessPolicies": {
                "url": "https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DescribeServiceAccessPolicies.html",
                "description": "Gets information about the access policies that control access to the domain's document and search endpoints.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "DescribeSuggesters": {
                "url": "https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DescribeSuggesters.html",
                "description": "Gets the suggesters configured for a domain.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "IndexDocuments": {
                "url": "https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_IndexDocuments.html",
                "description": "Tells the search domain to start indexing its documents using the latest indexing options.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "ListDomainNames": {
                "url": "https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_ListDomainNames.html",
                "description": "Lists all search domains owned by an account.",
                "accessLevel": "List",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "ListTags": {
                "url": "https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_ListTags.html",
                "description": "Displays all of the resource tags for an Amazon CloudSearch domain.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "RemoveTags": {
                "url": "https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_RemoveTags.html",
                "description": "Removes the specified resource tags from an Amazon ES domain.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "UpdateAvailabilityOptions": {
                "url": "https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_UpdateAvailabilityOptions.html",
                "description": "Configures the availability options for a domain.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "UpdateDomainEndpointOptions": {
                "url": "https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_UpdateDomainEndpointOptions.html",
                "description": "Configures the domain endpoint options for a domain.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "UpdateScalingParameters": {
                "url": "https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_UpdateScalingParameters.html",
                "description": "Configures scaling parameters for a domain.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "UpdateServiceAccessPolicies": {
                "url": "https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_UpdateServiceAccessPolicies.html",
                "description": "Configures the access rules that control access to the domain's document and search endpoints.",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "document": {
                "url": "https://docs.aws.amazon.com/cloudsearch/latest/developerguide/configuring-access.html#cloudsearch-actions",
                "description": "Allows access to the document service operations.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": false
                    }
                }
            },
            "search": {
                "url": "https://docs.aws.amazon.com/cloudsearch/latest/developerguide/configuring-access.html#cloudsearch-actions",
                "description": "Allows access to the search operations.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": false
                    }
                }
            },
            "suggest": {
                "url": "https://docs.aws.amazon.com/cloudsearch/latest/developerguide/configuring-access.html#cloudsearch-actions",
                "description": "Allows access to the suggest operations.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": false
                    }
                }
            }
        };
        this.resourceTypes = {
            "domain": {
                "name": "domain",
                "url": "https://docs.aws.amazon.com/cloudsearch/latest/developerguide/creating-domains.html",
                "arn": "arn:${Partition}:cloudsearch:${Region}:${Account}:domain/${DomainName}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Attaches resource tags to an Amazon CloudSearch domain.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_AddTags.html
     */
    toAddTags() {
        this.add('cloudsearch:AddTags');
        return this;
    }
    /**
     * Indexes the search suggestions.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_BuildSuggesters.html
     */
    toBuildSuggesters() {
        this.add('cloudsearch:BuildSuggesters');
        return this;
    }
    /**
     * Creates a new search domain.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_CreateDomain.html
     */
    toCreateDomain() {
        this.add('cloudsearch:CreateDomain');
        return this;
    }
    /**
     * Configures an analysis scheme that can be applied to a text or text-array field to define language-specific text processing options.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DefineAnalysisScheme.html
     */
    toDefineAnalysisScheme() {
        this.add('cloudsearch:DefineAnalysisScheme');
        return this;
    }
    /**
     * Configures an Expression for the search domain.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DefineExpression.html
     */
    toDefineExpression() {
        this.add('cloudsearch:DefineExpression');
        return this;
    }
    /**
     * Configures an IndexField for the search domain.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DefineIndexField.html
     */
    toDefineIndexField() {
        this.add('cloudsearch:DefineIndexField');
        return this;
    }
    /**
     * Configures a suggester for a domain.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DefineSuggester.html
     */
    toDefineSuggester() {
        this.add('cloudsearch:DefineSuggester');
        return this;
    }
    /**
     * Deletes an analysis scheme.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DeleteAnalysisScheme.html
     */
    toDeleteAnalysisScheme() {
        this.add('cloudsearch:DeleteAnalysisScheme');
        return this;
    }
    /**
     * Permanently deletes a search domain and all of its data.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DeleteDomain.html
     */
    toDeleteDomain() {
        this.add('cloudsearch:DeleteDomain');
        return this;
    }
    /**
     * Removes an Expression from the search domain.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DeleteExpression.html
     */
    toDeleteExpression() {
        this.add('cloudsearch:DeleteExpression');
        return this;
    }
    /**
     * Removes an IndexField from the search domain.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DeleteIndexField.html
     */
    toDeleteIndexField() {
        this.add('cloudsearch:DeleteIndexField');
        return this;
    }
    /**
     * Deletes a suggester.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DeleteSuggester.html
     */
    toDeleteSuggester() {
        this.add('cloudsearch:DeleteSuggester');
        return this;
    }
    /**
     * Gets the analysis schemes configured for a domain.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DescribeAnalysisSchemes.html
     */
    toDescribeAnalysisSchemes() {
        this.add('cloudsearch:DescribeAnalysisSchemes');
        return this;
    }
    /**
     * Gets the availability options configured for a domain.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DescribeAvailabilityOptions.html
     */
    toDescribeAvailabilityOptions() {
        this.add('cloudsearch:DescribeAvailabilityOptions');
        return this;
    }
    /**
     * Gets the domain endpoint options configured for a domain.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DescribeDomainEndpointOptions.html
     */
    toDescribeDomainEndpointOptions() {
        this.add('cloudsearch:DescribeDomainEndpointOptions');
        return this;
    }
    /**
     * Gets information about the search domains owned by this account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DescribeDomains.html
     */
    toDescribeDomains() {
        this.add('cloudsearch:DescribeDomains');
        return this;
    }
    /**
     * Gets the expressions configured for the search domain.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DescribeExpressions.html
     */
    toDescribeExpressions() {
        this.add('cloudsearch:DescribeExpressions');
        return this;
    }
    /**
     * Gets information about the index fields configured for the search domain.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DescribeIndexFields.html
     */
    toDescribeIndexFields() {
        this.add('cloudsearch:DescribeIndexFields');
        return this;
    }
    /**
     * Gets the scaling parameters configured for a domain.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DescribeScalingParameters.html
     */
    toDescribeScalingParameters() {
        this.add('cloudsearch:DescribeScalingParameters');
        return this;
    }
    /**
     * Gets information about the access policies that control access to the domain's document and search endpoints.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DescribeServiceAccessPolicies.html
     */
    toDescribeServiceAccessPolicies() {
        this.add('cloudsearch:DescribeServiceAccessPolicies');
        return this;
    }
    /**
     * Gets the suggesters configured for a domain.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_DescribeSuggesters.html
     */
    toDescribeSuggesters() {
        this.add('cloudsearch:DescribeSuggesters');
        return this;
    }
    /**
     * Tells the search domain to start indexing its documents using the latest indexing options.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_IndexDocuments.html
     */
    toIndexDocuments() {
        this.add('cloudsearch:IndexDocuments');
        return this;
    }
    /**
     * Lists all search domains owned by an account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_ListDomainNames.html
     */
    toListDomainNames() {
        this.add('cloudsearch:ListDomainNames');
        return this;
    }
    /**
     * Displays all of the resource tags for an Amazon CloudSearch domain.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_ListTags.html
     */
    toListTags() {
        this.add('cloudsearch:ListTags');
        return this;
    }
    /**
     * Removes the specified resource tags from an Amazon ES domain.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_RemoveTags.html
     */
    toRemoveTags() {
        this.add('cloudsearch:RemoveTags');
        return this;
    }
    /**
     * Configures the availability options for a domain.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_UpdateAvailabilityOptions.html
     */
    toUpdateAvailabilityOptions() {
        this.add('cloudsearch:UpdateAvailabilityOptions');
        return this;
    }
    /**
     * Configures the domain endpoint options for a domain.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_UpdateDomainEndpointOptions.html
     */
    toUpdateDomainEndpointOptions() {
        this.add('cloudsearch:UpdateDomainEndpointOptions');
        return this;
    }
    /**
     * Configures scaling parameters for a domain.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_UpdateScalingParameters.html
     */
    toUpdateScalingParameters() {
        this.add('cloudsearch:UpdateScalingParameters');
        return this;
    }
    /**
     * Configures the access rules that control access to the domain's document and search endpoints.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/API_UpdateServiceAccessPolicies.html
     */
    toUpdateServiceAccessPolicies() {
        this.add('cloudsearch:UpdateServiceAccessPolicies');
        return this;
    }
    /**
     * Allows access to the document service operations.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/configuring-access.html#cloudsearch-actions
     */
    todocument() {
        this.add('cloudsearch:document');
        return this;
    }
    /**
     * Allows access to the search operations.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/configuring-access.html#cloudsearch-actions
     */
    tosearch() {
        this.add('cloudsearch:search');
        return this;
    }
    /**
     * Allows access to the suggest operations.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/configuring-access.html#cloudsearch-actions
     */
    tosuggest() {
        this.add('cloudsearch:suggest');
        return this;
    }
    /**
     * Adds a resource of type domain to the statement
     *
     * https://docs.aws.amazon.com/cloudsearch/latest/developerguide/creating-domains.html
     *
     * @param domainName - Identifier for the domainName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onDomain(domainName, account, region, partition) {
        var arn = 'arn:${Partition}:cloudsearch:${Region}:${Account}:domain/${DomainName}';
        arn = arn.replace('${DomainName}', domainName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Cloudsearch = Cloudsearch;
//# sourceMappingURL=data:application/json;base64,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