"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CodeguruProfiler = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [codeguru-profiler](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoncodeguruprofiler.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class CodeguruProfiler extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [codeguru-profiler](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoncodeguruprofiler.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'codeguru-profiler';
        this.actionList = {
            "AddNotificationChannels": {
                "url": "https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_AddNotificationChannels.html",
                "description": "Grants permission to add up to 2 topic ARNs of existing AWS SNS topics to publish notifications",
                "accessLevel": "Write",
                "resourceTypes": {
                    "ProfilingGroup": {
                        "required": true
                    }
                }
            },
            "BatchGetFrameMetricData": {
                "url": "https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_BatchGetFrameMetricData.html",
                "description": "Grants permission to get the frame metric data for a Profiling Group",
                "accessLevel": "List",
                "resourceTypes": {
                    "ProfilingGroup": {
                        "required": true
                    }
                }
            },
            "ConfigureAgent": {
                "url": "https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_ConfigureAgent.html",
                "description": "Grants permission for an agent to register with the orchestration service and retrieve profiling configuration information",
                "accessLevel": "Write",
                "resourceTypes": {
                    "ProfilingGroup": {
                        "required": true
                    }
                }
            },
            "CreateProfilingGroup": {
                "url": "https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_CreateProfilingGroup.html",
                "description": "Grants permission to create a profiling group",
                "accessLevel": "Write",
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "DeleteProfilingGroup": {
                "url": "https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_DeleteProfilingGroup.html",
                "description": "Grants permission to delete a profiling group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "ProfilingGroup": {
                        "required": true
                    }
                }
            },
            "DescribeProfilingGroup": {
                "url": "https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_DescribeProfilingGroup.html",
                "description": "Grants permission to describe a profiling group",
                "accessLevel": "Read",
                "resourceTypes": {
                    "ProfilingGroup": {
                        "required": true
                    }
                }
            },
            "GetFindingsReportAccountSummary": {
                "url": "https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_GetFindingsReportAccountSummary.html",
                "description": "Grants permission to get a summary of recent recommendations for each profiling group in the account",
                "accessLevel": "Read"
            },
            "GetNotificationConfiguration": {
                "url": "https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_GetNotificationConfiguration.html",
                "description": "Grants permission to get the notification configuration",
                "accessLevel": "Read",
                "resourceTypes": {
                    "ProfilingGroup": {
                        "required": true
                    }
                }
            },
            "GetPolicy": {
                "url": "https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_GetPolicy.html",
                "description": "Grants permission to get the resource policy associated with the specified Profiling Group.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "ProfilingGroup": {
                        "required": true
                    }
                }
            },
            "GetProfile": {
                "url": "https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_GetProfile.html",
                "description": "Grants permission to get aggregated profiles for a specific profiling group",
                "accessLevel": "Read",
                "resourceTypes": {
                    "ProfilingGroup": {
                        "required": true
                    }
                }
            },
            "GetRecommendations": {
                "url": "https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_GetRecommendations.html",
                "description": "Grants permission to get recommendations",
                "accessLevel": "Read",
                "resourceTypes": {
                    "ProfilingGroup": {
                        "required": true
                    }
                }
            },
            "ListFindingsReports": {
                "url": "https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_ListFindingsReports.html",
                "description": "Grants permission to list the available recommendations reports for a specific profiling group",
                "accessLevel": "List",
                "resourceTypes": {
                    "ProfilingGroup": {
                        "required": true
                    }
                }
            },
            "ListProfileTimes": {
                "url": "https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_ListProfileTimes.html",
                "description": "Grants permission to list the start times of the available aggregated profiles for a specific profiling group",
                "accessLevel": "List",
                "resourceTypes": {
                    "ProfilingGroup": {
                        "required": true
                    }
                }
            },
            "ListProfilingGroups": {
                "url": "https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_ListProfilingGroups.html",
                "description": "Grants permission to list profiling groups in the account",
                "accessLevel": "List"
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_ListTagsForResource.html",
                "description": "Grants permission to list tags for a Profiling Group",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "ProfilingGroup": {
                        "required": true
                    }
                }
            },
            "PostAgentProfile": {
                "url": "https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_PostAgentProfile.html",
                "description": "Grants permission to submit a profile collected by an agent belonging to a specific profiling group for aggregation",
                "accessLevel": "Write",
                "resourceTypes": {
                    "ProfilingGroup": {
                        "required": true
                    }
                }
            },
            "PutPermission": {
                "url": "https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_PutPermission.html",
                "description": "Grants permission to update the list of principals allowed for an action group in the resource policy associated with the specified Profiling Group.",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "ProfilingGroup": {
                        "required": true
                    }
                }
            },
            "RemoveNotificationChannel": {
                "url": "https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_RemoveNotificationChannel.html",
                "description": "Grants permission to delete an already configured SNStopic arn from the notification configuration",
                "accessLevel": "Write",
                "resourceTypes": {
                    "ProfilingGroup": {
                        "required": true
                    }
                }
            },
            "RemovePermission": {
                "url": "https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_RemovePermission.html",
                "description": "Grants permission to remove the permission of specified Action Group from the resource policy associated with the specified Profiling Group.",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "ProfilingGroup": {
                        "required": true
                    }
                }
            },
            "SubmitFeedback": {
                "url": "https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_SubmitFeedback.html",
                "description": "Grants permission to submit user feedback for useful or non useful anomaly",
                "accessLevel": "Write",
                "resourceTypes": {
                    "ProfilingGroup": {
                        "required": true
                    }
                }
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_TagResource.html",
                "description": "Grants permission to add or overwrite tags to a Profiling Group",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "ProfilingGroup": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_UntagResource.html",
                "description": "Grants permission to remove tags from a Profiling Group",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "ProfilingGroup": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "UpdateProfilingGroup": {
                "url": "https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_UpdateProfilingGroup.html",
                "description": "Grants permission to update a specific profiling group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "ProfilingGroup": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "ProfilingGroup": {
                "name": "ProfilingGroup",
                "url": "https://docs.aws.amazon.com/codeguru/latest/profiler-ug/working-with-profiling-groups.html",
                "arn": "arn:${Partition}:codeguru-profiler:${Region}:${Account}:profilingGroup/${ProfilingGroupName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Grants permission to add up to 2 topic ARNs of existing AWS SNS topics to publish notifications
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_AddNotificationChannels.html
     */
    toAddNotificationChannels() {
        this.add('codeguru-profiler:AddNotificationChannels');
        return this;
    }
    /**
     * Grants permission to get the frame metric data for a Profiling Group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_BatchGetFrameMetricData.html
     */
    toBatchGetFrameMetricData() {
        this.add('codeguru-profiler:BatchGetFrameMetricData');
        return this;
    }
    /**
     * Grants permission for an agent to register with the orchestration service and retrieve profiling configuration information
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_ConfigureAgent.html
     */
    toConfigureAgent() {
        this.add('codeguru-profiler:ConfigureAgent');
        return this;
    }
    /**
     * Grants permission to create a profiling group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_CreateProfilingGroup.html
     */
    toCreateProfilingGroup() {
        this.add('codeguru-profiler:CreateProfilingGroup');
        return this;
    }
    /**
     * Grants permission to delete a profiling group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_DeleteProfilingGroup.html
     */
    toDeleteProfilingGroup() {
        this.add('codeguru-profiler:DeleteProfilingGroup');
        return this;
    }
    /**
     * Grants permission to describe a profiling group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_DescribeProfilingGroup.html
     */
    toDescribeProfilingGroup() {
        this.add('codeguru-profiler:DescribeProfilingGroup');
        return this;
    }
    /**
     * Grants permission to get a summary of recent recommendations for each profiling group in the account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_GetFindingsReportAccountSummary.html
     */
    toGetFindingsReportAccountSummary() {
        this.add('codeguru-profiler:GetFindingsReportAccountSummary');
        return this;
    }
    /**
     * Grants permission to get the notification configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_GetNotificationConfiguration.html
     */
    toGetNotificationConfiguration() {
        this.add('codeguru-profiler:GetNotificationConfiguration');
        return this;
    }
    /**
     * Grants permission to get the resource policy associated with the specified Profiling Group.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_GetPolicy.html
     */
    toGetPolicy() {
        this.add('codeguru-profiler:GetPolicy');
        return this;
    }
    /**
     * Grants permission to get aggregated profiles for a specific profiling group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_GetProfile.html
     */
    toGetProfile() {
        this.add('codeguru-profiler:GetProfile');
        return this;
    }
    /**
     * Grants permission to get recommendations
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_GetRecommendations.html
     */
    toGetRecommendations() {
        this.add('codeguru-profiler:GetRecommendations');
        return this;
    }
    /**
     * Grants permission to list the available recommendations reports for a specific profiling group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_ListFindingsReports.html
     */
    toListFindingsReports() {
        this.add('codeguru-profiler:ListFindingsReports');
        return this;
    }
    /**
     * Grants permission to list the start times of the available aggregated profiles for a specific profiling group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_ListProfileTimes.html
     */
    toListProfileTimes() {
        this.add('codeguru-profiler:ListProfileTimes');
        return this;
    }
    /**
     * Grants permission to list profiling groups in the account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_ListProfilingGroups.html
     */
    toListProfilingGroups() {
        this.add('codeguru-profiler:ListProfilingGroups');
        return this;
    }
    /**
     * Grants permission to list tags for a Profiling Group
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.add('codeguru-profiler:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to submit a profile collected by an agent belonging to a specific profiling group for aggregation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_PostAgentProfile.html
     */
    toPostAgentProfile() {
        this.add('codeguru-profiler:PostAgentProfile');
        return this;
    }
    /**
     * Grants permission to update the list of principals allowed for an action group in the resource policy associated with the specified Profiling Group.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_PutPermission.html
     */
    toPutPermission() {
        this.add('codeguru-profiler:PutPermission');
        return this;
    }
    /**
     * Grants permission to delete an already configured SNStopic arn from the notification configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_RemoveNotificationChannel.html
     */
    toRemoveNotificationChannel() {
        this.add('codeguru-profiler:RemoveNotificationChannel');
        return this;
    }
    /**
     * Grants permission to remove the permission of specified Action Group from the resource policy associated with the specified Profiling Group.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_RemovePermission.html
     */
    toRemovePermission() {
        this.add('codeguru-profiler:RemovePermission');
        return this;
    }
    /**
     * Grants permission to submit user feedback for useful or non useful anomaly
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_SubmitFeedback.html
     */
    toSubmitFeedback() {
        this.add('codeguru-profiler:SubmitFeedback');
        return this;
    }
    /**
     * Grants permission to add or overwrite tags to a Profiling Group
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_TagResource.html
     */
    toTagResource() {
        this.add('codeguru-profiler:TagResource');
        return this;
    }
    /**
     * Grants permission to remove tags from a Profiling Group
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_UntagResource.html
     */
    toUntagResource() {
        this.add('codeguru-profiler:UntagResource');
        return this;
    }
    /**
     * Grants permission to update a specific profiling group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-api/API_UpdateProfilingGroup.html
     */
    toUpdateProfilingGroup() {
        this.add('codeguru-profiler:UpdateProfilingGroup');
        return this;
    }
    /**
     * Adds a resource of type ProfilingGroup to the statement
     *
     * https://docs.aws.amazon.com/codeguru/latest/profiler-ug/working-with-profiling-groups.html
     *
     * @param profilingGroupName - Identifier for the profilingGroupName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onProfilingGroup(profilingGroupName, account, region, partition) {
        var arn = 'arn:${Partition}:codeguru-profiler:${Region}:${Account}:profilingGroup/${ProfilingGroupName}';
        arn = arn.replace('${ProfilingGroupName}', profilingGroupName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.CodeguruProfiler = CodeguruProfiler;
//# sourceMappingURL=data:application/json;base64,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