"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CognitoSync = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [cognito-sync](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoncognitosync.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class CognitoSync extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [cognito-sync](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoncognitosync.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'cognito-sync';
        this.actionList = {
            "BulkPublish": {
                "url": "https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_BulkPublish.html",
                "description": "Initiates a bulk publish of all existing datasets for an Identity Pool to the configured stream.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "identitypool": {
                        "required": true
                    }
                }
            },
            "DeleteDataset": {
                "url": "https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_DeleteDataset.html",
                "description": "Deletes the specific dataset.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dataset": {
                        "required": true
                    }
                }
            },
            "DescribeDataset": {
                "url": "https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_DescribeDataset.html",
                "description": "Gets meta data about a dataset by identity and dataset name.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "dataset": {
                        "required": true
                    }
                }
            },
            "DescribeIdentityPoolUsage": {
                "url": "https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_DescribeIdentityPoolUsage.html",
                "description": "Gets usage details (for example, data storage) about a particular identity pool.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "identitypool": {
                        "required": true
                    }
                }
            },
            "DescribeIdentityUsage": {
                "url": "https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_DescribeIdentityUsage.html",
                "description": "Gets usage information for an identity, including number of datasets and data usage.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "identity": {
                        "required": true
                    }
                }
            },
            "GetBulkPublishDetails": {
                "url": "https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_GetBulkPublishDetails.html",
                "description": "Get the status of the last BulkPublish operation for an identity pool.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "identitypool": {
                        "required": true
                    }
                }
            },
            "GetCognitoEvents": {
                "url": "https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_GetCognitoEvents.html",
                "description": "Gets the events and the corresponding Lambda functions associated with an identity pool.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "identitypool": {
                        "required": true
                    }
                }
            },
            "GetIdentityPoolConfiguration": {
                "url": "https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_GetIdentityPoolConfiguration.html",
                "description": "Gets the configuration settings of an identity pool.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "identitypool": {
                        "required": true
                    }
                }
            },
            "ListDatasets": {
                "url": "https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_ListDatasets.html",
                "description": "Lists datasets for an identity.",
                "accessLevel": "List",
                "resourceTypes": {
                    "dataset": {
                        "required": true
                    }
                }
            },
            "ListIdentityPoolUsage": {
                "url": "https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_ListIdentityPoolUsage.html",
                "description": "Gets a list of identity pools registered with Cognito.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "identitypool": {
                        "required": true
                    }
                }
            },
            "ListRecords": {
                "url": "https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_ListRecords.html",
                "description": "Gets paginated records, optionally changed after a particular sync count for a dataset and identity.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "dataset": {
                        "required": true
                    }
                }
            },
            "QueryRecords": {
                "url": "",
                "description": "A permission that grants the ability to query records.",
                "accessLevel": "Read"
            },
            "RegisterDevice": {
                "url": "https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_RegisterDevice.html",
                "description": "Registers a device to receive push sync notifications.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "identity": {
                        "required": true
                    }
                }
            },
            "SetCognitoEvents": {
                "url": "https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_SetCognitoEvents.html",
                "description": "Sets the AWS Lambda function for a given event type for an identity pool.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "identitypool": {
                        "required": true
                    }
                }
            },
            "SetDatasetConfiguration": {
                "url": "",
                "description": "A permission that grants ability to configure datasets.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dataset": {
                        "required": true
                    }
                }
            },
            "SetIdentityPoolConfiguration": {
                "url": "https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_SetIdentityPoolConfiguration.html",
                "description": "Sets the necessary configuration for push sync.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "identitypool": {
                        "required": true
                    }
                }
            },
            "SubscribeToDataset": {
                "url": "https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_SubscribeToDataset.html",
                "description": "Subscribes to receive notifications when a dataset is modified by another device.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dataset": {
                        "required": true
                    }
                }
            },
            "UnsubscribeFromDataset": {
                "url": "https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_UnsubscribeFromDataset.html",
                "description": "Unsubscribes from receiving notifications when a dataset is modified by another device.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dataset": {
                        "required": true
                    }
                }
            },
            "UpdateRecords": {
                "url": "https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_UpdateRecords.html",
                "description": "Posts updates to records and adds and deletes records for a dataset and user.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dataset": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "dataset": {
                "name": "dataset",
                "url": "https://docs.aws.amazon.com/cognito/latest/developerguide/synchronizing-data.html#understanding-datasets",
                "arn": "arn:${Partition}:cognito-sync:${Region}:${Account}:identitypool/${IdentityPoolId}/identity/${IdentityId}/dataset/${DatasetName}",
                "conditionKeys": []
            },
            "identity": {
                "name": "identity",
                "url": "https://docs.aws.amazon.com/cognito/latest/developerguide/identity-pools.html#authenticated-and-unauthenticated-identities",
                "arn": "arn:${Partition}:cognito-sync:${Region}:${Account}:identitypool/${IdentityPoolId}/identity/${IdentityId}",
                "conditionKeys": []
            },
            "identitypool": {
                "name": "identitypool",
                "url": "https://docs.aws.amazon.com/cognito/latest/developerguide/identity-pools.html",
                "arn": "arn:${Partition}:cognito-sync:${Region}:${Account}:identitypool/${IdentityPoolId}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Initiates a bulk publish of all existing datasets for an Identity Pool to the configured stream.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_BulkPublish.html
     */
    toBulkPublish() {
        this.add('cognito-sync:BulkPublish');
        return this;
    }
    /**
     * Deletes the specific dataset.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_DeleteDataset.html
     */
    toDeleteDataset() {
        this.add('cognito-sync:DeleteDataset');
        return this;
    }
    /**
     * Gets meta data about a dataset by identity and dataset name.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_DescribeDataset.html
     */
    toDescribeDataset() {
        this.add('cognito-sync:DescribeDataset');
        return this;
    }
    /**
     * Gets usage details (for example, data storage) about a particular identity pool.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_DescribeIdentityPoolUsage.html
     */
    toDescribeIdentityPoolUsage() {
        this.add('cognito-sync:DescribeIdentityPoolUsage');
        return this;
    }
    /**
     * Gets usage information for an identity, including number of datasets and data usage.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_DescribeIdentityUsage.html
     */
    toDescribeIdentityUsage() {
        this.add('cognito-sync:DescribeIdentityUsage');
        return this;
    }
    /**
     * Get the status of the last BulkPublish operation for an identity pool.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_GetBulkPublishDetails.html
     */
    toGetBulkPublishDetails() {
        this.add('cognito-sync:GetBulkPublishDetails');
        return this;
    }
    /**
     * Gets the events and the corresponding Lambda functions associated with an identity pool.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_GetCognitoEvents.html
     */
    toGetCognitoEvents() {
        this.add('cognito-sync:GetCognitoEvents');
        return this;
    }
    /**
     * Gets the configuration settings of an identity pool.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_GetIdentityPoolConfiguration.html
     */
    toGetIdentityPoolConfiguration() {
        this.add('cognito-sync:GetIdentityPoolConfiguration');
        return this;
    }
    /**
     * Lists datasets for an identity.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_ListDatasets.html
     */
    toListDatasets() {
        this.add('cognito-sync:ListDatasets');
        return this;
    }
    /**
     * Gets a list of identity pools registered with Cognito.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_ListIdentityPoolUsage.html
     */
    toListIdentityPoolUsage() {
        this.add('cognito-sync:ListIdentityPoolUsage');
        return this;
    }
    /**
     * Gets paginated records, optionally changed after a particular sync count for a dataset and identity.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_ListRecords.html
     */
    toListRecords() {
        this.add('cognito-sync:ListRecords');
        return this;
    }
    /**
     * A permission that grants the ability to query records.
     *
     * Access Level: Read
     */
    toQueryRecords() {
        this.add('cognito-sync:QueryRecords');
        return this;
    }
    /**
     * Registers a device to receive push sync notifications.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_RegisterDevice.html
     */
    toRegisterDevice() {
        this.add('cognito-sync:RegisterDevice');
        return this;
    }
    /**
     * Sets the AWS Lambda function for a given event type for an identity pool.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_SetCognitoEvents.html
     */
    toSetCognitoEvents() {
        this.add('cognito-sync:SetCognitoEvents');
        return this;
    }
    /**
     * A permission that grants ability to configure datasets.
     *
     * Access Level: Write
     */
    toSetDatasetConfiguration() {
        this.add('cognito-sync:SetDatasetConfiguration');
        return this;
    }
    /**
     * Sets the necessary configuration for push sync.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_SetIdentityPoolConfiguration.html
     */
    toSetIdentityPoolConfiguration() {
        this.add('cognito-sync:SetIdentityPoolConfiguration');
        return this;
    }
    /**
     * Subscribes to receive notifications when a dataset is modified by another device.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_SubscribeToDataset.html
     */
    toSubscribeToDataset() {
        this.add('cognito-sync:SubscribeToDataset');
        return this;
    }
    /**
     * Unsubscribes from receiving notifications when a dataset is modified by another device.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_UnsubscribeFromDataset.html
     */
    toUnsubscribeFromDataset() {
        this.add('cognito-sync:UnsubscribeFromDataset');
        return this;
    }
    /**
     * Posts updates to records and adds and deletes records for a dataset and user.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_UpdateRecords.html
     */
    toUpdateRecords() {
        this.add('cognito-sync:UpdateRecords');
        return this;
    }
    /**
     * Adds a resource of type dataset to the statement
     *
     * https://docs.aws.amazon.com/cognito/latest/developerguide/synchronizing-data.html#understanding-datasets
     *
     * @param identityPoolId - Identifier for the identityPoolId.
     * @param identityId - Identifier for the identityId.
     * @param datasetName - Identifier for the datasetName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onDataset(identityPoolId, identityId, datasetName, account, region, partition) {
        var arn = 'arn:${Partition}:cognito-sync:${Region}:${Account}:identitypool/${IdentityPoolId}/identity/${IdentityId}/dataset/${DatasetName}';
        arn = arn.replace('${IdentityPoolId}', identityPoolId);
        arn = arn.replace('${IdentityId}', identityId);
        arn = arn.replace('${DatasetName}', datasetName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type identity to the statement
     *
     * https://docs.aws.amazon.com/cognito/latest/developerguide/identity-pools.html#authenticated-and-unauthenticated-identities
     *
     * @param identityPoolId - Identifier for the identityPoolId.
     * @param identityId - Identifier for the identityId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onIdentity(identityPoolId, identityId, account, region, partition) {
        var arn = 'arn:${Partition}:cognito-sync:${Region}:${Account}:identitypool/${IdentityPoolId}/identity/${IdentityId}';
        arn = arn.replace('${IdentityPoolId}', identityPoolId);
        arn = arn.replace('${IdentityId}', identityId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type identitypool to the statement
     *
     * https://docs.aws.amazon.com/cognito/latest/developerguide/identity-pools.html
     *
     * @param identityPoolId - Identifier for the identityPoolId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onIdentitypool(identityPoolId, account, region, partition) {
        var arn = 'arn:${Partition}:cognito-sync:${Region}:${Account}:identitypool/${IdentityPoolId}';
        arn = arn.replace('${IdentityPoolId}', identityPoolId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.CognitoSync = CognitoSync;
//# sourceMappingURL=data:application/json;base64,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