"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Ec2 = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [ec2](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonec2.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Ec2 extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [ec2](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonec2.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'ec2';
        this.actionList = {
            "AcceptReservedInstancesExchangeQuote": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AcceptReservedInstancesExchangeQuote.html",
                "description": "Grants permission to accept a Convertible Reserved Instance exchange quote",
                "accessLevel": "Write"
            },
            "AcceptTransitGatewayPeeringAttachment": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AcceptTransitGatewayPeeringAttachment.html",
                "description": "Grants permission to accept a transit gateway peering attachment request",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway-attachment": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "AcceptTransitGatewayVpcAttachment": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AcceptTransitGatewayVpcAttachment.html",
                "description": "Grants permission to accept a request to attach a VPC to a transit gateway",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway-attachment": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "AcceptVpcEndpointConnections": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AcceptVpcEndpointConnections.html",
                "description": "Grants permission to accept one or more interface VPC endpoint connections to your VPC endpoint service",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vpc-endpoint-service": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "AcceptVpcPeeringConnection": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AcceptVpcPeeringConnection.html",
                "description": "Grants permission to accept a VPC peering connection request",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vpc": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Tenancy"
                        ]
                    },
                    "vpc-peering-connection": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:AccepterVpc",
                            "ec2:Region",
                            "ec2:RequesterVpc",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "AdvertiseByoipCidr": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AdvertiseByoipCidr.html",
                "description": "Grants permission to advertise an IP address range that is provisioned for use in AWS through bring your own IP addresses (BYOIP)",
                "accessLevel": "Write"
            },
            "AllocateAddress": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AllocateAddress.html",
                "description": "Grants permission to allocate an Elastic IP address (EIP) to your account",
                "accessLevel": "Write"
            },
            "AllocateHosts": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AllocateHosts.html",
                "description": "Grants permission to allocate a Dedicated Host to your account",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dedicated-host": {
                        "required": true
                    }
                }
            },
            "ApplySecurityGroupsToClientVpnTargetNetwork": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ApplySecurityGroupsToClientVpnTargetNetwork.html",
                "description": "Grants permission to apply a security group to the association between a Client VPN endpoint and a target network",
                "accessLevel": "Write",
                "resourceTypes": {
                    "client-vpn-endpoint": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "security-group": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "vpc": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "AssignIpv6Addresses": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssignIpv6Addresses.html",
                "description": "Grants permission to assign one or more IPv6 addresses to a network interface",
                "accessLevel": "Write"
            },
            "AssignPrivateIpAddresses": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssignPrivateIpAddresses.html",
                "description": "Grants permission to assign one or more secondary private IP addresses to a network interface",
                "accessLevel": "Write"
            },
            "AssociateAddress": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssociateAddress.html",
                "description": "Grants permission to associate an Elastic IP address (EIP) with an instance or a network interface",
                "accessLevel": "Write"
            },
            "AssociateClientVpnTargetNetwork": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssociateClientVpnTargetNetwork.html",
                "description": "Grants permission to associate a target network with a Client VPN endpoint",
                "accessLevel": "Write",
                "resourceTypes": {
                    "client-vpn-endpoint": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "subnet": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "AssociateDhcpOptions": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssociateDhcpOptions.html",
                "description": "Grants permission to associate or disassociate a set of DHCP options with a VPC",
                "accessLevel": "Write"
            },
            "AssociateIamInstanceProfile": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssociateIamInstanceProfile.html",
                "description": "Grants permission to associate an IAM instance profile with a running or stopped instance",
                "accessLevel": "Write",
                "dependentActions": [
                    "iam:PassRole"
                ],
                "resourceTypes": {
                    "instance": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:AvailabilityZone",
                            "ec2:EbsOptimized",
                            "ec2:InstanceProfile",
                            "ec2:InstanceType",
                            "ec2:PlacementGroup",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:RootDeviceType",
                            "ec2:Tenancy"
                        ]
                    }
                }
            },
            "AssociateRouteTable": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssociateRouteTable.html",
                "description": "Grants permission to associate a subnet or gateway with a route table",
                "accessLevel": "Write"
            },
            "AssociateSubnetCidrBlock": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssociateSubnetCidrBlock.html",
                "description": "Grants permission to associate a CIDR block with a subnet",
                "accessLevel": "Write"
            },
            "AssociateTransitGatewayMulticastDomain": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssociateTransitGatewayMulticastDomain.html",
                "description": "Grants permission to associate an attachment and list of subnets with a transit gateway multicast domain",
                "accessLevel": "Write",
                "resourceTypes": {
                    "subnet": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-attachment": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-multicast-domain": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "AssociateTransitGatewayRouteTable": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssociateTransitGatewayRouteTable.html",
                "description": "Grants permission to associate an attachment with a transit gateway route table",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway-attachment": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-route-table": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "AssociateVpcCidrBlock": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssociateVpcCidrBlock.html",
                "description": "Grants permission to associate a CIDR block with a VPC",
                "accessLevel": "Write"
            },
            "AttachClassicLinkVpc": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AttachClassicLinkVpc.html",
                "description": "Grants permission to link an EC2-Classic instance to a ClassicLink-enabled VPC through one or more of the VPC's security groups",
                "accessLevel": "Write",
                "resourceTypes": {
                    "instance": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:AvailabilityZone",
                            "ec2:EbsOptimized",
                            "ec2:InstanceProfile",
                            "ec2:InstanceType",
                            "ec2:PlacementGroup",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:RootDeviceType",
                            "ec2:Tenancy"
                        ]
                    },
                    "security-group": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    },
                    "vpc": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Tenancy"
                        ]
                    }
                }
            },
            "AttachInternetGateway": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AttachInternetGateway.html",
                "description": "Grants permission to attach an internet gateway to a VPC",
                "accessLevel": "Write"
            },
            "AttachNetworkInterface": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AttachNetworkInterface.html",
                "description": "Grants permission to attach a network interface to an instance",
                "accessLevel": "Write"
            },
            "AttachVolume": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AttachVolume.html",
                "description": "Grants permission to attach an EBS volume to a running or stopped instance and expose it to the instance with the specified device name",
                "accessLevel": "Write",
                "resourceTypes": {
                    "instance": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:AvailabilityZone",
                            "ec2:EbsOptimized",
                            "ec2:InstanceProfile",
                            "ec2:InstanceType",
                            "ec2:PlacementGroup",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:RootDeviceType",
                            "ec2:Tenancy"
                        ]
                    },
                    "volume": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:AvailabilityZone",
                            "ec2:Encrypted",
                            "ec2:ParentSnapshot",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:VolumeIops",
                            "ec2:VolumeSize",
                            "ec2:VolumeType"
                        ]
                    }
                }
            },
            "AttachVpnGateway": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AttachVpnGateway.html",
                "description": "Grants permission to attach a virtual private gateway to a VPC",
                "accessLevel": "Write"
            },
            "AuthorizeClientVpnIngress": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AuthorizeClientVpnIngress.html",
                "description": "Grants permission to add an inbound authorization rule to a Client VPN endpoint",
                "accessLevel": "Write",
                "resourceTypes": {
                    "client-vpn-endpoint": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "AuthorizeSecurityGroupEgress": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AuthorizeSecurityGroupEgress.html",
                "description": "Grants permission to add one or more outbound rules to a VPC security group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "security-group": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    }
                }
            },
            "AuthorizeSecurityGroupIngress": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AuthorizeSecurityGroupIngress.html",
                "description": "Grants permission to add one or more inbound rules to a security group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "security-group": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    }
                }
            },
            "BundleInstance": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_BundleInstance.html",
                "description": "Grants permission to bundle an instance store-backed Windows instance",
                "accessLevel": "Write"
            },
            "CancelBundleTask": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CancelBundleTask.html",
                "description": "Grants permission to cancel a bundling operation",
                "accessLevel": "Write"
            },
            "CancelCapacityReservation": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CancelCapacityReservation.html",
                "description": "Grants permission to cancel a Capacity Reservation and release the reserved capacity",
                "accessLevel": "Write",
                "resourceTypes": {
                    "capacity-reservation": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "CancelConversionTask": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CancelConversionTask.html",
                "description": "Grants permission to cancel an active conversion task",
                "accessLevel": "Write"
            },
            "CancelExportTask": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CancelExportTask.html",
                "description": "Grants permission to cancel an active export task",
                "accessLevel": "Write"
            },
            "CancelImportTask": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CancelImportTask.html",
                "description": "Grants permission to cancel an in-process import virtual machine or import snapshot task",
                "accessLevel": "Write"
            },
            "CancelReservedInstancesListing": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CancelReservedInstancesListing.html",
                "description": "Grants permission to cancel a Reserved Instance listing on the Reserved Instance Marketplace",
                "accessLevel": "Write"
            },
            "CancelSpotFleetRequests": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CancelSpotFleetRequests.html",
                "description": "Grants permission to cancel one or more Spot Fleet requests",
                "accessLevel": "Write"
            },
            "CancelSpotInstanceRequests": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CancelSpotInstanceRequests.html",
                "description": "Grants permission to cancel one or more Spot Instance requests",
                "accessLevel": "Write"
            },
            "ConfirmProductInstance": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ConfirmProductInstance.html",
                "description": "Grants permission to determine whether an owned product code is associated with an instance",
                "accessLevel": "Write"
            },
            "CopyFpgaImage": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CopyFpgaImage.html",
                "description": "Grants permission to copy a source Amazon FPGA image (AFI) to the current Region",
                "accessLevel": "Write"
            },
            "CopyImage": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CopyImage.html",
                "description": "Grants permission to copy an Amazon Machine Image (AMI) from a source Region to the current Region",
                "accessLevel": "Write"
            },
            "CopySnapshot": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CopySnapshot.html",
                "description": "Grants permission to copy a point-in-time snapshot of an EBS volume and store it in Amazon S3",
                "accessLevel": "Write",
                "resourceTypes": {
                    "snapshot": {
                        "required": true,
                        "conditions": [
                            "aws:TagKeys",
                            "aws:RequestTag/${TagKey}",
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateCapacityReservation": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateCapacityReservation.html",
                "description": "Grants permission to create a Capacity Reservation",
                "accessLevel": "Write",
                "resourceTypes": {
                    "capacity-reservation": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateCarrierGateway": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateCarrierGateway.html",
                "description": "Grants permission to create a carrier gateway and provides CSP connectivity to VPC customers.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "carrier-gateway": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateClientVpnEndpoint": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateClientVpnEndpoint.html",
                "description": "Grants permission to create a Client VPN endpoint",
                "accessLevel": "Write",
                "resourceTypes": {
                    "client-vpn-endpoint": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateClientVpnRoute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateClientVpnRoute.html",
                "description": "Grants permission to add a network route to a Client VPN endpoint's route table",
                "accessLevel": "Write",
                "resourceTypes": {
                    "client-vpn-endpoint": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "subnet": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "CreateCustomerGateway": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateCustomerGateway.html",
                "description": "Grants permission to create a customer gateway, which provides information to AWS about your customer gateway device",
                "accessLevel": "Write"
            },
            "CreateDefaultSubnet": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateDefaultSubnet.html",
                "description": "Grants permission to create a default subnet in a specified Availability Zone in a default VPC",
                "accessLevel": "Write"
            },
            "CreateDefaultVpc": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateDefaultVpc.html",
                "description": "Grants permission to create a default VPC with a default subnet in each Availability Zone",
                "accessLevel": "Write"
            },
            "CreateDhcpOptions": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateDhcpOptions.html",
                "description": "Grants permission to create a set of DHCP options for a VPC",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dhcp-options": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateEgressOnlyInternetGateway": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateEgressOnlyInternetGateway.html",
                "description": "Grants permission to create an egress-only internet gateway for a VPC",
                "accessLevel": "Write",
                "resourceTypes": {
                    "egress-only-internet-gateway": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateFleet": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateFleet.html",
                "description": "Grants permission to launch an EC2 Fleet",
                "accessLevel": "Write",
                "resourceTypes": {
                    "fleet": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateFlowLogs": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateFlowLogs.html",
                "description": "Grants permission to create one or more flow logs to capture IP traffic for a network interface",
                "accessLevel": "Write",
                "dependentActions": [
                    "iam:PassRole"
                ],
                "resourceTypes": {
                    "vpc-flow-log": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    },
                    "network-interface": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:AvailabilityZone",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Subnet",
                            "ec2:Vpc"
                        ]
                    },
                    "subnet": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:AvailabilityZone",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    },
                    "vpc": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Tenancy"
                        ]
                    }
                }
            },
            "CreateFpgaImage": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateFpgaImage.html",
                "description": "Grants permission to create an Amazon FPGA Image (AFI) from a design checkpoint (DCP)",
                "accessLevel": "Write",
                "resourceTypes": {
                    "fpga-image": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateImage": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateImage.html",
                "description": "Grants permission to create an Amazon EBS-backed AMI from a stopped or running Amazon EBS-backed instance",
                "accessLevel": "Write"
            },
            "CreateInstanceExportTask": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateInstanceExportTask.html",
                "description": "Grants permission to export a running or stopped instance to an Amazon S3 bucket",
                "accessLevel": "Write",
                "resourceTypes": {
                    "export-instance-task": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateInternetGateway": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateInternetGateway.html",
                "description": "Grants permission to create an internet gateway for a VPC",
                "accessLevel": "Write",
                "resourceTypes": {
                    "internet-gateway": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateKeyPair": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateKeyPair.html",
                "description": "Grants permission to create a 2048-bit RSA key pair",
                "accessLevel": "Write",
                "resourceTypes": {
                    "key-pair": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateLaunchTemplate": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateLaunchTemplate.html",
                "description": "Grants permission to create a launch template",
                "accessLevel": "Write",
                "resourceTypes": {
                    "launch-template": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateLaunchTemplateVersion": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateLaunchTemplateVersion.html",
                "description": "Grants permission to create a new version of a launch template",
                "accessLevel": "Write",
                "resourceTypes": {
                    "launch-template": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "CreateLocalGatewayRoute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateLocalGatewayRoute.html",
                "description": "Grants permission to create a static route for a local gateway route table",
                "accessLevel": "Write",
                "resourceTypes": {
                    "local-gateway-route-table": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "local-gateway-virtual-interface-group": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "CreateLocalGatewayRouteTableVpcAssociation": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateLocalGatewayRouteTableVpcAssociation.html",
                "description": "Grants permission to associate a VPC with a local gateway route table",
                "accessLevel": "Write",
                "resourceTypes": {
                    "local-gateway-route-table": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "local-gateway-route-table-vpc-association": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    },
                    "vpc": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Tenancy"
                        ]
                    }
                }
            },
            "CreateManagedPrefixList": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateManagedPrefixList.html",
                "description": "Grants permission to create a managed prefix list",
                "accessLevel": "Write",
                "resourceTypes": {
                    "prefix-list": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateNatGateway": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateNatGateway.html",
                "description": "Grants permission to create a NAT gateway in a subnet",
                "accessLevel": "Write",
                "resourceTypes": {
                    "natgateway": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateNetworkAcl": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateNetworkAcl.html",
                "description": "Grants permission to create a network ACL in a VPC",
                "accessLevel": "Write",
                "resourceTypes": {
                    "network-acl": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateNetworkAclEntry": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateNetworkAclEntry.html",
                "description": "Grants permission to create a numbered entry (a rule) in a network ACL",
                "accessLevel": "Write"
            },
            "CreateNetworkInterface": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateNetworkInterface.html",
                "description": "Grants permission to create a network interface in a subnet",
                "accessLevel": "Write",
                "resourceTypes": {
                    "network-interface": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateNetworkInterfacePermission": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateNetworkInterfacePermission.html",
                "description": "Grants permission to create a permission for an AWS-authorized user to perform certain operations on a network interface",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "network-interface": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:AuthorizedUser",
                            "ec2:AvailabilityZone",
                            "ec2:Permission",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Subnet",
                            "ec2:Vpc",
                            "ec2:AuthorizedService"
                        ]
                    }
                }
            },
            "CreatePlacementGroup": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreatePlacementGroup.html",
                "description": "Grants permission to create a placement group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "placement-group": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateReservedInstancesListing": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateReservedInstancesListing.html",
                "description": "Grants permission to create a listing for Standard Reserved Instances to be sold in the Reserved Instance Marketplace",
                "accessLevel": "Write"
            },
            "CreateRoute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateRoute.html",
                "description": "Grants permission to create a route in a VPC route table",
                "accessLevel": "Write",
                "resourceTypes": {
                    "route-table": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    }
                }
            },
            "CreateRouteTable": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateRouteTable.html",
                "description": "Grants permission to create a route table for a VPC",
                "accessLevel": "Write"
            },
            "CreateSecurityGroup": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateSecurityGroup.html",
                "description": "Grants permission to create a security group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "security-group": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateSnapshot": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateSnapshot.html",
                "description": "Grants permission to create a snapshot of an EBS volume and store it in Amazon S3",
                "accessLevel": "Write",
                "resourceTypes": {
                    "snapshot": {
                        "required": true,
                        "conditions": [
                            "aws:TagKeys",
                            "aws:RequestTag/${TagKey}",
                            "ec2:ParentVolume",
                            "ec2:Region"
                        ]
                    },
                    "volume": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Encrypted",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:VolumeIops",
                            "ec2:VolumeSize",
                            "ec2:VolumeType"
                        ]
                    }
                }
            },
            "CreateSnapshots": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateSnapshots.html",
                "description": "Grants permission to create crash-consistent snapshots of multiple EBS volumes and store them in Amazon S3",
                "accessLevel": "Write",
                "resourceTypes": {
                    "instance": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:AvailabilityZone",
                            "ec2:EbsOptimized",
                            "ec2:InstanceProfile",
                            "ec2:InstanceType",
                            "ec2:PlacementGroup",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:RootDeviceType",
                            "ec2:Tenancy"
                        ]
                    },
                    "snapshot": {
                        "required": true,
                        "conditions": [
                            "aws:TagKeys",
                            "aws:RequestTag/${TagKey}",
                            "ec2:ParentVolume",
                            "ec2:Region"
                        ]
                    },
                    "volume": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Encrypted",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:VolumeIops",
                            "ec2:VolumeSize",
                            "ec2:VolumeType"
                        ]
                    }
                }
            },
            "CreateSpotDatafeedSubscription": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateSpotDatafeedSubscription.html",
                "description": "Grants permission to create a data feed for Spot Instances to view Spot Instance usage logs",
                "accessLevel": "Write"
            },
            "CreateSubnet": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateSubnet.html",
                "description": "Grants permission to create a subnet in a VPC",
                "accessLevel": "Write",
                "resourceTypes": {
                    "subnet": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateTags": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTags.html",
                "description": "Grants permission to add or overwrite one or more tags for Amazon EC2 resources",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "capacity-reservation": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "client-vpn-endpoint": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "dedicated-host": {
                        "required": false
                    },
                    "dhcp-options": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "egress-only-internet-gateway": {
                        "required": false
                    },
                    "export-instance-task": {
                        "required": false
                    },
                    "fleet": {
                        "required": false
                    },
                    "fpga-image": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Owner",
                            "ec2:Public",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "image": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:ImageType",
                            "ec2:Owner",
                            "ec2:Public",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:RootDeviceType"
                        ]
                    },
                    "instance": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:AvailabilityZone",
                            "ec2:EbsOptimized",
                            "ec2:InstanceProfile",
                            "ec2:InstanceType",
                            "ec2:PlacementGroup",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:RootDeviceType",
                            "ec2:Tenancy"
                        ]
                    },
                    "internet-gateway": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "local-gateway": {
                        "required": false
                    },
                    "local-gateway-route-table": {
                        "required": false
                    },
                    "local-gateway-route-table-virtual-interface-group-association": {
                        "required": false
                    },
                    "local-gateway-route-table-vpc-association": {
                        "required": false
                    },
                    "local-gateway-virtual-interface": {
                        "required": false
                    },
                    "local-gateway-virtual-interface-group": {
                        "required": false
                    },
                    "natgateway": {
                        "required": false
                    },
                    "network-acl": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    },
                    "network-interface": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:AvailabilityZone",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Subnet",
                            "ec2:Vpc",
                            "ec2:AssociatePublicIpAddress"
                        ]
                    },
                    "reserved-instances": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:AvailabilityZone",
                            "ec2:InstanceType",
                            "ec2:Region",
                            "ec2:ReservedInstancesOfferingType",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Tenancy"
                        ]
                    },
                    "route-table": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    },
                    "security-group": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    },
                    "snapshot": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Owner",
                            "ec2:ParentVolume",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:SnapshotTime",
                            "ec2:VolumeSize"
                        ]
                    },
                    "spot-instance-request": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "subnet": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:AvailabilityZone",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    },
                    "traffic-mirror-filter": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "traffic-mirror-session": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "traffic-mirror-target": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-attachment": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-multicast-domain": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-route-table": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "volume": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:AvailabilityZone",
                            "ec2:Encrypted",
                            "ec2:ParentSnapshot",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:VolumeIops",
                            "ec2:VolumeSize",
                            "ec2:VolumeType"
                        ]
                    },
                    "vpc": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Tenancy"
                        ]
                    },
                    "vpc-endpoint": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "vpc-endpoint-service": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "vpc-flow-log": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "vpn-connection": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "vpn-gateway": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                },
                "conditions": [
                    "ec2:CreateAction"
                ]
            },
            "CreateTrafficMirrorFilter": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTrafficMirrorFilter.html",
                "description": "Grants permission to create a traffic mirror filter",
                "accessLevel": "Write",
                "resourceTypes": {
                    "traffic-mirror-filter": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateTrafficMirrorFilterRule": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTrafficMirrorFilterRule.html",
                "description": "Grants permission to create a traffic mirror filter rule",
                "accessLevel": "Write",
                "resourceTypes": {
                    "traffic-mirror-filter": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "traffic-mirror-filter-rule": {
                        "required": true,
                        "conditions": [
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateTrafficMirrorSession": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTrafficMirrorSession.html",
                "description": "Grants permission to create a traffic mirror session",
                "accessLevel": "Write",
                "resourceTypes": {
                    "network-interface": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "traffic-mirror-filter": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "traffic-mirror-session": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    },
                    "traffic-mirror-target": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "CreateTrafficMirrorTarget": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTrafficMirrorTarget.html",
                "description": "Grants permission to create a traffic mirror target",
                "accessLevel": "Write",
                "resourceTypes": {
                    "traffic-mirror-target": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    },
                    "network-interface": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "CreateTransitGateway": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTransitGateway.html",
                "description": "Grants permission to create a transit gateway",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateTransitGatewayMulticastDomain": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTransitGatewayMulticastDomain.html",
                "description": "Grants permission to create a multicast domain for a transit gateway",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-multicast-domain": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateTransitGatewayPeeringAttachment": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTransitGatewayPeeringAttachment.html",
                "description": "Grants permission to request a transit gateway peering attachment between a requester and accepter transit gateway",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-attachment": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateTransitGatewayPrefixListReference": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTransitGatewayPrefixListReference.html",
                "description": "Grants permission to create a transit gateway prefix list reference",
                "accessLevel": "Write",
                "resourceTypes": {
                    "prefix-list": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-route-table": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-attachment": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "CreateTransitGatewayRoute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTransitGatewayRoute.html",
                "description": "Grants permission to create a static route for a transit gateway route table",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway-route-table": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-attachment": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "CreateTransitGatewayRouteTable": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTransitGatewayRouteTable.html",
                "description": "Grants permission to create a route table for a transit gateway",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-route-table": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateTransitGatewayVpcAttachment": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTransitGatewayVpcAttachment.html",
                "description": "Grants permission to attach a VPC to a transit gateway",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-attachment": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    },
                    "vpc": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "subnet": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "CreateVolume": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVolume.html",
                "description": "Grants permission to create an EBS volume",
                "accessLevel": "Write",
                "resourceTypes": {
                    "volume": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:AvailabilityZone",
                            "ec2:Encrypted",
                            "ec2:ParentSnapshot",
                            "ec2:Region",
                            "ec2:VolumeIops",
                            "ec2:VolumeSize",
                            "ec2:VolumeType"
                        ]
                    }
                }
            },
            "CreateVpc": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVpc.html",
                "description": "Grants permission to create a VPC with a specified CIDR block",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vpc": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateVpcEndpoint": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVpcEndpoint.html",
                "description": "Grants permission to create a VPC endpoint for an AWS service",
                "accessLevel": "Write",
                "dependentActions": [
                    "route53:AssociateVPCWithHostedZone"
                ],
                "resourceTypes": {
                    "vpc": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "vpc-endpoint": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:VpceServiceName",
                            "ec2:VpceServiceOwner"
                        ]
                    },
                    "route-table": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "security-group": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "subnet": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "CreateVpcEndpointConnectionNotification": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVpcEndpointConnectionNotification.html",
                "description": "Grants permission to create a connection notification for a VPC endpoint or VPC endpoint service",
                "accessLevel": "Write"
            },
            "CreateVpcEndpointServiceConfiguration": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVpcEndpointServiceConfiguration.html",
                "description": "Grants permission to create a VPC endpoint service configuration to which service consumers (AWS accounts, IAM users, and IAM roles) can connect",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vpc-endpoint-service": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:VpceServicePrivateDnsName"
                        ]
                    }
                }
            },
            "CreateVpcPeeringConnection": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVpcPeeringConnection.html",
                "description": "Grants permission to request a VPC peering connection between two VPCs",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vpc": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Tenancy"
                        ]
                    },
                    "vpc-peering-connection": {
                        "required": true,
                        "conditions": [
                            "ec2:AccepterVpc",
                            "ec2:Region",
                            "ec2:RequesterVpc"
                        ]
                    }
                }
            },
            "CreateVpnConnection": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVpnConnection.html",
                "description": "Grants permission to create a VPN connection between a virtual private gateway or transit gateway and a customer gateway",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vpn-connection": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:AuthenticationType",
                            "ec2:DPDTimeoutSeconds",
                            "ec2:GatewayType",
                            "ec2:IKEVersions",
                            "ec2:InsideTunnelCidr",
                            "ec2:Phase1DHGroupNumbers",
                            "ec2:Phase2DHGroupNumbers",
                            "ec2:Phase1EncryptionAlgorithms",
                            "ec2:Phase2EncryptionAlgorithms",
                            "ec2:Phase1IntegrityAlgorithms",
                            "ec2:Phase2IntegrityAlgorithms",
                            "ec2:Phase1LifetimeSeconds",
                            "ec2:Phase2LifetimeSeconds",
                            "ec2:PresharedKeys",
                            "ec2:RekeyFuzzPercentage",
                            "ec2:RekeyMarginTimeSeconds",
                            "ec2:RoutingType"
                        ]
                    }
                }
            },
            "CreateVpnConnectionRoute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVpnConnectionRoute.html",
                "description": "Grants permission to create a static route for a VPN connection between a virtual private gateway and a customer gateway",
                "accessLevel": "Write"
            },
            "CreateVpnGateway": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVpnGateway.html",
                "description": "Grants permission to create a virtual private gateway",
                "accessLevel": "Write"
            },
            "DeleteCarrierGateway": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteCarrierGateway.html",
                "description": "Grants permission to delete a carrier gateway",
                "accessLevel": "Write",
                "resourceTypes": {
                    "carrier-gateway": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "aws:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteClientVpnEndpoint": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteClientVpnEndpoint.html",
                "description": "Grants permission to delete a Client VPN endpoint",
                "accessLevel": "Write",
                "resourceTypes": {
                    "client-vpn-endpoint": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteClientVpnRoute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteClientVpnRoute.html",
                "description": "Grants permission to delete a route from a Client VPN endpoint",
                "accessLevel": "Write",
                "resourceTypes": {
                    "client-vpn-endpoint": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "subnet": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteCustomerGateway": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteCustomerGateway.html",
                "description": "Grants permission to delete a customer gateway",
                "accessLevel": "Write",
                "resourceTypes": {
                    "customer-gateway": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteDhcpOptions": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteDhcpOptions.html",
                "description": "Grants permission to delete a set of DHCP options",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dhcp-options": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteEgressOnlyInternetGateway": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteEgressOnlyInternetGateway.html",
                "description": "Grants permission to delete an egress-only internet gateway",
                "accessLevel": "Write"
            },
            "DeleteFleets": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteFleets.html",
                "description": "Grants permission to delete one or more EC2 Fleets",
                "accessLevel": "Write"
            },
            "DeleteFlowLogs": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteFlowLogs.html",
                "description": "Grants permission to delete one or more flow logs",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vpc-flow-log": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteFpgaImage": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteFpgaImage.html",
                "description": "Grants permission to delete an Amazon FPGA Image (AFI)",
                "accessLevel": "Write"
            },
            "DeleteInternetGateway": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteInternetGateway.html",
                "description": "Grants permission to delete an internet gateway",
                "accessLevel": "Write",
                "resourceTypes": {
                    "internet-gateway": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteKeyPair": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteKeyPair.html",
                "description": "Grants permission to delete a key pair by removing the public key from Amazon EC2",
                "accessLevel": "Write"
            },
            "DeleteLaunchTemplate": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteLaunchTemplate.html",
                "description": "Grants permission to delete a launch template and its associated versions",
                "accessLevel": "Write",
                "resourceTypes": {
                    "launch-template": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteLaunchTemplateVersions": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteLaunchTemplateVersions.html",
                "description": "Grants permission to delete one or more versions of a launch template",
                "accessLevel": "Write",
                "resourceTypes": {
                    "launch-template": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteLocalGatewayRoute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteLocalGatewayRoute.html",
                "description": "Grants permission to delete a route from a local gateway route table",
                "accessLevel": "Write",
                "resourceTypes": {
                    "local-gateway-route-table": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteLocalGatewayRouteTableVpcAssociation": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteLocalGatewayRouteTableVpcAssociation.html",
                "description": "Grants permission to delete an association between a VPC and local gateway route table",
                "accessLevel": "Write",
                "resourceTypes": {
                    "local-gateway-route-table-vpc-association": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteManagedPrefixList": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteManagedPrefixList.html",
                "description": "Grants permission to delete a managed prefix list",
                "accessLevel": "Write",
                "resourceTypes": {
                    "prefix-list": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteNatGateway": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteNatGateway.html",
                "description": "Grants permission to delete a NAT gateway",
                "accessLevel": "Write"
            },
            "DeleteNetworkAcl": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteNetworkAcl.html",
                "description": "Grants permission to delete a network ACL",
                "accessLevel": "Write",
                "resourceTypes": {
                    "network-acl": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    }
                }
            },
            "DeleteNetworkAclEntry": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteNetworkAclEntry.html",
                "description": "Grants permission to delete an inbound or outbound entry (rule) from a network ACL",
                "accessLevel": "Write",
                "resourceTypes": {
                    "network-acl": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    }
                }
            },
            "DeleteNetworkInterface": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteNetworkInterface.html",
                "description": "Grants permission to delete a detached network interface",
                "accessLevel": "Write"
            },
            "DeleteNetworkInterfacePermission": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteNetworkInterfacePermission.html",
                "description": "Grants permission to delete a permission that is associated with a network interface",
                "accessLevel": "Permissions management"
            },
            "DeletePlacementGroup": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeletePlacementGroup.html",
                "description": "Grants permission to delete a placement group",
                "accessLevel": "Write"
            },
            "DeleteRoute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteRoute.html",
                "description": "Grants permission to delete a route from a route table",
                "accessLevel": "Write",
                "resourceTypes": {
                    "route-table": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    }
                }
            },
            "DeleteRouteTable": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteRouteTable.html",
                "description": "Grants permission to delete a route table",
                "accessLevel": "Write",
                "resourceTypes": {
                    "route-table": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    }
                }
            },
            "DeleteSecurityGroup": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteSecurityGroup.html",
                "description": "Grants permission to delete a security group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "security-group": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    }
                }
            },
            "DeleteSnapshot": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteSnapshot.html",
                "description": "Grants permission to delete a snapshot of an EBS volume",
                "accessLevel": "Write",
                "resourceTypes": {
                    "snapshot": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Owner",
                            "ec2:ParentVolume",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:SnapshotTime",
                            "ec2:VolumeSize"
                        ]
                    }
                }
            },
            "DeleteSpotDatafeedSubscription": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteSpotDatafeedSubscription.html",
                "description": "Grants permission to delete a data feed for Spot Instances",
                "accessLevel": "Write"
            },
            "DeleteSubnet": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteSubnet.html",
                "description": "Grants permission to delete a subnet",
                "accessLevel": "Write"
            },
            "DeleteTags": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTags.html",
                "description": "Grants permission to delete one or more tags from Amazon EC2 resources",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "capacity-reservation": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "client-vpn-endpoint": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "dedicated-host": {
                        "required": false
                    },
                    "dhcp-options": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "egress-only-internet-gateway": {
                        "required": false
                    },
                    "export-instance-task": {
                        "required": false
                    },
                    "fleet": {
                        "required": false
                    },
                    "fpga-image": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "image": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "instance": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "internet-gateway": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "local-gateway": {
                        "required": false
                    },
                    "local-gateway-route-table": {
                        "required": false
                    },
                    "local-gateway-route-table-virtual-interface-group-association": {
                        "required": false
                    },
                    "local-gateway-route-table-vpc-association": {
                        "required": false
                    },
                    "local-gateway-virtual-interface": {
                        "required": false
                    },
                    "local-gateway-virtual-interface-group": {
                        "required": false
                    },
                    "natgateway": {
                        "required": false
                    },
                    "network-acl": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "network-interface": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "reserved-instances": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "route-table": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "security-group": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "snapshot": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "spot-instance-request": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "subnet": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-attachment": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-multicast-domain": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-route-table": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "volume": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "vpc": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "vpc-endpoint": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "vpc-endpoint-service": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "vpc-flow-log": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "vpn-connection": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "vpn-gateway": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteTrafficMirrorFilter": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTrafficMirrorFilter.html",
                "description": "Grants permission to delete a traffic mirror filter",
                "accessLevel": "Write",
                "resourceTypes": {
                    "traffic-mirror-filter": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteTrafficMirrorFilterRule": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTrafficMirrorFilterRule.html",
                "description": "Grants permission to delete a traffic mirror filter rule",
                "accessLevel": "Write",
                "resourceTypes": {
                    "traffic-mirror-filter": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "traffic-mirror-filter-rule": {
                        "required": true,
                        "conditions": [
                            "ec2:Region"
                        ]
                    }
                }
            },
            "DeleteTrafficMirrorSession": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTrafficMirrorSession.html",
                "description": "Grants permission to delete a traffic mirror session",
                "accessLevel": "Write",
                "resourceTypes": {
                    "traffic-mirror-session": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteTrafficMirrorTarget": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTrafficMirrorTarget.html",
                "description": "Grants permission to delete a traffic mirror target",
                "accessLevel": "Write",
                "resourceTypes": {
                    "traffic-mirror-target": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteTransitGateway": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTransitGateway.html",
                "description": "Grants permission to delete a transit gateway",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteTransitGatewayMulticastDomain": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTransitGatewayMulticastDomain.html",
                "description": "Grants permissions to delete a transit gateway multicast domain",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway-multicast-domain": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteTransitGatewayPeeringAttachment": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTransitGatewayPeeringAttachment.html",
                "description": "Grants permission to delete a peering attachment from a transit gateway",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway-attachment": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteTransitGatewayPrefixListReference": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTransitGatewayPrefixListReference.html",
                "description": "Grants permission to delete a transit gateway prefix list reference",
                "accessLevel": "Write",
                "resourceTypes": {
                    "prefix-list": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-route-table": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteTransitGatewayRoute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTransitGatewayRoute.html",
                "description": "Grants permission to delete a route from a transit gateway route table",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway-route-table": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteTransitGatewayRouteTable": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTransitGatewayRouteTable.html",
                "description": "Grants permission to delete a transit gateway route table",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway-route-table": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteTransitGatewayVpcAttachment": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTransitGatewayVpcAttachment.html",
                "description": "Grants permission to delete a VPC attachment from a transit gateway",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway-attachment": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteVolume": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteVolume.html",
                "description": "Grants permission to delete an EBS volume",
                "accessLevel": "Write",
                "resourceTypes": {
                    "volume": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:AvailabilityZone",
                            "ec2:Encrypted",
                            "ec2:ParentSnapshot",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:VolumeIops",
                            "ec2:VolumeSize",
                            "ec2:VolumeType"
                        ]
                    }
                }
            },
            "DeleteVpc": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteVpc.html",
                "description": "Grants permission to delete a VPC",
                "accessLevel": "Write"
            },
            "DeleteVpcEndpointConnectionNotifications": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteVpcEndpointConnectionNotifications.html",
                "description": "Grants permission to delete one or more VPC endpoint connection notifications",
                "accessLevel": "Write"
            },
            "DeleteVpcEndpointServiceConfigurations": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteVpcEndpointServiceConfigurations.html",
                "description": "Grants permission to delete one or more VPC endpoint service configurations",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vpc-endpoint-service": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteVpcEndpoints": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteVpcEndpoints.html",
                "description": "Grants permission to delete one or more VPC endpoints",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vpc-endpoint": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteVpcPeeringConnection": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteVpcPeeringConnection.html",
                "description": "Grants permission to delete a VPC peering connection",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vpc-peering-connection": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:AccepterVpc",
                            "ec2:Region",
                            "ec2:RequesterVpc",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteVpnConnection": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteVpnConnection.html",
                "description": "Grants permission to delete a VPN connection",
                "accessLevel": "Write"
            },
            "DeleteVpnConnectionRoute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteVpnConnectionRoute.html",
                "description": "Grants permission to delete a static route for a VPN connection between a virtual private gateway and a customer gateway",
                "accessLevel": "Write"
            },
            "DeleteVpnGateway": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteVpnGateway.html",
                "description": "Grants permission to delete a virtual private gateway",
                "accessLevel": "Write"
            },
            "DeprovisionByoipCidr": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeprovisionByoipCidr.html",
                "description": "Grants permission to release an IP address range that was provisioned through bring your own IP addresses (BYOIP), and to delete the corresponding address pool",
                "accessLevel": "Write"
            },
            "DeregisterImage": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeregisterImage.html",
                "description": "Grants permission to deregister an Amazon Machine Image (AMI)",
                "accessLevel": "Write"
            },
            "DeregisterInstanceEventNotificationAttributes": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeregisterInstanceEventNotificationAttributes.html",
                "description": "Grants permission to remove tags from the set of tags to include in notifications about scheduled events for your instances",
                "accessLevel": "Write"
            },
            "DeregisterTransitGatewayMulticastGroupMembers": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeregisterTransitGatewayMulticastGroupMembers.html",
                "description": "Grants permission to deregister one or more network interface members from a group IP address in a transit gateway multicast domain",
                "accessLevel": "Write",
                "resourceTypes": {
                    "network-interface": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-multicast-domain": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeregisterTransitGatewayMulticastGroupSources": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeregisterTransitGatewayMulticastGroupSources.html",
                "description": "Grants permission to deregister one or more network interface sources from a group IP address in a transit gateway multicast domain",
                "accessLevel": "Write",
                "resourceTypes": {
                    "network-interface": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-multicast-domain": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DescribeAccountAttributes": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeAccountAttributes.html",
                "description": "Grants permission to describe the attributes of the AWS account",
                "accessLevel": "List"
            },
            "DescribeAddresses": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeAddresses.html",
                "description": "Grants permission to describe one or more Elastic IP addresses",
                "accessLevel": "List"
            },
            "DescribeAggregateIdFormat": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeAggregateIdFormat.html",
                "description": "Grants permission to describe the longer ID format settings for all resource types",
                "accessLevel": "List"
            },
            "DescribeAvailabilityZones": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeAvailabilityZones.html",
                "description": "Grants permission to describe one or more of the Availability Zones that are available to you",
                "accessLevel": "List"
            },
            "DescribeBundleTasks": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeBundleTasks.html",
                "description": "Grants permission to describe one or more bundling tasks",
                "accessLevel": "List"
            },
            "DescribeByoipCidrs": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeByoipCidrs.html",
                "description": "Grants permission to describe the IP address ranges that were provisioned through bring your own IP addresses (BYOIP)",
                "accessLevel": "List"
            },
            "DescribeCapacityReservations": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeCapacityReservations.html",
                "description": "Grants permission to describe one or more Capacity Reservations",
                "accessLevel": "List"
            },
            "DescribeCarrierGateways": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeCarrierGateways.html",
                "description": "Grants permission to describe one or more Carrier Gateways",
                "accessLevel": "List"
            },
            "DescribeClassicLinkInstances": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeClassicLinkInstances.html",
                "description": "Grants permission to describe one or more linked EC2-Classic instances",
                "accessLevel": "List"
            },
            "DescribeClientVpnAuthorizationRules": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeClientVpnAuthorizationRules.html",
                "description": "Grants permission to describe the authorization rules for a Client VPN endpoint",
                "accessLevel": "List"
            },
            "DescribeClientVpnConnections": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeClientVpnConnections.html",
                "description": "Grants permission to describe active client connections and connections that have been terminated within the last 60 minutes for a Client VPN endpoint",
                "accessLevel": "List"
            },
            "DescribeClientVpnEndpoints": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeClientVpnEndpoints.html",
                "description": "Grants permission to describe one or more Client VPN endpoints",
                "accessLevel": "List"
            },
            "DescribeClientVpnRoutes": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeClientVpnRoutes.html",
                "description": "Grants permission to describe the routes for a Client VPN endpoint",
                "accessLevel": "List"
            },
            "DescribeClientVpnTargetNetworks": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeClientVpnTargetNetworks.html",
                "description": "Grants permission to describe the target networks that are associated with a Client VPN endpoint",
                "accessLevel": "List"
            },
            "DescribeCoipPools": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeCoipPools.html",
                "description": "Grants permission to describe the specified customer-owned address pools or all of your customer-owned address pools",
                "accessLevel": "List"
            },
            "DescribeConversionTasks": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeConversionTasks.html",
                "description": "Grants permission to describe one or more conversion tasks",
                "accessLevel": "List"
            },
            "DescribeCustomerGateways": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeCustomerGateways.html",
                "description": "Grants permission to describe one or more customer gateways",
                "accessLevel": "List"
            },
            "DescribeDhcpOptions": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeDhcpOptions.html",
                "description": "Grants permission to describe one or more DHCP options sets",
                "accessLevel": "List"
            },
            "DescribeEgressOnlyInternetGateways": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeEgressOnlyInternetGateways.html",
                "description": "Grants permission to describe one or more egress-only internet gateways",
                "accessLevel": "List"
            },
            "DescribeElasticGpus": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeElasticGpus.html",
                "description": "Grants permission to describe an Elastic Graphics accelerator that is associated with an instance",
                "accessLevel": "Read"
            },
            "DescribeExportImageTasks": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeExportImageTasks.html",
                "description": "Grants permission to describe one or more export image tasks",
                "accessLevel": "List"
            },
            "DescribeExportTasks": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeExportTasks.html",
                "description": "Grants permission to describe one or more export instance tasks",
                "accessLevel": "List"
            },
            "DescribeFastSnapshotRestores": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeFastSnapshotRestores.html",
                "description": "Grants permission to describe the state of fast snapshot restores for snapshots",
                "accessLevel": "Read"
            },
            "DescribeFleetHistory": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeFleetHistory.html",
                "description": "Grants permission to describe the events for an EC2 Fleet during a specified time",
                "accessLevel": "List"
            },
            "DescribeFleetInstances": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeFleetInstances.html",
                "description": "Grants permission to describe the running instances for an EC2 Fleet",
                "accessLevel": "List"
            },
            "DescribeFleets": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeFleets.html",
                "description": "Grants permission to describe one or more EC2 Fleets",
                "accessLevel": "List"
            },
            "DescribeFlowLogs": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeFlowLogs.html",
                "description": "Grants permission to describe one or more flow logs",
                "accessLevel": "List"
            },
            "DescribeFpgaImageAttribute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeFpgaImageAttribute.html",
                "description": "Grants permission to describe the attributes of an Amazon FPGA Image (AFI)",
                "accessLevel": "List"
            },
            "DescribeFpgaImages": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeFpgaImages.html",
                "description": "Grants permission to describe one or more Amazon FPGA Images (AFIs)",
                "accessLevel": "List"
            },
            "DescribeHostReservationOfferings": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeHostReservationOfferings.html",
                "description": "Grants permission to describe the Dedicated Host Reservations that are available to purchase",
                "accessLevel": "List"
            },
            "DescribeHostReservations": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeHostReservations.html",
                "description": "Grants permission to describe the Dedicated Host Reservations that are associated with Dedicated Hosts in the AWS account",
                "accessLevel": "List"
            },
            "DescribeHosts": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeHosts.html",
                "description": "Grants permission to describe one or more Dedicated Hosts",
                "accessLevel": "List"
            },
            "DescribeIamInstanceProfileAssociations": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeIamInstanceProfileAssociations.html",
                "description": "Grants permission to describe the IAM instance profile associations",
                "accessLevel": "List"
            },
            "DescribeIdFormat": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeIdFormat.html",
                "description": "Grants permission to describe the ID format settings for resources",
                "accessLevel": "List"
            },
            "DescribeIdentityIdFormat": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeIdentityIdFormat.html",
                "description": "Grants permission to describe the ID format settings for resources for an IAM user, IAM role, or root user",
                "accessLevel": "List"
            },
            "DescribeImageAttribute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeImageAttribute.html",
                "description": "Grants permission to describe an attribute of an Amazon Machine Image (AMI)",
                "accessLevel": "List"
            },
            "DescribeImages": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeImages.html",
                "description": "Grants permission to describe one or more images (AMIs, AKIs, and ARIs)",
                "accessLevel": "List"
            },
            "DescribeImportImageTasks": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeImportImageTasks.html",
                "description": "Grants permission to describe import virtual machine or import snapshot tasks",
                "accessLevel": "List"
            },
            "DescribeImportSnapshotTasks": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeImportSnapshotTasks.html",
                "description": "Grants permission to describe import snapshot tasks",
                "accessLevel": "List"
            },
            "DescribeInstanceAttribute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeInstanceAttribute.html",
                "description": "Grants permission to describe the attributes of an instance",
                "accessLevel": "List"
            },
            "DescribeInstanceCreditSpecifications": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeInstanceCreditSpecifications.html",
                "description": "Grants permission to describe the credit option for CPU usage of one or more burstable performance instances",
                "accessLevel": "List"
            },
            "DescribeInstanceEventNotificationAttributes": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeInstanceEventNotificationAttributes.html",
                "description": "Grants permission to describe the set of tags to include in notifications about scheduled events for your instances",
                "accessLevel": "List"
            },
            "DescribeInstanceStatus": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeInstanceStatus.html",
                "description": "Grants permission to describe the status of one or more instances",
                "accessLevel": "List"
            },
            "DescribeInstanceTypeOfferings": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeInstanceTypeOfferings.html",
                "description": "Grants permission to describe the set of instance types that are offered in a location",
                "accessLevel": "List"
            },
            "DescribeInstanceTypes": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeInstanceTypes.html",
                "description": "Grants permission to describe the details of instance types that are offered in a location",
                "accessLevel": "List"
            },
            "DescribeInstances": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeInstances.html",
                "description": "Grants permission to describe one or more instances",
                "accessLevel": "List"
            },
            "DescribeInternetGateways": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeInternetGateways.html",
                "description": "Grants permission to describe one or more internet gateways",
                "accessLevel": "List"
            },
            "DescribeKeyPairs": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeKeyPairs.html",
                "description": "Grants permission to describe one or more key pairs",
                "accessLevel": "List"
            },
            "DescribeLaunchTemplateVersions": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeLaunchTemplateVersions.html",
                "description": "Grants permission to describe one or more launch template versions",
                "accessLevel": "List"
            },
            "DescribeLaunchTemplates": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeLaunchTemplates.html",
                "description": "Grants permission to describe one or more launch templates",
                "accessLevel": "List"
            },
            "DescribeLocalGatewayRouteTableVirtualInterfaceGroupAssociations": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeLocalGatewayRouteTableVirtualInterfaceGroupAssociations.html",
                "description": "Grants permission to describe the associations between virtual interface groups and local gateway route tables",
                "accessLevel": "List"
            },
            "DescribeLocalGatewayRouteTableVpcAssociations": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeLocalGatewayRouteTableVpcAssociations.html",
                "description": "Grants permission to describe an association between VPCs and local gateway route tables",
                "accessLevel": "List"
            },
            "DescribeLocalGatewayRouteTables": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeLocalGatewayRouteTables.html",
                "description": "Grants permission to describe one or more local gateway route tables",
                "accessLevel": "List"
            },
            "DescribeLocalGatewayVirtualInterfaceGroups": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeLocalGatewayVirtualInterfaceGroups.html",
                "description": "Grants permission to describe local gateway virtual interface groups",
                "accessLevel": "List"
            },
            "DescribeLocalGatewayVirtualInterfaces": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeLocalGatewayVirtualInterfaces.html",
                "description": "Grants permission to describe local gateway virtual interfaces",
                "accessLevel": "List"
            },
            "DescribeLocalGateways": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeLocalGateways.html",
                "description": "Grants permission to describe one or more local gateways",
                "accessLevel": "List"
            },
            "DescribeManagedPrefixLists": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeManagedPrefixLists.html",
                "description": "Grants permission to describe your managed prefix lists and any AWS-managed prefix lists",
                "accessLevel": "List"
            },
            "DescribeMovingAddresses": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeMovingAddresses.html",
                "description": "Grants permission to describe Elastic IP addresses that are being moved to the EC2-VPC platform",
                "accessLevel": "List"
            },
            "DescribeNatGateways": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeNatGateways.html",
                "description": "Grants permission to describe one or more NAT gateways",
                "accessLevel": "List"
            },
            "DescribeNetworkAcls": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeNetworkAcls.html",
                "description": "Grants permission to describe one or more network ACLs",
                "accessLevel": "List"
            },
            "DescribeNetworkInterfaceAttribute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeNetworkInterfaceAttribute.html",
                "description": "Grants permission to describe a network interface attribute",
                "accessLevel": "List"
            },
            "DescribeNetworkInterfacePermissions": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeNetworkInterfacePermissions.html",
                "description": "Grants permission to describe the permissions that are associated with a network interface",
                "accessLevel": "List"
            },
            "DescribeNetworkInterfaces": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeNetworkInterfaces.html",
                "description": "Grants permission to describe one or more network interfaces",
                "accessLevel": "List"
            },
            "DescribePlacementGroups": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribePlacementGroups.html",
                "description": "Grants permission to describe one or more placement groups",
                "accessLevel": "List"
            },
            "DescribePrefixLists": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribePrefixLists.html",
                "description": "Grants permission to describe available AWS services in a prefix list format",
                "accessLevel": "List"
            },
            "DescribePrincipalIdFormat": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribePrincipalIdFormat.html",
                "description": "Grants permission to describe the ID format settings for the root user and all IAM roles and IAM users that have explicitly specified a longer ID (17-character ID) preference",
                "accessLevel": "List"
            },
            "DescribePublicIpv4Pools": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribePublicIpv4Pools.html",
                "description": "Grants permission to describe one or more IPv4 address pools",
                "accessLevel": "List"
            },
            "DescribeRegions": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeRegions.html",
                "description": "Grants permission to describe one or more AWS Regions that are currently available in your account",
                "accessLevel": "List"
            },
            "DescribeReservedInstances": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeReservedInstances.html",
                "description": "Grants permission to describe one or more purchased Reserved Instances in your account",
                "accessLevel": "List"
            },
            "DescribeReservedInstancesListings": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeReservedInstancesListings.html",
                "description": "Grants permission to describe your account's Reserved Instance listings in the Reserved Instance Marketplace",
                "accessLevel": "List"
            },
            "DescribeReservedInstancesModifications": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeReservedInstancesModifications.html",
                "description": "Grants permission to describe the modifications made to one or more Reserved Instances",
                "accessLevel": "List"
            },
            "DescribeReservedInstancesOfferings": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeReservedInstancesOfferings.html",
                "description": "Grants permission to describe the Reserved Instance offerings that are available for purchase",
                "accessLevel": "List"
            },
            "DescribeRouteTables": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeRouteTables.html",
                "description": "Grants permission to describe one or more route tables",
                "accessLevel": "List"
            },
            "DescribeScheduledInstanceAvailability": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeScheduledInstanceAvailability.html",
                "description": "Grants permission to find available schedules for Scheduled Instances",
                "accessLevel": "Read"
            },
            "DescribeScheduledInstances": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeScheduledInstances.html",
                "description": "Grants permission to describe one or more Scheduled Instances in your account",
                "accessLevel": "Read"
            },
            "DescribeSecurityGroupReferences": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSecurityGroupReferences.html",
                "description": "Grants permission to describe the VPCs on the other side of a VPC peering connection that are referencing specified VPC security groups",
                "accessLevel": "List"
            },
            "DescribeSecurityGroups": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSecurityGroups.html",
                "description": "Grants permission to describe one or more security groups",
                "accessLevel": "List"
            },
            "DescribeSnapshotAttribute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSnapshotAttribute.html",
                "description": "Grants permission to describe an attribute of a snapshot",
                "accessLevel": "List"
            },
            "DescribeSnapshots": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSnapshots.html",
                "description": "Grants permission to describe one or more EBS snapshots",
                "accessLevel": "List"
            },
            "DescribeSpotDatafeedSubscription": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSpotDatafeedSubscription.html",
                "description": "Grants permission to describe the data feed for Spot Instances",
                "accessLevel": "List"
            },
            "DescribeSpotFleetInstances": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSpotFleetInstances.html",
                "description": "Grants permission to describe the running instances for a Spot Fleet",
                "accessLevel": "List"
            },
            "DescribeSpotFleetRequestHistory": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSpotFleetRequestHistory.html",
                "description": "Grants permission to describe the events for a Spot Fleet request during a specified time",
                "accessLevel": "List"
            },
            "DescribeSpotFleetRequests": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSpotFleetRequests.html",
                "description": "Grants permission to describe one or more Spot Fleet requests",
                "accessLevel": "List"
            },
            "DescribeSpotInstanceRequests": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSpotInstanceRequests.html",
                "description": "Grants permission to describe one or more Spot Instance requests",
                "accessLevel": "List"
            },
            "DescribeSpotPriceHistory": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSpotPriceHistory.html",
                "description": "Grants permission to describe the Spot Instance price history",
                "accessLevel": "List"
            },
            "DescribeStaleSecurityGroups": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeStaleSecurityGroups.html",
                "description": "Grants permission to describe the stale security group rules for security groups in a specified VPC",
                "accessLevel": "List"
            },
            "DescribeSubnets": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSubnets.html",
                "description": "Grants permission to describe one or more subnets",
                "accessLevel": "List"
            },
            "DescribeTags": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeTags.html",
                "description": "Grants permission to describe one or more tags for an Amazon EC2 resource",
                "accessLevel": "Read"
            },
            "DescribeTrafficMirrorFilters": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeTrafficMirrorFilters.html",
                "description": "Grants permission to describe one or more traffic mirror filters",
                "accessLevel": "List"
            },
            "DescribeTrafficMirrorSessions": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeTrafficMirrorSessions.html",
                "description": "Grants permission to describe one or more traffic mirror sessions",
                "accessLevel": "List"
            },
            "DescribeTrafficMirrorTargets": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeTrafficMirrorTargets.html",
                "description": "Grants permission to describe one or more traffic mirror targets",
                "accessLevel": "List"
            },
            "DescribeTransitGatewayAttachments": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeTransitGatewayAttachments.html",
                "description": "Grants permission to describe one or more attachments between resources and transit gateways",
                "accessLevel": "List"
            },
            "DescribeTransitGatewayMulticastDomains": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeTransitGatewayMulticastDomains.html",
                "description": "Grants permission to describe one or more transit gateway multicast domains",
                "accessLevel": "List"
            },
            "DescribeTransitGatewayPeeringAttachments": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeTransitGatewayPeeringAttachments.html",
                "description": "Grants permission to describe one or more transit gateway peering attachments",
                "accessLevel": "List"
            },
            "DescribeTransitGatewayRouteTables": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeTransitGatewayRouteTables.html",
                "description": "Grants permission to describe one or more transit gateway route tables",
                "accessLevel": "List"
            },
            "DescribeTransitGatewayVpcAttachments": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeTransitGatewayVpcAttachments.html",
                "description": "Grants permission to describe one or more VPC attachments on a transit gateway",
                "accessLevel": "List"
            },
            "DescribeTransitGateways": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeTransitGateways.html",
                "description": "Grants permission to describe one or more transit gateways",
                "accessLevel": "List"
            },
            "DescribeVolumeAttribute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVolumeAttribute.html",
                "description": "Grants permission to describe an attribute of an EBS volume",
                "accessLevel": "List"
            },
            "DescribeVolumeStatus": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVolumeStatus.html",
                "description": "Grants permission to describe the status of one or more EBS volumes",
                "accessLevel": "List"
            },
            "DescribeVolumes": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVolumes.html",
                "description": "Grants permission to describe one or more EBS volumes",
                "accessLevel": "List"
            },
            "DescribeVolumesModifications": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVolumesModifications.html",
                "description": "Grants permission to describe the current modification status of one or more EBS volumes",
                "accessLevel": "Read"
            },
            "DescribeVpcAttribute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcAttribute.html",
                "description": "Grants permission to describe an attribute of a VPC",
                "accessLevel": "List"
            },
            "DescribeVpcClassicLink": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcClassicLink.html",
                "description": "Grants permission to describe the ClassicLink status of one or more VPCs",
                "accessLevel": "List"
            },
            "DescribeVpcClassicLinkDnsSupport": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcClassicLinkDnsSupport.html",
                "description": "Grants permission to describe the ClassicLink DNS support status of one or more VPCs",
                "accessLevel": "List"
            },
            "DescribeVpcEndpointConnectionNotifications": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcEndpointConnectionNotifications.html",
                "description": "Grants permission to describe the connection notifications for VPC endpoints and VPC endpoint services",
                "accessLevel": "List"
            },
            "DescribeVpcEndpointConnections": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcEndpointConnections.html",
                "description": "Grants permission to describe the VPC endpoint connections to your VPC endpoint services",
                "accessLevel": "List"
            },
            "DescribeVpcEndpointServiceConfigurations": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcEndpointServiceConfigurations.html",
                "description": "Grants permission to describe VPC endpoint service configurations (your services)",
                "accessLevel": "List"
            },
            "DescribeVpcEndpointServicePermissions": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcEndpointServicePermissions.html",
                "description": "Grants permission to describe the principals (service consumers) that are permitted to discover your VPC endpoint service",
                "accessLevel": "List"
            },
            "DescribeVpcEndpointServices": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcEndpointServices.html",
                "description": "Grants permission to describe all supported AWS services that can be specified when creating a VPC endpoint",
                "accessLevel": "List"
            },
            "DescribeVpcEndpoints": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcEndpoints.html",
                "description": "Grants permission to describe one or more VPC endpoints",
                "accessLevel": "List"
            },
            "DescribeVpcPeeringConnections": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcPeeringConnections.html",
                "description": "Grants permission to describe one or more VPC peering connections",
                "accessLevel": "List"
            },
            "DescribeVpcs": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcs.html",
                "description": "Grants permission to describe one or more VPCs",
                "accessLevel": "List"
            },
            "DescribeVpnConnections": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpnConnections.html",
                "description": "Grants permission to describe one or more VPN connections",
                "accessLevel": "Read"
            },
            "DescribeVpnGateways": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpnGateways.html",
                "description": "Grants permission to describe one or more virtual private gateways",
                "accessLevel": "List"
            },
            "DetachClassicLinkVpc": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DetachClassicLinkVpc.html",
                "description": "Grants permission to unlink (detach) a linked EC2-Classic instance from a VPC",
                "accessLevel": "Write",
                "resourceTypes": {
                    "instance": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:AvailabilityZone",
                            "ec2:EbsOptimized",
                            "ec2:InstanceProfile",
                            "ec2:InstanceType",
                            "ec2:PlacementGroup",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:RootDeviceType",
                            "ec2:Tenancy"
                        ]
                    },
                    "vpc": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Tenancy"
                        ]
                    }
                }
            },
            "DetachInternetGateway": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DetachInternetGateway.html",
                "description": "Grants permission to detach an internet gateway from a VPC",
                "accessLevel": "Write"
            },
            "DetachNetworkInterface": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DetachNetworkInterface.html",
                "description": "Grants permission to detach a network interface from an instance",
                "accessLevel": "Write"
            },
            "DetachVolume": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DetachVolume.html",
                "description": "Grants permission to detach an EBS volume from an instance",
                "accessLevel": "Write",
                "resourceTypes": {
                    "instance": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:AvailabilityZone",
                            "ec2:EbsOptimized",
                            "ec2:InstanceProfile",
                            "ec2:InstanceType",
                            "ec2:PlacementGroup",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:RootDeviceType",
                            "ec2:Tenancy"
                        ]
                    },
                    "volume": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:AvailabilityZone",
                            "ec2:Encrypted",
                            "ec2:ParentSnapshot",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:VolumeIops",
                            "ec2:VolumeSize",
                            "ec2:VolumeType"
                        ]
                    }
                }
            },
            "DetachVpnGateway": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DetachVpnGateway.html",
                "description": "Grants permission to detach a virtual private gateway from a VPC",
                "accessLevel": "Write"
            },
            "DisableEbsEncryptionByDefault": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisableEbsEncryptionByDefault.html",
                "description": "Grants permission to disable EBS encryption by default for your account",
                "accessLevel": "Write"
            },
            "DisableFastSnapshotRestores": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisableFastSnapshotRestores.html",
                "description": "Grants permission to disable fast snapshot restores for one or more snapshots in specified Availability Zones",
                "accessLevel": "Write",
                "resourceTypes": {
                    "snapshot": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Owner",
                            "ec2:ParentVolume",
                            "ec2:Region",
                            "ec2:AvailabilityZone",
                            "ec2:SnapshotTime",
                            "ec2:Encrypted",
                            "ec2:VolumeSize",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DisableTransitGatewayRouteTablePropagation": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisableTransitGatewayRouteTablePropagation.html",
                "description": "Grants permission to disable a resource attachment from propagating routes to the specified propagation route table",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway-attachment": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-route-table": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DisableVgwRoutePropagation": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisableVgwRoutePropagation.html",
                "description": "Grants permission to disable a virtual private gateway from propagating routes to a specified route table of a VPC",
                "accessLevel": "Write"
            },
            "DisableVpcClassicLink": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisableVpcClassicLink.html",
                "description": "Grants permission to disable ClassicLink for a VPC",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vpc": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Tenancy"
                        ]
                    }
                }
            },
            "DisableVpcClassicLinkDnsSupport": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisableVpcClassicLinkDnsSupport.html",
                "description": "Grants permission to disable ClassicLink DNS support for a VPC",
                "accessLevel": "Write"
            },
            "DisassociateAddress": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisassociateAddress.html",
                "description": "Grants permission to disassociate an Elastic IP address from an instance or network interface",
                "accessLevel": "Write"
            },
            "DisassociateClientVpnTargetNetwork": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisassociateClientVpnTargetNetwork.html",
                "description": "Grants permission to disassociate a target network from a Client VPN endpoint",
                "accessLevel": "Write",
                "resourceTypes": {
                    "client-vpn-endpoint": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DisassociateIamInstanceProfile": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisassociateIamInstanceProfile.html",
                "description": "Grants permission to disassociate an IAM instance profile from a running or stopped instance",
                "accessLevel": "Write",
                "resourceTypes": {
                    "instance": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:AvailabilityZone",
                            "ec2:EbsOptimized",
                            "ec2:InstanceProfile",
                            "ec2:InstanceType",
                            "ec2:PlacementGroup",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:RootDeviceType",
                            "ec2:Tenancy"
                        ]
                    }
                }
            },
            "DisassociateRouteTable": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisassociateRouteTable.html",
                "description": "Grants permission to disassociate a subnet from a route table",
                "accessLevel": "Write"
            },
            "DisassociateSubnetCidrBlock": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisassociateSubnetCidrBlock.html",
                "description": "Grants permission to disassociate a CIDR block from a subnet",
                "accessLevel": "Write"
            },
            "DisassociateTransitGatewayMulticastDomain": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisassociateTransitGatewayMulticastDomain.html",
                "description": "Grants permission to disassociate one or more subnets from a transit gateway multicast domain",
                "accessLevel": "Write",
                "resourceTypes": {
                    "subnet": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-attachment": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-multicast-domain": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DisassociateTransitGatewayRouteTable": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisassociateTransitGatewayRouteTable.html",
                "description": "Grants permission to disassociate a resource attachment from a transit gateway route table",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway-attachment": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-route-table": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DisassociateVpcCidrBlock": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisassociateVpcCidrBlock.html",
                "description": "Grants permission to disassociate a CIDR block from a VPC",
                "accessLevel": "Write"
            },
            "EnableEbsEncryptionByDefault": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_EnableEbsEncryptionByDefault.html",
                "description": "Grants permission to enable EBS encryption by default for your account",
                "accessLevel": "Write"
            },
            "EnableFastSnapshotRestores": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_EnableFastSnapshotRestores.html",
                "description": "Grants permission to enable fast snapshot restores for one or more snapshots in specified Availability Zones",
                "accessLevel": "Write",
                "resourceTypes": {
                    "snapshot": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Owner",
                            "ec2:ParentVolume",
                            "ec2:Region",
                            "ec2:AvailabilityZone",
                            "ec2:SnapshotTime",
                            "ec2:Encrypted",
                            "ec2:VolumeSize",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "EnableTransitGatewayRouteTablePropagation": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_EnableTransitGatewayRouteTablePropagation.html",
                "description": "Grants permission to enable an attachment to propagate routes to a propagation route table",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway-attachment": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-route-table": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "EnableVgwRoutePropagation": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_EnableVgwRoutePropagation.html",
                "description": "Grants permission to enable a virtual private gateway to propagate routes to a VPC route table",
                "accessLevel": "Write"
            },
            "EnableVolumeIO": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_EnableVolumeIO.html",
                "description": "Grants permission to enable I/O operations for a volume that had I/O operations disabled",
                "accessLevel": "Write"
            },
            "EnableVpcClassicLink": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_EnableVpcClassicLink.html",
                "description": "Grants permission to enable a VPC for ClassicLink",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vpc": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Tenancy"
                        ]
                    }
                }
            },
            "EnableVpcClassicLinkDnsSupport": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_EnableVpcClassicLinkDnsSupport.html",
                "description": "Grants permission to enable a VPC to support DNS hostname resolution for ClassicLink",
                "accessLevel": "Write"
            },
            "ExportClientVpnClientCertificateRevocationList": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ExportClientVpnClientCertificateRevocationList.html",
                "description": "Grants permission to download the client certificate revocation list for a Client VPN endpoint",
                "accessLevel": "List"
            },
            "ExportClientVpnClientConfiguration": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ExportClientVpnClientConfiguration.html",
                "description": "Grants permission to download the contents of the Client VPN endpoint configuration file for a Client VPN endpoint",
                "accessLevel": "List"
            },
            "ExportImage": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ExportImage.html",
                "description": "Grants permission to export an Amazon Machine Image (AMI) to a VM file",
                "accessLevel": "Write"
            },
            "ExportTransitGatewayRoutes": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ExportTransitGatewayRoutes.html",
                "description": "Grants permission to export routes from a transit gateway route table to an Amazon S3 bucket",
                "accessLevel": "Write"
            },
            "GetCapacityReservationUsage": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetCapacityReservationUsage.html",
                "description": "Grants permission to get usage information about a Capacity Reservation",
                "accessLevel": "Read"
            },
            "GetCoipPoolUsage": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetCoipPoolUsage.html",
                "description": "Grants permission to describe the allocations from the specified customer-owned address pool",
                "accessLevel": "Read"
            },
            "GetConsoleOutput": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetConsoleOutput.html",
                "description": "Grants permission to get the console output for an instance",
                "accessLevel": "Read"
            },
            "GetConsoleScreenshot": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetConsoleScreenshot.html",
                "description": "Grants permission to retrieve a JPG-format screenshot of a running instance",
                "accessLevel": "Read",
                "resourceTypes": {
                    "instance": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:AvailabilityZone",
                            "ec2:EbsOptimized",
                            "ec2:InstanceProfile",
                            "ec2:InstanceType",
                            "ec2:PlacementGroup",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:RootDeviceType",
                            "ec2:Tenancy"
                        ]
                    }
                }
            },
            "GetDefaultCreditSpecification": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetDefaultCreditSpecification.html",
                "description": "Grants permission to get the default credit option for CPU usage of a burstable performance instance family",
                "accessLevel": "Read"
            },
            "GetEbsDefaultKmsKeyId": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetEbsDefaultKmsKeyId.html",
                "description": "Grants permission to get the ID of the default customer master key (CMK) for EBS encryption by default",
                "accessLevel": "Read"
            },
            "GetEbsEncryptionByDefault": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetEbsEncryptionByDefault.html",
                "description": "Grants permission to describe whether EBS encryption by default is enabled for your account",
                "accessLevel": "Read"
            },
            "GetHostReservationPurchasePreview": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetHostReservationPurchasePreview.html",
                "description": "Grants permission to preview a reservation purchase with configurations that match those of a Dedicated Host",
                "accessLevel": "Read"
            },
            "GetLaunchTemplateData": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetLaunchTemplateData.html",
                "description": "Grants permission to get the configuration data of the specified instance for use with a new launch template or launch template version",
                "accessLevel": "Read"
            },
            "GetManagedPrefixListAssociations": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetManagedPrefixListAssociations.html",
                "description": "Grants permission to get information about the resources that are associated with the specified managed prefix list",
                "accessLevel": "Read"
            },
            "GetManagedPrefixListEntries": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetManagedPrefixListEntries.html",
                "description": "Grants permission to get information about the entries for a specified managed prefix list",
                "accessLevel": "Read"
            },
            "GetPasswordData": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetPasswordData.html",
                "description": "Grants permission to retrieve the encrypted administrator password for a running Windows instance",
                "accessLevel": "Read"
            },
            "GetReservedInstancesExchangeQuote": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetReservedInstancesExchangeQuote.html",
                "description": "Grants permission to return a quote and exchange information for exchanging one or more Convertible Reserved Instances for a new Convertible Reserved Instance",
                "accessLevel": "Read"
            },
            "GetTransitGatewayAttachmentPropagations": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetTransitGatewayAttachmentPropagations.html",
                "description": "Grants permission to list the route tables to which a resource attachment propagates routes",
                "accessLevel": "List"
            },
            "GetTransitGatewayMulticastDomainAssociations": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetTransitGatewayMulticastDomainAssociations.html",
                "description": "Grants permission to get information about the associations for a transit gateway multicast domain",
                "accessLevel": "List"
            },
            "GetTransitGatewayPrefixListReferences": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetTransitGatewayPrefixListReference.html",
                "description": "Grants permission to get information about prefix list references for a transit gateway route table",
                "accessLevel": "List"
            },
            "GetTransitGatewayRouteTableAssociations": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetTransitGatewayRouteTableAssociations.html",
                "description": "Grants permission to get information about associations for a transit gateway route table",
                "accessLevel": "List"
            },
            "GetTransitGatewayRouteTablePropagations": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetTransitGatewayRouteTablePropagations.html",
                "description": "Grants permission to get information about the route table propagations for a transit gateway route table",
                "accessLevel": "List"
            },
            "ImportClientVpnClientCertificateRevocationList": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ImportClientVpnClientCertificateRevocationList.html",
                "description": "Grants permission to upload a client certificate revocation list to a Client VPN endpoint",
                "accessLevel": "Write",
                "resourceTypes": {
                    "client-vpn-endpoint": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "ImportImage": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ImportImage.html",
                "description": "Grants permission to import single or multi-volume disk images or EBS snapshots into an Amazon Machine Image (AMI)",
                "accessLevel": "Write"
            },
            "ImportInstance": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ImportInstance.html",
                "description": "Grants permission to create an import instance task using metadata from a disk image",
                "accessLevel": "Write"
            },
            "ImportKeyPair": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ImportKeyPair.html",
                "description": "Grants permission to import a public key from an RSA key pair that was created with a third-party tool",
                "accessLevel": "Write"
            },
            "ImportSnapshot": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ImportSnapshot.html",
                "description": "Grants permission to import a disk into an EBS snapshot",
                "accessLevel": "Write"
            },
            "ImportVolume": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ImportVolume.html",
                "description": "Grants permission to create an import volume task using metadata from a disk image",
                "accessLevel": "Write"
            },
            "ModifyCapacityReservation": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyCapacityReservation.html",
                "description": "Grants permission to modify a Capacity Reservation's capacity and the conditions under which it is to be released",
                "accessLevel": "Write",
                "resourceTypes": {
                    "capacity-reservation": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "ModifyClientVpnEndpoint": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyClientVpnEndpoint.html",
                "description": "Grants permission to modify a Client VPN endpoint",
                "accessLevel": "Write",
                "resourceTypes": {
                    "client-vpn-endpoint": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "ModifyDefaultCreditSpecification": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyDefaultCreditSpecification.html",
                "description": "Grants permission to change the account level default credit option for CPU usage of burstable performance instances",
                "accessLevel": "Write"
            },
            "ModifyEbsDefaultKmsKeyId": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyEbsDefaultKmsKeyId.html",
                "description": "Grants permission to change the default customer master key (CMK) for EBS encryption by default for your account",
                "accessLevel": "Write"
            },
            "ModifyFleet": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyFleet.html",
                "description": "Grants permission to modify an EC2 Fleet",
                "accessLevel": "Write"
            },
            "ModifyFpgaImageAttribute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyFpgaImageAttribute.html",
                "description": "Grants permission to modify an attribute of an Amazon FPGA Image (AFI)",
                "accessLevel": "Write"
            },
            "ModifyHosts": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyHosts.html",
                "description": "Grants permission to modify a Dedicated Host",
                "accessLevel": "Write"
            },
            "ModifyIdFormat": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyIdFormat.html",
                "description": "Grants permission to modify the ID format for a resource",
                "accessLevel": "Write"
            },
            "ModifyIdentityIdFormat": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyIdentityIdFormat.html",
                "description": "Grants permission to modify the ID format of a resource for a specific principal in your account",
                "accessLevel": "Write"
            },
            "ModifyImageAttribute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyImageAttribute.html",
                "description": "Grants permission to modify an attribute of an Amazon Machine Image (AMI)",
                "accessLevel": "Write"
            },
            "ModifyInstanceAttribute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyInstanceAttribute.html",
                "description": "Grants permission to modify an attribute of an instance",
                "accessLevel": "Write"
            },
            "ModifyInstanceCapacityReservationAttributes": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyInstanceCapacityReservationAttributes.html",
                "description": "Grants permission to modify the Capacity Reservation settings for a stopped instance",
                "accessLevel": "Write"
            },
            "ModifyInstanceCreditSpecification": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyInstanceCreditSpecification.html",
                "description": "Grants permission to modify the credit option for CPU usage on an instance",
                "accessLevel": "Write",
                "resourceTypes": {
                    "instance": {
                        "required": true
                    }
                }
            },
            "ModifyInstanceEventStartTime": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyInstanceEventStartTime.html",
                "description": "Grants permission to modify the start time for a scheduled EC2 instance event",
                "accessLevel": "Write",
                "resourceTypes": {
                    "instance": {
                        "required": true,
                        "conditions": [
                            "ec2:Region"
                        ]
                    }
                }
            },
            "ModifyInstanceMetadataOptions": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyInstanceMetadataOptions.html",
                "description": "Grants permission to modify the metadata options for an instance",
                "accessLevel": "Write"
            },
            "ModifyInstancePlacement": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyInstancePlacement.html",
                "description": "Grants permission to modify the placement attributes for an instance",
                "accessLevel": "Write"
            },
            "ModifyLaunchTemplate": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyLaunchTemplate.html",
                "description": "Grants permission to modify a launch template",
                "accessLevel": "Write",
                "resourceTypes": {
                    "launch-template": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "ModifyManagedPrefixList": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyManagedPrefixList.html",
                "description": "Grants permission to modify a managed prefix list",
                "accessLevel": "Write",
                "resourceTypes": {
                    "prefix-list": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "ModifyNetworkInterfaceAttribute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyNetworkInterfaceAttribute.html",
                "description": "Grants permission to modify an attribute of a network interface",
                "accessLevel": "Write"
            },
            "ModifyReservedInstances": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyReservedInstances.html",
                "description": "Grants permission to modify attributes of one or more Reserved Instances",
                "accessLevel": "Write"
            },
            "ModifySnapshotAttribute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifySnapshotAttribute.html",
                "description": "Grants permission to add or remove permission settings for a snapshot",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "snapshot": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Owner",
                            "ec2:ParentVolume",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:SnapshotTime",
                            "ec2:VolumeSize"
                        ]
                    }
                }
            },
            "ModifySpotFleetRequest": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifySpotFleetRequest.html",
                "description": "Grants permission to modify a Spot Fleet request",
                "accessLevel": "Write"
            },
            "ModifySubnetAttribute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifySubnetAttribute.html",
                "description": "Grants permission to modify an attribute of a subnet",
                "accessLevel": "Write"
            },
            "ModifyTrafficMirrorFilterNetworkServices": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyTrafficMirrorFilterNetworkServices.html",
                "description": "Grants permission to allow or restrict mirroring network services",
                "accessLevel": "Write",
                "resourceTypes": {
                    "traffic-mirror-filter": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "ModifyTrafficMirrorFilterRule": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyTrafficMirrorFilterRule.html",
                "description": "Grants permission to modify a traffic mirror rule",
                "accessLevel": "Write",
                "resourceTypes": {
                    "traffic-mirror-filter": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "traffic-mirror-filter-rule": {
                        "required": true,
                        "conditions": [
                            "ec2:Region"
                        ]
                    }
                }
            },
            "ModifyTrafficMirrorSession": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyTrafficMirrorSession.html",
                "description": "Grants permission to modify a traffic mirror session",
                "accessLevel": "Write",
                "resourceTypes": {
                    "traffic-mirror-session": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "traffic-mirror-filter": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "traffic-mirror-target": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "ModifyTransitGateway": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyTransitGateway.html",
                "description": "Grants permission to modify a transit gateway",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "ModifyTransitGatewayPrefixListReference": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyTransitGatewayPrefixListReference.html",
                "description": "Grants permission to modify a transit gateway prefix list reference",
                "accessLevel": "Write",
                "resourceTypes": {
                    "prefix-list": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-route-table": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "ModifyTransitGatewayVpcAttachment": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyTransitGatewayVpcAttachment.html",
                "description": "Grants permission to modify a VPC attachment on a transit gateway",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway-attachment": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "subnet": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "ModifyVolume": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVolume.html",
                "description": "Grants permission to modify the parameters of an EBS volume",
                "accessLevel": "Write"
            },
            "ModifyVolumeAttribute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVolumeAttribute.html",
                "description": "Grants permission to modify an attribute of a volume",
                "accessLevel": "Write"
            },
            "ModifyVpcAttribute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVpcAttribute.html",
                "description": "Grants permission to modify an attribute of a VPC",
                "accessLevel": "Write"
            },
            "ModifyVpcEndpoint": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVpcEndpoint.html",
                "description": "Grants permission to modify an attribute of a VPC endpoint",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vpc-endpoint": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "route-table": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "security-group": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "subnet": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "ModifyVpcEndpointConnectionNotification": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVpcEndpointConnectionNotification.html",
                "description": "Grants permission to modify a connection notification for a VPC endpoint or VPC endpoint service",
                "accessLevel": "Write"
            },
            "ModifyVpcEndpointServiceConfiguration": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVpcEndpointServiceConfiguration.html",
                "description": "Grants permission to modify the attributes of a VPC endpoint service configuration",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vpc-endpoint-service": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:VpceServicePrivateDnsName",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "ModifyVpcEndpointServicePermissions": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVpcEndpointServicePermissions.html",
                "description": "Grants permission to modify the permissions for a VPC endpoint service",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "vpc-endpoint-service": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "ModifyVpcPeeringConnectionOptions": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVpcPeeringConnectionOptions.html",
                "description": "Grants permission to modify the VPC peering connection options on one side of a VPC peering connection",
                "accessLevel": "Write"
            },
            "ModifyVpcTenancy": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVpcTenancy.html",
                "description": "Grants permission to modify the instance tenancy attribute of a VPC",
                "accessLevel": "Write"
            },
            "ModifyVpnConnection": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVpnConnection.html",
                "description": "Grants permission to modify the target gateway of a Site-to-Site VPN connection",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vpn-connection": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:GatewayType"
                        ]
                    }
                }
            },
            "ModifyVpnTunnelCertificate": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVpnTunnelCertificate",
                "description": "Grants permission to modify the certificate for a Site-to-Site VPN connection",
                "accessLevel": "Write"
            },
            "ModifyVpnTunnelOptions": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVpnTunnelOptions.html",
                "description": "Grants permission to modify the options for a Site-to-Site VPN connection",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vpn-connection": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:AuthenticationType",
                            "ec2:DPDTimeoutSeconds",
                            "ec2:IKEVersions",
                            "ec2:InsideTunnelCidr",
                            "ec2:Phase1DHGroupNumbers",
                            "ec2:Phase2DHGroupNumbers",
                            "ec2:Phase1EncryptionAlgorithms",
                            "ec2:Phase2EncryptionAlgorithms",
                            "ec2:Phase1IntegrityAlgorithms",
                            "ec2:Phase2IntegrityAlgorithms",
                            "ec2:Phase1LifetimeSeconds",
                            "ec2:Phase2LifetimeSeconds",
                            "ec2:PresharedKeys",
                            "ec2:RekeyFuzzPercentage",
                            "ec2:RekeyMarginTimeSeconds",
                            "ec2:RoutingType"
                        ]
                    }
                }
            },
            "MonitorInstances": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_MonitorInstances.html",
                "description": "Grants permission to enable detailed monitoring for a running instance",
                "accessLevel": "Write"
            },
            "MoveAddressToVpc": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_MoveAddressToVpc.html",
                "description": "Grants permission to move an Elastic IP address from the EC2-Classic platform to the EC2-VPC platform",
                "accessLevel": "Write"
            },
            "ProvisionByoipCidr": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ProvisionByoipCidr.html",
                "description": "Grants permission to provision an address range for use in AWS through bring your own IP addresses (BYOIP), and to create a corresponding address pool",
                "accessLevel": "Write"
            },
            "PurchaseHostReservation": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_PurchaseHostReservation.html",
                "description": "Grants permission to purchase a reservation with configurations that match those of a Dedicated Host",
                "accessLevel": "Write"
            },
            "PurchaseReservedInstancesOffering": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_PurchaseReservedInstancesOffering.html",
                "description": "Grants permission to purchase a Reserved Instance offering",
                "accessLevel": "Write"
            },
            "PurchaseScheduledInstances": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_PurchaseScheduledInstances.html",
                "description": "Grants permission to purchase one or more Scheduled Instances with a specified schedule",
                "accessLevel": "Write"
            },
            "RebootInstances": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RebootInstances.html",
                "description": "Grants permission to request a reboot of one or more instances",
                "accessLevel": "Write",
                "resourceTypes": {
                    "instance": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:AvailabilityZone",
                            "ec2:EbsOptimized",
                            "ec2:InstanceProfile",
                            "ec2:InstanceType",
                            "ec2:PlacementGroup",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:RootDeviceType",
                            "ec2:Tenancy"
                        ]
                    }
                }
            },
            "RegisterImage": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RegisterImage.html",
                "description": "Grants permission to register an Amazon Machine Image (AMI)",
                "accessLevel": "Write"
            },
            "RegisterInstanceEventNotificationAttributes": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RegisterInstanceEventNotificationAttributes.html",
                "description": "Grants permission to add tags to the set of tags to include in notifications about scheduled events for your instances",
                "accessLevel": "Write"
            },
            "RegisterTransitGatewayMulticastGroupMembers": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RegisterTransitGatewayMulticastGroupMembers.html",
                "description": "Grants permission to register one or more network interfaces as a member of a group IP address in a transit gateway multicast domain",
                "accessLevel": "Write",
                "resourceTypes": {
                    "network-interface": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-multicast-domain": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "RegisterTransitGatewayMulticastGroupSources": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RegisterTransitGatewayMulticastGroupSources.html",
                "description": "Grants permission to register one or more network interfaces as a source of a group IP address in a transit gateway multicast domain",
                "accessLevel": "Write",
                "resourceTypes": {
                    "network-interface": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-multicast-domain": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "RejectTransitGatewayPeeringAttachment": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RejectTransitGatewayPeeringAttachment.html",
                "description": "Grants permission to reject a transit gateway peering attachment request",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway-attachment": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "RejectTransitGatewayVpcAttachment": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RejectTransitGatewayVpcAttachment.html",
                "description": "Grants permission to reject a request to attach a VPC to a transit gateway",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway-attachment": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "RejectVpcEndpointConnections": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RejectVpcEndpointConnections.html",
                "description": "Grants permission to reject one or more VPC endpoint connection requests to a VPC endpoint service",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vpc-endpoint-service": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "RejectVpcPeeringConnection": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RejectVpcPeeringConnection.html",
                "description": "Grants permission to reject a VPC peering connection request",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vpc-peering-connection": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:AccepterVpc",
                            "ec2:Region",
                            "ec2:RequesterVpc",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "ReleaseAddress": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ReleaseAddress.html",
                "description": "Grants permission to release an Elastic IP address",
                "accessLevel": "Write"
            },
            "ReleaseHosts": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ReleaseHosts.html",
                "description": "Grants permission to release one or more On-Demand Dedicated Hosts",
                "accessLevel": "Write"
            },
            "ReplaceIamInstanceProfileAssociation": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ReplaceIamInstanceProfileAssociation.html",
                "description": "Grants permission to replace an IAM instance profile for an instance",
                "accessLevel": "Write",
                "dependentActions": [
                    "iam:PassRole"
                ],
                "resourceTypes": {
                    "instance": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:AvailabilityZone",
                            "ec2:EbsOptimized",
                            "ec2:InstanceProfile",
                            "ec2:InstanceType",
                            "ec2:PlacementGroup",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:RootDeviceType",
                            "ec2:Tenancy"
                        ]
                    }
                }
            },
            "ReplaceNetworkAclAssociation": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ReplaceNetworkAclAssociation.html",
                "description": "Grants permission to change which network ACL a subnet is associated with",
                "accessLevel": "Write"
            },
            "ReplaceNetworkAclEntry": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ReplaceNetworkAclEntry.html",
                "description": "Grants permission to replace an entry (rule) in a network ACL",
                "accessLevel": "Write"
            },
            "ReplaceRoute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ReplaceRoute.html",
                "description": "Grants permission to replace a route within a route table in a VPC",
                "accessLevel": "Write",
                "resourceTypes": {
                    "route-table": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    }
                }
            },
            "ReplaceRouteTableAssociation": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ReplaceRouteTableAssociation.html",
                "description": "Grants permission to change the route table that is associated with a subnet",
                "accessLevel": "Write"
            },
            "ReplaceTransitGatewayRoute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ReplaceTransitGatewayRoute.html",
                "description": "Grants permission to replace a route in a transit gateway route table",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway-route-table": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-attachment": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "ReportInstanceStatus": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ReportInstanceStatus.html",
                "description": "Grants permission to submit feedback about the status of an instance",
                "accessLevel": "Write"
            },
            "RequestSpotFleet": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RequestSpotFleet.html",
                "description": "Grants permission to create a Spot Fleet request",
                "accessLevel": "Write"
            },
            "RequestSpotInstances": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RequestSpotInstances.html",
                "description": "Grants permission to create a Spot Instance request",
                "accessLevel": "Write"
            },
            "ResetEbsDefaultKmsKeyId": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ResetEbsDefaultKmsKeyId.html",
                "description": "Grants permission to reset the default customer master key (CMK) for EBS encryption for your account to use the AWS-managed CMK for EBS",
                "accessLevel": "Write"
            },
            "ResetFpgaImageAttribute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ResetFpgaImageAttribute.html",
                "description": "Grants permission to reset an attribute of an Amazon FPGA Image (AFI) to its default value",
                "accessLevel": "Write"
            },
            "ResetImageAttribute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ResetImageAttribute.html",
                "description": "Grants permission to reset an attribute of an Amazon Machine Image (AMI) to its default value",
                "accessLevel": "Write"
            },
            "ResetInstanceAttribute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ResetInstanceAttribute.html",
                "description": "Grants permission to reset an attribute of an instance to its default value",
                "accessLevel": "Write"
            },
            "ResetNetworkInterfaceAttribute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ResetNetworkInterfaceAttribute.html",
                "description": "Grants permission to reset an attribute of a network interface",
                "accessLevel": "Write"
            },
            "ResetSnapshotAttribute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ResetSnapshotAttribute.html",
                "description": "Grants permission to reset permission settings for a snapshot",
                "accessLevel": "Permissions management"
            },
            "RestoreAddressToClassic": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RestoreAddressToClassic.html",
                "description": "Grants permission to restore an Elastic IP address that was previously moved to the EC2-VPC platform back to the EC2-Classic platform",
                "accessLevel": "Write"
            },
            "RestoreManagedPrefixListVersion": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RestoreManagedPrefixListVersion.html",
                "description": "Grants permission to restore the entries from a previous version of a managed prefix list to a new version of the prefix list",
                "accessLevel": "Write",
                "resourceTypes": {
                    "prefix-list": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "RevokeClientVpnIngress": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RevokeClientVpnIngress.html",
                "description": "Grants permission to remove an inbound authorization rule from a Client VPN endpoint",
                "accessLevel": "Write",
                "resourceTypes": {
                    "client-vpn-endpoint": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "RevokeSecurityGroupEgress": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RevokeSecurityGroupEgress.html",
                "description": "Grants permission to remove one or more outbound rules from a VPC security group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "security-group": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    }
                }
            },
            "RevokeSecurityGroupIngress": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RevokeSecurityGroupIngress.html",
                "description": "Grants permission to remove one or more inbound rules from a security group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "security-group": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    }
                }
            },
            "RunInstances": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RunInstances.html",
                "description": "Grants permission to launch one or more instances",
                "accessLevel": "Write",
                "resourceTypes": {
                    "image": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:ImageType",
                            "ec2:IsLaunchTemplateResource",
                            "ec2:LaunchTemplate",
                            "ec2:Owner",
                            "ec2:Public",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:RootDeviceType"
                        ]
                    },
                    "instance": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:AvailabilityZone",
                            "ec2:EbsOptimized",
                            "ec2:InstanceProfile",
                            "ec2:InstanceType",
                            "ec2:IsLaunchTemplateResource",
                            "ec2:LaunchTemplate",
                            "ec2:PlacementGroup",
                            "ec2:Region",
                            "ec2:RootDeviceType",
                            "ec2:Tenancy",
                            "ec2:MetadataHttpEndpoint",
                            "ec2:MetadataHttpTokens",
                            "ec2:MetadataHttpPutResponseHopLimit"
                        ]
                    },
                    "network-interface": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/",
                            "ec2:AvailabilityZone",
                            "ec2:IsLaunchTemplateResource",
                            "ec2:LaunchTemplate",
                            "ec2:Region",
                            "ec2:ResourceTag/",
                            "ec2:Subnet",
                            "ec2:Vpc",
                            "ec2:AssociatePublicIpAddress"
                        ]
                    },
                    "security-group": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:IsLaunchTemplateResource",
                            "ec2:LaunchTemplate",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    },
                    "subnet": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:AvailabilityZone",
                            "ec2:IsLaunchTemplateResource",
                            "ec2:LaunchTemplate",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    },
                    "volume": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:AvailabilityZone",
                            "ec2:Encrypted",
                            "ec2:IsLaunchTemplateResource",
                            "ec2:LaunchTemplate",
                            "ec2:ParentSnapshot",
                            "ec2:Region",
                            "ec2:VolumeIops",
                            "ec2:VolumeSize",
                            "ec2:VolumeType"
                        ]
                    },
                    "capacity-reservation": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "elastic-gpu": {
                        "required": false,
                        "conditions": [
                            "ec2:ElasticGpuType"
                        ]
                    },
                    "elastic-inference": {
                        "required": false
                    },
                    "key-pair": {
                        "required": false,
                        "conditions": [
                            "ec2:IsLaunchTemplateResource",
                            "ec2:LaunchTemplate",
                            "ec2:Region"
                        ]
                    },
                    "launch-template": {
                        "required": false,
                        "conditions": [
                            "ec2:IsLaunchTemplateResource",
                            "ec2:LaunchTemplate",
                            "ec2:Region"
                        ]
                    },
                    "placement-group": {
                        "required": false,
                        "conditions": [
                            "ec2:IsLaunchTemplateResource",
                            "ec2:LaunchTemplate",
                            "ec2:PlacementGroupStrategy",
                            "ec2:Region"
                        ]
                    },
                    "snapshot": {
                        "required": false,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:IsLaunchTemplateResource",
                            "ec2:LaunchTemplate",
                            "ec2:Owner",
                            "ec2:ParentVolume",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:SnapshotTime",
                            "ec2:VolumeSize"
                        ]
                    }
                }
            },
            "RunScheduledInstances": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RunScheduledInstances.html",
                "description": "Grants permission to launch one or more Scheduled Instances",
                "accessLevel": "Write"
            },
            "SearchLocalGatewayRoutes": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_SearchLocalGatewayRoutes.html",
                "description": "Grants permission to search for routes in a local gateway route table",
                "accessLevel": "List"
            },
            "SearchTransitGatewayMulticastGroups": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_SearchTransitGatewayMulticastGroups.html",
                "description": "Grants permission to search for groups, sources, and members in a transit gateway multicast domain",
                "accessLevel": "List"
            },
            "SearchTransitGatewayRoutes": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_SearchTransitGatewayRoutes.html",
                "description": "Grants permission to search for routes in a transit gateway route table",
                "accessLevel": "List"
            },
            "SendDiagnosticInterrupt": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_SendDiagnosticInterrupt.html",
                "description": "Grants permission to send a diagnostic interrupt to an Amazon EC2 instance",
                "accessLevel": "Write",
                "resourceTypes": {
                    "instance": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:AvailabilityZone",
                            "ec2:EbsOptimized",
                            "ec2:InstanceProfile",
                            "ec2:InstanceType",
                            "ec2:PlacementGroup",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:RootDeviceType",
                            "ec2:Tenancy"
                        ]
                    }
                }
            },
            "StartInstances": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_StartInstances.html",
                "description": "Grants permission to start a stopped instance",
                "accessLevel": "Write",
                "resourceTypes": {
                    "instance": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:AvailabilityZone",
                            "ec2:EbsOptimized",
                            "ec2:InstanceProfile",
                            "ec2:InstanceType",
                            "ec2:PlacementGroup",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:RootDeviceType",
                            "ec2:Tenancy"
                        ]
                    }
                }
            },
            "StartVpcEndpointServicePrivateDnsVerification": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_StartVpcEndpointServicePrivateDnsVerification.html",
                "description": "Grants permission to start the private DNS verification process for a VPC endpoint service",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vpc-endpoint-service": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "StopInstances": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_StopInstances.html",
                "description": "Grants permission to stop an Amazon EBS-backed instance",
                "accessLevel": "Write",
                "resourceTypes": {
                    "instance": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:AvailabilityZone",
                            "ec2:EbsOptimized",
                            "ec2:InstanceProfile",
                            "ec2:InstanceType",
                            "ec2:PlacementGroup",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:RootDeviceType",
                            "ec2:Tenancy"
                        ]
                    }
                }
            },
            "TerminateClientVpnConnections": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_TerminateClientVpnConnections.html",
                "description": "Grants permission to terminate active Client VPN endpoint connections",
                "accessLevel": "Write",
                "resourceTypes": {
                    "client-vpn-endpoint": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "TerminateInstances": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_TerminateInstances.html",
                "description": "Grants permission to shut down one or more instances",
                "accessLevel": "Write",
                "resourceTypes": {
                    "instance": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:AvailabilityZone",
                            "ec2:EbsOptimized",
                            "ec2:InstanceProfile",
                            "ec2:InstanceType",
                            "ec2:PlacementGroup",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:RootDeviceType",
                            "ec2:Tenancy"
                        ]
                    }
                }
            },
            "UnassignIpv6Addresses": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_UnassignIpv6Addresses.html",
                "description": "Grants permission to unassign one or more IPv6 addresses from a network interface",
                "accessLevel": "Write"
            },
            "UnassignPrivateIpAddresses": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_UnassignPrivateIpAddresses.html",
                "description": "Grants permission to unassign one or more secondary private IP addresses from a network interface",
                "accessLevel": "Write"
            },
            "UnmonitorInstances": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_UnmonitorInstances.html",
                "description": "Grants permission to disable detailed monitoring for a running instance",
                "accessLevel": "Write"
            },
            "UpdateSecurityGroupRuleDescriptionsEgress": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_UpdateSecurityGroupRuleDescriptionsEgress.html",
                "description": "Grants permission to update descriptions for one or more outbound rules in a VPC security group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "security-group": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    }
                }
            },
            "UpdateSecurityGroupRuleDescriptionsIngress": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_UpdateSecurityGroupRuleDescriptionsIngress.html",
                "description": "Grants permission to update descriptions for one or more inbound rules in a security group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "security-group": {
                        "required": true,
                        "conditions": [
                            "aws:ResourceTag/${TagKey}",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    }
                }
            },
            "WithdrawByoipCidr": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_WithdrawByoipCidr.html",
                "description": "Grants permission to stop advertising an address range that was provisioned for use in AWS through bring your own IP addresses (BYOIP)",
                "accessLevel": "Write"
            }
        };
        this.resourceTypes = {
            "capacity-reservation": {
                "name": "capacity-reservation",
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-capacity-reservations.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:capacity-reservation/${CapacityReservationId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "carrier-gateway": {
                "name": "carrier-gateway",
                "url": "https://docs.aws.amazon.com/vpc/latest/userguide/Carrier_Gateway.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:carrier-gateway/${CarrierGatewayId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "client-vpn-endpoint": {
                "name": "client-vpn-endpoint",
                "url": "https://docs.aws.amazon.com/vpn/latest/clientvpn-admin/what-is.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:client-vpn-endpoint/${ClientVpnEndpointId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "customer-gateway": {
                "name": "customer-gateway",
                "url": "https://docs.aws.amazon.com/vpn/latest/s2svpn/VPC_VPN.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:customer-gateway/${CustomerGatewayId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "dedicated-host": {
                "name": "dedicated-host",
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/dedicated-hosts-overview.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:dedicated-host/${HostId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:AutoPlacement",
                    "ec2:AvailabilityZone",
                    "ec2:HostRecovery",
                    "ec2:InstanceType",
                    "ec2:Quantity",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "dhcp-options": {
                "name": "dhcp-options",
                "url": "https://docs.aws.amazon.com/vpc/latest/userguide/VPC_DHCP_Options.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:dhcp-options/${DhcpOptionsId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "egress-only-internet-gateway": {
                "name": "egress-only-internet-gateway",
                "url": "https://docs.aws.amazon.com/vpc/latest/userguide/egress-only-internet-gateway.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:egress-only-internet-gateway/${EgressOnlyInternetGatewayId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "elastic-gpu": {
                "name": "elastic-gpu",
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/elastic-gpus.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:elastic-gpu/${ElasticGpuId}",
                "conditionKeys": [
                    "ec2:ElasticGpuType"
                ]
            },
            "elastic-inference": {
                "name": "elastic-inference",
                "url": "https://docs.aws.amazon.com/elastic-inference/latest/developerguide/what-is-ei.html",
                "arn": "arn:${Partition}:elastic-inference:${Region}:${Account}:elastic-inference-accelerator/${ElasticInferenceAcceleratorId}",
                "conditionKeys": []
            },
            "export-instance-task": {
                "name": "export-instance-task",
                "url": "https://docs.aws.amazon.com/vm-import/latest/userguide/vmexport.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:export-instance-task/${ExportTaskId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "fleet": {
                "name": "fleet",
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-fleet.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:fleet/${FleetId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "fpga-image": {
                "name": "fpga-image",
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#EC2_ARN_Format",
                "arn": "arn:${Partition}:ec2:${Region}::fpga-image/${FpgaImageId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Owner",
                    "ec2:Public",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "image": {
                "name": "image",
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/AMIs.html",
                "arn": "arn:${Partition}:ec2:${Region}::image/${ImageId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:ImageType",
                    "ec2:Owner",
                    "ec2:Public",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}",
                    "ec2:RootDeviceType"
                ]
            },
            "instance": {
                "name": "instance",
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/Instances.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:instance/${InstanceId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:AvailabilityZone",
                    "ec2:EbsOptimized",
                    "ec2:InstanceProfile",
                    "ec2:InstanceType",
                    "ec2:PlacementGroup",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}",
                    "ec2:RootDeviceType",
                    "ec2:Tenancy"
                ]
            },
            "internet-gateway": {
                "name": "internet-gateway",
                "url": "https://docs.aws.amazon.com/vpc/latest/userguide/VPC_Internet_Gateway.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:internet-gateway/${InternetGatewayId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "key-pair": {
                "name": "key-pair",
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-key-pairs.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:key-pair/${KeyPairName}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "launch-template": {
                "name": "launch-template",
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-launch-templates.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:launch-template/${LaunchTemplateId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "local-gateway": {
                "name": "local-gateway",
                "url": "https://docs.aws.amazon.com/outposts/latest/userguide/outposts-local-gateways.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:local-gateway/${LocalGatewayId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "local-gateway-route-table": {
                "name": "local-gateway-route-table",
                "url": "https://docs.aws.amazon.com/outposts/latest/userguide/outposts-local-gateways.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:local-gateway-route-table/${LocalGatewayRouteTableId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "local-gateway-route-table-virtual-interface-group-association": {
                "name": "local-gateway-route-table-virtual-interface-group-association",
                "url": "https://docs.aws.amazon.com/outposts/latest/userguide/outposts-local-gateways.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:local-gateway-route-table-virtual-interface-group-association/${LocalGatewayRouteTableVirtualInterfaceGroupAssociationId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "local-gateway-route-table-vpc-association": {
                "name": "local-gateway-route-table-vpc-association",
                "url": "https://docs.aws.amazon.com/outposts/latest/userguide/outposts-local-gateways.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:local-gateway-route-table-vpc-association/${LocalGatewayRouteTableVpcAssociationId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "local-gateway-virtual-interface": {
                "name": "local-gateway-virtual-interface",
                "url": "https://docs.aws.amazon.com/outposts/latest/userguide/outposts-local-gateways.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:local-gateway-virtual-interface/${LocalGatewayVirtualInterfaceId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "local-gateway-virtual-interface-group": {
                "name": "local-gateway-virtual-interface-group",
                "url": "https://docs.aws.amazon.com/outposts/latest/userguide/outposts-local-gateways.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:local-gateway-virtual-interface-group/${LocalGatewayVirtualInterfaceGroupId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "natgateway": {
                "name": "natgateway",
                "url": "https://docs.aws.amazon.com/vpc/latest/userguide/vpc-nat-gateway.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:natgateway/${NatGatewayId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "network-acl": {
                "name": "network-acl",
                "url": "https://docs.aws.amazon.com/vpc/latest/userguide/vpc-network-acls.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:network-acl/${NaclId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}",
                    "ec2:Vpc"
                ]
            },
            "network-interface": {
                "name": "network-interface",
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using-eni.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:network-interface/${NetworkInterfaceId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:AssociatePublicIpAddress",
                    "ec2:AuthorizedService",
                    "ec2:AvailabilityZone",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}",
                    "ec2:Subnet",
                    "ec2:Vpc"
                ]
            },
            "placement-group": {
                "name": "placement-group",
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/placement-groups.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:placement-group/${PlacementGroupName}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:PlacementGroupStrategy",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "prefix-list": {
                "name": "prefix-list",
                "url": "",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:prefix-list/${PrefixListId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "reserved-instances": {
                "name": "reserved-instances",
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-reserved-instances.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:reserved-instances/${ReservationId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:AvailabilityZone",
                    "ec2:InstanceType",
                    "ec2:Region",
                    "ec2:ReservedInstancesOfferingType",
                    "ec2:ResourceTag/${TagKey}",
                    "ec2:Tenancy"
                ]
            },
            "route-table": {
                "name": "route-table",
                "url": "https://docs.aws.amazon.com/vpc/latest/userguide/VPC_Route_Tables.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:route-table/${RouteTableId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}",
                    "ec2:Vpc"
                ]
            },
            "security-group": {
                "name": "security-group",
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-security-groups.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:security-group/${SecurityGroupId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}",
                    "ec2:Vpc"
                ]
            },
            "snapshot": {
                "name": "snapshot",
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSSnapshots.html",
                "arn": "arn:${Partition}:ec2:${Region}::snapshot/${SnapshotId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Owner",
                    "ec2:ParentVolume",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}",
                    "ec2:SnapshotTime",
                    "ec2:VolumeSize"
                ]
            },
            "spot-instance-request": {
                "name": "spot-instance-request",
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using-spot-instances.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:spot-instances-request/${SpotInstanceRequestId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "subnet": {
                "name": "subnet",
                "url": "https://docs.aws.amazon.com/vpc/latest/userguide/VPC_Subnets.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:subnet/${SubnetId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:AvailabilityZone",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}",
                    "ec2:Vpc"
                ]
            },
            "traffic-mirror-session": {
                "name": "traffic-mirror-session",
                "url": "https://docs.aws.amazon.com/vpc/latest/mirroring/traffic-mirroring-session.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:traffic-mirror-session/${TrafficMirrorSessionId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "traffic-mirror-target": {
                "name": "traffic-mirror-target",
                "url": "https://docs.aws.amazon.com/vpc/latest/mirroring/traffic-mirroring-target.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:traffic-mirror-target/${TrafficMirrorTargetId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "traffic-mirror-filter": {
                "name": "traffic-mirror-filter",
                "url": "https://docs.aws.amazon.com/vpc/latest/mirroring/traffic-mirroring-filter.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:traffic-mirror-filter/${TrafficMirrorFilterId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "traffic-mirror-filter-rule": {
                "name": "traffic-mirror-filter-rule",
                "url": "https://docs.aws.amazon.com/vpc/latest/mirroring/traffic-mirroring-filter.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:traffic-mirror-filter-rule/${TrafficMirrorFilterRuleId}",
                "conditionKeys": [
                    "ec2:Region"
                ]
            },
            "transit-gateway-attachment": {
                "name": "transit-gateway-attachment",
                "url": "https://docs.aws.amazon.com/vpc/latest/tgw/how-transit-gateways-work.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:transit-gateway-attachment/${TransitGatewayAttachmentId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "transit-gateway-multicast-domain": {
                "name": "transit-gateway-multicast-domain",
                "url": "https://docs.aws.amazon.com/vpc/latest/tgw/tgw-multicast-overview.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:transit-gateway-multicast-domain/${TransitGatewayMulticastDomainId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "transit-gateway-route-table": {
                "name": "transit-gateway-route-table",
                "url": "https://docs.aws.amazon.com/vpc/latest/tgw/how-transit-gateways-work.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:transit-gateway-route-table/${TransitGatewayRouteTableId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "transit-gateway": {
                "name": "transit-gateway",
                "url": "https://docs.aws.amazon.com/vpc/latest/tgw/how-transit-gateways-work.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:transit-gateway/${TransitGatewayId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "volume": {
                "name": "volume",
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ebs-volumes.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:volume/${VolumeId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:AvailabilityZone",
                    "ec2:Encrypted",
                    "ec2:ParentSnapshot",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}",
                    "ec2:VolumeIops",
                    "ec2:VolumeSize",
                    "ec2:VolumeType"
                ]
            },
            "vpc": {
                "name": "vpc",
                "url": "https://docs.aws.amazon.com/vpc/latest/userguide/VPC_Subnets.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:vpc/${VpcId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}",
                    "ec2:Tenancy"
                ]
            },
            "vpc-endpoint": {
                "name": "vpc-endpoint",
                "url": "https://docs.aws.amazon.com/vpc/latest/userguide/endpoint-services-overview.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:vpc-endpoint/${VpceId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}",
                    "ec2:VpceServiceName",
                    "ec2:VpceServiceOwner"
                ]
            },
            "vpc-endpoint-service": {
                "name": "vpc-endpoint-service",
                "url": "https://docs.aws.amazon.com/vpc/latest/userguide/endpoint-services-overview.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:vpc-endpoint-service/${VpceServiceId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}",
                    "ec2:VpceServicePrivateDnsName"
                ]
            },
            "vpc-flow-log": {
                "name": "vpc-flow-log",
                "url": "https://docs.aws.amazon.com/vpc/latest/userguide/flow-logs.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:vpc-flow-log/${VpcFlowLogId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "vpc-peering-connection": {
                "name": "vpc-peering-connection",
                "url": "https://docs.aws.amazon.com/vpc/latest/peering/what-is-vpc-peering.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:vpc-peering-connection/${VpcPeeringConnectionId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "ec2:AccepterVpc",
                    "ec2:Region",
                    "ec2:RequesterVpc",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "vpn-connection": {
                "name": "vpn-connection",
                "url": "https://docs.aws.amazon.com/vpn/latest/s2svpn/VPC_VPN.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:vpn-connection/${VpnConnectionId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:AuthenticationType",
                    "ec2:DPDTimeoutSeconds",
                    "ec2:GatewayType",
                    "ec2:IKEVersions",
                    "ec2:InsideTunnelCidr",
                    "ec2:Phase1DHGroupNumbers",
                    "ec2:Phase1EncryptionAlgorithms",
                    "ec2:Phase1IntegrityAlgorithms",
                    "ec2:Phase1LifetimeSeconds",
                    "ec2:Phase2DHGroupNumbers",
                    "ec2:Phase2EncryptionAlgorithms",
                    "ec2:Phase2IntegrityAlgorithms",
                    "ec2:Phase2LifetimeSeconds",
                    "ec2:PresharedKeys",
                    "ec2:Region",
                    "ec2:RekeyFuzzPercentage",
                    "ec2:RekeyMarginTimeSeconds",
                    "ec2:ResourceTag/${TagKey}",
                    "ec2:RoutingType"
                ]
            },
            "vpn-gateway": {
                "name": "vpn-gateway",
                "url": "https://docs.aws.amazon.com/vpn/latest/s2svpn/VPC_VPN.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:vpn-gateway/${VpnGatewayId}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Grants permission to accept a Convertible Reserved Instance exchange quote
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AcceptReservedInstancesExchangeQuote.html
     */
    toAcceptReservedInstancesExchangeQuote() {
        this.add('ec2:AcceptReservedInstancesExchangeQuote');
        return this;
    }
    /**
     * Grants permission to accept a transit gateway peering attachment request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AcceptTransitGatewayPeeringAttachment.html
     */
    toAcceptTransitGatewayPeeringAttachment() {
        this.add('ec2:AcceptTransitGatewayPeeringAttachment');
        return this;
    }
    /**
     * Grants permission to accept a request to attach a VPC to a transit gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AcceptTransitGatewayVpcAttachment.html
     */
    toAcceptTransitGatewayVpcAttachment() {
        this.add('ec2:AcceptTransitGatewayVpcAttachment');
        return this;
    }
    /**
     * Grants permission to accept one or more interface VPC endpoint connections to your VPC endpoint service
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AcceptVpcEndpointConnections.html
     */
    toAcceptVpcEndpointConnections() {
        this.add('ec2:AcceptVpcEndpointConnections');
        return this;
    }
    /**
     * Grants permission to accept a VPC peering connection request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AcceptVpcPeeringConnection.html
     */
    toAcceptVpcPeeringConnection() {
        this.add('ec2:AcceptVpcPeeringConnection');
        return this;
    }
    /**
     * Grants permission to advertise an IP address range that is provisioned for use in AWS through bring your own IP addresses (BYOIP)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AdvertiseByoipCidr.html
     */
    toAdvertiseByoipCidr() {
        this.add('ec2:AdvertiseByoipCidr');
        return this;
    }
    /**
     * Grants permission to allocate an Elastic IP address (EIP) to your account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AllocateAddress.html
     */
    toAllocateAddress() {
        this.add('ec2:AllocateAddress');
        return this;
    }
    /**
     * Grants permission to allocate a Dedicated Host to your account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AllocateHosts.html
     */
    toAllocateHosts() {
        this.add('ec2:AllocateHosts');
        return this;
    }
    /**
     * Grants permission to apply a security group to the association between a Client VPN endpoint and a target network
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ApplySecurityGroupsToClientVpnTargetNetwork.html
     */
    toApplySecurityGroupsToClientVpnTargetNetwork() {
        this.add('ec2:ApplySecurityGroupsToClientVpnTargetNetwork');
        return this;
    }
    /**
     * Grants permission to assign one or more IPv6 addresses to a network interface
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssignIpv6Addresses.html
     */
    toAssignIpv6Addresses() {
        this.add('ec2:AssignIpv6Addresses');
        return this;
    }
    /**
     * Grants permission to assign one or more secondary private IP addresses to a network interface
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssignPrivateIpAddresses.html
     */
    toAssignPrivateIpAddresses() {
        this.add('ec2:AssignPrivateIpAddresses');
        return this;
    }
    /**
     * Grants permission to associate an Elastic IP address (EIP) with an instance or a network interface
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssociateAddress.html
     */
    toAssociateAddress() {
        this.add('ec2:AssociateAddress');
        return this;
    }
    /**
     * Grants permission to associate a target network with a Client VPN endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssociateClientVpnTargetNetwork.html
     */
    toAssociateClientVpnTargetNetwork() {
        this.add('ec2:AssociateClientVpnTargetNetwork');
        return this;
    }
    /**
     * Grants permission to associate or disassociate a set of DHCP options with a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssociateDhcpOptions.html
     */
    toAssociateDhcpOptions() {
        this.add('ec2:AssociateDhcpOptions');
        return this;
    }
    /**
     * Grants permission to associate an IAM instance profile with a running or stopped instance
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssociateIamInstanceProfile.html
     */
    toAssociateIamInstanceProfile() {
        this.add('ec2:AssociateIamInstanceProfile');
        return this;
    }
    /**
     * Grants permission to associate a subnet or gateway with a route table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssociateRouteTable.html
     */
    toAssociateRouteTable() {
        this.add('ec2:AssociateRouteTable');
        return this;
    }
    /**
     * Grants permission to associate a CIDR block with a subnet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssociateSubnetCidrBlock.html
     */
    toAssociateSubnetCidrBlock() {
        this.add('ec2:AssociateSubnetCidrBlock');
        return this;
    }
    /**
     * Grants permission to associate an attachment and list of subnets with a transit gateway multicast domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssociateTransitGatewayMulticastDomain.html
     */
    toAssociateTransitGatewayMulticastDomain() {
        this.add('ec2:AssociateTransitGatewayMulticastDomain');
        return this;
    }
    /**
     * Grants permission to associate an attachment with a transit gateway route table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssociateTransitGatewayRouteTable.html
     */
    toAssociateTransitGatewayRouteTable() {
        this.add('ec2:AssociateTransitGatewayRouteTable');
        return this;
    }
    /**
     * Grants permission to associate a CIDR block with a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssociateVpcCidrBlock.html
     */
    toAssociateVpcCidrBlock() {
        this.add('ec2:AssociateVpcCidrBlock');
        return this;
    }
    /**
     * Grants permission to link an EC2-Classic instance to a ClassicLink-enabled VPC through one or more of the VPC's security groups
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AttachClassicLinkVpc.html
     */
    toAttachClassicLinkVpc() {
        this.add('ec2:AttachClassicLinkVpc');
        return this;
    }
    /**
     * Grants permission to attach an internet gateway to a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AttachInternetGateway.html
     */
    toAttachInternetGateway() {
        this.add('ec2:AttachInternetGateway');
        return this;
    }
    /**
     * Grants permission to attach a network interface to an instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AttachNetworkInterface.html
     */
    toAttachNetworkInterface() {
        this.add('ec2:AttachNetworkInterface');
        return this;
    }
    /**
     * Grants permission to attach an EBS volume to a running or stopped instance and expose it to the instance with the specified device name
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AttachVolume.html
     */
    toAttachVolume() {
        this.add('ec2:AttachVolume');
        return this;
    }
    /**
     * Grants permission to attach a virtual private gateway to a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AttachVpnGateway.html
     */
    toAttachVpnGateway() {
        this.add('ec2:AttachVpnGateway');
        return this;
    }
    /**
     * Grants permission to add an inbound authorization rule to a Client VPN endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AuthorizeClientVpnIngress.html
     */
    toAuthorizeClientVpnIngress() {
        this.add('ec2:AuthorizeClientVpnIngress');
        return this;
    }
    /**
     * Grants permission to add one or more outbound rules to a VPC security group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AuthorizeSecurityGroupEgress.html
     */
    toAuthorizeSecurityGroupEgress() {
        this.add('ec2:AuthorizeSecurityGroupEgress');
        return this;
    }
    /**
     * Grants permission to add one or more inbound rules to a security group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AuthorizeSecurityGroupIngress.html
     */
    toAuthorizeSecurityGroupIngress() {
        this.add('ec2:AuthorizeSecurityGroupIngress');
        return this;
    }
    /**
     * Grants permission to bundle an instance store-backed Windows instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_BundleInstance.html
     */
    toBundleInstance() {
        this.add('ec2:BundleInstance');
        return this;
    }
    /**
     * Grants permission to cancel a bundling operation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CancelBundleTask.html
     */
    toCancelBundleTask() {
        this.add('ec2:CancelBundleTask');
        return this;
    }
    /**
     * Grants permission to cancel a Capacity Reservation and release the reserved capacity
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CancelCapacityReservation.html
     */
    toCancelCapacityReservation() {
        this.add('ec2:CancelCapacityReservation');
        return this;
    }
    /**
     * Grants permission to cancel an active conversion task
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CancelConversionTask.html
     */
    toCancelConversionTask() {
        this.add('ec2:CancelConversionTask');
        return this;
    }
    /**
     * Grants permission to cancel an active export task
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CancelExportTask.html
     */
    toCancelExportTask() {
        this.add('ec2:CancelExportTask');
        return this;
    }
    /**
     * Grants permission to cancel an in-process import virtual machine or import snapshot task
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CancelImportTask.html
     */
    toCancelImportTask() {
        this.add('ec2:CancelImportTask');
        return this;
    }
    /**
     * Grants permission to cancel a Reserved Instance listing on the Reserved Instance Marketplace
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CancelReservedInstancesListing.html
     */
    toCancelReservedInstancesListing() {
        this.add('ec2:CancelReservedInstancesListing');
        return this;
    }
    /**
     * Grants permission to cancel one or more Spot Fleet requests
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CancelSpotFleetRequests.html
     */
    toCancelSpotFleetRequests() {
        this.add('ec2:CancelSpotFleetRequests');
        return this;
    }
    /**
     * Grants permission to cancel one or more Spot Instance requests
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CancelSpotInstanceRequests.html
     */
    toCancelSpotInstanceRequests() {
        this.add('ec2:CancelSpotInstanceRequests');
        return this;
    }
    /**
     * Grants permission to determine whether an owned product code is associated with an instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ConfirmProductInstance.html
     */
    toConfirmProductInstance() {
        this.add('ec2:ConfirmProductInstance');
        return this;
    }
    /**
     * Grants permission to copy a source Amazon FPGA image (AFI) to the current Region
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CopyFpgaImage.html
     */
    toCopyFpgaImage() {
        this.add('ec2:CopyFpgaImage');
        return this;
    }
    /**
     * Grants permission to copy an Amazon Machine Image (AMI) from a source Region to the current Region
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CopyImage.html
     */
    toCopyImage() {
        this.add('ec2:CopyImage');
        return this;
    }
    /**
     * Grants permission to copy a point-in-time snapshot of an EBS volume and store it in Amazon S3
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CopySnapshot.html
     */
    toCopySnapshot() {
        this.add('ec2:CopySnapshot');
        return this;
    }
    /**
     * Grants permission to create a Capacity Reservation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateCapacityReservation.html
     */
    toCreateCapacityReservation() {
        this.add('ec2:CreateCapacityReservation');
        return this;
    }
    /**
     * Grants permission to create a carrier gateway and provides CSP connectivity to VPC customers.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateCarrierGateway.html
     */
    toCreateCarrierGateway() {
        this.add('ec2:CreateCarrierGateway');
        return this;
    }
    /**
     * Grants permission to create a Client VPN endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateClientVpnEndpoint.html
     */
    toCreateClientVpnEndpoint() {
        this.add('ec2:CreateClientVpnEndpoint');
        return this;
    }
    /**
     * Grants permission to add a network route to a Client VPN endpoint's route table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateClientVpnRoute.html
     */
    toCreateClientVpnRoute() {
        this.add('ec2:CreateClientVpnRoute');
        return this;
    }
    /**
     * Grants permission to create a customer gateway, which provides information to AWS about your customer gateway device
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateCustomerGateway.html
     */
    toCreateCustomerGateway() {
        this.add('ec2:CreateCustomerGateway');
        return this;
    }
    /**
     * Grants permission to create a default subnet in a specified Availability Zone in a default VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateDefaultSubnet.html
     */
    toCreateDefaultSubnet() {
        this.add('ec2:CreateDefaultSubnet');
        return this;
    }
    /**
     * Grants permission to create a default VPC with a default subnet in each Availability Zone
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateDefaultVpc.html
     */
    toCreateDefaultVpc() {
        this.add('ec2:CreateDefaultVpc');
        return this;
    }
    /**
     * Grants permission to create a set of DHCP options for a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateDhcpOptions.html
     */
    toCreateDhcpOptions() {
        this.add('ec2:CreateDhcpOptions');
        return this;
    }
    /**
     * Grants permission to create an egress-only internet gateway for a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateEgressOnlyInternetGateway.html
     */
    toCreateEgressOnlyInternetGateway() {
        this.add('ec2:CreateEgressOnlyInternetGateway');
        return this;
    }
    /**
     * Grants permission to launch an EC2 Fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateFleet.html
     */
    toCreateFleet() {
        this.add('ec2:CreateFleet');
        return this;
    }
    /**
     * Grants permission to create one or more flow logs to capture IP traffic for a network interface
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateFlowLogs.html
     */
    toCreateFlowLogs() {
        this.add('ec2:CreateFlowLogs');
        return this;
    }
    /**
     * Grants permission to create an Amazon FPGA Image (AFI) from a design checkpoint (DCP)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateFpgaImage.html
     */
    toCreateFpgaImage() {
        this.add('ec2:CreateFpgaImage');
        return this;
    }
    /**
     * Grants permission to create an Amazon EBS-backed AMI from a stopped or running Amazon EBS-backed instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateImage.html
     */
    toCreateImage() {
        this.add('ec2:CreateImage');
        return this;
    }
    /**
     * Grants permission to export a running or stopped instance to an Amazon S3 bucket
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateInstanceExportTask.html
     */
    toCreateInstanceExportTask() {
        this.add('ec2:CreateInstanceExportTask');
        return this;
    }
    /**
     * Grants permission to create an internet gateway for a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateInternetGateway.html
     */
    toCreateInternetGateway() {
        this.add('ec2:CreateInternetGateway');
        return this;
    }
    /**
     * Grants permission to create a 2048-bit RSA key pair
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateKeyPair.html
     */
    toCreateKeyPair() {
        this.add('ec2:CreateKeyPair');
        return this;
    }
    /**
     * Grants permission to create a launch template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateLaunchTemplate.html
     */
    toCreateLaunchTemplate() {
        this.add('ec2:CreateLaunchTemplate');
        return this;
    }
    /**
     * Grants permission to create a new version of a launch template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateLaunchTemplateVersion.html
     */
    toCreateLaunchTemplateVersion() {
        this.add('ec2:CreateLaunchTemplateVersion');
        return this;
    }
    /**
     * Grants permission to create a static route for a local gateway route table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateLocalGatewayRoute.html
     */
    toCreateLocalGatewayRoute() {
        this.add('ec2:CreateLocalGatewayRoute');
        return this;
    }
    /**
     * Grants permission to associate a VPC with a local gateway route table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateLocalGatewayRouteTableVpcAssociation.html
     */
    toCreateLocalGatewayRouteTableVpcAssociation() {
        this.add('ec2:CreateLocalGatewayRouteTableVpcAssociation');
        return this;
    }
    /**
     * Grants permission to create a managed prefix list
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateManagedPrefixList.html
     */
    toCreateManagedPrefixList() {
        this.add('ec2:CreateManagedPrefixList');
        return this;
    }
    /**
     * Grants permission to create a NAT gateway in a subnet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateNatGateway.html
     */
    toCreateNatGateway() {
        this.add('ec2:CreateNatGateway');
        return this;
    }
    /**
     * Grants permission to create a network ACL in a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateNetworkAcl.html
     */
    toCreateNetworkAcl() {
        this.add('ec2:CreateNetworkAcl');
        return this;
    }
    /**
     * Grants permission to create a numbered entry (a rule) in a network ACL
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateNetworkAclEntry.html
     */
    toCreateNetworkAclEntry() {
        this.add('ec2:CreateNetworkAclEntry');
        return this;
    }
    /**
     * Grants permission to create a network interface in a subnet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateNetworkInterface.html
     */
    toCreateNetworkInterface() {
        this.add('ec2:CreateNetworkInterface');
        return this;
    }
    /**
     * Grants permission to create a permission for an AWS-authorized user to perform certain operations on a network interface
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateNetworkInterfacePermission.html
     */
    toCreateNetworkInterfacePermission() {
        this.add('ec2:CreateNetworkInterfacePermission');
        return this;
    }
    /**
     * Grants permission to create a placement group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreatePlacementGroup.html
     */
    toCreatePlacementGroup() {
        this.add('ec2:CreatePlacementGroup');
        return this;
    }
    /**
     * Grants permission to create a listing for Standard Reserved Instances to be sold in the Reserved Instance Marketplace
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateReservedInstancesListing.html
     */
    toCreateReservedInstancesListing() {
        this.add('ec2:CreateReservedInstancesListing');
        return this;
    }
    /**
     * Grants permission to create a route in a VPC route table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateRoute.html
     */
    toCreateRoute() {
        this.add('ec2:CreateRoute');
        return this;
    }
    /**
     * Grants permission to create a route table for a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateRouteTable.html
     */
    toCreateRouteTable() {
        this.add('ec2:CreateRouteTable');
        return this;
    }
    /**
     * Grants permission to create a security group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateSecurityGroup.html
     */
    toCreateSecurityGroup() {
        this.add('ec2:CreateSecurityGroup');
        return this;
    }
    /**
     * Grants permission to create a snapshot of an EBS volume and store it in Amazon S3
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateSnapshot.html
     */
    toCreateSnapshot() {
        this.add('ec2:CreateSnapshot');
        return this;
    }
    /**
     * Grants permission to create crash-consistent snapshots of multiple EBS volumes and store them in Amazon S3
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateSnapshots.html
     */
    toCreateSnapshots() {
        this.add('ec2:CreateSnapshots');
        return this;
    }
    /**
     * Grants permission to create a data feed for Spot Instances to view Spot Instance usage logs
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateSpotDatafeedSubscription.html
     */
    toCreateSpotDatafeedSubscription() {
        this.add('ec2:CreateSpotDatafeedSubscription');
        return this;
    }
    /**
     * Grants permission to create a subnet in a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateSubnet.html
     */
    toCreateSubnet() {
        this.add('ec2:CreateSubnet');
        return this;
    }
    /**
     * Grants permission to add or overwrite one or more tags for Amazon EC2 resources
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifCreateAction()
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTags.html
     */
    toCreateTags() {
        this.add('ec2:CreateTags');
        return this;
    }
    /**
     * Grants permission to create a traffic mirror filter
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTrafficMirrorFilter.html
     */
    toCreateTrafficMirrorFilter() {
        this.add('ec2:CreateTrafficMirrorFilter');
        return this;
    }
    /**
     * Grants permission to create a traffic mirror filter rule
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTrafficMirrorFilterRule.html
     */
    toCreateTrafficMirrorFilterRule() {
        this.add('ec2:CreateTrafficMirrorFilterRule');
        return this;
    }
    /**
     * Grants permission to create a traffic mirror session
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTrafficMirrorSession.html
     */
    toCreateTrafficMirrorSession() {
        this.add('ec2:CreateTrafficMirrorSession');
        return this;
    }
    /**
     * Grants permission to create a traffic mirror target
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTrafficMirrorTarget.html
     */
    toCreateTrafficMirrorTarget() {
        this.add('ec2:CreateTrafficMirrorTarget');
        return this;
    }
    /**
     * Grants permission to create a transit gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTransitGateway.html
     */
    toCreateTransitGateway() {
        this.add('ec2:CreateTransitGateway');
        return this;
    }
    /**
     * Grants permission to create a multicast domain for a transit gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTransitGatewayMulticastDomain.html
     */
    toCreateTransitGatewayMulticastDomain() {
        this.add('ec2:CreateTransitGatewayMulticastDomain');
        return this;
    }
    /**
     * Grants permission to request a transit gateway peering attachment between a requester and accepter transit gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTransitGatewayPeeringAttachment.html
     */
    toCreateTransitGatewayPeeringAttachment() {
        this.add('ec2:CreateTransitGatewayPeeringAttachment');
        return this;
    }
    /**
     * Grants permission to create a transit gateway prefix list reference
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTransitGatewayPrefixListReference.html
     */
    toCreateTransitGatewayPrefixListReference() {
        this.add('ec2:CreateTransitGatewayPrefixListReference');
        return this;
    }
    /**
     * Grants permission to create a static route for a transit gateway route table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTransitGatewayRoute.html
     */
    toCreateTransitGatewayRoute() {
        this.add('ec2:CreateTransitGatewayRoute');
        return this;
    }
    /**
     * Grants permission to create a route table for a transit gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTransitGatewayRouteTable.html
     */
    toCreateTransitGatewayRouteTable() {
        this.add('ec2:CreateTransitGatewayRouteTable');
        return this;
    }
    /**
     * Grants permission to attach a VPC to a transit gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTransitGatewayVpcAttachment.html
     */
    toCreateTransitGatewayVpcAttachment() {
        this.add('ec2:CreateTransitGatewayVpcAttachment');
        return this;
    }
    /**
     * Grants permission to create an EBS volume
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVolume.html
     */
    toCreateVolume() {
        this.add('ec2:CreateVolume');
        return this;
    }
    /**
     * Grants permission to create a VPC with a specified CIDR block
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVpc.html
     */
    toCreateVpc() {
        this.add('ec2:CreateVpc');
        return this;
    }
    /**
     * Grants permission to create a VPC endpoint for an AWS service
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - route53:AssociateVPCWithHostedZone
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVpcEndpoint.html
     */
    toCreateVpcEndpoint() {
        this.add('ec2:CreateVpcEndpoint');
        return this;
    }
    /**
     * Grants permission to create a connection notification for a VPC endpoint or VPC endpoint service
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVpcEndpointConnectionNotification.html
     */
    toCreateVpcEndpointConnectionNotification() {
        this.add('ec2:CreateVpcEndpointConnectionNotification');
        return this;
    }
    /**
     * Grants permission to create a VPC endpoint service configuration to which service consumers (AWS accounts, IAM users, and IAM roles) can connect
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVpcEndpointServiceConfiguration.html
     */
    toCreateVpcEndpointServiceConfiguration() {
        this.add('ec2:CreateVpcEndpointServiceConfiguration');
        return this;
    }
    /**
     * Grants permission to request a VPC peering connection between two VPCs
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVpcPeeringConnection.html
     */
    toCreateVpcPeeringConnection() {
        this.add('ec2:CreateVpcPeeringConnection');
        return this;
    }
    /**
     * Grants permission to create a VPN connection between a virtual private gateway or transit gateway and a customer gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVpnConnection.html
     */
    toCreateVpnConnection() {
        this.add('ec2:CreateVpnConnection');
        return this;
    }
    /**
     * Grants permission to create a static route for a VPN connection between a virtual private gateway and a customer gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVpnConnectionRoute.html
     */
    toCreateVpnConnectionRoute() {
        this.add('ec2:CreateVpnConnectionRoute');
        return this;
    }
    /**
     * Grants permission to create a virtual private gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVpnGateway.html
     */
    toCreateVpnGateway() {
        this.add('ec2:CreateVpnGateway');
        return this;
    }
    /**
     * Grants permission to delete a carrier gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteCarrierGateway.html
     */
    toDeleteCarrierGateway() {
        this.add('ec2:DeleteCarrierGateway');
        return this;
    }
    /**
     * Grants permission to delete a Client VPN endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteClientVpnEndpoint.html
     */
    toDeleteClientVpnEndpoint() {
        this.add('ec2:DeleteClientVpnEndpoint');
        return this;
    }
    /**
     * Grants permission to delete a route from a Client VPN endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteClientVpnRoute.html
     */
    toDeleteClientVpnRoute() {
        this.add('ec2:DeleteClientVpnRoute');
        return this;
    }
    /**
     * Grants permission to delete a customer gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteCustomerGateway.html
     */
    toDeleteCustomerGateway() {
        this.add('ec2:DeleteCustomerGateway');
        return this;
    }
    /**
     * Grants permission to delete a set of DHCP options
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteDhcpOptions.html
     */
    toDeleteDhcpOptions() {
        this.add('ec2:DeleteDhcpOptions');
        return this;
    }
    /**
     * Grants permission to delete an egress-only internet gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteEgressOnlyInternetGateway.html
     */
    toDeleteEgressOnlyInternetGateway() {
        this.add('ec2:DeleteEgressOnlyInternetGateway');
        return this;
    }
    /**
     * Grants permission to delete one or more EC2 Fleets
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteFleets.html
     */
    toDeleteFleets() {
        this.add('ec2:DeleteFleets');
        return this;
    }
    /**
     * Grants permission to delete one or more flow logs
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteFlowLogs.html
     */
    toDeleteFlowLogs() {
        this.add('ec2:DeleteFlowLogs');
        return this;
    }
    /**
     * Grants permission to delete an Amazon FPGA Image (AFI)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteFpgaImage.html
     */
    toDeleteFpgaImage() {
        this.add('ec2:DeleteFpgaImage');
        return this;
    }
    /**
     * Grants permission to delete an internet gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteInternetGateway.html
     */
    toDeleteInternetGateway() {
        this.add('ec2:DeleteInternetGateway');
        return this;
    }
    /**
     * Grants permission to delete a key pair by removing the public key from Amazon EC2
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteKeyPair.html
     */
    toDeleteKeyPair() {
        this.add('ec2:DeleteKeyPair');
        return this;
    }
    /**
     * Grants permission to delete a launch template and its associated versions
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteLaunchTemplate.html
     */
    toDeleteLaunchTemplate() {
        this.add('ec2:DeleteLaunchTemplate');
        return this;
    }
    /**
     * Grants permission to delete one or more versions of a launch template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteLaunchTemplateVersions.html
     */
    toDeleteLaunchTemplateVersions() {
        this.add('ec2:DeleteLaunchTemplateVersions');
        return this;
    }
    /**
     * Grants permission to delete a route from a local gateway route table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteLocalGatewayRoute.html
     */
    toDeleteLocalGatewayRoute() {
        this.add('ec2:DeleteLocalGatewayRoute');
        return this;
    }
    /**
     * Grants permission to delete an association between a VPC and local gateway route table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteLocalGatewayRouteTableVpcAssociation.html
     */
    toDeleteLocalGatewayRouteTableVpcAssociation() {
        this.add('ec2:DeleteLocalGatewayRouteTableVpcAssociation');
        return this;
    }
    /**
     * Grants permission to delete a managed prefix list
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteManagedPrefixList.html
     */
    toDeleteManagedPrefixList() {
        this.add('ec2:DeleteManagedPrefixList');
        return this;
    }
    /**
     * Grants permission to delete a NAT gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteNatGateway.html
     */
    toDeleteNatGateway() {
        this.add('ec2:DeleteNatGateway');
        return this;
    }
    /**
     * Grants permission to delete a network ACL
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteNetworkAcl.html
     */
    toDeleteNetworkAcl() {
        this.add('ec2:DeleteNetworkAcl');
        return this;
    }
    /**
     * Grants permission to delete an inbound or outbound entry (rule) from a network ACL
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteNetworkAclEntry.html
     */
    toDeleteNetworkAclEntry() {
        this.add('ec2:DeleteNetworkAclEntry');
        return this;
    }
    /**
     * Grants permission to delete a detached network interface
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteNetworkInterface.html
     */
    toDeleteNetworkInterface() {
        this.add('ec2:DeleteNetworkInterface');
        return this;
    }
    /**
     * Grants permission to delete a permission that is associated with a network interface
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteNetworkInterfacePermission.html
     */
    toDeleteNetworkInterfacePermission() {
        this.add('ec2:DeleteNetworkInterfacePermission');
        return this;
    }
    /**
     * Grants permission to delete a placement group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeletePlacementGroup.html
     */
    toDeletePlacementGroup() {
        this.add('ec2:DeletePlacementGroup');
        return this;
    }
    /**
     * Grants permission to delete a route from a route table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteRoute.html
     */
    toDeleteRoute() {
        this.add('ec2:DeleteRoute');
        return this;
    }
    /**
     * Grants permission to delete a route table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteRouteTable.html
     */
    toDeleteRouteTable() {
        this.add('ec2:DeleteRouteTable');
        return this;
    }
    /**
     * Grants permission to delete a security group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteSecurityGroup.html
     */
    toDeleteSecurityGroup() {
        this.add('ec2:DeleteSecurityGroup');
        return this;
    }
    /**
     * Grants permission to delete a snapshot of an EBS volume
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteSnapshot.html
     */
    toDeleteSnapshot() {
        this.add('ec2:DeleteSnapshot');
        return this;
    }
    /**
     * Grants permission to delete a data feed for Spot Instances
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteSpotDatafeedSubscription.html
     */
    toDeleteSpotDatafeedSubscription() {
        this.add('ec2:DeleteSpotDatafeedSubscription');
        return this;
    }
    /**
     * Grants permission to delete a subnet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteSubnet.html
     */
    toDeleteSubnet() {
        this.add('ec2:DeleteSubnet');
        return this;
    }
    /**
     * Grants permission to delete one or more tags from Amazon EC2 resources
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTags.html
     */
    toDeleteTags() {
        this.add('ec2:DeleteTags');
        return this;
    }
    /**
     * Grants permission to delete a traffic mirror filter
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTrafficMirrorFilter.html
     */
    toDeleteTrafficMirrorFilter() {
        this.add('ec2:DeleteTrafficMirrorFilter');
        return this;
    }
    /**
     * Grants permission to delete a traffic mirror filter rule
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTrafficMirrorFilterRule.html
     */
    toDeleteTrafficMirrorFilterRule() {
        this.add('ec2:DeleteTrafficMirrorFilterRule');
        return this;
    }
    /**
     * Grants permission to delete a traffic mirror session
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTrafficMirrorSession.html
     */
    toDeleteTrafficMirrorSession() {
        this.add('ec2:DeleteTrafficMirrorSession');
        return this;
    }
    /**
     * Grants permission to delete a traffic mirror target
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTrafficMirrorTarget.html
     */
    toDeleteTrafficMirrorTarget() {
        this.add('ec2:DeleteTrafficMirrorTarget');
        return this;
    }
    /**
     * Grants permission to delete a transit gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTransitGateway.html
     */
    toDeleteTransitGateway() {
        this.add('ec2:DeleteTransitGateway');
        return this;
    }
    /**
     * Grants permissions to delete a transit gateway multicast domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTransitGatewayMulticastDomain.html
     */
    toDeleteTransitGatewayMulticastDomain() {
        this.add('ec2:DeleteTransitGatewayMulticastDomain');
        return this;
    }
    /**
     * Grants permission to delete a peering attachment from a transit gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTransitGatewayPeeringAttachment.html
     */
    toDeleteTransitGatewayPeeringAttachment() {
        this.add('ec2:DeleteTransitGatewayPeeringAttachment');
        return this;
    }
    /**
     * Grants permission to delete a transit gateway prefix list reference
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTransitGatewayPrefixListReference.html
     */
    toDeleteTransitGatewayPrefixListReference() {
        this.add('ec2:DeleteTransitGatewayPrefixListReference');
        return this;
    }
    /**
     * Grants permission to delete a route from a transit gateway route table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTransitGatewayRoute.html
     */
    toDeleteTransitGatewayRoute() {
        this.add('ec2:DeleteTransitGatewayRoute');
        return this;
    }
    /**
     * Grants permission to delete a transit gateway route table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTransitGatewayRouteTable.html
     */
    toDeleteTransitGatewayRouteTable() {
        this.add('ec2:DeleteTransitGatewayRouteTable');
        return this;
    }
    /**
     * Grants permission to delete a VPC attachment from a transit gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTransitGatewayVpcAttachment.html
     */
    toDeleteTransitGatewayVpcAttachment() {
        this.add('ec2:DeleteTransitGatewayVpcAttachment');
        return this;
    }
    /**
     * Grants permission to delete an EBS volume
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteVolume.html
     */
    toDeleteVolume() {
        this.add('ec2:DeleteVolume');
        return this;
    }
    /**
     * Grants permission to delete a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteVpc.html
     */
    toDeleteVpc() {
        this.add('ec2:DeleteVpc');
        return this;
    }
    /**
     * Grants permission to delete one or more VPC endpoint connection notifications
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteVpcEndpointConnectionNotifications.html
     */
    toDeleteVpcEndpointConnectionNotifications() {
        this.add('ec2:DeleteVpcEndpointConnectionNotifications');
        return this;
    }
    /**
     * Grants permission to delete one or more VPC endpoint service configurations
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteVpcEndpointServiceConfigurations.html
     */
    toDeleteVpcEndpointServiceConfigurations() {
        this.add('ec2:DeleteVpcEndpointServiceConfigurations');
        return this;
    }
    /**
     * Grants permission to delete one or more VPC endpoints
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteVpcEndpoints.html
     */
    toDeleteVpcEndpoints() {
        this.add('ec2:DeleteVpcEndpoints');
        return this;
    }
    /**
     * Grants permission to delete a VPC peering connection
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteVpcPeeringConnection.html
     */
    toDeleteVpcPeeringConnection() {
        this.add('ec2:DeleteVpcPeeringConnection');
        return this;
    }
    /**
     * Grants permission to delete a VPN connection
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteVpnConnection.html
     */
    toDeleteVpnConnection() {
        this.add('ec2:DeleteVpnConnection');
        return this;
    }
    /**
     * Grants permission to delete a static route for a VPN connection between a virtual private gateway and a customer gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteVpnConnectionRoute.html
     */
    toDeleteVpnConnectionRoute() {
        this.add('ec2:DeleteVpnConnectionRoute');
        return this;
    }
    /**
     * Grants permission to delete a virtual private gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteVpnGateway.html
     */
    toDeleteVpnGateway() {
        this.add('ec2:DeleteVpnGateway');
        return this;
    }
    /**
     * Grants permission to release an IP address range that was provisioned through bring your own IP addresses (BYOIP), and to delete the corresponding address pool
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeprovisionByoipCidr.html
     */
    toDeprovisionByoipCidr() {
        this.add('ec2:DeprovisionByoipCidr');
        return this;
    }
    /**
     * Grants permission to deregister an Amazon Machine Image (AMI)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeregisterImage.html
     */
    toDeregisterImage() {
        this.add('ec2:DeregisterImage');
        return this;
    }
    /**
     * Grants permission to remove tags from the set of tags to include in notifications about scheduled events for your instances
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeregisterInstanceEventNotificationAttributes.html
     */
    toDeregisterInstanceEventNotificationAttributes() {
        this.add('ec2:DeregisterInstanceEventNotificationAttributes');
        return this;
    }
    /**
     * Grants permission to deregister one or more network interface members from a group IP address in a transit gateway multicast domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeregisterTransitGatewayMulticastGroupMembers.html
     */
    toDeregisterTransitGatewayMulticastGroupMembers() {
        this.add('ec2:DeregisterTransitGatewayMulticastGroupMembers');
        return this;
    }
    /**
     * Grants permission to deregister one or more network interface sources from a group IP address in a transit gateway multicast domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeregisterTransitGatewayMulticastGroupSources.html
     */
    toDeregisterTransitGatewayMulticastGroupSources() {
        this.add('ec2:DeregisterTransitGatewayMulticastGroupSources');
        return this;
    }
    /**
     * Grants permission to describe the attributes of the AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeAccountAttributes.html
     */
    toDescribeAccountAttributes() {
        this.add('ec2:DescribeAccountAttributes');
        return this;
    }
    /**
     * Grants permission to describe one or more Elastic IP addresses
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeAddresses.html
     */
    toDescribeAddresses() {
        this.add('ec2:DescribeAddresses');
        return this;
    }
    /**
     * Grants permission to describe the longer ID format settings for all resource types
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeAggregateIdFormat.html
     */
    toDescribeAggregateIdFormat() {
        this.add('ec2:DescribeAggregateIdFormat');
        return this;
    }
    /**
     * Grants permission to describe one or more of the Availability Zones that are available to you
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeAvailabilityZones.html
     */
    toDescribeAvailabilityZones() {
        this.add('ec2:DescribeAvailabilityZones');
        return this;
    }
    /**
     * Grants permission to describe one or more bundling tasks
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeBundleTasks.html
     */
    toDescribeBundleTasks() {
        this.add('ec2:DescribeBundleTasks');
        return this;
    }
    /**
     * Grants permission to describe the IP address ranges that were provisioned through bring your own IP addresses (BYOIP)
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeByoipCidrs.html
     */
    toDescribeByoipCidrs() {
        this.add('ec2:DescribeByoipCidrs');
        return this;
    }
    /**
     * Grants permission to describe one or more Capacity Reservations
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeCapacityReservations.html
     */
    toDescribeCapacityReservations() {
        this.add('ec2:DescribeCapacityReservations');
        return this;
    }
    /**
     * Grants permission to describe one or more Carrier Gateways
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeCarrierGateways.html
     */
    toDescribeCarrierGateways() {
        this.add('ec2:DescribeCarrierGateways');
        return this;
    }
    /**
     * Grants permission to describe one or more linked EC2-Classic instances
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeClassicLinkInstances.html
     */
    toDescribeClassicLinkInstances() {
        this.add('ec2:DescribeClassicLinkInstances');
        return this;
    }
    /**
     * Grants permission to describe the authorization rules for a Client VPN endpoint
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeClientVpnAuthorizationRules.html
     */
    toDescribeClientVpnAuthorizationRules() {
        this.add('ec2:DescribeClientVpnAuthorizationRules');
        return this;
    }
    /**
     * Grants permission to describe active client connections and connections that have been terminated within the last 60 minutes for a Client VPN endpoint
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeClientVpnConnections.html
     */
    toDescribeClientVpnConnections() {
        this.add('ec2:DescribeClientVpnConnections');
        return this;
    }
    /**
     * Grants permission to describe one or more Client VPN endpoints
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeClientVpnEndpoints.html
     */
    toDescribeClientVpnEndpoints() {
        this.add('ec2:DescribeClientVpnEndpoints');
        return this;
    }
    /**
     * Grants permission to describe the routes for a Client VPN endpoint
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeClientVpnRoutes.html
     */
    toDescribeClientVpnRoutes() {
        this.add('ec2:DescribeClientVpnRoutes');
        return this;
    }
    /**
     * Grants permission to describe the target networks that are associated with a Client VPN endpoint
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeClientVpnTargetNetworks.html
     */
    toDescribeClientVpnTargetNetworks() {
        this.add('ec2:DescribeClientVpnTargetNetworks');
        return this;
    }
    /**
     * Grants permission to describe the specified customer-owned address pools or all of your customer-owned address pools
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeCoipPools.html
     */
    toDescribeCoipPools() {
        this.add('ec2:DescribeCoipPools');
        return this;
    }
    /**
     * Grants permission to describe one or more conversion tasks
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeConversionTasks.html
     */
    toDescribeConversionTasks() {
        this.add('ec2:DescribeConversionTasks');
        return this;
    }
    /**
     * Grants permission to describe one or more customer gateways
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeCustomerGateways.html
     */
    toDescribeCustomerGateways() {
        this.add('ec2:DescribeCustomerGateways');
        return this;
    }
    /**
     * Grants permission to describe one or more DHCP options sets
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeDhcpOptions.html
     */
    toDescribeDhcpOptions() {
        this.add('ec2:DescribeDhcpOptions');
        return this;
    }
    /**
     * Grants permission to describe one or more egress-only internet gateways
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeEgressOnlyInternetGateways.html
     */
    toDescribeEgressOnlyInternetGateways() {
        this.add('ec2:DescribeEgressOnlyInternetGateways');
        return this;
    }
    /**
     * Grants permission to describe an Elastic Graphics accelerator that is associated with an instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeElasticGpus.html
     */
    toDescribeElasticGpus() {
        this.add('ec2:DescribeElasticGpus');
        return this;
    }
    /**
     * Grants permission to describe one or more export image tasks
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeExportImageTasks.html
     */
    toDescribeExportImageTasks() {
        this.add('ec2:DescribeExportImageTasks');
        return this;
    }
    /**
     * Grants permission to describe one or more export instance tasks
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeExportTasks.html
     */
    toDescribeExportTasks() {
        this.add('ec2:DescribeExportTasks');
        return this;
    }
    /**
     * Grants permission to describe the state of fast snapshot restores for snapshots
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeFastSnapshotRestores.html
     */
    toDescribeFastSnapshotRestores() {
        this.add('ec2:DescribeFastSnapshotRestores');
        return this;
    }
    /**
     * Grants permission to describe the events for an EC2 Fleet during a specified time
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeFleetHistory.html
     */
    toDescribeFleetHistory() {
        this.add('ec2:DescribeFleetHistory');
        return this;
    }
    /**
     * Grants permission to describe the running instances for an EC2 Fleet
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeFleetInstances.html
     */
    toDescribeFleetInstances() {
        this.add('ec2:DescribeFleetInstances');
        return this;
    }
    /**
     * Grants permission to describe one or more EC2 Fleets
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeFleets.html
     */
    toDescribeFleets() {
        this.add('ec2:DescribeFleets');
        return this;
    }
    /**
     * Grants permission to describe one or more flow logs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeFlowLogs.html
     */
    toDescribeFlowLogs() {
        this.add('ec2:DescribeFlowLogs');
        return this;
    }
    /**
     * Grants permission to describe the attributes of an Amazon FPGA Image (AFI)
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeFpgaImageAttribute.html
     */
    toDescribeFpgaImageAttribute() {
        this.add('ec2:DescribeFpgaImageAttribute');
        return this;
    }
    /**
     * Grants permission to describe one or more Amazon FPGA Images (AFIs)
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeFpgaImages.html
     */
    toDescribeFpgaImages() {
        this.add('ec2:DescribeFpgaImages');
        return this;
    }
    /**
     * Grants permission to describe the Dedicated Host Reservations that are available to purchase
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeHostReservationOfferings.html
     */
    toDescribeHostReservationOfferings() {
        this.add('ec2:DescribeHostReservationOfferings');
        return this;
    }
    /**
     * Grants permission to describe the Dedicated Host Reservations that are associated with Dedicated Hosts in the AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeHostReservations.html
     */
    toDescribeHostReservations() {
        this.add('ec2:DescribeHostReservations');
        return this;
    }
    /**
     * Grants permission to describe one or more Dedicated Hosts
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeHosts.html
     */
    toDescribeHosts() {
        this.add('ec2:DescribeHosts');
        return this;
    }
    /**
     * Grants permission to describe the IAM instance profile associations
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeIamInstanceProfileAssociations.html
     */
    toDescribeIamInstanceProfileAssociations() {
        this.add('ec2:DescribeIamInstanceProfileAssociations');
        return this;
    }
    /**
     * Grants permission to describe the ID format settings for resources
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeIdFormat.html
     */
    toDescribeIdFormat() {
        this.add('ec2:DescribeIdFormat');
        return this;
    }
    /**
     * Grants permission to describe the ID format settings for resources for an IAM user, IAM role, or root user
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeIdentityIdFormat.html
     */
    toDescribeIdentityIdFormat() {
        this.add('ec2:DescribeIdentityIdFormat');
        return this;
    }
    /**
     * Grants permission to describe an attribute of an Amazon Machine Image (AMI)
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeImageAttribute.html
     */
    toDescribeImageAttribute() {
        this.add('ec2:DescribeImageAttribute');
        return this;
    }
    /**
     * Grants permission to describe one or more images (AMIs, AKIs, and ARIs)
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeImages.html
     */
    toDescribeImages() {
        this.add('ec2:DescribeImages');
        return this;
    }
    /**
     * Grants permission to describe import virtual machine or import snapshot tasks
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeImportImageTasks.html
     */
    toDescribeImportImageTasks() {
        this.add('ec2:DescribeImportImageTasks');
        return this;
    }
    /**
     * Grants permission to describe import snapshot tasks
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeImportSnapshotTasks.html
     */
    toDescribeImportSnapshotTasks() {
        this.add('ec2:DescribeImportSnapshotTasks');
        return this;
    }
    /**
     * Grants permission to describe the attributes of an instance
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeInstanceAttribute.html
     */
    toDescribeInstanceAttribute() {
        this.add('ec2:DescribeInstanceAttribute');
        return this;
    }
    /**
     * Grants permission to describe the credit option for CPU usage of one or more burstable performance instances
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeInstanceCreditSpecifications.html
     */
    toDescribeInstanceCreditSpecifications() {
        this.add('ec2:DescribeInstanceCreditSpecifications');
        return this;
    }
    /**
     * Grants permission to describe the set of tags to include in notifications about scheduled events for your instances
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeInstanceEventNotificationAttributes.html
     */
    toDescribeInstanceEventNotificationAttributes() {
        this.add('ec2:DescribeInstanceEventNotificationAttributes');
        return this;
    }
    /**
     * Grants permission to describe the status of one or more instances
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeInstanceStatus.html
     */
    toDescribeInstanceStatus() {
        this.add('ec2:DescribeInstanceStatus');
        return this;
    }
    /**
     * Grants permission to describe the set of instance types that are offered in a location
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeInstanceTypeOfferings.html
     */
    toDescribeInstanceTypeOfferings() {
        this.add('ec2:DescribeInstanceTypeOfferings');
        return this;
    }
    /**
     * Grants permission to describe the details of instance types that are offered in a location
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeInstanceTypes.html
     */
    toDescribeInstanceTypes() {
        this.add('ec2:DescribeInstanceTypes');
        return this;
    }
    /**
     * Grants permission to describe one or more instances
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeInstances.html
     */
    toDescribeInstances() {
        this.add('ec2:DescribeInstances');
        return this;
    }
    /**
     * Grants permission to describe one or more internet gateways
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeInternetGateways.html
     */
    toDescribeInternetGateways() {
        this.add('ec2:DescribeInternetGateways');
        return this;
    }
    /**
     * Grants permission to describe one or more key pairs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeKeyPairs.html
     */
    toDescribeKeyPairs() {
        this.add('ec2:DescribeKeyPairs');
        return this;
    }
    /**
     * Grants permission to describe one or more launch template versions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeLaunchTemplateVersions.html
     */
    toDescribeLaunchTemplateVersions() {
        this.add('ec2:DescribeLaunchTemplateVersions');
        return this;
    }
    /**
     * Grants permission to describe one or more launch templates
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeLaunchTemplates.html
     */
    toDescribeLaunchTemplates() {
        this.add('ec2:DescribeLaunchTemplates');
        return this;
    }
    /**
     * Grants permission to describe the associations between virtual interface groups and local gateway route tables
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeLocalGatewayRouteTableVirtualInterfaceGroupAssociations.html
     */
    toDescribeLocalGatewayRouteTableVirtualInterfaceGroupAssociations() {
        this.add('ec2:DescribeLocalGatewayRouteTableVirtualInterfaceGroupAssociations');
        return this;
    }
    /**
     * Grants permission to describe an association between VPCs and local gateway route tables
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeLocalGatewayRouteTableVpcAssociations.html
     */
    toDescribeLocalGatewayRouteTableVpcAssociations() {
        this.add('ec2:DescribeLocalGatewayRouteTableVpcAssociations');
        return this;
    }
    /**
     * Grants permission to describe one or more local gateway route tables
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeLocalGatewayRouteTables.html
     */
    toDescribeLocalGatewayRouteTables() {
        this.add('ec2:DescribeLocalGatewayRouteTables');
        return this;
    }
    /**
     * Grants permission to describe local gateway virtual interface groups
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeLocalGatewayVirtualInterfaceGroups.html
     */
    toDescribeLocalGatewayVirtualInterfaceGroups() {
        this.add('ec2:DescribeLocalGatewayVirtualInterfaceGroups');
        return this;
    }
    /**
     * Grants permission to describe local gateway virtual interfaces
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeLocalGatewayVirtualInterfaces.html
     */
    toDescribeLocalGatewayVirtualInterfaces() {
        this.add('ec2:DescribeLocalGatewayVirtualInterfaces');
        return this;
    }
    /**
     * Grants permission to describe one or more local gateways
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeLocalGateways.html
     */
    toDescribeLocalGateways() {
        this.add('ec2:DescribeLocalGateways');
        return this;
    }
    /**
     * Grants permission to describe your managed prefix lists and any AWS-managed prefix lists
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeManagedPrefixLists.html
     */
    toDescribeManagedPrefixLists() {
        this.add('ec2:DescribeManagedPrefixLists');
        return this;
    }
    /**
     * Grants permission to describe Elastic IP addresses that are being moved to the EC2-VPC platform
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeMovingAddresses.html
     */
    toDescribeMovingAddresses() {
        this.add('ec2:DescribeMovingAddresses');
        return this;
    }
    /**
     * Grants permission to describe one or more NAT gateways
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeNatGateways.html
     */
    toDescribeNatGateways() {
        this.add('ec2:DescribeNatGateways');
        return this;
    }
    /**
     * Grants permission to describe one or more network ACLs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeNetworkAcls.html
     */
    toDescribeNetworkAcls() {
        this.add('ec2:DescribeNetworkAcls');
        return this;
    }
    /**
     * Grants permission to describe a network interface attribute
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeNetworkInterfaceAttribute.html
     */
    toDescribeNetworkInterfaceAttribute() {
        this.add('ec2:DescribeNetworkInterfaceAttribute');
        return this;
    }
    /**
     * Grants permission to describe the permissions that are associated with a network interface
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeNetworkInterfacePermissions.html
     */
    toDescribeNetworkInterfacePermissions() {
        this.add('ec2:DescribeNetworkInterfacePermissions');
        return this;
    }
    /**
     * Grants permission to describe one or more network interfaces
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeNetworkInterfaces.html
     */
    toDescribeNetworkInterfaces() {
        this.add('ec2:DescribeNetworkInterfaces');
        return this;
    }
    /**
     * Grants permission to describe one or more placement groups
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribePlacementGroups.html
     */
    toDescribePlacementGroups() {
        this.add('ec2:DescribePlacementGroups');
        return this;
    }
    /**
     * Grants permission to describe available AWS services in a prefix list format
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribePrefixLists.html
     */
    toDescribePrefixLists() {
        this.add('ec2:DescribePrefixLists');
        return this;
    }
    /**
     * Grants permission to describe the ID format settings for the root user and all IAM roles and IAM users that have explicitly specified a longer ID (17-character ID) preference
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribePrincipalIdFormat.html
     */
    toDescribePrincipalIdFormat() {
        this.add('ec2:DescribePrincipalIdFormat');
        return this;
    }
    /**
     * Grants permission to describe one or more IPv4 address pools
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribePublicIpv4Pools.html
     */
    toDescribePublicIpv4Pools() {
        this.add('ec2:DescribePublicIpv4Pools');
        return this;
    }
    /**
     * Grants permission to describe one or more AWS Regions that are currently available in your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeRegions.html
     */
    toDescribeRegions() {
        this.add('ec2:DescribeRegions');
        return this;
    }
    /**
     * Grants permission to describe one or more purchased Reserved Instances in your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeReservedInstances.html
     */
    toDescribeReservedInstances() {
        this.add('ec2:DescribeReservedInstances');
        return this;
    }
    /**
     * Grants permission to describe your account's Reserved Instance listings in the Reserved Instance Marketplace
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeReservedInstancesListings.html
     */
    toDescribeReservedInstancesListings() {
        this.add('ec2:DescribeReservedInstancesListings');
        return this;
    }
    /**
     * Grants permission to describe the modifications made to one or more Reserved Instances
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeReservedInstancesModifications.html
     */
    toDescribeReservedInstancesModifications() {
        this.add('ec2:DescribeReservedInstancesModifications');
        return this;
    }
    /**
     * Grants permission to describe the Reserved Instance offerings that are available for purchase
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeReservedInstancesOfferings.html
     */
    toDescribeReservedInstancesOfferings() {
        this.add('ec2:DescribeReservedInstancesOfferings');
        return this;
    }
    /**
     * Grants permission to describe one or more route tables
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeRouteTables.html
     */
    toDescribeRouteTables() {
        this.add('ec2:DescribeRouteTables');
        return this;
    }
    /**
     * Grants permission to find available schedules for Scheduled Instances
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeScheduledInstanceAvailability.html
     */
    toDescribeScheduledInstanceAvailability() {
        this.add('ec2:DescribeScheduledInstanceAvailability');
        return this;
    }
    /**
     * Grants permission to describe one or more Scheduled Instances in your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeScheduledInstances.html
     */
    toDescribeScheduledInstances() {
        this.add('ec2:DescribeScheduledInstances');
        return this;
    }
    /**
     * Grants permission to describe the VPCs on the other side of a VPC peering connection that are referencing specified VPC security groups
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSecurityGroupReferences.html
     */
    toDescribeSecurityGroupReferences() {
        this.add('ec2:DescribeSecurityGroupReferences');
        return this;
    }
    /**
     * Grants permission to describe one or more security groups
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSecurityGroups.html
     */
    toDescribeSecurityGroups() {
        this.add('ec2:DescribeSecurityGroups');
        return this;
    }
    /**
     * Grants permission to describe an attribute of a snapshot
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSnapshotAttribute.html
     */
    toDescribeSnapshotAttribute() {
        this.add('ec2:DescribeSnapshotAttribute');
        return this;
    }
    /**
     * Grants permission to describe one or more EBS snapshots
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSnapshots.html
     */
    toDescribeSnapshots() {
        this.add('ec2:DescribeSnapshots');
        return this;
    }
    /**
     * Grants permission to describe the data feed for Spot Instances
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSpotDatafeedSubscription.html
     */
    toDescribeSpotDatafeedSubscription() {
        this.add('ec2:DescribeSpotDatafeedSubscription');
        return this;
    }
    /**
     * Grants permission to describe the running instances for a Spot Fleet
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSpotFleetInstances.html
     */
    toDescribeSpotFleetInstances() {
        this.add('ec2:DescribeSpotFleetInstances');
        return this;
    }
    /**
     * Grants permission to describe the events for a Spot Fleet request during a specified time
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSpotFleetRequestHistory.html
     */
    toDescribeSpotFleetRequestHistory() {
        this.add('ec2:DescribeSpotFleetRequestHistory');
        return this;
    }
    /**
     * Grants permission to describe one or more Spot Fleet requests
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSpotFleetRequests.html
     */
    toDescribeSpotFleetRequests() {
        this.add('ec2:DescribeSpotFleetRequests');
        return this;
    }
    /**
     * Grants permission to describe one or more Spot Instance requests
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSpotInstanceRequests.html
     */
    toDescribeSpotInstanceRequests() {
        this.add('ec2:DescribeSpotInstanceRequests');
        return this;
    }
    /**
     * Grants permission to describe the Spot Instance price history
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSpotPriceHistory.html
     */
    toDescribeSpotPriceHistory() {
        this.add('ec2:DescribeSpotPriceHistory');
        return this;
    }
    /**
     * Grants permission to describe the stale security group rules for security groups in a specified VPC
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeStaleSecurityGroups.html
     */
    toDescribeStaleSecurityGroups() {
        this.add('ec2:DescribeStaleSecurityGroups');
        return this;
    }
    /**
     * Grants permission to describe one or more subnets
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSubnets.html
     */
    toDescribeSubnets() {
        this.add('ec2:DescribeSubnets');
        return this;
    }
    /**
     * Grants permission to describe one or more tags for an Amazon EC2 resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeTags.html
     */
    toDescribeTags() {
        this.add('ec2:DescribeTags');
        return this;
    }
    /**
     * Grants permission to describe one or more traffic mirror filters
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeTrafficMirrorFilters.html
     */
    toDescribeTrafficMirrorFilters() {
        this.add('ec2:DescribeTrafficMirrorFilters');
        return this;
    }
    /**
     * Grants permission to describe one or more traffic mirror sessions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeTrafficMirrorSessions.html
     */
    toDescribeTrafficMirrorSessions() {
        this.add('ec2:DescribeTrafficMirrorSessions');
        return this;
    }
    /**
     * Grants permission to describe one or more traffic mirror targets
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeTrafficMirrorTargets.html
     */
    toDescribeTrafficMirrorTargets() {
        this.add('ec2:DescribeTrafficMirrorTargets');
        return this;
    }
    /**
     * Grants permission to describe one or more attachments between resources and transit gateways
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeTransitGatewayAttachments.html
     */
    toDescribeTransitGatewayAttachments() {
        this.add('ec2:DescribeTransitGatewayAttachments');
        return this;
    }
    /**
     * Grants permission to describe one or more transit gateway multicast domains
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeTransitGatewayMulticastDomains.html
     */
    toDescribeTransitGatewayMulticastDomains() {
        this.add('ec2:DescribeTransitGatewayMulticastDomains');
        return this;
    }
    /**
     * Grants permission to describe one or more transit gateway peering attachments
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeTransitGatewayPeeringAttachments.html
     */
    toDescribeTransitGatewayPeeringAttachments() {
        this.add('ec2:DescribeTransitGatewayPeeringAttachments');
        return this;
    }
    /**
     * Grants permission to describe one or more transit gateway route tables
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeTransitGatewayRouteTables.html
     */
    toDescribeTransitGatewayRouteTables() {
        this.add('ec2:DescribeTransitGatewayRouteTables');
        return this;
    }
    /**
     * Grants permission to describe one or more VPC attachments on a transit gateway
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeTransitGatewayVpcAttachments.html
     */
    toDescribeTransitGatewayVpcAttachments() {
        this.add('ec2:DescribeTransitGatewayVpcAttachments');
        return this;
    }
    /**
     * Grants permission to describe one or more transit gateways
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeTransitGateways.html
     */
    toDescribeTransitGateways() {
        this.add('ec2:DescribeTransitGateways');
        return this;
    }
    /**
     * Grants permission to describe an attribute of an EBS volume
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVolumeAttribute.html
     */
    toDescribeVolumeAttribute() {
        this.add('ec2:DescribeVolumeAttribute');
        return this;
    }
    /**
     * Grants permission to describe the status of one or more EBS volumes
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVolumeStatus.html
     */
    toDescribeVolumeStatus() {
        this.add('ec2:DescribeVolumeStatus');
        return this;
    }
    /**
     * Grants permission to describe one or more EBS volumes
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVolumes.html
     */
    toDescribeVolumes() {
        this.add('ec2:DescribeVolumes');
        return this;
    }
    /**
     * Grants permission to describe the current modification status of one or more EBS volumes
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVolumesModifications.html
     */
    toDescribeVolumesModifications() {
        this.add('ec2:DescribeVolumesModifications');
        return this;
    }
    /**
     * Grants permission to describe an attribute of a VPC
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcAttribute.html
     */
    toDescribeVpcAttribute() {
        this.add('ec2:DescribeVpcAttribute');
        return this;
    }
    /**
     * Grants permission to describe the ClassicLink status of one or more VPCs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcClassicLink.html
     */
    toDescribeVpcClassicLink() {
        this.add('ec2:DescribeVpcClassicLink');
        return this;
    }
    /**
     * Grants permission to describe the ClassicLink DNS support status of one or more VPCs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcClassicLinkDnsSupport.html
     */
    toDescribeVpcClassicLinkDnsSupport() {
        this.add('ec2:DescribeVpcClassicLinkDnsSupport');
        return this;
    }
    /**
     * Grants permission to describe the connection notifications for VPC endpoints and VPC endpoint services
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcEndpointConnectionNotifications.html
     */
    toDescribeVpcEndpointConnectionNotifications() {
        this.add('ec2:DescribeVpcEndpointConnectionNotifications');
        return this;
    }
    /**
     * Grants permission to describe the VPC endpoint connections to your VPC endpoint services
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcEndpointConnections.html
     */
    toDescribeVpcEndpointConnections() {
        this.add('ec2:DescribeVpcEndpointConnections');
        return this;
    }
    /**
     * Grants permission to describe VPC endpoint service configurations (your services)
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcEndpointServiceConfigurations.html
     */
    toDescribeVpcEndpointServiceConfigurations() {
        this.add('ec2:DescribeVpcEndpointServiceConfigurations');
        return this;
    }
    /**
     * Grants permission to describe the principals (service consumers) that are permitted to discover your VPC endpoint service
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcEndpointServicePermissions.html
     */
    toDescribeVpcEndpointServicePermissions() {
        this.add('ec2:DescribeVpcEndpointServicePermissions');
        return this;
    }
    /**
     * Grants permission to describe all supported AWS services that can be specified when creating a VPC endpoint
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcEndpointServices.html
     */
    toDescribeVpcEndpointServices() {
        this.add('ec2:DescribeVpcEndpointServices');
        return this;
    }
    /**
     * Grants permission to describe one or more VPC endpoints
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcEndpoints.html
     */
    toDescribeVpcEndpoints() {
        this.add('ec2:DescribeVpcEndpoints');
        return this;
    }
    /**
     * Grants permission to describe one or more VPC peering connections
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcPeeringConnections.html
     */
    toDescribeVpcPeeringConnections() {
        this.add('ec2:DescribeVpcPeeringConnections');
        return this;
    }
    /**
     * Grants permission to describe one or more VPCs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcs.html
     */
    toDescribeVpcs() {
        this.add('ec2:DescribeVpcs');
        return this;
    }
    /**
     * Grants permission to describe one or more VPN connections
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpnConnections.html
     */
    toDescribeVpnConnections() {
        this.add('ec2:DescribeVpnConnections');
        return this;
    }
    /**
     * Grants permission to describe one or more virtual private gateways
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpnGateways.html
     */
    toDescribeVpnGateways() {
        this.add('ec2:DescribeVpnGateways');
        return this;
    }
    /**
     * Grants permission to unlink (detach) a linked EC2-Classic instance from a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DetachClassicLinkVpc.html
     */
    toDetachClassicLinkVpc() {
        this.add('ec2:DetachClassicLinkVpc');
        return this;
    }
    /**
     * Grants permission to detach an internet gateway from a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DetachInternetGateway.html
     */
    toDetachInternetGateway() {
        this.add('ec2:DetachInternetGateway');
        return this;
    }
    /**
     * Grants permission to detach a network interface from an instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DetachNetworkInterface.html
     */
    toDetachNetworkInterface() {
        this.add('ec2:DetachNetworkInterface');
        return this;
    }
    /**
     * Grants permission to detach an EBS volume from an instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DetachVolume.html
     */
    toDetachVolume() {
        this.add('ec2:DetachVolume');
        return this;
    }
    /**
     * Grants permission to detach a virtual private gateway from a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DetachVpnGateway.html
     */
    toDetachVpnGateway() {
        this.add('ec2:DetachVpnGateway');
        return this;
    }
    /**
     * Grants permission to disable EBS encryption by default for your account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisableEbsEncryptionByDefault.html
     */
    toDisableEbsEncryptionByDefault() {
        this.add('ec2:DisableEbsEncryptionByDefault');
        return this;
    }
    /**
     * Grants permission to disable fast snapshot restores for one or more snapshots in specified Availability Zones
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisableFastSnapshotRestores.html
     */
    toDisableFastSnapshotRestores() {
        this.add('ec2:DisableFastSnapshotRestores');
        return this;
    }
    /**
     * Grants permission to disable a resource attachment from propagating routes to the specified propagation route table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisableTransitGatewayRouteTablePropagation.html
     */
    toDisableTransitGatewayRouteTablePropagation() {
        this.add('ec2:DisableTransitGatewayRouteTablePropagation');
        return this;
    }
    /**
     * Grants permission to disable a virtual private gateway from propagating routes to a specified route table of a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisableVgwRoutePropagation.html
     */
    toDisableVgwRoutePropagation() {
        this.add('ec2:DisableVgwRoutePropagation');
        return this;
    }
    /**
     * Grants permission to disable ClassicLink for a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisableVpcClassicLink.html
     */
    toDisableVpcClassicLink() {
        this.add('ec2:DisableVpcClassicLink');
        return this;
    }
    /**
     * Grants permission to disable ClassicLink DNS support for a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisableVpcClassicLinkDnsSupport.html
     */
    toDisableVpcClassicLinkDnsSupport() {
        this.add('ec2:DisableVpcClassicLinkDnsSupport');
        return this;
    }
    /**
     * Grants permission to disassociate an Elastic IP address from an instance or network interface
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisassociateAddress.html
     */
    toDisassociateAddress() {
        this.add('ec2:DisassociateAddress');
        return this;
    }
    /**
     * Grants permission to disassociate a target network from a Client VPN endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisassociateClientVpnTargetNetwork.html
     */
    toDisassociateClientVpnTargetNetwork() {
        this.add('ec2:DisassociateClientVpnTargetNetwork');
        return this;
    }
    /**
     * Grants permission to disassociate an IAM instance profile from a running or stopped instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisassociateIamInstanceProfile.html
     */
    toDisassociateIamInstanceProfile() {
        this.add('ec2:DisassociateIamInstanceProfile');
        return this;
    }
    /**
     * Grants permission to disassociate a subnet from a route table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisassociateRouteTable.html
     */
    toDisassociateRouteTable() {
        this.add('ec2:DisassociateRouteTable');
        return this;
    }
    /**
     * Grants permission to disassociate a CIDR block from a subnet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisassociateSubnetCidrBlock.html
     */
    toDisassociateSubnetCidrBlock() {
        this.add('ec2:DisassociateSubnetCidrBlock');
        return this;
    }
    /**
     * Grants permission to disassociate one or more subnets from a transit gateway multicast domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisassociateTransitGatewayMulticastDomain.html
     */
    toDisassociateTransitGatewayMulticastDomain() {
        this.add('ec2:DisassociateTransitGatewayMulticastDomain');
        return this;
    }
    /**
     * Grants permission to disassociate a resource attachment from a transit gateway route table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisassociateTransitGatewayRouteTable.html
     */
    toDisassociateTransitGatewayRouteTable() {
        this.add('ec2:DisassociateTransitGatewayRouteTable');
        return this;
    }
    /**
     * Grants permission to disassociate a CIDR block from a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisassociateVpcCidrBlock.html
     */
    toDisassociateVpcCidrBlock() {
        this.add('ec2:DisassociateVpcCidrBlock');
        return this;
    }
    /**
     * Grants permission to enable EBS encryption by default for your account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_EnableEbsEncryptionByDefault.html
     */
    toEnableEbsEncryptionByDefault() {
        this.add('ec2:EnableEbsEncryptionByDefault');
        return this;
    }
    /**
     * Grants permission to enable fast snapshot restores for one or more snapshots in specified Availability Zones
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_EnableFastSnapshotRestores.html
     */
    toEnableFastSnapshotRestores() {
        this.add('ec2:EnableFastSnapshotRestores');
        return this;
    }
    /**
     * Grants permission to enable an attachment to propagate routes to a propagation route table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_EnableTransitGatewayRouteTablePropagation.html
     */
    toEnableTransitGatewayRouteTablePropagation() {
        this.add('ec2:EnableTransitGatewayRouteTablePropagation');
        return this;
    }
    /**
     * Grants permission to enable a virtual private gateway to propagate routes to a VPC route table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_EnableVgwRoutePropagation.html
     */
    toEnableVgwRoutePropagation() {
        this.add('ec2:EnableVgwRoutePropagation');
        return this;
    }
    /**
     * Grants permission to enable I/O operations for a volume that had I/O operations disabled
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_EnableVolumeIO.html
     */
    toEnableVolumeIO() {
        this.add('ec2:EnableVolumeIO');
        return this;
    }
    /**
     * Grants permission to enable a VPC for ClassicLink
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_EnableVpcClassicLink.html
     */
    toEnableVpcClassicLink() {
        this.add('ec2:EnableVpcClassicLink');
        return this;
    }
    /**
     * Grants permission to enable a VPC to support DNS hostname resolution for ClassicLink
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_EnableVpcClassicLinkDnsSupport.html
     */
    toEnableVpcClassicLinkDnsSupport() {
        this.add('ec2:EnableVpcClassicLinkDnsSupport');
        return this;
    }
    /**
     * Grants permission to download the client certificate revocation list for a Client VPN endpoint
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ExportClientVpnClientCertificateRevocationList.html
     */
    toExportClientVpnClientCertificateRevocationList() {
        this.add('ec2:ExportClientVpnClientCertificateRevocationList');
        return this;
    }
    /**
     * Grants permission to download the contents of the Client VPN endpoint configuration file for a Client VPN endpoint
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ExportClientVpnClientConfiguration.html
     */
    toExportClientVpnClientConfiguration() {
        this.add('ec2:ExportClientVpnClientConfiguration');
        return this;
    }
    /**
     * Grants permission to export an Amazon Machine Image (AMI) to a VM file
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ExportImage.html
     */
    toExportImage() {
        this.add('ec2:ExportImage');
        return this;
    }
    /**
     * Grants permission to export routes from a transit gateway route table to an Amazon S3 bucket
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ExportTransitGatewayRoutes.html
     */
    toExportTransitGatewayRoutes() {
        this.add('ec2:ExportTransitGatewayRoutes');
        return this;
    }
    /**
     * Grants permission to get usage information about a Capacity Reservation
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetCapacityReservationUsage.html
     */
    toGetCapacityReservationUsage() {
        this.add('ec2:GetCapacityReservationUsage');
        return this;
    }
    /**
     * Grants permission to describe the allocations from the specified customer-owned address pool
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetCoipPoolUsage.html
     */
    toGetCoipPoolUsage() {
        this.add('ec2:GetCoipPoolUsage');
        return this;
    }
    /**
     * Grants permission to get the console output for an instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetConsoleOutput.html
     */
    toGetConsoleOutput() {
        this.add('ec2:GetConsoleOutput');
        return this;
    }
    /**
     * Grants permission to retrieve a JPG-format screenshot of a running instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetConsoleScreenshot.html
     */
    toGetConsoleScreenshot() {
        this.add('ec2:GetConsoleScreenshot');
        return this;
    }
    /**
     * Grants permission to get the default credit option for CPU usage of a burstable performance instance family
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetDefaultCreditSpecification.html
     */
    toGetDefaultCreditSpecification() {
        this.add('ec2:GetDefaultCreditSpecification');
        return this;
    }
    /**
     * Grants permission to get the ID of the default customer master key (CMK) for EBS encryption by default
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetEbsDefaultKmsKeyId.html
     */
    toGetEbsDefaultKmsKeyId() {
        this.add('ec2:GetEbsDefaultKmsKeyId');
        return this;
    }
    /**
     * Grants permission to describe whether EBS encryption by default is enabled for your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetEbsEncryptionByDefault.html
     */
    toGetEbsEncryptionByDefault() {
        this.add('ec2:GetEbsEncryptionByDefault');
        return this;
    }
    /**
     * Grants permission to preview a reservation purchase with configurations that match those of a Dedicated Host
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetHostReservationPurchasePreview.html
     */
    toGetHostReservationPurchasePreview() {
        this.add('ec2:GetHostReservationPurchasePreview');
        return this;
    }
    /**
     * Grants permission to get the configuration data of the specified instance for use with a new launch template or launch template version
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetLaunchTemplateData.html
     */
    toGetLaunchTemplateData() {
        this.add('ec2:GetLaunchTemplateData');
        return this;
    }
    /**
     * Grants permission to get information about the resources that are associated with the specified managed prefix list
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetManagedPrefixListAssociations.html
     */
    toGetManagedPrefixListAssociations() {
        this.add('ec2:GetManagedPrefixListAssociations');
        return this;
    }
    /**
     * Grants permission to get information about the entries for a specified managed prefix list
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetManagedPrefixListEntries.html
     */
    toGetManagedPrefixListEntries() {
        this.add('ec2:GetManagedPrefixListEntries');
        return this;
    }
    /**
     * Grants permission to retrieve the encrypted administrator password for a running Windows instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetPasswordData.html
     */
    toGetPasswordData() {
        this.add('ec2:GetPasswordData');
        return this;
    }
    /**
     * Grants permission to return a quote and exchange information for exchanging one or more Convertible Reserved Instances for a new Convertible Reserved Instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetReservedInstancesExchangeQuote.html
     */
    toGetReservedInstancesExchangeQuote() {
        this.add('ec2:GetReservedInstancesExchangeQuote');
        return this;
    }
    /**
     * Grants permission to list the route tables to which a resource attachment propagates routes
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetTransitGatewayAttachmentPropagations.html
     */
    toGetTransitGatewayAttachmentPropagations() {
        this.add('ec2:GetTransitGatewayAttachmentPropagations');
        return this;
    }
    /**
     * Grants permission to get information about the associations for a transit gateway multicast domain
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetTransitGatewayMulticastDomainAssociations.html
     */
    toGetTransitGatewayMulticastDomainAssociations() {
        this.add('ec2:GetTransitGatewayMulticastDomainAssociations');
        return this;
    }
    /**
     * Grants permission to get information about prefix list references for a transit gateway route table
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetTransitGatewayPrefixListReference.html
     */
    toGetTransitGatewayPrefixListReferences() {
        this.add('ec2:GetTransitGatewayPrefixListReferences');
        return this;
    }
    /**
     * Grants permission to get information about associations for a transit gateway route table
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetTransitGatewayRouteTableAssociations.html
     */
    toGetTransitGatewayRouteTableAssociations() {
        this.add('ec2:GetTransitGatewayRouteTableAssociations');
        return this;
    }
    /**
     * Grants permission to get information about the route table propagations for a transit gateway route table
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetTransitGatewayRouteTablePropagations.html
     */
    toGetTransitGatewayRouteTablePropagations() {
        this.add('ec2:GetTransitGatewayRouteTablePropagations');
        return this;
    }
    /**
     * Grants permission to upload a client certificate revocation list to a Client VPN endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ImportClientVpnClientCertificateRevocationList.html
     */
    toImportClientVpnClientCertificateRevocationList() {
        this.add('ec2:ImportClientVpnClientCertificateRevocationList');
        return this;
    }
    /**
     * Grants permission to import single or multi-volume disk images or EBS snapshots into an Amazon Machine Image (AMI)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ImportImage.html
     */
    toImportImage() {
        this.add('ec2:ImportImage');
        return this;
    }
    /**
     * Grants permission to create an import instance task using metadata from a disk image
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ImportInstance.html
     */
    toImportInstance() {
        this.add('ec2:ImportInstance');
        return this;
    }
    /**
     * Grants permission to import a public key from an RSA key pair that was created with a third-party tool
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ImportKeyPair.html
     */
    toImportKeyPair() {
        this.add('ec2:ImportKeyPair');
        return this;
    }
    /**
     * Grants permission to import a disk into an EBS snapshot
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ImportSnapshot.html
     */
    toImportSnapshot() {
        this.add('ec2:ImportSnapshot');
        return this;
    }
    /**
     * Grants permission to create an import volume task using metadata from a disk image
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ImportVolume.html
     */
    toImportVolume() {
        this.add('ec2:ImportVolume');
        return this;
    }
    /**
     * Grants permission to modify a Capacity Reservation's capacity and the conditions under which it is to be released
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyCapacityReservation.html
     */
    toModifyCapacityReservation() {
        this.add('ec2:ModifyCapacityReservation');
        return this;
    }
    /**
     * Grants permission to modify a Client VPN endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyClientVpnEndpoint.html
     */
    toModifyClientVpnEndpoint() {
        this.add('ec2:ModifyClientVpnEndpoint');
        return this;
    }
    /**
     * Grants permission to change the account level default credit option for CPU usage of burstable performance instances
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyDefaultCreditSpecification.html
     */
    toModifyDefaultCreditSpecification() {
        this.add('ec2:ModifyDefaultCreditSpecification');
        return this;
    }
    /**
     * Grants permission to change the default customer master key (CMK) for EBS encryption by default for your account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyEbsDefaultKmsKeyId.html
     */
    toModifyEbsDefaultKmsKeyId() {
        this.add('ec2:ModifyEbsDefaultKmsKeyId');
        return this;
    }
    /**
     * Grants permission to modify an EC2 Fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyFleet.html
     */
    toModifyFleet() {
        this.add('ec2:ModifyFleet');
        return this;
    }
    /**
     * Grants permission to modify an attribute of an Amazon FPGA Image (AFI)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyFpgaImageAttribute.html
     */
    toModifyFpgaImageAttribute() {
        this.add('ec2:ModifyFpgaImageAttribute');
        return this;
    }
    /**
     * Grants permission to modify a Dedicated Host
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyHosts.html
     */
    toModifyHosts() {
        this.add('ec2:ModifyHosts');
        return this;
    }
    /**
     * Grants permission to modify the ID format for a resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyIdFormat.html
     */
    toModifyIdFormat() {
        this.add('ec2:ModifyIdFormat');
        return this;
    }
    /**
     * Grants permission to modify the ID format of a resource for a specific principal in your account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyIdentityIdFormat.html
     */
    toModifyIdentityIdFormat() {
        this.add('ec2:ModifyIdentityIdFormat');
        return this;
    }
    /**
     * Grants permission to modify an attribute of an Amazon Machine Image (AMI)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyImageAttribute.html
     */
    toModifyImageAttribute() {
        this.add('ec2:ModifyImageAttribute');
        return this;
    }
    /**
     * Grants permission to modify an attribute of an instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyInstanceAttribute.html
     */
    toModifyInstanceAttribute() {
        this.add('ec2:ModifyInstanceAttribute');
        return this;
    }
    /**
     * Grants permission to modify the Capacity Reservation settings for a stopped instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyInstanceCapacityReservationAttributes.html
     */
    toModifyInstanceCapacityReservationAttributes() {
        this.add('ec2:ModifyInstanceCapacityReservationAttributes');
        return this;
    }
    /**
     * Grants permission to modify the credit option for CPU usage on an instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyInstanceCreditSpecification.html
     */
    toModifyInstanceCreditSpecification() {
        this.add('ec2:ModifyInstanceCreditSpecification');
        return this;
    }
    /**
     * Grants permission to modify the start time for a scheduled EC2 instance event
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyInstanceEventStartTime.html
     */
    toModifyInstanceEventStartTime() {
        this.add('ec2:ModifyInstanceEventStartTime');
        return this;
    }
    /**
     * Grants permission to modify the metadata options for an instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyInstanceMetadataOptions.html
     */
    toModifyInstanceMetadataOptions() {
        this.add('ec2:ModifyInstanceMetadataOptions');
        return this;
    }
    /**
     * Grants permission to modify the placement attributes for an instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyInstancePlacement.html
     */
    toModifyInstancePlacement() {
        this.add('ec2:ModifyInstancePlacement');
        return this;
    }
    /**
     * Grants permission to modify a launch template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyLaunchTemplate.html
     */
    toModifyLaunchTemplate() {
        this.add('ec2:ModifyLaunchTemplate');
        return this;
    }
    /**
     * Grants permission to modify a managed prefix list
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyManagedPrefixList.html
     */
    toModifyManagedPrefixList() {
        this.add('ec2:ModifyManagedPrefixList');
        return this;
    }
    /**
     * Grants permission to modify an attribute of a network interface
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyNetworkInterfaceAttribute.html
     */
    toModifyNetworkInterfaceAttribute() {
        this.add('ec2:ModifyNetworkInterfaceAttribute');
        return this;
    }
    /**
     * Grants permission to modify attributes of one or more Reserved Instances
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyReservedInstances.html
     */
    toModifyReservedInstances() {
        this.add('ec2:ModifyReservedInstances');
        return this;
    }
    /**
     * Grants permission to add or remove permission settings for a snapshot
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifySnapshotAttribute.html
     */
    toModifySnapshotAttribute() {
        this.add('ec2:ModifySnapshotAttribute');
        return this;
    }
    /**
     * Grants permission to modify a Spot Fleet request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifySpotFleetRequest.html
     */
    toModifySpotFleetRequest() {
        this.add('ec2:ModifySpotFleetRequest');
        return this;
    }
    /**
     * Grants permission to modify an attribute of a subnet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifySubnetAttribute.html
     */
    toModifySubnetAttribute() {
        this.add('ec2:ModifySubnetAttribute');
        return this;
    }
    /**
     * Grants permission to allow or restrict mirroring network services
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyTrafficMirrorFilterNetworkServices.html
     */
    toModifyTrafficMirrorFilterNetworkServices() {
        this.add('ec2:ModifyTrafficMirrorFilterNetworkServices');
        return this;
    }
    /**
     * Grants permission to modify a traffic mirror rule
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyTrafficMirrorFilterRule.html
     */
    toModifyTrafficMirrorFilterRule() {
        this.add('ec2:ModifyTrafficMirrorFilterRule');
        return this;
    }
    /**
     * Grants permission to modify a traffic mirror session
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyTrafficMirrorSession.html
     */
    toModifyTrafficMirrorSession() {
        this.add('ec2:ModifyTrafficMirrorSession');
        return this;
    }
    /**
     * Grants permission to modify a transit gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyTransitGateway.html
     */
    toModifyTransitGateway() {
        this.add('ec2:ModifyTransitGateway');
        return this;
    }
    /**
     * Grants permission to modify a transit gateway prefix list reference
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyTransitGatewayPrefixListReference.html
     */
    toModifyTransitGatewayPrefixListReference() {
        this.add('ec2:ModifyTransitGatewayPrefixListReference');
        return this;
    }
    /**
     * Grants permission to modify a VPC attachment on a transit gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyTransitGatewayVpcAttachment.html
     */
    toModifyTransitGatewayVpcAttachment() {
        this.add('ec2:ModifyTransitGatewayVpcAttachment');
        return this;
    }
    /**
     * Grants permission to modify the parameters of an EBS volume
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVolume.html
     */
    toModifyVolume() {
        this.add('ec2:ModifyVolume');
        return this;
    }
    /**
     * Grants permission to modify an attribute of a volume
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVolumeAttribute.html
     */
    toModifyVolumeAttribute() {
        this.add('ec2:ModifyVolumeAttribute');
        return this;
    }
    /**
     * Grants permission to modify an attribute of a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVpcAttribute.html
     */
    toModifyVpcAttribute() {
        this.add('ec2:ModifyVpcAttribute');
        return this;
    }
    /**
     * Grants permission to modify an attribute of a VPC endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVpcEndpoint.html
     */
    toModifyVpcEndpoint() {
        this.add('ec2:ModifyVpcEndpoint');
        return this;
    }
    /**
     * Grants permission to modify a connection notification for a VPC endpoint or VPC endpoint service
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVpcEndpointConnectionNotification.html
     */
    toModifyVpcEndpointConnectionNotification() {
        this.add('ec2:ModifyVpcEndpointConnectionNotification');
        return this;
    }
    /**
     * Grants permission to modify the attributes of a VPC endpoint service configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVpcEndpointServiceConfiguration.html
     */
    toModifyVpcEndpointServiceConfiguration() {
        this.add('ec2:ModifyVpcEndpointServiceConfiguration');
        return this;
    }
    /**
     * Grants permission to modify the permissions for a VPC endpoint service
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVpcEndpointServicePermissions.html
     */
    toModifyVpcEndpointServicePermissions() {
        this.add('ec2:ModifyVpcEndpointServicePermissions');
        return this;
    }
    /**
     * Grants permission to modify the VPC peering connection options on one side of a VPC peering connection
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVpcPeeringConnectionOptions.html
     */
    toModifyVpcPeeringConnectionOptions() {
        this.add('ec2:ModifyVpcPeeringConnectionOptions');
        return this;
    }
    /**
     * Grants permission to modify the instance tenancy attribute of a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVpcTenancy.html
     */
    toModifyVpcTenancy() {
        this.add('ec2:ModifyVpcTenancy');
        return this;
    }
    /**
     * Grants permission to modify the target gateway of a Site-to-Site VPN connection
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVpnConnection.html
     */
    toModifyVpnConnection() {
        this.add('ec2:ModifyVpnConnection');
        return this;
    }
    /**
     * Grants permission to modify the certificate for a Site-to-Site VPN connection
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVpnTunnelCertificate
     */
    toModifyVpnTunnelCertificate() {
        this.add('ec2:ModifyVpnTunnelCertificate');
        return this;
    }
    /**
     * Grants permission to modify the options for a Site-to-Site VPN connection
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVpnTunnelOptions.html
     */
    toModifyVpnTunnelOptions() {
        this.add('ec2:ModifyVpnTunnelOptions');
        return this;
    }
    /**
     * Grants permission to enable detailed monitoring for a running instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_MonitorInstances.html
     */
    toMonitorInstances() {
        this.add('ec2:MonitorInstances');
        return this;
    }
    /**
     * Grants permission to move an Elastic IP address from the EC2-Classic platform to the EC2-VPC platform
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_MoveAddressToVpc.html
     */
    toMoveAddressToVpc() {
        this.add('ec2:MoveAddressToVpc');
        return this;
    }
    /**
     * Grants permission to provision an address range for use in AWS through bring your own IP addresses (BYOIP), and to create a corresponding address pool
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ProvisionByoipCidr.html
     */
    toProvisionByoipCidr() {
        this.add('ec2:ProvisionByoipCidr');
        return this;
    }
    /**
     * Grants permission to purchase a reservation with configurations that match those of a Dedicated Host
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_PurchaseHostReservation.html
     */
    toPurchaseHostReservation() {
        this.add('ec2:PurchaseHostReservation');
        return this;
    }
    /**
     * Grants permission to purchase a Reserved Instance offering
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_PurchaseReservedInstancesOffering.html
     */
    toPurchaseReservedInstancesOffering() {
        this.add('ec2:PurchaseReservedInstancesOffering');
        return this;
    }
    /**
     * Grants permission to purchase one or more Scheduled Instances with a specified schedule
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_PurchaseScheduledInstances.html
     */
    toPurchaseScheduledInstances() {
        this.add('ec2:PurchaseScheduledInstances');
        return this;
    }
    /**
     * Grants permission to request a reboot of one or more instances
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RebootInstances.html
     */
    toRebootInstances() {
        this.add('ec2:RebootInstances');
        return this;
    }
    /**
     * Grants permission to register an Amazon Machine Image (AMI)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RegisterImage.html
     */
    toRegisterImage() {
        this.add('ec2:RegisterImage');
        return this;
    }
    /**
     * Grants permission to add tags to the set of tags to include in notifications about scheduled events for your instances
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RegisterInstanceEventNotificationAttributes.html
     */
    toRegisterInstanceEventNotificationAttributes() {
        this.add('ec2:RegisterInstanceEventNotificationAttributes');
        return this;
    }
    /**
     * Grants permission to register one or more network interfaces as a member of a group IP address in a transit gateway multicast domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RegisterTransitGatewayMulticastGroupMembers.html
     */
    toRegisterTransitGatewayMulticastGroupMembers() {
        this.add('ec2:RegisterTransitGatewayMulticastGroupMembers');
        return this;
    }
    /**
     * Grants permission to register one or more network interfaces as a source of a group IP address in a transit gateway multicast domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RegisterTransitGatewayMulticastGroupSources.html
     */
    toRegisterTransitGatewayMulticastGroupSources() {
        this.add('ec2:RegisterTransitGatewayMulticastGroupSources');
        return this;
    }
    /**
     * Grants permission to reject a transit gateway peering attachment request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RejectTransitGatewayPeeringAttachment.html
     */
    toRejectTransitGatewayPeeringAttachment() {
        this.add('ec2:RejectTransitGatewayPeeringAttachment');
        return this;
    }
    /**
     * Grants permission to reject a request to attach a VPC to a transit gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RejectTransitGatewayVpcAttachment.html
     */
    toRejectTransitGatewayVpcAttachment() {
        this.add('ec2:RejectTransitGatewayVpcAttachment');
        return this;
    }
    /**
     * Grants permission to reject one or more VPC endpoint connection requests to a VPC endpoint service
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RejectVpcEndpointConnections.html
     */
    toRejectVpcEndpointConnections() {
        this.add('ec2:RejectVpcEndpointConnections');
        return this;
    }
    /**
     * Grants permission to reject a VPC peering connection request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RejectVpcPeeringConnection.html
     */
    toRejectVpcPeeringConnection() {
        this.add('ec2:RejectVpcPeeringConnection');
        return this;
    }
    /**
     * Grants permission to release an Elastic IP address
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ReleaseAddress.html
     */
    toReleaseAddress() {
        this.add('ec2:ReleaseAddress');
        return this;
    }
    /**
     * Grants permission to release one or more On-Demand Dedicated Hosts
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ReleaseHosts.html
     */
    toReleaseHosts() {
        this.add('ec2:ReleaseHosts');
        return this;
    }
    /**
     * Grants permission to replace an IAM instance profile for an instance
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ReplaceIamInstanceProfileAssociation.html
     */
    toReplaceIamInstanceProfileAssociation() {
        this.add('ec2:ReplaceIamInstanceProfileAssociation');
        return this;
    }
    /**
     * Grants permission to change which network ACL a subnet is associated with
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ReplaceNetworkAclAssociation.html
     */
    toReplaceNetworkAclAssociation() {
        this.add('ec2:ReplaceNetworkAclAssociation');
        return this;
    }
    /**
     * Grants permission to replace an entry (rule) in a network ACL
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ReplaceNetworkAclEntry.html
     */
    toReplaceNetworkAclEntry() {
        this.add('ec2:ReplaceNetworkAclEntry');
        return this;
    }
    /**
     * Grants permission to replace a route within a route table in a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ReplaceRoute.html
     */
    toReplaceRoute() {
        this.add('ec2:ReplaceRoute');
        return this;
    }
    /**
     * Grants permission to change the route table that is associated with a subnet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ReplaceRouteTableAssociation.html
     */
    toReplaceRouteTableAssociation() {
        this.add('ec2:ReplaceRouteTableAssociation');
        return this;
    }
    /**
     * Grants permission to replace a route in a transit gateway route table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ReplaceTransitGatewayRoute.html
     */
    toReplaceTransitGatewayRoute() {
        this.add('ec2:ReplaceTransitGatewayRoute');
        return this;
    }
    /**
     * Grants permission to submit feedback about the status of an instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ReportInstanceStatus.html
     */
    toReportInstanceStatus() {
        this.add('ec2:ReportInstanceStatus');
        return this;
    }
    /**
     * Grants permission to create a Spot Fleet request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RequestSpotFleet.html
     */
    toRequestSpotFleet() {
        this.add('ec2:RequestSpotFleet');
        return this;
    }
    /**
     * Grants permission to create a Spot Instance request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RequestSpotInstances.html
     */
    toRequestSpotInstances() {
        this.add('ec2:RequestSpotInstances');
        return this;
    }
    /**
     * Grants permission to reset the default customer master key (CMK) for EBS encryption for your account to use the AWS-managed CMK for EBS
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ResetEbsDefaultKmsKeyId.html
     */
    toResetEbsDefaultKmsKeyId() {
        this.add('ec2:ResetEbsDefaultKmsKeyId');
        return this;
    }
    /**
     * Grants permission to reset an attribute of an Amazon FPGA Image (AFI) to its default value
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ResetFpgaImageAttribute.html
     */
    toResetFpgaImageAttribute() {
        this.add('ec2:ResetFpgaImageAttribute');
        return this;
    }
    /**
     * Grants permission to reset an attribute of an Amazon Machine Image (AMI) to its default value
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ResetImageAttribute.html
     */
    toResetImageAttribute() {
        this.add('ec2:ResetImageAttribute');
        return this;
    }
    /**
     * Grants permission to reset an attribute of an instance to its default value
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ResetInstanceAttribute.html
     */
    toResetInstanceAttribute() {
        this.add('ec2:ResetInstanceAttribute');
        return this;
    }
    /**
     * Grants permission to reset an attribute of a network interface
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ResetNetworkInterfaceAttribute.html
     */
    toResetNetworkInterfaceAttribute() {
        this.add('ec2:ResetNetworkInterfaceAttribute');
        return this;
    }
    /**
     * Grants permission to reset permission settings for a snapshot
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ResetSnapshotAttribute.html
     */
    toResetSnapshotAttribute() {
        this.add('ec2:ResetSnapshotAttribute');
        return this;
    }
    /**
     * Grants permission to restore an Elastic IP address that was previously moved to the EC2-VPC platform back to the EC2-Classic platform
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RestoreAddressToClassic.html
     */
    toRestoreAddressToClassic() {
        this.add('ec2:RestoreAddressToClassic');
        return this;
    }
    /**
     * Grants permission to restore the entries from a previous version of a managed prefix list to a new version of the prefix list
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RestoreManagedPrefixListVersion.html
     */
    toRestoreManagedPrefixListVersion() {
        this.add('ec2:RestoreManagedPrefixListVersion');
        return this;
    }
    /**
     * Grants permission to remove an inbound authorization rule from a Client VPN endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RevokeClientVpnIngress.html
     */
    toRevokeClientVpnIngress() {
        this.add('ec2:RevokeClientVpnIngress');
        return this;
    }
    /**
     * Grants permission to remove one or more outbound rules from a VPC security group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RevokeSecurityGroupEgress.html
     */
    toRevokeSecurityGroupEgress() {
        this.add('ec2:RevokeSecurityGroupEgress');
        return this;
    }
    /**
     * Grants permission to remove one or more inbound rules from a security group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RevokeSecurityGroupIngress.html
     */
    toRevokeSecurityGroupIngress() {
        this.add('ec2:RevokeSecurityGroupIngress');
        return this;
    }
    /**
     * Grants permission to launch one or more instances
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RunInstances.html
     */
    toRunInstances() {
        this.add('ec2:RunInstances');
        return this;
    }
    /**
     * Grants permission to launch one or more Scheduled Instances
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RunScheduledInstances.html
     */
    toRunScheduledInstances() {
        this.add('ec2:RunScheduledInstances');
        return this;
    }
    /**
     * Grants permission to search for routes in a local gateway route table
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_SearchLocalGatewayRoutes.html
     */
    toSearchLocalGatewayRoutes() {
        this.add('ec2:SearchLocalGatewayRoutes');
        return this;
    }
    /**
     * Grants permission to search for groups, sources, and members in a transit gateway multicast domain
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_SearchTransitGatewayMulticastGroups.html
     */
    toSearchTransitGatewayMulticastGroups() {
        this.add('ec2:SearchTransitGatewayMulticastGroups');
        return this;
    }
    /**
     * Grants permission to search for routes in a transit gateway route table
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_SearchTransitGatewayRoutes.html
     */
    toSearchTransitGatewayRoutes() {
        this.add('ec2:SearchTransitGatewayRoutes');
        return this;
    }
    /**
     * Grants permission to send a diagnostic interrupt to an Amazon EC2 instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_SendDiagnosticInterrupt.html
     */
    toSendDiagnosticInterrupt() {
        this.add('ec2:SendDiagnosticInterrupt');
        return this;
    }
    /**
     * Grants permission to start a stopped instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_StartInstances.html
     */
    toStartInstances() {
        this.add('ec2:StartInstances');
        return this;
    }
    /**
     * Grants permission to start the private DNS verification process for a VPC endpoint service
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_StartVpcEndpointServicePrivateDnsVerification.html
     */
    toStartVpcEndpointServicePrivateDnsVerification() {
        this.add('ec2:StartVpcEndpointServicePrivateDnsVerification');
        return this;
    }
    /**
     * Grants permission to stop an Amazon EBS-backed instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_StopInstances.html
     */
    toStopInstances() {
        this.add('ec2:StopInstances');
        return this;
    }
    /**
     * Grants permission to terminate active Client VPN endpoint connections
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_TerminateClientVpnConnections.html
     */
    toTerminateClientVpnConnections() {
        this.add('ec2:TerminateClientVpnConnections');
        return this;
    }
    /**
     * Grants permission to shut down one or more instances
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_TerminateInstances.html
     */
    toTerminateInstances() {
        this.add('ec2:TerminateInstances');
        return this;
    }
    /**
     * Grants permission to unassign one or more IPv6 addresses from a network interface
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_UnassignIpv6Addresses.html
     */
    toUnassignIpv6Addresses() {
        this.add('ec2:UnassignIpv6Addresses');
        return this;
    }
    /**
     * Grants permission to unassign one or more secondary private IP addresses from a network interface
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_UnassignPrivateIpAddresses.html
     */
    toUnassignPrivateIpAddresses() {
        this.add('ec2:UnassignPrivateIpAddresses');
        return this;
    }
    /**
     * Grants permission to disable detailed monitoring for a running instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_UnmonitorInstances.html
     */
    toUnmonitorInstances() {
        this.add('ec2:UnmonitorInstances');
        return this;
    }
    /**
     * Grants permission to update descriptions for one or more outbound rules in a VPC security group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_UpdateSecurityGroupRuleDescriptionsEgress.html
     */
    toUpdateSecurityGroupRuleDescriptionsEgress() {
        this.add('ec2:UpdateSecurityGroupRuleDescriptionsEgress');
        return this;
    }
    /**
     * Grants permission to update descriptions for one or more inbound rules in a security group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_UpdateSecurityGroupRuleDescriptionsIngress.html
     */
    toUpdateSecurityGroupRuleDescriptionsIngress() {
        this.add('ec2:UpdateSecurityGroupRuleDescriptionsIngress');
        return this;
    }
    /**
     * Grants permission to stop advertising an address range that was provisioned for use in AWS through bring your own IP addresses (BYOIP)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_WithdrawByoipCidr.html
     */
    toWithdrawByoipCidr() {
        this.add('ec2:WithdrawByoipCidr');
        return this;
    }
    /**
     * Adds a resource of type capacity-reservation to the statement
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-capacity-reservations.html
     *
     * @param capacityReservationId - Identifier for the capacityReservationId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onCapacityReservation(capacityReservationId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:capacity-reservation/${CapacityReservationId}';
        arn = arn.replace('${CapacityReservationId}', capacityReservationId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type carrier-gateway to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/userguide/Carrier_Gateway.html
     *
     * @param carrierGatewayId - Identifier for the carrierGatewayId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onCarrierGateway(carrierGatewayId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:carrier-gateway/${CarrierGatewayId}';
        arn = arn.replace('${CarrierGatewayId}', carrierGatewayId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type client-vpn-endpoint to the statement
     *
     * https://docs.aws.amazon.com/vpn/latest/clientvpn-admin/what-is.html
     *
     * @param clientVpnEndpointId - Identifier for the clientVpnEndpointId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onClientVpnEndpoint(clientVpnEndpointId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:client-vpn-endpoint/${ClientVpnEndpointId}';
        arn = arn.replace('${ClientVpnEndpointId}', clientVpnEndpointId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type customer-gateway to the statement
     *
     * https://docs.aws.amazon.com/vpn/latest/s2svpn/VPC_VPN.html
     *
     * @param customerGatewayId - Identifier for the customerGatewayId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onCustomerGateway(customerGatewayId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:customer-gateway/${CustomerGatewayId}';
        arn = arn.replace('${CustomerGatewayId}', customerGatewayId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type dedicated-host to the statement
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/dedicated-hosts-overview.html
     *
     * @param hostId - Identifier for the hostId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifAutoPlacement()
     * - .ifAvailabilityZone()
     * - .ifHostRecovery()
     * - .ifInstanceType()
     * - .ifQuantity()
     * - .ifResourceTag()
     */
    onDedicatedHost(hostId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:dedicated-host/${HostId}';
        arn = arn.replace('${HostId}', hostId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type dhcp-options to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/userguide/VPC_DHCP_Options.html
     *
     * @param dhcpOptionsId - Identifier for the dhcpOptionsId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onDhcpOptions(dhcpOptionsId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:dhcp-options/${DhcpOptionsId}';
        arn = arn.replace('${DhcpOptionsId}', dhcpOptionsId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type egress-only-internet-gateway to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/userguide/egress-only-internet-gateway.html
     *
     * @param egressOnlyInternetGatewayId - Identifier for the egressOnlyInternetGatewayId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onEgressOnlyInternetGateway(egressOnlyInternetGatewayId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:egress-only-internet-gateway/${EgressOnlyInternetGatewayId}';
        arn = arn.replace('${EgressOnlyInternetGatewayId}', egressOnlyInternetGatewayId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type elastic-gpu to the statement
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/elastic-gpus.html
     *
     * @param elasticGpuId - Identifier for the elasticGpuId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifElasticGpuType()
     */
    onElasticGpu(elasticGpuId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:elastic-gpu/${ElasticGpuId}';
        arn = arn.replace('${ElasticGpuId}', elasticGpuId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type elastic-inference to the statement
     *
     * https://docs.aws.amazon.com/elastic-inference/latest/developerguide/what-is-ei.html
     *
     * @param elasticInferenceAcceleratorId - Identifier for the elasticInferenceAcceleratorId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onElasticInference(elasticInferenceAcceleratorId, account, region, partition) {
        var arn = 'arn:${Partition}:elastic-inference:${Region}:${Account}:elastic-inference-accelerator/${ElasticInferenceAcceleratorId}';
        arn = arn.replace('${ElasticInferenceAcceleratorId}', elasticInferenceAcceleratorId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type export-instance-task to the statement
     *
     * https://docs.aws.amazon.com/vm-import/latest/userguide/vmexport.html
     *
     * @param exportTaskId - Identifier for the exportTaskId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onExportInstanceTask(exportTaskId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:export-instance-task/${ExportTaskId}';
        arn = arn.replace('${ExportTaskId}', exportTaskId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type fleet to the statement
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-fleet.html
     *
     * @param fleetId - Identifier for the fleetId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onFleet(fleetId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:fleet/${FleetId}';
        arn = arn.replace('${FleetId}', fleetId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type fpga-image to the statement
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#EC2_ARN_Format
     *
     * @param fpgaImageId - Identifier for the fpgaImageId.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifOwner()
     * - .ifPublic()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onFpgaImage(fpgaImageId, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}::fpga-image/${FpgaImageId}';
        arn = arn.replace('${FpgaImageId}', fpgaImageId);
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type image to the statement
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/AMIs.html
     *
     * @param imageId - Identifier for the imageId.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifImageType()
     * - .ifOwner()
     * - .ifPublic()
     * - .ifRegion()
     * - .ifResourceTag()
     * - .ifRootDeviceType()
     */
    onImage(imageId, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}::image/${ImageId}';
        arn = arn.replace('${ImageId}', imageId);
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type instance to the statement
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/Instances.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifAvailabilityZone()
     * - .ifEbsOptimized()
     * - .ifInstanceProfile()
     * - .ifInstanceType()
     * - .ifPlacementGroup()
     * - .ifRegion()
     * - .ifResourceTag()
     * - .ifRootDeviceType()
     * - .ifTenancy()
     */
    onInstance(instanceId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:instance/${InstanceId}';
        arn = arn.replace('${InstanceId}', instanceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type internet-gateway to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/userguide/VPC_Internet_Gateway.html
     *
     * @param internetGatewayId - Identifier for the internetGatewayId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onInternetGateway(internetGatewayId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:internet-gateway/${InternetGatewayId}';
        arn = arn.replace('${InternetGatewayId}', internetGatewayId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type key-pair to the statement
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-key-pairs.html
     *
     * @param keyPairName - Identifier for the keyPairName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onKeyPair(keyPairName, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:key-pair/${KeyPairName}';
        arn = arn.replace('${KeyPairName}', keyPairName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type launch-template to the statement
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-launch-templates.html
     *
     * @param launchTemplateId - Identifier for the launchTemplateId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onLaunchTemplate(launchTemplateId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:launch-template/${LaunchTemplateId}';
        arn = arn.replace('${LaunchTemplateId}', launchTemplateId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type local-gateway to the statement
     *
     * https://docs.aws.amazon.com/outposts/latest/userguide/outposts-local-gateways.html
     *
     * @param localGatewayId - Identifier for the localGatewayId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onLocalGateway(localGatewayId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:local-gateway/${LocalGatewayId}';
        arn = arn.replace('${LocalGatewayId}', localGatewayId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type local-gateway-route-table to the statement
     *
     * https://docs.aws.amazon.com/outposts/latest/userguide/outposts-local-gateways.html
     *
     * @param localGatewayRouteTableId - Identifier for the localGatewayRouteTableId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onLocalGatewayRouteTable(localGatewayRouteTableId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:local-gateway-route-table/${LocalGatewayRouteTableId}';
        arn = arn.replace('${LocalGatewayRouteTableId}', localGatewayRouteTableId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type local-gateway-route-table-virtual-interface-group-association to the statement
     *
     * https://docs.aws.amazon.com/outposts/latest/userguide/outposts-local-gateways.html
     *
     * @param localGatewayRouteTableVirtualInterfaceGroupAssociationId - Identifier for the localGatewayRouteTableVirtualInterfaceGroupAssociationId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onLocalGatewayRouteTableVirtualInterfaceGroupAssociation(localGatewayRouteTableVirtualInterfaceGroupAssociationId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:local-gateway-route-table-virtual-interface-group-association/${LocalGatewayRouteTableVirtualInterfaceGroupAssociationId}';
        arn = arn.replace('${LocalGatewayRouteTableVirtualInterfaceGroupAssociationId}', localGatewayRouteTableVirtualInterfaceGroupAssociationId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type local-gateway-route-table-vpc-association to the statement
     *
     * https://docs.aws.amazon.com/outposts/latest/userguide/outposts-local-gateways.html
     *
     * @param localGatewayRouteTableVpcAssociationId - Identifier for the localGatewayRouteTableVpcAssociationId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onLocalGatewayRouteTableVpcAssociation(localGatewayRouteTableVpcAssociationId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:local-gateway-route-table-vpc-association/${LocalGatewayRouteTableVpcAssociationId}';
        arn = arn.replace('${LocalGatewayRouteTableVpcAssociationId}', localGatewayRouteTableVpcAssociationId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type local-gateway-virtual-interface to the statement
     *
     * https://docs.aws.amazon.com/outposts/latest/userguide/outposts-local-gateways.html
     *
     * @param localGatewayVirtualInterfaceId - Identifier for the localGatewayVirtualInterfaceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onLocalGatewayVirtualInterface(localGatewayVirtualInterfaceId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:local-gateway-virtual-interface/${LocalGatewayVirtualInterfaceId}';
        arn = arn.replace('${LocalGatewayVirtualInterfaceId}', localGatewayVirtualInterfaceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type local-gateway-virtual-interface-group to the statement
     *
     * https://docs.aws.amazon.com/outposts/latest/userguide/outposts-local-gateways.html
     *
     * @param localGatewayVirtualInterfaceGroupId - Identifier for the localGatewayVirtualInterfaceGroupId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onLocalGatewayVirtualInterfaceGroup(localGatewayVirtualInterfaceGroupId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:local-gateway-virtual-interface-group/${LocalGatewayVirtualInterfaceGroupId}';
        arn = arn.replace('${LocalGatewayVirtualInterfaceGroupId}', localGatewayVirtualInterfaceGroupId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type natgateway to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/userguide/vpc-nat-gateway.html
     *
     * @param natGatewayId - Identifier for the natGatewayId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onNatgateway(natGatewayId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:natgateway/${NatGatewayId}';
        arn = arn.replace('${NatGatewayId}', natGatewayId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type network-acl to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/userguide/vpc-network-acls.html
     *
     * @param naclId - Identifier for the naclId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     * - .ifVpc()
     */
    onNetworkAcl(naclId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:network-acl/${NaclId}';
        arn = arn.replace('${NaclId}', naclId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type network-interface to the statement
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using-eni.html
     *
     * @param networkInterfaceId - Identifier for the networkInterfaceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifAssociatePublicIpAddress()
     * - .ifAuthorizedService()
     * - .ifAvailabilityZone()
     * - .ifRegion()
     * - .ifResourceTag()
     * - .ifSubnet()
     * - .ifVpc()
     */
    onNetworkInterface(networkInterfaceId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:network-interface/${NetworkInterfaceId}';
        arn = arn.replace('${NetworkInterfaceId}', networkInterfaceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type placement-group to the statement
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/placement-groups.html
     *
     * @param placementGroupName - Identifier for the placementGroupName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifPlacementGroupStrategy()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onPlacementGroup(placementGroupName, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:placement-group/${PlacementGroupName}';
        arn = arn.replace('${PlacementGroupName}', placementGroupName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type prefix-list to the statement
     *
     * @param prefixListId - Identifier for the prefixListId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onPrefixList(prefixListId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:prefix-list/${PrefixListId}';
        arn = arn.replace('${PrefixListId}', prefixListId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type reserved-instances to the statement
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-reserved-instances.html
     *
     * @param reservationId - Identifier for the reservationId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifAvailabilityZone()
     * - .ifInstanceType()
     * - .ifRegion()
     * - .ifReservedInstancesOfferingType()
     * - .ifResourceTag()
     * - .ifTenancy()
     */
    onReservedInstances(reservationId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:reserved-instances/${ReservationId}';
        arn = arn.replace('${ReservationId}', reservationId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type route-table to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/userguide/VPC_Route_Tables.html
     *
     * @param routeTableId - Identifier for the routeTableId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     * - .ifVpc()
     */
    onRouteTable(routeTableId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:route-table/${RouteTableId}';
        arn = arn.replace('${RouteTableId}', routeTableId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type security-group to the statement
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-security-groups.html
     *
     * @param securityGroupId - Identifier for the securityGroupId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     * - .ifVpc()
     */
    onSecurityGroup(securityGroupId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:security-group/${SecurityGroupId}';
        arn = arn.replace('${SecurityGroupId}', securityGroupId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type snapshot to the statement
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSSnapshots.html
     *
     * @param snapshotId - Identifier for the snapshotId.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifOwner()
     * - .ifParentVolume()
     * - .ifRegion()
     * - .ifResourceTag()
     * - .ifSnapshotTime()
     * - .ifVolumeSize()
     */
    onSnapshot(snapshotId, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}::snapshot/${SnapshotId}';
        arn = arn.replace('${SnapshotId}', snapshotId);
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type spot-instance-request to the statement
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using-spot-instances.html
     *
     * @param spotInstanceRequestId - Identifier for the spotInstanceRequestId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onSpotInstanceRequest(spotInstanceRequestId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:spot-instances-request/${SpotInstanceRequestId}';
        arn = arn.replace('${SpotInstanceRequestId}', spotInstanceRequestId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type subnet to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/userguide/VPC_Subnets.html
     *
     * @param subnetId - Identifier for the subnetId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifAvailabilityZone()
     * - .ifRegion()
     * - .ifResourceTag()
     * - .ifVpc()
     */
    onSubnet(subnetId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:subnet/${SubnetId}';
        arn = arn.replace('${SubnetId}', subnetId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type traffic-mirror-session to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/mirroring/traffic-mirroring-session.html
     *
     * @param trafficMirrorSessionId - Identifier for the trafficMirrorSessionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onTrafficMirrorSession(trafficMirrorSessionId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:traffic-mirror-session/${TrafficMirrorSessionId}';
        arn = arn.replace('${TrafficMirrorSessionId}', trafficMirrorSessionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type traffic-mirror-target to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/mirroring/traffic-mirroring-target.html
     *
     * @param trafficMirrorTargetId - Identifier for the trafficMirrorTargetId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onTrafficMirrorTarget(trafficMirrorTargetId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:traffic-mirror-target/${TrafficMirrorTargetId}';
        arn = arn.replace('${TrafficMirrorTargetId}', trafficMirrorTargetId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type traffic-mirror-filter to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/mirroring/traffic-mirroring-filter.html
     *
     * @param trafficMirrorFilterId - Identifier for the trafficMirrorFilterId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onTrafficMirrorFilter(trafficMirrorFilterId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:traffic-mirror-filter/${TrafficMirrorFilterId}';
        arn = arn.replace('${TrafficMirrorFilterId}', trafficMirrorFilterId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type traffic-mirror-filter-rule to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/mirroring/traffic-mirroring-filter.html
     *
     * @param trafficMirrorFilterRuleId - Identifier for the trafficMirrorFilterRuleId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifRegion()
     */
    onTrafficMirrorFilterRule(trafficMirrorFilterRuleId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:traffic-mirror-filter-rule/${TrafficMirrorFilterRuleId}';
        arn = arn.replace('${TrafficMirrorFilterRuleId}', trafficMirrorFilterRuleId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type transit-gateway-attachment to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/tgw/how-transit-gateways-work.html
     *
     * @param transitGatewayAttachmentId - Identifier for the transitGatewayAttachmentId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onTransitGatewayAttachment(transitGatewayAttachmentId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:transit-gateway-attachment/${TransitGatewayAttachmentId}';
        arn = arn.replace('${TransitGatewayAttachmentId}', transitGatewayAttachmentId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type transit-gateway-multicast-domain to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/tgw/tgw-multicast-overview.html
     *
     * @param transitGatewayMulticastDomainId - Identifier for the transitGatewayMulticastDomainId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onTransitGatewayMulticastDomain(transitGatewayMulticastDomainId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:transit-gateway-multicast-domain/${TransitGatewayMulticastDomainId}';
        arn = arn.replace('${TransitGatewayMulticastDomainId}', transitGatewayMulticastDomainId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type transit-gateway-route-table to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/tgw/how-transit-gateways-work.html
     *
     * @param transitGatewayRouteTableId - Identifier for the transitGatewayRouteTableId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onTransitGatewayRouteTable(transitGatewayRouteTableId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:transit-gateway-route-table/${TransitGatewayRouteTableId}';
        arn = arn.replace('${TransitGatewayRouteTableId}', transitGatewayRouteTableId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type transit-gateway to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/tgw/how-transit-gateways-work.html
     *
     * @param transitGatewayId - Identifier for the transitGatewayId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onTransitGateway(transitGatewayId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:transit-gateway/${TransitGatewayId}';
        arn = arn.replace('${TransitGatewayId}', transitGatewayId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type volume to the statement
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ebs-volumes.html
     *
     * @param volumeId - Identifier for the volumeId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifAvailabilityZone()
     * - .ifEncrypted()
     * - .ifParentSnapshot()
     * - .ifRegion()
     * - .ifResourceTag()
     * - .ifVolumeIops()
     * - .ifVolumeSize()
     * - .ifVolumeType()
     */
    onVolume(volumeId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:volume/${VolumeId}';
        arn = arn.replace('${VolumeId}', volumeId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type vpc to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/userguide/VPC_Subnets.html
     *
     * @param vpcId - Identifier for the vpcId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     * - .ifTenancy()
     */
    onVpc(vpcId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:vpc/${VpcId}';
        arn = arn.replace('${VpcId}', vpcId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type vpc-endpoint to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/userguide/endpoint-services-overview.html
     *
     * @param vpceId - Identifier for the vpceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     * - .ifVpceServiceName()
     * - .ifVpceServiceOwner()
     */
    onVpcEndpoint(vpceId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:vpc-endpoint/${VpceId}';
        arn = arn.replace('${VpceId}', vpceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type vpc-endpoint-service to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/userguide/endpoint-services-overview.html
     *
     * @param vpceServiceId - Identifier for the vpceServiceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     * - .ifVpceServicePrivateDnsName()
     */
    onVpcEndpointService(vpceServiceId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:vpc-endpoint-service/${VpceServiceId}';
        arn = arn.replace('${VpceServiceId}', vpceServiceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type vpc-flow-log to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/userguide/flow-logs.html
     *
     * @param vpcFlowLogId - Identifier for the vpcFlowLogId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifRegion()
     * - .ifResourceTag()
     */
    onVpcFlowLog(vpcFlowLogId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:vpc-flow-log/${VpcFlowLogId}';
        arn = arn.replace('${VpcFlowLogId}', vpcFlowLogId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type vpc-peering-connection to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/peering/what-is-vpc-peering.html
     *
     * @param vpcPeeringConnectionId - Identifier for the vpcPeeringConnectionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAccepterVpc()
     * - .ifRegion()
     * - .ifRequesterVpc()
     * - .ifResourceTag()
     */
    onVpcPeeringConnection(vpcPeeringConnectionId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:vpc-peering-connection/${VpcPeeringConnectionId}';
        arn = arn.replace('${VpcPeeringConnectionId}', vpcPeeringConnectionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type vpn-connection to the statement
     *
     * https://docs.aws.amazon.com/vpn/latest/s2svpn/VPC_VPN.html
     *
     * @param vpnConnectionId - Identifier for the vpnConnectionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifAuthenticationType()
     * - .ifDPDTimeoutSeconds()
     * - .ifGatewayType()
     * - .ifIKEVersions()
     * - .ifInsideTunnelCidr()
     * - .ifPhase1DHGroupNumbers()
     * - .ifPhase1EncryptionAlgorithms()
     * - .ifPhase1IntegrityAlgorithms()
     * - .ifPhase1LifetimeSeconds()
     * - .ifPhase2DHGroupNumbers()
     * - .ifPhase2EncryptionAlgorithms()
     * - .ifPhase2IntegrityAlgorithms()
     * - .ifPhase2LifetimeSeconds()
     * - .ifPresharedKeys()
     * - .ifRegion()
     * - .ifRekeyFuzzPercentage()
     * - .ifRekeyMarginTimeSeconds()
     * - .ifResourceTag()
     * - .ifRoutingType()
     */
    onVpnConnection(vpnConnectionId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:vpn-connection/${VpnConnectionId}';
        arn = arn.replace('${VpnConnectionId}', vpnConnectionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type vpn-gateway to the statement
     *
     * https://docs.aws.amazon.com/vpn/latest/s2svpn/VPC_VPN.html
     *
     * @param vpnGatewayId - Identifier for the vpnGatewayId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onVpnGateway(vpnGatewayId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:vpn-gateway/${VpnGatewayId}';
        arn = arn.replace('${VpnGatewayId}', vpnGatewayId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters access by the ARN of an accepter VPC in a VPC peering connection
     *
     * https://docs.aws.amazon.com/vpc/latest/peering/security-iam.html
     *
     * Applies to actions:
     * - .toAcceptVpcPeeringConnection()
     * - .toCreateVpcPeeringConnection()
     * - .toDeleteVpcPeeringConnection()
     * - .toRejectVpcPeeringConnection()
     *
     * Applies to resource types:
     * - vpc-peering-connection
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifAccepterVpc(value, operator) {
        return this.if(`ec2:AccepterVpc`, value, operator || 'ArnEquals');
    }
    /**
     * Filters access by whether the user wants to associate a public IP address with the instance
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .toCreateTags()
     * - .toRunInstances()
     *
     * Applies to resource types:
     * - network-interface
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifAssociatePublicIpAddress(value) {
        return this.if(`ec2:AssociatePublicIpAddress`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the authentication type for the VPN tunnel endpoints
     *
     * https://docs.aws.amazon.com/vpn/latest/s2svpn/vpn-authentication-access-control.html
     *
     * Applies to actions:
     * - .toCreateVpnConnection()
     * - .toModifyVpnTunnelOptions()
     *
     * Applies to resource types:
     * - vpn-connection
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAuthenticationType(value, operator) {
        return this.if(`ec2:AuthenticationType`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the AWS service that has permission to use a resource
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .toCreateNetworkInterfacePermission()
     *
     * Applies to resource types:
     * - network-interface
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAuthorizedService(value, operator) {
        return this.if(`ec2:AuthorizedService`, value, operator || 'StringLike');
    }
    /**
     * Filters access by an IAM principal that has permission to use a resource
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .toCreateNetworkInterfacePermission()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAuthorizedUser(value, operator) {
        return this.if(`ec2:AuthorizedUser`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the Auto Placement properties of a Dedicated Host
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to resource types:
     * - dedicated-host
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAutoPlacement(value, operator) {
        return this.if(`ec2:AutoPlacement`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the name of an Availability Zone in an AWS Region
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .toAssociateIamInstanceProfile()
     * - .toAttachClassicLinkVpc()
     * - .toAttachVolume()
     * - .toCreateFlowLogs()
     * - .toCreateNetworkInterfacePermission()
     * - .toCreateSnapshots()
     * - .toCreateTags()
     * - .toCreateVolume()
     * - .toDeleteVolume()
     * - .toDetachClassicLinkVpc()
     * - .toDetachVolume()
     * - .toDisableFastSnapshotRestores()
     * - .toDisassociateIamInstanceProfile()
     * - .toEnableFastSnapshotRestores()
     * - .toGetConsoleScreenshot()
     * - .toRebootInstances()
     * - .toReplaceIamInstanceProfileAssociation()
     * - .toRunInstances()
     * - .toSendDiagnosticInterrupt()
     * - .toStartInstances()
     * - .toStopInstances()
     * - .toTerminateInstances()
     *
     * Applies to resource types:
     * - dedicated-host
     * - instance
     * - network-interface
     * - reserved-instances
     * - subnet
     * - volume
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAvailabilityZone(value, operator) {
        return this.if(`ec2:AvailabilityZone`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the name of a resource-creating API action
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/supported-iam-actions-tagging.html
     *
     * Applies to actions:
     * - .toCreateTags()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifCreateAction(value, operator) {
        return this.if(`ec2:CreateAction`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the duration after which DPD timeout occurs on a VPN tunnel
     *
     * https://docs.aws.amazon.com/vpn/latest/s2svpn/vpn-authentication-access-control.html
     *
     * Applies to actions:
     * - .toCreateVpnConnection()
     * - .toModifyVpnTunnelOptions()
     *
     * Applies to resource types:
     * - vpn-connection
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifDPDTimeoutSeconds(value, operator) {
        return this.if(`ec2:DPDTimeoutSeconds`, value, operator || 'NumericEquals');
    }
    /**
     * Filters access by whether the instance is enabled for EBS optimization
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .toAssociateIamInstanceProfile()
     * - .toAttachClassicLinkVpc()
     * - .toAttachVolume()
     * - .toCreateSnapshots()
     * - .toCreateTags()
     * - .toDetachClassicLinkVpc()
     * - .toDetachVolume()
     * - .toDisassociateIamInstanceProfile()
     * - .toGetConsoleScreenshot()
     * - .toRebootInstances()
     * - .toReplaceIamInstanceProfileAssociation()
     * - .toRunInstances()
     * - .toSendDiagnosticInterrupt()
     * - .toStartInstances()
     * - .toStopInstances()
     * - .toTerminateInstances()
     *
     * Applies to resource types:
     * - instance
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifEbsOptimized(value) {
        return this.if(`ec2:EbsOptimized`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the type of Elastic Graphics accelerator
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .toRunInstances()
     *
     * Applies to resource types:
     * - elastic-gpu
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifElasticGpuType(value, operator) {
        return this.if(`ec2:ElasticGpuType`, value, operator || 'StringLike');
    }
    /**
     * Filters access by whether the EBS volume is encrypted
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .toAttachVolume()
     * - .toCreateSnapshot()
     * - .toCreateSnapshots()
     * - .toCreateTags()
     * - .toCreateVolume()
     * - .toDeleteVolume()
     * - .toDetachVolume()
     * - .toDisableFastSnapshotRestores()
     * - .toEnableFastSnapshotRestores()
     * - .toRunInstances()
     *
     * Applies to resource types:
     * - volume
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifEncrypted(value) {
        return this.if(`ec2:Encrypted`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the gateway type for a VPN endpoint on the AWS side of a VPN connection
     *
     * https://docs.aws.amazon.com/vpn/latest/s2svpn/vpn-authentication-access-control.html
     *
     * Applies to actions:
     * - .toCreateVpnConnection()
     * - .toModifyVpnConnection()
     *
     * Applies to resource types:
     * - vpn-connection
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifGatewayType(value, operator) {
        return this.if(`ec2:GatewayType`, value, operator || 'StringLike');
    }
    /**
     * Filters access by whether host recovery is enabled for a Dedicated Host
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to resource types:
     * - dedicated-host
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifHostRecovery(value, operator) {
        return this.if(`ec2:HostRecovery`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the internet key exchange (IKE) versions that are permitted for a VPN tunnel
     *
     * https://docs.aws.amazon.com/vpn/latest/s2svpn/vpn-authentication-access-control.html
     *
     * Applies to actions:
     * - .toCreateVpnConnection()
     * - .toModifyVpnTunnelOptions()
     *
     * Applies to resource types:
     * - vpn-connection
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifIKEVersions(value, operator) {
        return this.if(`ec2:IKEVersions`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the type of image (machine, aki, or ari)
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .toCreateTags()
     * - .toRunInstances()
     *
     * Applies to resource types:
     * - image
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifImageType(value, operator) {
        return this.if(`ec2:ImageType`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the range of inside IP addresses for a VPN tunnel
     *
     * https://docs.aws.amazon.com/vpn/latest/s2svpn/vpn-authentication-access-control.html
     *
     * Applies to actions:
     * - .toCreateVpnConnection()
     * - .toModifyVpnTunnelOptions()
     *
     * Applies to resource types:
     * - vpn-connection
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifInsideTunnelCidr(value, operator) {
        return this.if(`ec2:InsideTunnelCidr`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the market or purchasing option of an instance (on-demand or spot)
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifInstanceMarketType(value, operator) {
        return this.if(`ec2:InstanceMarketType`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the ARN of an instance profile
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .toAssociateIamInstanceProfile()
     * - .toAttachClassicLinkVpc()
     * - .toAttachVolume()
     * - .toCreateSnapshots()
     * - .toCreateTags()
     * - .toDetachClassicLinkVpc()
     * - .toDetachVolume()
     * - .toDisassociateIamInstanceProfile()
     * - .toGetConsoleScreenshot()
     * - .toRebootInstances()
     * - .toReplaceIamInstanceProfileAssociation()
     * - .toRunInstances()
     * - .toSendDiagnosticInterrupt()
     * - .toStartInstances()
     * - .toStopInstances()
     * - .toTerminateInstances()
     *
     * Applies to resource types:
     * - instance
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifInstanceProfile(value, operator) {
        return this.if(`ec2:InstanceProfile`, value, operator || 'ArnEquals');
    }
    /**
     * Filters access by the type of instance
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .toAssociateIamInstanceProfile()
     * - .toAttachClassicLinkVpc()
     * - .toAttachVolume()
     * - .toCreateSnapshots()
     * - .toCreateTags()
     * - .toDetachClassicLinkVpc()
     * - .toDetachVolume()
     * - .toDisassociateIamInstanceProfile()
     * - .toGetConsoleScreenshot()
     * - .toRebootInstances()
     * - .toReplaceIamInstanceProfileAssociation()
     * - .toRunInstances()
     * - .toSendDiagnosticInterrupt()
     * - .toStartInstances()
     * - .toStopInstances()
     * - .toTerminateInstances()
     *
     * Applies to resource types:
     * - dedicated-host
     * - instance
     * - reserved-instances
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifInstanceType(value, operator) {
        return this.if(`ec2:InstanceType`, value, operator || 'StringLike');
    }
    /**
     * Filters access by whether users are able to override resources that are specified in the launch template
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .toRunInstances()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifIsLaunchTemplateResource(value) {
        return this.if(`ec2:IsLaunchTemplateResource`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the ARN of a launch template
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .toRunInstances()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifLaunchTemplate(value, operator) {
        return this.if(`ec2:LaunchTemplate`, value, operator || 'ArnEquals');
    }
    /**
     * Filters access by whether the HTTP endpoint is enabled for the instance metadata service
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .toRunInstances()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifMetadataHttpEndpoint(value, operator) {
        return this.if(`ec2:MetadataHttpEndpoint`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the allowed number of hops when calling the instance metadata service
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .toRunInstances()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifMetadataHttpPutResponseHopLimit(value, operator) {
        return this.if(`ec2:MetadataHttpPutResponseHopLimit`, value, operator || 'NumericEquals');
    }
    /**
     * Filters access by whether tokens are required when calling the instance metadata service (optional or required)
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .toRunInstances()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifMetadataHttpTokens(value, operator) {
        return this.if(`ec2:MetadataHttpTokens`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the owner of the resource (amazon, aws-marketplace, or an AWS account ID)
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .toCreateTags()
     * - .toDeleteSnapshot()
     * - .toDisableFastSnapshotRestores()
     * - .toEnableFastSnapshotRestores()
     * - .toModifySnapshotAttribute()
     * - .toRunInstances()
     *
     * Applies to resource types:
     * - fpga-image
     * - image
     * - snapshot
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifOwner(value, operator) {
        return this.if(`ec2:Owner`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the ARN of the parent snapshot
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .toAttachVolume()
     * - .toCreateTags()
     * - .toCreateVolume()
     * - .toDeleteVolume()
     * - .toDetachVolume()
     * - .toRunInstances()
     *
     * Applies to resource types:
     * - volume
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifParentSnapshot(value, operator) {
        return this.if(`ec2:ParentSnapshot`, value, operator || 'ArnEquals');
    }
    /**
     * Filters access by the ARN of the parent volume from which the snapshot was created
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .toCreateSnapshot()
     * - .toCreateSnapshots()
     * - .toCreateTags()
     * - .toDeleteSnapshot()
     * - .toDisableFastSnapshotRestores()
     * - .toEnableFastSnapshotRestores()
     * - .toModifySnapshotAttribute()
     * - .toRunInstances()
     *
     * Applies to resource types:
     * - snapshot
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifParentVolume(value, operator) {
        return this.if(`ec2:ParentVolume`, value, operator || 'ArnEquals');
    }
    /**
     * Filters access by the type of permission for a resource (INSTANCE-ATTACH or EIP-ASSOCIATE)
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .toCreateNetworkInterfacePermission()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPermission(value, operator) {
        return this.if(`ec2:Permission`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the Diffie-Hellman group numbers that are permitted for a VPN tunnel for the phase 1 IKE negotiations
     *
     * https://docs.aws.amazon.com/vpn/latest/s2svpn/vpn-authentication-access-control.html
     *
     * Applies to actions:
     * - .toCreateVpnConnection()
     * - .toModifyVpnTunnelOptions()
     *
     * Applies to resource types:
     * - vpn-connection
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifPhase1DHGroupNumbers(value, operator) {
        return this.if(`ec2:Phase1DHGroupNumbers`, value, operator || 'NumericEquals');
    }
    /**
     * Filters access by the encryption algorithms that are permitted for a VPN tunnel for the phase 1 IKE negotiations
     *
     * https://docs.aws.amazon.com/vpn/latest/s2svpn/vpn-authentication-access-control.htmls
     *
     * Applies to actions:
     * - .toCreateVpnConnection()
     * - .toModifyVpnTunnelOptions()
     *
     * Applies to resource types:
     * - vpn-connection
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPhase1EncryptionAlgorithms(value, operator) {
        return this.if(`ec2:Phase1EncryptionAlgorithms`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the integrity algorithms that are permitted for a VPN tunnel for the phase 1 IKE negotiations
     *
     * https://docs.aws.amazon.com/vpn/latest/s2svpn/vpn-authentication-access-control.html
     *
     * Applies to actions:
     * - .toCreateVpnConnection()
     * - .toModifyVpnTunnelOptions()
     *
     * Applies to resource types:
     * - vpn-connection
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPhase1IntegrityAlgorithms(value, operator) {
        return this.if(`ec2:Phase1IntegrityAlgorithms`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the lifetime in seconds for phase 1 of the IKE negotiations for a VPN tunnel
     *
     * https://docs.aws.amazon.com/vpn/latest/s2svpn/vpn-authentication-access-control.html
     *
     * Applies to actions:
     * - .toCreateVpnConnection()
     * - .toModifyVpnTunnelOptions()
     *
     * Applies to resource types:
     * - vpn-connection
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifPhase1LifetimeSeconds(value, operator) {
        return this.if(`ec2:Phase1LifetimeSeconds`, value, operator || 'NumericEquals');
    }
    /**
     * Filters access by the Diffie-Hellman group numbers that are permitted for a VPN tunnel for the phase 2 IKE negotiations
     *
     * https://docs.aws.amazon.com/vpn/latest/s2svpn/vpn-authentication-access-control.html
     *
     * Applies to actions:
     * - .toCreateVpnConnection()
     * - .toModifyVpnTunnelOptions()
     *
     * Applies to resource types:
     * - vpn-connection
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifPhase2DHGroupNumbers(value, operator) {
        return this.if(`ec2:Phase2DHGroupNumbers`, value, operator || 'NumericEquals');
    }
    /**
     * Filters access by the encryption algorithms that are permitted for a VPN tunnel for the phase 2 IKE negotiations
     *
     * https://docs.aws.amazon.com/vpn/latest/s2svpn/vpn-authentication-access-control.html
     *
     * Applies to actions:
     * - .toCreateVpnConnection()
     * - .toModifyVpnTunnelOptions()
     *
     * Applies to resource types:
     * - vpn-connection
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPhase2EncryptionAlgorithms(value, operator) {
        return this.if(`ec2:Phase2EncryptionAlgorithms`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the integrity algorithms that are permitted for a VPN tunnel for the phase 2 IKE negotiations
     *
     * https://docs.aws.amazon.com/vpn/latest/s2svpn/vpn-authentication-access-control.html
     *
     * Applies to actions:
     * - .toCreateVpnConnection()
     * - .toModifyVpnTunnelOptions()
     *
     * Applies to resource types:
     * - vpn-connection
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPhase2IntegrityAlgorithms(value, operator) {
        return this.if(`ec2:Phase2IntegrityAlgorithms`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the lifetime in seconds for phase 2 of the IKE negotiations for a VPN tunnel
     *
     * https://docs.aws.amazon.com/vpn/latest/s2svpn/vpn-authentication-access-control.html
     *
     * Applies to actions:
     * - .toCreateVpnConnection()
     * - .toModifyVpnTunnelOptions()
     *
     * Applies to resource types:
     * - vpn-connection
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifPhase2LifetimeSeconds(value, operator) {
        return this.if(`ec2:Phase2LifetimeSeconds`, value, operator || 'NumericEquals');
    }
    /**
     * Filters access by the ARN of the placement group
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .toAssociateIamInstanceProfile()
     * - .toAttachClassicLinkVpc()
     * - .toAttachVolume()
     * - .toCreateSnapshots()
     * - .toCreateTags()
     * - .toDetachClassicLinkVpc()
     * - .toDetachVolume()
     * - .toDisassociateIamInstanceProfile()
     * - .toGetConsoleScreenshot()
     * - .toRebootInstances()
     * - .toReplaceIamInstanceProfileAssociation()
     * - .toRunInstances()
     * - .toSendDiagnosticInterrupt()
     * - .toStartInstances()
     * - .toStopInstances()
     * - .toTerminateInstances()
     *
     * Applies to resource types:
     * - instance
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifPlacementGroup(value, operator) {
        return this.if(`ec2:PlacementGroup`, value, operator || 'ArnEquals');
    }
    /**
     * Filters access by the instance placement strategy used by the placement group (cluster, spread, or partition)
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .toRunInstances()
     *
     * Applies to resource types:
     * - placement-group
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPlacementGroupStrategy(value, operator) {
        return this.if(`ec2:PlacementGroupStrategy`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the pre-shared key (PSK) used to establish the initial IKE security association between a virtual private gateway and a customer gateway
     *
     * https://docs.aws.amazon.com/vpn/latest/s2svpn/vpn-authentication-access-control.html
     *
     * Applies to actions:
     * - .toCreateVpnConnection()
     * - .toModifyVpnTunnelOptions()
     *
     * Applies to resource types:
     * - vpn-connection
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPresharedKeys(value, operator) {
        return this.if(`ec2:PresharedKeys`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the product code that is associated with the AMI
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifProductCode(value, operator) {
        return this.if(`ec2:ProductCode`, value, operator || 'StringLike');
    }
    /**
     * Filters access by whether the image has public launch permissions
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .toCreateTags()
     * - .toRunInstances()
     *
     * Applies to resource types:
     * - fpga-image
     * - image
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifPublic(value) {
        return this.if(`ec2:Public`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the number of Dedicated Hosts in a request
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to resource types:
     * - dedicated-host
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifQuantity(value, operator) {
        return this.if(`ec2:Quantity`, value, operator || 'NumericEquals');
    }
    /**
     * Filters access by the name of the AWS Region
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .toAcceptTransitGatewayPeeringAttachment()
     * - .toAcceptTransitGatewayVpcAttachment()
     * - .toAcceptVpcEndpointConnections()
     * - .toAcceptVpcPeeringConnection()
     * - .toApplySecurityGroupsToClientVpnTargetNetwork()
     * - .toAssociateClientVpnTargetNetwork()
     * - .toAssociateIamInstanceProfile()
     * - .toAssociateTransitGatewayMulticastDomain()
     * - .toAssociateTransitGatewayRouteTable()
     * - .toAttachClassicLinkVpc()
     * - .toAttachVolume()
     * - .toAuthorizeClientVpnIngress()
     * - .toAuthorizeSecurityGroupEgress()
     * - .toAuthorizeSecurityGroupIngress()
     * - .toCancelCapacityReservation()
     * - .toCopySnapshot()
     * - .toCreateCapacityReservation()
     * - .toCreateCarrierGateway()
     * - .toCreateClientVpnEndpoint()
     * - .toCreateClientVpnRoute()
     * - .toCreateDhcpOptions()
     * - .toCreateEgressOnlyInternetGateway()
     * - .toCreateFleet()
     * - .toCreateFlowLogs()
     * - .toCreateFpgaImage()
     * - .toCreateInstanceExportTask()
     * - .toCreateInternetGateway()
     * - .toCreateKeyPair()
     * - .toCreateLaunchTemplate()
     * - .toCreateLaunchTemplateVersion()
     * - .toCreateLocalGatewayRoute()
     * - .toCreateLocalGatewayRouteTableVpcAssociation()
     * - .toCreateManagedPrefixList()
     * - .toCreateNatGateway()
     * - .toCreateNetworkAcl()
     * - .toCreateNetworkInterface()
     * - .toCreateNetworkInterfacePermission()
     * - .toCreatePlacementGroup()
     * - .toCreateRoute()
     * - .toCreateSecurityGroup()
     * - .toCreateSnapshot()
     * - .toCreateSnapshots()
     * - .toCreateSubnet()
     * - .toCreateTags()
     * - .toCreateTrafficMirrorFilter()
     * - .toCreateTrafficMirrorFilterRule()
     * - .toCreateTrafficMirrorSession()
     * - .toCreateTrafficMirrorTarget()
     * - .toCreateTransitGateway()
     * - .toCreateTransitGatewayMulticastDomain()
     * - .toCreateTransitGatewayPeeringAttachment()
     * - .toCreateTransitGatewayPrefixListReference()
     * - .toCreateTransitGatewayRoute()
     * - .toCreateTransitGatewayRouteTable()
     * - .toCreateTransitGatewayVpcAttachment()
     * - .toCreateVolume()
     * - .toCreateVpc()
     * - .toCreateVpcEndpoint()
     * - .toCreateVpcEndpointServiceConfiguration()
     * - .toCreateVpcPeeringConnection()
     * - .toCreateVpnConnection()
     * - .toDeleteCarrierGateway()
     * - .toDeleteClientVpnEndpoint()
     * - .toDeleteClientVpnRoute()
     * - .toDeleteCustomerGateway()
     * - .toDeleteDhcpOptions()
     * - .toDeleteFlowLogs()
     * - .toDeleteInternetGateway()
     * - .toDeleteLaunchTemplate()
     * - .toDeleteLaunchTemplateVersions()
     * - .toDeleteLocalGatewayRoute()
     * - .toDeleteLocalGatewayRouteTableVpcAssociation()
     * - .toDeleteManagedPrefixList()
     * - .toDeleteNetworkAcl()
     * - .toDeleteNetworkAclEntry()
     * - .toDeleteRoute()
     * - .toDeleteRouteTable()
     * - .toDeleteSecurityGroup()
     * - .toDeleteSnapshot()
     * - .toDeleteTags()
     * - .toDeleteTrafficMirrorFilter()
     * - .toDeleteTrafficMirrorFilterRule()
     * - .toDeleteTrafficMirrorSession()
     * - .toDeleteTrafficMirrorTarget()
     * - .toDeleteTransitGateway()
     * - .toDeleteTransitGatewayMulticastDomain()
     * - .toDeleteTransitGatewayPeeringAttachment()
     * - .toDeleteTransitGatewayPrefixListReference()
     * - .toDeleteTransitGatewayRoute()
     * - .toDeleteTransitGatewayRouteTable()
     * - .toDeleteTransitGatewayVpcAttachment()
     * - .toDeleteVolume()
     * - .toDeleteVpcEndpointServiceConfigurations()
     * - .toDeleteVpcEndpoints()
     * - .toDeleteVpcPeeringConnection()
     * - .toDeregisterTransitGatewayMulticastGroupMembers()
     * - .toDeregisterTransitGatewayMulticastGroupSources()
     * - .toDetachClassicLinkVpc()
     * - .toDetachVolume()
     * - .toDisableFastSnapshotRestores()
     * - .toDisableTransitGatewayRouteTablePropagation()
     * - .toDisableVpcClassicLink()
     * - .toDisassociateClientVpnTargetNetwork()
     * - .toDisassociateIamInstanceProfile()
     * - .toDisassociateTransitGatewayMulticastDomain()
     * - .toDisassociateTransitGatewayRouteTable()
     * - .toEnableFastSnapshotRestores()
     * - .toEnableTransitGatewayRouteTablePropagation()
     * - .toEnableVpcClassicLink()
     * - .toGetConsoleScreenshot()
     * - .toImportClientVpnClientCertificateRevocationList()
     * - .toModifyCapacityReservation()
     * - .toModifyClientVpnEndpoint()
     * - .toModifyInstanceEventStartTime()
     * - .toModifyLaunchTemplate()
     * - .toModifyManagedPrefixList()
     * - .toModifySnapshotAttribute()
     * - .toModifyTrafficMirrorFilterNetworkServices()
     * - .toModifyTrafficMirrorFilterRule()
     * - .toModifyTrafficMirrorSession()
     * - .toModifyTransitGateway()
     * - .toModifyTransitGatewayPrefixListReference()
     * - .toModifyTransitGatewayVpcAttachment()
     * - .toModifyVpcEndpoint()
     * - .toModifyVpcEndpointServiceConfiguration()
     * - .toModifyVpcEndpointServicePermissions()
     * - .toModifyVpnConnection()
     * - .toModifyVpnTunnelOptions()
     * - .toRebootInstances()
     * - .toRegisterTransitGatewayMulticastGroupMembers()
     * - .toRegisterTransitGatewayMulticastGroupSources()
     * - .toRejectTransitGatewayPeeringAttachment()
     * - .toRejectTransitGatewayVpcAttachment()
     * - .toRejectVpcEndpointConnections()
     * - .toRejectVpcPeeringConnection()
     * - .toReplaceIamInstanceProfileAssociation()
     * - .toReplaceRoute()
     * - .toReplaceTransitGatewayRoute()
     * - .toRestoreManagedPrefixListVersion()
     * - .toRevokeClientVpnIngress()
     * - .toRevokeSecurityGroupEgress()
     * - .toRevokeSecurityGroupIngress()
     * - .toRunInstances()
     * - .toSendDiagnosticInterrupt()
     * - .toStartInstances()
     * - .toStartVpcEndpointServicePrivateDnsVerification()
     * - .toStopInstances()
     * - .toTerminateClientVpnConnections()
     * - .toTerminateInstances()
     * - .toUpdateSecurityGroupRuleDescriptionsEgress()
     * - .toUpdateSecurityGroupRuleDescriptionsIngress()
     *
     * Applies to resource types:
     * - capacity-reservation
     * - carrier-gateway
     * - client-vpn-endpoint
     * - customer-gateway
     * - dhcp-options
     * - egress-only-internet-gateway
     * - export-instance-task
     * - fleet
     * - fpga-image
     * - image
     * - instance
     * - internet-gateway
     * - key-pair
     * - launch-template
     * - local-gateway
     * - local-gateway-route-table
     * - local-gateway-route-table-virtual-interface-group-association
     * - local-gateway-route-table-vpc-association
     * - local-gateway-virtual-interface
     * - local-gateway-virtual-interface-group
     * - natgateway
     * - network-acl
     * - network-interface
     * - placement-group
     * - prefix-list
     * - reserved-instances
     * - route-table
     * - security-group
     * - snapshot
     * - spot-instance-request
     * - subnet
     * - traffic-mirror-session
     * - traffic-mirror-target
     * - traffic-mirror-filter
     * - traffic-mirror-filter-rule
     * - transit-gateway-attachment
     * - transit-gateway-multicast-domain
     * - transit-gateway-route-table
     * - transit-gateway
     * - volume
     * - vpc
     * - vpc-endpoint
     * - vpc-endpoint-service
     * - vpc-flow-log
     * - vpc-peering-connection
     * - vpn-connection
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRegion(value, operator) {
        return this.if(`ec2:Region`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the percentage of increase of the rekey window (determined by the rekey margin time) within which the rekey time is randomly selected for a VPN tunnel
     *
     * https://docs.aws.amazon.com/vpn/latest/s2svpn/vpn-authentication-access-control.html
     *
     * Applies to actions:
     * - .toCreateVpnConnection()
     * - .toModifyVpnTunnelOptions()
     *
     * Applies to resource types:
     * - vpn-connection
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifRekeyFuzzPercentage(value, operator) {
        return this.if(`ec2:RekeyFuzzPercentage`, value, operator || 'NumericEquals');
    }
    /**
     * Filters access by the margin time before the phase 2 lifetime expires for a VPN tunnel
     *
     * https://docs.aws.amazon.com/vpn/latest/s2svpn/vpn-authentication-access-control.html
     *
     * Applies to actions:
     * - .toCreateVpnConnection()
     * - .toModifyVpnTunnelOptions()
     *
     * Applies to resource types:
     * - vpn-connection
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifRekeyMarginTimeSeconds(value, operator) {
        return this.if(`ec2:RekeyMarginTimeSeconds`, value, operator || 'NumericEquals');
    }
    /**
     * Filters access by the ARN of a requester VPC in a VPC peering connection
     *
     * https://docs.aws.amazon.com/vpc/latest/peering/security-iam.html
     *
     * Applies to actions:
     * - .toAcceptVpcPeeringConnection()
     * - .toCreateVpcPeeringConnection()
     * - .toDeleteVpcPeeringConnection()
     * - .toRejectVpcPeeringConnection()
     *
     * Applies to resource types:
     * - vpc-peering-connection
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifRequesterVpc(value, operator) {
        return this.if(`ec2:RequesterVpc`, value, operator || 'ArnEquals');
    }
    /**
     * Filters access by the payment option of the Reserved Instance offering (No Upfront, Partial Upfront, or All Upfront)
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-reserved-instances.html#ri-payment-options
     *
     * Applies to actions:
     * - .toCreateTags()
     *
     * Applies to resource types:
     * - reserved-instances
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifReservedInstancesOfferingType(value, operator) {
        return this.if(`ec2:ReservedInstancesOfferingType`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the preface string for a tag key and value pair that are attached to a resource
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/control-access-with-tags.html
     *
     * Applies to actions:
     * - .toRunInstances()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceTagExists(value, operator) {
        return this.if(`ec2:ResourceTag/`, value, operator || 'StringLike');
    }
    /**
     * Filters access by a tag key and value pair of a resource
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/control-access-with-tags.html
     *
     * Applies to actions:
     * - .toAcceptTransitGatewayPeeringAttachment()
     * - .toAcceptTransitGatewayVpcAttachment()
     * - .toAcceptVpcEndpointConnections()
     * - .toAcceptVpcPeeringConnection()
     * - .toApplySecurityGroupsToClientVpnTargetNetwork()
     * - .toAssociateClientVpnTargetNetwork()
     * - .toAssociateIamInstanceProfile()
     * - .toAssociateTransitGatewayMulticastDomain()
     * - .toAssociateTransitGatewayRouteTable()
     * - .toAttachClassicLinkVpc()
     * - .toAttachVolume()
     * - .toAuthorizeClientVpnIngress()
     * - .toAuthorizeSecurityGroupEgress()
     * - .toAuthorizeSecurityGroupIngress()
     * - .toCancelCapacityReservation()
     * - .toCreateClientVpnRoute()
     * - .toCreateFlowLogs()
     * - .toCreateLaunchTemplateVersion()
     * - .toCreateLocalGatewayRoute()
     * - .toCreateLocalGatewayRouteTableVpcAssociation()
     * - .toCreateNetworkInterfacePermission()
     * - .toCreateRoute()
     * - .toCreateSnapshot()
     * - .toCreateSnapshots()
     * - .toCreateTags()
     * - .toCreateTrafficMirrorFilterRule()
     * - .toCreateTrafficMirrorSession()
     * - .toCreateTrafficMirrorTarget()
     * - .toCreateTransitGatewayMulticastDomain()
     * - .toCreateTransitGatewayPeeringAttachment()
     * - .toCreateTransitGatewayPrefixListReference()
     * - .toCreateTransitGatewayRoute()
     * - .toCreateTransitGatewayRouteTable()
     * - .toCreateTransitGatewayVpcAttachment()
     * - .toCreateVpcEndpoint()
     * - .toCreateVpcPeeringConnection()
     * - .toDeleteCarrierGateway()
     * - .toDeleteClientVpnEndpoint()
     * - .toDeleteClientVpnRoute()
     * - .toDeleteCustomerGateway()
     * - .toDeleteDhcpOptions()
     * - .toDeleteFlowLogs()
     * - .toDeleteInternetGateway()
     * - .toDeleteLaunchTemplate()
     * - .toDeleteLaunchTemplateVersions()
     * - .toDeleteLocalGatewayRoute()
     * - .toDeleteLocalGatewayRouteTableVpcAssociation()
     * - .toDeleteManagedPrefixList()
     * - .toDeleteNetworkAcl()
     * - .toDeleteNetworkAclEntry()
     * - .toDeleteRoute()
     * - .toDeleteRouteTable()
     * - .toDeleteSecurityGroup()
     * - .toDeleteSnapshot()
     * - .toDeleteTags()
     * - .toDeleteTrafficMirrorFilter()
     * - .toDeleteTrafficMirrorFilterRule()
     * - .toDeleteTrafficMirrorSession()
     * - .toDeleteTrafficMirrorTarget()
     * - .toDeleteTransitGateway()
     * - .toDeleteTransitGatewayMulticastDomain()
     * - .toDeleteTransitGatewayPeeringAttachment()
     * - .toDeleteTransitGatewayPrefixListReference()
     * - .toDeleteTransitGatewayRoute()
     * - .toDeleteTransitGatewayRouteTable()
     * - .toDeleteTransitGatewayVpcAttachment()
     * - .toDeleteVolume()
     * - .toDeleteVpcEndpointServiceConfigurations()
     * - .toDeleteVpcEndpoints()
     * - .toDeleteVpcPeeringConnection()
     * - .toDeregisterTransitGatewayMulticastGroupMembers()
     * - .toDeregisterTransitGatewayMulticastGroupSources()
     * - .toDetachClassicLinkVpc()
     * - .toDetachVolume()
     * - .toDisableFastSnapshotRestores()
     * - .toDisableTransitGatewayRouteTablePropagation()
     * - .toDisableVpcClassicLink()
     * - .toDisassociateClientVpnTargetNetwork()
     * - .toDisassociateIamInstanceProfile()
     * - .toDisassociateTransitGatewayMulticastDomain()
     * - .toDisassociateTransitGatewayRouteTable()
     * - .toEnableFastSnapshotRestores()
     * - .toEnableTransitGatewayRouteTablePropagation()
     * - .toEnableVpcClassicLink()
     * - .toGetConsoleScreenshot()
     * - .toImportClientVpnClientCertificateRevocationList()
     * - .toModifyCapacityReservation()
     * - .toModifyClientVpnEndpoint()
     * - .toModifyLaunchTemplate()
     * - .toModifyManagedPrefixList()
     * - .toModifySnapshotAttribute()
     * - .toModifyTrafficMirrorFilterNetworkServices()
     * - .toModifyTrafficMirrorFilterRule()
     * - .toModifyTrafficMirrorSession()
     * - .toModifyTransitGateway()
     * - .toModifyTransitGatewayPrefixListReference()
     * - .toModifyTransitGatewayVpcAttachment()
     * - .toModifyVpcEndpoint()
     * - .toModifyVpcEndpointServiceConfiguration()
     * - .toModifyVpcEndpointServicePermissions()
     * - .toModifyVpnConnection()
     * - .toModifyVpnTunnelOptions()
     * - .toRebootInstances()
     * - .toRegisterTransitGatewayMulticastGroupMembers()
     * - .toRegisterTransitGatewayMulticastGroupSources()
     * - .toRejectTransitGatewayPeeringAttachment()
     * - .toRejectTransitGatewayVpcAttachment()
     * - .toRejectVpcEndpointConnections()
     * - .toRejectVpcPeeringConnection()
     * - .toReplaceIamInstanceProfileAssociation()
     * - .toReplaceRoute()
     * - .toReplaceTransitGatewayRoute()
     * - .toRestoreManagedPrefixListVersion()
     * - .toRevokeClientVpnIngress()
     * - .toRevokeSecurityGroupEgress()
     * - .toRevokeSecurityGroupIngress()
     * - .toRunInstances()
     * - .toSendDiagnosticInterrupt()
     * - .toStartInstances()
     * - .toStartVpcEndpointServicePrivateDnsVerification()
     * - .toStopInstances()
     * - .toTerminateClientVpnConnections()
     * - .toTerminateInstances()
     * - .toUpdateSecurityGroupRuleDescriptionsEgress()
     * - .toUpdateSecurityGroupRuleDescriptionsIngress()
     *
     * Applies to resource types:
     * - capacity-reservation
     * - carrier-gateway
     * - client-vpn-endpoint
     * - customer-gateway
     * - dedicated-host
     * - dhcp-options
     * - egress-only-internet-gateway
     * - export-instance-task
     * - fleet
     * - fpga-image
     * - image
     * - instance
     * - internet-gateway
     * - key-pair
     * - launch-template
     * - local-gateway
     * - local-gateway-route-table
     * - local-gateway-route-table-virtual-interface-group-association
     * - local-gateway-route-table-vpc-association
     * - local-gateway-virtual-interface
     * - local-gateway-virtual-interface-group
     * - natgateway
     * - network-acl
     * - network-interface
     * - placement-group
     * - prefix-list
     * - reserved-instances
     * - route-table
     * - security-group
     * - snapshot
     * - spot-instance-request
     * - subnet
     * - traffic-mirror-session
     * - traffic-mirror-target
     * - traffic-mirror-filter
     * - transit-gateway-attachment
     * - transit-gateway-multicast-domain
     * - transit-gateway-route-table
     * - transit-gateway
     * - volume
     * - vpc
     * - vpc-endpoint
     * - vpc-endpoint-service
     * - vpc-flow-log
     * - vpc-peering-connection
     * - vpn-connection
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceTag(tagKey, value, operator) {
        return this.if(`ec2:ResourceTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the version of the instance metadata service for retrieving IAM role credentials for EC2
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifRoleDelivery(value, operator) {
        return this.if(`ec2:RoleDelivery`, value, operator || 'NumericEquals');
    }
    /**
     * Filters access by the root device type of the instance (ebs or instance-store)
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .toAssociateIamInstanceProfile()
     * - .toAttachClassicLinkVpc()
     * - .toAttachVolume()
     * - .toCreateSnapshots()
     * - .toCreateTags()
     * - .toDetachClassicLinkVpc()
     * - .toDetachVolume()
     * - .toDisassociateIamInstanceProfile()
     * - .toGetConsoleScreenshot()
     * - .toRebootInstances()
     * - .toReplaceIamInstanceProfileAssociation()
     * - .toRunInstances()
     * - .toSendDiagnosticInterrupt()
     * - .toStartInstances()
     * - .toStopInstances()
     * - .toTerminateInstances()
     *
     * Applies to resource types:
     * - image
     * - instance
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRootDeviceType(value, operator) {
        return this.if(`ec2:RootDeviceType`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the routing type for the VPN connection
     *
     * https://docs.aws.amazon.com/vpn/latest/s2svpn/vpn-authentication-access-control.html
     *
     * Applies to actions:
     * - .toCreateVpnConnection()
     * - .toModifyVpnTunnelOptions()
     *
     * Applies to resource types:
     * - vpn-connection
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRoutingType(value, operator) {
        return this.if(`ec2:RoutingType`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the initiation time of a snapshot
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .toCreateTags()
     * - .toDeleteSnapshot()
     * - .toDisableFastSnapshotRestores()
     * - .toEnableFastSnapshotRestores()
     * - .toModifySnapshotAttribute()
     * - .toRunInstances()
     *
     * Applies to resource types:
     * - snapshot
     *
     * @param value The value(s) to check
     * @param operator Works with [date operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Date). **Default:** `DateEquals`
     */
    ifSnapshotTime(value, operator) {
        if (typeof value.getMonth === "function") {
            value = value.toISOString();
        }
        else if (Array.isArray(value)) {
            value = value.map((item) => {
                if (typeof item.getMonth === "function") {
                    item = item.toISOString();
                }
                return item;
            });
        }
        return this.if(`ec2:SnapshotTime`, value, operator || 'DateEquals');
    }
    /**
     * Filters access by the ARN of the instance from which the request originated
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifSourceInstanceARN(value, operator) {
        return this.if(`ec2:SourceInstanceARN`, value, operator || 'ArnEquals');
    }
    /**
     * Filters access by the ARN of the subnet
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .toCreateFlowLogs()
     * - .toCreateNetworkInterfacePermission()
     * - .toCreateTags()
     * - .toRunInstances()
     *
     * Applies to resource types:
     * - network-interface
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifSubnet(value, operator) {
        return this.if(`ec2:Subnet`, value, operator || 'ArnEquals');
    }
    /**
     * Filters access by the tenancy of the VPC or instance (default, dedicated, or host)
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .toAcceptVpcPeeringConnection()
     * - .toAssociateIamInstanceProfile()
     * - .toAttachClassicLinkVpc()
     * - .toAttachVolume()
     * - .toCreateFlowLogs()
     * - .toCreateLocalGatewayRouteTableVpcAssociation()
     * - .toCreateSnapshots()
     * - .toCreateTags()
     * - .toCreateVpcPeeringConnection()
     * - .toDetachClassicLinkVpc()
     * - .toDetachVolume()
     * - .toDisableVpcClassicLink()
     * - .toDisassociateIamInstanceProfile()
     * - .toEnableVpcClassicLink()
     * - .toGetConsoleScreenshot()
     * - .toRebootInstances()
     * - .toReplaceIamInstanceProfileAssociation()
     * - .toRunInstances()
     * - .toSendDiagnosticInterrupt()
     * - .toStartInstances()
     * - .toStopInstances()
     * - .toTerminateInstances()
     *
     * Applies to resource types:
     * - instance
     * - reserved-instances
     * - vpc
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTenancy(value, operator) {
        return this.if(`ec2:Tenancy`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the the number of input/output operations per second (IOPS) provisioned for the volume
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .toAttachVolume()
     * - .toCreateSnapshot()
     * - .toCreateSnapshots()
     * - .toCreateTags()
     * - .toCreateVolume()
     * - .toDeleteVolume()
     * - .toDetachVolume()
     * - .toRunInstances()
     *
     * Applies to resource types:
     * - volume
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifVolumeIops(value, operator) {
        return this.if(`ec2:VolumeIops`, value, operator || 'NumericEquals');
    }
    /**
     * Filters access by the size of the volume, in GiB.
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .toAttachVolume()
     * - .toCreateSnapshot()
     * - .toCreateSnapshots()
     * - .toCreateTags()
     * - .toCreateVolume()
     * - .toDeleteSnapshot()
     * - .toDeleteVolume()
     * - .toDetachVolume()
     * - .toDisableFastSnapshotRestores()
     * - .toEnableFastSnapshotRestores()
     * - .toModifySnapshotAttribute()
     * - .toRunInstances()
     *
     * Applies to resource types:
     * - snapshot
     * - volume
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifVolumeSize(value, operator) {
        return this.if(`ec2:VolumeSize`, value, operator || 'NumericEquals');
    }
    /**
     * Filters access by the type of volume (gp2, io1, io2, st1, sc1, or standard)
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .toAttachVolume()
     * - .toCreateSnapshot()
     * - .toCreateSnapshots()
     * - .toCreateTags()
     * - .toCreateVolume()
     * - .toDeleteVolume()
     * - .toDetachVolume()
     * - .toRunInstances()
     *
     * Applies to resource types:
     * - volume
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifVolumeType(value, operator) {
        return this.if(`ec2:VolumeType`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the ARN of the VPC
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .toAttachClassicLinkVpc()
     * - .toAuthorizeSecurityGroupEgress()
     * - .toAuthorizeSecurityGroupIngress()
     * - .toCreateFlowLogs()
     * - .toCreateNetworkInterfacePermission()
     * - .toCreateRoute()
     * - .toCreateTags()
     * - .toDeleteNetworkAcl()
     * - .toDeleteNetworkAclEntry()
     * - .toDeleteRoute()
     * - .toDeleteRouteTable()
     * - .toDeleteSecurityGroup()
     * - .toReplaceRoute()
     * - .toRevokeSecurityGroupEgress()
     * - .toRevokeSecurityGroupIngress()
     * - .toRunInstances()
     * - .toUpdateSecurityGroupRuleDescriptionsEgress()
     * - .toUpdateSecurityGroupRuleDescriptionsIngress()
     *
     * Applies to resource types:
     * - network-acl
     * - network-interface
     * - route-table
     * - security-group
     * - subnet
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifVpc(value, operator) {
        return this.if(`ec2:Vpc`, value, operator || 'ArnEquals');
    }
    /**
     * Filters access by the name of the VPC endpoint service
     *
     * Applies to actions:
     * - .toCreateVpcEndpoint()
     *
     * Applies to resource types:
     * - vpc-endpoint
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifVpceServiceName(value, operator) {
        return this.if(`ec2:VpceServiceName`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the service owner of the VPC endpoint service (amazon, aws-marketplace, or an AWS account ID)
     *
     * Applies to actions:
     * - .toCreateVpcEndpoint()
     *
     * Applies to resource types:
     * - vpc-endpoint
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifVpceServiceOwner(value, operator) {
        return this.if(`ec2:VpceServiceOwner`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the private DNS name of the VPC endpoint service
     *
     * Applies to actions:
     * - .toCreateVpcEndpointServiceConfiguration()
     * - .toModifyVpcEndpointServiceConfiguration()
     *
     * Applies to resource types:
     * - vpc-endpoint-service
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifVpceServicePrivateDnsName(value, operator) {
        return this.if(`ec2:VpceServicePrivateDnsName`, value, operator || 'StringLike');
    }
}
exports.Ec2 = Ec2;
//# sourceMappingURL=data:application/json;base64,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