"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Autoscaling = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [autoscaling](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonec2autoscaling.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Autoscaling extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [autoscaling](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonec2autoscaling.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'autoscaling';
        this.actionList = {
            "AttachInstances": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_AttachInstances.html",
                "description": "Attaches one or more EC2 instances to the specified Auto Scaling group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "autoScalingGroup": {
                        "required": true,
                        "conditions": [
                            "autoscaling:ResourceTag/${TagKey}",
                            "aws:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "AttachLoadBalancerTargetGroups": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_AttachLoadBalancerTargetGroups.html",
                "description": "Attaches one or more target groups to the specified Auto Scaling group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "autoScalingGroup": {
                        "required": true,
                        "conditions": [
                            "autoscaling:ResourceTag/${TagKey}",
                            "aws:ResourceTag/${TagKey}"
                        ]
                    }
                },
                "conditions": [
                    "autoscaling:TargetGroupARNs"
                ]
            },
            "AttachLoadBalancers": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_AttachLoadBalancers.html",
                "description": "Attaches one or more load balancers to the specified Auto Scaling group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "autoScalingGroup": {
                        "required": true,
                        "conditions": [
                            "autoscaling:ResourceTag/${TagKey}",
                            "aws:ResourceTag/${TagKey}"
                        ]
                    }
                },
                "conditions": [
                    "autoscaling:LoadBalancerNames"
                ]
            },
            "BatchDeleteScheduledAction": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_BatchDeleteScheduledAction.html",
                "description": "Deletes the specified scheduled actions.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "autoScalingGroup": {
                        "required": true,
                        "conditions": [
                            "autoscaling:ResourceTag/${TagKey}",
                            "aws:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "BatchPutScheduledUpdateGroupAction": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_BatchPutScheduledUpdateGroupAction.html",
                "description": "Creates or updates multiple scheduled scaling actions for an Auto Scaling group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "autoScalingGroup": {
                        "required": true,
                        "conditions": [
                            "autoscaling:ResourceTag/${TagKey}",
                            "aws:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "CancelInstanceRefresh": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_CancelInstanceRefresh.html",
                "description": "Grants permission to cancel an instance refresh operation in progress",
                "accessLevel": "Write",
                "resourceTypes": {
                    "autoScalingGroup": {
                        "required": true,
                        "conditions": [
                            "autoscaling:ResourceTag/${TagKey}",
                            "aws:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "CompleteLifecycleAction": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_CompleteLifecycleAction.html",
                "description": "Completes the lifecycle action for the specified token or instance with the specified result.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "autoScalingGroup": {
                        "required": true,
                        "conditions": [
                            "autoscaling:ResourceTag/${TagKey}",
                            "aws:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "CreateAutoScalingGroup": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_CreateAutoScalingGroup.html",
                "description": "Creates an Auto Scaling group with the specified name and attributes.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "autoScalingGroup": {
                        "required": true,
                        "conditions": [
                            "autoscaling:ResourceTag/${TagKey}",
                            "aws:ResourceTag/${TagKey}"
                        ]
                    }
                },
                "conditions": [
                    "autoscaling:InstanceTypes",
                    "autoscaling:LaunchConfigurationName",
                    "autoscaling:LaunchTemplateVersionSpecified",
                    "autoscaling:LoadBalancerNames",
                    "autoscaling:MaxSize",
                    "autoscaling:MinSize",
                    "autoscaling:TargetGroupARNs",
                    "autoscaling:VPCZoneIdentifiers",
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateLaunchConfiguration": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_CreateLaunchConfiguration.html",
                "description": "Creates a launch configuration.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "launchConfiguration": {
                        "required": true
                    }
                },
                "conditions": [
                    "autoscaling:ImageId",
                    "autoscaling:InstanceType",
                    "autoscaling:SpotPrice",
                    "autoscaling:MetadataHttpTokens",
                    "autoscaling:MetadataHttpPutResponseHopLimit",
                    "autoscaling:MetadataHttpEndpoint"
                ]
            },
            "CreateOrUpdateTags": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_CreateOrUpdateTags.html",
                "description": "Creates or updates tags for the specified Auto Scaling group.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "autoScalingGroup": {
                        "required": true,
                        "conditions": [
                            "autoscaling:ResourceTag/${TagKey}",
                            "aws:ResourceTag/${TagKey}"
                        ]
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "DeleteAutoScalingGroup": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DeleteAutoScalingGroup.html",
                "description": "Deletes the specified Auto Scaling group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "autoScalingGroup": {
                        "required": true,
                        "conditions": [
                            "autoscaling:ResourceTag/${TagKey}",
                            "aws:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteLaunchConfiguration": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DeleteLaunchConfiguration.html",
                "description": "Deletes the specified launch configuration.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "launchConfiguration": {
                        "required": true
                    }
                }
            },
            "DeleteLifecycleHook": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DeleteLifecycleHook.html",
                "description": "Deletes the specified lifecycle hook.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "autoScalingGroup": {
                        "required": true,
                        "conditions": [
                            "autoscaling:ResourceTag/${TagKey}",
                            "aws:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteNotificationConfiguration": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DeleteNotificationConfiguration.html",
                "description": "Deletes the specified notification.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "autoScalingGroup": {
                        "required": true,
                        "conditions": [
                            "autoscaling:ResourceTag/${TagKey}",
                            "aws:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeletePolicy": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DeletePolicy.html",
                "description": "Deletes the specified Auto Scaling policy.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "autoScalingGroup": {
                        "required": true,
                        "conditions": [
                            "autoscaling:ResourceTag/${TagKey}",
                            "aws:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteScheduledAction": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DeleteScheduledAction.html",
                "description": "Deletes the specified scheduled action.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "autoScalingGroup": {
                        "required": true,
                        "conditions": [
                            "autoscaling:ResourceTag/${TagKey}",
                            "aws:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteTags": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DeleteTags.html",
                "description": "Deletes the specified tags.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "autoScalingGroup": {
                        "required": true,
                        "conditions": [
                            "autoscaling:ResourceTag/${TagKey}",
                            "aws:ResourceTag/${TagKey}"
                        ]
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "DescribeAccountLimits": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeAccountLimits.html",
                "description": "Describes the current Auto Scaling resource limits for your AWS account.",
                "accessLevel": "List"
            },
            "DescribeAdjustmentTypes": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeAdjustmentTypes.html",
                "description": "Describes the policy adjustment types for use with PutScalingPolicy.",
                "accessLevel": "List"
            },
            "DescribeAutoScalingGroups": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeAutoScalingGroups.html",
                "description": "Describes one or more Auto Scaling groups. If a list of names is not provided, the call describes all Auto Scaling groups.",
                "accessLevel": "List"
            },
            "DescribeAutoScalingInstances": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeAutoScalingInstances.html",
                "description": "Describes one or more Auto Scaling instances. If a list is not provided, the call describes all instances.",
                "accessLevel": "List"
            },
            "DescribeAutoScalingNotificationTypes": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeAutoScalingNotificationTypes.html",
                "description": "Describes the notification types that are supported by Auto Scaling.",
                "accessLevel": "List"
            },
            "DescribeInstanceRefreshes": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeInstanceRefreshes.html",
                "description": "Grants permission to describe one or more instance refreshes for an Auto Scaling group",
                "accessLevel": "List"
            },
            "DescribeLaunchConfigurations": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeLaunchConfigurations.html",
                "description": "Describes one or more launch configurations. If you omit the list of names, then the call describes all launch configurations.",
                "accessLevel": "List"
            },
            "DescribeLifecycleHookTypes": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeLifecycleHookTypes.html",
                "description": "Describes the available types of lifecycle hooks.",
                "accessLevel": "List"
            },
            "DescribeLifecycleHooks": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeLifecycleHooks.html",
                "description": "Describes the lifecycle hooks for the specified Auto Scaling group.",
                "accessLevel": "List"
            },
            "DescribeLoadBalancerTargetGroups": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeLoadBalancerTargetGroups.html",
                "description": "Describes the target groups for the specified Auto Scaling group.",
                "accessLevel": "List"
            },
            "DescribeLoadBalancers": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeLoadBalancers.html",
                "description": "Describes the load balancers for the specified Auto Scaling group.",
                "accessLevel": "List"
            },
            "DescribeMetricCollectionTypes": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeMetricCollectionTypes.html",
                "description": "Describes the available CloudWatch metrics for Auto Scaling.",
                "accessLevel": "List"
            },
            "DescribeNotificationConfigurations": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeNotificationConfigurations.html",
                "description": "Describes the notification actions associated with the specified Auto Scaling group.",
                "accessLevel": "List"
            },
            "DescribePolicies": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribePolicies.html",
                "description": "Describes the policies for the specified Auto Scaling group.",
                "accessLevel": "List"
            },
            "DescribeScalingActivities": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeScalingActivities.html",
                "description": "Describes one or more scaling activities for the specified Auto Scaling group.",
                "accessLevel": "List"
            },
            "DescribeScalingProcessTypes": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeScalingProcessTypes.html",
                "description": "Describes the scaling process types for use with ResumeProcesses and SuspendProcesses.",
                "accessLevel": "List"
            },
            "DescribeScheduledActions": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeScheduledActions.html",
                "description": "Describes the actions scheduled for your Auto Scaling group that haven't run.",
                "accessLevel": "List"
            },
            "DescribeTags": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeTags.html",
                "description": "Describes the specified tags.",
                "accessLevel": "Read"
            },
            "DescribeTerminationPolicyTypes": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeTerminationPolicyTypes.html",
                "description": "Describes the termination policies supported by Auto Scaling.",
                "accessLevel": "List"
            },
            "DetachInstances": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DetachInstances.html",
                "description": "Removes one or more instances from the specified Auto Scaling group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "autoScalingGroup": {
                        "required": true,
                        "conditions": [
                            "autoscaling:ResourceTag/${TagKey}",
                            "aws:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DetachLoadBalancerTargetGroups": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DetachLoadBalancerTargetGroups.html",
                "description": "Detaches one or more target groups from the specified Auto Scaling group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "autoScalingGroup": {
                        "required": true,
                        "conditions": [
                            "autoscaling:ResourceTag/${TagKey}",
                            "aws:ResourceTag/${TagKey}"
                        ]
                    }
                },
                "conditions": [
                    "autoscaling:TargetGroupARNs"
                ]
            },
            "DetachLoadBalancers": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DetachLoadBalancers.html",
                "description": "Removes one or more load balancers from the specified Auto Scaling group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "autoScalingGroup": {
                        "required": true,
                        "conditions": [
                            "autoscaling:ResourceTag/${TagKey}",
                            "aws:ResourceTag/${TagKey}"
                        ]
                    }
                },
                "conditions": [
                    "autoscaling:LoadBalancerNames"
                ]
            },
            "DisableMetricsCollection": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DisableMetricsCollection.html",
                "description": "Disables monitoring of the specified metrics for the specified Auto Scaling group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "autoScalingGroup": {
                        "required": true,
                        "conditions": [
                            "autoscaling:ResourceTag/${TagKey}",
                            "aws:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "EnableMetricsCollection": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_EnableMetricsCollection.html",
                "description": "Enables monitoring of the specified metrics for the specified Auto Scaling group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "autoScalingGroup": {
                        "required": true,
                        "conditions": [
                            "autoscaling:ResourceTag/${TagKey}",
                            "aws:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "EnterStandby": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_EnterStandby.html",
                "description": "Moves the specified instances into Standby mode.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "autoScalingGroup": {
                        "required": true,
                        "conditions": [
                            "autoscaling:ResourceTag/${TagKey}",
                            "aws:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "ExecutePolicy": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_ExecutePolicy.html",
                "description": "Executes the specified policy.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "autoScalingGroup": {
                        "required": true,
                        "conditions": [
                            "autoscaling:ResourceTag/${TagKey}",
                            "aws:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "ExitStandby": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_ExitStandby.html",
                "description": "Moves the specified instances out of Standby mode.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "autoScalingGroup": {
                        "required": true,
                        "conditions": [
                            "autoscaling:ResourceTag/${TagKey}",
                            "aws:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "PutLifecycleHook": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_PutLifecycleHook.html",
                "description": "Creates or updates a lifecycle hook for the specified Auto Scaling Group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "autoScalingGroup": {
                        "required": true,
                        "conditions": [
                            "autoscaling:ResourceTag/${TagKey}",
                            "aws:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "PutNotificationConfiguration": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_PutNotificationConfiguration.html",
                "description": "Configures an Auto Scaling group to send notifications when specified events take place.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "autoScalingGroup": {
                        "required": true,
                        "conditions": [
                            "autoscaling:ResourceTag/${TagKey}",
                            "aws:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "PutScalingPolicy": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_PutScalingPolicy.html",
                "description": "Creates or updates a policy for an Auto Scaling group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "autoScalingGroup": {
                        "required": true,
                        "conditions": [
                            "autoscaling:ResourceTag/${TagKey}",
                            "aws:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "PutScheduledUpdateGroupAction": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_PutScheduledUpdateGroupAction.html",
                "description": "Creates or updates a scheduled scaling action for an Auto Scaling group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "autoScalingGroup": {
                        "required": true,
                        "conditions": [
                            "autoscaling:ResourceTag/${TagKey}",
                            "aws:ResourceTag/${TagKey}"
                        ]
                    }
                },
                "conditions": [
                    "autoscaling:MaxSize",
                    "autoscaling:MinSize"
                ]
            },
            "RecordLifecycleActionHeartbeat": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_RecordLifecycleActionHeartbeat.html",
                "description": "Records a heartbeat for the lifecycle action associated with the specified token or instance.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "autoScalingGroup": {
                        "required": true,
                        "conditions": [
                            "autoscaling:ResourceTag/${TagKey}",
                            "aws:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "ResumeProcesses": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_ResumeProcesses.html",
                "description": "Resumes the specified suspended Auto Scaling processes, or all suspended process, for the specified Auto Scaling group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "autoScalingGroup": {
                        "required": true,
                        "conditions": [
                            "autoscaling:ResourceTag/${TagKey}",
                            "aws:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "SetDesiredCapacity": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_SetDesiredCapacity.html",
                "description": "Sets the size of the specified Auto Scaling group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "autoScalingGroup": {
                        "required": true,
                        "conditions": [
                            "autoscaling:ResourceTag/${TagKey}",
                            "aws:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "SetInstanceHealth": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_SetInstanceHealth.html",
                "description": "Sets the health status of the specified instance.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "autoScalingGroup": {
                        "required": true,
                        "conditions": [
                            "autoscaling:ResourceTag/${TagKey}",
                            "aws:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "SetInstanceProtection": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_SetInstanceProtection.html",
                "description": "Updates the instance protection settings of the specified instances.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "autoScalingGroup": {
                        "required": true,
                        "conditions": [
                            "autoscaling:ResourceTag/${TagKey}",
                            "aws:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "StartInstanceRefresh": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_StartInstanceRefresh.html",
                "description": "Grants permission to start a new instance refresh operation",
                "accessLevel": "Write",
                "resourceTypes": {
                    "autoScalingGroup": {
                        "required": true,
                        "conditions": [
                            "autoscaling:ResourceTag/${TagKey}",
                            "aws:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "SuspendProcesses": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_SuspendProcesses.html",
                "description": "Suspends the specified Auto Scaling processes, or all processes, for the specified Auto Scaling group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "autoScalingGroup": {
                        "required": true,
                        "conditions": [
                            "autoscaling:ResourceTag/${TagKey}",
                            "aws:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "TerminateInstanceInAutoScalingGroup": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_TerminateInstanceInAutoScalingGroup.html",
                "description": "Terminates the specified instance and optionally adjusts the desired group size.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "autoScalingGroup": {
                        "required": true,
                        "conditions": [
                            "autoscaling:ResourceTag/${TagKey}",
                            "aws:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "UpdateAutoScalingGroup": {
                "url": "https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_UpdateAutoScalingGroup.html",
                "description": "Updates the configuration for the specified Auto Scaling group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "autoScalingGroup": {
                        "required": true,
                        "conditions": [
                            "autoscaling:ResourceTag/${TagKey}",
                            "aws:ResourceTag/${TagKey}"
                        ]
                    }
                },
                "conditions": [
                    "autoscaling:InstanceTypes",
                    "autoscaling:LaunchConfigurationName",
                    "autoscaling:LaunchTemplateVersionSpecified",
                    "autoscaling:MaxSize",
                    "autoscaling:MinSize",
                    "autoscaling:VPCZoneIdentifiers"
                ]
            }
        };
        this.resourceTypes = {
            "autoScalingGroup": {
                "name": "autoScalingGroup",
                "url": "https://docs.aws.amazon.com/autoscaling/latest/userguide/control-access-using-iam.html#policy-auto-scaling-resources",
                "arn": "arn:${Partition}:autoscaling:${Region}:${Account}:autoScalingGroup:${GroupId}:autoScalingGroupName/${GroupFriendlyName}",
                "conditionKeys": [
                    "autoscaling:ResourceTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "launchConfiguration": {
                "name": "launchConfiguration",
                "url": "https://docs.aws.amazon.com/autoscaling/latest/userguide/control-access-using-iam.html#policy-auto-scaling-resources",
                "arn": "arn:${Partition}:autoscaling:${Region}:${Account}:launchConfiguration:${Id}:launchConfigurationName/${LaunchConfigurationName}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Attaches one or more EC2 instances to the specified Auto Scaling group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_AttachInstances.html
     */
    toAttachInstances() {
        this.add('autoscaling:AttachInstances');
        return this;
    }
    /**
     * Attaches one or more target groups to the specified Auto Scaling group.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifTargetGroupARNs()
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_AttachLoadBalancerTargetGroups.html
     */
    toAttachLoadBalancerTargetGroups() {
        this.add('autoscaling:AttachLoadBalancerTargetGroups');
        return this;
    }
    /**
     * Attaches one or more load balancers to the specified Auto Scaling group.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifLoadBalancerNames()
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_AttachLoadBalancers.html
     */
    toAttachLoadBalancers() {
        this.add('autoscaling:AttachLoadBalancers');
        return this;
    }
    /**
     * Deletes the specified scheduled actions.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_BatchDeleteScheduledAction.html
     */
    toBatchDeleteScheduledAction() {
        this.add('autoscaling:BatchDeleteScheduledAction');
        return this;
    }
    /**
     * Creates or updates multiple scheduled scaling actions for an Auto Scaling group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_BatchPutScheduledUpdateGroupAction.html
     */
    toBatchPutScheduledUpdateGroupAction() {
        this.add('autoscaling:BatchPutScheduledUpdateGroupAction');
        return this;
    }
    /**
     * Grants permission to cancel an instance refresh operation in progress
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_CancelInstanceRefresh.html
     */
    toCancelInstanceRefresh() {
        this.add('autoscaling:CancelInstanceRefresh');
        return this;
    }
    /**
     * Completes the lifecycle action for the specified token or instance with the specified result.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_CompleteLifecycleAction.html
     */
    toCompleteLifecycleAction() {
        this.add('autoscaling:CompleteLifecycleAction');
        return this;
    }
    /**
     * Creates an Auto Scaling group with the specified name and attributes.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifInstanceTypes()
     * - .ifLaunchConfigurationName()
     * - .ifLaunchTemplateVersionSpecified()
     * - .ifLoadBalancerNames()
     * - .ifMaxSize()
     * - .ifMinSize()
     * - .ifTargetGroupARNs()
     * - .ifVPCZoneIdentifiers()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_CreateAutoScalingGroup.html
     */
    toCreateAutoScalingGroup() {
        this.add('autoscaling:CreateAutoScalingGroup');
        return this;
    }
    /**
     * Creates a launch configuration.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifImageId()
     * - .ifInstanceType()
     * - .ifSpotPrice()
     * - .ifMetadataHttpTokens()
     * - .ifMetadataHttpPutResponseHopLimit()
     * - .ifMetadataHttpEndpoint()
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_CreateLaunchConfiguration.html
     */
    toCreateLaunchConfiguration() {
        this.add('autoscaling:CreateLaunchConfiguration');
        return this;
    }
    /**
     * Creates or updates tags for the specified Auto Scaling group.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_CreateOrUpdateTags.html
     */
    toCreateOrUpdateTags() {
        this.add('autoscaling:CreateOrUpdateTags');
        return this;
    }
    /**
     * Deletes the specified Auto Scaling group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DeleteAutoScalingGroup.html
     */
    toDeleteAutoScalingGroup() {
        this.add('autoscaling:DeleteAutoScalingGroup');
        return this;
    }
    /**
     * Deletes the specified launch configuration.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DeleteLaunchConfiguration.html
     */
    toDeleteLaunchConfiguration() {
        this.add('autoscaling:DeleteLaunchConfiguration');
        return this;
    }
    /**
     * Deletes the specified lifecycle hook.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DeleteLifecycleHook.html
     */
    toDeleteLifecycleHook() {
        this.add('autoscaling:DeleteLifecycleHook');
        return this;
    }
    /**
     * Deletes the specified notification.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DeleteNotificationConfiguration.html
     */
    toDeleteNotificationConfiguration() {
        this.add('autoscaling:DeleteNotificationConfiguration');
        return this;
    }
    /**
     * Deletes the specified Auto Scaling policy.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DeletePolicy.html
     */
    toDeletePolicy() {
        this.add('autoscaling:DeletePolicy');
        return this;
    }
    /**
     * Deletes the specified scheduled action.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DeleteScheduledAction.html
     */
    toDeleteScheduledAction() {
        this.add('autoscaling:DeleteScheduledAction');
        return this;
    }
    /**
     * Deletes the specified tags.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DeleteTags.html
     */
    toDeleteTags() {
        this.add('autoscaling:DeleteTags');
        return this;
    }
    /**
     * Describes the current Auto Scaling resource limits for your AWS account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeAccountLimits.html
     */
    toDescribeAccountLimits() {
        this.add('autoscaling:DescribeAccountLimits');
        return this;
    }
    /**
     * Describes the policy adjustment types for use with PutScalingPolicy.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeAdjustmentTypes.html
     */
    toDescribeAdjustmentTypes() {
        this.add('autoscaling:DescribeAdjustmentTypes');
        return this;
    }
    /**
     * Describes one or more Auto Scaling groups. If a list of names is not provided, the call describes all Auto Scaling groups.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeAutoScalingGroups.html
     */
    toDescribeAutoScalingGroups() {
        this.add('autoscaling:DescribeAutoScalingGroups');
        return this;
    }
    /**
     * Describes one or more Auto Scaling instances. If a list is not provided, the call describes all instances.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeAutoScalingInstances.html
     */
    toDescribeAutoScalingInstances() {
        this.add('autoscaling:DescribeAutoScalingInstances');
        return this;
    }
    /**
     * Describes the notification types that are supported by Auto Scaling.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeAutoScalingNotificationTypes.html
     */
    toDescribeAutoScalingNotificationTypes() {
        this.add('autoscaling:DescribeAutoScalingNotificationTypes');
        return this;
    }
    /**
     * Grants permission to describe one or more instance refreshes for an Auto Scaling group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeInstanceRefreshes.html
     */
    toDescribeInstanceRefreshes() {
        this.add('autoscaling:DescribeInstanceRefreshes');
        return this;
    }
    /**
     * Describes one or more launch configurations. If you omit the list of names, then the call describes all launch configurations.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeLaunchConfigurations.html
     */
    toDescribeLaunchConfigurations() {
        this.add('autoscaling:DescribeLaunchConfigurations');
        return this;
    }
    /**
     * Describes the available types of lifecycle hooks.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeLifecycleHookTypes.html
     */
    toDescribeLifecycleHookTypes() {
        this.add('autoscaling:DescribeLifecycleHookTypes');
        return this;
    }
    /**
     * Describes the lifecycle hooks for the specified Auto Scaling group.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeLifecycleHooks.html
     */
    toDescribeLifecycleHooks() {
        this.add('autoscaling:DescribeLifecycleHooks');
        return this;
    }
    /**
     * Describes the target groups for the specified Auto Scaling group.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeLoadBalancerTargetGroups.html
     */
    toDescribeLoadBalancerTargetGroups() {
        this.add('autoscaling:DescribeLoadBalancerTargetGroups');
        return this;
    }
    /**
     * Describes the load balancers for the specified Auto Scaling group.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeLoadBalancers.html
     */
    toDescribeLoadBalancers() {
        this.add('autoscaling:DescribeLoadBalancers');
        return this;
    }
    /**
     * Describes the available CloudWatch metrics for Auto Scaling.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeMetricCollectionTypes.html
     */
    toDescribeMetricCollectionTypes() {
        this.add('autoscaling:DescribeMetricCollectionTypes');
        return this;
    }
    /**
     * Describes the notification actions associated with the specified Auto Scaling group.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeNotificationConfigurations.html
     */
    toDescribeNotificationConfigurations() {
        this.add('autoscaling:DescribeNotificationConfigurations');
        return this;
    }
    /**
     * Describes the policies for the specified Auto Scaling group.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribePolicies.html
     */
    toDescribePolicies() {
        this.add('autoscaling:DescribePolicies');
        return this;
    }
    /**
     * Describes one or more scaling activities for the specified Auto Scaling group.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeScalingActivities.html
     */
    toDescribeScalingActivities() {
        this.add('autoscaling:DescribeScalingActivities');
        return this;
    }
    /**
     * Describes the scaling process types for use with ResumeProcesses and SuspendProcesses.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeScalingProcessTypes.html
     */
    toDescribeScalingProcessTypes() {
        this.add('autoscaling:DescribeScalingProcessTypes');
        return this;
    }
    /**
     * Describes the actions scheduled for your Auto Scaling group that haven't run.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeScheduledActions.html
     */
    toDescribeScheduledActions() {
        this.add('autoscaling:DescribeScheduledActions');
        return this;
    }
    /**
     * Describes the specified tags.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeTags.html
     */
    toDescribeTags() {
        this.add('autoscaling:DescribeTags');
        return this;
    }
    /**
     * Describes the termination policies supported by Auto Scaling.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DescribeTerminationPolicyTypes.html
     */
    toDescribeTerminationPolicyTypes() {
        this.add('autoscaling:DescribeTerminationPolicyTypes');
        return this;
    }
    /**
     * Removes one or more instances from the specified Auto Scaling group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DetachInstances.html
     */
    toDetachInstances() {
        this.add('autoscaling:DetachInstances');
        return this;
    }
    /**
     * Detaches one or more target groups from the specified Auto Scaling group.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifTargetGroupARNs()
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DetachLoadBalancerTargetGroups.html
     */
    toDetachLoadBalancerTargetGroups() {
        this.add('autoscaling:DetachLoadBalancerTargetGroups');
        return this;
    }
    /**
     * Removes one or more load balancers from the specified Auto Scaling group.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifLoadBalancerNames()
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DetachLoadBalancers.html
     */
    toDetachLoadBalancers() {
        this.add('autoscaling:DetachLoadBalancers');
        return this;
    }
    /**
     * Disables monitoring of the specified metrics for the specified Auto Scaling group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_DisableMetricsCollection.html
     */
    toDisableMetricsCollection() {
        this.add('autoscaling:DisableMetricsCollection');
        return this;
    }
    /**
     * Enables monitoring of the specified metrics for the specified Auto Scaling group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_EnableMetricsCollection.html
     */
    toEnableMetricsCollection() {
        this.add('autoscaling:EnableMetricsCollection');
        return this;
    }
    /**
     * Moves the specified instances into Standby mode.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_EnterStandby.html
     */
    toEnterStandby() {
        this.add('autoscaling:EnterStandby');
        return this;
    }
    /**
     * Executes the specified policy.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_ExecutePolicy.html
     */
    toExecutePolicy() {
        this.add('autoscaling:ExecutePolicy');
        return this;
    }
    /**
     * Moves the specified instances out of Standby mode.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_ExitStandby.html
     */
    toExitStandby() {
        this.add('autoscaling:ExitStandby');
        return this;
    }
    /**
     * Creates or updates a lifecycle hook for the specified Auto Scaling Group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_PutLifecycleHook.html
     */
    toPutLifecycleHook() {
        this.add('autoscaling:PutLifecycleHook');
        return this;
    }
    /**
     * Configures an Auto Scaling group to send notifications when specified events take place.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_PutNotificationConfiguration.html
     */
    toPutNotificationConfiguration() {
        this.add('autoscaling:PutNotificationConfiguration');
        return this;
    }
    /**
     * Creates or updates a policy for an Auto Scaling group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_PutScalingPolicy.html
     */
    toPutScalingPolicy() {
        this.add('autoscaling:PutScalingPolicy');
        return this;
    }
    /**
     * Creates or updates a scheduled scaling action for an Auto Scaling group.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifMaxSize()
     * - .ifMinSize()
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_PutScheduledUpdateGroupAction.html
     */
    toPutScheduledUpdateGroupAction() {
        this.add('autoscaling:PutScheduledUpdateGroupAction');
        return this;
    }
    /**
     * Records a heartbeat for the lifecycle action associated with the specified token or instance.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_RecordLifecycleActionHeartbeat.html
     */
    toRecordLifecycleActionHeartbeat() {
        this.add('autoscaling:RecordLifecycleActionHeartbeat');
        return this;
    }
    /**
     * Resumes the specified suspended Auto Scaling processes, or all suspended process, for the specified Auto Scaling group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_ResumeProcesses.html
     */
    toResumeProcesses() {
        this.add('autoscaling:ResumeProcesses');
        return this;
    }
    /**
     * Sets the size of the specified Auto Scaling group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_SetDesiredCapacity.html
     */
    toSetDesiredCapacity() {
        this.add('autoscaling:SetDesiredCapacity');
        return this;
    }
    /**
     * Sets the health status of the specified instance.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_SetInstanceHealth.html
     */
    toSetInstanceHealth() {
        this.add('autoscaling:SetInstanceHealth');
        return this;
    }
    /**
     * Updates the instance protection settings of the specified instances.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_SetInstanceProtection.html
     */
    toSetInstanceProtection() {
        this.add('autoscaling:SetInstanceProtection');
        return this;
    }
    /**
     * Grants permission to start a new instance refresh operation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_StartInstanceRefresh.html
     */
    toStartInstanceRefresh() {
        this.add('autoscaling:StartInstanceRefresh');
        return this;
    }
    /**
     * Suspends the specified Auto Scaling processes, or all processes, for the specified Auto Scaling group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_SuspendProcesses.html
     */
    toSuspendProcesses() {
        this.add('autoscaling:SuspendProcesses');
        return this;
    }
    /**
     * Terminates the specified instance and optionally adjusts the desired group size.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_TerminateInstanceInAutoScalingGroup.html
     */
    toTerminateInstanceInAutoScalingGroup() {
        this.add('autoscaling:TerminateInstanceInAutoScalingGroup');
        return this;
    }
    /**
     * Updates the configuration for the specified Auto Scaling group.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceTypes()
     * - .ifLaunchConfigurationName()
     * - .ifLaunchTemplateVersionSpecified()
     * - .ifMaxSize()
     * - .ifMinSize()
     * - .ifVPCZoneIdentifiers()
     *
     * https://docs.aws.amazon.com/AutoScaling/latest/APIReference/API_UpdateAutoScalingGroup.html
     */
    toUpdateAutoScalingGroup() {
        this.add('autoscaling:UpdateAutoScalingGroup');
        return this;
    }
    /**
     * Adds a resource of type autoScalingGroup to the statement
     *
     * https://docs.aws.amazon.com/autoscaling/latest/userguide/control-access-using-iam.html#policy-auto-scaling-resources
     *
     * @param groupId - Identifier for the groupId.
     * @param groupFriendlyName - Identifier for the groupFriendlyName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifResourceTag()
     * - .ifAwsResourceTag()
     */
    onAutoScalingGroup(groupId, groupFriendlyName, account, region, partition) {
        var arn = 'arn:${Partition}:autoscaling:${Region}:${Account}:autoScalingGroup:${GroupId}:autoScalingGroupName/${GroupFriendlyName}';
        arn = arn.replace('${GroupId}', groupId);
        arn = arn.replace('${GroupFriendlyName}', groupFriendlyName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type launchConfiguration to the statement
     *
     * https://docs.aws.amazon.com/autoscaling/latest/userguide/control-access-using-iam.html#policy-auto-scaling-resources
     *
     * @param id - Identifier for the id.
     * @param launchConfigurationName - Identifier for the launchConfigurationName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onLaunchConfiguration(id, launchConfigurationName, account, region, partition) {
        var arn = 'arn:${Partition}:autoscaling:${Region}:${Account}:launchConfiguration:${Id}:launchConfigurationName/${LaunchConfigurationName}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${LaunchConfigurationName}', launchConfigurationName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * The AMI used to create the instance.
     *
     * https://docs.aws.amazon.com/autoscaling/latest/userguide/control-access-using-iam.html#policy-auto-scaling-condition-keys
     *
     * Applies to actions:
     * - .toCreateLaunchConfiguration()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifImageId(value, operator) {
        return this.if(`autoscaling:ImageId`, value, operator || 'StringLike');
    }
    /**
     * The type of instance, in terms of the hardware resources available.
     *
     * https://docs.aws.amazon.com/autoscaling/latest/userguide/control-access-using-iam.html#policy-auto-scaling-condition-keys
     *
     * Applies to actions:
     * - .toCreateLaunchConfiguration()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifInstanceType(value, operator) {
        return this.if(`autoscaling:InstanceType`, value, operator || 'StringLike');
    }
    /**
     * The types of instances, in terms of the hardware resources available.
     *
     * https://docs.aws.amazon.com/autoscaling/latest/userguide/control-access-using-iam.html#policy-auto-scaling-condition-keys
     *
     * Applies to actions:
     * - .toCreateAutoScalingGroup()
     * - .toUpdateAutoScalingGroup()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifInstanceTypes(value, operator) {
        return this.if(`autoscaling:InstanceTypes`, value, operator || 'StringLike');
    }
    /**
     * The name of a launch configuration.
     *
     * https://docs.aws.amazon.com/autoscaling/latest/userguide/control-access-using-iam.html#policy-auto-scaling-condition-keys
     *
     * Applies to actions:
     * - .toCreateAutoScalingGroup()
     * - .toUpdateAutoScalingGroup()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifLaunchConfigurationName(value, operator) {
        return this.if(`autoscaling:LaunchConfigurationName`, value, operator || 'StringLike');
    }
    /**
     * Filters access by whether users can specify any version of a launch template or only the Latest or Default version
     *
     * https://docs.aws.amazon.com/autoscaling/latest/userguide/control-access-using-iam.html#policy-auto-scaling-condition-keys
     *
     * Applies to actions:
     * - .toCreateAutoScalingGroup()
     * - .toUpdateAutoScalingGroup()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifLaunchTemplateVersionSpecified(value) {
        return this.if(`autoscaling:LaunchTemplateVersionSpecified`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * The name of the load balancer.
     *
     * https://docs.aws.amazon.com/autoscaling/latest/userguide/control-access-using-iam.html#policy-auto-scaling-condition-keys
     *
     * Applies to actions:
     * - .toAttachLoadBalancers()
     * - .toCreateAutoScalingGroup()
     * - .toDetachLoadBalancers()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifLoadBalancerNames(value, operator) {
        return this.if(`autoscaling:LoadBalancerNames`, value, operator || 'StringLike');
    }
    /**
     * The maximum scaling size.
     *
     * https://docs.aws.amazon.com/autoscaling/latest/userguide/control-access-using-iam.html#policy-auto-scaling-condition-keys
     *
     * Applies to actions:
     * - .toCreateAutoScalingGroup()
     * - .toPutScheduledUpdateGroupAction()
     * - .toUpdateAutoScalingGroup()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifMaxSize(value, operator) {
        return this.if(`autoscaling:MaxSize`, value, operator || 'NumericEquals');
    }
    /**
     * Filters access by whether the HTTP endpoint is enabled for the instance metadata service.
     *
     * https://docs.aws.amazon.com/autoscaling/latest/userguide/control-access-using-iam.html#policy-auto-scaling-condition-keys
     *
     * Applies to actions:
     * - .toCreateLaunchConfiguration()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifMetadataHttpEndpoint(value, operator) {
        return this.if(`autoscaling:MetadataHttpEndpoint`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the allowed number of hops when calling the instance metadata service.
     *
     * https://docs.aws.amazon.com/autoscaling/latest/userguide/control-access-using-iam.html#policy-auto-scaling-condition-keys
     *
     * Applies to actions:
     * - .toCreateLaunchConfiguration()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifMetadataHttpPutResponseHopLimit(value, operator) {
        return this.if(`autoscaling:MetadataHttpPutResponseHopLimit`, value, operator || 'NumericEquals');
    }
    /**
     * Filters access by whether tokens are required when calling the instance metadata service (optional or required)
     *
     * https://docs.aws.amazon.com/autoscaling/latest/userguide/control-access-using-iam.html#policy-auto-scaling-condition-keys
     *
     * Applies to actions:
     * - .toCreateLaunchConfiguration()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifMetadataHttpTokens(value, operator) {
        return this.if(`autoscaling:MetadataHttpTokens`, value, operator || 'StringLike');
    }
    /**
     * The minimum scaling size.
     *
     * https://docs.aws.amazon.com/autoscaling/latest/userguide/control-access-using-iam.html#policy-auto-scaling-condition-keys
     *
     * Applies to actions:
     * - .toCreateAutoScalingGroup()
     * - .toPutScheduledUpdateGroupAction()
     * - .toUpdateAutoScalingGroup()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifMinSize(value, operator) {
        return this.if(`autoscaling:MinSize`, value, operator || 'NumericEquals');
    }
    /**
     * The value of a tag attached to a resource.
     *
     * https://docs.aws.amazon.com/autoscaling/latest/userguide/control-access-using-iam.html#policy-auto-scaling-condition-keys
     *
     * Applies to actions:
     * - .toAttachInstances()
     * - .toAttachLoadBalancerTargetGroups()
     * - .toAttachLoadBalancers()
     * - .toBatchDeleteScheduledAction()
     * - .toBatchPutScheduledUpdateGroupAction()
     * - .toCancelInstanceRefresh()
     * - .toCompleteLifecycleAction()
     * - .toCreateAutoScalingGroup()
     * - .toCreateOrUpdateTags()
     * - .toDeleteAutoScalingGroup()
     * - .toDeleteLifecycleHook()
     * - .toDeleteNotificationConfiguration()
     * - .toDeletePolicy()
     * - .toDeleteScheduledAction()
     * - .toDeleteTags()
     * - .toDetachInstances()
     * - .toDetachLoadBalancerTargetGroups()
     * - .toDetachLoadBalancers()
     * - .toDisableMetricsCollection()
     * - .toEnableMetricsCollection()
     * - .toEnterStandby()
     * - .toExecutePolicy()
     * - .toExitStandby()
     * - .toPutLifecycleHook()
     * - .toPutNotificationConfiguration()
     * - .toPutScalingPolicy()
     * - .toPutScheduledUpdateGroupAction()
     * - .toRecordLifecycleActionHeartbeat()
     * - .toResumeProcesses()
     * - .toSetDesiredCapacity()
     * - .toSetInstanceHealth()
     * - .toSetInstanceProtection()
     * - .toStartInstanceRefresh()
     * - .toSuspendProcesses()
     * - .toTerminateInstanceInAutoScalingGroup()
     * - .toUpdateAutoScalingGroup()
     *
     * Applies to resource types:
     * - autoScalingGroup
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceTag(tagKey, value, operator) {
        return this.if(`autoscaling:ResourceTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * The spot price associated with an instance.
     *
     * https://docs.aws.amazon.com/autoscaling/latest/userguide/control-access-using-iam.html#policy-auto-scaling-condition-keys
     *
     * Applies to actions:
     * - .toCreateLaunchConfiguration()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifSpotPrice(value, operator) {
        return this.if(`autoscaling:SpotPrice`, value, operator || 'NumericEquals');
    }
    /**
     * The ARN of a target group.
     *
     * https://docs.aws.amazon.com/autoscaling/latest/userguide/control-access-using-iam.html#policy-auto-scaling-condition-keys
     *
     * Applies to actions:
     * - .toAttachLoadBalancerTargetGroups()
     * - .toCreateAutoScalingGroup()
     * - .toDetachLoadBalancerTargetGroups()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifTargetGroupARNs(value, operator) {
        return this.if(`autoscaling:TargetGroupARNs`, value, operator || 'ArnEquals');
    }
    /**
     * The identifier of a VPC zone.
     *
     * https://docs.aws.amazon.com/autoscaling/latest/userguide/control-access-using-iam.html#policy-auto-scaling-condition-keys
     *
     * Applies to actions:
     * - .toCreateAutoScalingGroup()
     * - .toUpdateAutoScalingGroup()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifVPCZoneIdentifiers(value, operator) {
        return this.if(`autoscaling:VPCZoneIdentifiers`, value, operator || 'StringLike');
    }
}
exports.Autoscaling = Autoscaling;
//# sourceMappingURL=data:application/json;base64,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