"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Ebs = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [ebs](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonelasticblockstore.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Ebs extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [ebs](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonelasticblockstore.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'ebs';
        this.actionList = {
            "GetSnapshotBlock": {
                "url": "https://docs.aws.amazon.com/ebs/latest/APIReference/API_GetSnapshotBlock.html",
                "description": "Grants permission to return the data of a block in an Amazon Elastic Block Store (EBS) snapshot",
                "accessLevel": "Read",
                "resourceTypes": {
                    "snapshot": {
                        "required": true
                    }
                }
            },
            "ListChangedBlocks": {
                "url": "https://docs.aws.amazon.com/ebs/latest/APIReference/API_ListChangedBlocks.html",
                "description": "Grants permission to list the block indexes and block tokens for blocks that are different between two Amazon Elastic Block Store (EBS) snapshots of the same volume/snapshot lineage",
                "accessLevel": "Read",
                "resourceTypes": {
                    "snapshot": {
                        "required": true
                    }
                }
            },
            "ListSnapshotBlocks": {
                "url": "https://docs.aws.amazon.com/ebs/latest/APIReference/API_ListSnapshotBlocks.html",
                "description": "Grants permission to list the block indexes and block tokens for blocks in an Amazon Elastic Block Store (EBS) snapshot.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "snapshot": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "snapshot": {
                "name": "snapshot",
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#EC2_ARN_Format",
                "arn": "arn:${Partition}:ec2:${Region}::snapshot/${SnapshotId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Grants permission to return the data of a block in an Amazon Elastic Block Store (EBS) snapshot
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ebs/latest/APIReference/API_GetSnapshotBlock.html
     */
    toGetSnapshotBlock() {
        this.add('ebs:GetSnapshotBlock');
        return this;
    }
    /**
     * Grants permission to list the block indexes and block tokens for blocks that are different between two Amazon Elastic Block Store (EBS) snapshots of the same volume/snapshot lineage
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ebs/latest/APIReference/API_ListChangedBlocks.html
     */
    toListChangedBlocks() {
        this.add('ebs:ListChangedBlocks');
        return this;
    }
    /**
     * Grants permission to list the block indexes and block tokens for blocks in an Amazon Elastic Block Store (EBS) snapshot.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ebs/latest/APIReference/API_ListSnapshotBlocks.html
     */
    toListSnapshotBlocks() {
        this.add('ebs:ListSnapshotBlocks');
        return this;
    }
    /**
     * Adds a resource of type snapshot to the statement
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#EC2_ARN_Format
     *
     * @param snapshotId - Identifier for the snapshotId.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSnapshot(snapshotId, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}::snapshot/${SnapshotId}';
        arn = arn.replace('${SnapshotId}', snapshotId);
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Ebs = Ebs;
//# sourceMappingURL=data:application/json;base64,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