"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Events = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [events](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoneventbridge.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Events extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [events](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoneventbridge.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'events';
        this.actionList = {
            "ActivateEventSource": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_ActivateEventSource.html",
                "description": "Activates a partner event source that has been deactivated. Once activated, your matching event bus will start receiving events from the event source.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "event-source": {
                        "required": true
                    }
                }
            },
            "CreateEventBus": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_CreateEventBus.html",
                "description": "Creates a new event bus within your account. This can be a custom event bus which you can use to receive events from your own custom applications and services, or it can be a partner event bus which can be matched to a partner event source.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "event-bus": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreatePartnerEventSource": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_CreatePartnerEventSource.html",
                "description": "Called by an AWS partner to create a partner event source.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "event-source": {
                        "required": true
                    }
                }
            },
            "DeactivateEventSource": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_DeactivateEventSource.html",
                "description": "Called by an AWS partner to create a partner event source.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "event-source": {
                        "required": true
                    }
                }
            },
            "DeleteEventBus": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_DeletePartnerEventSource",
                "description": "Deletes the specified custom event bus or partner event bus. All rules associated with this event bus are also deleted. You can't delete your account's default event bus.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "event-bus": {
                        "required": true
                    }
                }
            },
            "DeletePartnerEventSource": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_DeletePartnerEventSource",
                "description": "Called by an AWS partner to delete a partner event source.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "event-source": {
                        "required": true
                    }
                }
            },
            "DeleteRule": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_DeleteRule.html",
                "description": "Deletes a rule. You must remove all targets from a rule using RemoveTargets before you can delete the rule.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "rule": {
                        "required": true
                    }
                }
            },
            "DescribeEventBus": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_DescribeEventBus.html",
                "description": "Displays the external AWS accounts that are permitted to write events to your account using your account's event bus, and the associated policy.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "event-bus": {
                        "required": false
                    }
                }
            },
            "DescribeEventSource": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_DescribeEventSource.html",
                "description": "Describes the details of the specified partner event source that is shared with your account.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "event-source": {
                        "required": true
                    }
                }
            },
            "DescribePartnerEventSource": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_DescribePartnerEventSource.html",
                "description": "Called by an AWS partner describe the details of the specified partner event source that they have created.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "event-source": {
                        "required": true
                    }
                }
            },
            "DescribeRule": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_DescribeRule.html",
                "description": "Describes the details of the specified rule.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "rule": {
                        "required": true
                    }
                }
            },
            "DisableRule": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_DisableRule.html",
                "description": "Disables a rule. A disabled rule won't match any events, and won't self-trigger if it has a schedule expression.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "rule": {
                        "required": true
                    }
                }
            },
            "EnableRule": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_EnableRule.html",
                "description": "Enables a rule. If the rule does not exist, the operation fails.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "rule": {
                        "required": true
                    }
                }
            },
            "ListEventBuses": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_ListEventBuses.html",
                "description": "Lists all the event buses in your account, including the default event bus, custom event buses, and partner event buses.",
                "accessLevel": "List"
            },
            "ListEventSources": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_ListEventSources.html",
                "description": "Lists the event sources shared with this account.",
                "accessLevel": "List"
            },
            "ListPartnerEventSourceAccounts": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_ListPartnerEventSourceAccounts.html",
                "description": "Called by an AWS partner to display the AWS account ID that the specified partner event source is associated with.",
                "accessLevel": "List",
                "resourceTypes": {
                    "event-source": {
                        "required": true
                    }
                }
            },
            "ListPartnerEventSources": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_ListPartnerEventSources.html",
                "description": "Called by an AWS partner to list all the partner event sources that they have created.",
                "accessLevel": "List"
            },
            "ListRuleNamesByTarget": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_ListRuleNamesByTarget.html",
                "description": "Lists the names of the rules that the given target is put to.",
                "accessLevel": "List"
            },
            "ListRules": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_ListRules.html",
                "description": "Lists the Amazon EventBridge rules in your account.",
                "accessLevel": "List"
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_ListTagsForResource.html",
                "description": "This action lists tags for an Amazon EventBridge resource.",
                "accessLevel": "List",
                "resourceTypes": {
                    "event-bus": {
                        "required": false
                    },
                    "rule": {
                        "required": false
                    }
                }
            },
            "ListTargetsByRule": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_ListTargetsByRule.html",
                "description": "Lists of targets assigned to the rule.",
                "accessLevel": "List",
                "resourceTypes": {
                    "rule": {
                        "required": true
                    }
                }
            },
            "PutEvents": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_PutEvents.html",
                "description": "Sends custom events to Amazon EventBridge so that they can be matched to rules.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "event-bus": {
                        "required": true
                    }
                }
            },
            "PutPartnerEvents": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_PutPartnerEvents.html",
                "description": "Sends custom events to Amazon EventBridge so that they can be matched to rules.",
                "accessLevel": "Write"
            },
            "PutPermission": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_PutPermission.html",
                "description": "Running PutPermission permits the specified AWS account to put events to your account's default event bus.",
                "accessLevel": "Write"
            },
            "PutRule": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_PutRule.html",
                "description": "Creates or updates a rule. Rules are enabled by default, or based on value of the State parameter.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "rule": {
                        "required": true
                    }
                },
                "conditions": [
                    "events:detail.userIdentity.principalId",
                    "events:detail-type",
                    "events:source",
                    "events:detail.service",
                    "events:detail.eventTypeCode",
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "PutTargets": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_PutTargets.html",
                "description": "Adds target(s) to a rule. Targets are the resources that can be invoked when a rule is triggered.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "rule": {
                        "required": true
                    }
                },
                "conditions": [
                    "events:TargetArn"
                ]
            },
            "RemovePermission": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_RemovePermission.html",
                "description": "Revokes the permission of another AWS account to be able to put events to your default event bus.",
                "accessLevel": "Write"
            },
            "RemoveTargets": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_RemoveTargets.html",
                "description": "Removes target(s) from a rule so that when the rule is triggered, those targets will no longer be invoked.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "rule": {
                        "required": true
                    }
                }
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_TagResource.html",
                "description": "This action tags an Amazon EventBridge resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "event-bus": {
                        "required": false
                    },
                    "rule": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "TestEventPattern": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_TestEventPattern.html",
                "description": "Tests whether an event pattern matches the provided event.",
                "accessLevel": "Read"
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_UntagResource.html",
                "description": "This action removes a tag from an Amazon EventBridge resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "event-bus": {
                        "required": false
                    },
                    "rule": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            }
        };
        this.resourceTypes = {
            "event-source": {
                "name": "event-source",
                "url": "https://docs.aws.amazon.com/eventbridge/latest/userguide/iam-access-control-identity-based-eventbridge.html#eventbridge-arn-format",
                "arn": "arn:${Partition}:events:${Region}::event-source/${EventSourceName}",
                "conditionKeys": []
            },
            "event-bus": {
                "name": "event-bus",
                "url": "https://docs.aws.amazon.com/eventbridge/latest/userguide/iam-access-control-identity-based-eventbridge.html#eventbridge-arn-format",
                "arn": "arn:${Partition}:events:${Region}:${Account}:event-bus/${EventBusName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "rule": {
                "name": "rule",
                "url": "https://docs.aws.amazon.com/eventbridge/latest/userguide/iam-access-control-identity-based-eventbridge.html#eventbridge-arn-format",
                "arn": "arn:${Partition}:events:${Region}:${Account}:rule/${RuleName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Activates a partner event source that has been deactivated. Once activated, your matching event bus will start receiving events from the event source.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_ActivateEventSource.html
     */
    toActivateEventSource() {
        this.add('events:ActivateEventSource');
        return this;
    }
    /**
     * Creates a new event bus within your account. This can be a custom event bus which you can use to receive events from your own custom applications and services, or it can be a partner event bus which can be matched to a partner event source.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_CreateEventBus.html
     */
    toCreateEventBus() {
        this.add('events:CreateEventBus');
        return this;
    }
    /**
     * Called by an AWS partner to create a partner event source.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_CreatePartnerEventSource.html
     */
    toCreatePartnerEventSource() {
        this.add('events:CreatePartnerEventSource');
        return this;
    }
    /**
     * Called by an AWS partner to create a partner event source.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_DeactivateEventSource.html
     */
    toDeactivateEventSource() {
        this.add('events:DeactivateEventSource');
        return this;
    }
    /**
     * Deletes the specified custom event bus or partner event bus. All rules associated with this event bus are also deleted. You can't delete your account's default event bus.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_DeletePartnerEventSource
     */
    toDeleteEventBus() {
        this.add('events:DeleteEventBus');
        return this;
    }
    /**
     * Called by an AWS partner to delete a partner event source.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_DeletePartnerEventSource
     */
    toDeletePartnerEventSource() {
        this.add('events:DeletePartnerEventSource');
        return this;
    }
    /**
     * Deletes a rule. You must remove all targets from a rule using RemoveTargets before you can delete the rule.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_DeleteRule.html
     */
    toDeleteRule() {
        this.add('events:DeleteRule');
        return this;
    }
    /**
     * Displays the external AWS accounts that are permitted to write events to your account using your account's event bus, and the associated policy.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_DescribeEventBus.html
     */
    toDescribeEventBus() {
        this.add('events:DescribeEventBus');
        return this;
    }
    /**
     * Describes the details of the specified partner event source that is shared with your account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_DescribeEventSource.html
     */
    toDescribeEventSource() {
        this.add('events:DescribeEventSource');
        return this;
    }
    /**
     * Called by an AWS partner describe the details of the specified partner event source that they have created.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_DescribePartnerEventSource.html
     */
    toDescribePartnerEventSource() {
        this.add('events:DescribePartnerEventSource');
        return this;
    }
    /**
     * Describes the details of the specified rule.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_DescribeRule.html
     */
    toDescribeRule() {
        this.add('events:DescribeRule');
        return this;
    }
    /**
     * Disables a rule. A disabled rule won't match any events, and won't self-trigger if it has a schedule expression.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_DisableRule.html
     */
    toDisableRule() {
        this.add('events:DisableRule');
        return this;
    }
    /**
     * Enables a rule. If the rule does not exist, the operation fails.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_EnableRule.html
     */
    toEnableRule() {
        this.add('events:EnableRule');
        return this;
    }
    /**
     * Lists all the event buses in your account, including the default event bus, custom event buses, and partner event buses.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_ListEventBuses.html
     */
    toListEventBuses() {
        this.add('events:ListEventBuses');
        return this;
    }
    /**
     * Lists the event sources shared with this account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_ListEventSources.html
     */
    toListEventSources() {
        this.add('events:ListEventSources');
        return this;
    }
    /**
     * Called by an AWS partner to display the AWS account ID that the specified partner event source is associated with.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_ListPartnerEventSourceAccounts.html
     */
    toListPartnerEventSourceAccounts() {
        this.add('events:ListPartnerEventSourceAccounts');
        return this;
    }
    /**
     * Called by an AWS partner to list all the partner event sources that they have created.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_ListPartnerEventSources.html
     */
    toListPartnerEventSources() {
        this.add('events:ListPartnerEventSources');
        return this;
    }
    /**
     * Lists the names of the rules that the given target is put to.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_ListRuleNamesByTarget.html
     */
    toListRuleNamesByTarget() {
        this.add('events:ListRuleNamesByTarget');
        return this;
    }
    /**
     * Lists the Amazon EventBridge rules in your account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_ListRules.html
     */
    toListRules() {
        this.add('events:ListRules');
        return this;
    }
    /**
     * This action lists tags for an Amazon EventBridge resource.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.add('events:ListTagsForResource');
        return this;
    }
    /**
     * Lists of targets assigned to the rule.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_ListTargetsByRule.html
     */
    toListTargetsByRule() {
        this.add('events:ListTargetsByRule');
        return this;
    }
    /**
     * Sends custom events to Amazon EventBridge so that they can be matched to rules.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_PutEvents.html
     */
    toPutEvents() {
        this.add('events:PutEvents');
        return this;
    }
    /**
     * Sends custom events to Amazon EventBridge so that they can be matched to rules.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_PutPartnerEvents.html
     */
    toPutPartnerEvents() {
        this.add('events:PutPartnerEvents');
        return this;
    }
    /**
     * Running PutPermission permits the specified AWS account to put events to your account's default event bus.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_PutPermission.html
     */
    toPutPermission() {
        this.add('events:PutPermission');
        return this;
    }
    /**
     * Creates or updates a rule. Rules are enabled by default, or based on value of the State parameter.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifDetailUserIdentityPrincipalId()
     * - .ifDetailType()
     * - .ifSource()
     * - .ifDetailService()
     * - .ifDetailEventTypeCode()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_PutRule.html
     */
    toPutRule() {
        this.add('events:PutRule');
        return this;
    }
    /**
     * Adds target(s) to a rule. Targets are the resources that can be invoked when a rule is triggered.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifTargetArn()
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_PutTargets.html
     */
    toPutTargets() {
        this.add('events:PutTargets');
        return this;
    }
    /**
     * Revokes the permission of another AWS account to be able to put events to your default event bus.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_RemovePermission.html
     */
    toRemovePermission() {
        this.add('events:RemovePermission');
        return this;
    }
    /**
     * Removes target(s) from a rule so that when the rule is triggered, those targets will no longer be invoked.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_RemoveTargets.html
     */
    toRemoveTargets() {
        this.add('events:RemoveTargets');
        return this;
    }
    /**
     * This action tags an Amazon EventBridge resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        this.add('events:TagResource');
        return this;
    }
    /**
     * Tests whether an event pattern matches the provided event.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_TestEventPattern.html
     */
    toTestEventPattern() {
        this.add('events:TestEventPattern');
        return this;
    }
    /**
     * This action removes a tag from an Amazon EventBridge resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        this.add('events:UntagResource');
        return this;
    }
    /**
     * Adds a resource of type event-source to the statement
     *
     * https://docs.aws.amazon.com/eventbridge/latest/userguide/iam-access-control-identity-based-eventbridge.html#eventbridge-arn-format
     *
     * @param eventSourceName - Identifier for the eventSourceName.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onEventSource(eventSourceName, region, partition) {
        var arn = 'arn:${Partition}:events:${Region}::event-source/${EventSourceName}';
        arn = arn.replace('${EventSourceName}', eventSourceName);
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type event-bus to the statement
     *
     * https://docs.aws.amazon.com/eventbridge/latest/userguide/iam-access-control-identity-based-eventbridge.html#eventbridge-arn-format
     *
     * @param eventBusName - Identifier for the eventBusName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onEventBus(eventBusName, account, region, partition) {
        var arn = 'arn:${Partition}:events:${Region}:${Account}:event-bus/${EventBusName}';
        arn = arn.replace('${EventBusName}', eventBusName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type rule to the statement
     *
     * https://docs.aws.amazon.com/eventbridge/latest/userguide/iam-access-control-identity-based-eventbridge.html#eventbridge-arn-format
     *
     * @param ruleName - Identifier for the ruleName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRule(ruleName, account, region, partition) {
        var arn = 'arn:${Partition}:events:${Region}:${Account}:rule/${RuleName}';
        arn = arn.replace('${RuleName}', ruleName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * The ARN of a target that can be put to a rule.
     *
     * https://docs.aws.amazon.com/eventbridge/latest/userguide/policy-keys-eventbridge.html#limiting-access-to-targets
     *
     * Applies to actions:
     * - .toPutTargets()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifTargetArn(value, operator) {
        return this.if(`events:TargetArn`, value, operator || 'ArnEquals');
    }
    /**
     * Matches the literal string of the detail-type filed of the event.
     *
     * https://docs.aws.amazon.com/eventbridge/latest/userguide/policy-keys-eventbridge.html#events-pattern-detail-type
     *
     * Applies to actions:
     * - .toPutRule()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDetailType(value, operator) {
        return this.if(`events:detail-type`, value, operator || 'StringLike');
    }
    /**
     * Matches the literal string for the detail.eventTypeCode field of the event.
     *
     * https://docs.aws.amazon.com/eventbridge/latest/userguide/policy-keys-eventbridge.html#limit-rule-by-type-code
     *
     * Applies to actions:
     * - .toPutRule()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDetailEventTypeCode(value, operator) {
        return this.if(`events:detail.eventTypeCode`, value, operator || 'StringLike');
    }
    /**
     * Matches the literal string for the detail.service field of the event.
     *
     * https://docs.aws.amazon.com/eventbridge/latest/userguide/policy-keys-eventbridge.html#limit-rule-by-service
     *
     * Applies to actions:
     * - .toPutRule()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDetailService(value, operator) {
        return this.if(`events:detail.service`, value, operator || 'StringLike');
    }
    /**
     * Matches the literal string for the detail.useridentity.principalid field of the event.
     *
     * https://docs.aws.amazon.com/eventbridge/latest/userguide/policy-keys-eventbridge.html#consume-specific-events
     *
     * Applies to actions:
     * - .toPutRule()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDetailUserIdentityPrincipalId(value, operator) {
        return this.if(`events:detail.userIdentity.principalId`, value, operator || 'StringLike');
    }
    /**
     * The AWS service or AWS partner event source that generated the event. Matches the literal string of the source field of the event.
     *
     * https://docs.aws.amazon.com/eventbridge/latest/userguide/policy-keys-eventbridge.html#events-limit-access-control
     *
     * Applies to actions:
     * - .toPutRule()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSource(value, operator) {
        return this.if(`events:source`, value, operator || 'StringLike');
    }
}
exports.Events = Events;
//# sourceMappingURL=data:application/json;base64,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