"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Frauddetector = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [frauddetector](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonfrauddetector.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Frauddetector extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [frauddetector](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonfrauddetector.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'frauddetector';
        this.actionList = {
            "BatchCreateVariable": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_BatchCreateVariable",
                "description": "Creates a batch of variables.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "BatchGetVariable": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_BatchGetVariable",
                "description": "Gets a batch of variables.",
                "accessLevel": "List",
                "resourceTypes": {
                    "variable": {
                        "required": false
                    }
                }
            },
            "CreateDetectorVersion": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_CreateDetectorVersion",
                "description": "Creates a detector version. The detector version starts in a DRAFT status.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateModel": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_CreateModel",
                "description": "Creates a model using the specified model type.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "model": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateModelVersion": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_CreateModelVersion",
                "description": "Creates a version of the model using the specified model type and model id.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "model": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateRule": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_CreateRule.html",
                "description": "Creates a rule for use with the specified detector.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateVariable": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_CreateVariable.html",
                "description": "Creates a variable.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "DeleteDetector": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_DeleteDetector",
                "description": "Deletes the detector. Before deleting a detector, you must first delete all detector versions and rule versions associated with the detector.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    }
                }
            },
            "DeleteDetectorVersion": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_DeleteDetectorVersion",
                "description": "Deletes the detector version. You cannot delete detector versions that are in ACTIVE status.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "detector-version": {
                        "required": true
                    }
                }
            },
            "DeleteEvent": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_DeleteEvent",
                "description": "Deletes the specified event.",
                "accessLevel": "Write"
            },
            "DeleteRule": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_DeleteRule",
                "description": "Deletes the rule. You cannot delete a rule if it is used by an ACTIVE or INACTIVE detector version.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "rule": {
                        "required": true
                    }
                }
            },
            "DescribeDetector": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_DescribeDetector",
                "description": "Gets all versions for a specified detector.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    }
                }
            },
            "DescribeModelVersions": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_DescribeModelVersions",
                "description": "Gets all of the model versions for the specified model type or for the specified model type and model ID. You can also get details for a single, specified model version.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "model-version": {
                        "required": false
                    }
                }
            },
            "GetDetectorVersion": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_GetDetectorVersion",
                "description": "Gets a particular detector version.",
                "accessLevel": "List",
                "resourceTypes": {
                    "detector-version": {
                        "required": true
                    }
                }
            },
            "GetDetectors": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_GetDetectors",
                "description": "Gets all detectors or a single detector if a detectorId is specified. This is a paginated API. If you provide a null maxResults, this action retrieves a maximum of 10 records per page. If you provide a maxResults, the value must be between 5 and 10. To get the next page results, provide the pagination token from the GetDetectorsResponse as part of your request. A null pagination token fetches the records from the beginning.",
                "accessLevel": "List",
                "resourceTypes": {
                    "detector": {
                        "required": false
                    }
                }
            },
            "GetEntityTypes": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_GetEntityTypes",
                "description": "Gets all entity types or a specific entity type if a name is specified. This is a paginated API. If you provide a null maxResults, this action retrieves a maximum of 10 records per page. If you provide a maxResults, the value must be between 5 and 10. To get the next page results, provide the pagination token from the GetEntityTypesResponse as part of your request. A null pagination token fetches the records from the beginning.",
                "accessLevel": "List",
                "resourceTypes": {
                    "entity-type": {
                        "required": false
                    }
                }
            },
            "GetEventPrediction": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_GetEventPrediction",
                "description": "Evaluates an event against a detector version. If a version ID is not provided, the detector’s (ACTIVE) version is used.",
                "accessLevel": "Read"
            },
            "GetEventTypes": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_GetEventTypes",
                "description": "Gets all event types or a specific event type if name is provided. This is a paginated API. If you provide a null maxResults, this action retrieves a maximum of 10 records per page. If you provide a maxResults, the value must be between 5 and 10. To get the next page results, provide the pagination token from the GetEventTypesResponse as part of your request. A null pagination token fetches the records from the beginning.",
                "accessLevel": "List",
                "resourceTypes": {
                    "event-type": {
                        "required": false
                    }
                }
            },
            "GetExternalModels": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_GetExternalModels",
                "description": "Gets the details for one or more Amazon SageMaker models that have been imported into the service. This is a paginated API. If you provide a null maxResults, this actions retrieves a maximum of 10 records per page. If you provide a maxResults, the value must be between 5 and 10. To get the next page results, provide the pagination token from the GetExternalModelsResult as part of your request. A null pagination token fetches the records from the beginning.",
                "accessLevel": "List",
                "resourceTypes": {
                    "external-model": {
                        "required": false
                    }
                }
            },
            "GetKMSEncryptionKey": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_GetKMSEncryptionKey",
                "description": "Gets the encryption key if a Key Management Service (KMS) customer master key (CMK) has been specified to be used to encrypt content in Amazon Fraud Detector.",
                "accessLevel": "Read"
            },
            "GetLabels": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_GetLabels",
                "description": "Gets all labels or a specific label if name is provided. This is a paginated API. If you provide a null maxResults, this action retrieves a maximum of 50 records per page. If you provide a maxResults, the value must be between 10 and 50. To get the next page results, provide the pagination token from the GetGetLabelsResponse as part of your request. A null pagination token fetches the records from the beginning.",
                "accessLevel": "List",
                "resourceTypes": {
                    "label": {
                        "required": false
                    }
                }
            },
            "GetModelVersion": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_GetModelVersion",
                "description": "Gets the details of the specified model version.",
                "accessLevel": "List",
                "resourceTypes": {
                    "model-version": {
                        "required": true
                    }
                }
            },
            "GetModels": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_GetModels",
                "description": "Gets one or more models. Gets all models for the AWS account if no model type and no model id provided. Gets all models for the AWS account and model type, if the model type is specified but model id is not provided. Gets a specific model if (model type, model id) tuple is specified.",
                "accessLevel": "List",
                "resourceTypes": {
                    "model": {
                        "required": false
                    }
                }
            },
            "GetOutcomes": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_GetOutcomes",
                "description": "Gets one or more outcomes. This is a paginated API. If you provide a null maxResults, this actions retrieves a maximum of 100 records per page. If you provide a maxResults, the value must be between 50 and 100. To get the next page results, provide the pagination token from the GetOutcomesResult as part of your request. A null pagination token fetches the records from the beginning.",
                "accessLevel": "List",
                "resourceTypes": {
                    "outcome": {
                        "required": false
                    }
                }
            },
            "GetRules": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_GetRules.html",
                "description": "Get all rules for a detector (paginated) if ruleId and ruleVersion are not specified. Gets all rules for the detector and the ruleId if present (paginated). Gets a specific rule if both the ruleId and the ruleVersion are specified.",
                "accessLevel": "List",
                "resourceTypes": {
                    "rule": {
                        "required": false
                    }
                }
            },
            "GetVariables": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_GetVariables",
                "description": "Gets all of the variables or the specific variable. This is a paginated API. Providing null maxSizePerPage results in retrieving maximum of 100 records per page. If you provide maxSizePerPage the value must be between 50 and 100. To get the next page result, a provide a pagination token from GetVariablesResult as part of your request. Null pagination token fetches the records from the beginning.",
                "accessLevel": "List",
                "resourceTypes": {
                    "variable": {
                        "required": false
                    }
                }
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_ListTagsForResource.html",
                "description": "Lists all tags associated with the resource. This is a paginated API. To get the next page results, provide the pagination token from the response as part of your request. A null pagination token fetches the records from the beginning.",
                "accessLevel": "List",
                "resourceTypes": {
                    "detector": {
                        "required": false
                    },
                    "detector-version": {
                        "required": false
                    },
                    "entity-type": {
                        "required": false
                    },
                    "event-type": {
                        "required": false
                    },
                    "external-model": {
                        "required": false
                    },
                    "label": {
                        "required": false
                    },
                    "model": {
                        "required": false
                    },
                    "model-version": {
                        "required": false
                    },
                    "outcome": {
                        "required": false
                    },
                    "rule": {
                        "required": false
                    },
                    "variable": {
                        "required": false
                    }
                }
            },
            "PutDetector": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_PutDetector.html",
                "description": "Creates or updates a detector.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "PutEntityType": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_PutEntityType.html",
                "description": "Creates or updates an entity type. An entity represents who is performing the event. As part of a fraud prediction, you pass the entity ID to indicate the specific entity who performed the event. An entity type classifies the entity. Example classifications include customer, merchant, or account.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "entity-type": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "PutEventType": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_PutEventType.html",
                "description": "Creates or updates an event type. An event is a business activity that is evaluated for fraud risk. With Amazon Fraud Detector, you generate fraud predictions for events. An event type defines the structure for an event sent to Amazon Fraud Detector. This includes the variables sent as part of the event, the entity performing the event (such as a customer), and the labels that classify the event. Example event types include online payment transactions, account registrations, and authentications.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "event-type": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "PutExternalModel": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_PutExternalModel.html",
                "description": "Creates or updates an Amazon SageMaker model endpoint. You can also use this action to update the configuration of the model endpoint, including the IAM role and/or the mapped variables.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "external-model": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "PutKMSEncryptionKey": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_PutKMSEncryptionKey.html",
                "description": "Specifies the Key Management Service (KMS) customer master key (CMK) to be used to encrypt content in Amazon Fraud Detector.",
                "accessLevel": "Write"
            },
            "PutLabel": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_PutLabel.html",
                "description": "Creates or updates label. A label classifies an event as fraudulent or legitimate. Labels are associated with event types and used to train supervised machine learning models in Amazon Fraud Detector.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "label": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "PutOutcome": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_PutOutcome.html",
                "description": "Creates or updates an outcome.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "outcome": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_TagResource.html",
                "description": "Assigns tags to a resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "detector": {
                        "required": false
                    },
                    "detector-version": {
                        "required": false
                    },
                    "entity-type": {
                        "required": false
                    },
                    "event-type": {
                        "required": false
                    },
                    "external-model": {
                        "required": false
                    },
                    "label": {
                        "required": false
                    },
                    "model": {
                        "required": false
                    },
                    "model-version": {
                        "required": false
                    },
                    "outcome": {
                        "required": false
                    },
                    "rule": {
                        "required": false
                    },
                    "variable": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_UntagResource.html",
                "description": "Removes tags from a resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "detector": {
                        "required": false
                    },
                    "detector-version": {
                        "required": false
                    },
                    "entity-type": {
                        "required": false
                    },
                    "event-type": {
                        "required": false
                    },
                    "external-model": {
                        "required": false
                    },
                    "label": {
                        "required": false
                    },
                    "model": {
                        "required": false
                    },
                    "model-version": {
                        "required": false
                    },
                    "outcome": {
                        "required": false
                    },
                    "rule": {
                        "required": false
                    },
                    "variable": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "UpdateDetectorVersion": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_UpdateDetectorVersion",
                "description": "Updates a detector version. The detector version attributes that you can update include models, external model endpoints, rules, rule execution mode, and description. You can only update a DRAFT detector version.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "detector": {
                        "required": true
                    }
                }
            },
            "UpdateDetectorVersionMetadata": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_UpdateDetectorVersionMetadata",
                "description": "Updates the detector version's description. You can update the metadata for any detector version (DRAFT, ACTIVE, or INACTIVE).",
                "accessLevel": "Write",
                "resourceTypes": {
                    "detector-version": {
                        "required": true
                    }
                }
            },
            "UpdateDetectorVersionStatus": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_UpdateDetectorVersionStatus",
                "description": "Updates the detector version’s status. You can perform the following promotions or demotions using UpdateDetectorVersionStatus: DRAFT to ACTIVE, ACTIVE to INACTIVE, and INACTIVE to ACTIVE.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "detector-version": {
                        "required": true
                    }
                }
            },
            "UpdateModel": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_UpdateModel",
                "description": "Updates a model. You can update the description attribute using this action.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "model": {
                        "required": true
                    }
                }
            },
            "UpdateModelVersion": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_UpdateModelVersion",
                "description": "Updates a model version. Updating a model version retrains an existing model version using updated training data and produces a new minor version of the model. You can update the training data set location and data access role attributes using this action. This action creates and trains a new minor version of the model, for example version 1.01, 1.02, 1.03.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "model": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UpdateModelVersionStatus": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_UpdateModelVersionStatus",
                "description": "Updates the status of a model version.",
                "accessLevel": "Write"
            },
            "UpdateRuleMetadata": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_UpdateRuleMetadata.html",
                "description": "Updates a rule's metadata. The description attribute can be updated.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "rule": {
                        "required": true
                    }
                }
            },
            "UpdateRuleVersion": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_UpdateRuleVersion.html",
                "description": "Updates a rule version resulting in a new rule version. Updates a rule version resulting in a new rule version (version 1, 2, 3 ...).",
                "accessLevel": "Write",
                "resourceTypes": {
                    "rule": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UpdateVariable": {
                "url": "https://docs.aws.amazon.com/frauddetector/latest/api/API_UpdateVariable.html",
                "description": "Updates a variable.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "variable": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "detector": {
                "name": "detector",
                "url": "",
                "arn": "arn:${Partition}:frauddetector:${Region}:${Account}:detector/${ResourcePath}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "detector-version": {
                "name": "detector-version",
                "url": "",
                "arn": "arn:${Partition}:frauddetector:${Region}:${Account}:detector-version/${ResourcePath}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "entity-type": {
                "name": "entity-type",
                "url": "",
                "arn": "arn:${Partition}:frauddetector:${Region}:${Account}:entity-type/${ResourcePath}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "external-model": {
                "name": "external-model",
                "url": "",
                "arn": "arn:${Partition}:frauddetector:${Region}:${Account}:external-model/${ResourcePath}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "event-type": {
                "name": "event-type",
                "url": "",
                "arn": "arn:${Partition}:frauddetector:${Region}:${Account}:event-type/${ResourcePath}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "label": {
                "name": "label",
                "url": "",
                "arn": "arn:${Partition}:frauddetector:${Region}:${Account}:label/${ResourcePath}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "model": {
                "name": "model",
                "url": "",
                "arn": "arn:${Partition}:frauddetector:${Region}:${Account}:model/${ResourcePath}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "model-version": {
                "name": "model-version",
                "url": "",
                "arn": "arn:${Partition}:frauddetector:${Region}:${Account}:model-version/${ResourcePath}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "outcome": {
                "name": "outcome",
                "url": "",
                "arn": "arn:${Partition}:frauddetector:${Region}:${Account}:outcome/${ResourcePath}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "rule": {
                "name": "rule",
                "url": "",
                "arn": "arn:${Partition}:frauddetector:${Region}:${Account}:rule/${ResourcePath}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "variable": {
                "name": "variable",
                "url": "",
                "arn": "arn:${Partition}:frauddetector:${Region}:${Account}:variable/${ResourcePath}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Creates a batch of variables.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_BatchCreateVariable
     */
    toBatchCreateVariable() {
        this.add('frauddetector:BatchCreateVariable');
        return this;
    }
    /**
     * Gets a batch of variables.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_BatchGetVariable
     */
    toBatchGetVariable() {
        this.add('frauddetector:BatchGetVariable');
        return this;
    }
    /**
     * Creates a detector version. The detector version starts in a DRAFT status.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_CreateDetectorVersion
     */
    toCreateDetectorVersion() {
        this.add('frauddetector:CreateDetectorVersion');
        return this;
    }
    /**
     * Creates a model using the specified model type.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_CreateModel
     */
    toCreateModel() {
        this.add('frauddetector:CreateModel');
        return this;
    }
    /**
     * Creates a version of the model using the specified model type and model id.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_CreateModelVersion
     */
    toCreateModelVersion() {
        this.add('frauddetector:CreateModelVersion');
        return this;
    }
    /**
     * Creates a rule for use with the specified detector.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_CreateRule.html
     */
    toCreateRule() {
        this.add('frauddetector:CreateRule');
        return this;
    }
    /**
     * Creates a variable.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_CreateVariable.html
     */
    toCreateVariable() {
        this.add('frauddetector:CreateVariable');
        return this;
    }
    /**
     * Deletes the detector. Before deleting a detector, you must first delete all detector versions and rule versions associated with the detector.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_DeleteDetector
     */
    toDeleteDetector() {
        this.add('frauddetector:DeleteDetector');
        return this;
    }
    /**
     * Deletes the detector version. You cannot delete detector versions that are in ACTIVE status.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_DeleteDetectorVersion
     */
    toDeleteDetectorVersion() {
        this.add('frauddetector:DeleteDetectorVersion');
        return this;
    }
    /**
     * Deletes the specified event.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_DeleteEvent
     */
    toDeleteEvent() {
        this.add('frauddetector:DeleteEvent');
        return this;
    }
    /**
     * Deletes the rule. You cannot delete a rule if it is used by an ACTIVE or INACTIVE detector version.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_DeleteRule
     */
    toDeleteRule() {
        this.add('frauddetector:DeleteRule');
        return this;
    }
    /**
     * Gets all versions for a specified detector.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_DescribeDetector
     */
    toDescribeDetector() {
        this.add('frauddetector:DescribeDetector');
        return this;
    }
    /**
     * Gets all of the model versions for the specified model type or for the specified model type and model ID. You can also get details for a single, specified model version.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_DescribeModelVersions
     */
    toDescribeModelVersions() {
        this.add('frauddetector:DescribeModelVersions');
        return this;
    }
    /**
     * Gets a particular detector version.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_GetDetectorVersion
     */
    toGetDetectorVersion() {
        this.add('frauddetector:GetDetectorVersion');
        return this;
    }
    /**
     * Gets all detectors or a single detector if a detectorId is specified. This is a paginated API. If you provide a null maxResults, this action retrieves a maximum of 10 records per page. If you provide a maxResults, the value must be between 5 and 10. To get the next page results, provide the pagination token from the GetDetectorsResponse as part of your request. A null pagination token fetches the records from the beginning.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_GetDetectors
     */
    toGetDetectors() {
        this.add('frauddetector:GetDetectors');
        return this;
    }
    /**
     * Gets all entity types or a specific entity type if a name is specified. This is a paginated API. If you provide a null maxResults, this action retrieves a maximum of 10 records per page. If you provide a maxResults, the value must be between 5 and 10. To get the next page results, provide the pagination token from the GetEntityTypesResponse as part of your request. A null pagination token fetches the records from the beginning.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_GetEntityTypes
     */
    toGetEntityTypes() {
        this.add('frauddetector:GetEntityTypes');
        return this;
    }
    /**
     * Evaluates an event against a detector version. If a version ID is not provided, the detector’s (ACTIVE) version is used.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_GetEventPrediction
     */
    toGetEventPrediction() {
        this.add('frauddetector:GetEventPrediction');
        return this;
    }
    /**
     * Gets all event types or a specific event type if name is provided. This is a paginated API. If you provide a null maxResults, this action retrieves a maximum of 10 records per page. If you provide a maxResults, the value must be between 5 and 10. To get the next page results, provide the pagination token from the GetEventTypesResponse as part of your request. A null pagination token fetches the records from the beginning.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_GetEventTypes
     */
    toGetEventTypes() {
        this.add('frauddetector:GetEventTypes');
        return this;
    }
    /**
     * Gets the details for one or more Amazon SageMaker models that have been imported into the service. This is a paginated API. If you provide a null maxResults, this actions retrieves a maximum of 10 records per page. If you provide a maxResults, the value must be between 5 and 10. To get the next page results, provide the pagination token from the GetExternalModelsResult as part of your request. A null pagination token fetches the records from the beginning.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_GetExternalModels
     */
    toGetExternalModels() {
        this.add('frauddetector:GetExternalModels');
        return this;
    }
    /**
     * Gets the encryption key if a Key Management Service (KMS) customer master key (CMK) has been specified to be used to encrypt content in Amazon Fraud Detector.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_GetKMSEncryptionKey
     */
    toGetKMSEncryptionKey() {
        this.add('frauddetector:GetKMSEncryptionKey');
        return this;
    }
    /**
     * Gets all labels or a specific label if name is provided. This is a paginated API. If you provide a null maxResults, this action retrieves a maximum of 50 records per page. If you provide a maxResults, the value must be between 10 and 50. To get the next page results, provide the pagination token from the GetGetLabelsResponse as part of your request. A null pagination token fetches the records from the beginning.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_GetLabels
     */
    toGetLabels() {
        this.add('frauddetector:GetLabels');
        return this;
    }
    /**
     * Gets the details of the specified model version.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_GetModelVersion
     */
    toGetModelVersion() {
        this.add('frauddetector:GetModelVersion');
        return this;
    }
    /**
     * Gets one or more models. Gets all models for the AWS account if no model type and no model id provided. Gets all models for the AWS account and model type, if the model type is specified but model id is not provided. Gets a specific model if (model type, model id) tuple is specified.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_GetModels
     */
    toGetModels() {
        this.add('frauddetector:GetModels');
        return this;
    }
    /**
     * Gets one or more outcomes. This is a paginated API. If you provide a null maxResults, this actions retrieves a maximum of 100 records per page. If you provide a maxResults, the value must be between 50 and 100. To get the next page results, provide the pagination token from the GetOutcomesResult as part of your request. A null pagination token fetches the records from the beginning.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_GetOutcomes
     */
    toGetOutcomes() {
        this.add('frauddetector:GetOutcomes');
        return this;
    }
    /**
     * Get all rules for a detector (paginated) if ruleId and ruleVersion are not specified. Gets all rules for the detector and the ruleId if present (paginated). Gets a specific rule if both the ruleId and the ruleVersion are specified.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_GetRules.html
     */
    toGetRules() {
        this.add('frauddetector:GetRules');
        return this;
    }
    /**
     * Gets all of the variables or the specific variable. This is a paginated API. Providing null maxSizePerPage results in retrieving maximum of 100 records per page. If you provide maxSizePerPage the value must be between 50 and 100. To get the next page result, a provide a pagination token from GetVariablesResult as part of your request. Null pagination token fetches the records from the beginning.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_GetVariables
     */
    toGetVariables() {
        this.add('frauddetector:GetVariables');
        return this;
    }
    /**
     * Lists all tags associated with the resource. This is a paginated API. To get the next page results, provide the pagination token from the response as part of your request. A null pagination token fetches the records from the beginning.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.add('frauddetector:ListTagsForResource');
        return this;
    }
    /**
     * Creates or updates a detector.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_PutDetector.html
     */
    toPutDetector() {
        this.add('frauddetector:PutDetector');
        return this;
    }
    /**
     * Creates or updates an entity type. An entity represents who is performing the event. As part of a fraud prediction, you pass the entity ID to indicate the specific entity who performed the event. An entity type classifies the entity. Example classifications include customer, merchant, or account.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_PutEntityType.html
     */
    toPutEntityType() {
        this.add('frauddetector:PutEntityType');
        return this;
    }
    /**
     * Creates or updates an event type. An event is a business activity that is evaluated for fraud risk. With Amazon Fraud Detector, you generate fraud predictions for events. An event type defines the structure for an event sent to Amazon Fraud Detector. This includes the variables sent as part of the event, the entity performing the event (such as a customer), and the labels that classify the event. Example event types include online payment transactions, account registrations, and authentications.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_PutEventType.html
     */
    toPutEventType() {
        this.add('frauddetector:PutEventType');
        return this;
    }
    /**
     * Creates or updates an Amazon SageMaker model endpoint. You can also use this action to update the configuration of the model endpoint, including the IAM role and/or the mapped variables.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_PutExternalModel.html
     */
    toPutExternalModel() {
        this.add('frauddetector:PutExternalModel');
        return this;
    }
    /**
     * Specifies the Key Management Service (KMS) customer master key (CMK) to be used to encrypt content in Amazon Fraud Detector.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_PutKMSEncryptionKey.html
     */
    toPutKMSEncryptionKey() {
        this.add('frauddetector:PutKMSEncryptionKey');
        return this;
    }
    /**
     * Creates or updates label. A label classifies an event as fraudulent or legitimate. Labels are associated with event types and used to train supervised machine learning models in Amazon Fraud Detector.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_PutLabel.html
     */
    toPutLabel() {
        this.add('frauddetector:PutLabel');
        return this;
    }
    /**
     * Creates or updates an outcome.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_PutOutcome.html
     */
    toPutOutcome() {
        this.add('frauddetector:PutOutcome');
        return this;
    }
    /**
     * Assigns tags to a resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_TagResource.html
     */
    toTagResource() {
        this.add('frauddetector:TagResource');
        return this;
    }
    /**
     * Removes tags from a resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_UntagResource.html
     */
    toUntagResource() {
        this.add('frauddetector:UntagResource');
        return this;
    }
    /**
     * Updates a detector version. The detector version attributes that you can update include models, external model endpoints, rules, rule execution mode, and description. You can only update a DRAFT detector version.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_UpdateDetectorVersion
     */
    toUpdateDetectorVersion() {
        this.add('frauddetector:UpdateDetectorVersion');
        return this;
    }
    /**
     * Updates the detector version's description. You can update the metadata for any detector version (DRAFT, ACTIVE, or INACTIVE).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_UpdateDetectorVersionMetadata
     */
    toUpdateDetectorVersionMetadata() {
        this.add('frauddetector:UpdateDetectorVersionMetadata');
        return this;
    }
    /**
     * Updates the detector version’s status. You can perform the following promotions or demotions using UpdateDetectorVersionStatus: DRAFT to ACTIVE, ACTIVE to INACTIVE, and INACTIVE to ACTIVE.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_UpdateDetectorVersionStatus
     */
    toUpdateDetectorVersionStatus() {
        this.add('frauddetector:UpdateDetectorVersionStatus');
        return this;
    }
    /**
     * Updates a model. You can update the description attribute using this action.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_UpdateModel
     */
    toUpdateModel() {
        this.add('frauddetector:UpdateModel');
        return this;
    }
    /**
     * Updates a model version. Updating a model version retrains an existing model version using updated training data and produces a new minor version of the model. You can update the training data set location and data access role attributes using this action. This action creates and trains a new minor version of the model, for example version 1.01, 1.02, 1.03.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_UpdateModelVersion
     */
    toUpdateModelVersion() {
        this.add('frauddetector:UpdateModelVersion');
        return this;
    }
    /**
     * Updates the status of a model version.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_UpdateModelVersionStatus
     */
    toUpdateModelVersionStatus() {
        this.add('frauddetector:UpdateModelVersionStatus');
        return this;
    }
    /**
     * Updates a rule's metadata. The description attribute can be updated.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_UpdateRuleMetadata.html
     */
    toUpdateRuleMetadata() {
        this.add('frauddetector:UpdateRuleMetadata');
        return this;
    }
    /**
     * Updates a rule version resulting in a new rule version. Updates a rule version resulting in a new rule version (version 1, 2, 3 ...).
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_UpdateRuleVersion.html
     */
    toUpdateRuleVersion() {
        this.add('frauddetector:UpdateRuleVersion');
        return this;
    }
    /**
     * Updates a variable.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_UpdateVariable.html
     */
    toUpdateVariable() {
        this.add('frauddetector:UpdateVariable');
        return this;
    }
    /**
     * Adds a resource of type detector to the statement
     *
     * @param resourcePath - Identifier for the resourcePath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDetector(resourcePath, account, region, partition) {
        var arn = 'arn:${Partition}:frauddetector:${Region}:${Account}:detector/${ResourcePath}';
        arn = arn.replace('${ResourcePath}', resourcePath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type detector-version to the statement
     *
     * @param resourcePath - Identifier for the resourcePath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDetectorVersion(resourcePath, account, region, partition) {
        var arn = 'arn:${Partition}:frauddetector:${Region}:${Account}:detector-version/${ResourcePath}';
        arn = arn.replace('${ResourcePath}', resourcePath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type entity-type to the statement
     *
     * @param resourcePath - Identifier for the resourcePath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onEntityType(resourcePath, account, region, partition) {
        var arn = 'arn:${Partition}:frauddetector:${Region}:${Account}:entity-type/${ResourcePath}';
        arn = arn.replace('${ResourcePath}', resourcePath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type external-model to the statement
     *
     * @param resourcePath - Identifier for the resourcePath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onExternalModel(resourcePath, account, region, partition) {
        var arn = 'arn:${Partition}:frauddetector:${Region}:${Account}:external-model/${ResourcePath}';
        arn = arn.replace('${ResourcePath}', resourcePath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type event-type to the statement
     *
     * @param resourcePath - Identifier for the resourcePath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onEventType(resourcePath, account, region, partition) {
        var arn = 'arn:${Partition}:frauddetector:${Region}:${Account}:event-type/${ResourcePath}';
        arn = arn.replace('${ResourcePath}', resourcePath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type label to the statement
     *
     * @param resourcePath - Identifier for the resourcePath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onLabel(resourcePath, account, region, partition) {
        var arn = 'arn:${Partition}:frauddetector:${Region}:${Account}:label/${ResourcePath}';
        arn = arn.replace('${ResourcePath}', resourcePath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type model to the statement
     *
     * @param resourcePath - Identifier for the resourcePath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onModel(resourcePath, account, region, partition) {
        var arn = 'arn:${Partition}:frauddetector:${Region}:${Account}:model/${ResourcePath}';
        arn = arn.replace('${ResourcePath}', resourcePath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type model-version to the statement
     *
     * @param resourcePath - Identifier for the resourcePath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onModelVersion(resourcePath, account, region, partition) {
        var arn = 'arn:${Partition}:frauddetector:${Region}:${Account}:model-version/${ResourcePath}';
        arn = arn.replace('${ResourcePath}', resourcePath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type outcome to the statement
     *
     * @param resourcePath - Identifier for the resourcePath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onOutcome(resourcePath, account, region, partition) {
        var arn = 'arn:${Partition}:frauddetector:${Region}:${Account}:outcome/${ResourcePath}';
        arn = arn.replace('${ResourcePath}', resourcePath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type rule to the statement
     *
     * @param resourcePath - Identifier for the resourcePath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRule(resourcePath, account, region, partition) {
        var arn = 'arn:${Partition}:frauddetector:${Region}:${Account}:rule/${ResourcePath}';
        arn = arn.replace('${ResourcePath}', resourcePath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type variable to the statement
     *
     * @param resourcePath - Identifier for the resourcePath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onVariable(resourcePath, account, region, partition) {
        var arn = 'arn:${Partition}:frauddetector:${Region}:${Account}:variable/${ResourcePath}';
        arn = arn.replace('${ResourcePath}', resourcePath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Frauddetector = Frauddetector;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYW1hem9uZnJhdWRkZXRlY3Rvci5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImFtYXpvbmZyYXVkZGV0ZWN0b3IudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQUEsc0NBQW9FO0FBRXBFOzs7O0dBSUc7QUFDSCxNQUFhLGFBQWMsU0FBUSx3QkFBZTtJQXdwQmhEOzs7O09BSUc7SUFDSCxZQUFhLEdBQVk7UUFDdkIsS0FBSyxDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBN3BCTixrQkFBYSxHQUFHLGVBQWUsQ0FBQztRQUM3QixlQUFVLEdBQVk7WUFDOUIscUJBQXFCLEVBQUU7Z0JBQ3JCLEtBQUssRUFBRSw4RUFBOEU7Z0JBQ3JGLGFBQWEsRUFBRSwrQkFBK0I7Z0JBQzlDLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixZQUFZLEVBQUU7b0JBQ1osMEJBQTBCO29CQUMxQixhQUFhO2lCQUNkO2FBQ0Y7WUFDRCxrQkFBa0IsRUFBRTtnQkFDbEIsS0FBSyxFQUFFLDJFQUEyRTtnQkFDbEYsYUFBYSxFQUFFLDRCQUE0QjtnQkFDM0MsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixVQUFVLEVBQUU7d0JBQ1YsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCx1QkFBdUIsRUFBRTtnQkFDdkIsS0FBSyxFQUFFLGdGQUFnRjtnQkFDdkYsYUFBYSxFQUFFLDRFQUE0RTtnQkFDM0YsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixVQUFVLEVBQUU7d0JBQ1YsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2dCQUNELFlBQVksRUFBRTtvQkFDWiwwQkFBMEI7b0JBQzFCLGFBQWE7aUJBQ2Q7YUFDRjtZQUNELGFBQWEsRUFBRTtnQkFDYixLQUFLLEVBQUUsc0VBQXNFO2dCQUM3RSxhQUFhLEVBQUUsaURBQWlEO2dCQUNoRSxhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLE9BQU8sRUFBRTt3QkFDUCxVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7Z0JBQ0QsWUFBWSxFQUFFO29CQUNaLDBCQUEwQjtvQkFDMUIsYUFBYTtpQkFDZDthQUNGO1lBQ0Qsb0JBQW9CLEVBQUU7Z0JBQ3BCLEtBQUssRUFBRSw2RUFBNkU7Z0JBQ3BGLGFBQWEsRUFBRSw2RUFBNkU7Z0JBQzVGLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsT0FBTyxFQUFFO3dCQUNQLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjtnQkFDRCxZQUFZLEVBQUU7b0JBQ1osMEJBQTBCO29CQUMxQixhQUFhO2lCQUNkO2FBQ0Y7WUFDRCxZQUFZLEVBQUU7Z0JBQ1osS0FBSyxFQUFFLDBFQUEwRTtnQkFDakYsYUFBYSxFQUFFLHFEQUFxRDtnQkFDcEUsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixVQUFVLEVBQUU7d0JBQ1YsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2dCQUNELFlBQVksRUFBRTtvQkFDWiwwQkFBMEI7b0JBQzFCLGFBQWE7aUJBQ2Q7YUFDRjtZQUNELGdCQUFnQixFQUFFO2dCQUNoQixLQUFLLEVBQUUsOEVBQThFO2dCQUNyRixhQUFhLEVBQUUscUJBQXFCO2dCQUNwQyxhQUFhLEVBQUUsT0FBTztnQkFDdEIsWUFBWSxFQUFFO29CQUNaLDBCQUEwQjtvQkFDMUIsYUFBYTtpQkFDZDthQUNGO1lBQ0QsZ0JBQWdCLEVBQUU7Z0JBQ2hCLEtBQUssRUFBRSx5RUFBeUU7Z0JBQ2hGLGFBQWEsRUFBRSwrSUFBK0k7Z0JBQzlKLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsVUFBVSxFQUFFO3dCQUNWLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsdUJBQXVCLEVBQUU7Z0JBQ3ZCLEtBQUssRUFBRSxnRkFBZ0Y7Z0JBQ3ZGLGFBQWEsRUFBRSw4RkFBOEY7Z0JBQzdHLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2Ysa0JBQWtCLEVBQUU7d0JBQ2xCLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsYUFBYSxFQUFFO2dCQUNiLEtBQUssRUFBRSxzRUFBc0U7Z0JBQzdFLGFBQWEsRUFBRSw4QkFBOEI7Z0JBQzdDLGFBQWEsRUFBRSxPQUFPO2FBQ3ZCO1lBQ0QsWUFBWSxFQUFFO2dCQUNaLEtBQUssRUFBRSxxRUFBcUU7Z0JBQzVFLGFBQWEsRUFBRSxxR0FBcUc7Z0JBQ3BILGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsTUFBTSxFQUFFO3dCQUNOLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0Qsa0JBQWtCLEVBQUU7Z0JBQ2xCLEtBQUssRUFBRSwyRUFBMkU7Z0JBQ2xGLGFBQWEsRUFBRSw2Q0FBNkM7Z0JBQzVELGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsVUFBVSxFQUFFO3dCQUNWLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsdUJBQXVCLEVBQUU7Z0JBQ3ZCLEtBQUssRUFBRSxnRkFBZ0Y7Z0JBQ3ZGLGFBQWEsRUFBRSwyS0FBMks7Z0JBQzFMLGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsZUFBZSxFQUFFO3dCQUNmLFVBQVUsRUFBRSxLQUFLO3FCQUNsQjtpQkFDRjthQUNGO1lBQ0Qsb0JBQW9CLEVBQUU7Z0JBQ3BCLEtBQUssRUFBRSw2RUFBNkU7Z0JBQ3BGLGFBQWEsRUFBRSxxQ0FBcUM7Z0JBQ3BELGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2Ysa0JBQWtCLEVBQUU7d0JBQ2xCLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsY0FBYyxFQUFFO2dCQUNkLEtBQUssRUFBRSx1RUFBdUU7Z0JBQzlFLGFBQWEsRUFBRSw2YUFBNmE7Z0JBQzViLGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsVUFBVSxFQUFFO3dCQUNWLFVBQVUsRUFBRSxLQUFLO3FCQUNsQjtpQkFDRjthQUNGO1lBQ0QsZ0JBQWdCLEVBQUU7Z0JBQ2hCLEtBQUssRUFBRSx5RUFBeUU7Z0JBQ2hGLGFBQWEsRUFBRSxpYkFBaWI7Z0JBQ2hjLGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsYUFBYSxFQUFFO3dCQUNiLFVBQVUsRUFBRSxLQUFLO3FCQUNsQjtpQkFDRjthQUNGO1lBQ0Qsb0JBQW9CLEVBQUU7Z0JBQ3BCLEtBQUssRUFBRSw2RUFBNkU7Z0JBQ3BGLGFBQWEsRUFBRSwwSEFBMEg7Z0JBQ3pJLGFBQWEsRUFBRSxNQUFNO2FBQ3RCO1lBQ0QsZUFBZSxFQUFFO2dCQUNmLEtBQUssRUFBRSx3RUFBd0U7Z0JBQy9FLGFBQWEsRUFBRSwyYUFBMmE7Z0JBQzFiLGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsWUFBWSxFQUFFO3dCQUNaLFVBQVUsRUFBRSxLQUFLO3FCQUNsQjtpQkFDRjthQUNGO1lBQ0QsbUJBQW1CLEVBQUU7Z0JBQ25CLEtBQUssRUFBRSw0RUFBNEU7Z0JBQ25GLGFBQWEsRUFBRSw4Y0FBOGM7Z0JBQzdkLGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsZ0JBQWdCLEVBQUU7d0JBQ2hCLFVBQVUsRUFBRSxLQUFLO3FCQUNsQjtpQkFDRjthQUNGO1lBQ0QscUJBQXFCLEVBQUU7Z0JBQ3JCLEtBQUssRUFBRSw4RUFBOEU7Z0JBQ3JGLGFBQWEsRUFBRSxnS0FBZ0s7Z0JBQy9LLGFBQWEsRUFBRSxNQUFNO2FBQ3RCO1lBQ0QsV0FBVyxFQUFFO2dCQUNYLEtBQUssRUFBRSxvRUFBb0U7Z0JBQzNFLGFBQWEsRUFBRSxpYUFBaWE7Z0JBQ2hiLGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsT0FBTyxFQUFFO3dCQUNQLFVBQVUsRUFBRSxLQUFLO3FCQUNsQjtpQkFDRjthQUNGO1lBQ0QsaUJBQWlCLEVBQUU7Z0JBQ2pCLEtBQUssRUFBRSwwRUFBMEU7Z0JBQ2pGLGFBQWEsRUFBRSxrREFBa0Q7Z0JBQ2pFLGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsZUFBZSxFQUFFO3dCQUNmLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsV0FBVyxFQUFFO2dCQUNYLEtBQUssRUFBRSxvRUFBb0U7Z0JBQzNFLGFBQWEsRUFBRSw4UkFBOFI7Z0JBQzdTLGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsT0FBTyxFQUFFO3dCQUNQLFVBQVUsRUFBRSxLQUFLO3FCQUNsQjtpQkFDRjthQUNGO1lBQ0QsYUFBYSxFQUFFO2dCQUNiLEtBQUssRUFBRSxzRUFBc0U7Z0JBQzdFLGFBQWEsRUFBRSxtWUFBbVk7Z0JBQ2xaLGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsU0FBUyxFQUFFO3dCQUNULFVBQVUsRUFBRSxLQUFLO3FCQUNsQjtpQkFDRjthQUNGO1lBQ0QsVUFBVSxFQUFFO2dCQUNWLEtBQUssRUFBRSx3RUFBd0U7Z0JBQy9FLGFBQWEsRUFBRSx5T0FBeU87Z0JBQ3hQLGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsTUFBTSxFQUFFO3dCQUNOLFVBQVUsRUFBRSxLQUFLO3FCQUNsQjtpQkFDRjthQUNGO1lBQ0QsY0FBYyxFQUFFO2dCQUNkLEtBQUssRUFBRSx1RUFBdUU7Z0JBQzlFLGFBQWEsRUFBRSxnWkFBZ1o7Z0JBQy9aLGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsVUFBVSxFQUFFO3dCQUNWLFVBQVUsRUFBRSxLQUFLO3FCQUNsQjtpQkFDRjthQUNGO1lBQ0QscUJBQXFCLEVBQUU7Z0JBQ3JCLEtBQUssRUFBRSxtRkFBbUY7Z0JBQzFGLGFBQWEsRUFBRSw2T0FBNk87Z0JBQzVQLGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsVUFBVSxFQUFFO3dCQUNWLFVBQVUsRUFBRSxLQUFLO3FCQUNsQjtvQkFDRCxrQkFBa0IsRUFBRTt3QkFDbEIsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO29CQUNELGFBQWEsRUFBRTt3QkFDYixVQUFVLEVBQUUsS0FBSztxQkFDbEI7b0JBQ0QsWUFBWSxFQUFFO3dCQUNaLFVBQVUsRUFBRSxLQUFLO3FCQUNsQjtvQkFDRCxnQkFBZ0IsRUFBRTt3QkFDaEIsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO29CQUNELE9BQU8sRUFBRTt3QkFDUCxVQUFVLEVBQUUsS0FBSztxQkFDbEI7b0JBQ0QsT0FBTyxFQUFFO3dCQUNQLFVBQVUsRUFBRSxLQUFLO3FCQUNsQjtvQkFDRCxlQUFlLEVBQUU7d0JBQ2YsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO29CQUNELFNBQVMsRUFBRTt3QkFDVCxVQUFVLEVBQUUsS0FBSztxQkFDbEI7b0JBQ0QsTUFBTSxFQUFFO3dCQUNOLFVBQVUsRUFBRSxLQUFLO3FCQUNsQjtvQkFDRCxVQUFVLEVBQUU7d0JBQ1YsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCxhQUFhLEVBQUU7Z0JBQ2IsS0FBSyxFQUFFLDJFQUEyRTtnQkFDbEYsYUFBYSxFQUFFLGdDQUFnQztnQkFDL0MsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixVQUFVLEVBQUU7d0JBQ1YsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2dCQUNELFlBQVksRUFBRTtvQkFDWiwwQkFBMEI7b0JBQzFCLGFBQWE7aUJBQ2Q7YUFDRjtZQUNELGVBQWUsRUFBRTtnQkFDZixLQUFLLEVBQUUsNkVBQTZFO2dCQUNwRixhQUFhLEVBQUUsMlNBQTJTO2dCQUMxVCxhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLGFBQWEsRUFBRTt3QkFDYixVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7Z0JBQ0QsWUFBWSxFQUFFO29CQUNaLDBCQUEwQjtvQkFDMUIsYUFBYTtpQkFDZDthQUNGO1lBQ0QsY0FBYyxFQUFFO2dCQUNkLEtBQUssRUFBRSw0RUFBNEU7Z0JBQ25GLGFBQWEsRUFBRSxzZkFBc2Y7Z0JBQ3JnQixhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLFlBQVksRUFBRTt3QkFDWixVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7Z0JBQ0QsWUFBWSxFQUFFO29CQUNaLDBCQUEwQjtvQkFDMUIsYUFBYTtpQkFDZDthQUNGO1lBQ0Qsa0JBQWtCLEVBQUU7Z0JBQ2xCLEtBQUssRUFBRSxnRkFBZ0Y7Z0JBQ3ZGLGFBQWEsRUFBRSw0TEFBNEw7Z0JBQzNNLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsZ0JBQWdCLEVBQUU7d0JBQ2hCLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjtnQkFDRCxZQUFZLEVBQUU7b0JBQ1osMEJBQTBCO29CQUMxQixhQUFhO2lCQUNkO2FBQ0Y7WUFDRCxxQkFBcUIsRUFBRTtnQkFDckIsS0FBSyxFQUFFLG1GQUFtRjtnQkFDMUYsYUFBYSxFQUFFLDhIQUE4SDtnQkFDN0ksYUFBYSxFQUFFLE9BQU87YUFDdkI7WUFDRCxVQUFVLEVBQUU7Z0JBQ1YsS0FBSyxFQUFFLHdFQUF3RTtnQkFDL0UsYUFBYSxFQUFFLDBNQUEwTTtnQkFDek4sYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2dCQUNELFlBQVksRUFBRTtvQkFDWiwwQkFBMEI7b0JBQzFCLGFBQWE7aUJBQ2Q7YUFDRjtZQUNELFlBQVksRUFBRTtnQkFDWixLQUFLLEVBQUUsMEVBQTBFO2dCQUNqRixhQUFhLEVBQUUsZ0NBQWdDO2dCQUMvQyxhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLFNBQVMsRUFBRTt3QkFDVCxVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7Z0JBQ0QsWUFBWSxFQUFFO29CQUNaLDBCQUEwQjtvQkFDMUIsYUFBYTtpQkFDZDthQUNGO1lBQ0QsYUFBYSxFQUFFO2dCQUNiLEtBQUssRUFBRSwyRUFBMkU7Z0JBQ2xGLGFBQWEsRUFBRSw2QkFBNkI7Z0JBQzVDLGFBQWEsRUFBRSxTQUFTO2dCQUN4QixlQUFlLEVBQUU7b0JBQ2YsVUFBVSxFQUFFO3dCQUNWLFVBQVUsRUFBRSxLQUFLO3FCQUNsQjtvQkFDRCxrQkFBa0IsRUFBRTt3QkFDbEIsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO29CQUNELGFBQWEsRUFBRTt3QkFDYixVQUFVLEVBQUUsS0FBSztxQkFDbEI7b0JBQ0QsWUFBWSxFQUFFO3dCQUNaLFVBQVUsRUFBRSxLQUFLO3FCQUNsQjtvQkFDRCxnQkFBZ0IsRUFBRTt3QkFDaEIsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO29CQUNELE9BQU8sRUFBRTt3QkFDUCxVQUFVLEVBQUUsS0FBSztxQkFDbEI7b0JBQ0QsT0FBTyxFQUFFO3dCQUNQLFVBQVUsRUFBRSxLQUFLO3FCQUNsQjtvQkFDRCxlQUFlLEVBQUU7d0JBQ2YsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO29CQUNELFNBQVMsRUFBRTt3QkFDVCxVQUFVLEVBQUUsS0FBSztxQkFDbEI7b0JBQ0QsTUFBTSxFQUFFO3dCQUNOLFVBQVUsRUFBRSxLQUFLO3FCQUNsQjtvQkFDRCxVQUFVLEVBQUU7d0JBQ1YsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2dCQUNELFlBQVksRUFBRTtvQkFDWixhQUFhO29CQUNiLDBCQUEwQjtpQkFDM0I7YUFDRjtZQUNELGVBQWUsRUFBRTtnQkFDZixLQUFLLEVBQUUsNkVBQTZFO2dCQUNwRixhQUFhLEVBQUUsK0JBQStCO2dCQUM5QyxhQUFhLEVBQUUsU0FBUztnQkFDeEIsZUFBZSxFQUFFO29CQUNmLFVBQVUsRUFBRTt3QkFDVixVQUFVLEVBQUUsS0FBSztxQkFDbEI7b0JBQ0Qsa0JBQWtCLEVBQUU7d0JBQ2xCLFVBQVUsRUFBRSxLQUFLO3FCQUNsQjtvQkFDRCxhQUFhLEVBQUU7d0JBQ2IsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO29CQUNELFlBQVksRUFBRTt3QkFDWixVQUFVLEVBQUUsS0FBSztxQkFDbEI7b0JBQ0QsZ0JBQWdCLEVBQUU7d0JBQ2hCLFVBQVUsRUFBRSxLQUFLO3FCQUNsQjtvQkFDRCxPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO29CQUNELE9BQU8sRUFBRTt3QkFDUCxVQUFVLEVBQUUsS0FBSztxQkFDbEI7b0JBQ0QsZUFBZSxFQUFFO3dCQUNmLFVBQVUsRUFBRSxLQUFLO3FCQUNsQjtvQkFDRCxTQUFTLEVBQUU7d0JBQ1QsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO29CQUNELE1BQU0sRUFBRTt3QkFDTixVQUFVLEVBQUUsS0FBSztxQkFDbEI7b0JBQ0QsVUFBVSxFQUFFO3dCQUNWLFVBQVUsRUFBRSxLQUFLO3FCQUNsQjtpQkFDRjtnQkFDRCxZQUFZLEVBQUU7b0JBQ1osYUFBYTtvQkFDYiwwQkFBMEI7aUJBQzNCO2FBQ0Y7WUFDRCx1QkFBdUIsRUFBRTtnQkFDdkIsS0FBSyxFQUFFLGdGQUFnRjtnQkFDdkYsYUFBYSxFQUFFLHNOQUFzTjtnQkFDck8sYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixVQUFVLEVBQUU7d0JBQ1YsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCwrQkFBK0IsRUFBRTtnQkFDL0IsS0FBSyxFQUFFLHdGQUF3RjtnQkFDL0YsYUFBYSxFQUFFLGdJQUFnSTtnQkFDL0ksYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixrQkFBa0IsRUFBRTt3QkFDbEIsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCw2QkFBNkIsRUFBRTtnQkFDN0IsS0FBSyxFQUFFLHNGQUFzRjtnQkFDN0YsYUFBYSxFQUFFLDhMQUE4TDtnQkFDN00sYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixrQkFBa0IsRUFBRTt3QkFDbEIsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxhQUFhLEVBQUU7Z0JBQ2IsS0FBSyxFQUFFLHNFQUFzRTtnQkFDN0UsYUFBYSxFQUFFLDhFQUE4RTtnQkFDN0YsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxvQkFBb0IsRUFBRTtnQkFDcEIsS0FBSyxFQUFFLDZFQUE2RTtnQkFDcEYsYUFBYSxFQUFFLHlXQUF5VztnQkFDeFgsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2dCQUNELFlBQVksRUFBRTtvQkFDWiwwQkFBMEI7b0JBQzFCLGFBQWE7aUJBQ2Q7YUFDRjtZQUNELDBCQUEwQixFQUFFO2dCQUMxQixLQUFLLEVBQUUsbUZBQW1GO2dCQUMxRixhQUFhLEVBQUUsd0NBQXdDO2dCQUN2RCxhQUFhLEVBQUUsT0FBTzthQUN2QjtZQUNELG9CQUFvQixFQUFFO2dCQUNwQixLQUFLLEVBQUUsa0ZBQWtGO2dCQUN6RixhQUFhLEVBQUUsc0VBQXNFO2dCQUNyRixhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLE1BQU0sRUFBRTt3QkFDTixVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7YUFDRjtZQUNELG1CQUFtQixFQUFFO2dCQUNuQixLQUFLLEVBQUUsaUZBQWlGO2dCQUN4RixhQUFhLEVBQUUsdUlBQXVJO2dCQUN0SixhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLE1BQU0sRUFBRTt3QkFDTixVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7Z0JBQ0QsWUFBWSxFQUFFO29CQUNaLDBCQUEwQjtvQkFDMUIsYUFBYTtpQkFDZDthQUNGO1lBQ0QsZ0JBQWdCLEVBQUU7Z0JBQ2hCLEtBQUssRUFBRSw4RUFBOEU7Z0JBQ3JGLGFBQWEsRUFBRSxxQkFBcUI7Z0JBQ3BDLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsVUFBVSxFQUFFO3dCQUNWLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1NBQ0YsQ0FBQztRQUNRLGtCQUFhLEdBQWtCO1lBQ3ZDLFVBQVUsRUFBRTtnQkFDVixNQUFNLEVBQUUsVUFBVTtnQkFDbEIsS0FBSyxFQUFFLEVBQUU7Z0JBQ1QsS0FBSyxFQUFFLDhFQUE4RTtnQkFDckYsZUFBZSxFQUFFO29CQUNmLDJCQUEyQjtpQkFDNUI7YUFDRjtZQUNELGtCQUFrQixFQUFFO2dCQUNsQixNQUFNLEVBQUUsa0JBQWtCO2dCQUMxQixLQUFLLEVBQUUsRUFBRTtnQkFDVCxLQUFLLEVBQUUsc0ZBQXNGO2dCQUM3RixlQUFlLEVBQUU7b0JBQ2YsMkJBQTJCO2lCQUM1QjthQUNGO1lBQ0QsYUFBYSxFQUFFO2dCQUNiLE1BQU0sRUFBRSxhQUFhO2dCQUNyQixLQUFLLEVBQUUsRUFBRTtnQkFDVCxLQUFLLEVBQUUsaUZBQWlGO2dCQUN4RixlQUFlLEVBQUU7b0JBQ2YsMkJBQTJCO2lCQUM1QjthQUNGO1lBQ0QsZ0JBQWdCLEVBQUU7Z0JBQ2hCLE1BQU0sRUFBRSxnQkFBZ0I7Z0JBQ3hCLEtBQUssRUFBRSxFQUFFO2dCQUNULEtBQUssRUFBRSxvRkFBb0Y7Z0JBQzNGLGVBQWUsRUFBRTtvQkFDZiwyQkFBMkI7aUJBQzVCO2FBQ0Y7WUFDRCxZQUFZLEVBQUU7Z0JBQ1osTUFBTSxFQUFFLFlBQVk7Z0JBQ3BCLEtBQUssRUFBRSxFQUFFO2dCQUNULEtBQUssRUFBRSxnRkFBZ0Y7Z0JBQ3ZGLGVBQWUsRUFBRTtvQkFDZiwyQkFBMkI7aUJBQzVCO2FBQ0Y7WUFDRCxPQUFPLEVBQUU7Z0JBQ1AsTUFBTSxFQUFFLE9BQU87Z0JBQ2YsS0FBSyxFQUFFLEVBQUU7Z0JBQ1QsS0FBSyxFQUFFLDJFQUEyRTtnQkFDbEYsZUFBZSxFQUFFO29CQUNmLDJCQUEyQjtpQkFDNUI7YUFDRjtZQUNELE9BQU8sRUFBRTtnQkFDUCxNQUFNLEVBQUUsT0FBTztnQkFDZixLQUFLLEVBQUUsRUFBRTtnQkFDVCxLQUFLLEVBQUUsMkVBQTJFO2dCQUNsRixlQUFlLEVBQUU7b0JBQ2YsMkJBQTJCO2lCQUM1QjthQUNGO1lBQ0QsZUFBZSxFQUFFO2dCQUNmLE1BQU0sRUFBRSxlQUFlO2dCQUN2QixLQUFLLEVBQUUsRUFBRTtnQkFDVCxLQUFLLEVBQUUsbUZBQW1GO2dCQUMxRixlQUFlLEVBQUU7b0JBQ2YsMkJBQTJCO2lCQUM1QjthQUNGO1lBQ0QsU0FBUyxFQUFFO2dCQUNULE1BQU0sRUFBRSxTQUFTO2dCQUNqQixLQUFLLEVBQUUsRUFBRTtnQkFDVCxLQUFLLEVBQUUsNkVBQTZFO2dCQUNwRixlQUFlLEVBQUU7b0JBQ2YsMkJBQTJCO2lCQUM1QjthQUNGO1lBQ0QsTUFBTSxFQUFFO2dCQUNOLE1BQU0sRUFBRSxNQUFNO2dCQUNkLEtBQUssRUFBRSxFQUFFO2dCQUNULEtBQUssRUFBRSwwRUFBMEU7Z0JBQ2pGLGVBQWUsRUFBRTtvQkFDZiwyQkFBMkI7aUJBQzVCO2FBQ0Y7WUFDRCxVQUFVLEVBQUU7Z0JBQ1YsTUFBTSxFQUFFLFVBQVU7Z0JBQ2xCLEtBQUssRUFBRSxFQUFFO2dCQUNULEtBQUssRUFBRSw4RUFBOEU7Z0JBQ3JGLGVBQWUsRUFBRTtvQkFDZiwyQkFBMkI7aUJBQzVCO2FBQ0Y7U0FDRixDQUFDO0lBU0YsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSxxQkFBcUI7UUFDMUIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxtQ0FBbUMsQ0FBQyxDQUFDO1FBQzlDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGtCQUFrQjtRQUN2QixJQUFJLENBQUMsR0FBRyxDQUFDLGdDQUFnQyxDQUFDLENBQUM7UUFDM0MsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLHVCQUF1QjtRQUM1QixJQUFJLENBQUMsR0FBRyxDQUFDLHFDQUFxQyxDQUFDLENBQUM7UUFDaEQsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLGFBQWE7UUFDbEIsSUFBSSxDQUFDLEdBQUcsQ0FBQywyQkFBMkIsQ0FBQyxDQUFDO1FBQ3RDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSxvQkFBb0I7UUFDekIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxrQ0FBa0MsQ0FBQyxDQUFDO1FBQzdDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSxZQUFZO1FBQ2pCLElBQUksQ0FBQyxHQUFHLENBQUMsMEJBQTBCLENBQUMsQ0FBQztRQUNyQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksZ0JBQWdCO1FBQ3JCLElBQUksQ0FBQyxHQUFHLENBQUMsOEJBQThCLENBQUMsQ0FBQztRQUN6QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxnQkFBZ0I7UUFDckIsSUFBSSxDQUFDLEdBQUcsQ0FBQyw4QkFBOEIsQ0FBQyxDQUFDO1FBQ3pDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHVCQUF1QjtRQUM1QixJQUFJLENBQUMsR0FBRyxDQUFDLHFDQUFxQyxDQUFDLENBQUM7UUFDaEQsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksYUFBYTtRQUNsQixJQUFJLENBQUMsR0FBRyxDQUFDLDJCQUEyQixDQUFDLENBQUM7UUFDdEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksWUFBWTtRQUNqQixJQUFJLENBQUMsR0FBRyxDQUFDLDBCQUEwQixDQUFDLENBQUM7UUFDckMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksa0JBQWtCO1FBQ3ZCLElBQUksQ0FBQyxHQUFHLENBQUMsZ0NBQWdDLENBQUMsQ0FBQztRQUMzQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx1QkFBdUI7UUFDNUIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxxQ0FBcUMsQ0FBQyxDQUFDO1FBQ2hELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG9CQUFvQjtRQUN6QixJQUFJLENBQUMsR0FBRyxDQUFDLGtDQUFrQyxDQUFDLENBQUM7UUFDN0MsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksY0FBYztRQUNuQixJQUFJLENBQUMsR0FBRyxDQUFDLDRCQUE0QixDQUFDLENBQUM7UUFDdkMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZ0JBQWdCO1FBQ3JCLElBQUksQ0FBQyxHQUFHLENBQUMsOEJBQThCLENBQUMsQ0FBQztRQUN6QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxvQkFBb0I7UUFDekIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxrQ0FBa0MsQ0FBQyxDQUFDO1FBQzdDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGVBQWU7UUFDcEIsSUFBSSxDQUFDLEdBQUcsQ0FBQyw2QkFBNkIsQ0FBQyxDQUFDO1FBQ3hDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG1CQUFtQjtRQUN4QixJQUFJLENBQUMsR0FBRyxDQUFDLGlDQUFpQyxDQUFDLENBQUM7UUFDNUMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kscUJBQXFCO1FBQzFCLElBQUksQ0FBQyxHQUFHLENBQUMsbUNBQW1DLENBQUMsQ0FBQztRQUM5QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxXQUFXO1FBQ2hCLElBQUksQ0FBQyxHQUFHLENBQUMseUJBQXlCLENBQUMsQ0FBQztRQUNwQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxpQkFBaUI7UUFDdEIsSUFBSSxDQUFDLEdBQUcsQ0FBQywrQkFBK0IsQ0FBQyxDQUFDO1FBQzFDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLFdBQVc7UUFDaEIsSUFBSSxDQUFDLEdBQUcsQ0FBQyx5QkFBeUIsQ0FBQyxDQUFDO1FBQ3BDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGFBQWE7UUFDbEIsSUFBSSxDQUFDLEdBQUcsQ0FBQywyQkFBMkIsQ0FBQyxDQUFDO1FBQ3RDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLFVBQVU7UUFDZixJQUFJLENBQUMsR0FBRyxDQUFDLHdCQUF3QixDQUFDLENBQUM7UUFDbkMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksY0FBYztRQUNuQixJQUFJLENBQUMsR0FBRyxDQUFDLDRCQUE0QixDQUFDLENBQUM7UUFDdkMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kscUJBQXFCO1FBQzFCLElBQUksQ0FBQyxHQUFHLENBQUMsbUNBQW1DLENBQUMsQ0FBQztRQUM5QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksYUFBYTtRQUNsQixJQUFJLENBQUMsR0FBRyxDQUFDLDJCQUEyQixDQUFDLENBQUM7UUFDdEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLGVBQWU7UUFDcEIsSUFBSSxDQUFDLEdBQUcsQ0FBQyw2QkFBNkIsQ0FBQyxDQUFDO1FBQ3hDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSxjQUFjO1FBQ25CLElBQUksQ0FBQyxHQUFHLENBQUMsNEJBQTRCLENBQUMsQ0FBQztRQUN2QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksa0JBQWtCO1FBQ3ZCLElBQUksQ0FBQyxHQUFHLENBQUMsZ0NBQWdDLENBQUMsQ0FBQztRQUMzQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxxQkFBcUI7UUFDMUIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxtQ0FBbUMsQ0FBQyxDQUFDO1FBQzlDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSxVQUFVO1FBQ2YsSUFBSSxDQUFDLEdBQUcsQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDO1FBQ25DLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSxZQUFZO1FBQ2pCLElBQUksQ0FBQyxHQUFHLENBQUMsMEJBQTBCLENBQUMsQ0FBQztRQUNyQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksYUFBYTtRQUNsQixJQUFJLENBQUMsR0FBRyxDQUFDLDJCQUEyQixDQUFDLENBQUM7UUFDdEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLGVBQWU7UUFDcEIsSUFBSSxDQUFDLEdBQUcsQ0FBQyw2QkFBNkIsQ0FBQyxDQUFDO1FBQ3hDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHVCQUF1QjtRQUM1QixJQUFJLENBQUMsR0FBRyxDQUFDLHFDQUFxQyxDQUFDLENBQUM7UUFDaEQsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksK0JBQStCO1FBQ3BDLElBQUksQ0FBQyxHQUFHLENBQUMsNkNBQTZDLENBQUMsQ0FBQztRQUN4RCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSw2QkFBNkI7UUFDbEMsSUFBSSxDQUFDLEdBQUcsQ0FBQywyQ0FBMkMsQ0FBQyxDQUFDO1FBQ3RELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGFBQWE7UUFDbEIsSUFBSSxDQUFDLEdBQUcsQ0FBQywyQkFBMkIsQ0FBQyxDQUFDO1FBQ3RDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSxvQkFBb0I7UUFDekIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxrQ0FBa0MsQ0FBQyxDQUFDO1FBQzdDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDBCQUEwQjtRQUMvQixJQUFJLENBQUMsR0FBRyxDQUFDLHdDQUF3QyxDQUFDLENBQUM7UUFDbkQsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksb0JBQW9CO1FBQ3pCLElBQUksQ0FBQyxHQUFHLENBQUMsa0NBQWtDLENBQUMsQ0FBQztRQUM3QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksbUJBQW1CO1FBQ3hCLElBQUksQ0FBQyxHQUFHLENBQUMsaUNBQWlDLENBQUMsQ0FBQztRQUM1QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxnQkFBZ0I7UUFDckIsSUFBSSxDQUFDLEdBQUcsQ0FBQyw4QkFBOEIsQ0FBQyxDQUFDO1FBQ3pDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSxVQUFVLENBQUMsWUFBb0IsRUFBRSxPQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUMzRixJQUFJLEdBQUcsR0FBRyw4RUFBOEUsQ0FBQztRQUN6RixHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxpQkFBaUIsRUFBRSxZQUFZLENBQUMsQ0FBQztRQUNuRCxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxZQUFZLEVBQUUsT0FBTyxJQUFJLEdBQUcsQ0FBQyxDQUFDO1FBQ2hELEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFdBQVcsRUFBRSxNQUFNLElBQUksR0FBRyxDQUFDLENBQUM7UUFDOUMsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsY0FBYyxFQUFFLFNBQVMsSUFBSSxLQUFLLENBQUMsQ0FBQztRQUN0RCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLENBQUM7SUFDdEIsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSxpQkFBaUIsQ0FBQyxZQUFvQixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQ2xHLElBQUksR0FBRyxHQUFHLHNGQUFzRixDQUFDO1FBQ2pHLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLGlCQUFpQixFQUFFLFlBQVksQ0FBQyxDQUFDO1FBQ25ELEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFlBQVksRUFBRSxPQUFPLElBQUksR0FBRyxDQUFDLENBQUM7UUFDaEQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsV0FBVyxFQUFFLE1BQU0sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUM5QyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsU0FBUyxJQUFJLEtBQUssQ0FBQyxDQUFDO1FBQ3RELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN0QixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLFlBQVksQ0FBQyxZQUFvQixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQzdGLElBQUksR0FBRyxHQUFHLGlGQUFpRixDQUFDO1FBQzVGLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLGlCQUFpQixFQUFFLFlBQVksQ0FBQyxDQUFDO1FBQ25ELEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFlBQVksRUFBRSxPQUFPLElBQUksR0FBRyxDQUFDLENBQUM7UUFDaEQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsV0FBVyxFQUFFLE1BQU0sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUM5QyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsU0FBUyxJQUFJLEtBQUssQ0FBQyxDQUFDO1FBQ3RELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN0QixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLGVBQWUsQ0FBQyxZQUFvQixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQ2hHLElBQUksR0FBRyxHQUFHLG9GQUFvRixDQUFDO1FBQy9GLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLGlCQUFpQixFQUFFLFlBQVksQ0FBQyxDQUFDO1FBQ25ELEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFlBQVksRUFBRSxPQUFPLElBQUksR0FBRyxDQUFDLENBQUM7UUFDaEQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsV0FBVyxFQUFFLE1BQU0sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUM5QyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsU0FBUyxJQUFJLEtBQUssQ0FBQyxDQUFDO1FBQ3RELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN0QixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLFdBQVcsQ0FBQyxZQUFvQixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQzVGLElBQUksR0FBRyxHQUFHLGdGQUFnRixDQUFDO1FBQzNGLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLGlCQUFpQixFQUFFLFlBQVksQ0FBQyxDQUFDO1FBQ25ELEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFlBQVksRUFBRSxPQUFPLElBQUksR0FBRyxDQUFDLENBQUM7UUFDaEQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsV0FBVyxFQUFFLE1BQU0sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUM5QyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsU0FBUyxJQUFJLEtBQUssQ0FBQyxDQUFDO1FBQ3RELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN0QixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLE9BQU8sQ0FBQyxZQUFvQixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQ3hGLElBQUksR0FBRyxHQUFHLDJFQUEyRSxDQUFDO1FBQ3RGLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLGlCQUFpQixFQUFFLFlBQVksQ0FBQyxDQUFDO1FBQ25ELEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFlBQVksRUFBRSxPQUFPLElBQUksR0FBRyxDQUFDLENBQUM7UUFDaEQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsV0FBVyxFQUFFLE1BQU0sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUM5QyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsU0FBUyxJQUFJLEtBQUssQ0FBQyxDQUFDO1FBQ3RELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN0QixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLE9BQU8sQ0FBQyxZQUFvQixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQ3hGLElBQUksR0FBRyxHQUFHLDJFQUEyRSxDQUFDO1FBQ3RGLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLGlCQUFpQixFQUFFLFlBQVksQ0FBQyxDQUFDO1FBQ25ELEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFlBQVksRUFBRSxPQUFPLElBQUksR0FBRyxDQUFDLENBQUM7UUFDaEQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsV0FBVyxFQUFFLE1BQU0sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUM5QyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsU0FBUyxJQUFJLEtBQUssQ0FBQyxDQUFDO1FBQ3RELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN0QixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLGNBQWMsQ0FBQyxZQUFvQixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQy9GLElBQUksR0FBRyxHQUFHLG1GQUFtRixDQUFDO1FBQzlGLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLGlCQUFpQixFQUFFLFlBQVksQ0FBQyxDQUFDO1FBQ25ELEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFlBQVksRUFBRSxPQUFPLElBQUksR0FBRyxDQUFDLENBQUM7UUFDaEQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsV0FBVyxFQUFFLE1BQU0sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUM5QyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsU0FBUyxJQUFJLEtBQUssQ0FBQyxDQUFDO1FBQ3RELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN0QixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLFNBQVMsQ0FBQyxZQUFvQixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQzFGLElBQUksR0FBRyxHQUFHLDZFQUE2RSxDQUFDO1FBQ3hGLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLGlCQUFpQixFQUFFLFlBQVksQ0FBQyxDQUFDO1FBQ25ELEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFlBQVksRUFBRSxPQUFPLElBQUksR0FBRyxDQUFDLENBQUM7UUFDaEQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsV0FBVyxFQUFFLE1BQU0sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUM5QyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsU0FBUyxJQUFJLEtBQUssQ0FBQyxDQUFDO1FBQ3RELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN0QixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLE1BQU0sQ0FBQyxZQUFvQixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQ3ZGLElBQUksR0FBRyxHQUFHLDBFQUEwRSxDQUFDO1FBQ3JGLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLGlCQUFpQixFQUFFLFlBQVksQ0FBQyxDQUFDO1FBQ25ELEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFlBQVksRUFBRSxPQUFPLElBQUksR0FBRyxDQUFDLENBQUM7UUFDaEQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsV0FBVyxFQUFFLE1BQU0sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUM5QyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsU0FBUyxJQUFJLEtBQUssQ0FBQyxDQUFDO1FBQ3RELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN0QixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLFVBQVUsQ0FBQyxZQUFvQixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQzNGLElBQUksR0FBRyxHQUFHLDhFQUE4RSxDQUFDO1FBQ3pGLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLGlCQUFpQixFQUFFLFlBQVksQ0FBQyxDQUFDO1FBQ25ELEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFlBQVksRUFBRSxPQUFPLElBQUksR0FBRyxDQUFDLENBQUM7UUFDaEQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsV0FBVyxFQUFFLE1BQU0sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUM5QyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsU0FBUyxJQUFJLEtBQUssQ0FBQyxDQUFDO1FBQ3RELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN0QixDQUFDO0NBQ0Y7QUF4OUNELHNDQXc5Q0MiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBBY3Rpb25zLCBQb2xpY3lTdGF0ZW1lbnQsIFJlc291cmNlVHlwZXMgfSBmcm9tIFwiLi4vc2hhcmVkXCI7XG5cbi8qKlxuICogU3RhdGVtZW50IHByb3ZpZGVyIGZvciBzZXJ2aWNlIFtmcmF1ZGRldGVjdG9yXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvbGlzdF9hbWF6b25mcmF1ZGRldGVjdG9yLmh0bWwpLlxuICpcbiAqIEBwYXJhbSBzaWQgW1NJRF0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19zaWQuaHRtbCkgb2YgdGhlIHN0YXRlbWVudFxuICovXG5leHBvcnQgY2xhc3MgRnJhdWRkZXRlY3RvciBleHRlbmRzIFBvbGljeVN0YXRlbWVudCB7XG4gIHB1YmxpYyBzZXJ2aWNlUHJlZml4ID0gJ2ZyYXVkZGV0ZWN0b3InO1xuICBwcm90ZWN0ZWQgYWN0aW9uTGlzdDogQWN0aW9ucyA9IHtcbiAgICBcIkJhdGNoQ3JlYXRlVmFyaWFibGVcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZnJhdWRkZXRlY3Rvci9sYXRlc3QvYXBpL0FQSV9CYXRjaENyZWF0ZVZhcmlhYmxlXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiQ3JlYXRlcyBhIGJhdGNoIG9mIHZhcmlhYmxlcy5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJjb25kaXRpb25zXCI6IFtcbiAgICAgICAgXCJhd3M6UmVxdWVzdFRhZy8ke1RhZ0tleX1cIixcbiAgICAgICAgXCJhd3M6VGFnS2V5c1wiXG4gICAgICBdXG4gICAgfSxcbiAgICBcIkJhdGNoR2V0VmFyaWFibGVcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZnJhdWRkZXRlY3Rvci9sYXRlc3QvYXBpL0FQSV9CYXRjaEdldFZhcmlhYmxlXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR2V0cyBhIGJhdGNoIG9mIHZhcmlhYmxlcy5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJMaXN0XCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInZhcmlhYmxlXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiQ3JlYXRlRGV0ZWN0b3JWZXJzaW9uXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2ZyYXVkZGV0ZWN0b3IvbGF0ZXN0L2FwaS9BUElfQ3JlYXRlRGV0ZWN0b3JWZXJzaW9uXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiQ3JlYXRlcyBhIGRldGVjdG9yIHZlcnNpb24uIFRoZSBkZXRlY3RvciB2ZXJzaW9uIHN0YXJ0cyBpbiBhIERSQUZUIHN0YXR1cy5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJkZXRlY3RvclwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH0sXG4gICAgICBcImNvbmRpdGlvbnNcIjogW1xuICAgICAgICBcImF3czpSZXF1ZXN0VGFnLyR7VGFnS2V5fVwiLFxuICAgICAgICBcImF3czpUYWdLZXlzXCJcbiAgICAgIF1cbiAgICB9LFxuICAgIFwiQ3JlYXRlTW9kZWxcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZnJhdWRkZXRlY3Rvci9sYXRlc3QvYXBpL0FQSV9DcmVhdGVNb2RlbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkNyZWF0ZXMgYSBtb2RlbCB1c2luZyB0aGUgc3BlY2lmaWVkIG1vZGVsIHR5cGUuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwibW9kZWxcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9LFxuICAgICAgXCJjb25kaXRpb25zXCI6IFtcbiAgICAgICAgXCJhd3M6UmVxdWVzdFRhZy8ke1RhZ0tleX1cIixcbiAgICAgICAgXCJhd3M6VGFnS2V5c1wiXG4gICAgICBdXG4gICAgfSxcbiAgICBcIkNyZWF0ZU1vZGVsVmVyc2lvblwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9mcmF1ZGRldGVjdG9yL2xhdGVzdC9hcGkvQVBJX0NyZWF0ZU1vZGVsVmVyc2lvblwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkNyZWF0ZXMgYSB2ZXJzaW9uIG9mIHRoZSBtb2RlbCB1c2luZyB0aGUgc3BlY2lmaWVkIG1vZGVsIHR5cGUgYW5kIG1vZGVsIGlkLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcIm1vZGVsXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfSxcbiAgICAgIFwiY29uZGl0aW9uc1wiOiBbXG4gICAgICAgIFwiYXdzOlJlcXVlc3RUYWcvJHtUYWdLZXl9XCIsXG4gICAgICAgIFwiYXdzOlRhZ0tleXNcIlxuICAgICAgXVxuICAgIH0sXG4gICAgXCJDcmVhdGVSdWxlXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2ZyYXVkZGV0ZWN0b3IvbGF0ZXN0L2FwaS9BUElfQ3JlYXRlUnVsZS5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiQ3JlYXRlcyBhIHJ1bGUgZm9yIHVzZSB3aXRoIHRoZSBzcGVjaWZpZWQgZGV0ZWN0b3IuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiZGV0ZWN0b3JcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9LFxuICAgICAgXCJjb25kaXRpb25zXCI6IFtcbiAgICAgICAgXCJhd3M6UmVxdWVzdFRhZy8ke1RhZ0tleX1cIixcbiAgICAgICAgXCJhd3M6VGFnS2V5c1wiXG4gICAgICBdXG4gICAgfSxcbiAgICBcIkNyZWF0ZVZhcmlhYmxlXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2ZyYXVkZGV0ZWN0b3IvbGF0ZXN0L2FwaS9BUElfQ3JlYXRlVmFyaWFibGUuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkNyZWF0ZXMgYSB2YXJpYWJsZS5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJjb25kaXRpb25zXCI6IFtcbiAgICAgICAgXCJhd3M6UmVxdWVzdFRhZy8ke1RhZ0tleX1cIixcbiAgICAgICAgXCJhd3M6VGFnS2V5c1wiXG4gICAgICBdXG4gICAgfSxcbiAgICBcIkRlbGV0ZURldGVjdG9yXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2ZyYXVkZGV0ZWN0b3IvbGF0ZXN0L2FwaS9BUElfRGVsZXRlRGV0ZWN0b3JcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJEZWxldGVzIHRoZSBkZXRlY3Rvci4gQmVmb3JlIGRlbGV0aW5nIGEgZGV0ZWN0b3IsIHlvdSBtdXN0IGZpcnN0IGRlbGV0ZSBhbGwgZGV0ZWN0b3IgdmVyc2lvbnMgYW5kIHJ1bGUgdmVyc2lvbnMgYXNzb2NpYXRlZCB3aXRoIHRoZSBkZXRlY3Rvci5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJkZXRlY3RvclwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiRGVsZXRlRGV0ZWN0b3JWZXJzaW9uXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2ZyYXVkZGV0ZWN0b3IvbGF0ZXN0L2FwaS9BUElfRGVsZXRlRGV0ZWN0b3JWZXJzaW9uXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiRGVsZXRlcyB0aGUgZGV0ZWN0b3IgdmVyc2lvbi4gWW91IGNhbm5vdCBkZWxldGUgZGV0ZWN0b3IgdmVyc2lvbnMgdGhhdCBhcmUgaW4gQUNUSVZFIHN0YXR1cy5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJkZXRlY3Rvci12ZXJzaW9uXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJEZWxldGVFdmVudFwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9mcmF1ZGRldGVjdG9yL2xhdGVzdC9hcGkvQVBJX0RlbGV0ZUV2ZW50XCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiRGVsZXRlcyB0aGUgc3BlY2lmaWVkIGV2ZW50LlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCJcbiAgICB9LFxuICAgIFwiRGVsZXRlUnVsZVwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9mcmF1ZGRldGVjdG9yL2xhdGVzdC9hcGkvQVBJX0RlbGV0ZVJ1bGVcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJEZWxldGVzIHRoZSBydWxlLiBZb3UgY2Fubm90IGRlbGV0ZSBhIHJ1bGUgaWYgaXQgaXMgdXNlZCBieSBhbiBBQ1RJVkUgb3IgSU5BQ1RJVkUgZGV0ZWN0b3IgdmVyc2lvbi5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJydWxlXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJEZXNjcmliZURldGVjdG9yXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2ZyYXVkZGV0ZWN0b3IvbGF0ZXN0L2FwaS9BUElfRGVzY3JpYmVEZXRlY3RvclwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdldHMgYWxsIHZlcnNpb25zIGZvciBhIHNwZWNpZmllZCBkZXRlY3Rvci5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJSZWFkXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImRldGVjdG9yXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJEZXNjcmliZU1vZGVsVmVyc2lvbnNcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZnJhdWRkZXRlY3Rvci9sYXRlc3QvYXBpL0FQSV9EZXNjcmliZU1vZGVsVmVyc2lvbnNcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHZXRzIGFsbCBvZiB0aGUgbW9kZWwgdmVyc2lvbnMgZm9yIHRoZSBzcGVjaWZpZWQgbW9kZWwgdHlwZSBvciBmb3IgdGhlIHNwZWNpZmllZCBtb2RlbCB0eXBlIGFuZCBtb2RlbCBJRC4gWW91IGNhbiBhbHNvIGdldCBkZXRhaWxzIGZvciBhIHNpbmdsZSwgc3BlY2lmaWVkIG1vZGVsIHZlcnNpb24uXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUmVhZFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJtb2RlbC12ZXJzaW9uXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiR2V0RGV0ZWN0b3JWZXJzaW9uXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2ZyYXVkZGV0ZWN0b3IvbGF0ZXN0L2FwaS9BUElfR2V0RGV0ZWN0b3JWZXJzaW9uXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR2V0cyBhIHBhcnRpY3VsYXIgZGV0ZWN0b3IgdmVyc2lvbi5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJMaXN0XCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImRldGVjdG9yLXZlcnNpb25cIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkdldERldGVjdG9yc1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9mcmF1ZGRldGVjdG9yL2xhdGVzdC9hcGkvQVBJX0dldERldGVjdG9yc1wiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdldHMgYWxsIGRldGVjdG9ycyBvciBhIHNpbmdsZSBkZXRlY3RvciBpZiBhIGRldGVjdG9ySWQgaXMgc3BlY2lmaWVkLiBUaGlzIGlzIGEgcGFnaW5hdGVkIEFQSS4gSWYgeW91IHByb3ZpZGUgYSBudWxsIG1heFJlc3VsdHMsIHRoaXMgYWN0aW9uIHJldHJpZXZlcyBhIG1heGltdW0gb2YgMTAgcmVjb3JkcyBwZXIgcGFnZS4gSWYgeW91IHByb3ZpZGUgYSBtYXhSZXN1bHRzLCB0aGUgdmFsdWUgbXVzdCBiZSBiZXR3ZWVuIDUgYW5kIDEwLiBUbyBnZXQgdGhlIG5leHQgcGFnZSByZXN1bHRzLCBwcm92aWRlIHRoZSBwYWdpbmF0aW9uIHRva2VuIGZyb20gdGhlIEdldERldGVjdG9yc1Jlc3BvbnNlIGFzIHBhcnQgb2YgeW91ciByZXF1ZXN0LiBBIG51bGwgcGFnaW5hdGlvbiB0b2tlbiBmZXRjaGVzIHRoZSByZWNvcmRzIGZyb20gdGhlIGJlZ2lubmluZy5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJMaXN0XCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImRldGVjdG9yXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiR2V0RW50aXR5VHlwZXNcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZnJhdWRkZXRlY3Rvci9sYXRlc3QvYXBpL0FQSV9HZXRFbnRpdHlUeXBlc1wiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdldHMgYWxsIGVudGl0eSB0eXBlcyBvciBhIHNwZWNpZmljIGVudGl0eSB0eXBlIGlmIGEgbmFtZSBpcyBzcGVjaWZpZWQuIFRoaXMgaXMgYSBwYWdpbmF0ZWQgQVBJLiBJZiB5b3UgcHJvdmlkZSBhIG51bGwgbWF4UmVzdWx0cywgdGhpcyBhY3Rpb24gcmV0cmlldmVzIGEgbWF4aW11bSBvZiAxMCByZWNvcmRzIHBlciBwYWdlLiBJZiB5b3UgcHJvdmlkZSBhIG1heFJlc3VsdHMsIHRoZSB2YWx1ZSBtdXN0IGJlIGJldHdlZW4gNSBhbmQgMTAuIFRvIGdldCB0aGUgbmV4dCBwYWdlIHJlc3VsdHMsIHByb3ZpZGUgdGhlIHBhZ2luYXRpb24gdG9rZW4gZnJvbSB0aGUgR2V0RW50aXR5VHlwZXNSZXNwb25zZSBhcyBwYXJ0IG9mIHlvdXIgcmVxdWVzdC4gQSBudWxsIHBhZ2luYXRpb24gdG9rZW4gZmV0Y2hlcyB0aGUgcmVjb3JkcyBmcm9tIHRoZSBiZWdpbm5pbmcuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiTGlzdFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJlbnRpdHktdHlwZVwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkdldEV2ZW50UHJlZGljdGlvblwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9mcmF1ZGRldGVjdG9yL2xhdGVzdC9hcGkvQVBJX0dldEV2ZW50UHJlZGljdGlvblwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkV2YWx1YXRlcyBhbiBldmVudCBhZ2FpbnN0IGEgZGV0ZWN0b3IgdmVyc2lvbi4gSWYgYSB2ZXJzaW9uIElEIGlzIG5vdCBwcm92aWRlZCwgdGhlIGRldGVjdG9y4oCZcyAoQUNUSVZFKSB2ZXJzaW9uIGlzIHVzZWQuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUmVhZFwiXG4gICAgfSxcbiAgICBcIkdldEV2ZW50VHlwZXNcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZnJhdWRkZXRlY3Rvci9sYXRlc3QvYXBpL0FQSV9HZXRFdmVudFR5cGVzXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR2V0cyBhbGwgZXZlbnQgdHlwZXMgb3IgYSBzcGVjaWZpYyBldmVudCB0eXBlIGlmIG5hbWUgaXMgcHJvdmlkZWQuIFRoaXMgaXMgYSBwYWdpbmF0ZWQgQVBJLiBJZiB5b3UgcHJvdmlkZSBhIG51bGwgbWF4UmVzdWx0cywgdGhpcyBhY3Rpb24gcmV0cmlldmVzIGEgbWF4aW11bSBvZiAxMCByZWNvcmRzIHBlciBwYWdlLiBJZiB5b3UgcHJvdmlkZSBhIG1heFJlc3VsdHMsIHRoZSB2YWx1ZSBtdXN0IGJlIGJldHdlZW4gNSBhbmQgMTAuIFRvIGdldCB0aGUgbmV4dCBwYWdlIHJlc3VsdHMsIHByb3ZpZGUgdGhlIHBhZ2luYXRpb24gdG9rZW4gZnJvbSB0aGUgR2V0RXZlbnRUeXBlc1Jlc3BvbnNlIGFzIHBhcnQgb2YgeW91ciByZXF1ZXN0LiBBIG51bGwgcGFnaW5hdGlvbiB0b2tlbiBmZXRjaGVzIHRoZSByZWNvcmRzIGZyb20gdGhlIGJlZ2lubmluZy5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJMaXN0XCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImV2ZW50LXR5cGVcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJHZXRFeHRlcm5hbE1vZGVsc1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9mcmF1ZGRldGVjdG9yL2xhdGVzdC9hcGkvQVBJX0dldEV4dGVybmFsTW9kZWxzXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR2V0cyB0aGUgZGV0YWlscyBmb3Igb25lIG9yIG1vcmUgQW1hem9uIFNhZ2VNYWtlciBtb2RlbHMgdGhhdCBoYXZlIGJlZW4gaW1wb3J0ZWQgaW50byB0aGUgc2VydmljZS4gVGhpcyBpcyBhIHBhZ2luYXRlZCBBUEkuIElmIHlvdSBwcm92aWRlIGEgbnVsbCBtYXhSZXN1bHRzLCB0aGlzIGFjdGlvbnMgcmV0cmlldmVzIGEgbWF4aW11bSBvZiAxMCByZWNvcmRzIHBlciBwYWdlLiBJZiB5b3UgcHJvdmlkZSBhIG1heFJlc3VsdHMsIHRoZSB2YWx1ZSBtdXN0IGJlIGJldHdlZW4gNSBhbmQgMTAuIFRvIGdldCB0aGUgbmV4dCBwYWdlIHJlc3VsdHMsIHByb3ZpZGUgdGhlIHBhZ2luYXRpb24gdG9rZW4gZnJvbSB0aGUgR2V0RXh0ZXJuYWxNb2RlbHNSZXN1bHQgYXMgcGFydCBvZiB5b3VyIHJlcXVlc3QuIEEgbnVsbCBwYWdpbmF0aW9uIHRva2VuIGZldGNoZXMgdGhlIHJlY29yZHMgZnJvbSB0aGUgYmVnaW5uaW5nLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIkxpc3RcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiZXh0ZXJuYWwtbW9kZWxcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJHZXRLTVNFbmNyeXB0aW9uS2V5XCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2ZyYXVkZGV0ZWN0b3IvbGF0ZXN0L2FwaS9BUElfR2V0S01TRW5jcnlwdGlvbktleVwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdldHMgdGhlIGVuY3J5cHRpb24ga2V5IGlmIGEgS2V5IE1hbmFnZW1lbnQgU2VydmljZSAoS01TKSBjdXN0b21lciBtYXN0ZXIga2V5IChDTUspIGhhcyBiZWVuIHNwZWNpZmllZCB0byBiZSB1c2VkIHRvIGVuY3J5cHQgY29udGVudCBpbiBBbWF6b24gRnJhdWQgRGV0ZWN0b3IuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUmVhZFwiXG4gICAgfSxcbiAgICBcIkdldExhYmVsc1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9mcmF1ZGRldGVjdG9yL2xhdGVzdC9hcGkvQVBJX0dldExhYmVsc1wiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdldHMgYWxsIGxhYmVscyBvciBhIHNwZWNpZmljIGxhYmVsIGlmIG5hbWUgaXMgcHJvdmlkZWQuIFRoaXMgaXMgYSBwYWdpbmF0ZWQgQVBJLiBJZiB5b3UgcHJvdmlkZSBhIG51bGwgbWF4UmVzdWx0cywgdGhpcyBhY3Rpb24gcmV0cmlldmVzIGEgbWF4aW11bSBvZiA1MCByZWNvcmRzIHBlciBwYWdlLiBJZiB5b3UgcHJvdmlkZSBhIG1heFJlc3VsdHMsIHRoZSB2YWx1ZSBtdXN0IGJlIGJldHdlZW4gMTAgYW5kIDUwLiBUbyBnZXQgdGhlIG5leHQgcGFnZSByZXN1bHRzLCBwcm92aWRlIHRoZSBwYWdpbmF0aW9uIHRva2VuIGZyb20gdGhlIEdldEdldExhYmVsc1Jlc3BvbnNlIGFzIHBhcnQgb2YgeW91ciByZXF1ZXN0LiBBIG51bGwgcGFnaW5hdGlvbiB0b2tlbiBmZXRjaGVzIHRoZSByZWNvcmRzIGZyb20gdGhlIGJlZ2lubmluZy5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJMaXN0XCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImxhYmVsXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiR2V0TW9kZWxWZXJzaW9uXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2ZyYXVkZGV0ZWN0b3IvbGF0ZXN0L2FwaS9BUElfR2V0TW9kZWxWZXJzaW9uXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR2V0cyB0aGUgZGV0YWlscyBvZiB0aGUgc3BlY2lmaWVkIG1vZGVsIHZlcnNpb24uXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiTGlzdFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJtb2RlbC12ZXJzaW9uXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJHZXRNb2RlbHNcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZnJhdWRkZXRlY3Rvci9sYXRlc3QvYXBpL0FQSV9HZXRNb2RlbHNcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHZXRzIG9uZSBvciBtb3JlIG1vZGVscy4gR2V0cyBhbGwgbW9kZWxzIGZvciB0aGUgQVdTIGFjY291bnQgaWYgbm8gbW9kZWwgdHlwZSBhbmQgbm8gbW9kZWwgaWQgcHJvdmlkZWQuIEdldHMgYWxsIG1vZGVscyBmb3IgdGhlIEFXUyBhY2NvdW50IGFuZCBtb2RlbCB0eXBlLCBpZiB0aGUgbW9kZWwgdHlwZSBpcyBzcGVjaWZpZWQgYnV0IG1vZGVsIGlkIGlzIG5vdCBwcm92aWRlZC4gR2V0cyBhIHNwZWNpZmljIG1vZGVsIGlmIChtb2RlbCB0eXBlLCBtb2RlbCBpZCkgdHVwbGUgaXMgc3BlY2lmaWVkLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIkxpc3RcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwibW9kZWxcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJHZXRPdXRjb21lc1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9mcmF1ZGRldGVjdG9yL2xhdGVzdC9hcGkvQVBJX0dldE91dGNvbWVzXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR2V0cyBvbmUgb3IgbW9yZSBvdXRjb21lcy4gVGhpcyBpcyBhIHBhZ2luYXRlZCBBUEkuIElmIHlvdSBwcm92aWRlIGEgbnVsbCBtYXhSZXN1bHRzLCB0aGlzIGFjdGlvbnMgcmV0cmlldmVzIGEgbWF4aW11bSBvZiAxMDAgcmVjb3JkcyBwZXIgcGFnZS4gSWYgeW91IHByb3ZpZGUgYSBtYXhSZXN1bHRzLCB0aGUgdmFsdWUgbXVzdCBiZSBiZXR3ZWVuIDUwIGFuZCAxMDAuIFRvIGdldCB0aGUgbmV4dCBwYWdlIHJlc3VsdHMsIHByb3ZpZGUgdGhlIHBhZ2luYXRpb24gdG9rZW4gZnJvbSB0aGUgR2V0T3V0Y29tZXNSZXN1bHQgYXMgcGFydCBvZiB5b3VyIHJlcXVlc3QuIEEgbnVsbCBwYWdpbmF0aW9uIHRva2VuIGZldGNoZXMgdGhlIHJlY29yZHMgZnJvbSB0aGUgYmVnaW5uaW5nLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIkxpc3RcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwib3V0Y29tZVwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkdldFJ1bGVzXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2ZyYXVkZGV0ZWN0b3IvbGF0ZXN0L2FwaS9BUElfR2V0UnVsZXMuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdldCBhbGwgcnVsZXMgZm9yIGEgZGV0ZWN0b3IgKHBhZ2luYXRlZCkgaWYgcnVsZUlkIGFuZCBydWxlVmVyc2lvbiBhcmUgbm90IHNwZWNpZmllZC4gR2V0cyBhbGwgcnVsZXMgZm9yIHRoZSBkZXRlY3RvciBhbmQgdGhlIHJ1bGVJZCBpZiBwcmVzZW50IChwYWdpbmF0ZWQpLiBHZXRzIGEgc3BlY2lmaWMgcnVsZSBpZiBib3RoIHRoZSBydWxlSWQgYW5kIHRoZSBydWxlVmVyc2lvbiBhcmUgc3BlY2lmaWVkLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIkxpc3RcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwicnVsZVwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkdldFZhcmlhYmxlc1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9mcmF1ZGRldGVjdG9yL2xhdGVzdC9hcGkvQVBJX0dldFZhcmlhYmxlc1wiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdldHMgYWxsIG9mIHRoZSB2YXJpYWJsZXMgb3IgdGhlIHNwZWNpZmljIHZhcmlhYmxlLiBUaGlzIGlzIGEgcGFnaW5hdGVkIEFQSS4gUHJvdmlkaW5nIG51bGwgbWF4U2l6ZVBlclBhZ2UgcmVzdWx0cyBpbiByZXRyaWV2aW5nIG1heGltdW0gb2YgMTAwIHJlY29yZHMgcGVyIHBhZ2UuIElmIHlvdSBwcm92aWRlIG1heFNpemVQZXJQYWdlIHRoZSB2YWx1ZSBtdXN0IGJlIGJldHdlZW4gNTAgYW5kIDEwMC4gVG8gZ2V0IHRoZSBuZXh0IHBhZ2UgcmVzdWx0LCBhIHByb3ZpZGUgYSBwYWdpbmF0aW9uIHRva2VuIGZyb20gR2V0VmFyaWFibGVzUmVzdWx0IGFzIHBhcnQgb2YgeW91ciByZXF1ZXN0LiBOdWxsIHBhZ2luYXRpb24gdG9rZW4gZmV0Y2hlcyB0aGUgcmVjb3JkcyBmcm9tIHRoZSBiZWdpbm5pbmcuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiTGlzdFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJ2YXJpYWJsZVwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkxpc3RUYWdzRm9yUmVzb3VyY2VcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZnJhdWRkZXRlY3Rvci9sYXRlc3QvYXBpL0FQSV9MaXN0VGFnc0ZvclJlc291cmNlLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJMaXN0cyBhbGwgdGFncyBhc3NvY2lhdGVkIHdpdGggdGhlIHJlc291cmNlLiBUaGlzIGlzIGEgcGFnaW5hdGVkIEFQSS4gVG8gZ2V0IHRoZSBuZXh0IHBhZ2UgcmVzdWx0cywgcHJvdmlkZSB0aGUgcGFnaW5hdGlvbiB0b2tlbiBmcm9tIHRoZSByZXNwb25zZSBhcyBwYXJ0IG9mIHlvdXIgcmVxdWVzdC4gQSBudWxsIHBhZ2luYXRpb24gdG9rZW4gZmV0Y2hlcyB0aGUgcmVjb3JkcyBmcm9tIHRoZSBiZWdpbm5pbmcuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiTGlzdFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJkZXRlY3RvclwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9LFxuICAgICAgICBcImRldGVjdG9yLXZlcnNpb25cIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfSxcbiAgICAgICAgXCJlbnRpdHktdHlwZVwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9LFxuICAgICAgICBcImV2ZW50LXR5cGVcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfSxcbiAgICAgICAgXCJleHRlcm5hbC1tb2RlbFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9LFxuICAgICAgICBcImxhYmVsXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH0sXG4gICAgICAgIFwibW9kZWxcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfSxcbiAgICAgICAgXCJtb2RlbC12ZXJzaW9uXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH0sXG4gICAgICAgIFwib3V0Y29tZVwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9LFxuICAgICAgICBcInJ1bGVcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfSxcbiAgICAgICAgXCJ2YXJpYWJsZVwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIlB1dERldGVjdG9yXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2ZyYXVkZGV0ZWN0b3IvbGF0ZXN0L2FwaS9BUElfUHV0RGV0ZWN0b3IuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkNyZWF0ZXMgb3IgdXBkYXRlcyBhIGRldGVjdG9yLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImRldGVjdG9yXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfSxcbiAgICAgIFwiY29uZGl0aW9uc1wiOiBbXG4gICAgICAgIFwiYXdzOlJlcXVlc3RUYWcvJHtUYWdLZXl9XCIsXG4gICAgICAgIFwiYXdzOlRhZ0tleXNcIlxuICAgICAgXVxuICAgIH0sXG4gICAgXCJQdXRFbnRpdHlUeXBlXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2ZyYXVkZGV0ZWN0b3IvbGF0ZXN0L2FwaS9BUElfUHV0RW50aXR5VHlwZS5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiQ3JlYXRlcyBvciB1cGRhdGVzIGFuIGVudGl0eSB0eXBlLiBBbiBlbnRpdHkgcmVwcmVzZW50cyB3aG8gaXMgcGVyZm9ybWluZyB0aGUgZXZlbnQuIEFzIHBhcnQgb2YgYSBmcmF1ZCBwcmVkaWN0aW9uLCB5b3UgcGFzcyB0aGUgZW50aXR5IElEIHRvIGluZGljYXRlIHRoZSBzcGVjaWZpYyBlbnRpdHkgd2hvIHBlcmZvcm1lZCB0aGUgZXZlbnQuIEFuIGVudGl0eSB0eXBlIGNsYXNzaWZpZXMgdGhlIGVudGl0eS4gRXhhbXBsZSBjbGFzc2lmaWNhdGlvbnMgaW5jbHVkZSBjdXN0b21lciwgbWVyY2hhbnQsIG9yIGFjY291bnQuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiZW50aXR5LXR5cGVcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9LFxuICAgICAgXCJjb25kaXRpb25zXCI6IFtcbiAgICAgICAgXCJhd3M6UmVxdWVzdFRhZy8ke1RhZ0tleX1cIixcbiAgICAgICAgXCJhd3M6VGFnS2V5c1wiXG4gICAgICBdXG4gICAgfSxcbiAgICBcIlB1dEV2ZW50VHlwZVwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9mcmF1ZGRldGVjdG9yL2xhdGVzdC9hcGkvQVBJX1B1dEV2ZW50VHlwZS5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiQ3JlYXRlcyBvciB1cGRhdGVzIGFuIGV2ZW50IHR5cGUuIEFuIGV2ZW50IGlzIGEgYnVzaW5lc3MgYWN0aXZpdHkgdGhhdCBpcyBldmFsdWF0ZWQgZm9yIGZyYXVkIHJpc2suIFdpdGggQW1hem9uIEZyYXVkIERldGVjdG9yLCB5b3UgZ2VuZXJhdGUgZnJhdWQgcHJlZGljdGlvbnMgZm9yIGV2ZW50cy4gQW4gZXZlbnQgdHlwZSBkZWZpbmVzIHRoZSBzdHJ1Y3R1cmUgZm9yIGFuIGV2ZW50IHNlbnQgdG8gQW1hem9uIEZyYXVkIERldGVjdG9yLiBUaGlzIGluY2x1ZGVzIHRoZSB2YXJpYWJsZXMgc2VudCBhcyBwYXJ0IG9mIHRoZSBldmVudCwgdGhlIGVudGl0eSBwZXJmb3JtaW5nIHRoZSBldmVudCAoc3VjaCBhcyBhIGN1c3RvbWVyKSwgYW5kIHRoZSBsYWJlbHMgdGhhdCBjbGFzc2lmeSB0aGUgZXZlbnQuIEV4YW1wbGUgZXZlbnQgdHlwZXMgaW5jbHVkZSBvbmxpbmUgcGF5bWVudCB0cmFuc2FjdGlvbnMsIGFjY291bnQgcmVnaXN0cmF0aW9ucywgYW5kIGF1dGhlbnRpY2F0aW9ucy5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJldmVudC10eXBlXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfSxcbiAgICAgIFwiY29uZGl0aW9uc1wiOiBbXG4gICAgICAgIFwiYXdzOlJlcXVlc3RUYWcvJHtUYWdLZXl9XCIsXG4gICAgICAgIFwiYXdzOlRhZ0tleXNcIlxuICAgICAgXVxuICAgIH0sXG4gICAgXCJQdXRFeHRlcm5hbE1vZGVsXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2ZyYXVkZGV0ZWN0b3IvbGF0ZXN0L2FwaS9BUElfUHV0RXh0ZXJuYWxNb2RlbC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiQ3JlYXRlcyBvciB1cGRhdGVzIGFuIEFtYXpvbiBTYWdlTWFrZXIgbW9kZWwgZW5kcG9pbnQuIFlvdSBjYW4gYWxzbyB1c2UgdGhpcyBhY3Rpb24gdG8gdXBkYXRlIHRoZSBjb25maWd1cmF0aW9uIG9mIHRoZSBtb2RlbCBlbmRwb2ludCwgaW5jbHVkaW5nIHRoZSBJQU0gcm9sZSBhbmQvb3IgdGhlIG1hcHBlZCB2YXJpYWJsZXMuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiZXh0ZXJuYWwtbW9kZWxcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9LFxuICAgICAgXCJjb25kaXRpb25zXCI6IFtcbiAgICAgICAgXCJhd3M6UmVxdWVzdFRhZy8ke1RhZ0tleX1cIixcbiAgICAgICAgXCJhd3M6VGFnS2V5c1wiXG4gICAgICBdXG4gICAgfSxcbiAgICBcIlB1dEtNU0VuY3J5cHRpb25LZXlcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZnJhdWRkZXRlY3Rvci9sYXRlc3QvYXBpL0FQSV9QdXRLTVNFbmNyeXB0aW9uS2V5Lmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJTcGVjaWZpZXMgdGhlIEtleSBNYW5hZ2VtZW50IFNlcnZpY2UgKEtNUykgY3VzdG9tZXIgbWFzdGVyIGtleSAoQ01LKSB0byBiZSB1c2VkIHRvIGVuY3J5cHQgY29udGVudCBpbiBBbWF6b24gRnJhdWQgRGV0ZWN0b3IuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIlxuICAgIH0sXG4gICAgXCJQdXRMYWJlbFwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9mcmF1ZGRldGVjdG9yL2xhdGVzdC9hcGkvQVBJX1B1dExhYmVsLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJDcmVhdGVzIG9yIHVwZGF0ZXMgbGFiZWwuIEEgbGFiZWwgY2xhc3NpZmllcyBhbiBldmVudCBhcyBmcmF1ZHVsZW50IG9yIGxlZ2l0aW1hdGUuIExhYmVscyBhcmUgYXNzb2NpYXRlZCB3aXRoIGV2ZW50IHR5cGVzIGFuZCB1c2VkIHRvIHRyYWluIHN1cGVydmlzZWQgbWFjaGluZSBsZWFybmluZyBtb2RlbHMgaW4gQW1hem9uIEZyYXVkIERldGVjdG9yLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImxhYmVsXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfSxcbiAgICAgIFwiY29uZGl0aW9uc1wiOiBbXG4gICAgICAgIFwiYXdzOlJlcXVlc3RUYWcvJHtUYWdLZXl9XCIsXG4gICAgICAgIFwiYXdzOlRhZ0tleXNcIlxuICAgICAgXVxuICAgIH0sXG4gICAgXCJQdXRPdXRjb21lXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2ZyYXVkZGV0ZWN0b3IvbGF0ZXN0L2FwaS9BUElfUHV0T3V0Y29tZS5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiQ3JlYXRlcyBvciB1cGRhdGVzIGFuIG91dGNvbWUuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwib3V0Y29tZVwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH0sXG4gICAgICBcImNvbmRpdGlvbnNcIjogW1xuICAgICAgICBcImF3czpSZXF1ZXN0VGFnLyR7VGFnS2V5fVwiLFxuICAgICAgICBcImF3czpUYWdLZXlzXCJcbiAgICAgIF1cbiAgICB9LFxuICAgIFwiVGFnUmVzb3VyY2VcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZnJhdWRkZXRlY3Rvci9sYXRlc3QvYXBpL0FQSV9UYWdSZXNvdXJjZS5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiQXNzaWducyB0YWdzIHRvIGEgcmVzb3VyY2UuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiVGFnZ2luZ1wiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJkZXRlY3RvclwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9LFxuICAgICAgICBcImRldGVjdG9yLXZlcnNpb25cIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfSxcbiAgICAgICAgXCJlbnRpdHktdHlwZVwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9LFxuICAgICAgICBcImV2ZW50LXR5cGVcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfSxcbiAgICAgICAgXCJleHRlcm5hbC1tb2RlbFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9LFxuICAgICAgICBcImxhYmVsXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH0sXG4gICAgICAgIFwibW9kZWxcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfSxcbiAgICAgICAgXCJtb2RlbC12ZXJzaW9uXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH0sXG4gICAgICAgIFwib3V0Y29tZVwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9LFxuICAgICAgICBcInJ1bGVcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfSxcbiAgICAgICAgXCJ2YXJpYWJsZVwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9XG4gICAgICB9LFxuICAgICAgXCJjb25kaXRpb25zXCI6IFtcbiAgICAgICAgXCJhd3M6VGFnS2V5c1wiLFxuICAgICAgICBcImF3czpSZXF1ZXN0VGFnLyR7VGFnS2V5fVwiXG4gICAgICBdXG4gICAgfSxcbiAgICBcIlVudGFnUmVzb3VyY2VcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZnJhdWRkZXRlY3Rvci9sYXRlc3QvYXBpL0FQSV9VbnRhZ1Jlc291cmNlLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJSZW1vdmVzIHRhZ3MgZnJvbSBhIHJlc291cmNlLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlRhZ2dpbmdcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiZGV0ZWN0b3JcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfSxcbiAgICAgICAgXCJkZXRlY3Rvci12ZXJzaW9uXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH0sXG4gICAgICAgIFwiZW50aXR5LXR5cGVcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfSxcbiAgICAgICAgXCJldmVudC10eXBlXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH0sXG4gICAgICAgIFwiZXh0ZXJuYWwtbW9kZWxcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfSxcbiAgICAgICAgXCJsYWJlbFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9LFxuICAgICAgICBcIm1vZGVsXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH0sXG4gICAgICAgIFwibW9kZWwtdmVyc2lvblwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9LFxuICAgICAgICBcIm91dGNvbWVcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfSxcbiAgICAgICAgXCJydWxlXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH0sXG4gICAgICAgIFwidmFyaWFibGVcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfVxuICAgICAgfSxcbiAgICAgIFwiY29uZGl0aW9uc1wiOiBbXG4gICAgICAgIFwiYXdzOlRhZ0tleXNcIixcbiAgICAgICAgXCJhd3M6UmVxdWVzdFRhZy8ke1RhZ0tleX1cIlxuICAgICAgXVxuICAgIH0sXG4gICAgXCJVcGRhdGVEZXRlY3RvclZlcnNpb25cIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZnJhdWRkZXRlY3Rvci9sYXRlc3QvYXBpL0FQSV9VcGRhdGVEZXRlY3RvclZlcnNpb25cIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJVcGRhdGVzIGEgZGV0ZWN0b3IgdmVyc2lvbi4gVGhlIGRldGVjdG9yIHZlcnNpb24gYXR0cmlidXRlcyB0aGF0IHlvdSBjYW4gdXBkYXRlIGluY2x1ZGUgbW9kZWxzLCBleHRlcm5hbCBtb2RlbCBlbmRwb2ludHMsIHJ1bGVzLCBydWxlIGV4ZWN1dGlvbiBtb2RlLCBhbmQgZGVzY3JpcHRpb24uIFlvdSBjYW4gb25seSB1cGRhdGUgYSBEUkFGVCBkZXRlY3RvciB2ZXJzaW9uLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImRldGVjdG9yXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJVcGRhdGVEZXRlY3RvclZlcnNpb25NZXRhZGF0YVwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9mcmF1ZGRldGVjdG9yL2xhdGVzdC9hcGkvQVBJX1VwZGF0ZURldGVjdG9yVmVyc2lvbk1ldGFkYXRhXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiVXBkYXRlcyB0aGUgZGV0ZWN0b3IgdmVyc2lvbidzIGRlc2NyaXB0aW9uLiBZb3UgY2FuIHVwZGF0ZSB0aGUgbWV0YWRhdGEgZm9yIGFueSBkZXRlY3RvciB2ZXJzaW9uIChEUkFGVCwgQUNUSVZFLCBvciBJTkFDVElWRSkuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiZGV0ZWN0b3ItdmVyc2lvblwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiVXBkYXRlRGV0ZWN0b3JWZXJzaW9uU3RhdHVzXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2ZyYXVkZGV0ZWN0b3IvbGF0ZXN0L2FwaS9BUElfVXBkYXRlRGV0ZWN0b3JWZXJzaW9uU3RhdHVzXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiVXBkYXRlcyB0aGUgZGV0ZWN0b3IgdmVyc2lvbuKAmXMgc3RhdHVzLiBZb3UgY2FuIHBlcmZvcm0gdGhlIGZvbGxvd2luZyBwcm9tb3Rpb25zIG9yIGRlbW90aW9ucyB1c2luZyBVcGRhdGVEZXRlY3RvclZlcnNpb25TdGF0dXM6IERSQUZUIHRvIEFDVElWRSwgQUNUSVZFIHRvIElOQUNUSVZFLCBhbmQgSU5BQ1RJVkUgdG8gQUNUSVZFLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImRldGVjdG9yLXZlcnNpb25cIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIlVwZGF0ZU1vZGVsXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2ZyYXVkZGV0ZWN0b3IvbGF0ZXN0L2FwaS9BUElfVXBkYXRlTW9kZWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJVcGRhdGVzIGEgbW9kZWwuIFlvdSBjYW4gdXBkYXRlIHRoZSBkZXNjcmlwdGlvbiBhdHRyaWJ1dGUgdXNpbmcgdGhpcyBhY3Rpb24uXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwibW9kZWxcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIlVwZGF0ZU1vZGVsVmVyc2lvblwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9mcmF1ZGRldGVjdG9yL2xhdGVzdC9hcGkvQVBJX1VwZGF0ZU1vZGVsVmVyc2lvblwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlVwZGF0ZXMgYSBtb2RlbCB2ZXJzaW9uLiBVcGRhdGluZyBhIG1vZGVsIHZlcnNpb24gcmV0cmFpbnMgYW4gZXhpc3RpbmcgbW9kZWwgdmVyc2lvbiB1c2luZyB1cGRhdGVkIHRyYWluaW5nIGRhdGEgYW5kIHByb2R1Y2VzIGEgbmV3IG1pbm9yIHZlcnNpb24gb2YgdGhlIG1vZGVsLiBZb3UgY2FuIHVwZGF0ZSB0aGUgdHJhaW5pbmcgZGF0YSBzZXQgbG9jYXRpb24gYW5kIGRhdGEgYWNjZXNzIHJvbGUgYXR0cmlidXRlcyB1c2luZyB0aGlzIGFjdGlvbi4gVGhpcyBhY3Rpb24gY3JlYXRlcyBhbmQgdHJhaW5zIGEgbmV3IG1pbm9yIHZlcnNpb24gb2YgdGhlIG1vZGVsLCBmb3IgZXhhbXBsZSB2ZXJzaW9uIDEuMDEsIDEuMDIsIDEuMDMuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwibW9kZWxcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9LFxuICAgICAgXCJjb25kaXRpb25zXCI6IFtcbiAgICAgICAgXCJhd3M6UmVxdWVzdFRhZy8ke1RhZ0tleX1cIixcbiAgICAgICAgXCJhd3M6VGFnS2V5c1wiXG4gICAgICBdXG4gICAgfSxcbiAgICBcIlVwZGF0ZU1vZGVsVmVyc2lvblN0YXR1c1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9mcmF1ZGRldGVjdG9yL2xhdGVzdC9hcGkvQVBJX1VwZGF0ZU1vZGVsVmVyc2lvblN0YXR1c1wiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlVwZGF0ZXMgdGhlIHN0YXR1cyBvZiBhIG1vZGVsIHZlcnNpb24uXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIlxuICAgIH0sXG4gICAgXCJVcGRhdGVSdWxlTWV0YWRhdGFcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZnJhdWRkZXRlY3Rvci9sYXRlc3QvYXBpL0FQSV9VcGRhdGVSdWxlTWV0YWRhdGEuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlVwZGF0ZXMgYSBydWxlJ3MgbWV0YWRhdGEuIFRoZSBkZXNjcmlwdGlvbiBhdHRyaWJ1dGUgY2FuIGJlIHVwZGF0ZWQuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwicnVsZVwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiVXBkYXRlUnVsZVZlcnNpb25cIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZnJhdWRkZXRlY3Rvci9sYXRlc3QvYXBpL0FQSV9VcGRhdGVSdWxlVmVyc2lvbi5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiVXBkYXRlcyBhIHJ1bGUgdmVyc2lvbiByZXN1bHRpbmcgaW4gYSBuZXcgcnVsZSB2ZXJzaW9uLiBVcGRhdGVzIGEgcnVsZSB2ZXJzaW9uIHJlc3VsdGluZyBpbiBhIG5ldyBydWxlIHZlcnNpb24gKHZlcnNpb24gMSwgMiwgMyAuLi4pLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInJ1bGVcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9LFxuICAgICAgXCJjb25kaXRpb25zXCI6IFtcbiAgICAgICAgXCJhd3M6UmVxdWVzdFRhZy8ke1RhZ0tleX1cIixcbiAgICAgICAgXCJhd3M6VGFnS2V5c1wiXG4gICAgICBdXG4gICAgfSxcbiAgICBcIlVwZGF0ZVZhcmlhYmxlXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2ZyYXVkZGV0ZWN0b3IvbGF0ZXN0L2FwaS9BUElfVXBkYXRlVmFyaWFibGUuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlVwZGF0ZXMgYSB2YXJpYWJsZS5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJ2YXJpYWJsZVwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9XG4gIH07XG4gIHByb3RlY3RlZCByZXNvdXJjZVR5cGVzOiBSZXNvdXJjZVR5cGVzID0ge1xuICAgIFwiZGV0ZWN0b3JcIjoge1xuICAgICAgXCJuYW1lXCI6IFwiZGV0ZWN0b3JcIixcbiAgICAgIFwidXJsXCI6IFwiXCIsXG4gICAgICBcImFyblwiOiBcImFybjoke1BhcnRpdGlvbn06ZnJhdWRkZXRlY3Rvcjoke1JlZ2lvbn06JHtBY2NvdW50fTpkZXRlY3Rvci8ke1Jlc291cmNlUGF0aH1cIixcbiAgICAgIFwiY29uZGl0aW9uS2V5c1wiOiBbXG4gICAgICAgIFwiYXdzOlJlc291cmNlVGFnLyR7VGFnS2V5fVwiXG4gICAgICBdXG4gICAgfSxcbiAgICBcImRldGVjdG9yLXZlcnNpb25cIjoge1xuICAgICAgXCJuYW1lXCI6IFwiZGV0ZWN0b3ItdmVyc2lvblwiLFxuICAgICAgXCJ1cmxcIjogXCJcIixcbiAgICAgIFwiYXJuXCI6IFwiYXJuOiR7UGFydGl0aW9ufTpmcmF1ZGRldGVjdG9yOiR7UmVnaW9ufToke0FjY291bnR9OmRldGVjdG9yLXZlcnNpb24vJHtSZXNvdXJjZVBhdGh9XCIsXG4gICAgICBcImNvbmRpdGlvbktleXNcIjogW1xuICAgICAgICBcImF3czpSZXNvdXJjZVRhZy8ke1RhZ0tleX1cIlxuICAgICAgXVxuICAgIH0sXG4gICAgXCJlbnRpdHktdHlwZVwiOiB7XG4gICAgICBcIm5hbWVcIjogXCJlbnRpdHktdHlwZVwiLFxuICAgICAgXCJ1cmxcIjogXCJcIixcbiAgICAgIFwiYXJuXCI6IFwiYXJuOiR7UGFydGl0aW9ufTpmcmF1ZGRldGVjdG9yOiR7UmVnaW9ufToke0FjY291bnR9OmVudGl0eS10eXBlLyR7UmVzb3VyY2VQYXRofVwiLFxuICAgICAgXCJjb25kaXRpb25LZXlzXCI6IFtcbiAgICAgICAgXCJhd3M6UmVzb3VyY2VUYWcvJHtUYWdLZXl9XCJcbiAgICAgIF1cbiAgICB9LFxuICAgIFwiZXh0ZXJuYWwtbW9kZWxcIjoge1xuICAgICAgXCJuYW1lXCI6IFwiZXh0ZXJuYWwtbW9kZWxcIixcbiAgICAgIFwidXJsXCI6IFwiXCIsXG4gICAgICBcImFyblwiOiBcImFybjoke1BhcnRpdGlvbn06ZnJhdWRkZXRlY3Rvcjoke1JlZ2lvbn06JHtBY2NvdW50fTpleHRlcm5hbC1tb2RlbC8ke1Jlc291cmNlUGF0aH1cIixcbiAgICAgIFwiY29uZGl0aW9uS2V5c1wiOiBbXG4gICAgICAgIFwiYXdzOlJlc291cmNlVGFnLyR7VGFnS2V5fVwiXG4gICAgICBdXG4gICAgfSxcbiAgICBcImV2ZW50LXR5cGVcIjoge1xuICAgICAgXCJuYW1lXCI6IFwiZXZlbnQtdHlwZVwiLFxuICAgICAgXCJ1cmxcIjogXCJcIixcbiAgICAgIFwiYXJuXCI6IFwiYXJuOiR7UGFydGl0aW9ufTpmcmF1ZGRldGVjdG9yOiR7UmVnaW9ufToke0FjY291bnR9OmV2ZW50LXR5cGUvJHtSZXNvdXJjZVBhdGh9XCIsXG4gICAgICBcImNvbmRpdGlvbktleXNcIjogW1xuICAgICAgICBcImF3czpSZXNvdXJjZVRhZy8ke1RhZ0tleX1cIlxuICAgICAgXVxuICAgIH0sXG4gICAgXCJsYWJlbFwiOiB7XG4gICAgICBcIm5hbWVcIjogXCJsYWJlbFwiLFxuICAgICAgXCJ1cmxcIjogXCJcIixcbiAgICAgIFwiYXJuXCI6IFwiYXJuOiR7UGFydGl0aW9ufTpmcmF1ZGRldGVjdG9yOiR7UmVnaW9ufToke0FjY291bnR9OmxhYmVsLyR7UmVzb3VyY2VQYXRofVwiLFxuICAgICAgXCJjb25kaXRpb25LZXlzXCI6IFtcbiAgICAgICAgXCJhd3M6UmVzb3VyY2VUYWcvJHtUYWdLZXl9XCJcbiAgICAgIF1cbiAgICB9LFxuICAgIFwibW9kZWxcIjoge1xuICAgICAgXCJuYW1lXCI6IFwibW9kZWxcIixcbiAgICAgIFwidXJsXCI6IFwiXCIsXG4gICAgICBcImFyblwiOiBcImFybjoke1BhcnRpdGlvbn06ZnJhdWRkZXRlY3Rvcjoke1JlZ2lvbn06JHtBY2NvdW50fTptb2RlbC8ke1Jlc291cmNlUGF0aH1cIixcbiAgICAgIFwiY29uZGl0aW9uS2V5c1wiOiBbXG4gICAgICAgIFwiYXdzOlJlc291cmNlVGFnLyR7VGFnS2V5fVwiXG4gICAgICBdXG4gICAgfSxcbiAgICBcIm1vZGVsLXZlcnNpb25cIjoge1xuICAgICAgXCJuYW1lXCI6IFwibW9kZWwtdmVyc2lvblwiLFxuICAgICAgXCJ1cmxcIjogXCJcIixcbiAgICAgIFwiYXJuXCI6IFwiYXJuOiR7UGFydGl0aW9ufTpmcmF1ZGRldGVjdG9yOiR7UmVnaW9ufToke0FjY291bnR9Om1vZGVsLXZlcnNpb24vJHtSZXNvdXJjZVBhdGh9XCIsXG4gICAgICBcImNvbmRpdGlvbktleXNcIjogW1xuICAgICAgICBcImF3czpSZXNvdXJjZVRhZy8ke1RhZ0tleX1cIlxuICAgICAgXVxuICAgIH0sXG4gICAgXCJvdXRjb21lXCI6IHtcbiAgICAgIFwibmFtZVwiOiBcIm91dGNvbWVcIixcbiAgICAgIFwidXJsXCI6IFwiXCIsXG4gICAgICBcImFyblwiOiBcImFybjoke1BhcnRpdGlvbn06ZnJhdWRkZXRlY3Rvcjoke1JlZ2lvbn06JHtBY2NvdW50fTpvdXRjb21lLyR7UmVzb3VyY2VQYXRofVwiLFxuICAgICAgXCJjb25kaXRpb25LZXlzXCI6IFtcbiAgICAgICAgXCJhd3M6UmVzb3VyY2VUYWcvJHtUYWdLZXl9XCJcbiAgICAgIF1cbiAgICB9LFxuICAgIFwicnVsZVwiOiB7XG4gICAgICBcIm5hbWVcIjogXCJydWxlXCIsXG4gICAgICBcInVybFwiOiBcIlwiLFxuICAgICAgXCJhcm5cIjogXCJhcm46JHtQYXJ0aXRpb259OmZyYXVkZGV0ZWN0b3I6JHtSZWdpb259OiR7QWNjb3VudH06cnVsZS8ke1Jlc291cmNlUGF0aH1cIixcbiAgICAgIFwiY29uZGl0aW9uS2V5c1wiOiBbXG4gICAgICAgIFwiYXdzOlJlc291cmNlVGFnLyR7VGFnS2V5fVwiXG4gICAgICBdXG4gICAgfSxcbiAgICBcInZhcmlhYmxlXCI6IHtcbiAgICAgIFwibmFtZVwiOiBcInZhcmlhYmxlXCIsXG4gICAgICBcInVybFwiOiBcIlwiLFxuICAgICAgXCJhcm5cIjogXCJhcm46JHtQYXJ0aXRpb259OmZyYXVkZGV0ZWN0b3I6JHtSZWdpb259OiR7QWNjb3VudH06dmFyaWFibGUvJHtSZXNvdXJjZVBhdGh9XCIsXG4gICAgICBcImNvbmRpdGlvbktleXNcIjogW1xuICAgICAgICBcImF3czpSZXNvdXJjZVRhZy8ke1RhZ0tleX1cIlxuICAgICAgXVxuICAgIH1cbiAgfTtcblxuICAvKipcbiAgICogU3RhdGVtZW50IHByb3ZpZGVyIGZvciBzZXJ2aWNlIFtmcmF1ZGRldGVjdG9yXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvbGlzdF9hbWF6b25mcmF1ZGRldGVjdG9yLmh0bWwpLlxuICAgKlxuICAgKiBAcGFyYW0gc2lkIFtTSURdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfc2lkLmh0bWwpIG9mIHRoZSBzdGF0ZW1lbnRcbiAgICovXG4gIGNvbnN0cnVjdG9yIChzaWQ/OiBzdHJpbmcpIHtcbiAgICBzdXBlcihzaWQpO1xuICB9XG5cbiAgLyoqXG4gICAqIENyZWF0ZXMgYSBiYXRjaCBvZiB2YXJpYWJsZXMuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2ZyYXVkZGV0ZWN0b3IvbGF0ZXN0L2FwaS9BUElfQmF0Y2hDcmVhdGVWYXJpYWJsZVxuICAgKi9cbiAgcHVibGljIHRvQmF0Y2hDcmVhdGVWYXJpYWJsZSgpIHtcbiAgICB0aGlzLmFkZCgnZnJhdWRkZXRlY3RvcjpCYXRjaENyZWF0ZVZhcmlhYmxlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR2V0cyBhIGJhdGNoIG9mIHZhcmlhYmxlcy5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9mcmF1ZGRldGVjdG9yL2xhdGVzdC9hcGkvQVBJX0JhdGNoR2V0VmFyaWFibGVcbiAgICovXG4gIHB1YmxpYyB0b0JhdGNoR2V0VmFyaWFibGUoKSB7XG4gICAgdGhpcy5hZGQoJ2ZyYXVkZGV0ZWN0b3I6QmF0Y2hHZXRWYXJpYWJsZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIENyZWF0ZXMgYSBkZXRlY3RvciB2ZXJzaW9uLiBUaGUgZGV0ZWN0b3IgdmVyc2lvbiBzdGFydHMgaW4gYSBEUkFGVCBzdGF0dXMuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2ZyYXVkZGV0ZWN0b3IvbGF0ZXN0L2FwaS9BUElfQ3JlYXRlRGV0ZWN0b3JWZXJzaW9uXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVEZXRlY3RvclZlcnNpb24oKSB7XG4gICAgdGhpcy5hZGQoJ2ZyYXVkZGV0ZWN0b3I6Q3JlYXRlRGV0ZWN0b3JWZXJzaW9uJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogQ3JlYXRlcyBhIG1vZGVsIHVzaW5nIHRoZSBzcGVjaWZpZWQgbW9kZWwgdHlwZS5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZnJhdWRkZXRlY3Rvci9sYXRlc3QvYXBpL0FQSV9DcmVhdGVNb2RlbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlTW9kZWwoKSB7XG4gICAgdGhpcy5hZGQoJ2ZyYXVkZGV0ZWN0b3I6Q3JlYXRlTW9kZWwnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBDcmVhdGVzIGEgdmVyc2lvbiBvZiB0aGUgbW9kZWwgdXNpbmcgdGhlIHNwZWNpZmllZCBtb2RlbCB0eXBlIGFuZCBtb2RlbCBpZC5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZnJhdWRkZXRlY3Rvci9sYXRlc3QvYXBpL0FQSV9DcmVhdGVNb2RlbFZlcnNpb25cbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZU1vZGVsVmVyc2lvbigpIHtcbiAgICB0aGlzLmFkZCgnZnJhdWRkZXRlY3RvcjpDcmVhdGVNb2RlbFZlcnNpb24nKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBDcmVhdGVzIGEgcnVsZSBmb3IgdXNlIHdpdGggdGhlIHNwZWNpZmllZCBkZXRlY3Rvci5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZnJhdWRkZXRlY3Rvci9sYXRlc3QvYXBpL0FQSV9DcmVhdGVSdWxlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZVJ1bGUoKSB7XG4gICAgdGhpcy5hZGQoJ2ZyYXVkZGV0ZWN0b3I6Q3JlYXRlUnVsZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIENyZWF0ZXMgYSB2YXJpYWJsZS5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZnJhdWRkZXRlY3Rvci9sYXRlc3QvYXBpL0FQSV9DcmVhdGVWYXJpYWJsZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVWYXJpYWJsZSgpIHtcbiAgICB0aGlzLmFkZCgnZnJhdWRkZXRlY3RvcjpDcmVhdGVWYXJpYWJsZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIERlbGV0ZXMgdGhlIGRldGVjdG9yLiBCZWZvcmUgZGVsZXRpbmcgYSBkZXRlY3RvciwgeW91IG11c3QgZmlyc3QgZGVsZXRlIGFsbCBkZXRlY3RvciB2ZXJzaW9ucyBhbmQgcnVsZSB2ZXJzaW9ucyBhc3NvY2lhdGVkIHdpdGggdGhlIGRldGVjdG9yLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9mcmF1ZGRldGVjdG9yL2xhdGVzdC9hcGkvQVBJX0RlbGV0ZURldGVjdG9yXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVEZXRlY3RvcigpIHtcbiAgICB0aGlzLmFkZCgnZnJhdWRkZXRlY3RvcjpEZWxldGVEZXRlY3RvcicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIERlbGV0ZXMgdGhlIGRldGVjdG9yIHZlcnNpb24uIFlvdSBjYW5ub3QgZGVsZXRlIGRldGVjdG9yIHZlcnNpb25zIHRoYXQgYXJlIGluIEFDVElWRSBzdGF0dXMuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2ZyYXVkZGV0ZWN0b3IvbGF0ZXN0L2FwaS9BUElfRGVsZXRlRGV0ZWN0b3JWZXJzaW9uXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVEZXRlY3RvclZlcnNpb24oKSB7XG4gICAgdGhpcy5hZGQoJ2ZyYXVkZGV0ZWN0b3I6RGVsZXRlRGV0ZWN0b3JWZXJzaW9uJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogRGVsZXRlcyB0aGUgc3BlY2lmaWVkIGV2ZW50LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9mcmF1ZGRldGVjdG9yL2xhdGVzdC9hcGkvQVBJX0RlbGV0ZUV2ZW50XG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVFdmVudCgpIHtcbiAgICB0aGlzLmFkZCgnZnJhdWRkZXRlY3RvcjpEZWxldGVFdmVudCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIERlbGV0ZXMgdGhlIHJ1bGUuIFlvdSBjYW5ub3QgZGVsZXRlIGEgcnVsZSBpZiBpdCBpcyB1c2VkIGJ5IGFuIEFDVElWRSBvciBJTkFDVElWRSBkZXRlY3RvciB2ZXJzaW9uLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9mcmF1ZGRldGVjdG9yL2xhdGVzdC9hcGkvQVBJX0RlbGV0ZVJ1bGVcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZVJ1bGUoKSB7XG4gICAgdGhpcy5hZGQoJ2ZyYXVkZGV0ZWN0b3I6RGVsZXRlUnVsZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdldHMgYWxsIHZlcnNpb25zIGZvciBhIHNwZWNpZmllZCBkZXRlY3Rvci5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9mcmF1ZGRldGVjdG9yL2xhdGVzdC9hcGkvQVBJX0Rlc2NyaWJlRGV0ZWN0b3JcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlRGV0ZWN0b3IoKSB7XG4gICAgdGhpcy5hZGQoJ2ZyYXVkZGV0ZWN0b3I6RGVzY3JpYmVEZXRlY3RvcicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdldHMgYWxsIG9mIHRoZSBtb2RlbCB2ZXJzaW9ucyBmb3IgdGhlIHNwZWNpZmllZCBtb2RlbCB0eXBlIG9yIGZvciB0aGUgc3BlY2lmaWVkIG1vZGVsIHR5cGUgYW5kIG1vZGVsIElELiBZb3UgY2FuIGFsc28gZ2V0IGRldGFpbHMgZm9yIGEgc2luZ2xlLCBzcGVjaWZpZWQgbW9kZWwgdmVyc2lvbi5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9mcmF1ZGRldGVjdG9yL2xhdGVzdC9hcGkvQVBJX0Rlc2NyaWJlTW9kZWxWZXJzaW9uc1xuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVNb2RlbFZlcnNpb25zKCkge1xuICAgIHRoaXMuYWRkKCdmcmF1ZGRldGVjdG9yOkRlc2NyaWJlTW9kZWxWZXJzaW9ucycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdldHMgYSBwYXJ0aWN1bGFyIGRldGVjdG9yIHZlcnNpb24uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZnJhdWRkZXRlY3Rvci9sYXRlc3QvYXBpL0FQSV9HZXREZXRlY3RvclZlcnNpb25cbiAgICovXG4gIHB1YmxpYyB0b0dldERldGVjdG9yVmVyc2lvbigpIHtcbiAgICB0aGlzLmFkZCgnZnJhdWRkZXRlY3RvcjpHZXREZXRlY3RvclZlcnNpb24nKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHZXRzIGFsbCBkZXRlY3RvcnMgb3IgYSBzaW5nbGUgZGV0ZWN0b3IgaWYgYSBkZXRlY3RvcklkIGlzIHNwZWNpZmllZC4gVGhpcyBpcyBhIHBhZ2luYXRlZCBBUEkuIElmIHlvdSBwcm92aWRlIGEgbnVsbCBtYXhSZXN1bHRzLCB0aGlzIGFjdGlvbiByZXRyaWV2ZXMgYSBtYXhpbXVtIG9mIDEwIHJlY29yZHMgcGVyIHBhZ2UuIElmIHlvdSBwcm92aWRlIGEgbWF4UmVzdWx0cywgdGhlIHZhbHVlIG11c3QgYmUgYmV0d2VlbiA1IGFuZCAxMC4gVG8gZ2V0IHRoZSBuZXh0IHBhZ2UgcmVzdWx0cywgcHJvdmlkZSB0aGUgcGFnaW5hdGlvbiB0b2tlbiBmcm9tIHRoZSBHZXREZXRlY3RvcnNSZXNwb25zZSBhcyBwYXJ0IG9mIHlvdXIgcmVxdWVzdC4gQSBudWxsIHBhZ2luYXRpb24gdG9rZW4gZmV0Y2hlcyB0aGUgcmVjb3JkcyBmcm9tIHRoZSBiZWdpbm5pbmcuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZnJhdWRkZXRlY3Rvci9sYXRlc3QvYXBpL0FQSV9HZXREZXRlY3RvcnNcbiAgICovXG4gIHB1YmxpYyB0b0dldERldGVjdG9ycygpIHtcbiAgICB0aGlzLmFkZCgnZnJhdWRkZXRlY3RvcjpHZXREZXRlY3RvcnMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHZXRzIGFsbCBlbnRpdHkgdHlwZXMgb3IgYSBzcGVjaWZpYyBlbnRpdHkgdHlwZSBpZiBhIG5hbWUgaXMgc3BlY2lmaWVkLiBUaGlzIGlzIGEgcGFnaW5hdGVkIEFQSS4gSWYgeW91IHByb3ZpZGUgYSBudWxsIG1heFJlc3VsdHMsIHRoaXMgYWN0aW9uIHJldHJpZXZlcyBhIG1heGltdW0gb2YgMTAgcmVjb3JkcyBwZXIgcGFnZS4gSWYgeW91IHByb3ZpZGUgYSBtYXhSZXN1bHRzLCB0aGUgdmFsdWUgbXVzdCBiZSBiZXR3ZWVuIDUgYW5kIDEwLiBUbyBnZXQgdGhlIG5leHQgcGFnZSByZXN1bHRzLCBwcm92aWRlIHRoZSBwYWdpbmF0aW9uIHRva2VuIGZyb20gdGhlIEdldEVudGl0eVR5cGVzUmVzcG9uc2UgYXMgcGFydCBvZiB5b3VyIHJlcXVlc3QuIEEgbnVsbCBwYWdpbmF0aW9uIHRva2VuIGZldGNoZXMgdGhlIHJlY29yZHMgZnJvbSB0aGUgYmVnaW5uaW5nLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2ZyYXVkZGV0ZWN0b3IvbGF0ZXN0L2FwaS9BUElfR2V0RW50aXR5VHlwZXNcbiAgICovXG4gIHB1YmxpYyB0b0dldEVudGl0eVR5cGVzKCkge1xuICAgIHRoaXMuYWRkKCdmcmF1ZGRldGVjdG9yOkdldEVudGl0eVR5cGVzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogRXZhbHVhdGVzIGFuIGV2ZW50IGFnYWluc3QgYSBkZXRlY3RvciB2ZXJzaW9uLiBJZiBhIHZlcnNpb24gSUQgaXMgbm90IHByb3ZpZGVkLCB0aGUgZGV0ZWN0b3LigJlzIChBQ1RJVkUpIHZlcnNpb24gaXMgdXNlZC5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9mcmF1ZGRldGVjdG9yL2xhdGVzdC9hcGkvQVBJX0dldEV2ZW50UHJlZGljdGlvblxuICAgKi9cbiAgcHVibGljIHRvR2V0RXZlbnRQcmVkaWN0aW9uKCkge1xuICAgIHRoaXMuYWRkKCdmcmF1ZGRldGVjdG9yOkdldEV2ZW50UHJlZGljdGlvbicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdldHMgYWxsIGV2ZW50IHR5cGVzIG9yIGEgc3BlY2lmaWMgZXZlbnQgdHlwZSBpZiBuYW1lIGlzIHByb3ZpZGVkLiBUaGlzIGlzIGEgcGFnaW5hdGVkIEFQSS4gSWYgeW91IHByb3ZpZGUgYSBudWxsIG1heFJlc3VsdHMsIHRoaXMgYWN0aW9uIHJldHJpZXZlcyBhIG1heGltdW0gb2YgMTAgcmVjb3JkcyBwZXIgcGFnZS4gSWYgeW91IHByb3ZpZGUgYSBtYXhSZXN1bHRzLCB0aGUgdmFsdWUgbXVzdCBiZSBiZXR3ZWVuIDUgYW5kIDEwLiBUbyBnZXQgdGhlIG5leHQgcGFnZSByZXN1bHRzLCBwcm92aWRlIHRoZSBwYWdpbmF0aW9uIHRva2VuIGZyb20gdGhlIEdldEV2ZW50VHlwZXNSZXNwb25zZSBhcyBwYXJ0IG9mIHlvdXIgcmVxdWVzdC4gQSBudWxsIHBhZ2luYXRpb24gdG9rZW4gZmV0Y2hlcyB0aGUgcmVjb3JkcyBmcm9tIHRoZSBiZWdpbm5pbmcuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZnJhdWRkZXRlY3Rvci9sYXRlc3QvYXBpL0FQSV9HZXRFdmVudFR5cGVzXG4gICAqL1xuICBwdWJsaWMgdG9HZXRFdmVudFR5cGVzKCkge1xuICAgIHRoaXMuYWRkKCdmcmF1ZGRldGVjdG9yOkdldEV2ZW50VHlwZXMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHZXRzIHRoZSBkZXRhaWxzIGZvciBvbmUgb3IgbW9yZSBBbWF6b24gU2FnZU1ha2VyIG1vZGVscyB0aGF0IGhhdmUgYmVlbiBpbXBvcnRlZCBpbnRvIHRoZSBzZXJ2aWNlLiBUaGlzIGlzIGEgcGFnaW5hdGVkIEFQSS4gSWYgeW91IHByb3ZpZGUgYSBudWxsIG1heFJlc3VsdHMsIHRoaXMgYWN0aW9ucyByZXRyaWV2ZXMgYSBtYXhpbXVtIG9mIDEwIHJlY29yZHMgcGVyIHBhZ2UuIElmIHlvdSBwcm92aWRlIGEgbWF4UmVzdWx0cywgdGhlIHZhbHVlIG11c3QgYmUgYmV0d2VlbiA1IGFuZCAxMC4gVG8gZ2V0IHRoZSBuZXh0IHBhZ2UgcmVzdWx0cywgcHJvdmlkZSB0aGUgcGFnaW5hdGlvbiB0b2tlbiBmcm9tIHRoZSBHZXRFeHRlcm5hbE1vZGVsc1Jlc3VsdCBhcyBwYXJ0IG9mIHlvdXIgcmVxdWVzdC4gQSBudWxsIHBhZ2luYXRpb24gdG9rZW4gZmV0Y2hlcyB0aGUgcmVjb3JkcyBmcm9tIHRoZSBiZWdpbm5pbmcuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZnJhdWRkZXRlY3Rvci9sYXRlc3QvYXBpL0FQSV9HZXRFeHRlcm5hbE1vZGVsc1xuICAgKi9cbiAgcHVibGljIHRvR2V0RXh0ZXJuYWxNb2RlbHMoKSB7XG4gICAgdGhpcy5hZGQoJ2ZyYXVkZGV0ZWN0b3I6R2V0RXh0ZXJuYWxNb2RlbHMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHZXRzIHRoZSBlbmNyeXB0aW9uIGtleSBpZiBhIEtleSBNYW5hZ2VtZW50IFNlcnZpY2UgKEtNUykgY3VzdG9tZXIgbWFzdGVyIGtleSAoQ01LKSBoYXMgYmVlbiBzcGVjaWZpZWQgdG8gYmUgdXNlZCB0byBlbmNyeXB0IGNvbnRlbnQgaW4gQW1hem9uIEZyYXVkIERldGVjdG9yLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2ZyYXVkZGV0ZWN0b3IvbGF0ZXN0L2FwaS9BUElfR2V0S01TRW5jcnlwdGlvbktleVxuICAgKi9cbiAgcHVibGljIHRvR2V0S01TRW5jcnlwdGlvbktleSgpIHtcbiAgICB0aGlzLmFkZCgnZnJhdWRkZXRlY3RvcjpHZXRLTVNFbmNyeXB0aW9uS2V5Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR2V0cyBhbGwgbGFiZWxzIG9yIGEgc3BlY2lmaWMgbGFiZWwgaWYgbmFtZSBpcyBwcm92aWRlZC4gVGhpcyBpcyBhIHBhZ2luYXRlZCBBUEkuIElmIHlvdSBwcm92aWRlIGEgbnVsbCBtYXhSZXN1bHRzLCB0aGlzIGFjdGlvbiByZXRyaWV2ZXMgYSBtYXhpbXVtIG9mIDUwIHJlY29yZHMgcGVyIHBhZ2UuIElmIHlvdSBwcm92aWRlIGEgbWF4UmVzdWx0cywgdGhlIHZhbHVlIG11c3QgYmUgYmV0d2VlbiAxMCBhbmQgNTAuIFRvIGdldCB0aGUgbmV4dCBwYWdlIHJlc3VsdHMsIHByb3ZpZGUgdGhlIHBhZ2luYXRpb24gdG9rZW4gZnJvbSB0aGUgR2V0R2V0TGFiZWxzUmVzcG9uc2UgYXMgcGFydCBvZiB5b3VyIHJlcXVlc3QuIEEgbnVsbCBwYWdpbmF0aW9uIHRva2VuIGZldGNoZXMgdGhlIHJlY29yZHMgZnJvbSB0aGUgYmVnaW5uaW5nLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2ZyYXVkZGV0ZWN0b3IvbGF0ZXN0L2FwaS9BUElfR2V0TGFiZWxzXG4gICAqL1xuICBwdWJsaWMgdG9HZXRMYWJlbHMoKSB7XG4gICAgdGhpcy5hZGQoJ2ZyYXVkZGV0ZWN0b3I6R2V0TGFiZWxzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR2V0cyB0aGUgZGV0YWlscyBvZiB0aGUgc3BlY2lmaWVkIG1vZGVsIHZlcnNpb24uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZnJhdWRkZXRlY3Rvci9sYXRlc3QvYXBpL0FQSV9HZXRNb2RlbFZlcnNpb25cbiAgICovXG4gIHB1YmxpYyB0b0dldE1vZGVsVmVyc2lvbigpIHtcbiAgICB0aGlzLmFkZCgnZnJhdWRkZXRlY3RvcjpHZXRNb2RlbFZlcnNpb24nKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHZXRzIG9uZSBvciBtb3JlIG1vZGVscy4gR2V0cyBhbGwgbW9kZWxzIGZvciB0aGUgQVdTIGFjY291bnQgaWYgbm8gbW9kZWwgdHlwZSBhbmQgbm8gbW9kZWwgaWQgcHJvdmlkZWQuIEdldHMgYWxsIG1vZGVscyBmb3IgdGhlIEFXUyBhY2NvdW50IGFuZCBtb2RlbCB0eXBlLCBpZiB0aGUgbW9kZWwgdHlwZSBpcyBzcGVjaWZpZWQgYnV0IG1vZGVsIGlkIGlzIG5vdCBwcm92aWRlZC4gR2V0cyBhIHNwZWNpZmljIG1vZGVsIGlmIChtb2RlbCB0eXBlLCBtb2RlbCBpZCkgdHVwbGUgaXMgc3BlY2lmaWVkLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2ZyYXVkZGV0ZWN0b3IvbGF0ZXN0L2FwaS9BUElfR2V0TW9kZWxzXG4gICAqL1xuICBwdWJsaWMgdG9HZXRNb2RlbHMoKSB7XG4gICAgdGhpcy5hZGQoJ2ZyYXVkZGV0ZWN0b3I6R2V0TW9kZWxzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR2V0cyBvbmUgb3IgbW9yZSBvdXRjb21lcy4gVGhpcyBpcyBhIHBhZ2luYXRlZCBBUEkuIElmIHlvdSBwcm92aWRlIGEgbnVsbCBtYXhSZXN1bHRzLCB0aGlzIGFjdGlvbnMgcmV0cmlldmVzIGEgbWF4aW11bSBvZiAxMDAgcmVjb3JkcyBwZXIgcGFnZS4gSWYgeW91IHByb3ZpZGUgYSBtYXhSZXN1bHRzLCB0aGUgdmFsdWUgbXVzdCBiZSBiZXR3ZWVuIDUwIGFuZCAxMDAuIFRvIGdldCB0aGUgbmV4dCBwYWdlIHJlc3VsdHMsIHByb3ZpZGUgdGhlIHBhZ2luYXRpb24gdG9rZW4gZnJvbSB0aGUgR2V0T3V0Y29tZXNSZXN1bHQgYXMgcGFydCBvZiB5b3VyIHJlcXVlc3QuIEEgbnVsbCBwYWdpbmF0aW9uIHRva2VuIGZldGNoZXMgdGhlIHJlY29yZHMgZnJvbSB0aGUgYmVnaW5uaW5nLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2ZyYXVkZGV0ZWN0b3IvbGF0ZXN0L2FwaS9BUElfR2V0T3V0Y29tZXNcbiAgICovXG4gIHB1YmxpYyB0b0dldE91dGNvbWVzKCkge1xuICAgIHRoaXMuYWRkKCdmcmF1ZGRldGVjdG9yOkdldE91dGNvbWVzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR2V0IGFsbCBydWxlcyBmb3IgYSBkZXRlY3RvciAocGFnaW5hdGVkKSBpZiBydWxlSWQgYW5kIHJ1bGVWZXJzaW9uIGFyZSBub3Qgc3BlY2lmaWVkLiBHZXRzIGFsbCBydWxlcyBmb3IgdGhlIGRldGVjdG9yIGFuZCB0aGUgcnVsZUlkIGlmIHByZXNlbnQgKHBhZ2luYXRlZCkuIEdldHMgYSBzcGVjaWZpYyBydWxlIGlmIGJvdGggdGhlIHJ1bGVJZCBhbmQgdGhlIHJ1bGVWZXJzaW9uIGFyZSBzcGVjaWZpZWQuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZnJhdWRkZXRlY3Rvci9sYXRlc3QvYXBpL0FQSV9HZXRSdWxlcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRSdWxlcygpIHtcbiAgICB0aGlzLmFkZCgnZnJhdWRkZXRlY3RvcjpHZXRSdWxlcycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdldHMgYWxsIG9mIHRoZSB2YXJpYWJsZXMgb3IgdGhlIHNwZWNpZmljIHZhcmlhYmxlLiBUaGlzIGlzIGEgcGFnaW5hdGVkIEFQSS4gUHJvdmlkaW5nIG51bGwgbWF4U2l6ZVBlclBhZ2UgcmVzdWx0cyBpbiByZXRyaWV2aW5nIG1heGltdW0gb2YgMTAwIHJlY29yZHMgcGVyIHBhZ2UuIElmIHlvdSBwcm92aWRlIG1heFNpemVQZXJQYWdlIHRoZSB2YWx1ZSBtdXN0IGJlIGJldHdlZW4gNTAgYW5kIDEwMC4gVG8gZ2V0IHRoZSBuZXh0IHBhZ2UgcmVzdWx0LCBhIHByb3ZpZGUgYSBwYWdpbmF0aW9uIHRva2VuIGZyb20gR2V0VmFyaWFibGVzUmVzdWx0IGFzIHBhcnQgb2YgeW91ciByZXF1ZXN0LiBOdWxsIHBhZ2luYXRpb24gdG9rZW4gZmV0Y2hlcyB0aGUgcmVjb3JkcyBmcm9tIHRoZSBiZWdpbm5pbmcuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZnJhdWRkZXRlY3Rvci9sYXRlc3QvYXBpL0FQSV9HZXRWYXJpYWJsZXNcbiAgICovXG4gIHB1YmxpYyB0b0dldFZhcmlhYmxlcygpIHtcbiAgICB0aGlzLmFkZCgnZnJhdWRkZXRlY3RvcjpHZXRWYXJpYWJsZXMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBMaXN0cyBhbGwgdGFncyBhc3NvY2lhdGVkIHdpdGggdGhlIHJlc291cmNlLiBUaGlzIGlzIGEgcGFnaW5hdGVkIEFQSS4gVG8gZ2V0IHRoZSBuZXh0IHBhZ2UgcmVzdWx0cywgcHJvdmlkZSB0aGUgcGFnaW5hdGlvbiB0b2tlbiBmcm9tIHRoZSByZXNwb25zZSBhcyBwYXJ0IG9mIHlvdXIgcmVxdWVzdC4gQSBudWxsIHBhZ2luYXRpb24gdG9rZW4gZmV0Y2hlcyB0aGUgcmVjb3JkcyBmcm9tIHRoZSBiZWdpbm5pbmcuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZnJhdWRkZXRlY3Rvci9sYXRlc3QvYXBpL0FQSV9MaXN0VGFnc0ZvclJlc291cmNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RUYWdzRm9yUmVzb3VyY2UoKSB7XG4gICAgdGhpcy5hZGQoJ2ZyYXVkZGV0ZWN0b3I6TGlzdFRhZ3NGb3JSZXNvdXJjZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIENyZWF0ZXMgb3IgdXBkYXRlcyBhIGRldGVjdG9yLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9mcmF1ZGRldGVjdG9yL2xhdGVzdC9hcGkvQVBJX1B1dERldGVjdG9yLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1B1dERldGVjdG9yKCkge1xuICAgIHRoaXMuYWRkKCdmcmF1ZGRldGVjdG9yOlB1dERldGVjdG9yJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogQ3JlYXRlcyBvciB1cGRhdGVzIGFuIGVudGl0eSB0eXBlLiBBbiBlbnRpdHkgcmVwcmVzZW50cyB3aG8gaXMgcGVyZm9ybWluZyB0aGUgZXZlbnQuIEFzIHBhcnQgb2YgYSBmcmF1ZCBwcmVkaWN0aW9uLCB5b3UgcGFzcyB0aGUgZW50aXR5IElEIHRvIGluZGljYXRlIHRoZSBzcGVjaWZpYyBlbnRpdHkgd2hvIHBlcmZvcm1lZCB0aGUgZXZlbnQuIEFuIGVudGl0eSB0eXBlIGNsYXNzaWZpZXMgdGhlIGVudGl0eS4gRXhhbXBsZSBjbGFzc2lmaWNhdGlvbnMgaW5jbHVkZSBjdXN0b21lciwgbWVyY2hhbnQsIG9yIGFjY291bnQuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2ZyYXVkZGV0ZWN0b3IvbGF0ZXN0L2FwaS9BUElfUHV0RW50aXR5VHlwZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9QdXRFbnRpdHlUeXBlKCkge1xuICAgIHRoaXMuYWRkKCdmcmF1ZGRldGVjdG9yOlB1dEVudGl0eVR5cGUnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBDcmVhdGVzIG9yIHVwZGF0ZXMgYW4gZXZlbnQgdHlwZS4gQW4gZXZlbnQgaXMgYSBidXNpbmVzcyBhY3Rpdml0eSB0aGF0IGlzIGV2YWx1YXRlZCBmb3IgZnJhdWQgcmlzay4gV2l0aCBBbWF6b24gRnJhdWQgRGV0ZWN0b3IsIHlvdSBnZW5lcmF0ZSBmcmF1ZCBwcmVkaWN0aW9ucyBmb3IgZXZlbnRzLiBBbiBldmVudCB0eXBlIGRlZmluZXMgdGhlIHN0cnVjdHVyZSBmb3IgYW4gZXZlbnQgc2VudCB0byBBbWF6b24gRnJhdWQgRGV0ZWN0b3IuIFRoaXMgaW5jbHVkZXMgdGhlIHZhcmlhYmxlcyBzZW50IGFzIHBhcnQgb2YgdGhlIGV2ZW50LCB0aGUgZW50aXR5IHBlcmZvcm1pbmcgdGhlIGV2ZW50IChzdWNoIGFzIGEgY3VzdG9tZXIpLCBhbmQgdGhlIGxhYmVscyB0aGF0IGNsYXNzaWZ5IHRoZSBldmVudC4gRXhhbXBsZSBldmVudCB0eXBlcyBpbmNsdWRlIG9ubGluZSBwYXltZW50IHRyYW5zYWN0aW9ucywgYWNjb3VudCByZWdpc3RyYXRpb25zLCBhbmQgYXV0aGVudGljYXRpb25zLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9mcmF1ZGRldGVjdG9yL2xhdGVzdC9hcGkvQVBJX1B1dEV2ZW50VHlwZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9QdXRFdmVudFR5cGUoKSB7XG4gICAgdGhpcy5hZGQoJ2ZyYXVkZGV0ZWN0b3I6UHV0RXZlbnRUeXBlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogQ3JlYXRlcyBvciB1cGRhdGVzIGFuIEFtYXpvbiBTYWdlTWFrZXIgbW9kZWwgZW5kcG9pbnQuIFlvdSBjYW4gYWxzbyB1c2UgdGhpcyBhY3Rpb24gdG8gdXBkYXRlIHRoZSBjb25maWd1cmF0aW9uIG9mIHRoZSBtb2RlbCBlbmRwb2ludCwgaW5jbHVkaW5nIHRoZSBJQU0gcm9sZSBhbmQvb3IgdGhlIG1hcHBlZCB2YXJpYWJsZXMuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2ZyYXVkZGV0ZWN0b3IvbGF0ZXN0L2FwaS9BUElfUHV0RXh0ZXJuYWxNb2RlbC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9QdXRFeHRlcm5hbE1vZGVsKCkge1xuICAgIHRoaXMuYWRkKCdmcmF1ZGRldGVjdG9yOlB1dEV4dGVybmFsTW9kZWwnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBTcGVjaWZpZXMgdGhlIEtleSBNYW5hZ2VtZW50IFNlcnZpY2UgKEtNUykgY3VzdG9tZXIgbWFzdGVyIGtleSAoQ01LKSB0byBiZSB1c2VkIHRvIGVuY3J5cHQgY29udGVudCBpbiBBbWF6b24gRnJhdWQgRGV0ZWN0b3IuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2ZyYXVkZGV0ZWN0b3IvbGF0ZXN0L2FwaS9BUElfUHV0S01TRW5jcnlwdGlvbktleS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9QdXRLTVNFbmNyeXB0aW9uS2V5KCkge1xuICAgIHRoaXMuYWRkKCdmcmF1ZGRldGVjdG9yOlB1dEtNU0VuY3J5cHRpb25LZXknKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBDcmVhdGVzIG9yIHVwZGF0ZXMgbGFiZWwuIEEgbGFiZWwgY2xhc3NpZmllcyBhbiBldmVudCBhcyBmcmF1ZHVsZW50IG9yIGxlZ2l0aW1hdGUuIExhYmVscyBhcmUgYXNzb2NpYXRlZCB3aXRoIGV2ZW50IHR5cGVzIGFuZCB1c2VkIHRvIHRyYWluIHN1cGVydmlzZWQgbWFjaGluZSBsZWFybmluZyBtb2RlbHMgaW4gQW1hem9uIEZyYXVkIERldGVjdG9yLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9mcmF1ZGRldGVjdG9yL2xhdGVzdC9hcGkvQVBJX1B1dExhYmVsLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1B1dExhYmVsKCkge1xuICAgIHRoaXMuYWRkKCdmcmF1ZGRldGVjdG9yOlB1dExhYmVsJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogQ3JlYXRlcyBvciB1cGRhdGVzIGFuIG91dGNvbWUuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2ZyYXVkZGV0ZWN0b3IvbGF0ZXN0L2FwaS9BUElfUHV0T3V0Y29tZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9QdXRPdXRjb21lKCkge1xuICAgIHRoaXMuYWRkKCdmcmF1ZGRldGVjdG9yOlB1dE91dGNvbWUnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBBc3NpZ25zIHRhZ3MgdG8gYSByZXNvdXJjZS5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBUYWdnaW5nXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9mcmF1ZGRldGVjdG9yL2xhdGVzdC9hcGkvQVBJX1RhZ1Jlc291cmNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1RhZ1Jlc291cmNlKCkge1xuICAgIHRoaXMuYWRkKCdmcmF1ZGRldGVjdG9yOlRhZ1Jlc291cmNlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmVtb3ZlcyB0YWdzIGZyb20gYSByZXNvdXJjZS5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBUYWdnaW5nXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9mcmF1ZGRldGVjdG9yL2xhdGVzdC9hcGkvQVBJX1VudGFnUmVzb3VyY2UuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVW50YWdSZXNvdXJjZSgpIHtcbiAgICB0aGlzLmFkZCgnZnJhdWRkZXRlY3RvcjpVbnRhZ1Jlc291cmNlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogVXBkYXRlcyBhIGRldGVjdG9yIHZlcnNpb24uIFRoZSBkZXRlY3RvciB2ZXJzaW9uIGF0dHJpYnV0ZXMgdGhhdCB5b3UgY2FuIHVwZGF0ZSBpbmNsdWRlIG1vZGVscywgZXh0ZXJuYWwgbW9kZWwgZW5kcG9pbnRzLCBydWxlcywgcnVsZSBleGVjdXRpb24gbW9kZSwgYW5kIGRlc2NyaXB0aW9uLiBZb3UgY2FuIG9ubHkgdXBkYXRlIGEgRFJBRlQgZGV0ZWN0b3IgdmVyc2lvbi5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZnJhdWRkZXRlY3Rvci9sYXRlc3QvYXBpL0FQSV9VcGRhdGVEZXRlY3RvclZlcnNpb25cbiAgICovXG4gIHB1YmxpYyB0b1VwZGF0ZURldGVjdG9yVmVyc2lvbigpIHtcbiAgICB0aGlzLmFkZCgnZnJhdWRkZXRlY3RvcjpVcGRhdGVEZXRlY3RvclZlcnNpb24nKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBVcGRhdGVzIHRoZSBkZXRlY3RvciB2ZXJzaW9uJ3MgZGVzY3JpcHRpb24uIFlvdSBjYW4gdXBkYXRlIHRoZSBtZXRhZGF0YSBmb3IgYW55IGRldGVjdG9yIHZlcnNpb24gKERSQUZULCBBQ1RJVkUsIG9yIElOQUNUSVZFKS5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZnJhdWRkZXRlY3Rvci9sYXRlc3QvYXBpL0FQSV9VcGRhdGVEZXRlY3RvclZlcnNpb25NZXRhZGF0YVxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlRGV0ZWN0b3JWZXJzaW9uTWV0YWRhdGEoKSB7XG4gICAgdGhpcy5hZGQoJ2ZyYXVkZGV0ZWN0b3I6VXBkYXRlRGV0ZWN0b3JWZXJzaW9uTWV0YWRhdGEnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBVcGRhdGVzIHRoZSBkZXRlY3RvciB2ZXJzaW9u4oCZcyBzdGF0dXMuIFlvdSBjYW4gcGVyZm9ybSB0aGUgZm9sbG93aW5nIHByb21vdGlvbnMgb3IgZGVtb3Rpb25zIHVzaW5nIFVwZGF0ZURldGVjdG9yVmVyc2lvblN0YXR1czogRFJBRlQgdG8gQUNUSVZFLCBBQ1RJVkUgdG8gSU5BQ1RJVkUsIGFuZCBJTkFDVElWRSB0byBBQ1RJVkUuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2ZyYXVkZGV0ZWN0b3IvbGF0ZXN0L2FwaS9BUElfVXBkYXRlRGV0ZWN0b3JWZXJzaW9uU3RhdHVzXG4gICAqL1xuICBwdWJsaWMgdG9VcGRhdGVEZXRlY3RvclZlcnNpb25TdGF0dXMoKSB7XG4gICAgdGhpcy5hZGQoJ2ZyYXVkZGV0ZWN0b3I6VXBkYXRlRGV0ZWN0b3JWZXJzaW9uU3RhdHVzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogVXBkYXRlcyBhIG1vZGVsLiBZb3UgY2FuIHVwZGF0ZSB0aGUgZGVzY3JpcHRpb24gYXR0cmlidXRlIHVzaW5nIHRoaXMgYWN0aW9uLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9mcmF1ZGRldGVjdG9yL2xhdGVzdC9hcGkvQVBJX1VwZGF0ZU1vZGVsXG4gICAqL1xuICBwdWJsaWMgdG9VcGRhdGVNb2RlbCgpIHtcbiAgICB0aGlzLmFkZCgnZnJhdWRkZXRlY3RvcjpVcGRhdGVNb2RlbCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFVwZGF0ZXMgYSBtb2RlbCB2ZXJzaW9uLiBVcGRhdGluZyBhIG1vZGVsIHZlcnNpb24gcmV0cmFpbnMgYW4gZXhpc3RpbmcgbW9kZWwgdmVyc2lvbiB1c2luZyB1cGRhdGVkIHRyYWluaW5nIGRhdGEgYW5kIHByb2R1Y2VzIGEgbmV3IG1pbm9yIHZlcnNpb24gb2YgdGhlIG1vZGVsLiBZb3UgY2FuIHVwZGF0ZSB0aGUgdHJhaW5pbmcgZGF0YSBzZXQgbG9jYXRpb24gYW5kIGRhdGEgYWNjZXNzIHJvbGUgYXR0cmlidXRlcyB1c2luZyB0aGlzIGFjdGlvbi4gVGhpcyBhY3Rpb24gY3JlYXRlcyBhbmQgdHJhaW5zIGEgbmV3IG1pbm9yIHZlcnNpb24gb2YgdGhlIG1vZGVsLCBmb3IgZXhhbXBsZSB2ZXJzaW9uIDEuMDEsIDEuMDIsIDEuMDMuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2ZyYXVkZGV0ZWN0b3IvbGF0ZXN0L2FwaS9BUElfVXBkYXRlTW9kZWxWZXJzaW9uXG4gICAqL1xuICBwdWJsaWMgdG9VcGRhdGVNb2RlbFZlcnNpb24oKSB7XG4gICAgdGhpcy5hZGQoJ2ZyYXVkZGV0ZWN0b3I6VXBkYXRlTW9kZWxWZXJzaW9uJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogVXBkYXRlcyB0aGUgc3RhdHVzIG9mIGEgbW9kZWwgdmVyc2lvbi5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZnJhdWRkZXRlY3Rvci9sYXRlc3QvYXBpL0FQSV9VcGRhdGVNb2RlbFZlcnNpb25TdGF0dXNcbiAgICovXG4gIHB1YmxpYyB0b1VwZGF0ZU1vZGVsVmVyc2lvblN0YXR1cygpIHtcbiAgICB0aGlzLmFkZCgnZnJhdWRkZXRlY3RvcjpVcGRhdGVNb2RlbFZlcnNpb25TdGF0dXMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBVcGRhdGVzIGEgcnVsZSdzIG1ldGFkYXRhLiBUaGUgZGVzY3JpcHRpb24gYXR0cmlidXRlIGNhbiBiZSB1cGRhdGVkLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9mcmF1ZGRldGVjdG9yL2xhdGVzdC9hcGkvQVBJX1VwZGF0ZVJ1bGVNZXRhZGF0YS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VcGRhdGVSdWxlTWV0YWRhdGEoKSB7XG4gICAgdGhpcy5hZGQoJ2ZyYXVkZGV0ZWN0b3I6VXBkYXRlUnVsZU1ldGFkYXRhJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogVXBkYXRlcyBhIHJ1bGUgdmVyc2lvbiByZXN1bHRpbmcgaW4gYSBuZXcgcnVsZSB2ZXJzaW9uLiBVcGRhdGVzIGEgcnVsZSB2ZXJzaW9uIHJlc3VsdGluZyBpbiBhIG5ldyBydWxlIHZlcnNpb24gKHZlcnNpb24gMSwgMiwgMyAuLi4pLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9mcmF1ZGRldGVjdG9yL2xhdGVzdC9hcGkvQVBJX1VwZGF0ZVJ1bGVWZXJzaW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1VwZGF0ZVJ1bGVWZXJzaW9uKCkge1xuICAgIHRoaXMuYWRkKCdmcmF1ZGRldGVjdG9yOlVwZGF0ZVJ1bGVWZXJzaW9uJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogVXBkYXRlcyBhIHZhcmlhYmxlLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9mcmF1ZGRldGVjdG9yL2xhdGVzdC9hcGkvQVBJX1VwZGF0ZVZhcmlhYmxlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1VwZGF0ZVZhcmlhYmxlKCkge1xuICAgIHRoaXMuYWRkKCdmcmF1ZGRldGVjdG9yOlVwZGF0ZVZhcmlhYmxlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgZGV0ZWN0b3IgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBAcGFyYW0gcmVzb3VyY2VQYXRoIC0gSWRlbnRpZmllciBmb3IgdGhlIHJlc291cmNlUGF0aC5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgYWNjb3VudHMuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCByZWdpb25zLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYC5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25EZXRlY3RvcihyZXNvdXJjZVBhdGg6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICB2YXIgYXJuID0gJ2Fybjoke1BhcnRpdGlvbn06ZnJhdWRkZXRlY3Rvcjoke1JlZ2lvbn06JHtBY2NvdW50fTpkZXRlY3Rvci8ke1Jlc291cmNlUGF0aH0nO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1Jlc291cmNlUGF0aH0nLCByZXNvdXJjZVBhdGgpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0FjY291bnR9JywgYWNjb3VudCB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1JlZ2lvbn0nLCByZWdpb24gfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtQYXJ0aXRpb259JywgcGFydGl0aW9uIHx8ICdhd3MnKTtcbiAgICByZXR1cm4gdGhpcy5vbihhcm4pO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIGRldGVjdG9yLXZlcnNpb24gdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBAcGFyYW0gcmVzb3VyY2VQYXRoIC0gSWRlbnRpZmllciBmb3IgdGhlIHJlc291cmNlUGF0aC5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgYWNjb3VudHMuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCByZWdpb25zLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYC5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25EZXRlY3RvclZlcnNpb24ocmVzb3VyY2VQYXRoOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgdmFyIGFybiA9ICdhcm46JHtQYXJ0aXRpb259OmZyYXVkZGV0ZWN0b3I6JHtSZWdpb259OiR7QWNjb3VudH06ZGV0ZWN0b3ItdmVyc2lvbi8ke1Jlc291cmNlUGF0aH0nO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1Jlc291cmNlUGF0aH0nLCByZXNvdXJjZVBhdGgpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0FjY291bnR9JywgYWNjb3VudCB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1JlZ2lvbn0nLCByZWdpb24gfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtQYXJ0aXRpb259JywgcGFydGl0aW9uIHx8ICdhd3MnKTtcbiAgICByZXR1cm4gdGhpcy5vbihhcm4pO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIGVudGl0eS10eXBlIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogQHBhcmFtIHJlc291cmNlUGF0aCAtIElkZW50aWZpZXIgZm9yIHRoZSByZXNvdXJjZVBhdGguXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIGFjY291bnRzLlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgcmVnaW9ucy5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AuXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uRW50aXR5VHlwZShyZXNvdXJjZVBhdGg6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICB2YXIgYXJuID0gJ2Fybjoke1BhcnRpdGlvbn06ZnJhdWRkZXRlY3Rvcjoke1JlZ2lvbn06JHtBY2NvdW50fTplbnRpdHktdHlwZS8ke1Jlc291cmNlUGF0aH0nO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1Jlc291cmNlUGF0aH0nLCByZXNvdXJjZVBhdGgpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0FjY291bnR9JywgYWNjb3VudCB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1JlZ2lvbn0nLCByZWdpb24gfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtQYXJ0aXRpb259JywgcGFydGl0aW9uIHx8ICdhd3MnKTtcbiAgICByZXR1cm4gdGhpcy5vbihhcm4pO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIGV4dGVybmFsLW1vZGVsIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogQHBhcmFtIHJlc291cmNlUGF0aCAtIElkZW50aWZpZXIgZm9yIHRoZSByZXNvdXJjZVBhdGguXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIGFjY291bnRzLlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgcmVnaW9ucy5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AuXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uRXh0ZXJuYWxNb2RlbChyZXNvdXJjZVBhdGg6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICB2YXIgYXJuID0gJ2Fybjoke1BhcnRpdGlvbn06ZnJhdWRkZXRlY3Rvcjoke1JlZ2lvbn06JHtBY2NvdW50fTpleHRlcm5hbC1tb2RlbC8ke1Jlc291cmNlUGF0aH0nO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1Jlc291cmNlUGF0aH0nLCByZXNvdXJjZVBhdGgpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0FjY291bnR9JywgYWNjb3VudCB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1JlZ2lvbn0nLCByZWdpb24gfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtQYXJ0aXRpb259JywgcGFydGl0aW9uIHx8ICdhd3MnKTtcbiAgICByZXR1cm4gdGhpcy5vbihhcm4pO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIGV2ZW50LXR5cGUgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBAcGFyYW0gcmVzb3VyY2VQYXRoIC0gSWRlbnRpZmllciBmb3IgdGhlIHJlc291cmNlUGF0aC5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgYWNjb3VudHMuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCByZWdpb25zLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYC5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25FdmVudFR5cGUocmVzb3VyY2VQYXRoOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgdmFyIGFybiA9ICdhcm46JHtQYXJ0aXRpb259OmZyYXVkZGV0ZWN0b3I6JHtSZWdpb259OiR7QWNjb3VudH06ZXZlbnQtdHlwZS8ke1Jlc291cmNlUGF0aH0nO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1Jlc291cmNlUGF0aH0nLCByZXNvdXJjZVBhdGgpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0FjY291bnR9JywgYWNjb3VudCB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1JlZ2lvbn0nLCByZWdpb24gfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtQYXJ0aXRpb259JywgcGFydGl0aW9uIHx8ICdhd3MnKTtcbiAgICByZXR1cm4gdGhpcy5vbihhcm4pO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIGxhYmVsIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogQHBhcmFtIHJlc291cmNlUGF0aCAtIElkZW50aWZpZXIgZm9yIHRoZSByZXNvdXJjZVBhdGguXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIGFjY291bnRzLlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgcmVnaW9ucy5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AuXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uTGFiZWwocmVzb3VyY2VQYXRoOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgdmFyIGFybiA9ICdhcm46JHtQYXJ0aXRpb259OmZyYXVkZGV0ZWN0b3I6JHtSZWdpb259OiR7QWNjb3VudH06bGFiZWwvJHtSZXNvdXJjZVBhdGh9JztcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtSZXNvdXJjZVBhdGh9JywgcmVzb3VyY2VQYXRoKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtBY2NvdW50fScsIGFjY291bnQgfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtSZWdpb259JywgcmVnaW9uIHx8ICcqJyk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UGFydGl0aW9ufScsIHBhcnRpdGlvbiB8fCAnYXdzJyk7XG4gICAgcmV0dXJuIHRoaXMub24oYXJuKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBtb2RlbCB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIEBwYXJhbSByZXNvdXJjZVBhdGggLSBJZGVudGlmaWVyIGZvciB0aGUgcmVzb3VyY2VQYXRoLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCBhY2NvdW50cy5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIHJlZ2lvbnMuXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvbk1vZGVsKHJlc291cmNlUGF0aDogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHZhciBhcm4gPSAnYXJuOiR7UGFydGl0aW9ufTpmcmF1ZGRldGVjdG9yOiR7UmVnaW9ufToke0FjY291bnR9Om1vZGVsLyR7UmVzb3VyY2VQYXRofSc7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UmVzb3VyY2VQYXRofScsIHJlc291cmNlUGF0aCk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7QWNjb3VudH0nLCBhY2NvdW50IHx8ICcqJyk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UmVnaW9ufScsIHJlZ2lvbiB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1BhcnRpdGlvbn0nLCBwYXJ0aXRpb24gfHwgJ2F3cycpO1xuICAgIHJldHVybiB0aGlzLm9uKGFybik7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgbW9kZWwtdmVyc2lvbiB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIEBwYXJhbSByZXNvdXJjZVBhdGggLSBJZGVudGlmaWVyIGZvciB0aGUgcmVzb3VyY2VQYXRoLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCBhY2NvdW50cy5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIHJlZ2lvbnMuXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvbk1vZGVsVmVyc2lvbihyZXNvdXJjZVBhdGg6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICB2YXIgYXJuID0gJ2Fybjoke1BhcnRpdGlvbn06ZnJhdWRkZXRlY3Rvcjoke1JlZ2lvbn06JHtBY2NvdW50fTptb2RlbC12ZXJzaW9uLyR7UmVzb3VyY2VQYXRofSc7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UmVzb3VyY2VQYXRofScsIHJlc291cmNlUGF0aCk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7QWNjb3VudH0nLCBhY2NvdW50IHx8ICcqJyk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UmVnaW9ufScsIHJlZ2lvbiB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1BhcnRpdGlvbn0nLCBwYXJ0aXRpb24gfHwgJ2F3cycpO1xuICAgIHJldHVybiB0aGlzLm9uKGFybik7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgb3V0Y29tZSB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIEBwYXJhbSByZXNvdXJjZVBhdGggLSBJZGVudGlmaWVyIGZvciB0aGUgcmVzb3VyY2VQYXRoLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCBhY2NvdW50cy5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIHJlZ2lvbnMuXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvbk91dGNvbWUocmVzb3VyY2VQYXRoOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgdmFyIGFybiA9ICdhcm46JHtQYXJ0aXRpb259OmZyYXVkZGV0ZWN0b3I6JHtSZWdpb259OiR7QWNjb3VudH06b3V0Y29tZS8ke1Jlc291cmNlUGF0aH0nO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1Jlc291cmNlUGF0aH0nLCByZXNvdXJjZVBhdGgpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0FjY291bnR9JywgYWNjb3VudCB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1JlZ2lvbn0nLCByZWdpb24gfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtQYXJ0aXRpb259JywgcGFydGl0aW9uIHx8ICdhd3MnKTtcbiAgICByZXR1cm4gdGhpcy5vbihhcm4pO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIHJ1bGUgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBAcGFyYW0gcmVzb3VyY2VQYXRoIC0gSWRlbnRpZmllciBmb3IgdGhlIHJlc291cmNlUGF0aC5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgYWNjb3VudHMuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCByZWdpb25zLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYC5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25SdWxlKHJlc291cmNlUGF0aDogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHZhciBhcm4gPSAnYXJuOiR7UGFydGl0aW9ufTpmcmF1ZGRldGVjdG9yOiR7UmVnaW9ufToke0FjY291bnR9OnJ1bGUvJHtSZXNvdXJjZVBhdGh9JztcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtSZXNvdXJjZVBhdGh9JywgcmVzb3VyY2VQYXRoKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtBY2NvdW50fScsIGFjY291bnQgfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtSZWdpb259JywgcmVnaW9uIHx8ICcqJyk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UGFydGl0aW9ufScsIHBhcnRpdGlvbiB8fCAnYXdzJyk7XG4gICAgcmV0dXJuIHRoaXMub24oYXJuKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSB2YXJpYWJsZSB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIEBwYXJhbSByZXNvdXJjZVBhdGggLSBJZGVudGlmaWVyIGZvciB0aGUgcmVzb3VyY2VQYXRoLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCBhY2NvdW50cy5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIHJlZ2lvbnMuXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvblZhcmlhYmxlKHJlc291cmNlUGF0aDogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHZhciBhcm4gPSAnYXJuOiR7UGFydGl0aW9ufTpmcmF1ZGRldGVjdG9yOiR7UmVnaW9ufToke0FjY291bnR9OnZhcmlhYmxlLyR7UmVzb3VyY2VQYXRofSc7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UmVzb3VyY2VQYXRofScsIHJlc291cmNlUGF0aCk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7QWNjb3VudH0nLCBhY2NvdW50IHx8ICcqJyk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UmVnaW9ufScsIHJlZ2lvbiB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1BhcnRpdGlvbn0nLCBwYXJ0aXRpb24gfHwgJ2F3cycpO1xuICAgIHJldHVybiB0aGlzLm9uKGFybik7XG4gIH1cbn1cbiJdfQ==