"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Firehose = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [firehose](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonkinesisfirehose.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Firehose extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [firehose](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonkinesisfirehose.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'firehose';
        this.actionList = {
            "CreateDeliveryStream": {
                "url": "https://docs.aws.amazon.com/firehose/latest/APIReference/API_CreateDeliveryStream.html",
                "description": "Creates a delivery stream.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "deliverystream": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "DeleteDeliveryStream": {
                "url": "https://docs.aws.amazon.com/firehose/latest/APIReference/API_DeleteDeliveryStream.html",
                "description": "Deletes a delivery stream and its data.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "deliverystream": {
                        "required": true
                    }
                }
            },
            "DescribeDeliveryStream": {
                "url": "https://docs.aws.amazon.com/firehose/latest/APIReference/API_DescribeDeliveryStream.html",
                "description": "Describes the specified delivery stream and gets the status.",
                "accessLevel": "List",
                "resourceTypes": {
                    "deliverystream": {
                        "required": true
                    }
                }
            },
            "ListDeliveryStreams": {
                "url": "https://docs.aws.amazon.com/firehose/latest/APIReference/API_ListDeliveryStreams.html",
                "description": "Lists your delivery streams.",
                "accessLevel": "List"
            },
            "ListTagsForDeliveryStream": {
                "url": "https://docs.aws.amazon.com/firehose/latest/APIReference/API_ListTagsForDeliveryStream.html",
                "description": "Lists the tags for the specified delivery stream.",
                "accessLevel": "List",
                "resourceTypes": {
                    "deliverystream": {
                        "required": true
                    }
                }
            },
            "PutRecord": {
                "url": "https://docs.aws.amazon.com/firehose/latest/APIReference/API_PutRecord.html",
                "description": "Writes a single data record into an Amazon Kinesis Firehose delivery stream.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "deliverystream": {
                        "required": true
                    }
                }
            },
            "PutRecordBatch": {
                "url": "https://docs.aws.amazon.com/firehose/latest/APIReference/API_PutRecordBatch.html",
                "description": "Writes multiple data records into a delivery stream in a single call, which can achieve higher throughput per producer than when writing single records.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "deliverystream": {
                        "required": true
                    }
                }
            },
            "StartDeliveryStreamEncryption": {
                "url": "https://docs.aws.amazon.com/firehose/latest/APIReference/API_StartDeliveryStreamEncryption.html",
                "description": "Enables server-side encryption (SSE) for the delivery stream.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "deliverystream": {
                        "required": true
                    }
                }
            },
            "StopDeliveryStreamEncryption": {
                "url": "https://docs.aws.amazon.com/firehose/latest/APIReference/API_StopDeliveryStreamEncryption.html",
                "description": "Disables the specified destination of the specified delivery stream.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "deliverystream": {
                        "required": true
                    }
                }
            },
            "TagDeliveryStream": {
                "url": "https://docs.aws.amazon.com/firehose/latest/APIReference/API_TagDeliveryStream.html",
                "description": "Adds or updates tags for the specified delivery stream.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "deliverystream": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UntagDeliveryStream": {
                "url": "https://docs.aws.amazon.com/firehose/latest/APIReference/API_UntagDeliveryStream.html",
                "description": "Removes tags from the specified delivery stream.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "deliverystream": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "UpdateDestination": {
                "url": "https://docs.aws.amazon.com/firehose/latest/APIReference/API_UpdateDestination.html",
                "description": "Updates the specified destination of the specified delivery stream.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "deliverystream": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "deliverystream": {
                "name": "deliverystream",
                "url": "https://docs.aws.amazon.com/firehose/latest/dev/basic-create.html",
                "arn": "arn:${Partition}:firehose:${Region}:${Account}:deliverystream/${DeliveryStreamName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Creates a delivery stream.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/firehose/latest/APIReference/API_CreateDeliveryStream.html
     */
    toCreateDeliveryStream() {
        this.add('firehose:CreateDeliveryStream');
        return this;
    }
    /**
     * Deletes a delivery stream and its data.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/firehose/latest/APIReference/API_DeleteDeliveryStream.html
     */
    toDeleteDeliveryStream() {
        this.add('firehose:DeleteDeliveryStream');
        return this;
    }
    /**
     * Describes the specified delivery stream and gets the status.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/firehose/latest/APIReference/API_DescribeDeliveryStream.html
     */
    toDescribeDeliveryStream() {
        this.add('firehose:DescribeDeliveryStream');
        return this;
    }
    /**
     * Lists your delivery streams.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/firehose/latest/APIReference/API_ListDeliveryStreams.html
     */
    toListDeliveryStreams() {
        this.add('firehose:ListDeliveryStreams');
        return this;
    }
    /**
     * Lists the tags for the specified delivery stream.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/firehose/latest/APIReference/API_ListTagsForDeliveryStream.html
     */
    toListTagsForDeliveryStream() {
        this.add('firehose:ListTagsForDeliveryStream');
        return this;
    }
    /**
     * Writes a single data record into an Amazon Kinesis Firehose delivery stream.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/firehose/latest/APIReference/API_PutRecord.html
     */
    toPutRecord() {
        this.add('firehose:PutRecord');
        return this;
    }
    /**
     * Writes multiple data records into a delivery stream in a single call, which can achieve higher throughput per producer than when writing single records.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/firehose/latest/APIReference/API_PutRecordBatch.html
     */
    toPutRecordBatch() {
        this.add('firehose:PutRecordBatch');
        return this;
    }
    /**
     * Enables server-side encryption (SSE) for the delivery stream.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/firehose/latest/APIReference/API_StartDeliveryStreamEncryption.html
     */
    toStartDeliveryStreamEncryption() {
        this.add('firehose:StartDeliveryStreamEncryption');
        return this;
    }
    /**
     * Disables the specified destination of the specified delivery stream.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/firehose/latest/APIReference/API_StopDeliveryStreamEncryption.html
     */
    toStopDeliveryStreamEncryption() {
        this.add('firehose:StopDeliveryStreamEncryption');
        return this;
    }
    /**
     * Adds or updates tags for the specified delivery stream.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/firehose/latest/APIReference/API_TagDeliveryStream.html
     */
    toTagDeliveryStream() {
        this.add('firehose:TagDeliveryStream');
        return this;
    }
    /**
     * Removes tags from the specified delivery stream.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/firehose/latest/APIReference/API_UntagDeliveryStream.html
     */
    toUntagDeliveryStream() {
        this.add('firehose:UntagDeliveryStream');
        return this;
    }
    /**
     * Updates the specified destination of the specified delivery stream.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/firehose/latest/APIReference/API_UpdateDestination.html
     */
    toUpdateDestination() {
        this.add('firehose:UpdateDestination');
        return this;
    }
    /**
     * Adds a resource of type deliverystream to the statement
     *
     * https://docs.aws.amazon.com/firehose/latest/dev/basic-create.html
     *
     * @param deliveryStreamName - Identifier for the deliveryStreamName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDeliverystream(deliveryStreamName, account, region, partition) {
        var arn = 'arn:${Partition}:firehose:${Region}:${Account}:deliverystream/${DeliveryStreamName}';
        arn = arn.replace('${DeliveryStreamName}', deliveryStreamName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Firehose = Firehose;
//# sourceMappingURL=data:application/json;base64,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