"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Lex = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [lex](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonlex.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Lex extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [lex](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonlex.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'lex';
        this.actionList = {
            "CreateBotVersion": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_CreateBotVersion.html",
                "description": "Creates a new version based on the $LATEST version of the specified bot.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bot version": {
                        "required": true
                    }
                }
            },
            "CreateIntentVersion": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_CreateIntentVersion.html",
                "description": "Creates a new version based on the $LATEST version of the specified intent.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "intent version": {
                        "required": true
                    }
                }
            },
            "CreateSlotTypeVersion": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_CreateSlotTypeVersion.html",
                "description": "Creates a new version based on the $LATEST version of the specified slot type.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "slottype version": {
                        "required": true
                    }
                }
            },
            "DeleteBot": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_DeleteBot.html",
                "description": "Deletes all versions of a bot.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bot version": {
                        "required": true
                    }
                }
            },
            "DeleteBotAlias": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_DeleteBotAlias.html",
                "description": "Deletes an alias for a specific bot.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bot alias": {
                        "required": true
                    }
                }
            },
            "DeleteBotChannelAssociation": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_DeleteBotChannelAssociation.html",
                "description": "Deletes the association between a Amazon Lex bot alias and a messaging platform.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "channel": {
                        "required": true
                    }
                }
            },
            "DeleteBotVersion": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_DeleteBotVersion.html",
                "description": "Deletes a specific version of a bot.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bot version": {
                        "required": true
                    }
                }
            },
            "DeleteIntent": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_DeleteIntent.html",
                "description": "Deletes all versions of an intent.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "intent version": {
                        "required": true
                    }
                }
            },
            "DeleteIntentVersion": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_DeleteIntentVersion.html",
                "description": "Deletes a specific version of an intent.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "intent version": {
                        "required": true
                    }
                }
            },
            "DeleteSession": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_runtime_DeleteSession.html",
                "description": "Removes session information for a specified bot, alias, and user ID.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bot alias": {
                        "required": false
                    },
                    "bot version": {
                        "required": false
                    }
                }
            },
            "DeleteSlotType": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_DeleteSlotType.html",
                "description": "Deletes all versions of a slot type.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "slottype version": {
                        "required": true
                    }
                }
            },
            "DeleteSlotTypeVersion": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_DeleteSlotTypeVersion.html",
                "description": "Deletes a specific version of a slot type.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "slottype version": {
                        "required": true
                    }
                }
            },
            "DeleteUtterances": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_DeleteUtterances.html",
                "description": "Deletes the information Amazon Lex maintains for utterances on a specific bot and userId.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bot version": {
                        "required": true
                    }
                }
            },
            "GetBot": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_GetBot.html",
                "description": "Returns information for a specific bot. In addition to the bot name, the bot version or alias is required.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bot alias": {
                        "required": false
                    },
                    "bot version": {
                        "required": false
                    }
                }
            },
            "GetBotAlias": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_GetBotAlias.html",
                "description": "Returns information about a Amazon Lex bot alias.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bot alias": {
                        "required": true
                    }
                }
            },
            "GetBotAliases": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_GetBotAliases.html",
                "description": "Returns a list of aliases for a given Amazon Lex bot.",
                "accessLevel": "List"
            },
            "GetBotChannelAssociation": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_GetBotChannelAssociation.html",
                "description": "Returns information about the association between a Amazon Lex bot and a messaging platform.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "channel": {
                        "required": true
                    }
                }
            },
            "GetBotChannelAssociations": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_GetBotChannelAssociations.html",
                "description": "Returns a list of all of the channels associated with a single bot.",
                "accessLevel": "List",
                "resourceTypes": {
                    "channel": {
                        "required": true
                    }
                }
            },
            "GetBotVersions": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_GetBotVersions.html",
                "description": "Returns information for all versions of a specific bot.",
                "accessLevel": "List",
                "resourceTypes": {
                    "bot version": {
                        "required": true
                    }
                }
            },
            "GetBots": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_GetBots.html",
                "description": "Returns information for the $LATEST version of all bots, subject to filters provided by the client.",
                "accessLevel": "List"
            },
            "GetBuiltinIntent": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_GetBuiltinIntent.html",
                "description": "Returns information about a built-in intent.",
                "accessLevel": "Read"
            },
            "GetBuiltinIntents": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_GetBuiltinIntents.html",
                "description": "Gets a list of built-in intents that meet the specified criteria.",
                "accessLevel": "Read"
            },
            "GetBuiltinSlotTypes": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_GetBuiltinSlotTypes.html",
                "description": "Gets a list of built-in slot types that meet the specified criteria.",
                "accessLevel": "Read"
            },
            "GetExport": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_GetExport.html",
                "description": "Exports Amazon Lex Resource in a requested format.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bot version": {
                        "required": true
                    }
                }
            },
            "GetImport": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_GetImport.html",
                "description": "Gets information about an import job started with StartImport.",
                "accessLevel": "Read"
            },
            "GetIntent": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_GetIntent.html",
                "description": "Returns information for a specific intent. In addition to the intent name, you must also specify the intent version.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "intent version": {
                        "required": true
                    }
                }
            },
            "GetIntentVersions": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_GetIntentVersions.html",
                "description": "Returns information for all versions of a specific intent.",
                "accessLevel": "List",
                "resourceTypes": {
                    "intent version": {
                        "required": true
                    }
                }
            },
            "GetIntents": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_GetIntents.html",
                "description": "Returns information for the $LATEST version of all intents, subject to filters provided by the client.",
                "accessLevel": "List"
            },
            "GetSession": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_runtime_GetSession.html",
                "description": "Returns session information for a specified bot, alias, and user ID.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bot alias": {
                        "required": false
                    },
                    "bot version": {
                        "required": false
                    }
                }
            },
            "GetSlotType": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_GetSlotType.html",
                "description": "Returns information about a specific version of a slot type. In addition to specifying the slot type name, you must also specify the slot type version.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "slottype version": {
                        "required": true
                    }
                }
            },
            "GetSlotTypeVersions": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_GetSlotTypeVersions.html",
                "description": "Returns information for all versions of a specific slot type.",
                "accessLevel": "List",
                "resourceTypes": {
                    "slottype version": {
                        "required": true
                    }
                }
            },
            "GetSlotTypes": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_GetSlotTypes.html",
                "description": "Returns information for the $LATEST version of all slot types, subject to filters provided by the client.",
                "accessLevel": "List"
            },
            "GetUtterancesView": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_GetUtterancesView.html",
                "description": "Returns a view of aggregate utterance data for versions of a bot for a recent time period.",
                "accessLevel": "List",
                "resourceTypes": {
                    "bot version": {
                        "required": true
                    }
                }
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_ListTagsForResource.html",
                "description": "Lists tags for a Lex resource",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bot": {
                        "required": false
                    },
                    "bot alias": {
                        "required": false
                    },
                    "channel": {
                        "required": false
                    }
                }
            },
            "PostContent": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_runtime_PostContent.html",
                "description": "Sends user input (text or speech) to Amazon Lex.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bot alias": {
                        "required": false
                    },
                    "bot version": {
                        "required": false
                    }
                }
            },
            "PostText": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_runtime_PostText.html",
                "description": "Sends user input (text-only) to Amazon Lex.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bot alias": {
                        "required": false
                    },
                    "bot version": {
                        "required": false
                    }
                }
            },
            "PutBot": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_PutBot.html",
                "description": "Creates or updates the $LATEST version of a Amazon Lex conversational bot.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bot version": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "PutBotAlias": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_PutBotAlias.html",
                "description": "Creates or updates an alias for the specific bot.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bot alias": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "PutIntent": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_PutIntent.html",
                "description": "Creates or updates the $LATEST version of an intent.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "intent version": {
                        "required": true
                    }
                }
            },
            "PutSession": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_runtime_PutSession.html",
                "description": "Creates a new session or modifies an existing session with an Amazon Lex bot.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bot alias": {
                        "required": false
                    },
                    "bot version": {
                        "required": false
                    }
                }
            },
            "PutSlotType": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_PutSlotType.html",
                "description": "Creates or updates the $LATEST version of a slot type.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "slottype version": {
                        "required": true
                    }
                }
            },
            "StartImport": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_StartImport.html",
                "description": "Starts a job to import a resource to Amazon Lex.",
                "accessLevel": "Write"
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_TagResource.html",
                "description": "Adds or overwrites tags to a Lex resource",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "bot": {
                        "required": false
                    },
                    "bot alias": {
                        "required": false
                    },
                    "channel": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_UntagResource.html",
                "description": "Removes tags from a Lex resource",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "bot": {
                        "required": false
                    },
                    "bot alias": {
                        "required": false
                    },
                    "channel": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            }
        };
        this.resourceTypes = {
            "bot": {
                "name": "bot",
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_BotMetadata.html",
                "arn": "arn:${Partition}:lex:${Region}:${Account}:bot:${BotName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "bot version": {
                "name": "bot version",
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_BotMetadata.html",
                "arn": "arn:${Partition}:lex:${Region}:${Account}:bot:${BotName}:${BotVersion}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "bot alias": {
                "name": "bot alias",
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_BotAliasMetadata.html",
                "arn": "arn:${Partition}:lex:${Region}:${Account}:bot:${BotName}:${BotAlias}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "channel": {
                "name": "channel",
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_BotChannelAssociation.html",
                "arn": "arn:${Partition}:lex:${Region}:${Account}:bot-channel:${BotName}:${BotAlias}:${ChannelName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "intent version": {
                "name": "intent version",
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_Intent.html",
                "arn": "arn:${Partition}:lex:${Region}:${Account}:intent:${IntentName}:${IntentVersion}",
                "conditionKeys": []
            },
            "slottype version": {
                "name": "slottype version",
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_SlotTypeMetadata.html",
                "arn": "arn:${Partition}:lex:${Region}:${Account}:slottype:${SlotName}:${SlotVersion}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Creates a new version based on the $LATEST version of the specified bot.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_CreateBotVersion.html
     */
    toCreateBotVersion() {
        this.add('lex:CreateBotVersion');
        return this;
    }
    /**
     * Creates a new version based on the $LATEST version of the specified intent.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_CreateIntentVersion.html
     */
    toCreateIntentVersion() {
        this.add('lex:CreateIntentVersion');
        return this;
    }
    /**
     * Creates a new version based on the $LATEST version of the specified slot type.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_CreateSlotTypeVersion.html
     */
    toCreateSlotTypeVersion() {
        this.add('lex:CreateSlotTypeVersion');
        return this;
    }
    /**
     * Deletes all versions of a bot.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_DeleteBot.html
     */
    toDeleteBot() {
        this.add('lex:DeleteBot');
        return this;
    }
    /**
     * Deletes an alias for a specific bot.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_DeleteBotAlias.html
     */
    toDeleteBotAlias() {
        this.add('lex:DeleteBotAlias');
        return this;
    }
    /**
     * Deletes the association between a Amazon Lex bot alias and a messaging platform.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_DeleteBotChannelAssociation.html
     */
    toDeleteBotChannelAssociation() {
        this.add('lex:DeleteBotChannelAssociation');
        return this;
    }
    /**
     * Deletes a specific version of a bot.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_DeleteBotVersion.html
     */
    toDeleteBotVersion() {
        this.add('lex:DeleteBotVersion');
        return this;
    }
    /**
     * Deletes all versions of an intent.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_DeleteIntent.html
     */
    toDeleteIntent() {
        this.add('lex:DeleteIntent');
        return this;
    }
    /**
     * Deletes a specific version of an intent.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_DeleteIntentVersion.html
     */
    toDeleteIntentVersion() {
        this.add('lex:DeleteIntentVersion');
        return this;
    }
    /**
     * Removes session information for a specified bot, alias, and user ID.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_runtime_DeleteSession.html
     */
    toDeleteSession() {
        this.add('lex:DeleteSession');
        return this;
    }
    /**
     * Deletes all versions of a slot type.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_DeleteSlotType.html
     */
    toDeleteSlotType() {
        this.add('lex:DeleteSlotType');
        return this;
    }
    /**
     * Deletes a specific version of a slot type.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_DeleteSlotTypeVersion.html
     */
    toDeleteSlotTypeVersion() {
        this.add('lex:DeleteSlotTypeVersion');
        return this;
    }
    /**
     * Deletes the information Amazon Lex maintains for utterances on a specific bot and userId.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_DeleteUtterances.html
     */
    toDeleteUtterances() {
        this.add('lex:DeleteUtterances');
        return this;
    }
    /**
     * Returns information for a specific bot. In addition to the bot name, the bot version or alias is required.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_GetBot.html
     */
    toGetBot() {
        this.add('lex:GetBot');
        return this;
    }
    /**
     * Returns information about a Amazon Lex bot alias.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_GetBotAlias.html
     */
    toGetBotAlias() {
        this.add('lex:GetBotAlias');
        return this;
    }
    /**
     * Returns a list of aliases for a given Amazon Lex bot.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_GetBotAliases.html
     */
    toGetBotAliases() {
        this.add('lex:GetBotAliases');
        return this;
    }
    /**
     * Returns information about the association between a Amazon Lex bot and a messaging platform.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_GetBotChannelAssociation.html
     */
    toGetBotChannelAssociation() {
        this.add('lex:GetBotChannelAssociation');
        return this;
    }
    /**
     * Returns a list of all of the channels associated with a single bot.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_GetBotChannelAssociations.html
     */
    toGetBotChannelAssociations() {
        this.add('lex:GetBotChannelAssociations');
        return this;
    }
    /**
     * Returns information for all versions of a specific bot.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_GetBotVersions.html
     */
    toGetBotVersions() {
        this.add('lex:GetBotVersions');
        return this;
    }
    /**
     * Returns information for the $LATEST version of all bots, subject to filters provided by the client.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_GetBots.html
     */
    toGetBots() {
        this.add('lex:GetBots');
        return this;
    }
    /**
     * Returns information about a built-in intent.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_GetBuiltinIntent.html
     */
    toGetBuiltinIntent() {
        this.add('lex:GetBuiltinIntent');
        return this;
    }
    /**
     * Gets a list of built-in intents that meet the specified criteria.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_GetBuiltinIntents.html
     */
    toGetBuiltinIntents() {
        this.add('lex:GetBuiltinIntents');
        return this;
    }
    /**
     * Gets a list of built-in slot types that meet the specified criteria.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_GetBuiltinSlotTypes.html
     */
    toGetBuiltinSlotTypes() {
        this.add('lex:GetBuiltinSlotTypes');
        return this;
    }
    /**
     * Exports Amazon Lex Resource in a requested format.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_GetExport.html
     */
    toGetExport() {
        this.add('lex:GetExport');
        return this;
    }
    /**
     * Gets information about an import job started with StartImport.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_GetImport.html
     */
    toGetImport() {
        this.add('lex:GetImport');
        return this;
    }
    /**
     * Returns information for a specific intent. In addition to the intent name, you must also specify the intent version.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_GetIntent.html
     */
    toGetIntent() {
        this.add('lex:GetIntent');
        return this;
    }
    /**
     * Returns information for all versions of a specific intent.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_GetIntentVersions.html
     */
    toGetIntentVersions() {
        this.add('lex:GetIntentVersions');
        return this;
    }
    /**
     * Returns information for the $LATEST version of all intents, subject to filters provided by the client.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_GetIntents.html
     */
    toGetIntents() {
        this.add('lex:GetIntents');
        return this;
    }
    /**
     * Returns session information for a specified bot, alias, and user ID.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_runtime_GetSession.html
     */
    toGetSession() {
        this.add('lex:GetSession');
        return this;
    }
    /**
     * Returns information about a specific version of a slot type. In addition to specifying the slot type name, you must also specify the slot type version.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_GetSlotType.html
     */
    toGetSlotType() {
        this.add('lex:GetSlotType');
        return this;
    }
    /**
     * Returns information for all versions of a specific slot type.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_GetSlotTypeVersions.html
     */
    toGetSlotTypeVersions() {
        this.add('lex:GetSlotTypeVersions');
        return this;
    }
    /**
     * Returns information for the $LATEST version of all slot types, subject to filters provided by the client.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_GetSlotTypes.html
     */
    toGetSlotTypes() {
        this.add('lex:GetSlotTypes');
        return this;
    }
    /**
     * Returns a view of aggregate utterance data for versions of a bot for a recent time period.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_GetUtterancesView.html
     */
    toGetUtterancesView() {
        this.add('lex:GetUtterancesView');
        return this;
    }
    /**
     * Lists tags for a Lex resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.add('lex:ListTagsForResource');
        return this;
    }
    /**
     * Sends user input (text or speech) to Amazon Lex.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_runtime_PostContent.html
     */
    toPostContent() {
        this.add('lex:PostContent');
        return this;
    }
    /**
     * Sends user input (text-only) to Amazon Lex.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_runtime_PostText.html
     */
    toPostText() {
        this.add('lex:PostText');
        return this;
    }
    /**
     * Creates or updates the $LATEST version of a Amazon Lex conversational bot.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_PutBot.html
     */
    toPutBot() {
        this.add('lex:PutBot');
        return this;
    }
    /**
     * Creates or updates an alias for the specific bot.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_PutBotAlias.html
     */
    toPutBotAlias() {
        this.add('lex:PutBotAlias');
        return this;
    }
    /**
     * Creates or updates the $LATEST version of an intent.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_PutIntent.html
     */
    toPutIntent() {
        this.add('lex:PutIntent');
        return this;
    }
    /**
     * Creates a new session or modifies an existing session with an Amazon Lex bot.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_runtime_PutSession.html
     */
    toPutSession() {
        this.add('lex:PutSession');
        return this;
    }
    /**
     * Creates or updates the $LATEST version of a slot type.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_PutSlotType.html
     */
    toPutSlotType() {
        this.add('lex:PutSlotType');
        return this;
    }
    /**
     * Starts a job to import a resource to Amazon Lex.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_StartImport.html
     */
    toStartImport() {
        this.add('lex:StartImport');
        return this;
    }
    /**
     * Adds or overwrites tags to a Lex resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_TagResource.html
     */
    toTagResource() {
        this.add('lex:TagResource');
        return this;
    }
    /**
     * Removes tags from a Lex resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_UntagResource.html
     */
    toUntagResource() {
        this.add('lex:UntagResource');
        return this;
    }
    /**
     * Adds a resource of type bot to the statement
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_BotMetadata.html
     *
     * @param botName - Identifier for the botName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onBot(botName, account, region, partition) {
        var arn = 'arn:${Partition}:lex:${Region}:${Account}:bot:${BotName}';
        arn = arn.replace('${BotName}', botName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type bot version to the statement
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_BotMetadata.html
     *
     * @param botName - Identifier for the botName.
     * @param botVersion - Identifier for the botVersion.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onBotVersion(botName, botVersion, account, region, partition) {
        var arn = 'arn:${Partition}:lex:${Region}:${Account}:bot:${BotName}:${BotVersion}';
        arn = arn.replace('${BotName}', botName);
        arn = arn.replace('${BotVersion}', botVersion);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type bot alias to the statement
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_BotAliasMetadata.html
     *
     * @param botName - Identifier for the botName.
     * @param botAlias - Identifier for the botAlias.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onBotAlias(botName, botAlias, account, region, partition) {
        var arn = 'arn:${Partition}:lex:${Region}:${Account}:bot:${BotName}:${BotAlias}';
        arn = arn.replace('${BotName}', botName);
        arn = arn.replace('${BotAlias}', botAlias);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type channel to the statement
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_BotChannelAssociation.html
     *
     * @param botName - Identifier for the botName.
     * @param botAlias - Identifier for the botAlias.
     * @param channelName - Identifier for the channelName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onChannel(botName, botAlias, channelName, account, region, partition) {
        var arn = 'arn:${Partition}:lex:${Region}:${Account}:bot-channel:${BotName}:${BotAlias}:${ChannelName}';
        arn = arn.replace('${BotName}', botName);
        arn = arn.replace('${BotAlias}', botAlias);
        arn = arn.replace('${ChannelName}', channelName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type intent version to the statement
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_Intent.html
     *
     * @param intentName - Identifier for the intentName.
     * @param intentVersion - Identifier for the intentVersion.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onIntentVersion(intentName, intentVersion, account, region, partition) {
        var arn = 'arn:${Partition}:lex:${Region}:${Account}:intent:${IntentName}:${IntentVersion}';
        arn = arn.replace('${IntentName}', intentName);
        arn = arn.replace('${IntentVersion}', intentVersion);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type slottype version to the statement
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_SlotTypeMetadata.html
     *
     * @param slotName - Identifier for the slotName.
     * @param slotVersion - Identifier for the slotVersion.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onSlottypeVersion(slotName, slotVersion, account, region, partition) {
        var arn = 'arn:${Partition}:lex:${Region}:${Account}:slottype:${SlotName}:${SlotVersion}';
        arn = arn.replace('${SlotName}', slotName);
        arn = arn.replace('${SlotVersion}', slotVersion);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Enables you to control access based on the intents included in the request.
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAssociatedIntents(value, operator) {
        return this.if(`lex:associatedIntents`, value, operator || 'StringLike');
    }
    /**
     * Enables you to control access based on the slot types included in the request.
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAssociatedSlotTypes(value, operator) {
        return this.if(`lex:associatedSlotTypes`, value, operator || 'StringLike');
    }
    /**
     * Enables you to control access based on the channel type included in the request.
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifChannelType(value, operator) {
        return this.if(`lex:channelType`, value, operator || 'StringLike');
    }
}
exports.Lex = Lex;
//# sourceMappingURL=data:application/json;base64,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