"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Lightsail = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [lightsail](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonlightsail.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Lightsail extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [lightsail](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonlightsail.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'lightsail';
        this.actionList = {
            "AllocateStaticIp": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_AllocateStaticIp.html",
                "description": "Creates a static IP address that can be attached to an instance.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "StaticIp": {
                        "required": true
                    }
                }
            },
            "AttachDisk": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_AttachDisk.html",
                "description": "Attaches a disk to an instance.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Disk": {
                        "required": true
                    },
                    "Instance": {
                        "required": true
                    }
                }
            },
            "AttachInstancesToLoadBalancer": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_AttachInstancesToLoadBalancer.html",
                "description": "Attaches one or more instances to a load balancer.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    },
                    "LoadBalancer": {
                        "required": true
                    }
                }
            },
            "AttachLoadBalancerTlsCertificate": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_AttachLoadBalancerTlsCertificate.html",
                "description": "Attaches a TLS certificate to a load balancer.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "LoadBalancer": {
                        "required": true
                    }
                }
            },
            "AttachStaticIp": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_AttachStaticIp.html",
                "description": "Attaches a static IP address to an instance.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    },
                    "StaticIp": {
                        "required": true
                    }
                }
            },
            "CloseInstancePublicPorts": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CloseInstancePublicPorts.html",
                "description": "Closes a public port of an instance.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    }
                }
            },
            "CopySnapshot": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CopySnapshot.html",
                "description": "Copies a snapshot from one AWS Region to another in Amazon Lightsail.",
                "accessLevel": "Write"
            },
            "CreateCloudFormationStack": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateCloudFormationStack.html",
                "description": "Creates a new Amazon EC2 instance from an exported Amazon Lightsail snapshot.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "ExportSnapshotRecord": {
                        "required": true
                    }
                }
            },
            "CreateDisk": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateDisk.html",
                "description": "Creates a disk.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Disk": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateDiskFromSnapshot": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateDiskFromSnapshot.html",
                "description": "Creates a disk from snapshot.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Disk": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateDiskSnapshot": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateDiskSnapshot.html",
                "description": "Creates a disk snapshot.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Disk": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateDomain": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateDomain.html",
                "description": "Creates a domain resource for the specified domain name.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Domain": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateDomainEntry": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateDomainEntry.html",
                "description": "Creates one or more DNS record entries for a domain resource: Address (A), canonical name (CNAME), mail exchanger (MX), name server (NS), start of authority (SOA), service locator (SRV), or text (TXT).",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Domain": {
                        "required": true
                    }
                }
            },
            "CreateInstanceSnapshot": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateInstanceSnapshot.html",
                "description": "Creates an instance snapshot.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    },
                    "InstanceSnapshot": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateInstances": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateInstances.html",
                "description": "Creates one or more instances.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "KeyPair": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateInstancesFromSnapshot": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateInstancesFromSnapshot.html",
                "description": "Creates one or more instances based on an instance snapshot.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    },
                    "InstanceSnapshot": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateKeyPair": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateKeyPair.html",
                "description": "Creates a key pair used to authenticate and connect to an instance.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "KeyPair": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateLoadBalancer": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateLoadBalancer.html",
                "description": "Creates a load balancer.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "LoadBalancer": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateLoadBalancerTlsCertificate": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateLoadBalancerTlsCertificate.html",
                "description": "Creates a load balancer TLS certificate.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "LoadBalancer": {
                        "required": true
                    }
                }
            },
            "CreateRelationalDatabase": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateRelationalDatabase.html",
                "description": "Creates a new relational database.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "RelationalDatabase": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateRelationalDatabaseFromSnapshot": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateRelationalDatabaseFromSnapshot.html",
                "description": "Creates a new relational database from a snapshot.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "RelationalDatabase": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateRelationalDatabaseSnapshot": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateRelationalDatabaseSnapshot.html",
                "description": "Creates a relational database snapshot.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "RelationalDatabaseSnapshot": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "DeleteDisk": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteDisk.html",
                "description": "Deletes a disk.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Disk": {
                        "required": true
                    }
                }
            },
            "DeleteDiskSnapshot": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteDiskSnapshot.html",
                "description": "Deletes a disk snapshot.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Disk": {
                        "required": true
                    }
                }
            },
            "DeleteDomain": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteDomain.html",
                "description": "Deletes a domain resource and all of its DNS records.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Domain": {
                        "required": true
                    }
                }
            },
            "DeleteDomainEntry": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteDomainEntry.html",
                "description": "Deletes a DNS record entry for a domain resource.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Domain": {
                        "required": true
                    }
                }
            },
            "DeleteInstance": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteInstance.html",
                "description": "Deletes an instance.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    }
                }
            },
            "DeleteInstanceSnapshot": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteInstanceSnapshot.html",
                "description": "Deletes an instance snapshot.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "InstanceSnapshot": {
                        "required": true
                    }
                }
            },
            "DeleteKeyPair": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteKeyPair.html",
                "description": "Deletes a key pair used to authenticate and connect to an instance.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "KeyPair": {
                        "required": true
                    }
                }
            },
            "DeleteKnownHostKeys": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteKnownHostKeys.html",
                "description": "Deletes the known host key or certificate used by the Amazon Lightsail browser-based SSH or RDP clients to authenticate an instance.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    }
                }
            },
            "DeleteLoadBalancer": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteLoadBalancer.html",
                "description": "Deletes a load balancer.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "LoadBalancer": {
                        "required": true
                    }
                }
            },
            "DeleteLoadBalancerTlsCertificate": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteLoadBalancerTlsCertificate.html",
                "description": "Deletes a load balancer TLS certificate.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "LoadBalancer": {
                        "required": true
                    }
                }
            },
            "DeleteRelationalDatabase": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteRelationalDatabase.html",
                "description": "Deletes a relational database.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "RelationalDatabase": {
                        "required": true
                    }
                }
            },
            "DeleteRelationalDatabaseSnapshot": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteRelationalDatabaseSnapshot.html",
                "description": "Deletes relational database snapshot.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "RelationalDatabaseSnapshot": {
                        "required": true
                    }
                }
            },
            "DetachDisk": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DetachDisk.html",
                "description": "Detaches a disk from an instance.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Disk": {
                        "required": true
                    }
                }
            },
            "DetachInstancesFromLoadBalancer": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DetachInstancesFromLoadBalancer.html",
                "description": "Detaches one or more instances from a load balancer.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    },
                    "LoadBalancer": {
                        "required": true
                    }
                }
            },
            "DetachStaticIp": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DetachStaticIp.html",
                "description": "Detaches a static IP from an instance to which it is attached.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    },
                    "StaticIp": {
                        "required": true
                    }
                }
            },
            "DownloadDefaultKeyPair": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DownloadDefaultKeyPair.html",
                "description": "Downloads the default key pair used to authenticate and connect to instances in a specific AWS Region.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "KeyPair": {
                        "required": true
                    }
                }
            },
            "ExportSnapshot": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_ExportSnapshot.html",
                "description": "Exports an Amazon Lightsail snapshot to Amazon EC2.",
                "accessLevel": "Write"
            },
            "GetActiveNames": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetActiveNames.html",
                "description": "Returns the names of all active (not deleted) resources.",
                "accessLevel": "Read"
            },
            "GetBlueprints": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetBlueprints.html",
                "description": "Returns a list of instance images, or blueprints. You can use a blueprint to create a new instance already running a specific operating system, as well as a pre-installed application or development stack. The software that runs on your instance depends on the blueprint you define when creating the instance.",
                "accessLevel": "List"
            },
            "GetBundles": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetBundles.html",
                "description": "Returns a list of instance bundles. You can use a bundle to create a new instance with a set of performance specifications, such as CPU count, disk size, RAM size, and network transfer allowance. The cost of your instance depends on the bundle you define when creating the instance.",
                "accessLevel": "List"
            },
            "GetCloudFormationStackRecords": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetCloudFormationStackRecords.html",
                "description": "Returns information about all CloudFormation stacks used to create Amazon EC2 resources from exported Amazon Lightsail snapshots.",
                "accessLevel": "List",
                "resourceTypes": {
                    "CloudFormationStackRecord": {
                        "required": true
                    }
                }
            },
            "GetDisk": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetDisk.html",
                "description": "Returns information about a disk.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Disk": {
                        "required": true
                    }
                }
            },
            "GetDiskSnapshot": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetDiskSnapshot.html",
                "description": "Returns information about a disk snapshot.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Disk": {
                        "required": true
                    }
                }
            },
            "GetDiskSnapshots": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetDiskSnapshots.html",
                "description": "Returns information about all disk snapshots.",
                "accessLevel": "List",
                "resourceTypes": {
                    "Disk": {
                        "required": true
                    }
                }
            },
            "GetDisks": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetDisks.html",
                "description": "Returns information about all disks.",
                "accessLevel": "List"
            },
            "GetDomain": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetDomain.html",
                "description": "Returns DNS records for a domain resource.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Domain": {
                        "required": true
                    }
                }
            },
            "GetDomains": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetDomains.html",
                "description": "Returns DNS records for all domain resources.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Domain": {
                        "required": true
                    }
                }
            },
            "GetExportSnapshotRecords": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetExportSnapshotRecords.html",
                "description": "Returns information about all records to export Amazon Lightsail snapshots to Amazon EC2.",
                "accessLevel": "List",
                "resourceTypes": {
                    "ExportSnapshotRecord": {
                        "required": true
                    }
                }
            },
            "GetInstance": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetInstance.html",
                "description": "Returns information about an instance.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    }
                }
            },
            "GetInstanceAccessDetails": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetInstanceAccessDetails.html",
                "description": "Returns temporary keys you can use to authenticate and connect to an instance.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    }
                }
            },
            "GetInstanceMetricData": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetInstanceMetricData.html",
                "description": "Returns the data points for the specified metric of an instance.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    }
                }
            },
            "GetInstancePortStates": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetInstancePortStates.html",
                "description": "Returns the port states of an instance.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    }
                }
            },
            "GetInstanceSnapshot": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetInstanceSnapshot.html",
                "description": "Returns information about an instance snapshot.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "InstanceSnapshot": {
                        "required": true
                    }
                }
            },
            "GetInstanceSnapshots": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetInstanceSnapshots.html",
                "description": "Returns information about all instance snapshots.",
                "accessLevel": "List",
                "resourceTypes": {
                    "InstanceSnapshot": {
                        "required": true
                    }
                }
            },
            "GetInstanceState": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetInstanceState.html",
                "description": "Returns the state of an instance.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    }
                }
            },
            "GetInstances": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetInstances.html",
                "description": "Returns information about all instances.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    }
                }
            },
            "GetKeyPair": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetKeyPair.html",
                "description": "Returns information about a key pair.",
                "accessLevel": "List",
                "resourceTypes": {
                    "KeyPair": {
                        "required": true
                    }
                }
            },
            "GetKeyPairs": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetKeyPairs.html",
                "description": "Returns information about all key pairs.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "KeyPair": {
                        "required": true
                    }
                }
            },
            "GetLoadBalancer": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetLoadBalancer.html",
                "description": "Returns information about a load balancer.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "LoadBalancer": {
                        "required": true
                    }
                }
            },
            "GetLoadBalancerMetricData": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetLoadBalancerMetricData.html",
                "description": "Returns the data points for the specified metric of a load balancer.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "LoadBalancer": {
                        "required": true
                    }
                }
            },
            "GetLoadBalancerTlsCertificates": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteLoadBalancerTlsCertificate.html",
                "description": "Returns information about a load balancer TLS certificate.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "LoadBalancer": {
                        "required": true
                    }
                }
            },
            "GetLoadBalancers": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetLoadBalancers.html",
                "description": "Returns information about load balancers.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "LoadBalancer": {
                        "required": true
                    }
                }
            },
            "GetOperation": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetOperation.html",
                "description": "Returns information about an operation. Operations include events such as when you create an instance, allocate a static IP, attach a static IP, and so on.",
                "accessLevel": "Read"
            },
            "GetOperations": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetOperations.html",
                "description": "Returns information about all operations. Operations include events such as when you create an instance, allocate a static IP, attach a static IP, and so on.",
                "accessLevel": "Read"
            },
            "GetOperationsForResource": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetOperationsForResource.html",
                "description": "Returns operations for a resource.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Domain": {
                        "required": false
                    },
                    "Instance": {
                        "required": false
                    },
                    "InstanceSnapshot": {
                        "required": false
                    },
                    "KeyPair": {
                        "required": false
                    },
                    "StaticIp": {
                        "required": false
                    }
                }
            },
            "GetRegions": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRegions.html",
                "description": "Returns a list of all valid AWS Regions for Amazon Lightsail.",
                "accessLevel": "List"
            },
            "GetRelationalDatabase": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabase.html",
                "description": "Returns information about a relational database.",
                "accessLevel": "List",
                "resourceTypes": {
                    "RelationalDatabase": {
                        "required": true
                    }
                }
            },
            "GetRelationalDatabaseBlueprints": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseBlueprints.html",
                "description": "Returns a list of relational database images, or blueprints. You can use a blueprint to create a new database running a specific database engine. The database engine that runs on your database depends on the blueprint you define when creating the relational database.",
                "accessLevel": "List"
            },
            "GetRelationalDatabaseBundles": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseBundles.html",
                "description": "Returns a list of relational database bundles. You can use a bundle to create a new database with a set of performance specifications, such as CPU count, disk size, RAM size, network transfer allowance, and standard of high availability. The cost of your database depends on the bundle you define when creating the relational database.",
                "accessLevel": "List"
            },
            "GetRelationalDatabaseEvents": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseEvents.html",
                "description": "Returns events for a relational database.",
                "accessLevel": "Read"
            },
            "GetRelationalDatabaseLogEvents": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseLogEvents.html",
                "description": "Returns events for the specified log stream of a relational database.",
                "accessLevel": "Read"
            },
            "GetRelationalDatabaseLogStreams": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseLogStreams.html",
                "description": "Returns the log streams available for a relational database.",
                "accessLevel": "Read"
            },
            "GetRelationalDatabaseMasterUserPassword": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseMasterUserPassword.html",
                "description": "Returns the master user password of a relational database.",
                "accessLevel": "Write"
            },
            "GetRelationalDatabaseMetricData": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseMetricData.html",
                "description": "Returns the data points for the specified metric of a relational database.",
                "accessLevel": "Read"
            },
            "GetRelationalDatabaseParameters": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseParameters.html",
                "description": "Returns the parameters of a relational database.",
                "accessLevel": "List"
            },
            "GetRelationalDatabaseSnapshot": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseSnapshot.html",
                "description": "Returns information about a relational database snapshot.",
                "accessLevel": "List",
                "resourceTypes": {
                    "RelationalDatabase": {
                        "required": true
                    }
                }
            },
            "GetRelationalDatabaseSnapshots": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseSnapshots.html",
                "description": "Returns information about all relational database snapshots.",
                "accessLevel": "List",
                "resourceTypes": {
                    "RelationalDatabase": {
                        "required": true
                    }
                }
            },
            "GetRelationalDatabases": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabases.html",
                "description": "Return information about all relational databases.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "RelationalDatabase": {
                        "required": true
                    }
                }
            },
            "GetStaticIp": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetStaticIp.html",
                "description": "Returns information about a static IP.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "StaticIp": {
                        "required": true
                    }
                }
            },
            "GetStaticIps": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetStaticIps.html",
                "description": "Returns information about all static IPs.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "StaticIp": {
                        "required": true
                    }
                }
            },
            "ImportKeyPair": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_ImportKeyPair.html",
                "description": "Imports a public key from a key pair.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "KeyPair": {
                        "required": true
                    }
                }
            },
            "IsVpcPeered": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_IsVpcPeered.html",
                "description": "Returns a boolean value indicating whether the Amazon Lightsail virtual private cloud (VPC) is peered.",
                "accessLevel": "Read"
            },
            "OpenInstancePublicPorts": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_OpenInstancePublicPorts.html",
                "description": "Adds, or opens a public port of an instance.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    }
                }
            },
            "PeerVpc": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_PeerVpc.html",
                "description": "Tries to peer the Amazon Lightsail virtual private cloud (VPC) with the default VPC.",
                "accessLevel": "Write"
            },
            "PutInstancePublicPorts": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_PutInstancePublicPorts.html",
                "description": "Sets the specified open ports for an instance, and closes all ports for every protocol not included in the request.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    }
                }
            },
            "RebootInstance": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_RebootInstance.html",
                "description": "Reboots an instance that is in a running state.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    }
                }
            },
            "RebootRelationalDatabase": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_RebootRelationalDatabase.html",
                "description": "Reboots a relational database that is in a running state.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "RelationalDatabase": {
                        "required": true
                    }
                }
            },
            "ReleaseStaticIp": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_ReleaseStaticIp.html",
                "description": "Deletes a static IP.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "StaticIp": {
                        "required": true
                    }
                }
            },
            "StartInstance": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_StartInstance.html",
                "description": "Starts an instance that is in a stopped state.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    }
                }
            },
            "StartRelationalDatabase": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_StartRelationalDatabase.html",
                "description": "Starts a relational database that is in a stopped state.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "RelationalDatabase": {
                        "required": true
                    }
                }
            },
            "StopInstance": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_StopInstance.html",
                "description": "Stops an instance that is in a running state.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Instance": {
                        "required": true
                    }
                }
            },
            "StopRelationalDatabase": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_StopRelationalDatabase.html",
                "description": "Stops a relational database that is in a running state.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "RelationalDatabase": {
                        "required": true
                    }
                }
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_TagResource.html",
                "description": "Tags a resource.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Disk": {
                        "required": false
                    },
                    "DiskSnapshot": {
                        "required": false
                    },
                    "Domain": {
                        "required": false
                    },
                    "Instance": {
                        "required": false
                    },
                    "InstanceSnapshot": {
                        "required": false
                    },
                    "KeyPair": {
                        "required": false
                    },
                    "LoadBalancer": {
                        "required": false
                    },
                    "RelationalDatabase": {
                        "required": false
                    },
                    "RelationalDatabaseSnapshot": {
                        "required": false
                    },
                    "StaticIp": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UnpeerVpc": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_UnpeerVpc.html",
                "description": "Attempts to unpeer the Amazon Lightsail virtual private cloud (VPC) from the default VPC.",
                "accessLevel": "Write"
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_UntagResource.html",
                "description": "Untags a resource.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Disk": {
                        "required": false
                    },
                    "DiskSnapshot": {
                        "required": false
                    },
                    "Domain": {
                        "required": false
                    },
                    "Instance": {
                        "required": false
                    },
                    "InstanceSnapshot": {
                        "required": false
                    },
                    "KeyPair": {
                        "required": false
                    },
                    "LoadBalancer": {
                        "required": false
                    },
                    "RelationalDatabase": {
                        "required": false
                    },
                    "RelationalDatabaseSnapshot": {
                        "required": false
                    },
                    "StaticIp": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UpdateDomainEntry": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_UpdateDomainEntry.html",
                "description": "Updates a domain recordset after it is created.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Domain": {
                        "required": true
                    }
                }
            },
            "UpdateLoadBalancerAttribute": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DetachInstancesFromLoadBalancer.html",
                "description": "Updates a load balancer attribute, such as the health check path and session stickiness.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "LoadBalancer": {
                        "required": true
                    }
                }
            },
            "UpdateRelationalDatabase": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_UpdateRelationalDatabase.html",
                "description": "Updates a relational database.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "RelationalDatabase": {
                        "required": true
                    }
                }
            },
            "UpdateRelationalDatabaseParameters": {
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_UpdateRelationalDatabaseParameters.html",
                "description": "Updates the parameters of a relational database.",
                "accessLevel": "Write"
            }
        };
        this.resourceTypes = {
            "Domain": {
                "name": "Domain",
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_Domain.html",
                "arn": "arn:${Partition}:lightsail:${Region}:${Account}:Domain/${Id}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "Instance": {
                "name": "Instance",
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_Instance.html",
                "arn": "arn:${Partition}:lightsail:${Region}:${Account}:Instance/${Id}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "InstanceSnapshot": {
                "name": "InstanceSnapshot",
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_InstanceSnapshot.html",
                "arn": "arn:${Partition}:lightsail:${Region}:${Account}:InstanceSnapshot/${Id}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "KeyPair": {
                "name": "KeyPair",
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_KeyPair.html",
                "arn": "arn:${Partition}:lightsail:${Region}:${Account}:KeyPair/${Id}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "StaticIp": {
                "name": "StaticIp",
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_StaticIp.html",
                "arn": "arn:${Partition}:lightsail:${Region}:${Account}:StaticIp/${Id}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "Disk": {
                "name": "Disk",
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_Disk.html",
                "arn": "arn:${Partition}:lightsail:${Region}:${Account}:Disk/${Id}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "DiskSnapshot": {
                "name": "DiskSnapshot",
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DiskSnapshot.html",
                "arn": "arn:${Partition}:lightsail:${Region}:${Account}:DiskSnapshot/${Id}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "LoadBalancer": {
                "name": "LoadBalancer",
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_LoadBalancer.html",
                "arn": "arn:${Partition}:lightsail:${Region}:${Account}:LoadBalancer/${Id}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "PeeredVpc": {
                "name": "PeeredVpc",
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/PeeredVpc.html",
                "arn": "arn:${Partition}:lightsail:${Region}:${Account}:PeeredVpc/${Id}",
                "conditionKeys": []
            },
            "LoadBalancerTlsCertificate": {
                "name": "LoadBalancerTlsCertificate",
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/LoadBalancerTlsCertificate.html",
                "arn": "arn:${Partition}:lightsail:${Region}:${Account}:LoadBalancerTlsCertificate/${Id}",
                "conditionKeys": []
            },
            "ExportSnapshotRecord": {
                "name": "ExportSnapshotRecord",
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_ExportSnapshotRecord.html",
                "arn": "arn:${Partition}:lightsail:${Region}:${Account}:ExportSnapshotRecord/${Id}",
                "conditionKeys": []
            },
            "CloudFormationStackRecord": {
                "name": "CloudFormationStackRecord",
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CloudFormationStackRecord.html",
                "arn": "arn:${Partition}:lightsail:${Region}:${Account}:CloudFormationStackRecord/${Id}",
                "conditionKeys": []
            },
            "RelationalDatabase": {
                "name": "RelationalDatabase",
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_RelationalDatabase.html",
                "arn": "arn:${Partition}:lightsail:${Region}:${Account}:RelationalDatabase/${Id}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "RelationalDatabaseSnapshot": {
                "name": "RelationalDatabaseSnapshot",
                "url": "https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_RelationalDatabaseSnapshot.html",
                "arn": "arn:${Partition}:lightsail:${Region}:${Account}:RelationalDatabaseSnapshot/${Id}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Creates a static IP address that can be attached to an instance.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_AllocateStaticIp.html
     */
    toAllocateStaticIp() {
        this.add('lightsail:AllocateStaticIp');
        return this;
    }
    /**
     * Attaches a disk to an instance.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_AttachDisk.html
     */
    toAttachDisk() {
        this.add('lightsail:AttachDisk');
        return this;
    }
    /**
     * Attaches one or more instances to a load balancer.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_AttachInstancesToLoadBalancer.html
     */
    toAttachInstancesToLoadBalancer() {
        this.add('lightsail:AttachInstancesToLoadBalancer');
        return this;
    }
    /**
     * Attaches a TLS certificate to a load balancer.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_AttachLoadBalancerTlsCertificate.html
     */
    toAttachLoadBalancerTlsCertificate() {
        this.add('lightsail:AttachLoadBalancerTlsCertificate');
        return this;
    }
    /**
     * Attaches a static IP address to an instance.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_AttachStaticIp.html
     */
    toAttachStaticIp() {
        this.add('lightsail:AttachStaticIp');
        return this;
    }
    /**
     * Closes a public port of an instance.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CloseInstancePublicPorts.html
     */
    toCloseInstancePublicPorts() {
        this.add('lightsail:CloseInstancePublicPorts');
        return this;
    }
    /**
     * Copies a snapshot from one AWS Region to another in Amazon Lightsail.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CopySnapshot.html
     */
    toCopySnapshot() {
        this.add('lightsail:CopySnapshot');
        return this;
    }
    /**
     * Creates a new Amazon EC2 instance from an exported Amazon Lightsail snapshot.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateCloudFormationStack.html
     */
    toCreateCloudFormationStack() {
        this.add('lightsail:CreateCloudFormationStack');
        return this;
    }
    /**
     * Creates a disk.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateDisk.html
     */
    toCreateDisk() {
        this.add('lightsail:CreateDisk');
        return this;
    }
    /**
     * Creates a disk from snapshot.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateDiskFromSnapshot.html
     */
    toCreateDiskFromSnapshot() {
        this.add('lightsail:CreateDiskFromSnapshot');
        return this;
    }
    /**
     * Creates a disk snapshot.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateDiskSnapshot.html
     */
    toCreateDiskSnapshot() {
        this.add('lightsail:CreateDiskSnapshot');
        return this;
    }
    /**
     * Creates a domain resource for the specified domain name.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateDomain.html
     */
    toCreateDomain() {
        this.add('lightsail:CreateDomain');
        return this;
    }
    /**
     * Creates one or more DNS record entries for a domain resource: Address (A), canonical name (CNAME), mail exchanger (MX), name server (NS), start of authority (SOA), service locator (SRV), or text (TXT).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateDomainEntry.html
     */
    toCreateDomainEntry() {
        this.add('lightsail:CreateDomainEntry');
        return this;
    }
    /**
     * Creates an instance snapshot.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateInstanceSnapshot.html
     */
    toCreateInstanceSnapshot() {
        this.add('lightsail:CreateInstanceSnapshot');
        return this;
    }
    /**
     * Creates one or more instances.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateInstances.html
     */
    toCreateInstances() {
        this.add('lightsail:CreateInstances');
        return this;
    }
    /**
     * Creates one or more instances based on an instance snapshot.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateInstancesFromSnapshot.html
     */
    toCreateInstancesFromSnapshot() {
        this.add('lightsail:CreateInstancesFromSnapshot');
        return this;
    }
    /**
     * Creates a key pair used to authenticate and connect to an instance.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateKeyPair.html
     */
    toCreateKeyPair() {
        this.add('lightsail:CreateKeyPair');
        return this;
    }
    /**
     * Creates a load balancer.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateLoadBalancer.html
     */
    toCreateLoadBalancer() {
        this.add('lightsail:CreateLoadBalancer');
        return this;
    }
    /**
     * Creates a load balancer TLS certificate.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateLoadBalancerTlsCertificate.html
     */
    toCreateLoadBalancerTlsCertificate() {
        this.add('lightsail:CreateLoadBalancerTlsCertificate');
        return this;
    }
    /**
     * Creates a new relational database.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateRelationalDatabase.html
     */
    toCreateRelationalDatabase() {
        this.add('lightsail:CreateRelationalDatabase');
        return this;
    }
    /**
     * Creates a new relational database from a snapshot.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateRelationalDatabaseFromSnapshot.html
     */
    toCreateRelationalDatabaseFromSnapshot() {
        this.add('lightsail:CreateRelationalDatabaseFromSnapshot');
        return this;
    }
    /**
     * Creates a relational database snapshot.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateRelationalDatabaseSnapshot.html
     */
    toCreateRelationalDatabaseSnapshot() {
        this.add('lightsail:CreateRelationalDatabaseSnapshot');
        return this;
    }
    /**
     * Deletes a disk.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteDisk.html
     */
    toDeleteDisk() {
        this.add('lightsail:DeleteDisk');
        return this;
    }
    /**
     * Deletes a disk snapshot.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteDiskSnapshot.html
     */
    toDeleteDiskSnapshot() {
        this.add('lightsail:DeleteDiskSnapshot');
        return this;
    }
    /**
     * Deletes a domain resource and all of its DNS records.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteDomain.html
     */
    toDeleteDomain() {
        this.add('lightsail:DeleteDomain');
        return this;
    }
    /**
     * Deletes a DNS record entry for a domain resource.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteDomainEntry.html
     */
    toDeleteDomainEntry() {
        this.add('lightsail:DeleteDomainEntry');
        return this;
    }
    /**
     * Deletes an instance.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteInstance.html
     */
    toDeleteInstance() {
        this.add('lightsail:DeleteInstance');
        return this;
    }
    /**
     * Deletes an instance snapshot.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteInstanceSnapshot.html
     */
    toDeleteInstanceSnapshot() {
        this.add('lightsail:DeleteInstanceSnapshot');
        return this;
    }
    /**
     * Deletes a key pair used to authenticate and connect to an instance.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteKeyPair.html
     */
    toDeleteKeyPair() {
        this.add('lightsail:DeleteKeyPair');
        return this;
    }
    /**
     * Deletes the known host key or certificate used by the Amazon Lightsail browser-based SSH or RDP clients to authenticate an instance.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteKnownHostKeys.html
     */
    toDeleteKnownHostKeys() {
        this.add('lightsail:DeleteKnownHostKeys');
        return this;
    }
    /**
     * Deletes a load balancer.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteLoadBalancer.html
     */
    toDeleteLoadBalancer() {
        this.add('lightsail:DeleteLoadBalancer');
        return this;
    }
    /**
     * Deletes a load balancer TLS certificate.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteLoadBalancerTlsCertificate.html
     */
    toDeleteLoadBalancerTlsCertificate() {
        this.add('lightsail:DeleteLoadBalancerTlsCertificate');
        return this;
    }
    /**
     * Deletes a relational database.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteRelationalDatabase.html
     */
    toDeleteRelationalDatabase() {
        this.add('lightsail:DeleteRelationalDatabase');
        return this;
    }
    /**
     * Deletes relational database snapshot.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteRelationalDatabaseSnapshot.html
     */
    toDeleteRelationalDatabaseSnapshot() {
        this.add('lightsail:DeleteRelationalDatabaseSnapshot');
        return this;
    }
    /**
     * Detaches a disk from an instance.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DetachDisk.html
     */
    toDetachDisk() {
        this.add('lightsail:DetachDisk');
        return this;
    }
    /**
     * Detaches one or more instances from a load balancer.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DetachInstancesFromLoadBalancer.html
     */
    toDetachInstancesFromLoadBalancer() {
        this.add('lightsail:DetachInstancesFromLoadBalancer');
        return this;
    }
    /**
     * Detaches a static IP from an instance to which it is attached.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DetachStaticIp.html
     */
    toDetachStaticIp() {
        this.add('lightsail:DetachStaticIp');
        return this;
    }
    /**
     * Downloads the default key pair used to authenticate and connect to instances in a specific AWS Region.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DownloadDefaultKeyPair.html
     */
    toDownloadDefaultKeyPair() {
        this.add('lightsail:DownloadDefaultKeyPair');
        return this;
    }
    /**
     * Exports an Amazon Lightsail snapshot to Amazon EC2.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_ExportSnapshot.html
     */
    toExportSnapshot() {
        this.add('lightsail:ExportSnapshot');
        return this;
    }
    /**
     * Returns the names of all active (not deleted) resources.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetActiveNames.html
     */
    toGetActiveNames() {
        this.add('lightsail:GetActiveNames');
        return this;
    }
    /**
     * Returns a list of instance images, or blueprints. You can use a blueprint to create a new instance already running a specific operating system, as well as a pre-installed application or development stack. The software that runs on your instance depends on the blueprint you define when creating the instance.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetBlueprints.html
     */
    toGetBlueprints() {
        this.add('lightsail:GetBlueprints');
        return this;
    }
    /**
     * Returns a list of instance bundles. You can use a bundle to create a new instance with a set of performance specifications, such as CPU count, disk size, RAM size, and network transfer allowance. The cost of your instance depends on the bundle you define when creating the instance.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetBundles.html
     */
    toGetBundles() {
        this.add('lightsail:GetBundles');
        return this;
    }
    /**
     * Returns information about all CloudFormation stacks used to create Amazon EC2 resources from exported Amazon Lightsail snapshots.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetCloudFormationStackRecords.html
     */
    toGetCloudFormationStackRecords() {
        this.add('lightsail:GetCloudFormationStackRecords');
        return this;
    }
    /**
     * Returns information about a disk.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetDisk.html
     */
    toGetDisk() {
        this.add('lightsail:GetDisk');
        return this;
    }
    /**
     * Returns information about a disk snapshot.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetDiskSnapshot.html
     */
    toGetDiskSnapshot() {
        this.add('lightsail:GetDiskSnapshot');
        return this;
    }
    /**
     * Returns information about all disk snapshots.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetDiskSnapshots.html
     */
    toGetDiskSnapshots() {
        this.add('lightsail:GetDiskSnapshots');
        return this;
    }
    /**
     * Returns information about all disks.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetDisks.html
     */
    toGetDisks() {
        this.add('lightsail:GetDisks');
        return this;
    }
    /**
     * Returns DNS records for a domain resource.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetDomain.html
     */
    toGetDomain() {
        this.add('lightsail:GetDomain');
        return this;
    }
    /**
     * Returns DNS records for all domain resources.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetDomains.html
     */
    toGetDomains() {
        this.add('lightsail:GetDomains');
        return this;
    }
    /**
     * Returns information about all records to export Amazon Lightsail snapshots to Amazon EC2.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetExportSnapshotRecords.html
     */
    toGetExportSnapshotRecords() {
        this.add('lightsail:GetExportSnapshotRecords');
        return this;
    }
    /**
     * Returns information about an instance.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetInstance.html
     */
    toGetInstance() {
        this.add('lightsail:GetInstance');
        return this;
    }
    /**
     * Returns temporary keys you can use to authenticate and connect to an instance.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetInstanceAccessDetails.html
     */
    toGetInstanceAccessDetails() {
        this.add('lightsail:GetInstanceAccessDetails');
        return this;
    }
    /**
     * Returns the data points for the specified metric of an instance.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetInstanceMetricData.html
     */
    toGetInstanceMetricData() {
        this.add('lightsail:GetInstanceMetricData');
        return this;
    }
    /**
     * Returns the port states of an instance.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetInstancePortStates.html
     */
    toGetInstancePortStates() {
        this.add('lightsail:GetInstancePortStates');
        return this;
    }
    /**
     * Returns information about an instance snapshot.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetInstanceSnapshot.html
     */
    toGetInstanceSnapshot() {
        this.add('lightsail:GetInstanceSnapshot');
        return this;
    }
    /**
     * Returns information about all instance snapshots.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetInstanceSnapshots.html
     */
    toGetInstanceSnapshots() {
        this.add('lightsail:GetInstanceSnapshots');
        return this;
    }
    /**
     * Returns the state of an instance.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetInstanceState.html
     */
    toGetInstanceState() {
        this.add('lightsail:GetInstanceState');
        return this;
    }
    /**
     * Returns information about all instances.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetInstances.html
     */
    toGetInstances() {
        this.add('lightsail:GetInstances');
        return this;
    }
    /**
     * Returns information about a key pair.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetKeyPair.html
     */
    toGetKeyPair() {
        this.add('lightsail:GetKeyPair');
        return this;
    }
    /**
     * Returns information about all key pairs.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetKeyPairs.html
     */
    toGetKeyPairs() {
        this.add('lightsail:GetKeyPairs');
        return this;
    }
    /**
     * Returns information about a load balancer.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetLoadBalancer.html
     */
    toGetLoadBalancer() {
        this.add('lightsail:GetLoadBalancer');
        return this;
    }
    /**
     * Returns the data points for the specified metric of a load balancer.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetLoadBalancerMetricData.html
     */
    toGetLoadBalancerMetricData() {
        this.add('lightsail:GetLoadBalancerMetricData');
        return this;
    }
    /**
     * Returns information about a load balancer TLS certificate.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DeleteLoadBalancerTlsCertificate.html
     */
    toGetLoadBalancerTlsCertificates() {
        this.add('lightsail:GetLoadBalancerTlsCertificates');
        return this;
    }
    /**
     * Returns information about load balancers.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetLoadBalancers.html
     */
    toGetLoadBalancers() {
        this.add('lightsail:GetLoadBalancers');
        return this;
    }
    /**
     * Returns information about an operation. Operations include events such as when you create an instance, allocate a static IP, attach a static IP, and so on.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetOperation.html
     */
    toGetOperation() {
        this.add('lightsail:GetOperation');
        return this;
    }
    /**
     * Returns information about all operations. Operations include events such as when you create an instance, allocate a static IP, attach a static IP, and so on.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetOperations.html
     */
    toGetOperations() {
        this.add('lightsail:GetOperations');
        return this;
    }
    /**
     * Returns operations for a resource.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetOperationsForResource.html
     */
    toGetOperationsForResource() {
        this.add('lightsail:GetOperationsForResource');
        return this;
    }
    /**
     * Returns a list of all valid AWS Regions for Amazon Lightsail.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRegions.html
     */
    toGetRegions() {
        this.add('lightsail:GetRegions');
        return this;
    }
    /**
     * Returns information about a relational database.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabase.html
     */
    toGetRelationalDatabase() {
        this.add('lightsail:GetRelationalDatabase');
        return this;
    }
    /**
     * Returns a list of relational database images, or blueprints. You can use a blueprint to create a new database running a specific database engine. The database engine that runs on your database depends on the blueprint you define when creating the relational database.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseBlueprints.html
     */
    toGetRelationalDatabaseBlueprints() {
        this.add('lightsail:GetRelationalDatabaseBlueprints');
        return this;
    }
    /**
     * Returns a list of relational database bundles. You can use a bundle to create a new database with a set of performance specifications, such as CPU count, disk size, RAM size, network transfer allowance, and standard of high availability. The cost of your database depends on the bundle you define when creating the relational database.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseBundles.html
     */
    toGetRelationalDatabaseBundles() {
        this.add('lightsail:GetRelationalDatabaseBundles');
        return this;
    }
    /**
     * Returns events for a relational database.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseEvents.html
     */
    toGetRelationalDatabaseEvents() {
        this.add('lightsail:GetRelationalDatabaseEvents');
        return this;
    }
    /**
     * Returns events for the specified log stream of a relational database.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseLogEvents.html
     */
    toGetRelationalDatabaseLogEvents() {
        this.add('lightsail:GetRelationalDatabaseLogEvents');
        return this;
    }
    /**
     * Returns the log streams available for a relational database.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseLogStreams.html
     */
    toGetRelationalDatabaseLogStreams() {
        this.add('lightsail:GetRelationalDatabaseLogStreams');
        return this;
    }
    /**
     * Returns the master user password of a relational database.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseMasterUserPassword.html
     */
    toGetRelationalDatabaseMasterUserPassword() {
        this.add('lightsail:GetRelationalDatabaseMasterUserPassword');
        return this;
    }
    /**
     * Returns the data points for the specified metric of a relational database.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseMetricData.html
     */
    toGetRelationalDatabaseMetricData() {
        this.add('lightsail:GetRelationalDatabaseMetricData');
        return this;
    }
    /**
     * Returns the parameters of a relational database.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseParameters.html
     */
    toGetRelationalDatabaseParameters() {
        this.add('lightsail:GetRelationalDatabaseParameters');
        return this;
    }
    /**
     * Returns information about a relational database snapshot.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseSnapshot.html
     */
    toGetRelationalDatabaseSnapshot() {
        this.add('lightsail:GetRelationalDatabaseSnapshot');
        return this;
    }
    /**
     * Returns information about all relational database snapshots.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabaseSnapshots.html
     */
    toGetRelationalDatabaseSnapshots() {
        this.add('lightsail:GetRelationalDatabaseSnapshots');
        return this;
    }
    /**
     * Return information about all relational databases.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetRelationalDatabases.html
     */
    toGetRelationalDatabases() {
        this.add('lightsail:GetRelationalDatabases');
        return this;
    }
    /**
     * Returns information about a static IP.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetStaticIp.html
     */
    toGetStaticIp() {
        this.add('lightsail:GetStaticIp');
        return this;
    }
    /**
     * Returns information about all static IPs.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_GetStaticIps.html
     */
    toGetStaticIps() {
        this.add('lightsail:GetStaticIps');
        return this;
    }
    /**
     * Imports a public key from a key pair.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_ImportKeyPair.html
     */
    toImportKeyPair() {
        this.add('lightsail:ImportKeyPair');
        return this;
    }
    /**
     * Returns a boolean value indicating whether the Amazon Lightsail virtual private cloud (VPC) is peered.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_IsVpcPeered.html
     */
    toIsVpcPeered() {
        this.add('lightsail:IsVpcPeered');
        return this;
    }
    /**
     * Adds, or opens a public port of an instance.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_OpenInstancePublicPorts.html
     */
    toOpenInstancePublicPorts() {
        this.add('lightsail:OpenInstancePublicPorts');
        return this;
    }
    /**
     * Tries to peer the Amazon Lightsail virtual private cloud (VPC) with the default VPC.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_PeerVpc.html
     */
    toPeerVpc() {
        this.add('lightsail:PeerVpc');
        return this;
    }
    /**
     * Sets the specified open ports for an instance, and closes all ports for every protocol not included in the request.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_PutInstancePublicPorts.html
     */
    toPutInstancePublicPorts() {
        this.add('lightsail:PutInstancePublicPorts');
        return this;
    }
    /**
     * Reboots an instance that is in a running state.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_RebootInstance.html
     */
    toRebootInstance() {
        this.add('lightsail:RebootInstance');
        return this;
    }
    /**
     * Reboots a relational database that is in a running state.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_RebootRelationalDatabase.html
     */
    toRebootRelationalDatabase() {
        this.add('lightsail:RebootRelationalDatabase');
        return this;
    }
    /**
     * Deletes a static IP.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_ReleaseStaticIp.html
     */
    toReleaseStaticIp() {
        this.add('lightsail:ReleaseStaticIp');
        return this;
    }
    /**
     * Starts an instance that is in a stopped state.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_StartInstance.html
     */
    toStartInstance() {
        this.add('lightsail:StartInstance');
        return this;
    }
    /**
     * Starts a relational database that is in a stopped state.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_StartRelationalDatabase.html
     */
    toStartRelationalDatabase() {
        this.add('lightsail:StartRelationalDatabase');
        return this;
    }
    /**
     * Stops an instance that is in a running state.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_StopInstance.html
     */
    toStopInstance() {
        this.add('lightsail:StopInstance');
        return this;
    }
    /**
     * Stops a relational database that is in a running state.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_StopRelationalDatabase.html
     */
    toStopRelationalDatabase() {
        this.add('lightsail:StopRelationalDatabase');
        return this;
    }
    /**
     * Tags a resource.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_TagResource.html
     */
    toTagResource() {
        this.add('lightsail:TagResource');
        return this;
    }
    /**
     * Attempts to unpeer the Amazon Lightsail virtual private cloud (VPC) from the default VPC.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_UnpeerVpc.html
     */
    toUnpeerVpc() {
        this.add('lightsail:UnpeerVpc');
        return this;
    }
    /**
     * Untags a resource.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_UntagResource.html
     */
    toUntagResource() {
        this.add('lightsail:UntagResource');
        return this;
    }
    /**
     * Updates a domain recordset after it is created.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_UpdateDomainEntry.html
     */
    toUpdateDomainEntry() {
        this.add('lightsail:UpdateDomainEntry');
        return this;
    }
    /**
     * Updates a load balancer attribute, such as the health check path and session stickiness.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DetachInstancesFromLoadBalancer.html
     */
    toUpdateLoadBalancerAttribute() {
        this.add('lightsail:UpdateLoadBalancerAttribute');
        return this;
    }
    /**
     * Updates a relational database.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_UpdateRelationalDatabase.html
     */
    toUpdateRelationalDatabase() {
        this.add('lightsail:UpdateRelationalDatabase');
        return this;
    }
    /**
     * Updates the parameters of a relational database.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_UpdateRelationalDatabaseParameters.html
     */
    toUpdateRelationalDatabaseParameters() {
        this.add('lightsail:UpdateRelationalDatabaseParameters');
        return this;
    }
    /**
     * Adds a resource of type Domain to the statement
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_Domain.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDomain(id, account, region, partition) {
        var arn = 'arn:${Partition}:lightsail:${Region}:${Account}:Domain/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type Instance to the statement
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_Instance.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onInstance(id, account, region, partition) {
        var arn = 'arn:${Partition}:lightsail:${Region}:${Account}:Instance/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type InstanceSnapshot to the statement
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_InstanceSnapshot.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onInstanceSnapshot(id, account, region, partition) {
        var arn = 'arn:${Partition}:lightsail:${Region}:${Account}:InstanceSnapshot/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type KeyPair to the statement
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_KeyPair.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onKeyPair(id, account, region, partition) {
        var arn = 'arn:${Partition}:lightsail:${Region}:${Account}:KeyPair/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type StaticIp to the statement
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_StaticIp.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onStaticIp(id, account, region, partition) {
        var arn = 'arn:${Partition}:lightsail:${Region}:${Account}:StaticIp/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type Disk to the statement
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_Disk.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDisk(id, account, region, partition) {
        var arn = 'arn:${Partition}:lightsail:${Region}:${Account}:Disk/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type DiskSnapshot to the statement
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_DiskSnapshot.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDiskSnapshot(id, account, region, partition) {
        var arn = 'arn:${Partition}:lightsail:${Region}:${Account}:DiskSnapshot/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type LoadBalancer to the statement
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_LoadBalancer.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onLoadBalancer(id, account, region, partition) {
        var arn = 'arn:${Partition}:lightsail:${Region}:${Account}:LoadBalancer/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type PeeredVpc to the statement
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/PeeredVpc.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onPeeredVpc(id, account, region, partition) {
        var arn = 'arn:${Partition}:lightsail:${Region}:${Account}:PeeredVpc/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type LoadBalancerTlsCertificate to the statement
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/LoadBalancerTlsCertificate.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onLoadBalancerTlsCertificate(id, account, region, partition) {
        var arn = 'arn:${Partition}:lightsail:${Region}:${Account}:LoadBalancerTlsCertificate/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type ExportSnapshotRecord to the statement
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_ExportSnapshotRecord.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onExportSnapshotRecord(id, account, region, partition) {
        var arn = 'arn:${Partition}:lightsail:${Region}:${Account}:ExportSnapshotRecord/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type CloudFormationStackRecord to the statement
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CloudFormationStackRecord.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onCloudFormationStackRecord(id, account, region, partition) {
        var arn = 'arn:${Partition}:lightsail:${Region}:${Account}:CloudFormationStackRecord/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type RelationalDatabase to the statement
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_RelationalDatabase.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRelationalDatabase(id, account, region, partition) {
        var arn = 'arn:${Partition}:lightsail:${Region}:${Account}:RelationalDatabase/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type RelationalDatabaseSnapshot to the statement
     *
     * https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_RelationalDatabaseSnapshot.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRelationalDatabaseSnapshot(id, account, region, partition) {
        var arn = 'arn:${Partition}:lightsail:${Region}:${Account}:RelationalDatabaseSnapshot/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Lightsail = Lightsail;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYW1hem9ubGlnaHRzYWlsLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiYW1hem9ubGlnaHRzYWlsLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBLHNDQUFvRTtBQUVwRTs7OztHQUlHO0FBQ0gsTUFBYSxTQUFVLFNBQVEsd0JBQWU7SUF1b0M1Qzs7OztPQUlHO0lBQ0gsWUFBYSxHQUFZO1FBQ3ZCLEtBQUssQ0FBQyxHQUFHLENBQUMsQ0FBQztRQTVvQ04sa0JBQWEsR0FBRyxXQUFXLENBQUM7UUFDekIsZUFBVSxHQUFZO1lBQzlCLGtCQUFrQixFQUFFO2dCQUNsQixLQUFLLEVBQUUsMEZBQTBGO2dCQUNqRyxhQUFhLEVBQUUsa0VBQWtFO2dCQUNqRixhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLFVBQVUsRUFBRTt3QkFDVixVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7YUFDRjtZQUNELFlBQVksRUFBRTtnQkFDWixLQUFLLEVBQUUsb0ZBQW9GO2dCQUMzRixhQUFhLEVBQUUsaUNBQWlDO2dCQUNoRCxhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLE1BQU0sRUFBRTt3QkFDTixVQUFVLEVBQUUsSUFBSTtxQkFDakI7b0JBQ0QsVUFBVSxFQUFFO3dCQUNWLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsK0JBQStCLEVBQUU7Z0JBQy9CLEtBQUssRUFBRSx1R0FBdUc7Z0JBQzlHLGFBQWEsRUFBRSxvREFBb0Q7Z0JBQ25FLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsVUFBVSxFQUFFO3dCQUNWLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtvQkFDRCxjQUFjLEVBQUU7d0JBQ2QsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxrQ0FBa0MsRUFBRTtnQkFDbEMsS0FBSyxFQUFFLDBHQUEwRztnQkFDakgsYUFBYSxFQUFFLGdEQUFnRDtnQkFDL0QsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixjQUFjLEVBQUU7d0JBQ2QsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxnQkFBZ0IsRUFBRTtnQkFDaEIsS0FBSyxFQUFFLHdGQUF3RjtnQkFDL0YsYUFBYSxFQUFFLDhDQUE4QztnQkFDN0QsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixVQUFVLEVBQUU7d0JBQ1YsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO29CQUNELFVBQVUsRUFBRTt3QkFDVixVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7YUFDRjtZQUNELDBCQUEwQixFQUFFO2dCQUMxQixLQUFLLEVBQUUsa0dBQWtHO2dCQUN6RyxhQUFhLEVBQUUsc0NBQXNDO2dCQUNyRCxhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLFVBQVUsRUFBRTt3QkFDVixVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7YUFDRjtZQUNELGNBQWMsRUFBRTtnQkFDZCxLQUFLLEVBQUUsc0ZBQXNGO2dCQUM3RixhQUFhLEVBQUUsdUVBQXVFO2dCQUN0RixhQUFhLEVBQUUsT0FBTzthQUN2QjtZQUNELDJCQUEyQixFQUFFO2dCQUMzQixLQUFLLEVBQUUsbUdBQW1HO2dCQUMxRyxhQUFhLEVBQUUsK0VBQStFO2dCQUM5RixhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLHNCQUFzQixFQUFFO3dCQUN0QixVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7YUFDRjtZQUNELFlBQVksRUFBRTtnQkFDWixLQUFLLEVBQUUsb0ZBQW9GO2dCQUMzRixhQUFhLEVBQUUsaUJBQWlCO2dCQUNoQyxhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLE1BQU0sRUFBRTt3QkFDTixVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7Z0JBQ0QsWUFBWSxFQUFFO29CQUNaLDBCQUEwQjtvQkFDMUIsYUFBYTtpQkFDZDthQUNGO1lBQ0Qsd0JBQXdCLEVBQUU7Z0JBQ3hCLEtBQUssRUFBRSxnR0FBZ0c7Z0JBQ3ZHLGFBQWEsRUFBRSwrQkFBK0I7Z0JBQzlDLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsTUFBTSxFQUFFO3dCQUNOLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjtnQkFDRCxZQUFZLEVBQUU7b0JBQ1osMEJBQTBCO29CQUMxQixhQUFhO2lCQUNkO2FBQ0Y7WUFDRCxvQkFBb0IsRUFBRTtnQkFDcEIsS0FBSyxFQUFFLDRGQUE0RjtnQkFDbkcsYUFBYSxFQUFFLDBCQUEwQjtnQkFDekMsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixNQUFNLEVBQUU7d0JBQ04sVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2dCQUNELFlBQVksRUFBRTtvQkFDWiwwQkFBMEI7b0JBQzFCLGFBQWE7aUJBQ2Q7YUFDRjtZQUNELGNBQWMsRUFBRTtnQkFDZCxLQUFLLEVBQUUsc0ZBQXNGO2dCQUM3RixhQUFhLEVBQUUsMERBQTBEO2dCQUN6RSxhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLFFBQVEsRUFBRTt3QkFDUixVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7Z0JBQ0QsWUFBWSxFQUFFO29CQUNaLDBCQUEwQjtvQkFDMUIsYUFBYTtpQkFDZDthQUNGO1lBQ0QsbUJBQW1CLEVBQUU7Z0JBQ25CLEtBQUssRUFBRSwyRkFBMkY7Z0JBQ2xHLGFBQWEsRUFBRSwyTUFBMk07Z0JBQzFOLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsUUFBUSxFQUFFO3dCQUNSLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0Qsd0JBQXdCLEVBQUU7Z0JBQ3hCLEtBQUssRUFBRSxnR0FBZ0c7Z0JBQ3ZHLGFBQWEsRUFBRSwrQkFBK0I7Z0JBQzlDLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsVUFBVSxFQUFFO3dCQUNWLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtvQkFDRCxrQkFBa0IsRUFBRTt3QkFDbEIsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2dCQUNELFlBQVksRUFBRTtvQkFDWiwwQkFBMEI7b0JBQzFCLGFBQWE7aUJBQ2Q7YUFDRjtZQUNELGlCQUFpQixFQUFFO2dCQUNqQixLQUFLLEVBQUUseUZBQXlGO2dCQUNoRyxhQUFhLEVBQUUsZ0NBQWdDO2dCQUMvQyxhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLFNBQVMsRUFBRTt3QkFDVCxVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7Z0JBQ0QsWUFBWSxFQUFFO29CQUNaLDBCQUEwQjtvQkFDMUIsYUFBYTtpQkFDZDthQUNGO1lBQ0QsNkJBQTZCLEVBQUU7Z0JBQzdCLEtBQUssRUFBRSxxR0FBcUc7Z0JBQzVHLGFBQWEsRUFBRSw4REFBOEQ7Z0JBQzdFLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsVUFBVSxFQUFFO3dCQUNWLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtvQkFDRCxrQkFBa0IsRUFBRTt3QkFDbEIsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2dCQUNELFlBQVksRUFBRTtvQkFDWiwwQkFBMEI7b0JBQzFCLGFBQWE7aUJBQ2Q7YUFDRjtZQUNELGVBQWUsRUFBRTtnQkFDZixLQUFLLEVBQUUsdUZBQXVGO2dCQUM5RixhQUFhLEVBQUUscUVBQXFFO2dCQUNwRixhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLFNBQVMsRUFBRTt3QkFDVCxVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7Z0JBQ0QsWUFBWSxFQUFFO29CQUNaLDBCQUEwQjtvQkFDMUIsYUFBYTtpQkFDZDthQUNGO1lBQ0Qsb0JBQW9CLEVBQUU7Z0JBQ3BCLEtBQUssRUFBRSw0RkFBNEY7Z0JBQ25HLGFBQWEsRUFBRSwwQkFBMEI7Z0JBQ3pDLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsY0FBYyxFQUFFO3dCQUNkLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjtnQkFDRCxZQUFZLEVBQUU7b0JBQ1osMEJBQTBCO29CQUMxQixhQUFhO2lCQUNkO2FBQ0Y7WUFDRCxrQ0FBa0MsRUFBRTtnQkFDbEMsS0FBSyxFQUFFLDBHQUEwRztnQkFDakgsYUFBYSxFQUFFLDBDQUEwQztnQkFDekQsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixjQUFjLEVBQUU7d0JBQ2QsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCwwQkFBMEIsRUFBRTtnQkFDMUIsS0FBSyxFQUFFLGtHQUFrRztnQkFDekcsYUFBYSxFQUFFLG9DQUFvQztnQkFDbkQsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixvQkFBb0IsRUFBRTt3QkFDcEIsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2dCQUNELFlBQVksRUFBRTtvQkFDWiwwQkFBMEI7b0JBQzFCLGFBQWE7aUJBQ2Q7YUFDRjtZQUNELHNDQUFzQyxFQUFFO2dCQUN0QyxLQUFLLEVBQUUsOEdBQThHO2dCQUNySCxhQUFhLEVBQUUsb0RBQW9EO2dCQUNuRSxhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLG9CQUFvQixFQUFFO3dCQUNwQixVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7Z0JBQ0QsWUFBWSxFQUFFO29CQUNaLDBCQUEwQjtvQkFDMUIsYUFBYTtpQkFDZDthQUNGO1lBQ0Qsa0NBQWtDLEVBQUU7Z0JBQ2xDLEtBQUssRUFBRSwwR0FBMEc7Z0JBQ2pILGFBQWEsRUFBRSx5Q0FBeUM7Z0JBQ3hELGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsNEJBQTRCLEVBQUU7d0JBQzVCLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjtnQkFDRCxZQUFZLEVBQUU7b0JBQ1osMEJBQTBCO29CQUMxQixhQUFhO2lCQUNkO2FBQ0Y7WUFDRCxZQUFZLEVBQUU7Z0JBQ1osS0FBSyxFQUFFLG9GQUFvRjtnQkFDM0YsYUFBYSxFQUFFLGlCQUFpQjtnQkFDaEMsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixNQUFNLEVBQUU7d0JBQ04sVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxvQkFBb0IsRUFBRTtnQkFDcEIsS0FBSyxFQUFFLDRGQUE0RjtnQkFDbkcsYUFBYSxFQUFFLDBCQUEwQjtnQkFDekMsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixNQUFNLEVBQUU7d0JBQ04sVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxjQUFjLEVBQUU7Z0JBQ2QsS0FBSyxFQUFFLHNGQUFzRjtnQkFDN0YsYUFBYSxFQUFFLHVEQUF1RDtnQkFDdEUsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixRQUFRLEVBQUU7d0JBQ1IsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxtQkFBbUIsRUFBRTtnQkFDbkIsS0FBSyxFQUFFLDJGQUEyRjtnQkFDbEcsYUFBYSxFQUFFLG1EQUFtRDtnQkFDbEUsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixRQUFRLEVBQUU7d0JBQ1IsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxnQkFBZ0IsRUFBRTtnQkFDaEIsS0FBSyxFQUFFLHdGQUF3RjtnQkFDL0YsYUFBYSxFQUFFLHNCQUFzQjtnQkFDckMsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixVQUFVLEVBQUU7d0JBQ1YsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCx3QkFBd0IsRUFBRTtnQkFDeEIsS0FBSyxFQUFFLGdHQUFnRztnQkFDdkcsYUFBYSxFQUFFLCtCQUErQjtnQkFDOUMsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixrQkFBa0IsRUFBRTt3QkFDbEIsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxlQUFlLEVBQUU7Z0JBQ2YsS0FBSyxFQUFFLHVGQUF1RjtnQkFDOUYsYUFBYSxFQUFFLHFFQUFxRTtnQkFDcEYsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixTQUFTLEVBQUU7d0JBQ1QsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxxQkFBcUIsRUFBRTtnQkFDckIsS0FBSyxFQUFFLDZGQUE2RjtnQkFDcEcsYUFBYSxFQUFFLHNJQUFzSTtnQkFDckosYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixVQUFVLEVBQUU7d0JBQ1YsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxvQkFBb0IsRUFBRTtnQkFDcEIsS0FBSyxFQUFFLDRGQUE0RjtnQkFDbkcsYUFBYSxFQUFFLDBCQUEwQjtnQkFDekMsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixjQUFjLEVBQUU7d0JBQ2QsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxrQ0FBa0MsRUFBRTtnQkFDbEMsS0FBSyxFQUFFLDBHQUEwRztnQkFDakgsYUFBYSxFQUFFLDBDQUEwQztnQkFDekQsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixjQUFjLEVBQUU7d0JBQ2QsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCwwQkFBMEIsRUFBRTtnQkFDMUIsS0FBSyxFQUFFLGtHQUFrRztnQkFDekcsYUFBYSxFQUFFLGdDQUFnQztnQkFDL0MsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixvQkFBb0IsRUFBRTt3QkFDcEIsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxrQ0FBa0MsRUFBRTtnQkFDbEMsS0FBSyxFQUFFLDBHQUEwRztnQkFDakgsYUFBYSxFQUFFLHVDQUF1QztnQkFDdEQsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZiw0QkFBNEIsRUFBRTt3QkFDNUIsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxZQUFZLEVBQUU7Z0JBQ1osS0FBSyxFQUFFLG9GQUFvRjtnQkFDM0YsYUFBYSxFQUFFLG1DQUFtQztnQkFDbEQsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixNQUFNLEVBQUU7d0JBQ04sVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxpQ0FBaUMsRUFBRTtnQkFDakMsS0FBSyxFQUFFLHlHQUF5RztnQkFDaEgsYUFBYSxFQUFFLHNEQUFzRDtnQkFDckUsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixVQUFVLEVBQUU7d0JBQ1YsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO29CQUNELGNBQWMsRUFBRTt3QkFDZCxVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7YUFDRjtZQUNELGdCQUFnQixFQUFFO2dCQUNoQixLQUFLLEVBQUUsd0ZBQXdGO2dCQUMvRixhQUFhLEVBQUUsZ0VBQWdFO2dCQUMvRSxhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLFVBQVUsRUFBRTt3QkFDVixVQUFVLEVBQUUsSUFBSTtxQkFDakI7b0JBQ0QsVUFBVSxFQUFFO3dCQUNWLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0Qsd0JBQXdCLEVBQUU7Z0JBQ3hCLEtBQUssRUFBRSxnR0FBZ0c7Z0JBQ3ZHLGFBQWEsRUFBRSx3R0FBd0c7Z0JBQ3ZILGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsU0FBUyxFQUFFO3dCQUNULFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsZ0JBQWdCLEVBQUU7Z0JBQ2hCLEtBQUssRUFBRSx3RkFBd0Y7Z0JBQy9GLGFBQWEsRUFBRSxxREFBcUQ7Z0JBQ3BFLGFBQWEsRUFBRSxPQUFPO2FBQ3ZCO1lBQ0QsZ0JBQWdCLEVBQUU7Z0JBQ2hCLEtBQUssRUFBRSx3RkFBd0Y7Z0JBQy9GLGFBQWEsRUFBRSwwREFBMEQ7Z0JBQ3pFLGFBQWEsRUFBRSxNQUFNO2FBQ3RCO1lBQ0QsZUFBZSxFQUFFO2dCQUNmLEtBQUssRUFBRSx1RkFBdUY7Z0JBQzlGLGFBQWEsRUFBRSxzVEFBc1Q7Z0JBQ3JVLGFBQWEsRUFBRSxNQUFNO2FBQ3RCO1lBQ0QsWUFBWSxFQUFFO2dCQUNaLEtBQUssRUFBRSxvRkFBb0Y7Z0JBQzNGLGFBQWEsRUFBRSw0UkFBNFI7Z0JBQzNTLGFBQWEsRUFBRSxNQUFNO2FBQ3RCO1lBQ0QsK0JBQStCLEVBQUU7Z0JBQy9CLEtBQUssRUFBRSx1R0FBdUc7Z0JBQzlHLGFBQWEsRUFBRSxtSUFBbUk7Z0JBQ2xKLGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsMkJBQTJCLEVBQUU7d0JBQzNCLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsU0FBUyxFQUFFO2dCQUNULEtBQUssRUFBRSxpRkFBaUY7Z0JBQ3hGLGFBQWEsRUFBRSxtQ0FBbUM7Z0JBQ2xELGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsTUFBTSxFQUFFO3dCQUNOLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsaUJBQWlCLEVBQUU7Z0JBQ2pCLEtBQUssRUFBRSx5RkFBeUY7Z0JBQ2hHLGFBQWEsRUFBRSw0Q0FBNEM7Z0JBQzNELGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsTUFBTSxFQUFFO3dCQUNOLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0Qsa0JBQWtCLEVBQUU7Z0JBQ2xCLEtBQUssRUFBRSwwRkFBMEY7Z0JBQ2pHLGFBQWEsRUFBRSwrQ0FBK0M7Z0JBQzlELGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsTUFBTSxFQUFFO3dCQUNOLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsVUFBVSxFQUFFO2dCQUNWLEtBQUssRUFBRSxrRkFBa0Y7Z0JBQ3pGLGFBQWEsRUFBRSxzQ0FBc0M7Z0JBQ3JELGFBQWEsRUFBRSxNQUFNO2FBQ3RCO1lBQ0QsV0FBVyxFQUFFO2dCQUNYLEtBQUssRUFBRSxtRkFBbUY7Z0JBQzFGLGFBQWEsRUFBRSw0Q0FBNEM7Z0JBQzNELGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsUUFBUSxFQUFFO3dCQUNSLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsWUFBWSxFQUFFO2dCQUNaLEtBQUssRUFBRSxvRkFBb0Y7Z0JBQzNGLGFBQWEsRUFBRSwrQ0FBK0M7Z0JBQzlELGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsUUFBUSxFQUFFO3dCQUNSLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsMEJBQTBCLEVBQUU7Z0JBQzFCLEtBQUssRUFBRSxrR0FBa0c7Z0JBQ3pHLGFBQWEsRUFBRSwyRkFBMkY7Z0JBQzFHLGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2Ysc0JBQXNCLEVBQUU7d0JBQ3RCLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsYUFBYSxFQUFFO2dCQUNiLEtBQUssRUFBRSxxRkFBcUY7Z0JBQzVGLGFBQWEsRUFBRSx3Q0FBd0M7Z0JBQ3ZELGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsVUFBVSxFQUFFO3dCQUNWLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsMEJBQTBCLEVBQUU7Z0JBQzFCLEtBQUssRUFBRSxrR0FBa0c7Z0JBQ3pHLGFBQWEsRUFBRSxnRkFBZ0Y7Z0JBQy9GLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsVUFBVSxFQUFFO3dCQUNWLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsdUJBQXVCLEVBQUU7Z0JBQ3ZCLEtBQUssRUFBRSwrRkFBK0Y7Z0JBQ3RHLGFBQWEsRUFBRSxrRUFBa0U7Z0JBQ2pGLGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsVUFBVSxFQUFFO3dCQUNWLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsdUJBQXVCLEVBQUU7Z0JBQ3ZCLEtBQUssRUFBRSwrRkFBK0Y7Z0JBQ3RHLGFBQWEsRUFBRSx5Q0FBeUM7Z0JBQ3hELGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsVUFBVSxFQUFFO3dCQUNWLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QscUJBQXFCLEVBQUU7Z0JBQ3JCLEtBQUssRUFBRSw2RkFBNkY7Z0JBQ3BHLGFBQWEsRUFBRSxpREFBaUQ7Z0JBQ2hFLGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2Ysa0JBQWtCLEVBQUU7d0JBQ2xCLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0Qsc0JBQXNCLEVBQUU7Z0JBQ3RCLEtBQUssRUFBRSw4RkFBOEY7Z0JBQ3JHLGFBQWEsRUFBRSxtREFBbUQ7Z0JBQ2xFLGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2Ysa0JBQWtCLEVBQUU7d0JBQ2xCLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0Qsa0JBQWtCLEVBQUU7Z0JBQ2xCLEtBQUssRUFBRSwwRkFBMEY7Z0JBQ2pHLGFBQWEsRUFBRSxtQ0FBbUM7Z0JBQ2xELGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsVUFBVSxFQUFFO3dCQUNWLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsY0FBYyxFQUFFO2dCQUNkLEtBQUssRUFBRSxzRkFBc0Y7Z0JBQzdGLGFBQWEsRUFBRSwwQ0FBMEM7Z0JBQ3pELGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsVUFBVSxFQUFFO3dCQUNWLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsWUFBWSxFQUFFO2dCQUNaLEtBQUssRUFBRSxvRkFBb0Y7Z0JBQzNGLGFBQWEsRUFBRSx1Q0FBdUM7Z0JBQ3RELGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsU0FBUyxFQUFFO3dCQUNULFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsYUFBYSxFQUFFO2dCQUNiLEtBQUssRUFBRSxxRkFBcUY7Z0JBQzVGLGFBQWEsRUFBRSwwQ0FBMEM7Z0JBQ3pELGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsU0FBUyxFQUFFO3dCQUNULFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsaUJBQWlCLEVBQUU7Z0JBQ2pCLEtBQUssRUFBRSx5RkFBeUY7Z0JBQ2hHLGFBQWEsRUFBRSw0Q0FBNEM7Z0JBQzNELGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsY0FBYyxFQUFFO3dCQUNkLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsMkJBQTJCLEVBQUU7Z0JBQzNCLEtBQUssRUFBRSxtR0FBbUc7Z0JBQzFHLGFBQWEsRUFBRSxzRUFBc0U7Z0JBQ3JGLGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsY0FBYyxFQUFFO3dCQUNkLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsZ0NBQWdDLEVBQUU7Z0JBQ2hDLEtBQUssRUFBRSwwR0FBMEc7Z0JBQ2pILGFBQWEsRUFBRSw0REFBNEQ7Z0JBQzNFLGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsY0FBYyxFQUFFO3dCQUNkLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0Qsa0JBQWtCLEVBQUU7Z0JBQ2xCLEtBQUssRUFBRSwwRkFBMEY7Z0JBQ2pHLGFBQWEsRUFBRSwyQ0FBMkM7Z0JBQzFELGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsY0FBYyxFQUFFO3dCQUNkLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsY0FBYyxFQUFFO2dCQUNkLEtBQUssRUFBRSxzRkFBc0Y7Z0JBQzdGLGFBQWEsRUFBRSw2SkFBNko7Z0JBQzVLLGFBQWEsRUFBRSxNQUFNO2FBQ3RCO1lBQ0QsZUFBZSxFQUFFO2dCQUNmLEtBQUssRUFBRSx1RkFBdUY7Z0JBQzlGLGFBQWEsRUFBRSwrSkFBK0o7Z0JBQzlLLGFBQWEsRUFBRSxNQUFNO2FBQ3RCO1lBQ0QsMEJBQTBCLEVBQUU7Z0JBQzFCLEtBQUssRUFBRSxrR0FBa0c7Z0JBQ3pHLGFBQWEsRUFBRSxvQ0FBb0M7Z0JBQ25ELGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsUUFBUSxFQUFFO3dCQUNSLFVBQVUsRUFBRSxLQUFLO3FCQUNsQjtvQkFDRCxVQUFVLEVBQUU7d0JBQ1YsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO29CQUNELGtCQUFrQixFQUFFO3dCQUNsQixVQUFVLEVBQUUsS0FBSztxQkFDbEI7b0JBQ0QsU0FBUyxFQUFFO3dCQUNULFVBQVUsRUFBRSxLQUFLO3FCQUNsQjtvQkFDRCxVQUFVLEVBQUU7d0JBQ1YsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2FBQ0Y7WUFDRCxZQUFZLEVBQUU7Z0JBQ1osS0FBSyxFQUFFLG9GQUFvRjtnQkFDM0YsYUFBYSxFQUFFLCtEQUErRDtnQkFDOUUsYUFBYSxFQUFFLE1BQU07YUFDdEI7WUFDRCx1QkFBdUIsRUFBRTtnQkFDdkIsS0FBSyxFQUFFLCtGQUErRjtnQkFDdEcsYUFBYSxFQUFFLGtEQUFrRDtnQkFDakUsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixvQkFBb0IsRUFBRTt3QkFDcEIsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxpQ0FBaUMsRUFBRTtnQkFDakMsS0FBSyxFQUFFLHlHQUF5RztnQkFDaEgsYUFBYSxFQUFFLDZRQUE2UTtnQkFDNVIsYUFBYSxFQUFFLE1BQU07YUFDdEI7WUFDRCw4QkFBOEIsRUFBRTtnQkFDOUIsS0FBSyxFQUFFLHNHQUFzRztnQkFDN0csYUFBYSxFQUFFLGlWQUFpVjtnQkFDaFcsYUFBYSxFQUFFLE1BQU07YUFDdEI7WUFDRCw2QkFBNkIsRUFBRTtnQkFDN0IsS0FBSyxFQUFFLHFHQUFxRztnQkFDNUcsYUFBYSxFQUFFLDJDQUEyQztnQkFDMUQsYUFBYSxFQUFFLE1BQU07YUFDdEI7WUFDRCxnQ0FBZ0MsRUFBRTtnQkFDaEMsS0FBSyxFQUFFLHdHQUF3RztnQkFDL0csYUFBYSxFQUFFLHVFQUF1RTtnQkFDdEYsYUFBYSxFQUFFLE1BQU07YUFDdEI7WUFDRCxpQ0FBaUMsRUFBRTtnQkFDakMsS0FBSyxFQUFFLHlHQUF5RztnQkFDaEgsYUFBYSxFQUFFLDhEQUE4RDtnQkFDN0UsYUFBYSxFQUFFLE1BQU07YUFDdEI7WUFDRCx5Q0FBeUMsRUFBRTtnQkFDekMsS0FBSyxFQUFFLGlIQUFpSDtnQkFDeEgsYUFBYSxFQUFFLDREQUE0RDtnQkFDM0UsYUFBYSxFQUFFLE9BQU87YUFDdkI7WUFDRCxpQ0FBaUMsRUFBRTtnQkFDakMsS0FBSyxFQUFFLHlHQUF5RztnQkFDaEgsYUFBYSxFQUFFLDRFQUE0RTtnQkFDM0YsYUFBYSxFQUFFLE1BQU07YUFDdEI7WUFDRCxpQ0FBaUMsRUFBRTtnQkFDakMsS0FBSyxFQUFFLHlHQUF5RztnQkFDaEgsYUFBYSxFQUFFLGtEQUFrRDtnQkFDakUsYUFBYSxFQUFFLE1BQU07YUFDdEI7WUFDRCwrQkFBK0IsRUFBRTtnQkFDL0IsS0FBSyxFQUFFLHVHQUF1RztnQkFDOUcsYUFBYSxFQUFFLDJEQUEyRDtnQkFDMUUsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixvQkFBb0IsRUFBRTt3QkFDcEIsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxnQ0FBZ0MsRUFBRTtnQkFDaEMsS0FBSyxFQUFFLHdHQUF3RztnQkFDL0csYUFBYSxFQUFFLDhEQUE4RDtnQkFDN0UsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixvQkFBb0IsRUFBRTt3QkFDcEIsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCx3QkFBd0IsRUFBRTtnQkFDeEIsS0FBSyxFQUFFLGdHQUFnRztnQkFDdkcsYUFBYSxFQUFFLG9EQUFvRDtnQkFDbkUsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixvQkFBb0IsRUFBRTt3QkFDcEIsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxhQUFhLEVBQUU7Z0JBQ2IsS0FBSyxFQUFFLHFGQUFxRjtnQkFDNUYsYUFBYSxFQUFFLHdDQUF3QztnQkFDdkQsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixVQUFVLEVBQUU7d0JBQ1YsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxjQUFjLEVBQUU7Z0JBQ2QsS0FBSyxFQUFFLHNGQUFzRjtnQkFDN0YsYUFBYSxFQUFFLDJDQUEyQztnQkFDMUQsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixVQUFVLEVBQUU7d0JBQ1YsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxlQUFlLEVBQUU7Z0JBQ2YsS0FBSyxFQUFFLHVGQUF1RjtnQkFDOUYsYUFBYSxFQUFFLHVDQUF1QztnQkFDdEQsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixTQUFTLEVBQUU7d0JBQ1QsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxhQUFhLEVBQUU7Z0JBQ2IsS0FBSyxFQUFFLHFGQUFxRjtnQkFDNUYsYUFBYSxFQUFFLHdHQUF3RztnQkFDdkgsYUFBYSxFQUFFLE1BQU07YUFDdEI7WUFDRCx5QkFBeUIsRUFBRTtnQkFDekIsS0FBSyxFQUFFLGlHQUFpRztnQkFDeEcsYUFBYSxFQUFFLDhDQUE4QztnQkFDN0QsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixVQUFVLEVBQUU7d0JBQ1YsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxTQUFTLEVBQUU7Z0JBQ1QsS0FBSyxFQUFFLGlGQUFpRjtnQkFDeEYsYUFBYSxFQUFFLHNGQUFzRjtnQkFDckcsYUFBYSxFQUFFLE9BQU87YUFDdkI7WUFDRCx3QkFBd0IsRUFBRTtnQkFDeEIsS0FBSyxFQUFFLGdHQUFnRztnQkFDdkcsYUFBYSxFQUFFLHFIQUFxSDtnQkFDcEksYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixVQUFVLEVBQUU7d0JBQ1YsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxnQkFBZ0IsRUFBRTtnQkFDaEIsS0FBSyxFQUFFLHdGQUF3RjtnQkFDL0YsYUFBYSxFQUFFLGlEQUFpRDtnQkFDaEUsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixVQUFVLEVBQUU7d0JBQ1YsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCwwQkFBMEIsRUFBRTtnQkFDMUIsS0FBSyxFQUFFLGtHQUFrRztnQkFDekcsYUFBYSxFQUFFLDJEQUEyRDtnQkFDMUUsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixvQkFBb0IsRUFBRTt3QkFDcEIsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxpQkFBaUIsRUFBRTtnQkFDakIsS0FBSyxFQUFFLHlGQUF5RjtnQkFDaEcsYUFBYSxFQUFFLHNCQUFzQjtnQkFDckMsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixVQUFVLEVBQUU7d0JBQ1YsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxlQUFlLEVBQUU7Z0JBQ2YsS0FBSyxFQUFFLHVGQUF1RjtnQkFDOUYsYUFBYSxFQUFFLGdEQUFnRDtnQkFDL0QsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixVQUFVLEVBQUU7d0JBQ1YsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCx5QkFBeUIsRUFBRTtnQkFDekIsS0FBSyxFQUFFLGlHQUFpRztnQkFDeEcsYUFBYSxFQUFFLDBEQUEwRDtnQkFDekUsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixvQkFBb0IsRUFBRTt3QkFDcEIsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxjQUFjLEVBQUU7Z0JBQ2QsS0FBSyxFQUFFLHNGQUFzRjtnQkFDN0YsYUFBYSxFQUFFLCtDQUErQztnQkFDOUQsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixVQUFVLEVBQUU7d0JBQ1YsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCx3QkFBd0IsRUFBRTtnQkFDeEIsS0FBSyxFQUFFLGdHQUFnRztnQkFDdkcsYUFBYSxFQUFFLHlEQUF5RDtnQkFDeEUsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixvQkFBb0IsRUFBRTt3QkFDcEIsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxhQUFhLEVBQUU7Z0JBQ2IsS0FBSyxFQUFFLHFGQUFxRjtnQkFDNUYsYUFBYSxFQUFFLGtCQUFrQjtnQkFDakMsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixNQUFNLEVBQUU7d0JBQ04sVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO29CQUNELGNBQWMsRUFBRTt3QkFDZCxVQUFVLEVBQUUsS0FBSztxQkFDbEI7b0JBQ0QsUUFBUSxFQUFFO3dCQUNSLFVBQVUsRUFBRSxLQUFLO3FCQUNsQjtvQkFDRCxVQUFVLEVBQUU7d0JBQ1YsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO29CQUNELGtCQUFrQixFQUFFO3dCQUNsQixVQUFVLEVBQUUsS0FBSztxQkFDbEI7b0JBQ0QsU0FBUyxFQUFFO3dCQUNULFVBQVUsRUFBRSxLQUFLO3FCQUNsQjtvQkFDRCxjQUFjLEVBQUU7d0JBQ2QsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO29CQUNELG9CQUFvQixFQUFFO3dCQUNwQixVQUFVLEVBQUUsS0FBSztxQkFDbEI7b0JBQ0QsNEJBQTRCLEVBQUU7d0JBQzVCLFVBQVUsRUFBRSxLQUFLO3FCQUNsQjtvQkFDRCxVQUFVLEVBQUU7d0JBQ1YsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO2lCQUNGO2dCQUNELFlBQVksRUFBRTtvQkFDWiwwQkFBMEI7b0JBQzFCLGFBQWE7aUJBQ2Q7YUFDRjtZQUNELFdBQVcsRUFBRTtnQkFDWCxLQUFLLEVBQUUsbUZBQW1GO2dCQUMxRixhQUFhLEVBQUUsMkZBQTJGO2dCQUMxRyxhQUFhLEVBQUUsT0FBTzthQUN2QjtZQUNELGVBQWUsRUFBRTtnQkFDZixLQUFLLEVBQUUsdUZBQXVGO2dCQUM5RixhQUFhLEVBQUUsb0JBQW9CO2dCQUNuQyxhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLE1BQU0sRUFBRTt3QkFDTixVQUFVLEVBQUUsS0FBSztxQkFDbEI7b0JBQ0QsY0FBYyxFQUFFO3dCQUNkLFVBQVUsRUFBRSxLQUFLO3FCQUNsQjtvQkFDRCxRQUFRLEVBQUU7d0JBQ1IsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO29CQUNELFVBQVUsRUFBRTt3QkFDVixVQUFVLEVBQUUsS0FBSztxQkFDbEI7b0JBQ0Qsa0JBQWtCLEVBQUU7d0JBQ2xCLFVBQVUsRUFBRSxLQUFLO3FCQUNsQjtvQkFDRCxTQUFTLEVBQUU7d0JBQ1QsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO29CQUNELGNBQWMsRUFBRTt3QkFDZCxVQUFVLEVBQUUsS0FBSztxQkFDbEI7b0JBQ0Qsb0JBQW9CLEVBQUU7d0JBQ3BCLFVBQVUsRUFBRSxLQUFLO3FCQUNsQjtvQkFDRCw0QkFBNEIsRUFBRTt3QkFDNUIsVUFBVSxFQUFFLEtBQUs7cUJBQ2xCO29CQUNELFVBQVUsRUFBRTt3QkFDVixVQUFVLEVBQUUsS0FBSztxQkFDbEI7aUJBQ0Y7Z0JBQ0QsWUFBWSxFQUFFO29CQUNaLDBCQUEwQjtvQkFDMUIsYUFBYTtpQkFDZDthQUNGO1lBQ0QsbUJBQW1CLEVBQUU7Z0JBQ25CLEtBQUssRUFBRSwyRkFBMkY7Z0JBQ2xHLGFBQWEsRUFBRSxpREFBaUQ7Z0JBQ2hFLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsUUFBUSxFQUFFO3dCQUNSLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsNkJBQTZCLEVBQUU7Z0JBQzdCLEtBQUssRUFBRSx5R0FBeUc7Z0JBQ2hILGFBQWEsRUFBRSwwRkFBMEY7Z0JBQ3pHLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsY0FBYyxFQUFFO3dCQUNkLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsMEJBQTBCLEVBQUU7Z0JBQzFCLEtBQUssRUFBRSxrR0FBa0c7Z0JBQ3pHLGFBQWEsRUFBRSxnQ0FBZ0M7Z0JBQy9DLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2Ysb0JBQW9CLEVBQUU7d0JBQ3BCLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0Qsb0NBQW9DLEVBQUU7Z0JBQ3BDLEtBQUssRUFBRSw0R0FBNEc7Z0JBQ25ILGFBQWEsRUFBRSxrREFBa0Q7Z0JBQ2pFLGFBQWEsRUFBRSxPQUFPO2FBQ3ZCO1NBQ0YsQ0FBQztRQUNRLGtCQUFhLEdBQWtCO1lBQ3ZDLFFBQVEsRUFBRTtnQkFDUixNQUFNLEVBQUUsUUFBUTtnQkFDaEIsS0FBSyxFQUFFLGdGQUFnRjtnQkFDdkYsS0FBSyxFQUFFLDhEQUE4RDtnQkFDckUsZUFBZSxFQUFFO29CQUNmLDJCQUEyQjtpQkFDNUI7YUFDRjtZQUNELFVBQVUsRUFBRTtnQkFDVixNQUFNLEVBQUUsVUFBVTtnQkFDbEIsS0FBSyxFQUFFLGtGQUFrRjtnQkFDekYsS0FBSyxFQUFFLGdFQUFnRTtnQkFDdkUsZUFBZSxFQUFFO29CQUNmLDJCQUEyQjtpQkFDNUI7YUFDRjtZQUNELGtCQUFrQixFQUFFO2dCQUNsQixNQUFNLEVBQUUsa0JBQWtCO2dCQUMxQixLQUFLLEVBQUUsMEZBQTBGO2dCQUNqRyxLQUFLLEVBQUUsd0VBQXdFO2dCQUMvRSxlQUFlLEVBQUU7b0JBQ2YsMkJBQTJCO2lCQUM1QjthQUNGO1lBQ0QsU0FBUyxFQUFFO2dCQUNULE1BQU0sRUFBRSxTQUFTO2dCQUNqQixLQUFLLEVBQUUsaUZBQWlGO2dCQUN4RixLQUFLLEVBQUUsK0RBQStEO2dCQUN0RSxlQUFlLEVBQUU7b0JBQ2YsMkJBQTJCO2lCQUM1QjthQUNGO1lBQ0QsVUFBVSxFQUFFO2dCQUNWLE1BQU0sRUFBRSxVQUFVO2dCQUNsQixLQUFLLEVBQUUsa0ZBQWtGO2dCQUN6RixLQUFLLEVBQUUsZ0VBQWdFO2dCQUN2RSxlQUFlLEVBQUU7b0JBQ2YsMkJBQTJCO2lCQUM1QjthQUNGO1lBQ0QsTUFBTSxFQUFFO2dCQUNOLE1BQU0sRUFBRSxNQUFNO2dCQUNkLEtBQUssRUFBRSw4RUFBOEU7Z0JBQ3JGLEtBQUssRUFBRSw0REFBNEQ7Z0JBQ25FLGVBQWUsRUFBRTtvQkFDZiwyQkFBMkI7aUJBQzVCO2FBQ0Y7WUFDRCxjQUFjLEVBQUU7Z0JBQ2QsTUFBTSxFQUFFLGNBQWM7Z0JBQ3RCLEtBQUssRUFBRSxzRkFBc0Y7Z0JBQzdGLEtBQUssRUFBRSxvRUFBb0U7Z0JBQzNFLGVBQWUsRUFBRTtvQkFDZiwyQkFBMkI7aUJBQzVCO2FBQ0Y7WUFDRCxjQUFjLEVBQUU7Z0JBQ2QsTUFBTSxFQUFFLGNBQWM7Z0JBQ3RCLEtBQUssRUFBRSxzRkFBc0Y7Z0JBQzdGLEtBQUssRUFBRSxvRUFBb0U7Z0JBQzNFLGVBQWUsRUFBRTtvQkFDZiwyQkFBMkI7aUJBQzVCO2FBQ0Y7WUFDRCxXQUFXLEVBQUU7Z0JBQ1gsTUFBTSxFQUFFLFdBQVc7Z0JBQ25CLEtBQUssRUFBRSwrRUFBK0U7Z0JBQ3RGLEtBQUssRUFBRSxpRUFBaUU7Z0JBQ3hFLGVBQWUsRUFBRSxFQUFFO2FBQ3BCO1lBQ0QsNEJBQTRCLEVBQUU7Z0JBQzVCLE1BQU0sRUFBRSw0QkFBNEI7Z0JBQ3BDLEtBQUssRUFBRSxnR0FBZ0c7Z0JBQ3ZHLEtBQUssRUFBRSxrRkFBa0Y7Z0JBQ3pGLGVBQWUsRUFBRSxFQUFFO2FBQ3BCO1lBQ0Qsc0JBQXNCLEVBQUU7Z0JBQ3RCLE1BQU0sRUFBRSxzQkFBc0I7Z0JBQzlCLEtBQUssRUFBRSw4RkFBOEY7Z0JBQ3JHLEtBQUssRUFBRSw0RUFBNEU7Z0JBQ25GLGVBQWUsRUFBRSxFQUFFO2FBQ3BCO1lBQ0QsMkJBQTJCLEVBQUU7Z0JBQzNCLE1BQU0sRUFBRSwyQkFBMkI7Z0JBQ25DLEtBQUssRUFBRSxtR0FBbUc7Z0JBQzFHLEtBQUssRUFBRSxpRkFBaUY7Z0JBQ3hGLGVBQWUsRUFBRSxFQUFFO2FBQ3BCO1lBQ0Qsb0JBQW9CLEVBQUU7Z0JBQ3BCLE1BQU0sRUFBRSxvQkFBb0I7Z0JBQzVCLEtBQUssRUFBRSw0RkFBNEY7Z0JBQ25HLEtBQUssRUFBRSwwRUFBMEU7Z0JBQ2pGLGVBQWUsRUFBRTtvQkFDZiwyQkFBMkI7aUJBQzVCO2FBQ0Y7WUFDRCw0QkFBNEIsRUFBRTtnQkFDNUIsTUFBTSxFQUFFLDRCQUE0QjtnQkFDcEMsS0FBSyxFQUFFLG9HQUFvRztnQkFDM0csS0FBSyxFQUFFLGtGQUFrRjtnQkFDekYsZUFBZSxFQUFFO29CQUNmLDJCQUEyQjtpQkFDNUI7YUFDRjtTQUNGLENBQUM7SUFTRixDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksa0JBQWtCO1FBQ3ZCLElBQUksQ0FBQyxHQUFHLENBQUMsNEJBQTRCLENBQUMsQ0FBQztRQUN2QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxZQUFZO1FBQ2pCLElBQUksQ0FBQyxHQUFHLENBQUMsc0JBQXNCLENBQUMsQ0FBQztRQUNqQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSwrQkFBK0I7UUFDcEMsSUFBSSxDQUFDLEdBQUcsQ0FBQyx5Q0FBeUMsQ0FBQyxDQUFDO1FBQ3BELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGtDQUFrQztRQUN2QyxJQUFJLENBQUMsR0FBRyxDQUFDLDRDQUE0QyxDQUFDLENBQUM7UUFDdkQsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZ0JBQWdCO1FBQ3JCLElBQUksQ0FBQyxHQUFHLENBQUMsMEJBQTBCLENBQUMsQ0FBQztRQUNyQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSwwQkFBMEI7UUFDL0IsSUFBSSxDQUFDLEdBQUcsQ0FBQyxvQ0FBb0MsQ0FBQyxDQUFDO1FBQy9DLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGNBQWM7UUFDbkIsSUFBSSxDQUFDLEdBQUcsQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDO1FBQ25DLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDJCQUEyQjtRQUNoQyxJQUFJLENBQUMsR0FBRyxDQUFDLHFDQUFxQyxDQUFDLENBQUM7UUFDaEQsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLFlBQVk7UUFDakIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDO1FBQ2pDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSx3QkFBd0I7UUFDN0IsSUFBSSxDQUFDLEdBQUcsQ0FBQyxrQ0FBa0MsQ0FBQyxDQUFDO1FBQzdDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSxvQkFBb0I7UUFDekIsSUFBSSxDQUFDLEdBQUcsQ0FBQyw4QkFBOEIsQ0FBQyxDQUFDO1FBQ3pDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSxjQUFjO1FBQ25CLElBQUksQ0FBQyxHQUFHLENBQUMsd0JBQXdCLENBQUMsQ0FBQztRQUNuQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxtQkFBbUI7UUFDeEIsSUFBSSxDQUFDLEdBQUcsQ0FBQyw2QkFBNkIsQ0FBQyxDQUFDO1FBQ3hDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSx3QkFBd0I7UUFDN0IsSUFBSSxDQUFDLEdBQUcsQ0FBQyxrQ0FBa0MsQ0FBQyxDQUFDO1FBQzdDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSxpQkFBaUI7UUFDdEIsSUFBSSxDQUFDLEdBQUcsQ0FBQywyQkFBMkIsQ0FBQyxDQUFDO1FBQ3RDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSw2QkFBNkI7UUFDbEMsSUFBSSxDQUFDLEdBQUcsQ0FBQyx1Q0FBdUMsQ0FBQyxDQUFDO1FBQ2xELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSxlQUFlO1FBQ3BCLElBQUksQ0FBQyxHQUFHLENBQUMseUJBQXlCLENBQUMsQ0FBQztRQUNwQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksb0JBQW9CO1FBQ3pCLElBQUksQ0FBQyxHQUFHLENBQUMsOEJBQThCLENBQUMsQ0FBQztRQUN6QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxrQ0FBa0M7UUFDdkMsSUFBSSxDQUFDLEdBQUcsQ0FBQyw0Q0FBNEMsQ0FBQyxDQUFDO1FBQ3ZELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSwwQkFBMEI7UUFDL0IsSUFBSSxDQUFDLEdBQUcsQ0FBQyxvQ0FBb0MsQ0FBQyxDQUFDO1FBQy9DLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSxzQ0FBc0M7UUFDM0MsSUFBSSxDQUFDLEdBQUcsQ0FBQyxnREFBZ0QsQ0FBQyxDQUFDO1FBQzNELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSxrQ0FBa0M7UUFDdkMsSUFBSSxDQUFDLEdBQUcsQ0FBQyw0Q0FBNEMsQ0FBQyxDQUFDO1FBQ3ZELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLFlBQVk7UUFDakIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDO1FBQ2pDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG9CQUFvQjtRQUN6QixJQUFJLENBQUMsR0FBRyxDQUFDLDhCQUE4QixDQUFDLENBQUM7UUFDekMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksY0FBYztRQUNuQixJQUFJLENBQUMsR0FBRyxDQUFDLHdCQUF3QixDQUFDLENBQUM7UUFDbkMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksbUJBQW1CO1FBQ3hCLElBQUksQ0FBQyxHQUFHLENBQUMsNkJBQTZCLENBQUMsQ0FBQztRQUN4QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxnQkFBZ0I7UUFDckIsSUFBSSxDQUFDLEdBQUcsQ0FBQywwQkFBMEIsQ0FBQyxDQUFDO1FBQ3JDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHdCQUF3QjtRQUM3QixJQUFJLENBQUMsR0FBRyxDQUFDLGtDQUFrQyxDQUFDLENBQUM7UUFDN0MsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZUFBZTtRQUNwQixJQUFJLENBQUMsR0FBRyxDQUFDLHlCQUF5QixDQUFDLENBQUM7UUFDcEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kscUJBQXFCO1FBQzFCLElBQUksQ0FBQyxHQUFHLENBQUMsK0JBQStCLENBQUMsQ0FBQztRQUMxQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxvQkFBb0I7UUFDekIsSUFBSSxDQUFDLEdBQUcsQ0FBQyw4QkFBOEIsQ0FBQyxDQUFDO1FBQ3pDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGtDQUFrQztRQUN2QyxJQUFJLENBQUMsR0FBRyxDQUFDLDRDQUE0QyxDQUFDLENBQUM7UUFDdkQsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksMEJBQTBCO1FBQy9CLElBQUksQ0FBQyxHQUFHLENBQUMsb0NBQW9DLENBQUMsQ0FBQztRQUMvQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxrQ0FBa0M7UUFDdkMsSUFBSSxDQUFDLEdBQUcsQ0FBQyw0Q0FBNEMsQ0FBQyxDQUFDO1FBQ3ZELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLFlBQVk7UUFDakIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDO1FBQ2pDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGlDQUFpQztRQUN0QyxJQUFJLENBQUMsR0FBRyxDQUFDLDJDQUEyQyxDQUFDLENBQUM7UUFDdEQsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZ0JBQWdCO1FBQ3JCLElBQUksQ0FBQyxHQUFHLENBQUMsMEJBQTBCLENBQUMsQ0FBQztRQUNyQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx3QkFBd0I7UUFDN0IsSUFBSSxDQUFDLEdBQUcsQ0FBQyxrQ0FBa0MsQ0FBQyxDQUFDO1FBQzdDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGdCQUFnQjtRQUNyQixJQUFJLENBQUMsR0FBRyxDQUFDLDBCQUEwQixDQUFDLENBQUM7UUFDckMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZ0JBQWdCO1FBQ3JCLElBQUksQ0FBQyxHQUFHLENBQUMsMEJBQTBCLENBQUMsQ0FBQztRQUNyQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxlQUFlO1FBQ3BCLElBQUksQ0FBQyxHQUFHLENBQUMseUJBQXlCLENBQUMsQ0FBQztRQUNwQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxZQUFZO1FBQ2pCLElBQUksQ0FBQyxHQUFHLENBQUMsc0JBQXNCLENBQUMsQ0FBQztRQUNqQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSwrQkFBK0I7UUFDcEMsSUFBSSxDQUFDLEdBQUcsQ0FBQyx5Q0FBeUMsQ0FBQyxDQUFDO1FBQ3BELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLFNBQVM7UUFDZCxJQUFJLENBQUMsR0FBRyxDQUFDLG1CQUFtQixDQUFDLENBQUM7UUFDOUIsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksaUJBQWlCO1FBQ3RCLElBQUksQ0FBQyxHQUFHLENBQUMsMkJBQTJCLENBQUMsQ0FBQztRQUN0QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxrQkFBa0I7UUFDdkIsSUFBSSxDQUFDLEdBQUcsQ0FBQyw0QkFBNEIsQ0FBQyxDQUFDO1FBQ3ZDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLFVBQVU7UUFDZixJQUFJLENBQUMsR0FBRyxDQUFDLG9CQUFvQixDQUFDLENBQUM7UUFDL0IsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksV0FBVztRQUNoQixJQUFJLENBQUMsR0FBRyxDQUFDLHFCQUFxQixDQUFDLENBQUM7UUFDaEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksWUFBWTtRQUNqQixJQUFJLENBQUMsR0FBRyxDQUFDLHNCQUFzQixDQUFDLENBQUM7UUFDakMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksMEJBQTBCO1FBQy9CLElBQUksQ0FBQyxHQUFHLENBQUMsb0NBQW9DLENBQUMsQ0FBQztRQUMvQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxhQUFhO1FBQ2xCLElBQUksQ0FBQyxHQUFHLENBQUMsdUJBQXVCLENBQUMsQ0FBQztRQUNsQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSwwQkFBMEI7UUFDL0IsSUFBSSxDQUFDLEdBQUcsQ0FBQyxvQ0FBb0MsQ0FBQyxDQUFDO1FBQy9DLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHVCQUF1QjtRQUM1QixJQUFJLENBQUMsR0FBRyxDQUFDLGlDQUFpQyxDQUFDLENBQUM7UUFDNUMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksdUJBQXVCO1FBQzVCLElBQUksQ0FBQyxHQUFHLENBQUMsaUNBQWlDLENBQUMsQ0FBQztRQUM1QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxxQkFBcUI7UUFDMUIsSUFBSSxDQUFDLEdBQUcsQ0FBQywrQkFBK0IsQ0FBQyxDQUFDO1FBQzFDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHNCQUFzQjtRQUMzQixJQUFJLENBQUMsR0FBRyxDQUFDLGdDQUFnQyxDQUFDLENBQUM7UUFDM0MsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksa0JBQWtCO1FBQ3ZCLElBQUksQ0FBQyxHQUFHLENBQUMsNEJBQTRCLENBQUMsQ0FBQztRQUN2QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxjQUFjO1FBQ25CLElBQUksQ0FBQyxHQUFHLENBQUMsd0JBQXdCLENBQUMsQ0FBQztRQUNuQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxZQUFZO1FBQ2pCLElBQUksQ0FBQyxHQUFHLENBQUMsc0JBQXNCLENBQUMsQ0FBQztRQUNqQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxhQUFhO1FBQ2xCLElBQUksQ0FBQyxHQUFHLENBQUMsdUJBQXVCLENBQUMsQ0FBQztRQUNsQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxpQkFBaUI7UUFDdEIsSUFBSSxDQUFDLEdBQUcsQ0FBQywyQkFBMkIsQ0FBQyxDQUFDO1FBQ3RDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDJCQUEyQjtRQUNoQyxJQUFJLENBQUMsR0FBRyxDQUFDLHFDQUFxQyxDQUFDLENBQUM7UUFDaEQsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZ0NBQWdDO1FBQ3JDLElBQUksQ0FBQyxHQUFHLENBQUMsMENBQTBDLENBQUMsQ0FBQztRQUNyRCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxrQkFBa0I7UUFDdkIsSUFBSSxDQUFDLEdBQUcsQ0FBQyw0QkFBNEIsQ0FBQyxDQUFDO1FBQ3ZDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGNBQWM7UUFDbkIsSUFBSSxDQUFDLEdBQUcsQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDO1FBQ25DLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGVBQWU7UUFDcEIsSUFBSSxDQUFDLEdBQUcsQ0FBQyx5QkFBeUIsQ0FBQyxDQUFDO1FBQ3BDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDBCQUEwQjtRQUMvQixJQUFJLENBQUMsR0FBRyxDQUFDLG9DQUFvQyxDQUFDLENBQUM7UUFDL0MsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksWUFBWTtRQUNqQixJQUFJLENBQUMsR0FBRyxDQUFDLHNCQUFzQixDQUFDLENBQUM7UUFDakMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksdUJBQXVCO1FBQzVCLElBQUksQ0FBQyxHQUFHLENBQUMsaUNBQWlDLENBQUMsQ0FBQztRQUM1QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxpQ0FBaUM7UUFDdEMsSUFBSSxDQUFDLEdBQUcsQ0FBQywyQ0FBMkMsQ0FBQyxDQUFDO1FBQ3RELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDhCQUE4QjtRQUNuQyxJQUFJLENBQUMsR0FBRyxDQUFDLHdDQUF3QyxDQUFDLENBQUM7UUFDbkQsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksNkJBQTZCO1FBQ2xDLElBQUksQ0FBQyxHQUFHLENBQUMsdUNBQXVDLENBQUMsQ0FBQztRQUNsRCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxnQ0FBZ0M7UUFDckMsSUFBSSxDQUFDLEdBQUcsQ0FBQywwQ0FBMEMsQ0FBQyxDQUFDO1FBQ3JELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGlDQUFpQztRQUN0QyxJQUFJLENBQUMsR0FBRyxDQUFDLDJDQUEyQyxDQUFDLENBQUM7UUFDdEQsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kseUNBQXlDO1FBQzlDLElBQUksQ0FBQyxHQUFHLENBQUMsbURBQW1ELENBQUMsQ0FBQztRQUM5RCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxpQ0FBaUM7UUFDdEMsSUFBSSxDQUFDLEdBQUcsQ0FBQywyQ0FBMkMsQ0FBQyxDQUFDO1FBQ3RELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGlDQUFpQztRQUN0QyxJQUFJLENBQUMsR0FBRyxDQUFDLDJDQUEyQyxDQUFDLENBQUM7UUFDdEQsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksK0JBQStCO1FBQ3BDLElBQUksQ0FBQyxHQUFHLENBQUMseUNBQXlDLENBQUMsQ0FBQztRQUNwRCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxnQ0FBZ0M7UUFDckMsSUFBSSxDQUFDLEdBQUcsQ0FBQywwQ0FBMEMsQ0FBQyxDQUFDO1FBQ3JELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHdCQUF3QjtRQUM3QixJQUFJLENBQUMsR0FBRyxDQUFDLGtDQUFrQyxDQUFDLENBQUM7UUFDN0MsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksYUFBYTtRQUNsQixJQUFJLENBQUMsR0FBRyxDQUFDLHVCQUF1QixDQUFDLENBQUM7UUFDbEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksY0FBYztRQUNuQixJQUFJLENBQUMsR0FBRyxDQUFDLHdCQUF3QixDQUFDLENBQUM7UUFDbkMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZUFBZTtRQUNwQixJQUFJLENBQUMsR0FBRyxDQUFDLHlCQUF5QixDQUFDLENBQUM7UUFDcEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksYUFBYTtRQUNsQixJQUFJLENBQUMsR0FBRyxDQUFDLHVCQUF1QixDQUFDLENBQUM7UUFDbEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kseUJBQXlCO1FBQzlCLElBQUksQ0FBQyxHQUFHLENBQUMsbUNBQW1DLENBQUMsQ0FBQztRQUM5QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxTQUFTO1FBQ2QsSUFBSSxDQUFDLEdBQUcsQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDO1FBQzlCLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHdCQUF3QjtRQUM3QixJQUFJLENBQUMsR0FBRyxDQUFDLGtDQUFrQyxDQUFDLENBQUM7UUFDN0MsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZ0JBQWdCO1FBQ3JCLElBQUksQ0FBQyxHQUFHLENBQUMsMEJBQTBCLENBQUMsQ0FBQztRQUNyQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSwwQkFBMEI7UUFDL0IsSUFBSSxDQUFDLEdBQUcsQ0FBQyxvQ0FBb0MsQ0FBQyxDQUFDO1FBQy9DLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGlCQUFpQjtRQUN0QixJQUFJLENBQUMsR0FBRyxDQUFDLDJCQUEyQixDQUFDLENBQUM7UUFDdEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZUFBZTtRQUNwQixJQUFJLENBQUMsR0FBRyxDQUFDLHlCQUF5QixDQUFDLENBQUM7UUFDcEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kseUJBQXlCO1FBQzlCLElBQUksQ0FBQyxHQUFHLENBQUMsbUNBQW1DLENBQUMsQ0FBQztRQUM5QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxjQUFjO1FBQ25CLElBQUksQ0FBQyxHQUFHLENBQUMsd0JBQXdCLENBQUMsQ0FBQztRQUNuQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx3QkFBd0I7UUFDN0IsSUFBSSxDQUFDLEdBQUcsQ0FBQyxrQ0FBa0MsQ0FBQyxDQUFDO1FBQzdDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSxhQUFhO1FBQ2xCLElBQUksQ0FBQyxHQUFHLENBQUMsdUJBQXVCLENBQUMsQ0FBQztRQUNsQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxXQUFXO1FBQ2hCLElBQUksQ0FBQyxHQUFHLENBQUMscUJBQXFCLENBQUMsQ0FBQztRQUNoQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksZUFBZTtRQUNwQixJQUFJLENBQUMsR0FBRyxDQUFDLHlCQUF5QixDQUFDLENBQUM7UUFDcEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksbUJBQW1CO1FBQ3hCLElBQUksQ0FBQyxHQUFHLENBQUMsNkJBQTZCLENBQUMsQ0FBQztRQUN4QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSw2QkFBNkI7UUFDbEMsSUFBSSxDQUFDLEdBQUcsQ0FBQyx1Q0FBdUMsQ0FBQyxDQUFDO1FBQ2xELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDBCQUEwQjtRQUMvQixJQUFJLENBQUMsR0FBRyxDQUFDLG9DQUFvQyxDQUFDLENBQUM7UUFDL0MsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksb0NBQW9DO1FBQ3pDLElBQUksQ0FBQyxHQUFHLENBQUMsOENBQThDLENBQUMsQ0FBQztRQUN6RCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7O09BWUc7SUFDSSxRQUFRLENBQUMsRUFBVSxFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQy9FLElBQUksR0FBRyxHQUFHLDhEQUE4RCxDQUFDO1FBQ3pFLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLE9BQU8sRUFBRSxFQUFFLENBQUMsQ0FBQztRQUMvQixHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxZQUFZLEVBQUUsT0FBTyxJQUFJLEdBQUcsQ0FBQyxDQUFDO1FBQ2hELEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFdBQVcsRUFBRSxNQUFNLElBQUksR0FBRyxDQUFDLENBQUM7UUFDOUMsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsY0FBYyxFQUFFLFNBQVMsSUFBSSxLQUFLLENBQUMsQ0FBQztRQUN0RCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLENBQUM7SUFDdEIsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7T0FZRztJQUNJLFVBQVUsQ0FBQyxFQUFVLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDakYsSUFBSSxHQUFHLEdBQUcsZ0VBQWdFLENBQUM7UUFDM0UsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsT0FBTyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBQy9CLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFlBQVksRUFBRSxPQUFPLElBQUksR0FBRyxDQUFDLENBQUM7UUFDaEQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsV0FBVyxFQUFFLE1BQU0sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUM5QyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsU0FBUyxJQUFJLEtBQUssQ0FBQyxDQUFDO1FBQ3RELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN0QixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksa0JBQWtCLENBQUMsRUFBVSxFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQ3pGLElBQUksR0FBRyxHQUFHLHdFQUF3RSxDQUFDO1FBQ25GLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLE9BQU8sRUFBRSxFQUFFLENBQUMsQ0FBQztRQUMvQixHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxZQUFZLEVBQUUsT0FBTyxJQUFJLEdBQUcsQ0FBQyxDQUFDO1FBQ2hELEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFdBQVcsRUFBRSxNQUFNLElBQUksR0FBRyxDQUFDLENBQUM7UUFDOUMsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsY0FBYyxFQUFFLFNBQVMsSUFBSSxLQUFLLENBQUMsQ0FBQztRQUN0RCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLENBQUM7SUFDdEIsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7T0FZRztJQUNJLFNBQVMsQ0FBQyxFQUFVLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDaEYsSUFBSSxHQUFHLEdBQUcsK0RBQStELENBQUM7UUFDMUUsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsT0FBTyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBQy9CLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFlBQVksRUFBRSxPQUFPLElBQUksR0FBRyxDQUFDLENBQUM7UUFDaEQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsV0FBVyxFQUFFLE1BQU0sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUM5QyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsU0FBUyxJQUFJLEtBQUssQ0FBQyxDQUFDO1FBQ3RELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN0QixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksVUFBVSxDQUFDLEVBQVUsRUFBRSxPQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUNqRixJQUFJLEdBQUcsR0FBRyxnRUFBZ0UsQ0FBQztRQUMzRSxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxPQUFPLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFDL0IsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsWUFBWSxFQUFFLE9BQU8sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUNoRCxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxXQUFXLEVBQUUsTUFBTSxJQUFJLEdBQUcsQ0FBQyxDQUFDO1FBQzlDLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLGNBQWMsRUFBRSxTQUFTLElBQUksS0FBSyxDQUFDLENBQUM7UUFDdEQsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLEdBQUcsQ0FBQyxDQUFDO0lBQ3RCLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7O09BWUc7SUFDSSxNQUFNLENBQUMsRUFBVSxFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQzdFLElBQUksR0FBRyxHQUFHLDREQUE0RCxDQUFDO1FBQ3ZFLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLE9BQU8sRUFBRSxFQUFFLENBQUMsQ0FBQztRQUMvQixHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxZQUFZLEVBQUUsT0FBTyxJQUFJLEdBQUcsQ0FBQyxDQUFDO1FBQ2hELEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFdBQVcsRUFBRSxNQUFNLElBQUksR0FBRyxDQUFDLENBQUM7UUFDOUMsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsY0FBYyxFQUFFLFNBQVMsSUFBSSxLQUFLLENBQUMsQ0FBQztRQUN0RCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLENBQUM7SUFDdEIsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7T0FZRztJQUNJLGNBQWMsQ0FBQyxFQUFVLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDckYsSUFBSSxHQUFHLEdBQUcsb0VBQW9FLENBQUM7UUFDL0UsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsT0FBTyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBQy9CLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFlBQVksRUFBRSxPQUFPLElBQUksR0FBRyxDQUFDLENBQUM7UUFDaEQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsV0FBVyxFQUFFLE1BQU0sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUM5QyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsU0FBUyxJQUFJLEtBQUssQ0FBQyxDQUFDO1FBQ3RELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN0QixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksY0FBYyxDQUFDLEVBQVUsRUFBRSxPQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUNyRixJQUFJLEdBQUcsR0FBRyxvRUFBb0UsQ0FBQztRQUMvRSxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxPQUFPLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFDL0IsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsWUFBWSxFQUFFLE9BQU8sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUNoRCxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxXQUFXLEVBQUUsTUFBTSxJQUFJLEdBQUcsQ0FBQyxDQUFDO1FBQzlDLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLGNBQWMsRUFBRSxTQUFTLElBQUksS0FBSyxDQUFDLENBQUM7UUFDdEQsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLEdBQUcsQ0FBQyxDQUFDO0lBQ3RCLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxXQUFXLENBQUMsRUFBVSxFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQ2xGLElBQUksR0FBRyxHQUFHLGlFQUFpRSxDQUFDO1FBQzVFLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLE9BQU8sRUFBRSxFQUFFLENBQUMsQ0FBQztRQUMvQixHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxZQUFZLEVBQUUsT0FBTyxJQUFJLEdBQUcsQ0FBQyxDQUFDO1FBQ2hELEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFdBQVcsRUFBRSxNQUFNLElBQUksR0FBRyxDQUFDLENBQUM7UUFDOUMsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsY0FBYyxFQUFFLFNBQVMsSUFBSSxLQUFLLENBQUMsQ0FBQztRQUN0RCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLENBQUM7SUFDdEIsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLDRCQUE0QixDQUFDLEVBQVUsRUFBRSxPQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUNuRyxJQUFJLEdBQUcsR0FBRyxrRkFBa0YsQ0FBQztRQUM3RixHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxPQUFPLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFDL0IsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsWUFBWSxFQUFFLE9BQU8sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUNoRCxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxXQUFXLEVBQUUsTUFBTSxJQUFJLEdBQUcsQ0FBQyxDQUFDO1FBQzlDLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLGNBQWMsRUFBRSxTQUFTLElBQUksS0FBSyxDQUFDLENBQUM7UUFDdEQsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLEdBQUcsQ0FBQyxDQUFDO0lBQ3RCLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxzQkFBc0IsQ0FBQyxFQUFVLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDN0YsSUFBSSxHQUFHLEdBQUcsNEVBQTRFLENBQUM7UUFDdkYsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsT0FBTyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBQy9CLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFlBQVksRUFBRSxPQUFPLElBQUksR0FBRyxDQUFDLENBQUM7UUFDaEQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsV0FBVyxFQUFFLE1BQU0sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUM5QyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsU0FBUyxJQUFJLEtBQUssQ0FBQyxDQUFDO1FBQ3RELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN0QixDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksMkJBQTJCLENBQUMsRUFBVSxFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQ2xHLElBQUksR0FBRyxHQUFHLGlGQUFpRixDQUFDO1FBQzVGLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLE9BQU8sRUFBRSxFQUFFLENBQUMsQ0FBQztRQUMvQixHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxZQUFZLEVBQUUsT0FBTyxJQUFJLEdBQUcsQ0FBQyxDQUFDO1FBQ2hELEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFdBQVcsRUFBRSxNQUFNLElBQUksR0FBRyxDQUFDLENBQUM7UUFDOUMsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsY0FBYyxFQUFFLFNBQVMsSUFBSSxLQUFLLENBQUMsQ0FBQztRQUN0RCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLENBQUM7SUFDdEIsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7T0FZRztJQUNJLG9CQUFvQixDQUFDLEVBQVUsRUFBRSxPQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUMzRixJQUFJLEdBQUcsR0FBRywwRUFBMEUsQ0FBQztRQUNyRixHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxPQUFPLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFDL0IsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsWUFBWSxFQUFFLE9BQU8sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUNoRCxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxXQUFXLEVBQUUsTUFBTSxJQUFJLEdBQUcsQ0FBQyxDQUFDO1FBQzlDLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLGNBQWMsRUFBRSxTQUFTLElBQUksS0FBSyxDQUFDLENBQUM7UUFDdEQsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLEdBQUcsQ0FBQyxDQUFDO0lBQ3RCLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7O09BWUc7SUFDSSw0QkFBNEIsQ0FBQyxFQUFVLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDbkcsSUFBSSxHQUFHLEdBQUcsa0ZBQWtGLENBQUM7UUFDN0YsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsT0FBTyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBQy9CLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFlBQVksRUFBRSxPQUFPLElBQUksR0FBRyxDQUFDLENBQUM7UUFDaEQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsV0FBVyxFQUFFLE1BQU0sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUM5QyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsU0FBUyxJQUFJLEtBQUssQ0FBQyxDQUFDO1FBQ3RELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN0QixDQUFDO0NBQ0Y7QUEzcUZELDhCQTJxRkMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBBY3Rpb25zLCBQb2xpY3lTdGF0ZW1lbnQsIFJlc291cmNlVHlwZXMgfSBmcm9tIFwiLi4vc2hhcmVkXCI7XG5cbi8qKlxuICogU3RhdGVtZW50IHByb3ZpZGVyIGZvciBzZXJ2aWNlIFtsaWdodHNhaWxdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9saXN0X2FtYXpvbmxpZ2h0c2FpbC5odG1sKS5cbiAqXG4gKiBAcGFyYW0gc2lkIFtTSURdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfc2lkLmh0bWwpIG9mIHRoZSBzdGF0ZW1lbnRcbiAqL1xuZXhwb3J0IGNsYXNzIExpZ2h0c2FpbCBleHRlbmRzIFBvbGljeVN0YXRlbWVudCB7XG4gIHB1YmxpYyBzZXJ2aWNlUHJlZml4ID0gJ2xpZ2h0c2FpbCc7XG4gIHByb3RlY3RlZCBhY3Rpb25MaXN0OiBBY3Rpb25zID0ge1xuICAgIFwiQWxsb2NhdGVTdGF0aWNJcFwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9saWdodHNhaWwvMjAxNi0xMS0yOC9hcGktcmVmZXJlbmNlL0FQSV9BbGxvY2F0ZVN0YXRpY0lwLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJDcmVhdGVzIGEgc3RhdGljIElQIGFkZHJlc3MgdGhhdCBjYW4gYmUgYXR0YWNoZWQgdG8gYW4gaW5zdGFuY2UuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiU3RhdGljSXBcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkF0dGFjaERpc2tcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGlnaHRzYWlsLzIwMTYtMTEtMjgvYXBpLXJlZmVyZW5jZS9BUElfQXR0YWNoRGlzay5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiQXR0YWNoZXMgYSBkaXNrIHRvIGFuIGluc3RhbmNlLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcIkRpc2tcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9LFxuICAgICAgICBcIkluc3RhbmNlXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJBdHRhY2hJbnN0YW5jZXNUb0xvYWRCYWxhbmNlclwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9saWdodHNhaWwvMjAxNi0xMS0yOC9hcGktcmVmZXJlbmNlL0FQSV9BdHRhY2hJbnN0YW5jZXNUb0xvYWRCYWxhbmNlci5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiQXR0YWNoZXMgb25lIG9yIG1vcmUgaW5zdGFuY2VzIHRvIGEgbG9hZCBiYWxhbmNlci5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJJbnN0YW5jZVwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH0sXG4gICAgICAgIFwiTG9hZEJhbGFuY2VyXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJBdHRhY2hMb2FkQmFsYW5jZXJUbHNDZXJ0aWZpY2F0ZVwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9saWdodHNhaWwvMjAxNi0xMS0yOC9hcGktcmVmZXJlbmNlL0FQSV9BdHRhY2hMb2FkQmFsYW5jZXJUbHNDZXJ0aWZpY2F0ZS5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiQXR0YWNoZXMgYSBUTFMgY2VydGlmaWNhdGUgdG8gYSBsb2FkIGJhbGFuY2VyLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcIkxvYWRCYWxhbmNlclwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiQXR0YWNoU3RhdGljSXBcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGlnaHRzYWlsLzIwMTYtMTEtMjgvYXBpLXJlZmVyZW5jZS9BUElfQXR0YWNoU3RhdGljSXAuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkF0dGFjaGVzIGEgc3RhdGljIElQIGFkZHJlc3MgdG8gYW4gaW5zdGFuY2UuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiSW5zdGFuY2VcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9LFxuICAgICAgICBcIlN0YXRpY0lwXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJDbG9zZUluc3RhbmNlUHVibGljUG9ydHNcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGlnaHRzYWlsLzIwMTYtMTEtMjgvYXBpLXJlZmVyZW5jZS9BUElfQ2xvc2VJbnN0YW5jZVB1YmxpY1BvcnRzLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJDbG9zZXMgYSBwdWJsaWMgcG9ydCBvZiBhbiBpbnN0YW5jZS5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJJbnN0YW5jZVwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiQ29weVNuYXBzaG90XCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xpZ2h0c2FpbC8yMDE2LTExLTI4L2FwaS1yZWZlcmVuY2UvQVBJX0NvcHlTbmFwc2hvdC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiQ29waWVzIGEgc25hcHNob3QgZnJvbSBvbmUgQVdTIFJlZ2lvbiB0byBhbm90aGVyIGluIEFtYXpvbiBMaWdodHNhaWwuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIlxuICAgIH0sXG4gICAgXCJDcmVhdGVDbG91ZEZvcm1hdGlvblN0YWNrXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xpZ2h0c2FpbC8yMDE2LTExLTI4L2FwaS1yZWZlcmVuY2UvQVBJX0NyZWF0ZUNsb3VkRm9ybWF0aW9uU3RhY2suaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkNyZWF0ZXMgYSBuZXcgQW1hem9uIEVDMiBpbnN0YW5jZSBmcm9tIGFuIGV4cG9ydGVkIEFtYXpvbiBMaWdodHNhaWwgc25hcHNob3QuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiRXhwb3J0U25hcHNob3RSZWNvcmRcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkNyZWF0ZURpc2tcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGlnaHRzYWlsLzIwMTYtMTEtMjgvYXBpLXJlZmVyZW5jZS9BUElfQ3JlYXRlRGlzay5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiQ3JlYXRlcyBhIGRpc2suXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiRGlza1wiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH0sXG4gICAgICBcImNvbmRpdGlvbnNcIjogW1xuICAgICAgICBcImF3czpSZXF1ZXN0VGFnLyR7VGFnS2V5fVwiLFxuICAgICAgICBcImF3czpUYWdLZXlzXCJcbiAgICAgIF1cbiAgICB9LFxuICAgIFwiQ3JlYXRlRGlza0Zyb21TbmFwc2hvdFwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9saWdodHNhaWwvMjAxNi0xMS0yOC9hcGktcmVmZXJlbmNlL0FQSV9DcmVhdGVEaXNrRnJvbVNuYXBzaG90Lmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJDcmVhdGVzIGEgZGlzayBmcm9tIHNuYXBzaG90LlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcIkRpc2tcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9LFxuICAgICAgXCJjb25kaXRpb25zXCI6IFtcbiAgICAgICAgXCJhd3M6UmVxdWVzdFRhZy8ke1RhZ0tleX1cIixcbiAgICAgICAgXCJhd3M6VGFnS2V5c1wiXG4gICAgICBdXG4gICAgfSxcbiAgICBcIkNyZWF0ZURpc2tTbmFwc2hvdFwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9saWdodHNhaWwvMjAxNi0xMS0yOC9hcGktcmVmZXJlbmNlL0FQSV9DcmVhdGVEaXNrU25hcHNob3QuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkNyZWF0ZXMgYSBkaXNrIHNuYXBzaG90LlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcIkRpc2tcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9LFxuICAgICAgXCJjb25kaXRpb25zXCI6IFtcbiAgICAgICAgXCJhd3M6UmVxdWVzdFRhZy8ke1RhZ0tleX1cIixcbiAgICAgICAgXCJhd3M6VGFnS2V5c1wiXG4gICAgICBdXG4gICAgfSxcbiAgICBcIkNyZWF0ZURvbWFpblwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9saWdodHNhaWwvMjAxNi0xMS0yOC9hcGktcmVmZXJlbmNlL0FQSV9DcmVhdGVEb21haW4uaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkNyZWF0ZXMgYSBkb21haW4gcmVzb3VyY2UgZm9yIHRoZSBzcGVjaWZpZWQgZG9tYWluIG5hbWUuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiRG9tYWluXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfSxcbiAgICAgIFwiY29uZGl0aW9uc1wiOiBbXG4gICAgICAgIFwiYXdzOlJlcXVlc3RUYWcvJHtUYWdLZXl9XCIsXG4gICAgICAgIFwiYXdzOlRhZ0tleXNcIlxuICAgICAgXVxuICAgIH0sXG4gICAgXCJDcmVhdGVEb21haW5FbnRyeVwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9saWdodHNhaWwvMjAxNi0xMS0yOC9hcGktcmVmZXJlbmNlL0FQSV9DcmVhdGVEb21haW5FbnRyeS5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiQ3JlYXRlcyBvbmUgb3IgbW9yZSBETlMgcmVjb3JkIGVudHJpZXMgZm9yIGEgZG9tYWluIHJlc291cmNlOiBBZGRyZXNzIChBKSwgY2Fub25pY2FsIG5hbWUgKENOQU1FKSwgbWFpbCBleGNoYW5nZXIgKE1YKSwgbmFtZSBzZXJ2ZXIgKE5TKSwgc3RhcnQgb2YgYXV0aG9yaXR5IChTT0EpLCBzZXJ2aWNlIGxvY2F0b3IgKFNSViksIG9yIHRleHQgKFRYVCkuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiRG9tYWluXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJDcmVhdGVJbnN0YW5jZVNuYXBzaG90XCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xpZ2h0c2FpbC8yMDE2LTExLTI4L2FwaS1yZWZlcmVuY2UvQVBJX0NyZWF0ZUluc3RhbmNlU25hcHNob3QuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkNyZWF0ZXMgYW4gaW5zdGFuY2Ugc25hcHNob3QuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiSW5zdGFuY2VcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9LFxuICAgICAgICBcIkluc3RhbmNlU25hcHNob3RcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9LFxuICAgICAgXCJjb25kaXRpb25zXCI6IFtcbiAgICAgICAgXCJhd3M6UmVxdWVzdFRhZy8ke1RhZ0tleX1cIixcbiAgICAgICAgXCJhd3M6VGFnS2V5c1wiXG4gICAgICBdXG4gICAgfSxcbiAgICBcIkNyZWF0ZUluc3RhbmNlc1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9saWdodHNhaWwvMjAxNi0xMS0yOC9hcGktcmVmZXJlbmNlL0FQSV9DcmVhdGVJbnN0YW5jZXMuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkNyZWF0ZXMgb25lIG9yIG1vcmUgaW5zdGFuY2VzLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcIktleVBhaXJcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9LFxuICAgICAgXCJjb25kaXRpb25zXCI6IFtcbiAgICAgICAgXCJhd3M6UmVxdWVzdFRhZy8ke1RhZ0tleX1cIixcbiAgICAgICAgXCJhd3M6VGFnS2V5c1wiXG4gICAgICBdXG4gICAgfSxcbiAgICBcIkNyZWF0ZUluc3RhbmNlc0Zyb21TbmFwc2hvdFwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9saWdodHNhaWwvMjAxNi0xMS0yOC9hcGktcmVmZXJlbmNlL0FQSV9DcmVhdGVJbnN0YW5jZXNGcm9tU25hcHNob3QuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkNyZWF0ZXMgb25lIG9yIG1vcmUgaW5zdGFuY2VzIGJhc2VkIG9uIGFuIGluc3RhbmNlIHNuYXBzaG90LlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcIkluc3RhbmNlXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfSxcbiAgICAgICAgXCJJbnN0YW5jZVNuYXBzaG90XCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfSxcbiAgICAgIFwiY29uZGl0aW9uc1wiOiBbXG4gICAgICAgIFwiYXdzOlJlcXVlc3RUYWcvJHtUYWdLZXl9XCIsXG4gICAgICAgIFwiYXdzOlRhZ0tleXNcIlxuICAgICAgXVxuICAgIH0sXG4gICAgXCJDcmVhdGVLZXlQYWlyXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xpZ2h0c2FpbC8yMDE2LTExLTI4L2FwaS1yZWZlcmVuY2UvQVBJX0NyZWF0ZUtleVBhaXIuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkNyZWF0ZXMgYSBrZXkgcGFpciB1c2VkIHRvIGF1dGhlbnRpY2F0ZSBhbmQgY29ubmVjdCB0byBhbiBpbnN0YW5jZS5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJLZXlQYWlyXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfSxcbiAgICAgIFwiY29uZGl0aW9uc1wiOiBbXG4gICAgICAgIFwiYXdzOlJlcXVlc3RUYWcvJHtUYWdLZXl9XCIsXG4gICAgICAgIFwiYXdzOlRhZ0tleXNcIlxuICAgICAgXVxuICAgIH0sXG4gICAgXCJDcmVhdGVMb2FkQmFsYW5jZXJcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGlnaHRzYWlsLzIwMTYtMTEtMjgvYXBpLXJlZmVyZW5jZS9BUElfQ3JlYXRlTG9hZEJhbGFuY2VyLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJDcmVhdGVzIGEgbG9hZCBiYWxhbmNlci5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJMb2FkQmFsYW5jZXJcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9LFxuICAgICAgXCJjb25kaXRpb25zXCI6IFtcbiAgICAgICAgXCJhd3M6UmVxdWVzdFRhZy8ke1RhZ0tleX1cIixcbiAgICAgICAgXCJhd3M6VGFnS2V5c1wiXG4gICAgICBdXG4gICAgfSxcbiAgICBcIkNyZWF0ZUxvYWRCYWxhbmNlclRsc0NlcnRpZmljYXRlXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xpZ2h0c2FpbC8yMDE2LTExLTI4L2FwaS1yZWZlcmVuY2UvQVBJX0NyZWF0ZUxvYWRCYWxhbmNlclRsc0NlcnRpZmljYXRlLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJDcmVhdGVzIGEgbG9hZCBiYWxhbmNlciBUTFMgY2VydGlmaWNhdGUuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiTG9hZEJhbGFuY2VyXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJDcmVhdGVSZWxhdGlvbmFsRGF0YWJhc2VcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGlnaHRzYWlsLzIwMTYtMTEtMjgvYXBpLXJlZmVyZW5jZS9BUElfQ3JlYXRlUmVsYXRpb25hbERhdGFiYXNlLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJDcmVhdGVzIGEgbmV3IHJlbGF0aW9uYWwgZGF0YWJhc2UuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiUmVsYXRpb25hbERhdGFiYXNlXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfSxcbiAgICAgIFwiY29uZGl0aW9uc1wiOiBbXG4gICAgICAgIFwiYXdzOlJlcXVlc3RUYWcvJHtUYWdLZXl9XCIsXG4gICAgICAgIFwiYXdzOlRhZ0tleXNcIlxuICAgICAgXVxuICAgIH0sXG4gICAgXCJDcmVhdGVSZWxhdGlvbmFsRGF0YWJhc2VGcm9tU25hcHNob3RcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGlnaHRzYWlsLzIwMTYtMTEtMjgvYXBpLXJlZmVyZW5jZS9BUElfQ3JlYXRlUmVsYXRpb25hbERhdGFiYXNlRnJvbVNuYXBzaG90Lmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJDcmVhdGVzIGEgbmV3IHJlbGF0aW9uYWwgZGF0YWJhc2UgZnJvbSBhIHNuYXBzaG90LlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcIlJlbGF0aW9uYWxEYXRhYmFzZVwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH0sXG4gICAgICBcImNvbmRpdGlvbnNcIjogW1xuICAgICAgICBcImF3czpSZXF1ZXN0VGFnLyR7VGFnS2V5fVwiLFxuICAgICAgICBcImF3czpUYWdLZXlzXCJcbiAgICAgIF1cbiAgICB9LFxuICAgIFwiQ3JlYXRlUmVsYXRpb25hbERhdGFiYXNlU25hcHNob3RcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGlnaHRzYWlsLzIwMTYtMTEtMjgvYXBpLXJlZmVyZW5jZS9BUElfQ3JlYXRlUmVsYXRpb25hbERhdGFiYXNlU25hcHNob3QuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkNyZWF0ZXMgYSByZWxhdGlvbmFsIGRhdGFiYXNlIHNuYXBzaG90LlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcIlJlbGF0aW9uYWxEYXRhYmFzZVNuYXBzaG90XCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfSxcbiAgICAgIFwiY29uZGl0aW9uc1wiOiBbXG4gICAgICAgIFwiYXdzOlJlcXVlc3RUYWcvJHtUYWdLZXl9XCIsXG4gICAgICAgIFwiYXdzOlRhZ0tleXNcIlxuICAgICAgXVxuICAgIH0sXG4gICAgXCJEZWxldGVEaXNrXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xpZ2h0c2FpbC8yMDE2LTExLTI4L2FwaS1yZWZlcmVuY2UvQVBJX0RlbGV0ZURpc2suaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkRlbGV0ZXMgYSBkaXNrLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcIkRpc2tcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkRlbGV0ZURpc2tTbmFwc2hvdFwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9saWdodHNhaWwvMjAxNi0xMS0yOC9hcGktcmVmZXJlbmNlL0FQSV9EZWxldGVEaXNrU25hcHNob3QuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkRlbGV0ZXMgYSBkaXNrIHNuYXBzaG90LlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcIkRpc2tcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkRlbGV0ZURvbWFpblwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9saWdodHNhaWwvMjAxNi0xMS0yOC9hcGktcmVmZXJlbmNlL0FQSV9EZWxldGVEb21haW4uaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkRlbGV0ZXMgYSBkb21haW4gcmVzb3VyY2UgYW5kIGFsbCBvZiBpdHMgRE5TIHJlY29yZHMuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiRG9tYWluXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJEZWxldGVEb21haW5FbnRyeVwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9saWdodHNhaWwvMjAxNi0xMS0yOC9hcGktcmVmZXJlbmNlL0FQSV9EZWxldGVEb21haW5FbnRyeS5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiRGVsZXRlcyBhIEROUyByZWNvcmQgZW50cnkgZm9yIGEgZG9tYWluIHJlc291cmNlLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcIkRvbWFpblwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiRGVsZXRlSW5zdGFuY2VcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGlnaHRzYWlsLzIwMTYtMTEtMjgvYXBpLXJlZmVyZW5jZS9BUElfRGVsZXRlSW5zdGFuY2UuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkRlbGV0ZXMgYW4gaW5zdGFuY2UuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiSW5zdGFuY2VcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkRlbGV0ZUluc3RhbmNlU25hcHNob3RcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGlnaHRzYWlsLzIwMTYtMTEtMjgvYXBpLXJlZmVyZW5jZS9BUElfRGVsZXRlSW5zdGFuY2VTbmFwc2hvdC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiRGVsZXRlcyBhbiBpbnN0YW5jZSBzbmFwc2hvdC5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJJbnN0YW5jZVNuYXBzaG90XCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJEZWxldGVLZXlQYWlyXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xpZ2h0c2FpbC8yMDE2LTExLTI4L2FwaS1yZWZlcmVuY2UvQVBJX0RlbGV0ZUtleVBhaXIuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkRlbGV0ZXMgYSBrZXkgcGFpciB1c2VkIHRvIGF1dGhlbnRpY2F0ZSBhbmQgY29ubmVjdCB0byBhbiBpbnN0YW5jZS5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJLZXlQYWlyXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJEZWxldGVLbm93bkhvc3RLZXlzXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xpZ2h0c2FpbC8yMDE2LTExLTI4L2FwaS1yZWZlcmVuY2UvQVBJX0RlbGV0ZUtub3duSG9zdEtleXMuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkRlbGV0ZXMgdGhlIGtub3duIGhvc3Qga2V5IG9yIGNlcnRpZmljYXRlIHVzZWQgYnkgdGhlIEFtYXpvbiBMaWdodHNhaWwgYnJvd3Nlci1iYXNlZCBTU0ggb3IgUkRQIGNsaWVudHMgdG8gYXV0aGVudGljYXRlIGFuIGluc3RhbmNlLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcIkluc3RhbmNlXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJEZWxldGVMb2FkQmFsYW5jZXJcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGlnaHRzYWlsLzIwMTYtMTEtMjgvYXBpLXJlZmVyZW5jZS9BUElfRGVsZXRlTG9hZEJhbGFuY2VyLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJEZWxldGVzIGEgbG9hZCBiYWxhbmNlci5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJMb2FkQmFsYW5jZXJcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkRlbGV0ZUxvYWRCYWxhbmNlclRsc0NlcnRpZmljYXRlXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xpZ2h0c2FpbC8yMDE2LTExLTI4L2FwaS1yZWZlcmVuY2UvQVBJX0RlbGV0ZUxvYWRCYWxhbmNlclRsc0NlcnRpZmljYXRlLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJEZWxldGVzIGEgbG9hZCBiYWxhbmNlciBUTFMgY2VydGlmaWNhdGUuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiTG9hZEJhbGFuY2VyXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJEZWxldGVSZWxhdGlvbmFsRGF0YWJhc2VcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGlnaHRzYWlsLzIwMTYtMTEtMjgvYXBpLXJlZmVyZW5jZS9BUElfRGVsZXRlUmVsYXRpb25hbERhdGFiYXNlLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJEZWxldGVzIGEgcmVsYXRpb25hbCBkYXRhYmFzZS5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJSZWxhdGlvbmFsRGF0YWJhc2VcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkRlbGV0ZVJlbGF0aW9uYWxEYXRhYmFzZVNuYXBzaG90XCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xpZ2h0c2FpbC8yMDE2LTExLTI4L2FwaS1yZWZlcmVuY2UvQVBJX0RlbGV0ZVJlbGF0aW9uYWxEYXRhYmFzZVNuYXBzaG90Lmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJEZWxldGVzIHJlbGF0aW9uYWwgZGF0YWJhc2Ugc25hcHNob3QuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiUmVsYXRpb25hbERhdGFiYXNlU25hcHNob3RcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkRldGFjaERpc2tcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGlnaHRzYWlsLzIwMTYtMTEtMjgvYXBpLXJlZmVyZW5jZS9BUElfRGV0YWNoRGlzay5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiRGV0YWNoZXMgYSBkaXNrIGZyb20gYW4gaW5zdGFuY2UuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiRGlza1wiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiRGV0YWNoSW5zdGFuY2VzRnJvbUxvYWRCYWxhbmNlclwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9saWdodHNhaWwvMjAxNi0xMS0yOC9hcGktcmVmZXJlbmNlL0FQSV9EZXRhY2hJbnN0YW5jZXNGcm9tTG9hZEJhbGFuY2VyLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJEZXRhY2hlcyBvbmUgb3IgbW9yZSBpbnN0YW5jZXMgZnJvbSBhIGxvYWQgYmFsYW5jZXIuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiSW5zdGFuY2VcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9LFxuICAgICAgICBcIkxvYWRCYWxhbmNlclwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiRGV0YWNoU3RhdGljSXBcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGlnaHRzYWlsLzIwMTYtMTEtMjgvYXBpLXJlZmVyZW5jZS9BUElfRGV0YWNoU3RhdGljSXAuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkRldGFjaGVzIGEgc3RhdGljIElQIGZyb20gYW4gaW5zdGFuY2UgdG8gd2hpY2ggaXQgaXMgYXR0YWNoZWQuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiSW5zdGFuY2VcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9LFxuICAgICAgICBcIlN0YXRpY0lwXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJEb3dubG9hZERlZmF1bHRLZXlQYWlyXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xpZ2h0c2FpbC8yMDE2LTExLTI4L2FwaS1yZWZlcmVuY2UvQVBJX0Rvd25sb2FkRGVmYXVsdEtleVBhaXIuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkRvd25sb2FkcyB0aGUgZGVmYXVsdCBrZXkgcGFpciB1c2VkIHRvIGF1dGhlbnRpY2F0ZSBhbmQgY29ubmVjdCB0byBpbnN0YW5jZXMgaW4gYSBzcGVjaWZpYyBBV1MgUmVnaW9uLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcIktleVBhaXJcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkV4cG9ydFNuYXBzaG90XCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xpZ2h0c2FpbC8yMDE2LTExLTI4L2FwaS1yZWZlcmVuY2UvQVBJX0V4cG9ydFNuYXBzaG90Lmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJFeHBvcnRzIGFuIEFtYXpvbiBMaWdodHNhaWwgc25hcHNob3QgdG8gQW1hem9uIEVDMi5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiXG4gICAgfSxcbiAgICBcIkdldEFjdGl2ZU5hbWVzXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xpZ2h0c2FpbC8yMDE2LTExLTI4L2FwaS1yZWZlcmVuY2UvQVBJX0dldEFjdGl2ZU5hbWVzLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJSZXR1cm5zIHRoZSBuYW1lcyBvZiBhbGwgYWN0aXZlIChub3QgZGVsZXRlZCkgcmVzb3VyY2VzLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlJlYWRcIlxuICAgIH0sXG4gICAgXCJHZXRCbHVlcHJpbnRzXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xpZ2h0c2FpbC8yMDE2LTExLTI4L2FwaS1yZWZlcmVuY2UvQVBJX0dldEJsdWVwcmludHMuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlJldHVybnMgYSBsaXN0IG9mIGluc3RhbmNlIGltYWdlcywgb3IgYmx1ZXByaW50cy4gWW91IGNhbiB1c2UgYSBibHVlcHJpbnQgdG8gY3JlYXRlIGEgbmV3IGluc3RhbmNlIGFscmVhZHkgcnVubmluZyBhIHNwZWNpZmljIG9wZXJhdGluZyBzeXN0ZW0sIGFzIHdlbGwgYXMgYSBwcmUtaW5zdGFsbGVkIGFwcGxpY2F0aW9uIG9yIGRldmVsb3BtZW50IHN0YWNrLiBUaGUgc29mdHdhcmUgdGhhdCBydW5zIG9uIHlvdXIgaW5zdGFuY2UgZGVwZW5kcyBvbiB0aGUgYmx1ZXByaW50IHlvdSBkZWZpbmUgd2hlbiBjcmVhdGluZyB0aGUgaW5zdGFuY2UuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiTGlzdFwiXG4gICAgfSxcbiAgICBcIkdldEJ1bmRsZXNcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGlnaHRzYWlsLzIwMTYtMTEtMjgvYXBpLXJlZmVyZW5jZS9BUElfR2V0QnVuZGxlcy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiUmV0dXJucyBhIGxpc3Qgb2YgaW5zdGFuY2UgYnVuZGxlcy4gWW91IGNhbiB1c2UgYSBidW5kbGUgdG8gY3JlYXRlIGEgbmV3IGluc3RhbmNlIHdpdGggYSBzZXQgb2YgcGVyZm9ybWFuY2Ugc3BlY2lmaWNhdGlvbnMsIHN1Y2ggYXMgQ1BVIGNvdW50LCBkaXNrIHNpemUsIFJBTSBzaXplLCBhbmQgbmV0d29yayB0cmFuc2ZlciBhbGxvd2FuY2UuIFRoZSBjb3N0IG9mIHlvdXIgaW5zdGFuY2UgZGVwZW5kcyBvbiB0aGUgYnVuZGxlIHlvdSBkZWZpbmUgd2hlbiBjcmVhdGluZyB0aGUgaW5zdGFuY2UuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiTGlzdFwiXG4gICAgfSxcbiAgICBcIkdldENsb3VkRm9ybWF0aW9uU3RhY2tSZWNvcmRzXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xpZ2h0c2FpbC8yMDE2LTExLTI4L2FwaS1yZWZlcmVuY2UvQVBJX0dldENsb3VkRm9ybWF0aW9uU3RhY2tSZWNvcmRzLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJSZXR1cm5zIGluZm9ybWF0aW9uIGFib3V0IGFsbCBDbG91ZEZvcm1hdGlvbiBzdGFja3MgdXNlZCB0byBjcmVhdGUgQW1hem9uIEVDMiByZXNvdXJjZXMgZnJvbSBleHBvcnRlZCBBbWF6b24gTGlnaHRzYWlsIHNuYXBzaG90cy5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJMaXN0XCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcIkNsb3VkRm9ybWF0aW9uU3RhY2tSZWNvcmRcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkdldERpc2tcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGlnaHRzYWlsLzIwMTYtMTEtMjgvYXBpLXJlZmVyZW5jZS9BUElfR2V0RGlzay5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiUmV0dXJucyBpbmZvcm1hdGlvbiBhYm91dCBhIGRpc2suXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUmVhZFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJEaXNrXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJHZXREaXNrU25hcHNob3RcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGlnaHRzYWlsLzIwMTYtMTEtMjgvYXBpLXJlZmVyZW5jZS9BUElfR2V0RGlza1NuYXBzaG90Lmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJSZXR1cm5zIGluZm9ybWF0aW9uIGFib3V0IGEgZGlzayBzbmFwc2hvdC5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJSZWFkXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcIkRpc2tcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkdldERpc2tTbmFwc2hvdHNcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGlnaHRzYWlsLzIwMTYtMTEtMjgvYXBpLXJlZmVyZW5jZS9BUElfR2V0RGlza1NuYXBzaG90cy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiUmV0dXJucyBpbmZvcm1hdGlvbiBhYm91dCBhbGwgZGlzayBzbmFwc2hvdHMuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiTGlzdFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJEaXNrXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJHZXREaXNrc1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9saWdodHNhaWwvMjAxNi0xMS0yOC9hcGktcmVmZXJlbmNlL0FQSV9HZXREaXNrcy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiUmV0dXJucyBpbmZvcm1hdGlvbiBhYm91dCBhbGwgZGlza3MuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiTGlzdFwiXG4gICAgfSxcbiAgICBcIkdldERvbWFpblwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9saWdodHNhaWwvMjAxNi0xMS0yOC9hcGktcmVmZXJlbmNlL0FQSV9HZXREb21haW4uaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlJldHVybnMgRE5TIHJlY29yZHMgZm9yIGEgZG9tYWluIHJlc291cmNlLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlJlYWRcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiRG9tYWluXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJHZXREb21haW5zXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xpZ2h0c2FpbC8yMDE2LTExLTI4L2FwaS1yZWZlcmVuY2UvQVBJX0dldERvbWFpbnMuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlJldHVybnMgRE5TIHJlY29yZHMgZm9yIGFsbCBkb21haW4gcmVzb3VyY2VzLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlJlYWRcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiRG9tYWluXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJHZXRFeHBvcnRTbmFwc2hvdFJlY29yZHNcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGlnaHRzYWlsLzIwMTYtMTEtMjgvYXBpLXJlZmVyZW5jZS9BUElfR2V0RXhwb3J0U25hcHNob3RSZWNvcmRzLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJSZXR1cm5zIGluZm9ybWF0aW9uIGFib3V0IGFsbCByZWNvcmRzIHRvIGV4cG9ydCBBbWF6b24gTGlnaHRzYWlsIHNuYXBzaG90cyB0byBBbWF6b24gRUMyLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIkxpc3RcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiRXhwb3J0U25hcHNob3RSZWNvcmRcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkdldEluc3RhbmNlXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xpZ2h0c2FpbC8yMDE2LTExLTI4L2FwaS1yZWZlcmVuY2UvQVBJX0dldEluc3RhbmNlLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJSZXR1cm5zIGluZm9ybWF0aW9uIGFib3V0IGFuIGluc3RhbmNlLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlJlYWRcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiSW5zdGFuY2VcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkdldEluc3RhbmNlQWNjZXNzRGV0YWlsc1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9saWdodHNhaWwvMjAxNi0xMS0yOC9hcGktcmVmZXJlbmNlL0FQSV9HZXRJbnN0YW5jZUFjY2Vzc0RldGFpbHMuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlJldHVybnMgdGVtcG9yYXJ5IGtleXMgeW91IGNhbiB1c2UgdG8gYXV0aGVudGljYXRlIGFuZCBjb25uZWN0IHRvIGFuIGluc3RhbmNlLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcIkluc3RhbmNlXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJHZXRJbnN0YW5jZU1ldHJpY0RhdGFcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGlnaHRzYWlsLzIwMTYtMTEtMjgvYXBpLXJlZmVyZW5jZS9BUElfR2V0SW5zdGFuY2VNZXRyaWNEYXRhLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJSZXR1cm5zIHRoZSBkYXRhIHBvaW50cyBmb3IgdGhlIHNwZWNpZmllZCBtZXRyaWMgb2YgYW4gaW5zdGFuY2UuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUmVhZFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJJbnN0YW5jZVwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiR2V0SW5zdGFuY2VQb3J0U3RhdGVzXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xpZ2h0c2FpbC8yMDE2LTExLTI4L2FwaS1yZWZlcmVuY2UvQVBJX0dldEluc3RhbmNlUG9ydFN0YXRlcy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiUmV0dXJucyB0aGUgcG9ydCBzdGF0ZXMgb2YgYW4gaW5zdGFuY2UuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUmVhZFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJJbnN0YW5jZVwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiR2V0SW5zdGFuY2VTbmFwc2hvdFwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9saWdodHNhaWwvMjAxNi0xMS0yOC9hcGktcmVmZXJlbmNlL0FQSV9HZXRJbnN0YW5jZVNuYXBzaG90Lmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJSZXR1cm5zIGluZm9ybWF0aW9uIGFib3V0IGFuIGluc3RhbmNlIHNuYXBzaG90LlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlJlYWRcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiSW5zdGFuY2VTbmFwc2hvdFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiR2V0SW5zdGFuY2VTbmFwc2hvdHNcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGlnaHRzYWlsLzIwMTYtMTEtMjgvYXBpLXJlZmVyZW5jZS9BUElfR2V0SW5zdGFuY2VTbmFwc2hvdHMuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlJldHVybnMgaW5mb3JtYXRpb24gYWJvdXQgYWxsIGluc3RhbmNlIHNuYXBzaG90cy5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJMaXN0XCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcIkluc3RhbmNlU25hcHNob3RcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkdldEluc3RhbmNlU3RhdGVcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGlnaHRzYWlsLzIwMTYtMTEtMjgvYXBpLXJlZmVyZW5jZS9BUElfR2V0SW5zdGFuY2VTdGF0ZS5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiUmV0dXJucyB0aGUgc3RhdGUgb2YgYW4gaW5zdGFuY2UuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUmVhZFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJJbnN0YW5jZVwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiR2V0SW5zdGFuY2VzXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xpZ2h0c2FpbC8yMDE2LTExLTI4L2FwaS1yZWZlcmVuY2UvQVBJX0dldEluc3RhbmNlcy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiUmV0dXJucyBpbmZvcm1hdGlvbiBhYm91dCBhbGwgaW5zdGFuY2VzLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlJlYWRcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiSW5zdGFuY2VcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkdldEtleVBhaXJcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGlnaHRzYWlsLzIwMTYtMTEtMjgvYXBpLXJlZmVyZW5jZS9BUElfR2V0S2V5UGFpci5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiUmV0dXJucyBpbmZvcm1hdGlvbiBhYm91dCBhIGtleSBwYWlyLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIkxpc3RcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiS2V5UGFpclwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiR2V0S2V5UGFpcnNcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGlnaHRzYWlsLzIwMTYtMTEtMjgvYXBpLXJlZmVyZW5jZS9BUElfR2V0S2V5UGFpcnMuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlJldHVybnMgaW5mb3JtYXRpb24gYWJvdXQgYWxsIGtleSBwYWlycy5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJSZWFkXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcIktleVBhaXJcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkdldExvYWRCYWxhbmNlclwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9saWdodHNhaWwvMjAxNi0xMS0yOC9hcGktcmVmZXJlbmNlL0FQSV9HZXRMb2FkQmFsYW5jZXIuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlJldHVybnMgaW5mb3JtYXRpb24gYWJvdXQgYSBsb2FkIGJhbGFuY2VyLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlJlYWRcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiTG9hZEJhbGFuY2VyXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJHZXRMb2FkQmFsYW5jZXJNZXRyaWNEYXRhXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xpZ2h0c2FpbC8yMDE2LTExLTI4L2FwaS1yZWZlcmVuY2UvQVBJX0dldExvYWRCYWxhbmNlck1ldHJpY0RhdGEuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlJldHVybnMgdGhlIGRhdGEgcG9pbnRzIGZvciB0aGUgc3BlY2lmaWVkIG1ldHJpYyBvZiBhIGxvYWQgYmFsYW5jZXIuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUmVhZFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJMb2FkQmFsYW5jZXJcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkdldExvYWRCYWxhbmNlclRsc0NlcnRpZmljYXRlc1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9saWdodHNhaWwvMjAxNi0xMS0yOC9hcGktcmVmZXJlbmNlL0FQSV9EZWxldGVMb2FkQmFsYW5jZXJUbHNDZXJ0aWZpY2F0ZS5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiUmV0dXJucyBpbmZvcm1hdGlvbiBhYm91dCBhIGxvYWQgYmFsYW5jZXIgVExTIGNlcnRpZmljYXRlLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlJlYWRcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiTG9hZEJhbGFuY2VyXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJHZXRMb2FkQmFsYW5jZXJzXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xpZ2h0c2FpbC8yMDE2LTExLTI4L2FwaS1yZWZlcmVuY2UvQVBJX0dldExvYWRCYWxhbmNlcnMuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlJldHVybnMgaW5mb3JtYXRpb24gYWJvdXQgbG9hZCBiYWxhbmNlcnMuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUmVhZFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJMb2FkQmFsYW5jZXJcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkdldE9wZXJhdGlvblwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9saWdodHNhaWwvMjAxNi0xMS0yOC9hcGktcmVmZXJlbmNlL0FQSV9HZXRPcGVyYXRpb24uaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlJldHVybnMgaW5mb3JtYXRpb24gYWJvdXQgYW4gb3BlcmF0aW9uLiBPcGVyYXRpb25zIGluY2x1ZGUgZXZlbnRzIHN1Y2ggYXMgd2hlbiB5b3UgY3JlYXRlIGFuIGluc3RhbmNlLCBhbGxvY2F0ZSBhIHN0YXRpYyBJUCwgYXR0YWNoIGEgc3RhdGljIElQLCBhbmQgc28gb24uXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUmVhZFwiXG4gICAgfSxcbiAgICBcIkdldE9wZXJhdGlvbnNcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGlnaHRzYWlsLzIwMTYtMTEtMjgvYXBpLXJlZmVyZW5jZS9BUElfR2V0T3BlcmF0aW9ucy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiUmV0dXJucyBpbmZvcm1hdGlvbiBhYm91dCBhbGwgb3BlcmF0aW9ucy4gT3BlcmF0aW9ucyBpbmNsdWRlIGV2ZW50cyBzdWNoIGFzIHdoZW4geW91IGNyZWF0ZSBhbiBpbnN0YW5jZSwgYWxsb2NhdGUgYSBzdGF0aWMgSVAsIGF0dGFjaCBhIHN0YXRpYyBJUCwgYW5kIHNvIG9uLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlJlYWRcIlxuICAgIH0sXG4gICAgXCJHZXRPcGVyYXRpb25zRm9yUmVzb3VyY2VcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGlnaHRzYWlsLzIwMTYtMTEtMjgvYXBpLXJlZmVyZW5jZS9BUElfR2V0T3BlcmF0aW9uc0ZvclJlc291cmNlLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJSZXR1cm5zIG9wZXJhdGlvbnMgZm9yIGEgcmVzb3VyY2UuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUmVhZFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJEb21haW5cIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfSxcbiAgICAgICAgXCJJbnN0YW5jZVwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9LFxuICAgICAgICBcIkluc3RhbmNlU25hcHNob3RcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfSxcbiAgICAgICAgXCJLZXlQYWlyXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH0sXG4gICAgICAgIFwiU3RhdGljSXBcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJHZXRSZWdpb25zXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xpZ2h0c2FpbC8yMDE2LTExLTI4L2FwaS1yZWZlcmVuY2UvQVBJX0dldFJlZ2lvbnMuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlJldHVybnMgYSBsaXN0IG9mIGFsbCB2YWxpZCBBV1MgUmVnaW9ucyBmb3IgQW1hem9uIExpZ2h0c2FpbC5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJMaXN0XCJcbiAgICB9LFxuICAgIFwiR2V0UmVsYXRpb25hbERhdGFiYXNlXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xpZ2h0c2FpbC8yMDE2LTExLTI4L2FwaS1yZWZlcmVuY2UvQVBJX0dldFJlbGF0aW9uYWxEYXRhYmFzZS5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiUmV0dXJucyBpbmZvcm1hdGlvbiBhYm91dCBhIHJlbGF0aW9uYWwgZGF0YWJhc2UuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiTGlzdFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJSZWxhdGlvbmFsRGF0YWJhc2VcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkdldFJlbGF0aW9uYWxEYXRhYmFzZUJsdWVwcmludHNcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGlnaHRzYWlsLzIwMTYtMTEtMjgvYXBpLXJlZmVyZW5jZS9BUElfR2V0UmVsYXRpb25hbERhdGFiYXNlQmx1ZXByaW50cy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiUmV0dXJucyBhIGxpc3Qgb2YgcmVsYXRpb25hbCBkYXRhYmFzZSBpbWFnZXMsIG9yIGJsdWVwcmludHMuIFlvdSBjYW4gdXNlIGEgYmx1ZXByaW50IHRvIGNyZWF0ZSBhIG5ldyBkYXRhYmFzZSBydW5uaW5nIGEgc3BlY2lmaWMgZGF0YWJhc2UgZW5naW5lLiBUaGUgZGF0YWJhc2UgZW5naW5lIHRoYXQgcnVucyBvbiB5b3VyIGRhdGFiYXNlIGRlcGVuZHMgb24gdGhlIGJsdWVwcmludCB5b3UgZGVmaW5lIHdoZW4gY3JlYXRpbmcgdGhlIHJlbGF0aW9uYWwgZGF0YWJhc2UuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiTGlzdFwiXG4gICAgfSxcbiAgICBcIkdldFJlbGF0aW9uYWxEYXRhYmFzZUJ1bmRsZXNcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGlnaHRzYWlsLzIwMTYtMTEtMjgvYXBpLXJlZmVyZW5jZS9BUElfR2V0UmVsYXRpb25hbERhdGFiYXNlQnVuZGxlcy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiUmV0dXJucyBhIGxpc3Qgb2YgcmVsYXRpb25hbCBkYXRhYmFzZSBidW5kbGVzLiBZb3UgY2FuIHVzZSBhIGJ1bmRsZSB0byBjcmVhdGUgYSBuZXcgZGF0YWJhc2Ugd2l0aCBhIHNldCBvZiBwZXJmb3JtYW5jZSBzcGVjaWZpY2F0aW9ucywgc3VjaCBhcyBDUFUgY291bnQsIGRpc2sgc2l6ZSwgUkFNIHNpemUsIG5ldHdvcmsgdHJhbnNmZXIgYWxsb3dhbmNlLCBhbmQgc3RhbmRhcmQgb2YgaGlnaCBhdmFpbGFiaWxpdHkuIFRoZSBjb3N0IG9mIHlvdXIgZGF0YWJhc2UgZGVwZW5kcyBvbiB0aGUgYnVuZGxlIHlvdSBkZWZpbmUgd2hlbiBjcmVhdGluZyB0aGUgcmVsYXRpb25hbCBkYXRhYmFzZS5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJMaXN0XCJcbiAgICB9LFxuICAgIFwiR2V0UmVsYXRpb25hbERhdGFiYXNlRXZlbnRzXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xpZ2h0c2FpbC8yMDE2LTExLTI4L2FwaS1yZWZlcmVuY2UvQVBJX0dldFJlbGF0aW9uYWxEYXRhYmFzZUV2ZW50cy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiUmV0dXJucyBldmVudHMgZm9yIGEgcmVsYXRpb25hbCBkYXRhYmFzZS5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJSZWFkXCJcbiAgICB9LFxuICAgIFwiR2V0UmVsYXRpb25hbERhdGFiYXNlTG9nRXZlbnRzXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xpZ2h0c2FpbC8yMDE2LTExLTI4L2FwaS1yZWZlcmVuY2UvQVBJX0dldFJlbGF0aW9uYWxEYXRhYmFzZUxvZ0V2ZW50cy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiUmV0dXJucyBldmVudHMgZm9yIHRoZSBzcGVjaWZpZWQgbG9nIHN0cmVhbSBvZiBhIHJlbGF0aW9uYWwgZGF0YWJhc2UuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUmVhZFwiXG4gICAgfSxcbiAgICBcIkdldFJlbGF0aW9uYWxEYXRhYmFzZUxvZ1N0cmVhbXNcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGlnaHRzYWlsLzIwMTYtMTEtMjgvYXBpLXJlZmVyZW5jZS9BUElfR2V0UmVsYXRpb25hbERhdGFiYXNlTG9nU3RyZWFtcy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiUmV0dXJucyB0aGUgbG9nIHN0cmVhbXMgYXZhaWxhYmxlIGZvciBhIHJlbGF0aW9uYWwgZGF0YWJhc2UuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUmVhZFwiXG4gICAgfSxcbiAgICBcIkdldFJlbGF0aW9uYWxEYXRhYmFzZU1hc3RlclVzZXJQYXNzd29yZFwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9saWdodHNhaWwvMjAxNi0xMS0yOC9hcGktcmVmZXJlbmNlL0FQSV9HZXRSZWxhdGlvbmFsRGF0YWJhc2VNYXN0ZXJVc2VyUGFzc3dvcmQuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlJldHVybnMgdGhlIG1hc3RlciB1c2VyIHBhc3N3b3JkIG9mIGEgcmVsYXRpb25hbCBkYXRhYmFzZS5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiXG4gICAgfSxcbiAgICBcIkdldFJlbGF0aW9uYWxEYXRhYmFzZU1ldHJpY0RhdGFcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGlnaHRzYWlsLzIwMTYtMTEtMjgvYXBpLXJlZmVyZW5jZS9BUElfR2V0UmVsYXRpb25hbERhdGFiYXNlTWV0cmljRGF0YS5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiUmV0dXJucyB0aGUgZGF0YSBwb2ludHMgZm9yIHRoZSBzcGVjaWZpZWQgbWV0cmljIG9mIGEgcmVsYXRpb25hbCBkYXRhYmFzZS5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJSZWFkXCJcbiAgICB9LFxuICAgIFwiR2V0UmVsYXRpb25hbERhdGFiYXNlUGFyYW1ldGVyc1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9saWdodHNhaWwvMjAxNi0xMS0yOC9hcGktcmVmZXJlbmNlL0FQSV9HZXRSZWxhdGlvbmFsRGF0YWJhc2VQYXJhbWV0ZXJzLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJSZXR1cm5zIHRoZSBwYXJhbWV0ZXJzIG9mIGEgcmVsYXRpb25hbCBkYXRhYmFzZS5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJMaXN0XCJcbiAgICB9LFxuICAgIFwiR2V0UmVsYXRpb25hbERhdGFiYXNlU25hcHNob3RcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGlnaHRzYWlsLzIwMTYtMTEtMjgvYXBpLXJlZmVyZW5jZS9BUElfR2V0UmVsYXRpb25hbERhdGFiYXNlU25hcHNob3QuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlJldHVybnMgaW5mb3JtYXRpb24gYWJvdXQgYSByZWxhdGlvbmFsIGRhdGFiYXNlIHNuYXBzaG90LlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIkxpc3RcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiUmVsYXRpb25hbERhdGFiYXNlXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJHZXRSZWxhdGlvbmFsRGF0YWJhc2VTbmFwc2hvdHNcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGlnaHRzYWlsLzIwMTYtMTEtMjgvYXBpLXJlZmVyZW5jZS9BUElfR2V0UmVsYXRpb25hbERhdGFiYXNlU25hcHNob3RzLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJSZXR1cm5zIGluZm9ybWF0aW9uIGFib3V0IGFsbCByZWxhdGlvbmFsIGRhdGFiYXNlIHNuYXBzaG90cy5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJMaXN0XCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcIlJlbGF0aW9uYWxEYXRhYmFzZVwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiR2V0UmVsYXRpb25hbERhdGFiYXNlc1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9saWdodHNhaWwvMjAxNi0xMS0yOC9hcGktcmVmZXJlbmNlL0FQSV9HZXRSZWxhdGlvbmFsRGF0YWJhc2VzLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJSZXR1cm4gaW5mb3JtYXRpb24gYWJvdXQgYWxsIHJlbGF0aW9uYWwgZGF0YWJhc2VzLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlJlYWRcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiUmVsYXRpb25hbERhdGFiYXNlXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJHZXRTdGF0aWNJcFwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9saWdodHNhaWwvMjAxNi0xMS0yOC9hcGktcmVmZXJlbmNlL0FQSV9HZXRTdGF0aWNJcC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiUmV0dXJucyBpbmZvcm1hdGlvbiBhYm91dCBhIHN0YXRpYyBJUC5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJSZWFkXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcIlN0YXRpY0lwXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJHZXRTdGF0aWNJcHNcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGlnaHRzYWlsLzIwMTYtMTEtMjgvYXBpLXJlZmVyZW5jZS9BUElfR2V0U3RhdGljSXBzLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJSZXR1cm5zIGluZm9ybWF0aW9uIGFib3V0IGFsbCBzdGF0aWMgSVBzLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlJlYWRcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiU3RhdGljSXBcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkltcG9ydEtleVBhaXJcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGlnaHRzYWlsLzIwMTYtMTEtMjgvYXBpLXJlZmVyZW5jZS9BUElfSW1wb3J0S2V5UGFpci5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiSW1wb3J0cyBhIHB1YmxpYyBrZXkgZnJvbSBhIGtleSBwYWlyLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcIktleVBhaXJcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIklzVnBjUGVlcmVkXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xpZ2h0c2FpbC8yMDE2LTExLTI4L2FwaS1yZWZlcmVuY2UvQVBJX0lzVnBjUGVlcmVkLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJSZXR1cm5zIGEgYm9vbGVhbiB2YWx1ZSBpbmRpY2F0aW5nIHdoZXRoZXIgdGhlIEFtYXpvbiBMaWdodHNhaWwgdmlydHVhbCBwcml2YXRlIGNsb3VkIChWUEMpIGlzIHBlZXJlZC5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJSZWFkXCJcbiAgICB9LFxuICAgIFwiT3Blbkluc3RhbmNlUHVibGljUG9ydHNcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGlnaHRzYWlsLzIwMTYtMTEtMjgvYXBpLXJlZmVyZW5jZS9BUElfT3Blbkluc3RhbmNlUHVibGljUG9ydHMuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkFkZHMsIG9yIG9wZW5zIGEgcHVibGljIHBvcnQgb2YgYW4gaW5zdGFuY2UuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiSW5zdGFuY2VcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIlBlZXJWcGNcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGlnaHRzYWlsLzIwMTYtMTEtMjgvYXBpLXJlZmVyZW5jZS9BUElfUGVlclZwYy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiVHJpZXMgdG8gcGVlciB0aGUgQW1hem9uIExpZ2h0c2FpbCB2aXJ0dWFsIHByaXZhdGUgY2xvdWQgKFZQQykgd2l0aCB0aGUgZGVmYXVsdCBWUEMuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIlxuICAgIH0sXG4gICAgXCJQdXRJbnN0YW5jZVB1YmxpY1BvcnRzXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xpZ2h0c2FpbC8yMDE2LTExLTI4L2FwaS1yZWZlcmVuY2UvQVBJX1B1dEluc3RhbmNlUHVibGljUG9ydHMuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlNldHMgdGhlIHNwZWNpZmllZCBvcGVuIHBvcnRzIGZvciBhbiBpbnN0YW5jZSwgYW5kIGNsb3NlcyBhbGwgcG9ydHMgZm9yIGV2ZXJ5IHByb3RvY29sIG5vdCBpbmNsdWRlZCBpbiB0aGUgcmVxdWVzdC5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJJbnN0YW5jZVwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiUmVib290SW5zdGFuY2VcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGlnaHRzYWlsLzIwMTYtMTEtMjgvYXBpLXJlZmVyZW5jZS9BUElfUmVib290SW5zdGFuY2UuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlJlYm9vdHMgYW4gaW5zdGFuY2UgdGhhdCBpcyBpbiBhIHJ1bm5pbmcgc3RhdGUuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiSW5zdGFuY2VcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIlJlYm9vdFJlbGF0aW9uYWxEYXRhYmFzZVwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9saWdodHNhaWwvMjAxNi0xMS0yOC9hcGktcmVmZXJlbmNlL0FQSV9SZWJvb3RSZWxhdGlvbmFsRGF0YWJhc2UuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlJlYm9vdHMgYSByZWxhdGlvbmFsIGRhdGFiYXNlIHRoYXQgaXMgaW4gYSBydW5uaW5nIHN0YXRlLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcIlJlbGF0aW9uYWxEYXRhYmFzZVwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiUmVsZWFzZVN0YXRpY0lwXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xpZ2h0c2FpbC8yMDE2LTExLTI4L2FwaS1yZWZlcmVuY2UvQVBJX1JlbGVhc2VTdGF0aWNJcC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiRGVsZXRlcyBhIHN0YXRpYyBJUC5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJTdGF0aWNJcFwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiU3RhcnRJbnN0YW5jZVwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9saWdodHNhaWwvMjAxNi0xMS0yOC9hcGktcmVmZXJlbmNlL0FQSV9TdGFydEluc3RhbmNlLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJTdGFydHMgYW4gaW5zdGFuY2UgdGhhdCBpcyBpbiBhIHN0b3BwZWQgc3RhdGUuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiSW5zdGFuY2VcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIlN0YXJ0UmVsYXRpb25hbERhdGFiYXNlXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xpZ2h0c2FpbC8yMDE2LTExLTI4L2FwaS1yZWZlcmVuY2UvQVBJX1N0YXJ0UmVsYXRpb25hbERhdGFiYXNlLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJTdGFydHMgYSByZWxhdGlvbmFsIGRhdGFiYXNlIHRoYXQgaXMgaW4gYSBzdG9wcGVkIHN0YXRlLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcIlJlbGF0aW9uYWxEYXRhYmFzZVwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiU3RvcEluc3RhbmNlXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xpZ2h0c2FpbC8yMDE2LTExLTI4L2FwaS1yZWZlcmVuY2UvQVBJX1N0b3BJbnN0YW5jZS5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiU3RvcHMgYW4gaW5zdGFuY2UgdGhhdCBpcyBpbiBhIHJ1bm5pbmcgc3RhdGUuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiSW5zdGFuY2VcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIlN0b3BSZWxhdGlvbmFsRGF0YWJhc2VcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGlnaHRzYWlsLzIwMTYtMTEtMjgvYXBpLXJlZmVyZW5jZS9BUElfU3RvcFJlbGF0aW9uYWxEYXRhYmFzZS5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiU3RvcHMgYSByZWxhdGlvbmFsIGRhdGFiYXNlIHRoYXQgaXMgaW4gYSBydW5uaW5nIHN0YXRlLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcIlJlbGF0aW9uYWxEYXRhYmFzZVwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiVGFnUmVzb3VyY2VcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGlnaHRzYWlsLzIwMTYtMTEtMjgvYXBpLXJlZmVyZW5jZS9BUElfVGFnUmVzb3VyY2UuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlRhZ3MgYSByZXNvdXJjZS5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJEaXNrXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH0sXG4gICAgICAgIFwiRGlza1NuYXBzaG90XCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH0sXG4gICAgICAgIFwiRG9tYWluXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH0sXG4gICAgICAgIFwiSW5zdGFuY2VcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfSxcbiAgICAgICAgXCJJbnN0YW5jZVNuYXBzaG90XCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH0sXG4gICAgICAgIFwiS2V5UGFpclwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9LFxuICAgICAgICBcIkxvYWRCYWxhbmNlclwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9LFxuICAgICAgICBcIlJlbGF0aW9uYWxEYXRhYmFzZVwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9LFxuICAgICAgICBcIlJlbGF0aW9uYWxEYXRhYmFzZVNuYXBzaG90XCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH0sXG4gICAgICAgIFwiU3RhdGljSXBcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfVxuICAgICAgfSxcbiAgICAgIFwiY29uZGl0aW9uc1wiOiBbXG4gICAgICAgIFwiYXdzOlJlcXVlc3RUYWcvJHtUYWdLZXl9XCIsXG4gICAgICAgIFwiYXdzOlRhZ0tleXNcIlxuICAgICAgXVxuICAgIH0sXG4gICAgXCJVbnBlZXJWcGNcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGlnaHRzYWlsLzIwMTYtMTEtMjgvYXBpLXJlZmVyZW5jZS9BUElfVW5wZWVyVnBjLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJBdHRlbXB0cyB0byB1bnBlZXIgdGhlIEFtYXpvbiBMaWdodHNhaWwgdmlydHVhbCBwcml2YXRlIGNsb3VkIChWUEMpIGZyb20gdGhlIGRlZmF1bHQgVlBDLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCJcbiAgICB9LFxuICAgIFwiVW50YWdSZXNvdXJjZVwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9saWdodHNhaWwvMjAxNi0xMS0yOC9hcGktcmVmZXJlbmNlL0FQSV9VbnRhZ1Jlc291cmNlLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJVbnRhZ3MgYSByZXNvdXJjZS5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJEaXNrXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH0sXG4gICAgICAgIFwiRGlza1NuYXBzaG90XCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH0sXG4gICAgICAgIFwiRG9tYWluXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH0sXG4gICAgICAgIFwiSW5zdGFuY2VcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfSxcbiAgICAgICAgXCJJbnN0YW5jZVNuYXBzaG90XCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH0sXG4gICAgICAgIFwiS2V5UGFpclwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9LFxuICAgICAgICBcIkxvYWRCYWxhbmNlclwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9LFxuICAgICAgICBcIlJlbGF0aW9uYWxEYXRhYmFzZVwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiBmYWxzZVxuICAgICAgICB9LFxuICAgICAgICBcIlJlbGF0aW9uYWxEYXRhYmFzZVNuYXBzaG90XCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IGZhbHNlXG4gICAgICAgIH0sXG4gICAgICAgIFwiU3RhdGljSXBcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogZmFsc2VcbiAgICAgICAgfVxuICAgICAgfSxcbiAgICAgIFwiY29uZGl0aW9uc1wiOiBbXG4gICAgICAgIFwiYXdzOlJlcXVlc3RUYWcvJHtUYWdLZXl9XCIsXG4gICAgICAgIFwiYXdzOlRhZ0tleXNcIlxuICAgICAgXVxuICAgIH0sXG4gICAgXCJVcGRhdGVEb21haW5FbnRyeVwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9saWdodHNhaWwvMjAxNi0xMS0yOC9hcGktcmVmZXJlbmNlL0FQSV9VcGRhdGVEb21haW5FbnRyeS5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiVXBkYXRlcyBhIGRvbWFpbiByZWNvcmRzZXQgYWZ0ZXIgaXQgaXMgY3JlYXRlZC5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJEb21haW5cIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIlVwZGF0ZUxvYWRCYWxhbmNlckF0dHJpYnV0ZVwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9saWdodHNhaWwvMjAxNi0xMS0yOC9hcGktcmVmZXJlbmNlL0FQSV9EZXRhY2hJbnN0YW5jZXNGcm9tTG9hZEJhbGFuY2VyLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJVcGRhdGVzIGEgbG9hZCBiYWxhbmNlciBhdHRyaWJ1dGUsIHN1Y2ggYXMgdGhlIGhlYWx0aCBjaGVjayBwYXRoIGFuZCBzZXNzaW9uIHN0aWNraW5lc3MuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiTG9hZEJhbGFuY2VyXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJVcGRhdGVSZWxhdGlvbmFsRGF0YWJhc2VcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGlnaHRzYWlsLzIwMTYtMTEtMjgvYXBpLXJlZmVyZW5jZS9BUElfVXBkYXRlUmVsYXRpb25hbERhdGFiYXNlLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJVcGRhdGVzIGEgcmVsYXRpb25hbCBkYXRhYmFzZS5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJSZWxhdGlvbmFsRGF0YWJhc2VcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIlVwZGF0ZVJlbGF0aW9uYWxEYXRhYmFzZVBhcmFtZXRlcnNcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGlnaHRzYWlsLzIwMTYtMTEtMjgvYXBpLXJlZmVyZW5jZS9BUElfVXBkYXRlUmVsYXRpb25hbERhdGFiYXNlUGFyYW1ldGVycy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiVXBkYXRlcyB0aGUgcGFyYW1ldGVycyBvZiBhIHJlbGF0aW9uYWwgZGF0YWJhc2UuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIlxuICAgIH1cbiAgfTtcbiAgcHJvdGVjdGVkIHJlc291cmNlVHlwZXM6IFJlc291cmNlVHlwZXMgPSB7XG4gICAgXCJEb21haW5cIjoge1xuICAgICAgXCJuYW1lXCI6IFwiRG9tYWluXCIsXG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9saWdodHNhaWwvMjAxNi0xMS0yOC9hcGktcmVmZXJlbmNlL0FQSV9Eb21haW4uaHRtbFwiLFxuICAgICAgXCJhcm5cIjogXCJhcm46JHtQYXJ0aXRpb259OmxpZ2h0c2FpbDoke1JlZ2lvbn06JHtBY2NvdW50fTpEb21haW4vJHtJZH1cIixcbiAgICAgIFwiY29uZGl0aW9uS2V5c1wiOiBbXG4gICAgICAgIFwiYXdzOlJlc291cmNlVGFnLyR7VGFnS2V5fVwiXG4gICAgICBdXG4gICAgfSxcbiAgICBcIkluc3RhbmNlXCI6IHtcbiAgICAgIFwibmFtZVwiOiBcIkluc3RhbmNlXCIsXG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9saWdodHNhaWwvMjAxNi0xMS0yOC9hcGktcmVmZXJlbmNlL0FQSV9JbnN0YW5jZS5odG1sXCIsXG4gICAgICBcImFyblwiOiBcImFybjoke1BhcnRpdGlvbn06bGlnaHRzYWlsOiR7UmVnaW9ufToke0FjY291bnR9Okluc3RhbmNlLyR7SWR9XCIsXG4gICAgICBcImNvbmRpdGlvbktleXNcIjogW1xuICAgICAgICBcImF3czpSZXNvdXJjZVRhZy8ke1RhZ0tleX1cIlxuICAgICAgXVxuICAgIH0sXG4gICAgXCJJbnN0YW5jZVNuYXBzaG90XCI6IHtcbiAgICAgIFwibmFtZVwiOiBcIkluc3RhbmNlU25hcHNob3RcIixcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xpZ2h0c2FpbC8yMDE2LTExLTI4L2FwaS1yZWZlcmVuY2UvQVBJX0luc3RhbmNlU25hcHNob3QuaHRtbFwiLFxuICAgICAgXCJhcm5cIjogXCJhcm46JHtQYXJ0aXRpb259OmxpZ2h0c2FpbDoke1JlZ2lvbn06JHtBY2NvdW50fTpJbnN0YW5jZVNuYXBzaG90LyR7SWR9XCIsXG4gICAgICBcImNvbmRpdGlvbktleXNcIjogW1xuICAgICAgICBcImF3czpSZXNvdXJjZVRhZy8ke1RhZ0tleX1cIlxuICAgICAgXVxuICAgIH0sXG4gICAgXCJLZXlQYWlyXCI6IHtcbiAgICAgIFwibmFtZVwiOiBcIktleVBhaXJcIixcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xpZ2h0c2FpbC8yMDE2LTExLTI4L2FwaS1yZWZlcmVuY2UvQVBJX0tleVBhaXIuaHRtbFwiLFxuICAgICAgXCJhcm5cIjogXCJhcm46JHtQYXJ0aXRpb259OmxpZ2h0c2FpbDoke1JlZ2lvbn06JHtBY2NvdW50fTpLZXlQYWlyLyR7SWR9XCIsXG4gICAgICBcImNvbmRpdGlvbktleXNcIjogW1xuICAgICAgICBcImF3czpSZXNvdXJjZVRhZy8ke1RhZ0tleX1cIlxuICAgICAgXVxuICAgIH0sXG4gICAgXCJTdGF0aWNJcFwiOiB7XG4gICAgICBcIm5hbWVcIjogXCJTdGF0aWNJcFwiLFxuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGlnaHRzYWlsLzIwMTYtMTEtMjgvYXBpLXJlZmVyZW5jZS9BUElfU3RhdGljSXAuaHRtbFwiLFxuICAgICAgXCJhcm5cIjogXCJhcm46JHtQYXJ0aXRpb259OmxpZ2h0c2FpbDoke1JlZ2lvbn06JHtBY2NvdW50fTpTdGF0aWNJcC8ke0lkfVwiLFxuICAgICAgXCJjb25kaXRpb25LZXlzXCI6IFtcbiAgICAgICAgXCJhd3M6UmVzb3VyY2VUYWcvJHtUYWdLZXl9XCJcbiAgICAgIF1cbiAgICB9LFxuICAgIFwiRGlza1wiOiB7XG4gICAgICBcIm5hbWVcIjogXCJEaXNrXCIsXG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9saWdodHNhaWwvMjAxNi0xMS0yOC9hcGktcmVmZXJlbmNlL0FQSV9EaXNrLmh0bWxcIixcbiAgICAgIFwiYXJuXCI6IFwiYXJuOiR7UGFydGl0aW9ufTpsaWdodHNhaWw6JHtSZWdpb259OiR7QWNjb3VudH06RGlzay8ke0lkfVwiLFxuICAgICAgXCJjb25kaXRpb25LZXlzXCI6IFtcbiAgICAgICAgXCJhd3M6UmVzb3VyY2VUYWcvJHtUYWdLZXl9XCJcbiAgICAgIF1cbiAgICB9LFxuICAgIFwiRGlza1NuYXBzaG90XCI6IHtcbiAgICAgIFwibmFtZVwiOiBcIkRpc2tTbmFwc2hvdFwiLFxuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGlnaHRzYWlsLzIwMTYtMTEtMjgvYXBpLXJlZmVyZW5jZS9BUElfRGlza1NuYXBzaG90Lmh0bWxcIixcbiAgICAgIFwiYXJuXCI6IFwiYXJuOiR7UGFydGl0aW9ufTpsaWdodHNhaWw6JHtSZWdpb259OiR7QWNjb3VudH06RGlza1NuYXBzaG90LyR7SWR9XCIsXG4gICAgICBcImNvbmRpdGlvbktleXNcIjogW1xuICAgICAgICBcImF3czpSZXNvdXJjZVRhZy8ke1RhZ0tleX1cIlxuICAgICAgXVxuICAgIH0sXG4gICAgXCJMb2FkQmFsYW5jZXJcIjoge1xuICAgICAgXCJuYW1lXCI6IFwiTG9hZEJhbGFuY2VyXCIsXG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9saWdodHNhaWwvMjAxNi0xMS0yOC9hcGktcmVmZXJlbmNlL0FQSV9Mb2FkQmFsYW5jZXIuaHRtbFwiLFxuICAgICAgXCJhcm5cIjogXCJhcm46JHtQYXJ0aXRpb259OmxpZ2h0c2FpbDoke1JlZ2lvbn06JHtBY2NvdW50fTpMb2FkQmFsYW5jZXIvJHtJZH1cIixcbiAgICAgIFwiY29uZGl0aW9uS2V5c1wiOiBbXG4gICAgICAgIFwiYXdzOlJlc291cmNlVGFnLyR7VGFnS2V5fVwiXG4gICAgICBdXG4gICAgfSxcbiAgICBcIlBlZXJlZFZwY1wiOiB7XG4gICAgICBcIm5hbWVcIjogXCJQZWVyZWRWcGNcIixcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xpZ2h0c2FpbC8yMDE2LTExLTI4L2FwaS1yZWZlcmVuY2UvUGVlcmVkVnBjLmh0bWxcIixcbiAgICAgIFwiYXJuXCI6IFwiYXJuOiR7UGFydGl0aW9ufTpsaWdodHNhaWw6JHtSZWdpb259OiR7QWNjb3VudH06UGVlcmVkVnBjLyR7SWR9XCIsXG4gICAgICBcImNvbmRpdGlvbktleXNcIjogW11cbiAgICB9LFxuICAgIFwiTG9hZEJhbGFuY2VyVGxzQ2VydGlmaWNhdGVcIjoge1xuICAgICAgXCJuYW1lXCI6IFwiTG9hZEJhbGFuY2VyVGxzQ2VydGlmaWNhdGVcIixcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xpZ2h0c2FpbC8yMDE2LTExLTI4L2FwaS1yZWZlcmVuY2UvTG9hZEJhbGFuY2VyVGxzQ2VydGlmaWNhdGUuaHRtbFwiLFxuICAgICAgXCJhcm5cIjogXCJhcm46JHtQYXJ0aXRpb259OmxpZ2h0c2FpbDoke1JlZ2lvbn06JHtBY2NvdW50fTpMb2FkQmFsYW5jZXJUbHNDZXJ0aWZpY2F0ZS8ke0lkfVwiLFxuICAgICAgXCJjb25kaXRpb25LZXlzXCI6IFtdXG4gICAgfSxcbiAgICBcIkV4cG9ydFNuYXBzaG90UmVjb3JkXCI6IHtcbiAgICAgIFwibmFtZVwiOiBcIkV4cG9ydFNuYXBzaG90UmVjb3JkXCIsXG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9saWdodHNhaWwvMjAxNi0xMS0yOC9hcGktcmVmZXJlbmNlL0FQSV9FeHBvcnRTbmFwc2hvdFJlY29yZC5odG1sXCIsXG4gICAgICBcImFyblwiOiBcImFybjoke1BhcnRpdGlvbn06bGlnaHRzYWlsOiR7UmVnaW9ufToke0FjY291bnR9OkV4cG9ydFNuYXBzaG90UmVjb3JkLyR7SWR9XCIsXG4gICAgICBcImNvbmRpdGlvbktleXNcIjogW11cbiAgICB9LFxuICAgIFwiQ2xvdWRGb3JtYXRpb25TdGFja1JlY29yZFwiOiB7XG4gICAgICBcIm5hbWVcIjogXCJDbG91ZEZvcm1hdGlvblN0YWNrUmVjb3JkXCIsXG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9saWdodHNhaWwvMjAxNi0xMS0yOC9hcGktcmVmZXJlbmNlL0FQSV9DbG91ZEZvcm1hdGlvblN0YWNrUmVjb3JkLmh0bWxcIixcbiAgICAgIFwiYXJuXCI6IFwiYXJuOiR7UGFydGl0aW9ufTpsaWdodHNhaWw6JHtSZWdpb259OiR7QWNjb3VudH06Q2xvdWRGb3JtYXRpb25TdGFja1JlY29yZC8ke0lkfVwiLFxuICAgICAgXCJjb25kaXRpb25LZXlzXCI6IFtdXG4gICAgfSxcbiAgICBcIlJlbGF0aW9uYWxEYXRhYmFzZVwiOiB7XG4gICAgICBcIm5hbWVcIjogXCJSZWxhdGlvbmFsRGF0YWJhc2VcIixcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xpZ2h0c2FpbC8yMDE2LTExLTI4L2FwaS1yZWZlcmVuY2UvQVBJX1JlbGF0aW9uYWxEYXRhYmFzZS5odG1sXCIsXG4gICAgICBcImFyblwiOiBcImFybjoke1BhcnRpdGlvbn06bGlnaHRzYWlsOiR7UmVnaW9ufToke0FjY291bnR9OlJlbGF0aW9uYWxEYXRhYmFzZS8ke0lkfVwiLFxuICAgICAgXCJjb25kaXRpb25LZXlzXCI6IFtcbiAgICAgICAgXCJhd3M6UmVzb3VyY2VUYWcvJHtUYWdLZXl9XCJcbiAgICAgIF1cbiAgICB9LFxuICAgIFwiUmVsYXRpb25hbERhdGFiYXNlU25hcHNob3RcIjoge1xuICAgICAgXCJuYW1lXCI6IFwiUmVsYXRpb25hbERhdGFiYXNlU25hcHNob3RcIixcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xpZ2h0c2FpbC8yMDE2LTExLTI4L2FwaS1yZWZlcmVuY2UvQVBJX1JlbGF0aW9uYWxEYXRhYmFzZVNuYXBzaG90Lmh0bWxcIixcbiAgICAgIFwiYXJuXCI6IFwiYXJuOiR7UGFydGl0aW9ufTpsaWdodHNhaWw6JHtSZWdpb259OiR7QWNjb3VudH06UmVsYXRpb25hbERhdGFiYXNlU25hcHNob3QvJHtJZH1cIixcbiAgICAgIFwiY29uZGl0aW9uS2V5c1wiOiBbXG4gICAgICAgIFwiYXdzOlJlc291cmNlVGFnLyR7VGFnS2V5fVwiXG4gICAgICBdXG4gICAgfVxuICB9O1xuXG4gIC8qKlxuICAgKiBTdGF0ZW1lbnQgcHJvdmlkZXIgZm9yIHNlcnZpY2UgW2xpZ2h0c2FpbF0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL2xpc3RfYW1hem9ubGlnaHRzYWlsLmh0bWwpLlxuICAgKlxuICAgKiBAcGFyYW0gc2lkIFtTSURdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfc2lkLmh0bWwpIG9mIHRoZSBzdGF0ZW1lbnRcbiAgICovXG4gIGNvbnN0cnVjdG9yIChzaWQ/OiBzdHJpbmcpIHtcbiAgICBzdXBlcihzaWQpO1xuICB9XG5cbiAgLyoqXG4gICAqIENyZWF0ZXMgYSBzdGF0aWMgSVAgYWRkcmVzcyB0aGF0IGNhbiBiZSBhdHRhY2hlZCB0byBhbiBpbnN0YW5jZS5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGlnaHRzYWlsLzIwMTYtMTEtMjgvYXBpLXJlZmVyZW5jZS9BUElfQWxsb2NhdGVTdGF0aWNJcC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9BbGxvY2F0ZVN0YXRpY0lwKCkge1xuICAgIHRoaXMuYWRkKCdsaWdodHNhaWw6QWxsb2NhdGVTdGF0aWNJcCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEF0dGFjaGVzIGEgZGlzayB0byBhbiBpbnN0YW5jZS5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGlnaHRzYWlsLzIwMTYtMTEtMjgvYXBpLXJlZmVyZW5jZS9BUElfQXR0YWNoRGlzay5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9BdHRhY2hEaXNrKCkge1xuICAgIHRoaXMuYWRkKCdsaWdodHNhaWw6QXR0YWNoRGlzaycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEF0dGFjaGVzIG9uZSBvciBtb3JlIGluc3RhbmNlcyB0byBhIGxvYWQgYmFsYW5jZXIuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xpZ2h0c2FpbC8yMDE2LTExLTI4L2FwaS1yZWZlcmVuY2UvQVBJX0F0dGFjaEluc3RhbmNlc1RvTG9hZEJhbGFuY2VyLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0F0dGFjaEluc3RhbmNlc1RvTG9hZEJhbGFuY2VyKCkge1xuICAgIHRoaXMuYWRkKCdsaWdodHNhaWw6QXR0YWNoSW5zdGFuY2VzVG9Mb2FkQmFsYW5jZXInKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBBdHRhY2hlcyBhIFRMUyBjZXJ0aWZpY2F0ZSB0byBhIGxvYWQgYmFsYW5jZXIuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xpZ2h0c2FpbC8yMDE2LTExLTI4L2FwaS1yZWZlcmVuY2UvQVBJX0F0dGFjaExvYWRCYWxhbmNlclRsc0NlcnRpZmljYXRlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0F0dGFjaExvYWRCYWxhbmNlclRsc0NlcnRpZmljYXRlKCkge1xuICAgIHRoaXMuYWRkKCdsaWdodHNhaWw6QXR0YWNoTG9hZEJhbGFuY2VyVGxzQ2VydGlmaWNhdGUnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBBdHRhY2hlcyBhIHN0YXRpYyBJUCBhZGRyZXNzIHRvIGFuIGluc3RhbmNlLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9saWdodHNhaWwvMjAxNi0xMS0yOC9hcGktcmVmZXJlbmNlL0FQSV9BdHRhY2hTdGF0aWNJcC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9BdHRhY2hTdGF0aWNJcCgpIHtcbiAgICB0aGlzLmFkZCgnbGlnaHRzYWlsOkF0dGFjaFN0YXRpY0lwJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogQ2xvc2VzIGEgcHVibGljIHBvcnQgb2YgYW4gaW5zdGFuY2UuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xpZ2h0c2FpbC8yMDE2LTExLTI4L2FwaS1yZWZlcmVuY2UvQVBJX0Nsb3NlSW5zdGFuY2VQdWJsaWNQb3J0cy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DbG9zZUluc3RhbmNlUHVibGljUG9ydHMoKSB7XG4gICAgdGhpcy5hZGQoJ2xpZ2h0c2FpbDpDbG9zZUluc3RhbmNlUHVibGljUG9ydHMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBDb3BpZXMgYSBzbmFwc2hvdCBmcm9tIG9uZSBBV1MgUmVnaW9uIHRvIGFub3RoZXIgaW4gQW1hem9uIExpZ2h0c2FpbC5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGlnaHRzYWlsLzIwMTYtMTEtMjgvYXBpLXJlZmVyZW5jZS9BUElfQ29weVNuYXBzaG90Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NvcHlTbmFwc2hvdCgpIHtcbiAgICB0aGlzLmFkZCgnbGlnaHRzYWlsOkNvcHlTbmFwc2hvdCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIENyZWF0ZXMgYSBuZXcgQW1hem9uIEVDMiBpbnN0YW5jZSBmcm9tIGFuIGV4cG9ydGVkIEFtYXpvbiBMaWdodHNhaWwgc25hcHNob3QuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xpZ2h0c2FpbC8yMDE2LTExLTI4L2FwaS1yZWZlcmVuY2UvQVBJX0NyZWF0ZUNsb3VkRm9ybWF0aW9uU3RhY2suaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlQ2xvdWRGb3JtYXRpb25TdGFjaygpIHtcbiAgICB0aGlzLmFkZCgnbGlnaHRzYWlsOkNyZWF0ZUNsb3VkRm9ybWF0aW9uU3RhY2snKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBDcmVhdGVzIGEgZGlzay5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGlnaHRzYWlsLzIwMTYtMTEtMjgvYXBpLXJlZmVyZW5jZS9BUElfQ3JlYXRlRGlzay5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVEaXNrKCkge1xuICAgIHRoaXMuYWRkKCdsaWdodHNhaWw6Q3JlYXRlRGlzaycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIENyZWF0ZXMgYSBkaXNrIGZyb20gc25hcHNob3QuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xpZ2h0c2FpbC8yMDE2LTExLTI4L2FwaS1yZWZlcmVuY2UvQVBJX0NyZWF0ZURpc2tGcm9tU25hcHNob3QuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlRGlza0Zyb21TbmFwc2hvdCgpIHtcbiAgICB0aGlzLmFkZCgnbGlnaHRzYWlsOkNyZWF0ZURpc2tGcm9tU25hcHNob3QnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBDcmVhdGVzIGEgZGlzayBzbmFwc2hvdC5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGlnaHRzYWlsLzIwMTYtMTEtMjgvYXBpLXJlZmVyZW5jZS9BUElfQ3JlYXRlRGlza1NuYXBzaG90Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZURpc2tTbmFwc2hvdCgpIHtcbiAgICB0aGlzLmFkZCgnbGlnaHRzYWlsOkNyZWF0ZURpc2tTbmFwc2hvdCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIENyZWF0ZXMgYSBkb21haW4gcmVzb3VyY2UgZm9yIHRoZSBzcGVjaWZpZWQgZG9tYWluIG5hbWUuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xpZ2h0c2FpbC8yMDE2LTExLTI4L2FwaS1yZWZlcmVuY2UvQVBJX0NyZWF0ZURvbWFpbi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVEb21haW4oKSB7XG4gICAgdGhpcy5hZGQoJ2xpZ2h0c2FpbDpDcmVhdGVEb21haW4nKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBDcmVhdGVzIG9uZSBvciBtb3JlIEROUyByZWNvcmQgZW50cmllcyBmb3IgYSBkb21haW4gcmVzb3VyY2U6IEFkZHJlc3MgKEEpLCBjYW5vbmljYWwgbmFtZSAoQ05BTUUpLCBtYWlsIGV4Y2hhbmdlciAoTVgpLCBuYW1lIHNlcnZlciAoTlMpLCBzdGFydCBvZiBhdXRob3JpdHkgKFNPQSksIHNlcnZpY2UgbG9jYXRvciAoU1JWKSwgb3IgdGV4dCAoVFhUKS5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGlnaHRzYWlsLzIwMTYtMTEtMjgvYXBpLXJlZmVyZW5jZS9BUElfQ3JlYXRlRG9tYWluRW50cnkuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlRG9tYWluRW50cnkoKSB7XG4gICAgdGhpcy5hZGQoJ2xpZ2h0c2FpbDpDcmVhdGVEb21haW5FbnRyeScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIENyZWF0ZXMgYW4gaW5zdGFuY2Ugc25hcHNob3QuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xpZ2h0c2FpbC8yMDE2LTExLTI4L2FwaS1yZWZlcmVuY2UvQVBJX0NyZWF0ZUluc3RhbmNlU25hcHNob3QuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlSW5zdGFuY2VTbmFwc2hvdCgpIHtcbiAgICB0aGlzLmFkZCgnbGlnaHRzYWlsOkNyZWF0ZUluc3RhbmNlU25hcHNob3QnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBDcmVhdGVzIG9uZSBvciBtb3JlIGluc3RhbmNlcy5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGlnaHRzYWlsLzIwMTYtMTEtMjgvYXBpLXJlZmVyZW5jZS9BUElfQ3JlYXRlSW5zdGFuY2VzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZUluc3RhbmNlcygpIHtcbiAgICB0aGlzLmFkZCgnbGlnaHRzYWlsOkNyZWF0ZUluc3RhbmNlcycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIENyZWF0ZXMgb25lIG9yIG1vcmUgaW5zdGFuY2VzIGJhc2VkIG9uIGFuIGluc3RhbmNlIHNuYXBzaG90LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9saWdodHNhaWwvMjAxNi0xMS0yOC9hcGktcmVmZXJlbmNlL0FQSV9DcmVhdGVJbnN0YW5jZXNGcm9tU25hcHNob3QuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlSW5zdGFuY2VzRnJvbVNuYXBzaG90KCkge1xuICAgIHRoaXMuYWRkKCdsaWdodHNhaWw6Q3JlYXRlSW5zdGFuY2VzRnJvbVNuYXBzaG90Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogQ3JlYXRlcyBhIGtleSBwYWlyIHVzZWQgdG8gYXV0aGVudGljYXRlIGFuZCBjb25uZWN0IHRvIGFuIGluc3RhbmNlLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9saWdodHNhaWwvMjAxNi0xMS0yOC9hcGktcmVmZXJlbmNlL0FQSV9DcmVhdGVLZXlQYWlyLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZUtleVBhaXIoKSB7XG4gICAgdGhpcy5hZGQoJ2xpZ2h0c2FpbDpDcmVhdGVLZXlQYWlyJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogQ3JlYXRlcyBhIGxvYWQgYmFsYW5jZXIuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xpZ2h0c2FpbC8yMDE2LTExLTI4L2FwaS1yZWZlcmVuY2UvQVBJX0NyZWF0ZUxvYWRCYWxhbmNlci5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVMb2FkQmFsYW5jZXIoKSB7XG4gICAgdGhpcy5hZGQoJ2xpZ2h0c2FpbDpDcmVhdGVMb2FkQmFsYW5jZXInKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBDcmVhdGVzIGEgbG9hZCBiYWxhbmNlciBUTFMgY2VydGlmaWNhdGUuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xpZ2h0c2FpbC8yMDE2LTExLTI4L2FwaS1yZWZlcmVuY2UvQVBJX0NyZWF0ZUxvYWRCYWxhbmNlclRsc0NlcnRpZmljYXRlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZUxvYWRCYWxhbmNlclRsc0NlcnRpZmljYXRlKCkge1xuICAgIHRoaXMuYWRkKCdsaWdodHNhaWw6Q3JlYXRlTG9hZEJhbGFuY2VyVGxzQ2VydGlmaWNhdGUnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBDcmVhdGVzIGEgbmV3IHJlbGF0aW9uYWwgZGF0YWJhc2UuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xpZ2h0c2FpbC8yMDE2LTExLTI4L2FwaS1yZWZlcmVuY2UvQVBJX0NyZWF0ZVJlbGF0aW9uYWxEYXRhYmFzZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVSZWxhdGlvbmFsRGF0YWJhc2UoKSB7XG4gICAgdGhpcy5hZGQoJ2xpZ2h0c2FpbDpDcmVhdGVSZWxhdGlvbmFsRGF0YWJhc2UnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBDcmVhdGVzIGEgbmV3IHJlbGF0aW9uYWwgZGF0YWJhc2UgZnJvbSBhIHNuYXBzaG90LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9saWdodHNhaWwvMjAxNi0xMS0yOC9hcGktcmVmZXJlbmNlL0FQSV9DcmVhdGVSZWxhdGlvbmFsRGF0YWJhc2VGcm9tU25hcHNob3QuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlUmVsYXRpb25hbERhdGFiYXNlRnJvbVNuYXBzaG90KCkge1xuICAgIHRoaXMuYWRkKCdsaWdodHNhaWw6Q3JlYXRlUmVsYXRpb25hbERhdGFiYXNlRnJvbVNuYXBzaG90Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogQ3JlYXRlcyBhIHJlbGF0aW9uYWwgZGF0YWJhc2Ugc25hcHNob3QuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xpZ2h0c2FpbC8yMDE2LTExLTI4L2FwaS1yZWZlcmVuY2UvQVBJX0NyZWF0ZVJlbGF0aW9uYWxEYXRhYmFzZVNuYXBzaG90Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZVJlbGF0aW9uYWxEYXRhYmFzZVNuYXBzaG90KCkge1xuICAgIHRoaXMuYWRkKCdsaWdodHNhaWw6Q3JlYXRlUmVsYXRpb25hbERhdGFiYXNlU25hcHNob3QnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBEZWxldGVzIGEgZGlzay5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGlnaHRzYWlsLzIwMTYtMTEtMjgvYXBpLXJlZmVyZW5jZS9BUElfRGVsZXRlRGlzay5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVEaXNrKCkge1xuICAgIHRoaXMuYWRkKCdsaWdodHNhaWw6RGVsZXRlRGlzaycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIERlbGV0ZXMgYSBkaXNrIHNuYXBzaG90LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9saWdodHNhaWwvMjAxNi0xMS0yOC9hcGktcmVmZXJlbmNlL0FQSV9EZWxldGVEaXNrU25hcHNob3QuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlRGlza1NuYXBzaG90KCkge1xuICAgIHRoaXMuYWRkKCdsaWdodHNhaWw6RGVsZXRlRGlza1NuYXBzaG90Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogRGVsZXRlcyBhIGRvbWFpbiByZXNvdXJjZSBhbmQgYWxsIG9mIGl0cyBETlMgcmVjb3Jkcy5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGlnaHRzYWlsLzIwMTYtMTEtMjgvYXBpLXJlZmVyZW5jZS9BUElfRGVsZXRlRG9tYWluLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZURvbWFpbigpIHtcbiAgICB0aGlzLmFkZCgnbGlnaHRzYWlsOkRlbGV0ZURvbWFpbicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIERlbGV0ZXMgYSBETlMgcmVjb3JkIGVudHJ5IGZvciBhIGRvbWFpbiByZXNvdXJjZS5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGlnaHRzYWlsLzIwMTYtMTEtMjgvYXBpLXJlZmVyZW5jZS9BUElfRGVsZXRlRG9tYWluRW50cnkuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlRG9tYWluRW50cnkoKSB7XG4gICAgdGhpcy5hZGQoJ2xpZ2h0c2FpbDpEZWxldGVEb21haW5FbnRyeScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIERlbGV0ZXMgYW4gaW5zdGFuY2UuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xpZ2h0c2FpbC8yMDE2LTExLTI4L2FwaS1yZWZlcmVuY2UvQVBJX0RlbGV0ZUluc3RhbmNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZUluc3RhbmNlKCkge1xuICAgIHRoaXMuYWRkKCdsaWdodHNhaWw6RGVsZXRlSW5zdGFuY2UnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBEZWxldGVzIGFuIGluc3RhbmNlIHNuYXBzaG90LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9saWdodHNhaWwvMjAxNi0xMS0yOC9hcGktcmVmZXJlbmNlL0FQSV9EZWxldGVJbnN0YW5jZVNuYXBzaG90Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZUluc3RhbmNlU25hcHNob3QoKSB7XG4gICAgdGhpcy5hZGQoJ2xpZ2h0c2FpbDpEZWxldGVJbnN0YW5jZVNuYXBzaG90Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogRGVsZXRlcyBhIGtleSBwYWlyIHVzZWQgdG8gYXV0aGVudGljYXRlIGFuZCBjb25uZWN0IHRvIGFuIGluc3RhbmNlLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9saWdodHNhaWwvMjAxNi0xMS0yOC9hcGktcmVmZXJlbmNlL0FQSV9EZWxldGVLZXlQYWlyLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZUtleVBhaXIoKSB7XG4gICAgdGhpcy5hZGQoJ2xpZ2h0c2FpbDpEZWxldGVLZXlQYWlyJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogRGVsZXRlcyB0aGUga25vd24gaG9zdCBrZXkgb3IgY2VydGlmaWNhdGUgdXNlZCBieSB0aGUgQW1hem9uIExpZ2h0c2FpbCBicm93c2VyLWJhc2VkIFNTSCBvciBSRFAgY2xpZW50cyB0byBhdXRoZW50aWNhdGUgYW4gaW5zdGFuY2UuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xpZ2h0c2FpbC8yMDE2LTExLTI4L2FwaS1yZWZlcmVuY2UvQVBJX0RlbGV0ZUtub3duSG9zdEtleXMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlS25vd25Ib3N0S2V5cygpIHtcbiAgICB0aGlzLmFkZCgnbGlnaHRzYWlsOkRlbGV0ZUtub3duSG9zdEtleXMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBEZWxldGVzIGEgbG9hZCBiYWxhbmNlci5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGlnaHRzYWlsLzIwMTYtMTEtMjgvYXBpLXJlZmVyZW5jZS9BUElfRGVsZXRlTG9hZEJhbGFuY2VyLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZUxvYWRCYWxhbmNlcigpIHtcbiAgICB0aGlzLmFkZCgnbGlnaHRzYWlsOkRlbGV0ZUxvYWRCYWxhbmNlcicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIERlbGV0ZXMgYSBsb2FkIGJhbGFuY2VyIFRMUyBjZXJ0aWZpY2F0ZS5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGlnaHRzYWlsLzIwMTYtMTEtMjgvYXBpLXJlZmVyZW5jZS9BUElfRGVsZXRlTG9hZEJhbGFuY2VyVGxzQ2VydGlmaWNhdGUuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlTG9hZEJhbGFuY2VyVGxzQ2VydGlmaWNhdGUoKSB7XG4gICAgdGhpcy5hZGQoJ2xpZ2h0c2FpbDpEZWxldGVMb2FkQmFsYW5jZXJUbHNDZXJ0aWZpY2F0ZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIERlbGV0ZXMgYSByZWxhdGlvbmFsIGRhdGFiYXNlLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9saWdodHNhaWwvMjAxNi0xMS0yOC9hcGktcmVmZXJlbmNlL0FQSV9EZWxldGVSZWxhdGlvbmFsRGF0YWJhc2UuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlUmVsYXRpb25hbERhdGFiYXNlKCkge1xuICAgIHRoaXMuYWRkKCdsaWdodHNhaWw6RGVsZXRlUmVsYXRpb25hbERhdGFiYXNlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogRGVsZXRlcyByZWxhdGlvbmFsIGRhdGFiYXNlIHNuYXBzaG90LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9saWdodHNhaWwvMjAxNi0xMS0yOC9hcGktcmVmZXJlbmNlL0FQSV9EZWxldGVSZWxhdGlvbmFsRGF0YWJhc2VTbmFwc2hvdC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVSZWxhdGlvbmFsRGF0YWJhc2VTbmFwc2hvdCgpIHtcbiAgICB0aGlzLmFkZCgnbGlnaHRzYWlsOkRlbGV0ZVJlbGF0aW9uYWxEYXRhYmFzZVNuYXBzaG90Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogRGV0YWNoZXMgYSBkaXNrIGZyb20gYW4gaW5zdGFuY2UuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xpZ2h0c2FpbC8yMDE2LTExLTI4L2FwaS1yZWZlcmVuY2UvQVBJX0RldGFjaERpc2suaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGV0YWNoRGlzaygpIHtcbiAgICB0aGlzLmFkZCgnbGlnaHRzYWlsOkRldGFjaERpc2snKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBEZXRhY2hlcyBvbmUgb3IgbW9yZSBpbnN0YW5jZXMgZnJvbSBhIGxvYWQgYmFsYW5jZXIuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xpZ2h0c2FpbC8yMDE2LTExLTI4L2FwaS1yZWZlcmVuY2UvQVBJX0RldGFjaEluc3RhbmNlc0Zyb21Mb2FkQmFsYW5jZXIuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGV0YWNoSW5zdGFuY2VzRnJvbUxvYWRCYWxhbmNlcigpIHtcbiAgICB0aGlzLmFkZCgnbGlnaHRzYWlsOkRldGFjaEluc3RhbmNlc0Zyb21Mb2FkQmFsYW5jZXInKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBEZXRhY2hlcyBhIHN0YXRpYyBJUCBmcm9tIGFuIGluc3RhbmNlIHRvIHdoaWNoIGl0IGlzIGF0dGFjaGVkLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9saWdodHNhaWwvMjAxNi0xMS0yOC9hcGktcmVmZXJlbmNlL0FQSV9EZXRhY2hTdGF0aWNJcC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZXRhY2hTdGF0aWNJcCgpIHtcbiAgICB0aGlzLmFkZCgnbGlnaHRzYWlsOkRldGFjaFN0YXRpY0lwJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogRG93bmxvYWRzIHRoZSBkZWZhdWx0IGtleSBwYWlyIHVzZWQgdG8gYXV0aGVudGljYXRlIGFuZCBjb25uZWN0IHRvIGluc3RhbmNlcyBpbiBhIHNwZWNpZmljIEFXUyBSZWdpb24uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xpZ2h0c2FpbC8yMDE2LTExLTI4L2FwaS1yZWZlcmVuY2UvQVBJX0Rvd25sb2FkRGVmYXVsdEtleVBhaXIuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRG93bmxvYWREZWZhdWx0S2V5UGFpcigpIHtcbiAgICB0aGlzLmFkZCgnbGlnaHRzYWlsOkRvd25sb2FkRGVmYXVsdEtleVBhaXInKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBFeHBvcnRzIGFuIEFtYXpvbiBMaWdodHNhaWwgc25hcHNob3QgdG8gQW1hem9uIEVDMi5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGlnaHRzYWlsLzIwMTYtMTEtMjgvYXBpLXJlZmVyZW5jZS9BUElfRXhwb3J0U25hcHNob3QuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRXhwb3J0U25hcHNob3QoKSB7XG4gICAgdGhpcy5hZGQoJ2xpZ2h0c2FpbDpFeHBvcnRTbmFwc2hvdCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgdGhlIG5hbWVzIG9mIGFsbCBhY3RpdmUgKG5vdCBkZWxldGVkKSByZXNvdXJjZXMuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGlnaHRzYWlsLzIwMTYtMTEtMjgvYXBpLXJlZmVyZW5jZS9BUElfR2V0QWN0aXZlTmFtZXMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0QWN0aXZlTmFtZXMoKSB7XG4gICAgdGhpcy5hZGQoJ2xpZ2h0c2FpbDpHZXRBY3RpdmVOYW1lcycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgYSBsaXN0IG9mIGluc3RhbmNlIGltYWdlcywgb3IgYmx1ZXByaW50cy4gWW91IGNhbiB1c2UgYSBibHVlcHJpbnQgdG8gY3JlYXRlIGEgbmV3IGluc3RhbmNlIGFscmVhZHkgcnVubmluZyBhIHNwZWNpZmljIG9wZXJhdGluZyBzeXN0ZW0sIGFzIHdlbGwgYXMgYSBwcmUtaW5zdGFsbGVkIGFwcGxpY2F0aW9uIG9yIGRldmVsb3BtZW50IHN0YWNrLiBUaGUgc29mdHdhcmUgdGhhdCBydW5zIG9uIHlvdXIgaW5zdGFuY2UgZGVwZW5kcyBvbiB0aGUgYmx1ZXByaW50IHlvdSBkZWZpbmUgd2hlbiBjcmVhdGluZyB0aGUgaW5zdGFuY2UuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGlnaHRzYWlsLzIwMTYtMTEtMjgvYXBpLXJlZmVyZW5jZS9BUElfR2V0Qmx1ZXByaW50cy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRCbHVlcHJpbnRzKCkge1xuICAgIHRoaXMuYWRkKCdsaWdodHNhaWw6R2V0Qmx1ZXByaW50cycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgYSBsaXN0IG9mIGluc3RhbmNlIGJ1bmRsZXMuIFlvdSBjYW4gdXNlIGEgYnVuZGxlIHRvIGNyZWF0ZSBhIG5ldyBpbnN0YW5jZSB3aXRoIGEgc2V0IG9mIHBlcmZvcm1hbmNlIHNwZWNpZmljYXRpb25zLCBzdWNoIGFzIENQVSBjb3VudCwgZGlzayBzaXplLCBSQU0gc2l6ZSwgYW5kIG5ldHdvcmsgdHJhbnNmZXIgYWxsb3dhbmNlLiBUaGUgY29zdCBvZiB5b3VyIGluc3RhbmNlIGRlcGVuZHMgb24gdGhlIGJ1bmRsZSB5b3UgZGVmaW5lIHdoZW4gY3JlYXRpbmcgdGhlIGluc3RhbmNlLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xpZ2h0c2FpbC8yMDE2LTExLTI4L2FwaS1yZWZlcmVuY2UvQVBJX0dldEJ1bmRsZXMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0QnVuZGxlcygpIHtcbiAgICB0aGlzLmFkZCgnbGlnaHRzYWlsOkdldEJ1bmRsZXMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIGluZm9ybWF0aW9uIGFib3V0IGFsbCBDbG91ZEZvcm1hdGlvbiBzdGFja3MgdXNlZCB0byBjcmVhdGUgQW1hem9uIEVDMiByZXNvdXJjZXMgZnJvbSBleHBvcnRlZCBBbWF6b24gTGlnaHRzYWlsIHNuYXBzaG90cy5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9saWdodHNhaWwvMjAxNi0xMS0yOC9hcGktcmVmZXJlbmNlL0FQSV9HZXRDbG91ZEZvcm1hdGlvblN0YWNrUmVjb3Jkcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRDbG91ZEZvcm1hdGlvblN0YWNrUmVjb3JkcygpIHtcbiAgICB0aGlzLmFkZCgnbGlnaHRzYWlsOkdldENsb3VkRm9ybWF0aW9uU3RhY2tSZWNvcmRzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyBpbmZvcm1hdGlvbiBhYm91dCBhIGRpc2suXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGlnaHRzYWlsLzIwMTYtMTEtMjgvYXBpLXJlZmVyZW5jZS9BUElfR2V0RGlzay5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXREaXNrKCkge1xuICAgIHRoaXMuYWRkKCdsaWdodHNhaWw6R2V0RGlzaycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgaW5mb3JtYXRpb24gYWJvdXQgYSBkaXNrIHNuYXBzaG90LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xpZ2h0c2FpbC8yMDE2LTExLTI4L2FwaS1yZWZlcmVuY2UvQVBJX0dldERpc2tTbmFwc2hvdC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXREaXNrU25hcHNob3QoKSB7XG4gICAgdGhpcy5hZGQoJ2xpZ2h0c2FpbDpHZXREaXNrU25hcHNob3QnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIGluZm9ybWF0aW9uIGFib3V0IGFsbCBkaXNrIHNuYXBzaG90cy5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9saWdodHNhaWwvMjAxNi0xMS0yOC9hcGktcmVmZXJlbmNlL0FQSV9HZXREaXNrU25hcHNob3RzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldERpc2tTbmFwc2hvdHMoKSB7XG4gICAgdGhpcy5hZGQoJ2xpZ2h0c2FpbDpHZXREaXNrU25hcHNob3RzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyBpbmZvcm1hdGlvbiBhYm91dCBhbGwgZGlza3MuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGlnaHRzYWlsLzIwMTYtMTEtMjgvYXBpLXJlZmVyZW5jZS9BUElfR2V0RGlza3MuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0RGlza3MoKSB7XG4gICAgdGhpcy5hZGQoJ2xpZ2h0c2FpbDpHZXREaXNrcycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgRE5TIHJlY29yZHMgZm9yIGEgZG9tYWluIHJlc291cmNlLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xpZ2h0c2FpbC8yMDE2LTExLTI4L2FwaS1yZWZlcmVuY2UvQVBJX0dldERvbWFpbi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXREb21haW4oKSB7XG4gICAgdGhpcy5hZGQoJ2xpZ2h0c2FpbDpHZXREb21haW4nKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIEROUyByZWNvcmRzIGZvciBhbGwgZG9tYWluIHJlc291cmNlcy5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9saWdodHNhaWwvMjAxNi0xMS0yOC9hcGktcmVmZXJlbmNlL0FQSV9HZXREb21haW5zLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldERvbWFpbnMoKSB7XG4gICAgdGhpcy5hZGQoJ2xpZ2h0c2FpbDpHZXREb21haW5zJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyBpbmZvcm1hdGlvbiBhYm91dCBhbGwgcmVjb3JkcyB0byBleHBvcnQgQW1hem9uIExpZ2h0c2FpbCBzbmFwc2hvdHMgdG8gQW1hem9uIEVDMi5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9saWdodHNhaWwvMjAxNi0xMS0yOC9hcGktcmVmZXJlbmNlL0FQSV9HZXRFeHBvcnRTbmFwc2hvdFJlY29yZHMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0RXhwb3J0U25hcHNob3RSZWNvcmRzKCkge1xuICAgIHRoaXMuYWRkKCdsaWdodHNhaWw6R2V0RXhwb3J0U25hcHNob3RSZWNvcmRzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyBpbmZvcm1hdGlvbiBhYm91dCBhbiBpbnN0YW5jZS5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9saWdodHNhaWwvMjAxNi0xMS0yOC9hcGktcmVmZXJlbmNlL0FQSV9HZXRJbnN0YW5jZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRJbnN0YW5jZSgpIHtcbiAgICB0aGlzLmFkZCgnbGlnaHRzYWlsOkdldEluc3RhbmNlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyB0ZW1wb3Jhcnkga2V5cyB5b3UgY2FuIHVzZSB0byBhdXRoZW50aWNhdGUgYW5kIGNvbm5lY3QgdG8gYW4gaW5zdGFuY2UuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xpZ2h0c2FpbC8yMDE2LTExLTI4L2FwaS1yZWZlcmVuY2UvQVBJX0dldEluc3RhbmNlQWNjZXNzRGV0YWlscy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRJbnN0YW5jZUFjY2Vzc0RldGFpbHMoKSB7XG4gICAgdGhpcy5hZGQoJ2xpZ2h0c2FpbDpHZXRJbnN0YW5jZUFjY2Vzc0RldGFpbHMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIHRoZSBkYXRhIHBvaW50cyBmb3IgdGhlIHNwZWNpZmllZCBtZXRyaWMgb2YgYW4gaW5zdGFuY2UuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGlnaHRzYWlsLzIwMTYtMTEtMjgvYXBpLXJlZmVyZW5jZS9BUElfR2V0SW5zdGFuY2VNZXRyaWNEYXRhLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldEluc3RhbmNlTWV0cmljRGF0YSgpIHtcbiAgICB0aGlzLmFkZCgnbGlnaHRzYWlsOkdldEluc3RhbmNlTWV0cmljRGF0YScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgdGhlIHBvcnQgc3RhdGVzIG9mIGFuIGluc3RhbmNlLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xpZ2h0c2FpbC8yMDE2LTExLTI4L2FwaS1yZWZlcmVuY2UvQVBJX0dldEluc3RhbmNlUG9ydFN0YXRlcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRJbnN0YW5jZVBvcnRTdGF0ZXMoKSB7XG4gICAgdGhpcy5hZGQoJ2xpZ2h0c2FpbDpHZXRJbnN0YW5jZVBvcnRTdGF0ZXMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIGluZm9ybWF0aW9uIGFib3V0IGFuIGluc3RhbmNlIHNuYXBzaG90LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xpZ2h0c2FpbC8yMDE2LTExLTI4L2FwaS1yZWZlcmVuY2UvQVBJX0dldEluc3RhbmNlU25hcHNob3QuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0SW5zdGFuY2VTbmFwc2hvdCgpIHtcbiAgICB0aGlzLmFkZCgnbGlnaHRzYWlsOkdldEluc3RhbmNlU25hcHNob3QnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIGluZm9ybWF0aW9uIGFib3V0IGFsbCBpbnN0YW5jZSBzbmFwc2hvdHMuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGlnaHRzYWlsLzIwMTYtMTEtMjgvYXBpLXJlZmVyZW5jZS9BUElfR2V0SW5zdGFuY2VTbmFwc2hvdHMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0SW5zdGFuY2VTbmFwc2hvdHMoKSB7XG4gICAgdGhpcy5hZGQoJ2xpZ2h0c2FpbDpHZXRJbnN0YW5jZVNuYXBzaG90cycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgdGhlIHN0YXRlIG9mIGFuIGluc3RhbmNlLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xpZ2h0c2FpbC8yMDE2LTExLTI4L2FwaS1yZWZlcmVuY2UvQVBJX0dldEluc3RhbmNlU3RhdGUuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0SW5zdGFuY2VTdGF0ZSgpIHtcbiAgICB0aGlzLmFkZCgnbGlnaHRzYWlsOkdldEluc3RhbmNlU3RhdGUnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIGluZm9ybWF0aW9uIGFib3V0IGFsbCBpbnN0YW5jZXMuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGlnaHRzYWlsLzIwMTYtMTEtMjgvYXBpLXJlZmVyZW5jZS9BUElfR2V0SW5zdGFuY2VzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldEluc3RhbmNlcygpIHtcbiAgICB0aGlzLmFkZCgnbGlnaHRzYWlsOkdldEluc3RhbmNlcycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgaW5mb3JtYXRpb24gYWJvdXQgYSBrZXkgcGFpci5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9saWdodHNhaWwvMjAxNi0xMS0yOC9hcGktcmVmZXJlbmNlL0FQSV9HZXRLZXlQYWlyLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldEtleVBhaXIoKSB7XG4gICAgdGhpcy5hZGQoJ2xpZ2h0c2FpbDpHZXRLZXlQYWlyJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyBpbmZvcm1hdGlvbiBhYm91dCBhbGwga2V5IHBhaXJzLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xpZ2h0c2FpbC8yMDE2LTExLTI4L2FwaS1yZWZlcmVuY2UvQVBJX0dldEtleVBhaXJzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldEtleVBhaXJzKCkge1xuICAgIHRoaXMuYWRkKCdsaWdodHNhaWw6R2V0S2V5UGFpcnMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIGluZm9ybWF0aW9uIGFib3V0IGEgbG9hZCBiYWxhbmNlci5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9saWdodHNhaWwvMjAxNi0xMS0yOC9hcGktcmVmZXJlbmNlL0FQSV9HZXRMb2FkQmFsYW5jZXIuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0TG9hZEJhbGFuY2VyKCkge1xuICAgIHRoaXMuYWRkKCdsaWdodHNhaWw6R2V0TG9hZEJhbGFuY2VyJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyB0aGUgZGF0YSBwb2ludHMgZm9yIHRoZSBzcGVjaWZpZWQgbWV0cmljIG9mIGEgbG9hZCBiYWxhbmNlci5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9saWdodHNhaWwvMjAxNi0xMS0yOC9hcGktcmVmZXJlbmNlL0FQSV9HZXRMb2FkQmFsYW5jZXJNZXRyaWNEYXRhLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldExvYWRCYWxhbmNlck1ldHJpY0RhdGEoKSB7XG4gICAgdGhpcy5hZGQoJ2xpZ2h0c2FpbDpHZXRMb2FkQmFsYW5jZXJNZXRyaWNEYXRhJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyBpbmZvcm1hdGlvbiBhYm91dCBhIGxvYWQgYmFsYW5jZXIgVExTIGNlcnRpZmljYXRlLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xpZ2h0c2FpbC8yMDE2LTExLTI4L2FwaS1yZWZlcmVuY2UvQVBJX0RlbGV0ZUxvYWRCYWxhbmNlclRsc0NlcnRpZmljYXRlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldExvYWRCYWxhbmNlclRsc0NlcnRpZmljYXRlcygpIHtcbiAgICB0aGlzLmFkZCgnbGlnaHRzYWlsOkdldExvYWRCYWxhbmNlclRsc0NlcnRpZmljYXRlcycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgaW5mb3JtYXRpb24gYWJvdXQgbG9hZCBiYWxhbmNlcnMuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGlnaHRzYWlsLzIwMTYtMTEtMjgvYXBpLXJlZmVyZW5jZS9BUElfR2V0TG9hZEJhbGFuY2Vycy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRMb2FkQmFsYW5jZXJzKCkge1xuICAgIHRoaXMuYWRkKCdsaWdodHNhaWw6R2V0TG9hZEJhbGFuY2VycycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgaW5mb3JtYXRpb24gYWJvdXQgYW4gb3BlcmF0aW9uLiBPcGVyYXRpb25zIGluY2x1ZGUgZXZlbnRzIHN1Y2ggYXMgd2hlbiB5b3UgY3JlYXRlIGFuIGluc3RhbmNlLCBhbGxvY2F0ZSBhIHN0YXRpYyBJUCwgYXR0YWNoIGEgc3RhdGljIElQLCBhbmQgc28gb24uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGlnaHRzYWlsLzIwMTYtMTEtMjgvYXBpLXJlZmVyZW5jZS9BUElfR2V0T3BlcmF0aW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldE9wZXJhdGlvbigpIHtcbiAgICB0aGlzLmFkZCgnbGlnaHRzYWlsOkdldE9wZXJhdGlvbicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgaW5mb3JtYXRpb24gYWJvdXQgYWxsIG9wZXJhdGlvbnMuIE9wZXJhdGlvbnMgaW5jbHVkZSBldmVudHMgc3VjaCBhcyB3aGVuIHlvdSBjcmVhdGUgYW4gaW5zdGFuY2UsIGFsbG9jYXRlIGEgc3RhdGljIElQLCBhdHRhY2ggYSBzdGF0aWMgSVAsIGFuZCBzbyBvbi5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9saWdodHNhaWwvMjAxNi0xMS0yOC9hcGktcmVmZXJlbmNlL0FQSV9HZXRPcGVyYXRpb25zLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldE9wZXJhdGlvbnMoKSB7XG4gICAgdGhpcy5hZGQoJ2xpZ2h0c2FpbDpHZXRPcGVyYXRpb25zJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyBvcGVyYXRpb25zIGZvciBhIHJlc291cmNlLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xpZ2h0c2FpbC8yMDE2LTExLTI4L2FwaS1yZWZlcmVuY2UvQVBJX0dldE9wZXJhdGlvbnNGb3JSZXNvdXJjZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRPcGVyYXRpb25zRm9yUmVzb3VyY2UoKSB7XG4gICAgdGhpcy5hZGQoJ2xpZ2h0c2FpbDpHZXRPcGVyYXRpb25zRm9yUmVzb3VyY2UnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIGEgbGlzdCBvZiBhbGwgdmFsaWQgQVdTIFJlZ2lvbnMgZm9yIEFtYXpvbiBMaWdodHNhaWwuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGlnaHRzYWlsLzIwMTYtMTEtMjgvYXBpLXJlZmVyZW5jZS9BUElfR2V0UmVnaW9ucy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRSZWdpb25zKCkge1xuICAgIHRoaXMuYWRkKCdsaWdodHNhaWw6R2V0UmVnaW9ucycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgaW5mb3JtYXRpb24gYWJvdXQgYSByZWxhdGlvbmFsIGRhdGFiYXNlLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xpZ2h0c2FpbC8yMDE2LTExLTI4L2FwaS1yZWZlcmVuY2UvQVBJX0dldFJlbGF0aW9uYWxEYXRhYmFzZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRSZWxhdGlvbmFsRGF0YWJhc2UoKSB7XG4gICAgdGhpcy5hZGQoJ2xpZ2h0c2FpbDpHZXRSZWxhdGlvbmFsRGF0YWJhc2UnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIGEgbGlzdCBvZiByZWxhdGlvbmFsIGRhdGFiYXNlIGltYWdlcywgb3IgYmx1ZXByaW50cy4gWW91IGNhbiB1c2UgYSBibHVlcHJpbnQgdG8gY3JlYXRlIGEgbmV3IGRhdGFiYXNlIHJ1bm5pbmcgYSBzcGVjaWZpYyBkYXRhYmFzZSBlbmdpbmUuIFRoZSBkYXRhYmFzZSBlbmdpbmUgdGhhdCBydW5zIG9uIHlvdXIgZGF0YWJhc2UgZGVwZW5kcyBvbiB0aGUgYmx1ZXByaW50IHlvdSBkZWZpbmUgd2hlbiBjcmVhdGluZyB0aGUgcmVsYXRpb25hbCBkYXRhYmFzZS5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9saWdodHNhaWwvMjAxNi0xMS0yOC9hcGktcmVmZXJlbmNlL0FQSV9HZXRSZWxhdGlvbmFsRGF0YWJhc2VCbHVlcHJpbnRzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldFJlbGF0aW9uYWxEYXRhYmFzZUJsdWVwcmludHMoKSB7XG4gICAgdGhpcy5hZGQoJ2xpZ2h0c2FpbDpHZXRSZWxhdGlvbmFsRGF0YWJhc2VCbHVlcHJpbnRzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyBhIGxpc3Qgb2YgcmVsYXRpb25hbCBkYXRhYmFzZSBidW5kbGVzLiBZb3UgY2FuIHVzZSBhIGJ1bmRsZSB0byBjcmVhdGUgYSBuZXcgZGF0YWJhc2Ugd2l0aCBhIHNldCBvZiBwZXJmb3JtYW5jZSBzcGVjaWZpY2F0aW9ucywgc3VjaCBhcyBDUFUgY291bnQsIGRpc2sgc2l6ZSwgUkFNIHNpemUsIG5ldHdvcmsgdHJhbnNmZXIgYWxsb3dhbmNlLCBhbmQgc3RhbmRhcmQgb2YgaGlnaCBhdmFpbGFiaWxpdHkuIFRoZSBjb3N0IG9mIHlvdXIgZGF0YWJhc2UgZGVwZW5kcyBvbiB0aGUgYnVuZGxlIHlvdSBkZWZpbmUgd2hlbiBjcmVhdGluZyB0aGUgcmVsYXRpb25hbCBkYXRhYmFzZS5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9saWdodHNhaWwvMjAxNi0xMS0yOC9hcGktcmVmZXJlbmNlL0FQSV9HZXRSZWxhdGlvbmFsRGF0YWJhc2VCdW5kbGVzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldFJlbGF0aW9uYWxEYXRhYmFzZUJ1bmRsZXMoKSB7XG4gICAgdGhpcy5hZGQoJ2xpZ2h0c2FpbDpHZXRSZWxhdGlvbmFsRGF0YWJhc2VCdW5kbGVzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyBldmVudHMgZm9yIGEgcmVsYXRpb25hbCBkYXRhYmFzZS5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9saWdodHNhaWwvMjAxNi0xMS0yOC9hcGktcmVmZXJlbmNlL0FQSV9HZXRSZWxhdGlvbmFsRGF0YWJhc2VFdmVudHMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0UmVsYXRpb25hbERhdGFiYXNlRXZlbnRzKCkge1xuICAgIHRoaXMuYWRkKCdsaWdodHNhaWw6R2V0UmVsYXRpb25hbERhdGFiYXNlRXZlbnRzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyBldmVudHMgZm9yIHRoZSBzcGVjaWZpZWQgbG9nIHN0cmVhbSBvZiBhIHJlbGF0aW9uYWwgZGF0YWJhc2UuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGlnaHRzYWlsLzIwMTYtMTEtMjgvYXBpLXJlZmVyZW5jZS9BUElfR2V0UmVsYXRpb25hbERhdGFiYXNlTG9nRXZlbnRzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldFJlbGF0aW9uYWxEYXRhYmFzZUxvZ0V2ZW50cygpIHtcbiAgICB0aGlzLmFkZCgnbGlnaHRzYWlsOkdldFJlbGF0aW9uYWxEYXRhYmFzZUxvZ0V2ZW50cycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgdGhlIGxvZyBzdHJlYW1zIGF2YWlsYWJsZSBmb3IgYSByZWxhdGlvbmFsIGRhdGFiYXNlLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xpZ2h0c2FpbC8yMDE2LTExLTI4L2FwaS1yZWZlcmVuY2UvQVBJX0dldFJlbGF0aW9uYWxEYXRhYmFzZUxvZ1N0cmVhbXMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0UmVsYXRpb25hbERhdGFiYXNlTG9nU3RyZWFtcygpIHtcbiAgICB0aGlzLmFkZCgnbGlnaHRzYWlsOkdldFJlbGF0aW9uYWxEYXRhYmFzZUxvZ1N0cmVhbXMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIHRoZSBtYXN0ZXIgdXNlciBwYXNzd29yZCBvZiBhIHJlbGF0aW9uYWwgZGF0YWJhc2UuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xpZ2h0c2FpbC8yMDE2LTExLTI4L2FwaS1yZWZlcmVuY2UvQVBJX0dldFJlbGF0aW9uYWxEYXRhYmFzZU1hc3RlclVzZXJQYXNzd29yZC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRSZWxhdGlvbmFsRGF0YWJhc2VNYXN0ZXJVc2VyUGFzc3dvcmQoKSB7XG4gICAgdGhpcy5hZGQoJ2xpZ2h0c2FpbDpHZXRSZWxhdGlvbmFsRGF0YWJhc2VNYXN0ZXJVc2VyUGFzc3dvcmQnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIHRoZSBkYXRhIHBvaW50cyBmb3IgdGhlIHNwZWNpZmllZCBtZXRyaWMgb2YgYSByZWxhdGlvbmFsIGRhdGFiYXNlLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xpZ2h0c2FpbC8yMDE2LTExLTI4L2FwaS1yZWZlcmVuY2UvQVBJX0dldFJlbGF0aW9uYWxEYXRhYmFzZU1ldHJpY0RhdGEuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0UmVsYXRpb25hbERhdGFiYXNlTWV0cmljRGF0YSgpIHtcbiAgICB0aGlzLmFkZCgnbGlnaHRzYWlsOkdldFJlbGF0aW9uYWxEYXRhYmFzZU1ldHJpY0RhdGEnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIHRoZSBwYXJhbWV0ZXJzIG9mIGEgcmVsYXRpb25hbCBkYXRhYmFzZS5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9saWdodHNhaWwvMjAxNi0xMS0yOC9hcGktcmVmZXJlbmNlL0FQSV9HZXRSZWxhdGlvbmFsRGF0YWJhc2VQYXJhbWV0ZXJzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldFJlbGF0aW9uYWxEYXRhYmFzZVBhcmFtZXRlcnMoKSB7XG4gICAgdGhpcy5hZGQoJ2xpZ2h0c2FpbDpHZXRSZWxhdGlvbmFsRGF0YWJhc2VQYXJhbWV0ZXJzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyBpbmZvcm1hdGlvbiBhYm91dCBhIHJlbGF0aW9uYWwgZGF0YWJhc2Ugc25hcHNob3QuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGlnaHRzYWlsLzIwMTYtMTEtMjgvYXBpLXJlZmVyZW5jZS9BUElfR2V0UmVsYXRpb25hbERhdGFiYXNlU25hcHNob3QuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0UmVsYXRpb25hbERhdGFiYXNlU25hcHNob3QoKSB7XG4gICAgdGhpcy5hZGQoJ2xpZ2h0c2FpbDpHZXRSZWxhdGlvbmFsRGF0YWJhc2VTbmFwc2hvdCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgaW5mb3JtYXRpb24gYWJvdXQgYWxsIHJlbGF0aW9uYWwgZGF0YWJhc2Ugc25hcHNob3RzLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xpZ2h0c2FpbC8yMDE2LTExLTI4L2FwaS1yZWZlcmVuY2UvQVBJX0dldFJlbGF0aW9uYWxEYXRhYmFzZVNuYXBzaG90cy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRSZWxhdGlvbmFsRGF0YWJhc2VTbmFwc2hvdHMoKSB7XG4gICAgdGhpcy5hZGQoJ2xpZ2h0c2FpbDpHZXRSZWxhdGlvbmFsRGF0YWJhc2VTbmFwc2hvdHMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm4gaW5mb3JtYXRpb24gYWJvdXQgYWxsIHJlbGF0aW9uYWwgZGF0YWJhc2VzLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xpZ2h0c2FpbC8yMDE2LTExLTI4L2FwaS1yZWZlcmVuY2UvQVBJX0dldFJlbGF0aW9uYWxEYXRhYmFzZXMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0UmVsYXRpb25hbERhdGFiYXNlcygpIHtcbiAgICB0aGlzLmFkZCgnbGlnaHRzYWlsOkdldFJlbGF0aW9uYWxEYXRhYmFzZXMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIGluZm9ybWF0aW9uIGFib3V0IGEgc3RhdGljIElQLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xpZ2h0c2FpbC8yMDE2LTExLTI4L2FwaS1yZWZlcmVuY2UvQVBJX0dldFN0YXRpY0lwLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldFN0YXRpY0lwKCkge1xuICAgIHRoaXMuYWRkKCdsaWdodHNhaWw6R2V0U3RhdGljSXAnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIGluZm9ybWF0aW9uIGFib3V0IGFsbCBzdGF0aWMgSVBzLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xpZ2h0c2FpbC8yMDE2LTExLTI4L2FwaS1yZWZlcmVuY2UvQVBJX0dldFN0YXRpY0lwcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRTdGF0aWNJcHMoKSB7XG4gICAgdGhpcy5hZGQoJ2xpZ2h0c2FpbDpHZXRTdGF0aWNJcHMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBJbXBvcnRzIGEgcHVibGljIGtleSBmcm9tIGEga2V5IHBhaXIuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xpZ2h0c2FpbC8yMDE2LTExLTI4L2FwaS1yZWZlcmVuY2UvQVBJX0ltcG9ydEtleVBhaXIuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvSW1wb3J0S2V5UGFpcigpIHtcbiAgICB0aGlzLmFkZCgnbGlnaHRzYWlsOkltcG9ydEtleVBhaXInKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIGEgYm9vbGVhbiB2YWx1ZSBpbmRpY2F0aW5nIHdoZXRoZXIgdGhlIEFtYXpvbiBMaWdodHNhaWwgdmlydHVhbCBwcml2YXRlIGNsb3VkIChWUEMpIGlzIHBlZXJlZC5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9saWdodHNhaWwvMjAxNi0xMS0yOC9hcGktcmVmZXJlbmNlL0FQSV9Jc1ZwY1BlZXJlZC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9Jc1ZwY1BlZXJlZCgpIHtcbiAgICB0aGlzLmFkZCgnbGlnaHRzYWlsOklzVnBjUGVlcmVkJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogQWRkcywgb3Igb3BlbnMgYSBwdWJsaWMgcG9ydCBvZiBhbiBpbnN0YW5jZS5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGlnaHRzYWlsLzIwMTYtMTEtMjgvYXBpLXJlZmVyZW5jZS9BUElfT3Blbkluc3RhbmNlUHVibGljUG9ydHMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvT3Blbkluc3RhbmNlUHVibGljUG9ydHMoKSB7XG4gICAgdGhpcy5hZGQoJ2xpZ2h0c2FpbDpPcGVuSW5zdGFuY2VQdWJsaWNQb3J0cycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFRyaWVzIHRvIHBlZXIgdGhlIEFtYXpvbiBMaWdodHNhaWwgdmlydHVhbCBwcml2YXRlIGNsb3VkIChWUEMpIHdpdGggdGhlIGRlZmF1bHQgVlBDLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9saWdodHNhaWwvMjAxNi0xMS0yOC9hcGktcmVmZXJlbmNlL0FQSV9QZWVyVnBjLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1BlZXJWcGMoKSB7XG4gICAgdGhpcy5hZGQoJ2xpZ2h0c2FpbDpQZWVyVnBjJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogU2V0cyB0aGUgc3BlY2lmaWVkIG9wZW4gcG9ydHMgZm9yIGFuIGluc3RhbmNlLCBhbmQgY2xvc2VzIGFsbCBwb3J0cyBmb3IgZXZlcnkgcHJvdG9jb2wgbm90IGluY2x1ZGVkIGluIHRoZSByZXF1ZXN0LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9saWdodHNhaWwvMjAxNi0xMS0yOC9hcGktcmVmZXJlbmNlL0FQSV9QdXRJbnN0YW5jZVB1YmxpY1BvcnRzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1B1dEluc3RhbmNlUHVibGljUG9ydHMoKSB7XG4gICAgdGhpcy5hZGQoJ2xpZ2h0c2FpbDpQdXRJbnN0YW5jZVB1YmxpY1BvcnRzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmVib290cyBhbiBpbnN0YW5jZSB0aGF0IGlzIGluIGEgcnVubmluZyBzdGF0ZS5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGlnaHRzYWlsLzIwMTYtMTEtMjgvYXBpLXJlZmVyZW5jZS9BUElfUmVib290SW5zdGFuY2UuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvUmVib290SW5zdGFuY2UoKSB7XG4gICAgdGhpcy5hZGQoJ2xpZ2h0c2FpbDpSZWJvb3RJbnN0YW5jZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJlYm9vdHMgYSByZWxhdGlvbmFsIGRhdGFiYXNlIHRoYXQgaXMgaW4gYSBydW5uaW5nIHN0YXRlLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9saWdodHNhaWwvMjAxNi0xMS0yOC9hcGktcmVmZXJlbmNlL0FQSV9SZWJvb3RSZWxhdGlvbmFsRGF0YWJhc2UuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvUmVib290UmVsYXRpb25hbERhdGFiYXNlKCkge1xuICAgIHRoaXMuYWRkKCdsaWdodHNhaWw6UmVib290UmVsYXRpb25hbERhdGFiYXNlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogRGVsZXRlcyBhIHN0YXRpYyBJUC5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGlnaHRzYWlsLzIwMTYtMTEtMjgvYXBpLXJlZmVyZW5jZS9BUElfUmVsZWFzZVN0YXRpY0lwLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1JlbGVhc2VTdGF0aWNJcCgpIHtcbiAgICB0aGlzLmFkZCgnbGlnaHRzYWlsOlJlbGVhc2VTdGF0aWNJcCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFN0YXJ0cyBhbiBpbnN0YW5jZSB0aGF0IGlzIGluIGEgc3RvcHBlZCBzdGF0ZS5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGlnaHRzYWlsLzIwMTYtMTEtMjgvYXBpLXJlZmVyZW5jZS9BUElfU3RhcnRJbnN0YW5jZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9TdGFydEluc3RhbmNlKCkge1xuICAgIHRoaXMuYWRkKCdsaWdodHNhaWw6U3RhcnRJbnN0YW5jZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFN0YXJ0cyBhIHJlbGF0aW9uYWwgZGF0YWJhc2UgdGhhdCBpcyBpbiBhIHN0b3BwZWQgc3RhdGUuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xpZ2h0c2FpbC8yMDE2LTExLTI4L2FwaS1yZWZlcmVuY2UvQVBJX1N0YXJ0UmVsYXRpb25hbERhdGFiYXNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1N0YXJ0UmVsYXRpb25hbERhdGFiYXNlKCkge1xuICAgIHRoaXMuYWRkKCdsaWdodHNhaWw6U3RhcnRSZWxhdGlvbmFsRGF0YWJhc2UnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBTdG9wcyBhbiBpbnN0YW5jZSB0aGF0IGlzIGluIGEgcnVubmluZyBzdGF0ZS5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGlnaHRzYWlsLzIwMTYtMTEtMjgvYXBpLXJlZmVyZW5jZS9BUElfU3RvcEluc3RhbmNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1N0b3BJbnN0YW5jZSgpIHtcbiAgICB0aGlzLmFkZCgnbGlnaHRzYWlsOlN0b3BJbnN0YW5jZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFN0b3BzIGEgcmVsYXRpb25hbCBkYXRhYmFzZSB0aGF0IGlzIGluIGEgcnVubmluZyBzdGF0ZS5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGlnaHRzYWlsLzIwMTYtMTEtMjgvYXBpLXJlZmVyZW5jZS9BUElfU3RvcFJlbGF0aW9uYWxEYXRhYmFzZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9TdG9wUmVsYXRpb25hbERhdGFiYXNlKCkge1xuICAgIHRoaXMuYWRkKCdsaWdodHNhaWw6U3RvcFJlbGF0aW9uYWxEYXRhYmFzZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFRhZ3MgYSByZXNvdXJjZS5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGlnaHRzYWlsLzIwMTYtMTEtMjgvYXBpLXJlZmVyZW5jZS9BUElfVGFnUmVzb3VyY2UuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVGFnUmVzb3VyY2UoKSB7XG4gICAgdGhpcy5hZGQoJ2xpZ2h0c2FpbDpUYWdSZXNvdXJjZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEF0dGVtcHRzIHRvIHVucGVlciB0aGUgQW1hem9uIExpZ2h0c2FpbCB2aXJ0dWFsIHByaXZhdGUgY2xvdWQgKFZQQykgZnJvbSB0aGUgZGVmYXVsdCBWUEMuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xpZ2h0c2FpbC8yMDE2LTExLTI4L2FwaS1yZWZlcmVuY2UvQVBJX1VucGVlclZwYy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VbnBlZXJWcGMoKSB7XG4gICAgdGhpcy5hZGQoJ2xpZ2h0c2FpbDpVbnBlZXJWcGMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBVbnRhZ3MgYSByZXNvdXJjZS5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGlnaHRzYWlsLzIwMTYtMTEtMjgvYXBpLXJlZmVyZW5jZS9BUElfVW50YWdSZXNvdXJjZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VbnRhZ1Jlc291cmNlKCkge1xuICAgIHRoaXMuYWRkKCdsaWdodHNhaWw6VW50YWdSZXNvdXJjZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFVwZGF0ZXMgYSBkb21haW4gcmVjb3Jkc2V0IGFmdGVyIGl0IGlzIGNyZWF0ZWQuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xpZ2h0c2FpbC8yMDE2LTExLTI4L2FwaS1yZWZlcmVuY2UvQVBJX1VwZGF0ZURvbWFpbkVudHJ5Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1VwZGF0ZURvbWFpbkVudHJ5KCkge1xuICAgIHRoaXMuYWRkKCdsaWdodHNhaWw6VXBkYXRlRG9tYWluRW50cnknKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBVcGRhdGVzIGEgbG9hZCBiYWxhbmNlciBhdHRyaWJ1dGUsIHN1Y2ggYXMgdGhlIGhlYWx0aCBjaGVjayBwYXRoIGFuZCBzZXNzaW9uIHN0aWNraW5lc3MuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xpZ2h0c2FpbC8yMDE2LTExLTI4L2FwaS1yZWZlcmVuY2UvQVBJX0RldGFjaEluc3RhbmNlc0Zyb21Mb2FkQmFsYW5jZXIuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlTG9hZEJhbGFuY2VyQXR0cmlidXRlKCkge1xuICAgIHRoaXMuYWRkKCdsaWdodHNhaWw6VXBkYXRlTG9hZEJhbGFuY2VyQXR0cmlidXRlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogVXBkYXRlcyBhIHJlbGF0aW9uYWwgZGF0YWJhc2UuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xpZ2h0c2FpbC8yMDE2LTExLTI4L2FwaS1yZWZlcmVuY2UvQVBJX1VwZGF0ZVJlbGF0aW9uYWxEYXRhYmFzZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VcGRhdGVSZWxhdGlvbmFsRGF0YWJhc2UoKSB7XG4gICAgdGhpcy5hZGQoJ2xpZ2h0c2FpbDpVcGRhdGVSZWxhdGlvbmFsRGF0YWJhc2UnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBVcGRhdGVzIHRoZSBwYXJhbWV0ZXJzIG9mIGEgcmVsYXRpb25hbCBkYXRhYmFzZS5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGlnaHRzYWlsLzIwMTYtMTEtMjgvYXBpLXJlZmVyZW5jZS9BUElfVXBkYXRlUmVsYXRpb25hbERhdGFiYXNlUGFyYW1ldGVycy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VcGRhdGVSZWxhdGlvbmFsRGF0YWJhc2VQYXJhbWV0ZXJzKCkge1xuICAgIHRoaXMuYWRkKCdsaWdodHNhaWw6VXBkYXRlUmVsYXRpb25hbERhdGFiYXNlUGFyYW1ldGVycycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIERvbWFpbiB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9saWdodHNhaWwvMjAxNi0xMS0yOC9hcGktcmVmZXJlbmNlL0FQSV9Eb21haW4uaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gaWQgLSBJZGVudGlmaWVyIGZvciB0aGUgaWQuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIGFjY291bnRzLlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgcmVnaW9ucy5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AuXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uRG9tYWluKGlkOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgdmFyIGFybiA9ICdhcm46JHtQYXJ0aXRpb259OmxpZ2h0c2FpbDoke1JlZ2lvbn06JHtBY2NvdW50fTpEb21haW4vJHtJZH0nO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0lkfScsIGlkKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtBY2NvdW50fScsIGFjY291bnQgfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtSZWdpb259JywgcmVnaW9uIHx8ICcqJyk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UGFydGl0aW9ufScsIHBhcnRpdGlvbiB8fCAnYXdzJyk7XG4gICAgcmV0dXJuIHRoaXMub24oYXJuKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBJbnN0YW5jZSB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9saWdodHNhaWwvMjAxNi0xMS0yOC9hcGktcmVmZXJlbmNlL0FQSV9JbnN0YW5jZS5odG1sXG4gICAqXG4gICAqIEBwYXJhbSBpZCAtIElkZW50aWZpZXIgZm9yIHRoZSBpZC5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgYWNjb3VudHMuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCByZWdpb25zLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYC5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25JbnN0YW5jZShpZDogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHZhciBhcm4gPSAnYXJuOiR7UGFydGl0aW9ufTpsaWdodHNhaWw6JHtSZWdpb259OiR7QWNjb3VudH06SW5zdGFuY2UvJHtJZH0nO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0lkfScsIGlkKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtBY2NvdW50fScsIGFjY291bnQgfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtSZWdpb259JywgcmVnaW9uIHx8ICcqJyk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UGFydGl0aW9ufScsIHBhcnRpdGlvbiB8fCAnYXdzJyk7XG4gICAgcmV0dXJuIHRoaXMub24oYXJuKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBJbnN0YW5jZVNuYXBzaG90IHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xpZ2h0c2FpbC8yMDE2LTExLTI4L2FwaS1yZWZlcmVuY2UvQVBJX0luc3RhbmNlU25hcHNob3QuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gaWQgLSBJZGVudGlmaWVyIGZvciB0aGUgaWQuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIGFjY291bnRzLlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgcmVnaW9ucy5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AuXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uSW5zdGFuY2VTbmFwc2hvdChpZDogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHZhciBhcm4gPSAnYXJuOiR7UGFydGl0aW9ufTpsaWdodHNhaWw6JHtSZWdpb259OiR7QWNjb3VudH06SW5zdGFuY2VTbmFwc2hvdC8ke0lkfSc7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7SWR9JywgaWQpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0FjY291bnR9JywgYWNjb3VudCB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1JlZ2lvbn0nLCByZWdpb24gfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtQYXJ0aXRpb259JywgcGFydGl0aW9uIHx8ICdhd3MnKTtcbiAgICByZXR1cm4gdGhpcy5vbihhcm4pO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIEtleVBhaXIgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGlnaHRzYWlsLzIwMTYtMTEtMjgvYXBpLXJlZmVyZW5jZS9BUElfS2V5UGFpci5odG1sXG4gICAqXG4gICAqIEBwYXJhbSBpZCAtIElkZW50aWZpZXIgZm9yIHRoZSBpZC5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgYWNjb3VudHMuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCByZWdpb25zLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYC5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25LZXlQYWlyKGlkOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgdmFyIGFybiA9ICdhcm46JHtQYXJ0aXRpb259OmxpZ2h0c2FpbDoke1JlZ2lvbn06JHtBY2NvdW50fTpLZXlQYWlyLyR7SWR9JztcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtJZH0nLCBpZCk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7QWNjb3VudH0nLCBhY2NvdW50IHx8ICcqJyk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UmVnaW9ufScsIHJlZ2lvbiB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1BhcnRpdGlvbn0nLCBwYXJ0aXRpb24gfHwgJ2F3cycpO1xuICAgIHJldHVybiB0aGlzLm9uKGFybik7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgU3RhdGljSXAgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGlnaHRzYWlsLzIwMTYtMTEtMjgvYXBpLXJlZmVyZW5jZS9BUElfU3RhdGljSXAuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gaWQgLSBJZGVudGlmaWVyIGZvciB0aGUgaWQuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIGFjY291bnRzLlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgcmVnaW9ucy5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AuXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uU3RhdGljSXAoaWQ6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICB2YXIgYXJuID0gJ2Fybjoke1BhcnRpdGlvbn06bGlnaHRzYWlsOiR7UmVnaW9ufToke0FjY291bnR9OlN0YXRpY0lwLyR7SWR9JztcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtJZH0nLCBpZCk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7QWNjb3VudH0nLCBhY2NvdW50IHx8ICcqJyk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UmVnaW9ufScsIHJlZ2lvbiB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1BhcnRpdGlvbn0nLCBwYXJ0aXRpb24gfHwgJ2F3cycpO1xuICAgIHJldHVybiB0aGlzLm9uKGFybik7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgRGlzayB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9saWdodHNhaWwvMjAxNi0xMS0yOC9hcGktcmVmZXJlbmNlL0FQSV9EaXNrLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIGlkIC0gSWRlbnRpZmllciBmb3IgdGhlIGlkLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCBhY2NvdW50cy5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIHJlZ2lvbnMuXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvbkRpc2soaWQ6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICB2YXIgYXJuID0gJ2Fybjoke1BhcnRpdGlvbn06bGlnaHRzYWlsOiR7UmVnaW9ufToke0FjY291bnR9OkRpc2svJHtJZH0nO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0lkfScsIGlkKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtBY2NvdW50fScsIGFjY291bnQgfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtSZWdpb259JywgcmVnaW9uIHx8ICcqJyk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UGFydGl0aW9ufScsIHBhcnRpdGlvbiB8fCAnYXdzJyk7XG4gICAgcmV0dXJuIHRoaXMub24oYXJuKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBEaXNrU25hcHNob3QgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGlnaHRzYWlsLzIwMTYtMTEtMjgvYXBpLXJlZmVyZW5jZS9BUElfRGlza1NuYXBzaG90Lmh0bWxcbiAgICpcbiAgICogQHBhcmFtIGlkIC0gSWRlbnRpZmllciBmb3IgdGhlIGlkLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCBhY2NvdW50cy5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIHJlZ2lvbnMuXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvbkRpc2tTbmFwc2hvdChpZDogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHZhciBhcm4gPSAnYXJuOiR7UGFydGl0aW9ufTpsaWdodHNhaWw6JHtSZWdpb259OiR7QWNjb3VudH06RGlza1NuYXBzaG90LyR7SWR9JztcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtJZH0nLCBpZCk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7QWNjb3VudH0nLCBhY2NvdW50IHx8ICcqJyk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UmVnaW9ufScsIHJlZ2lvbiB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1BhcnRpdGlvbn0nLCBwYXJ0aXRpb24gfHwgJ2F3cycpO1xuICAgIHJldHVybiB0aGlzLm9uKGFybik7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgTG9hZEJhbGFuY2VyIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xpZ2h0c2FpbC8yMDE2LTExLTI4L2FwaS1yZWZlcmVuY2UvQVBJX0xvYWRCYWxhbmNlci5odG1sXG4gICAqXG4gICAqIEBwYXJhbSBpZCAtIElkZW50aWZpZXIgZm9yIHRoZSBpZC5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgYWNjb3VudHMuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCByZWdpb25zLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYC5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25Mb2FkQmFsYW5jZXIoaWQ6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICB2YXIgYXJuID0gJ2Fybjoke1BhcnRpdGlvbn06bGlnaHRzYWlsOiR7UmVnaW9ufToke0FjY291bnR9OkxvYWRCYWxhbmNlci8ke0lkfSc7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7SWR9JywgaWQpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0FjY291bnR9JywgYWNjb3VudCB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1JlZ2lvbn0nLCByZWdpb24gfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtQYXJ0aXRpb259JywgcGFydGl0aW9uIHx8ICdhd3MnKTtcbiAgICByZXR1cm4gdGhpcy5vbihhcm4pO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIFBlZXJlZFZwYyB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9saWdodHNhaWwvMjAxNi0xMS0yOC9hcGktcmVmZXJlbmNlL1BlZXJlZFZwYy5odG1sXG4gICAqXG4gICAqIEBwYXJhbSBpZCAtIElkZW50aWZpZXIgZm9yIHRoZSBpZC5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgYWNjb3VudHMuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCByZWdpb25zLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYC5cbiAgICovXG4gIHB1YmxpYyBvblBlZXJlZFZwYyhpZDogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHZhciBhcm4gPSAnYXJuOiR7UGFydGl0aW9ufTpsaWdodHNhaWw6JHtSZWdpb259OiR7QWNjb3VudH06UGVlcmVkVnBjLyR7SWR9JztcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtJZH0nLCBpZCk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7QWNjb3VudH0nLCBhY2NvdW50IHx8ICcqJyk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UmVnaW9ufScsIHJlZ2lvbiB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1BhcnRpdGlvbn0nLCBwYXJ0aXRpb24gfHwgJ2F3cycpO1xuICAgIHJldHVybiB0aGlzLm9uKGFybik7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgTG9hZEJhbGFuY2VyVGxzQ2VydGlmaWNhdGUgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbGlnaHRzYWlsLzIwMTYtMTEtMjgvYXBpLXJlZmVyZW5jZS9Mb2FkQmFsYW5jZXJUbHNDZXJ0aWZpY2F0ZS5odG1sXG4gICAqXG4gICAqIEBwYXJhbSBpZCAtIElkZW50aWZpZXIgZm9yIHRoZSBpZC5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgYWNjb3VudHMuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCByZWdpb25zLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYC5cbiAgICovXG4gIHB1YmxpYyBvbkxvYWRCYWxhbmNlclRsc0NlcnRpZmljYXRlKGlkOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgdmFyIGFybiA9ICdhcm46JHtQYXJ0aXRpb259OmxpZ2h0c2FpbDoke1JlZ2lvbn06JHtBY2NvdW50fTpMb2FkQmFsYW5jZXJUbHNDZXJ0aWZpY2F0ZS8ke0lkfSc7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7SWR9JywgaWQpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0FjY291bnR9JywgYWNjb3VudCB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1JlZ2lvbn0nLCByZWdpb24gfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtQYXJ0aXRpb259JywgcGFydGl0aW9uIHx8ICdhd3MnKTtcbiAgICByZXR1cm4gdGhpcy5vbihhcm4pO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIEV4cG9ydFNuYXBzaG90UmVjb3JkIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2xpZ2h0c2FpbC8yMDE2LTExLTI4L2FwaS1yZWZlcmVuY2UvQVBJX0V4cG9ydFNuYXBzaG90UmVjb3JkLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIGlkIC0gSWRlbnRpZmllciBmb3IgdGhlIGlkLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCBhY2NvdW50cy5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIHJlZ2lvbnMuXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLlxuICAgKi9cbiAgcHVibGljIG9uRXhwb3J0U25hcHNob3RSZWNvcmQoaWQ6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICB2YXIgYXJuID0gJ2Fybjoke1BhcnRpdGlvbn06bGlnaHRzYWlsOiR7UmVnaW9ufToke0FjY291bnR9OkV4cG9ydFNuYXBzaG90UmVjb3JkLyR7SWR9JztcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtJZH0nLCBpZCk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7QWNjb3VudH0nLCBhY2NvdW50IHx8ICcqJyk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UmVnaW9ufScsIHJlZ2lvbiB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1BhcnRpdGlvbn0nLCBwYXJ0aXRpb24gfHwgJ2F3cycpO1xuICAgIHJldHVybiB0aGlzLm9uKGFybik7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgQ2xvdWRGb3JtYXRpb25TdGFja1JlY29yZCB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9saWdodHNhaWwvMjAxNi0xMS0yOC9hcGktcmVmZXJlbmNlL0FQSV9DbG91ZEZvcm1hdGlvblN0YWNrUmVjb3JkLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIGlkIC0gSWRlbnRpZmllciBmb3IgdGhlIGlkLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCBhY2NvdW50cy5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIHJlZ2lvbnMuXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLlxuICAgKi9cbiAgcHVibGljIG9uQ2xvdWRGb3JtYXRpb25TdGFja1JlY29yZChpZDogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHZhciBhcm4gPSAnYXJuOiR7UGFydGl0aW9ufTpsaWdodHNhaWw6JHtSZWdpb259OiR7QWNjb3VudH06Q2xvdWRGb3JtYXRpb25TdGFja1JlY29yZC8ke0lkfSc7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7SWR9JywgaWQpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0FjY291bnR9JywgYWNjb3VudCB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1JlZ2lvbn0nLCByZWdpb24gfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtQYXJ0aXRpb259JywgcGFydGl0aW9uIHx8ICdhd3MnKTtcbiAgICByZXR1cm4gdGhpcy5vbihhcm4pO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIFJlbGF0aW9uYWxEYXRhYmFzZSB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9saWdodHNhaWwvMjAxNi0xMS0yOC9hcGktcmVmZXJlbmNlL0FQSV9SZWxhdGlvbmFsRGF0YWJhc2UuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gaWQgLSBJZGVudGlmaWVyIGZvciB0aGUgaWQuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIGFjY291bnRzLlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgcmVnaW9ucy5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AuXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uUmVsYXRpb25hbERhdGFiYXNlKGlkOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgdmFyIGFybiA9ICdhcm46JHtQYXJ0aXRpb259OmxpZ2h0c2FpbDoke1JlZ2lvbn06JHtBY2NvdW50fTpSZWxhdGlvbmFsRGF0YWJhc2UvJHtJZH0nO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0lkfScsIGlkKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtBY2NvdW50fScsIGFjY291bnQgfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtSZWdpb259JywgcmVnaW9uIHx8ICcqJyk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UGFydGl0aW9ufScsIHBhcnRpdGlvbiB8fCAnYXdzJyk7XG4gICAgcmV0dXJuIHRoaXMub24oYXJuKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBSZWxhdGlvbmFsRGF0YWJhc2VTbmFwc2hvdCB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9saWdodHNhaWwvMjAxNi0xMS0yOC9hcGktcmVmZXJlbmNlL0FQSV9SZWxhdGlvbmFsRGF0YWJhc2VTbmFwc2hvdC5odG1sXG4gICAqXG4gICAqIEBwYXJhbSBpZCAtIElkZW50aWZpZXIgZm9yIHRoZSBpZC5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgYWNjb3VudHMuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCByZWdpb25zLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYC5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25SZWxhdGlvbmFsRGF0YWJhc2VTbmFwc2hvdChpZDogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHZhciBhcm4gPSAnYXJuOiR7UGFydGl0aW9ufTpsaWdodHNhaWw6JHtSZWdpb259OiR7QWNjb3VudH06UmVsYXRpb25hbERhdGFiYXNlU25hcHNob3QvJHtJZH0nO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0lkfScsIGlkKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtBY2NvdW50fScsIGFjY291bnQgfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtSZWdpb259JywgcmVnaW9uIHx8ICcqJyk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UGFydGl0aW9ufScsIHBhcnRpdGlvbiB8fCAnYXdzJyk7XG4gICAgcmV0dXJuIHRoaXMub24oYXJuKTtcbiAgfVxufVxuIl19