"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Qldb = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [qldb](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonqldb.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Qldb extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [qldb](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonqldb.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'qldb';
        this.actionList = {
            "CancelJournalKinesisStream": {
                "url": "https://docs.aws.amazon.com/qldb/latest/developerguide/API_CancelJournalKinesisStream.html",
                "description": "Grants permission to cancel a journal kinesis stream",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stream": {
                        "required": true
                    }
                }
            },
            "CreateLedger": {
                "url": "https://docs.aws.amazon.com/qldb/latest/developerguide/API_CreateLedger.html",
                "description": "Grants permission to create a ledger",
                "accessLevel": "Write",
                "resourceTypes": {
                    "ledger": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "DeleteLedger": {
                "url": "https://docs.aws.amazon.com/qldb/latest/developerguide/API_DeleteLedger.html",
                "description": "Grants permission to delete a ledger",
                "accessLevel": "Write",
                "resourceTypes": {
                    "ledger": {
                        "required": true
                    }
                }
            },
            "DescribeJournalKinesisStream": {
                "url": "https://docs.aws.amazon.com/qldb/latest/developerguide/API_DescribeJournalKinesisStream.html",
                "description": "Grants permission to describe information about a journal kinesis stream",
                "accessLevel": "Read",
                "resourceTypes": {
                    "stream": {
                        "required": true
                    }
                }
            },
            "DescribeJournalS3Export": {
                "url": "https://docs.aws.amazon.com/qldb/latest/developerguide/API_DescribeJournalS3Export.html",
                "description": "Grants permission to describe information about a journal export job",
                "accessLevel": "Read",
                "resourceTypes": {
                    "ledger": {
                        "required": true
                    }
                }
            },
            "DescribeLedger": {
                "url": "https://docs.aws.amazon.com/qldb/latest/developerguide/API_DescribeLedger.html",
                "description": "Grants permission to describe a ledger",
                "accessLevel": "Read",
                "resourceTypes": {
                    "ledger": {
                        "required": true
                    }
                }
            },
            "ExecuteStatement": {
                "url": "https://docs.aws.amazon.com/qldb/latest/developerguide/console_QLDB.html",
                "description": "Grants permission to send commands to a ledger via the console",
                "accessLevel": "Write",
                "resourceTypes": {
                    "ledger": {
                        "required": true
                    }
                }
            },
            "ExportJournalToS3": {
                "url": "https://docs.aws.amazon.com/qldb/latest/developerguide/API_ExportJournalToS3.html",
                "description": "Grants permission to export journal contents to an Amazon S3 bucket",
                "accessLevel": "Write",
                "resourceTypes": {
                    "ledger": {
                        "required": true
                    }
                }
            },
            "GetBlock": {
                "url": "https://docs.aws.amazon.com/qldb/latest/developerguide/API_GetBlock.html",
                "description": "Grants permission to retrieve a block from a ledger for a given BlockAddress",
                "accessLevel": "Read",
                "resourceTypes": {
                    "ledger": {
                        "required": true
                    }
                }
            },
            "GetDigest": {
                "url": "https://docs.aws.amazon.com/qldb/latest/developerguide/API_GetDigest.html",
                "description": "Grants permission to retrieve a digest from a ledger for a given BlockAddress",
                "accessLevel": "Read",
                "resourceTypes": {
                    "ledger": {
                        "required": true
                    }
                }
            },
            "GetRevision": {
                "url": "https://docs.aws.amazon.com/qldb/latest/developerguide/API_GetRevision.html",
                "description": "Grants permission to retrieve a revision for a given document ID and a given BlockAddress",
                "accessLevel": "Read",
                "resourceTypes": {
                    "ledger": {
                        "required": true
                    }
                }
            },
            "InsertSampleData": {
                "url": "https://docs.aws.amazon.com/qldb/latest/developerguide/console_QLDB.html",
                "description": "Grants permission to insert sample application data via the console",
                "accessLevel": "Write",
                "resourceTypes": {
                    "ledger": {
                        "required": true
                    }
                }
            },
            "ListJournalKinesisStreamsForLedger": {
                "url": "https://docs.aws.amazon.com/qldb/latest/developerguide/API_ListJournalKinesisStreamsForLedger.html",
                "description": "Grants permission to list journal kinesis streams for a specified ledger",
                "accessLevel": "List",
                "resourceTypes": {
                    "stream": {
                        "required": true
                    }
                }
            },
            "ListJournalS3Exports": {
                "url": "https://docs.aws.amazon.com/qldb/latest/developerguide/API_ListJournalS3Exports.html",
                "description": "Grants permission to list journal export jobs for all ledgers",
                "accessLevel": "List"
            },
            "ListJournalS3ExportsForLedger": {
                "url": "https://docs.aws.amazon.com/qldb/latest/developerguide/API_ListJournalS3ExportsForLedger.html",
                "description": "Grants permission to list journal export jobs for a specified ledger",
                "accessLevel": "List",
                "resourceTypes": {
                    "ledger": {
                        "required": true
                    }
                }
            },
            "ListLedgers": {
                "url": "https://docs.aws.amazon.com/qldb/latest/developerguide/API_ListLedgers.html",
                "description": "Grants permission to list existing ledgers",
                "accessLevel": "List"
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/qldb/latest/developerguide/API_ListTagsForResource.html",
                "description": "Grants permission to list tags for a resource",
                "accessLevel": "Read",
                "resourceTypes": {
                    "ledger": {
                        "required": false
                    }
                }
            },
            "SendCommand": {
                "url": "https://docs.aws.amazon.com/qldb/latest/developerguide/Using.API.html",
                "description": "Grants permission to send commands to a ledger",
                "accessLevel": "Write",
                "resourceTypes": {
                    "ledger": {
                        "required": true
                    }
                }
            },
            "ShowCatalog": {
                "url": "https://docs.aws.amazon.com/qldb/latest/developerguide/console_QLDB.html",
                "description": "Grants permission to view a ledger's catalog via the console",
                "accessLevel": "Write",
                "resourceTypes": {
                    "ledger": {
                        "required": true
                    }
                }
            },
            "StreamJournalToKinesis": {
                "url": "https://docs.aws.amazon.com/qldb/latest/developerguide/API_StreamJournalToKinesis.html",
                "description": "Grants permission to stream journal contents to a Kinesis Data Stream",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stream": {
                        "required": true
                    }
                }
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/qldb/latest/developerguide/API_TagResource.html",
                "description": "Grants permission to add one or more tags to a resource",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "ledger": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/qldb/latest/developerguide/API_UntagResource.html",
                "description": "Grants permission to remove one or more tags to a resource",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "ledger": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "UpdateLedger": {
                "url": "https://docs.aws.amazon.com/qldb/latest/developerguide/API_UpdateLedger.html",
                "description": "Grants permission to update properties on a ledger",
                "accessLevel": "Write",
                "resourceTypes": {
                    "ledger": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "ledger": {
                "name": "ledger",
                "url": "https://docs.aws.amazon.com/qldb/latest/developerguide/what-is.html",
                "arn": "arn:${Partition}:qldb:${Region}:${Account}:ledger/${LedgerName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "stream": {
                "name": "stream",
                "url": "https://docs.aws.amazon.com/qldb/latest/developerguide/what-is.html",
                "arn": "arn:${Partition}:qldb:${Region}:${Account}:stream/${LedgerName}/${StreamId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Grants permission to cancel a journal kinesis stream
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/API_CancelJournalKinesisStream.html
     */
    toCancelJournalKinesisStream() {
        this.add('qldb:CancelJournalKinesisStream');
        return this;
    }
    /**
     * Grants permission to create a ledger
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/API_CreateLedger.html
     */
    toCreateLedger() {
        this.add('qldb:CreateLedger');
        return this;
    }
    /**
     * Grants permission to delete a ledger
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/API_DeleteLedger.html
     */
    toDeleteLedger() {
        this.add('qldb:DeleteLedger');
        return this;
    }
    /**
     * Grants permission to describe information about a journal kinesis stream
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/API_DescribeJournalKinesisStream.html
     */
    toDescribeJournalKinesisStream() {
        this.add('qldb:DescribeJournalKinesisStream');
        return this;
    }
    /**
     * Grants permission to describe information about a journal export job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/API_DescribeJournalS3Export.html
     */
    toDescribeJournalS3Export() {
        this.add('qldb:DescribeJournalS3Export');
        return this;
    }
    /**
     * Grants permission to describe a ledger
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/API_DescribeLedger.html
     */
    toDescribeLedger() {
        this.add('qldb:DescribeLedger');
        return this;
    }
    /**
     * Grants permission to send commands to a ledger via the console
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/console_QLDB.html
     */
    toExecuteStatement() {
        this.add('qldb:ExecuteStatement');
        return this;
    }
    /**
     * Grants permission to export journal contents to an Amazon S3 bucket
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/API_ExportJournalToS3.html
     */
    toExportJournalToS3() {
        this.add('qldb:ExportJournalToS3');
        return this;
    }
    /**
     * Grants permission to retrieve a block from a ledger for a given BlockAddress
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/API_GetBlock.html
     */
    toGetBlock() {
        this.add('qldb:GetBlock');
        return this;
    }
    /**
     * Grants permission to retrieve a digest from a ledger for a given BlockAddress
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/API_GetDigest.html
     */
    toGetDigest() {
        this.add('qldb:GetDigest');
        return this;
    }
    /**
     * Grants permission to retrieve a revision for a given document ID and a given BlockAddress
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/API_GetRevision.html
     */
    toGetRevision() {
        this.add('qldb:GetRevision');
        return this;
    }
    /**
     * Grants permission to insert sample application data via the console
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/console_QLDB.html
     */
    toInsertSampleData() {
        this.add('qldb:InsertSampleData');
        return this;
    }
    /**
     * Grants permission to list journal kinesis streams for a specified ledger
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/API_ListJournalKinesisStreamsForLedger.html
     */
    toListJournalKinesisStreamsForLedger() {
        this.add('qldb:ListJournalKinesisStreamsForLedger');
        return this;
    }
    /**
     * Grants permission to list journal export jobs for all ledgers
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/API_ListJournalS3Exports.html
     */
    toListJournalS3Exports() {
        this.add('qldb:ListJournalS3Exports');
        return this;
    }
    /**
     * Grants permission to list journal export jobs for a specified ledger
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/API_ListJournalS3ExportsForLedger.html
     */
    toListJournalS3ExportsForLedger() {
        this.add('qldb:ListJournalS3ExportsForLedger');
        return this;
    }
    /**
     * Grants permission to list existing ledgers
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/API_ListLedgers.html
     */
    toListLedgers() {
        this.add('qldb:ListLedgers');
        return this;
    }
    /**
     * Grants permission to list tags for a resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.add('qldb:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to send commands to a ledger
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/Using.API.html
     */
    toSendCommand() {
        this.add('qldb:SendCommand');
        return this;
    }
    /**
     * Grants permission to view a ledger's catalog via the console
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/console_QLDB.html
     */
    toShowCatalog() {
        this.add('qldb:ShowCatalog');
        return this;
    }
    /**
     * Grants permission to stream journal contents to a Kinesis Data Stream
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/API_StreamJournalToKinesis.html
     */
    toStreamJournalToKinesis() {
        this.add('qldb:StreamJournalToKinesis');
        return this;
    }
    /**
     * Grants permission to add one or more tags to a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/API_TagResource.html
     */
    toTagResource() {
        this.add('qldb:TagResource');
        return this;
    }
    /**
     * Grants permission to remove one or more tags to a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/API_UntagResource.html
     */
    toUntagResource() {
        this.add('qldb:UntagResource');
        return this;
    }
    /**
     * Grants permission to update properties on a ledger
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/API_UpdateLedger.html
     */
    toUpdateLedger() {
        this.add('qldb:UpdateLedger');
        return this;
    }
    /**
     * Adds a resource of type ledger to the statement
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/what-is.html
     *
     * @param ledgerName - Identifier for the ledgerName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onLedger(ledgerName, account, region, partition) {
        var arn = 'arn:${Partition}:qldb:${Region}:${Account}:ledger/${LedgerName}';
        arn = arn.replace('${LedgerName}', ledgerName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type stream to the statement
     *
     * https://docs.aws.amazon.com/qldb/latest/developerguide/what-is.html
     *
     * @param ledgerName - Identifier for the ledgerName.
     * @param streamId - Identifier for the streamId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onStream(ledgerName, streamId, account, region, partition) {
        var arn = 'arn:${Partition}:qldb:${Region}:${Account}:stream/${LedgerName}/${StreamId}';
        arn = arn.replace('${LedgerName}', ledgerName);
        arn = arn.replace('${StreamId}', streamId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Qldb = Qldb;
//# sourceMappingURL=data:application/json;base64,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