"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Route53 = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [route53](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonroute53.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Route53 extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [route53](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonroute53.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'route53';
        this.actionList = {
            "AssociateVPCWithHostedZone": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_AssociateVPCWithHostedZone.html",
                "description": "Grants permission to associate an additional Amazon VPC with a private hosted zone",
                "accessLevel": "Write",
                "resourceTypes": {
                    "hostedzone": {
                        "required": true
                    }
                }
            },
            "ChangeResourceRecordSets": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_ChangeResourceRecordSets.html",
                "description": "Grants permission to create, update, or delete a record, which contains authoritative DNS information for a specified domain or subdomain name",
                "accessLevel": "Write",
                "resourceTypes": {
                    "hostedzone": {
                        "required": true
                    }
                }
            },
            "ChangeTagsForResource": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_ChangeTagsForResource.html",
                "description": "Grants permission to add, edit, or delete tags for a health check or a hosted zone",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "healthcheck": {
                        "required": true
                    },
                    "hostedzone": {
                        "required": true
                    }
                }
            },
            "CreateHealthCheck": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_CreateHealthCheck.html",
                "description": "Grants permission to create a new health check, which monitors the health and performance of your web applications, web servers, and other resources",
                "accessLevel": "Write"
            },
            "CreateHostedZone": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_CreateHostedZone.html",
                "description": "Grants permission to create a public hosted zone, which you use to specify how the Domain Name System (DNS) routes traffic on the Internet for a domain, such as example.com, and its subdomains",
                "accessLevel": "Write"
            },
            "CreateQueryLoggingConfig": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_CreateQueryLoggingConfig.html",
                "description": "Grants permission to create a configuration for DNS query logging",
                "accessLevel": "Write",
                "resourceTypes": {
                    "hostedzone": {
                        "required": true
                    }
                }
            },
            "CreateReusableDelegationSet": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_CreateReusableDelegationSet.html",
                "description": "Grants permission to create a delegation set (a group of four name servers) that can be reused by multiple hosted zones",
                "accessLevel": "Write"
            },
            "CreateTrafficPolicy": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_CreateTrafficPolicy.html",
                "description": "Grants permission to create a traffic policy, which you use to create multiple DNS records for one domain name (such as example.com) or one subdomain name (such as www.example.com)",
                "accessLevel": "Write"
            },
            "CreateTrafficPolicyInstance": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_CreateTrafficPolicyInstance.html",
                "description": "Grants permission to create records in a specified hosted zone based on the settings in a specified traffic policy version",
                "accessLevel": "Write",
                "resourceTypes": {
                    "hostedzone": {
                        "required": true
                    },
                    "trafficpolicy": {
                        "required": true
                    }
                }
            },
            "CreateTrafficPolicyVersion": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_CreateTrafficPolicyVersion.html",
                "description": "Grants permission to create a new version of an existing traffic policy",
                "accessLevel": "Write",
                "resourceTypes": {
                    "trafficpolicy": {
                        "required": true
                    }
                }
            },
            "CreateVPCAssociationAuthorization": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_CreateVPCAssociationAuthorization.html",
                "description": "Grants permission to authorize the AWS account that created a specified VPC to submit an AssociateVPCWithHostedZone request, which associates the VPC with a specified hosted zone that was created by a different account",
                "accessLevel": "Write",
                "resourceTypes": {
                    "hostedzone": {
                        "required": true
                    }
                }
            },
            "DeleteHealthCheck": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_DeleteHealthCheck.html",
                "description": "Grants permission to delete a health check",
                "accessLevel": "Write",
                "resourceTypes": {
                    "healthcheck": {
                        "required": true
                    }
                }
            },
            "DeleteHostedZone": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_DeleteHostedZone.html",
                "description": "Grants permission to delete a hosted zone",
                "accessLevel": "Write",
                "resourceTypes": {
                    "hostedzone": {
                        "required": true
                    }
                }
            },
            "DeleteQueryLoggingConfig": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_DeleteQueryLoggingConfig.html",
                "description": "Grants permission to delete a configuration for DNS query logging",
                "accessLevel": "Write",
                "resourceTypes": {
                    "queryloggingconfig": {
                        "required": true
                    }
                }
            },
            "DeleteReusableDelegationSet": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_DeleteReusableDelegationSet.html",
                "description": "Grants permission to delete a reusable delegation set",
                "accessLevel": "Write",
                "resourceTypes": {
                    "delegationset": {
                        "required": true
                    }
                }
            },
            "DeleteTrafficPolicy": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_DeleteTrafficPolicy.html",
                "description": "Grants permission to delete a traffic policy",
                "accessLevel": "Write",
                "resourceTypes": {
                    "trafficpolicy": {
                        "required": true
                    }
                }
            },
            "DeleteTrafficPolicyInstance": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_DeleteTrafficPolicyInstance.html",
                "description": "Grants permission to delete a traffic policy instance and all the records that Route 53 created when you created the instance",
                "accessLevel": "Write",
                "resourceTypes": {
                    "trafficpolicyinstance": {
                        "required": true
                    }
                }
            },
            "DeleteVPCAssociationAuthorization": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_DeleteVPCAssociationAuthorization.html",
                "description": "Grants permission to remove authorization for associating an Amazon Virtual Private Cloud with a Route 53 private hosted zone",
                "accessLevel": "Write",
                "resourceTypes": {
                    "hostedzone": {
                        "required": true
                    }
                }
            },
            "DisassociateVPCFromHostedZone": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_DisassociateVPCFromHostedZone.html",
                "description": "Grants permission to disassociate an Amazon Virtual Private Cloud from a Route 53 private hosted zone",
                "accessLevel": "Write"
            },
            "GetAccountLimit": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetAccountLimit.html",
                "description": "Grants permission to get the specified limit for the current account, for example, the maximum number of health checks that you can create using the account",
                "accessLevel": "Read"
            },
            "GetChange": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetChange.html",
                "description": "Grants permission to get the current status of a request to create, update, or delete one or more records",
                "accessLevel": "List",
                "resourceTypes": {
                    "change": {
                        "required": true
                    }
                }
            },
            "GetCheckerIpRanges": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetCheckerIpRanges.html",
                "description": "Grants permission to get a list of the IP ranges that are used by Route 53 health checkers to check the health of your resources",
                "accessLevel": "List"
            },
            "GetGeoLocation": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetGeoLocation.html",
                "description": "Grants permission to get information about whether a specified geographic location is supported for Route 53 geolocation records",
                "accessLevel": "List"
            },
            "GetHealthCheck": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetHealthCheck.html",
                "description": "Grants permission to get information about a specified health check",
                "accessLevel": "Read",
                "resourceTypes": {
                    "healthcheck": {
                        "required": true
                    }
                }
            },
            "GetHealthCheckCount": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetHealthCheckCount.html",
                "description": "Grants permission to get the number of health checks that are associated with the current AWS account",
                "accessLevel": "List"
            },
            "GetHealthCheckLastFailureReason": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetHealthCheckLastFailureReason.html",
                "description": "Grants permission to get the reason that a specified health check failed most recently",
                "accessLevel": "List",
                "resourceTypes": {
                    "healthcheck": {
                        "required": true
                    }
                }
            },
            "GetHealthCheckStatus": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetHealthCheckStatus.html",
                "description": "Grants permission to get the status of a specified health check",
                "accessLevel": "List",
                "resourceTypes": {
                    "healthcheck": {
                        "required": true
                    }
                }
            },
            "GetHostedZone": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetHostedZone.html",
                "description": "Grants permission to get information about a specified hosted zone including the four name servers that Route 53 assigned to the hosted zone",
                "accessLevel": "List",
                "resourceTypes": {
                    "hostedzone": {
                        "required": true
                    }
                }
            },
            "GetHostedZoneCount": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetHostedZoneCount.html",
                "description": "Grants permission to get the number of hosted zones that are associated with the current AWS account",
                "accessLevel": "List"
            },
            "GetHostedZoneLimit": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetHostedZoneLimit.html",
                "description": "Grants permission to get the specified limit for a specified hosted zone",
                "accessLevel": "Read",
                "resourceTypes": {
                    "hostedzone": {
                        "required": true
                    }
                }
            },
            "GetQueryLoggingConfig": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetQueryLoggingConfig.html",
                "description": "Grants permission to get information about a specified configuration for DNS query logging",
                "accessLevel": "Read",
                "resourceTypes": {
                    "queryloggingconfig": {
                        "required": true
                    }
                }
            },
            "GetReusableDelegationSet": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetReusableDelegationSet.html",
                "description": "Grants permission to get information about a specified reusable delegation set, including the four name servers that are assigned to the delegation set",
                "accessLevel": "List",
                "resourceTypes": {
                    "delegationset": {
                        "required": true
                    }
                }
            },
            "GetReusableDelegationSetLimit": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetReusableDelegationSetLimit.html",
                "description": "Grants permission to get the maximum number of hosted zones that you can associate with the specified reusable delegation set",
                "accessLevel": "Read",
                "resourceTypes": {
                    "delegationset": {
                        "required": true
                    }
                }
            },
            "GetTrafficPolicy": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetTrafficPolicy.html",
                "description": "Grants permission to get information about a specified traffic policy version",
                "accessLevel": "Read",
                "resourceTypes": {
                    "trafficpolicy": {
                        "required": true
                    }
                }
            },
            "GetTrafficPolicyInstance": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetTrafficPolicyInstance.html",
                "description": "Grants permission to get information about a specified traffic policy instance",
                "accessLevel": "Read",
                "resourceTypes": {
                    "trafficpolicyinstance": {
                        "required": true
                    }
                }
            },
            "GetTrafficPolicyInstanceCount": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetTrafficPolicyInstanceCount.html",
                "description": "Grants permission to get the number of traffic policy instances that are associated with the current AWS account",
                "accessLevel": "Read"
            },
            "ListGeoLocations": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListGeoLocations.html",
                "description": "Grants permission to get a list of geographic locations that Route 53 supports for geolocation",
                "accessLevel": "List"
            },
            "ListHealthChecks": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListHealthChecks.html",
                "description": "Grants permission to get a list of the health checks that are associated with the current AWS account",
                "accessLevel": "List"
            },
            "ListHostedZones": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListHostedZones.html",
                "description": "Grants permission to get a list of the public and private hosted zones that are associated with the current AWS account",
                "accessLevel": "List"
            },
            "ListHostedZonesByName": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListHostedZonesByName.html",
                "description": "Grants permission to get a list of your hosted zones in lexicographic order. Hosted zones are sorted by name with the labels reversed, for example, com.example.www.",
                "accessLevel": "List"
            },
            "ListQueryLoggingConfigs": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListQueryLoggingConfigs.html",
                "description": "Grants permission to list the configurations for DNS query logging that are associated with the current AWS account or the configuration that is associated with a specified hosted zone.",
                "accessLevel": "List",
                "resourceTypes": {
                    "hostedzone": {
                        "required": false
                    }
                }
            },
            "ListResourceRecordSets": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListResourceRecordSets.html",
                "description": "Grants permission to list the records in a specified hosted zone",
                "accessLevel": "List",
                "resourceTypes": {
                    "hostedzone": {
                        "required": true
                    }
                }
            },
            "ListReusableDelegationSets": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListReusableDelegationSets.html",
                "description": "Grants permission to list the reusable delegation sets that are associated with the current AWS account.",
                "accessLevel": "List"
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListTagsForResource.html",
                "description": "Grants permission to list tags for one health check or hosted zone",
                "accessLevel": "Read",
                "resourceTypes": {
                    "healthcheck": {
                        "required": false
                    },
                    "hostedzone": {
                        "required": false
                    }
                }
            },
            "ListTagsForResources": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListTagsForResources.html",
                "description": "Grants permission to list tags for up to 10 health checks or hosted zones",
                "accessLevel": "Read",
                "resourceTypes": {
                    "healthcheck": {
                        "required": false
                    },
                    "hostedzone": {
                        "required": false
                    }
                }
            },
            "ListTrafficPolicies": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListTrafficPolicies.html",
                "description": "Grants permission to get information about the latest version for every traffic policy that is associated with the current AWS account. Policies are listed in the order in which they were created.",
                "accessLevel": "Read"
            },
            "ListTrafficPolicyInstances": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListTrafficPolicyInstances.html",
                "description": "Grants permission to get information about the traffic policy instances that you created by using the current AWS account",
                "accessLevel": "Read"
            },
            "ListTrafficPolicyInstancesByHostedZone": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListTrafficPolicyInstancesByHostedZone.html",
                "description": "Grants permission to get information about the traffic policy instances that you created in a specified hosted zone",
                "accessLevel": "Read",
                "resourceTypes": {
                    "hostedzone": {
                        "required": true
                    }
                }
            },
            "ListTrafficPolicyInstancesByPolicy": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListTrafficPolicyInstancesByPolicy.html",
                "description": "Grants permission to get information about the traffic policy instances that you created using a specified traffic policy version",
                "accessLevel": "Read",
                "resourceTypes": {
                    "trafficpolicy": {
                        "required": true
                    }
                }
            },
            "ListTrafficPolicyVersions": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListTrafficPolicyVersions.html",
                "description": "Grants permission to get information about all the versions for a specified traffic policy",
                "accessLevel": "Read",
                "resourceTypes": {
                    "trafficpolicy": {
                        "required": true
                    }
                }
            },
            "ListVPCAssociationAuthorizations": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListVPCAssociationAuthorizations.html",
                "description": "Grants permission to get a list of the VPCs that were created by other accounts and that can be associated with a specified hosted zone",
                "accessLevel": "Read",
                "resourceTypes": {
                    "hostedzone": {
                        "required": true
                    }
                }
            },
            "TestDNSAnswer": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_TestDNSAnswer.html",
                "description": "Grants permission to get the value that Route 53 returns in response to a DNS query for a specified record name and type",
                "accessLevel": "Read"
            },
            "UpdateHealthCheck": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_UpdateHealthCheck.html",
                "description": "Grants permission to update an existing health check",
                "accessLevel": "Write",
                "resourceTypes": {
                    "healthcheck": {
                        "required": true
                    }
                }
            },
            "UpdateHostedZoneComment": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_UpdateHostedZoneComment.html",
                "description": "Grants permission to update the comment for a specified hosted zone",
                "accessLevel": "Write",
                "resourceTypes": {
                    "hostedzone": {
                        "required": true
                    }
                }
            },
            "UpdateTrafficPolicyComment": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_UpdateTrafficPolicyComment.html",
                "description": "Grants permission to update the comment for a specified traffic policy version",
                "accessLevel": "Write",
                "resourceTypes": {
                    "trafficpolicy": {
                        "required": true
                    }
                }
            },
            "UpdateTrafficPolicyInstance": {
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_UpdateTrafficPolicyInstance.html",
                "description": "Grants permission to update the records in a specified hosted zone that were created based on the settings in a specified traffic policy version",
                "accessLevel": "Write",
                "resourceTypes": {
                    "trafficpolicyinstance": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "change": {
                "name": "change",
                "url": "https://docs.aws.amazon.com/Route53/latest/APIReference/API_Change.html",
                "arn": "arn:${Partition}:route53:::change/${Id}",
                "conditionKeys": []
            },
            "delegationset": {
                "name": "delegationset",
                "url": "https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/route-53-concepts.html#route-53-concepts-reusable-delegation-set",
                "arn": "arn:${Partition}:route53:::delegationset/${Id}",
                "conditionKeys": []
            },
            "healthcheck": {
                "name": "healthcheck",
                "url": "https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/route-53-concepts.html#route-53-concepts-health-check",
                "arn": "arn:${Partition}:route53:::healthcheck/${Id}",
                "conditionKeys": []
            },
            "hostedzone": {
                "name": "hostedzone",
                "url": "https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/route-53-concepts.html#route-53-concepts-hosted-zone",
                "arn": "arn:${Partition}:route53:::hostedzone/${Id}",
                "conditionKeys": []
            },
            "trafficpolicy": {
                "name": "trafficpolicy",
                "url": "https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/traffic-policies.html",
                "arn": "arn:${Partition}:route53:::trafficpolicy/${Id}",
                "conditionKeys": []
            },
            "trafficpolicyinstance": {
                "name": "trafficpolicyinstance",
                "url": "https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/traffic-policy-records.html",
                "arn": "arn:${Partition}:route53:::trafficpolicyinstance/${Id}",
                "conditionKeys": []
            },
            "queryloggingconfig": {
                "name": "queryloggingconfig",
                "url": "https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/query-logs.html",
                "arn": "arn:${Partition}:route53:::queryloggingconfig/${Id}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Grants permission to associate an additional Amazon VPC with a private hosted zone
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_AssociateVPCWithHostedZone.html
     */
    toAssociateVPCWithHostedZone() {
        this.add('route53:AssociateVPCWithHostedZone');
        return this;
    }
    /**
     * Grants permission to create, update, or delete a record, which contains authoritative DNS information for a specified domain or subdomain name
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_ChangeResourceRecordSets.html
     */
    toChangeResourceRecordSets() {
        this.add('route53:ChangeResourceRecordSets');
        return this;
    }
    /**
     * Grants permission to add, edit, or delete tags for a health check or a hosted zone
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_ChangeTagsForResource.html
     */
    toChangeTagsForResource() {
        this.add('route53:ChangeTagsForResource');
        return this;
    }
    /**
     * Grants permission to create a new health check, which monitors the health and performance of your web applications, web servers, and other resources
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_CreateHealthCheck.html
     */
    toCreateHealthCheck() {
        this.add('route53:CreateHealthCheck');
        return this;
    }
    /**
     * Grants permission to create a public hosted zone, which you use to specify how the Domain Name System (DNS) routes traffic on the Internet for a domain, such as example.com, and its subdomains
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_CreateHostedZone.html
     */
    toCreateHostedZone() {
        this.add('route53:CreateHostedZone');
        return this;
    }
    /**
     * Grants permission to create a configuration for DNS query logging
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_CreateQueryLoggingConfig.html
     */
    toCreateQueryLoggingConfig() {
        this.add('route53:CreateQueryLoggingConfig');
        return this;
    }
    /**
     * Grants permission to create a delegation set (a group of four name servers) that can be reused by multiple hosted zones
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_CreateReusableDelegationSet.html
     */
    toCreateReusableDelegationSet() {
        this.add('route53:CreateReusableDelegationSet');
        return this;
    }
    /**
     * Grants permission to create a traffic policy, which you use to create multiple DNS records for one domain name (such as example.com) or one subdomain name (such as www.example.com)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_CreateTrafficPolicy.html
     */
    toCreateTrafficPolicy() {
        this.add('route53:CreateTrafficPolicy');
        return this;
    }
    /**
     * Grants permission to create records in a specified hosted zone based on the settings in a specified traffic policy version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_CreateTrafficPolicyInstance.html
     */
    toCreateTrafficPolicyInstance() {
        this.add('route53:CreateTrafficPolicyInstance');
        return this;
    }
    /**
     * Grants permission to create a new version of an existing traffic policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_CreateTrafficPolicyVersion.html
     */
    toCreateTrafficPolicyVersion() {
        this.add('route53:CreateTrafficPolicyVersion');
        return this;
    }
    /**
     * Grants permission to authorize the AWS account that created a specified VPC to submit an AssociateVPCWithHostedZone request, which associates the VPC with a specified hosted zone that was created by a different account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_CreateVPCAssociationAuthorization.html
     */
    toCreateVPCAssociationAuthorization() {
        this.add('route53:CreateVPCAssociationAuthorization');
        return this;
    }
    /**
     * Grants permission to delete a health check
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_DeleteHealthCheck.html
     */
    toDeleteHealthCheck() {
        this.add('route53:DeleteHealthCheck');
        return this;
    }
    /**
     * Grants permission to delete a hosted zone
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_DeleteHostedZone.html
     */
    toDeleteHostedZone() {
        this.add('route53:DeleteHostedZone');
        return this;
    }
    /**
     * Grants permission to delete a configuration for DNS query logging
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_DeleteQueryLoggingConfig.html
     */
    toDeleteQueryLoggingConfig() {
        this.add('route53:DeleteQueryLoggingConfig');
        return this;
    }
    /**
     * Grants permission to delete a reusable delegation set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_DeleteReusableDelegationSet.html
     */
    toDeleteReusableDelegationSet() {
        this.add('route53:DeleteReusableDelegationSet');
        return this;
    }
    /**
     * Grants permission to delete a traffic policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_DeleteTrafficPolicy.html
     */
    toDeleteTrafficPolicy() {
        this.add('route53:DeleteTrafficPolicy');
        return this;
    }
    /**
     * Grants permission to delete a traffic policy instance and all the records that Route 53 created when you created the instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_DeleteTrafficPolicyInstance.html
     */
    toDeleteTrafficPolicyInstance() {
        this.add('route53:DeleteTrafficPolicyInstance');
        return this;
    }
    /**
     * Grants permission to remove authorization for associating an Amazon Virtual Private Cloud with a Route 53 private hosted zone
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_DeleteVPCAssociationAuthorization.html
     */
    toDeleteVPCAssociationAuthorization() {
        this.add('route53:DeleteVPCAssociationAuthorization');
        return this;
    }
    /**
     * Grants permission to disassociate an Amazon Virtual Private Cloud from a Route 53 private hosted zone
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_DisassociateVPCFromHostedZone.html
     */
    toDisassociateVPCFromHostedZone() {
        this.add('route53:DisassociateVPCFromHostedZone');
        return this;
    }
    /**
     * Grants permission to get the specified limit for the current account, for example, the maximum number of health checks that you can create using the account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetAccountLimit.html
     */
    toGetAccountLimit() {
        this.add('route53:GetAccountLimit');
        return this;
    }
    /**
     * Grants permission to get the current status of a request to create, update, or delete one or more records
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetChange.html
     */
    toGetChange() {
        this.add('route53:GetChange');
        return this;
    }
    /**
     * Grants permission to get a list of the IP ranges that are used by Route 53 health checkers to check the health of your resources
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetCheckerIpRanges.html
     */
    toGetCheckerIpRanges() {
        this.add('route53:GetCheckerIpRanges');
        return this;
    }
    /**
     * Grants permission to get information about whether a specified geographic location is supported for Route 53 geolocation records
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetGeoLocation.html
     */
    toGetGeoLocation() {
        this.add('route53:GetGeoLocation');
        return this;
    }
    /**
     * Grants permission to get information about a specified health check
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetHealthCheck.html
     */
    toGetHealthCheck() {
        this.add('route53:GetHealthCheck');
        return this;
    }
    /**
     * Grants permission to get the number of health checks that are associated with the current AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetHealthCheckCount.html
     */
    toGetHealthCheckCount() {
        this.add('route53:GetHealthCheckCount');
        return this;
    }
    /**
     * Grants permission to get the reason that a specified health check failed most recently
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetHealthCheckLastFailureReason.html
     */
    toGetHealthCheckLastFailureReason() {
        this.add('route53:GetHealthCheckLastFailureReason');
        return this;
    }
    /**
     * Grants permission to get the status of a specified health check
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetHealthCheckStatus.html
     */
    toGetHealthCheckStatus() {
        this.add('route53:GetHealthCheckStatus');
        return this;
    }
    /**
     * Grants permission to get information about a specified hosted zone including the four name servers that Route 53 assigned to the hosted zone
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetHostedZone.html
     */
    toGetHostedZone() {
        this.add('route53:GetHostedZone');
        return this;
    }
    /**
     * Grants permission to get the number of hosted zones that are associated with the current AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetHostedZoneCount.html
     */
    toGetHostedZoneCount() {
        this.add('route53:GetHostedZoneCount');
        return this;
    }
    /**
     * Grants permission to get the specified limit for a specified hosted zone
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetHostedZoneLimit.html
     */
    toGetHostedZoneLimit() {
        this.add('route53:GetHostedZoneLimit');
        return this;
    }
    /**
     * Grants permission to get information about a specified configuration for DNS query logging
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetQueryLoggingConfig.html
     */
    toGetQueryLoggingConfig() {
        this.add('route53:GetQueryLoggingConfig');
        return this;
    }
    /**
     * Grants permission to get information about a specified reusable delegation set, including the four name servers that are assigned to the delegation set
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetReusableDelegationSet.html
     */
    toGetReusableDelegationSet() {
        this.add('route53:GetReusableDelegationSet');
        return this;
    }
    /**
     * Grants permission to get the maximum number of hosted zones that you can associate with the specified reusable delegation set
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetReusableDelegationSetLimit.html
     */
    toGetReusableDelegationSetLimit() {
        this.add('route53:GetReusableDelegationSetLimit');
        return this;
    }
    /**
     * Grants permission to get information about a specified traffic policy version
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetTrafficPolicy.html
     */
    toGetTrafficPolicy() {
        this.add('route53:GetTrafficPolicy');
        return this;
    }
    /**
     * Grants permission to get information about a specified traffic policy instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetTrafficPolicyInstance.html
     */
    toGetTrafficPolicyInstance() {
        this.add('route53:GetTrafficPolicyInstance');
        return this;
    }
    /**
     * Grants permission to get the number of traffic policy instances that are associated with the current AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_GetTrafficPolicyInstanceCount.html
     */
    toGetTrafficPolicyInstanceCount() {
        this.add('route53:GetTrafficPolicyInstanceCount');
        return this;
    }
    /**
     * Grants permission to get a list of geographic locations that Route 53 supports for geolocation
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListGeoLocations.html
     */
    toListGeoLocations() {
        this.add('route53:ListGeoLocations');
        return this;
    }
    /**
     * Grants permission to get a list of the health checks that are associated with the current AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListHealthChecks.html
     */
    toListHealthChecks() {
        this.add('route53:ListHealthChecks');
        return this;
    }
    /**
     * Grants permission to get a list of the public and private hosted zones that are associated with the current AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListHostedZones.html
     */
    toListHostedZones() {
        this.add('route53:ListHostedZones');
        return this;
    }
    /**
     * Grants permission to get a list of your hosted zones in lexicographic order. Hosted zones are sorted by name with the labels reversed, for example, com.example.www.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListHostedZonesByName.html
     */
    toListHostedZonesByName() {
        this.add('route53:ListHostedZonesByName');
        return this;
    }
    /**
     * Grants permission to list the configurations for DNS query logging that are associated with the current AWS account or the configuration that is associated with a specified hosted zone.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListQueryLoggingConfigs.html
     */
    toListQueryLoggingConfigs() {
        this.add('route53:ListQueryLoggingConfigs');
        return this;
    }
    /**
     * Grants permission to list the records in a specified hosted zone
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListResourceRecordSets.html
     */
    toListResourceRecordSets() {
        this.add('route53:ListResourceRecordSets');
        return this;
    }
    /**
     * Grants permission to list the reusable delegation sets that are associated with the current AWS account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListReusableDelegationSets.html
     */
    toListReusableDelegationSets() {
        this.add('route53:ListReusableDelegationSets');
        return this;
    }
    /**
     * Grants permission to list tags for one health check or hosted zone
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.add('route53:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to list tags for up to 10 health checks or hosted zones
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListTagsForResources.html
     */
    toListTagsForResources() {
        this.add('route53:ListTagsForResources');
        return this;
    }
    /**
     * Grants permission to get information about the latest version for every traffic policy that is associated with the current AWS account. Policies are listed in the order in which they were created.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListTrafficPolicies.html
     */
    toListTrafficPolicies() {
        this.add('route53:ListTrafficPolicies');
        return this;
    }
    /**
     * Grants permission to get information about the traffic policy instances that you created by using the current AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListTrafficPolicyInstances.html
     */
    toListTrafficPolicyInstances() {
        this.add('route53:ListTrafficPolicyInstances');
        return this;
    }
    /**
     * Grants permission to get information about the traffic policy instances that you created in a specified hosted zone
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListTrafficPolicyInstancesByHostedZone.html
     */
    toListTrafficPolicyInstancesByHostedZone() {
        this.add('route53:ListTrafficPolicyInstancesByHostedZone');
        return this;
    }
    /**
     * Grants permission to get information about the traffic policy instances that you created using a specified traffic policy version
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListTrafficPolicyInstancesByPolicy.html
     */
    toListTrafficPolicyInstancesByPolicy() {
        this.add('route53:ListTrafficPolicyInstancesByPolicy');
        return this;
    }
    /**
     * Grants permission to get information about all the versions for a specified traffic policy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListTrafficPolicyVersions.html
     */
    toListTrafficPolicyVersions() {
        this.add('route53:ListTrafficPolicyVersions');
        return this;
    }
    /**
     * Grants permission to get a list of the VPCs that were created by other accounts and that can be associated with a specified hosted zone
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_ListVPCAssociationAuthorizations.html
     */
    toListVPCAssociationAuthorizations() {
        this.add('route53:ListVPCAssociationAuthorizations');
        return this;
    }
    /**
     * Grants permission to get the value that Route 53 returns in response to a DNS query for a specified record name and type
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_TestDNSAnswer.html
     */
    toTestDNSAnswer() {
        this.add('route53:TestDNSAnswer');
        return this;
    }
    /**
     * Grants permission to update an existing health check
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_UpdateHealthCheck.html
     */
    toUpdateHealthCheck() {
        this.add('route53:UpdateHealthCheck');
        return this;
    }
    /**
     * Grants permission to update the comment for a specified hosted zone
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_UpdateHostedZoneComment.html
     */
    toUpdateHostedZoneComment() {
        this.add('route53:UpdateHostedZoneComment');
        return this;
    }
    /**
     * Grants permission to update the comment for a specified traffic policy version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_UpdateTrafficPolicyComment.html
     */
    toUpdateTrafficPolicyComment() {
        this.add('route53:UpdateTrafficPolicyComment');
        return this;
    }
    /**
     * Grants permission to update the records in a specified hosted zone that were created based on the settings in a specified traffic policy version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_UpdateTrafficPolicyInstance.html
     */
    toUpdateTrafficPolicyInstance() {
        this.add('route53:UpdateTrafficPolicyInstance');
        return this;
    }
    /**
     * Adds a resource of type change to the statement
     *
     * https://docs.aws.amazon.com/Route53/latest/APIReference/API_Change.html
     *
     * @param id - Identifier for the id.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onChange(id, partition) {
        var arn = 'arn:${Partition}:route53:::change/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type delegationset to the statement
     *
     * https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/route-53-concepts.html#route-53-concepts-reusable-delegation-set
     *
     * @param id - Identifier for the id.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onDelegationset(id, partition) {
        var arn = 'arn:${Partition}:route53:::delegationset/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type healthcheck to the statement
     *
     * https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/route-53-concepts.html#route-53-concepts-health-check
     *
     * @param id - Identifier for the id.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onHealthcheck(id, partition) {
        var arn = 'arn:${Partition}:route53:::healthcheck/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type hostedzone to the statement
     *
     * https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/route-53-concepts.html#route-53-concepts-hosted-zone
     *
     * @param id - Identifier for the id.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onHostedzone(id, partition) {
        var arn = 'arn:${Partition}:route53:::hostedzone/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type trafficpolicy to the statement
     *
     * https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/traffic-policies.html
     *
     * @param id - Identifier for the id.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onTrafficpolicy(id, partition) {
        var arn = 'arn:${Partition}:route53:::trafficpolicy/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type trafficpolicyinstance to the statement
     *
     * https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/traffic-policy-records.html
     *
     * @param id - Identifier for the id.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onTrafficpolicyinstance(id, partition) {
        var arn = 'arn:${Partition}:route53:::trafficpolicyinstance/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type queryloggingconfig to the statement
     *
     * https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/query-logs.html
     *
     * @param id - Identifier for the id.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onQueryloggingconfig(id, partition) {
        var arn = 'arn:${Partition}:route53:::queryloggingconfig/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Route53 = Route53;
//# sourceMappingURL=data:application/json;base64,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