"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Sagemaker = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [sagemaker](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Sagemaker extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [sagemaker](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'sagemaker';
        this.actionList = {
            "AddTags": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_AddTags.html",
                "description": "Adds or overwrites one or more tags for the specified Amazon SageMaker resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "app": {
                        "required": false
                    },
                    "automl-job": {
                        "required": false
                    },
                    "domain": {
                        "required": false
                    },
                    "endpoint": {
                        "required": false
                    },
                    "endpoint-config": {
                        "required": false
                    },
                    "experiment": {
                        "required": false
                    },
                    "experiment-trial": {
                        "required": false
                    },
                    "experiment-trial-component": {
                        "required": false
                    },
                    "flow-definition": {
                        "required": false
                    },
                    "human-task-ui": {
                        "required": false
                    },
                    "hyper-parameter-tuning-job": {
                        "required": false
                    },
                    "labeling-job": {
                        "required": false
                    },
                    "model": {
                        "required": false
                    },
                    "monitoring-schedule": {
                        "required": false
                    },
                    "notebook-instance": {
                        "required": false
                    },
                    "processing-job": {
                        "required": false
                    },
                    "training-job": {
                        "required": false
                    },
                    "transform-job": {
                        "required": false
                    },
                    "user-profile": {
                        "required": false
                    },
                    "workteam": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "AssociateTrialComponent": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_AssociateTrialComponent.html",
                "description": "Associate a trial component with a trial.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "experiment-trial": {
                        "required": true
                    },
                    "experiment-trial-component": {
                        "required": true
                    }
                }
            },
            "BatchGetMetrics": {
                "url": "",
                "description": "Retrieve metrics associated with SageMaker Resources such as Training Jobs. This API is not publicly exposed at this point, however admins can control this action",
                "accessLevel": "Read",
                "resourceTypes": {
                    "training-job": {
                        "required": true
                    }
                }
            },
            "BatchPutMetrics": {
                "url": "",
                "description": "Publish metrics associated with a SageMaker Resource such as a Training Job. This API is not publicly exposed at this point, however admins can control this action",
                "accessLevel": "Write",
                "resourceTypes": {
                    "training-job": {
                        "required": true
                    }
                }
            },
            "CreateAlgorithm": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateAlgorithm.html",
                "description": "Create an algorithm.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "algorithm": {
                        "required": true
                    }
                }
            },
            "CreateApp": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateApp.html",
                "description": "Grants permission to create an App for a SageMaker Studio UserProfile",
                "accessLevel": "Write",
                "resourceTypes": {
                    "app": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "sagemaker:InstanceTypes"
                ]
            },
            "CreateAutoMLJob": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateAutoMLJob.html",
                "description": "Creates automl job.",
                "accessLevel": "Write",
                "dependentActions": [
                    "iam:PassRole"
                ],
                "resourceTypes": {
                    "automl-job": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "sagemaker:InterContainerTrafficEncryption",
                    "sagemaker:OutputKmsKey",
                    "sagemaker:VolumeKmsKey",
                    "sagemaker:VpcSecurityGroupIds",
                    "sagemaker:VpcSubnets"
                ]
            },
            "CreateCodeRepository": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateCodeRepository.html",
                "description": "Create a code repository.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "code-repository": {
                        "required": true
                    }
                }
            },
            "CreateCompilationJob": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateCompilationJob.html",
                "description": "Create a compilation job.",
                "accessLevel": "Write",
                "dependentActions": [
                    "iam:PassRole"
                ],
                "resourceTypes": {
                    "compilation-job": {
                        "required": true
                    }
                }
            },
            "CreateDomain": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateDomain.html",
                "description": "Grants permission to create a Domain for SageMaker Studio",
                "accessLevel": "Write",
                "dependentActions": [
                    "iam:CreateServiceLinkedRole",
                    "iam:PassRole"
                ],
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "sagemaker:AppNetworkAccess",
                    "sagemaker:InstanceTypes",
                    "sagemaker:VpcSecurityGroupIds",
                    "sagemaker:VpcSubnets",
                    "sagemaker:DomainSharingOutputKmsKey",
                    "sagemaker:HomeEfsFileSystemKmsKey"
                ]
            },
            "CreateEndpoint": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateEndpoint.html",
                "description": "Creates an endpoint using the endpoint configuration specified in the request.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "endpoint": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateEndpointConfig": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateEndpointConfig.html",
                "description": "Creates an endpoint configuration that can be deployed using Amazon SageMaker hosting services.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "endpoint-config": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "sagemaker:AcceleratorTypes",
                    "sagemaker:InstanceTypes",
                    "sagemaker:ModelArn",
                    "sagemaker:VolumeKmsKey"
                ]
            },
            "CreateExperiment": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateExperiment.html",
                "description": "Create an experiment.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "experiment": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateFlowDefinition": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateFlowDefinition.html",
                "description": "Creates a flow definition, which defines settings for a human workflow.",
                "accessLevel": "Write",
                "dependentActions": [
                    "iam:PassRole"
                ],
                "resourceTypes": {
                    "flow-definition": {
                        "required": true
                    }
                },
                "conditions": [
                    "sagemaker:WorkteamArn",
                    "sagemaker:WorkteamType",
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateHumanTaskUi": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateHumanTaskUi.html",
                "description": "Defines the settings you will use for the human review workflow user interface.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "human-task-ui": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateHyperParameterTuningJob": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateHyperParameterTuningJob.html",
                "description": "Creates hyper parameter tuning job that can be deployed using Amazon SageMaker.",
                "accessLevel": "Write",
                "dependentActions": [
                    "iam:PassRole"
                ],
                "resourceTypes": {
                    "hyper-parameter-tuning-job": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "sagemaker:FileSystemAccessMode",
                    "sagemaker:FileSystemDirectoryPath",
                    "sagemaker:FileSystemId",
                    "sagemaker:FileSystemType",
                    "sagemaker:InstanceTypes",
                    "sagemaker:InterContainerTrafficEncryption",
                    "sagemaker:MaxRuntimeInSeconds",
                    "sagemaker:NetworkIsolation",
                    "sagemaker:OutputKmsKey",
                    "sagemaker:VolumeKmsKey",
                    "sagemaker:VpcSecurityGroupIds",
                    "sagemaker:VpcSubnets"
                ]
            },
            "CreateLabelingJob": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateLabelingJob.html",
                "description": "Starts a labeling job. A labeling job takes unlabeled data in and produces labeled data as output, which can be used for training SageMaker models.",
                "accessLevel": "Write",
                "dependentActions": [
                    "iam:PassRole"
                ],
                "resourceTypes": {
                    "labeling-job": {
                        "required": true
                    }
                },
                "conditions": [
                    "sagemaker:WorkteamArn",
                    "sagemaker:WorkteamType",
                    "sagemaker:VolumeKmsKey",
                    "sagemaker:OutputKmsKey",
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateModel": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateModel.html",
                "description": "Creates a model in Amazon SageMaker. In the request, you specify a name for the model and describe one or more containers.",
                "accessLevel": "Write",
                "dependentActions": [
                    "iam:PassRole"
                ],
                "resourceTypes": {
                    "model": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "sagemaker:NetworkIsolation",
                    "sagemaker:VpcSecurityGroupIds",
                    "sagemaker:VpcSubnets"
                ]
            },
            "CreateModelPackage": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateModelPackage.html",
                "description": "Create a model package.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "model-package": {
                        "required": true
                    }
                }
            },
            "CreateMonitoringSchedule": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateMonitoringSchedule.html",
                "description": "Creates a monitoring schedule.",
                "accessLevel": "Write",
                "dependentActions": [
                    "iam:PassRole"
                ],
                "resourceTypes": {
                    "monitoring-schedule": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "sagemaker:InstanceTypes",
                    "sagemaker:MaxRuntimeInSeconds",
                    "sagemaker:NetworkIsolation",
                    "sagemaker:OutputKmsKey",
                    "sagemaker:VolumeKmsKey",
                    "sagemaker:VpcSecurityGroupIds",
                    "sagemaker:VpcSubnets"
                ]
            },
            "CreateNotebookInstance": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateNotebookInstance.html",
                "description": "Creates an Amazon SageMaker notebook instance. A notebook instance is an Amazon EC2 instance running on a Jupyter Notebook.",
                "accessLevel": "Write",
                "dependentActions": [
                    "iam:PassRole"
                ],
                "resourceTypes": {
                    "notebook-instance": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "sagemaker:AcceleratorTypes",
                    "sagemaker:DirectInternetAccess",
                    "sagemaker:InstanceTypes",
                    "sagemaker:RootAccess",
                    "sagemaker:VolumeKmsKey",
                    "sagemaker:VpcSecurityGroupIds",
                    "sagemaker:VpcSubnets"
                ]
            },
            "CreateNotebookInstanceLifecycleConfig": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateNotebookInstanceLifecycleConfig.html",
                "description": "Creates an notebook instance lifecycle configuration that can be deployed using Amazon SageMaker.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "notebook-instance-lifecycle-config": {
                        "required": true
                    }
                }
            },
            "CreatePresignedDomainUrl": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreatePresignedDomainUrl.html",
                "description": "Grants permission to return a URL that you can use from your browser to connect to the Domain as a specified UserProfile when AuthMode is 'IAM'",
                "accessLevel": "Write",
                "resourceTypes": {
                    "user-profile": {
                        "required": true
                    }
                }
            },
            "CreatePresignedNotebookInstanceUrl": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreatePresignedNotebookInstanceUrl.html",
                "description": "Returns a URL that you can use from your browser to connect to the Notebook Instance.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "notebook-instance": {
                        "required": true
                    }
                }
            },
            "CreateProcessingJob": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateProcessingJob.html",
                "description": "Starts a processing job. After processing completes, Amazon SageMaker saves the resulting artifacts and other optional output to an Amazon S3 location that you specify.",
                "accessLevel": "Write",
                "dependentActions": [
                    "iam:PassRole"
                ],
                "resourceTypes": {
                    "processing-job": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "sagemaker:InstanceTypes",
                    "sagemaker:MaxRuntimeInSeconds",
                    "sagemaker:NetworkIsolation",
                    "sagemaker:OutputKmsKey",
                    "sagemaker:VolumeKmsKey",
                    "sagemaker:VpcSecurityGroupIds",
                    "sagemaker:VpcSubnets",
                    "sagemaker:InterContainerTrafficEncryption"
                ]
            },
            "CreateTrainingJob": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateTrainingJob.html",
                "description": "Starts a model training job. After training completes, Amazon SageMaker saves the resulting model artifacts and other optional output to an Amazon S3 location that you specify.",
                "accessLevel": "Write",
                "dependentActions": [
                    "iam:PassRole"
                ],
                "resourceTypes": {
                    "training-job": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "sagemaker:FileSystemAccessMode",
                    "sagemaker:FileSystemDirectoryPath",
                    "sagemaker:FileSystemId",
                    "sagemaker:FileSystemType",
                    "sagemaker:InstanceTypes",
                    "sagemaker:InterContainerTrafficEncryption",
                    "sagemaker:MaxRuntimeInSeconds",
                    "sagemaker:NetworkIsolation",
                    "sagemaker:OutputKmsKey",
                    "sagemaker:VolumeKmsKey",
                    "sagemaker:VpcSecurityGroupIds",
                    "sagemaker:VpcSubnets"
                ]
            },
            "CreateTransformJob": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateTransformJob.html",
                "description": "Starts a transform job. After the results are obtained, Amazon SageMaker saves them to an Amazon S3 location that you specify.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transform-job": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "sagemaker:InstanceTypes",
                    "sagemaker:ModelArn",
                    "sagemaker:OutputKmsKey",
                    "sagemaker:VolumeKmsKey"
                ]
            },
            "CreateTrial": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateTrial.html",
                "description": "Create a trial.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "experiment-trial": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateTrialComponent": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateTrialComponent.html",
                "description": "Create a trial component.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "experiment-trial-component": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateUserProfile": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateUserProfile.html",
                "description": "Grants permission to create a UserProfile for a SageMaker Studio Domain",
                "accessLevel": "Write",
                "dependentActions": [
                    "iam:PassRole"
                ],
                "resourceTypes": {
                    "user-profile": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "sagemaker:VpcSecurityGroupIds",
                    "sagemaker:InstanceTypes",
                    "sagemaker:DomainSharingOutputKmsKey"
                ]
            },
            "CreateWorkforce": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateWorkforce.html",
                "description": "Create a workforce.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "workforce": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateWorkteam": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateWorkteam.html",
                "description": "Create a workteam.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "workteam": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "DeleteAlgorithm": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DeleteAlgorithm.html",
                "description": "Deletes an algorithm.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "algorithm": {
                        "required": true
                    }
                }
            },
            "DeleteApp": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DeleteApp.html",
                "description": "Grants permission to delete an App",
                "accessLevel": "Write",
                "resourceTypes": {
                    "app": {
                        "required": true
                    }
                }
            },
            "DeleteCodeRepository": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DeleteCodeRepository.html",
                "description": "Deletes a code repository.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "code-repository": {
                        "required": true
                    }
                }
            },
            "DeleteDomain": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DeleteDomain.html",
                "description": "Grants permission to delete a Domain",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "DeleteEndpoint": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DeleteEndpoint.html",
                "description": "Deletes an endpoint. Amazon SageMaker frees up all the resources that were deployed when the endpoint was created.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "endpoint": {
                        "required": true
                    }
                }
            },
            "DeleteEndpointConfig": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DeleteEndpointConfig.html",
                "description": "Deletes the endpoint configuration created using the CreateEndpointConfig API. The DeleteEndpointConfig API deletes only the specified configuration. It does not delete any endpoints created using the configuration.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "endpoint-config": {
                        "required": true
                    }
                }
            },
            "DeleteExperiment": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DeleteExperiment.html",
                "description": "Deletes an experiment.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "experiment": {
                        "required": true
                    }
                }
            },
            "DeleteFlowDefinition": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DeleteFlowDefinition.html",
                "description": "Deltes the specified flow definition.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "flow-definition": {
                        "required": true
                    }
                }
            },
            "DeleteHumanLoop": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DeleteHumanLoop.html",
                "description": "Deletes the specified human loop.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "human-loop": {
                        "required": true
                    }
                }
            },
            "DeleteModel": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DeleteModel.html",
                "description": "Deletes a model created using the CreateModel API. The DeleteModel API deletes only the model entry in Amazon SageMaker that you created by calling the CreateModel API. It does not delete model artifacts, inference code, or the IAM role that you specified when creating the model.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "model": {
                        "required": true
                    }
                }
            },
            "DeleteModelPackage": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DeleteModelPackage.html",
                "description": "Deletes a model package.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "model-package": {
                        "required": true
                    }
                }
            },
            "DeleteMonitoringSchedule": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DeleteMonitoringSchedule.html",
                "description": "Deletes a monitoring schedule. Amazon SageMaker will no longer run the scheduled monitoring.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "monitoring-schedule": {
                        "required": true
                    }
                }
            },
            "DeleteNotebookInstance": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DeleteNotebookInstance.html",
                "description": "Deletes an Amazon SageMaker notebook instance. Before you can delete a notebook instance, you must call the StopNotebookInstance API.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "notebook-instance": {
                        "required": true
                    }
                }
            },
            "DeleteNotebookInstanceLifecycleConfig": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DeleteNotebookInstanceLifecycleConfig.html",
                "description": "Deletes an notebook instance lifecycle configuration that can be deployed using Amazon SageMaker.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "notebook-instance-lifecycle-config": {
                        "required": true
                    }
                }
            },
            "DeleteTags": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DeleteTags.html",
                "description": "Deletes the specified set of tags from an Amazon SageMaker resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "app": {
                        "required": false
                    },
                    "automl-job": {
                        "required": false
                    },
                    "compilation-job": {
                        "required": false
                    },
                    "domain": {
                        "required": false
                    },
                    "endpoint": {
                        "required": false
                    },
                    "endpoint-config": {
                        "required": false
                    },
                    "experiment": {
                        "required": false
                    },
                    "experiment-trial": {
                        "required": false
                    },
                    "experiment-trial-component": {
                        "required": false
                    },
                    "flow-definition": {
                        "required": false
                    },
                    "human-task-ui": {
                        "required": false
                    },
                    "hyper-parameter-tuning-job": {
                        "required": false
                    },
                    "labeling-job": {
                        "required": false
                    },
                    "model": {
                        "required": false
                    },
                    "monitoring-schedule": {
                        "required": false
                    },
                    "notebook-instance": {
                        "required": false
                    },
                    "processing-job": {
                        "required": false
                    },
                    "training-job": {
                        "required": false
                    },
                    "transform-job": {
                        "required": false
                    },
                    "user-profile": {
                        "required": false
                    },
                    "workteam": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "DeleteTrial": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DeleteTrial.html",
                "description": "Deletes a trial.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "experiment-trial": {
                        "required": true
                    }
                }
            },
            "DeleteTrialComponent": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DeleteTrialComponent.html",
                "description": "Deletes a trial component.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "experiment-trial-component": {
                        "required": true
                    }
                }
            },
            "DeleteUserProfile": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DeleteUserProfile.html",
                "description": "Grants permission to delete a UserProfile",
                "accessLevel": "Write",
                "resourceTypes": {
                    "user-profile": {
                        "required": true
                    }
                }
            },
            "DeleteWorkforce": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DeleteWorkforce.html",
                "description": "Deletes a workforce.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "workforce": {
                        "required": true
                    }
                }
            },
            "DeleteWorkteam": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DeleteWorkteam.html",
                "description": "Deletes a workteam.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "workteam": {
                        "required": true
                    }
                }
            },
            "DescribeAlgorithm": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeAlgorithm.html",
                "description": "Returns information about an algorithm.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "algorithm": {
                        "required": true
                    }
                }
            },
            "DescribeApp": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeApp.html",
                "description": "Grants permission to describe an App",
                "accessLevel": "Read",
                "resourceTypes": {
                    "app": {
                        "required": true
                    }
                }
            },
            "DescribeAutoMLJob": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeAutoMLJob.html",
                "description": "Describes an automl job that was created via CreateAutoMLJob API.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "automl-job": {
                        "required": true
                    }
                }
            },
            "DescribeCodeRepository": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeCodeRepository.html",
                "description": "Returns information about a code repository.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "code-repository": {
                        "required": true
                    }
                }
            },
            "DescribeCompilationJob": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeCompilationJob.html",
                "description": "Returns information about a compilation job.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "compilation-job": {
                        "required": true
                    }
                }
            },
            "DescribeDomain": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeDomain.html",
                "description": "Grants permission to describe a Domain",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "DescribeEndpoint": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeEndpoint.html",
                "description": "Returns the description of an endpoint.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "endpoint": {
                        "required": true
                    }
                }
            },
            "DescribeEndpointConfig": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeEndpointConfig.html",
                "description": "Returns the description of an endpoint configuration, which was created using the CreateEndpointConfig API.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "endpoint-config": {
                        "required": true
                    }
                }
            },
            "DescribeExperiment": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeExperiment.html",
                "description": "Returns information about an experiment.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "experiment": {
                        "required": true
                    }
                }
            },
            "DescribeFlowDefinition": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeFlowDefinition.html",
                "description": "Returns detailed information about the specified flow definition.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "flow-definition": {
                        "required": true
                    }
                }
            },
            "DescribeHumanLoop": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeHumanLoop.html",
                "description": "Returns detailed information about the specified human loop.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "human-loop": {
                        "required": true
                    }
                }
            },
            "DescribeHumanTaskUi": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeHumanTaskUi.html",
                "description": "Returns detailed information about the specified human review workflow user interface.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "human-task-ui": {
                        "required": true
                    }
                }
            },
            "DescribeHyperParameterTuningJob": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeHyperParameterTuningJob.html",
                "description": "Describes a hyper parameter tuning job that was created via CreateHyperParameterTuningJob API.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "hyper-parameter-tuning-job": {
                        "required": true
                    }
                }
            },
            "DescribeLabelingJob": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeLabelingJob.html",
                "description": "Returns information about a labeling job.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "labeling-job": {
                        "required": true
                    }
                }
            },
            "DescribeModel": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeModel.html",
                "description": "Describes a model that you created using the CreateModel API.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "model": {
                        "required": true
                    }
                }
            },
            "DescribeModelPackage": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeModelPackage.html",
                "description": "Returns information about a model package.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "model-package": {
                        "required": true
                    }
                }
            },
            "DescribeMonitoringSchedule": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeMonitoringSchedule.html",
                "description": "Returns information about a monitoring schedule.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "monitoring-schedule": {
                        "required": true
                    }
                }
            },
            "DescribeNotebookInstance": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeNotebookInstance.html",
                "description": "Returns information about a notebook instance.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "notebook-instance": {
                        "required": true
                    }
                }
            },
            "DescribeNotebookInstanceLifecycleConfig": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeNotebookInstanceLifecycleConfig.html",
                "description": "Describes an notebook instance lifecycle configuration that was created via CreateNotebookInstanceLifecycleConfig API.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "notebook-instance-lifecycle-config": {
                        "required": true
                    }
                }
            },
            "DescribeProcessingJob": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeProcessingJob.html",
                "description": "Returns information about a processing job.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "processing-job": {
                        "required": true
                    }
                }
            },
            "DescribeSubscribedWorkteam": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeSubscribedWorkteam.html",
                "description": "Returns information about a subscribed workteam.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "workteam": {
                        "required": true
                    }
                }
            },
            "DescribeTrainingJob": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeTrainingJob.html",
                "description": "Returns information about a training job.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "training-job": {
                        "required": true
                    }
                }
            },
            "DescribeTransformJob": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeTransformJob.html",
                "description": "Returns information about a transform job.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "transform-job": {
                        "required": true
                    }
                }
            },
            "DescribeTrial": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeTrial.html",
                "description": "Returns information about a trial.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "experiment-trial": {
                        "required": true
                    }
                }
            },
            "DescribeTrialComponent": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeTrialComponent.html",
                "description": "Returns information about a trial component.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "experiment-trial-component": {
                        "required": true
                    }
                }
            },
            "DescribeUserProfile": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeUserProfile.html",
                "description": "Grants permission to describe a UserProfile",
                "accessLevel": "Read",
                "resourceTypes": {
                    "user-profile": {
                        "required": true
                    }
                }
            },
            "DescribeWorkforce": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeWorkforce.html",
                "description": "Returns information about a workforce.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "workforce": {
                        "required": true
                    }
                }
            },
            "DescribeWorkteam": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeWorkteam.html",
                "description": "Returns information about a workteam.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "workteam": {
                        "required": true
                    }
                }
            },
            "DisassociateTrialComponent": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_DisassociateTrialComponent.html",
                "description": "Disassociate a trial component with a trial.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "experiment-trial": {
                        "required": true
                    },
                    "experiment-trial-component": {
                        "required": true
                    },
                    "processing-job": {
                        "required": true
                    }
                }
            },
            "GetSearchSuggestions": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_GetSearchSuggestions.html",
                "description": "Get search suggestions when provided with keyword.",
                "accessLevel": "Read"
            },
            "InvokeEndpoint": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_InvokeEndpoint.html",
                "description": "After you deploy a model into production using Amazon SageMaker hosting services, your client applications use this API to get inferences from the model hosted at the specified endpoint.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "endpoint": {
                        "required": true
                    }
                },
                "conditions": [
                    "sagemaker:TargetModel"
                ]
            },
            "ListAlgorithms": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListAlgorithms.html",
                "description": "Lists algorithms.",
                "accessLevel": "List"
            },
            "ListApps": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListApps.html",
                "description": "Grants permission to list the Apps in your account",
                "accessLevel": "List"
            },
            "ListAutoMLJobs": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListAutoMLJobs.html",
                "description": "Lists automl jobs created via the CreateAutoMLJob.",
                "accessLevel": "List"
            },
            "ListCandidatesForAutoMLJob": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListCandidatesForAutoMLJob.html",
                "description": "Lists candidates for automl job created via the CreateAutoMLJob.",
                "accessLevel": "List"
            },
            "ListCodeRepositories": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListCodeRepositories.html",
                "description": "Lists code repositories.",
                "accessLevel": "List"
            },
            "ListCompilationJobs": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListCompilationJobs.html",
                "description": "Lists compilation jobs.",
                "accessLevel": "List"
            },
            "ListDomains": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListDomains.html",
                "description": "Grants permission to list the Domains in your account",
                "accessLevel": "List"
            },
            "ListEndpointConfigs": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListEndpointConfigs.html",
                "description": "Lists endpoint configurations.",
                "accessLevel": "List"
            },
            "ListEndpoints": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListEndpoints.html",
                "description": "Lists endpoints.",
                "accessLevel": "List"
            },
            "ListExperiments": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListExperiments.html",
                "description": "Lists experiments.",
                "accessLevel": "List"
            },
            "ListFlowDefinitions": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListFlowDefinitions.html",
                "description": "Returns summary information about flow definitions, given the specified parameters.",
                "accessLevel": "List"
            },
            "ListHumanLoops": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListHumanLoops.html",
                "description": "Returns summary information about human loops, given the specified parameters.",
                "accessLevel": "List"
            },
            "ListHumanTaskUis": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListHumanTaskUis.html",
                "description": "Returns summary information about human review workflow user interfaces, given the specified parameters.",
                "accessLevel": "List"
            },
            "ListHyperParameterTuningJobs": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListHyperParameterTuningJobs.html",
                "description": "Lists hyper parameter tuning jobs that was created using Amazon SageMaker.",
                "accessLevel": "List"
            },
            "ListLabelingJobs": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListLabelingJobs.html",
                "description": "Lists labeling jobs.",
                "accessLevel": "List"
            },
            "ListLabelingJobsForWorkteam": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListLabelingJobs.html",
                "description": "Lists labeling jobs for workteam.",
                "accessLevel": "List",
                "resourceTypes": {
                    "workteam": {
                        "required": true
                    }
                }
            },
            "ListModelPackages": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListModelPackages.html",
                "description": "Lists model packages.",
                "accessLevel": "List"
            },
            "ListModels": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListModels.html",
                "description": "Lists the models created with the CreateModel API.",
                "accessLevel": "List"
            },
            "ListMonitoringExecutions": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListMonitoringExecutions.html",
                "description": "Lists monitoring executions.",
                "accessLevel": "List"
            },
            "ListMonitoringSchedules": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListMonitoringSchedules.html",
                "description": "Lists monitoring schedules.",
                "accessLevel": "List"
            },
            "ListNotebookInstanceLifecycleConfigs": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListNotebookInstanceLifecycleConfigs.html",
                "description": "Lists notebook instance lifecycle configurations that can be deployed using Amazon SageMaker.",
                "accessLevel": "List"
            },
            "ListNotebookInstances": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListNotebookInstances.html",
                "description": "Returns a list of the Amazon SageMaker notebook instances in the requester's account in an AWS Region.",
                "accessLevel": "List"
            },
            "ListProcessingJobs": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListProcessingJobs.html",
                "description": "Lists processing jobs.",
                "accessLevel": "List"
            },
            "ListSubscribedWorkteams": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListSubscribedWorkteams.html",
                "description": "Lists subscribed workteams.",
                "accessLevel": "List"
            },
            "ListTags": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListTags.html",
                "description": "Returns the tag set associated with the specified resource.",
                "accessLevel": "List",
                "resourceTypes": {
                    "app": {
                        "required": false
                    },
                    "automl-job": {
                        "required": false
                    },
                    "domain": {
                        "required": false
                    },
                    "endpoint": {
                        "required": false
                    },
                    "endpoint-config": {
                        "required": false
                    },
                    "experiment": {
                        "required": false
                    },
                    "experiment-trial": {
                        "required": false
                    },
                    "experiment-trial-component": {
                        "required": false
                    },
                    "flow-definition": {
                        "required": false
                    },
                    "human-task-ui": {
                        "required": false
                    },
                    "hyper-parameter-tuning-job": {
                        "required": false
                    },
                    "labeling-job": {
                        "required": false
                    },
                    "model": {
                        "required": false
                    },
                    "monitoring-schedule": {
                        "required": false
                    },
                    "notebook-instance": {
                        "required": false
                    },
                    "training-job": {
                        "required": false
                    },
                    "transform-job": {
                        "required": false
                    },
                    "user-profile": {
                        "required": false
                    },
                    "workteam": {
                        "required": false
                    }
                }
            },
            "ListTrainingJobs": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListTrainingJobs.html",
                "description": "Lists training jobs.",
                "accessLevel": "List"
            },
            "ListTrainingJobsForHyperParameterTuningJob": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_ ListTrainingJobsForHyperParameterTuningJob.html",
                "description": "Lists training jobs for a hyper parameter tuning job that was created using Amazon SageMaker.",
                "accessLevel": "List",
                "resourceTypes": {
                    "hyper-parameter-tuning-job": {
                        "required": true
                    }
                }
            },
            "ListTransformJobs": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListTransformJobs.html",
                "description": "Lists transform jobs.",
                "accessLevel": "List"
            },
            "ListTrialComponents": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListTrialComponents.html",
                "description": "Lists trial components.",
                "accessLevel": "List"
            },
            "ListTrials": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListTrials.html",
                "description": "Lists trials.",
                "accessLevel": "List"
            },
            "ListUserProfiles": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListUserProfiles.html",
                "description": "Grants permission to list the UserProfiles in your account",
                "accessLevel": "List"
            },
            "ListWorkforces": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListWorkforces.html",
                "description": "Lists workforces.",
                "accessLevel": "List"
            },
            "ListWorkteams": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListWorkteams.html",
                "description": "Lists workteams.",
                "accessLevel": "List"
            },
            "RenderUiTemplate": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_RenderUiTemplate.html",
                "description": "Render a UI template used for a human annotation task.",
                "accessLevel": "Read",
                "dependentActions": [
                    "iam:PassRole"
                ]
            },
            "Search": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_Search.html",
                "description": "Search for SageMaker objects.",
                "accessLevel": "Read"
            },
            "StartHumanLoop": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_StartHumanLoop.html",
                "description": "Starts a human loop.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "flow-definition": {
                        "required": true
                    }
                }
            },
            "StartMonitoringSchedule": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_StartMonitoringSchedule.html",
                "description": "Starts a monitoring schedule.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "monitoring-schedule": {
                        "required": true
                    }
                }
            },
            "StartNotebookInstance": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_StartNotebookInstance.html",
                "description": "Launches an EC2 instance with the latest version of the libraries and attaches your EBS volume.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "notebook-instance": {
                        "required": true
                    }
                }
            },
            "StopAutoMLJob": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_StopAutoMLJob.html",
                "description": "Stops a running automl job created via the CreateAutoMLJob.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "automl-job": {
                        "required": true
                    }
                }
            },
            "StopCompilationJob": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_StopCompilationJob.html",
                "description": "Stops a compilation job.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "compilation-job": {
                        "required": true
                    }
                }
            },
            "StopHumanLoop": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_StopHumanLoop.html",
                "description": "Stops the specified human loop.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "human-loop": {
                        "required": true
                    }
                }
            },
            "StopHyperParameterTuningJob": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_StopHyperParameterTuningJob.html",
                "description": "Stops a running hyper parameter tuning job create via the CreateHyperParameterTuningJob.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "hyper-parameter-tuning-job": {
                        "required": true
                    }
                }
            },
            "StopLabelingJob": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_StopLabelingJob.html",
                "description": "Stops a labeling job. Any labels already generated will be exported before stopping.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "labeling-job": {
                        "required": true
                    }
                }
            },
            "StopMonitoringSchedule": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_StopMonitoringSchedule.html",
                "description": "Stops a monitoring schedule.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "monitoring-schedule": {
                        "required": true
                    }
                }
            },
            "StopNotebookInstance": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_StopNotebookInstance.html",
                "description": "Terminates the EC2 instance. Before terminating the instance, Amazon SageMaker disconnects the EBS volume from it. Amazon SageMaker preserves the EBS volume.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "notebook-instance": {
                        "required": true
                    }
                }
            },
            "StopProcessingJob": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_StopProcessingJob.html",
                "description": "Stops a processing job. To stop a job, Amazon SageMaker sends the algorithm the SIGTERM signal, which delays job termination for 120 seconds.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "processing-job": {
                        "required": true
                    }
                }
            },
            "StopTrainingJob": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_StopTrainingJob.html",
                "description": "Stops a training job. To stop a job, Amazon SageMaker sends the algorithm the SIGTERM signal, which delays job termination for 120 seconds.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "training-job": {
                        "required": true
                    }
                }
            },
            "StopTransformJob": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_StopTransformJob.html",
                "description": "Stops a transform job. When Amazon SageMaker receives a StopTransformJob request, the status of the job changes to Stopping. After Amazon SageMaker stops the job, the status is set to Stopped",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transform-job": {
                        "required": true
                    }
                }
            },
            "UpdateCodeRepository": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_UpdateCodeRepository.html",
                "description": "Updates a code repository.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "code-repository": {
                        "required": true
                    }
                }
            },
            "UpdateDomain": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_UpdateDomain.html",
                "description": "Grants permission to update a Domain",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                },
                "conditions": [
                    "sagemaker:VpcSecurityGroupIds",
                    "sagemaker:InstanceTypes",
                    "sagemaker:DomainSharingOutputKmsKey"
                ]
            },
            "UpdateEndpoint": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_UpdateEndpoint.html",
                "description": "Updates an endpoint to use the endpoint configuration specified in the request.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "endpoint": {
                        "required": true
                    }
                }
            },
            "UpdateEndpointWeightsAndCapacities": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_UpdateEndpointWeightsAndCapacities.html",
                "description": "Updates variant weight, capacity, or both of one or more variants associated with an endpoint.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "endpoint": {
                        "required": true
                    }
                }
            },
            "UpdateExperiment": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_UpdateExperiment.html",
                "description": "Updates an experiment.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "experiment": {
                        "required": true
                    }
                }
            },
            "UpdateMonitoringSchedule": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_UpdateMonitoringSchedule.html",
                "description": "Updates a monitoring schedule.",
                "accessLevel": "Write",
                "dependentActions": [
                    "iam:PassRole"
                ],
                "resourceTypes": {
                    "monitoring-schedule": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "sagemaker:InstanceTypes",
                    "sagemaker:MaxRuntimeInSeconds",
                    "sagemaker:NetworkIsolation",
                    "sagemaker:OutputKmsKey",
                    "sagemaker:VolumeKmsKey",
                    "sagemaker:VpcSecurityGroupIds",
                    "sagemaker:VpcSubnets"
                ]
            },
            "UpdateNotebookInstance": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_UpdateNotebookInstance.html",
                "description": "Updates a notebook instance. Notebook instance updates include upgrading or downgrading the EC2 instance used for your notebook instance to accommodate changes in your workload requirements. You can also update the VPC security groups.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "notebook-instance": {
                        "required": true
                    }
                },
                "conditions": [
                    "sagemaker:AcceleratorTypes",
                    "sagemaker:InstanceTypes",
                    "sagemaker:RootAccess"
                ]
            },
            "UpdateNotebookInstanceLifecycleConfig": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_ UpdateNotebookInstanceLifecycleConfig.html",
                "description": "Updates a notebook instance lifecycle configuration created with the CreateNotebookInstanceLifecycleConfig API.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "notebook-instance-lifecycle-config": {
                        "required": true
                    }
                }
            },
            "UpdateTrial": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_UpdateTrial.html",
                "description": "Updates a trial.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "experiment-trial": {
                        "required": true
                    }
                }
            },
            "UpdateTrialComponent": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_UpdateTrialComponent.html",
                "description": "Updates a trial component.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "experiment-trial-component": {
                        "required": true
                    }
                }
            },
            "UpdateUserProfile": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_UpdateUserProfile.html",
                "description": "Grants permission to update a UserProfile",
                "accessLevel": "Write",
                "resourceTypes": {
                    "user-profile": {
                        "required": true
                    }
                },
                "conditions": [
                    "sagemaker:InstanceTypes",
                    "sagemaker:VpcSecurityGroupIds",
                    "sagemaker:InstanceTypes",
                    "sagemaker:DomainSharingOutputKmsKey"
                ]
            },
            "UpdateWorkforce": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_UpdateWorkforce.html",
                "description": "Updates a workforce.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "workforce": {
                        "required": true
                    }
                }
            },
            "UpdateWorkteam": {
                "url": "https://docs.aws.amazon.com/sagemaker/latest/dg/API_UpdateWorkteam.html",
                "description": "Updates a workteam.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "workteam": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "human-loop": {
                "name": "human-loop",
                "url": "",
                "arn": "arn:${Partition}:sagemaker:${Region}:${Account}:human-loop/${HumanLoopName}",
                "conditionKeys": []
            },
            "flow-definition": {
                "name": "flow-definition",
                "url": "",
                "arn": "arn:${Partition}:sagemaker:${Region}:${Account}:flow-definition/${FlowDefinitionName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "sagemaker:ResourceTag/${TagKey}"
                ]
            },
            "human-task-ui": {
                "name": "human-task-ui",
                "url": "",
                "arn": "arn:${Partition}:sagemaker:${Region}:${Account}:human-task-ui/${HumanTaskUiName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "sagemaker:ResourceTag/${TagKey}"
                ]
            },
            "labeling-job": {
                "name": "labeling-job",
                "url": "",
                "arn": "arn:${Partition}:sagemaker:${Region}:${Account}:labeling-job/${LabelingJobName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "sagemaker:ResourceTag/${TagKey}"
                ]
            },
            "workteam": {
                "name": "workteam",
                "url": "",
                "arn": "arn:${Partition}:sagemaker:${Region}:${Account}:workteam/${WorkteamName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "sagemaker:ResourceTag/${TagKey}"
                ]
            },
            "workforce": {
                "name": "workforce",
                "url": "",
                "arn": "arn:${Partition}:sagemaker:${Region}:${Account}:workforce/${WorkforceName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "sagemaker:ResourceTag/${TagKey}"
                ]
            },
            "domain": {
                "name": "domain",
                "url": "",
                "arn": "arn:${Partition}:sagemaker:${Region}:${Account}:domain/${DomainId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "sagemaker:ResourceTag/${TagKey}"
                ]
            },
            "user-profile": {
                "name": "user-profile",
                "url": "",
                "arn": "arn:${Partition}:sagemaker:${Region}:${Account}:user-profile/${DomainId}/${UserProfileName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "sagemaker:ResourceTag/${TagKey}"
                ]
            },
            "app": {
                "name": "app",
                "url": "",
                "arn": "arn:${Partition}:sagemaker:${Region}:${Account}:app/${DomainId}/${UserProfileName}/${AppType}/${AppName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "sagemaker:ResourceTag/${TagKey}"
                ]
            },
            "notebook-instance": {
                "name": "notebook-instance",
                "url": "",
                "arn": "arn:${Partition}:sagemaker:${Region}:${Account}:notebook-instance/${NotebookInstanceName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "sagemaker:ResourceTag/${TagKey}"
                ]
            },
            "notebook-instance-lifecycle-config": {
                "name": "notebook-instance-lifecycle-config",
                "url": "",
                "arn": "arn:${Partition}:sagemaker:${Region}:${Account}:notebook-instance-lifecycle-config/${NotebookInstanceLifecycleConfigName}",
                "conditionKeys": []
            },
            "code-repository": {
                "name": "code-repository",
                "url": "",
                "arn": "arn:${Partition}:sagemaker:${Region}:${Account}:code-repository/${CodeRepositoryName}",
                "conditionKeys": []
            },
            "algorithm": {
                "name": "algorithm",
                "url": "",
                "arn": "arn:${Partition}:sagemaker:${Region}:${Account}:algorithm/${AlgorithmName}",
                "conditionKeys": []
            },
            "training-job": {
                "name": "training-job",
                "url": "",
                "arn": "arn:${Partition}:sagemaker:${Region}:${Account}:training-job/${TrainingJobName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "sagemaker:ResourceTag/${TagKey}"
                ]
            },
            "processing-job": {
                "name": "processing-job",
                "url": "",
                "arn": "arn:${Partition}:sagemaker:${Region}:${Account}:processing-job/${ProcessingJobName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "sagemaker:ResourceTag/${TagKey}"
                ]
            },
            "hyper-parameter-tuning-job": {
                "name": "hyper-parameter-tuning-job",
                "url": "",
                "arn": "arn:${Partition}:sagemaker:${Region}:${Account}:hyper-parameter-tuning-job/${HyperParameterTuningJobName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "sagemaker:ResourceTag/${TagKey}"
                ]
            },
            "model-package": {
                "name": "model-package",
                "url": "",
                "arn": "arn:${Partition}:sagemaker:${Region}:${Account}:model-package/${ModelPackageName}",
                "conditionKeys": []
            },
            "model": {
                "name": "model",
                "url": "",
                "arn": "arn:${Partition}:sagemaker:${Region}:${Account}:model/${ModelName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "sagemaker:ResourceTag/${TagKey}"
                ]
            },
            "endpoint-config": {
                "name": "endpoint-config",
                "url": "",
                "arn": "arn:${Partition}:sagemaker:${Region}:${Account}:endpoint-config/${EndpointConfigName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "sagemaker:ResourceTag/${TagKey}"
                ]
            },
            "endpoint": {
                "name": "endpoint",
                "url": "",
                "arn": "arn:${Partition}:sagemaker:${Region}:${Account}:endpoint/${EndpointName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "sagemaker:ResourceTag/${TagKey}"
                ]
            },
            "transform-job": {
                "name": "transform-job",
                "url": "",
                "arn": "arn:${Partition}:sagemaker:${Region}:${Account}:transform-job/${TransformJobName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "sagemaker:ResourceTag/${TagKey}"
                ]
            },
            "compilation-job": {
                "name": "compilation-job",
                "url": "",
                "arn": "arn:${Partition}:sagemaker:${Region}:${Account}:compilation-job/${CompilationJobName}",
                "conditionKeys": []
            },
            "automl-job": {
                "name": "automl-job",
                "url": "",
                "arn": "arn:${Partition}:sagemaker:${Region}:${Account}:automl-job/${AutoMLJobJobName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "sagemaker:ResourceTag/${TagKey}"
                ]
            },
            "monitoring-schedule": {
                "name": "monitoring-schedule",
                "url": "",
                "arn": "arn:${Partition}:sagemaker:${Region}:${Account}:monitoring-schedule/${MonitoringScheduleName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "sagemaker:ResourceTag/${TagKey}"
                ]
            },
            "experiment": {
                "name": "experiment",
                "url": "",
                "arn": "arn:${Partition}:sagemaker:${Region}:${Account}:experiment/${ExperimentName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "sagemaker:ResourceTag/${TagKey}"
                ]
            },
            "experiment-trial": {
                "name": "experiment-trial",
                "url": "",
                "arn": "arn:${Partition}:sagemaker:${Region}:${Account}:experiment-trial/${TrialName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "sagemaker:ResourceTag/${TagKey}"
                ]
            },
            "experiment-trial-component": {
                "name": "experiment-trial-component",
                "url": "",
                "arn": "arn:${Partition}:sagemaker:${Region}:${Account}:experiment-trial-component/${TrialComponentName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "sagemaker:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Adds or overwrites one or more tags for the specified Amazon SageMaker resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_AddTags.html
     */
    toAddTags() {
        this.add('sagemaker:AddTags');
        return this;
    }
    /**
     * Associate a trial component with a trial.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_AssociateTrialComponent.html
     */
    toAssociateTrialComponent() {
        this.add('sagemaker:AssociateTrialComponent');
        return this;
    }
    /**
     * Retrieve metrics associated with SageMaker Resources such as Training Jobs. This API is not publicly exposed at this point, however admins can control this action
     *
     * Access Level: Read
     */
    toBatchGetMetrics() {
        this.add('sagemaker:BatchGetMetrics');
        return this;
    }
    /**
     * Publish metrics associated with a SageMaker Resource such as a Training Job. This API is not publicly exposed at this point, however admins can control this action
     *
     * Access Level: Write
     */
    toBatchPutMetrics() {
        this.add('sagemaker:BatchPutMetrics');
        return this;
    }
    /**
     * Create an algorithm.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateAlgorithm.html
     */
    toCreateAlgorithm() {
        this.add('sagemaker:CreateAlgorithm');
        return this;
    }
    /**
     * Grants permission to create an App for a SageMaker Studio UserProfile
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifInstanceTypes()
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateApp.html
     */
    toCreateApp() {
        this.add('sagemaker:CreateApp');
        return this;
    }
    /**
     * Creates automl job.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifInterContainerTrafficEncryption()
     * - .ifOutputKmsKey()
     * - .ifVolumeKmsKey()
     * - .ifVpcSecurityGroupIds()
     * - .ifVpcSubnets()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateAutoMLJob.html
     */
    toCreateAutoMLJob() {
        this.add('sagemaker:CreateAutoMLJob');
        return this;
    }
    /**
     * Create a code repository.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateCodeRepository.html
     */
    toCreateCodeRepository() {
        this.add('sagemaker:CreateCodeRepository');
        return this;
    }
    /**
     * Create a compilation job.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateCompilationJob.html
     */
    toCreateCompilationJob() {
        this.add('sagemaker:CreateCompilationJob');
        return this;
    }
    /**
     * Grants permission to create a Domain for SageMaker Studio
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifAppNetworkAccess()
     * - .ifInstanceTypes()
     * - .ifVpcSecurityGroupIds()
     * - .ifVpcSubnets()
     * - .ifDomainSharingOutputKmsKey()
     * - .ifHomeEfsFileSystemKmsKey()
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateDomain.html
     */
    toCreateDomain() {
        this.add('sagemaker:CreateDomain');
        return this;
    }
    /**
     * Creates an endpoint using the endpoint configuration specified in the request.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateEndpoint.html
     */
    toCreateEndpoint() {
        this.add('sagemaker:CreateEndpoint');
        return this;
    }
    /**
     * Creates an endpoint configuration that can be deployed using Amazon SageMaker hosting services.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifAcceleratorTypes()
     * - .ifInstanceTypes()
     * - .ifModelArn()
     * - .ifVolumeKmsKey()
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateEndpointConfig.html
     */
    toCreateEndpointConfig() {
        this.add('sagemaker:CreateEndpointConfig');
        return this;
    }
    /**
     * Create an experiment.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateExperiment.html
     */
    toCreateExperiment() {
        this.add('sagemaker:CreateExperiment');
        return this;
    }
    /**
     * Creates a flow definition, which defines settings for a human workflow.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifWorkteamArn()
     * - .ifWorkteamType()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateFlowDefinition.html
     */
    toCreateFlowDefinition() {
        this.add('sagemaker:CreateFlowDefinition');
        return this;
    }
    /**
     * Defines the settings you will use for the human review workflow user interface.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateHumanTaskUi.html
     */
    toCreateHumanTaskUi() {
        this.add('sagemaker:CreateHumanTaskUi');
        return this;
    }
    /**
     * Creates hyper parameter tuning job that can be deployed using Amazon SageMaker.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifFileSystemAccessMode()
     * - .ifFileSystemDirectoryPath()
     * - .ifFileSystemId()
     * - .ifFileSystemType()
     * - .ifInstanceTypes()
     * - .ifInterContainerTrafficEncryption()
     * - .ifMaxRuntimeInSeconds()
     * - .ifNetworkIsolation()
     * - .ifOutputKmsKey()
     * - .ifVolumeKmsKey()
     * - .ifVpcSecurityGroupIds()
     * - .ifVpcSubnets()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateHyperParameterTuningJob.html
     */
    toCreateHyperParameterTuningJob() {
        this.add('sagemaker:CreateHyperParameterTuningJob');
        return this;
    }
    /**
     * Starts a labeling job. A labeling job takes unlabeled data in and produces labeled data as output, which can be used for training SageMaker models.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifWorkteamArn()
     * - .ifWorkteamType()
     * - .ifVolumeKmsKey()
     * - .ifOutputKmsKey()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateLabelingJob.html
     */
    toCreateLabelingJob() {
        this.add('sagemaker:CreateLabelingJob');
        return this;
    }
    /**
     * Creates a model in Amazon SageMaker. In the request, you specify a name for the model and describe one or more containers.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifNetworkIsolation()
     * - .ifVpcSecurityGroupIds()
     * - .ifVpcSubnets()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateModel.html
     */
    toCreateModel() {
        this.add('sagemaker:CreateModel');
        return this;
    }
    /**
     * Create a model package.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateModelPackage.html
     */
    toCreateModelPackage() {
        this.add('sagemaker:CreateModelPackage');
        return this;
    }
    /**
     * Creates a monitoring schedule.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifInstanceTypes()
     * - .ifMaxRuntimeInSeconds()
     * - .ifNetworkIsolation()
     * - .ifOutputKmsKey()
     * - .ifVolumeKmsKey()
     * - .ifVpcSecurityGroupIds()
     * - .ifVpcSubnets()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateMonitoringSchedule.html
     */
    toCreateMonitoringSchedule() {
        this.add('sagemaker:CreateMonitoringSchedule');
        return this;
    }
    /**
     * Creates an Amazon SageMaker notebook instance. A notebook instance is an Amazon EC2 instance running on a Jupyter Notebook.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifAcceleratorTypes()
     * - .ifDirectInternetAccess()
     * - .ifInstanceTypes()
     * - .ifRootAccess()
     * - .ifVolumeKmsKey()
     * - .ifVpcSecurityGroupIds()
     * - .ifVpcSubnets()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateNotebookInstance.html
     */
    toCreateNotebookInstance() {
        this.add('sagemaker:CreateNotebookInstance');
        return this;
    }
    /**
     * Creates an notebook instance lifecycle configuration that can be deployed using Amazon SageMaker.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateNotebookInstanceLifecycleConfig.html
     */
    toCreateNotebookInstanceLifecycleConfig() {
        this.add('sagemaker:CreateNotebookInstanceLifecycleConfig');
        return this;
    }
    /**
     * Grants permission to return a URL that you can use from your browser to connect to the Domain as a specified UserProfile when AuthMode is 'IAM'
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreatePresignedDomainUrl.html
     */
    toCreatePresignedDomainUrl() {
        this.add('sagemaker:CreatePresignedDomainUrl');
        return this;
    }
    /**
     * Returns a URL that you can use from your browser to connect to the Notebook Instance.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreatePresignedNotebookInstanceUrl.html
     */
    toCreatePresignedNotebookInstanceUrl() {
        this.add('sagemaker:CreatePresignedNotebookInstanceUrl');
        return this;
    }
    /**
     * Starts a processing job. After processing completes, Amazon SageMaker saves the resulting artifacts and other optional output to an Amazon S3 location that you specify.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifInstanceTypes()
     * - .ifMaxRuntimeInSeconds()
     * - .ifNetworkIsolation()
     * - .ifOutputKmsKey()
     * - .ifVolumeKmsKey()
     * - .ifVpcSecurityGroupIds()
     * - .ifVpcSubnets()
     * - .ifInterContainerTrafficEncryption()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateProcessingJob.html
     */
    toCreateProcessingJob() {
        this.add('sagemaker:CreateProcessingJob');
        return this;
    }
    /**
     * Starts a model training job. After training completes, Amazon SageMaker saves the resulting model artifacts and other optional output to an Amazon S3 location that you specify.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifFileSystemAccessMode()
     * - .ifFileSystemDirectoryPath()
     * - .ifFileSystemId()
     * - .ifFileSystemType()
     * - .ifInstanceTypes()
     * - .ifInterContainerTrafficEncryption()
     * - .ifMaxRuntimeInSeconds()
     * - .ifNetworkIsolation()
     * - .ifOutputKmsKey()
     * - .ifVolumeKmsKey()
     * - .ifVpcSecurityGroupIds()
     * - .ifVpcSubnets()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateTrainingJob.html
     */
    toCreateTrainingJob() {
        this.add('sagemaker:CreateTrainingJob');
        return this;
    }
    /**
     * Starts a transform job. After the results are obtained, Amazon SageMaker saves them to an Amazon S3 location that you specify.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifInstanceTypes()
     * - .ifModelArn()
     * - .ifOutputKmsKey()
     * - .ifVolumeKmsKey()
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateTransformJob.html
     */
    toCreateTransformJob() {
        this.add('sagemaker:CreateTransformJob');
        return this;
    }
    /**
     * Create a trial.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateTrial.html
     */
    toCreateTrial() {
        this.add('sagemaker:CreateTrial');
        return this;
    }
    /**
     * Create a trial component.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateTrialComponent.html
     */
    toCreateTrialComponent() {
        this.add('sagemaker:CreateTrialComponent');
        return this;
    }
    /**
     * Grants permission to create a UserProfile for a SageMaker Studio Domain
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifVpcSecurityGroupIds()
     * - .ifInstanceTypes()
     * - .ifDomainSharingOutputKmsKey()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateUserProfile.html
     */
    toCreateUserProfile() {
        this.add('sagemaker:CreateUserProfile');
        return this;
    }
    /**
     * Create a workforce.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateWorkforce.html
     */
    toCreateWorkforce() {
        this.add('sagemaker:CreateWorkforce');
        return this;
    }
    /**
     * Create a workteam.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateWorkteam.html
     */
    toCreateWorkteam() {
        this.add('sagemaker:CreateWorkteam');
        return this;
    }
    /**
     * Deletes an algorithm.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DeleteAlgorithm.html
     */
    toDeleteAlgorithm() {
        this.add('sagemaker:DeleteAlgorithm');
        return this;
    }
    /**
     * Grants permission to delete an App
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DeleteApp.html
     */
    toDeleteApp() {
        this.add('sagemaker:DeleteApp');
        return this;
    }
    /**
     * Deletes a code repository.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DeleteCodeRepository.html
     */
    toDeleteCodeRepository() {
        this.add('sagemaker:DeleteCodeRepository');
        return this;
    }
    /**
     * Grants permission to delete a Domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DeleteDomain.html
     */
    toDeleteDomain() {
        this.add('sagemaker:DeleteDomain');
        return this;
    }
    /**
     * Deletes an endpoint. Amazon SageMaker frees up all the resources that were deployed when the endpoint was created.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DeleteEndpoint.html
     */
    toDeleteEndpoint() {
        this.add('sagemaker:DeleteEndpoint');
        return this;
    }
    /**
     * Deletes the endpoint configuration created using the CreateEndpointConfig API. The DeleteEndpointConfig API deletes only the specified configuration. It does not delete any endpoints created using the configuration.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DeleteEndpointConfig.html
     */
    toDeleteEndpointConfig() {
        this.add('sagemaker:DeleteEndpointConfig');
        return this;
    }
    /**
     * Deletes an experiment.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DeleteExperiment.html
     */
    toDeleteExperiment() {
        this.add('sagemaker:DeleteExperiment');
        return this;
    }
    /**
     * Deltes the specified flow definition.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DeleteFlowDefinition.html
     */
    toDeleteFlowDefinition() {
        this.add('sagemaker:DeleteFlowDefinition');
        return this;
    }
    /**
     * Deletes the specified human loop.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DeleteHumanLoop.html
     */
    toDeleteHumanLoop() {
        this.add('sagemaker:DeleteHumanLoop');
        return this;
    }
    /**
     * Deletes a model created using the CreateModel API. The DeleteModel API deletes only the model entry in Amazon SageMaker that you created by calling the CreateModel API. It does not delete model artifacts, inference code, or the IAM role that you specified when creating the model.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DeleteModel.html
     */
    toDeleteModel() {
        this.add('sagemaker:DeleteModel');
        return this;
    }
    /**
     * Deletes a model package.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DeleteModelPackage.html
     */
    toDeleteModelPackage() {
        this.add('sagemaker:DeleteModelPackage');
        return this;
    }
    /**
     * Deletes a monitoring schedule. Amazon SageMaker will no longer run the scheduled monitoring.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DeleteMonitoringSchedule.html
     */
    toDeleteMonitoringSchedule() {
        this.add('sagemaker:DeleteMonitoringSchedule');
        return this;
    }
    /**
     * Deletes an Amazon SageMaker notebook instance. Before you can delete a notebook instance, you must call the StopNotebookInstance API.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DeleteNotebookInstance.html
     */
    toDeleteNotebookInstance() {
        this.add('sagemaker:DeleteNotebookInstance');
        return this;
    }
    /**
     * Deletes an notebook instance lifecycle configuration that can be deployed using Amazon SageMaker.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DeleteNotebookInstanceLifecycleConfig.html
     */
    toDeleteNotebookInstanceLifecycleConfig() {
        this.add('sagemaker:DeleteNotebookInstanceLifecycleConfig');
        return this;
    }
    /**
     * Deletes the specified set of tags from an Amazon SageMaker resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DeleteTags.html
     */
    toDeleteTags() {
        this.add('sagemaker:DeleteTags');
        return this;
    }
    /**
     * Deletes a trial.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DeleteTrial.html
     */
    toDeleteTrial() {
        this.add('sagemaker:DeleteTrial');
        return this;
    }
    /**
     * Deletes a trial component.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DeleteTrialComponent.html
     */
    toDeleteTrialComponent() {
        this.add('sagemaker:DeleteTrialComponent');
        return this;
    }
    /**
     * Grants permission to delete a UserProfile
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DeleteUserProfile.html
     */
    toDeleteUserProfile() {
        this.add('sagemaker:DeleteUserProfile');
        return this;
    }
    /**
     * Deletes a workforce.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DeleteWorkforce.html
     */
    toDeleteWorkforce() {
        this.add('sagemaker:DeleteWorkforce');
        return this;
    }
    /**
     * Deletes a workteam.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DeleteWorkteam.html
     */
    toDeleteWorkteam() {
        this.add('sagemaker:DeleteWorkteam');
        return this;
    }
    /**
     * Returns information about an algorithm.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeAlgorithm.html
     */
    toDescribeAlgorithm() {
        this.add('sagemaker:DescribeAlgorithm');
        return this;
    }
    /**
     * Grants permission to describe an App
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeApp.html
     */
    toDescribeApp() {
        this.add('sagemaker:DescribeApp');
        return this;
    }
    /**
     * Describes an automl job that was created via CreateAutoMLJob API.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeAutoMLJob.html
     */
    toDescribeAutoMLJob() {
        this.add('sagemaker:DescribeAutoMLJob');
        return this;
    }
    /**
     * Returns information about a code repository.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeCodeRepository.html
     */
    toDescribeCodeRepository() {
        this.add('sagemaker:DescribeCodeRepository');
        return this;
    }
    /**
     * Returns information about a compilation job.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeCompilationJob.html
     */
    toDescribeCompilationJob() {
        this.add('sagemaker:DescribeCompilationJob');
        return this;
    }
    /**
     * Grants permission to describe a Domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeDomain.html
     */
    toDescribeDomain() {
        this.add('sagemaker:DescribeDomain');
        return this;
    }
    /**
     * Returns the description of an endpoint.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeEndpoint.html
     */
    toDescribeEndpoint() {
        this.add('sagemaker:DescribeEndpoint');
        return this;
    }
    /**
     * Returns the description of an endpoint configuration, which was created using the CreateEndpointConfig API.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeEndpointConfig.html
     */
    toDescribeEndpointConfig() {
        this.add('sagemaker:DescribeEndpointConfig');
        return this;
    }
    /**
     * Returns information about an experiment.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeExperiment.html
     */
    toDescribeExperiment() {
        this.add('sagemaker:DescribeExperiment');
        return this;
    }
    /**
     * Returns detailed information about the specified flow definition.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeFlowDefinition.html
     */
    toDescribeFlowDefinition() {
        this.add('sagemaker:DescribeFlowDefinition');
        return this;
    }
    /**
     * Returns detailed information about the specified human loop.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeHumanLoop.html
     */
    toDescribeHumanLoop() {
        this.add('sagemaker:DescribeHumanLoop');
        return this;
    }
    /**
     * Returns detailed information about the specified human review workflow user interface.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeHumanTaskUi.html
     */
    toDescribeHumanTaskUi() {
        this.add('sagemaker:DescribeHumanTaskUi');
        return this;
    }
    /**
     * Describes a hyper parameter tuning job that was created via CreateHyperParameterTuningJob API.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeHyperParameterTuningJob.html
     */
    toDescribeHyperParameterTuningJob() {
        this.add('sagemaker:DescribeHyperParameterTuningJob');
        return this;
    }
    /**
     * Returns information about a labeling job.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeLabelingJob.html
     */
    toDescribeLabelingJob() {
        this.add('sagemaker:DescribeLabelingJob');
        return this;
    }
    /**
     * Describes a model that you created using the CreateModel API.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeModel.html
     */
    toDescribeModel() {
        this.add('sagemaker:DescribeModel');
        return this;
    }
    /**
     * Returns information about a model package.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeModelPackage.html
     */
    toDescribeModelPackage() {
        this.add('sagemaker:DescribeModelPackage');
        return this;
    }
    /**
     * Returns information about a monitoring schedule.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeMonitoringSchedule.html
     */
    toDescribeMonitoringSchedule() {
        this.add('sagemaker:DescribeMonitoringSchedule');
        return this;
    }
    /**
     * Returns information about a notebook instance.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeNotebookInstance.html
     */
    toDescribeNotebookInstance() {
        this.add('sagemaker:DescribeNotebookInstance');
        return this;
    }
    /**
     * Describes an notebook instance lifecycle configuration that was created via CreateNotebookInstanceLifecycleConfig API.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeNotebookInstanceLifecycleConfig.html
     */
    toDescribeNotebookInstanceLifecycleConfig() {
        this.add('sagemaker:DescribeNotebookInstanceLifecycleConfig');
        return this;
    }
    /**
     * Returns information about a processing job.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeProcessingJob.html
     */
    toDescribeProcessingJob() {
        this.add('sagemaker:DescribeProcessingJob');
        return this;
    }
    /**
     * Returns information about a subscribed workteam.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeSubscribedWorkteam.html
     */
    toDescribeSubscribedWorkteam() {
        this.add('sagemaker:DescribeSubscribedWorkteam');
        return this;
    }
    /**
     * Returns information about a training job.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeTrainingJob.html
     */
    toDescribeTrainingJob() {
        this.add('sagemaker:DescribeTrainingJob');
        return this;
    }
    /**
     * Returns information about a transform job.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeTransformJob.html
     */
    toDescribeTransformJob() {
        this.add('sagemaker:DescribeTransformJob');
        return this;
    }
    /**
     * Returns information about a trial.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeTrial.html
     */
    toDescribeTrial() {
        this.add('sagemaker:DescribeTrial');
        return this;
    }
    /**
     * Returns information about a trial component.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeTrialComponent.html
     */
    toDescribeTrialComponent() {
        this.add('sagemaker:DescribeTrialComponent');
        return this;
    }
    /**
     * Grants permission to describe a UserProfile
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeUserProfile.html
     */
    toDescribeUserProfile() {
        this.add('sagemaker:DescribeUserProfile');
        return this;
    }
    /**
     * Returns information about a workforce.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeWorkforce.html
     */
    toDescribeWorkforce() {
        this.add('sagemaker:DescribeWorkforce');
        return this;
    }
    /**
     * Returns information about a workteam.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DescribeWorkteam.html
     */
    toDescribeWorkteam() {
        this.add('sagemaker:DescribeWorkteam');
        return this;
    }
    /**
     * Disassociate a trial component with a trial.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_DisassociateTrialComponent.html
     */
    toDisassociateTrialComponent() {
        this.add('sagemaker:DisassociateTrialComponent');
        return this;
    }
    /**
     * Get search suggestions when provided with keyword.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_GetSearchSuggestions.html
     */
    toGetSearchSuggestions() {
        this.add('sagemaker:GetSearchSuggestions');
        return this;
    }
    /**
     * After you deploy a model into production using Amazon SageMaker hosting services, your client applications use this API to get inferences from the model hosted at the specified endpoint.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifTargetModel()
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_InvokeEndpoint.html
     */
    toInvokeEndpoint() {
        this.add('sagemaker:InvokeEndpoint');
        return this;
    }
    /**
     * Lists algorithms.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListAlgorithms.html
     */
    toListAlgorithms() {
        this.add('sagemaker:ListAlgorithms');
        return this;
    }
    /**
     * Grants permission to list the Apps in your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListApps.html
     */
    toListApps() {
        this.add('sagemaker:ListApps');
        return this;
    }
    /**
     * Lists automl jobs created via the CreateAutoMLJob.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListAutoMLJobs.html
     */
    toListAutoMLJobs() {
        this.add('sagemaker:ListAutoMLJobs');
        return this;
    }
    /**
     * Lists candidates for automl job created via the CreateAutoMLJob.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListCandidatesForAutoMLJob.html
     */
    toListCandidatesForAutoMLJob() {
        this.add('sagemaker:ListCandidatesForAutoMLJob');
        return this;
    }
    /**
     * Lists code repositories.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListCodeRepositories.html
     */
    toListCodeRepositories() {
        this.add('sagemaker:ListCodeRepositories');
        return this;
    }
    /**
     * Lists compilation jobs.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListCompilationJobs.html
     */
    toListCompilationJobs() {
        this.add('sagemaker:ListCompilationJobs');
        return this;
    }
    /**
     * Grants permission to list the Domains in your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListDomains.html
     */
    toListDomains() {
        this.add('sagemaker:ListDomains');
        return this;
    }
    /**
     * Lists endpoint configurations.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListEndpointConfigs.html
     */
    toListEndpointConfigs() {
        this.add('sagemaker:ListEndpointConfigs');
        return this;
    }
    /**
     * Lists endpoints.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListEndpoints.html
     */
    toListEndpoints() {
        this.add('sagemaker:ListEndpoints');
        return this;
    }
    /**
     * Lists experiments.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListExperiments.html
     */
    toListExperiments() {
        this.add('sagemaker:ListExperiments');
        return this;
    }
    /**
     * Returns summary information about flow definitions, given the specified parameters.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListFlowDefinitions.html
     */
    toListFlowDefinitions() {
        this.add('sagemaker:ListFlowDefinitions');
        return this;
    }
    /**
     * Returns summary information about human loops, given the specified parameters.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListHumanLoops.html
     */
    toListHumanLoops() {
        this.add('sagemaker:ListHumanLoops');
        return this;
    }
    /**
     * Returns summary information about human review workflow user interfaces, given the specified parameters.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListHumanTaskUis.html
     */
    toListHumanTaskUis() {
        this.add('sagemaker:ListHumanTaskUis');
        return this;
    }
    /**
     * Lists hyper parameter tuning jobs that was created using Amazon SageMaker.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListHyperParameterTuningJobs.html
     */
    toListHyperParameterTuningJobs() {
        this.add('sagemaker:ListHyperParameterTuningJobs');
        return this;
    }
    /**
     * Lists labeling jobs.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListLabelingJobs.html
     */
    toListLabelingJobs() {
        this.add('sagemaker:ListLabelingJobs');
        return this;
    }
    /**
     * Lists labeling jobs for workteam.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListLabelingJobs.html
     */
    toListLabelingJobsForWorkteam() {
        this.add('sagemaker:ListLabelingJobsForWorkteam');
        return this;
    }
    /**
     * Lists model packages.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListModelPackages.html
     */
    toListModelPackages() {
        this.add('sagemaker:ListModelPackages');
        return this;
    }
    /**
     * Lists the models created with the CreateModel API.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListModels.html
     */
    toListModels() {
        this.add('sagemaker:ListModels');
        return this;
    }
    /**
     * Lists monitoring executions.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListMonitoringExecutions.html
     */
    toListMonitoringExecutions() {
        this.add('sagemaker:ListMonitoringExecutions');
        return this;
    }
    /**
     * Lists monitoring schedules.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListMonitoringSchedules.html
     */
    toListMonitoringSchedules() {
        this.add('sagemaker:ListMonitoringSchedules');
        return this;
    }
    /**
     * Lists notebook instance lifecycle configurations that can be deployed using Amazon SageMaker.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListNotebookInstanceLifecycleConfigs.html
     */
    toListNotebookInstanceLifecycleConfigs() {
        this.add('sagemaker:ListNotebookInstanceLifecycleConfigs');
        return this;
    }
    /**
     * Returns a list of the Amazon SageMaker notebook instances in the requester's account in an AWS Region.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListNotebookInstances.html
     */
    toListNotebookInstances() {
        this.add('sagemaker:ListNotebookInstances');
        return this;
    }
    /**
     * Lists processing jobs.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListProcessingJobs.html
     */
    toListProcessingJobs() {
        this.add('sagemaker:ListProcessingJobs');
        return this;
    }
    /**
     * Lists subscribed workteams.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListSubscribedWorkteams.html
     */
    toListSubscribedWorkteams() {
        this.add('sagemaker:ListSubscribedWorkteams');
        return this;
    }
    /**
     * Returns the tag set associated with the specified resource.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListTags.html
     */
    toListTags() {
        this.add('sagemaker:ListTags');
        return this;
    }
    /**
     * Lists training jobs.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListTrainingJobs.html
     */
    toListTrainingJobs() {
        this.add('sagemaker:ListTrainingJobs');
        return this;
    }
    /**
     * Lists training jobs for a hyper parameter tuning job that was created using Amazon SageMaker.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_ ListTrainingJobsForHyperParameterTuningJob.html
     */
    toListTrainingJobsForHyperParameterTuningJob() {
        this.add('sagemaker:ListTrainingJobsForHyperParameterTuningJob');
        return this;
    }
    /**
     * Lists transform jobs.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListTransformJobs.html
     */
    toListTransformJobs() {
        this.add('sagemaker:ListTransformJobs');
        return this;
    }
    /**
     * Lists trial components.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListTrialComponents.html
     */
    toListTrialComponents() {
        this.add('sagemaker:ListTrialComponents');
        return this;
    }
    /**
     * Lists trials.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListTrials.html
     */
    toListTrials() {
        this.add('sagemaker:ListTrials');
        return this;
    }
    /**
     * Grants permission to list the UserProfiles in your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListUserProfiles.html
     */
    toListUserProfiles() {
        this.add('sagemaker:ListUserProfiles');
        return this;
    }
    /**
     * Lists workforces.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListWorkforces.html
     */
    toListWorkforces() {
        this.add('sagemaker:ListWorkforces');
        return this;
    }
    /**
     * Lists workteams.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_ListWorkteams.html
     */
    toListWorkteams() {
        this.add('sagemaker:ListWorkteams');
        return this;
    }
    /**
     * Render a UI template used for a human annotation task.
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_RenderUiTemplate.html
     */
    toRenderUiTemplate() {
        this.add('sagemaker:RenderUiTemplate');
        return this;
    }
    /**
     * Search for SageMaker objects.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_Search.html
     */
    toSearch() {
        this.add('sagemaker:Search');
        return this;
    }
    /**
     * Starts a human loop.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_StartHumanLoop.html
     */
    toStartHumanLoop() {
        this.add('sagemaker:StartHumanLoop');
        return this;
    }
    /**
     * Starts a monitoring schedule.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_StartMonitoringSchedule.html
     */
    toStartMonitoringSchedule() {
        this.add('sagemaker:StartMonitoringSchedule');
        return this;
    }
    /**
     * Launches an EC2 instance with the latest version of the libraries and attaches your EBS volume.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_StartNotebookInstance.html
     */
    toStartNotebookInstance() {
        this.add('sagemaker:StartNotebookInstance');
        return this;
    }
    /**
     * Stops a running automl job created via the CreateAutoMLJob.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_StopAutoMLJob.html
     */
    toStopAutoMLJob() {
        this.add('sagemaker:StopAutoMLJob');
        return this;
    }
    /**
     * Stops a compilation job.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_StopCompilationJob.html
     */
    toStopCompilationJob() {
        this.add('sagemaker:StopCompilationJob');
        return this;
    }
    /**
     * Stops the specified human loop.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_StopHumanLoop.html
     */
    toStopHumanLoop() {
        this.add('sagemaker:StopHumanLoop');
        return this;
    }
    /**
     * Stops a running hyper parameter tuning job create via the CreateHyperParameterTuningJob.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_StopHyperParameterTuningJob.html
     */
    toStopHyperParameterTuningJob() {
        this.add('sagemaker:StopHyperParameterTuningJob');
        return this;
    }
    /**
     * Stops a labeling job. Any labels already generated will be exported before stopping.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_StopLabelingJob.html
     */
    toStopLabelingJob() {
        this.add('sagemaker:StopLabelingJob');
        return this;
    }
    /**
     * Stops a monitoring schedule.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_StopMonitoringSchedule.html
     */
    toStopMonitoringSchedule() {
        this.add('sagemaker:StopMonitoringSchedule');
        return this;
    }
    /**
     * Terminates the EC2 instance. Before terminating the instance, Amazon SageMaker disconnects the EBS volume from it. Amazon SageMaker preserves the EBS volume.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_StopNotebookInstance.html
     */
    toStopNotebookInstance() {
        this.add('sagemaker:StopNotebookInstance');
        return this;
    }
    /**
     * Stops a processing job. To stop a job, Amazon SageMaker sends the algorithm the SIGTERM signal, which delays job termination for 120 seconds.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_StopProcessingJob.html
     */
    toStopProcessingJob() {
        this.add('sagemaker:StopProcessingJob');
        return this;
    }
    /**
     * Stops a training job. To stop a job, Amazon SageMaker sends the algorithm the SIGTERM signal, which delays job termination for 120 seconds.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_StopTrainingJob.html
     */
    toStopTrainingJob() {
        this.add('sagemaker:StopTrainingJob');
        return this;
    }
    /**
     * Stops a transform job. When Amazon SageMaker receives a StopTransformJob request, the status of the job changes to Stopping. After Amazon SageMaker stops the job, the status is set to Stopped
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_StopTransformJob.html
     */
    toStopTransformJob() {
        this.add('sagemaker:StopTransformJob');
        return this;
    }
    /**
     * Updates a code repository.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_UpdateCodeRepository.html
     */
    toUpdateCodeRepository() {
        this.add('sagemaker:UpdateCodeRepository');
        return this;
    }
    /**
     * Grants permission to update a Domain
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifVpcSecurityGroupIds()
     * - .ifInstanceTypes()
     * - .ifDomainSharingOutputKmsKey()
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_UpdateDomain.html
     */
    toUpdateDomain() {
        this.add('sagemaker:UpdateDomain');
        return this;
    }
    /**
     * Updates an endpoint to use the endpoint configuration specified in the request.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_UpdateEndpoint.html
     */
    toUpdateEndpoint() {
        this.add('sagemaker:UpdateEndpoint');
        return this;
    }
    /**
     * Updates variant weight, capacity, or both of one or more variants associated with an endpoint.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_UpdateEndpointWeightsAndCapacities.html
     */
    toUpdateEndpointWeightsAndCapacities() {
        this.add('sagemaker:UpdateEndpointWeightsAndCapacities');
        return this;
    }
    /**
     * Updates an experiment.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_UpdateExperiment.html
     */
    toUpdateExperiment() {
        this.add('sagemaker:UpdateExperiment');
        return this;
    }
    /**
     * Updates a monitoring schedule.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifInstanceTypes()
     * - .ifMaxRuntimeInSeconds()
     * - .ifNetworkIsolation()
     * - .ifOutputKmsKey()
     * - .ifVolumeKmsKey()
     * - .ifVpcSecurityGroupIds()
     * - .ifVpcSubnets()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_UpdateMonitoringSchedule.html
     */
    toUpdateMonitoringSchedule() {
        this.add('sagemaker:UpdateMonitoringSchedule');
        return this;
    }
    /**
     * Updates a notebook instance. Notebook instance updates include upgrading or downgrading the EC2 instance used for your notebook instance to accommodate changes in your workload requirements. You can also update the VPC security groups.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAcceleratorTypes()
     * - .ifInstanceTypes()
     * - .ifRootAccess()
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_UpdateNotebookInstance.html
     */
    toUpdateNotebookInstance() {
        this.add('sagemaker:UpdateNotebookInstance');
        return this;
    }
    /**
     * Updates a notebook instance lifecycle configuration created with the CreateNotebookInstanceLifecycleConfig API.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_ UpdateNotebookInstanceLifecycleConfig.html
     */
    toUpdateNotebookInstanceLifecycleConfig() {
        this.add('sagemaker:UpdateNotebookInstanceLifecycleConfig');
        return this;
    }
    /**
     * Updates a trial.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_UpdateTrial.html
     */
    toUpdateTrial() {
        this.add('sagemaker:UpdateTrial');
        return this;
    }
    /**
     * Updates a trial component.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_UpdateTrialComponent.html
     */
    toUpdateTrialComponent() {
        this.add('sagemaker:UpdateTrialComponent');
        return this;
    }
    /**
     * Grants permission to update a UserProfile
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceTypes()
     * - .ifVpcSecurityGroupIds()
     * - .ifInstanceTypes()
     * - .ifDomainSharingOutputKmsKey()
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_UpdateUserProfile.html
     */
    toUpdateUserProfile() {
        this.add('sagemaker:UpdateUserProfile');
        return this;
    }
    /**
     * Updates a workforce.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_UpdateWorkforce.html
     */
    toUpdateWorkforce() {
        this.add('sagemaker:UpdateWorkforce');
        return this;
    }
    /**
     * Updates a workteam.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sagemaker/latest/dg/API_UpdateWorkteam.html
     */
    toUpdateWorkteam() {
        this.add('sagemaker:UpdateWorkteam');
        return this;
    }
    /**
     * Adds a resource of type human-loop to the statement
     *
     * @param humanLoopName - Identifier for the humanLoopName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onHumanLoop(humanLoopName, account, region, partition) {
        var arn = 'arn:${Partition}:sagemaker:${Region}:${Account}:human-loop/${HumanLoopName}';
        arn = arn.replace('${HumanLoopName}', humanLoopName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type flow-definition to the statement
     *
     * @param flowDefinitionName - Identifier for the flowDefinitionName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onFlowDefinition(flowDefinitionName, account, region, partition) {
        var arn = 'arn:${Partition}:sagemaker:${Region}:${Account}:flow-definition/${FlowDefinitionName}';
        arn = arn.replace('${FlowDefinitionName}', flowDefinitionName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type human-task-ui to the statement
     *
     * @param humanTaskUiName - Identifier for the humanTaskUiName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onHumanTaskUi(humanTaskUiName, account, region, partition) {
        var arn = 'arn:${Partition}:sagemaker:${Region}:${Account}:human-task-ui/${HumanTaskUiName}';
        arn = arn.replace('${HumanTaskUiName}', humanTaskUiName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type labeling-job to the statement
     *
     * @param labelingJobName - Identifier for the labelingJobName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onLabelingJob(labelingJobName, account, region, partition) {
        var arn = 'arn:${Partition}:sagemaker:${Region}:${Account}:labeling-job/${LabelingJobName}';
        arn = arn.replace('${LabelingJobName}', labelingJobName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type workteam to the statement
     *
     * @param workteamName - Identifier for the workteamName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onWorkteam(workteamName, account, region, partition) {
        var arn = 'arn:${Partition}:sagemaker:${Region}:${Account}:workteam/${WorkteamName}';
        arn = arn.replace('${WorkteamName}', workteamName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type workforce to the statement
     *
     * @param workforceName - Identifier for the workforceName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onWorkforce(workforceName, account, region, partition) {
        var arn = 'arn:${Partition}:sagemaker:${Region}:${Account}:workforce/${WorkforceName}';
        arn = arn.replace('${WorkforceName}', workforceName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type domain to the statement
     *
     * @param domainId - Identifier for the domainId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onDomain(domainId, account, region, partition) {
        var arn = 'arn:${Partition}:sagemaker:${Region}:${Account}:domain/${DomainId}';
        arn = arn.replace('${DomainId}', domainId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type user-profile to the statement
     *
     * @param domainId - Identifier for the domainId.
     * @param userProfileName - Identifier for the userProfileName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onUserProfile(domainId, userProfileName, account, region, partition) {
        var arn = 'arn:${Partition}:sagemaker:${Region}:${Account}:user-profile/${DomainId}/${UserProfileName}';
        arn = arn.replace('${DomainId}', domainId);
        arn = arn.replace('${UserProfileName}', userProfileName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type app to the statement
     *
     * @param domainId - Identifier for the domainId.
     * @param userProfileName - Identifier for the userProfileName.
     * @param appType - Identifier for the appType.
     * @param appName - Identifier for the appName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onApp(domainId, userProfileName, appType, appName, account, region, partition) {
        var arn = 'arn:${Partition}:sagemaker:${Region}:${Account}:app/${DomainId}/${UserProfileName}/${AppType}/${AppName}';
        arn = arn.replace('${DomainId}', domainId);
        arn = arn.replace('${UserProfileName}', userProfileName);
        arn = arn.replace('${AppType}', appType);
        arn = arn.replace('${AppName}', appName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type notebook-instance to the statement
     *
     * @param notebookInstanceName - Identifier for the notebookInstanceName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onNotebookInstance(notebookInstanceName, account, region, partition) {
        var arn = 'arn:${Partition}:sagemaker:${Region}:${Account}:notebook-instance/${NotebookInstanceName}';
        arn = arn.replace('${NotebookInstanceName}', notebookInstanceName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type notebook-instance-lifecycle-config to the statement
     *
     * @param notebookInstanceLifecycleConfigName - Identifier for the notebookInstanceLifecycleConfigName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onNotebookInstanceLifecycleConfig(notebookInstanceLifecycleConfigName, account, region, partition) {
        var arn = 'arn:${Partition}:sagemaker:${Region}:${Account}:notebook-instance-lifecycle-config/${NotebookInstanceLifecycleConfigName}';
        arn = arn.replace('${NotebookInstanceLifecycleConfigName}', notebookInstanceLifecycleConfigName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type code-repository to the statement
     *
     * @param codeRepositoryName - Identifier for the codeRepositoryName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onCodeRepository(codeRepositoryName, account, region, partition) {
        var arn = 'arn:${Partition}:sagemaker:${Region}:${Account}:code-repository/${CodeRepositoryName}';
        arn = arn.replace('${CodeRepositoryName}', codeRepositoryName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type algorithm to the statement
     *
     * @param algorithmName - Identifier for the algorithmName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onAlgorithm(algorithmName, account, region, partition) {
        var arn = 'arn:${Partition}:sagemaker:${Region}:${Account}:algorithm/${AlgorithmName}';
        arn = arn.replace('${AlgorithmName}', algorithmName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type training-job to the statement
     *
     * @param trainingJobName - Identifier for the trainingJobName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onTrainingJob(trainingJobName, account, region, partition) {
        var arn = 'arn:${Partition}:sagemaker:${Region}:${Account}:training-job/${TrainingJobName}';
        arn = arn.replace('${TrainingJobName}', trainingJobName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type processing-job to the statement
     *
     * @param processingJobName - Identifier for the processingJobName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onProcessingJob(processingJobName, account, region, partition) {
        var arn = 'arn:${Partition}:sagemaker:${Region}:${Account}:processing-job/${ProcessingJobName}';
        arn = arn.replace('${ProcessingJobName}', processingJobName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type hyper-parameter-tuning-job to the statement
     *
     * @param hyperParameterTuningJobName - Identifier for the hyperParameterTuningJobName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onHyperParameterTuningJob(hyperParameterTuningJobName, account, region, partition) {
        var arn = 'arn:${Partition}:sagemaker:${Region}:${Account}:hyper-parameter-tuning-job/${HyperParameterTuningJobName}';
        arn = arn.replace('${HyperParameterTuningJobName}', hyperParameterTuningJobName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type model-package to the statement
     *
     * @param modelPackageName - Identifier for the modelPackageName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onModelPackage(modelPackageName, account, region, partition) {
        var arn = 'arn:${Partition}:sagemaker:${Region}:${Account}:model-package/${ModelPackageName}';
        arn = arn.replace('${ModelPackageName}', modelPackageName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type model to the statement
     *
     * @param modelName - Identifier for the modelName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onModel(modelName, account, region, partition) {
        var arn = 'arn:${Partition}:sagemaker:${Region}:${Account}:model/${ModelName}';
        arn = arn.replace('${ModelName}', modelName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type endpoint-config to the statement
     *
     * @param endpointConfigName - Identifier for the endpointConfigName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onEndpointConfig(endpointConfigName, account, region, partition) {
        var arn = 'arn:${Partition}:sagemaker:${Region}:${Account}:endpoint-config/${EndpointConfigName}';
        arn = arn.replace('${EndpointConfigName}', endpointConfigName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type endpoint to the statement
     *
     * @param endpointName - Identifier for the endpointName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onEndpoint(endpointName, account, region, partition) {
        var arn = 'arn:${Partition}:sagemaker:${Region}:${Account}:endpoint/${EndpointName}';
        arn = arn.replace('${EndpointName}', endpointName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type transform-job to the statement
     *
     * @param transformJobName - Identifier for the transformJobName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onTransformJob(transformJobName, account, region, partition) {
        var arn = 'arn:${Partition}:sagemaker:${Region}:${Account}:transform-job/${TransformJobName}';
        arn = arn.replace('${TransformJobName}', transformJobName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type compilation-job to the statement
     *
     * @param compilationJobName - Identifier for the compilationJobName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onCompilationJob(compilationJobName, account, region, partition) {
        var arn = 'arn:${Partition}:sagemaker:${Region}:${Account}:compilation-job/${CompilationJobName}';
        arn = arn.replace('${CompilationJobName}', compilationJobName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type automl-job to the statement
     *
     * @param autoMLJobJobName - Identifier for the autoMLJobJobName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onAutomlJob(autoMLJobJobName, account, region, partition) {
        var arn = 'arn:${Partition}:sagemaker:${Region}:${Account}:automl-job/${AutoMLJobJobName}';
        arn = arn.replace('${AutoMLJobJobName}', autoMLJobJobName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type monitoring-schedule to the statement
     *
     * @param monitoringScheduleName - Identifier for the monitoringScheduleName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onMonitoringSchedule(monitoringScheduleName, account, region, partition) {
        var arn = 'arn:${Partition}:sagemaker:${Region}:${Account}:monitoring-schedule/${MonitoringScheduleName}';
        arn = arn.replace('${MonitoringScheduleName}', monitoringScheduleName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type experiment to the statement
     *
     * @param experimentName - Identifier for the experimentName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onExperiment(experimentName, account, region, partition) {
        var arn = 'arn:${Partition}:sagemaker:${Region}:${Account}:experiment/${ExperimentName}';
        arn = arn.replace('${ExperimentName}', experimentName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type experiment-trial to the statement
     *
     * @param trialName - Identifier for the trialName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onExperimentTrial(trialName, account, region, partition) {
        var arn = 'arn:${Partition}:sagemaker:${Region}:${Account}:experiment-trial/${TrialName}';
        arn = arn.replace('${TrialName}', trialName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type experiment-trial-component to the statement
     *
     * @param trialComponentName - Identifier for the trialComponentName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onExperimentTrialComponent(trialComponentName, account, region, partition) {
        var arn = 'arn:${Partition}:sagemaker:${Region}:${Account}:experiment-trial-component/${TrialComponentName}';
        arn = arn.replace('${TrialComponentName}', trialComponentName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * The list of all accelerator types associated with the resource in the request.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .toCreateEndpointConfig()
     * - .toCreateNotebookInstance()
     * - .toUpdateNotebookInstance()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAcceleratorTypes(value, operator) {
        return this.if(`sagemaker:AcceleratorTypes`, value, operator || 'StringLike');
    }
    /**
     * App network access associated with the resource in the request.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .toCreateDomain()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAppNetworkAccess(value, operator) {
        return this.if(`sagemaker:AppNetworkAccess`, value, operator || 'StringLike');
    }
    /**
     * The direct internet access associated with the resource in the request.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .toCreateNotebookInstance()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDirectInternetAccess(value, operator) {
        return this.if(`sagemaker:DirectInternetAccess`, value, operator || 'StringLike');
    }
    /**
     * The Domain sharing output KMS key associated with the resource in the request.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .toCreateDomain()
     * - .toCreateUserProfile()
     * - .toUpdateDomain()
     * - .toUpdateUserProfile()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifDomainSharingOutputKmsKey(value, operator) {
        return this.if(`sagemaker:DomainSharingOutputKmsKey`, value, operator || 'ArnEquals');
    }
    /**
     * File system access mode associated with the resource in the request.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .toCreateHyperParameterTuningJob()
     * - .toCreateTrainingJob()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifFileSystemAccessMode(value, operator) {
        return this.if(`sagemaker:FileSystemAccessMode`, value, operator || 'StringLike');
    }
    /**
     * File system directory path associated with the resource in the request.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .toCreateHyperParameterTuningJob()
     * - .toCreateTrainingJob()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifFileSystemDirectoryPath(value, operator) {
        return this.if(`sagemaker:FileSystemDirectoryPath`, value, operator || 'StringLike');
    }
    /**
     * A file system ID associated with the resource in the request.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .toCreateHyperParameterTuningJob()
     * - .toCreateTrainingJob()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifFileSystemId(value, operator) {
        return this.if(`sagemaker:FileSystemId`, value, operator || 'StringLike');
    }
    /**
     * File system type associated with the resource in the request.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .toCreateHyperParameterTuningJob()
     * - .toCreateTrainingJob()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifFileSystemType(value, operator) {
        return this.if(`sagemaker:FileSystemType`, value, operator || 'StringLike');
    }
    /**
     * The KMS Key Id of the EFS File System used for UserProfile home directories, which is associated with the resource in the request.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .toCreateDomain()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifHomeEfsFileSystemKmsKey(value, operator) {
        return this.if(`sagemaker:HomeEfsFileSystemKmsKey`, value, operator || 'ArnEquals');
    }
    /**
     * The list of all instance types associated with the resource in the request.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .toCreateApp()
     * - .toCreateDomain()
     * - .toCreateEndpointConfig()
     * - .toCreateHyperParameterTuningJob()
     * - .toCreateMonitoringSchedule()
     * - .toCreateNotebookInstance()
     * - .toCreateProcessingJob()
     * - .toCreateTrainingJob()
     * - .toCreateTransformJob()
     * - .toCreateUserProfile()
     * - .toUpdateDomain()
     * - .toUpdateMonitoringSchedule()
     * - .toUpdateNotebookInstance()
     * - .toUpdateUserProfile()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifInstanceTypes(value, operator) {
        return this.if(`sagemaker:InstanceTypes`, value, operator || 'StringLike');
    }
    /**
     * The inter container traffic encryption associated with the resource in the request.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .toCreateAutoMLJob()
     * - .toCreateHyperParameterTuningJob()
     * - .toCreateProcessingJob()
     * - .toCreateTrainingJob()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifInterContainerTrafficEncryption(value) {
        return this.if(`sagemaker:InterContainerTrafficEncryption`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * The max runtime in seconds associated with the resource in the request.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .toCreateHyperParameterTuningJob()
     * - .toCreateMonitoringSchedule()
     * - .toCreateProcessingJob()
     * - .toCreateTrainingJob()
     * - .toUpdateMonitoringSchedule()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifMaxRuntimeInSeconds(value, operator) {
        return this.if(`sagemaker:MaxRuntimeInSeconds`, value, operator || 'NumericEquals');
    }
    /**
     * The model arn associated with the resource in the request.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .toCreateEndpointConfig()
     * - .toCreateTransformJob()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifModelArn(value, operator) {
        return this.if(`sagemaker:ModelArn`, value, operator || 'ArnEquals');
    }
    /**
     * The network isolation associated with the resource in the request.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .toCreateHyperParameterTuningJob()
     * - .toCreateModel()
     * - .toCreateMonitoringSchedule()
     * - .toCreateProcessingJob()
     * - .toCreateTrainingJob()
     * - .toUpdateMonitoringSchedule()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifNetworkIsolation(value) {
        return this.if(`sagemaker:NetworkIsolation`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * The output kms key associated with the resource in the request.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .toCreateAutoMLJob()
     * - .toCreateHyperParameterTuningJob()
     * - .toCreateLabelingJob()
     * - .toCreateMonitoringSchedule()
     * - .toCreateProcessingJob()
     * - .toCreateTrainingJob()
     * - .toCreateTransformJob()
     * - .toUpdateMonitoringSchedule()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifOutputKmsKey(value, operator) {
        return this.if(`sagemaker:OutputKmsKey`, value, operator || 'ArnEquals');
    }
    /**
     * The preface string for a tag key and value pair attached to a resource.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceTagExists(value, operator) {
        return this.if(`sagemaker:ResourceTag/`, value, operator || 'StringLike');
    }
    /**
     * A tag key and value pair.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to resource types:
     * - flow-definition
     * - human-task-ui
     * - labeling-job
     * - workteam
     * - workforce
     * - domain
     * - user-profile
     * - app
     * - notebook-instance
     * - training-job
     * - processing-job
     * - hyper-parameter-tuning-job
     * - model
     * - endpoint-config
     * - endpoint
     * - transform-job
     * - automl-job
     * - monitoring-schedule
     * - experiment
     * - experiment-trial
     * - experiment-trial-component
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceTag(tagKey, value, operator) {
        return this.if(`sagemaker:ResourceTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * The root access associated with the resource in the request.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .toCreateNotebookInstance()
     * - .toUpdateNotebookInstance()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRootAccess(value, operator) {
        return this.if(`sagemaker:RootAccess`, value, operator || 'StringLike');
    }
    /**
     * The target model associated with the Multi-Model Endpoint in the request.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .toInvokeEndpoint()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTargetModel(value, operator) {
        return this.if(`sagemaker:TargetModel`, value, operator || 'StringLike');
    }
    /**
     * The volume kms key associated with the resource in the request.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .toCreateAutoMLJob()
     * - .toCreateEndpointConfig()
     * - .toCreateHyperParameterTuningJob()
     * - .toCreateLabelingJob()
     * - .toCreateMonitoringSchedule()
     * - .toCreateNotebookInstance()
     * - .toCreateProcessingJob()
     * - .toCreateTrainingJob()
     * - .toCreateTransformJob()
     * - .toUpdateMonitoringSchedule()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifVolumeKmsKey(value, operator) {
        return this.if(`sagemaker:VolumeKmsKey`, value, operator || 'ArnEquals');
    }
    /**
     * The list of all vpc security group ids associated with the resource in the request.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .toCreateAutoMLJob()
     * - .toCreateDomain()
     * - .toCreateHyperParameterTuningJob()
     * - .toCreateModel()
     * - .toCreateMonitoringSchedule()
     * - .toCreateNotebookInstance()
     * - .toCreateProcessingJob()
     * - .toCreateTrainingJob()
     * - .toCreateUserProfile()
     * - .toUpdateDomain()
     * - .toUpdateMonitoringSchedule()
     * - .toUpdateUserProfile()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifVpcSecurityGroupIds(value, operator) {
        return this.if(`sagemaker:VpcSecurityGroupIds`, value, operator || 'StringLike');
    }
    /**
     * The list of all vpc subnets associated with the resource in the request.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .toCreateAutoMLJob()
     * - .toCreateDomain()
     * - .toCreateHyperParameterTuningJob()
     * - .toCreateModel()
     * - .toCreateMonitoringSchedule()
     * - .toCreateNotebookInstance()
     * - .toCreateProcessingJob()
     * - .toCreateTrainingJob()
     * - .toUpdateMonitoringSchedule()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifVpcSubnets(value, operator) {
        return this.if(`sagemaker:VpcSubnets`, value, operator || 'StringLike');
    }
    /**
     * The workteam arn associated to the request.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .toCreateFlowDefinition()
     * - .toCreateLabelingJob()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifWorkteamArn(value, operator) {
        return this.if(`sagemaker:WorkteamArn`, value, operator || 'ArnEquals');
    }
    /**
     * The workteam type associated to the request. This can be public-crowd, private-crowd or vendor-crowd.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsagemaker.html#amazonsagemaker-policy-keys
     *
     * Applies to actions:
     * - .toCreateFlowDefinition()
     * - .toCreateLabelingJob()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifWorkteamType(value, operator) {
        return this.if(`sagemaker:WorkteamType`, value, operator || 'StringLike');
    }
}
exports.Sagemaker = Sagemaker;
//# sourceMappingURL=data:application/json;base64,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