"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AppmeshPreview = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [appmesh-preview](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsappmeshpreview.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class AppmeshPreview extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [appmesh-preview](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsappmeshpreview.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'appmesh-preview';
        this.actionList = {
            "CreateGatewayRoute": {
                "url": "https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_CreateGatewayRoute.html",
                "description": "Creates a gateway route that is associated with a virtual gateway.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "gatewayRoute": {
                        "required": true
                    },
                    "virtualService": {
                        "required": false
                    }
                }
            },
            "CreateMesh": {
                "url": "https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_CreateMesh.html",
                "description": "Creates a service mesh.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "mesh": {
                        "required": true
                    }
                }
            },
            "CreateRoute": {
                "url": "https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_CreateRoute.html",
                "description": "Creates a route that is associated with a virtual router.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "route": {
                        "required": true
                    },
                    "virtualNode": {
                        "required": false
                    }
                }
            },
            "CreateVirtualGateway": {
                "url": "https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_CreateVirtualGateway.html",
                "description": "Creates a virtual gateway within a service mesh.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "virtualGateway": {
                        "required": true
                    }
                }
            },
            "CreateVirtualNode": {
                "url": "https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_CreateVirtualNode.html",
                "description": "Creates a virtual node within a service mesh.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "virtualNode": {
                        "required": true
                    },
                    "virtualService": {
                        "required": false
                    }
                }
            },
            "CreateVirtualRouter": {
                "url": "https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_CreateVirtualRouter.html",
                "description": "Creates a virtual router within a service mesh.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "virtualRouter": {
                        "required": true
                    }
                }
            },
            "CreateVirtualService": {
                "url": "https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_CreateVirtualService.html",
                "description": "Creates a virtual service within a service mesh.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "virtualService": {
                        "required": true
                    },
                    "virtualNode": {
                        "required": false
                    },
                    "virtualRouter": {
                        "required": false
                    }
                }
            },
            "DeleteGatewayRoute": {
                "url": "https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DeleteGatewayRoute.html",
                "description": "Deletes an existing gateway route.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "gatewayRoute": {
                        "required": true
                    }
                }
            },
            "DeleteMesh": {
                "url": "https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DeleteMesh.html",
                "description": "Deletes an existing service mesh.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "mesh": {
                        "required": true
                    }
                }
            },
            "DeleteRoute": {
                "url": "https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DeleteRoute.html",
                "description": "Deletes an existing route.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "route": {
                        "required": true
                    }
                }
            },
            "DeleteVirtualGateway": {
                "url": "https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DeleteVirtualGateway.html",
                "description": "Deletes an existing virtual gateway.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "virtualGateway": {
                        "required": true
                    }
                }
            },
            "DeleteVirtualNode": {
                "url": "https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DeleteVirtualNode.html",
                "description": "Deletes an existing virtual node.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "virtualNode": {
                        "required": true
                    }
                }
            },
            "DeleteVirtualRouter": {
                "url": "https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DeleteVirtualRouter.html",
                "description": "Deletes an existing virtual router.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "virtualRouter": {
                        "required": true
                    }
                }
            },
            "DeleteVirtualService": {
                "url": "https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DeleteVirtualService.html",
                "description": "Deletes an existing virtual service.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "virtualService": {
                        "required": true
                    }
                }
            },
            "DescribeGatewayRoute": {
                "url": "https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DescribeGatewayRoute.html",
                "description": "Describes an existing gateway route.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "gatewayRoute": {
                        "required": true
                    }
                }
            },
            "DescribeMesh": {
                "url": "https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DescribeMesh.html",
                "description": "Describes an existing service mesh.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "mesh": {
                        "required": true
                    }
                }
            },
            "DescribeRoute": {
                "url": "https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DescribeRoute.html",
                "description": "Describes an existing route.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "route": {
                        "required": true
                    }
                }
            },
            "DescribeVirtualGateway": {
                "url": "https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DescribeVirtualGateway.html",
                "description": "Describes an existing virtual gateway.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "virtualGateway": {
                        "required": true
                    }
                }
            },
            "DescribeVirtualNode": {
                "url": "https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DescribeVirtualNode.html",
                "description": "Describes an existing virtual node.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "virtualNode": {
                        "required": true
                    }
                }
            },
            "DescribeVirtualRouter": {
                "url": "https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DescribeVirtualRouter.html",
                "description": "Describes an existing virtual router.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "virtualRouter": {
                        "required": true
                    }
                }
            },
            "DescribeVirtualService": {
                "url": "https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DescribeVirtualService.html",
                "description": "Describes an existing virtual service.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "virtualService": {
                        "required": true
                    }
                }
            },
            "ListGatewayRoutes": {
                "url": "https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_ListGatewayRoutes.html",
                "description": "Returns a list of existing gateway routes in a service mesh.",
                "accessLevel": "List",
                "resourceTypes": {
                    "virtualGateway": {
                        "required": true
                    }
                }
            },
            "ListMeshes": {
                "url": "https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_ListMeshes.html",
                "description": "Returns a list of existing service meshes.",
                "accessLevel": "List"
            },
            "ListRoutes": {
                "url": "https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_ListRoutes.html",
                "description": "Returns a list of existing routes in a service mesh.",
                "accessLevel": "List",
                "resourceTypes": {
                    "virtualRouter": {
                        "required": true
                    }
                }
            },
            "ListVirtualGateways": {
                "url": "https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_ListVirtualGateways.html",
                "description": "Returns a list of existing virtual gateways in a service mesh.",
                "accessLevel": "List",
                "resourceTypes": {
                    "mesh": {
                        "required": true
                    }
                }
            },
            "ListVirtualNodes": {
                "url": "https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_ListVirtualNodes.html",
                "description": "Returns a list of existing virtual nodes.",
                "accessLevel": "List",
                "resourceTypes": {
                    "mesh": {
                        "required": true
                    }
                }
            },
            "ListVirtualRouters": {
                "url": "https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_ListVirtualRouters.html",
                "description": "Returns a list of existing virtual routers in a service mesh.",
                "accessLevel": "List",
                "resourceTypes": {
                    "mesh": {
                        "required": true
                    }
                }
            },
            "ListVirtualServices": {
                "url": "https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_ListVirtualServices.html",
                "description": "Returns a list of existing virtual services in a service mesh.",
                "accessLevel": "List",
                "resourceTypes": {
                    "mesh": {
                        "required": true
                    }
                }
            },
            "StreamAggregatedResources": {
                "url": "https://docs.aws.amazon.com/app-mesh/latest/userguide/envoy.html",
                "description": "Allows an Envoy Proxy to receive streamed resources for an App Mesh endpoint (VirtualNode/VirtualGateway).",
                "accessLevel": "Read",
                "resourceTypes": {
                    "virtualGateway": {
                        "required": false
                    },
                    "virtualNode": {
                        "required": false
                    }
                }
            },
            "UpdateGatewayRoute": {
                "url": "https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_UpdateGatewayRoute.html",
                "description": "Updates an existing gateway route for a specified service mesh and virtual gateway.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "gatewayRoute": {
                        "required": true
                    },
                    "virtualService": {
                        "required": false
                    }
                }
            },
            "UpdateMesh": {
                "url": "https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_UpdateMesh.html",
                "description": "Updates an existing service mesh.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "mesh": {
                        "required": true
                    }
                }
            },
            "UpdateRoute": {
                "url": "https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_UpdateRoute.html",
                "description": "Updates an existing route for a specified service mesh and virtual router.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "route": {
                        "required": true
                    },
                    "virtualNode": {
                        "required": false
                    }
                }
            },
            "UpdateVirtualGateway": {
                "url": "https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_UpdateVirtualGateway.html",
                "description": "Updates an existing virtual gateway in a specified service mesh.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "virtualGateway": {
                        "required": true
                    }
                }
            },
            "UpdateVirtualNode": {
                "url": "https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_UpdateVirtualNode.html",
                "description": "Updates an existing virtual node in a specified service mesh.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "virtualNode": {
                        "required": true
                    }
                }
            },
            "UpdateVirtualRouter": {
                "url": "https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_UpdateVirtualRouter.html",
                "description": "Updates an existing virtual router in a specified service mesh.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "virtualRouter": {
                        "required": true
                    }
                }
            },
            "UpdateVirtualService": {
                "url": "https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_UpdateVirtualService.html",
                "description": "Updates an existing virtual service in a specified service mesh.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "mesh": {
                        "required": true
                    },
                    "virtualNode": {
                        "required": false
                    },
                    "virtualRouter": {
                        "required": false
                    }
                }
            }
        };
        this.resourceTypes = {
            "mesh": {
                "name": "mesh",
                "url": "https://docs.aws.amazon.com/app-mesh/latest/userguide/meshes.html",
                "arn": "arn:${Partition}:appmesh-preview:${Region}:${Account}:mesh/${MeshName}",
                "conditionKeys": []
            },
            "virtualService": {
                "name": "virtualService",
                "url": "https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_services.html",
                "arn": "arn:${Partition}:appmesh-preview:${Region}:${Account}:mesh/${MeshName}/virtualService/${VirtualServiceName}",
                "conditionKeys": []
            },
            "virtualNode": {
                "name": "virtualNode",
                "url": "https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_nodes.html",
                "arn": "arn:${Partition}:appmesh-preview:${Region}:${Account}:mesh/${MeshName}/virtualNode/${VirtualNodeName}",
                "conditionKeys": []
            },
            "virtualRouter": {
                "name": "virtualRouter",
                "url": "https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_routers.html",
                "arn": "arn:${Partition}:appmesh-preview:${Region}:${Account}:mesh/${MeshName}/virtualRouter/${VirtualRouterName}",
                "conditionKeys": []
            },
            "route": {
                "name": "route",
                "url": "https://docs.aws.amazon.com/app-mesh/latest/userguide/routes.html",
                "arn": "arn:${Partition}:appmesh-preview:${Region}:${Account}:mesh/${MeshName}/virtualRouter/${VirtualRouterName}/route/${RouteName}",
                "conditionKeys": []
            },
            "virtualGateway": {
                "name": "virtualGateway",
                "url": "https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_gateways.html",
                "arn": "arn:${Partition}:appmesh-preview:${Region}:${Account}:mesh/${MeshName}/virtualGateway/${VirtualGatewayName}",
                "conditionKeys": []
            },
            "gatewayRoute": {
                "name": "gatewayRoute",
                "url": "https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_gateways.html",
                "arn": "arn:${Partition}:appmesh-preview:${Region}:${Account}:mesh/${MeshName}/virtualGateway/${VirtualGatewayName}/gatewayRoute/${GatewayRouteName}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Creates a gateway route that is associated with a virtual gateway.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_CreateGatewayRoute.html
     */
    toCreateGatewayRoute() {
        this.add('appmesh-preview:CreateGatewayRoute');
        return this;
    }
    /**
     * Creates a service mesh.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_CreateMesh.html
     */
    toCreateMesh() {
        this.add('appmesh-preview:CreateMesh');
        return this;
    }
    /**
     * Creates a route that is associated with a virtual router.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_CreateRoute.html
     */
    toCreateRoute() {
        this.add('appmesh-preview:CreateRoute');
        return this;
    }
    /**
     * Creates a virtual gateway within a service mesh.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_CreateVirtualGateway.html
     */
    toCreateVirtualGateway() {
        this.add('appmesh-preview:CreateVirtualGateway');
        return this;
    }
    /**
     * Creates a virtual node within a service mesh.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_CreateVirtualNode.html
     */
    toCreateVirtualNode() {
        this.add('appmesh-preview:CreateVirtualNode');
        return this;
    }
    /**
     * Creates a virtual router within a service mesh.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_CreateVirtualRouter.html
     */
    toCreateVirtualRouter() {
        this.add('appmesh-preview:CreateVirtualRouter');
        return this;
    }
    /**
     * Creates a virtual service within a service mesh.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_CreateVirtualService.html
     */
    toCreateVirtualService() {
        this.add('appmesh-preview:CreateVirtualService');
        return this;
    }
    /**
     * Deletes an existing gateway route.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DeleteGatewayRoute.html
     */
    toDeleteGatewayRoute() {
        this.add('appmesh-preview:DeleteGatewayRoute');
        return this;
    }
    /**
     * Deletes an existing service mesh.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DeleteMesh.html
     */
    toDeleteMesh() {
        this.add('appmesh-preview:DeleteMesh');
        return this;
    }
    /**
     * Deletes an existing route.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DeleteRoute.html
     */
    toDeleteRoute() {
        this.add('appmesh-preview:DeleteRoute');
        return this;
    }
    /**
     * Deletes an existing virtual gateway.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DeleteVirtualGateway.html
     */
    toDeleteVirtualGateway() {
        this.add('appmesh-preview:DeleteVirtualGateway');
        return this;
    }
    /**
     * Deletes an existing virtual node.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DeleteVirtualNode.html
     */
    toDeleteVirtualNode() {
        this.add('appmesh-preview:DeleteVirtualNode');
        return this;
    }
    /**
     * Deletes an existing virtual router.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DeleteVirtualRouter.html
     */
    toDeleteVirtualRouter() {
        this.add('appmesh-preview:DeleteVirtualRouter');
        return this;
    }
    /**
     * Deletes an existing virtual service.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DeleteVirtualService.html
     */
    toDeleteVirtualService() {
        this.add('appmesh-preview:DeleteVirtualService');
        return this;
    }
    /**
     * Describes an existing gateway route.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DescribeGatewayRoute.html
     */
    toDescribeGatewayRoute() {
        this.add('appmesh-preview:DescribeGatewayRoute');
        return this;
    }
    /**
     * Describes an existing service mesh.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DescribeMesh.html
     */
    toDescribeMesh() {
        this.add('appmesh-preview:DescribeMesh');
        return this;
    }
    /**
     * Describes an existing route.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DescribeRoute.html
     */
    toDescribeRoute() {
        this.add('appmesh-preview:DescribeRoute');
        return this;
    }
    /**
     * Describes an existing virtual gateway.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DescribeVirtualGateway.html
     */
    toDescribeVirtualGateway() {
        this.add('appmesh-preview:DescribeVirtualGateway');
        return this;
    }
    /**
     * Describes an existing virtual node.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DescribeVirtualNode.html
     */
    toDescribeVirtualNode() {
        this.add('appmesh-preview:DescribeVirtualNode');
        return this;
    }
    /**
     * Describes an existing virtual router.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DescribeVirtualRouter.html
     */
    toDescribeVirtualRouter() {
        this.add('appmesh-preview:DescribeVirtualRouter');
        return this;
    }
    /**
     * Describes an existing virtual service.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_DescribeVirtualService.html
     */
    toDescribeVirtualService() {
        this.add('appmesh-preview:DescribeVirtualService');
        return this;
    }
    /**
     * Returns a list of existing gateway routes in a service mesh.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_ListGatewayRoutes.html
     */
    toListGatewayRoutes() {
        this.add('appmesh-preview:ListGatewayRoutes');
        return this;
    }
    /**
     * Returns a list of existing service meshes.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_ListMeshes.html
     */
    toListMeshes() {
        this.add('appmesh-preview:ListMeshes');
        return this;
    }
    /**
     * Returns a list of existing routes in a service mesh.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_ListRoutes.html
     */
    toListRoutes() {
        this.add('appmesh-preview:ListRoutes');
        return this;
    }
    /**
     * Returns a list of existing virtual gateways in a service mesh.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_ListVirtualGateways.html
     */
    toListVirtualGateways() {
        this.add('appmesh-preview:ListVirtualGateways');
        return this;
    }
    /**
     * Returns a list of existing virtual nodes.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_ListVirtualNodes.html
     */
    toListVirtualNodes() {
        this.add('appmesh-preview:ListVirtualNodes');
        return this;
    }
    /**
     * Returns a list of existing virtual routers in a service mesh.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_ListVirtualRouters.html
     */
    toListVirtualRouters() {
        this.add('appmesh-preview:ListVirtualRouters');
        return this;
    }
    /**
     * Returns a list of existing virtual services in a service mesh.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_ListVirtualServices.html
     */
    toListVirtualServices() {
        this.add('appmesh-preview:ListVirtualServices');
        return this;
    }
    /**
     * Allows an Envoy Proxy to receive streamed resources for an App Mesh endpoint (VirtualNode/VirtualGateway).
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/app-mesh/latest/userguide/envoy.html
     */
    toStreamAggregatedResources() {
        this.add('appmesh-preview:StreamAggregatedResources');
        return this;
    }
    /**
     * Updates an existing gateway route for a specified service mesh and virtual gateway.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_UpdateGatewayRoute.html
     */
    toUpdateGatewayRoute() {
        this.add('appmesh-preview:UpdateGatewayRoute');
        return this;
    }
    /**
     * Updates an existing service mesh.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_UpdateMesh.html
     */
    toUpdateMesh() {
        this.add('appmesh-preview:UpdateMesh');
        return this;
    }
    /**
     * Updates an existing route for a specified service mesh and virtual router.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_UpdateRoute.html
     */
    toUpdateRoute() {
        this.add('appmesh-preview:UpdateRoute');
        return this;
    }
    /**
     * Updates an existing virtual gateway in a specified service mesh.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_UpdateVirtualGateway.html
     */
    toUpdateVirtualGateway() {
        this.add('appmesh-preview:UpdateVirtualGateway');
        return this;
    }
    /**
     * Updates an existing virtual node in a specified service mesh.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_UpdateVirtualNode.html
     */
    toUpdateVirtualNode() {
        this.add('appmesh-preview:UpdateVirtualNode');
        return this;
    }
    /**
     * Updates an existing virtual router in a specified service mesh.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_UpdateVirtualRouter.html
     */
    toUpdateVirtualRouter() {
        this.add('appmesh-preview:UpdateVirtualRouter');
        return this;
    }
    /**
     * Updates an existing virtual service in a specified service mesh.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_UpdateVirtualService.html
     */
    toUpdateVirtualService() {
        this.add('appmesh-preview:UpdateVirtualService');
        return this;
    }
    /**
     * Adds a resource of type mesh to the statement
     *
     * https://docs.aws.amazon.com/app-mesh/latest/userguide/meshes.html
     *
     * @param meshName - Identifier for the meshName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onMesh(meshName, account, region, partition) {
        var arn = 'arn:${Partition}:appmesh-preview:${Region}:${Account}:mesh/${MeshName}';
        arn = arn.replace('${MeshName}', meshName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type virtualService to the statement
     *
     * https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_services.html
     *
     * @param meshName - Identifier for the meshName.
     * @param virtualServiceName - Identifier for the virtualServiceName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onVirtualService(meshName, virtualServiceName, account, region, partition) {
        var arn = 'arn:${Partition}:appmesh-preview:${Region}:${Account}:mesh/${MeshName}/virtualService/${VirtualServiceName}';
        arn = arn.replace('${MeshName}', meshName);
        arn = arn.replace('${VirtualServiceName}', virtualServiceName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type virtualNode to the statement
     *
     * https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_nodes.html
     *
     * @param meshName - Identifier for the meshName.
     * @param virtualNodeName - Identifier for the virtualNodeName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onVirtualNode(meshName, virtualNodeName, account, region, partition) {
        var arn = 'arn:${Partition}:appmesh-preview:${Region}:${Account}:mesh/${MeshName}/virtualNode/${VirtualNodeName}';
        arn = arn.replace('${MeshName}', meshName);
        arn = arn.replace('${VirtualNodeName}', virtualNodeName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type virtualRouter to the statement
     *
     * https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_routers.html
     *
     * @param meshName - Identifier for the meshName.
     * @param virtualRouterName - Identifier for the virtualRouterName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onVirtualRouter(meshName, virtualRouterName, account, region, partition) {
        var arn = 'arn:${Partition}:appmesh-preview:${Region}:${Account}:mesh/${MeshName}/virtualRouter/${VirtualRouterName}';
        arn = arn.replace('${MeshName}', meshName);
        arn = arn.replace('${VirtualRouterName}', virtualRouterName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type route to the statement
     *
     * https://docs.aws.amazon.com/app-mesh/latest/userguide/routes.html
     *
     * @param meshName - Identifier for the meshName.
     * @param virtualRouterName - Identifier for the virtualRouterName.
     * @param routeName - Identifier for the routeName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onRoute(meshName, virtualRouterName, routeName, account, region, partition) {
        var arn = 'arn:${Partition}:appmesh-preview:${Region}:${Account}:mesh/${MeshName}/virtualRouter/${VirtualRouterName}/route/${RouteName}';
        arn = arn.replace('${MeshName}', meshName);
        arn = arn.replace('${VirtualRouterName}', virtualRouterName);
        arn = arn.replace('${RouteName}', routeName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type virtualGateway to the statement
     *
     * https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_gateways.html
     *
     * @param meshName - Identifier for the meshName.
     * @param virtualGatewayName - Identifier for the virtualGatewayName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onVirtualGateway(meshName, virtualGatewayName, account, region, partition) {
        var arn = 'arn:${Partition}:appmesh-preview:${Region}:${Account}:mesh/${MeshName}/virtualGateway/${VirtualGatewayName}';
        arn = arn.replace('${MeshName}', meshName);
        arn = arn.replace('${VirtualGatewayName}', virtualGatewayName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type gatewayRoute to the statement
     *
     * https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_gateways.html
     *
     * @param meshName - Identifier for the meshName.
     * @param virtualGatewayName - Identifier for the virtualGatewayName.
     * @param gatewayRouteName - Identifier for the gatewayRouteName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onGatewayRoute(meshName, virtualGatewayName, gatewayRouteName, account, region, partition) {
        var arn = 'arn:${Partition}:appmesh-preview:${Region}:${Account}:mesh/${MeshName}/virtualGateway/${VirtualGatewayName}/gatewayRoute/${GatewayRouteName}';
        arn = arn.replace('${MeshName}', meshName);
        arn = arn.replace('${VirtualGatewayName}', virtualGatewayName);
        arn = arn.replace('${GatewayRouteName}', gatewayRouteName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.AppmeshPreview = AppmeshPreview;
//# sourceMappingURL=data:application/json;base64,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