"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Appsync = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [appsync](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsappsync.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Appsync extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [appsync](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsappsync.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'appsync';
        this.actionList = {
            "CreateApiKey": {
                "url": "https://docs.aws.amazon.com/appsync/latest/APIReference/API_CreateApiKey.html",
                "description": "Creates a unique key that you can distribute to clients who are executing your API.",
                "accessLevel": "Write"
            },
            "CreateDataSource": {
                "url": "https://docs.aws.amazon.com/appsync/latest/APIReference/API_CreateDataSource.html",
                "description": "Creates a DataSource object.",
                "accessLevel": "Write"
            },
            "CreateFunction": {
                "url": "https://docs.aws.amazon.com/appsync/latest/APIReference/API_CreateFunction.html",
                "description": "Create a new Function object.",
                "accessLevel": "Write"
            },
            "CreateGraphqlApi": {
                "url": "https://docs.aws.amazon.com/appsync/latest/APIReference/API_CreateGraphqlApi.html",
                "description": "Creates a GraphqlApi object, which is the top level AppSync resource.",
                "accessLevel": "Tagging",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateResolver": {
                "url": "https://docs.aws.amazon.com/appsync/latest/APIReference/API_CreateResolver.html",
                "description": "Creates a Resolver object. A resolver converts incoming requests into a format that a data source can understand, and converts the data source's responses into GraphQL.",
                "accessLevel": "Write"
            },
            "CreateType": {
                "url": "https://docs.aws.amazon.com/appsync/latest/APIReference/API_CreateType.html",
                "description": "Creates a Type object.",
                "accessLevel": "Write"
            },
            "DeleteApiKey": {
                "url": "https://docs.aws.amazon.com/appsync/latest/APIReference/API_DeleteApiKey.html",
                "description": "Deletes an API key.",
                "accessLevel": "Write"
            },
            "DeleteDataSource": {
                "url": "https://docs.aws.amazon.com/appsync/latest/APIReference/API_DeleteDataSource.html",
                "description": "Deletes a DataSource object.",
                "accessLevel": "Write"
            },
            "DeleteFunction": {
                "url": "https://docs.aws.amazon.com/appsync/latest/APIReference/API_DeleteFunction.html",
                "description": "Deletes a Function object.",
                "accessLevel": "Write"
            },
            "DeleteGraphqlApi": {
                "url": "https://docs.aws.amazon.com/appsync/latest/APIReference/API_DeleteGraphqlApi.html",
                "description": "Deletes a GraphqlApi object. This will also clean up every AppSync resource below that API.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "graphqlapi": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "DeleteResolver": {
                "url": "https://docs.aws.amazon.com/appsync/latest/APIReference/API_DeleteResolver.html",
                "description": "Deletes a Resolver object.",
                "accessLevel": "Write"
            },
            "DeleteType": {
                "url": "https://docs.aws.amazon.com/appsync/latest/APIReference/API_DeleteType.html",
                "description": "Deletes a Type object.",
                "accessLevel": "Write"
            },
            "GetDataSource": {
                "url": "https://docs.aws.amazon.com/appsync/latest/APIReference/API_GetDataSource.html",
                "description": "Retrieves a DataSource object.",
                "accessLevel": "Read"
            },
            "GetFunction": {
                "url": "https://docs.aws.amazon.com/appsync/latest/APIReference/API_GetFunction.html",
                "description": "Retrieves a Function object.",
                "accessLevel": "Read"
            },
            "GetGraphqlApi": {
                "url": "https://docs.aws.amazon.com/appsync/latest/APIReference/API_GetGraphqlApi.html",
                "description": "Retrieves a GraphqlApi object.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "graphqlapi": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "GetIntrospectionSchema": {
                "url": "https://docs.aws.amazon.com/appsync/latest/APIReference/API_GetIntrospectionSchema.html",
                "description": "Retrieves the introspection schema for a GraphQL API.",
                "accessLevel": "Read"
            },
            "GetResolver": {
                "url": "https://docs.aws.amazon.com/appsync/latest/APIReference/API_GetResolver.html",
                "description": "Retrieves a Resolver object.",
                "accessLevel": "Read"
            },
            "GetSchemaCreationStatus": {
                "url": "https://docs.aws.amazon.com/appsync/latest/APIReference/API_GetSchemaCreationStatus.html",
                "description": "Retrieves the current status of a schema creation operation.",
                "accessLevel": "Read"
            },
            "GetType": {
                "url": "https://docs.aws.amazon.com/appsync/latest/APIReference/API_GetType.html",
                "description": "Retrieves a Type object.",
                "accessLevel": "Read"
            },
            "GraphQL": {
                "url": "https://docs.aws.amazon.com/appsync/latest/devguide/using-your-api.html",
                "description": "Sends a GraphQL query to a GraphQL API.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "field": {
                        "required": true
                    },
                    "graphqlapi": {
                        "required": true
                    }
                }
            },
            "ListApiKeys": {
                "url": "https://docs.aws.amazon.com/appsync/latest/APIReference/API_ListApiKeys.html",
                "description": "Lists the API keys for a given API.",
                "accessLevel": "List"
            },
            "ListDataSources": {
                "url": "https://docs.aws.amazon.com/appsync/latest/APIReference/API_ListDataSources.html",
                "description": "Lists the data sources for a given API.",
                "accessLevel": "List"
            },
            "ListFunctions": {
                "url": "https://docs.aws.amazon.com/appsync/latest/APIReference/API_ListFunctions.html",
                "description": "Lists the functions for a given API.",
                "accessLevel": "List"
            },
            "ListGraphqlApis": {
                "url": "https://docs.aws.amazon.com/appsync/latest/APIReference/API_ListGraphqlApis.html",
                "description": "Lists your GraphQL APIs.",
                "accessLevel": "List"
            },
            "ListResolvers": {
                "url": "https://docs.aws.amazon.com/appsync/latest/APIReference/API_ListResolvers.html",
                "description": "Lists the resolvers for a given API and type.",
                "accessLevel": "List"
            },
            "ListResolversByFunction": {
                "url": "https://docs.aws.amazon.com/appsync/latest/APIReference/API_ListResolversByFunction.html",
                "description": "List the resolvers that are associated with a specific function.",
                "accessLevel": "List"
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/appsync/latest/APIReference/API_ListTagsForResource.html",
                "description": "List the tags for a resource.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "graphqlapi": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "ListTypes": {
                "url": "https://docs.aws.amazon.com/appsync/latest/APIReference/API_ListTypes.html",
                "description": "Lists the types for a given API.",
                "accessLevel": "List"
            },
            "StartSchemaCreation": {
                "url": "https://docs.aws.amazon.com/appsync/latest/APIReference/API_StartSchemaCreation.html",
                "description": "Adds a new schema to your GraphQL API. This operation is asynchronous - GetSchemaCreationStatus can show when it has completed.",
                "accessLevel": "Write"
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/appsync/latest/APIReference/API_TagResource.html",
                "description": "Tag a resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "graphqlapi": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/appsync/latest/APIReference/API_UntagResource.html",
                "description": "Untag a resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "graphqlapi": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "UpdateApiKey": {
                "url": "https://docs.aws.amazon.com/appsync/latest/APIReference/API_UpdateApiKey.html",
                "description": "Updates an API key for a given API.",
                "accessLevel": "Write"
            },
            "UpdateDataSource": {
                "url": "https://docs.aws.amazon.com/appsync/latest/APIReference/API_UpdateDataSource.html",
                "description": "Updates a DataSource object.",
                "accessLevel": "Write"
            },
            "UpdateFunction": {
                "url": "https://docs.aws.amazon.com/appsync/latest/APIReference/API_UpdateFunction.html",
                "description": "Updates an existing Function object.",
                "accessLevel": "Write"
            },
            "UpdateGraphqlApi": {
                "url": "https://docs.aws.amazon.com/appsync/latest/APIReference/API_UpdateGraphqlApi.html",
                "description": "Updates a GraphqlApi object.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "graphqlapi": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "UpdateResolver": {
                "url": "https://docs.aws.amazon.com/appsync/latest/APIReference/API_UpdateResolver.html",
                "description": "Updates a Resolver object.",
                "accessLevel": "Write"
            },
            "UpdateType": {
                "url": "https://docs.aws.amazon.com/appsync/latest/APIReference/API_UpdateType.html",
                "description": "Updates a Type object.",
                "accessLevel": "Write"
            }
        };
        this.resourceTypes = {
            "datasource": {
                "name": "datasource",
                "url": "https://docs.aws.amazon.com/appsync/latest/devguide/attaching-a-data-source.html",
                "arn": "arn:${Partition}:appsync:${Region}:${Account}:apis/${GraphQLAPIId}/datasources/${DatasourceName}",
                "conditionKeys": []
            },
            "graphqlapi": {
                "name": "graphqlapi",
                "url": "https://docs.aws.amazon.com/appsync/latest/devguide/designing-a-graphql-api.html",
                "arn": "arn:${Partition}:appsync:${Region}:${Account}:apis/${GraphQLAPIId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "field": {
                "name": "field",
                "url": "https://docs.aws.amazon.com/appsync/latest/devguide/configuring-resolvers.html",
                "arn": "arn:${Partition}:appsync:${Region}:${Account}:apis/${GraphQLAPIId}/types/${TypeName}/fields/${FieldName}",
                "conditionKeys": []
            },
            "type": {
                "name": "type",
                "url": "https://docs.aws.amazon.com/appsync/latest/devguide/designing-your-schema.html#adding-a-root-query-type",
                "arn": "arn:${Partition}:appsync:${Region}:${Account}:apis/${GraphQLAPIId}/types/${TypeName}",
                "conditionKeys": []
            },
            "function": {
                "name": "function",
                "url": "https://docs.aws.amazon.com/appsync/latest/devguide/pipeline-resolvers.html",
                "arn": "arn:${Partition}:appsync:${Region}:${Account}:apis/${GraphQLAPIId}/functions/${FunctionId}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Creates a unique key that you can distribute to clients who are executing your API.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_CreateApiKey.html
     */
    toCreateApiKey() {
        this.add('appsync:CreateApiKey');
        return this;
    }
    /**
     * Creates a DataSource object.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_CreateDataSource.html
     */
    toCreateDataSource() {
        this.add('appsync:CreateDataSource');
        return this;
    }
    /**
     * Create a new Function object.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_CreateFunction.html
     */
    toCreateFunction() {
        this.add('appsync:CreateFunction');
        return this;
    }
    /**
     * Creates a GraphqlApi object, which is the top level AppSync resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_CreateGraphqlApi.html
     */
    toCreateGraphqlApi() {
        this.add('appsync:CreateGraphqlApi');
        return this;
    }
    /**
     * Creates a Resolver object. A resolver converts incoming requests into a format that a data source can understand, and converts the data source's responses into GraphQL.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_CreateResolver.html
     */
    toCreateResolver() {
        this.add('appsync:CreateResolver');
        return this;
    }
    /**
     * Creates a Type object.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_CreateType.html
     */
    toCreateType() {
        this.add('appsync:CreateType');
        return this;
    }
    /**
     * Deletes an API key.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_DeleteApiKey.html
     */
    toDeleteApiKey() {
        this.add('appsync:DeleteApiKey');
        return this;
    }
    /**
     * Deletes a DataSource object.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_DeleteDataSource.html
     */
    toDeleteDataSource() {
        this.add('appsync:DeleteDataSource');
        return this;
    }
    /**
     * Deletes a Function object.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_DeleteFunction.html
     */
    toDeleteFunction() {
        this.add('appsync:DeleteFunction');
        return this;
    }
    /**
     * Deletes a GraphqlApi object. This will also clean up every AppSync resource below that API.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_DeleteGraphqlApi.html
     */
    toDeleteGraphqlApi() {
        this.add('appsync:DeleteGraphqlApi');
        return this;
    }
    /**
     * Deletes a Resolver object.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_DeleteResolver.html
     */
    toDeleteResolver() {
        this.add('appsync:DeleteResolver');
        return this;
    }
    /**
     * Deletes a Type object.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_DeleteType.html
     */
    toDeleteType() {
        this.add('appsync:DeleteType');
        return this;
    }
    /**
     * Retrieves a DataSource object.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_GetDataSource.html
     */
    toGetDataSource() {
        this.add('appsync:GetDataSource');
        return this;
    }
    /**
     * Retrieves a Function object.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_GetFunction.html
     */
    toGetFunction() {
        this.add('appsync:GetFunction');
        return this;
    }
    /**
     * Retrieves a GraphqlApi object.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_GetGraphqlApi.html
     */
    toGetGraphqlApi() {
        this.add('appsync:GetGraphqlApi');
        return this;
    }
    /**
     * Retrieves the introspection schema for a GraphQL API.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_GetIntrospectionSchema.html
     */
    toGetIntrospectionSchema() {
        this.add('appsync:GetIntrospectionSchema');
        return this;
    }
    /**
     * Retrieves a Resolver object.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_GetResolver.html
     */
    toGetResolver() {
        this.add('appsync:GetResolver');
        return this;
    }
    /**
     * Retrieves the current status of a schema creation operation.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_GetSchemaCreationStatus.html
     */
    toGetSchemaCreationStatus() {
        this.add('appsync:GetSchemaCreationStatus');
        return this;
    }
    /**
     * Retrieves a Type object.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_GetType.html
     */
    toGetType() {
        this.add('appsync:GetType');
        return this;
    }
    /**
     * Sends a GraphQL query to a GraphQL API.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/devguide/using-your-api.html
     */
    toGraphQL() {
        this.add('appsync:GraphQL');
        return this;
    }
    /**
     * Lists the API keys for a given API.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_ListApiKeys.html
     */
    toListApiKeys() {
        this.add('appsync:ListApiKeys');
        return this;
    }
    /**
     * Lists the data sources for a given API.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_ListDataSources.html
     */
    toListDataSources() {
        this.add('appsync:ListDataSources');
        return this;
    }
    /**
     * Lists the functions for a given API.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_ListFunctions.html
     */
    toListFunctions() {
        this.add('appsync:ListFunctions');
        return this;
    }
    /**
     * Lists your GraphQL APIs.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_ListGraphqlApis.html
     */
    toListGraphqlApis() {
        this.add('appsync:ListGraphqlApis');
        return this;
    }
    /**
     * Lists the resolvers for a given API and type.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_ListResolvers.html
     */
    toListResolvers() {
        this.add('appsync:ListResolvers');
        return this;
    }
    /**
     * List the resolvers that are associated with a specific function.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_ListResolversByFunction.html
     */
    toListResolversByFunction() {
        this.add('appsync:ListResolversByFunction');
        return this;
    }
    /**
     * List the tags for a resource.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.add('appsync:ListTagsForResource');
        return this;
    }
    /**
     * Lists the types for a given API.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_ListTypes.html
     */
    toListTypes() {
        this.add('appsync:ListTypes');
        return this;
    }
    /**
     * Adds a new schema to your GraphQL API. This operation is asynchronous - GetSchemaCreationStatus can show when it has completed.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_StartSchemaCreation.html
     */
    toStartSchemaCreation() {
        this.add('appsync:StartSchemaCreation');
        return this;
    }
    /**
     * Tag a resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        this.add('appsync:TagResource');
        return this;
    }
    /**
     * Untag a resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        this.add('appsync:UntagResource');
        return this;
    }
    /**
     * Updates an API key for a given API.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_UpdateApiKey.html
     */
    toUpdateApiKey() {
        this.add('appsync:UpdateApiKey');
        return this;
    }
    /**
     * Updates a DataSource object.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_UpdateDataSource.html
     */
    toUpdateDataSource() {
        this.add('appsync:UpdateDataSource');
        return this;
    }
    /**
     * Updates an existing Function object.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_UpdateFunction.html
     */
    toUpdateFunction() {
        this.add('appsync:UpdateFunction');
        return this;
    }
    /**
     * Updates a GraphqlApi object.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_UpdateGraphqlApi.html
     */
    toUpdateGraphqlApi() {
        this.add('appsync:UpdateGraphqlApi');
        return this;
    }
    /**
     * Updates a Resolver object.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_UpdateResolver.html
     */
    toUpdateResolver() {
        this.add('appsync:UpdateResolver');
        return this;
    }
    /**
     * Updates a Type object.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_UpdateType.html
     */
    toUpdateType() {
        this.add('appsync:UpdateType');
        return this;
    }
    /**
     * Adds a resource of type datasource to the statement
     *
     * https://docs.aws.amazon.com/appsync/latest/devguide/attaching-a-data-source.html
     *
     * @param graphQLAPIId - Identifier for the graphQLAPIId.
     * @param datasourceName - Identifier for the datasourceName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onDatasource(graphQLAPIId, datasourceName, account, region, partition) {
        var arn = 'arn:${Partition}:appsync:${Region}:${Account}:apis/${GraphQLAPIId}/datasources/${DatasourceName}';
        arn = arn.replace('${GraphQLAPIId}', graphQLAPIId);
        arn = arn.replace('${DatasourceName}', datasourceName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type graphqlapi to the statement
     *
     * https://docs.aws.amazon.com/appsync/latest/devguide/designing-a-graphql-api.html
     *
     * @param graphQLAPIId - Identifier for the graphQLAPIId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onGraphqlapi(graphQLAPIId, account, region, partition) {
        var arn = 'arn:${Partition}:appsync:${Region}:${Account}:apis/${GraphQLAPIId}';
        arn = arn.replace('${GraphQLAPIId}', graphQLAPIId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type field to the statement
     *
     * https://docs.aws.amazon.com/appsync/latest/devguide/configuring-resolvers.html
     *
     * @param graphQLAPIId - Identifier for the graphQLAPIId.
     * @param typeName - Identifier for the typeName.
     * @param fieldName - Identifier for the fieldName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onField(graphQLAPIId, typeName, fieldName, account, region, partition) {
        var arn = 'arn:${Partition}:appsync:${Region}:${Account}:apis/${GraphQLAPIId}/types/${TypeName}/fields/${FieldName}';
        arn = arn.replace('${GraphQLAPIId}', graphQLAPIId);
        arn = arn.replace('${TypeName}', typeName);
        arn = arn.replace('${FieldName}', fieldName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type type to the statement
     *
     * https://docs.aws.amazon.com/appsync/latest/devguide/designing-your-schema.html#adding-a-root-query-type
     *
     * @param graphQLAPIId - Identifier for the graphQLAPIId.
     * @param typeName - Identifier for the typeName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onType(graphQLAPIId, typeName, account, region, partition) {
        var arn = 'arn:${Partition}:appsync:${Region}:${Account}:apis/${GraphQLAPIId}/types/${TypeName}';
        arn = arn.replace('${GraphQLAPIId}', graphQLAPIId);
        arn = arn.replace('${TypeName}', typeName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type function to the statement
     *
     * https://docs.aws.amazon.com/appsync/latest/devguide/pipeline-resolvers.html
     *
     * @param graphQLAPIId - Identifier for the graphQLAPIId.
     * @param functionId - Identifier for the functionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onFunction(graphQLAPIId, functionId, account, region, partition) {
        var arn = 'arn:${Partition}:appsync:${Region}:${Account}:apis/${GraphQLAPIId}/functions/${FunctionId}';
        arn = arn.replace('${GraphQLAPIId}', graphQLAPIId);
        arn = arn.replace('${FunctionId}', functionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Appsync = Appsync;
//# sourceMappingURL=data:application/json;base64,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