"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Backup = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [backup](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsbackup.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Backup extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [backup](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsbackup.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'backup';
        this.actionList = {
            "CopyIntoBackupVault": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_CopyIntoBackupVault.html",
                "description": "Copy into a backup vault",
                "accessLevel": "Write"
            },
            "CreateBackupPlan": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_CreateBackupPlan.html",
                "description": "Creates a new backup plan",
                "accessLevel": "Write",
                "resourceTypes": {
                    "backupPlan": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateBackupSelection": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_CreateBackupSelection.html",
                "description": "Creates a new resource assignment in a backup plan.",
                "accessLevel": "Write",
                "dependentActions": [
                    "iam:PassRole"
                ],
                "resourceTypes": {
                    "backupPlan": {
                        "required": true
                    }
                }
            },
            "CreateBackupVault": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_CreateBackupVault.html",
                "description": "Creates a new backup vault.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "backupVault": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "DeleteBackupPlan": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DeleteBackupPlan.html",
                "description": "Deletes a backup plan.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "backupPlan": {
                        "required": true
                    }
                }
            },
            "DeleteBackupSelection": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DeleteBackupSelection.html",
                "description": "Deletes a resource assignment from a backup plan.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "backupPlan": {
                        "required": true
                    }
                }
            },
            "DeleteBackupVault": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DeleteBackupVault.html",
                "description": "Deletes a backup vault.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "backupVault": {
                        "required": true
                    }
                }
            },
            "DeleteBackupVaultAccessPolicy": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DeleteBackupVaultAccessPolicy.html",
                "description": "Deletes backup vault access policy.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "backupVault": {
                        "required": true
                    }
                }
            },
            "DeleteBackupVaultNotifications": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DeleteBackupVaultNotifications.html",
                "description": "Remove notifications from backup vault.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "backupVault": {
                        "required": true
                    }
                }
            },
            "DeleteRecoveryPoint": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DeleteRecoveryPoint.html",
                "description": "Deletes a recovery point from a backup vault.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "recoveryPoint": {
                        "required": true
                    }
                }
            },
            "DescribeBackupJob": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DescribeBackupJob.html",
                "description": "Describes a backup job",
                "accessLevel": "Read"
            },
            "DescribeBackupVault": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DescribeBackupVault.html",
                "description": "Creates a new backup vault with the specified name.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "backupVault": {
                        "required": true
                    }
                }
            },
            "DescribeCopyJob": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DescribeCopyJob.html",
                "description": "Describes a copy job",
                "accessLevel": "Read",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "DescribeProtectedResource": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DescribeProtectedResource.html",
                "description": "Describes a protected resource.",
                "accessLevel": "Read"
            },
            "DescribeRecoveryPoint": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DescribeRecoveryPoint.html",
                "description": "Describes a recovery point.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "recoveryPoint": {
                        "required": true
                    }
                }
            },
            "DescribeRegionSettings": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DescribeRegionSettings.html",
                "description": "Describes region settings",
                "accessLevel": "Read"
            },
            "DescribeRestoreJob": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DescribeRestoreJob.html",
                "description": "Describes a restore job.",
                "accessLevel": "Read"
            },
            "ExportBackupPlanTemplate": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ExportBackupPlanTemplate.html",
                "description": "Exports a backup plan as a JSON.",
                "accessLevel": "Read"
            },
            "GetBackupPlan": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_GetBackupPlan.html",
                "description": "Gets a backup plan.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "backupPlan": {
                        "required": true
                    }
                }
            },
            "GetBackupPlanFromJSON": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_GetBackupPlanFromJSON.html",
                "description": "Transforms a JSON to a backup plan.",
                "accessLevel": "Read"
            },
            "GetBackupPlanFromTemplate": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_GetBackupPlanFromTemplate.html",
                "description": "Transforms a template to a backup plan.",
                "accessLevel": "Read"
            },
            "GetBackupSelection": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_GetBackupSelection.html",
                "description": "Gets a backup plan resource assignment.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "backupPlan": {
                        "required": true
                    }
                }
            },
            "GetBackupVaultAccessPolicy": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_GetBackupVaultAccessPolicy.html",
                "description": "Gets backup vault access policy.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "backupVault": {
                        "required": true
                    }
                }
            },
            "GetBackupVaultNotifications": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_GetBackupVaultNotifications.html",
                "description": "Gets backup vault notifications.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "backupVault": {
                        "required": true
                    }
                }
            },
            "GetRecoveryPointRestoreMetadata": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_GetRecoveryPointRestoreMetadata.html",
                "description": "Gets recovery point restore metadata.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "recoveryPoint": {
                        "required": true
                    }
                }
            },
            "GetSupportedResourceTypes": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_GetSupportedResourceTypes.html",
                "description": "Gets supported resource types.",
                "accessLevel": "Read"
            },
            "ListBackupJobs": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListBackupJobs.html",
                "description": "Lists backup jobs.",
                "accessLevel": "List"
            },
            "ListBackupPlanTemplates": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListBackupPlanTemplates.html",
                "description": "Lists backup plan templates provided by AWS Backup.",
                "accessLevel": "List"
            },
            "ListBackupPlanVersions": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListBackupPlanVersions.html",
                "description": "Lists backup plan versions.",
                "accessLevel": "List",
                "resourceTypes": {
                    "backupPlan": {
                        "required": true
                    }
                }
            },
            "ListBackupPlans": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListBackupPlans.html",
                "description": "Lists backup plans.",
                "accessLevel": "List"
            },
            "ListBackupSelections": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListBackupSelections.html",
                "description": "Lists resource assignments for a specific backup plan.",
                "accessLevel": "List",
                "resourceTypes": {
                    "backupPlan": {
                        "required": true
                    }
                }
            },
            "ListBackupVaults": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListBackupVaults.html",
                "description": "Lists backup vaults.",
                "accessLevel": "List"
            },
            "ListCopyJobs": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListCopyJobs.html",
                "description": "List copy jobs",
                "accessLevel": "List"
            },
            "ListProtectedResources": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListProtectedResources.html",
                "description": "Lists protected resources by AWS Backup.",
                "accessLevel": "List"
            },
            "ListRecoveryPointsByBackupVault": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListRecoveryPointsByBackupVault.html",
                "description": "Lists recovery points inside a backup vault.",
                "accessLevel": "List",
                "resourceTypes": {
                    "backupVault": {
                        "required": true
                    }
                }
            },
            "ListRecoveryPointsByResource": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListRecoveryPointsByResource.html",
                "description": "Lists recovery points for a resource.",
                "accessLevel": "List"
            },
            "ListRestoreJobs": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListRestoreJobs.html",
                "description": "Lists restore jobs.",
                "accessLevel": "List"
            },
            "ListTags": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListTags.html",
                "description": "Lists tags for a resource.",
                "accessLevel": "List",
                "resourceTypes": {
                    "backupPlan": {
                        "required": false
                    },
                    "backupVault": {
                        "required": false
                    },
                    "recoveryPoint": {
                        "required": false
                    }
                }
            },
            "PutBackupVaultAccessPolicy": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_PutBackupVaultAccessPolicy.html",
                "description": "Adds an access policy to the backup vault.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "backupVault": {
                        "required": true
                    }
                }
            },
            "PutBackupVaultNotifications": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_PutBackupVaultNotifications.html",
                "description": "Adds an SNS topic to the backup vault.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "backupVault": {
                        "required": true
                    }
                }
            },
            "StartBackupJob": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_StartBackupJob.html",
                "description": "Starts a new backup job.",
                "accessLevel": "Write",
                "dependentActions": [
                    "iam:PassRole"
                ],
                "resourceTypes": {
                    "backupVault": {
                        "required": true
                    }
                }
            },
            "StartCopyJob": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_StartCopyJob.html",
                "description": "Copy a backup from a source region to a destination region.",
                "accessLevel": "Write",
                "dependentActions": [
                    "iam:PassRole"
                ],
                "resourceTypes": {
                    "recoveryPoint": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "StartRestoreJob": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_StartRestoreJob.html",
                "description": "Starts a new restore job.",
                "accessLevel": "Write",
                "dependentActions": [
                    "iam:PassRole"
                ],
                "resourceTypes": {
                    "recoveryPoint": {
                        "required": true
                    }
                }
            },
            "StopBackupJob": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_StopBackupJob.html",
                "description": "Stops a backup job.",
                "accessLevel": "Write"
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_TagResource.html",
                "description": "Tags a resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "backupPlan": {
                        "required": false
                    },
                    "backupVault": {
                        "required": false
                    },
                    "recoveryPoint": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_UntagResource.html",
                "description": "Untags a resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "backupPlan": {
                        "required": false
                    },
                    "backupVault": {
                        "required": false
                    },
                    "recoveryPoint": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "UpdateBackupPlan": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_UpdateBackupPlan.html",
                "description": "Updates a backup plan.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "backupPlan": {
                        "required": true
                    }
                }
            },
            "UpdateRecoveryPointLifecycle": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_UpdateRecoveryPointLifecycle.html",
                "description": "Updates the lifecycle of the recovery point.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "recoveryPoint": {
                        "required": true
                    }
                }
            },
            "UpdateRegionSettings": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_UpdateRegionSettings.html",
                "description": "Describes region settings",
                "accessLevel": "Write"
            }
        };
        this.resourceTypes = {
            "backupVault": {
                "name": "backupVault",
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/vaults.html",
                "arn": "arn:${Partition}:backup:${Region}:${Account}:backup-vault:${BackupVaultName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "backupPlan": {
                "name": "backupPlan",
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/about-backup-plans.html",
                "arn": "arn:${Partition}:backup:${Region}:${Account}:backup-plan:${BackupPlanId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "recoveryPoint": {
                "name": "recoveryPoint",
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/recovery-points.html",
                "arn": "arn:${Partition}:backup:${Region}:${Account}:recovery-point:${RecoveryPointId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Copy into a backup vault
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_CopyIntoBackupVault.html
     */
    toCopyIntoBackupVault() {
        this.add('backup:CopyIntoBackupVault');
        return this;
    }
    /**
     * Creates a new backup plan
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_CreateBackupPlan.html
     */
    toCreateBackupPlan() {
        this.add('backup:CreateBackupPlan');
        return this;
    }
    /**
     * Creates a new resource assignment in a backup plan.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_CreateBackupSelection.html
     */
    toCreateBackupSelection() {
        this.add('backup:CreateBackupSelection');
        return this;
    }
    /**
     * Creates a new backup vault.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_CreateBackupVault.html
     */
    toCreateBackupVault() {
        this.add('backup:CreateBackupVault');
        return this;
    }
    /**
     * Deletes a backup plan.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DeleteBackupPlan.html
     */
    toDeleteBackupPlan() {
        this.add('backup:DeleteBackupPlan');
        return this;
    }
    /**
     * Deletes a resource assignment from a backup plan.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DeleteBackupSelection.html
     */
    toDeleteBackupSelection() {
        this.add('backup:DeleteBackupSelection');
        return this;
    }
    /**
     * Deletes a backup vault.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DeleteBackupVault.html
     */
    toDeleteBackupVault() {
        this.add('backup:DeleteBackupVault');
        return this;
    }
    /**
     * Deletes backup vault access policy.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DeleteBackupVaultAccessPolicy.html
     */
    toDeleteBackupVaultAccessPolicy() {
        this.add('backup:DeleteBackupVaultAccessPolicy');
        return this;
    }
    /**
     * Remove notifications from backup vault.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DeleteBackupVaultNotifications.html
     */
    toDeleteBackupVaultNotifications() {
        this.add('backup:DeleteBackupVaultNotifications');
        return this;
    }
    /**
     * Deletes a recovery point from a backup vault.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DeleteRecoveryPoint.html
     */
    toDeleteRecoveryPoint() {
        this.add('backup:DeleteRecoveryPoint');
        return this;
    }
    /**
     * Describes a backup job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DescribeBackupJob.html
     */
    toDescribeBackupJob() {
        this.add('backup:DescribeBackupJob');
        return this;
    }
    /**
     * Creates a new backup vault with the specified name.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DescribeBackupVault.html
     */
    toDescribeBackupVault() {
        this.add('backup:DescribeBackupVault');
        return this;
    }
    /**
     * Describes a copy job
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DescribeCopyJob.html
     */
    toDescribeCopyJob() {
        this.add('backup:DescribeCopyJob');
        return this;
    }
    /**
     * Describes a protected resource.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DescribeProtectedResource.html
     */
    toDescribeProtectedResource() {
        this.add('backup:DescribeProtectedResource');
        return this;
    }
    /**
     * Describes a recovery point.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DescribeRecoveryPoint.html
     */
    toDescribeRecoveryPoint() {
        this.add('backup:DescribeRecoveryPoint');
        return this;
    }
    /**
     * Describes region settings
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DescribeRegionSettings.html
     */
    toDescribeRegionSettings() {
        this.add('backup:DescribeRegionSettings');
        return this;
    }
    /**
     * Describes a restore job.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DescribeRestoreJob.html
     */
    toDescribeRestoreJob() {
        this.add('backup:DescribeRestoreJob');
        return this;
    }
    /**
     * Exports a backup plan as a JSON.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ExportBackupPlanTemplate.html
     */
    toExportBackupPlanTemplate() {
        this.add('backup:ExportBackupPlanTemplate');
        return this;
    }
    /**
     * Gets a backup plan.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_GetBackupPlan.html
     */
    toGetBackupPlan() {
        this.add('backup:GetBackupPlan');
        return this;
    }
    /**
     * Transforms a JSON to a backup plan.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_GetBackupPlanFromJSON.html
     */
    toGetBackupPlanFromJSON() {
        this.add('backup:GetBackupPlanFromJSON');
        return this;
    }
    /**
     * Transforms a template to a backup plan.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_GetBackupPlanFromTemplate.html
     */
    toGetBackupPlanFromTemplate() {
        this.add('backup:GetBackupPlanFromTemplate');
        return this;
    }
    /**
     * Gets a backup plan resource assignment.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_GetBackupSelection.html
     */
    toGetBackupSelection() {
        this.add('backup:GetBackupSelection');
        return this;
    }
    /**
     * Gets backup vault access policy.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_GetBackupVaultAccessPolicy.html
     */
    toGetBackupVaultAccessPolicy() {
        this.add('backup:GetBackupVaultAccessPolicy');
        return this;
    }
    /**
     * Gets backup vault notifications.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_GetBackupVaultNotifications.html
     */
    toGetBackupVaultNotifications() {
        this.add('backup:GetBackupVaultNotifications');
        return this;
    }
    /**
     * Gets recovery point restore metadata.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_GetRecoveryPointRestoreMetadata.html
     */
    toGetRecoveryPointRestoreMetadata() {
        this.add('backup:GetRecoveryPointRestoreMetadata');
        return this;
    }
    /**
     * Gets supported resource types.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_GetSupportedResourceTypes.html
     */
    toGetSupportedResourceTypes() {
        this.add('backup:GetSupportedResourceTypes');
        return this;
    }
    /**
     * Lists backup jobs.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListBackupJobs.html
     */
    toListBackupJobs() {
        this.add('backup:ListBackupJobs');
        return this;
    }
    /**
     * Lists backup plan templates provided by AWS Backup.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListBackupPlanTemplates.html
     */
    toListBackupPlanTemplates() {
        this.add('backup:ListBackupPlanTemplates');
        return this;
    }
    /**
     * Lists backup plan versions.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListBackupPlanVersions.html
     */
    toListBackupPlanVersions() {
        this.add('backup:ListBackupPlanVersions');
        return this;
    }
    /**
     * Lists backup plans.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListBackupPlans.html
     */
    toListBackupPlans() {
        this.add('backup:ListBackupPlans');
        return this;
    }
    /**
     * Lists resource assignments for a specific backup plan.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListBackupSelections.html
     */
    toListBackupSelections() {
        this.add('backup:ListBackupSelections');
        return this;
    }
    /**
     * Lists backup vaults.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListBackupVaults.html
     */
    toListBackupVaults() {
        this.add('backup:ListBackupVaults');
        return this;
    }
    /**
     * List copy jobs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListCopyJobs.html
     */
    toListCopyJobs() {
        this.add('backup:ListCopyJobs');
        return this;
    }
    /**
     * Lists protected resources by AWS Backup.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListProtectedResources.html
     */
    toListProtectedResources() {
        this.add('backup:ListProtectedResources');
        return this;
    }
    /**
     * Lists recovery points inside a backup vault.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListRecoveryPointsByBackupVault.html
     */
    toListRecoveryPointsByBackupVault() {
        this.add('backup:ListRecoveryPointsByBackupVault');
        return this;
    }
    /**
     * Lists recovery points for a resource.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListRecoveryPointsByResource.html
     */
    toListRecoveryPointsByResource() {
        this.add('backup:ListRecoveryPointsByResource');
        return this;
    }
    /**
     * Lists restore jobs.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListRestoreJobs.html
     */
    toListRestoreJobs() {
        this.add('backup:ListRestoreJobs');
        return this;
    }
    /**
     * Lists tags for a resource.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListTags.html
     */
    toListTags() {
        this.add('backup:ListTags');
        return this;
    }
    /**
     * Adds an access policy to the backup vault.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_PutBackupVaultAccessPolicy.html
     */
    toPutBackupVaultAccessPolicy() {
        this.add('backup:PutBackupVaultAccessPolicy');
        return this;
    }
    /**
     * Adds an SNS topic to the backup vault.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_PutBackupVaultNotifications.html
     */
    toPutBackupVaultNotifications() {
        this.add('backup:PutBackupVaultNotifications');
        return this;
    }
    /**
     * Starts a new backup job.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_StartBackupJob.html
     */
    toStartBackupJob() {
        this.add('backup:StartBackupJob');
        return this;
    }
    /**
     * Copy a backup from a source region to a destination region.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_StartCopyJob.html
     */
    toStartCopyJob() {
        this.add('backup:StartCopyJob');
        return this;
    }
    /**
     * Starts a new restore job.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_StartRestoreJob.html
     */
    toStartRestoreJob() {
        this.add('backup:StartRestoreJob');
        return this;
    }
    /**
     * Stops a backup job.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_StopBackupJob.html
     */
    toStopBackupJob() {
        this.add('backup:StopBackupJob');
        return this;
    }
    /**
     * Tags a resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_TagResource.html
     */
    toTagResource() {
        this.add('backup:TagResource');
        return this;
    }
    /**
     * Untags a resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_UntagResource.html
     */
    toUntagResource() {
        this.add('backup:UntagResource');
        return this;
    }
    /**
     * Updates a backup plan.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_UpdateBackupPlan.html
     */
    toUpdateBackupPlan() {
        this.add('backup:UpdateBackupPlan');
        return this;
    }
    /**
     * Updates the lifecycle of the recovery point.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_UpdateRecoveryPointLifecycle.html
     */
    toUpdateRecoveryPointLifecycle() {
        this.add('backup:UpdateRecoveryPointLifecycle');
        return this;
    }
    /**
     * Describes region settings
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_UpdateRegionSettings.html
     */
    toUpdateRegionSettings() {
        this.add('backup:UpdateRegionSettings');
        return this;
    }
    /**
     * Adds a resource of type backupVault to the statement
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/vaults.html
     *
     * @param backupVaultName - Identifier for the backupVaultName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onBackupVault(backupVaultName, account, region, partition) {
        var arn = 'arn:${Partition}:backup:${Region}:${Account}:backup-vault:${BackupVaultName}';
        arn = arn.replace('${BackupVaultName}', backupVaultName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type backupPlan to the statement
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/about-backup-plans.html
     *
     * @param backupPlanId - Identifier for the backupPlanId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onBackupPlan(backupPlanId, account, region, partition) {
        var arn = 'arn:${Partition}:backup:${Region}:${Account}:backup-plan:${BackupPlanId}';
        arn = arn.replace('${BackupPlanId}', backupPlanId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type recoveryPoint to the statement
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/recovery-points.html
     *
     * @param recoveryPointId - Identifier for the recoveryPointId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRecoveryPoint(recoveryPointId, account, region, partition) {
        var arn = 'arn:${Partition}:backup:${Region}:${Account}:recovery-point:${RecoveryPointId}';
        arn = arn.replace('${RecoveryPointId}', recoveryPointId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Backup = Backup;
//# sourceMappingURL=data:application/json;base64,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