"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Budgets = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [budgets](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsbudgetservice.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Budgets extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [budgets](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsbudgetservice.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'budgets';
        this.actionList = {
            "ModifyBudget": {
                "url": "https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/billing-permissions-ref.html#user-permissions",
                "description": "Grants permissions to modify budgets and budget details",
                "accessLevel": "Write",
                "resourceTypes": {
                    "budget": {
                        "required": true
                    }
                }
            },
            "ViewBudget": {
                "url": "https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/billing-permissions-ref.html#user-permissions",
                "description": "Grants permissions to view budgets and budget details",
                "accessLevel": "Read",
                "resourceTypes": {
                    "budget": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "budget": {
                "name": "budget",
                "url": "https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/budgets-managing-costs.html",
                "arn": "arn:${Partition}:budgets::${Account}:budget/${BudgetName}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Grants permissions to modify budgets and budget details
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/billing-permissions-ref.html#user-permissions
     */
    toModifyBudget() {
        this.add('budgets:ModifyBudget');
        return this;
    }
    /**
     * Grants permissions to view budgets and budget details
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/billing-permissions-ref.html#user-permissions
     */
    toViewBudget() {
        this.add('budgets:ViewBudget');
        return this;
    }
    /**
     * Adds a resource of type budget to the statement
     *
     * https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/budgets-managing-costs.html
     *
     * @param budgetName - Identifier for the budgetName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onBudget(budgetName, account, partition) {
        var arn = 'arn:${Partition}:budgets::${Account}:budget/${BudgetName}';
        arn = arn.replace('${BudgetName}', budgetName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Budgets = Budgets;
//# sourceMappingURL=data:application/json;base64,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