"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Acm = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [acm](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscertificatemanager.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Acm extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [acm](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscertificatemanager.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'acm';
        this.actionList = {
            "AddTagsToCertificate": {
                "url": "https://docs.aws.amazon.com/acm/latest/APIReference/API_AddTagsToCertificate.html",
                "description": "Adds one or more tags to a certificate.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "certificate": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "DeleteCertificate": {
                "url": "https://docs.aws.amazon.com/acm/latest/APIReference/API_DeleteCertificate.html",
                "description": "Deletes a certificate and its associated private key.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "certificate": {
                        "required": true
                    }
                }
            },
            "DescribeCertificate": {
                "url": "https://docs.aws.amazon.com/acm/latest/APIReference/API_DescribeCertificate.html",
                "description": "Returns a list of the fields contained in the specified certificate.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "certificate": {
                        "required": true
                    }
                }
            },
            "ExportCertificate": {
                "url": "https://docs.aws.amazon.com/acm/latest/APIReference/API_ExportCertificate.html",
                "description": "Exports a private certificate issued by a private certificate authority (CA) for use anywhere.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "certificate": {
                        "required": true
                    }
                }
            },
            "GetCertificate": {
                "url": "https://docs.aws.amazon.com/acm/latest/APIReference/API_GetCertificate.html",
                "description": "Retrieves a certificate and certificate chain for the certificate specified by an ARN.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "certificate": {
                        "required": true
                    }
                }
            },
            "ImportCertificate": {
                "url": "https://docs.aws.amazon.com/acm/latest/APIReference/API_ImportCertificate.html",
                "description": "Imports a 3rd party SSL/TLS certificate into AWS Certificate Manager (ACM).",
                "accessLevel": "Write",
                "resourceTypes": {
                    "certificate": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "ListCertificates": {
                "url": "https://docs.aws.amazon.com/acm/latest/APIReference/API_ListCertificates.html",
                "description": "Retrieves a list of the certificate ARNs and the domain name for each ARN.",
                "accessLevel": "List"
            },
            "ListTagsForCertificate": {
                "url": "https://docs.aws.amazon.com/acm/latest/APIReference/API_ListTagsForCertificate.html",
                "description": "Lists the tags that have been applied to the certificate.",
                "accessLevel": "Read"
            },
            "RemoveTagsFromCertificate": {
                "url": "https://docs.aws.amazon.com/acm/latest/APIReference/API_RemoveTagsFromCertificate.html",
                "description": "Remove one or more tags from a certificate. A tag consists of a key-value pair",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "certificate": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "RenewCertificate": {
                "url": "https://docs.aws.amazon.com/acm/latest/APIReference/API_RenewCertificate.html",
                "description": "Renews an eligable private certificate.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "certificate": {
                        "required": true
                    }
                }
            },
            "RequestCertificate": {
                "url": "https://docs.aws.amazon.com/acm/latest/APIReference/API_RequestCertificate.html",
                "description": "Requests a public or private certificate.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "ResendValidationEmail": {
                "url": "https://docs.aws.amazon.com/acm/latest/APIReference/API_ResendValidationEmail.html",
                "description": "Resends an email to request domain ownership validation.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "certificate": {
                        "required": true
                    }
                }
            },
            "UpdateCertificateOptions": {
                "url": "https://docs.aws.amazon.com/acm/latest/APIReference/API_UpdateCertificateOptions.html",
                "description": "Updates a certificate. Use to specify whether to opt in to or out of certificate transparency logging.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "certificate": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "certificate": {
                "name": "certificate",
                "url": "https://docs.aws.amazon.com/acm/latest/userguide/authen-overview.html#acm-resources-operations",
                "arn": "arn:${Partition}:acm:${Region}:${Account}:certificate/${CertificateId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Adds one or more tags to a certificate.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/acm/latest/APIReference/API_AddTagsToCertificate.html
     */
    toAddTagsToCertificate() {
        this.add('acm:AddTagsToCertificate');
        return this;
    }
    /**
     * Deletes a certificate and its associated private key.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/acm/latest/APIReference/API_DeleteCertificate.html
     */
    toDeleteCertificate() {
        this.add('acm:DeleteCertificate');
        return this;
    }
    /**
     * Returns a list of the fields contained in the specified certificate.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/acm/latest/APIReference/API_DescribeCertificate.html
     */
    toDescribeCertificate() {
        this.add('acm:DescribeCertificate');
        return this;
    }
    /**
     * Exports a private certificate issued by a private certificate authority (CA) for use anywhere.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/acm/latest/APIReference/API_ExportCertificate.html
     */
    toExportCertificate() {
        this.add('acm:ExportCertificate');
        return this;
    }
    /**
     * Retrieves a certificate and certificate chain for the certificate specified by an ARN.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/acm/latest/APIReference/API_GetCertificate.html
     */
    toGetCertificate() {
        this.add('acm:GetCertificate');
        return this;
    }
    /**
     * Imports a 3rd party SSL/TLS certificate into AWS Certificate Manager (ACM).
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/acm/latest/APIReference/API_ImportCertificate.html
     */
    toImportCertificate() {
        this.add('acm:ImportCertificate');
        return this;
    }
    /**
     * Retrieves a list of the certificate ARNs and the domain name for each ARN.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/acm/latest/APIReference/API_ListCertificates.html
     */
    toListCertificates() {
        this.add('acm:ListCertificates');
        return this;
    }
    /**
     * Lists the tags that have been applied to the certificate.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/acm/latest/APIReference/API_ListTagsForCertificate.html
     */
    toListTagsForCertificate() {
        this.add('acm:ListTagsForCertificate');
        return this;
    }
    /**
     * Remove one or more tags from a certificate. A tag consists of a key-value pair
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/acm/latest/APIReference/API_RemoveTagsFromCertificate.html
     */
    toRemoveTagsFromCertificate() {
        this.add('acm:RemoveTagsFromCertificate');
        return this;
    }
    /**
     * Renews an eligable private certificate.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/acm/latest/APIReference/API_RenewCertificate.html
     */
    toRenewCertificate() {
        this.add('acm:RenewCertificate');
        return this;
    }
    /**
     * Requests a public or private certificate.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/acm/latest/APIReference/API_RequestCertificate.html
     */
    toRequestCertificate() {
        this.add('acm:RequestCertificate');
        return this;
    }
    /**
     * Resends an email to request domain ownership validation.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/acm/latest/APIReference/API_ResendValidationEmail.html
     */
    toResendValidationEmail() {
        this.add('acm:ResendValidationEmail');
        return this;
    }
    /**
     * Updates a certificate. Use to specify whether to opt in to or out of certificate transparency logging.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/acm/latest/APIReference/API_UpdateCertificateOptions.html
     */
    toUpdateCertificateOptions() {
        this.add('acm:UpdateCertificateOptions');
        return this;
    }
    /**
     * Adds a resource of type certificate to the statement
     *
     * https://docs.aws.amazon.com/acm/latest/userguide/authen-overview.html#acm-resources-operations
     *
     * @param certificateId - Identifier for the certificateId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCertificate(certificateId, account, region, partition) {
        var arn = 'arn:${Partition}:acm:${Region}:${Account}:certificate/${CertificateId}';
        arn = arn.replace('${CertificateId}', certificateId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Acm = Acm;
//# sourceMappingURL=data:application/json;base64,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