"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AcmPca = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [acm-pca](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscertificatemanagerprivatecertificateauthority.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class AcmPca extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [acm-pca](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscertificatemanagerprivatecertificateauthority.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'acm-pca';
        this.actionList = {
            "CreateCertificateAuthority": {
                "url": "https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_CreateCertificateAuthority.html",
                "description": "Creates an ACM Private CA and its associated private key and configuration.",
                "accessLevel": "Tagging",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateCertificateAuthorityAuditReport": {
                "url": "https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_CreateCertificateAuthorityAuditReport.html",
                "description": "Creates an audit report for an ACM Private CA.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "certificate-authority": {
                        "required": true
                    }
                }
            },
            "CreatePermission": {
                "url": "https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_CreatePermission.html",
                "description": "Creates a permission for an ACM Private CA.",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "certificate-authority": {
                        "required": true
                    }
                }
            },
            "DeleteCertificateAuthority": {
                "url": "https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_DeleteCertificateAuthority.html",
                "description": "Deletes an ACM Private CA and its associated private key and configuration.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "certificate-authority": {
                        "required": true
                    }
                }
            },
            "DeletePermission": {
                "url": "https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_DeletePermission.html",
                "description": "Deletes a permission for an ACM Private CA.",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "certificate-authority": {
                        "required": true
                    }
                }
            },
            "DeletePolicy": {
                "url": "https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_DeletePolicy.html",
                "description": "Deletes the policy for an ACM Private CA.",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "certificate-authority": {
                        "required": true
                    }
                }
            },
            "DescribeCertificateAuthority": {
                "url": "https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_DescribeCertificateAuthority.html",
                "description": "Returns a list of the configuration and status fields contained in the specified ACM Private CA.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "certificate-authority": {
                        "required": true
                    }
                }
            },
            "DescribeCertificateAuthorityAuditReport": {
                "url": "https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_DescribeCertificateAuthorityAuditReport.html",
                "description": "Returns the status and information about an ACM Private CA audit report.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "certificate-authority": {
                        "required": true
                    }
                }
            },
            "GetCertificate": {
                "url": "https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_GetCertificate.html",
                "description": "Retrieves an ACM Private CA certificate and certificate chain for the certificate authority specified by an ARN.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "certificate-authority": {
                        "required": true
                    }
                }
            },
            "GetCertificateAuthorityCertificate": {
                "url": "https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_GetCertificateAuthorityCertificate.html",
                "description": "Retrieves an ACM Private CA certificate and certificate chain for the certificate authority specified by an ARN.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "certificate-authority": {
                        "required": true
                    }
                }
            },
            "GetCertificateAuthorityCsr": {
                "url": "https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_GetCertificateAuthorityCsr.html",
                "description": "Retrieves an ACM Private CA certificate signing request (CSR) for the certificate-authority specified by an ARN.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "certificate-authority": {
                        "required": true
                    }
                }
            },
            "GetPolicy": {
                "url": "https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_GetPolicy.html",
                "description": "Retrieves the policy on an ACM Private CA.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "certificate-authority": {
                        "required": true
                    }
                }
            },
            "ImportCertificateAuthorityCertificate": {
                "url": "https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_ImportCertificateAuthorityCertificate.html",
                "description": "Imports an SSL/TLS certificate into ACM Private CA for use as the CA certificate of an ACM Private CA.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "certificate-authority": {
                        "required": true
                    }
                }
            },
            "IssueCertificate": {
                "url": "https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_IssueCertificate.html",
                "description": "Issues an ACM Private CA certificate.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "certificate-authority": {
                        "required": true
                    }
                },
                "conditions": [
                    "acm-pca:TemplateArn"
                ]
            },
            "ListCertificateAuthorities": {
                "url": "https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_ListCertificateAuthorities.html",
                "description": "Retrieves a list of the ACM Private CA certificate authority ARNs, and a summary of the status of each CA in the calling account.",
                "accessLevel": "List"
            },
            "ListPermissions": {
                "url": "https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_ListPermissions.html",
                "description": "Lists the permissions that have been applied to the ACM Private CA certificate authority.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "certificate-authority": {
                        "required": true
                    }
                }
            },
            "ListTags": {
                "url": "https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_ListTags.html",
                "description": "Lists the tags that have been applied to the ACM Private CA certificate authority.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "certificate-authority": {
                        "required": true
                    }
                }
            },
            "PutPolicy": {
                "url": "https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_PutPolicy.html",
                "description": "Puts a policy on an ACM Private CA.",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "certificate-authority": {
                        "required": true
                    }
                }
            },
            "RestoreCertificateAuthority": {
                "url": "https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_RestoreCertificateAuthority.html",
                "description": "Restores an ACM Private CA from the deleted state to the state it was in when deleted.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "certificate-authority": {
                        "required": true
                    }
                }
            },
            "RevokeCertificate": {
                "url": "https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_RevokeCertificate.html",
                "description": "Revokes a certificate issued by an ACM Private CA.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "certificate-authority": {
                        "required": true
                    }
                }
            },
            "TagCertificateAuthority": {
                "url": "https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_TagCertificateAuthority.html",
                "description": "Adds one or more tags to an ACM Private CA.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "certificate-authority": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "UntagCertificateAuthority": {
                "url": "https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_UntagCertificateAuthority.html",
                "description": "Remove one or more tags from an ACM Private CA.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "certificate-authority": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "UpdateCertificateAuthority": {
                "url": "https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_UpdateCertificateAuthority.html",
                "description": "Updates the configuration of an ACM Private CA.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "certificate-authority": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "certificate-authority": {
                "name": "certificate-authority",
                "url": "https://docs.aws.amazon.com/acm-pca/latest/userguide/authen-overview.html#acm-pca-resources-operations",
                "arn": "arn:${Partition}:acm-pca:${Region}:${Account}:certificate-authority/${CertificateAuthorityId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Creates an ACM Private CA and its associated private key and configuration.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_CreateCertificateAuthority.html
     */
    toCreateCertificateAuthority() {
        this.add('acm-pca:CreateCertificateAuthority');
        return this;
    }
    /**
     * Creates an audit report for an ACM Private CA.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_CreateCertificateAuthorityAuditReport.html
     */
    toCreateCertificateAuthorityAuditReport() {
        this.add('acm-pca:CreateCertificateAuthorityAuditReport');
        return this;
    }
    /**
     * Creates a permission for an ACM Private CA.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_CreatePermission.html
     */
    toCreatePermission() {
        this.add('acm-pca:CreatePermission');
        return this;
    }
    /**
     * Deletes an ACM Private CA and its associated private key and configuration.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_DeleteCertificateAuthority.html
     */
    toDeleteCertificateAuthority() {
        this.add('acm-pca:DeleteCertificateAuthority');
        return this;
    }
    /**
     * Deletes a permission for an ACM Private CA.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_DeletePermission.html
     */
    toDeletePermission() {
        this.add('acm-pca:DeletePermission');
        return this;
    }
    /**
     * Deletes the policy for an ACM Private CA.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_DeletePolicy.html
     */
    toDeletePolicy() {
        this.add('acm-pca:DeletePolicy');
        return this;
    }
    /**
     * Returns a list of the configuration and status fields contained in the specified ACM Private CA.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_DescribeCertificateAuthority.html
     */
    toDescribeCertificateAuthority() {
        this.add('acm-pca:DescribeCertificateAuthority');
        return this;
    }
    /**
     * Returns the status and information about an ACM Private CA audit report.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_DescribeCertificateAuthorityAuditReport.html
     */
    toDescribeCertificateAuthorityAuditReport() {
        this.add('acm-pca:DescribeCertificateAuthorityAuditReport');
        return this;
    }
    /**
     * Retrieves an ACM Private CA certificate and certificate chain for the certificate authority specified by an ARN.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_GetCertificate.html
     */
    toGetCertificate() {
        this.add('acm-pca:GetCertificate');
        return this;
    }
    /**
     * Retrieves an ACM Private CA certificate and certificate chain for the certificate authority specified by an ARN.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_GetCertificateAuthorityCertificate.html
     */
    toGetCertificateAuthorityCertificate() {
        this.add('acm-pca:GetCertificateAuthorityCertificate');
        return this;
    }
    /**
     * Retrieves an ACM Private CA certificate signing request (CSR) for the certificate-authority specified by an ARN.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_GetCertificateAuthorityCsr.html
     */
    toGetCertificateAuthorityCsr() {
        this.add('acm-pca:GetCertificateAuthorityCsr');
        return this;
    }
    /**
     * Retrieves the policy on an ACM Private CA.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_GetPolicy.html
     */
    toGetPolicy() {
        this.add('acm-pca:GetPolicy');
        return this;
    }
    /**
     * Imports an SSL/TLS certificate into ACM Private CA for use as the CA certificate of an ACM Private CA.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_ImportCertificateAuthorityCertificate.html
     */
    toImportCertificateAuthorityCertificate() {
        this.add('acm-pca:ImportCertificateAuthorityCertificate');
        return this;
    }
    /**
     * Issues an ACM Private CA certificate.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifTemplateArn()
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_IssueCertificate.html
     */
    toIssueCertificate() {
        this.add('acm-pca:IssueCertificate');
        return this;
    }
    /**
     * Retrieves a list of the ACM Private CA certificate authority ARNs, and a summary of the status of each CA in the calling account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_ListCertificateAuthorities.html
     */
    toListCertificateAuthorities() {
        this.add('acm-pca:ListCertificateAuthorities');
        return this;
    }
    /**
     * Lists the permissions that have been applied to the ACM Private CA certificate authority.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_ListPermissions.html
     */
    toListPermissions() {
        this.add('acm-pca:ListPermissions');
        return this;
    }
    /**
     * Lists the tags that have been applied to the ACM Private CA certificate authority.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_ListTags.html
     */
    toListTags() {
        this.add('acm-pca:ListTags');
        return this;
    }
    /**
     * Puts a policy on an ACM Private CA.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_PutPolicy.html
     */
    toPutPolicy() {
        this.add('acm-pca:PutPolicy');
        return this;
    }
    /**
     * Restores an ACM Private CA from the deleted state to the state it was in when deleted.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_RestoreCertificateAuthority.html
     */
    toRestoreCertificateAuthority() {
        this.add('acm-pca:RestoreCertificateAuthority');
        return this;
    }
    /**
     * Revokes a certificate issued by an ACM Private CA.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_RevokeCertificate.html
     */
    toRevokeCertificate() {
        this.add('acm-pca:RevokeCertificate');
        return this;
    }
    /**
     * Adds one or more tags to an ACM Private CA.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_TagCertificateAuthority.html
     */
    toTagCertificateAuthority() {
        this.add('acm-pca:TagCertificateAuthority');
        return this;
    }
    /**
     * Remove one or more tags from an ACM Private CA.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_UntagCertificateAuthority.html
     */
    toUntagCertificateAuthority() {
        this.add('acm-pca:UntagCertificateAuthority');
        return this;
    }
    /**
     * Updates the configuration of an ACM Private CA.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_UpdateCertificateAuthority.html
     */
    toUpdateCertificateAuthority() {
        this.add('acm-pca:UpdateCertificateAuthority');
        return this;
    }
    /**
     * Adds a resource of type certificate-authority to the statement
     *
     * https://docs.aws.amazon.com/acm-pca/latest/userguide/authen-overview.html#acm-pca-resources-operations
     *
     * @param certificateAuthorityId - Identifier for the certificateAuthorityId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCertificateAuthority(certificateAuthorityId, account, region, partition) {
        var arn = 'arn:${Partition}:acm-pca:${Region}:${Account}:certificate-authority/${CertificateAuthorityId}';
        arn = arn.replace('${CertificateAuthorityId}', certificateAuthorityId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters issue certificate requests based on the presence of TemplateArn in the request.
     *
     * Applies to actions:
     * - .toIssueCertificate()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTemplateArn(value, operator) {
        return this.if(`acm-pca:TemplateArn`, value, operator || 'StringLike');
    }
}
exports.AcmPca = AcmPca;
//# sourceMappingURL=data:application/json;base64,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