"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Codecommit = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [codecommit](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscodecommit.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Codecommit extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [codecommit](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscodecommit.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'codecommit';
        this.actionList = {
            "AssociateApprovalRuleTemplateWithRepository": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_AssociateApprovalRuleTemplateWithRepository.html",
                "description": "Grants permission to associate an approval rule template with a repository",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "BatchAssociateApprovalRuleTemplateWithRepositories": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_BatchAssociateApprovalRuleTemplateWithRepositories.html",
                "description": "Grants permission to associate an approval rule template with multiple repositories in a single operation",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "BatchDescribeMergeConflicts": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_BatchDescribeMergeConflicts.html",
                "description": "Grants permission to get information about multiple merge conflicts when attempting to merge two commits using either the three-way merge or the squash merge option",
                "accessLevel": "Read",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "BatchDisassociateApprovalRuleTemplateFromRepositories": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_BatchDisassociateApprovalRuleTemplateFromRepositories.html",
                "description": "Grants permission to remove the association between an approval rule template and multiple repositories in a single operation",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "BatchGetCommits": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_BatchGetCommits.html",
                "description": "Grants permission to get return information about one or more commits in an AWS CodeCommit repository.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "BatchGetPullRequests": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-permissions-reference.html#aa-pr",
                "description": "Grants permission to return information about one or more pull requests in an AWS CodeCommit repository",
                "accessLevel": "Read",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "BatchGetRepositories": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_BatchGetRepositories.html",
                "description": "Grants permission to get information about multiple repositories",
                "accessLevel": "Read",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "CancelUploadArchive": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-permissions-reference.html#aa-acp",
                "description": "Grants permission to cancel the uploading of an archive to a pipeline in AWS CodePipeline",
                "accessLevel": "Read",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "CreateApprovalRuleTemplate": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_CreateApprovalRuleTemplate.html",
                "description": "Grants permission to create an approval rule template that will automatically create approval rules in pull requests that match the conditions defined in the template; does not grant permission to create approval rules for individual pull requests",
                "accessLevel": "Write"
            },
            "CreateBranch": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_CreateBranch.html",
                "description": "Grants permission to create a branch in an AWS CodeCommit repository with this API; does not control Git create branch actions",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                },
                "conditions": [
                    "codecommit:References"
                ]
            },
            "CreateCommit": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_CreateCommit.html",
                "description": "Grants permission to add, copy, move or update single or multiple files in a branch in an AWS CodeCommit repository, and generate a commit for the changes in the specified branch.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                },
                "conditions": [
                    "codecommit:References"
                ]
            },
            "CreatePullRequest": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_CreatePullRequest.html",
                "description": "Grants permission to create a pull request in the specified repository",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "CreatePullRequestApprovalRule": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_CreatePullRequestApprovalRule.html",
                "description": "Grants permission to create an approval rule specific to an individual pull request; does not grant permission to create approval rule templates",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "CreateRepository": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_CreateRepository.html",
                "description": "Grants permission to create an AWS CodeCommit repository",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateUnreferencedMergeCommit": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_CreateUnreferencedMergeCommit.html",
                "description": "Grants permission to create an unreferenced commit that contains the result of merging two commits using either the three-way or the squash merge option; does not control Git merge actions",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                },
                "conditions": [
                    "codecommit:References"
                ]
            },
            "DeleteApprovalRuleTemplate": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_DeleteApprovalRuleTemplate.html",
                "description": "Grants permission to delete an approval rule template",
                "accessLevel": "Write"
            },
            "DeleteBranch": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_DeleteBranch.html",
                "description": "Grants permission to delete a branch in an AWS CodeCommit repository with this API; does not control Git delete branch actions",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                },
                "conditions": [
                    "codecommit:References"
                ]
            },
            "DeleteCommentContent": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_DeleteCommentContent.html",
                "description": "Grants permission to delete the content of a comment made on a change, file, or commit in a repository",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "DeleteFile": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_DeleteFile.html",
                "description": "Grants permission to delete a specified file from a specified branch",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                },
                "conditions": [
                    "codecommit:References"
                ]
            },
            "DeletePullRequestApprovalRule": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_DeletePullRequestApprovalRule.html",
                "description": "Grants permission to delete approval rule created for a pull request if the rule was not created by an approval rule template",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "DeleteRepository": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_DeleteRepository.html",
                "description": "Grants permission to delete an AWS CodeCommit repository",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "DescribeMergeConflicts": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_DescribeMergeConflicts.html",
                "description": "Grants permission to get information about specific merge conflicts when attempting to merge two commits using either the three-way or the squash merge option",
                "accessLevel": "Read",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "DescribePullRequestEvents": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_DescribePullRequestEvents.html",
                "description": "Grants permission to return information about one or more pull request events",
                "accessLevel": "Read",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "DisassociateApprovalRuleTemplateFromRepository": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_DisassociateApprovalRuleTemplateFromRepository.html",
                "description": "Grants permission to remove the association between an approval rule template and a repository",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "EvaluatePullRequestApprovalRules": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_EvaluatePullRequestApprovalRules.html",
                "description": "Grants permission to evaluate whether a pull request is mergable based on its current approval state and approval rule requirements",
                "accessLevel": "Read",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "GetApprovalRuleTemplate": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetApprovalRuleTemplate.html",
                "description": "Grants permission to return information about an approval rule template",
                "accessLevel": "Read"
            },
            "GetBlob": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetBlob.html",
                "description": "Grants permission to view the encoded content of an individual file in an AWS CodeCommit repository from the AWS CodeCommit console",
                "accessLevel": "Read",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "GetBranch": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetBranch.html",
                "description": "Grants permission to get details about a branch in an AWS CodeCommit repository with this API; does not control Git branch actions",
                "accessLevel": "Read",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "GetComment": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetComment.html",
                "description": "Grants permission to get the content of a comment made on a change, file, or commit in a repository",
                "accessLevel": "Read",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "GetCommentReactions": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetCommentReactions.html",
                "description": "Grants permission to get the reactions on a comment",
                "accessLevel": "Read",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "GetCommentsForComparedCommit": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetCommentsForComparedCommit.html",
                "description": "Grants permission to get information about comments made on the comparison between two commits",
                "accessLevel": "Read",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "GetCommentsForPullRequest": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetCommentsForPullRequest.html",
                "description": "Grants permission to get comments made on a pull request",
                "accessLevel": "Read",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "GetCommit": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetCommit.html",
                "description": "Grants permission to return information about a commit, including commit message and committer information, with this API; does not control Git log actions",
                "accessLevel": "Read",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "GetCommitHistory": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-permissions-reference.html#aa-code",
                "description": "Grants permission to get information about the history of commits in a repository",
                "accessLevel": "Read",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "GetCommitsFromMergeBase": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-permissions-reference.html#aa-pr",
                "description": "Grants permission to get information about the difference between commits in the context of a potential merge",
                "accessLevel": "Read",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "GetDifferences": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetDifferences.html",
                "description": "Grants permission to view information about the differences between valid commit specifiers such as a branch, tag, HEAD, commit ID, or other fully qualified reference",
                "accessLevel": "Read",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "GetFile": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetFile.html",
                "description": "Grants permission to return the base-64 encoded contents of a specified file and its metadata",
                "accessLevel": "Read",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "GetFolder": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetFolder.html",
                "description": "Grants permission to return the contents of a specified folder in a repository",
                "accessLevel": "Read",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "GetMergeCommit": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetMergeCommit.html",
                "description": "Grants permission to get information about a merge commit created by one of the merge options for pull requests that creates merge commits. Not all merge options create merge commits. This permission does not control Git merge actions",
                "accessLevel": "Read",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                },
                "conditions": [
                    "codecommit:References"
                ]
            },
            "GetMergeConflicts": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetMergeConflicts.html",
                "description": "Grants permission to get information about merge conflicts between the before and after commit IDs for a pull request in a repository",
                "accessLevel": "Read",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "GetMergeOptions": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetMergeOptions.html",
                "description": "Grants permission to get information about merge options for pull requests that can be used to merge two commits; does not control Git merge actions",
                "accessLevel": "Read",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "GetObjectIdentifier": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-permissions-reference.html#aa-code",
                "description": "Grants permission to resolve blobs, trees, and commits to their identifier",
                "accessLevel": "Read",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "GetPullRequest": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetPullRequest.html",
                "description": "Grants permission to get information about a pull request in a specified repository",
                "accessLevel": "Read",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "GetPullRequestApprovalStates": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetPullRequestApprovalStates.html",
                "description": "Grants permission to retrieve the current approvals on an inputted pull request",
                "accessLevel": "Read",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "GetPullRequestOverrideState": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetPullRequestOverrideState.html",
                "description": "Grants permission to retrieve the current override state of a given pull request",
                "accessLevel": "Read",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "GetReferences": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-permissions-reference.html#aa-code",
                "description": "Grants permission to get details about references in an AWS CodeCommit repository; does not control Git reference actions",
                "accessLevel": "Read",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "GetRepository": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetRepository.html",
                "description": "Grants permission to get information about an AWS CodeCommit repository",
                "accessLevel": "Read",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "GetRepositoryTriggers": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetRepositoryTriggers.html",
                "description": "Grants permission to get information about triggers configured for a repository",
                "accessLevel": "Read",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "GetTree": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-permissions-reference.html#aa-code",
                "description": "Grants permission to view the contents of a specified tree in an AWS CodeCommit repository from the AWS CodeCommit console",
                "accessLevel": "Read",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "GetUploadArchiveStatus": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-permissions-reference.html#aa-acp",
                "description": "Grants permission to get status information about an archive upload to a pipeline in AWS CodePipeline",
                "accessLevel": "Read",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "GitPull": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-permissions-reference.html#aa-git",
                "description": "Grants permission to pull information from an AWS CodeCommit repository to a local repo",
                "accessLevel": "Read",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "GitPush": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-permissions-reference.html#aa-git",
                "description": "Grants permission to push information from a local repo to an AWS CodeCommit repository",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                },
                "conditions": [
                    "codecommit:References"
                ]
            },
            "ListApprovalRuleTemplates": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_ListApprovalRuleTemplates.html",
                "description": "Grants permission to list all approval rule templates in an AWS Region for the AWS account",
                "accessLevel": "List"
            },
            "ListAssociatedApprovalRuleTemplatesForRepository": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_ListAssociatedApprovalRuleTemplatesForRepository.html",
                "description": "Grants permission to list approval rule templates that are associated with a repository",
                "accessLevel": "List",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "ListBranches": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_ListBranches.html",
                "description": "Grants permission to list branches for an AWS CodeCommit repository with this API; does not control Git branch actions",
                "accessLevel": "List",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "ListPullRequests": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_ListPullRequests.html",
                "description": "Grants permission to list pull requests for a specified repository",
                "accessLevel": "List",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "ListRepositories": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_ListRepositories.html",
                "description": "Grants permission to list information about AWS CodeCommit repositories in the current Region for your AWS account",
                "accessLevel": "List"
            },
            "ListRepositoriesForApprovalRuleTemplate": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_ListRepositoriesForApprovalRuleTemplate.html",
                "description": "Grants permission to list repositories that are associated with an approval rule template",
                "accessLevel": "List"
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_ListTagsForResource.html",
                "description": "Grants permission to list the resource attached to a CodeCommit resource ARN",
                "accessLevel": "List",
                "resourceTypes": {
                    "repository": {
                        "required": false
                    }
                }
            },
            "MergeBranchesByFastForward": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_MergeBranchesByFastForward.html",
                "description": "Grants permission to merge two commits into the specified destination branch using the fast-forward merge option",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                },
                "conditions": [
                    "codecommit:References"
                ]
            },
            "MergeBranchesBySquash": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_MergeBranchesBySquash.html",
                "description": "Grants permission to merge two commits into the specified destination branch using the squash merge option",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                },
                "conditions": [
                    "codecommit:References"
                ]
            },
            "MergeBranchesByThreeWay": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_MergeBranchesByThreeWay.html",
                "description": "Grants permission to merge two commits into the specified destination branch using the three-way merge option",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                },
                "conditions": [
                    "codecommit:References"
                ]
            },
            "MergePullRequestByFastForward": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_MergePullRequestByFastForward.html",
                "description": "Grants permission to close a pull request and attempt to merge it into the specified destination branch for that pull request at the specified commit using the fast-forward merge option",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                },
                "conditions": [
                    "codecommit:References"
                ]
            },
            "MergePullRequestBySquash": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_MergePullRequestBySquash.html",
                "description": "Grants permission to close a pull request and attempt to merge it into the specified destination branch for that pull request at the specified commit using the squash merge option",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                },
                "conditions": [
                    "codecommit:References"
                ]
            },
            "MergePullRequestByThreeWay": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_MergePullRequestByThreeWay.html",
                "description": "Grants permission to close a pull request and attempt to merge it into the specified destination branch for that pull request at the specified commit using the three-way merge option",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                },
                "conditions": [
                    "codecommit:References"
                ]
            },
            "OverridePullRequestApprovalRules": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_OverridePullRequestApprovalRules.html",
                "description": "Grants permission to override all approval rules for a pull request, including approval rules created by a template",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "PostCommentForComparedCommit": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_PostCommentForComparedCommit.html",
                "description": "Grants permission to post a comment on the comparison between two commits",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "PostCommentForPullRequest": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_PostCommentForPullRequest.html",
                "description": "Grants permission to post a comment on a pull request",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "PostCommentReply": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_PostCommentReply.html",
                "description": "Grants permission to post a comment in reply to a comment on a comparison between commits or a pull request",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "PutCommentReaction": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_PutCommentReaction.html",
                "description": "Grants permission to post a reaction on a comment",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "PutFile": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_PutFile.html",
                "description": "Grants permission to add or update a file in a branch in an AWS CodeCommit repository, and generate a commit for the addition in the specified branch",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                },
                "conditions": [
                    "codecommit:References"
                ]
            },
            "PutRepositoryTriggers": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_PutRepositoryTriggers.html",
                "description": "Grants permission to create, update, or delete triggers for a repository",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_TagResource.html",
                "description": "Grants permission to attach resource tags to a CodeCommit resource ARN",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}",
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "TestRepositoryTriggers": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_TestRepositoryTriggers.html",
                "description": "Grants permission to test the functionality of repository triggers by sending information to the trigger target",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_UntagResource.html",
                "description": "Grants permission to disassociate resource tags from a CodeCommit resource ARN",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "UpdateApprovalRuleTemplateContent": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_UpdateApprovalRuleTemplateContent.html",
                "description": "Grants permission to update the content of approval rule templates; does not grant permission to update content of approval rules created specifically for pull requests",
                "accessLevel": "Write"
            },
            "UpdateApprovalRuleTemplateDescription": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_UpdateApprovalRuleTemplateDescription.html",
                "description": "Grants permission to update the description of approval rule templates",
                "accessLevel": "Write"
            },
            "UpdateApprovalRuleTemplateName": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_UpdateApprovalRuleTemplateName.html",
                "description": "Grants permission to update the name of approval rule templates",
                "accessLevel": "Write"
            },
            "UpdateComment": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_UpdateComment.html",
                "description": "Grants permission to update the contents of a comment if the identity matches the identity used to create the comment",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "UpdateDefaultBranch": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_UpdateDefaultBranch.html",
                "description": "Grants permission to change the default branch in an AWS CodeCommit repository",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "UpdatePullRequestApprovalRuleContent": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_UpdatePullRequestApprovalRuleContent.html",
                "description": "Grants permission to update the content for approval rules created for a specific pull requests; does not grant permission to update approval rule content for rules created with an approval rule template",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "UpdatePullRequestApprovalState": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_UpdatePullRequestApprovalState.html",
                "description": "Grants permission to update the approval state for pull requests",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "UpdatePullRequestDescription": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_UpdatePullRequestDescription.html",
                "description": "Grants permission to update the description of a pull request",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "UpdatePullRequestStatus": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_UpdatePullRequestStatus.html",
                "description": "Grants permission to update the status of a pull request",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "UpdatePullRequestTitle": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_UpdatePullRequestTitle.html",
                "description": "Grants permission to update the title of a pull request",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "UpdateRepositoryDescription": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_UpdateRepositoryDescription.html",
                "description": "Grants permission to change the description of an AWS CodeCommit repository",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "UpdateRepositoryName": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/APIReference/API_UpdateRepositoryName.html",
                "description": "Grants permission to change the name of an AWS CodeCommit repository",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            },
            "UploadArchive": {
                "url": "https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-permissions-reference.html#aa-acp",
                "description": "Grants permission to the service role for AWS CodePipeline to upload repository changes into a pipeline",
                "accessLevel": "Write",
                "resourceTypes": {
                    "repository": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "repository": {
                "name": "repository",
                "url": "https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-iam-access-control-identity-based.html#arn-formats",
                "arn": "arn:${Partition}:codecommit:${Region}:${Account}:${RepositoryName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Grants permission to associate an approval rule template with a repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_AssociateApprovalRuleTemplateWithRepository.html
     */
    toAssociateApprovalRuleTemplateWithRepository() {
        this.add('codecommit:AssociateApprovalRuleTemplateWithRepository');
        return this;
    }
    /**
     * Grants permission to associate an approval rule template with multiple repositories in a single operation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_BatchAssociateApprovalRuleTemplateWithRepositories.html
     */
    toBatchAssociateApprovalRuleTemplateWithRepositories() {
        this.add('codecommit:BatchAssociateApprovalRuleTemplateWithRepositories');
        return this;
    }
    /**
     * Grants permission to get information about multiple merge conflicts when attempting to merge two commits using either the three-way merge or the squash merge option
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_BatchDescribeMergeConflicts.html
     */
    toBatchDescribeMergeConflicts() {
        this.add('codecommit:BatchDescribeMergeConflicts');
        return this;
    }
    /**
     * Grants permission to remove the association between an approval rule template and multiple repositories in a single operation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_BatchDisassociateApprovalRuleTemplateFromRepositories.html
     */
    toBatchDisassociateApprovalRuleTemplateFromRepositories() {
        this.add('codecommit:BatchDisassociateApprovalRuleTemplateFromRepositories');
        return this;
    }
    /**
     * Grants permission to get return information about one or more commits in an AWS CodeCommit repository.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_BatchGetCommits.html
     */
    toBatchGetCommits() {
        this.add('codecommit:BatchGetCommits');
        return this;
    }
    /**
     * Grants permission to return information about one or more pull requests in an AWS CodeCommit repository
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-permissions-reference.html#aa-pr
     */
    toBatchGetPullRequests() {
        this.add('codecommit:BatchGetPullRequests');
        return this;
    }
    /**
     * Grants permission to get information about multiple repositories
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_BatchGetRepositories.html
     */
    toBatchGetRepositories() {
        this.add('codecommit:BatchGetRepositories');
        return this;
    }
    /**
     * Grants permission to cancel the uploading of an archive to a pipeline in AWS CodePipeline
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-permissions-reference.html#aa-acp
     */
    toCancelUploadArchive() {
        this.add('codecommit:CancelUploadArchive');
        return this;
    }
    /**
     * Grants permission to create an approval rule template that will automatically create approval rules in pull requests that match the conditions defined in the template; does not grant permission to create approval rules for individual pull requests
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_CreateApprovalRuleTemplate.html
     */
    toCreateApprovalRuleTemplate() {
        this.add('codecommit:CreateApprovalRuleTemplate');
        return this;
    }
    /**
     * Grants permission to create a branch in an AWS CodeCommit repository with this API; does not control Git create branch actions
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifReferences()
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_CreateBranch.html
     */
    toCreateBranch() {
        this.add('codecommit:CreateBranch');
        return this;
    }
    /**
     * Grants permission to add, copy, move or update single or multiple files in a branch in an AWS CodeCommit repository, and generate a commit for the changes in the specified branch.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifReferences()
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_CreateCommit.html
     */
    toCreateCommit() {
        this.add('codecommit:CreateCommit');
        return this;
    }
    /**
     * Grants permission to create a pull request in the specified repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_CreatePullRequest.html
     */
    toCreatePullRequest() {
        this.add('codecommit:CreatePullRequest');
        return this;
    }
    /**
     * Grants permission to create an approval rule specific to an individual pull request; does not grant permission to create approval rule templates
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_CreatePullRequestApprovalRule.html
     */
    toCreatePullRequestApprovalRule() {
        this.add('codecommit:CreatePullRequestApprovalRule');
        return this;
    }
    /**
     * Grants permission to create an AWS CodeCommit repository
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_CreateRepository.html
     */
    toCreateRepository() {
        this.add('codecommit:CreateRepository');
        return this;
    }
    /**
     * Grants permission to create an unreferenced commit that contains the result of merging two commits using either the three-way or the squash merge option; does not control Git merge actions
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifReferences()
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_CreateUnreferencedMergeCommit.html
     */
    toCreateUnreferencedMergeCommit() {
        this.add('codecommit:CreateUnreferencedMergeCommit');
        return this;
    }
    /**
     * Grants permission to delete an approval rule template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_DeleteApprovalRuleTemplate.html
     */
    toDeleteApprovalRuleTemplate() {
        this.add('codecommit:DeleteApprovalRuleTemplate');
        return this;
    }
    /**
     * Grants permission to delete a branch in an AWS CodeCommit repository with this API; does not control Git delete branch actions
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifReferences()
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_DeleteBranch.html
     */
    toDeleteBranch() {
        this.add('codecommit:DeleteBranch');
        return this;
    }
    /**
     * Grants permission to delete the content of a comment made on a change, file, or commit in a repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_DeleteCommentContent.html
     */
    toDeleteCommentContent() {
        this.add('codecommit:DeleteCommentContent');
        return this;
    }
    /**
     * Grants permission to delete a specified file from a specified branch
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifReferences()
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_DeleteFile.html
     */
    toDeleteFile() {
        this.add('codecommit:DeleteFile');
        return this;
    }
    /**
     * Grants permission to delete approval rule created for a pull request if the rule was not created by an approval rule template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_DeletePullRequestApprovalRule.html
     */
    toDeletePullRequestApprovalRule() {
        this.add('codecommit:DeletePullRequestApprovalRule');
        return this;
    }
    /**
     * Grants permission to delete an AWS CodeCommit repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_DeleteRepository.html
     */
    toDeleteRepository() {
        this.add('codecommit:DeleteRepository');
        return this;
    }
    /**
     * Grants permission to get information about specific merge conflicts when attempting to merge two commits using either the three-way or the squash merge option
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_DescribeMergeConflicts.html
     */
    toDescribeMergeConflicts() {
        this.add('codecommit:DescribeMergeConflicts');
        return this;
    }
    /**
     * Grants permission to return information about one or more pull request events
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_DescribePullRequestEvents.html
     */
    toDescribePullRequestEvents() {
        this.add('codecommit:DescribePullRequestEvents');
        return this;
    }
    /**
     * Grants permission to remove the association between an approval rule template and a repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_DisassociateApprovalRuleTemplateFromRepository.html
     */
    toDisassociateApprovalRuleTemplateFromRepository() {
        this.add('codecommit:DisassociateApprovalRuleTemplateFromRepository');
        return this;
    }
    /**
     * Grants permission to evaluate whether a pull request is mergable based on its current approval state and approval rule requirements
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_EvaluatePullRequestApprovalRules.html
     */
    toEvaluatePullRequestApprovalRules() {
        this.add('codecommit:EvaluatePullRequestApprovalRules');
        return this;
    }
    /**
     * Grants permission to return information about an approval rule template
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetApprovalRuleTemplate.html
     */
    toGetApprovalRuleTemplate() {
        this.add('codecommit:GetApprovalRuleTemplate');
        return this;
    }
    /**
     * Grants permission to view the encoded content of an individual file in an AWS CodeCommit repository from the AWS CodeCommit console
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetBlob.html
     */
    toGetBlob() {
        this.add('codecommit:GetBlob');
        return this;
    }
    /**
     * Grants permission to get details about a branch in an AWS CodeCommit repository with this API; does not control Git branch actions
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetBranch.html
     */
    toGetBranch() {
        this.add('codecommit:GetBranch');
        return this;
    }
    /**
     * Grants permission to get the content of a comment made on a change, file, or commit in a repository
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetComment.html
     */
    toGetComment() {
        this.add('codecommit:GetComment');
        return this;
    }
    /**
     * Grants permission to get the reactions on a comment
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetCommentReactions.html
     */
    toGetCommentReactions() {
        this.add('codecommit:GetCommentReactions');
        return this;
    }
    /**
     * Grants permission to get information about comments made on the comparison between two commits
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetCommentsForComparedCommit.html
     */
    toGetCommentsForComparedCommit() {
        this.add('codecommit:GetCommentsForComparedCommit');
        return this;
    }
    /**
     * Grants permission to get comments made on a pull request
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetCommentsForPullRequest.html
     */
    toGetCommentsForPullRequest() {
        this.add('codecommit:GetCommentsForPullRequest');
        return this;
    }
    /**
     * Grants permission to return information about a commit, including commit message and committer information, with this API; does not control Git log actions
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetCommit.html
     */
    toGetCommit() {
        this.add('codecommit:GetCommit');
        return this;
    }
    /**
     * Grants permission to get information about the history of commits in a repository
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-permissions-reference.html#aa-code
     */
    toGetCommitHistory() {
        this.add('codecommit:GetCommitHistory');
        return this;
    }
    /**
     * Grants permission to get information about the difference between commits in the context of a potential merge
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-permissions-reference.html#aa-pr
     */
    toGetCommitsFromMergeBase() {
        this.add('codecommit:GetCommitsFromMergeBase');
        return this;
    }
    /**
     * Grants permission to view information about the differences between valid commit specifiers such as a branch, tag, HEAD, commit ID, or other fully qualified reference
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetDifferences.html
     */
    toGetDifferences() {
        this.add('codecommit:GetDifferences');
        return this;
    }
    /**
     * Grants permission to return the base-64 encoded contents of a specified file and its metadata
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetFile.html
     */
    toGetFile() {
        this.add('codecommit:GetFile');
        return this;
    }
    /**
     * Grants permission to return the contents of a specified folder in a repository
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetFolder.html
     */
    toGetFolder() {
        this.add('codecommit:GetFolder');
        return this;
    }
    /**
     * Grants permission to get information about a merge commit created by one of the merge options for pull requests that creates merge commits. Not all merge options create merge commits. This permission does not control Git merge actions
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifReferences()
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetMergeCommit.html
     */
    toGetMergeCommit() {
        this.add('codecommit:GetMergeCommit');
        return this;
    }
    /**
     * Grants permission to get information about merge conflicts between the before and after commit IDs for a pull request in a repository
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetMergeConflicts.html
     */
    toGetMergeConflicts() {
        this.add('codecommit:GetMergeConflicts');
        return this;
    }
    /**
     * Grants permission to get information about merge options for pull requests that can be used to merge two commits; does not control Git merge actions
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetMergeOptions.html
     */
    toGetMergeOptions() {
        this.add('codecommit:GetMergeOptions');
        return this;
    }
    /**
     * Grants permission to resolve blobs, trees, and commits to their identifier
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-permissions-reference.html#aa-code
     */
    toGetObjectIdentifier() {
        this.add('codecommit:GetObjectIdentifier');
        return this;
    }
    /**
     * Grants permission to get information about a pull request in a specified repository
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetPullRequest.html
     */
    toGetPullRequest() {
        this.add('codecommit:GetPullRequest');
        return this;
    }
    /**
     * Grants permission to retrieve the current approvals on an inputted pull request
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetPullRequestApprovalStates.html
     */
    toGetPullRequestApprovalStates() {
        this.add('codecommit:GetPullRequestApprovalStates');
        return this;
    }
    /**
     * Grants permission to retrieve the current override state of a given pull request
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetPullRequestOverrideState.html
     */
    toGetPullRequestOverrideState() {
        this.add('codecommit:GetPullRequestOverrideState');
        return this;
    }
    /**
     * Grants permission to get details about references in an AWS CodeCommit repository; does not control Git reference actions
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-permissions-reference.html#aa-code
     */
    toGetReferences() {
        this.add('codecommit:GetReferences');
        return this;
    }
    /**
     * Grants permission to get information about an AWS CodeCommit repository
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetRepository.html
     */
    toGetRepository() {
        this.add('codecommit:GetRepository');
        return this;
    }
    /**
     * Grants permission to get information about triggers configured for a repository
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_GetRepositoryTriggers.html
     */
    toGetRepositoryTriggers() {
        this.add('codecommit:GetRepositoryTriggers');
        return this;
    }
    /**
     * Grants permission to view the contents of a specified tree in an AWS CodeCommit repository from the AWS CodeCommit console
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-permissions-reference.html#aa-code
     */
    toGetTree() {
        this.add('codecommit:GetTree');
        return this;
    }
    /**
     * Grants permission to get status information about an archive upload to a pipeline in AWS CodePipeline
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-permissions-reference.html#aa-acp
     */
    toGetUploadArchiveStatus() {
        this.add('codecommit:GetUploadArchiveStatus');
        return this;
    }
    /**
     * Grants permission to pull information from an AWS CodeCommit repository to a local repo
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-permissions-reference.html#aa-git
     */
    toGitPull() {
        this.add('codecommit:GitPull');
        return this;
    }
    /**
     * Grants permission to push information from a local repo to an AWS CodeCommit repository
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifReferences()
     *
     * https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-permissions-reference.html#aa-git
     */
    toGitPush() {
        this.add('codecommit:GitPush');
        return this;
    }
    /**
     * Grants permission to list all approval rule templates in an AWS Region for the AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_ListApprovalRuleTemplates.html
     */
    toListApprovalRuleTemplates() {
        this.add('codecommit:ListApprovalRuleTemplates');
        return this;
    }
    /**
     * Grants permission to list approval rule templates that are associated with a repository
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_ListAssociatedApprovalRuleTemplatesForRepository.html
     */
    toListAssociatedApprovalRuleTemplatesForRepository() {
        this.add('codecommit:ListAssociatedApprovalRuleTemplatesForRepository');
        return this;
    }
    /**
     * Grants permission to list branches for an AWS CodeCommit repository with this API; does not control Git branch actions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_ListBranches.html
     */
    toListBranches() {
        this.add('codecommit:ListBranches');
        return this;
    }
    /**
     * Grants permission to list pull requests for a specified repository
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_ListPullRequests.html
     */
    toListPullRequests() {
        this.add('codecommit:ListPullRequests');
        return this;
    }
    /**
     * Grants permission to list information about AWS CodeCommit repositories in the current Region for your AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_ListRepositories.html
     */
    toListRepositories() {
        this.add('codecommit:ListRepositories');
        return this;
    }
    /**
     * Grants permission to list repositories that are associated with an approval rule template
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_ListRepositoriesForApprovalRuleTemplate.html
     */
    toListRepositoriesForApprovalRuleTemplate() {
        this.add('codecommit:ListRepositoriesForApprovalRuleTemplate');
        return this;
    }
    /**
     * Grants permission to list the resource attached to a CodeCommit resource ARN
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.add('codecommit:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to merge two commits into the specified destination branch using the fast-forward merge option
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifReferences()
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_MergeBranchesByFastForward.html
     */
    toMergeBranchesByFastForward() {
        this.add('codecommit:MergeBranchesByFastForward');
        return this;
    }
    /**
     * Grants permission to merge two commits into the specified destination branch using the squash merge option
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifReferences()
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_MergeBranchesBySquash.html
     */
    toMergeBranchesBySquash() {
        this.add('codecommit:MergeBranchesBySquash');
        return this;
    }
    /**
     * Grants permission to merge two commits into the specified destination branch using the three-way merge option
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifReferences()
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_MergeBranchesByThreeWay.html
     */
    toMergeBranchesByThreeWay() {
        this.add('codecommit:MergeBranchesByThreeWay');
        return this;
    }
    /**
     * Grants permission to close a pull request and attempt to merge it into the specified destination branch for that pull request at the specified commit using the fast-forward merge option
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifReferences()
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_MergePullRequestByFastForward.html
     */
    toMergePullRequestByFastForward() {
        this.add('codecommit:MergePullRequestByFastForward');
        return this;
    }
    /**
     * Grants permission to close a pull request and attempt to merge it into the specified destination branch for that pull request at the specified commit using the squash merge option
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifReferences()
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_MergePullRequestBySquash.html
     */
    toMergePullRequestBySquash() {
        this.add('codecommit:MergePullRequestBySquash');
        return this;
    }
    /**
     * Grants permission to close a pull request and attempt to merge it into the specified destination branch for that pull request at the specified commit using the three-way merge option
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifReferences()
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_MergePullRequestByThreeWay.html
     */
    toMergePullRequestByThreeWay() {
        this.add('codecommit:MergePullRequestByThreeWay');
        return this;
    }
    /**
     * Grants permission to override all approval rules for a pull request, including approval rules created by a template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_OverridePullRequestApprovalRules.html
     */
    toOverridePullRequestApprovalRules() {
        this.add('codecommit:OverridePullRequestApprovalRules');
        return this;
    }
    /**
     * Grants permission to post a comment on the comparison between two commits
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_PostCommentForComparedCommit.html
     */
    toPostCommentForComparedCommit() {
        this.add('codecommit:PostCommentForComparedCommit');
        return this;
    }
    /**
     * Grants permission to post a comment on a pull request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_PostCommentForPullRequest.html
     */
    toPostCommentForPullRequest() {
        this.add('codecommit:PostCommentForPullRequest');
        return this;
    }
    /**
     * Grants permission to post a comment in reply to a comment on a comparison between commits or a pull request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_PostCommentReply.html
     */
    toPostCommentReply() {
        this.add('codecommit:PostCommentReply');
        return this;
    }
    /**
     * Grants permission to post a reaction on a comment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_PutCommentReaction.html
     */
    toPutCommentReaction() {
        this.add('codecommit:PutCommentReaction');
        return this;
    }
    /**
     * Grants permission to add or update a file in a branch in an AWS CodeCommit repository, and generate a commit for the addition in the specified branch
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifReferences()
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_PutFile.html
     */
    toPutFile() {
        this.add('codecommit:PutFile');
        return this;
    }
    /**
     * Grants permission to create, update, or delete triggers for a repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_PutRepositoryTriggers.html
     */
    toPutRepositoryTriggers() {
        this.add('codecommit:PutRepositoryTriggers');
        return this;
    }
    /**
     * Grants permission to attach resource tags to a CodeCommit resource ARN
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        this.add('codecommit:TagResource');
        return this;
    }
    /**
     * Grants permission to test the functionality of repository triggers by sending information to the trigger target
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_TestRepositoryTriggers.html
     */
    toTestRepositoryTriggers() {
        this.add('codecommit:TestRepositoryTriggers');
        return this;
    }
    /**
     * Grants permission to disassociate resource tags from a CodeCommit resource ARN
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        this.add('codecommit:UntagResource');
        return this;
    }
    /**
     * Grants permission to update the content of approval rule templates; does not grant permission to update content of approval rules created specifically for pull requests
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_UpdateApprovalRuleTemplateContent.html
     */
    toUpdateApprovalRuleTemplateContent() {
        this.add('codecommit:UpdateApprovalRuleTemplateContent');
        return this;
    }
    /**
     * Grants permission to update the description of approval rule templates
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_UpdateApprovalRuleTemplateDescription.html
     */
    toUpdateApprovalRuleTemplateDescription() {
        this.add('codecommit:UpdateApprovalRuleTemplateDescription');
        return this;
    }
    /**
     * Grants permission to update the name of approval rule templates
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_UpdateApprovalRuleTemplateName.html
     */
    toUpdateApprovalRuleTemplateName() {
        this.add('codecommit:UpdateApprovalRuleTemplateName');
        return this;
    }
    /**
     * Grants permission to update the contents of a comment if the identity matches the identity used to create the comment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_UpdateComment.html
     */
    toUpdateComment() {
        this.add('codecommit:UpdateComment');
        return this;
    }
    /**
     * Grants permission to change the default branch in an AWS CodeCommit repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_UpdateDefaultBranch.html
     */
    toUpdateDefaultBranch() {
        this.add('codecommit:UpdateDefaultBranch');
        return this;
    }
    /**
     * Grants permission to update the content for approval rules created for a specific pull requests; does not grant permission to update approval rule content for rules created with an approval rule template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_UpdatePullRequestApprovalRuleContent.html
     */
    toUpdatePullRequestApprovalRuleContent() {
        this.add('codecommit:UpdatePullRequestApprovalRuleContent');
        return this;
    }
    /**
     * Grants permission to update the approval state for pull requests
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_UpdatePullRequestApprovalState.html
     */
    toUpdatePullRequestApprovalState() {
        this.add('codecommit:UpdatePullRequestApprovalState');
        return this;
    }
    /**
     * Grants permission to update the description of a pull request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_UpdatePullRequestDescription.html
     */
    toUpdatePullRequestDescription() {
        this.add('codecommit:UpdatePullRequestDescription');
        return this;
    }
    /**
     * Grants permission to update the status of a pull request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_UpdatePullRequestStatus.html
     */
    toUpdatePullRequestStatus() {
        this.add('codecommit:UpdatePullRequestStatus');
        return this;
    }
    /**
     * Grants permission to update the title of a pull request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_UpdatePullRequestTitle.html
     */
    toUpdatePullRequestTitle() {
        this.add('codecommit:UpdatePullRequestTitle');
        return this;
    }
    /**
     * Grants permission to change the description of an AWS CodeCommit repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_UpdateRepositoryDescription.html
     */
    toUpdateRepositoryDescription() {
        this.add('codecommit:UpdateRepositoryDescription');
        return this;
    }
    /**
     * Grants permission to change the name of an AWS CodeCommit repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/APIReference/API_UpdateRepositoryName.html
     */
    toUpdateRepositoryName() {
        this.add('codecommit:UpdateRepositoryName');
        return this;
    }
    /**
     * Grants permission to the service role for AWS CodePipeline to upload repository changes into a pipeline
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-permissions-reference.html#aa-acp
     */
    toUploadArchive() {
        this.add('codecommit:UploadArchive');
        return this;
    }
    /**
     * Adds a resource of type repository to the statement
     *
     * https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-iam-access-control-identity-based.html#arn-formats
     *
     * @param repositoryName - Identifier for the repositoryName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRepository(repositoryName, account, region, partition) {
        var arn = 'arn:${Partition}:codecommit:${Region}:${Account}:${RepositoryName}';
        arn = arn.replace('${RepositoryName}', repositoryName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters access by Git reference to specified AWS CodeCommit actions
     *
     * https://docs.aws.amazon.com/codecommit/latest/userguide/how-to-conditional-branch.html
     *
     * Applies to actions:
     * - .toCreateBranch()
     * - .toCreateCommit()
     * - .toCreateUnreferencedMergeCommit()
     * - .toDeleteBranch()
     * - .toDeleteFile()
     * - .toGetMergeCommit()
     * - .toGitPush()
     * - .toMergeBranchesByFastForward()
     * - .toMergeBranchesBySquash()
     * - .toMergeBranchesByThreeWay()
     * - .toMergePullRequestByFastForward()
     * - .toMergePullRequestBySquash()
     * - .toMergePullRequestByThreeWay()
     * - .toPutFile()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifReferences(value, operator) {
        return this.if(`codecommit:References`, value, operator || 'StringLike');
    }
}
exports.Codecommit = Codecommit;
//# sourceMappingURL=data:application/json;base64,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