"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Codedeploy = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [codedeploy](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscodedeploy.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Codedeploy extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [codedeploy](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscodedeploy.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'codedeploy';
        this.actionList = {
            "AddTagsToOnPremisesInstances": {
                "url": "https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_AddTagsToOnPremisesInstances.html",
                "description": "Add tags to one or more on-premises instances.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "instance": {
                        "required": true
                    }
                }
            },
            "BatchGetApplicationRevisions": {
                "url": "https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_BatchGetApplicationRevisions.html",
                "description": "Gets information about one or more application revisions.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "application": {
                        "required": true
                    }
                }
            },
            "BatchGetApplications": {
                "url": "https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_BatchGetApplications.html",
                "description": "Get information about multiple applications associated with the IAM user.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "application": {
                        "required": true
                    }
                }
            },
            "BatchGetDeploymentGroups": {
                "url": "https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_BatchGetDeploymentGroups.html",
                "description": "Get information about one or more deployment groups.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "deploymentgroup": {
                        "required": true
                    }
                }
            },
            "BatchGetDeploymentInstances": {
                "url": "https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_BatchGetDeploymentInstances.html",
                "description": "Gets information about one or more instance that are part of a deployment group.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "deploymentgroup": {
                        "required": true
                    }
                }
            },
            "BatchGetDeploymentTargets": {
                "url": "https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_BatchGetDeploymentTargets.html",
                "description": "Returns an array of one or more targets associated with a deployment. This method works with all compute types and should be used instead of the deprecated BatchGetDeploymentInstances. The maximum number of targets that can be returned is 25.",
                "accessLevel": "Read"
            },
            "BatchGetDeployments": {
                "url": "https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_BatchGetDeployments.html",
                "description": "Get information about multiple deployments associated with the IAM user.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "deploymentgroup": {
                        "required": true
                    }
                }
            },
            "BatchGetOnPremisesInstances": {
                "url": "https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_BatchGetOnPremisesInstances.html",
                "description": "Get information about one or more on-premises instances.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "instance": {
                        "required": true
                    }
                }
            },
            "ContinueDeployment": {
                "url": "https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_ContinueDeployment.html",
                "description": "Starts the process of rerouting traffic from instances in the original environment to instances in thereplacement environment without waiting for a specified wait time to elapse.",
                "accessLevel": "Write"
            },
            "CreateApplication": {
                "url": "https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_CreateApplication.html",
                "description": "Create an application associated with the IAM user.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "application": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateCloudFormationDeployment": {
                "url": "",
                "description": "Create CloudFormation deployment to cooperate ochestration for a CloudFormation stack update.",
                "accessLevel": "Write"
            },
            "CreateDeployment": {
                "url": "https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_CreateDeployment.html",
                "description": "Create a deployment for an application associated with the IAM user.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "deploymentgroup": {
                        "required": true
                    }
                }
            },
            "CreateDeploymentConfig": {
                "url": "https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_CreateDeploymentConfig.html",
                "description": "Create a custom deployment configuration associated with the IAM user.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "deploymentconfig": {
                        "required": true
                    }
                }
            },
            "CreateDeploymentGroup": {
                "url": "https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_CreateDeploymentGroup.html",
                "description": "Create a deployment group for an application associated with the IAM user.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "deploymentgroup": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "DeleteApplication": {
                "url": "https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_DeleteApplication.html",
                "description": "Delete an application associated with the IAM user.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "application": {
                        "required": true
                    }
                }
            },
            "DeleteDeploymentConfig": {
                "url": "https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_DeleteDeploymentConfig.html",
                "description": "Delete a custom deployment configuration associated with the IAM user.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "deploymentconfig": {
                        "required": true
                    }
                }
            },
            "DeleteDeploymentGroup": {
                "url": "https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_DeleteDeploymentGroup.html",
                "description": "Delete a deployment group for an application associated with the IAM user.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "deploymentgroup": {
                        "required": true
                    }
                }
            },
            "DeleteGitHubAccountToken": {
                "url": "https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_DeleteGitHubAccountToken.html",
                "description": "Deletes a GitHub account connection.",
                "accessLevel": "Write"
            },
            "DeleteResourcesByExternalId": {
                "url": "https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_DeleteResourcesByExternalId.html",
                "description": "Delete resources associated with the given external Id.",
                "accessLevel": "Write"
            },
            "DeregisterOnPremisesInstance": {
                "url": "https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_DeregisterOnPremisesInstance.html",
                "description": "Deregister an on-premises instance.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "instance": {
                        "required": true
                    }
                }
            },
            "GetApplication": {
                "url": "https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_GetApplication.html",
                "description": "Get information about a single application associated with the IAM user.",
                "accessLevel": "List",
                "resourceTypes": {
                    "application": {
                        "required": true
                    }
                }
            },
            "GetApplicationRevision": {
                "url": "https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_GetApplicationRevision.html",
                "description": "Get information about a single application revision for an application associated with the IAM user.",
                "accessLevel": "List",
                "resourceTypes": {
                    "application": {
                        "required": true
                    }
                }
            },
            "GetDeployment": {
                "url": "https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_GetDeployment.html",
                "description": "Get information about a single deployment to a deployment group for an application associated with the IAM user.",
                "accessLevel": "List",
                "resourceTypes": {
                    "deploymentgroup": {
                        "required": true
                    }
                }
            },
            "GetDeploymentConfig": {
                "url": "https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_GetDeploymentConfig.html",
                "description": "Get information about a single deployment configuration associated with the IAM user.",
                "accessLevel": "List",
                "resourceTypes": {
                    "deploymentconfig": {
                        "required": true
                    }
                }
            },
            "GetDeploymentGroup": {
                "url": "https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_GetDeploymentGroup.html",
                "description": "Get information about a single deployment group for an application associated with the IAM user.",
                "accessLevel": "List",
                "resourceTypes": {
                    "deploymentgroup": {
                        "required": true
                    }
                }
            },
            "GetDeploymentInstance": {
                "url": "https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_GetDeploymentInstance.html",
                "description": "Get information about a single instance in a deployment associated with the IAM user.",
                "accessLevel": "List",
                "resourceTypes": {
                    "deploymentgroup": {
                        "required": true
                    }
                }
            },
            "GetDeploymentTarget": {
                "url": "https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_GetDeploymentTarget.html",
                "description": "Returns information about a deployment target.",
                "accessLevel": "Read"
            },
            "GetOnPremisesInstance": {
                "url": "https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_GetOnPremisesInstance.html",
                "description": "Get information about a single on-premises instance.",
                "accessLevel": "List",
                "resourceTypes": {
                    "instance": {
                        "required": true
                    }
                }
            },
            "ListApplicationRevisions": {
                "url": "https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_ListApplicationRevisions.html",
                "description": "Get information about all application revisions for an application associated with the IAM user.",
                "accessLevel": "List",
                "resourceTypes": {
                    "application": {
                        "required": true
                    }
                }
            },
            "ListApplications": {
                "url": "https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_ListApplications.html",
                "description": "Get information about all applications associated with the IAM user.",
                "accessLevel": "List"
            },
            "ListDeploymentConfigs": {
                "url": "https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_ListDeploymentConfigs.html",
                "description": "Get information about all deployment configurations associated with the IAM user.",
                "accessLevel": "List"
            },
            "ListDeploymentGroups": {
                "url": "https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_ListDeploymentGroups.html",
                "description": "Get information about all deployment groups for an application associated with the IAM user.",
                "accessLevel": "List",
                "resourceTypes": {
                    "application": {
                        "required": true
                    }
                }
            },
            "ListDeploymentInstances": {
                "url": "https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_ListDeploymentInstances.html",
                "description": "Get information about all instances in a deployment associated with the IAM user.",
                "accessLevel": "List",
                "resourceTypes": {
                    "deploymentgroup": {
                        "required": true
                    }
                }
            },
            "ListDeploymentTargets": {
                "url": "https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_ListDeploymentTargets.html",
                "description": "Returns an array of target IDs that are associated a deployment.",
                "accessLevel": "List"
            },
            "ListDeployments": {
                "url": "https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_ListDeployments.html",
                "description": "Get information about all deployments to a deployment group associated with the IAM user, or to get all deployments associated with the IAM user.",
                "accessLevel": "List",
                "resourceTypes": {
                    "deploymentgroup": {
                        "required": true
                    }
                }
            },
            "ListGitHubAccountTokenNames": {
                "url": "https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_ ListGitHubAccountTokenNames.html",
                "description": "Lists the names of stored connections to GitHub accounts.",
                "accessLevel": "List"
            },
            "ListOnPremisesInstances": {
                "url": "https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_ListOnPremisesInstances.html",
                "description": "Get a list of one or more on-premises instance names.",
                "accessLevel": "List"
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_ ListTagsForResource.html",
                "description": "Returns a list of tags for the resource identified by a specified ARN. Tags are used to organize and categorize your CodeDeploy resources.",
                "accessLevel": "List",
                "resourceTypes": {
                    "application": {
                        "required": false
                    },
                    "deploymentgroup": {
                        "required": false
                    }
                }
            },
            "PutLifecycleEventHookExecutionStatus": {
                "url": "https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_PutLifecycleEventHookExecutionStatus.html",
                "description": "Notify a lifecycle event hook execution status for associated deployment with the IAM user.",
                "accessLevel": "Write"
            },
            "RegisterApplicationRevision": {
                "url": "https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_RegisterApplicationRevision.html",
                "description": "Register information about an application revision for an application associated with the IAM user.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "application": {
                        "required": true
                    }
                }
            },
            "RegisterOnPremisesInstance": {
                "url": "https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_RegisterOnPremisesInstance.html",
                "description": "Register an on-premises instance.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "instance": {
                        "required": true
                    }
                }
            },
            "RemoveTagsFromOnPremisesInstances": {
                "url": "https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_RemoveTagsFromOnPremisesInstances.html",
                "description": "Remove tags from one or more on-premises instances.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "instance": {
                        "required": true
                    }
                }
            },
            "SkipWaitTimeForInstanceTermination": {
                "url": "https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_ SkipWaitTimeForInstanceTermination.html",
                "description": "In a blue/green deployment, overrides any specified wait time and starts terminating instances immediately after the traffic routing is complete.",
                "accessLevel": "Write"
            },
            "StopDeployment": {
                "url": "https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_StopDeployment.html",
                "description": "Description for StopDeployment",
                "accessLevel": "Write"
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_TagResource.html",
                "description": "Associates the list of tags in the input Tags parameter with the resource identified by the ResourceArn input parameter.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "application": {
                        "required": false
                    },
                    "deploymentgroup": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_UntagResource.html",
                "description": "Disassociates a resource from a list of tags. The resource is identified by the ResourceArn input parameter. The tags are identfied by the list of keys in the TagKeys input parameter.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "application": {
                        "required": false
                    },
                    "deploymentgroup": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "UpdateApplication": {
                "url": "https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_UpdateApplication.html",
                "description": "Description for UpdateApplication",
                "accessLevel": "Write",
                "resourceTypes": {
                    "application": {
                        "required": true
                    }
                }
            },
            "UpdateDeploymentGroup": {
                "url": "https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_UpdateDeploymentGroup.html",
                "description": "Change information about a single deployment group for an application associated with the IAM user.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "deploymentgroup": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "application": {
                "name": "application",
                "url": "https://docs.aws.amazon.com/codedeploy/latest/userguide/auth-and-access-control-permissions-reference.html",
                "arn": "arn:${Partition}:codedeploy:${Region}:${Account}:application:${ApplicationName}",
                "conditionKeys": []
            },
            "deploymentconfig": {
                "name": "deploymentconfig",
                "url": "https://docs.aws.amazon.com/codedeploy/latest/userguide/auth-and-access-control-permissions-reference.html",
                "arn": "arn:${Partition}:codedeploy:${Region}:${Account}:deploymentconfig:${DeploymentConfigurationName}",
                "conditionKeys": []
            },
            "deploymentgroup": {
                "name": "deploymentgroup",
                "url": "https://docs.aws.amazon.com/codedeploy/latest/userguide/auth-and-access-control-permissions-reference.html",
                "arn": "arn:${Partition}:codedeploy:${Region}:${Account}:deploymentgroup:${ApplicationName}/${DeploymentGroupName}",
                "conditionKeys": []
            },
            "instance": {
                "name": "instance",
                "url": "https://docs.aws.amazon.com/codedeploy/latest/userguide/auth-and-access-control-permissions-reference.html",
                "arn": "arn:${Partition}:codedeploy:${Region}:${Account}:instance:${InstanceName}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Add tags to one or more on-premises instances.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_AddTagsToOnPremisesInstances.html
     */
    toAddTagsToOnPremisesInstances() {
        this.add('codedeploy:AddTagsToOnPremisesInstances');
        return this;
    }
    /**
     * Gets information about one or more application revisions.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_BatchGetApplicationRevisions.html
     */
    toBatchGetApplicationRevisions() {
        this.add('codedeploy:BatchGetApplicationRevisions');
        return this;
    }
    /**
     * Get information about multiple applications associated with the IAM user.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_BatchGetApplications.html
     */
    toBatchGetApplications() {
        this.add('codedeploy:BatchGetApplications');
        return this;
    }
    /**
     * Get information about one or more deployment groups.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_BatchGetDeploymentGroups.html
     */
    toBatchGetDeploymentGroups() {
        this.add('codedeploy:BatchGetDeploymentGroups');
        return this;
    }
    /**
     * Gets information about one or more instance that are part of a deployment group.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_BatchGetDeploymentInstances.html
     */
    toBatchGetDeploymentInstances() {
        this.add('codedeploy:BatchGetDeploymentInstances');
        return this;
    }
    /**
     * Returns an array of one or more targets associated with a deployment. This method works with all compute types and should be used instead of the deprecated BatchGetDeploymentInstances. The maximum number of targets that can be returned is 25.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_BatchGetDeploymentTargets.html
     */
    toBatchGetDeploymentTargets() {
        this.add('codedeploy:BatchGetDeploymentTargets');
        return this;
    }
    /**
     * Get information about multiple deployments associated with the IAM user.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_BatchGetDeployments.html
     */
    toBatchGetDeployments() {
        this.add('codedeploy:BatchGetDeployments');
        return this;
    }
    /**
     * Get information about one or more on-premises instances.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_BatchGetOnPremisesInstances.html
     */
    toBatchGetOnPremisesInstances() {
        this.add('codedeploy:BatchGetOnPremisesInstances');
        return this;
    }
    /**
     * Starts the process of rerouting traffic from instances in the original environment to instances in thereplacement environment without waiting for a specified wait time to elapse.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_ContinueDeployment.html
     */
    toContinueDeployment() {
        this.add('codedeploy:ContinueDeployment');
        return this;
    }
    /**
     * Create an application associated with the IAM user.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_CreateApplication.html
     */
    toCreateApplication() {
        this.add('codedeploy:CreateApplication');
        return this;
    }
    /**
     * Create CloudFormation deployment to cooperate ochestration for a CloudFormation stack update.
     *
     * Access Level: Write
     */
    toCreateCloudFormationDeployment() {
        this.add('codedeploy:CreateCloudFormationDeployment');
        return this;
    }
    /**
     * Create a deployment for an application associated with the IAM user.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_CreateDeployment.html
     */
    toCreateDeployment() {
        this.add('codedeploy:CreateDeployment');
        return this;
    }
    /**
     * Create a custom deployment configuration associated with the IAM user.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_CreateDeploymentConfig.html
     */
    toCreateDeploymentConfig() {
        this.add('codedeploy:CreateDeploymentConfig');
        return this;
    }
    /**
     * Create a deployment group for an application associated with the IAM user.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_CreateDeploymentGroup.html
     */
    toCreateDeploymentGroup() {
        this.add('codedeploy:CreateDeploymentGroup');
        return this;
    }
    /**
     * Delete an application associated with the IAM user.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_DeleteApplication.html
     */
    toDeleteApplication() {
        this.add('codedeploy:DeleteApplication');
        return this;
    }
    /**
     * Delete a custom deployment configuration associated with the IAM user.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_DeleteDeploymentConfig.html
     */
    toDeleteDeploymentConfig() {
        this.add('codedeploy:DeleteDeploymentConfig');
        return this;
    }
    /**
     * Delete a deployment group for an application associated with the IAM user.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_DeleteDeploymentGroup.html
     */
    toDeleteDeploymentGroup() {
        this.add('codedeploy:DeleteDeploymentGroup');
        return this;
    }
    /**
     * Deletes a GitHub account connection.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_DeleteGitHubAccountToken.html
     */
    toDeleteGitHubAccountToken() {
        this.add('codedeploy:DeleteGitHubAccountToken');
        return this;
    }
    /**
     * Delete resources associated with the given external Id.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_DeleteResourcesByExternalId.html
     */
    toDeleteResourcesByExternalId() {
        this.add('codedeploy:DeleteResourcesByExternalId');
        return this;
    }
    /**
     * Deregister an on-premises instance.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_DeregisterOnPremisesInstance.html
     */
    toDeregisterOnPremisesInstance() {
        this.add('codedeploy:DeregisterOnPremisesInstance');
        return this;
    }
    /**
     * Get information about a single application associated with the IAM user.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_GetApplication.html
     */
    toGetApplication() {
        this.add('codedeploy:GetApplication');
        return this;
    }
    /**
     * Get information about a single application revision for an application associated with the IAM user.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_GetApplicationRevision.html
     */
    toGetApplicationRevision() {
        this.add('codedeploy:GetApplicationRevision');
        return this;
    }
    /**
     * Get information about a single deployment to a deployment group for an application associated with the IAM user.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_GetDeployment.html
     */
    toGetDeployment() {
        this.add('codedeploy:GetDeployment');
        return this;
    }
    /**
     * Get information about a single deployment configuration associated with the IAM user.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_GetDeploymentConfig.html
     */
    toGetDeploymentConfig() {
        this.add('codedeploy:GetDeploymentConfig');
        return this;
    }
    /**
     * Get information about a single deployment group for an application associated with the IAM user.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_GetDeploymentGroup.html
     */
    toGetDeploymentGroup() {
        this.add('codedeploy:GetDeploymentGroup');
        return this;
    }
    /**
     * Get information about a single instance in a deployment associated with the IAM user.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_GetDeploymentInstance.html
     */
    toGetDeploymentInstance() {
        this.add('codedeploy:GetDeploymentInstance');
        return this;
    }
    /**
     * Returns information about a deployment target.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_GetDeploymentTarget.html
     */
    toGetDeploymentTarget() {
        this.add('codedeploy:GetDeploymentTarget');
        return this;
    }
    /**
     * Get information about a single on-premises instance.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_GetOnPremisesInstance.html
     */
    toGetOnPremisesInstance() {
        this.add('codedeploy:GetOnPremisesInstance');
        return this;
    }
    /**
     * Get information about all application revisions for an application associated with the IAM user.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_ListApplicationRevisions.html
     */
    toListApplicationRevisions() {
        this.add('codedeploy:ListApplicationRevisions');
        return this;
    }
    /**
     * Get information about all applications associated with the IAM user.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_ListApplications.html
     */
    toListApplications() {
        this.add('codedeploy:ListApplications');
        return this;
    }
    /**
     * Get information about all deployment configurations associated with the IAM user.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_ListDeploymentConfigs.html
     */
    toListDeploymentConfigs() {
        this.add('codedeploy:ListDeploymentConfigs');
        return this;
    }
    /**
     * Get information about all deployment groups for an application associated with the IAM user.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_ListDeploymentGroups.html
     */
    toListDeploymentGroups() {
        this.add('codedeploy:ListDeploymentGroups');
        return this;
    }
    /**
     * Get information about all instances in a deployment associated with the IAM user.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_ListDeploymentInstances.html
     */
    toListDeploymentInstances() {
        this.add('codedeploy:ListDeploymentInstances');
        return this;
    }
    /**
     * Returns an array of target IDs that are associated a deployment.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_ListDeploymentTargets.html
     */
    toListDeploymentTargets() {
        this.add('codedeploy:ListDeploymentTargets');
        return this;
    }
    /**
     * Get information about all deployments to a deployment group associated with the IAM user, or to get all deployments associated with the IAM user.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_ListDeployments.html
     */
    toListDeployments() {
        this.add('codedeploy:ListDeployments');
        return this;
    }
    /**
     * Lists the names of stored connections to GitHub accounts.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_ ListGitHubAccountTokenNames.html
     */
    toListGitHubAccountTokenNames() {
        this.add('codedeploy:ListGitHubAccountTokenNames');
        return this;
    }
    /**
     * Get a list of one or more on-premises instance names.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_ListOnPremisesInstances.html
     */
    toListOnPremisesInstances() {
        this.add('codedeploy:ListOnPremisesInstances');
        return this;
    }
    /**
     * Returns a list of tags for the resource identified by a specified ARN. Tags are used to organize and categorize your CodeDeploy resources.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_ ListTagsForResource.html
     */
    toListTagsForResource() {
        this.add('codedeploy:ListTagsForResource');
        return this;
    }
    /**
     * Notify a lifecycle event hook execution status for associated deployment with the IAM user.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_PutLifecycleEventHookExecutionStatus.html
     */
    toPutLifecycleEventHookExecutionStatus() {
        this.add('codedeploy:PutLifecycleEventHookExecutionStatus');
        return this;
    }
    /**
     * Register information about an application revision for an application associated with the IAM user.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_RegisterApplicationRevision.html
     */
    toRegisterApplicationRevision() {
        this.add('codedeploy:RegisterApplicationRevision');
        return this;
    }
    /**
     * Register an on-premises instance.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_RegisterOnPremisesInstance.html
     */
    toRegisterOnPremisesInstance() {
        this.add('codedeploy:RegisterOnPremisesInstance');
        return this;
    }
    /**
     * Remove tags from one or more on-premises instances.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_RemoveTagsFromOnPremisesInstances.html
     */
    toRemoveTagsFromOnPremisesInstances() {
        this.add('codedeploy:RemoveTagsFromOnPremisesInstances');
        return this;
    }
    /**
     * In a blue/green deployment, overrides any specified wait time and starts terminating instances immediately after the traffic routing is complete.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_ SkipWaitTimeForInstanceTermination.html
     */
    toSkipWaitTimeForInstanceTermination() {
        this.add('codedeploy:SkipWaitTimeForInstanceTermination');
        return this;
    }
    /**
     * Description for StopDeployment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_StopDeployment.html
     */
    toStopDeployment() {
        this.add('codedeploy:StopDeployment');
        return this;
    }
    /**
     * Associates the list of tags in the input Tags parameter with the resource identified by the ResourceArn input parameter.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        this.add('codedeploy:TagResource');
        return this;
    }
    /**
     * Disassociates a resource from a list of tags. The resource is identified by the ResourceArn input parameter. The tags are identfied by the list of keys in the TagKeys input parameter.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        this.add('codedeploy:UntagResource');
        return this;
    }
    /**
     * Description for UpdateApplication
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_UpdateApplication.html
     */
    toUpdateApplication() {
        this.add('codedeploy:UpdateApplication');
        return this;
    }
    /**
     * Change information about a single deployment group for an application associated with the IAM user.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_UpdateDeploymentGroup.html
     */
    toUpdateDeploymentGroup() {
        this.add('codedeploy:UpdateDeploymentGroup');
        return this;
    }
    /**
     * Adds a resource of type application to the statement
     *
     * https://docs.aws.amazon.com/codedeploy/latest/userguide/auth-and-access-control-permissions-reference.html
     *
     * @param applicationName - Identifier for the applicationName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onApplication(applicationName, account, region, partition) {
        var arn = 'arn:${Partition}:codedeploy:${Region}:${Account}:application:${ApplicationName}';
        arn = arn.replace('${ApplicationName}', applicationName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type deploymentconfig to the statement
     *
     * https://docs.aws.amazon.com/codedeploy/latest/userguide/auth-and-access-control-permissions-reference.html
     *
     * @param deploymentConfigurationName - Identifier for the deploymentConfigurationName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onDeploymentconfig(deploymentConfigurationName, account, region, partition) {
        var arn = 'arn:${Partition}:codedeploy:${Region}:${Account}:deploymentconfig:${DeploymentConfigurationName}';
        arn = arn.replace('${DeploymentConfigurationName}', deploymentConfigurationName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type deploymentgroup to the statement
     *
     * https://docs.aws.amazon.com/codedeploy/latest/userguide/auth-and-access-control-permissions-reference.html
     *
     * @param applicationName - Identifier for the applicationName.
     * @param deploymentGroupName - Identifier for the deploymentGroupName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onDeploymentgroup(applicationName, deploymentGroupName, account, region, partition) {
        var arn = 'arn:${Partition}:codedeploy:${Region}:${Account}:deploymentgroup:${ApplicationName}/${DeploymentGroupName}';
        arn = arn.replace('${ApplicationName}', applicationName);
        arn = arn.replace('${DeploymentGroupName}', deploymentGroupName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type instance to the statement
     *
     * https://docs.aws.amazon.com/codedeploy/latest/userguide/auth-and-access-control-permissions-reference.html
     *
     * @param instanceName - Identifier for the instanceName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onInstance(instanceName, account, region, partition) {
        var arn = 'arn:${Partition}:codedeploy:${Region}:${Account}:instance:${InstanceName}';
        arn = arn.replace('${InstanceName}', instanceName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Codedeploy = Codedeploy;
//# sourceMappingURL=data:application/json;base64,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