"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Codepipeline = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [codepipeline](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscodepipeline.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Codepipeline extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [codepipeline](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscodepipeline.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'codepipeline';
        this.actionList = {
            "AcknowledgeJob": {
                "url": "https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_AcknowledgeJob.html",
                "description": "Grants permission to view information about a specified job and whether that job has been received by the job worker",
                "accessLevel": "Write"
            },
            "AcknowledgeThirdPartyJob": {
                "url": "https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_AcknowledgeThirdPartyJob.html",
                "description": "Grants permission to confirm that a job worker has received the specified job (partner actions only)",
                "accessLevel": "Write"
            },
            "CreateCustomActionType": {
                "url": "https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_CreateCustomActionType.html",
                "description": "Grants permission to create a custom action that you can use in the pipelines associated with your AWS account",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "actiontype": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreatePipeline": {
                "url": "https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_CreatePipeline.html",
                "description": "Grants permission to create a uniquely named pipeline",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "pipeline": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "DeleteCustomActionType": {
                "url": "https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_DeleteCustomActionType.html",
                "description": "Grants permission to delete a custom action",
                "accessLevel": "Write",
                "resourceTypes": {
                    "actiontype": {
                        "required": true
                    }
                }
            },
            "DeletePipeline": {
                "url": "https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_DeletePipeline.html",
                "description": "Grants permission to delete a specified pipeline",
                "accessLevel": "Write",
                "resourceTypes": {
                    "pipeline": {
                        "required": true
                    }
                }
            },
            "DeleteWebhook": {
                "url": "https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_DeleteWebhook.html",
                "description": "Grants permission to delete a specified webhook",
                "accessLevel": "Write",
                "resourceTypes": {
                    "webhook": {
                        "required": true
                    }
                }
            },
            "DeregisterWebhookWithThirdParty": {
                "url": "https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_DeregisterWebhookWithThirdParty.html",
                "description": "Grants permission to remove the registration of a webhook with the third party specified in its configuration",
                "accessLevel": "Write",
                "resourceTypes": {
                    "webhook": {
                        "required": true
                    }
                }
            },
            "DisableStageTransition": {
                "url": "https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_DisableStageTransition.html",
                "description": "Grants permission to prevent revisions from transitioning to the next stage in a pipeline",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stage": {
                        "required": true
                    }
                }
            },
            "EnableStageTransition": {
                "url": "https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_EnableStageTransition.html",
                "description": "Grants permission to allow revisions to transition to the next stage in a pipeline",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stage": {
                        "required": true
                    }
                }
            },
            "GetJobDetails": {
                "url": "https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_GetJobDetails.html",
                "description": "Grants permission to view information about a job (custom actions only)",
                "accessLevel": "Read"
            },
            "GetPipeline": {
                "url": "https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_GetPipeline.html",
                "description": "Grants permission to retrieve information about a pipeline structure",
                "accessLevel": "Read",
                "resourceTypes": {
                    "pipeline": {
                        "required": true
                    }
                }
            },
            "GetPipelineExecution": {
                "url": "https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_GetPipelineExecution.html",
                "description": "Grants permission to view information about an execution of a pipeline, including details about artifacts, the pipeline execution ID, and the name, version, and status of the pipeline",
                "accessLevel": "Read",
                "resourceTypes": {
                    "pipeline": {
                        "required": true
                    }
                }
            },
            "GetPipelineState": {
                "url": "https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_GetPipelineState.html",
                "description": "Grants permission to view information about the current state of the stages and actions of a pipeline",
                "accessLevel": "Read",
                "resourceTypes": {
                    "pipeline": {
                        "required": true
                    }
                }
            },
            "GetThirdPartyJobDetails": {
                "url": "https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_GetThirdPartyJobDetails.html",
                "description": "Grants permission to view the details of a job for a third-party action (partner actions only)",
                "accessLevel": "Read"
            },
            "ListActionExecutions": {
                "url": "https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_ListActionExecutions.html",
                "description": "Grants permission to list the action executions that have occurred in a pipeline",
                "accessLevel": "Read",
                "resourceTypes": {
                    "pipeline": {
                        "required": true
                    }
                }
            },
            "ListActionTypes": {
                "url": "https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_ListActionTypes.html",
                "description": "Grants permission to list a summary of all the action types available for pipelines in your account",
                "accessLevel": "Read",
                "resourceTypes": {
                    "actiontype": {
                        "required": true
                    }
                }
            },
            "ListPipelineExecutions": {
                "url": "https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_ListPipelineExecutions.html",
                "description": "Grants permission to list a summary of the most recent executions for a pipeline",
                "accessLevel": "List",
                "resourceTypes": {
                    "pipeline": {
                        "required": true
                    }
                }
            },
            "ListPipelines": {
                "url": "https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_ListPipelines.html",
                "description": "Grants permission to list a summary of all the pipelines associated with your AWS account",
                "accessLevel": "List",
                "resourceTypes": {
                    "pipeline": {
                        "required": true
                    }
                }
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_ListTagsForResource.html",
                "description": "Grants permission to list tags for a CodePipeline resource",
                "accessLevel": "Read",
                "resourceTypes": {
                    "actiontype": {
                        "required": false
                    },
                    "pipeline": {
                        "required": false
                    },
                    "webhook": {
                        "required": false
                    }
                }
            },
            "ListWebhooks": {
                "url": "https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_ListWebhooks.html",
                "description": "Grants permission to list all of the webhooks associated with your AWS account",
                "accessLevel": "List",
                "resourceTypes": {
                    "webhook": {
                        "required": true
                    }
                }
            },
            "PollForJobs": {
                "url": "https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_PollForJobs.html",
                "description": "Grants permission to view information about any jobs for CodePipeline to act on",
                "accessLevel": "Write",
                "resourceTypes": {
                    "actiontype": {
                        "required": true
                    }
                }
            },
            "PollForThirdPartyJobs": {
                "url": "https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_PollForThirdPartyJobs.html",
                "description": "Grants permission to determine whether there are any third-party jobs for a job worker to act on (partner actions only)",
                "accessLevel": "Write"
            },
            "PutActionRevision": {
                "url": "https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_PutActionRevision.html",
                "description": "Grants permission to edit actions in a pipeline",
                "accessLevel": "Write",
                "resourceTypes": {
                    "action": {
                        "required": true
                    }
                }
            },
            "PutApprovalResult": {
                "url": "https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_PutApprovalResult.html",
                "description": "Grants permission to provide a response (Approved or Rejected) to a manual approval request in CodePipeline",
                "accessLevel": "Write",
                "resourceTypes": {
                    "action": {
                        "required": true
                    }
                }
            },
            "PutJobFailureResult": {
                "url": "https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_PutJobFailureResult.html",
                "description": "Grants permission to represent the failure of a job as returned to the pipeline by a job worker (custom actions only)",
                "accessLevel": "Write"
            },
            "PutJobSuccessResult": {
                "url": "https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_PutJobSuccessResult.html",
                "description": "Grants permission to represent the success of a job as returned to the pipeline by a job worker (custom actions only)",
                "accessLevel": "Write"
            },
            "PutThirdPartyJobFailureResult": {
                "url": "https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_PutThirdPartyJobFailureResult.html",
                "description": "Grants permission to represent the failure of a third-party job as returned to the pipeline by a job worker (partner actions only)",
                "accessLevel": "Write"
            },
            "PutThirdPartyJobSuccessResult": {
                "url": "https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_PutThirdPartyJobSuccessResult.html",
                "description": "Grants permission to represent the success of a third-party job as returned to the pipeline by a job worker (partner actions only)",
                "accessLevel": "Write"
            },
            "PutWebhook": {
                "url": "https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_PutWebhook.html",
                "description": "Grants permission to create or update a webhook",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "pipeline": {
                        "required": true
                    },
                    "webhook": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "RegisterWebhookWithThirdParty": {
                "url": "https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_RegisterWebhookWithThirdParty.html",
                "description": "Grants permission to register a webhook with the third party specified in its configuration",
                "accessLevel": "Write",
                "resourceTypes": {
                    "webhook": {
                        "required": true
                    }
                }
            },
            "RetryStageExecution": {
                "url": "https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_RetryStageExecution.html",
                "description": "Grants permission to resume the pipeline execution by retrying the last failed actions in a stage",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stage": {
                        "required": true
                    }
                }
            },
            "StartPipelineExecution": {
                "url": "https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_StartPipelineExecution.html",
                "description": "Grants permission to run the most recent revision through the pipeline",
                "accessLevel": "Write",
                "resourceTypes": {
                    "pipeline": {
                        "required": true
                    }
                }
            },
            "StopPipelineExecution": {
                "url": "https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_StopPipelineExecution.html",
                "description": "Grants permission to stop an in-progress pipeline execution",
                "accessLevel": "Write",
                "resourceTypes": {
                    "pipeline": {
                        "required": true
                    }
                }
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_TagResource.html",
                "description": "Grants permission to tag a CodePipeline resource",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "actiontype": {
                        "required": false
                    },
                    "pipeline": {
                        "required": false
                    },
                    "webhook": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_UntagResource.html",
                "description": "Grants permission to remove a tag from a CodePipeline resource",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "actiontype": {
                        "required": false
                    },
                    "pipeline": {
                        "required": false
                    },
                    "webhook": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "UpdatePipeline": {
                "url": "https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_UpdatePipeline.html",
                "description": "Grants permission to update a pipeline with changes to the structure of the pipeline",
                "accessLevel": "Write",
                "resourceTypes": {
                    "pipeline": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "action": {
                "name": "action",
                "url": "https://docs.aws.amazon.com/codepipeline/latest/userguide/iam-access-control-identity-based.html#ACP_ARN_Format",
                "arn": "arn:${Partition}:codepipeline:${Region}:${Account}:${PipelineName}/${StageName}/${ActionName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "actiontype": {
                "name": "actiontype",
                "url": "https://docs.aws.amazon.com/codepipeline/latest/userguide/iam-access-control-identity-based.html#ACP_ARN_Format",
                "arn": "arn:${Partition}:codepipeline:${Region}:${Account}:actiontype:${Owner}/${Category}/${Provider}/${Version}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "pipeline": {
                "name": "pipeline",
                "url": "https://docs.aws.amazon.com/codepipeline/latest/userguide/iam-access-control-identity-based.html#ACP_ARN_Format",
                "arn": "arn:${Partition}:codepipeline:${Region}:${Account}:${PipelineName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "stage": {
                "name": "stage",
                "url": "https://docs.aws.amazon.com/codepipeline/latest/userguide/iam-access-control-identity-based.html#ACP_ARN_Format",
                "arn": "arn:${Partition}:codepipeline:${Region}:${Account}:${PipelineName}/${StageName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "webhook": {
                "name": "webhook",
                "url": "https://docs.aws.amazon.com/codepipeline/latest/userguide/iam-access-control-identity-based.html#ACP_ARN_Format",
                "arn": "arn:${Partition}:codepipeline:${Region}:${Account}:webhook:${WebhookName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Grants permission to view information about a specified job and whether that job has been received by the job worker
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_AcknowledgeJob.html
     */
    toAcknowledgeJob() {
        this.add('codepipeline:AcknowledgeJob');
        return this;
    }
    /**
     * Grants permission to confirm that a job worker has received the specified job (partner actions only)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_AcknowledgeThirdPartyJob.html
     */
    toAcknowledgeThirdPartyJob() {
        this.add('codepipeline:AcknowledgeThirdPartyJob');
        return this;
    }
    /**
     * Grants permission to create a custom action that you can use in the pipelines associated with your AWS account
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_CreateCustomActionType.html
     */
    toCreateCustomActionType() {
        this.add('codepipeline:CreateCustomActionType');
        return this;
    }
    /**
     * Grants permission to create a uniquely named pipeline
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_CreatePipeline.html
     */
    toCreatePipeline() {
        this.add('codepipeline:CreatePipeline');
        return this;
    }
    /**
     * Grants permission to delete a custom action
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_DeleteCustomActionType.html
     */
    toDeleteCustomActionType() {
        this.add('codepipeline:DeleteCustomActionType');
        return this;
    }
    /**
     * Grants permission to delete a specified pipeline
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_DeletePipeline.html
     */
    toDeletePipeline() {
        this.add('codepipeline:DeletePipeline');
        return this;
    }
    /**
     * Grants permission to delete a specified webhook
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_DeleteWebhook.html
     */
    toDeleteWebhook() {
        this.add('codepipeline:DeleteWebhook');
        return this;
    }
    /**
     * Grants permission to remove the registration of a webhook with the third party specified in its configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_DeregisterWebhookWithThirdParty.html
     */
    toDeregisterWebhookWithThirdParty() {
        this.add('codepipeline:DeregisterWebhookWithThirdParty');
        return this;
    }
    /**
     * Grants permission to prevent revisions from transitioning to the next stage in a pipeline
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_DisableStageTransition.html
     */
    toDisableStageTransition() {
        this.add('codepipeline:DisableStageTransition');
        return this;
    }
    /**
     * Grants permission to allow revisions to transition to the next stage in a pipeline
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_EnableStageTransition.html
     */
    toEnableStageTransition() {
        this.add('codepipeline:EnableStageTransition');
        return this;
    }
    /**
     * Grants permission to view information about a job (custom actions only)
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_GetJobDetails.html
     */
    toGetJobDetails() {
        this.add('codepipeline:GetJobDetails');
        return this;
    }
    /**
     * Grants permission to retrieve information about a pipeline structure
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_GetPipeline.html
     */
    toGetPipeline() {
        this.add('codepipeline:GetPipeline');
        return this;
    }
    /**
     * Grants permission to view information about an execution of a pipeline, including details about artifacts, the pipeline execution ID, and the name, version, and status of the pipeline
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_GetPipelineExecution.html
     */
    toGetPipelineExecution() {
        this.add('codepipeline:GetPipelineExecution');
        return this;
    }
    /**
     * Grants permission to view information about the current state of the stages and actions of a pipeline
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_GetPipelineState.html
     */
    toGetPipelineState() {
        this.add('codepipeline:GetPipelineState');
        return this;
    }
    /**
     * Grants permission to view the details of a job for a third-party action (partner actions only)
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_GetThirdPartyJobDetails.html
     */
    toGetThirdPartyJobDetails() {
        this.add('codepipeline:GetThirdPartyJobDetails');
        return this;
    }
    /**
     * Grants permission to list the action executions that have occurred in a pipeline
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_ListActionExecutions.html
     */
    toListActionExecutions() {
        this.add('codepipeline:ListActionExecutions');
        return this;
    }
    /**
     * Grants permission to list a summary of all the action types available for pipelines in your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_ListActionTypes.html
     */
    toListActionTypes() {
        this.add('codepipeline:ListActionTypes');
        return this;
    }
    /**
     * Grants permission to list a summary of the most recent executions for a pipeline
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_ListPipelineExecutions.html
     */
    toListPipelineExecutions() {
        this.add('codepipeline:ListPipelineExecutions');
        return this;
    }
    /**
     * Grants permission to list a summary of all the pipelines associated with your AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_ListPipelines.html
     */
    toListPipelines() {
        this.add('codepipeline:ListPipelines');
        return this;
    }
    /**
     * Grants permission to list tags for a CodePipeline resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.add('codepipeline:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to list all of the webhooks associated with your AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_ListWebhooks.html
     */
    toListWebhooks() {
        this.add('codepipeline:ListWebhooks');
        return this;
    }
    /**
     * Grants permission to view information about any jobs for CodePipeline to act on
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_PollForJobs.html
     */
    toPollForJobs() {
        this.add('codepipeline:PollForJobs');
        return this;
    }
    /**
     * Grants permission to determine whether there are any third-party jobs for a job worker to act on (partner actions only)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_PollForThirdPartyJobs.html
     */
    toPollForThirdPartyJobs() {
        this.add('codepipeline:PollForThirdPartyJobs');
        return this;
    }
    /**
     * Grants permission to edit actions in a pipeline
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_PutActionRevision.html
     */
    toPutActionRevision() {
        this.add('codepipeline:PutActionRevision');
        return this;
    }
    /**
     * Grants permission to provide a response (Approved or Rejected) to a manual approval request in CodePipeline
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_PutApprovalResult.html
     */
    toPutApprovalResult() {
        this.add('codepipeline:PutApprovalResult');
        return this;
    }
    /**
     * Grants permission to represent the failure of a job as returned to the pipeline by a job worker (custom actions only)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_PutJobFailureResult.html
     */
    toPutJobFailureResult() {
        this.add('codepipeline:PutJobFailureResult');
        return this;
    }
    /**
     * Grants permission to represent the success of a job as returned to the pipeline by a job worker (custom actions only)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_PutJobSuccessResult.html
     */
    toPutJobSuccessResult() {
        this.add('codepipeline:PutJobSuccessResult');
        return this;
    }
    /**
     * Grants permission to represent the failure of a third-party job as returned to the pipeline by a job worker (partner actions only)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_PutThirdPartyJobFailureResult.html
     */
    toPutThirdPartyJobFailureResult() {
        this.add('codepipeline:PutThirdPartyJobFailureResult');
        return this;
    }
    /**
     * Grants permission to represent the success of a third-party job as returned to the pipeline by a job worker (partner actions only)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_PutThirdPartyJobSuccessResult.html
     */
    toPutThirdPartyJobSuccessResult() {
        this.add('codepipeline:PutThirdPartyJobSuccessResult');
        return this;
    }
    /**
     * Grants permission to create or update a webhook
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_PutWebhook.html
     */
    toPutWebhook() {
        this.add('codepipeline:PutWebhook');
        return this;
    }
    /**
     * Grants permission to register a webhook with the third party specified in its configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_RegisterWebhookWithThirdParty.html
     */
    toRegisterWebhookWithThirdParty() {
        this.add('codepipeline:RegisterWebhookWithThirdParty');
        return this;
    }
    /**
     * Grants permission to resume the pipeline execution by retrying the last failed actions in a stage
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_RetryStageExecution.html
     */
    toRetryStageExecution() {
        this.add('codepipeline:RetryStageExecution');
        return this;
    }
    /**
     * Grants permission to run the most recent revision through the pipeline
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_StartPipelineExecution.html
     */
    toStartPipelineExecution() {
        this.add('codepipeline:StartPipelineExecution');
        return this;
    }
    /**
     * Grants permission to stop an in-progress pipeline execution
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_StopPipelineExecution.html
     */
    toStopPipelineExecution() {
        this.add('codepipeline:StopPipelineExecution');
        return this;
    }
    /**
     * Grants permission to tag a CodePipeline resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        this.add('codepipeline:TagResource');
        return this;
    }
    /**
     * Grants permission to remove a tag from a CodePipeline resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        this.add('codepipeline:UntagResource');
        return this;
    }
    /**
     * Grants permission to update a pipeline with changes to the structure of the pipeline
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_UpdatePipeline.html
     */
    toUpdatePipeline() {
        this.add('codepipeline:UpdatePipeline');
        return this;
    }
    /**
     * Adds a resource of type action to the statement
     *
     * https://docs.aws.amazon.com/codepipeline/latest/userguide/iam-access-control-identity-based.html#ACP_ARN_Format
     *
     * @param pipelineName - Identifier for the pipelineName.
     * @param stageName - Identifier for the stageName.
     * @param actionName - Identifier for the actionName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAction(pipelineName, stageName, actionName, account, region, partition) {
        var arn = 'arn:${Partition}:codepipeline:${Region}:${Account}:${PipelineName}/${StageName}/${ActionName}';
        arn = arn.replace('${PipelineName}', pipelineName);
        arn = arn.replace('${StageName}', stageName);
        arn = arn.replace('${ActionName}', actionName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type actiontype to the statement
     *
     * https://docs.aws.amazon.com/codepipeline/latest/userguide/iam-access-control-identity-based.html#ACP_ARN_Format
     *
     * @param owner - Identifier for the owner.
     * @param category - Identifier for the category.
     * @param provider - Identifier for the provider.
     * @param version - Identifier for the version.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onActiontype(owner, category, provider, version, account, region, partition) {
        var arn = 'arn:${Partition}:codepipeline:${Region}:${Account}:actiontype:${Owner}/${Category}/${Provider}/${Version}';
        arn = arn.replace('${Owner}', owner);
        arn = arn.replace('${Category}', category);
        arn = arn.replace('${Provider}', provider);
        arn = arn.replace('${Version}', version);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type pipeline to the statement
     *
     * https://docs.aws.amazon.com/codepipeline/latest/userguide/iam-access-control-identity-based.html#ACP_ARN_Format
     *
     * @param pipelineName - Identifier for the pipelineName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onPipeline(pipelineName, account, region, partition) {
        var arn = 'arn:${Partition}:codepipeline:${Region}:${Account}:${PipelineName}';
        arn = arn.replace('${PipelineName}', pipelineName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type stage to the statement
     *
     * https://docs.aws.amazon.com/codepipeline/latest/userguide/iam-access-control-identity-based.html#ACP_ARN_Format
     *
     * @param pipelineName - Identifier for the pipelineName.
     * @param stageName - Identifier for the stageName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onStage(pipelineName, stageName, account, region, partition) {
        var arn = 'arn:${Partition}:codepipeline:${Region}:${Account}:${PipelineName}/${StageName}';
        arn = arn.replace('${PipelineName}', pipelineName);
        arn = arn.replace('${StageName}', stageName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type webhook to the statement
     *
     * https://docs.aws.amazon.com/codepipeline/latest/userguide/iam-access-control-identity-based.html#ACP_ARN_Format
     *
     * @param webhookName - Identifier for the webhookName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onWebhook(webhookName, account, region, partition) {
        var arn = 'arn:${Partition}:codepipeline:${Region}:${Account}:webhook:${WebhookName}';
        arn = arn.replace('${WebhookName}', webhookName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Codepipeline = Codepipeline;
//# sourceMappingURL=data:application/json;base64,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