"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Config = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [config](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsconfig.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Config extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [config](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsconfig.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'config';
        this.actionList = {
            "BatchGetAggregateResourceConfig": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_BatchGetAggregateResourceConfig.html",
                "description": "Returns the current configuration items for resources that are present in your AWS Config aggregator",
                "accessLevel": "Read",
                "resourceTypes": {
                    "ConfigurationAggregator": {
                        "required": true
                    }
                }
            },
            "BatchGetResourceConfig": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_BatchGetResourceConfig.html",
                "description": "Returns the current configuration for one or more requested resources",
                "accessLevel": "Read"
            },
            "DeleteAggregationAuthorization": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_DeleteAggregationAuthorization.html",
                "description": "Deletes the authorization granted to the specified configuration aggregator account in a specified region",
                "accessLevel": "Write",
                "resourceTypes": {
                    "AggregationAuthorization": {
                        "required": true
                    }
                }
            },
            "DeleteConfigRule": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_DeleteConfigRule.html",
                "description": "Deletes the specified AWS Config rule and all of its evaluation results",
                "accessLevel": "Write",
                "resourceTypes": {
                    "ConfigRule": {
                        "required": true
                    }
                }
            },
            "DeleteConfigurationAggregator": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_DeleteConfigurationAggregator.html",
                "description": "Deletes the specified configuration aggregator and the aggregated data associated with the aggregator",
                "accessLevel": "Write",
                "resourceTypes": {
                    "ConfigurationAggregator": {
                        "required": true
                    }
                }
            },
            "DeleteConfigurationRecorder": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_DeleteConfigurationRecorder.html",
                "description": "Deletes the configuration recorder",
                "accessLevel": "Write"
            },
            "DeleteConformancePack": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_DeleteConformancePack.html",
                "description": "Deletes the specified conformance pack and all the AWS Config rules and all evaluation results within that conformance pack.",
                "accessLevel": "Write"
            },
            "DeleteDeliveryChannel": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_DeleteDeliveryChannel.html",
                "description": "Deletes the delivery channel",
                "accessLevel": "Write"
            },
            "DeleteEvaluationResults": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_DeleteEvaluationResults.html",
                "description": "Deletes the evaluation results for the specified Config rule",
                "accessLevel": "Write",
                "resourceTypes": {
                    "ConfigRule": {
                        "required": true
                    }
                }
            },
            "DeleteOrganizationConfigRule": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_DeleteOrganizationConfigRule.html",
                "description": "Deletes the specified organization config rule and all of its evaluation results from all member accounts in that organization.",
                "accessLevel": "Write"
            },
            "DeleteOrganizationConformancePack": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_DeleteOrganizationConformancePack.html",
                "description": "Deletes the specified organization conformance pack and all of its evaluation results from all member accounts in that organization.",
                "accessLevel": "Write"
            },
            "DeletePendingAggregationRequest": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_DeletePendingAggregationRequest.html",
                "description": "Deletes pending authorization requests for a specified aggregator account in a specified region",
                "accessLevel": "Write"
            },
            "DeleteRemediationConfiguration": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_DeleteRemediationConfiguration.html",
                "description": "Deletes the remediation configuration",
                "accessLevel": "Write",
                "resourceTypes": {
                    "RemediationConfiguration": {
                        "required": true
                    }
                }
            },
            "DeleteRemediationExceptions": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_DeleteRemediationExceptions.html",
                "description": "Deletes one or more remediation exceptions for specific resource keys for a specific AWS Config Rule.",
                "accessLevel": "Write"
            },
            "DeleteRetentionConfiguration": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_DeleteRetentionConfiguration.html",
                "description": "Deletes the retention configuration",
                "accessLevel": "Write"
            },
            "DeliverConfigSnapshot": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_DeliverConfigSnapshot.html",
                "description": "Schedules delivery of a configuration snapshot to the Amazon S3 bucket in the specified delivery channel",
                "accessLevel": "Read"
            },
            "DescribeAggregateComplianceByConfigRules": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeAggregateComplianceByConfigRules.html",
                "description": "Returns a list of compliant and noncompliant rules with the number of resources for compliant and noncompliant rules",
                "accessLevel": "List",
                "resourceTypes": {
                    "ConfigurationAggregator": {
                        "required": true
                    }
                }
            },
            "DescribeAggregationAuthorizations": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeAggregationAuthorizations.html",
                "description": "Returns a list of authorizations granted to various aggregator accounts and regions",
                "accessLevel": "List"
            },
            "DescribeComplianceByConfigRule": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeComplianceByConfigRule.html",
                "description": "Indicates whether the specified AWS Config rules are compliant",
                "accessLevel": "List",
                "resourceTypes": {
                    "ConfigRule": {
                        "required": true
                    }
                }
            },
            "DescribeComplianceByResource": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeComplianceByResource.html",
                "description": "Indicates whether the specified AWS resources are compliant",
                "accessLevel": "List"
            },
            "DescribeConfigRuleEvaluationStatus": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeConfigRuleEvaluationStatus.html",
                "description": "Returns status information for each of your AWS managed Config rules",
                "accessLevel": "List",
                "resourceTypes": {
                    "ConfigRule": {
                        "required": true
                    }
                }
            },
            "DescribeConfigRules": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeConfigRules.html",
                "description": "Returns details about your AWS Config rules",
                "accessLevel": "List",
                "resourceTypes": {
                    "ConfigRule": {
                        "required": true
                    }
                }
            },
            "DescribeConfigurationAggregatorSourcesStatus": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeConfigurationAggregatorSourcesStatus.html",
                "description": "Returns status information for sources within an aggregator",
                "accessLevel": "List",
                "resourceTypes": {
                    "ConfigurationAggregator": {
                        "required": true
                    }
                }
            },
            "DescribeConfigurationAggregators": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeConfigurationAggregators.html",
                "description": "Returns the details of one or more configuration aggregators",
                "accessLevel": "List"
            },
            "DescribeConfigurationRecorderStatus": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeConfigurationRecorderStatus.html",
                "description": "Returns the current status of the specified configuration recorder",
                "accessLevel": "List"
            },
            "DescribeConfigurationRecorders": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeConfigurationRecorders.html",
                "description": "Returns the name of one or more specified configuration recorders",
                "accessLevel": "List"
            },
            "DescribeConformancePackCompliance": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeConformancePackCompliance.html",
                "description": "Returns compliance information for each rule in that conformance pack.",
                "accessLevel": "Read"
            },
            "DescribeConformancePackStatus": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeConformancePackStatus.html",
                "description": "Provides one or more conformance packs deployment status.",
                "accessLevel": "Read"
            },
            "DescribeConformancePacks": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeConformancePacks.html",
                "description": "Returns a list of one or more conformance packs.",
                "accessLevel": "Read"
            },
            "DescribeDeliveryChannelStatus": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeDeliveryChannelStatus.html",
                "description": "Returns the current status of the specified delivery channel",
                "accessLevel": "List"
            },
            "DescribeDeliveryChannels": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeDeliveryChannels.html",
                "description": "Returns details about the specified delivery channel",
                "accessLevel": "List"
            },
            "DescribeOrganizationConfigRuleStatuses": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeOrganizationConfigRuleStatuses.html",
                "description": "Provides organization config rule deployment status for an organization.",
                "accessLevel": "Read"
            },
            "DescribeOrganizationConfigRules": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeOrganizationConfigRules.html",
                "description": "Returns a list of organization config rules.",
                "accessLevel": "Read"
            },
            "DescribeOrganizationConformancePackStatuses": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeOrganizationConformancePackStatuses.html",
                "description": "Provides organization conformance pack deployment status for an organization.",
                "accessLevel": "Read"
            },
            "DescribeOrganizationConformancePacks": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeOrganizationConformancePacks.html",
                "description": "Returns a list of organization conformance packs.",
                "accessLevel": "Read"
            },
            "DescribePendingAggregationRequests": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_DescribePendingAggregationRequests.html",
                "description": "Returns a list of all pending aggregation requests",
                "accessLevel": "List"
            },
            "DescribeRemediationConfigurations": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeRemediationConfigurations.html",
                "description": "Returns the details of one or more remediation configurations",
                "accessLevel": "List",
                "resourceTypes": {
                    "RemediationConfiguration": {
                        "required": true
                    }
                }
            },
            "DescribeRemediationExceptions": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeRemediationExceptions.html",
                "description": "Returns the details of one or more remediation exceptions.",
                "accessLevel": "List"
            },
            "DescribeRemediationExecutionStatus": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeRemediationExecutionStatus.html",
                "description": "Provides a detailed view of a Remediation Execution for a set of resources including state, timestamps and any error messages for steps that have failed",
                "accessLevel": "List",
                "resourceTypes": {
                    "RemediationConfiguration": {
                        "required": true
                    }
                }
            },
            "DescribeRetentionConfigurations": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeRetentionConfigurations.html",
                "description": "Returns the details of one or more retention configurations",
                "accessLevel": "List"
            },
            "GetAggregateComplianceDetailsByConfigRule": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_GetAggregateComplianceDetailsByConfigRule.html",
                "description": "Returns the evaluation results for the specified AWS Config rule for a specific resource in a rule",
                "accessLevel": "Read",
                "resourceTypes": {
                    "ConfigurationAggregator": {
                        "required": true
                    }
                }
            },
            "GetAggregateConfigRuleComplianceSummary": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_GetAggregateConfigRuleComplianceSummary.html",
                "description": "Returns the number of compliant and noncompliant rules for one or more accounts and regions in an aggregator",
                "accessLevel": "Read",
                "resourceTypes": {
                    "ConfigurationAggregator": {
                        "required": true
                    }
                }
            },
            "GetAggregateDiscoveredResourceCounts": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_GetAggregateDiscoveredResourceCounts.html",
                "description": "Returns the resource counts across accounts and regions that are present in your AWS Config aggregator",
                "accessLevel": "Read",
                "resourceTypes": {
                    "ConfigurationAggregator": {
                        "required": true
                    }
                }
            },
            "GetAggregateResourceConfig": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_GetAggregateResourceConfig.html",
                "description": "Returns configuration item that is aggregated for your specific resource in a specific source account and region",
                "accessLevel": "Read",
                "resourceTypes": {
                    "ConfigurationAggregator": {
                        "required": true
                    }
                }
            },
            "GetComplianceDetailsByConfigRule": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_GetComplianceDetailsByConfigRule.html",
                "description": "Returns the evaluation results for the specified AWS Config rule",
                "accessLevel": "Read",
                "resourceTypes": {
                    "ConfigRule": {
                        "required": true
                    }
                }
            },
            "GetComplianceDetailsByResource": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_GetComplianceDetailsByResource.html",
                "description": "Returns the evaluation results for the specified AWS resource",
                "accessLevel": "Read"
            },
            "GetComplianceSummaryByConfigRule": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_GetComplianceSummaryByConfigRule.html",
                "description": "Returns the number of AWS Config rules that are compliant and noncompliant, up to a maximum of 25 for each",
                "accessLevel": "Read"
            },
            "GetComplianceSummaryByResourceType": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_GetComplianceSummaryByResourceType.html",
                "description": "Returns the number of resources that are compliant and the number that are noncompliant",
                "accessLevel": "Read"
            },
            "GetConformancePackComplianceDetails": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_GetConformancePackComplianceDetails.html",
                "description": "Returns compliance details of a conformance pack for all AWS resources that are monitered by conformance pack.",
                "accessLevel": "Read"
            },
            "GetConformancePackComplianceSummary": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_GetConformancePackComplianceSummary.html",
                "description": "Provides compliance summary for one or more conformance packs.",
                "accessLevel": "Read"
            },
            "GetDiscoveredResourceCounts": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_GetDiscoveredResourceCounts.html",
                "description": "Returns the resource types, the number of each resource type, and the total number of resources that AWS Config is recording in this region for your AWS account",
                "accessLevel": "Read"
            },
            "GetOrganizationConfigRuleDetailedStatus": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_GetOrganizationConfigRuleDetailedStatus.html",
                "description": "Returns detailed status for each member account within an organization for a given organization config rule.",
                "accessLevel": "Read"
            },
            "GetOrganizationConformancePackDetailedStatus": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_GetOrganizationConformancePackDetailedStatus.html",
                "description": "Returns detailed status for each member account within an organization for a given organization conformance pack.",
                "accessLevel": "Read"
            },
            "GetResourceConfigHistory": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_GetResourceConfigHistory.html",
                "description": "Returns a list of configuration items for the specified resource",
                "accessLevel": "Read"
            },
            "ListAggregateDiscoveredResources": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_ListAggregateDiscoveredResources.html",
                "description": "Accepts a resource type and returns a list of resource identifiers that are aggregated for a specific resource type across accounts and regions",
                "accessLevel": "List",
                "resourceTypes": {
                    "ConfigurationAggregator": {
                        "required": true
                    }
                }
            },
            "ListDiscoveredResources": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_ListDiscoveredResources.html",
                "description": "Accepts a resource type and returns a list of resource identifiers for the resources of that type",
                "accessLevel": "List"
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_ListTagsForResource.html",
                "description": "List the tags for AWS Config resource",
                "accessLevel": "List",
                "resourceTypes": {
                    "AggregationAuthorization": {
                        "required": false
                    },
                    "ConfigRule": {
                        "required": false
                    },
                    "ConfigurationAggregator": {
                        "required": false
                    }
                }
            },
            "PutAggregationAuthorization": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_PutAggregationAuthorization.html",
                "description": "Authorizes the aggregator account and region to collect data from the source account and region",
                "accessLevel": "Write",
                "resourceTypes": {
                    "AggregationAuthorization": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "PutConfigRule": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_PutConfigRule.html",
                "description": "Adds or updates an AWS Config rule for evaluating whether your AWS resources comply with your desired configurations",
                "accessLevel": "Write",
                "resourceTypes": {
                    "ConfigRule": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "PutConfigurationAggregator": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_PutConfigurationAggregator.html",
                "description": "Creates and updates the configuration aggregator with the selected source accounts and regions",
                "accessLevel": "Write",
                "resourceTypes": {
                    "ConfigurationAggregator": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "PutConfigurationRecorder": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_PutConfigurationRecorder.html",
                "description": "Creates a new configuration recorder to record the selected resource configurations",
                "accessLevel": "Write"
            },
            "PutConformancePack": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_PutConformancePack.html",
                "description": "Creates or updates a conformance pack.",
                "accessLevel": "Write"
            },
            "PutDeliveryChannel": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_PutDeliveryChannel.html",
                "description": "Creates a delivery channel object to deliver configuration information to an Amazon S3 bucket and Amazon SNS topic",
                "accessLevel": "Write"
            },
            "PutEvaluations": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_PutEvaluations.html",
                "description": "Used by an AWS Lambda function to deliver evaluation results to AWS Config",
                "accessLevel": "Write"
            },
            "PutOrganizationConfigRule": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_PutOrganizationConfigRule.html",
                "description": "Adds or updates organization config rule for your entire organization evaluating whether your AWS resources comply with your desired configurations.",
                "accessLevel": "Write"
            },
            "PutOrganizationConformancePack": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_PutOrganizationConformancePack.html",
                "description": "Adds or updates organization conformance pack for your entire organization evaluating whether your AWS resources comply with your desired configurations.",
                "accessLevel": "Write"
            },
            "PutRemediationConfigurations": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_PutRemediationConfigurations.html",
                "description": "Adds or updates the remediation configuration with a specific AWS Config rule with the selected target or action",
                "accessLevel": "Write",
                "resourceTypes": {
                    "RemediationConfiguration": {
                        "required": true
                    }
                }
            },
            "PutRemediationExceptions": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_PutRemediationExceptions.html",
                "description": "Adds or updates remediation exceptions for specific resources for a specific AWS Config rule.",
                "accessLevel": "Write"
            },
            "PutRetentionConfiguration": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_PutRetentionConfiguration.html",
                "description": "Creates and updates the retention configuration with details about retention period (number of days) that AWS Config stores your historical information",
                "accessLevel": "Write"
            },
            "SelectAggregateResourceConfig": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_SelectAggregateResourceConfig.html",
                "description": "Accepts a structured query language (SQL) SELECT command and an aggregator to query configuration state of AWS resources across multiple accounts and regions, performs the corresponding search, and returns resource configurations matching the properties.",
                "accessLevel": "Read"
            },
            "SelectResourceConfig": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_SelectResourceConfig.html",
                "description": "Accepts a structured query language (SQL) SELECT command, performs the corresponding search, and returns resource configurations matching the properties",
                "accessLevel": "Read"
            },
            "StartConfigRulesEvaluation": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_StartConfigRulesEvaluation.html",
                "description": "Evaluates your resources against the specified Config rules",
                "accessLevel": "Write",
                "resourceTypes": {
                    "ConfigRule": {
                        "required": true
                    }
                }
            },
            "StartConfigurationRecorder": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_StartConfigurationRecorder.html",
                "description": "Starts recording configurations of the AWS resources you have selected to record in your AWS account",
                "accessLevel": "Write"
            },
            "StartRemediationExecution": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_StartRemediationExecution.html",
                "description": "Runs an on-demand remediation for the specified AWS Config rules against the last known remediation configuration",
                "accessLevel": "Write",
                "resourceTypes": {
                    "RemediationConfiguration": {
                        "required": true
                    }
                }
            },
            "StopConfigurationRecorder": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_StopConfigurationRecorder.html",
                "description": "Stops recording configurations of the AWS resources you have selected to record in your AWS account",
                "accessLevel": "Write"
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_TagResource.html",
                "description": "Associates the specified tags to a resource with the specified resourceArn",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "AggregationAuthorization": {
                        "required": false
                    },
                    "ConfigRule": {
                        "required": false
                    },
                    "ConfigurationAggregator": {
                        "required": false
                    },
                    "ConformancePack": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/config/latest/APIReference/API_UntagResource.html",
                "description": "Deletes specified tags from a resource",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "AggregationAuthorization": {
                        "required": false
                    },
                    "ConfigRule": {
                        "required": false
                    },
                    "ConfigurationAggregator": {
                        "required": false
                    },
                    "ConformancePack": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            }
        };
        this.resourceTypes = {
            "AggregationAuthorization": {
                "name": "AggregationAuthorization",
                "url": "https://docs.aws.amazon.com/config/latest/developerguide/config-concepts.htmlAPI_AggregationAuthorization.html",
                "arn": "arn:${Partition}:config:${Region}:${Account}:aggregation-authorization/${AggregatorAccount}/${AggregatorRegion}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "ConfigurationAggregator": {
                "name": "ConfigurationAggregator",
                "url": "https://docs.aws.amazon.com/config/latest/developerguide/config-concepts.htmlAPI_ConfigurationAggregator.html",
                "arn": "arn:${Partition}:config:${Region}:${Account}:config-aggregator/${AggregatorId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "ConfigRule": {
                "name": "ConfigRule",
                "url": "https://docs.aws.amazon.com/config/latest/developerguide/config-concepts.htmlAPI_ConfigRule.html",
                "arn": "arn:${Partition}:config:${Region}:${Account}:config-rule/${ConfigRuleId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "ConformancePack": {
                "name": "ConformancePack",
                "url": "https://docs.aws.amazon.com/config/latest/developerguide/config-concepts.htmlAPI_ConformancePack.html",
                "arn": "arn:${Partition}:config:${Region}:${Account}:conformance-pack/${ConformancePackName}/${ConformancePackId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "OrganizationConfigRule": {
                "name": "OrganizationConfigRule",
                "url": "https://docs.aws.amazon.com/config/latest/developerguide/config-concepts.htmlAPI_OrganizationConfigRule.html",
                "arn": "arn:${Partition}:config:${Region}:${Account}:organization-config-rule/${OrganizationConfigRuleId}",
                "conditionKeys": []
            },
            "OrganizationConformancePack": {
                "name": "OrganizationConformancePack",
                "url": "https://docs.aws.amazon.com/config/latest/developerguide/config-concepts.htmlAPI_OrganizationConformancePack.html",
                "arn": "arn:${Partition}:config:${Region}:${Account}:organization-conformance-pack/${OrganizationConformancePackId}",
                "conditionKeys": []
            },
            "RemediationConfiguration": {
                "name": "RemediationConfiguration",
                "url": "https://docs.aws.amazon.com/config/latest/developerguide/config-concepts.htmlAPI_RemediationConfiguration.html",
                "arn": "arn:${Partition}:config:${Region}:${Account}:remediation-configuration/${RemediationConfigurationId}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Returns the current configuration items for resources that are present in your AWS Config aggregator
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_BatchGetAggregateResourceConfig.html
     */
    toBatchGetAggregateResourceConfig() {
        this.add('config:BatchGetAggregateResourceConfig');
        return this;
    }
    /**
     * Returns the current configuration for one or more requested resources
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_BatchGetResourceConfig.html
     */
    toBatchGetResourceConfig() {
        this.add('config:BatchGetResourceConfig');
        return this;
    }
    /**
     * Deletes the authorization granted to the specified configuration aggregator account in a specified region
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DeleteAggregationAuthorization.html
     */
    toDeleteAggregationAuthorization() {
        this.add('config:DeleteAggregationAuthorization');
        return this;
    }
    /**
     * Deletes the specified AWS Config rule and all of its evaluation results
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DeleteConfigRule.html
     */
    toDeleteConfigRule() {
        this.add('config:DeleteConfigRule');
        return this;
    }
    /**
     * Deletes the specified configuration aggregator and the aggregated data associated with the aggregator
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DeleteConfigurationAggregator.html
     */
    toDeleteConfigurationAggregator() {
        this.add('config:DeleteConfigurationAggregator');
        return this;
    }
    /**
     * Deletes the configuration recorder
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DeleteConfigurationRecorder.html
     */
    toDeleteConfigurationRecorder() {
        this.add('config:DeleteConfigurationRecorder');
        return this;
    }
    /**
     * Deletes the specified conformance pack and all the AWS Config rules and all evaluation results within that conformance pack.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DeleteConformancePack.html
     */
    toDeleteConformancePack() {
        this.add('config:DeleteConformancePack');
        return this;
    }
    /**
     * Deletes the delivery channel
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DeleteDeliveryChannel.html
     */
    toDeleteDeliveryChannel() {
        this.add('config:DeleteDeliveryChannel');
        return this;
    }
    /**
     * Deletes the evaluation results for the specified Config rule
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DeleteEvaluationResults.html
     */
    toDeleteEvaluationResults() {
        this.add('config:DeleteEvaluationResults');
        return this;
    }
    /**
     * Deletes the specified organization config rule and all of its evaluation results from all member accounts in that organization.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DeleteOrganizationConfigRule.html
     */
    toDeleteOrganizationConfigRule() {
        this.add('config:DeleteOrganizationConfigRule');
        return this;
    }
    /**
     * Deletes the specified organization conformance pack and all of its evaluation results from all member accounts in that organization.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DeleteOrganizationConformancePack.html
     */
    toDeleteOrganizationConformancePack() {
        this.add('config:DeleteOrganizationConformancePack');
        return this;
    }
    /**
     * Deletes pending authorization requests for a specified aggregator account in a specified region
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DeletePendingAggregationRequest.html
     */
    toDeletePendingAggregationRequest() {
        this.add('config:DeletePendingAggregationRequest');
        return this;
    }
    /**
     * Deletes the remediation configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DeleteRemediationConfiguration.html
     */
    toDeleteRemediationConfiguration() {
        this.add('config:DeleteRemediationConfiguration');
        return this;
    }
    /**
     * Deletes one or more remediation exceptions for specific resource keys for a specific AWS Config Rule.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DeleteRemediationExceptions.html
     */
    toDeleteRemediationExceptions() {
        this.add('config:DeleteRemediationExceptions');
        return this;
    }
    /**
     * Deletes the retention configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DeleteRetentionConfiguration.html
     */
    toDeleteRetentionConfiguration() {
        this.add('config:DeleteRetentionConfiguration');
        return this;
    }
    /**
     * Schedules delivery of a configuration snapshot to the Amazon S3 bucket in the specified delivery channel
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DeliverConfigSnapshot.html
     */
    toDeliverConfigSnapshot() {
        this.add('config:DeliverConfigSnapshot');
        return this;
    }
    /**
     * Returns a list of compliant and noncompliant rules with the number of resources for compliant and noncompliant rules
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeAggregateComplianceByConfigRules.html
     */
    toDescribeAggregateComplianceByConfigRules() {
        this.add('config:DescribeAggregateComplianceByConfigRules');
        return this;
    }
    /**
     * Returns a list of authorizations granted to various aggregator accounts and regions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeAggregationAuthorizations.html
     */
    toDescribeAggregationAuthorizations() {
        this.add('config:DescribeAggregationAuthorizations');
        return this;
    }
    /**
     * Indicates whether the specified AWS Config rules are compliant
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeComplianceByConfigRule.html
     */
    toDescribeComplianceByConfigRule() {
        this.add('config:DescribeComplianceByConfigRule');
        return this;
    }
    /**
     * Indicates whether the specified AWS resources are compliant
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeComplianceByResource.html
     */
    toDescribeComplianceByResource() {
        this.add('config:DescribeComplianceByResource');
        return this;
    }
    /**
     * Returns status information for each of your AWS managed Config rules
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeConfigRuleEvaluationStatus.html
     */
    toDescribeConfigRuleEvaluationStatus() {
        this.add('config:DescribeConfigRuleEvaluationStatus');
        return this;
    }
    /**
     * Returns details about your AWS Config rules
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeConfigRules.html
     */
    toDescribeConfigRules() {
        this.add('config:DescribeConfigRules');
        return this;
    }
    /**
     * Returns status information for sources within an aggregator
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeConfigurationAggregatorSourcesStatus.html
     */
    toDescribeConfigurationAggregatorSourcesStatus() {
        this.add('config:DescribeConfigurationAggregatorSourcesStatus');
        return this;
    }
    /**
     * Returns the details of one or more configuration aggregators
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeConfigurationAggregators.html
     */
    toDescribeConfigurationAggregators() {
        this.add('config:DescribeConfigurationAggregators');
        return this;
    }
    /**
     * Returns the current status of the specified configuration recorder
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeConfigurationRecorderStatus.html
     */
    toDescribeConfigurationRecorderStatus() {
        this.add('config:DescribeConfigurationRecorderStatus');
        return this;
    }
    /**
     * Returns the name of one or more specified configuration recorders
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeConfigurationRecorders.html
     */
    toDescribeConfigurationRecorders() {
        this.add('config:DescribeConfigurationRecorders');
        return this;
    }
    /**
     * Returns compliance information for each rule in that conformance pack.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeConformancePackCompliance.html
     */
    toDescribeConformancePackCompliance() {
        this.add('config:DescribeConformancePackCompliance');
        return this;
    }
    /**
     * Provides one or more conformance packs deployment status.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeConformancePackStatus.html
     */
    toDescribeConformancePackStatus() {
        this.add('config:DescribeConformancePackStatus');
        return this;
    }
    /**
     * Returns a list of one or more conformance packs.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeConformancePacks.html
     */
    toDescribeConformancePacks() {
        this.add('config:DescribeConformancePacks');
        return this;
    }
    /**
     * Returns the current status of the specified delivery channel
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeDeliveryChannelStatus.html
     */
    toDescribeDeliveryChannelStatus() {
        this.add('config:DescribeDeliveryChannelStatus');
        return this;
    }
    /**
     * Returns details about the specified delivery channel
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeDeliveryChannels.html
     */
    toDescribeDeliveryChannels() {
        this.add('config:DescribeDeliveryChannels');
        return this;
    }
    /**
     * Provides organization config rule deployment status for an organization.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeOrganizationConfigRuleStatuses.html
     */
    toDescribeOrganizationConfigRuleStatuses() {
        this.add('config:DescribeOrganizationConfigRuleStatuses');
        return this;
    }
    /**
     * Returns a list of organization config rules.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeOrganizationConfigRules.html
     */
    toDescribeOrganizationConfigRules() {
        this.add('config:DescribeOrganizationConfigRules');
        return this;
    }
    /**
     * Provides organization conformance pack deployment status for an organization.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeOrganizationConformancePackStatuses.html
     */
    toDescribeOrganizationConformancePackStatuses() {
        this.add('config:DescribeOrganizationConformancePackStatuses');
        return this;
    }
    /**
     * Returns a list of organization conformance packs.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeOrganizationConformancePacks.html
     */
    toDescribeOrganizationConformancePacks() {
        this.add('config:DescribeOrganizationConformancePacks');
        return this;
    }
    /**
     * Returns a list of all pending aggregation requests
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribePendingAggregationRequests.html
     */
    toDescribePendingAggregationRequests() {
        this.add('config:DescribePendingAggregationRequests');
        return this;
    }
    /**
     * Returns the details of one or more remediation configurations
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeRemediationConfigurations.html
     */
    toDescribeRemediationConfigurations() {
        this.add('config:DescribeRemediationConfigurations');
        return this;
    }
    /**
     * Returns the details of one or more remediation exceptions.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeRemediationExceptions.html
     */
    toDescribeRemediationExceptions() {
        this.add('config:DescribeRemediationExceptions');
        return this;
    }
    /**
     * Provides a detailed view of a Remediation Execution for a set of resources including state, timestamps and any error messages for steps that have failed
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeRemediationExecutionStatus.html
     */
    toDescribeRemediationExecutionStatus() {
        this.add('config:DescribeRemediationExecutionStatus');
        return this;
    }
    /**
     * Returns the details of one or more retention configurations
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_DescribeRetentionConfigurations.html
     */
    toDescribeRetentionConfigurations() {
        this.add('config:DescribeRetentionConfigurations');
        return this;
    }
    /**
     * Returns the evaluation results for the specified AWS Config rule for a specific resource in a rule
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_GetAggregateComplianceDetailsByConfigRule.html
     */
    toGetAggregateComplianceDetailsByConfigRule() {
        this.add('config:GetAggregateComplianceDetailsByConfigRule');
        return this;
    }
    /**
     * Returns the number of compliant and noncompliant rules for one or more accounts and regions in an aggregator
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_GetAggregateConfigRuleComplianceSummary.html
     */
    toGetAggregateConfigRuleComplianceSummary() {
        this.add('config:GetAggregateConfigRuleComplianceSummary');
        return this;
    }
    /**
     * Returns the resource counts across accounts and regions that are present in your AWS Config aggregator
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_GetAggregateDiscoveredResourceCounts.html
     */
    toGetAggregateDiscoveredResourceCounts() {
        this.add('config:GetAggregateDiscoveredResourceCounts');
        return this;
    }
    /**
     * Returns configuration item that is aggregated for your specific resource in a specific source account and region
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_GetAggregateResourceConfig.html
     */
    toGetAggregateResourceConfig() {
        this.add('config:GetAggregateResourceConfig');
        return this;
    }
    /**
     * Returns the evaluation results for the specified AWS Config rule
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_GetComplianceDetailsByConfigRule.html
     */
    toGetComplianceDetailsByConfigRule() {
        this.add('config:GetComplianceDetailsByConfigRule');
        return this;
    }
    /**
     * Returns the evaluation results for the specified AWS resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_GetComplianceDetailsByResource.html
     */
    toGetComplianceDetailsByResource() {
        this.add('config:GetComplianceDetailsByResource');
        return this;
    }
    /**
     * Returns the number of AWS Config rules that are compliant and noncompliant, up to a maximum of 25 for each
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_GetComplianceSummaryByConfigRule.html
     */
    toGetComplianceSummaryByConfigRule() {
        this.add('config:GetComplianceSummaryByConfigRule');
        return this;
    }
    /**
     * Returns the number of resources that are compliant and the number that are noncompliant
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_GetComplianceSummaryByResourceType.html
     */
    toGetComplianceSummaryByResourceType() {
        this.add('config:GetComplianceSummaryByResourceType');
        return this;
    }
    /**
     * Returns compliance details of a conformance pack for all AWS resources that are monitered by conformance pack.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_GetConformancePackComplianceDetails.html
     */
    toGetConformancePackComplianceDetails() {
        this.add('config:GetConformancePackComplianceDetails');
        return this;
    }
    /**
     * Provides compliance summary for one or more conformance packs.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_GetConformancePackComplianceSummary.html
     */
    toGetConformancePackComplianceSummary() {
        this.add('config:GetConformancePackComplianceSummary');
        return this;
    }
    /**
     * Returns the resource types, the number of each resource type, and the total number of resources that AWS Config is recording in this region for your AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_GetDiscoveredResourceCounts.html
     */
    toGetDiscoveredResourceCounts() {
        this.add('config:GetDiscoveredResourceCounts');
        return this;
    }
    /**
     * Returns detailed status for each member account within an organization for a given organization config rule.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_GetOrganizationConfigRuleDetailedStatus.html
     */
    toGetOrganizationConfigRuleDetailedStatus() {
        this.add('config:GetOrganizationConfigRuleDetailedStatus');
        return this;
    }
    /**
     * Returns detailed status for each member account within an organization for a given organization conformance pack.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_GetOrganizationConformancePackDetailedStatus.html
     */
    toGetOrganizationConformancePackDetailedStatus() {
        this.add('config:GetOrganizationConformancePackDetailedStatus');
        return this;
    }
    /**
     * Returns a list of configuration items for the specified resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_GetResourceConfigHistory.html
     */
    toGetResourceConfigHistory() {
        this.add('config:GetResourceConfigHistory');
        return this;
    }
    /**
     * Accepts a resource type and returns a list of resource identifiers that are aggregated for a specific resource type across accounts and regions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_ListAggregateDiscoveredResources.html
     */
    toListAggregateDiscoveredResources() {
        this.add('config:ListAggregateDiscoveredResources');
        return this;
    }
    /**
     * Accepts a resource type and returns a list of resource identifiers for the resources of that type
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_ListDiscoveredResources.html
     */
    toListDiscoveredResources() {
        this.add('config:ListDiscoveredResources');
        return this;
    }
    /**
     * List the tags for AWS Config resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.add('config:ListTagsForResource');
        return this;
    }
    /**
     * Authorizes the aggregator account and region to collect data from the source account and region
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_PutAggregationAuthorization.html
     */
    toPutAggregationAuthorization() {
        this.add('config:PutAggregationAuthorization');
        return this;
    }
    /**
     * Adds or updates an AWS Config rule for evaluating whether your AWS resources comply with your desired configurations
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_PutConfigRule.html
     */
    toPutConfigRule() {
        this.add('config:PutConfigRule');
        return this;
    }
    /**
     * Creates and updates the configuration aggregator with the selected source accounts and regions
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_PutConfigurationAggregator.html
     */
    toPutConfigurationAggregator() {
        this.add('config:PutConfigurationAggregator');
        return this;
    }
    /**
     * Creates a new configuration recorder to record the selected resource configurations
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_PutConfigurationRecorder.html
     */
    toPutConfigurationRecorder() {
        this.add('config:PutConfigurationRecorder');
        return this;
    }
    /**
     * Creates or updates a conformance pack.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_PutConformancePack.html
     */
    toPutConformancePack() {
        this.add('config:PutConformancePack');
        return this;
    }
    /**
     * Creates a delivery channel object to deliver configuration information to an Amazon S3 bucket and Amazon SNS topic
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_PutDeliveryChannel.html
     */
    toPutDeliveryChannel() {
        this.add('config:PutDeliveryChannel');
        return this;
    }
    /**
     * Used by an AWS Lambda function to deliver evaluation results to AWS Config
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_PutEvaluations.html
     */
    toPutEvaluations() {
        this.add('config:PutEvaluations');
        return this;
    }
    /**
     * Adds or updates organization config rule for your entire organization evaluating whether your AWS resources comply with your desired configurations.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_PutOrganizationConfigRule.html
     */
    toPutOrganizationConfigRule() {
        this.add('config:PutOrganizationConfigRule');
        return this;
    }
    /**
     * Adds or updates organization conformance pack for your entire organization evaluating whether your AWS resources comply with your desired configurations.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_PutOrganizationConformancePack.html
     */
    toPutOrganizationConformancePack() {
        this.add('config:PutOrganizationConformancePack');
        return this;
    }
    /**
     * Adds or updates the remediation configuration with a specific AWS Config rule with the selected target or action
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_PutRemediationConfigurations.html
     */
    toPutRemediationConfigurations() {
        this.add('config:PutRemediationConfigurations');
        return this;
    }
    /**
     * Adds or updates remediation exceptions for specific resources for a specific AWS Config rule.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_PutRemediationExceptions.html
     */
    toPutRemediationExceptions() {
        this.add('config:PutRemediationExceptions');
        return this;
    }
    /**
     * Creates and updates the retention configuration with details about retention period (number of days) that AWS Config stores your historical information
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_PutRetentionConfiguration.html
     */
    toPutRetentionConfiguration() {
        this.add('config:PutRetentionConfiguration');
        return this;
    }
    /**
     * Accepts a structured query language (SQL) SELECT command and an aggregator to query configuration state of AWS resources across multiple accounts and regions, performs the corresponding search, and returns resource configurations matching the properties.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_SelectAggregateResourceConfig.html
     */
    toSelectAggregateResourceConfig() {
        this.add('config:SelectAggregateResourceConfig');
        return this;
    }
    /**
     * Accepts a structured query language (SQL) SELECT command, performs the corresponding search, and returns resource configurations matching the properties
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_SelectResourceConfig.html
     */
    toSelectResourceConfig() {
        this.add('config:SelectResourceConfig');
        return this;
    }
    /**
     * Evaluates your resources against the specified Config rules
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_StartConfigRulesEvaluation.html
     */
    toStartConfigRulesEvaluation() {
        this.add('config:StartConfigRulesEvaluation');
        return this;
    }
    /**
     * Starts recording configurations of the AWS resources you have selected to record in your AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_StartConfigurationRecorder.html
     */
    toStartConfigurationRecorder() {
        this.add('config:StartConfigurationRecorder');
        return this;
    }
    /**
     * Runs an on-demand remediation for the specified AWS Config rules against the last known remediation configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_StartRemediationExecution.html
     */
    toStartRemediationExecution() {
        this.add('config:StartRemediationExecution');
        return this;
    }
    /**
     * Stops recording configurations of the AWS resources you have selected to record in your AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_StopConfigurationRecorder.html
     */
    toStopConfigurationRecorder() {
        this.add('config:StopConfigurationRecorder');
        return this;
    }
    /**
     * Associates the specified tags to a resource with the specified resourceArn
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        this.add('config:TagResource');
        return this;
    }
    /**
     * Deletes specified tags from a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/config/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        this.add('config:UntagResource');
        return this;
    }
    /**
     * Adds a resource of type AggregationAuthorization to the statement
     *
     * https://docs.aws.amazon.com/config/latest/developerguide/config-concepts.htmlAPI_AggregationAuthorization.html
     *
     * @param aggregatorAccount - Identifier for the aggregatorAccount.
     * @param aggregatorRegion - Identifier for the aggregatorRegion.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAggregationAuthorization(aggregatorAccount, aggregatorRegion, account, region, partition) {
        var arn = 'arn:${Partition}:config:${Region}:${Account}:aggregation-authorization/${AggregatorAccount}/${AggregatorRegion}';
        arn = arn.replace('${AggregatorAccount}', aggregatorAccount);
        arn = arn.replace('${AggregatorRegion}', aggregatorRegion);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type ConfigurationAggregator to the statement
     *
     * https://docs.aws.amazon.com/config/latest/developerguide/config-concepts.htmlAPI_ConfigurationAggregator.html
     *
     * @param aggregatorId - Identifier for the aggregatorId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onConfigurationAggregator(aggregatorId, account, region, partition) {
        var arn = 'arn:${Partition}:config:${Region}:${Account}:config-aggregator/${AggregatorId}';
        arn = arn.replace('${AggregatorId}', aggregatorId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type ConfigRule to the statement
     *
     * https://docs.aws.amazon.com/config/latest/developerguide/config-concepts.htmlAPI_ConfigRule.html
     *
     * @param configRuleId - Identifier for the configRuleId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onConfigRule(configRuleId, account, region, partition) {
        var arn = 'arn:${Partition}:config:${Region}:${Account}:config-rule/${ConfigRuleId}';
        arn = arn.replace('${ConfigRuleId}', configRuleId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type ConformancePack to the statement
     *
     * https://docs.aws.amazon.com/config/latest/developerguide/config-concepts.htmlAPI_ConformancePack.html
     *
     * @param conformancePackName - Identifier for the conformancePackName.
     * @param conformancePackId - Identifier for the conformancePackId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onConformancePack(conformancePackName, conformancePackId, account, region, partition) {
        var arn = 'arn:${Partition}:config:${Region}:${Account}:conformance-pack/${ConformancePackName}/${ConformancePackId}';
        arn = arn.replace('${ConformancePackName}', conformancePackName);
        arn = arn.replace('${ConformancePackId}', conformancePackId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type OrganizationConfigRule to the statement
     *
     * https://docs.aws.amazon.com/config/latest/developerguide/config-concepts.htmlAPI_OrganizationConfigRule.html
     *
     * @param organizationConfigRuleId - Identifier for the organizationConfigRuleId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onOrganizationConfigRule(organizationConfigRuleId, account, region, partition) {
        var arn = 'arn:${Partition}:config:${Region}:${Account}:organization-config-rule/${OrganizationConfigRuleId}';
        arn = arn.replace('${OrganizationConfigRuleId}', organizationConfigRuleId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type OrganizationConformancePack to the statement
     *
     * https://docs.aws.amazon.com/config/latest/developerguide/config-concepts.htmlAPI_OrganizationConformancePack.html
     *
     * @param organizationConformancePackId - Identifier for the organizationConformancePackId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onOrganizationConformancePack(organizationConformancePackId, account, region, partition) {
        var arn = 'arn:${Partition}:config:${Region}:${Account}:organization-conformance-pack/${OrganizationConformancePackId}';
        arn = arn.replace('${OrganizationConformancePackId}', organizationConformancePackId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type RemediationConfiguration to the statement
     *
     * https://docs.aws.amazon.com/config/latest/developerguide/config-concepts.htmlAPI_RemediationConfiguration.html
     *
     * @param remediationConfigurationId - Identifier for the remediationConfigurationId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onRemediationConfiguration(remediationConfigurationId, account, region, partition) {
        var arn = 'arn:${Partition}:config:${Region}:${Account}:remediation-configuration/${RemediationConfigurationId}';
        arn = arn.replace('${RemediationConfigurationId}', remediationConfigurationId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Config = Config;
//# sourceMappingURL=data:application/json;base64,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