"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Dms = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [dms](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsdatabasemigrationservice.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Dms extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [dms](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsdatabasemigrationservice.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'dms';
        this.actionList = {
            "AddTagsToResource": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_AddTagsToResource.html",
                "description": "Adds metadata tags to a DMS resource, including replication instance, endpoint, security group, and migration task",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "Certificate": {
                        "required": false
                    },
                    "Endpoint": {
                        "required": false
                    },
                    "EventSubscription": {
                        "required": false
                    },
                    "ReplicationInstance": {
                        "required": false
                    },
                    "ReplicationSubnetGroup": {
                        "required": false
                    },
                    "ReplicationTask": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "dms:req-tag/${TagKey}"
                ]
            },
            "ApplyPendingMaintenanceAction": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_ApplyPendingMaintenanceAction.html",
                "description": "Applies a pending maintenance action to a resource (for example, to a replication instance).",
                "accessLevel": "Write",
                "resourceTypes": {
                    "ReplicationInstance": {
                        "required": true
                    }
                }
            },
            "CreateEndpoint": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_CreateEndpoint.html",
                "description": "Creates an endpoint using the provided settings",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "dms:req-tag/${TagKey}"
                ]
            },
            "CreateEventSubscription": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_CreateEventSubscription.html",
                "description": "Creates an AWS DMS event notification subscription.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "dms:req-tag/${TagKey}"
                ]
            },
            "CreateReplicationInstance": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_CreateReplicationInstance.html",
                "description": "Creates the replication instance using the specified parameters",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "dms:req-tag/${TagKey}"
                ]
            },
            "CreateReplicationSubnetGroup": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_CreateReplicationSubnetGroup.html",
                "description": "Creates a replication subnet group given a list of the subnet IDs in a VPC",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "dms:req-tag/${TagKey}"
                ]
            },
            "CreateReplicationTask": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_CreateReplicationTask.html",
                "description": "Creates a replication task using the specified parameters",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Endpoint": {
                        "required": true
                    },
                    "ReplicationInstance": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "dms:req-tag/${TagKey}"
                ]
            },
            "DeleteCertificate": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_DeleteCertificate.html",
                "description": "Deletes the specified certificate",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Certificate": {
                        "required": true
                    }
                }
            },
            "DeleteEndpoint": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_DeleteEndpoint.html",
                "description": "Deletes the specified endpoint",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Endpoint": {
                        "required": true
                    }
                }
            },
            "DeleteEventSubscription": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_DeleteEventSubscription.html",
                "description": "Deletes an AWS DMS event subscription.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "EventSubscription": {
                        "required": true
                    }
                }
            },
            "DeleteReplicationInstance": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_DeleteReplicationInstance.html",
                "description": "Deletes the specified replication instance",
                "accessLevel": "Write",
                "resourceTypes": {
                    "ReplicationInstance": {
                        "required": true
                    }
                }
            },
            "DeleteReplicationSubnetGroup": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_DeleteReplicationSubnetGroup.html",
                "description": "Deletes a subnet group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "ReplicationSubnetGroup": {
                        "required": true
                    }
                }
            },
            "DeleteReplicationTask": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_DeleteReplicationTask.html",
                "description": "Deletes the specified replication task",
                "accessLevel": "Write",
                "resourceTypes": {
                    "ReplicationTask": {
                        "required": true
                    }
                }
            },
            "DescribeAccountAttributes": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeAccountAttributes.html",
                "description": "Lists all of the AWS DMS attributes for a customer account",
                "accessLevel": "Read"
            },
            "DescribeCertificates": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeCertificates.html",
                "description": "Provides a description of the certificate.",
                "accessLevel": "Read"
            },
            "DescribeConnections": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeConnections.html",
                "description": "Describes the status of the connections that have been made between the replication instance and an endpoint",
                "accessLevel": "Read"
            },
            "DescribeEndpointTypes": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeEndpointTypes.html",
                "description": "Returns information about the type of endpoints available",
                "accessLevel": "Read"
            },
            "DescribeEndpoints": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeEndpoints.html",
                "description": "Returns information about the endpoints for your account in the current region",
                "accessLevel": "Read"
            },
            "DescribeEventCategories": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeEventCategories.html",
                "description": "Lists categories for all event source types, or, if specified, for a specified source type.",
                "accessLevel": "Read"
            },
            "DescribeEventSubscriptions": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeEventSubscriptions.html",
                "description": "Lists all the event subscriptions for a customer account.",
                "accessLevel": "Read"
            },
            "DescribeEvents": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeEvents.html",
                "description": "Lists events for a given source identifier and source type.",
                "accessLevel": "Read"
            },
            "DescribeOrderableReplicationInstances": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeOrderableReplicationInstances.html",
                "description": "Returns information about the replication instance types that can be created in the specified region",
                "accessLevel": "Read"
            },
            "DescribeRefreshSchemasStatus": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeRefreshSchemasStatus.html",
                "description": "Returns the status of the RefreshSchemas operation",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Endpoint": {
                        "required": true
                    }
                }
            },
            "DescribeReplicationInstanceTaskLogs": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeReplicationInstanceTaskLogs.html",
                "description": "Returns information about the task logs for the specified task.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "ReplicationInstance": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "DescribeReplicationInstances": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeReplicationInstances.html",
                "description": "Returns information about replication instances for your account in the current region",
                "accessLevel": "Read"
            },
            "DescribeReplicationSubnetGroups": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeReplicationSubnetGroups.html",
                "description": "Returns information about the replication subnet groups",
                "accessLevel": "Read"
            },
            "DescribeReplicationTaskAssessmentResults": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeReplicationTaskAssessmentResults.html",
                "description": "Returns the task assessment results from Amazon S3. This action always returns the latest results.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "ReplicationTask": {
                        "required": false
                    }
                }
            },
            "DescribeReplicationTasks": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeReplicationTasks.html",
                "description": "Returns information about replication tasks for your account in the current region",
                "accessLevel": "Read"
            },
            "DescribeSchemas": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeSchemas.html",
                "description": "Returns information about the schema for the specified endpoint",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Endpoint": {
                        "required": true
                    }
                }
            },
            "DescribeTableStatistics": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeTableStatistics.html",
                "description": "Returns table statistics on the database migration task, including table name, rows inserted, rows updated, and rows deleted",
                "accessLevel": "Read",
                "resourceTypes": {
                    "ReplicationTask": {
                        "required": true
                    }
                }
            },
            "ImportCertificate": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_ImportCertificate.html",
                "description": "Uploads the specified certificate.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_ListTagsForResource.html",
                "description": "Lists all tags for an AWS DMS resource",
                "accessLevel": "List",
                "resourceTypes": {
                    "Certificate": {
                        "required": false
                    },
                    "Endpoint": {
                        "required": false
                    },
                    "EventSubscription": {
                        "required": false
                    },
                    "ReplicationInstance": {
                        "required": false
                    },
                    "ReplicationSubnetGroup": {
                        "required": false
                    },
                    "ReplicationTask": {
                        "required": false
                    }
                }
            },
            "ModifyEndpoint": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_ModifyEndpoint.html",
                "description": "Modifies the specified endpoint",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Endpoint": {
                        "required": true
                    },
                    "Certificate": {
                        "required": false
                    }
                }
            },
            "ModifyEventSubscription": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_ModifyEventSubscription.html",
                "description": "Modifies an existing AWS DMS event notification subscription.",
                "accessLevel": "Write"
            },
            "ModifyReplicationInstance": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_ModifyReplicationInstance.html",
                "description": "Modifies the replication instance to apply new settings",
                "accessLevel": "Write",
                "resourceTypes": {
                    "ReplicationInstance": {
                        "required": true
                    }
                }
            },
            "ModifyReplicationSubnetGroup": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_ModifyReplicationSubnetGroup.html",
                "description": "Modifies the settings for the specified replication subnet group",
                "accessLevel": "Write"
            },
            "ModifyReplicationTask": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_ModifyReplicationTask.html",
                "description": "Modifies the specified replication task.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "ReplicationTask": {
                        "required": true
                    }
                }
            },
            "RebootReplicationInstance": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_RebootReplicationInstance.html",
                "description": "Reboots a replication instance. Rebooting results in a momentary outage, until the replication instance becomes available again.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "ReplicationInstance": {
                        "required": true
                    }
                }
            },
            "RefreshSchemas": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_RefreshSchemas.html",
                "description": "Populates the schema for the specified endpoint",
                "accessLevel": "Write",
                "resourceTypes": {
                    "Endpoint": {
                        "required": true
                    },
                    "ReplicationInstance": {
                        "required": true
                    }
                }
            },
            "ReloadTables": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_ReloadTables.html",
                "description": "Reloads the target database table with the source data.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "ReplicationTask": {
                        "required": true
                    }
                }
            },
            "RemoveTagsFromResource": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_RemoveTagsFromResource.html",
                "description": "Removes metadata tags from a DMS resource",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "Certificate": {
                        "required": false
                    },
                    "Endpoint": {
                        "required": false
                    },
                    "EventSubscription": {
                        "required": false
                    },
                    "ReplicationInstance": {
                        "required": false
                    },
                    "ReplicationSubnetGroup": {
                        "required": false
                    },
                    "ReplicationTask": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "StartReplicationTask": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_StartReplicationTask.html",
                "description": "Starts the replication task",
                "accessLevel": "Write",
                "resourceTypes": {
                    "ReplicationTask": {
                        "required": true
                    }
                }
            },
            "StartReplicationTaskAssessment": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_StartReplicationTaskAssessment.html",
                "description": "Starts the replication task assessment for unsupported data types in the source database.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "ReplicationTask": {
                        "required": true
                    }
                }
            },
            "StopReplicationTask": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_StopReplicationTask.html",
                "description": "Stops the replication task",
                "accessLevel": "Write",
                "resourceTypes": {
                    "ReplicationTask": {
                        "required": true
                    }
                }
            },
            "TestConnection": {
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_TestConnection.html",
                "description": "Tests the connection between the replication instance and the endpoint",
                "accessLevel": "Read",
                "resourceTypes": {
                    "Endpoint": {
                        "required": true
                    },
                    "ReplicationInstance": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "ReplicationInstance": {
                "name": "ReplicationInstance",
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_ReplicationInstance.html",
                "arn": "arn:${Partition}:dms:${Region}:${Account}:rep:${ResourceName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "dms:rep-tag/${TagKey}"
                ]
            },
            "ReplicationTask": {
                "name": "ReplicationTask",
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_ReplicationTask.html",
                "arn": "arn:${Partition}:dms:${Region}:${Account}:task:${ResourceName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "dms:task-tag/${TagKey}"
                ]
            },
            "Endpoint": {
                "name": "Endpoint",
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_Endpoint.html",
                "arn": "arn:${Partition}:dms:${Region}:${Account}:endpoint:${ResourceName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "dms:endpoint-tag/${TagKey}"
                ]
            },
            "Certificate": {
                "name": "Certificate",
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_Certificate.html",
                "arn": "arn:${Partition}:dms:${Region}:${Account}:cert:${ResourceName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "dms:cert-tag/${TagKey}"
                ]
            },
            "EventSubscription": {
                "name": "EventSubscription",
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_EventSubscription.html",
                "arn": "arn:${Partition}:dms:${Region}:${Account}:es:${ResourceName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "dms:es-tag/${TagKey}"
                ]
            },
            "ReplicationSubnetGroup": {
                "name": "ReplicationSubnetGroup",
                "url": "https://docs.aws.amazon.com/dms/latest/APIReference/API_ReplicationSubnetGroup.html",
                "arn": "arn:${Partition}:dms:${Region}:${Account}:subgrp:${ResourceName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}",
                    "dms:subgrp-tag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Adds metadata tags to a DMS resource, including replication instance, endpoint, security group, and migration task
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_AddTagsToResource.html
     */
    toAddTagsToResource() {
        this.add('dms:AddTagsToResource');
        return this;
    }
    /**
     * Applies a pending maintenance action to a resource (for example, to a replication instance).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_ApplyPendingMaintenanceAction.html
     */
    toApplyPendingMaintenanceAction() {
        this.add('dms:ApplyPendingMaintenanceAction');
        return this;
    }
    /**
     * Creates an endpoint using the provided settings
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_CreateEndpoint.html
     */
    toCreateEndpoint() {
        this.add('dms:CreateEndpoint');
        return this;
    }
    /**
     * Creates an AWS DMS event notification subscription.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_CreateEventSubscription.html
     */
    toCreateEventSubscription() {
        this.add('dms:CreateEventSubscription');
        return this;
    }
    /**
     * Creates the replication instance using the specified parameters
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_CreateReplicationInstance.html
     */
    toCreateReplicationInstance() {
        this.add('dms:CreateReplicationInstance');
        return this;
    }
    /**
     * Creates a replication subnet group given a list of the subnet IDs in a VPC
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_CreateReplicationSubnetGroup.html
     */
    toCreateReplicationSubnetGroup() {
        this.add('dms:CreateReplicationSubnetGroup');
        return this;
    }
    /**
     * Creates a replication task using the specified parameters
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_CreateReplicationTask.html
     */
    toCreateReplicationTask() {
        this.add('dms:CreateReplicationTask');
        return this;
    }
    /**
     * Deletes the specified certificate
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DeleteCertificate.html
     */
    toDeleteCertificate() {
        this.add('dms:DeleteCertificate');
        return this;
    }
    /**
     * Deletes the specified endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DeleteEndpoint.html
     */
    toDeleteEndpoint() {
        this.add('dms:DeleteEndpoint');
        return this;
    }
    /**
     * Deletes an AWS DMS event subscription.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DeleteEventSubscription.html
     */
    toDeleteEventSubscription() {
        this.add('dms:DeleteEventSubscription');
        return this;
    }
    /**
     * Deletes the specified replication instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DeleteReplicationInstance.html
     */
    toDeleteReplicationInstance() {
        this.add('dms:DeleteReplicationInstance');
        return this;
    }
    /**
     * Deletes a subnet group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DeleteReplicationSubnetGroup.html
     */
    toDeleteReplicationSubnetGroup() {
        this.add('dms:DeleteReplicationSubnetGroup');
        return this;
    }
    /**
     * Deletes the specified replication task
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DeleteReplicationTask.html
     */
    toDeleteReplicationTask() {
        this.add('dms:DeleteReplicationTask');
        return this;
    }
    /**
     * Lists all of the AWS DMS attributes for a customer account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeAccountAttributes.html
     */
    toDescribeAccountAttributes() {
        this.add('dms:DescribeAccountAttributes');
        return this;
    }
    /**
     * Provides a description of the certificate.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeCertificates.html
     */
    toDescribeCertificates() {
        this.add('dms:DescribeCertificates');
        return this;
    }
    /**
     * Describes the status of the connections that have been made between the replication instance and an endpoint
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeConnections.html
     */
    toDescribeConnections() {
        this.add('dms:DescribeConnections');
        return this;
    }
    /**
     * Returns information about the type of endpoints available
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeEndpointTypes.html
     */
    toDescribeEndpointTypes() {
        this.add('dms:DescribeEndpointTypes');
        return this;
    }
    /**
     * Returns information about the endpoints for your account in the current region
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeEndpoints.html
     */
    toDescribeEndpoints() {
        this.add('dms:DescribeEndpoints');
        return this;
    }
    /**
     * Lists categories for all event source types, or, if specified, for a specified source type.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeEventCategories.html
     */
    toDescribeEventCategories() {
        this.add('dms:DescribeEventCategories');
        return this;
    }
    /**
     * Lists all the event subscriptions for a customer account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeEventSubscriptions.html
     */
    toDescribeEventSubscriptions() {
        this.add('dms:DescribeEventSubscriptions');
        return this;
    }
    /**
     * Lists events for a given source identifier and source type.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeEvents.html
     */
    toDescribeEvents() {
        this.add('dms:DescribeEvents');
        return this;
    }
    /**
     * Returns information about the replication instance types that can be created in the specified region
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeOrderableReplicationInstances.html
     */
    toDescribeOrderableReplicationInstances() {
        this.add('dms:DescribeOrderableReplicationInstances');
        return this;
    }
    /**
     * Returns the status of the RefreshSchemas operation
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeRefreshSchemasStatus.html
     */
    toDescribeRefreshSchemasStatus() {
        this.add('dms:DescribeRefreshSchemasStatus');
        return this;
    }
    /**
     * Returns information about the task logs for the specified task.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeReplicationInstanceTaskLogs.html
     */
    toDescribeReplicationInstanceTaskLogs() {
        this.add('dms:DescribeReplicationInstanceTaskLogs');
        return this;
    }
    /**
     * Returns information about replication instances for your account in the current region
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeReplicationInstances.html
     */
    toDescribeReplicationInstances() {
        this.add('dms:DescribeReplicationInstances');
        return this;
    }
    /**
     * Returns information about the replication subnet groups
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeReplicationSubnetGroups.html
     */
    toDescribeReplicationSubnetGroups() {
        this.add('dms:DescribeReplicationSubnetGroups');
        return this;
    }
    /**
     * Returns the task assessment results from Amazon S3. This action always returns the latest results.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeReplicationTaskAssessmentResults.html
     */
    toDescribeReplicationTaskAssessmentResults() {
        this.add('dms:DescribeReplicationTaskAssessmentResults');
        return this;
    }
    /**
     * Returns information about replication tasks for your account in the current region
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeReplicationTasks.html
     */
    toDescribeReplicationTasks() {
        this.add('dms:DescribeReplicationTasks');
        return this;
    }
    /**
     * Returns information about the schema for the specified endpoint
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeSchemas.html
     */
    toDescribeSchemas() {
        this.add('dms:DescribeSchemas');
        return this;
    }
    /**
     * Returns table statistics on the database migration task, including table name, rows inserted, rows updated, and rows deleted
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeTableStatistics.html
     */
    toDescribeTableStatistics() {
        this.add('dms:DescribeTableStatistics');
        return this;
    }
    /**
     * Uploads the specified certificate.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_ImportCertificate.html
     */
    toImportCertificate() {
        this.add('dms:ImportCertificate');
        return this;
    }
    /**
     * Lists all tags for an AWS DMS resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.add('dms:ListTagsForResource');
        return this;
    }
    /**
     * Modifies the specified endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_ModifyEndpoint.html
     */
    toModifyEndpoint() {
        this.add('dms:ModifyEndpoint');
        return this;
    }
    /**
     * Modifies an existing AWS DMS event notification subscription.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_ModifyEventSubscription.html
     */
    toModifyEventSubscription() {
        this.add('dms:ModifyEventSubscription');
        return this;
    }
    /**
     * Modifies the replication instance to apply new settings
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_ModifyReplicationInstance.html
     */
    toModifyReplicationInstance() {
        this.add('dms:ModifyReplicationInstance');
        return this;
    }
    /**
     * Modifies the settings for the specified replication subnet group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_ModifyReplicationSubnetGroup.html
     */
    toModifyReplicationSubnetGroup() {
        this.add('dms:ModifyReplicationSubnetGroup');
        return this;
    }
    /**
     * Modifies the specified replication task.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_ModifyReplicationTask.html
     */
    toModifyReplicationTask() {
        this.add('dms:ModifyReplicationTask');
        return this;
    }
    /**
     * Reboots a replication instance. Rebooting results in a momentary outage, until the replication instance becomes available again.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_RebootReplicationInstance.html
     */
    toRebootReplicationInstance() {
        this.add('dms:RebootReplicationInstance');
        return this;
    }
    /**
     * Populates the schema for the specified endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_RefreshSchemas.html
     */
    toRefreshSchemas() {
        this.add('dms:RefreshSchemas');
        return this;
    }
    /**
     * Reloads the target database table with the source data.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_ReloadTables.html
     */
    toReloadTables() {
        this.add('dms:ReloadTables');
        return this;
    }
    /**
     * Removes metadata tags from a DMS resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_RemoveTagsFromResource.html
     */
    toRemoveTagsFromResource() {
        this.add('dms:RemoveTagsFromResource');
        return this;
    }
    /**
     * Starts the replication task
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_StartReplicationTask.html
     */
    toStartReplicationTask() {
        this.add('dms:StartReplicationTask');
        return this;
    }
    /**
     * Starts the replication task assessment for unsupported data types in the source database.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_StartReplicationTaskAssessment.html
     */
    toStartReplicationTaskAssessment() {
        this.add('dms:StartReplicationTaskAssessment');
        return this;
    }
    /**
     * Stops the replication task
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_StopReplicationTask.html
     */
    toStopReplicationTask() {
        this.add('dms:StopReplicationTask');
        return this;
    }
    /**
     * Tests the connection between the replication instance and the endpoint
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_TestConnection.html
     */
    toTestConnection() {
        this.add('dms:TestConnection');
        return this;
    }
    /**
     * Adds a resource of type ReplicationInstance to the statement
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_ReplicationInstance.html
     *
     * @param resourceName - Identifier for the resourceName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifRepTag()
     */
    onReplicationInstance(resourceName, account, region, partition) {
        var arn = 'arn:${Partition}:dms:${Region}:${Account}:rep:${ResourceName}';
        arn = arn.replace('${ResourceName}', resourceName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type ReplicationTask to the statement
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_ReplicationTask.html
     *
     * @param resourceName - Identifier for the resourceName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifTaskTag()
     */
    onReplicationTask(resourceName, account, region, partition) {
        var arn = 'arn:${Partition}:dms:${Region}:${Account}:task:${ResourceName}';
        arn = arn.replace('${ResourceName}', resourceName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type Endpoint to the statement
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_Endpoint.html
     *
     * @param resourceName - Identifier for the resourceName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifEndpointTag()
     */
    onEndpoint(resourceName, account, region, partition) {
        var arn = 'arn:${Partition}:dms:${Region}:${Account}:endpoint:${ResourceName}';
        arn = arn.replace('${ResourceName}', resourceName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type Certificate to the statement
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_Certificate.html
     *
     * @param resourceName - Identifier for the resourceName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifCertTag()
     */
    onCertificate(resourceName, account, region, partition) {
        var arn = 'arn:${Partition}:dms:${Region}:${Account}:cert:${ResourceName}';
        arn = arn.replace('${ResourceName}', resourceName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type EventSubscription to the statement
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_EventSubscription.html
     *
     * @param resourceName - Identifier for the resourceName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifEsTag()
     */
    onEventSubscription(resourceName, account, region, partition) {
        var arn = 'arn:${Partition}:dms:${Region}:${Account}:es:${ResourceName}';
        arn = arn.replace('${ResourceName}', resourceName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type ReplicationSubnetGroup to the statement
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_ReplicationSubnetGroup.html
     *
     * @param resourceName - Identifier for the resourceName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifSubgrpTag()
     */
    onReplicationSubnetGroup(resourceName, account, region, partition) {
        var arn = 'arn:${Partition}:dms:${Region}:${Account}:subgrp:${ResourceName}';
        arn = arn.replace('${ResourceName}', resourceName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters actions based on the presence of tag keys in the request for Certificate
     *
     * Applies to resource types:
     * - Certificate
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifCertTag(tagKey, value, operator) {
        return this.if(`dms:cert-tag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the presence of tag keys in the request for Endpoint
     *
     * Applies to resource types:
     * - Endpoint
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEndpointTag(tagKey, value, operator) {
        return this.if(`dms:endpoint-tag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the presence of tag keys in the request for EventSubscription
     *
     * Applies to resource types:
     * - EventSubscription
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEsTag(tagKey, value, operator) {
        return this.if(`dms:es-tag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the presence of tag keys in the request for ReplicationInstance
     *
     * Applies to resource types:
     * - ReplicationInstance
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRepTag(tagKey, value, operator) {
        return this.if(`dms:rep-tag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the presence of tag key-value pairs in the request
     *
     * Applies to actions:
     * - .toAddTagsToResource()
     * - .toCreateEndpoint()
     * - .toCreateEventSubscription()
     * - .toCreateReplicationInstance()
     * - .toCreateReplicationSubnetGroup()
     * - .toCreateReplicationTask()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifReqTag(tagKey, value, operator) {
        return this.if(`dms:req-tag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the presence of tag keys in the request for ReplicationSubnetGroup
     *
     * Applies to resource types:
     * - ReplicationSubnetGroup
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSubgrpTag(tagKey, value, operator) {
        return this.if(`dms:subgrp-tag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters actions based on the presence of tag keys in the request for ReplicationTask
     *
     * Applies to resource types:
     * - ReplicationTask
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTaskTag(tagKey, value, operator) {
        return this.if(`dms:task-tag/${tagKey}`, value, operator || 'StringLike');
    }
}
exports.Dms = Dms;
//# sourceMappingURL=data:application/json;base64,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