"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Directconnect = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [directconnect](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsdirectconnect.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Directconnect extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [directconnect](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsdirectconnect.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'directconnect';
        this.actionList = {
            "AcceptDirectConnectGatewayAssociationProposal": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_AcceptDirectConnectGatewayAssociationProposal.html",
                "description": "Accepts a proposal request to attach a virtual private gateway to a Direct Connect gateway.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dx-gateway": {
                        "required": true
                    }
                }
            },
            "AllocateConnectionOnInterconnect": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_AllocateConnectionOnInterconnect.html",
                "description": "Creates a hosted connection on an interconnect.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxcon": {
                        "required": true
                    }
                }
            },
            "AllocateHostedConnection": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_AllocateHostedConnection.html",
                "description": "Creates a new hosted connection between a AWS Direct Connect partner's network and a specific AWS Direct Connect location.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxcon": {
                        "required": false
                    },
                    "dxlag": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "AllocatePrivateVirtualInterface": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_AllocatePrivateVirtualInterface.html",
                "description": "Provisions a private virtual interface to be owned by a different customer.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxcon": {
                        "required": false
                    },
                    "dxlag": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "AllocatePublicVirtualInterface": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_AllocatePublicVirtualInterface.html",
                "description": "Provisions a public virtual interface to be owned by a different customer.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxcon": {
                        "required": false
                    },
                    "dxlag": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "AllocateTransitVirtualInterface": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_AllocateTransitVirtualInterface.html",
                "description": "Provisions a transit virtual interface to be owned by a different customer.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxcon": {
                        "required": false
                    },
                    "dxlag": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "AssociateConnectionWithLag": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_AssociateConnectionWithLag.html",
                "description": "Associates a connection with a LAG.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxcon": {
                        "required": true
                    },
                    "dxlag": {
                        "required": true
                    }
                }
            },
            "AssociateHostedConnection": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_AssociateHostedConnection.html",
                "description": "Associates a hosted connection and its virtual interfaces with a link aggregation group (LAG) or interconnect.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxcon": {
                        "required": false
                    },
                    "dxlag": {
                        "required": false
                    }
                }
            },
            "AssociateVirtualInterface": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_AssociateVirtualInterface.html",
                "description": "Associates a virtual interface with a specified link aggregation group (LAG) or connection.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxvif": {
                        "required": true
                    },
                    "dxcon": {
                        "required": false
                    },
                    "dxlag": {
                        "required": false
                    }
                }
            },
            "ConfirmConnection": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_ConfirmConnection.html",
                "description": "Confirm the creation of a hosted connection on an interconnect.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxcon": {
                        "required": true
                    }
                }
            },
            "ConfirmPrivateVirtualInterface": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_ConfirmPrivateVirtualInterface.html",
                "description": "Accept ownership of a private virtual interface created by another customer.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxvif": {
                        "required": true
                    }
                }
            },
            "ConfirmPublicVirtualInterface": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_ConfirmPublicVirtualInterface.html",
                "description": "Accept ownership of a public virtual interface created by another customer",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxvif": {
                        "required": true
                    }
                }
            },
            "ConfirmTransitVirtualInterface": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_ConfirmTransitVirtualInterface.html",
                "description": "Accept ownership of a transit virtual interface created by another customer",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxvif": {
                        "required": true
                    }
                }
            },
            "CreateBGPPeer": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_CreateBGPPeer.html",
                "description": "Creates a BGP peer on the specified virtual interface.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxvif": {
                        "required": true
                    }
                }
            },
            "CreateConnection": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_CreateConnection.html",
                "description": "Creates a new connection between the customer network and a specific AWS Direct Connect location.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxlag": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateDirectConnectGateway": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_CreateDirectConnectGateway.html",
                "description": "Creates a Direct Connect gateway, which is an intermediate object that enables you to connect a set of virtual interfaces and virtual private gateways.",
                "accessLevel": "Write"
            },
            "CreateDirectConnectGatewayAssociation": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_CreateDirectConnectGatewayAssociation.html",
                "description": "Creates an association between a Direct Connect gateway and a virtual private gateway.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dx-gateway": {
                        "required": true
                    }
                }
            },
            "CreateDirectConnectGatewayAssociationProposal": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_CreateDirectConnectGatewayAssociationProposal.html",
                "description": "Creates a proposal to associate the specified virtual private gateway with the specified Direct Connect gateway.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dx-gateway": {
                        "required": true
                    }
                }
            },
            "CreateInterconnect": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_CreateInterconnect.html",
                "description": "Creates a new interconnect between a AWS Direct Connect partner's network and a specific AWS Direct Connect location.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxlag": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateLag": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_CreateLag.html",
                "description": "Creates a link aggregation group (LAG) with the specified number of bundled physical connections between the customer network and a specific AWS Direct Connect location.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxcon": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreatePrivateVirtualInterface": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_CreatePrivateVirtualInterface.html",
                "description": "Creates a new private virtual interface.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxcon": {
                        "required": false
                    },
                    "dxlag": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreatePublicVirtualInterface": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_CreatePublicVirtualInterface.html",
                "description": "Creates a new public virtual interface.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxcon": {
                        "required": false
                    },
                    "dxlag": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateTransitVirtualInterface": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_CreateTransitVirtualInterface.html",
                "description": "Creates a new transit virtual interface.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxcon": {
                        "required": false
                    },
                    "dxlag": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "DeleteBGPPeer": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DeleteBGPPeer.html",
                "description": "Deletes the specified BGP peer on the specified virtual interface with the specified customer address and ASN.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxvif": {
                        "required": true
                    }
                }
            },
            "DeleteConnection": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DeleteConnection.html",
                "description": "Deletes the connection.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxcon": {
                        "required": true
                    }
                }
            },
            "DeleteDirectConnectGateway": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DeleteDirectConnectGateway.html",
                "description": "Deletes the specified Direct Connect gateway.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dx-gateway": {
                        "required": true
                    }
                }
            },
            "DeleteDirectConnectGatewayAssociation": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DeleteDirectConnectGatewayAssociation.html",
                "description": "Deletes the association between the specified Direct Connect gateway and virtual private gateway.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dx-gateway": {
                        "required": true
                    }
                }
            },
            "DeleteDirectConnectGatewayAssociationProposal": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DeleteDirectConnectGatewayAssociationProposal.html",
                "description": "Deletes the association proposal request between the specified Direct Connect gateway and virtual private gateway.",
                "accessLevel": "Write"
            },
            "DeleteInterconnect": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DeleteInterconnect.html",
                "description": "Deletes the specified interconnect.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxcon": {
                        "required": true
                    }
                }
            },
            "DeleteLag": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DeleteLag.html",
                "description": "Deletes the specified link aggregation group (LAG).",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxlag": {
                        "required": true
                    }
                }
            },
            "DeleteVirtualInterface": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DeleteVirtualInterface.html",
                "description": "Deletes a virtual interface.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxvif": {
                        "required": true
                    }
                }
            },
            "DescribeConnectionLoa": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeConnectionLoa.html",
                "description": "Returns the LOA-CFA for a Connection.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "dxcon": {
                        "required": true
                    }
                }
            },
            "DescribeConnections": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeConnections.html",
                "description": "Displays all connections in this region.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "dxcon": {
                        "required": false
                    }
                }
            },
            "DescribeConnectionsOnInterconnect": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeConnectionsOnInterconnect.html",
                "description": "Return a list of connections that have been provisioned on the given interconnect.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "dxcon": {
                        "required": true
                    }
                }
            },
            "DescribeDirectConnectGatewayAssociationProposals": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeDirectConnectGatewayAssociationProposals.html",
                "description": "Describes one or more association proposals for connection between a virtual private gateway and a Direct Connect gateway.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "dx-gateway": {
                        "required": false
                    }
                }
            },
            "DescribeDirectConnectGatewayAssociations": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeDirectConnectGatewayAssociations.html",
                "description": "Lists the associations between your Direct Connect gateways and virtual private gateways.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "dx-gateway": {
                        "required": false
                    }
                }
            },
            "DescribeDirectConnectGatewayAttachments": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeDirectConnectGatewayAttachments.html",
                "description": "Lists the attachments between your Direct Connect gateways and virtual interfaces.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "dx-gateway": {
                        "required": false
                    }
                }
            },
            "DescribeDirectConnectGateways": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeDirectConnectGateways.html",
                "description": "Lists all your Direct Connect gateways or only the specified Direct Connect gateway.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "dx-gateway": {
                        "required": false
                    }
                }
            },
            "DescribeHostedConnections": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeHostedConnections.html",
                "description": "Lists the hosted connections that have been provisioned on the specified interconnect or link aggregation group (LAG).",
                "accessLevel": "Read",
                "resourceTypes": {
                    "dxcon": {
                        "required": false
                    },
                    "dxlag": {
                        "required": false
                    }
                }
            },
            "DescribeInterconnectLoa": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeInterconnectLoa.html",
                "description": "Returns the LOA-CFA for an Interconnect.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "dxcon": {
                        "required": true
                    }
                }
            },
            "DescribeInterconnects": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeInterconnects.html",
                "description": "Returns a list of interconnects owned by the AWS account.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "dxcon": {
                        "required": false
                    }
                }
            },
            "DescribeLags": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeLags.html",
                "description": "Describes all your link aggregation groups (LAG) or the specified LAG.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "dxlag": {
                        "required": false
                    }
                }
            },
            "DescribeLoa": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeLoa.html",
                "description": "Gets the LOA-CFA for a connection, interconnect, or link aggregation group (LAG).",
                "accessLevel": "Read",
                "resourceTypes": {
                    "dxcon": {
                        "required": false
                    },
                    "dxlag": {
                        "required": false
                    }
                }
            },
            "DescribeLocations": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeLocations.html",
                "description": "Returns the list of AWS Direct Connect locations in the current AWS region.",
                "accessLevel": "List"
            },
            "DescribeTags": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeTags.html",
                "description": "Describes the tags associated with the specified AWS Direct Connect resources.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "dxcon": {
                        "required": false
                    },
                    "dxlag": {
                        "required": false
                    },
                    "dxvif": {
                        "required": false
                    }
                }
            },
            "DescribeVirtualGateways": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeVirtualGateways.html",
                "description": "Returns a list of virtual private gateways owned by the AWS account.",
                "accessLevel": "Read"
            },
            "DescribeVirtualInterfaces": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeVirtualInterfaces.html",
                "description": "Displays all virtual interfaces for an AWS account.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "dxcon": {
                        "required": false
                    },
                    "dxlag": {
                        "required": false
                    },
                    "dxvif": {
                        "required": false
                    }
                }
            },
            "DisassociateConnectionFromLag": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DisassociateConnectionFromLag.html",
                "description": "Disassociates a connection from a link aggregation group (LAG).",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxcon": {
                        "required": true
                    },
                    "dxlag": {
                        "required": true
                    }
                }
            },
            "ListVirtualInterfaceTestHistory": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_ListVirtualInterfaceTestHistory.html",
                "description": "Lists the virtual interface failover test history.",
                "accessLevel": "List",
                "resourceTypes": {
                    "dxvif": {
                        "required": true
                    }
                }
            },
            "StartBgpFailoverTest": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_StartBgpFailoverTest.html",
                "description": "Starts the virtual interface failover test that verifies your configuration meets your resiliency requirements by placing the BGP peering session in the DOWN state. You can then send traffic to verify that there are no outages.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxvif": {
                        "required": true
                    }
                }
            },
            "StopBgpFailoverTest": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_StopBgpFailoverTest.html",
                "description": "Stops the virtual interface failover test.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxvif": {
                        "required": true
                    }
                }
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_TagResource.html",
                "description": "Adds the specified tags to the specified AWS Direct Connect resource. Each resource can have a maximum of 50 tags.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "dxcon": {
                        "required": false
                    },
                    "dxlag": {
                        "required": false
                    },
                    "dxvif": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_UntagResource.html",
                "description": "Removes one or more tags from the specified AWS Direct Connect resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "dxcon": {
                        "required": false
                    },
                    "dxlag": {
                        "required": false
                    },
                    "dxvif": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "UpdateDirectConnectGatewayAssociation": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_UpdateDirectConnectGatewayAssociation.html",
                "description": "Updates the specified attributes of the Direct Connect gateway association.",
                "accessLevel": "Write"
            },
            "UpdateLag": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_UpdateLag.html",
                "description": "Updates the attributes of the specified link aggregation group (LAG).",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxlag": {
                        "required": true
                    }
                }
            },
            "UpdateVirtualInterfaceAttributes": {
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_UpdateVirtualInterfaceAttributes.html",
                "description": "Updates the specified attributes of the specified virtual private interface.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dxvif": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "dxcon": {
                "name": "dxcon",
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_Connection.html",
                "arn": "arn:${Partition}:directconnect:${Region}:${Account}:dxcon/${ConnectionId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "dxlag": {
                "name": "dxlag",
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_Lag.html",
                "arn": "arn:${Partition}:directconnect:${Region}:${Account}:dxlag/${LagId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "dxvif": {
                "name": "dxvif",
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_VirtualInterface.html",
                "arn": "arn:${Partition}:directconnect:${Region}:${Account}:dxvif/${VirtualInterfaceId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "dx-gateway": {
                "name": "dx-gateway",
                "url": "https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DirectConnectGateway.html",
                "arn": "arn:${Partition}:directconnect::${Account}:dx-gateway/${DirectConnectGatewayId}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Accepts a proposal request to attach a virtual private gateway to a Direct Connect gateway.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_AcceptDirectConnectGatewayAssociationProposal.html
     */
    toAcceptDirectConnectGatewayAssociationProposal() {
        this.add('directconnect:AcceptDirectConnectGatewayAssociationProposal');
        return this;
    }
    /**
     * Creates a hosted connection on an interconnect.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_AllocateConnectionOnInterconnect.html
     */
    toAllocateConnectionOnInterconnect() {
        this.add('directconnect:AllocateConnectionOnInterconnect');
        return this;
    }
    /**
     * Creates a new hosted connection between a AWS Direct Connect partner's network and a specific AWS Direct Connect location.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_AllocateHostedConnection.html
     */
    toAllocateHostedConnection() {
        this.add('directconnect:AllocateHostedConnection');
        return this;
    }
    /**
     * Provisions a private virtual interface to be owned by a different customer.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_AllocatePrivateVirtualInterface.html
     */
    toAllocatePrivateVirtualInterface() {
        this.add('directconnect:AllocatePrivateVirtualInterface');
        return this;
    }
    /**
     * Provisions a public virtual interface to be owned by a different customer.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_AllocatePublicVirtualInterface.html
     */
    toAllocatePublicVirtualInterface() {
        this.add('directconnect:AllocatePublicVirtualInterface');
        return this;
    }
    /**
     * Provisions a transit virtual interface to be owned by a different customer.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_AllocateTransitVirtualInterface.html
     */
    toAllocateTransitVirtualInterface() {
        this.add('directconnect:AllocateTransitVirtualInterface');
        return this;
    }
    /**
     * Associates a connection with a LAG.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_AssociateConnectionWithLag.html
     */
    toAssociateConnectionWithLag() {
        this.add('directconnect:AssociateConnectionWithLag');
        return this;
    }
    /**
     * Associates a hosted connection and its virtual interfaces with a link aggregation group (LAG) or interconnect.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_AssociateHostedConnection.html
     */
    toAssociateHostedConnection() {
        this.add('directconnect:AssociateHostedConnection');
        return this;
    }
    /**
     * Associates a virtual interface with a specified link aggregation group (LAG) or connection.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_AssociateVirtualInterface.html
     */
    toAssociateVirtualInterface() {
        this.add('directconnect:AssociateVirtualInterface');
        return this;
    }
    /**
     * Confirm the creation of a hosted connection on an interconnect.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_ConfirmConnection.html
     */
    toConfirmConnection() {
        this.add('directconnect:ConfirmConnection');
        return this;
    }
    /**
     * Accept ownership of a private virtual interface created by another customer.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_ConfirmPrivateVirtualInterface.html
     */
    toConfirmPrivateVirtualInterface() {
        this.add('directconnect:ConfirmPrivateVirtualInterface');
        return this;
    }
    /**
     * Accept ownership of a public virtual interface created by another customer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_ConfirmPublicVirtualInterface.html
     */
    toConfirmPublicVirtualInterface() {
        this.add('directconnect:ConfirmPublicVirtualInterface');
        return this;
    }
    /**
     * Accept ownership of a transit virtual interface created by another customer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_ConfirmTransitVirtualInterface.html
     */
    toConfirmTransitVirtualInterface() {
        this.add('directconnect:ConfirmTransitVirtualInterface');
        return this;
    }
    /**
     * Creates a BGP peer on the specified virtual interface.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_CreateBGPPeer.html
     */
    toCreateBGPPeer() {
        this.add('directconnect:CreateBGPPeer');
        return this;
    }
    /**
     * Creates a new connection between the customer network and a specific AWS Direct Connect location.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_CreateConnection.html
     */
    toCreateConnection() {
        this.add('directconnect:CreateConnection');
        return this;
    }
    /**
     * Creates a Direct Connect gateway, which is an intermediate object that enables you to connect a set of virtual interfaces and virtual private gateways.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_CreateDirectConnectGateway.html
     */
    toCreateDirectConnectGateway() {
        this.add('directconnect:CreateDirectConnectGateway');
        return this;
    }
    /**
     * Creates an association between a Direct Connect gateway and a virtual private gateway.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_CreateDirectConnectGatewayAssociation.html
     */
    toCreateDirectConnectGatewayAssociation() {
        this.add('directconnect:CreateDirectConnectGatewayAssociation');
        return this;
    }
    /**
     * Creates a proposal to associate the specified virtual private gateway with the specified Direct Connect gateway.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_CreateDirectConnectGatewayAssociationProposal.html
     */
    toCreateDirectConnectGatewayAssociationProposal() {
        this.add('directconnect:CreateDirectConnectGatewayAssociationProposal');
        return this;
    }
    /**
     * Creates a new interconnect between a AWS Direct Connect partner's network and a specific AWS Direct Connect location.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_CreateInterconnect.html
     */
    toCreateInterconnect() {
        this.add('directconnect:CreateInterconnect');
        return this;
    }
    /**
     * Creates a link aggregation group (LAG) with the specified number of bundled physical connections between the customer network and a specific AWS Direct Connect location.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_CreateLag.html
     */
    toCreateLag() {
        this.add('directconnect:CreateLag');
        return this;
    }
    /**
     * Creates a new private virtual interface.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_CreatePrivateVirtualInterface.html
     */
    toCreatePrivateVirtualInterface() {
        this.add('directconnect:CreatePrivateVirtualInterface');
        return this;
    }
    /**
     * Creates a new public virtual interface.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_CreatePublicVirtualInterface.html
     */
    toCreatePublicVirtualInterface() {
        this.add('directconnect:CreatePublicVirtualInterface');
        return this;
    }
    /**
     * Creates a new transit virtual interface.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_CreateTransitVirtualInterface.html
     */
    toCreateTransitVirtualInterface() {
        this.add('directconnect:CreateTransitVirtualInterface');
        return this;
    }
    /**
     * Deletes the specified BGP peer on the specified virtual interface with the specified customer address and ASN.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DeleteBGPPeer.html
     */
    toDeleteBGPPeer() {
        this.add('directconnect:DeleteBGPPeer');
        return this;
    }
    /**
     * Deletes the connection.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DeleteConnection.html
     */
    toDeleteConnection() {
        this.add('directconnect:DeleteConnection');
        return this;
    }
    /**
     * Deletes the specified Direct Connect gateway.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DeleteDirectConnectGateway.html
     */
    toDeleteDirectConnectGateway() {
        this.add('directconnect:DeleteDirectConnectGateway');
        return this;
    }
    /**
     * Deletes the association between the specified Direct Connect gateway and virtual private gateway.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DeleteDirectConnectGatewayAssociation.html
     */
    toDeleteDirectConnectGatewayAssociation() {
        this.add('directconnect:DeleteDirectConnectGatewayAssociation');
        return this;
    }
    /**
     * Deletes the association proposal request between the specified Direct Connect gateway and virtual private gateway.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DeleteDirectConnectGatewayAssociationProposal.html
     */
    toDeleteDirectConnectGatewayAssociationProposal() {
        this.add('directconnect:DeleteDirectConnectGatewayAssociationProposal');
        return this;
    }
    /**
     * Deletes the specified interconnect.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DeleteInterconnect.html
     */
    toDeleteInterconnect() {
        this.add('directconnect:DeleteInterconnect');
        return this;
    }
    /**
     * Deletes the specified link aggregation group (LAG).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DeleteLag.html
     */
    toDeleteLag() {
        this.add('directconnect:DeleteLag');
        return this;
    }
    /**
     * Deletes a virtual interface.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DeleteVirtualInterface.html
     */
    toDeleteVirtualInterface() {
        this.add('directconnect:DeleteVirtualInterface');
        return this;
    }
    /**
     * Returns the LOA-CFA for a Connection.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeConnectionLoa.html
     */
    toDescribeConnectionLoa() {
        this.add('directconnect:DescribeConnectionLoa');
        return this;
    }
    /**
     * Displays all connections in this region.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeConnections.html
     */
    toDescribeConnections() {
        this.add('directconnect:DescribeConnections');
        return this;
    }
    /**
     * Return a list of connections that have been provisioned on the given interconnect.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeConnectionsOnInterconnect.html
     */
    toDescribeConnectionsOnInterconnect() {
        this.add('directconnect:DescribeConnectionsOnInterconnect');
        return this;
    }
    /**
     * Describes one or more association proposals for connection between a virtual private gateway and a Direct Connect gateway.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeDirectConnectGatewayAssociationProposals.html
     */
    toDescribeDirectConnectGatewayAssociationProposals() {
        this.add('directconnect:DescribeDirectConnectGatewayAssociationProposals');
        return this;
    }
    /**
     * Lists the associations between your Direct Connect gateways and virtual private gateways.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeDirectConnectGatewayAssociations.html
     */
    toDescribeDirectConnectGatewayAssociations() {
        this.add('directconnect:DescribeDirectConnectGatewayAssociations');
        return this;
    }
    /**
     * Lists the attachments between your Direct Connect gateways and virtual interfaces.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeDirectConnectGatewayAttachments.html
     */
    toDescribeDirectConnectGatewayAttachments() {
        this.add('directconnect:DescribeDirectConnectGatewayAttachments');
        return this;
    }
    /**
     * Lists all your Direct Connect gateways or only the specified Direct Connect gateway.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeDirectConnectGateways.html
     */
    toDescribeDirectConnectGateways() {
        this.add('directconnect:DescribeDirectConnectGateways');
        return this;
    }
    /**
     * Lists the hosted connections that have been provisioned on the specified interconnect or link aggregation group (LAG).
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeHostedConnections.html
     */
    toDescribeHostedConnections() {
        this.add('directconnect:DescribeHostedConnections');
        return this;
    }
    /**
     * Returns the LOA-CFA for an Interconnect.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeInterconnectLoa.html
     */
    toDescribeInterconnectLoa() {
        this.add('directconnect:DescribeInterconnectLoa');
        return this;
    }
    /**
     * Returns a list of interconnects owned by the AWS account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeInterconnects.html
     */
    toDescribeInterconnects() {
        this.add('directconnect:DescribeInterconnects');
        return this;
    }
    /**
     * Describes all your link aggregation groups (LAG) or the specified LAG.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeLags.html
     */
    toDescribeLags() {
        this.add('directconnect:DescribeLags');
        return this;
    }
    /**
     * Gets the LOA-CFA for a connection, interconnect, or link aggregation group (LAG).
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeLoa.html
     */
    toDescribeLoa() {
        this.add('directconnect:DescribeLoa');
        return this;
    }
    /**
     * Returns the list of AWS Direct Connect locations in the current AWS region.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeLocations.html
     */
    toDescribeLocations() {
        this.add('directconnect:DescribeLocations');
        return this;
    }
    /**
     * Describes the tags associated with the specified AWS Direct Connect resources.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeTags.html
     */
    toDescribeTags() {
        this.add('directconnect:DescribeTags');
        return this;
    }
    /**
     * Returns a list of virtual private gateways owned by the AWS account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeVirtualGateways.html
     */
    toDescribeVirtualGateways() {
        this.add('directconnect:DescribeVirtualGateways');
        return this;
    }
    /**
     * Displays all virtual interfaces for an AWS account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DescribeVirtualInterfaces.html
     */
    toDescribeVirtualInterfaces() {
        this.add('directconnect:DescribeVirtualInterfaces');
        return this;
    }
    /**
     * Disassociates a connection from a link aggregation group (LAG).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DisassociateConnectionFromLag.html
     */
    toDisassociateConnectionFromLag() {
        this.add('directconnect:DisassociateConnectionFromLag');
        return this;
    }
    /**
     * Lists the virtual interface failover test history.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_ListVirtualInterfaceTestHistory.html
     */
    toListVirtualInterfaceTestHistory() {
        this.add('directconnect:ListVirtualInterfaceTestHistory');
        return this;
    }
    /**
     * Starts the virtual interface failover test that verifies your configuration meets your resiliency requirements by placing the BGP peering session in the DOWN state. You can then send traffic to verify that there are no outages.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_StartBgpFailoverTest.html
     */
    toStartBgpFailoverTest() {
        this.add('directconnect:StartBgpFailoverTest');
        return this;
    }
    /**
     * Stops the virtual interface failover test.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_StopBgpFailoverTest.html
     */
    toStopBgpFailoverTest() {
        this.add('directconnect:StopBgpFailoverTest');
        return this;
    }
    /**
     * Adds the specified tags to the specified AWS Direct Connect resource. Each resource can have a maximum of 50 tags.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        this.add('directconnect:TagResource');
        return this;
    }
    /**
     * Removes one or more tags from the specified AWS Direct Connect resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        this.add('directconnect:UntagResource');
        return this;
    }
    /**
     * Updates the specified attributes of the Direct Connect gateway association.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_UpdateDirectConnectGatewayAssociation.html
     */
    toUpdateDirectConnectGatewayAssociation() {
        this.add('directconnect:UpdateDirectConnectGatewayAssociation');
        return this;
    }
    /**
     * Updates the attributes of the specified link aggregation group (LAG).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_UpdateLag.html
     */
    toUpdateLag() {
        this.add('directconnect:UpdateLag');
        return this;
    }
    /**
     * Updates the specified attributes of the specified virtual private interface.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_UpdateVirtualInterfaceAttributes.html
     */
    toUpdateVirtualInterfaceAttributes() {
        this.add('directconnect:UpdateVirtualInterfaceAttributes');
        return this;
    }
    /**
     * Adds a resource of type dxcon to the statement
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_Connection.html
     *
     * @param connectionId - Identifier for the connectionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDxcon(connectionId, account, region, partition) {
        var arn = 'arn:${Partition}:directconnect:${Region}:${Account}:dxcon/${ConnectionId}';
        arn = arn.replace('${ConnectionId}', connectionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type dxlag to the statement
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_Lag.html
     *
     * @param lagId - Identifier for the lagId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDxlag(lagId, account, region, partition) {
        var arn = 'arn:${Partition}:directconnect:${Region}:${Account}:dxlag/${LagId}';
        arn = arn.replace('${LagId}', lagId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type dxvif to the statement
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_VirtualInterface.html
     *
     * @param virtualInterfaceId - Identifier for the virtualInterfaceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDxvif(virtualInterfaceId, account, region, partition) {
        var arn = 'arn:${Partition}:directconnect:${Region}:${Account}:dxvif/${VirtualInterfaceId}';
        arn = arn.replace('${VirtualInterfaceId}', virtualInterfaceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type dx-gateway to the statement
     *
     * https://docs.aws.amazon.com/directconnect/latest/APIReference/API_DirectConnectGateway.html
     *
     * @param directConnectGatewayId - Identifier for the directConnectGatewayId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onDxGateway(directConnectGatewayId, account, partition) {
        var arn = 'arn:${Partition}:directconnect::${Account}:dx-gateway/${DirectConnectGatewayId}';
        arn = arn.replace('${DirectConnectGatewayId}', directConnectGatewayId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Directconnect = Directconnect;
//# sourceMappingURL=data:application/json;base64,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