"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.MediapackageVod = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [mediapackage-vod](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awselementalmediapackagevod.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class MediapackageVod extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [mediapackage-vod](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awselementalmediapackagevod.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'mediapackage-vod';
        this.actionList = {
            "CreateAsset": {
                "url": "https://docs.aws.amazon.com/mediapackage-vod/latest/apireference/assets.html#assetspost",
                "description": "Grants permission to create an asset in AWS Elemental MediaPackage",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreatePackagingConfiguration": {
                "url": "https://docs.aws.amazon.com/mediapackage-vod/latest/apireference/packaging_configurations.html#packaging_configurationspost",
                "description": "Grants permission to create a packaging configuration in AWS Elemental MediaPackage",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreatePackagingGroup": {
                "url": "https://docs.aws.amazon.com/mediapackage-vod/latest/apireference/packaging_groups.html#packaging_groupspost",
                "description": "Grants permission to create a packaging group in AWS Elemental MediaPackage",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "DeleteAsset": {
                "url": "https://docs.aws.amazon.com/mediapackage-vod/latest/apireference/assets-id.html#assets-iddelete",
                "description": "Grants permission to delete an asset in AWS Elemental MediaPackage",
                "accessLevel": "Write",
                "resourceTypes": {
                    "assets": {
                        "required": true
                    }
                }
            },
            "DeletePackagingConfiguration": {
                "url": "https://docs.aws.amazon.com/mediapackage-vod/latest/apireference/packaging_configurations-id.html#packaging_configurations-iddelete",
                "description": "Grants permission to delete a packaging configuration in AWS Elemental MediaPackage",
                "accessLevel": "Write",
                "resourceTypes": {
                    "packaging-configurations": {
                        "required": true
                    }
                }
            },
            "DeletePackagingGroup": {
                "url": "https://docs.aws.amazon.com/mediapackage-vod/latest/apireference/packaging_groups-id.html#packaging_groups-iddelete",
                "description": "Grants permission to delete a packaging group in AWS Elemental MediaPackage",
                "accessLevel": "Write",
                "resourceTypes": {
                    "packaging-groups": {
                        "required": true
                    }
                }
            },
            "DescribeAsset": {
                "url": "https://docs.aws.amazon.com/mediapackage-vod/latest/apireference/assets-id.html#assets-idget",
                "description": "Grants permission to view the details of an asset in AWS Elemental MediaPackage",
                "accessLevel": "Read",
                "resourceTypes": {
                    "assets": {
                        "required": true
                    }
                }
            },
            "DescribePackagingConfiguration": {
                "url": "https://docs.aws.amazon.com/mediapackage-vod/latest/apireference/packaging_configurations-id.html#packaging_configurations-idget",
                "description": "Grants permission to view the details of a packaging configuration in AWS Elemental MediaPackage",
                "accessLevel": "Read",
                "resourceTypes": {
                    "packaging-configurations": {
                        "required": true
                    }
                }
            },
            "DescribePackagingGroup": {
                "url": "https://docs.aws.amazon.com/mediapackage-vod/latest/apireference/packaging_groups-id.html#packaging_groups-idget",
                "description": "Grants permission to view the details of a packaging group in AWS Elemental MediaPackage",
                "accessLevel": "Read",
                "resourceTypes": {
                    "packaging-groups": {
                        "required": true
                    }
                }
            },
            "ListAssets": {
                "url": "https://docs.aws.amazon.com/mediapackage-vod/latest/apireference/assets.html#assetsget",
                "description": "Grants permission to view a list of assets in AWS Elemental MediaPackage",
                "accessLevel": "List"
            },
            "ListPackagingConfigurations": {
                "url": "https://docs.aws.amazon.com/mediapackage-vod/latest/apireference/packaging_configurations.html#packaging_configurationsget",
                "description": "Grants permission to view a list of packaging configurations in AWS Elemental MediaPackage",
                "accessLevel": "List"
            },
            "ListPackagingGroups": {
                "url": "https://docs.aws.amazon.com/mediapackage-vod/latest/apireference/packaging_groups.html#packaging_groupsget",
                "description": "Grants permission to view a list of packaging groups in AWS Elemental MediaPackage",
                "accessLevel": "List"
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/mediapackage-vod/latest/apireference/tags-resource-arn.html#tags-resource-arnget",
                "description": "Grants permission to list the tags assigned to a PackagingGroup, PackagingConfiguration, or Asset.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "assets": {
                        "required": false
                    },
                    "packaging-configurations": {
                        "required": false
                    },
                    "packaging-groups": {
                        "required": false
                    }
                }
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/mediapackage-vod/latest/apireference/tags-resource-arn.html#tags-resource-arnpost",
                "description": "Grants permission to assign tags to a PackagingGroup, PackagingConfiguration, or Asset.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "assets": {
                        "required": false
                    },
                    "packaging-configurations": {
                        "required": false
                    },
                    "packaging-groups": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/mediapackage-vod/latest/apireference/tags-resource-arn.html#tags-resource-arndelete",
                "description": "Grants permission to delete tags from a PackagingGroup, PackagingConfiguration, or Asset.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "assets": {
                        "required": false
                    },
                    "packaging-configurations": {
                        "required": false
                    },
                    "packaging-groups": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            }
        };
        this.resourceTypes = {
            "assets": {
                "name": "assets",
                "url": "https://docs.aws.amazon.com/mediapackage/latest/ug/asset.html",
                "arn": "arn:${Partition}:mediapackage-vod:${Region}:${Account}:assets/${AssetIdentifier}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "packaging-configurations": {
                "name": "packaging-configurations",
                "url": "https://docs.aws.amazon.com/mediapackage/latest/ug/pkg-cfig.html",
                "arn": "arn:${Partition}:mediapackage-vod:${Region}:${Account}:packaging-configurations/${PackagingConfigurationIdentifier}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "packaging-groups": {
                "name": "packaging-groups",
                "url": "https://docs.aws.amazon.com/mediapackage/latest/ug/pkg-group.html",
                "arn": "arn:${Partition}:mediapackage-vod:${Region}:${Account}:packaging-groups/${PackagingGroupIdentifier}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Grants permission to create an asset in AWS Elemental MediaPackage
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/mediapackage-vod/latest/apireference/assets.html#assetspost
     */
    toCreateAsset() {
        this.add('mediapackage-vod:CreateAsset');
        return this;
    }
    /**
     * Grants permission to create a packaging configuration in AWS Elemental MediaPackage
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/mediapackage-vod/latest/apireference/packaging_configurations.html#packaging_configurationspost
     */
    toCreatePackagingConfiguration() {
        this.add('mediapackage-vod:CreatePackagingConfiguration');
        return this;
    }
    /**
     * Grants permission to create a packaging group in AWS Elemental MediaPackage
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/mediapackage-vod/latest/apireference/packaging_groups.html#packaging_groupspost
     */
    toCreatePackagingGroup() {
        this.add('mediapackage-vod:CreatePackagingGroup');
        return this;
    }
    /**
     * Grants permission to delete an asset in AWS Elemental MediaPackage
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediapackage-vod/latest/apireference/assets-id.html#assets-iddelete
     */
    toDeleteAsset() {
        this.add('mediapackage-vod:DeleteAsset');
        return this;
    }
    /**
     * Grants permission to delete a packaging configuration in AWS Elemental MediaPackage
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediapackage-vod/latest/apireference/packaging_configurations-id.html#packaging_configurations-iddelete
     */
    toDeletePackagingConfiguration() {
        this.add('mediapackage-vod:DeletePackagingConfiguration');
        return this;
    }
    /**
     * Grants permission to delete a packaging group in AWS Elemental MediaPackage
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediapackage-vod/latest/apireference/packaging_groups-id.html#packaging_groups-iddelete
     */
    toDeletePackagingGroup() {
        this.add('mediapackage-vod:DeletePackagingGroup');
        return this;
    }
    /**
     * Grants permission to view the details of an asset in AWS Elemental MediaPackage
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediapackage-vod/latest/apireference/assets-id.html#assets-idget
     */
    toDescribeAsset() {
        this.add('mediapackage-vod:DescribeAsset');
        return this;
    }
    /**
     * Grants permission to view the details of a packaging configuration in AWS Elemental MediaPackage
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediapackage-vod/latest/apireference/packaging_configurations-id.html#packaging_configurations-idget
     */
    toDescribePackagingConfiguration() {
        this.add('mediapackage-vod:DescribePackagingConfiguration');
        return this;
    }
    /**
     * Grants permission to view the details of a packaging group in AWS Elemental MediaPackage
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediapackage-vod/latest/apireference/packaging_groups-id.html#packaging_groups-idget
     */
    toDescribePackagingGroup() {
        this.add('mediapackage-vod:DescribePackagingGroup');
        return this;
    }
    /**
     * Grants permission to view a list of assets in AWS Elemental MediaPackage
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mediapackage-vod/latest/apireference/assets.html#assetsget
     */
    toListAssets() {
        this.add('mediapackage-vod:ListAssets');
        return this;
    }
    /**
     * Grants permission to view a list of packaging configurations in AWS Elemental MediaPackage
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mediapackage-vod/latest/apireference/packaging_configurations.html#packaging_configurationsget
     */
    toListPackagingConfigurations() {
        this.add('mediapackage-vod:ListPackagingConfigurations');
        return this;
    }
    /**
     * Grants permission to view a list of packaging groups in AWS Elemental MediaPackage
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mediapackage-vod/latest/apireference/packaging_groups.html#packaging_groupsget
     */
    toListPackagingGroups() {
        this.add('mediapackage-vod:ListPackagingGroups');
        return this;
    }
    /**
     * Grants permission to list the tags assigned to a PackagingGroup, PackagingConfiguration, or Asset.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediapackage-vod/latest/apireference/tags-resource-arn.html#tags-resource-arnget
     */
    toListTagsForResource() {
        this.add('mediapackage-vod:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to assign tags to a PackagingGroup, PackagingConfiguration, or Asset.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/mediapackage-vod/latest/apireference/tags-resource-arn.html#tags-resource-arnpost
     */
    toTagResource() {
        this.add('mediapackage-vod:TagResource');
        return this;
    }
    /**
     * Grants permission to delete tags from a PackagingGroup, PackagingConfiguration, or Asset.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/mediapackage-vod/latest/apireference/tags-resource-arn.html#tags-resource-arndelete
     */
    toUntagResource() {
        this.add('mediapackage-vod:UntagResource');
        return this;
    }
    /**
     * Adds a resource of type assets to the statement
     *
     * https://docs.aws.amazon.com/mediapackage/latest/ug/asset.html
     *
     * @param assetIdentifier - Identifier for the assetIdentifier.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAssets(assetIdentifier, account, region, partition) {
        var arn = 'arn:${Partition}:mediapackage-vod:${Region}:${Account}:assets/${AssetIdentifier}';
        arn = arn.replace('${AssetIdentifier}', assetIdentifier);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type packaging-configurations to the statement
     *
     * https://docs.aws.amazon.com/mediapackage/latest/ug/pkg-cfig.html
     *
     * @param packagingConfigurationIdentifier - Identifier for the packagingConfigurationIdentifier.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onPackagingConfigurations(packagingConfigurationIdentifier, account, region, partition) {
        var arn = 'arn:${Partition}:mediapackage-vod:${Region}:${Account}:packaging-configurations/${PackagingConfigurationIdentifier}';
        arn = arn.replace('${PackagingConfigurationIdentifier}', packagingConfigurationIdentifier);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type packaging-groups to the statement
     *
     * https://docs.aws.amazon.com/mediapackage/latest/ug/pkg-group.html
     *
     * @param packagingGroupIdentifier - Identifier for the packagingGroupIdentifier.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onPackagingGroups(packagingGroupIdentifier, account, region, partition) {
        var arn = 'arn:${Partition}:mediapackage-vod:${Region}:${Account}:packaging-groups/${PackagingGroupIdentifier}';
        arn = arn.replace('${PackagingGroupIdentifier}', packagingGroupIdentifier);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.MediapackageVod = MediapackageVod;
//# sourceMappingURL=data:application/json;base64,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